import logging
import sys


def get_dev_logger(name='dev.radosgw_agent'):
    """
    A simple utility to be able to log things that are meant for developer-eyes
    and not for user facing.

    All developer logs must be prepended with `dev` so this utility ensures
    that is the case. To use it::

        dev_log = get_dev_logger(__name__)

    Or::

        dev_log = get_dev_logger('dev.custom_name')
    """
    if not name.startswith('dev'):
        return logging.getLogger('%s.%s' % ('dev', name))
    return logging.getLogger(name)


BLACK, RED, GREEN, YELLOW, BLUE, MAGENTA, CYAN, WHITE = range(8)

COLORS = {
    'WARNING': YELLOW,
    'INFO': WHITE,
    'DEBUG': BLUE,
    'CRITICAL': RED,
    'ERROR': RED,
    'FATAL': RED,
}

RESET_SEQ = "\033[0m"
COLOR_SEQ = "\033[1;%dm"
BOLD_SEQ = "\033[1m"

BASE_COLOR_FORMAT = "%(asctime)s %(process)d\
 [$BOLD%(name)s$RESET][%(color_levelname)-17s] %(message)s"

BASE_FORMAT = "%(asctime)s %(process)d [%(name)s][%(levelname)-6s] %(message)s"


def supports_color():
    """
    Returns True if the running system's terminal supports color, and False
    otherwise.
    """
    unsupported_platform = (sys.platform in ('win32', 'Pocket PC'))
    # isatty is not always implemented
    is_a_tty = hasattr(sys.stdout, 'isatty') and sys.stdout.isatty()
    if unsupported_platform or not is_a_tty:
        return False
    return True


def color_message(message):
    message = message.replace("$RESET", RESET_SEQ).replace("$BOLD", BOLD_SEQ)
    return message


class ColoredFormatter(logging.Formatter):
    """
    A very basic logging formatter that not only applies color to the levels of
    the ouput but will also truncate the level names so that they do not alter
    the visuals of logging when presented on the terminal.
    """

    def __init__(self, msg):
        logging.Formatter.__init__(self, msg)

    def format(self, record):
        levelname = record.levelname
        truncated_level = record.levelname[:6]
        levelname_color = COLOR_SEQ % (30 + COLORS[levelname]) + truncated_level + RESET_SEQ
        record.color_levelname = levelname_color
        return logging.Formatter.format(self, record)


def color_format():
    """
    Main entry point to get a colored formatter, it will use the
    BASE_FORMAT by default and fall back to no colors if the system
    does not support it
    """
    str_format = BASE_COLOR_FORMAT if supports_color() else BASE_FORMAT
    color_format = color_message(str_format)
    return ColoredFormatter(color_format)
