// -*- mode:C++; tab-width:8; c-basic-offset:2; indent-tabs-mode:t -*-
// vim: ts=8 sw=2 smarttab

#include <random>

#include <boost/iterator/counting_iterator.hpp>

#include "test/crimson/gtest_seastar.h"
#include "test/crimson/seastore/transaction_manager_test_state.h"

#include "crimson/os/seastore/cache.h"
#include "crimson/os/seastore/transaction_manager.h"
#include "crimson/os/seastore/segment_manager/ephemeral.h"
#include "crimson/os/seastore/segment_manager.h"

#include "test/crimson/seastore/test_block.h"

using namespace crimson;
using namespace crimson::os;
using namespace crimson::os::seastore;

namespace {
  [[maybe_unused]] seastar::logger& logger() {
    return crimson::get_logger(ceph_subsys_test);
  }
}

struct test_extent_record_t {
  test_extent_desc_t desc;
  unsigned refcount = 0;
  test_extent_record_t() = default;
  test_extent_record_t(
    const test_extent_desc_t &desc,
    unsigned refcount) : desc(desc), refcount(refcount) {}

  void update(const test_extent_desc_t &to) {
    desc = to;
  }

  bool operator==(const test_extent_desc_t &rhs) const {
    return desc == rhs;
  }
  bool operator!=(const test_extent_desc_t &rhs) const {
    return desc != rhs;
  }
};

template<>
struct fmt::formatter<test_extent_record_t> : fmt::formatter<std::string_view> {
  template <typename FormatContext>
  auto format(const test_extent_record_t& r, FormatContext& ctx) const {
    return fmt::format_to(ctx.out(), "test_extent_record_t({}, refcount={})",
			  r.desc, r.refcount);
  }
};

struct transaction_manager_test_t :
  public seastar_test_suite_t,
  TMTestState,
  ::testing::WithParamInterface<const char*> {

  std::random_device rd;
  std::mt19937 gen;

  transaction_manager_test_t(std::size_t num_main_devices, std::size_t num_cold_devices)
    : TMTestState(num_main_devices, num_cold_devices), gen(rd()) {
  }

  laddr_t get_random_laddr(size_t block_size, laddr_t limit) {
    return block_size *
      std::uniform_int_distribution<>(0, (limit / block_size) - 1)(gen);
  }

  char get_random_contents() {
    return static_cast<char>(std::uniform_int_distribution<>(0, 255)(gen));
  }

  seastar::future<> set_up_fut() final {
    std::string j_type = GetParam();
    if (j_type == "segmented") {
      return tm_setup(journal_type_t::SEGMENTED);
    } else if (j_type == "circularbounded") {
      return tm_setup(journal_type_t::RANDOM_BLOCK);
    } else {
      ceph_assert(0 == "no support");
    }
  }

  seastar::future<> tear_down_fut() final {
    return tm_teardown();
  }

  struct test_extents_t : std::map<laddr_t, test_extent_record_t> {
    using delta_t = std::map<laddr_t, std::optional<test_extent_record_t>>;
    std::map<laddr_t, uint64_t> laddr_write_seq;

    struct delta_overlay_t {
      const test_extents_t &extents;
      const delta_t &delta;

      delta_overlay_t(
	const test_extents_t &extents,
	const delta_t &delta)
	: extents(extents), delta(delta) {}


      class iterator {
	friend class test_extents_t;

	const delta_overlay_t &parent;
	test_extents_t::const_iterator biter;
	delta_t::const_iterator oiter;
	std::optional<std::pair<laddr_t, test_extent_record_t>> cur;

	iterator(
	  const delta_overlay_t &parent,
	  test_extents_t::const_iterator biter,
	  delta_t::const_iterator oiter)
	  : parent(parent), biter(biter), oiter(oiter) {}

	laddr_t get_bkey() {
	  return biter == parent.extents.end() ? L_ADDR_MAX : biter->first;
	}

	laddr_t get_okey() {
	  return oiter == parent.delta.end() ? L_ADDR_MAX : oiter->first;
	}

	bool is_end() {
	  return oiter == parent.delta.end() && biter == parent.extents.end();
	}

	bool is_valid() {
	  return is_end() ||
	    ((get_okey() < get_bkey()) && (oiter->second)) ||
	    (get_okey() > get_bkey());
	}

	auto get_pair() {
	  assert(is_valid());
	  assert(!is_end());
	  auto okey = get_okey();
	  auto bkey = get_bkey();
	  return (
	    bkey < okey ?
	    std::pair<laddr_t, test_extent_record_t>(*biter) :
	    std::make_pair(okey, *(oiter->second)));
	}

	void adjust() {
	  while (!is_valid()) {
	    if (get_okey() < get_bkey()) {
	      assert(!oiter->second);
	      ++oiter;
	    } else {
	      assert(get_okey() == get_bkey());
	      ++biter;
	    }
	  }
	  assert(is_valid());
	  if (!is_end()) {
	    cur = get_pair();
	  } else {
	    cur = std::nullopt;
	  }
	}

      public:
	iterator(const iterator &) = default;
	iterator(iterator &&) = default;

	iterator &operator++() {
	  assert(is_valid());
	  assert(!is_end());
	  if (get_bkey() < get_okey()) {
	    ++biter;
	  } else {
	    ++oiter;
	  }
	  adjust();
	  return *this;
	}

	bool operator==(const iterator &o) const {
	  return o.biter == biter && o.oiter == oiter;
	}
	bool operator!=(const iterator &o) const {
	  return !(*this == o);
	}

	auto operator*() {
	  assert(!is_end());
	  return *cur;
	}
	auto operator->() {
	  assert(!is_end());
	  return &*cur;
	}
      };

      iterator begin() {
	auto ret = iterator{*this, extents.begin(), delta.begin()};
	ret.adjust();
	return ret;
      }

      iterator end() {
	auto ret = iterator{*this, extents.end(), delta.end()};
	// adjust unnecessary
	return ret;
      }

      iterator lower_bound(laddr_t l) {
	auto ret = iterator{*this, extents.lower_bound(l), delta.lower_bound(l)};
	ret.adjust();
	return ret;
      }

      iterator upper_bound(laddr_t l) {
	auto ret = iterator{*this, extents.upper_bound(l), delta.upper_bound(l)};
	ret.adjust();
	return ret;
      }

      iterator find(laddr_t l) {
	auto ret = lower_bound(l);
	if (ret == end() || ret->first != l) {
	  return end();
	} else {
	  return ret;
	}
      }
    };
  private:
    void check_available(
      laddr_t addr, extent_len_t len, const delta_t &delta
    ) const {
      delta_overlay_t overlay(*this, delta);
      for (const auto &i: overlay) {
	if (i.first < addr) {
	  EXPECT_FALSE(i.first + i.second.desc.len > addr);
	} else {
	  EXPECT_FALSE(addr + len > i.first);
	}
      }
    }

    void check_hint(
      laddr_t hint,
      laddr_t addr,
      extent_len_t len,
      delta_t &delta) const {
      delta_overlay_t overlay(*this, delta);
      auto iter = overlay.lower_bound(hint);
      laddr_t last = hint;
      while (true) {
	if (iter == overlay.end() || iter->first > addr) {
	  EXPECT_EQ(addr, last);
	  break;
	}
	EXPECT_FALSE(iter->first - last > len);
	last = iter->first + iter->second.desc.len;
	++iter;
      }
    }

    std::optional<test_extent_record_t> &populate_delta(
      laddr_t addr, delta_t &delta, const test_extent_desc_t *desc) const {
      auto diter = delta.find(addr);
      if (diter != delta.end())
	return diter->second;

      auto iter = find(addr);
      if (iter == end()) {
	assert(desc);
	auto ret = delta.emplace(
	  std::make_pair(addr, test_extent_record_t{*desc, 0}));
	assert(ret.second);
	return ret.first->second;
      } else {
	auto ret = delta.emplace(*iter);
	assert(ret.second);
	return ret.first->second;
      }
    }
  public:
    delta_overlay_t get_overlay(const delta_t &delta) const {
      return delta_overlay_t{*this, delta};
    }

    void insert(TestBlock &extent, delta_t &delta) const {
      check_available(extent.get_laddr(), extent.get_length(), delta);
      delta[extent.get_laddr()] =
	test_extent_record_t{extent.get_desc(), 1};
    }

    void alloced(laddr_t hint, TestBlock &extent, delta_t &delta) const {
      check_hint(hint, extent.get_laddr(), extent.get_length(), delta);
      insert(extent, delta);
    }

    bool contains(laddr_t addr, const delta_t &delta) const {
      delta_overlay_t overlay(*this, delta);
      return overlay.find(addr) != overlay.end();
    }

    test_extent_record_t get(laddr_t addr, const delta_t &delta) const {
      delta_overlay_t overlay(*this, delta);
      auto iter = overlay.find(addr);
      assert(iter != overlay.end());
      return iter->second;
    }

    void update(
      laddr_t addr,
      const test_extent_desc_t &desc,
      delta_t &delta) const {
      auto &rec = populate_delta(addr, delta, &desc);
      assert(rec);
      rec->desc = desc;
    }

    int inc_ref(
      laddr_t addr,
      delta_t &delta) const {
      auto &rec = populate_delta(addr, delta, nullptr);
      assert(rec);
      return ++rec->refcount;
    }

    int dec_ref(
      laddr_t addr,
      delta_t &delta) const {
      auto &rec = populate_delta(addr, delta, nullptr);
      assert(rec);
      assert(rec->refcount > 0);
      rec->refcount--;
      if (rec->refcount == 0) {
	delta[addr] = std::nullopt;
	return 0;
      } else {
	return rec->refcount;
      }
    }

    void consume(const delta_t &delta, const uint64_t write_seq = 0) {
      for (const auto &i : delta) {
	if (i.second) {
	  if (laddr_write_seq.find(i.first) == laddr_write_seq.end() ||
	      laddr_write_seq[i.first] <= write_seq) {
	    (*this)[i.first] = *i.second;
	    laddr_write_seq[i.first] = write_seq;
	  }
	} else {
	  erase(i.first);
	}
      }
    }

  } test_mappings;

  struct test_transaction_t {
    TransactionRef t;
    test_extents_t::delta_t mapping_delta;
  };

  test_transaction_t create_transaction() {
    return { create_mutate_transaction(), {} };
  }

  test_transaction_t create_read_test_transaction() {
    return {create_read_transaction(), {} };
  }

  test_transaction_t create_weak_test_transaction() {
    return { create_weak_transaction(), {} };
  }

  TestBlockRef alloc_extent(
    test_transaction_t &t,
    laddr_t hint,
    extent_len_t len,
    char contents) {
    auto extent = with_trans_intr(*(t.t), [&](auto& trans) {
      return tm->alloc_extent<TestBlock>(trans, hint, len);
    }).unsafe_get0();
    extent->set_contents(contents);
    EXPECT_FALSE(test_mappings.contains(extent->get_laddr(), t.mapping_delta));
    EXPECT_EQ(len, extent->get_length());
    test_mappings.alloced(hint, *extent, t.mapping_delta);
    return extent;
  }

  TestBlockRef alloc_extent(
    test_transaction_t &t,
    laddr_t hint,
    extent_len_t len) {
    return alloc_extent(
      t,
      hint,
      len,
      get_random_contents());
  }

  bool check_usage() {
    return epm->check_usage();
  }

  void replay() {
    EXPECT_TRUE(check_usage());
    restart();
  }

  void check() {
    check_mappings();
    check_usage();
  }

  void check_mappings() {
    auto t = create_weak_test_transaction();
    check_mappings(t);
  }

  TestBlockRef get_extent(
    test_transaction_t &t,
    laddr_t addr,
    extent_len_t len) {
    ceph_assert(test_mappings.contains(addr, t.mapping_delta));
    ceph_assert(test_mappings.get(addr, t.mapping_delta).desc.len == len);

    auto ext = with_trans_intr(*(t.t), [&](auto& trans) {
      return tm->read_extent<TestBlock>(trans, addr, len);
    }).unsafe_get0();
    EXPECT_EQ(addr, ext->get_laddr());
    return ext;
  }

  TestBlockRef try_get_extent(
    test_transaction_t &t,
    laddr_t addr) {
    ceph_assert(test_mappings.contains(addr, t.mapping_delta));

    using ertr = with_trans_ertr<TransactionManager::read_extent_iertr>;
    using ret = ertr::future<TestBlockRef>;
    auto ext = with_trans_intr(*(t.t), [&](auto& trans) {
      return tm->read_extent<TestBlock>(trans, addr);
    }).safe_then([](auto ext) -> ret {
      return ertr::make_ready_future<TestBlockRef>(ext);
    }).handle_error(
      [](const crimson::ct_error::eagain &e) {
	return seastar::make_ready_future<TestBlockRef>();
      },
      crimson::ct_error::assert_all{
	"get_extent got invalid error"
      }
    ).get0();
    if (ext) {
      EXPECT_EQ(addr, ext->get_laddr());
    }
    return ext;
  }

  TestBlockRef try_get_extent(
    test_transaction_t &t,
    laddr_t addr,
    extent_len_t len) {
    ceph_assert(test_mappings.contains(addr, t.mapping_delta));
    ceph_assert(test_mappings.get(addr, t.mapping_delta).desc.len == len);

    using ertr = with_trans_ertr<TransactionManager::read_extent_iertr>;
    using ret = ertr::future<TestBlockRef>;
    auto ext = with_trans_intr(*(t.t), [&](auto& trans) {
      return tm->read_extent<TestBlock>(trans, addr, len);
    }).safe_then([](auto ext) -> ret {
      return ertr::make_ready_future<TestBlockRef>(ext);
    }).handle_error(
      [](const crimson::ct_error::eagain &e) {
	return seastar::make_ready_future<TestBlockRef>();
      },
      crimson::ct_error::assert_all{
	"get_extent got invalid error"
      }
    ).get0();
    if (ext) {
      EXPECT_EQ(addr, ext->get_laddr());
    }
    return ext;
  }

  test_block_mutator_t mutator;
  TestBlockRef mutate_extent(
    test_transaction_t &t,
    TestBlockRef ref) {
    ceph_assert(test_mappings.contains(ref->get_laddr(), t.mapping_delta));
    ceph_assert(
      test_mappings.get(ref->get_laddr(), t.mapping_delta).desc.len ==
      ref->get_length());

    auto ext = tm->get_mutable_extent(*t.t, ref)->cast<TestBlock>();
    EXPECT_EQ(ext->get_laddr(), ref->get_laddr());
    EXPECT_EQ(ext->get_desc(), ref->get_desc());
    mutator.mutate(*ext, gen);

    test_mappings.update(ext->get_laddr(), ext->get_desc(), t.mapping_delta);
    return ext;
  }

  TestBlockRef mutate_addr(
    test_transaction_t &t,
    laddr_t offset,
    size_t length) {
    auto ext = get_extent(t, offset, length);
    mutate_extent(t, ext);
    return ext;
  }

  void inc_ref(test_transaction_t &t, laddr_t offset) {
    ceph_assert(test_mappings.contains(offset, t.mapping_delta));
    ceph_assert(test_mappings.get(offset, t.mapping_delta).refcount > 0);

    auto refcnt = with_trans_intr(*(t.t), [&](auto& trans) {
      return tm->inc_ref(trans, offset);
    }).unsafe_get0();
    auto check_refcnt = test_mappings.inc_ref(offset, t.mapping_delta);
    EXPECT_EQ(refcnt, check_refcnt);
  }

  void dec_ref(test_transaction_t &t, laddr_t offset) {
    ceph_assert(test_mappings.contains(offset, t.mapping_delta));
    ceph_assert(test_mappings.get(offset, t.mapping_delta).refcount > 0);

    auto refcnt = with_trans_intr(*(t.t), [&](auto& trans) {
      return tm->dec_ref(trans, offset);
    }).unsafe_get0();
    auto check_refcnt = test_mappings.dec_ref(offset, t.mapping_delta);
    EXPECT_EQ(refcnt, check_refcnt);
    if (refcnt == 0)
      logger().debug("dec_ref: {} at refcount 0", offset);
  }

  void check_mappings(test_transaction_t &t) {
    auto overlay = test_mappings.get_overlay(t.mapping_delta);
    for (const auto &i: overlay) {
      logger().debug("check_mappings: {}->{}", i.first, i.second);
      auto ext = get_extent(t, i.first, i.second.desc.len);
      EXPECT_EQ(i.second, ext->get_desc());
    }
    with_trans_intr(
      *t.t,
      [this, &overlay](auto &t) {
	return lba_manager->scan_mappings(
	  t,
	  0,
	  L_ADDR_MAX,
	  [iter=overlay.begin(), &overlay](auto l, auto p, auto len) mutable {
	    EXPECT_NE(iter, overlay.end());
	    logger().debug(
	      "check_mappings: scan {}",
	      l);
	    EXPECT_EQ(l, iter->first);
	    ++iter;
	  });
      }).unsafe_get0();
    (void)with_trans_intr(
      *t.t,
      [=, this](auto &t) {
	return lba_manager->check_child_trackers(t);
      }).unsafe_get0();
  }

  bool try_submit_transaction(test_transaction_t t) {
    using ertr = with_trans_ertr<TransactionManager::submit_transaction_iertr>;
    using ret = ertr::future<bool>;
    uint64_t write_seq = 0;
    bool success = submit_transaction_fut_with_seq(*t.t
    ).safe_then([&write_seq](auto seq) -> ret {
      write_seq = seq;
      return ertr::make_ready_future<bool>(true);
    }).handle_error(
      [](const crimson::ct_error::eagain &e) {
	return seastar::make_ready_future<bool>(false);
      },
      crimson::ct_error::assert_all{
	"try_submit_transaction hit invalid error"
      }
    ).then([this](auto ret) {
      return epm->run_background_work_until_halt(
      ).then([ret] { return ret; });
    }).get0();

    if (success) {
      test_mappings.consume(t.mapping_delta, write_seq);
    }

    return success;
  }

  void submit_transaction(test_transaction_t &&t) {
    bool success = try_submit_transaction(std::move(t));
    EXPECT_TRUE(success);
  }

  void submit_transaction_expect_conflict(test_transaction_t &&t) {
    bool success = try_submit_transaction(std::move(t));
    EXPECT_FALSE(success);
  }

  auto allocate_sequentially(const size_t size, const int num, bool run_clean = true) {
    return repeat_eagain([this, size, num] {
      return seastar::do_with(
	create_transaction(),
	[this, size, num](auto &t) {
	  return with_trans_intr(
	    *t.t,
	    [&t, this, size, num](auto &) {
	      return trans_intr::do_for_each(
		boost::make_counting_iterator(0),
		boost::make_counting_iterator(num),
		[&t, this, size](auto) {
		  return tm->alloc_extent<TestBlock>(
		    *(t.t), L_ADDR_MIN, size
		  ).si_then([&t, this, size](auto extent) {
		    extent->set_contents(get_random_contents());
		    EXPECT_FALSE(
		      test_mappings.contains(extent->get_laddr(), t.mapping_delta));
		    EXPECT_EQ(size, extent->get_length());
		    test_mappings.alloced(extent->get_laddr(), *extent, t.mapping_delta);
		    return seastar::now();
		  });
		}).si_then([&t, this] {
		  return tm->submit_transaction(*t.t);
		});
	    }).safe_then([&t, this] {
	      test_mappings.consume(t.mapping_delta);
	    });
	});
    }).safe_then([this, run_clean]() {
      if (run_clean) {
        return epm->run_background_work_until_halt();
      } else {
        return epm->background_process.trimmer->trim();
      }
    }).handle_error(
      crimson::ct_error::assert_all{
	"Invalid error in SeaStore::list_collections"
      }
    );
  }

  void test_parallel_extent_read() {
    constexpr size_t TOTAL = 4<<20;
    constexpr size_t BSIZE = 4<<10;
    constexpr size_t BLOCKS = TOTAL / BSIZE;
    run_async([this] {
      for (unsigned i = 0; i < BLOCKS; ++i) {
	auto t = create_transaction();
	auto extent = alloc_extent(
	  t,
	  i * BSIZE,
	  BSIZE);
	ASSERT_EQ(i * BSIZE, extent->get_laddr());
	submit_transaction(std::move(t));
      }

      seastar::do_with(
	create_read_test_transaction(),
	[this](auto &t) {
	return with_trans_intr(*(t.t), [this](auto &t) {
	  return trans_intr::parallel_for_each(
	    boost::make_counting_iterator(0lu),
	    boost::make_counting_iterator(BLOCKS),
	    [this, &t](auto i) {
	    return tm->read_extent<TestBlock>(t, i * BSIZE, BSIZE
	    ).si_then([](auto) {
	      return seastar::now();
	    });
	  });
	});
      }).unsafe_get0();
    });
  }

  void test_random_writes_concurrent() {
    constexpr unsigned WRITE_STREAMS = 256;

    constexpr size_t TOTAL = 4<<20;
    constexpr size_t BSIZE = 4<<10;
    constexpr size_t BLOCKS = TOTAL / BSIZE;
    run_async([this] {
      std::for_each(
        boost::make_counting_iterator(0u),
        boost::make_counting_iterator(WRITE_STREAMS),
        [&](auto idx) {
          for (unsigned i = idx; i < BLOCKS; i += WRITE_STREAMS) {
            while (true) {
              auto t = create_transaction();
              auto extent = alloc_extent(
                t,
                i * BSIZE,
                BSIZE);
              ASSERT_EQ(i * BSIZE, extent->get_laddr());
              if (try_submit_transaction(std::move(t)))
                break;
            }
          }
        });

      int writes = 0;
      unsigned failures = 0;
      seastar::parallel_for_each(
        boost::make_counting_iterator(0u),
        boost::make_counting_iterator(WRITE_STREAMS),
        [&](auto) {
          return seastar::async([&] {
            while (writes < 300) {
              auto t = create_transaction();
              auto ext = try_get_extent(
                t,
                get_random_laddr(BSIZE, TOTAL),
                BSIZE);
              if (!ext){
                failures++;
                continue;
              }
              auto mut = mutate_extent(t, ext);
              auto success = try_submit_transaction(std::move(t));
              writes += success;
              failures += !success;
            }
          });
        }).get0();
      replay();
      logger().info("random_writes_concurrent: checking");
      check();
      logger().info(
        "random_writes_concurrent: {} suceeded, {} failed",
        writes,
        failures
      );
    });
  }

  void test_evict() {
    // only support segmented backend currently
    ASSERT_EQ(epm->get_main_backend_type(), backend_type_t::SEGMENTED);
    ASSERT_TRUE(epm->background_process.has_cold_tier());
    constexpr size_t device_size =
      segment_manager::DEFAULT_TEST_EPHEMERAL.size;
    constexpr size_t block_size =
      segment_manager::DEFAULT_TEST_EPHEMERAL.block_size;
    constexpr size_t segment_size =
      segment_manager::DEFAULT_TEST_EPHEMERAL.segment_size;
    ASSERT_GE(segment_size, block_size * 20);

    run_async([this] {
      // indicates there is no available segments to reclaim
      double stop_ratio = (double)segment_size / (double)device_size / 2;
      // 1 segment
      double default_ratio = stop_ratio * 2;
      // 1.25 segment
      double fast_ratio = stop_ratio * 2.5;

      epm->background_process
        .eviction_state
        .init(stop_ratio, default_ratio, fast_ratio);

      // these variables are described in
      // EPM::BackgroundProcess::eviction_state_t::maybe_update_eviction_mode
      size_t ratio_A_size = segment_size / 2 - block_size * 10;
      size_t ratio_B_size = segment_size / 2 + block_size * 10;
      size_t ratio_C_size = segment_size + block_size;
      size_t ratio_D_size = segment_size * 1.25 + block_size;

      auto run_until = [this](size_t size) -> seastar::future<> {
        return seastar::repeat([this, size] {
          size_t current_size = epm->background_process
                                    .main_cleaner->get_stat().data_stored;
          if (current_size >= size) {
            return seastar::futurize_invoke([] {
              return seastar::stop_iteration::yes;
            });
          } else {
            int num = (size - current_size) / block_size;
            return seastar::do_for_each(
              boost::make_counting_iterator(0),
              boost::make_counting_iterator(num),
              [this](auto) {
	        // don't start background process to test the behavior
                // of generation changes during alloc new extents
                return allocate_sequentially(block_size, 1, false);
              }).then([] {
                return seastar::stop_iteration::no;
              });
          }
        });
      };

      std::vector<extent_types_t> all_extent_types{
        extent_types_t::ROOT,
        extent_types_t::LADDR_INTERNAL,
        extent_types_t::LADDR_LEAF,
        extent_types_t::OMAP_INNER,
        extent_types_t::OMAP_LEAF,
        extent_types_t::ONODE_BLOCK_STAGED,
        extent_types_t::COLL_BLOCK,
        extent_types_t::OBJECT_DATA_BLOCK,
        extent_types_t::RETIRED_PLACEHOLDER,
        extent_types_t::ALLOC_INFO,
        extent_types_t::JOURNAL_TAIL,
        extent_types_t::TEST_BLOCK,
        extent_types_t::TEST_BLOCK_PHYSICAL,
        extent_types_t::BACKREF_INTERNAL,
        extent_types_t::BACKREF_LEAF
      };

      std::vector<rewrite_gen_t> all_generations;
      for (auto i = INIT_GENERATION; i < REWRITE_GENERATIONS; i++) {
        all_generations.push_back(i);
      }

      // input target-generation -> expected generation after the adjustment
      using generation_mapping_t = std::map<rewrite_gen_t, rewrite_gen_t>;
      std::map<extent_types_t, generation_mapping_t> expected_generations;

      // this loop should be consistent with EPM::adjust_generation
      for (auto t : all_extent_types) {
        expected_generations[t] = {};
        if (!is_logical_type(t)) {
          for (auto gen : all_generations) {
            expected_generations[t][gen] = INLINE_GENERATION;
          }
        } else {
	  if (get_extent_category(t) == data_category_t::METADATA) {
	    expected_generations[t][INIT_GENERATION] = INLINE_GENERATION;
	  } else {
	    expected_generations[t][INIT_GENERATION] = OOL_GENERATION;
	  }

          for (auto i = INIT_GENERATION + 1; i < REWRITE_GENERATIONS; i++) {
	    expected_generations[t][i] = i;
          }
        }
      }

      auto update_data_gen_mapping = [&](std::function<rewrite_gen_t(rewrite_gen_t)> func) {
        for (auto t : all_extent_types) {
          if (!is_logical_type(t)) {
            continue;
          }
          for (auto i = INIT_GENERATION + 1; i < REWRITE_GENERATIONS; i++) {
            expected_generations[t][i] = func(i);
          }
        }
        // since background process didn't start in allocate_sequentially
        // we update eviction mode manually.
        epm->background_process.maybe_update_eviction_mode();
      };

      auto test_gen = [&](const char *caller) {
        for (auto t : all_extent_types) {
          for (auto gen : all_generations) {
            auto epm_gen = epm->adjust_generation(
              get_extent_category(t),
              t,
              placement_hint_t::HOT,
              gen);
            if (expected_generations[t][gen] != epm_gen) {
              logger().error("caller: {}, extent type: {}, input generation: {}, "
			     "expected generation : {}, adjust result from EPM: {}",
			     caller, t, gen, expected_generations[t][gen], epm_gen);
            }
            EXPECT_EQ(expected_generations[t][gen], epm_gen);
          }
        }
      };

      // verify that no data should go to the cold tier
      update_data_gen_mapping([](rewrite_gen_t gen) -> rewrite_gen_t {
        if (gen == MIN_COLD_GENERATION) {
          return MIN_COLD_GENERATION - 1;
        } else {
          return gen;
        }
      });
      test_gen("init");

      run_until(ratio_A_size).get();
      EXPECT_TRUE(epm->background_process.eviction_state.is_stop_mode());
      test_gen("exceed ratio A");
      epm->run_background_work_until_halt().get();

      run_until(ratio_B_size).get();
      EXPECT_TRUE(epm->background_process.eviction_state.is_stop_mode());
      test_gen("exceed ratio B");
      epm->run_background_work_until_halt().get();

      // verify that data may go to the cold tier
      run_until(ratio_C_size).get();
      update_data_gen_mapping([](rewrite_gen_t gen) { return gen; });
      EXPECT_TRUE(epm->background_process.eviction_state.is_default_mode());
      test_gen("exceed ratio C");
      epm->run_background_work_until_halt().get();

      // verify that data must go to the cold tier
      run_until(ratio_D_size).get();
      update_data_gen_mapping([](rewrite_gen_t gen) {
        if (gen >= MIN_REWRITE_GENERATION && gen < MIN_COLD_GENERATION) {
          return MIN_COLD_GENERATION;
        } else {
          return gen;
        }
      });
      EXPECT_TRUE(epm->background_process.eviction_state.is_fast_mode());
      test_gen("exceed ratio D");

      auto main_size = epm->background_process.main_cleaner->get_stat().data_stored;
      auto cold_size = epm->background_process.cold_cleaner->get_stat().data_stored;
      EXPECT_EQ(cold_size, 0);
      epm->run_background_work_until_halt().get();
      auto new_main_size = epm->background_process.main_cleaner->get_stat().data_stored;
      auto new_cold_size = epm->background_process.cold_cleaner->get_stat().data_stored;
      EXPECT_GE(main_size, new_main_size);
      EXPECT_NE(new_cold_size, 0);

      update_data_gen_mapping([](rewrite_gen_t gen) { return gen; });
      EXPECT_TRUE(epm->background_process.eviction_state.is_default_mode());
      test_gen("finish evict");
    });
  }

  std::optional<TestBlockRef> map_existing_extent(
    test_transaction_t &t,
    laddr_t hint,
    paddr_t existing_paddr,
    extent_len_t length) {
    if (t.t->is_conflicted()) {
      return std::nullopt;
    }
    auto extent = with_trans_intr(*(t.t), [&](auto& trans) {
      return tm->map_existing_extent<TestBlock>(trans, hint, existing_paddr, length);
    }).handle_error(crimson::ct_error::eagain::handle([] {
      return TCachedExtentRef<TestBlock>(new TestBlock(0));
    }), crimson::ct_error::pass_further_all{}).unsafe_get0();
    if (t.t->is_conflicted()) {
      return std::nullopt;
    }
    EXPECT_TRUE(extent->get_length() != 0);
    EXPECT_FALSE(test_mappings.contains(extent->get_laddr(), t.mapping_delta));
    EXPECT_EQ(length, extent->get_length());
    test_mappings.alloced(hint, *extent, t.mapping_delta);
    return std::make_optional(std::move(extent));
  }

  void test_map_existing_extent() {
    run_async([this] {
      constexpr size_t offset = 16 << 10;
      constexpr size_t length = 16 << 10;
      {
	auto t = create_transaction();
	auto extent = alloc_extent(t, offset, length);
	submit_transaction(std::move(t));
      }
      {
	auto t = create_transaction();
	auto extent = get_extent(t, offset, length);
	auto base_paddr = extent->get_paddr();
	dec_ref(t, offset);
	auto extent1 = map_existing_extent(t, offset, base_paddr, 4 << 10);
	ASSERT_TRUE(extent1.has_value());
	auto extent2 = map_existing_extent(t, offset + (4 << 10), base_paddr.add_offset(4 << 10), 4 << 10);
	ASSERT_TRUE(extent2.has_value());
	auto extent3 = map_existing_extent(t, offset + (8 << 10), base_paddr.add_offset(8 << 10), 8 << 10);
	ASSERT_TRUE(extent3.has_value());
	ASSERT_TRUE((*extent1)->is_exist_clean());
	ASSERT_TRUE((*extent2)->is_exist_clean());
	ASSERT_TRUE((*extent3)->is_exist_clean());
	auto extent4 = mutate_extent(t, (*extent3));
	ASSERT_TRUE(extent4->is_exist_mutation_pending());
	ASSERT_TRUE((*extent3).get() == extent4.get());
	submit_transaction(std::move(t));
	check();
      }
      replay();
      check();
    });
  }

  void test_map_existing_extent_concurrent() {
    run_async([this] {
      constexpr unsigned REMAP_NUM = 32;
      constexpr size_t offset = 0;
      constexpr size_t length = 256 << 10;
      {
	auto t = create_transaction();
	auto extent = alloc_extent(t, offset, length);
	ASSERT_EQ(length, extent->get_length());
	submit_transaction(std::move(t));
      }
      int success = 0;
      int early_exit = 0;
      int conflicted = 0;

      seastar::parallel_for_each(
        boost::make_counting_iterator(0u),
	boost::make_counting_iterator(REMAP_NUM),
	[&](auto) {
	  return seastar::async([&] {
	    uint32_t pieces = std::uniform_int_distribution<>(1, 31)(gen);
	    std::set<uint32_t> split_points;
	    for (uint32_t i = 0; i < pieces; i++) {
	      auto p = std::uniform_int_distribution<>(1, 256)(gen);
	      split_points.insert(p - p % 4);
	    }

	    auto t = create_transaction();
	    auto ext0 = try_get_extent(t, offset);
	    if (!ext0 || ext0->get_length() != length) {
	      early_exit++;
	      return;
	    }
	    auto paddr = ext0->get_paddr();
	    dec_ref(t, offset);

	    auto base = 0;
	    ASSERT_TRUE(!split_points.empty());
	    for (auto off : split_points) {
	      if (off == 0) {
		continue;
	      }

	      auto ext_ = map_existing_extent(t, base << 10, paddr.add_offset(base << 10), (off - base) << 10);
	      if (!ext_) {
		conflicted++;
		return;
	      }
	      auto ext = *ext_;
	      ASSERT_TRUE(ext->is_exist_clean());
	      if (get_random_contents() % 2 == 0) {
		auto ext1 = mutate_extent(t, ext);
		ASSERT_TRUE(ext1->is_exist_mutation_pending());
	      }
	      base = off;
	    }

	    base <<= 10;
	    if (base != length) {
	      auto ext_ = map_existing_extent(t, base, paddr.add_offset(base), length - base);
	      if (!ext_) {
		conflicted++;
		return;
	      }
	      auto ext = *ext_;
	      ASSERT_TRUE(ext->is_exist_clean());
	      if (get_random_contents() % 2 == 0) {
		auto ext1 = mutate_extent(t, ext);
		ASSERT_TRUE(ext1->is_exist_mutation_pending());
	      }
	    }
	    if (try_submit_transaction(std::move(t))) {
	      success++;
	      logger().info("transaction {} submit the transction", static_cast<void*>(t.t.get()));
	    } else {
	      conflicted++;
	    }
	  });
	}).handle_exception([](std::exception_ptr e) {
	  logger().info("{}", e);
	}).get0();
      logger().info("test_map_existing_extent_concurrent: early_exit {} conflicted {} success {}", early_exit, conflicted, success);
      ASSERT_TRUE(success == 1);
      ASSERT_EQ(success + conflicted + early_exit, REMAP_NUM);
      replay();
      check();
    });
  }
};

struct tm_single_device_test_t :
  public transaction_manager_test_t {

  tm_single_device_test_t() : transaction_manager_test_t(1, 0) {}
};

struct tm_multi_device_test_t :
  public transaction_manager_test_t {

  tm_multi_device_test_t() : transaction_manager_test_t(3, 0) {}
};

struct tm_multi_tier_device_test_t :
  public transaction_manager_test_t {

  tm_multi_tier_device_test_t() : transaction_manager_test_t(1, 2) {}
};

TEST_P(tm_single_device_test_t, basic)
{
  constexpr laddr_t SIZE = 4096;
  run_async([this] {
    constexpr laddr_t ADDR = 0xFF * SIZE;
    {
      auto t = create_transaction();
      auto extent = alloc_extent(
	t,
	ADDR,
	SIZE,
	'a');
      ASSERT_EQ(ADDR, extent->get_laddr());
      check_mappings(t);
      check();
      submit_transaction(std::move(t));
      check();
    }
  });
}

TEST_P(tm_single_device_test_t, mutate)
{
  constexpr laddr_t SIZE = 4096;
  run_async([this] {
    constexpr laddr_t ADDR = 0xFF * SIZE;
    {
      auto t = create_transaction();
      auto extent = alloc_extent(
	t,
	ADDR,
	SIZE,
	'a');
      ASSERT_EQ(ADDR, extent->get_laddr());
      check_mappings(t);
      check();
      submit_transaction(std::move(t));
      check();
    }
    ASSERT_TRUE(check_usage());
    replay();
    {
      auto t = create_transaction();
      auto ext = get_extent(
	t,
	ADDR,
	SIZE);
      auto mut = mutate_extent(t, ext);
      check_mappings(t);
      check();
      submit_transaction(std::move(t));
      check();
    }
    ASSERT_TRUE(check_usage());
    replay();
    check();
  });
}

TEST_P(tm_single_device_test_t, allocate_lba_conflict)
{
  constexpr laddr_t SIZE = 4096;
  run_async([this] {
    constexpr laddr_t ADDR = 0xFF * SIZE;
    constexpr laddr_t ADDR2 = 0xFE * SIZE;
    auto t = create_transaction();
    auto t2 = create_transaction();

    // These should conflict as they should both modify the lba root
    auto extent = alloc_extent(
      t,
      ADDR,
      SIZE,
      'a');
    ASSERT_EQ(ADDR, extent->get_laddr());
    check_mappings(t);
    check();

    auto extent2 = alloc_extent(
      t2,
      ADDR2,
      SIZE,
      'a');
    ASSERT_EQ(ADDR2, extent2->get_laddr());
    check_mappings(t2);
    extent2.reset();

    submit_transaction(std::move(t2));
    submit_transaction_expect_conflict(std::move(t));
  });
}

TEST_P(tm_single_device_test_t, mutate_lba_conflict)
{
  constexpr laddr_t SIZE = 4096;
  run_async([this] {
    {
      auto t = create_transaction();
      for (unsigned i = 0; i < 300; ++i) {
	auto extent = alloc_extent(
	  t,
	  laddr_t(i * SIZE),
	  SIZE);
      }
      check_mappings(t);
      submit_transaction(std::move(t));
      check();
    }

    constexpr laddr_t ADDR = 150 * SIZE;
    {
      auto t = create_transaction();
      auto t2 = create_transaction();

      mutate_addr(t, ADDR, SIZE);
      mutate_addr(t2, ADDR, SIZE);

      submit_transaction(std::move(t));
      submit_transaction_expect_conflict(std::move(t2));
    }
    check();

    {
      auto t = create_transaction();
      mutate_addr(t, ADDR, SIZE);
      submit_transaction(std::move(t));
    }
    check();
  });
}

TEST_P(tm_single_device_test_t, concurrent_mutate_lba_no_conflict)
{
  constexpr laddr_t SIZE = 4096;
  constexpr size_t NUM = 500;
  constexpr laddr_t addr = 0;
  constexpr laddr_t addr2 = SIZE * (NUM - 1);
  run_async([this] {
    {
      auto t = create_transaction();
      for (unsigned i = 0; i < NUM; ++i) {
	auto extent = alloc_extent(
	  t,
	  laddr_t(i * SIZE),
	  SIZE);
      }
      submit_transaction(std::move(t));
    }

    {
      auto t = create_transaction();
      auto t2 = create_transaction();

      mutate_addr(t, addr, SIZE);
      mutate_addr(t2, addr2, SIZE);

      submit_transaction(std::move(t));
      submit_transaction(std::move(t2));
    }
    check();
  });
}

TEST_P(tm_single_device_test_t, create_remove_same_transaction)
{
  constexpr laddr_t SIZE = 4096;
  run_async([this] {
    constexpr laddr_t ADDR = 0xFF * SIZE;
    {
      auto t = create_transaction();
      auto extent = alloc_extent(
	t,
	ADDR,
	SIZE,
	'a');
      ASSERT_EQ(ADDR, extent->get_laddr());
      check_mappings(t);
      dec_ref(t, ADDR);
      check_mappings(t);

      extent = alloc_extent(
	t,
	ADDR,
	SIZE,
	'a');

      submit_transaction(std::move(t));
      check();
    }
    replay();
    check();
  });
}

TEST_P(tm_single_device_test_t, split_merge_read_same_transaction)
{
  constexpr laddr_t SIZE = 4096;
  run_async([this] {
    {
      auto t = create_transaction();
      for (unsigned i = 0; i < 300; ++i) {
	auto extent = alloc_extent(
	  t,
	  laddr_t(i * SIZE),
	  SIZE);
      }
      check_mappings(t);
      submit_transaction(std::move(t));
      check();
    }
    {
      auto t = create_transaction();
      for (unsigned i = 0; i < 240; ++i) {
	dec_ref(
	  t,
	  laddr_t(i * SIZE));
      }
      check_mappings(t);
      submit_transaction(std::move(t));
      check();
    }
  });
}

TEST_P(tm_single_device_test_t, inc_dec_ref)
{
  constexpr laddr_t SIZE = 4096;
  run_async([this] {
    constexpr laddr_t ADDR = 0xFF * SIZE;
    {
      auto t = create_transaction();
      auto extent = alloc_extent(
	t,
	ADDR,
	SIZE,
	'a');
      ASSERT_EQ(ADDR, extent->get_laddr());
      check_mappings(t);
      check();
      submit_transaction(std::move(t));
      check();
    }
    replay();
    {
      auto t = create_transaction();
      inc_ref(t, ADDR);
      check_mappings(t);
      check();
      submit_transaction(std::move(t));
      check();
    }
    {
      auto t = create_transaction();
      dec_ref(t, ADDR);
      check_mappings(t);
      check();
      submit_transaction(std::move(t));
      check();
    }
    replay();
    {
      auto t = create_transaction();
      dec_ref(t, ADDR);
      check_mappings(t);
      check();
      submit_transaction(std::move(t));
      check();
    }
  });
}

TEST_P(tm_single_device_test_t, cause_lba_split)
{
  constexpr laddr_t SIZE = 4096;
  run_async([this] {
    for (unsigned i = 0; i < 200; ++i) {
      auto t = create_transaction();
      auto extent = alloc_extent(
	t,
	i * SIZE,
	SIZE,
	(char)(i & 0xFF));
      ASSERT_EQ(i * SIZE, extent->get_laddr());
      submit_transaction(std::move(t));
    }
    check();
  });
}

TEST_P(tm_single_device_test_t, random_writes)
{
  constexpr size_t TOTAL = 4<<20;
  constexpr size_t BSIZE = 4<<10;
  constexpr size_t PADDING_SIZE = 256<<10;
  constexpr size_t BLOCKS = TOTAL / BSIZE;
  run_async([this] {
    for (unsigned i = 0; i < BLOCKS; ++i) {
      auto t = create_transaction();
      auto extent = alloc_extent(
	t,
	i * BSIZE,
	BSIZE);
      ASSERT_EQ(i * BSIZE, extent->get_laddr());
      submit_transaction(std::move(t));
    }

    for (unsigned i = 0; i < 4; ++i) {
      for (unsigned j = 0; j < 65; ++j) {
	auto t = create_transaction();
	for (unsigned k = 0; k < 2; ++k) {
	  auto ext = get_extent(
	    t,
	    get_random_laddr(BSIZE, TOTAL),
	    BSIZE);
	  auto mut = mutate_extent(t, ext);
	  // pad out transaction
	  auto padding = alloc_extent(
	    t,
	    TOTAL + (k * PADDING_SIZE),
	    PADDING_SIZE);
	  dec_ref(t, padding->get_laddr());
	}
	submit_transaction(std::move(t));
      }
      replay();
      logger().info("random_writes: {} checking", i);
      check();
      logger().info("random_writes: {} done replaying/checking", i);
    }
  });
}

TEST_P(tm_single_device_test_t, find_hole_assert_trigger)
{
  constexpr unsigned max = 10;
  constexpr size_t BSIZE = 4<<10;
  int num = 40;
  run([&, this] {
    return seastar::parallel_for_each(
      boost::make_counting_iterator(0u),
      boost::make_counting_iterator(max),
      [&, this](auto idx) {
        return allocate_sequentially(BSIZE, num);
    });
  });
}

TEST_P(tm_single_device_test_t, random_writes_concurrent)
{
  test_random_writes_concurrent();
}

TEST_P(tm_multi_device_test_t, random_writes_concurrent)
{
  test_random_writes_concurrent();
}

TEST_P(tm_multi_tier_device_test_t, evict)
{
  test_evict();
}

TEST_P(tm_single_device_test_t, parallel_extent_read)
{
  test_parallel_extent_read();
}

TEST_P(tm_single_device_test_t, test_map_existing_extent)
{
  test_map_existing_extent();
}
TEST_P(tm_single_device_test_t, test_map_existing_extent_concurrent)
{
  test_map_existing_extent_concurrent();
}
INSTANTIATE_TEST_SUITE_P(
  transaction_manager_test,
  tm_single_device_test_t,
  ::testing::Values (
    "segmented",
    "circularbounded"
  )
);

INSTANTIATE_TEST_SUITE_P(
  transaction_manager_test,
  tm_multi_device_test_t,
  ::testing::Values (
    "segmented"
  )
);

INSTANTIATE_TEST_SUITE_P(
  transaction_manager_test,
  tm_multi_tier_device_test_t,
  ::testing::Values (
    "segmented"
  )
);
