/*
 * CMapService.cpp, part of VCMI engine
 *
 * Authors: listed in file AUTHORS in main folder
 *
 * License: GNU General Public License v2.0 or later
 * Full text of license available in license.txt file, in main folder
 *
 */
#include "StdInc.h"
#include "CMapService.h"

#include "MapFormatSettings.h"

#include "../json/JsonUtils.h"
#include "../filesystem/Filesystem.h"
#include "../filesystem/CBinaryReader.h"
#include "../filesystem/CCompressedStream.h"
#include "../filesystem/CMemoryStream.h"
#include "../filesystem/CMemoryBuffer.h"
#include "../modding/CModHandler.h"
#include "../modding/ModDescription.h"
#include "../modding/ModScope.h"
#include "../GameLibrary.h"

#include "CMap.h"
#include "MapFormat.h"

#include "MapFormatH3M.h"
#include "MapFormatJson.h"
#include "../callback/EditorCallback.h"

VCMI_LIB_NAMESPACE_BEGIN


std::unique_ptr<CMap> CMapService::loadMap(const ResourcePath & name, IGameInfoCallback * cb) const
{
	std::string modName = LIBRARY->modh->findResourceOrigin(name);
	std::string encoding = LIBRARY->modh->findResourceEncoding(name);

	auto stream = getStreamFromFS(name);
	auto loader = getMapLoader(stream, name.getName(), modName, encoding);
	// If JSON loader, let it know whether this is editor callback
	if(auto jsonLoader = dynamic_cast<CMapLoaderJson*>(loader.get()))
		jsonLoader->setRunningInMapEditor(dynamic_cast<EditorCallback*>(cb) != nullptr);

	return loader->loadMap(cb);
}

std::unique_ptr<CMapHeader> CMapService::loadMapHeader(const ResourcePath & name, bool isEditor) const
{
	std::string modName = LIBRARY->modh->findResourceOrigin(name);
	std::string encoding = LIBRARY->modh->findResourceEncoding(name);

	auto stream = getStreamFromFS(name);
	auto loader = getMapLoader(stream, name.getName(), modName, encoding);
	// default: not editor; callers may pass explicit overload if needed
	if(auto jsonLoader = dynamic_cast<CMapLoaderJson*>(loader.get()))
		jsonLoader->setRunningInMapEditor(false);
	return loader->loadMapHeader();
}

std::unique_ptr<CMap> CMapService::loadMap(const uint8_t * buffer, int size, const std::string & name,  const std::string & modName, const std::string & encoding, IGameInfoCallback * cb) const
{
	auto stream = getStreamFromMem(buffer, size);
	auto loader = getMapLoader(stream, name, modName, encoding);
	if(auto jsonLoader = dynamic_cast<CMapLoaderJson*>(loader.get()))
		jsonLoader->setRunningInMapEditor(dynamic_cast<EditorCallback*>(cb) != nullptr);
	std::unique_ptr<CMap> map(loader->loadMap(cb));
	std::unique_ptr<CMapHeader> header(map.get());

	//might be original campaign and require patch
	getMapPatcher(name)->patchMapHeader(header);
	header.release();

	return map;
}

std::unique_ptr<CMapHeader> CMapService::loadMapHeader(const uint8_t * buffer, int size, const std::string & name, const std::string & modName, const std::string & encoding) const
{
	auto stream = getStreamFromMem(buffer, size);
	auto loader = getMapLoader(stream, name, modName, encoding);
	if(auto jsonLoader = dynamic_cast<CMapLoaderJson*>(loader.get()))
		jsonLoader->setRunningInMapEditor(false);
	std::unique_ptr<CMapHeader> header = loader->loadMapHeader();

	//might be original campaign and require patch
	getMapPatcher(name)->patchMapHeader(header);
	return header;
}

void CMapService::saveMap(const std::unique_ptr<CMap> & map, boost::filesystem::path fullPath) const
{
	CMemoryBuffer serializeBuffer;
	{
		CMapSaverJson saver(&serializeBuffer);
		saver.saveMap(map);
	}
	{
		boost::filesystem::remove(fullPath);
		std::ofstream tmp(fullPath.c_str(), std::ofstream::binary);

		tmp.write(reinterpret_cast<const char *>(serializeBuffer.getBuffer().data()), serializeBuffer.getSize());
		tmp.flush();
		tmp.close();
	}
}

ModCompatibilityInfo CMapService::verifyMapHeaderMods(const CMapHeader & map)
{
	const auto & activeMods = LIBRARY->modh->getActiveMods();
	
	ModCompatibilityInfo missingMods;
	ModCompatibilityInfo missingModsFiltered;
	for(const auto & mapMod : map.mods)
	{
		if(vstd::contains(activeMods, mapMod.first))
		{
			const auto & modInfo = LIBRARY->modh->getModInfo(mapMod.first);
			if(modInfo.getVersion().compatible(mapMod.second.version))
				continue;
		}
		missingMods[mapMod.first] = mapMod.second;
	}
	
	//filter child mods
	for(const auto & mapMod : missingMods)
	{
		if(!mapMod.second.parent.empty() && missingMods.count(mapMod.second.parent))
			continue;
		missingModsFiltered.insert(mapMod);
	}
	
	return missingModsFiltered;
}

std::unique_ptr<CInputStream> CMapService::getStreamFromFS(const ResourcePath & name)
{
	return CResourceHandler::get()->load(name);
}

std::unique_ptr<CInputStream> CMapService::getStreamFromMem(const uint8_t * buffer, int size)
{
	return std::unique_ptr<CInputStream>(new CMemoryStream(buffer, size));
}

std::unique_ptr<IMapLoader> CMapService::getMapLoader(std::unique_ptr<CInputStream> & stream, std::string mapName, std::string modName, std::string encoding)
{
	// Read map header
	CBinaryReader reader(stream.get());
	ui32 header = reader.readUInt32();
	reader.getStream()->seek(0);

	//check for ZIP magic. Zip files are VCMI maps
	switch(header)
	{
	case 0x06054b50:
	case 0x04034b50:
	case 0x02014b50:
		return std::unique_ptr<IMapLoader>(new CMapLoaderJson(stream.get(), mapName));
		break;
	default:
		// Check which map format is used
		// gzip header is 3 bytes only in size
		switch(header & 0xffffff)
		{
			// gzip header magic number, reversed for LE
			case 0x00088B1F:
				stream = std::unique_ptr<CInputStream>(new CCompressedStream(std::move(stream), true));
				return std::unique_ptr<IMapLoader>(new CMapLoaderH3M(mapName, modName, encoding, stream.get()));
			case static_cast<int>(EMapFormat::WOG) :
			case static_cast<int>(EMapFormat::AB)  :
			case static_cast<int>(EMapFormat::ROE) :
			case static_cast<int>(EMapFormat::SOD) :
			case static_cast<int>(EMapFormat::CHR) :
			case static_cast<int>(EMapFormat::HOTA) :
				return std::unique_ptr<IMapLoader>(new CMapLoaderH3M(mapName, modName, encoding, stream.get()));
			default :
				throw std::runtime_error("Unknown map format");
		}
	}
}

std::unique_ptr<IMapPatcher> CMapService::getMapPatcher(std::string scenarioName)
{
	boost::to_lower(scenarioName);
	logGlobal->debug("Request to patch map %s", scenarioName);
	return std::make_unique<CMapPatcher>(LIBRARY->mapFormat->mapOverrides(scenarioName));
}

VCMI_LIB_NAMESPACE_END
