/*
 * This program is released under the Common Public License V1.0
 *
 * You should have received a copy of Common Public License V1.0 along with
 * with this program.
 *
 * DRBG conforming to NIST SP800-90A
 *
 * Author(s): Patrick Steuer <patrick.steuer@de.ibm.com>
 *
 * Copyright IBM Corp. 2015
 */

#ifndef S390_DRBG_SHA512_H
#define S390_DRBG_SHA512_H

#include <stdint.h>

#include "ica_api.h"

#define DRBG_SHA512_SEED_LEN (888 / 8)

/*
 * SHA-512 DRBG mechanism working state type (see POP)
 */
struct drbg_sha512_ws{
	uint32_t rsvd0;				/* padding */
	uint32_t reseed_ctr;			/* reseed counter */
	uint64_t stream_bytes;			/* no. of generated bytes */
	unsigned char rsvd1;			/* padding */
	unsigned char v[DRBG_SHA512_SEED_LEN];	/* V */
	unsigned char rsvd2;			/* padding */
	unsigned char c[DRBG_SHA512_SEED_LEN];	/* C */
};

/*
 * SHA-512 DRBG mechanism functions
 */
int drbg_sha512_instantiate(void **ws,
			    int sec_strength,
			    const unsigned char *pers,
			    size_t pers_len,
			    const unsigned char *entropy,
			    size_t entropy_len,
			    const unsigned char *nonce,
			    size_t nonce_len);

int drbg_sha512_instantiate_ppno(void **ws,
				 int sec_strength,
				 const unsigned char *pers,
				 size_t pers_len,
				 const unsigned char *entropy,
				 size_t entropy_len,
				 const unsigned char *nonce,
				 size_t nonce_len);

int drbg_sha512_reseed(void *ws,
		       const unsigned char *add,
		       size_t add_len,
		       const unsigned char *entropy,
		       size_t entropy_len);

int drbg_sha512_reseed_ppno(void *ws,
			    const unsigned char *add,
			    size_t add_len,
			    const unsigned char *entropy,
			    size_t entropy_len);

int drbg_sha512_generate(void *ws,
			 const unsigned char *add,
			 size_t add_len,
			 unsigned char *prnd,
			 size_t prnd_len);

int drbg_sha512_generate_ppno(void *ws,
			      const unsigned char *add,
			      size_t add_len,
			      unsigned char *prnd,
			      size_t prnd_len);

int drbg_sha512_uninstantiate(void **ws,
			      bool test_mode);

int drbg_sha512_health_test(void *func,
			    int sec,
			    bool pr);

#endif
