/* $Id: hamsi_helper.c 202 2010-05-31 15:46:48Z tp $ */
/*
 * Helper code for Hamsi (input block expansion). This code is
 * automatically generated and includes precomputed tables for
 * expansion code which handles 2 to 8 bits at a time.
 *
 * This file is included from hamsi.c, and is not meant to be compiled
 * independently.
 *
 * ==========================(LICENSE BEGIN)============================
 *
 * Copyright (c) 2007-2010  Projet RNRT SAPHIR
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * ===========================(LICENSE END)=============================
 *
 * @author   Thomas Pornin <thomas.pornin@cryptolog.com>
 */

#ifdef __cplusplus
extern "C"{
#endif

#if SPH_HAMSI_EXPAND_SMALL == 1

/* Note: this table lists bits within each byte from least
   siginificant to most significant. */
static const sph_u32 T256[32][8] = {
	{ SPH_C32(0x74951000), SPH_C32(0x5a2b467e), SPH_C32(0x88fd1d2b),
	  SPH_C32(0x1ee68292), SPH_C32(0xcba90000), SPH_C32(0x90273769),
	  SPH_C32(0xbbdcf407), SPH_C32(0xd0f4af61) },
	{ SPH_C32(0xcba90000), SPH_C32(0x90273769), SPH_C32(0xbbdcf407),
	  SPH_C32(0xd0f4af61), SPH_C32(0xbf3c1000), SPH_C32(0xca0c7117),
	  SPH_C32(0x3321e92c), SPH_C32(0xce122df3) },
	{ SPH_C32(0xe92a2000), SPH_C32(0xb4578cfc), SPH_C32(0x11fa3a57),
	  SPH_C32(0x3dc90524), SPH_C32(0x97530000), SPH_C32(0x204f6ed3),
	  SPH_C32(0x77b9e80f), SPH_C32(0xa1ec5ec1) },
	{ SPH_C32(0x97530000), SPH_C32(0x204f6ed3), SPH_C32(0x77b9e80f),
	  SPH_C32(0xa1ec5ec1), SPH_C32(0x7e792000), SPH_C32(0x9418e22f),
	  SPH_C32(0x6643d258), SPH_C32(0x9c255be5) },
	{ SPH_C32(0x121b4000), SPH_C32(0x5b17d9e8), SPH_C32(0x8dfacfab),
	  SPH_C32(0xce36cc72), SPH_C32(0xe6570000), SPH_C32(0x4bb33a25),
	  SPH_C32(0x848598ba), SPH_C32(0x1041003e) },
	{ SPH_C32(0xe6570000), SPH_C32(0x4bb33a25), SPH_C32(0x848598ba),
	  SPH_C32(0x1041003e), SPH_C32(0xf44c4000), SPH_C32(0x10a4e3cd),
	  SPH_C32(0x097f5711), SPH_C32(0xde77cc4c) },
	{ SPH_C32(0xe4788000), SPH_C32(0x859673c1), SPH_C32(0xb5fb2452),
	  SPH_C32(0x29cc5edf), SPH_C32(0x045f0000), SPH_C32(0x9c4a93c9),
	  SPH_C32(0x62fc79d0), SPH_C32(0x731ebdc2) },
	{ SPH_C32(0x045f0000), SPH_C32(0x9c4a93c9), SPH_C32(0x62fc79d0),
	  SPH_C32(0x731ebdc2), SPH_C32(0xe0278000), SPH_C32(0x19dce008),
	  SPH_C32(0xd7075d82), SPH_C32(0x5ad2e31d) },
	{ SPH_C32(0xb7a40100), SPH_C32(0x8a1f31d8), SPH_C32(0x8589d8ab),
	  SPH_C32(0xe6c46464), SPH_C32(0x734c0000), SPH_C32(0x956fa7d6),
	  SPH_C32(0xa29d1297), SPH_C32(0x6ee56854) },
	{ SPH_C32(0x734c0000), SPH_C32(0x956fa7d6), SPH_C32(0xa29d1297),
	  SPH_C32(0x6ee56854), SPH_C32(0xc4e80100), SPH_C32(0x1f70960e),
	  SPH_C32(0x2714ca3c), SPH_C32(0x88210c30) },
	{ SPH_C32(0xa7b80200), SPH_C32(0x1f128433), SPH_C32(0x60e5f9f2),
	  SPH_C32(0x9e147576), SPH_C32(0xee260000), SPH_C32(0x124b683e),
	  SPH_C32(0x80c2d68f), SPH_C32(0x3bf3ab2c) },
	{ SPH_C32(0xee260000), SPH_C32(0x124b683e), SPH_C32(0x80c2d68f),
	  SPH_C32(0x3bf3ab2c), SPH_C32(0x499e0200), SPH_C32(0x0d59ec0d),
	  SPH_C32(0xe0272f7d), SPH_C32(0xa5e7de5a) },
	{ SPH_C32(0x8f3e0400), SPH_C32(0x0d9dc877), SPH_C32(0x6fc548e1),
	  SPH_C32(0x898d2cd6), SPH_C32(0x14bd0000), SPH_C32(0x2fba37ff),
	  SPH_C32(0x6a72e5bb), SPH_C32(0x247febe6) },
	{ SPH_C32(0x14bd0000), SPH_C32(0x2fba37ff), SPH_C32(0x6a72e5bb),
	  SPH_C32(0x247febe6), SPH_C32(0x9b830400), SPH_C32(0x2227ff88),
	  SPH_C32(0x05b7ad5a), SPH_C32(0xadf2c730) },
	{ SPH_C32(0xde320800), SPH_C32(0x288350fe), SPH_C32(0x71852ac7),
	  SPH_C32(0xa6bf9f96), SPH_C32(0xe18b0000), SPH_C32(0x5459887d),
	  SPH_C32(0xbf1283d3), SPH_C32(0x1b666a73) },
	{ SPH_C32(0xe18b0000), SPH_C32(0x5459887d), SPH_C32(0xbf1283d3),
	  SPH_C32(0x1b666a73), SPH_C32(0x3fb90800), SPH_C32(0x7cdad883),
	  SPH_C32(0xce97a914), SPH_C32(0xbdd9f5e5) },
	{ SPH_C32(0x515c0010), SPH_C32(0x40f372fb), SPH_C32(0xfce72602),
	  SPH_C32(0x71575061), SPH_C32(0x2e390000), SPH_C32(0x64dd6689),
	  SPH_C32(0x3cd406fc), SPH_C32(0xb1f490bc) },
	{ SPH_C32(0x2e390000), SPH_C32(0x64dd6689), SPH_C32(0x3cd406fc),
	  SPH_C32(0xb1f490bc), SPH_C32(0x7f650010), SPH_C32(0x242e1472),
	  SPH_C32(0xc03320fe), SPH_C32(0xc0a3c0dd) },
	{ SPH_C32(0xa2b80020), SPH_C32(0x81e7e5f6), SPH_C32(0xf9ce4c04),
	  SPH_C32(0xe2afa0c0), SPH_C32(0x5c720000), SPH_C32(0xc9bacd12),
	  SPH_C32(0x79a90df9), SPH_C32(0x63e92178) },
	{ SPH_C32(0x5c720000), SPH_C32(0xc9bacd12), SPH_C32(0x79a90df9),
	  SPH_C32(0x63e92178), SPH_C32(0xfeca0020), SPH_C32(0x485d28e4),
	  SPH_C32(0x806741fd), SPH_C32(0x814681b8) },
	{ SPH_C32(0x4dce0040), SPH_C32(0x3b5bec7e), SPH_C32(0x36656ba8),
	  SPH_C32(0x23633a05), SPH_C32(0x78ab0000), SPH_C32(0xa0cd5a34),
	  SPH_C32(0x5d5ca0f7), SPH_C32(0x727784cb) },
	{ SPH_C32(0x78ab0000), SPH_C32(0xa0cd5a34), SPH_C32(0x5d5ca0f7),
	  SPH_C32(0x727784cb), SPH_C32(0x35650040), SPH_C32(0x9b96b64a),
	  SPH_C32(0x6b39cb5f), SPH_C32(0x5114bece) },
	{ SPH_C32(0x5bd20080), SPH_C32(0x450f18ec), SPH_C32(0xc2c46c55),
	  SPH_C32(0xf362b233), SPH_C32(0x39a60000), SPH_C32(0x4ab753eb),
	  SPH_C32(0xd14e094b), SPH_C32(0xb772b42b) },
	{ SPH_C32(0x39a60000), SPH_C32(0x4ab753eb), SPH_C32(0xd14e094b),
	  SPH_C32(0xb772b42b), SPH_C32(0x62740080), SPH_C32(0x0fb84b07),
	  SPH_C32(0x138a651e), SPH_C32(0x44100618) },
	{ SPH_C32(0xc04e0001), SPH_C32(0x33b9c010), SPH_C32(0xae0ebb05),
	  SPH_C32(0xb5a4c63b), SPH_C32(0xc8f10000), SPH_C32(0x0b2de782),
	  SPH_C32(0x6bf648a4), SPH_C32(0x539cbdbf) },
	{ SPH_C32(0xc8f10000), SPH_C32(0x0b2de782), SPH_C32(0x6bf648a4),
	  SPH_C32(0x539cbdbf), SPH_C32(0x08bf0001), SPH_C32(0x38942792),
	  SPH_C32(0xc5f8f3a1), SPH_C32(0xe6387b84) },
	{ SPH_C32(0x88230002), SPH_C32(0x5fe7a7b3), SPH_C32(0x99e585aa),
	  SPH_C32(0x8d75f7f1), SPH_C32(0x51ac0000), SPH_C32(0x25e30f14),
	  SPH_C32(0x79e22a4c), SPH_C32(0x1298bd46) },
	{ SPH_C32(0x51ac0000), SPH_C32(0x25e30f14), SPH_C32(0x79e22a4c),
	  SPH_C32(0x1298bd46), SPH_C32(0xd98f0002), SPH_C32(0x7a04a8a7),
	  SPH_C32(0xe007afe6), SPH_C32(0x9fed4ab7) },
	{ SPH_C32(0xd0080004), SPH_C32(0x8c768f77), SPH_C32(0x9dc5b050),
	  SPH_C32(0xaf4a29da), SPH_C32(0x6ba90000), SPH_C32(0x40ebf9aa),
	  SPH_C32(0x98321c3d), SPH_C32(0x76acc733) },
	{ SPH_C32(0x6ba90000), SPH_C32(0x40ebf9aa), SPH_C32(0x98321c3d),
	  SPH_C32(0x76acc733), SPH_C32(0xbba10004), SPH_C32(0xcc9d76dd),
	  SPH_C32(0x05f7ac6d), SPH_C32(0xd9e6eee9) },
	{ SPH_C32(0xa8ae0008), SPH_C32(0x2079397d), SPH_C32(0xfe739301),
	  SPH_C32(0xb8a92831), SPH_C32(0x171c0000), SPH_C32(0xb26e3344),
	  SPH_C32(0x9e6a837e), SPH_C32(0x58f8485f) },
	{ SPH_C32(0x171c0000), SPH_C32(0xb26e3344), SPH_C32(0x9e6a837e),
	  SPH_C32(0x58f8485f), SPH_C32(0xbfb20008), SPH_C32(0x92170a39),
	  SPH_C32(0x6019107f), SPH_C32(0xe051606e) }
};

#define INPUT_SMALL   do { \
		const sph_u32 *tp = &T256[0][0]; \
		unsigned u, v; \
		m0 = 0; \
		m1 = 0; \
		m2 = 0; \
		m3 = 0; \
		m4 = 0; \
		m5 = 0; \
		m6 = 0; \
		m7 = 0; \
		for (u = 0; u < 4; u ++) { \
			unsigned db = buf[u]; \
			for (v = 0; v < 8; v ++, db >>= 1) { \
				sph_u32 dm = SPH_T32(-(sph_u32)(db & 1)); \
				m0 ^= dm & *tp ++; \
				m1 ^= dm & *tp ++; \
				m2 ^= dm & *tp ++; \
				m3 ^= dm & *tp ++; \
				m4 ^= dm & *tp ++; \
				m5 ^= dm & *tp ++; \
				m6 ^= dm & *tp ++; \
				m7 ^= dm & *tp ++; \
			} \
		} \
	} while (0)

#endif

#if SPH_HAMSI_EXPAND_SMALL == 2

static const sph_u32 T256_0[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xe4788000), SPH_C32(0x859673c1), SPH_C32(0xb5fb2452),
	  SPH_C32(0x29cc5edf), SPH_C32(0x045f0000), SPH_C32(0x9c4a93c9),
	  SPH_C32(0x62fc79d0), SPH_C32(0x731ebdc2) },
	{ SPH_C32(0x045f0000), SPH_C32(0x9c4a93c9), SPH_C32(0x62fc79d0),
	  SPH_C32(0x731ebdc2), SPH_C32(0xe0278000), SPH_C32(0x19dce008),
	  SPH_C32(0xd7075d82), SPH_C32(0x5ad2e31d) },
	{ SPH_C32(0xe0278000), SPH_C32(0x19dce008), SPH_C32(0xd7075d82),
	  SPH_C32(0x5ad2e31d), SPH_C32(0xe4788000), SPH_C32(0x859673c1),
	  SPH_C32(0xb5fb2452), SPH_C32(0x29cc5edf) }
};

static const sph_u32 T256_2[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x121b4000), SPH_C32(0x5b17d9e8), SPH_C32(0x8dfacfab),
	  SPH_C32(0xce36cc72), SPH_C32(0xe6570000), SPH_C32(0x4bb33a25),
	  SPH_C32(0x848598ba), SPH_C32(0x1041003e) },
	{ SPH_C32(0xe6570000), SPH_C32(0x4bb33a25), SPH_C32(0x848598ba),
	  SPH_C32(0x1041003e), SPH_C32(0xf44c4000), SPH_C32(0x10a4e3cd),
	  SPH_C32(0x097f5711), SPH_C32(0xde77cc4c) },
	{ SPH_C32(0xf44c4000), SPH_C32(0x10a4e3cd), SPH_C32(0x097f5711),
	  SPH_C32(0xde77cc4c), SPH_C32(0x121b4000), SPH_C32(0x5b17d9e8),
	  SPH_C32(0x8dfacfab), SPH_C32(0xce36cc72) }
};

static const sph_u32 T256_4[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xe92a2000), SPH_C32(0xb4578cfc), SPH_C32(0x11fa3a57),
	  SPH_C32(0x3dc90524), SPH_C32(0x97530000), SPH_C32(0x204f6ed3),
	  SPH_C32(0x77b9e80f), SPH_C32(0xa1ec5ec1) },
	{ SPH_C32(0x97530000), SPH_C32(0x204f6ed3), SPH_C32(0x77b9e80f),
	  SPH_C32(0xa1ec5ec1), SPH_C32(0x7e792000), SPH_C32(0x9418e22f),
	  SPH_C32(0x6643d258), SPH_C32(0x9c255be5) },
	{ SPH_C32(0x7e792000), SPH_C32(0x9418e22f), SPH_C32(0x6643d258),
	  SPH_C32(0x9c255be5), SPH_C32(0xe92a2000), SPH_C32(0xb4578cfc),
	  SPH_C32(0x11fa3a57), SPH_C32(0x3dc90524) }
};

static const sph_u32 T256_6[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x74951000), SPH_C32(0x5a2b467e), SPH_C32(0x88fd1d2b),
	  SPH_C32(0x1ee68292), SPH_C32(0xcba90000), SPH_C32(0x90273769),
	  SPH_C32(0xbbdcf407), SPH_C32(0xd0f4af61) },
	{ SPH_C32(0xcba90000), SPH_C32(0x90273769), SPH_C32(0xbbdcf407),
	  SPH_C32(0xd0f4af61), SPH_C32(0xbf3c1000), SPH_C32(0xca0c7117),
	  SPH_C32(0x3321e92c), SPH_C32(0xce122df3) },
	{ SPH_C32(0xbf3c1000), SPH_C32(0xca0c7117), SPH_C32(0x3321e92c),
	  SPH_C32(0xce122df3), SPH_C32(0x74951000), SPH_C32(0x5a2b467e),
	  SPH_C32(0x88fd1d2b), SPH_C32(0x1ee68292) }
};

static const sph_u32 T256_8[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xde320800), SPH_C32(0x288350fe), SPH_C32(0x71852ac7),
	  SPH_C32(0xa6bf9f96), SPH_C32(0xe18b0000), SPH_C32(0x5459887d),
	  SPH_C32(0xbf1283d3), SPH_C32(0x1b666a73) },
	{ SPH_C32(0xe18b0000), SPH_C32(0x5459887d), SPH_C32(0xbf1283d3),
	  SPH_C32(0x1b666a73), SPH_C32(0x3fb90800), SPH_C32(0x7cdad883),
	  SPH_C32(0xce97a914), SPH_C32(0xbdd9f5e5) },
	{ SPH_C32(0x3fb90800), SPH_C32(0x7cdad883), SPH_C32(0xce97a914),
	  SPH_C32(0xbdd9f5e5), SPH_C32(0xde320800), SPH_C32(0x288350fe),
	  SPH_C32(0x71852ac7), SPH_C32(0xa6bf9f96) }
};

static const sph_u32 T256_10[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x8f3e0400), SPH_C32(0x0d9dc877), SPH_C32(0x6fc548e1),
	  SPH_C32(0x898d2cd6), SPH_C32(0x14bd0000), SPH_C32(0x2fba37ff),
	  SPH_C32(0x6a72e5bb), SPH_C32(0x247febe6) },
	{ SPH_C32(0x14bd0000), SPH_C32(0x2fba37ff), SPH_C32(0x6a72e5bb),
	  SPH_C32(0x247febe6), SPH_C32(0x9b830400), SPH_C32(0x2227ff88),
	  SPH_C32(0x05b7ad5a), SPH_C32(0xadf2c730) },
	{ SPH_C32(0x9b830400), SPH_C32(0x2227ff88), SPH_C32(0x05b7ad5a),
	  SPH_C32(0xadf2c730), SPH_C32(0x8f3e0400), SPH_C32(0x0d9dc877),
	  SPH_C32(0x6fc548e1), SPH_C32(0x898d2cd6) }
};

static const sph_u32 T256_12[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xa7b80200), SPH_C32(0x1f128433), SPH_C32(0x60e5f9f2),
	  SPH_C32(0x9e147576), SPH_C32(0xee260000), SPH_C32(0x124b683e),
	  SPH_C32(0x80c2d68f), SPH_C32(0x3bf3ab2c) },
	{ SPH_C32(0xee260000), SPH_C32(0x124b683e), SPH_C32(0x80c2d68f),
	  SPH_C32(0x3bf3ab2c), SPH_C32(0x499e0200), SPH_C32(0x0d59ec0d),
	  SPH_C32(0xe0272f7d), SPH_C32(0xa5e7de5a) },
	{ SPH_C32(0x499e0200), SPH_C32(0x0d59ec0d), SPH_C32(0xe0272f7d),
	  SPH_C32(0xa5e7de5a), SPH_C32(0xa7b80200), SPH_C32(0x1f128433),
	  SPH_C32(0x60e5f9f2), SPH_C32(0x9e147576) }
};

static const sph_u32 T256_14[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xb7a40100), SPH_C32(0x8a1f31d8), SPH_C32(0x8589d8ab),
	  SPH_C32(0xe6c46464), SPH_C32(0x734c0000), SPH_C32(0x956fa7d6),
	  SPH_C32(0xa29d1297), SPH_C32(0x6ee56854) },
	{ SPH_C32(0x734c0000), SPH_C32(0x956fa7d6), SPH_C32(0xa29d1297),
	  SPH_C32(0x6ee56854), SPH_C32(0xc4e80100), SPH_C32(0x1f70960e),
	  SPH_C32(0x2714ca3c), SPH_C32(0x88210c30) },
	{ SPH_C32(0xc4e80100), SPH_C32(0x1f70960e), SPH_C32(0x2714ca3c),
	  SPH_C32(0x88210c30), SPH_C32(0xb7a40100), SPH_C32(0x8a1f31d8),
	  SPH_C32(0x8589d8ab), SPH_C32(0xe6c46464) }
};

static const sph_u32 T256_16[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x5bd20080), SPH_C32(0x450f18ec), SPH_C32(0xc2c46c55),
	  SPH_C32(0xf362b233), SPH_C32(0x39a60000), SPH_C32(0x4ab753eb),
	  SPH_C32(0xd14e094b), SPH_C32(0xb772b42b) },
	{ SPH_C32(0x39a60000), SPH_C32(0x4ab753eb), SPH_C32(0xd14e094b),
	  SPH_C32(0xb772b42b), SPH_C32(0x62740080), SPH_C32(0x0fb84b07),
	  SPH_C32(0x138a651e), SPH_C32(0x44100618) },
	{ SPH_C32(0x62740080), SPH_C32(0x0fb84b07), SPH_C32(0x138a651e),
	  SPH_C32(0x44100618), SPH_C32(0x5bd20080), SPH_C32(0x450f18ec),
	  SPH_C32(0xc2c46c55), SPH_C32(0xf362b233) }
};

static const sph_u32 T256_18[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x4dce0040), SPH_C32(0x3b5bec7e), SPH_C32(0x36656ba8),
	  SPH_C32(0x23633a05), SPH_C32(0x78ab0000), SPH_C32(0xa0cd5a34),
	  SPH_C32(0x5d5ca0f7), SPH_C32(0x727784cb) },
	{ SPH_C32(0x78ab0000), SPH_C32(0xa0cd5a34), SPH_C32(0x5d5ca0f7),
	  SPH_C32(0x727784cb), SPH_C32(0x35650040), SPH_C32(0x9b96b64a),
	  SPH_C32(0x6b39cb5f), SPH_C32(0x5114bece) },
	{ SPH_C32(0x35650040), SPH_C32(0x9b96b64a), SPH_C32(0x6b39cb5f),
	  SPH_C32(0x5114bece), SPH_C32(0x4dce0040), SPH_C32(0x3b5bec7e),
	  SPH_C32(0x36656ba8), SPH_C32(0x23633a05) }
};

static const sph_u32 T256_20[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xa2b80020), SPH_C32(0x81e7e5f6), SPH_C32(0xf9ce4c04),
	  SPH_C32(0xe2afa0c0), SPH_C32(0x5c720000), SPH_C32(0xc9bacd12),
	  SPH_C32(0x79a90df9), SPH_C32(0x63e92178) },
	{ SPH_C32(0x5c720000), SPH_C32(0xc9bacd12), SPH_C32(0x79a90df9),
	  SPH_C32(0x63e92178), SPH_C32(0xfeca0020), SPH_C32(0x485d28e4),
	  SPH_C32(0x806741fd), SPH_C32(0x814681b8) },
	{ SPH_C32(0xfeca0020), SPH_C32(0x485d28e4), SPH_C32(0x806741fd),
	  SPH_C32(0x814681b8), SPH_C32(0xa2b80020), SPH_C32(0x81e7e5f6),
	  SPH_C32(0xf9ce4c04), SPH_C32(0xe2afa0c0) }
};

static const sph_u32 T256_22[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x515c0010), SPH_C32(0x40f372fb), SPH_C32(0xfce72602),
	  SPH_C32(0x71575061), SPH_C32(0x2e390000), SPH_C32(0x64dd6689),
	  SPH_C32(0x3cd406fc), SPH_C32(0xb1f490bc) },
	{ SPH_C32(0x2e390000), SPH_C32(0x64dd6689), SPH_C32(0x3cd406fc),
	  SPH_C32(0xb1f490bc), SPH_C32(0x7f650010), SPH_C32(0x242e1472),
	  SPH_C32(0xc03320fe), SPH_C32(0xc0a3c0dd) },
	{ SPH_C32(0x7f650010), SPH_C32(0x242e1472), SPH_C32(0xc03320fe),
	  SPH_C32(0xc0a3c0dd), SPH_C32(0x515c0010), SPH_C32(0x40f372fb),
	  SPH_C32(0xfce72602), SPH_C32(0x71575061) }
};

static const sph_u32 T256_24[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xa8ae0008), SPH_C32(0x2079397d), SPH_C32(0xfe739301),
	  SPH_C32(0xb8a92831), SPH_C32(0x171c0000), SPH_C32(0xb26e3344),
	  SPH_C32(0x9e6a837e), SPH_C32(0x58f8485f) },
	{ SPH_C32(0x171c0000), SPH_C32(0xb26e3344), SPH_C32(0x9e6a837e),
	  SPH_C32(0x58f8485f), SPH_C32(0xbfb20008), SPH_C32(0x92170a39),
	  SPH_C32(0x6019107f), SPH_C32(0xe051606e) },
	{ SPH_C32(0xbfb20008), SPH_C32(0x92170a39), SPH_C32(0x6019107f),
	  SPH_C32(0xe051606e), SPH_C32(0xa8ae0008), SPH_C32(0x2079397d),
	  SPH_C32(0xfe739301), SPH_C32(0xb8a92831) }
};

static const sph_u32 T256_26[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xd0080004), SPH_C32(0x8c768f77), SPH_C32(0x9dc5b050),
	  SPH_C32(0xaf4a29da), SPH_C32(0x6ba90000), SPH_C32(0x40ebf9aa),
	  SPH_C32(0x98321c3d), SPH_C32(0x76acc733) },
	{ SPH_C32(0x6ba90000), SPH_C32(0x40ebf9aa), SPH_C32(0x98321c3d),
	  SPH_C32(0x76acc733), SPH_C32(0xbba10004), SPH_C32(0xcc9d76dd),
	  SPH_C32(0x05f7ac6d), SPH_C32(0xd9e6eee9) },
	{ SPH_C32(0xbba10004), SPH_C32(0xcc9d76dd), SPH_C32(0x05f7ac6d),
	  SPH_C32(0xd9e6eee9), SPH_C32(0xd0080004), SPH_C32(0x8c768f77),
	  SPH_C32(0x9dc5b050), SPH_C32(0xaf4a29da) }
};

static const sph_u32 T256_28[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x88230002), SPH_C32(0x5fe7a7b3), SPH_C32(0x99e585aa),
	  SPH_C32(0x8d75f7f1), SPH_C32(0x51ac0000), SPH_C32(0x25e30f14),
	  SPH_C32(0x79e22a4c), SPH_C32(0x1298bd46) },
	{ SPH_C32(0x51ac0000), SPH_C32(0x25e30f14), SPH_C32(0x79e22a4c),
	  SPH_C32(0x1298bd46), SPH_C32(0xd98f0002), SPH_C32(0x7a04a8a7),
	  SPH_C32(0xe007afe6), SPH_C32(0x9fed4ab7) },
	{ SPH_C32(0xd98f0002), SPH_C32(0x7a04a8a7), SPH_C32(0xe007afe6),
	  SPH_C32(0x9fed4ab7), SPH_C32(0x88230002), SPH_C32(0x5fe7a7b3),
	  SPH_C32(0x99e585aa), SPH_C32(0x8d75f7f1) }
};

static const sph_u32 T256_30[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xc04e0001), SPH_C32(0x33b9c010), SPH_C32(0xae0ebb05),
	  SPH_C32(0xb5a4c63b), SPH_C32(0xc8f10000), SPH_C32(0x0b2de782),
	  SPH_C32(0x6bf648a4), SPH_C32(0x539cbdbf) },
	{ SPH_C32(0xc8f10000), SPH_C32(0x0b2de782), SPH_C32(0x6bf648a4),
	  SPH_C32(0x539cbdbf), SPH_C32(0x08bf0001), SPH_C32(0x38942792),
	  SPH_C32(0xc5f8f3a1), SPH_C32(0xe6387b84) },
	{ SPH_C32(0x08bf0001), SPH_C32(0x38942792), SPH_C32(0xc5f8f3a1),
	  SPH_C32(0xe6387b84), SPH_C32(0xc04e0001), SPH_C32(0x33b9c010),
	  SPH_C32(0xae0ebb05), SPH_C32(0xb5a4c63b) }
};

#define INPUT_SMALL   do { \
		unsigned acc = buf[0]; \
		const sph_u32 *rp; \
		rp = &T256_0[acc >> 6][0]; \
		m0 = rp[0]; \
		m1 = rp[1]; \
		m2 = rp[2]; \
		m3 = rp[3]; \
		m4 = rp[4]; \
		m5 = rp[5]; \
		m6 = rp[6]; \
		m7 = rp[7]; \
		rp = &T256_2[(acc >> 4) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_4[(acc >> 2) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_6[acc & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		acc = buf[1]; \
		rp = &T256_8[acc >> 6][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_10[(acc >> 4) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_12[(acc >> 2) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_14[acc & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		acc = buf[2]; \
		rp = &T256_16[acc >> 6][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_18[(acc >> 4) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_20[(acc >> 2) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_22[acc & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		acc = buf[3]; \
		rp = &T256_24[acc >> 6][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_26[(acc >> 4) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_28[(acc >> 2) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_30[acc & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
	} while (0)

#endif

#if SPH_HAMSI_EXPAND_SMALL == 3

static const sph_u32 T256_0[8][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xe6570000), SPH_C32(0x4bb33a25), SPH_C32(0x848598ba),
	  SPH_C32(0x1041003e), SPH_C32(0xf44c4000), SPH_C32(0x10a4e3cd),
	  SPH_C32(0x097f5711), SPH_C32(0xde77cc4c) },
	{ SPH_C32(0xe4788000), SPH_C32(0x859673c1), SPH_C32(0xb5fb2452),
	  SPH_C32(0x29cc5edf), SPH_C32(0x045f0000), SPH_C32(0x9c4a93c9),
	  SPH_C32(0x62fc79d0), SPH_C32(0x731ebdc2) },
	{ SPH_C32(0x022f8000), SPH_C32(0xce2549e4), SPH_C32(0x317ebce8),
	  SPH_C32(0x398d5ee1), SPH_C32(0xf0134000), SPH_C32(0x8cee7004),
	  SPH_C32(0x6b832ec1), SPH_C32(0xad69718e) },
	{ SPH_C32(0x045f0000), SPH_C32(0x9c4a93c9), SPH_C32(0x62fc79d0),
	  SPH_C32(0x731ebdc2), SPH_C32(0xe0278000), SPH_C32(0x19dce008),
	  SPH_C32(0xd7075d82), SPH_C32(0x5ad2e31d) },
	{ SPH_C32(0xe2080000), SPH_C32(0xd7f9a9ec), SPH_C32(0xe679e16a),
	  SPH_C32(0x635fbdfc), SPH_C32(0x146bc000), SPH_C32(0x097803c5),
	  SPH_C32(0xde780a93), SPH_C32(0x84a52f51) },
	{ SPH_C32(0xe0278000), SPH_C32(0x19dce008), SPH_C32(0xd7075d82),
	  SPH_C32(0x5ad2e31d), SPH_C32(0xe4788000), SPH_C32(0x859673c1),
	  SPH_C32(0xb5fb2452), SPH_C32(0x29cc5edf) },
	{ SPH_C32(0x06708000), SPH_C32(0x526fda2d), SPH_C32(0x5382c538),
	  SPH_C32(0x4a93e323), SPH_C32(0x1034c000), SPH_C32(0x9532900c),
	  SPH_C32(0xbc847343), SPH_C32(0xf7bb9293) }
};

static const sph_u32 T256_3[8][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xe92a2000), SPH_C32(0xb4578cfc), SPH_C32(0x11fa3a57),
	  SPH_C32(0x3dc90524), SPH_C32(0x97530000), SPH_C32(0x204f6ed3),
	  SPH_C32(0x77b9e80f), SPH_C32(0xa1ec5ec1) },
	{ SPH_C32(0x97530000), SPH_C32(0x204f6ed3), SPH_C32(0x77b9e80f),
	  SPH_C32(0xa1ec5ec1), SPH_C32(0x7e792000), SPH_C32(0x9418e22f),
	  SPH_C32(0x6643d258), SPH_C32(0x9c255be5) },
	{ SPH_C32(0x7e792000), SPH_C32(0x9418e22f), SPH_C32(0x6643d258),
	  SPH_C32(0x9c255be5), SPH_C32(0xe92a2000), SPH_C32(0xb4578cfc),
	  SPH_C32(0x11fa3a57), SPH_C32(0x3dc90524) },
	{ SPH_C32(0x121b4000), SPH_C32(0x5b17d9e8), SPH_C32(0x8dfacfab),
	  SPH_C32(0xce36cc72), SPH_C32(0xe6570000), SPH_C32(0x4bb33a25),
	  SPH_C32(0x848598ba), SPH_C32(0x1041003e) },
	{ SPH_C32(0xfb316000), SPH_C32(0xef405514), SPH_C32(0x9c00f5fc),
	  SPH_C32(0xf3ffc956), SPH_C32(0x71040000), SPH_C32(0x6bfc54f6),
	  SPH_C32(0xf33c70b5), SPH_C32(0xb1ad5eff) },
	{ SPH_C32(0x85484000), SPH_C32(0x7b58b73b), SPH_C32(0xfa4327a4),
	  SPH_C32(0x6fda92b3), SPH_C32(0x982e2000), SPH_C32(0xdfabd80a),
	  SPH_C32(0xe2c64ae2), SPH_C32(0x8c645bdb) },
	{ SPH_C32(0x6c626000), SPH_C32(0xcf0f3bc7), SPH_C32(0xebb91df3),
	  SPH_C32(0x52139797), SPH_C32(0x0f7d2000), SPH_C32(0xffe4b6d9),
	  SPH_C32(0x957fa2ed), SPH_C32(0x2d88051a) }
};

static const sph_u32 T256_6[8][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xe18b0000), SPH_C32(0x5459887d), SPH_C32(0xbf1283d3),
	  SPH_C32(0x1b666a73), SPH_C32(0x3fb90800), SPH_C32(0x7cdad883),
	  SPH_C32(0xce97a914), SPH_C32(0xbdd9f5e5) },
	{ SPH_C32(0x74951000), SPH_C32(0x5a2b467e), SPH_C32(0x88fd1d2b),
	  SPH_C32(0x1ee68292), SPH_C32(0xcba90000), SPH_C32(0x90273769),
	  SPH_C32(0xbbdcf407), SPH_C32(0xd0f4af61) },
	{ SPH_C32(0x951e1000), SPH_C32(0x0e72ce03), SPH_C32(0x37ef9ef8),
	  SPH_C32(0x0580e8e1), SPH_C32(0xf4100800), SPH_C32(0xecfdefea),
	  SPH_C32(0x754b5d13), SPH_C32(0x6d2d5a84) },
	{ SPH_C32(0xcba90000), SPH_C32(0x90273769), SPH_C32(0xbbdcf407),
	  SPH_C32(0xd0f4af61), SPH_C32(0xbf3c1000), SPH_C32(0xca0c7117),
	  SPH_C32(0x3321e92c), SPH_C32(0xce122df3) },
	{ SPH_C32(0x2a220000), SPH_C32(0xc47ebf14), SPH_C32(0x04ce77d4),
	  SPH_C32(0xcb92c512), SPH_C32(0x80851800), SPH_C32(0xb6d6a994),
	  SPH_C32(0xfdb64038), SPH_C32(0x73cbd816) },
	{ SPH_C32(0xbf3c1000), SPH_C32(0xca0c7117), SPH_C32(0x3321e92c),
	  SPH_C32(0xce122df3), SPH_C32(0x74951000), SPH_C32(0x5a2b467e),
	  SPH_C32(0x88fd1d2b), SPH_C32(0x1ee68292) },
	{ SPH_C32(0x5eb71000), SPH_C32(0x9e55f96a), SPH_C32(0x8c336aff),
	  SPH_C32(0xd5744780), SPH_C32(0x4b2c1800), SPH_C32(0x26f19efd),
	  SPH_C32(0x466ab43f), SPH_C32(0xa33f7777) }
};

static const sph_u32 T256_9[8][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x8f3e0400), SPH_C32(0x0d9dc877), SPH_C32(0x6fc548e1),
	  SPH_C32(0x898d2cd6), SPH_C32(0x14bd0000), SPH_C32(0x2fba37ff),
	  SPH_C32(0x6a72e5bb), SPH_C32(0x247febe6) },
	{ SPH_C32(0x14bd0000), SPH_C32(0x2fba37ff), SPH_C32(0x6a72e5bb),
	  SPH_C32(0x247febe6), SPH_C32(0x9b830400), SPH_C32(0x2227ff88),
	  SPH_C32(0x05b7ad5a), SPH_C32(0xadf2c730) },
	{ SPH_C32(0x9b830400), SPH_C32(0x2227ff88), SPH_C32(0x05b7ad5a),
	  SPH_C32(0xadf2c730), SPH_C32(0x8f3e0400), SPH_C32(0x0d9dc877),
	  SPH_C32(0x6fc548e1), SPH_C32(0x898d2cd6) },
	{ SPH_C32(0xde320800), SPH_C32(0x288350fe), SPH_C32(0x71852ac7),
	  SPH_C32(0xa6bf9f96), SPH_C32(0xe18b0000), SPH_C32(0x5459887d),
	  SPH_C32(0xbf1283d3), SPH_C32(0x1b666a73) },
	{ SPH_C32(0x510c0c00), SPH_C32(0x251e9889), SPH_C32(0x1e406226),
	  SPH_C32(0x2f32b340), SPH_C32(0xf5360000), SPH_C32(0x7be3bf82),
	  SPH_C32(0xd5606668), SPH_C32(0x3f198195) },
	{ SPH_C32(0xca8f0800), SPH_C32(0x07396701), SPH_C32(0x1bf7cf7c),
	  SPH_C32(0x82c07470), SPH_C32(0x7a080400), SPH_C32(0x767e77f5),
	  SPH_C32(0xbaa52e89), SPH_C32(0xb694ad43) },
	{ SPH_C32(0x45b10c00), SPH_C32(0x0aa4af76), SPH_C32(0x7432879d),
	  SPH_C32(0x0b4d58a6), SPH_C32(0x6eb50400), SPH_C32(0x59c4400a),
	  SPH_C32(0xd0d7cb32), SPH_C32(0x92eb46a5) }
};

static const sph_u32 T256_12[8][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x734c0000), SPH_C32(0x956fa7d6), SPH_C32(0xa29d1297),
	  SPH_C32(0x6ee56854), SPH_C32(0xc4e80100), SPH_C32(0x1f70960e),
	  SPH_C32(0x2714ca3c), SPH_C32(0x88210c30) },
	{ SPH_C32(0xa7b80200), SPH_C32(0x1f128433), SPH_C32(0x60e5f9f2),
	  SPH_C32(0x9e147576), SPH_C32(0xee260000), SPH_C32(0x124b683e),
	  SPH_C32(0x80c2d68f), SPH_C32(0x3bf3ab2c) },
	{ SPH_C32(0xd4f40200), SPH_C32(0x8a7d23e5), SPH_C32(0xc278eb65),
	  SPH_C32(0xf0f11d22), SPH_C32(0x2ace0100), SPH_C32(0x0d3bfe30),
	  SPH_C32(0xa7d61cb3), SPH_C32(0xb3d2a71c) },
	{ SPH_C32(0xee260000), SPH_C32(0x124b683e), SPH_C32(0x80c2d68f),
	  SPH_C32(0x3bf3ab2c), SPH_C32(0x499e0200), SPH_C32(0x0d59ec0d),
	  SPH_C32(0xe0272f7d), SPH_C32(0xa5e7de5a) },
	{ SPH_C32(0x9d6a0000), SPH_C32(0x8724cfe8), SPH_C32(0x225fc418),
	  SPH_C32(0x5516c378), SPH_C32(0x8d760300), SPH_C32(0x12297a03),
	  SPH_C32(0xc733e541), SPH_C32(0x2dc6d26a) },
	{ SPH_C32(0x499e0200), SPH_C32(0x0d59ec0d), SPH_C32(0xe0272f7d),
	  SPH_C32(0xa5e7de5a), SPH_C32(0xa7b80200), SPH_C32(0x1f128433),
	  SPH_C32(0x60e5f9f2), SPH_C32(0x9e147576) },
	{ SPH_C32(0x3ad20200), SPH_C32(0x98364bdb), SPH_C32(0x42ba3dea),
	  SPH_C32(0xcb02b60e), SPH_C32(0x63500300), SPH_C32(0x0062123d),
	  SPH_C32(0x47f133ce), SPH_C32(0x16357946) }
};

static const sph_u32 T256_15[8][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x5bd20080), SPH_C32(0x450f18ec), SPH_C32(0xc2c46c55),
	  SPH_C32(0xf362b233), SPH_C32(0x39a60000), SPH_C32(0x4ab753eb),
	  SPH_C32(0xd14e094b), SPH_C32(0xb772b42b) },
	{ SPH_C32(0x39a60000), SPH_C32(0x4ab753eb), SPH_C32(0xd14e094b),
	  SPH_C32(0xb772b42b), SPH_C32(0x62740080), SPH_C32(0x0fb84b07),
	  SPH_C32(0x138a651e), SPH_C32(0x44100618) },
	{ SPH_C32(0x62740080), SPH_C32(0x0fb84b07), SPH_C32(0x138a651e),
	  SPH_C32(0x44100618), SPH_C32(0x5bd20080), SPH_C32(0x450f18ec),
	  SPH_C32(0xc2c46c55), SPH_C32(0xf362b233) },
	{ SPH_C32(0xb7a40100), SPH_C32(0x8a1f31d8), SPH_C32(0x8589d8ab),
	  SPH_C32(0xe6c46464), SPH_C32(0x734c0000), SPH_C32(0x956fa7d6),
	  SPH_C32(0xa29d1297), SPH_C32(0x6ee56854) },
	{ SPH_C32(0xec760180), SPH_C32(0xcf102934), SPH_C32(0x474db4fe),
	  SPH_C32(0x15a6d657), SPH_C32(0x4aea0000), SPH_C32(0xdfd8f43d),
	  SPH_C32(0x73d31bdc), SPH_C32(0xd997dc7f) },
	{ SPH_C32(0x8e020100), SPH_C32(0xc0a86233), SPH_C32(0x54c7d1e0),
	  SPH_C32(0x51b6d04f), SPH_C32(0x11380080), SPH_C32(0x9ad7ecd1),
	  SPH_C32(0xb1177789), SPH_C32(0x2af56e4c) },
	{ SPH_C32(0xd5d00180), SPH_C32(0x85a77adf), SPH_C32(0x9603bdb5),
	  SPH_C32(0xa2d4627c), SPH_C32(0x289e0080), SPH_C32(0xd060bf3a),
	  SPH_C32(0x60597ec2), SPH_C32(0x9d87da67) }
};

static const sph_u32 T256_18[8][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x5c720000), SPH_C32(0xc9bacd12), SPH_C32(0x79a90df9),
	  SPH_C32(0x63e92178), SPH_C32(0xfeca0020), SPH_C32(0x485d28e4),
	  SPH_C32(0x806741fd), SPH_C32(0x814681b8) },
	{ SPH_C32(0x4dce0040), SPH_C32(0x3b5bec7e), SPH_C32(0x36656ba8),
	  SPH_C32(0x23633a05), SPH_C32(0x78ab0000), SPH_C32(0xa0cd5a34),
	  SPH_C32(0x5d5ca0f7), SPH_C32(0x727784cb) },
	{ SPH_C32(0x11bc0040), SPH_C32(0xf2e1216c), SPH_C32(0x4fcc6651),
	  SPH_C32(0x408a1b7d), SPH_C32(0x86610020), SPH_C32(0xe89072d0),
	  SPH_C32(0xdd3be10a), SPH_C32(0xf3310573) },
	{ SPH_C32(0x78ab0000), SPH_C32(0xa0cd5a34), SPH_C32(0x5d5ca0f7),
	  SPH_C32(0x727784cb), SPH_C32(0x35650040), SPH_C32(0x9b96b64a),
	  SPH_C32(0x6b39cb5f), SPH_C32(0x5114bece) },
	{ SPH_C32(0x24d90000), SPH_C32(0x69779726), SPH_C32(0x24f5ad0e),
	  SPH_C32(0x119ea5b3), SPH_C32(0xcbaf0060), SPH_C32(0xd3cb9eae),
	  SPH_C32(0xeb5e8aa2), SPH_C32(0xd0523f76) },
	{ SPH_C32(0x35650040), SPH_C32(0x9b96b64a), SPH_C32(0x6b39cb5f),
	  SPH_C32(0x5114bece), SPH_C32(0x4dce0040), SPH_C32(0x3b5bec7e),
	  SPH_C32(0x36656ba8), SPH_C32(0x23633a05) },
	{ SPH_C32(0x69170040), SPH_C32(0x522c7b58), SPH_C32(0x1290c6a6),
	  SPH_C32(0x32fd9fb6), SPH_C32(0xb3040060), SPH_C32(0x7306c49a),
	  SPH_C32(0xb6022a55), SPH_C32(0xa225bbbd) }
};

static const sph_u32 T256_21[8][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x515c0010), SPH_C32(0x40f372fb), SPH_C32(0xfce72602),
	  SPH_C32(0x71575061), SPH_C32(0x2e390000), SPH_C32(0x64dd6689),
	  SPH_C32(0x3cd406fc), SPH_C32(0xb1f490bc) },
	{ SPH_C32(0x2e390000), SPH_C32(0x64dd6689), SPH_C32(0x3cd406fc),
	  SPH_C32(0xb1f490bc), SPH_C32(0x7f650010), SPH_C32(0x242e1472),
	  SPH_C32(0xc03320fe), SPH_C32(0xc0a3c0dd) },
	{ SPH_C32(0x7f650010), SPH_C32(0x242e1472), SPH_C32(0xc03320fe),
	  SPH_C32(0xc0a3c0dd), SPH_C32(0x515c0010), SPH_C32(0x40f372fb),
	  SPH_C32(0xfce72602), SPH_C32(0x71575061) },
	{ SPH_C32(0xa2b80020), SPH_C32(0x81e7e5f6), SPH_C32(0xf9ce4c04),
	  SPH_C32(0xe2afa0c0), SPH_C32(0x5c720000), SPH_C32(0xc9bacd12),
	  SPH_C32(0x79a90df9), SPH_C32(0x63e92178) },
	{ SPH_C32(0xf3e40030), SPH_C32(0xc114970d), SPH_C32(0x05296a06),
	  SPH_C32(0x93f8f0a1), SPH_C32(0x724b0000), SPH_C32(0xad67ab9b),
	  SPH_C32(0x457d0b05), SPH_C32(0xd21db1c4) },
	{ SPH_C32(0x8c810020), SPH_C32(0xe53a837f), SPH_C32(0xc51a4af8),
	  SPH_C32(0x535b307c), SPH_C32(0x23170010), SPH_C32(0xed94d960),
	  SPH_C32(0xb99a2d07), SPH_C32(0xa34ae1a5) },
	{ SPH_C32(0xdddd0030), SPH_C32(0xa5c9f184), SPH_C32(0x39fd6cfa),
	  SPH_C32(0x220c601d), SPH_C32(0x0d2e0010), SPH_C32(0x8949bfe9),
	  SPH_C32(0x854e2bfb), SPH_C32(0x12be7119) }
};

static const sph_u32 T256_24[8][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x6ba90000), SPH_C32(0x40ebf9aa), SPH_C32(0x98321c3d),
	  SPH_C32(0x76acc733), SPH_C32(0xbba10004), SPH_C32(0xcc9d76dd),
	  SPH_C32(0x05f7ac6d), SPH_C32(0xd9e6eee9) },
	{ SPH_C32(0xa8ae0008), SPH_C32(0x2079397d), SPH_C32(0xfe739301),
	  SPH_C32(0xb8a92831), SPH_C32(0x171c0000), SPH_C32(0xb26e3344),
	  SPH_C32(0x9e6a837e), SPH_C32(0x58f8485f) },
	{ SPH_C32(0xc3070008), SPH_C32(0x6092c0d7), SPH_C32(0x66418f3c),
	  SPH_C32(0xce05ef02), SPH_C32(0xacbd0004), SPH_C32(0x7ef34599),
	  SPH_C32(0x9b9d2f13), SPH_C32(0x811ea6b6) },
	{ SPH_C32(0x171c0000), SPH_C32(0xb26e3344), SPH_C32(0x9e6a837e),
	  SPH_C32(0x58f8485f), SPH_C32(0xbfb20008), SPH_C32(0x92170a39),
	  SPH_C32(0x6019107f), SPH_C32(0xe051606e) },
	{ SPH_C32(0x7cb50000), SPH_C32(0xf285caee), SPH_C32(0x06589f43),
	  SPH_C32(0x2e548f6c), SPH_C32(0x0413000c), SPH_C32(0x5e8a7ce4),
	  SPH_C32(0x65eebc12), SPH_C32(0x39b78e87) },
	{ SPH_C32(0xbfb20008), SPH_C32(0x92170a39), SPH_C32(0x6019107f),
	  SPH_C32(0xe051606e), SPH_C32(0xa8ae0008), SPH_C32(0x2079397d),
	  SPH_C32(0xfe739301), SPH_C32(0xb8a92831) },
	{ SPH_C32(0xd41b0008), SPH_C32(0xd2fcf393), SPH_C32(0xf82b0c42),
	  SPH_C32(0x96fda75d), SPH_C32(0x130f000c), SPH_C32(0xece44fa0),
	  SPH_C32(0xfb843f6c), SPH_C32(0x614fc6d8) }
};

static const sph_u32 T256_27[8][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x88230002), SPH_C32(0x5fe7a7b3), SPH_C32(0x99e585aa),
	  SPH_C32(0x8d75f7f1), SPH_C32(0x51ac0000), SPH_C32(0x25e30f14),
	  SPH_C32(0x79e22a4c), SPH_C32(0x1298bd46) },
	{ SPH_C32(0x51ac0000), SPH_C32(0x25e30f14), SPH_C32(0x79e22a4c),
	  SPH_C32(0x1298bd46), SPH_C32(0xd98f0002), SPH_C32(0x7a04a8a7),
	  SPH_C32(0xe007afe6), SPH_C32(0x9fed4ab7) },
	{ SPH_C32(0xd98f0002), SPH_C32(0x7a04a8a7), SPH_C32(0xe007afe6),
	  SPH_C32(0x9fed4ab7), SPH_C32(0x88230002), SPH_C32(0x5fe7a7b3),
	  SPH_C32(0x99e585aa), SPH_C32(0x8d75f7f1) },
	{ SPH_C32(0xd0080004), SPH_C32(0x8c768f77), SPH_C32(0x9dc5b050),
	  SPH_C32(0xaf4a29da), SPH_C32(0x6ba90000), SPH_C32(0x40ebf9aa),
	  SPH_C32(0x98321c3d), SPH_C32(0x76acc733) },
	{ SPH_C32(0x582b0006), SPH_C32(0xd39128c4), SPH_C32(0x042035fa),
	  SPH_C32(0x223fde2b), SPH_C32(0x3a050000), SPH_C32(0x6508f6be),
	  SPH_C32(0xe1d03671), SPH_C32(0x64347a75) },
	{ SPH_C32(0x81a40004), SPH_C32(0xa9958063), SPH_C32(0xe4279a1c),
	  SPH_C32(0xbdd2949c), SPH_C32(0xb2260002), SPH_C32(0x3aef510d),
	  SPH_C32(0x7835b3db), SPH_C32(0xe9418d84) },
	{ SPH_C32(0x09870006), SPH_C32(0xf67227d0), SPH_C32(0x7dc21fb6),
	  SPH_C32(0x30a7636d), SPH_C32(0xe38a0002), SPH_C32(0x1f0c5e19),
	  SPH_C32(0x01d79997), SPH_C32(0xfbd930c2) }
};

static const sph_u32 T256_30[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xc04e0001), SPH_C32(0x33b9c010), SPH_C32(0xae0ebb05),
	  SPH_C32(0xb5a4c63b), SPH_C32(0xc8f10000), SPH_C32(0x0b2de782),
	  SPH_C32(0x6bf648a4), SPH_C32(0x539cbdbf) },
	{ SPH_C32(0xc8f10000), SPH_C32(0x0b2de782), SPH_C32(0x6bf648a4),
	  SPH_C32(0x539cbdbf), SPH_C32(0x08bf0001), SPH_C32(0x38942792),
	  SPH_C32(0xc5f8f3a1), SPH_C32(0xe6387b84) },
	{ SPH_C32(0x08bf0001), SPH_C32(0x38942792), SPH_C32(0xc5f8f3a1),
	  SPH_C32(0xe6387b84), SPH_C32(0xc04e0001), SPH_C32(0x33b9c010),
	  SPH_C32(0xae0ebb05), SPH_C32(0xb5a4c63b) }
};

#define INPUT_SMALL   do { \
		unsigned acc = buf[0]; \
		const sph_u32 *rp; \
		rp = &T256_0[acc >> 5][0]; \
		m0 = rp[0]; \
		m1 = rp[1]; \
		m2 = rp[2]; \
		m3 = rp[3]; \
		m4 = rp[4]; \
		m5 = rp[5]; \
		m6 = rp[6]; \
		m7 = rp[7]; \
		rp = &T256_3[(acc >> 2) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		acc = (acc << 8) | buf[1]; \
		rp = &T256_6[(acc >> 7) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_9[(acc >> 4) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_12[(acc >> 1) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		acc = (acc << 8) | buf[2]; \
		rp = &T256_15[(acc >> 6) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_18[(acc >> 3) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_21[acc & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		acc = buf[3]; \
		rp = &T256_24[acc >> 5][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_27[(acc >> 2) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_30[acc & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
	} while (0)

#endif

#if SPH_HAMSI_EXPAND_SMALL == 4

static const sph_u32 T256_0[16][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x121b4000), SPH_C32(0x5b17d9e8), SPH_C32(0x8dfacfab),
	  SPH_C32(0xce36cc72), SPH_C32(0xe6570000), SPH_C32(0x4bb33a25),
	  SPH_C32(0x848598ba), SPH_C32(0x1041003e) },
	{ SPH_C32(0xe6570000), SPH_C32(0x4bb33a25), SPH_C32(0x848598ba),
	  SPH_C32(0x1041003e), SPH_C32(0xf44c4000), SPH_C32(0x10a4e3cd),
	  SPH_C32(0x097f5711), SPH_C32(0xde77cc4c) },
	{ SPH_C32(0xf44c4000), SPH_C32(0x10a4e3cd), SPH_C32(0x097f5711),
	  SPH_C32(0xde77cc4c), SPH_C32(0x121b4000), SPH_C32(0x5b17d9e8),
	  SPH_C32(0x8dfacfab), SPH_C32(0xce36cc72) },
	{ SPH_C32(0xe4788000), SPH_C32(0x859673c1), SPH_C32(0xb5fb2452),
	  SPH_C32(0x29cc5edf), SPH_C32(0x045f0000), SPH_C32(0x9c4a93c9),
	  SPH_C32(0x62fc79d0), SPH_C32(0x731ebdc2) },
	{ SPH_C32(0xf663c000), SPH_C32(0xde81aa29), SPH_C32(0x3801ebf9),
	  SPH_C32(0xe7fa92ad), SPH_C32(0xe2080000), SPH_C32(0xd7f9a9ec),
	  SPH_C32(0xe679e16a), SPH_C32(0x635fbdfc) },
	{ SPH_C32(0x022f8000), SPH_C32(0xce2549e4), SPH_C32(0x317ebce8),
	  SPH_C32(0x398d5ee1), SPH_C32(0xf0134000), SPH_C32(0x8cee7004),
	  SPH_C32(0x6b832ec1), SPH_C32(0xad69718e) },
	{ SPH_C32(0x1034c000), SPH_C32(0x9532900c), SPH_C32(0xbc847343),
	  SPH_C32(0xf7bb9293), SPH_C32(0x16444000), SPH_C32(0xc75d4a21),
	  SPH_C32(0xef06b67b), SPH_C32(0xbd2871b0) },
	{ SPH_C32(0x045f0000), SPH_C32(0x9c4a93c9), SPH_C32(0x62fc79d0),
	  SPH_C32(0x731ebdc2), SPH_C32(0xe0278000), SPH_C32(0x19dce008),
	  SPH_C32(0xd7075d82), SPH_C32(0x5ad2e31d) },
	{ SPH_C32(0x16444000), SPH_C32(0xc75d4a21), SPH_C32(0xef06b67b),
	  SPH_C32(0xbd2871b0), SPH_C32(0x06708000), SPH_C32(0x526fda2d),
	  SPH_C32(0x5382c538), SPH_C32(0x4a93e323) },
	{ SPH_C32(0xe2080000), SPH_C32(0xd7f9a9ec), SPH_C32(0xe679e16a),
	  SPH_C32(0x635fbdfc), SPH_C32(0x146bc000), SPH_C32(0x097803c5),
	  SPH_C32(0xde780a93), SPH_C32(0x84a52f51) },
	{ SPH_C32(0xf0134000), SPH_C32(0x8cee7004), SPH_C32(0x6b832ec1),
	  SPH_C32(0xad69718e), SPH_C32(0xf23cc000), SPH_C32(0x42cb39e0),
	  SPH_C32(0x5afd9229), SPH_C32(0x94e42f6f) },
	{ SPH_C32(0xe0278000), SPH_C32(0x19dce008), SPH_C32(0xd7075d82),
	  SPH_C32(0x5ad2e31d), SPH_C32(0xe4788000), SPH_C32(0x859673c1),
	  SPH_C32(0xb5fb2452), SPH_C32(0x29cc5edf) },
	{ SPH_C32(0xf23cc000), SPH_C32(0x42cb39e0), SPH_C32(0x5afd9229),
	  SPH_C32(0x94e42f6f), SPH_C32(0x022f8000), SPH_C32(0xce2549e4),
	  SPH_C32(0x317ebce8), SPH_C32(0x398d5ee1) },
	{ SPH_C32(0x06708000), SPH_C32(0x526fda2d), SPH_C32(0x5382c538),
	  SPH_C32(0x4a93e323), SPH_C32(0x1034c000), SPH_C32(0x9532900c),
	  SPH_C32(0xbc847343), SPH_C32(0xf7bb9293) },
	{ SPH_C32(0x146bc000), SPH_C32(0x097803c5), SPH_C32(0xde780a93),
	  SPH_C32(0x84a52f51), SPH_C32(0xf663c000), SPH_C32(0xde81aa29),
	  SPH_C32(0x3801ebf9), SPH_C32(0xe7fa92ad) }
};

static const sph_u32 T256_4[16][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x74951000), SPH_C32(0x5a2b467e), SPH_C32(0x88fd1d2b),
	  SPH_C32(0x1ee68292), SPH_C32(0xcba90000), SPH_C32(0x90273769),
	  SPH_C32(0xbbdcf407), SPH_C32(0xd0f4af61) },
	{ SPH_C32(0xcba90000), SPH_C32(0x90273769), SPH_C32(0xbbdcf407),
	  SPH_C32(0xd0f4af61), SPH_C32(0xbf3c1000), SPH_C32(0xca0c7117),
	  SPH_C32(0x3321e92c), SPH_C32(0xce122df3) },
	{ SPH_C32(0xbf3c1000), SPH_C32(0xca0c7117), SPH_C32(0x3321e92c),
	  SPH_C32(0xce122df3), SPH_C32(0x74951000), SPH_C32(0x5a2b467e),
	  SPH_C32(0x88fd1d2b), SPH_C32(0x1ee68292) },
	{ SPH_C32(0xe92a2000), SPH_C32(0xb4578cfc), SPH_C32(0x11fa3a57),
	  SPH_C32(0x3dc90524), SPH_C32(0x97530000), SPH_C32(0x204f6ed3),
	  SPH_C32(0x77b9e80f), SPH_C32(0xa1ec5ec1) },
	{ SPH_C32(0x9dbf3000), SPH_C32(0xee7cca82), SPH_C32(0x9907277c),
	  SPH_C32(0x232f87b6), SPH_C32(0x5cfa0000), SPH_C32(0xb06859ba),
	  SPH_C32(0xcc651c08), SPH_C32(0x7118f1a0) },
	{ SPH_C32(0x22832000), SPH_C32(0x2470bb95), SPH_C32(0xaa26ce50),
	  SPH_C32(0xed3daa45), SPH_C32(0x286f1000), SPH_C32(0xea431fc4),
	  SPH_C32(0x44980123), SPH_C32(0x6ffe7332) },
	{ SPH_C32(0x56163000), SPH_C32(0x7e5bfdeb), SPH_C32(0x22dbd37b),
	  SPH_C32(0xf3db28d7), SPH_C32(0xe3c61000), SPH_C32(0x7a6428ad),
	  SPH_C32(0xff44f524), SPH_C32(0xbf0adc53) },
	{ SPH_C32(0x97530000), SPH_C32(0x204f6ed3), SPH_C32(0x77b9e80f),
	  SPH_C32(0xa1ec5ec1), SPH_C32(0x7e792000), SPH_C32(0x9418e22f),
	  SPH_C32(0x6643d258), SPH_C32(0x9c255be5) },
	{ SPH_C32(0xe3c61000), SPH_C32(0x7a6428ad), SPH_C32(0xff44f524),
	  SPH_C32(0xbf0adc53), SPH_C32(0xb5d02000), SPH_C32(0x043fd546),
	  SPH_C32(0xdd9f265f), SPH_C32(0x4cd1f484) },
	{ SPH_C32(0x5cfa0000), SPH_C32(0xb06859ba), SPH_C32(0xcc651c08),
	  SPH_C32(0x7118f1a0), SPH_C32(0xc1453000), SPH_C32(0x5e149338),
	  SPH_C32(0x55623b74), SPH_C32(0x52377616) },
	{ SPH_C32(0x286f1000), SPH_C32(0xea431fc4), SPH_C32(0x44980123),
	  SPH_C32(0x6ffe7332), SPH_C32(0x0aec3000), SPH_C32(0xce33a451),
	  SPH_C32(0xeebecf73), SPH_C32(0x82c3d977) },
	{ SPH_C32(0x7e792000), SPH_C32(0x9418e22f), SPH_C32(0x6643d258),
	  SPH_C32(0x9c255be5), SPH_C32(0xe92a2000), SPH_C32(0xb4578cfc),
	  SPH_C32(0x11fa3a57), SPH_C32(0x3dc90524) },
	{ SPH_C32(0x0aec3000), SPH_C32(0xce33a451), SPH_C32(0xeebecf73),
	  SPH_C32(0x82c3d977), SPH_C32(0x22832000), SPH_C32(0x2470bb95),
	  SPH_C32(0xaa26ce50), SPH_C32(0xed3daa45) },
	{ SPH_C32(0xb5d02000), SPH_C32(0x043fd546), SPH_C32(0xdd9f265f),
	  SPH_C32(0x4cd1f484), SPH_C32(0x56163000), SPH_C32(0x7e5bfdeb),
	  SPH_C32(0x22dbd37b), SPH_C32(0xf3db28d7) },
	{ SPH_C32(0xc1453000), SPH_C32(0x5e149338), SPH_C32(0x55623b74),
	  SPH_C32(0x52377616), SPH_C32(0x9dbf3000), SPH_C32(0xee7cca82),
	  SPH_C32(0x9907277c), SPH_C32(0x232f87b6) }
};

static const sph_u32 T256_8[16][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x8f3e0400), SPH_C32(0x0d9dc877), SPH_C32(0x6fc548e1),
	  SPH_C32(0x898d2cd6), SPH_C32(0x14bd0000), SPH_C32(0x2fba37ff),
	  SPH_C32(0x6a72e5bb), SPH_C32(0x247febe6) },
	{ SPH_C32(0x14bd0000), SPH_C32(0x2fba37ff), SPH_C32(0x6a72e5bb),
	  SPH_C32(0x247febe6), SPH_C32(0x9b830400), SPH_C32(0x2227ff88),
	  SPH_C32(0x05b7ad5a), SPH_C32(0xadf2c730) },
	{ SPH_C32(0x9b830400), SPH_C32(0x2227ff88), SPH_C32(0x05b7ad5a),
	  SPH_C32(0xadf2c730), SPH_C32(0x8f3e0400), SPH_C32(0x0d9dc877),
	  SPH_C32(0x6fc548e1), SPH_C32(0x898d2cd6) },
	{ SPH_C32(0xde320800), SPH_C32(0x288350fe), SPH_C32(0x71852ac7),
	  SPH_C32(0xa6bf9f96), SPH_C32(0xe18b0000), SPH_C32(0x5459887d),
	  SPH_C32(0xbf1283d3), SPH_C32(0x1b666a73) },
	{ SPH_C32(0x510c0c00), SPH_C32(0x251e9889), SPH_C32(0x1e406226),
	  SPH_C32(0x2f32b340), SPH_C32(0xf5360000), SPH_C32(0x7be3bf82),
	  SPH_C32(0xd5606668), SPH_C32(0x3f198195) },
	{ SPH_C32(0xca8f0800), SPH_C32(0x07396701), SPH_C32(0x1bf7cf7c),
	  SPH_C32(0x82c07470), SPH_C32(0x7a080400), SPH_C32(0x767e77f5),
	  SPH_C32(0xbaa52e89), SPH_C32(0xb694ad43) },
	{ SPH_C32(0x45b10c00), SPH_C32(0x0aa4af76), SPH_C32(0x7432879d),
	  SPH_C32(0x0b4d58a6), SPH_C32(0x6eb50400), SPH_C32(0x59c4400a),
	  SPH_C32(0xd0d7cb32), SPH_C32(0x92eb46a5) },
	{ SPH_C32(0xe18b0000), SPH_C32(0x5459887d), SPH_C32(0xbf1283d3),
	  SPH_C32(0x1b666a73), SPH_C32(0x3fb90800), SPH_C32(0x7cdad883),
	  SPH_C32(0xce97a914), SPH_C32(0xbdd9f5e5) },
	{ SPH_C32(0x6eb50400), SPH_C32(0x59c4400a), SPH_C32(0xd0d7cb32),
	  SPH_C32(0x92eb46a5), SPH_C32(0x2b040800), SPH_C32(0x5360ef7c),
	  SPH_C32(0xa4e54caf), SPH_C32(0x99a61e03) },
	{ SPH_C32(0xf5360000), SPH_C32(0x7be3bf82), SPH_C32(0xd5606668),
	  SPH_C32(0x3f198195), SPH_C32(0xa43a0c00), SPH_C32(0x5efd270b),
	  SPH_C32(0xcb20044e), SPH_C32(0x102b32d5) },
	{ SPH_C32(0x7a080400), SPH_C32(0x767e77f5), SPH_C32(0xbaa52e89),
	  SPH_C32(0xb694ad43), SPH_C32(0xb0870c00), SPH_C32(0x714710f4),
	  SPH_C32(0xa152e1f5), SPH_C32(0x3454d933) },
	{ SPH_C32(0x3fb90800), SPH_C32(0x7cdad883), SPH_C32(0xce97a914),
	  SPH_C32(0xbdd9f5e5), SPH_C32(0xde320800), SPH_C32(0x288350fe),
	  SPH_C32(0x71852ac7), SPH_C32(0xa6bf9f96) },
	{ SPH_C32(0xb0870c00), SPH_C32(0x714710f4), SPH_C32(0xa152e1f5),
	  SPH_C32(0x3454d933), SPH_C32(0xca8f0800), SPH_C32(0x07396701),
	  SPH_C32(0x1bf7cf7c), SPH_C32(0x82c07470) },
	{ SPH_C32(0x2b040800), SPH_C32(0x5360ef7c), SPH_C32(0xa4e54caf),
	  SPH_C32(0x99a61e03), SPH_C32(0x45b10c00), SPH_C32(0x0aa4af76),
	  SPH_C32(0x7432879d), SPH_C32(0x0b4d58a6) },
	{ SPH_C32(0xa43a0c00), SPH_C32(0x5efd270b), SPH_C32(0xcb20044e),
	  SPH_C32(0x102b32d5), SPH_C32(0x510c0c00), SPH_C32(0x251e9889),
	  SPH_C32(0x1e406226), SPH_C32(0x2f32b340) }
};

static const sph_u32 T256_12[16][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xb7a40100), SPH_C32(0x8a1f31d8), SPH_C32(0x8589d8ab),
	  SPH_C32(0xe6c46464), SPH_C32(0x734c0000), SPH_C32(0x956fa7d6),
	  SPH_C32(0xa29d1297), SPH_C32(0x6ee56854) },
	{ SPH_C32(0x734c0000), SPH_C32(0x956fa7d6), SPH_C32(0xa29d1297),
	  SPH_C32(0x6ee56854), SPH_C32(0xc4e80100), SPH_C32(0x1f70960e),
	  SPH_C32(0x2714ca3c), SPH_C32(0x88210c30) },
	{ SPH_C32(0xc4e80100), SPH_C32(0x1f70960e), SPH_C32(0x2714ca3c),
	  SPH_C32(0x88210c30), SPH_C32(0xb7a40100), SPH_C32(0x8a1f31d8),
	  SPH_C32(0x8589d8ab), SPH_C32(0xe6c46464) },
	{ SPH_C32(0xa7b80200), SPH_C32(0x1f128433), SPH_C32(0x60e5f9f2),
	  SPH_C32(0x9e147576), SPH_C32(0xee260000), SPH_C32(0x124b683e),
	  SPH_C32(0x80c2d68f), SPH_C32(0x3bf3ab2c) },
	{ SPH_C32(0x101c0300), SPH_C32(0x950db5eb), SPH_C32(0xe56c2159),
	  SPH_C32(0x78d01112), SPH_C32(0x9d6a0000), SPH_C32(0x8724cfe8),
	  SPH_C32(0x225fc418), SPH_C32(0x5516c378) },
	{ SPH_C32(0xd4f40200), SPH_C32(0x8a7d23e5), SPH_C32(0xc278eb65),
	  SPH_C32(0xf0f11d22), SPH_C32(0x2ace0100), SPH_C32(0x0d3bfe30),
	  SPH_C32(0xa7d61cb3), SPH_C32(0xb3d2a71c) },
	{ SPH_C32(0x63500300), SPH_C32(0x0062123d), SPH_C32(0x47f133ce),
	  SPH_C32(0x16357946), SPH_C32(0x59820100), SPH_C32(0x985459e6),
	  SPH_C32(0x054b0e24), SPH_C32(0xdd37cf48) },
	{ SPH_C32(0xee260000), SPH_C32(0x124b683e), SPH_C32(0x80c2d68f),
	  SPH_C32(0x3bf3ab2c), SPH_C32(0x499e0200), SPH_C32(0x0d59ec0d),
	  SPH_C32(0xe0272f7d), SPH_C32(0xa5e7de5a) },
	{ SPH_C32(0x59820100), SPH_C32(0x985459e6), SPH_C32(0x054b0e24),
	  SPH_C32(0xdd37cf48), SPH_C32(0x3ad20200), SPH_C32(0x98364bdb),
	  SPH_C32(0x42ba3dea), SPH_C32(0xcb02b60e) },
	{ SPH_C32(0x9d6a0000), SPH_C32(0x8724cfe8), SPH_C32(0x225fc418),
	  SPH_C32(0x5516c378), SPH_C32(0x8d760300), SPH_C32(0x12297a03),
	  SPH_C32(0xc733e541), SPH_C32(0x2dc6d26a) },
	{ SPH_C32(0x2ace0100), SPH_C32(0x0d3bfe30), SPH_C32(0xa7d61cb3),
	  SPH_C32(0xb3d2a71c), SPH_C32(0xfe3a0300), SPH_C32(0x8746ddd5),
	  SPH_C32(0x65aef7d6), SPH_C32(0x4323ba3e) },
	{ SPH_C32(0x499e0200), SPH_C32(0x0d59ec0d), SPH_C32(0xe0272f7d),
	  SPH_C32(0xa5e7de5a), SPH_C32(0xa7b80200), SPH_C32(0x1f128433),
	  SPH_C32(0x60e5f9f2), SPH_C32(0x9e147576) },
	{ SPH_C32(0xfe3a0300), SPH_C32(0x8746ddd5), SPH_C32(0x65aef7d6),
	  SPH_C32(0x4323ba3e), SPH_C32(0xd4f40200), SPH_C32(0x8a7d23e5),
	  SPH_C32(0xc278eb65), SPH_C32(0xf0f11d22) },
	{ SPH_C32(0x3ad20200), SPH_C32(0x98364bdb), SPH_C32(0x42ba3dea),
	  SPH_C32(0xcb02b60e), SPH_C32(0x63500300), SPH_C32(0x0062123d),
	  SPH_C32(0x47f133ce), SPH_C32(0x16357946) },
	{ SPH_C32(0x8d760300), SPH_C32(0x12297a03), SPH_C32(0xc733e541),
	  SPH_C32(0x2dc6d26a), SPH_C32(0x101c0300), SPH_C32(0x950db5eb),
	  SPH_C32(0xe56c2159), SPH_C32(0x78d01112) }
};

static const sph_u32 T256_16[16][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x4dce0040), SPH_C32(0x3b5bec7e), SPH_C32(0x36656ba8),
	  SPH_C32(0x23633a05), SPH_C32(0x78ab0000), SPH_C32(0xa0cd5a34),
	  SPH_C32(0x5d5ca0f7), SPH_C32(0x727784cb) },
	{ SPH_C32(0x78ab0000), SPH_C32(0xa0cd5a34), SPH_C32(0x5d5ca0f7),
	  SPH_C32(0x727784cb), SPH_C32(0x35650040), SPH_C32(0x9b96b64a),
	  SPH_C32(0x6b39cb5f), SPH_C32(0x5114bece) },
	{ SPH_C32(0x35650040), SPH_C32(0x9b96b64a), SPH_C32(0x6b39cb5f),
	  SPH_C32(0x5114bece), SPH_C32(0x4dce0040), SPH_C32(0x3b5bec7e),
	  SPH_C32(0x36656ba8), SPH_C32(0x23633a05) },
	{ SPH_C32(0x5bd20080), SPH_C32(0x450f18ec), SPH_C32(0xc2c46c55),
	  SPH_C32(0xf362b233), SPH_C32(0x39a60000), SPH_C32(0x4ab753eb),
	  SPH_C32(0xd14e094b), SPH_C32(0xb772b42b) },
	{ SPH_C32(0x161c00c0), SPH_C32(0x7e54f492), SPH_C32(0xf4a107fd),
	  SPH_C32(0xd0018836), SPH_C32(0x410d0000), SPH_C32(0xea7a09df),
	  SPH_C32(0x8c12a9bc), SPH_C32(0xc50530e0) },
	{ SPH_C32(0x23790080), SPH_C32(0xe5c242d8), SPH_C32(0x9f98cca2),
	  SPH_C32(0x811536f8), SPH_C32(0x0cc30040), SPH_C32(0xd121e5a1),
	  SPH_C32(0xba77c214), SPH_C32(0xe6660ae5) },
	{ SPH_C32(0x6eb700c0), SPH_C32(0xde99aea6), SPH_C32(0xa9fda70a),
	  SPH_C32(0xa2760cfd), SPH_C32(0x74680040), SPH_C32(0x71ecbf95),
	  SPH_C32(0xe72b62e3), SPH_C32(0x94118e2e) },
	{ SPH_C32(0x39a60000), SPH_C32(0x4ab753eb), SPH_C32(0xd14e094b),
	  SPH_C32(0xb772b42b), SPH_C32(0x62740080), SPH_C32(0x0fb84b07),
	  SPH_C32(0x138a651e), SPH_C32(0x44100618) },
	{ SPH_C32(0x74680040), SPH_C32(0x71ecbf95), SPH_C32(0xe72b62e3),
	  SPH_C32(0x94118e2e), SPH_C32(0x1adf0080), SPH_C32(0xaf751133),
	  SPH_C32(0x4ed6c5e9), SPH_C32(0x366782d3) },
	{ SPH_C32(0x410d0000), SPH_C32(0xea7a09df), SPH_C32(0x8c12a9bc),
	  SPH_C32(0xc50530e0), SPH_C32(0x571100c0), SPH_C32(0x942efd4d),
	  SPH_C32(0x78b3ae41), SPH_C32(0x1504b8d6) },
	{ SPH_C32(0x0cc30040), SPH_C32(0xd121e5a1), SPH_C32(0xba77c214),
	  SPH_C32(0xe6660ae5), SPH_C32(0x2fba00c0), SPH_C32(0x34e3a779),
	  SPH_C32(0x25ef0eb6), SPH_C32(0x67733c1d) },
	{ SPH_C32(0x62740080), SPH_C32(0x0fb84b07), SPH_C32(0x138a651e),
	  SPH_C32(0x44100618), SPH_C32(0x5bd20080), SPH_C32(0x450f18ec),
	  SPH_C32(0xc2c46c55), SPH_C32(0xf362b233) },
	{ SPH_C32(0x2fba00c0), SPH_C32(0x34e3a779), SPH_C32(0x25ef0eb6),
	  SPH_C32(0x67733c1d), SPH_C32(0x23790080), SPH_C32(0xe5c242d8),
	  SPH_C32(0x9f98cca2), SPH_C32(0x811536f8) },
	{ SPH_C32(0x1adf0080), SPH_C32(0xaf751133), SPH_C32(0x4ed6c5e9),
	  SPH_C32(0x366782d3), SPH_C32(0x6eb700c0), SPH_C32(0xde99aea6),
	  SPH_C32(0xa9fda70a), SPH_C32(0xa2760cfd) },
	{ SPH_C32(0x571100c0), SPH_C32(0x942efd4d), SPH_C32(0x78b3ae41),
	  SPH_C32(0x1504b8d6), SPH_C32(0x161c00c0), SPH_C32(0x7e54f492),
	  SPH_C32(0xf4a107fd), SPH_C32(0xd0018836) }
};

static const sph_u32 T256_20[16][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x515c0010), SPH_C32(0x40f372fb), SPH_C32(0xfce72602),
	  SPH_C32(0x71575061), SPH_C32(0x2e390000), SPH_C32(0x64dd6689),
	  SPH_C32(0x3cd406fc), SPH_C32(0xb1f490bc) },
	{ SPH_C32(0x2e390000), SPH_C32(0x64dd6689), SPH_C32(0x3cd406fc),
	  SPH_C32(0xb1f490bc), SPH_C32(0x7f650010), SPH_C32(0x242e1472),
	  SPH_C32(0xc03320fe), SPH_C32(0xc0a3c0dd) },
	{ SPH_C32(0x7f650010), SPH_C32(0x242e1472), SPH_C32(0xc03320fe),
	  SPH_C32(0xc0a3c0dd), SPH_C32(0x515c0010), SPH_C32(0x40f372fb),
	  SPH_C32(0xfce72602), SPH_C32(0x71575061) },
	{ SPH_C32(0xa2b80020), SPH_C32(0x81e7e5f6), SPH_C32(0xf9ce4c04),
	  SPH_C32(0xe2afa0c0), SPH_C32(0x5c720000), SPH_C32(0xc9bacd12),
	  SPH_C32(0x79a90df9), SPH_C32(0x63e92178) },
	{ SPH_C32(0xf3e40030), SPH_C32(0xc114970d), SPH_C32(0x05296a06),
	  SPH_C32(0x93f8f0a1), SPH_C32(0x724b0000), SPH_C32(0xad67ab9b),
	  SPH_C32(0x457d0b05), SPH_C32(0xd21db1c4) },
	{ SPH_C32(0x8c810020), SPH_C32(0xe53a837f), SPH_C32(0xc51a4af8),
	  SPH_C32(0x535b307c), SPH_C32(0x23170010), SPH_C32(0xed94d960),
	  SPH_C32(0xb99a2d07), SPH_C32(0xa34ae1a5) },
	{ SPH_C32(0xdddd0030), SPH_C32(0xa5c9f184), SPH_C32(0x39fd6cfa),
	  SPH_C32(0x220c601d), SPH_C32(0x0d2e0010), SPH_C32(0x8949bfe9),
	  SPH_C32(0x854e2bfb), SPH_C32(0x12be7119) },
	{ SPH_C32(0x5c720000), SPH_C32(0xc9bacd12), SPH_C32(0x79a90df9),
	  SPH_C32(0x63e92178), SPH_C32(0xfeca0020), SPH_C32(0x485d28e4),
	  SPH_C32(0x806741fd), SPH_C32(0x814681b8) },
	{ SPH_C32(0x0d2e0010), SPH_C32(0x8949bfe9), SPH_C32(0x854e2bfb),
	  SPH_C32(0x12be7119), SPH_C32(0xd0f30020), SPH_C32(0x2c804e6d),
	  SPH_C32(0xbcb34701), SPH_C32(0x30b21104) },
	{ SPH_C32(0x724b0000), SPH_C32(0xad67ab9b), SPH_C32(0x457d0b05),
	  SPH_C32(0xd21db1c4), SPH_C32(0x81af0030), SPH_C32(0x6c733c96),
	  SPH_C32(0x40546103), SPH_C32(0x41e54165) },
	{ SPH_C32(0x23170010), SPH_C32(0xed94d960), SPH_C32(0xb99a2d07),
	  SPH_C32(0xa34ae1a5), SPH_C32(0xaf960030), SPH_C32(0x08ae5a1f),
	  SPH_C32(0x7c8067ff), SPH_C32(0xf011d1d9) },
	{ SPH_C32(0xfeca0020), SPH_C32(0x485d28e4), SPH_C32(0x806741fd),
	  SPH_C32(0x814681b8), SPH_C32(0xa2b80020), SPH_C32(0x81e7e5f6),
	  SPH_C32(0xf9ce4c04), SPH_C32(0xe2afa0c0) },
	{ SPH_C32(0xaf960030), SPH_C32(0x08ae5a1f), SPH_C32(0x7c8067ff),
	  SPH_C32(0xf011d1d9), SPH_C32(0x8c810020), SPH_C32(0xe53a837f),
	  SPH_C32(0xc51a4af8), SPH_C32(0x535b307c) },
	{ SPH_C32(0xd0f30020), SPH_C32(0x2c804e6d), SPH_C32(0xbcb34701),
	  SPH_C32(0x30b21104), SPH_C32(0xdddd0030), SPH_C32(0xa5c9f184),
	  SPH_C32(0x39fd6cfa), SPH_C32(0x220c601d) },
	{ SPH_C32(0x81af0030), SPH_C32(0x6c733c96), SPH_C32(0x40546103),
	  SPH_C32(0x41e54165), SPH_C32(0xf3e40030), SPH_C32(0xc114970d),
	  SPH_C32(0x05296a06), SPH_C32(0x93f8f0a1) }
};

static const sph_u32 T256_24[16][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xd0080004), SPH_C32(0x8c768f77), SPH_C32(0x9dc5b050),
	  SPH_C32(0xaf4a29da), SPH_C32(0x6ba90000), SPH_C32(0x40ebf9aa),
	  SPH_C32(0x98321c3d), SPH_C32(0x76acc733) },
	{ SPH_C32(0x6ba90000), SPH_C32(0x40ebf9aa), SPH_C32(0x98321c3d),
	  SPH_C32(0x76acc733), SPH_C32(0xbba10004), SPH_C32(0xcc9d76dd),
	  SPH_C32(0x05f7ac6d), SPH_C32(0xd9e6eee9) },
	{ SPH_C32(0xbba10004), SPH_C32(0xcc9d76dd), SPH_C32(0x05f7ac6d),
	  SPH_C32(0xd9e6eee9), SPH_C32(0xd0080004), SPH_C32(0x8c768f77),
	  SPH_C32(0x9dc5b050), SPH_C32(0xaf4a29da) },
	{ SPH_C32(0xa8ae0008), SPH_C32(0x2079397d), SPH_C32(0xfe739301),
	  SPH_C32(0xb8a92831), SPH_C32(0x171c0000), SPH_C32(0xb26e3344),
	  SPH_C32(0x9e6a837e), SPH_C32(0x58f8485f) },
	{ SPH_C32(0x78a6000c), SPH_C32(0xac0fb60a), SPH_C32(0x63b62351),
	  SPH_C32(0x17e301eb), SPH_C32(0x7cb50000), SPH_C32(0xf285caee),
	  SPH_C32(0x06589f43), SPH_C32(0x2e548f6c) },
	{ SPH_C32(0xc3070008), SPH_C32(0x6092c0d7), SPH_C32(0x66418f3c),
	  SPH_C32(0xce05ef02), SPH_C32(0xacbd0004), SPH_C32(0x7ef34599),
	  SPH_C32(0x9b9d2f13), SPH_C32(0x811ea6b6) },
	{ SPH_C32(0x130f000c), SPH_C32(0xece44fa0), SPH_C32(0xfb843f6c),
	  SPH_C32(0x614fc6d8), SPH_C32(0xc7140004), SPH_C32(0x3e18bc33),
	  SPH_C32(0x03af332e), SPH_C32(0xf7b26185) },
	{ SPH_C32(0x171c0000), SPH_C32(0xb26e3344), SPH_C32(0x9e6a837e),
	  SPH_C32(0x58f8485f), SPH_C32(0xbfb20008), SPH_C32(0x92170a39),
	  SPH_C32(0x6019107f), SPH_C32(0xe051606e) },
	{ SPH_C32(0xc7140004), SPH_C32(0x3e18bc33), SPH_C32(0x03af332e),
	  SPH_C32(0xf7b26185), SPH_C32(0xd41b0008), SPH_C32(0xd2fcf393),
	  SPH_C32(0xf82b0c42), SPH_C32(0x96fda75d) },
	{ SPH_C32(0x7cb50000), SPH_C32(0xf285caee), SPH_C32(0x06589f43),
	  SPH_C32(0x2e548f6c), SPH_C32(0x0413000c), SPH_C32(0x5e8a7ce4),
	  SPH_C32(0x65eebc12), SPH_C32(0x39b78e87) },
	{ SPH_C32(0xacbd0004), SPH_C32(0x7ef34599), SPH_C32(0x9b9d2f13),
	  SPH_C32(0x811ea6b6), SPH_C32(0x6fba000c), SPH_C32(0x1e61854e),
	  SPH_C32(0xfddca02f), SPH_C32(0x4f1b49b4) },
	{ SPH_C32(0xbfb20008), SPH_C32(0x92170a39), SPH_C32(0x6019107f),
	  SPH_C32(0xe051606e), SPH_C32(0xa8ae0008), SPH_C32(0x2079397d),
	  SPH_C32(0xfe739301), SPH_C32(0xb8a92831) },
	{ SPH_C32(0x6fba000c), SPH_C32(0x1e61854e), SPH_C32(0xfddca02f),
	  SPH_C32(0x4f1b49b4), SPH_C32(0xc3070008), SPH_C32(0x6092c0d7),
	  SPH_C32(0x66418f3c), SPH_C32(0xce05ef02) },
	{ SPH_C32(0xd41b0008), SPH_C32(0xd2fcf393), SPH_C32(0xf82b0c42),
	  SPH_C32(0x96fda75d), SPH_C32(0x130f000c), SPH_C32(0xece44fa0),
	  SPH_C32(0xfb843f6c), SPH_C32(0x614fc6d8) },
	{ SPH_C32(0x0413000c), SPH_C32(0x5e8a7ce4), SPH_C32(0x65eebc12),
	  SPH_C32(0x39b78e87), SPH_C32(0x78a6000c), SPH_C32(0xac0fb60a),
	  SPH_C32(0x63b62351), SPH_C32(0x17e301eb) }
};

static const sph_u32 T256_28[16][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xc04e0001), SPH_C32(0x33b9c010), SPH_C32(0xae0ebb05),
	  SPH_C32(0xb5a4c63b), SPH_C32(0xc8f10000), SPH_C32(0x0b2de782),
	  SPH_C32(0x6bf648a4), SPH_C32(0x539cbdbf) },
	{ SPH_C32(0xc8f10000), SPH_C32(0x0b2de782), SPH_C32(0x6bf648a4),
	  SPH_C32(0x539cbdbf), SPH_C32(0x08bf0001), SPH_C32(0x38942792),
	  SPH_C32(0xc5f8f3a1), SPH_C32(0xe6387b84) },
	{ SPH_C32(0x08bf0001), SPH_C32(0x38942792), SPH_C32(0xc5f8f3a1),
	  SPH_C32(0xe6387b84), SPH_C32(0xc04e0001), SPH_C32(0x33b9c010),
	  SPH_C32(0xae0ebb05), SPH_C32(0xb5a4c63b) },
	{ SPH_C32(0x88230002), SPH_C32(0x5fe7a7b3), SPH_C32(0x99e585aa),
	  SPH_C32(0x8d75f7f1), SPH_C32(0x51ac0000), SPH_C32(0x25e30f14),
	  SPH_C32(0x79e22a4c), SPH_C32(0x1298bd46) },
	{ SPH_C32(0x486d0003), SPH_C32(0x6c5e67a3), SPH_C32(0x37eb3eaf),
	  SPH_C32(0x38d131ca), SPH_C32(0x995d0000), SPH_C32(0x2ecee896),
	  SPH_C32(0x121462e8), SPH_C32(0x410400f9) },
	{ SPH_C32(0x40d20002), SPH_C32(0x54ca4031), SPH_C32(0xf213cd0e),
	  SPH_C32(0xdee94a4e), SPH_C32(0x59130001), SPH_C32(0x1d772886),
	  SPH_C32(0xbc1ad9ed), SPH_C32(0xf4a0c6c2) },
	{ SPH_C32(0x809c0003), SPH_C32(0x67738021), SPH_C32(0x5c1d760b),
	  SPH_C32(0x6b4d8c75), SPH_C32(0x91e20001), SPH_C32(0x165acf04),
	  SPH_C32(0xd7ec9149), SPH_C32(0xa73c7b7d) },
	{ SPH_C32(0x51ac0000), SPH_C32(0x25e30f14), SPH_C32(0x79e22a4c),
	  SPH_C32(0x1298bd46), SPH_C32(0xd98f0002), SPH_C32(0x7a04a8a7),
	  SPH_C32(0xe007afe6), SPH_C32(0x9fed4ab7) },
	{ SPH_C32(0x91e20001), SPH_C32(0x165acf04), SPH_C32(0xd7ec9149),
	  SPH_C32(0xa73c7b7d), SPH_C32(0x117e0002), SPH_C32(0x71294f25),
	  SPH_C32(0x8bf1e742), SPH_C32(0xcc71f708) },
	{ SPH_C32(0x995d0000), SPH_C32(0x2ecee896), SPH_C32(0x121462e8),
	  SPH_C32(0x410400f9), SPH_C32(0xd1300003), SPH_C32(0x42908f35),
	  SPH_C32(0x25ff5c47), SPH_C32(0x79d53133) },
	{ SPH_C32(0x59130001), SPH_C32(0x1d772886), SPH_C32(0xbc1ad9ed),
	  SPH_C32(0xf4a0c6c2), SPH_C32(0x19c10003), SPH_C32(0x49bd68b7),
	  SPH_C32(0x4e0914e3), SPH_C32(0x2a498c8c) },
	{ SPH_C32(0xd98f0002), SPH_C32(0x7a04a8a7), SPH_C32(0xe007afe6),
	  SPH_C32(0x9fed4ab7), SPH_C32(0x88230002), SPH_C32(0x5fe7a7b3),
	  SPH_C32(0x99e585aa), SPH_C32(0x8d75f7f1) },
	{ SPH_C32(0x19c10003), SPH_C32(0x49bd68b7), SPH_C32(0x4e0914e3),
	  SPH_C32(0x2a498c8c), SPH_C32(0x40d20002), SPH_C32(0x54ca4031),
	  SPH_C32(0xf213cd0e), SPH_C32(0xdee94a4e) },
	{ SPH_C32(0x117e0002), SPH_C32(0x71294f25), SPH_C32(0x8bf1e742),
	  SPH_C32(0xcc71f708), SPH_C32(0x809c0003), SPH_C32(0x67738021),
	  SPH_C32(0x5c1d760b), SPH_C32(0x6b4d8c75) },
	{ SPH_C32(0xd1300003), SPH_C32(0x42908f35), SPH_C32(0x25ff5c47),
	  SPH_C32(0x79d53133), SPH_C32(0x486d0003), SPH_C32(0x6c5e67a3),
	  SPH_C32(0x37eb3eaf), SPH_C32(0x38d131ca) }
};

#define INPUT_SMALL   do { \
		unsigned acc = buf[0]; \
		const sph_u32 *rp; \
		rp = &T256_0[acc >> 4][0]; \
		m0 = rp[0]; \
		m1 = rp[1]; \
		m2 = rp[2]; \
		m3 = rp[3]; \
		m4 = rp[4]; \
		m5 = rp[5]; \
		m6 = rp[6]; \
		m7 = rp[7]; \
		rp = &T256_4[acc & 0x0f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		acc = buf[1]; \
		rp = &T256_8[acc >> 4][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_12[acc & 0x0f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		acc = buf[2]; \
		rp = &T256_16[acc >> 4][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_20[acc & 0x0f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		acc = buf[3]; \
		rp = &T256_24[acc >> 4][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_28[acc & 0x0f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
	} while (0)

#endif

#if SPH_HAMSI_EXPAND_SMALL == 5

static const sph_u32 T256_0[32][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x97530000), SPH_C32(0x204f6ed3), SPH_C32(0x77b9e80f),
	  SPH_C32(0xa1ec5ec1), SPH_C32(0x7e792000), SPH_C32(0x9418e22f),
	  SPH_C32(0x6643d258), SPH_C32(0x9c255be5) },
	{ SPH_C32(0x121b4000), SPH_C32(0x5b17d9e8), SPH_C32(0x8dfacfab),
	  SPH_C32(0xce36cc72), SPH_C32(0xe6570000), SPH_C32(0x4bb33a25),
	  SPH_C32(0x848598ba), SPH_C32(0x1041003e) },
	{ SPH_C32(0x85484000), SPH_C32(0x7b58b73b), SPH_C32(0xfa4327a4),
	  SPH_C32(0x6fda92b3), SPH_C32(0x982e2000), SPH_C32(0xdfabd80a),
	  SPH_C32(0xe2c64ae2), SPH_C32(0x8c645bdb) },
	{ SPH_C32(0xe6570000), SPH_C32(0x4bb33a25), SPH_C32(0x848598ba),
	  SPH_C32(0x1041003e), SPH_C32(0xf44c4000), SPH_C32(0x10a4e3cd),
	  SPH_C32(0x097f5711), SPH_C32(0xde77cc4c) },
	{ SPH_C32(0x71040000), SPH_C32(0x6bfc54f6), SPH_C32(0xf33c70b5),
	  SPH_C32(0xb1ad5eff), SPH_C32(0x8a356000), SPH_C32(0x84bc01e2),
	  SPH_C32(0x6f3c8549), SPH_C32(0x425297a9) },
	{ SPH_C32(0xf44c4000), SPH_C32(0x10a4e3cd), SPH_C32(0x097f5711),
	  SPH_C32(0xde77cc4c), SPH_C32(0x121b4000), SPH_C32(0x5b17d9e8),
	  SPH_C32(0x8dfacfab), SPH_C32(0xce36cc72) },
	{ SPH_C32(0x631f4000), SPH_C32(0x30eb8d1e), SPH_C32(0x7ec6bf1e),
	  SPH_C32(0x7f9b928d), SPH_C32(0x6c626000), SPH_C32(0xcf0f3bc7),
	  SPH_C32(0xebb91df3), SPH_C32(0x52139797) },
	{ SPH_C32(0xe4788000), SPH_C32(0x859673c1), SPH_C32(0xb5fb2452),
	  SPH_C32(0x29cc5edf), SPH_C32(0x045f0000), SPH_C32(0x9c4a93c9),
	  SPH_C32(0x62fc79d0), SPH_C32(0x731ebdc2) },
	{ SPH_C32(0x732b8000), SPH_C32(0xa5d91d12), SPH_C32(0xc242cc5d),
	  SPH_C32(0x8820001e), SPH_C32(0x7a262000), SPH_C32(0x085271e6),
	  SPH_C32(0x04bfab88), SPH_C32(0xef3be627) },
	{ SPH_C32(0xf663c000), SPH_C32(0xde81aa29), SPH_C32(0x3801ebf9),
	  SPH_C32(0xe7fa92ad), SPH_C32(0xe2080000), SPH_C32(0xd7f9a9ec),
	  SPH_C32(0xe679e16a), SPH_C32(0x635fbdfc) },
	{ SPH_C32(0x6130c000), SPH_C32(0xfecec4fa), SPH_C32(0x4fb803f6),
	  SPH_C32(0x4616cc6c), SPH_C32(0x9c712000), SPH_C32(0x43e14bc3),
	  SPH_C32(0x803a3332), SPH_C32(0xff7ae619) },
	{ SPH_C32(0x022f8000), SPH_C32(0xce2549e4), SPH_C32(0x317ebce8),
	  SPH_C32(0x398d5ee1), SPH_C32(0xf0134000), SPH_C32(0x8cee7004),
	  SPH_C32(0x6b832ec1), SPH_C32(0xad69718e) },
	{ SPH_C32(0x957c8000), SPH_C32(0xee6a2737), SPH_C32(0x46c754e7),
	  SPH_C32(0x98610020), SPH_C32(0x8e6a6000), SPH_C32(0x18f6922b),
	  SPH_C32(0x0dc0fc99), SPH_C32(0x314c2a6b) },
	{ SPH_C32(0x1034c000), SPH_C32(0x9532900c), SPH_C32(0xbc847343),
	  SPH_C32(0xf7bb9293), SPH_C32(0x16444000), SPH_C32(0xc75d4a21),
	  SPH_C32(0xef06b67b), SPH_C32(0xbd2871b0) },
	{ SPH_C32(0x8767c000), SPH_C32(0xb57dfedf), SPH_C32(0xcb3d9b4c),
	  SPH_C32(0x5657cc52), SPH_C32(0x683d6000), SPH_C32(0x5345a80e),
	  SPH_C32(0x89456423), SPH_C32(0x210d2a55) },
	{ SPH_C32(0x045f0000), SPH_C32(0x9c4a93c9), SPH_C32(0x62fc79d0),
	  SPH_C32(0x731ebdc2), SPH_C32(0xe0278000), SPH_C32(0x19dce008),
	  SPH_C32(0xd7075d82), SPH_C32(0x5ad2e31d) },
	{ SPH_C32(0x930c0000), SPH_C32(0xbc05fd1a), SPH_C32(0x154591df),
	  SPH_C32(0xd2f2e303), SPH_C32(0x9e5ea000), SPH_C32(0x8dc40227),
	  SPH_C32(0xb1448fda), SPH_C32(0xc6f7b8f8) },
	{ SPH_C32(0x16444000), SPH_C32(0xc75d4a21), SPH_C32(0xef06b67b),
	  SPH_C32(0xbd2871b0), SPH_C32(0x06708000), SPH_C32(0x526fda2d),
	  SPH_C32(0x5382c538), SPH_C32(0x4a93e323) },
	{ SPH_C32(0x81174000), SPH_C32(0xe71224f2), SPH_C32(0x98bf5e74),
	  SPH_C32(0x1cc42f71), SPH_C32(0x7809a000), SPH_C32(0xc6773802),
	  SPH_C32(0x35c11760), SPH_C32(0xd6b6b8c6) },
	{ SPH_C32(0xe2080000), SPH_C32(0xd7f9a9ec), SPH_C32(0xe679e16a),
	  SPH_C32(0x635fbdfc), SPH_C32(0x146bc000), SPH_C32(0x097803c5),
	  SPH_C32(0xde780a93), SPH_C32(0x84a52f51) },
	{ SPH_C32(0x755b0000), SPH_C32(0xf7b6c73f), SPH_C32(0x91c00965),
	  SPH_C32(0xc2b3e33d), SPH_C32(0x6a12e000), SPH_C32(0x9d60e1ea),
	  SPH_C32(0xb83bd8cb), SPH_C32(0x188074b4) },
	{ SPH_C32(0xf0134000), SPH_C32(0x8cee7004), SPH_C32(0x6b832ec1),
	  SPH_C32(0xad69718e), SPH_C32(0xf23cc000), SPH_C32(0x42cb39e0),
	  SPH_C32(0x5afd9229), SPH_C32(0x94e42f6f) },
	{ SPH_C32(0x67404000), SPH_C32(0xaca11ed7), SPH_C32(0x1c3ac6ce),
	  SPH_C32(0x0c852f4f), SPH_C32(0x8c45e000), SPH_C32(0xd6d3dbcf),
	  SPH_C32(0x3cbe4071), SPH_C32(0x08c1748a) },
	{ SPH_C32(0xe0278000), SPH_C32(0x19dce008), SPH_C32(0xd7075d82),
	  SPH_C32(0x5ad2e31d), SPH_C32(0xe4788000), SPH_C32(0x859673c1),
	  SPH_C32(0xb5fb2452), SPH_C32(0x29cc5edf) },
	{ SPH_C32(0x77748000), SPH_C32(0x39938edb), SPH_C32(0xa0beb58d),
	  SPH_C32(0xfb3ebddc), SPH_C32(0x9a01a000), SPH_C32(0x118e91ee),
	  SPH_C32(0xd3b8f60a), SPH_C32(0xb5e9053a) },
	{ SPH_C32(0xf23cc000), SPH_C32(0x42cb39e0), SPH_C32(0x5afd9229),
	  SPH_C32(0x94e42f6f), SPH_C32(0x022f8000), SPH_C32(0xce2549e4),
	  SPH_C32(0x317ebce8), SPH_C32(0x398d5ee1) },
	{ SPH_C32(0x656fc000), SPH_C32(0x62845733), SPH_C32(0x2d447a26),
	  SPH_C32(0x350871ae), SPH_C32(0x7c56a000), SPH_C32(0x5a3dabcb),
	  SPH_C32(0x573d6eb0), SPH_C32(0xa5a80504) },
	{ SPH_C32(0x06708000), SPH_C32(0x526fda2d), SPH_C32(0x5382c538),
	  SPH_C32(0x4a93e323), SPH_C32(0x1034c000), SPH_C32(0x9532900c),
	  SPH_C32(0xbc847343), SPH_C32(0xf7bb9293) },
	{ SPH_C32(0x91238000), SPH_C32(0x7220b4fe), SPH_C32(0x243b2d37),
	  SPH_C32(0xeb7fbde2), SPH_C32(0x6e4de000), SPH_C32(0x012a7223),
	  SPH_C32(0xdac7a11b), SPH_C32(0x6b9ec976) },
	{ SPH_C32(0x146bc000), SPH_C32(0x097803c5), SPH_C32(0xde780a93),
	  SPH_C32(0x84a52f51), SPH_C32(0xf663c000), SPH_C32(0xde81aa29),
	  SPH_C32(0x3801ebf9), SPH_C32(0xe7fa92ad) },
	{ SPH_C32(0x8338c000), SPH_C32(0x29376d16), SPH_C32(0xa9c1e29c),
	  SPH_C32(0x25497190), SPH_C32(0x881ae000), SPH_C32(0x4a994806),
	  SPH_C32(0x5e4239a1), SPH_C32(0x7bdfc948) }
};

static const sph_u32 T256_5[32][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xde320800), SPH_C32(0x288350fe), SPH_C32(0x71852ac7),
	  SPH_C32(0xa6bf9f96), SPH_C32(0xe18b0000), SPH_C32(0x5459887d),
	  SPH_C32(0xbf1283d3), SPH_C32(0x1b666a73) },
	{ SPH_C32(0xe18b0000), SPH_C32(0x5459887d), SPH_C32(0xbf1283d3),
	  SPH_C32(0x1b666a73), SPH_C32(0x3fb90800), SPH_C32(0x7cdad883),
	  SPH_C32(0xce97a914), SPH_C32(0xbdd9f5e5) },
	{ SPH_C32(0x3fb90800), SPH_C32(0x7cdad883), SPH_C32(0xce97a914),
	  SPH_C32(0xbdd9f5e5), SPH_C32(0xde320800), SPH_C32(0x288350fe),
	  SPH_C32(0x71852ac7), SPH_C32(0xa6bf9f96) },
	{ SPH_C32(0x74951000), SPH_C32(0x5a2b467e), SPH_C32(0x88fd1d2b),
	  SPH_C32(0x1ee68292), SPH_C32(0xcba90000), SPH_C32(0x90273769),
	  SPH_C32(0xbbdcf407), SPH_C32(0xd0f4af61) },
	{ SPH_C32(0xaaa71800), SPH_C32(0x72a81680), SPH_C32(0xf97837ec),
	  SPH_C32(0xb8591d04), SPH_C32(0x2a220000), SPH_C32(0xc47ebf14),
	  SPH_C32(0x04ce77d4), SPH_C32(0xcb92c512) },
	{ SPH_C32(0x951e1000), SPH_C32(0x0e72ce03), SPH_C32(0x37ef9ef8),
	  SPH_C32(0x0580e8e1), SPH_C32(0xf4100800), SPH_C32(0xecfdefea),
	  SPH_C32(0x754b5d13), SPH_C32(0x6d2d5a84) },
	{ SPH_C32(0x4b2c1800), SPH_C32(0x26f19efd), SPH_C32(0x466ab43f),
	  SPH_C32(0xa33f7777), SPH_C32(0x159b0800), SPH_C32(0xb8a46797),
	  SPH_C32(0xca59dec0), SPH_C32(0x764b30f7) },
	{ SPH_C32(0xcba90000), SPH_C32(0x90273769), SPH_C32(0xbbdcf407),
	  SPH_C32(0xd0f4af61), SPH_C32(0xbf3c1000), SPH_C32(0xca0c7117),
	  SPH_C32(0x3321e92c), SPH_C32(0xce122df3) },
	{ SPH_C32(0x159b0800), SPH_C32(0xb8a46797), SPH_C32(0xca59dec0),
	  SPH_C32(0x764b30f7), SPH_C32(0x5eb71000), SPH_C32(0x9e55f96a),
	  SPH_C32(0x8c336aff), SPH_C32(0xd5744780) },
	{ SPH_C32(0x2a220000), SPH_C32(0xc47ebf14), SPH_C32(0x04ce77d4),
	  SPH_C32(0xcb92c512), SPH_C32(0x80851800), SPH_C32(0xb6d6a994),
	  SPH_C32(0xfdb64038), SPH_C32(0x73cbd816) },
	{ SPH_C32(0xf4100800), SPH_C32(0xecfdefea), SPH_C32(0x754b5d13),
	  SPH_C32(0x6d2d5a84), SPH_C32(0x610e1800), SPH_C32(0xe28f21e9),
	  SPH_C32(0x42a4c3eb), SPH_C32(0x68adb265) },
	{ SPH_C32(0xbf3c1000), SPH_C32(0xca0c7117), SPH_C32(0x3321e92c),
	  SPH_C32(0xce122df3), SPH_C32(0x74951000), SPH_C32(0x5a2b467e),
	  SPH_C32(0x88fd1d2b), SPH_C32(0x1ee68292) },
	{ SPH_C32(0x610e1800), SPH_C32(0xe28f21e9), SPH_C32(0x42a4c3eb),
	  SPH_C32(0x68adb265), SPH_C32(0x951e1000), SPH_C32(0x0e72ce03),
	  SPH_C32(0x37ef9ef8), SPH_C32(0x0580e8e1) },
	{ SPH_C32(0x5eb71000), SPH_C32(0x9e55f96a), SPH_C32(0x8c336aff),
	  SPH_C32(0xd5744780), SPH_C32(0x4b2c1800), SPH_C32(0x26f19efd),
	  SPH_C32(0x466ab43f), SPH_C32(0xa33f7777) },
	{ SPH_C32(0x80851800), SPH_C32(0xb6d6a994), SPH_C32(0xfdb64038),
	  SPH_C32(0x73cbd816), SPH_C32(0xaaa71800), SPH_C32(0x72a81680),
	  SPH_C32(0xf97837ec), SPH_C32(0xb8591d04) },
	{ SPH_C32(0xe92a2000), SPH_C32(0xb4578cfc), SPH_C32(0x11fa3a57),
	  SPH_C32(0x3dc90524), SPH_C32(0x97530000), SPH_C32(0x204f6ed3),
	  SPH_C32(0x77b9e80f), SPH_C32(0xa1ec5ec1) },
	{ SPH_C32(0x37182800), SPH_C32(0x9cd4dc02), SPH_C32(0x607f1090),
	  SPH_C32(0x9b769ab2), SPH_C32(0x76d80000), SPH_C32(0x7416e6ae),
	  SPH_C32(0xc8ab6bdc), SPH_C32(0xba8a34b2) },
	{ SPH_C32(0x08a12000), SPH_C32(0xe00e0481), SPH_C32(0xaee8b984),
	  SPH_C32(0x26af6f57), SPH_C32(0xa8ea0800), SPH_C32(0x5c95b650),
	  SPH_C32(0xb92e411b), SPH_C32(0x1c35ab24) },
	{ SPH_C32(0xd6932800), SPH_C32(0xc88d547f), SPH_C32(0xdf6d9343),
	  SPH_C32(0x8010f0c1), SPH_C32(0x49610800), SPH_C32(0x08cc3e2d),
	  SPH_C32(0x063cc2c8), SPH_C32(0x0753c157) },
	{ SPH_C32(0x9dbf3000), SPH_C32(0xee7cca82), SPH_C32(0x9907277c),
	  SPH_C32(0x232f87b6), SPH_C32(0x5cfa0000), SPH_C32(0xb06859ba),
	  SPH_C32(0xcc651c08), SPH_C32(0x7118f1a0) },
	{ SPH_C32(0x438d3800), SPH_C32(0xc6ff9a7c), SPH_C32(0xe8820dbb),
	  SPH_C32(0x85901820), SPH_C32(0xbd710000), SPH_C32(0xe431d1c7),
	  SPH_C32(0x73779fdb), SPH_C32(0x6a7e9bd3) },
	{ SPH_C32(0x7c343000), SPH_C32(0xba2542ff), SPH_C32(0x2615a4af),
	  SPH_C32(0x3849edc5), SPH_C32(0x63430800), SPH_C32(0xccb28139),
	  SPH_C32(0x02f2b51c), SPH_C32(0xccc10445) },
	{ SPH_C32(0xa2063800), SPH_C32(0x92a61201), SPH_C32(0x57908e68),
	  SPH_C32(0x9ef67253), SPH_C32(0x82c80800), SPH_C32(0x98eb0944),
	  SPH_C32(0xbde036cf), SPH_C32(0xd7a76e36) },
	{ SPH_C32(0x22832000), SPH_C32(0x2470bb95), SPH_C32(0xaa26ce50),
	  SPH_C32(0xed3daa45), SPH_C32(0x286f1000), SPH_C32(0xea431fc4),
	  SPH_C32(0x44980123), SPH_C32(0x6ffe7332) },
	{ SPH_C32(0xfcb12800), SPH_C32(0x0cf3eb6b), SPH_C32(0xdba3e497),
	  SPH_C32(0x4b8235d3), SPH_C32(0xc9e41000), SPH_C32(0xbe1a97b9),
	  SPH_C32(0xfb8a82f0), SPH_C32(0x74981941) },
	{ SPH_C32(0xc3082000), SPH_C32(0x702933e8), SPH_C32(0x15344d83),
	  SPH_C32(0xf65bc036), SPH_C32(0x17d61800), SPH_C32(0x9699c747),
	  SPH_C32(0x8a0fa837), SPH_C32(0xd22786d7) },
	{ SPH_C32(0x1d3a2800), SPH_C32(0x58aa6316), SPH_C32(0x64b16744),
	  SPH_C32(0x50e45fa0), SPH_C32(0xf65d1800), SPH_C32(0xc2c04f3a),
	  SPH_C32(0x351d2be4), SPH_C32(0xc941eca4) },
	{ SPH_C32(0x56163000), SPH_C32(0x7e5bfdeb), SPH_C32(0x22dbd37b),
	  SPH_C32(0xf3db28d7), SPH_C32(0xe3c61000), SPH_C32(0x7a6428ad),
	  SPH_C32(0xff44f524), SPH_C32(0xbf0adc53) },
	{ SPH_C32(0x88243800), SPH_C32(0x56d8ad15), SPH_C32(0x535ef9bc),
	  SPH_C32(0x5564b741), SPH_C32(0x024d1000), SPH_C32(0x2e3da0d0),
	  SPH_C32(0x405676f7), SPH_C32(0xa46cb620) },
	{ SPH_C32(0xb79d3000), SPH_C32(0x2a027596), SPH_C32(0x9dc950a8),
	  SPH_C32(0xe8bd42a4), SPH_C32(0xdc7f1800), SPH_C32(0x06bef02e),
	  SPH_C32(0x31d35c30), SPH_C32(0x02d329b6) },
	{ SPH_C32(0x69af3800), SPH_C32(0x02812568), SPH_C32(0xec4c7a6f),
	  SPH_C32(0x4e02dd32), SPH_C32(0x3df41800), SPH_C32(0x52e77853),
	  SPH_C32(0x8ec1dfe3), SPH_C32(0x19b543c5) }
};

static const sph_u32 T256_10[32][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x734c0000), SPH_C32(0x956fa7d6), SPH_C32(0xa29d1297),
	  SPH_C32(0x6ee56854), SPH_C32(0xc4e80100), SPH_C32(0x1f70960e),
	  SPH_C32(0x2714ca3c), SPH_C32(0x88210c30) },
	{ SPH_C32(0xa7b80200), SPH_C32(0x1f128433), SPH_C32(0x60e5f9f2),
	  SPH_C32(0x9e147576), SPH_C32(0xee260000), SPH_C32(0x124b683e),
	  SPH_C32(0x80c2d68f), SPH_C32(0x3bf3ab2c) },
	{ SPH_C32(0xd4f40200), SPH_C32(0x8a7d23e5), SPH_C32(0xc278eb65),
	  SPH_C32(0xf0f11d22), SPH_C32(0x2ace0100), SPH_C32(0x0d3bfe30),
	  SPH_C32(0xa7d61cb3), SPH_C32(0xb3d2a71c) },
	{ SPH_C32(0xee260000), SPH_C32(0x124b683e), SPH_C32(0x80c2d68f),
	  SPH_C32(0x3bf3ab2c), SPH_C32(0x499e0200), SPH_C32(0x0d59ec0d),
	  SPH_C32(0xe0272f7d), SPH_C32(0xa5e7de5a) },
	{ SPH_C32(0x9d6a0000), SPH_C32(0x8724cfe8), SPH_C32(0x225fc418),
	  SPH_C32(0x5516c378), SPH_C32(0x8d760300), SPH_C32(0x12297a03),
	  SPH_C32(0xc733e541), SPH_C32(0x2dc6d26a) },
	{ SPH_C32(0x499e0200), SPH_C32(0x0d59ec0d), SPH_C32(0xe0272f7d),
	  SPH_C32(0xa5e7de5a), SPH_C32(0xa7b80200), SPH_C32(0x1f128433),
	  SPH_C32(0x60e5f9f2), SPH_C32(0x9e147576) },
	{ SPH_C32(0x3ad20200), SPH_C32(0x98364bdb), SPH_C32(0x42ba3dea),
	  SPH_C32(0xcb02b60e), SPH_C32(0x63500300), SPH_C32(0x0062123d),
	  SPH_C32(0x47f133ce), SPH_C32(0x16357946) },
	{ SPH_C32(0x8f3e0400), SPH_C32(0x0d9dc877), SPH_C32(0x6fc548e1),
	  SPH_C32(0x898d2cd6), SPH_C32(0x14bd0000), SPH_C32(0x2fba37ff),
	  SPH_C32(0x6a72e5bb), SPH_C32(0x247febe6) },
	{ SPH_C32(0xfc720400), SPH_C32(0x98f26fa1), SPH_C32(0xcd585a76),
	  SPH_C32(0xe7684482), SPH_C32(0xd0550100), SPH_C32(0x30caa1f1),
	  SPH_C32(0x4d662f87), SPH_C32(0xac5ee7d6) },
	{ SPH_C32(0x28860600), SPH_C32(0x128f4c44), SPH_C32(0x0f20b113),
	  SPH_C32(0x179959a0), SPH_C32(0xfa9b0000), SPH_C32(0x3df15fc1),
	  SPH_C32(0xeab03334), SPH_C32(0x1f8c40ca) },
	{ SPH_C32(0x5bca0600), SPH_C32(0x87e0eb92), SPH_C32(0xadbda384),
	  SPH_C32(0x797c31f4), SPH_C32(0x3e730100), SPH_C32(0x2281c9cf),
	  SPH_C32(0xcda4f908), SPH_C32(0x97ad4cfa) },
	{ SPH_C32(0x61180400), SPH_C32(0x1fd6a049), SPH_C32(0xef079e6e),
	  SPH_C32(0xb27e87fa), SPH_C32(0x5d230200), SPH_C32(0x22e3dbf2),
	  SPH_C32(0x8a55cac6), SPH_C32(0x819835bc) },
	{ SPH_C32(0x12540400), SPH_C32(0x8ab9079f), SPH_C32(0x4d9a8cf9),
	  SPH_C32(0xdc9befae), SPH_C32(0x99cb0300), SPH_C32(0x3d934dfc),
	  SPH_C32(0xad4100fa), SPH_C32(0x09b9398c) },
	{ SPH_C32(0xc6a00600), SPH_C32(0x00c4247a), SPH_C32(0x8fe2679c),
	  SPH_C32(0x2c6af28c), SPH_C32(0xb3050200), SPH_C32(0x30a8b3cc),
	  SPH_C32(0x0a971c49), SPH_C32(0xba6b9e90) },
	{ SPH_C32(0xb5ec0600), SPH_C32(0x95ab83ac), SPH_C32(0x2d7f750b),
	  SPH_C32(0x428f9ad8), SPH_C32(0x77ed0300), SPH_C32(0x2fd825c2),
	  SPH_C32(0x2d83d675), SPH_C32(0x324a92a0) },
	{ SPH_C32(0x14bd0000), SPH_C32(0x2fba37ff), SPH_C32(0x6a72e5bb),
	  SPH_C32(0x247febe6), SPH_C32(0x9b830400), SPH_C32(0x2227ff88),
	  SPH_C32(0x05b7ad5a), SPH_C32(0xadf2c730) },
	{ SPH_C32(0x67f10000), SPH_C32(0xbad59029), SPH_C32(0xc8eff72c),
	  SPH_C32(0x4a9a83b2), SPH_C32(0x5f6b0500), SPH_C32(0x3d576986),
	  SPH_C32(0x22a36766), SPH_C32(0x25d3cb00) },
	{ SPH_C32(0xb3050200), SPH_C32(0x30a8b3cc), SPH_C32(0x0a971c49),
	  SPH_C32(0xba6b9e90), SPH_C32(0x75a50400), SPH_C32(0x306c97b6),
	  SPH_C32(0x85757bd5), SPH_C32(0x96016c1c) },
	{ SPH_C32(0xc0490200), SPH_C32(0xa5c7141a), SPH_C32(0xa80a0ede),
	  SPH_C32(0xd48ef6c4), SPH_C32(0xb14d0500), SPH_C32(0x2f1c01b8),
	  SPH_C32(0xa261b1e9), SPH_C32(0x1e20602c) },
	{ SPH_C32(0xfa9b0000), SPH_C32(0x3df15fc1), SPH_C32(0xeab03334),
	  SPH_C32(0x1f8c40ca), SPH_C32(0xd21d0600), SPH_C32(0x2f7e1385),
	  SPH_C32(0xe5908227), SPH_C32(0x0815196a) },
	{ SPH_C32(0x89d70000), SPH_C32(0xa89ef817), SPH_C32(0x482d21a3),
	  SPH_C32(0x7169289e), SPH_C32(0x16f50700), SPH_C32(0x300e858b),
	  SPH_C32(0xc284481b), SPH_C32(0x8034155a) },
	{ SPH_C32(0x5d230200), SPH_C32(0x22e3dbf2), SPH_C32(0x8a55cac6),
	  SPH_C32(0x819835bc), SPH_C32(0x3c3b0600), SPH_C32(0x3d357bbb),
	  SPH_C32(0x655254a8), SPH_C32(0x33e6b246) },
	{ SPH_C32(0x2e6f0200), SPH_C32(0xb78c7c24), SPH_C32(0x28c8d851),
	  SPH_C32(0xef7d5de8), SPH_C32(0xf8d30700), SPH_C32(0x2245edb5),
	  SPH_C32(0x42469e94), SPH_C32(0xbbc7be76) },
	{ SPH_C32(0x9b830400), SPH_C32(0x2227ff88), SPH_C32(0x05b7ad5a),
	  SPH_C32(0xadf2c730), SPH_C32(0x8f3e0400), SPH_C32(0x0d9dc877),
	  SPH_C32(0x6fc548e1), SPH_C32(0x898d2cd6) },
	{ SPH_C32(0xe8cf0400), SPH_C32(0xb748585e), SPH_C32(0xa72abfcd),
	  SPH_C32(0xc317af64), SPH_C32(0x4bd60500), SPH_C32(0x12ed5e79),
	  SPH_C32(0x48d182dd), SPH_C32(0x01ac20e6) },
	{ SPH_C32(0x3c3b0600), SPH_C32(0x3d357bbb), SPH_C32(0x655254a8),
	  SPH_C32(0x33e6b246), SPH_C32(0x61180400), SPH_C32(0x1fd6a049),
	  SPH_C32(0xef079e6e), SPH_C32(0xb27e87fa) },
	{ SPH_C32(0x4f770600), SPH_C32(0xa85adc6d), SPH_C32(0xc7cf463f),
	  SPH_C32(0x5d03da12), SPH_C32(0xa5f00500), SPH_C32(0x00a63647),
	  SPH_C32(0xc8135452), SPH_C32(0x3a5f8bca) },
	{ SPH_C32(0x75a50400), SPH_C32(0x306c97b6), SPH_C32(0x85757bd5),
	  SPH_C32(0x96016c1c), SPH_C32(0xc6a00600), SPH_C32(0x00c4247a),
	  SPH_C32(0x8fe2679c), SPH_C32(0x2c6af28c) },
	{ SPH_C32(0x06e90400), SPH_C32(0xa5033060), SPH_C32(0x27e86942),
	  SPH_C32(0xf8e40448), SPH_C32(0x02480700), SPH_C32(0x1fb4b274),
	  SPH_C32(0xa8f6ada0), SPH_C32(0xa44bfebc) },
	{ SPH_C32(0xd21d0600), SPH_C32(0x2f7e1385), SPH_C32(0xe5908227),
	  SPH_C32(0x0815196a), SPH_C32(0x28860600), SPH_C32(0x128f4c44),
	  SPH_C32(0x0f20b113), SPH_C32(0x179959a0) },
	{ SPH_C32(0xa1510600), SPH_C32(0xba11b453), SPH_C32(0x470d90b0),
	  SPH_C32(0x66f0713e), SPH_C32(0xec6e0700), SPH_C32(0x0dffda4a),
	  SPH_C32(0x28347b2f), SPH_C32(0x9fb85590) }
};

static const sph_u32 T256_15[32][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x4dce0040), SPH_C32(0x3b5bec7e), SPH_C32(0x36656ba8),
	  SPH_C32(0x23633a05), SPH_C32(0x78ab0000), SPH_C32(0xa0cd5a34),
	  SPH_C32(0x5d5ca0f7), SPH_C32(0x727784cb) },
	{ SPH_C32(0x78ab0000), SPH_C32(0xa0cd5a34), SPH_C32(0x5d5ca0f7),
	  SPH_C32(0x727784cb), SPH_C32(0x35650040), SPH_C32(0x9b96b64a),
	  SPH_C32(0x6b39cb5f), SPH_C32(0x5114bece) },
	{ SPH_C32(0x35650040), SPH_C32(0x9b96b64a), SPH_C32(0x6b39cb5f),
	  SPH_C32(0x5114bece), SPH_C32(0x4dce0040), SPH_C32(0x3b5bec7e),
	  SPH_C32(0x36656ba8), SPH_C32(0x23633a05) },
	{ SPH_C32(0x5bd20080), SPH_C32(0x450f18ec), SPH_C32(0xc2c46c55),
	  SPH_C32(0xf362b233), SPH_C32(0x39a60000), SPH_C32(0x4ab753eb),
	  SPH_C32(0xd14e094b), SPH_C32(0xb772b42b) },
	{ SPH_C32(0x161c00c0), SPH_C32(0x7e54f492), SPH_C32(0xf4a107fd),
	  SPH_C32(0xd0018836), SPH_C32(0x410d0000), SPH_C32(0xea7a09df),
	  SPH_C32(0x8c12a9bc), SPH_C32(0xc50530e0) },
	{ SPH_C32(0x23790080), SPH_C32(0xe5c242d8), SPH_C32(0x9f98cca2),
	  SPH_C32(0x811536f8), SPH_C32(0x0cc30040), SPH_C32(0xd121e5a1),
	  SPH_C32(0xba77c214), SPH_C32(0xe6660ae5) },
	{ SPH_C32(0x6eb700c0), SPH_C32(0xde99aea6), SPH_C32(0xa9fda70a),
	  SPH_C32(0xa2760cfd), SPH_C32(0x74680040), SPH_C32(0x71ecbf95),
	  SPH_C32(0xe72b62e3), SPH_C32(0x94118e2e) },
	{ SPH_C32(0x39a60000), SPH_C32(0x4ab753eb), SPH_C32(0xd14e094b),
	  SPH_C32(0xb772b42b), SPH_C32(0x62740080), SPH_C32(0x0fb84b07),
	  SPH_C32(0x138a651e), SPH_C32(0x44100618) },
	{ SPH_C32(0x74680040), SPH_C32(0x71ecbf95), SPH_C32(0xe72b62e3),
	  SPH_C32(0x94118e2e), SPH_C32(0x1adf0080), SPH_C32(0xaf751133),
	  SPH_C32(0x4ed6c5e9), SPH_C32(0x366782d3) },
	{ SPH_C32(0x410d0000), SPH_C32(0xea7a09df), SPH_C32(0x8c12a9bc),
	  SPH_C32(0xc50530e0), SPH_C32(0x571100c0), SPH_C32(0x942efd4d),
	  SPH_C32(0x78b3ae41), SPH_C32(0x1504b8d6) },
	{ SPH_C32(0x0cc30040), SPH_C32(0xd121e5a1), SPH_C32(0xba77c214),
	  SPH_C32(0xe6660ae5), SPH_C32(0x2fba00c0), SPH_C32(0x34e3a779),
	  SPH_C32(0x25ef0eb6), SPH_C32(0x67733c1d) },
	{ SPH_C32(0x62740080), SPH_C32(0x0fb84b07), SPH_C32(0x138a651e),
	  SPH_C32(0x44100618), SPH_C32(0x5bd20080), SPH_C32(0x450f18ec),
	  SPH_C32(0xc2c46c55), SPH_C32(0xf362b233) },
	{ SPH_C32(0x2fba00c0), SPH_C32(0x34e3a779), SPH_C32(0x25ef0eb6),
	  SPH_C32(0x67733c1d), SPH_C32(0x23790080), SPH_C32(0xe5c242d8),
	  SPH_C32(0x9f98cca2), SPH_C32(0x811536f8) },
	{ SPH_C32(0x1adf0080), SPH_C32(0xaf751133), SPH_C32(0x4ed6c5e9),
	  SPH_C32(0x366782d3), SPH_C32(0x6eb700c0), SPH_C32(0xde99aea6),
	  SPH_C32(0xa9fda70a), SPH_C32(0xa2760cfd) },
	{ SPH_C32(0x571100c0), SPH_C32(0x942efd4d), SPH_C32(0x78b3ae41),
	  SPH_C32(0x1504b8d6), SPH_C32(0x161c00c0), SPH_C32(0x7e54f492),
	  SPH_C32(0xf4a107fd), SPH_C32(0xd0018836) },
	{ SPH_C32(0xb7a40100), SPH_C32(0x8a1f31d8), SPH_C32(0x8589d8ab),
	  SPH_C32(0xe6c46464), SPH_C32(0x734c0000), SPH_C32(0x956fa7d6),
	  SPH_C32(0xa29d1297), SPH_C32(0x6ee56854) },
	{ SPH_C32(0xfa6a0140), SPH_C32(0xb144dda6), SPH_C32(0xb3ecb303),
	  SPH_C32(0xc5a75e61), SPH_C32(0x0be70000), SPH_C32(0x35a2fde2),
	  SPH_C32(0xffc1b260), SPH_C32(0x1c92ec9f) },
	{ SPH_C32(0xcf0f0100), SPH_C32(0x2ad26bec), SPH_C32(0xd8d5785c),
	  SPH_C32(0x94b3e0af), SPH_C32(0x46290040), SPH_C32(0x0ef9119c),
	  SPH_C32(0xc9a4d9c8), SPH_C32(0x3ff1d69a) },
	{ SPH_C32(0x82c10140), SPH_C32(0x11898792), SPH_C32(0xeeb013f4),
	  SPH_C32(0xb7d0daaa), SPH_C32(0x3e820040), SPH_C32(0xae344ba8),
	  SPH_C32(0x94f8793f), SPH_C32(0x4d865251) },
	{ SPH_C32(0xec760180), SPH_C32(0xcf102934), SPH_C32(0x474db4fe),
	  SPH_C32(0x15a6d657), SPH_C32(0x4aea0000), SPH_C32(0xdfd8f43d),
	  SPH_C32(0x73d31bdc), SPH_C32(0xd997dc7f) },
	{ SPH_C32(0xa1b801c0), SPH_C32(0xf44bc54a), SPH_C32(0x7128df56),
	  SPH_C32(0x36c5ec52), SPH_C32(0x32410000), SPH_C32(0x7f15ae09),
	  SPH_C32(0x2e8fbb2b), SPH_C32(0xabe058b4) },
	{ SPH_C32(0x94dd0180), SPH_C32(0x6fdd7300), SPH_C32(0x1a111409),
	  SPH_C32(0x67d1529c), SPH_C32(0x7f8f0040), SPH_C32(0x444e4277),
	  SPH_C32(0x18ead083), SPH_C32(0x888362b1) },
	{ SPH_C32(0xd91301c0), SPH_C32(0x54869f7e), SPH_C32(0x2c747fa1),
	  SPH_C32(0x44b26899), SPH_C32(0x07240040), SPH_C32(0xe4831843),
	  SPH_C32(0x45b67074), SPH_C32(0xfaf4e67a) },
	{ SPH_C32(0x8e020100), SPH_C32(0xc0a86233), SPH_C32(0x54c7d1e0),
	  SPH_C32(0x51b6d04f), SPH_C32(0x11380080), SPH_C32(0x9ad7ecd1),
	  SPH_C32(0xb1177789), SPH_C32(0x2af56e4c) },
	{ SPH_C32(0xc3cc0140), SPH_C32(0xfbf38e4d), SPH_C32(0x62a2ba48),
	  SPH_C32(0x72d5ea4a), SPH_C32(0x69930080), SPH_C32(0x3a1ab6e5),
	  SPH_C32(0xec4bd77e), SPH_C32(0x5882ea87) },
	{ SPH_C32(0xf6a90100), SPH_C32(0x60653807), SPH_C32(0x099b7117),
	  SPH_C32(0x23c15484), SPH_C32(0x245d00c0), SPH_C32(0x01415a9b),
	  SPH_C32(0xda2ebcd6), SPH_C32(0x7be1d082) },
	{ SPH_C32(0xbb670140), SPH_C32(0x5b3ed479), SPH_C32(0x3ffe1abf),
	  SPH_C32(0x00a26e81), SPH_C32(0x5cf600c0), SPH_C32(0xa18c00af),
	  SPH_C32(0x87721c21), SPH_C32(0x09965449) },
	{ SPH_C32(0xd5d00180), SPH_C32(0x85a77adf), SPH_C32(0x9603bdb5),
	  SPH_C32(0xa2d4627c), SPH_C32(0x289e0080), SPH_C32(0xd060bf3a),
	  SPH_C32(0x60597ec2), SPH_C32(0x9d87da67) },
	{ SPH_C32(0x981e01c0), SPH_C32(0xbefc96a1), SPH_C32(0xa066d61d),
	  SPH_C32(0x81b75879), SPH_C32(0x50350080), SPH_C32(0x70ade50e),
	  SPH_C32(0x3d05de35), SPH_C32(0xeff05eac) },
	{ SPH_C32(0xad7b0180), SPH_C32(0x256a20eb), SPH_C32(0xcb5f1d42),
	  SPH_C32(0xd0a3e6b7), SPH_C32(0x1dfb00c0), SPH_C32(0x4bf60970),
	  SPH_C32(0x0b60b59d), SPH_C32(0xcc9364a9) },
	{ SPH_C32(0xe0b501c0), SPH_C32(0x1e31cc95), SPH_C32(0xfd3a76ea),
	  SPH_C32(0xf3c0dcb2), SPH_C32(0x655000c0), SPH_C32(0xeb3b5344),
	  SPH_C32(0x563c156a), SPH_C32(0xbee4e062) }
};

static const sph_u32 T256_20[32][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x171c0000), SPH_C32(0xb26e3344), SPH_C32(0x9e6a837e),
	  SPH_C32(0x58f8485f), SPH_C32(0xbfb20008), SPH_C32(0x92170a39),
	  SPH_C32(0x6019107f), SPH_C32(0xe051606e) },
	{ SPH_C32(0x515c0010), SPH_C32(0x40f372fb), SPH_C32(0xfce72602),
	  SPH_C32(0x71575061), SPH_C32(0x2e390000), SPH_C32(0x64dd6689),
	  SPH_C32(0x3cd406fc), SPH_C32(0xb1f490bc) },
	{ SPH_C32(0x46400010), SPH_C32(0xf29d41bf), SPH_C32(0x628da57c),
	  SPH_C32(0x29af183e), SPH_C32(0x918b0008), SPH_C32(0xf6ca6cb0),
	  SPH_C32(0x5ccd1683), SPH_C32(0x51a5f0d2) },
	{ SPH_C32(0x2e390000), SPH_C32(0x64dd6689), SPH_C32(0x3cd406fc),
	  SPH_C32(0xb1f490bc), SPH_C32(0x7f650010), SPH_C32(0x242e1472),
	  SPH_C32(0xc03320fe), SPH_C32(0xc0a3c0dd) },
	{ SPH_C32(0x39250000), SPH_C32(0xd6b355cd), SPH_C32(0xa2be8582),
	  SPH_C32(0xe90cd8e3), SPH_C32(0xc0d70018), SPH_C32(0xb6391e4b),
	  SPH_C32(0xa02a3081), SPH_C32(0x20f2a0b3) },
	{ SPH_C32(0x7f650010), SPH_C32(0x242e1472), SPH_C32(0xc03320fe),
	  SPH_C32(0xc0a3c0dd), SPH_C32(0x515c0010), SPH_C32(0x40f372fb),
	  SPH_C32(0xfce72602), SPH_C32(0x71575061) },
	{ SPH_C32(0x68790010), SPH_C32(0x96402736), SPH_C32(0x5e59a380),
	  SPH_C32(0x985b8882), SPH_C32(0xeeee0018), SPH_C32(0xd2e478c2),
	  SPH_C32(0x9cfe367d), SPH_C32(0x9106300f) },
	{ SPH_C32(0xa2b80020), SPH_C32(0x81e7e5f6), SPH_C32(0xf9ce4c04),
	  SPH_C32(0xe2afa0c0), SPH_C32(0x5c720000), SPH_C32(0xc9bacd12),
	  SPH_C32(0x79a90df9), SPH_C32(0x63e92178) },
	{ SPH_C32(0xb5a40020), SPH_C32(0x3389d6b2), SPH_C32(0x67a4cf7a),
	  SPH_C32(0xba57e89f), SPH_C32(0xe3c00008), SPH_C32(0x5badc72b),
	  SPH_C32(0x19b01d86), SPH_C32(0x83b84116) },
	{ SPH_C32(0xf3e40030), SPH_C32(0xc114970d), SPH_C32(0x05296a06),
	  SPH_C32(0x93f8f0a1), SPH_C32(0x724b0000), SPH_C32(0xad67ab9b),
	  SPH_C32(0x457d0b05), SPH_C32(0xd21db1c4) },
	{ SPH_C32(0xe4f80030), SPH_C32(0x737aa449), SPH_C32(0x9b43e978),
	  SPH_C32(0xcb00b8fe), SPH_C32(0xcdf90008), SPH_C32(0x3f70a1a2),
	  SPH_C32(0x25641b7a), SPH_C32(0x324cd1aa) },
	{ SPH_C32(0x8c810020), SPH_C32(0xe53a837f), SPH_C32(0xc51a4af8),
	  SPH_C32(0x535b307c), SPH_C32(0x23170010), SPH_C32(0xed94d960),
	  SPH_C32(0xb99a2d07), SPH_C32(0xa34ae1a5) },
	{ SPH_C32(0x9b9d0020), SPH_C32(0x5754b03b), SPH_C32(0x5b70c986),
	  SPH_C32(0x0ba37823), SPH_C32(0x9ca50018), SPH_C32(0x7f83d359),
	  SPH_C32(0xd9833d78), SPH_C32(0x431b81cb) },
	{ SPH_C32(0xdddd0030), SPH_C32(0xa5c9f184), SPH_C32(0x39fd6cfa),
	  SPH_C32(0x220c601d), SPH_C32(0x0d2e0010), SPH_C32(0x8949bfe9),
	  SPH_C32(0x854e2bfb), SPH_C32(0x12be7119) },
	{ SPH_C32(0xcac10030), SPH_C32(0x17a7c2c0), SPH_C32(0xa797ef84),
	  SPH_C32(0x7af42842), SPH_C32(0xb29c0018), SPH_C32(0x1b5eb5d0),
	  SPH_C32(0xe5573b84), SPH_C32(0xf2ef1177) },
	{ SPH_C32(0x5c720000), SPH_C32(0xc9bacd12), SPH_C32(0x79a90df9),
	  SPH_C32(0x63e92178), SPH_C32(0xfeca0020), SPH_C32(0x485d28e4),
	  SPH_C32(0x806741fd), SPH_C32(0x814681b8) },
	{ SPH_C32(0x4b6e0000), SPH_C32(0x7bd4fe56), SPH_C32(0xe7c38e87),
	  SPH_C32(0x3b116927), SPH_C32(0x41780028), SPH_C32(0xda4a22dd),
	  SPH_C32(0xe07e5182), SPH_C32(0x6117e1d6) },
	{ SPH_C32(0x0d2e0010), SPH_C32(0x8949bfe9), SPH_C32(0x854e2bfb),
	  SPH_C32(0x12be7119), SPH_C32(0xd0f30020), SPH_C32(0x2c804e6d),
	  SPH_C32(0xbcb34701), SPH_C32(0x30b21104) },
	{ SPH_C32(0x1a320010), SPH_C32(0x3b278cad), SPH_C32(0x1b24a885),
	  SPH_C32(0x4a463946), SPH_C32(0x6f410028), SPH_C32(0xbe974454),
	  SPH_C32(0xdcaa577e), SPH_C32(0xd0e3716a) },
	{ SPH_C32(0x724b0000), SPH_C32(0xad67ab9b), SPH_C32(0x457d0b05),
	  SPH_C32(0xd21db1c4), SPH_C32(0x81af0030), SPH_C32(0x6c733c96),
	  SPH_C32(0x40546103), SPH_C32(0x41e54165) },
	{ SPH_C32(0x65570000), SPH_C32(0x1f0998df), SPH_C32(0xdb17887b),
	  SPH_C32(0x8ae5f99b), SPH_C32(0x3e1d0038), SPH_C32(0xfe6436af),
	  SPH_C32(0x204d717c), SPH_C32(0xa1b4210b) },
	{ SPH_C32(0x23170010), SPH_C32(0xed94d960), SPH_C32(0xb99a2d07),
	  SPH_C32(0xa34ae1a5), SPH_C32(0xaf960030), SPH_C32(0x08ae5a1f),
	  SPH_C32(0x7c8067ff), SPH_C32(0xf011d1d9) },
	{ SPH_C32(0x340b0010), SPH_C32(0x5ffaea24), SPH_C32(0x27f0ae79),
	  SPH_C32(0xfbb2a9fa), SPH_C32(0x10240038), SPH_C32(0x9ab95026),
	  SPH_C32(0x1c997780), SPH_C32(0x1040b1b7) },
	{ SPH_C32(0xfeca0020), SPH_C32(0x485d28e4), SPH_C32(0x806741fd),
	  SPH_C32(0x814681b8), SPH_C32(0xa2b80020), SPH_C32(0x81e7e5f6),
	  SPH_C32(0xf9ce4c04), SPH_C32(0xe2afa0c0) },
	{ SPH_C32(0xe9d60020), SPH_C32(0xfa331ba0), SPH_C32(0x1e0dc283),
	  SPH_C32(0xd9bec9e7), SPH_C32(0x1d0a0028), SPH_C32(0x13f0efcf),
	  SPH_C32(0x99d75c7b), SPH_C32(0x02fec0ae) },
	{ SPH_C32(0xaf960030), SPH_C32(0x08ae5a1f), SPH_C32(0x7c8067ff),
	  SPH_C32(0xf011d1d9), SPH_C32(0x8c810020), SPH_C32(0xe53a837f),
	  SPH_C32(0xc51a4af8), SPH_C32(0x535b307c) },
	{ SPH_C32(0xb88a0030), SPH_C32(0xbac0695b), SPH_C32(0xe2eae481),
	  SPH_C32(0xa8e99986), SPH_C32(0x33330028), SPH_C32(0x772d8946),
	  SPH_C32(0xa5035a87), SPH_C32(0xb30a5012) },
	{ SPH_C32(0xd0f30020), SPH_C32(0x2c804e6d), SPH_C32(0xbcb34701),
	  SPH_C32(0x30b21104), SPH_C32(0xdddd0030), SPH_C32(0xa5c9f184),
	  SPH_C32(0x39fd6cfa), SPH_C32(0x220c601d) },
	{ SPH_C32(0xc7ef0020), SPH_C32(0x9eee7d29), SPH_C32(0x22d9c47f),
	  SPH_C32(0x684a595b), SPH_C32(0x626f0038), SPH_C32(0x37defbbd),
	  SPH_C32(0x59e47c85), SPH_C32(0xc25d0073) },
	{ SPH_C32(0x81af0030), SPH_C32(0x6c733c96), SPH_C32(0x40546103),
	  SPH_C32(0x41e54165), SPH_C32(0xf3e40030), SPH_C32(0xc114970d),
	  SPH_C32(0x05296a06), SPH_C32(0x93f8f0a1) },
	{ SPH_C32(0x96b30030), SPH_C32(0xde1d0fd2), SPH_C32(0xde3ee27d),
	  SPH_C32(0x191d093a), SPH_C32(0x4c560038), SPH_C32(0x53039d34),
	  SPH_C32(0x65307a79), SPH_C32(0x73a990cf) }
};

static const sph_u32 T256_25[32][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x88230002), SPH_C32(0x5fe7a7b3), SPH_C32(0x99e585aa),
	  SPH_C32(0x8d75f7f1), SPH_C32(0x51ac0000), SPH_C32(0x25e30f14),
	  SPH_C32(0x79e22a4c), SPH_C32(0x1298bd46) },
	{ SPH_C32(0x51ac0000), SPH_C32(0x25e30f14), SPH_C32(0x79e22a4c),
	  SPH_C32(0x1298bd46), SPH_C32(0xd98f0002), SPH_C32(0x7a04a8a7),
	  SPH_C32(0xe007afe6), SPH_C32(0x9fed4ab7) },
	{ SPH_C32(0xd98f0002), SPH_C32(0x7a04a8a7), SPH_C32(0xe007afe6),
	  SPH_C32(0x9fed4ab7), SPH_C32(0x88230002), SPH_C32(0x5fe7a7b3),
	  SPH_C32(0x99e585aa), SPH_C32(0x8d75f7f1) },
	{ SPH_C32(0xd0080004), SPH_C32(0x8c768f77), SPH_C32(0x9dc5b050),
	  SPH_C32(0xaf4a29da), SPH_C32(0x6ba90000), SPH_C32(0x40ebf9aa),
	  SPH_C32(0x98321c3d), SPH_C32(0x76acc733) },
	{ SPH_C32(0x582b0006), SPH_C32(0xd39128c4), SPH_C32(0x042035fa),
	  SPH_C32(0x223fde2b), SPH_C32(0x3a050000), SPH_C32(0x6508f6be),
	  SPH_C32(0xe1d03671), SPH_C32(0x64347a75) },
	{ SPH_C32(0x81a40004), SPH_C32(0xa9958063), SPH_C32(0xe4279a1c),
	  SPH_C32(0xbdd2949c), SPH_C32(0xb2260002), SPH_C32(0x3aef510d),
	  SPH_C32(0x7835b3db), SPH_C32(0xe9418d84) },
	{ SPH_C32(0x09870006), SPH_C32(0xf67227d0), SPH_C32(0x7dc21fb6),
	  SPH_C32(0x30a7636d), SPH_C32(0xe38a0002), SPH_C32(0x1f0c5e19),
	  SPH_C32(0x01d79997), SPH_C32(0xfbd930c2) },
	{ SPH_C32(0x6ba90000), SPH_C32(0x40ebf9aa), SPH_C32(0x98321c3d),
	  SPH_C32(0x76acc733), SPH_C32(0xbba10004), SPH_C32(0xcc9d76dd),
	  SPH_C32(0x05f7ac6d), SPH_C32(0xd9e6eee9) },
	{ SPH_C32(0xe38a0002), SPH_C32(0x1f0c5e19), SPH_C32(0x01d79997),
	  SPH_C32(0xfbd930c2), SPH_C32(0xea0d0004), SPH_C32(0xe97e79c9),
	  SPH_C32(0x7c158621), SPH_C32(0xcb7e53af) },
	{ SPH_C32(0x3a050000), SPH_C32(0x6508f6be), SPH_C32(0xe1d03671),
	  SPH_C32(0x64347a75), SPH_C32(0x622e0006), SPH_C32(0xb699de7a),
	  SPH_C32(0xe5f0038b), SPH_C32(0x460ba45e) },
	{ SPH_C32(0xb2260002), SPH_C32(0x3aef510d), SPH_C32(0x7835b3db),
	  SPH_C32(0xe9418d84), SPH_C32(0x33820006), SPH_C32(0x937ad16e),
	  SPH_C32(0x9c1229c7), SPH_C32(0x54931918) },
	{ SPH_C32(0xbba10004), SPH_C32(0xcc9d76dd), SPH_C32(0x05f7ac6d),
	  SPH_C32(0xd9e6eee9), SPH_C32(0xd0080004), SPH_C32(0x8c768f77),
	  SPH_C32(0x9dc5b050), SPH_C32(0xaf4a29da) },
	{ SPH_C32(0x33820006), SPH_C32(0x937ad16e), SPH_C32(0x9c1229c7),
	  SPH_C32(0x54931918), SPH_C32(0x81a40004), SPH_C32(0xa9958063),
	  SPH_C32(0xe4279a1c), SPH_C32(0xbdd2949c) },
	{ SPH_C32(0xea0d0004), SPH_C32(0xe97e79c9), SPH_C32(0x7c158621),
	  SPH_C32(0xcb7e53af), SPH_C32(0x09870006), SPH_C32(0xf67227d0),
	  SPH_C32(0x7dc21fb6), SPH_C32(0x30a7636d) },
	{ SPH_C32(0x622e0006), SPH_C32(0xb699de7a), SPH_C32(0xe5f0038b),
	  SPH_C32(0x460ba45e), SPH_C32(0x582b0006), SPH_C32(0xd39128c4),
	  SPH_C32(0x042035fa), SPH_C32(0x223fde2b) },
	{ SPH_C32(0xa8ae0008), SPH_C32(0x2079397d), SPH_C32(0xfe739301),
	  SPH_C32(0xb8a92831), SPH_C32(0x171c0000), SPH_C32(0xb26e3344),
	  SPH_C32(0x9e6a837e), SPH_C32(0x58f8485f) },
	{ SPH_C32(0x208d000a), SPH_C32(0x7f9e9ece), SPH_C32(0x679616ab),
	  SPH_C32(0x35dcdfc0), SPH_C32(0x46b00000), SPH_C32(0x978d3c50),
	  SPH_C32(0xe788a932), SPH_C32(0x4a60f519) },
	{ SPH_C32(0xf9020008), SPH_C32(0x059a3669), SPH_C32(0x8791b94d),
	  SPH_C32(0xaa319577), SPH_C32(0xce930002), SPH_C32(0xc86a9be3),
	  SPH_C32(0x7e6d2c98), SPH_C32(0xc71502e8) },
	{ SPH_C32(0x7121000a), SPH_C32(0x5a7d91da), SPH_C32(0x1e743ce7),
	  SPH_C32(0x27446286), SPH_C32(0x9f3f0002), SPH_C32(0xed8994f7),
	  SPH_C32(0x078f06d4), SPH_C32(0xd58dbfae) },
	{ SPH_C32(0x78a6000c), SPH_C32(0xac0fb60a), SPH_C32(0x63b62351),
	  SPH_C32(0x17e301eb), SPH_C32(0x7cb50000), SPH_C32(0xf285caee),
	  SPH_C32(0x06589f43), SPH_C32(0x2e548f6c) },
	{ SPH_C32(0xf085000e), SPH_C32(0xf3e811b9), SPH_C32(0xfa53a6fb),
	  SPH_C32(0x9a96f61a), SPH_C32(0x2d190000), SPH_C32(0xd766c5fa),
	  SPH_C32(0x7fbab50f), SPH_C32(0x3ccc322a) },
	{ SPH_C32(0x290a000c), SPH_C32(0x89ecb91e), SPH_C32(0x1a54091d),
	  SPH_C32(0x057bbcad), SPH_C32(0xa53a0002), SPH_C32(0x88816249),
	  SPH_C32(0xe65f30a5), SPH_C32(0xb1b9c5db) },
	{ SPH_C32(0xa129000e), SPH_C32(0xd60b1ead), SPH_C32(0x83b18cb7),
	  SPH_C32(0x880e4b5c), SPH_C32(0xf4960002), SPH_C32(0xad626d5d),
	  SPH_C32(0x9fbd1ae9), SPH_C32(0xa321789d) },
	{ SPH_C32(0xc3070008), SPH_C32(0x6092c0d7), SPH_C32(0x66418f3c),
	  SPH_C32(0xce05ef02), SPH_C32(0xacbd0004), SPH_C32(0x7ef34599),
	  SPH_C32(0x9b9d2f13), SPH_C32(0x811ea6b6) },
	{ SPH_C32(0x4b24000a), SPH_C32(0x3f756764), SPH_C32(0xffa40a96),
	  SPH_C32(0x437018f3), SPH_C32(0xfd110004), SPH_C32(0x5b104a8d),
	  SPH_C32(0xe27f055f), SPH_C32(0x93861bf0) },
	{ SPH_C32(0x92ab0008), SPH_C32(0x4571cfc3), SPH_C32(0x1fa3a570),
	  SPH_C32(0xdc9d5244), SPH_C32(0x75320006), SPH_C32(0x04f7ed3e),
	  SPH_C32(0x7b9a80f5), SPH_C32(0x1ef3ec01) },
	{ SPH_C32(0x1a88000a), SPH_C32(0x1a966870), SPH_C32(0x864620da),
	  SPH_C32(0x51e8a5b5), SPH_C32(0x249e0006), SPH_C32(0x2114e22a),
	  SPH_C32(0x0278aab9), SPH_C32(0x0c6b5147) },
	{ SPH_C32(0x130f000c), SPH_C32(0xece44fa0), SPH_C32(0xfb843f6c),
	  SPH_C32(0x614fc6d8), SPH_C32(0xc7140004), SPH_C32(0x3e18bc33),
	  SPH_C32(0x03af332e), SPH_C32(0xf7b26185) },
	{ SPH_C32(0x9b2c000e), SPH_C32(0xb303e813), SPH_C32(0x6261bac6),
	  SPH_C32(0xec3a3129), SPH_C32(0x96b80004), SPH_C32(0x1bfbb327),
	  SPH_C32(0x7a4d1962), SPH_C32(0xe52adcc3) },
	{ SPH_C32(0x42a3000c), SPH_C32(0xc90740b4), SPH_C32(0x82661520),
	  SPH_C32(0x73d77b9e), SPH_C32(0x1e9b0006), SPH_C32(0x441c1494),
	  SPH_C32(0xe3a89cc8), SPH_C32(0x685f2b32) },
	{ SPH_C32(0xca80000e), SPH_C32(0x96e0e707), SPH_C32(0x1b83908a),
	  SPH_C32(0xfea28c6f), SPH_C32(0x4f370006), SPH_C32(0x61ff1b80),
	  SPH_C32(0x9a4ab684), SPH_C32(0x7ac79674) }
};

static const sph_u32 T256_30[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xc04e0001), SPH_C32(0x33b9c010), SPH_C32(0xae0ebb05),
	  SPH_C32(0xb5a4c63b), SPH_C32(0xc8f10000), SPH_C32(0x0b2de782),
	  SPH_C32(0x6bf648a4), SPH_C32(0x539cbdbf) },
	{ SPH_C32(0xc8f10000), SPH_C32(0x0b2de782), SPH_C32(0x6bf648a4),
	  SPH_C32(0x539cbdbf), SPH_C32(0x08bf0001), SPH_C32(0x38942792),
	  SPH_C32(0xc5f8f3a1), SPH_C32(0xe6387b84) },
	{ SPH_C32(0x08bf0001), SPH_C32(0x38942792), SPH_C32(0xc5f8f3a1),
	  SPH_C32(0xe6387b84), SPH_C32(0xc04e0001), SPH_C32(0x33b9c010),
	  SPH_C32(0xae0ebb05), SPH_C32(0xb5a4c63b) }
};

#define INPUT_SMALL   do { \
		unsigned acc = buf[0]; \
		const sph_u32 *rp; \
		rp = &T256_0[acc >> 3][0]; \
		m0 = rp[0]; \
		m1 = rp[1]; \
		m2 = rp[2]; \
		m3 = rp[3]; \
		m4 = rp[4]; \
		m5 = rp[5]; \
		m6 = rp[6]; \
		m7 = rp[7]; \
		acc = (acc << 8) | buf[1]; \
		rp = &T256_5[(acc >> 6) & 0x1f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_10[(acc >> 1) & 0x1f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		acc = (acc << 8) | buf[2]; \
		rp = &T256_15[(acc >> 4) & 0x1f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		acc = (acc << 8) | buf[3]; \
		rp = &T256_20[(acc >> 7) & 0x1f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_25[(acc >> 2) & 0x1f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_30[acc & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
	} while (0)

#endif

#if SPH_HAMSI_EXPAND_SMALL == 6

static const sph_u32 T256_0[64][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xe92a2000), SPH_C32(0xb4578cfc), SPH_C32(0x11fa3a57),
	  SPH_C32(0x3dc90524), SPH_C32(0x97530000), SPH_C32(0x204f6ed3),
	  SPH_C32(0x77b9e80f), SPH_C32(0xa1ec5ec1) },
	{ SPH_C32(0x97530000), SPH_C32(0x204f6ed3), SPH_C32(0x77b9e80f),
	  SPH_C32(0xa1ec5ec1), SPH_C32(0x7e792000), SPH_C32(0x9418e22f),
	  SPH_C32(0x6643d258), SPH_C32(0x9c255be5) },
	{ SPH_C32(0x7e792000), SPH_C32(0x9418e22f), SPH_C32(0x6643d258),
	  SPH_C32(0x9c255be5), SPH_C32(0xe92a2000), SPH_C32(0xb4578cfc),
	  SPH_C32(0x11fa3a57), SPH_C32(0x3dc90524) },
	{ SPH_C32(0x121b4000), SPH_C32(0x5b17d9e8), SPH_C32(0x8dfacfab),
	  SPH_C32(0xce36cc72), SPH_C32(0xe6570000), SPH_C32(0x4bb33a25),
	  SPH_C32(0x848598ba), SPH_C32(0x1041003e) },
	{ SPH_C32(0xfb316000), SPH_C32(0xef405514), SPH_C32(0x9c00f5fc),
	  SPH_C32(0xf3ffc956), SPH_C32(0x71040000), SPH_C32(0x6bfc54f6),
	  SPH_C32(0xf33c70b5), SPH_C32(0xb1ad5eff) },
	{ SPH_C32(0x85484000), SPH_C32(0x7b58b73b), SPH_C32(0xfa4327a4),
	  SPH_C32(0x6fda92b3), SPH_C32(0x982e2000), SPH_C32(0xdfabd80a),
	  SPH_C32(0xe2c64ae2), SPH_C32(0x8c645bdb) },
	{ SPH_C32(0x6c626000), SPH_C32(0xcf0f3bc7), SPH_C32(0xebb91df3),
	  SPH_C32(0x52139797), SPH_C32(0x0f7d2000), SPH_C32(0xffe4b6d9),
	  SPH_C32(0x957fa2ed), SPH_C32(0x2d88051a) },
	{ SPH_C32(0xe6570000), SPH_C32(0x4bb33a25), SPH_C32(0x848598ba),
	  SPH_C32(0x1041003e), SPH_C32(0xf44c4000), SPH_C32(0x10a4e3cd),
	  SPH_C32(0x097f5711), SPH_C32(0xde77cc4c) },
	{ SPH_C32(0x0f7d2000), SPH_C32(0xffe4b6d9), SPH_C32(0x957fa2ed),
	  SPH_C32(0x2d88051a), SPH_C32(0x631f4000), SPH_C32(0x30eb8d1e),
	  SPH_C32(0x7ec6bf1e), SPH_C32(0x7f9b928d) },
	{ SPH_C32(0x71040000), SPH_C32(0x6bfc54f6), SPH_C32(0xf33c70b5),
	  SPH_C32(0xb1ad5eff), SPH_C32(0x8a356000), SPH_C32(0x84bc01e2),
	  SPH_C32(0x6f3c8549), SPH_C32(0x425297a9) },
	{ SPH_C32(0x982e2000), SPH_C32(0xdfabd80a), SPH_C32(0xe2c64ae2),
	  SPH_C32(0x8c645bdb), SPH_C32(0x1d666000), SPH_C32(0xa4f36f31),
	  SPH_C32(0x18856d46), SPH_C32(0xe3bec968) },
	{ SPH_C32(0xf44c4000), SPH_C32(0x10a4e3cd), SPH_C32(0x097f5711),
	  SPH_C32(0xde77cc4c), SPH_C32(0x121b4000), SPH_C32(0x5b17d9e8),
	  SPH_C32(0x8dfacfab), SPH_C32(0xce36cc72) },
	{ SPH_C32(0x1d666000), SPH_C32(0xa4f36f31), SPH_C32(0x18856d46),
	  SPH_C32(0xe3bec968), SPH_C32(0x85484000), SPH_C32(0x7b58b73b),
	  SPH_C32(0xfa4327a4), SPH_C32(0x6fda92b3) },
	{ SPH_C32(0x631f4000), SPH_C32(0x30eb8d1e), SPH_C32(0x7ec6bf1e),
	  SPH_C32(0x7f9b928d), SPH_C32(0x6c626000), SPH_C32(0xcf0f3bc7),
	  SPH_C32(0xebb91df3), SPH_C32(0x52139797) },
	{ SPH_C32(0x8a356000), SPH_C32(0x84bc01e2), SPH_C32(0x6f3c8549),
	  SPH_C32(0x425297a9), SPH_C32(0xfb316000), SPH_C32(0xef405514),
	  SPH_C32(0x9c00f5fc), SPH_C32(0xf3ffc956) },
	{ SPH_C32(0xe4788000), SPH_C32(0x859673c1), SPH_C32(0xb5fb2452),
	  SPH_C32(0x29cc5edf), SPH_C32(0x045f0000), SPH_C32(0x9c4a93c9),
	  SPH_C32(0x62fc79d0), SPH_C32(0x731ebdc2) },
	{ SPH_C32(0x0d52a000), SPH_C32(0x31c1ff3d), SPH_C32(0xa4011e05),
	  SPH_C32(0x14055bfb), SPH_C32(0x930c0000), SPH_C32(0xbc05fd1a),
	  SPH_C32(0x154591df), SPH_C32(0xd2f2e303) },
	{ SPH_C32(0x732b8000), SPH_C32(0xa5d91d12), SPH_C32(0xc242cc5d),
	  SPH_C32(0x8820001e), SPH_C32(0x7a262000), SPH_C32(0x085271e6),
	  SPH_C32(0x04bfab88), SPH_C32(0xef3be627) },
	{ SPH_C32(0x9a01a000), SPH_C32(0x118e91ee), SPH_C32(0xd3b8f60a),
	  SPH_C32(0xb5e9053a), SPH_C32(0xed752000), SPH_C32(0x281d1f35),
	  SPH_C32(0x73064387), SPH_C32(0x4ed7b8e6) },
	{ SPH_C32(0xf663c000), SPH_C32(0xde81aa29), SPH_C32(0x3801ebf9),
	  SPH_C32(0xe7fa92ad), SPH_C32(0xe2080000), SPH_C32(0xd7f9a9ec),
	  SPH_C32(0xe679e16a), SPH_C32(0x635fbdfc) },
	{ SPH_C32(0x1f49e000), SPH_C32(0x6ad626d5), SPH_C32(0x29fbd1ae),
	  SPH_C32(0xda339789), SPH_C32(0x755b0000), SPH_C32(0xf7b6c73f),
	  SPH_C32(0x91c00965), SPH_C32(0xc2b3e33d) },
	{ SPH_C32(0x6130c000), SPH_C32(0xfecec4fa), SPH_C32(0x4fb803f6),
	  SPH_C32(0x4616cc6c), SPH_C32(0x9c712000), SPH_C32(0x43e14bc3),
	  SPH_C32(0x803a3332), SPH_C32(0xff7ae619) },
	{ SPH_C32(0x881ae000), SPH_C32(0x4a994806), SPH_C32(0x5e4239a1),
	  SPH_C32(0x7bdfc948), SPH_C32(0x0b222000), SPH_C32(0x63ae2510),
	  SPH_C32(0xf783db3d), SPH_C32(0x5e96b8d8) },
	{ SPH_C32(0x022f8000), SPH_C32(0xce2549e4), SPH_C32(0x317ebce8),
	  SPH_C32(0x398d5ee1), SPH_C32(0xf0134000), SPH_C32(0x8cee7004),
	  SPH_C32(0x6b832ec1), SPH_C32(0xad69718e) },
	{ SPH_C32(0xeb05a000), SPH_C32(0x7a72c518), SPH_C32(0x208486bf),
	  SPH_C32(0x04445bc5), SPH_C32(0x67404000), SPH_C32(0xaca11ed7),
	  SPH_C32(0x1c3ac6ce), SPH_C32(0x0c852f4f) },
	{ SPH_C32(0x957c8000), SPH_C32(0xee6a2737), SPH_C32(0x46c754e7),
	  SPH_C32(0x98610020), SPH_C32(0x8e6a6000), SPH_C32(0x18f6922b),
	  SPH_C32(0x0dc0fc99), SPH_C32(0x314c2a6b) },
	{ SPH_C32(0x7c56a000), SPH_C32(0x5a3dabcb), SPH_C32(0x573d6eb0),
	  SPH_C32(0xa5a80504), SPH_C32(0x19396000), SPH_C32(0x38b9fcf8),
	  SPH_C32(0x7a791496), SPH_C32(0x90a074aa) },
	{ SPH_C32(0x1034c000), SPH_C32(0x9532900c), SPH_C32(0xbc847343),
	  SPH_C32(0xf7bb9293), SPH_C32(0x16444000), SPH_C32(0xc75d4a21),
	  SPH_C32(0xef06b67b), SPH_C32(0xbd2871b0) },
	{ SPH_C32(0xf91ee000), SPH_C32(0x21651cf0), SPH_C32(0xad7e4914),
	  SPH_C32(0xca7297b7), SPH_C32(0x81174000), SPH_C32(0xe71224f2),
	  SPH_C32(0x98bf5e74), SPH_C32(0x1cc42f71) },
	{ SPH_C32(0x8767c000), SPH_C32(0xb57dfedf), SPH_C32(0xcb3d9b4c),
	  SPH_C32(0x5657cc52), SPH_C32(0x683d6000), SPH_C32(0x5345a80e),
	  SPH_C32(0x89456423), SPH_C32(0x210d2a55) },
	{ SPH_C32(0x6e4de000), SPH_C32(0x012a7223), SPH_C32(0xdac7a11b),
	  SPH_C32(0x6b9ec976), SPH_C32(0xff6e6000), SPH_C32(0x730ac6dd),
	  SPH_C32(0xfefc8c2c), SPH_C32(0x80e17494) },
	{ SPH_C32(0x045f0000), SPH_C32(0x9c4a93c9), SPH_C32(0x62fc79d0),
	  SPH_C32(0x731ebdc2), SPH_C32(0xe0278000), SPH_C32(0x19dce008),
	  SPH_C32(0xd7075d82), SPH_C32(0x5ad2e31d) },
	{ SPH_C32(0xed752000), SPH_C32(0x281d1f35), SPH_C32(0x73064387),
	  SPH_C32(0x4ed7b8e6), SPH_C32(0x77748000), SPH_C32(0x39938edb),
	  SPH_C32(0xa0beb58d), SPH_C32(0xfb3ebddc) },
	{ SPH_C32(0x930c0000), SPH_C32(0xbc05fd1a), SPH_C32(0x154591df),
	  SPH_C32(0xd2f2e303), SPH_C32(0x9e5ea000), SPH_C32(0x8dc40227),
	  SPH_C32(0xb1448fda), SPH_C32(0xc6f7b8f8) },
	{ SPH_C32(0x7a262000), SPH_C32(0x085271e6), SPH_C32(0x04bfab88),
	  SPH_C32(0xef3be627), SPH_C32(0x090da000), SPH_C32(0xad8b6cf4),
	  SPH_C32(0xc6fd67d5), SPH_C32(0x671be639) },
	{ SPH_C32(0x16444000), SPH_C32(0xc75d4a21), SPH_C32(0xef06b67b),
	  SPH_C32(0xbd2871b0), SPH_C32(0x06708000), SPH_C32(0x526fda2d),
	  SPH_C32(0x5382c538), SPH_C32(0x4a93e323) },
	{ SPH_C32(0xff6e6000), SPH_C32(0x730ac6dd), SPH_C32(0xfefc8c2c),
	  SPH_C32(0x80e17494), SPH_C32(0x91238000), SPH_C32(0x7220b4fe),
	  SPH_C32(0x243b2d37), SPH_C32(0xeb7fbde2) },
	{ SPH_C32(0x81174000), SPH_C32(0xe71224f2), SPH_C32(0x98bf5e74),
	  SPH_C32(0x1cc42f71), SPH_C32(0x7809a000), SPH_C32(0xc6773802),
	  SPH_C32(0x35c11760), SPH_C32(0xd6b6b8c6) },
	{ SPH_C32(0x683d6000), SPH_C32(0x5345a80e), SPH_C32(0x89456423),
	  SPH_C32(0x210d2a55), SPH_C32(0xef5aa000), SPH_C32(0xe63856d1),
	  SPH_C32(0x4278ff6f), SPH_C32(0x775ae607) },
	{ SPH_C32(0xe2080000), SPH_C32(0xd7f9a9ec), SPH_C32(0xe679e16a),
	  SPH_C32(0x635fbdfc), SPH_C32(0x146bc000), SPH_C32(0x097803c5),
	  SPH_C32(0xde780a93), SPH_C32(0x84a52f51) },
	{ SPH_C32(0x0b222000), SPH_C32(0x63ae2510), SPH_C32(0xf783db3d),
	  SPH_C32(0x5e96b8d8), SPH_C32(0x8338c000), SPH_C32(0x29376d16),
	  SPH_C32(0xa9c1e29c), SPH_C32(0x25497190) },
	{ SPH_C32(0x755b0000), SPH_C32(0xf7b6c73f), SPH_C32(0x91c00965),
	  SPH_C32(0xc2b3e33d), SPH_C32(0x6a12e000), SPH_C32(0x9d60e1ea),
	  SPH_C32(0xb83bd8cb), SPH_C32(0x188074b4) },
	{ SPH_C32(0x9c712000), SPH_C32(0x43e14bc3), SPH_C32(0x803a3332),
	  SPH_C32(0xff7ae619), SPH_C32(0xfd41e000), SPH_C32(0xbd2f8f39),
	  SPH_C32(0xcf8230c4), SPH_C32(0xb96c2a75) },
	{ SPH_C32(0xf0134000), SPH_C32(0x8cee7004), SPH_C32(0x6b832ec1),
	  SPH_C32(0xad69718e), SPH_C32(0xf23cc000), SPH_C32(0x42cb39e0),
	  SPH_C32(0x5afd9229), SPH_C32(0x94e42f6f) },
	{ SPH_C32(0x19396000), SPH_C32(0x38b9fcf8), SPH_C32(0x7a791496),
	  SPH_C32(0x90a074aa), SPH_C32(0x656fc000), SPH_C32(0x62845733),
	  SPH_C32(0x2d447a26), SPH_C32(0x350871ae) },
	{ SPH_C32(0x67404000), SPH_C32(0xaca11ed7), SPH_C32(0x1c3ac6ce),
	  SPH_C32(0x0c852f4f), SPH_C32(0x8c45e000), SPH_C32(0xd6d3dbcf),
	  SPH_C32(0x3cbe4071), SPH_C32(0x08c1748a) },
	{ SPH_C32(0x8e6a6000), SPH_C32(0x18f6922b), SPH_C32(0x0dc0fc99),
	  SPH_C32(0x314c2a6b), SPH_C32(0x1b16e000), SPH_C32(0xf69cb51c),
	  SPH_C32(0x4b07a87e), SPH_C32(0xa92d2a4b) },
	{ SPH_C32(0xe0278000), SPH_C32(0x19dce008), SPH_C32(0xd7075d82),
	  SPH_C32(0x5ad2e31d), SPH_C32(0xe4788000), SPH_C32(0x859673c1),
	  SPH_C32(0xb5fb2452), SPH_C32(0x29cc5edf) },
	{ SPH_C32(0x090da000), SPH_C32(0xad8b6cf4), SPH_C32(0xc6fd67d5),
	  SPH_C32(0x671be639), SPH_C32(0x732b8000), SPH_C32(0xa5d91d12),
	  SPH_C32(0xc242cc5d), SPH_C32(0x8820001e) },
	{ SPH_C32(0x77748000), SPH_C32(0x39938edb), SPH_C32(0xa0beb58d),
	  SPH_C32(0xfb3ebddc), SPH_C32(0x9a01a000), SPH_C32(0x118e91ee),
	  SPH_C32(0xd3b8f60a), SPH_C32(0xb5e9053a) },
	{ SPH_C32(0x9e5ea000), SPH_C32(0x8dc40227), SPH_C32(0xb1448fda),
	  SPH_C32(0xc6f7b8f8), SPH_C32(0x0d52a000), SPH_C32(0x31c1ff3d),
	  SPH_C32(0xa4011e05), SPH_C32(0x14055bfb) },
	{ SPH_C32(0xf23cc000), SPH_C32(0x42cb39e0), SPH_C32(0x5afd9229),
	  SPH_C32(0x94e42f6f), SPH_C32(0x022f8000), SPH_C32(0xce2549e4),
	  SPH_C32(0x317ebce8), SPH_C32(0x398d5ee1) },
	{ SPH_C32(0x1b16e000), SPH_C32(0xf69cb51c), SPH_C32(0x4b07a87e),
	  SPH_C32(0xa92d2a4b), SPH_C32(0x957c8000), SPH_C32(0xee6a2737),
	  SPH_C32(0x46c754e7), SPH_C32(0x98610020) },
	{ SPH_C32(0x656fc000), SPH_C32(0x62845733), SPH_C32(0x2d447a26),
	  SPH_C32(0x350871ae), SPH_C32(0x7c56a000), SPH_C32(0x5a3dabcb),
	  SPH_C32(0x573d6eb0), SPH_C32(0xa5a80504) },
	{ SPH_C32(0x8c45e000), SPH_C32(0xd6d3dbcf), SPH_C32(0x3cbe4071),
	  SPH_C32(0x08c1748a), SPH_C32(0xeb05a000), SPH_C32(0x7a72c518),
	  SPH_C32(0x208486bf), SPH_C32(0x04445bc5) },
	{ SPH_C32(0x06708000), SPH_C32(0x526fda2d), SPH_C32(0x5382c538),
	  SPH_C32(0x4a93e323), SPH_C32(0x1034c000), SPH_C32(0x9532900c),
	  SPH_C32(0xbc847343), SPH_C32(0xf7bb9293) },
	{ SPH_C32(0xef5aa000), SPH_C32(0xe63856d1), SPH_C32(0x4278ff6f),
	  SPH_C32(0x775ae607), SPH_C32(0x8767c000), SPH_C32(0xb57dfedf),
	  SPH_C32(0xcb3d9b4c), SPH_C32(0x5657cc52) },
	{ SPH_C32(0x91238000), SPH_C32(0x7220b4fe), SPH_C32(0x243b2d37),
	  SPH_C32(0xeb7fbde2), SPH_C32(0x6e4de000), SPH_C32(0x012a7223),
	  SPH_C32(0xdac7a11b), SPH_C32(0x6b9ec976) },
	{ SPH_C32(0x7809a000), SPH_C32(0xc6773802), SPH_C32(0x35c11760),
	  SPH_C32(0xd6b6b8c6), SPH_C32(0xf91ee000), SPH_C32(0x21651cf0),
	  SPH_C32(0xad7e4914), SPH_C32(0xca7297b7) },
	{ SPH_C32(0x146bc000), SPH_C32(0x097803c5), SPH_C32(0xde780a93),
	  SPH_C32(0x84a52f51), SPH_C32(0xf663c000), SPH_C32(0xde81aa29),
	  SPH_C32(0x3801ebf9), SPH_C32(0xe7fa92ad) },
	{ SPH_C32(0xfd41e000), SPH_C32(0xbd2f8f39), SPH_C32(0xcf8230c4),
	  SPH_C32(0xb96c2a75), SPH_C32(0x6130c000), SPH_C32(0xfecec4fa),
	  SPH_C32(0x4fb803f6), SPH_C32(0x4616cc6c) },
	{ SPH_C32(0x8338c000), SPH_C32(0x29376d16), SPH_C32(0xa9c1e29c),
	  SPH_C32(0x25497190), SPH_C32(0x881ae000), SPH_C32(0x4a994806),
	  SPH_C32(0x5e4239a1), SPH_C32(0x7bdfc948) },
	{ SPH_C32(0x6a12e000), SPH_C32(0x9d60e1ea), SPH_C32(0xb83bd8cb),
	  SPH_C32(0x188074b4), SPH_C32(0x1f49e000), SPH_C32(0x6ad626d5),
	  SPH_C32(0x29fbd1ae), SPH_C32(0xda339789) }
};

static const sph_u32 T256_6[64][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x8f3e0400), SPH_C32(0x0d9dc877), SPH_C32(0x6fc548e1),
	  SPH_C32(0x898d2cd6), SPH_C32(0x14bd0000), SPH_C32(0x2fba37ff),
	  SPH_C32(0x6a72e5bb), SPH_C32(0x247febe6) },
	{ SPH_C32(0x14bd0000), SPH_C32(0x2fba37ff), SPH_C32(0x6a72e5bb),
	  SPH_C32(0x247febe6), SPH_C32(0x9b830400), SPH_C32(0x2227ff88),
	  SPH_C32(0x05b7ad5a), SPH_C32(0xadf2c730) },
	{ SPH_C32(0x9b830400), SPH_C32(0x2227ff88), SPH_C32(0x05b7ad5a),
	  SPH_C32(0xadf2c730), SPH_C32(0x8f3e0400), SPH_C32(0x0d9dc877),
	  SPH_C32(0x6fc548e1), SPH_C32(0x898d2cd6) },
	{ SPH_C32(0xde320800), SPH_C32(0x288350fe), SPH_C32(0x71852ac7),
	  SPH_C32(0xa6bf9f96), SPH_C32(0xe18b0000), SPH_C32(0x5459887d),
	  SPH_C32(0xbf1283d3), SPH_C32(0x1b666a73) },
	{ SPH_C32(0x510c0c00), SPH_C32(0x251e9889), SPH_C32(0x1e406226),
	  SPH_C32(0x2f32b340), SPH_C32(0xf5360000), SPH_C32(0x7be3bf82),
	  SPH_C32(0xd5606668), SPH_C32(0x3f198195) },
	{ SPH_C32(0xca8f0800), SPH_C32(0x07396701), SPH_C32(0x1bf7cf7c),
	  SPH_C32(0x82c07470), SPH_C32(0x7a080400), SPH_C32(0x767e77f5),
	  SPH_C32(0xbaa52e89), SPH_C32(0xb694ad43) },
	{ SPH_C32(0x45b10c00), SPH_C32(0x0aa4af76), SPH_C32(0x7432879d),
	  SPH_C32(0x0b4d58a6), SPH_C32(0x6eb50400), SPH_C32(0x59c4400a),
	  SPH_C32(0xd0d7cb32), SPH_C32(0x92eb46a5) },
	{ SPH_C32(0xe18b0000), SPH_C32(0x5459887d), SPH_C32(0xbf1283d3),
	  SPH_C32(0x1b666a73), SPH_C32(0x3fb90800), SPH_C32(0x7cdad883),
	  SPH_C32(0xce97a914), SPH_C32(0xbdd9f5e5) },
	{ SPH_C32(0x6eb50400), SPH_C32(0x59c4400a), SPH_C32(0xd0d7cb32),
	  SPH_C32(0x92eb46a5), SPH_C32(0x2b040800), SPH_C32(0x5360ef7c),
	  SPH_C32(0xa4e54caf), SPH_C32(0x99a61e03) },
	{ SPH_C32(0xf5360000), SPH_C32(0x7be3bf82), SPH_C32(0xd5606668),
	  SPH_C32(0x3f198195), SPH_C32(0xa43a0c00), SPH_C32(0x5efd270b),
	  SPH_C32(0xcb20044e), SPH_C32(0x102b32d5) },
	{ SPH_C32(0x7a080400), SPH_C32(0x767e77f5), SPH_C32(0xbaa52e89),
	  SPH_C32(0xb694ad43), SPH_C32(0xb0870c00), SPH_C32(0x714710f4),
	  SPH_C32(0xa152e1f5), SPH_C32(0x3454d933) },
	{ SPH_C32(0x3fb90800), SPH_C32(0x7cdad883), SPH_C32(0xce97a914),
	  SPH_C32(0xbdd9f5e5), SPH_C32(0xde320800), SPH_C32(0x288350fe),
	  SPH_C32(0x71852ac7), SPH_C32(0xa6bf9f96) },
	{ SPH_C32(0xb0870c00), SPH_C32(0x714710f4), SPH_C32(0xa152e1f5),
	  SPH_C32(0x3454d933), SPH_C32(0xca8f0800), SPH_C32(0x07396701),
	  SPH_C32(0x1bf7cf7c), SPH_C32(0x82c07470) },
	{ SPH_C32(0x2b040800), SPH_C32(0x5360ef7c), SPH_C32(0xa4e54caf),
	  SPH_C32(0x99a61e03), SPH_C32(0x45b10c00), SPH_C32(0x0aa4af76),
	  SPH_C32(0x7432879d), SPH_C32(0x0b4d58a6) },
	{ SPH_C32(0xa43a0c00), SPH_C32(0x5efd270b), SPH_C32(0xcb20044e),
	  SPH_C32(0x102b32d5), SPH_C32(0x510c0c00), SPH_C32(0x251e9889),
	  SPH_C32(0x1e406226), SPH_C32(0x2f32b340) },
	{ SPH_C32(0x74951000), SPH_C32(0x5a2b467e), SPH_C32(0x88fd1d2b),
	  SPH_C32(0x1ee68292), SPH_C32(0xcba90000), SPH_C32(0x90273769),
	  SPH_C32(0xbbdcf407), SPH_C32(0xd0f4af61) },
	{ SPH_C32(0xfbab1400), SPH_C32(0x57b68e09), SPH_C32(0xe73855ca),
	  SPH_C32(0x976bae44), SPH_C32(0xdf140000), SPH_C32(0xbf9d0096),
	  SPH_C32(0xd1ae11bc), SPH_C32(0xf48b4487) },
	{ SPH_C32(0x60281000), SPH_C32(0x75917181), SPH_C32(0xe28ff890),
	  SPH_C32(0x3a996974), SPH_C32(0x502a0400), SPH_C32(0xb200c8e1),
	  SPH_C32(0xbe6b595d), SPH_C32(0x7d066851) },
	{ SPH_C32(0xef161400), SPH_C32(0x780cb9f6), SPH_C32(0x8d4ab071),
	  SPH_C32(0xb31445a2), SPH_C32(0x44970400), SPH_C32(0x9dbaff1e),
	  SPH_C32(0xd419bce6), SPH_C32(0x597983b7) },
	{ SPH_C32(0xaaa71800), SPH_C32(0x72a81680), SPH_C32(0xf97837ec),
	  SPH_C32(0xb8591d04), SPH_C32(0x2a220000), SPH_C32(0xc47ebf14),
	  SPH_C32(0x04ce77d4), SPH_C32(0xcb92c512) },
	{ SPH_C32(0x25991c00), SPH_C32(0x7f35def7), SPH_C32(0x96bd7f0d),
	  SPH_C32(0x31d431d2), SPH_C32(0x3e9f0000), SPH_C32(0xebc488eb),
	  SPH_C32(0x6ebc926f), SPH_C32(0xefed2ef4) },
	{ SPH_C32(0xbe1a1800), SPH_C32(0x5d12217f), SPH_C32(0x930ad257),
	  SPH_C32(0x9c26f6e2), SPH_C32(0xb1a10400), SPH_C32(0xe659409c),
	  SPH_C32(0x0179da8e), SPH_C32(0x66600222) },
	{ SPH_C32(0x31241c00), SPH_C32(0x508fe908), SPH_C32(0xfccf9ab6),
	  SPH_C32(0x15abda34), SPH_C32(0xa51c0400), SPH_C32(0xc9e37763),
	  SPH_C32(0x6b0b3f35), SPH_C32(0x421fe9c4) },
	{ SPH_C32(0x951e1000), SPH_C32(0x0e72ce03), SPH_C32(0x37ef9ef8),
	  SPH_C32(0x0580e8e1), SPH_C32(0xf4100800), SPH_C32(0xecfdefea),
	  SPH_C32(0x754b5d13), SPH_C32(0x6d2d5a84) },
	{ SPH_C32(0x1a201400), SPH_C32(0x03ef0674), SPH_C32(0x582ad619),
	  SPH_C32(0x8c0dc437), SPH_C32(0xe0ad0800), SPH_C32(0xc347d815),
	  SPH_C32(0x1f39b8a8), SPH_C32(0x4952b162) },
	{ SPH_C32(0x81a31000), SPH_C32(0x21c8f9fc), SPH_C32(0x5d9d7b43),
	  SPH_C32(0x21ff0307), SPH_C32(0x6f930c00), SPH_C32(0xceda1062),
	  SPH_C32(0x70fcf049), SPH_C32(0xc0df9db4) },
	{ SPH_C32(0x0e9d1400), SPH_C32(0x2c55318b), SPH_C32(0x325833a2),
	  SPH_C32(0xa8722fd1), SPH_C32(0x7b2e0c00), SPH_C32(0xe160279d),
	  SPH_C32(0x1a8e15f2), SPH_C32(0xe4a07652) },
	{ SPH_C32(0x4b2c1800), SPH_C32(0x26f19efd), SPH_C32(0x466ab43f),
	  SPH_C32(0xa33f7777), SPH_C32(0x159b0800), SPH_C32(0xb8a46797),
	  SPH_C32(0xca59dec0), SPH_C32(0x764b30f7) },
	{ SPH_C32(0xc4121c00), SPH_C32(0x2b6c568a), SPH_C32(0x29affcde),
	  SPH_C32(0x2ab25ba1), SPH_C32(0x01260800), SPH_C32(0x971e5068),
	  SPH_C32(0xa02b3b7b), SPH_C32(0x5234db11) },
	{ SPH_C32(0x5f911800), SPH_C32(0x094ba902), SPH_C32(0x2c185184),
	  SPH_C32(0x87409c91), SPH_C32(0x8e180c00), SPH_C32(0x9a83981f),
	  SPH_C32(0xcfee739a), SPH_C32(0xdbb9f7c7) },
	{ SPH_C32(0xd0af1c00), SPH_C32(0x04d66175), SPH_C32(0x43dd1965),
	  SPH_C32(0x0ecdb047), SPH_C32(0x9aa50c00), SPH_C32(0xb539afe0),
	  SPH_C32(0xa59c9621), SPH_C32(0xffc61c21) },
	{ SPH_C32(0xcba90000), SPH_C32(0x90273769), SPH_C32(0xbbdcf407),
	  SPH_C32(0xd0f4af61), SPH_C32(0xbf3c1000), SPH_C32(0xca0c7117),
	  SPH_C32(0x3321e92c), SPH_C32(0xce122df3) },
	{ SPH_C32(0x44970400), SPH_C32(0x9dbaff1e), SPH_C32(0xd419bce6),
	  SPH_C32(0x597983b7), SPH_C32(0xab811000), SPH_C32(0xe5b646e8),
	  SPH_C32(0x59530c97), SPH_C32(0xea6dc615) },
	{ SPH_C32(0xdf140000), SPH_C32(0xbf9d0096), SPH_C32(0xd1ae11bc),
	  SPH_C32(0xf48b4487), SPH_C32(0x24bf1400), SPH_C32(0xe82b8e9f),
	  SPH_C32(0x36964476), SPH_C32(0x63e0eac3) },
	{ SPH_C32(0x502a0400), SPH_C32(0xb200c8e1), SPH_C32(0xbe6b595d),
	  SPH_C32(0x7d066851), SPH_C32(0x30021400), SPH_C32(0xc791b960),
	  SPH_C32(0x5ce4a1cd), SPH_C32(0x479f0125) },
	{ SPH_C32(0x159b0800), SPH_C32(0xb8a46797), SPH_C32(0xca59dec0),
	  SPH_C32(0x764b30f7), SPH_C32(0x5eb71000), SPH_C32(0x9e55f96a),
	  SPH_C32(0x8c336aff), SPH_C32(0xd5744780) },
	{ SPH_C32(0x9aa50c00), SPH_C32(0xb539afe0), SPH_C32(0xa59c9621),
	  SPH_C32(0xffc61c21), SPH_C32(0x4a0a1000), SPH_C32(0xb1efce95),
	  SPH_C32(0xe6418f44), SPH_C32(0xf10bac66) },
	{ SPH_C32(0x01260800), SPH_C32(0x971e5068), SPH_C32(0xa02b3b7b),
	  SPH_C32(0x5234db11), SPH_C32(0xc5341400), SPH_C32(0xbc7206e2),
	  SPH_C32(0x8984c7a5), SPH_C32(0x788680b0) },
	{ SPH_C32(0x8e180c00), SPH_C32(0x9a83981f), SPH_C32(0xcfee739a),
	  SPH_C32(0xdbb9f7c7), SPH_C32(0xd1891400), SPH_C32(0x93c8311d),
	  SPH_C32(0xe3f6221e), SPH_C32(0x5cf96b56) },
	{ SPH_C32(0x2a220000), SPH_C32(0xc47ebf14), SPH_C32(0x04ce77d4),
	  SPH_C32(0xcb92c512), SPH_C32(0x80851800), SPH_C32(0xb6d6a994),
	  SPH_C32(0xfdb64038), SPH_C32(0x73cbd816) },
	{ SPH_C32(0xa51c0400), SPH_C32(0xc9e37763), SPH_C32(0x6b0b3f35),
	  SPH_C32(0x421fe9c4), SPH_C32(0x94381800), SPH_C32(0x996c9e6b),
	  SPH_C32(0x97c4a583), SPH_C32(0x57b433f0) },
	{ SPH_C32(0x3e9f0000), SPH_C32(0xebc488eb), SPH_C32(0x6ebc926f),
	  SPH_C32(0xefed2ef4), SPH_C32(0x1b061c00), SPH_C32(0x94f1561c),
	  SPH_C32(0xf801ed62), SPH_C32(0xde391f26) },
	{ SPH_C32(0xb1a10400), SPH_C32(0xe659409c), SPH_C32(0x0179da8e),
	  SPH_C32(0x66600222), SPH_C32(0x0fbb1c00), SPH_C32(0xbb4b61e3),
	  SPH_C32(0x927308d9), SPH_C32(0xfa46f4c0) },
	{ SPH_C32(0xf4100800), SPH_C32(0xecfdefea), SPH_C32(0x754b5d13),
	  SPH_C32(0x6d2d5a84), SPH_C32(0x610e1800), SPH_C32(0xe28f21e9),
	  SPH_C32(0x42a4c3eb), SPH_C32(0x68adb265) },
	{ SPH_C32(0x7b2e0c00), SPH_C32(0xe160279d), SPH_C32(0x1a8e15f2),
	  SPH_C32(0xe4a07652), SPH_C32(0x75b31800), SPH_C32(0xcd351616),
	  SPH_C32(0x28d62650), SPH_C32(0x4cd25983) },
	{ SPH_C32(0xe0ad0800), SPH_C32(0xc347d815), SPH_C32(0x1f39b8a8),
	  SPH_C32(0x4952b162), SPH_C32(0xfa8d1c00), SPH_C32(0xc0a8de61),
	  SPH_C32(0x47136eb1), SPH_C32(0xc55f7555) },
	{ SPH_C32(0x6f930c00), SPH_C32(0xceda1062), SPH_C32(0x70fcf049),
	  SPH_C32(0xc0df9db4), SPH_C32(0xee301c00), SPH_C32(0xef12e99e),
	  SPH_C32(0x2d618b0a), SPH_C32(0xe1209eb3) },
	{ SPH_C32(0xbf3c1000), SPH_C32(0xca0c7117), SPH_C32(0x3321e92c),
	  SPH_C32(0xce122df3), SPH_C32(0x74951000), SPH_C32(0x5a2b467e),
	  SPH_C32(0x88fd1d2b), SPH_C32(0x1ee68292) },
	{ SPH_C32(0x30021400), SPH_C32(0xc791b960), SPH_C32(0x5ce4a1cd),
	  SPH_C32(0x479f0125), SPH_C32(0x60281000), SPH_C32(0x75917181),
	  SPH_C32(0xe28ff890), SPH_C32(0x3a996974) },
	{ SPH_C32(0xab811000), SPH_C32(0xe5b646e8), SPH_C32(0x59530c97),
	  SPH_C32(0xea6dc615), SPH_C32(0xef161400), SPH_C32(0x780cb9f6),
	  SPH_C32(0x8d4ab071), SPH_C32(0xb31445a2) },
	{ SPH_C32(0x24bf1400), SPH_C32(0xe82b8e9f), SPH_C32(0x36964476),
	  SPH_C32(0x63e0eac3), SPH_C32(0xfbab1400), SPH_C32(0x57b68e09),
	  SPH_C32(0xe73855ca), SPH_C32(0x976bae44) },
	{ SPH_C32(0x610e1800), SPH_C32(0xe28f21e9), SPH_C32(0x42a4c3eb),
	  SPH_C32(0x68adb265), SPH_C32(0x951e1000), SPH_C32(0x0e72ce03),
	  SPH_C32(0x37ef9ef8), SPH_C32(0x0580e8e1) },
	{ SPH_C32(0xee301c00), SPH_C32(0xef12e99e), SPH_C32(0x2d618b0a),
	  SPH_C32(0xe1209eb3), SPH_C32(0x81a31000), SPH_C32(0x21c8f9fc),
	  SPH_C32(0x5d9d7b43), SPH_C32(0x21ff0307) },
	{ SPH_C32(0x75b31800), SPH_C32(0xcd351616), SPH_C32(0x28d62650),
	  SPH_C32(0x4cd25983), SPH_C32(0x0e9d1400), SPH_C32(0x2c55318b),
	  SPH_C32(0x325833a2), SPH_C32(0xa8722fd1) },
	{ SPH_C32(0xfa8d1c00), SPH_C32(0xc0a8de61), SPH_C32(0x47136eb1),
	  SPH_C32(0xc55f7555), SPH_C32(0x1a201400), SPH_C32(0x03ef0674),
	  SPH_C32(0x582ad619), SPH_C32(0x8c0dc437) },
	{ SPH_C32(0x5eb71000), SPH_C32(0x9e55f96a), SPH_C32(0x8c336aff),
	  SPH_C32(0xd5744780), SPH_C32(0x4b2c1800), SPH_C32(0x26f19efd),
	  SPH_C32(0x466ab43f), SPH_C32(0xa33f7777) },
	{ SPH_C32(0xd1891400), SPH_C32(0x93c8311d), SPH_C32(0xe3f6221e),
	  SPH_C32(0x5cf96b56), SPH_C32(0x5f911800), SPH_C32(0x094ba902),
	  SPH_C32(0x2c185184), SPH_C32(0x87409c91) },
	{ SPH_C32(0x4a0a1000), SPH_C32(0xb1efce95), SPH_C32(0xe6418f44),
	  SPH_C32(0xf10bac66), SPH_C32(0xd0af1c00), SPH_C32(0x04d66175),
	  SPH_C32(0x43dd1965), SPH_C32(0x0ecdb047) },
	{ SPH_C32(0xc5341400), SPH_C32(0xbc7206e2), SPH_C32(0x8984c7a5),
	  SPH_C32(0x788680b0), SPH_C32(0xc4121c00), SPH_C32(0x2b6c568a),
	  SPH_C32(0x29affcde), SPH_C32(0x2ab25ba1) },
	{ SPH_C32(0x80851800), SPH_C32(0xb6d6a994), SPH_C32(0xfdb64038),
	  SPH_C32(0x73cbd816), SPH_C32(0xaaa71800), SPH_C32(0x72a81680),
	  SPH_C32(0xf97837ec), SPH_C32(0xb8591d04) },
	{ SPH_C32(0x0fbb1c00), SPH_C32(0xbb4b61e3), SPH_C32(0x927308d9),
	  SPH_C32(0xfa46f4c0), SPH_C32(0xbe1a1800), SPH_C32(0x5d12217f),
	  SPH_C32(0x930ad257), SPH_C32(0x9c26f6e2) },
	{ SPH_C32(0x94381800), SPH_C32(0x996c9e6b), SPH_C32(0x97c4a583),
	  SPH_C32(0x57b433f0), SPH_C32(0x31241c00), SPH_C32(0x508fe908),
	  SPH_C32(0xfccf9ab6), SPH_C32(0x15abda34) },
	{ SPH_C32(0x1b061c00), SPH_C32(0x94f1561c), SPH_C32(0xf801ed62),
	  SPH_C32(0xde391f26), SPH_C32(0x25991c00), SPH_C32(0x7f35def7),
	  SPH_C32(0x96bd7f0d), SPH_C32(0x31d431d2) }
};

static const sph_u32 T256_12[64][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x5bd20080), SPH_C32(0x450f18ec), SPH_C32(0xc2c46c55),
	  SPH_C32(0xf362b233), SPH_C32(0x39a60000), SPH_C32(0x4ab753eb),
	  SPH_C32(0xd14e094b), SPH_C32(0xb772b42b) },
	{ SPH_C32(0x39a60000), SPH_C32(0x4ab753eb), SPH_C32(0xd14e094b),
	  SPH_C32(0xb772b42b), SPH_C32(0x62740080), SPH_C32(0x0fb84b07),
	  SPH_C32(0x138a651e), SPH_C32(0x44100618) },
	{ SPH_C32(0x62740080), SPH_C32(0x0fb84b07), SPH_C32(0x138a651e),
	  SPH_C32(0x44100618), SPH_C32(0x5bd20080), SPH_C32(0x450f18ec),
	  SPH_C32(0xc2c46c55), SPH_C32(0xf362b233) },
	{ SPH_C32(0xb7a40100), SPH_C32(0x8a1f31d8), SPH_C32(0x8589d8ab),
	  SPH_C32(0xe6c46464), SPH_C32(0x734c0000), SPH_C32(0x956fa7d6),
	  SPH_C32(0xa29d1297), SPH_C32(0x6ee56854) },
	{ SPH_C32(0xec760180), SPH_C32(0xcf102934), SPH_C32(0x474db4fe),
	  SPH_C32(0x15a6d657), SPH_C32(0x4aea0000), SPH_C32(0xdfd8f43d),
	  SPH_C32(0x73d31bdc), SPH_C32(0xd997dc7f) },
	{ SPH_C32(0x8e020100), SPH_C32(0xc0a86233), SPH_C32(0x54c7d1e0),
	  SPH_C32(0x51b6d04f), SPH_C32(0x11380080), SPH_C32(0x9ad7ecd1),
	  SPH_C32(0xb1177789), SPH_C32(0x2af56e4c) },
	{ SPH_C32(0xd5d00180), SPH_C32(0x85a77adf), SPH_C32(0x9603bdb5),
	  SPH_C32(0xa2d4627c), SPH_C32(0x289e0080), SPH_C32(0xd060bf3a),
	  SPH_C32(0x60597ec2), SPH_C32(0x9d87da67) },
	{ SPH_C32(0x734c0000), SPH_C32(0x956fa7d6), SPH_C32(0xa29d1297),
	  SPH_C32(0x6ee56854), SPH_C32(0xc4e80100), SPH_C32(0x1f70960e),
	  SPH_C32(0x2714ca3c), SPH_C32(0x88210c30) },
	{ SPH_C32(0x289e0080), SPH_C32(0xd060bf3a), SPH_C32(0x60597ec2),
	  SPH_C32(0x9d87da67), SPH_C32(0xfd4e0100), SPH_C32(0x55c7c5e5),
	  SPH_C32(0xf65ac377), SPH_C32(0x3f53b81b) },
	{ SPH_C32(0x4aea0000), SPH_C32(0xdfd8f43d), SPH_C32(0x73d31bdc),
	  SPH_C32(0xd997dc7f), SPH_C32(0xa69c0180), SPH_C32(0x10c8dd09),
	  SPH_C32(0x349eaf22), SPH_C32(0xcc310a28) },
	{ SPH_C32(0x11380080), SPH_C32(0x9ad7ecd1), SPH_C32(0xb1177789),
	  SPH_C32(0x2af56e4c), SPH_C32(0x9f3a0180), SPH_C32(0x5a7f8ee2),
	  SPH_C32(0xe5d0a669), SPH_C32(0x7b43be03) },
	{ SPH_C32(0xc4e80100), SPH_C32(0x1f70960e), SPH_C32(0x2714ca3c),
	  SPH_C32(0x88210c30), SPH_C32(0xb7a40100), SPH_C32(0x8a1f31d8),
	  SPH_C32(0x8589d8ab), SPH_C32(0xe6c46464) },
	{ SPH_C32(0x9f3a0180), SPH_C32(0x5a7f8ee2), SPH_C32(0xe5d0a669),
	  SPH_C32(0x7b43be03), SPH_C32(0x8e020100), SPH_C32(0xc0a86233),
	  SPH_C32(0x54c7d1e0), SPH_C32(0x51b6d04f) },
	{ SPH_C32(0xfd4e0100), SPH_C32(0x55c7c5e5), SPH_C32(0xf65ac377),
	  SPH_C32(0x3f53b81b), SPH_C32(0xd5d00180), SPH_C32(0x85a77adf),
	  SPH_C32(0x9603bdb5), SPH_C32(0xa2d4627c) },
	{ SPH_C32(0xa69c0180), SPH_C32(0x10c8dd09), SPH_C32(0x349eaf22),
	  SPH_C32(0xcc310a28), SPH_C32(0xec760180), SPH_C32(0xcf102934),
	  SPH_C32(0x474db4fe), SPH_C32(0x15a6d657) },
	{ SPH_C32(0xa7b80200), SPH_C32(0x1f128433), SPH_C32(0x60e5f9f2),
	  SPH_C32(0x9e147576), SPH_C32(0xee260000), SPH_C32(0x124b683e),
	  SPH_C32(0x80c2d68f), SPH_C32(0x3bf3ab2c) },
	{ SPH_C32(0xfc6a0280), SPH_C32(0x5a1d9cdf), SPH_C32(0xa22195a7),
	  SPH_C32(0x6d76c745), SPH_C32(0xd7800000), SPH_C32(0x58fc3bd5),
	  SPH_C32(0x518cdfc4), SPH_C32(0x8c811f07) },
	{ SPH_C32(0x9e1e0200), SPH_C32(0x55a5d7d8), SPH_C32(0xb1abf0b9),
	  SPH_C32(0x2966c15d), SPH_C32(0x8c520080), SPH_C32(0x1df32339),
	  SPH_C32(0x9348b391), SPH_C32(0x7fe3ad34) },
	{ SPH_C32(0xc5cc0280), SPH_C32(0x10aacf34), SPH_C32(0x736f9cec),
	  SPH_C32(0xda04736e), SPH_C32(0xb5f40080), SPH_C32(0x574470d2),
	  SPH_C32(0x4206bada), SPH_C32(0xc891191f) },
	{ SPH_C32(0x101c0300), SPH_C32(0x950db5eb), SPH_C32(0xe56c2159),
	  SPH_C32(0x78d01112), SPH_C32(0x9d6a0000), SPH_C32(0x8724cfe8),
	  SPH_C32(0x225fc418), SPH_C32(0x5516c378) },
	{ SPH_C32(0x4bce0380), SPH_C32(0xd002ad07), SPH_C32(0x27a84d0c),
	  SPH_C32(0x8bb2a321), SPH_C32(0xa4cc0000), SPH_C32(0xcd939c03),
	  SPH_C32(0xf311cd53), SPH_C32(0xe2647753) },
	{ SPH_C32(0x29ba0300), SPH_C32(0xdfbae600), SPH_C32(0x34222812),
	  SPH_C32(0xcfa2a539), SPH_C32(0xff1e0080), SPH_C32(0x889c84ef),
	  SPH_C32(0x31d5a106), SPH_C32(0x1106c560) },
	{ SPH_C32(0x72680380), SPH_C32(0x9ab5feec), SPH_C32(0xf6e64447),
	  SPH_C32(0x3cc0170a), SPH_C32(0xc6b80080), SPH_C32(0xc22bd704),
	  SPH_C32(0xe09ba84d), SPH_C32(0xa674714b) },
	{ SPH_C32(0xd4f40200), SPH_C32(0x8a7d23e5), SPH_C32(0xc278eb65),
	  SPH_C32(0xf0f11d22), SPH_C32(0x2ace0100), SPH_C32(0x0d3bfe30),
	  SPH_C32(0xa7d61cb3), SPH_C32(0xb3d2a71c) },
	{ SPH_C32(0x8f260280), SPH_C32(0xcf723b09), SPH_C32(0x00bc8730),
	  SPH_C32(0x0393af11), SPH_C32(0x13680100), SPH_C32(0x478caddb),
	  SPH_C32(0x769815f8), SPH_C32(0x04a01337) },
	{ SPH_C32(0xed520200), SPH_C32(0xc0ca700e), SPH_C32(0x1336e22e),
	  SPH_C32(0x4783a909), SPH_C32(0x48ba0180), SPH_C32(0x0283b537),
	  SPH_C32(0xb45c79ad), SPH_C32(0xf7c2a104) },
	{ SPH_C32(0xb6800280), SPH_C32(0x85c568e2), SPH_C32(0xd1f28e7b),
	  SPH_C32(0xb4e11b3a), SPH_C32(0x711c0180), SPH_C32(0x4834e6dc),
	  SPH_C32(0x651270e6), SPH_C32(0x40b0152f) },
	{ SPH_C32(0x63500300), SPH_C32(0x0062123d), SPH_C32(0x47f133ce),
	  SPH_C32(0x16357946), SPH_C32(0x59820100), SPH_C32(0x985459e6),
	  SPH_C32(0x054b0e24), SPH_C32(0xdd37cf48) },
	{ SPH_C32(0x38820380), SPH_C32(0x456d0ad1), SPH_C32(0x85355f9b),
	  SPH_C32(0xe557cb75), SPH_C32(0x60240100), SPH_C32(0xd2e30a0d),
	  SPH_C32(0xd405076f), SPH_C32(0x6a457b63) },
	{ SPH_C32(0x5af60300), SPH_C32(0x4ad541d6), SPH_C32(0x96bf3a85),
	  SPH_C32(0xa147cd6d), SPH_C32(0x3bf60180), SPH_C32(0x97ec12e1),
	  SPH_C32(0x16c16b3a), SPH_C32(0x9927c950) },
	{ SPH_C32(0x01240380), SPH_C32(0x0fda593a), SPH_C32(0x547b56d0),
	  SPH_C32(0x52257f5e), SPH_C32(0x02500180), SPH_C32(0xdd5b410a),
	  SPH_C32(0xc78f6271), SPH_C32(0x2e557d7b) },
	{ SPH_C32(0xee260000), SPH_C32(0x124b683e), SPH_C32(0x80c2d68f),
	  SPH_C32(0x3bf3ab2c), SPH_C32(0x499e0200), SPH_C32(0x0d59ec0d),
	  SPH_C32(0xe0272f7d), SPH_C32(0xa5e7de5a) },
	{ SPH_C32(0xb5f40080), SPH_C32(0x574470d2), SPH_C32(0x4206bada),
	  SPH_C32(0xc891191f), SPH_C32(0x70380200), SPH_C32(0x47eebfe6),
	  SPH_C32(0x31692636), SPH_C32(0x12956a71) },
	{ SPH_C32(0xd7800000), SPH_C32(0x58fc3bd5), SPH_C32(0x518cdfc4),
	  SPH_C32(0x8c811f07), SPH_C32(0x2bea0280), SPH_C32(0x02e1a70a),
	  SPH_C32(0xf3ad4a63), SPH_C32(0xe1f7d842) },
	{ SPH_C32(0x8c520080), SPH_C32(0x1df32339), SPH_C32(0x9348b391),
	  SPH_C32(0x7fe3ad34), SPH_C32(0x124c0280), SPH_C32(0x4856f4e1),
	  SPH_C32(0x22e34328), SPH_C32(0x56856c69) },
	{ SPH_C32(0x59820100), SPH_C32(0x985459e6), SPH_C32(0x054b0e24),
	  SPH_C32(0xdd37cf48), SPH_C32(0x3ad20200), SPH_C32(0x98364bdb),
	  SPH_C32(0x42ba3dea), SPH_C32(0xcb02b60e) },
	{ SPH_C32(0x02500180), SPH_C32(0xdd5b410a), SPH_C32(0xc78f6271),
	  SPH_C32(0x2e557d7b), SPH_C32(0x03740200), SPH_C32(0xd2811830),
	  SPH_C32(0x93f434a1), SPH_C32(0x7c700225) },
	{ SPH_C32(0x60240100), SPH_C32(0xd2e30a0d), SPH_C32(0xd405076f),
	  SPH_C32(0x6a457b63), SPH_C32(0x58a60280), SPH_C32(0x978e00dc),
	  SPH_C32(0x513058f4), SPH_C32(0x8f12b016) },
	{ SPH_C32(0x3bf60180), SPH_C32(0x97ec12e1), SPH_C32(0x16c16b3a),
	  SPH_C32(0x9927c950), SPH_C32(0x61000280), SPH_C32(0xdd395337),
	  SPH_C32(0x807e51bf), SPH_C32(0x3860043d) },
	{ SPH_C32(0x9d6a0000), SPH_C32(0x8724cfe8), SPH_C32(0x225fc418),
	  SPH_C32(0x5516c378), SPH_C32(0x8d760300), SPH_C32(0x12297a03),
	  SPH_C32(0xc733e541), SPH_C32(0x2dc6d26a) },
	{ SPH_C32(0xc6b80080), SPH_C32(0xc22bd704), SPH_C32(0xe09ba84d),
	  SPH_C32(0xa674714b), SPH_C32(0xb4d00300), SPH_C32(0x589e29e8),
	  SPH_C32(0x167dec0a), SPH_C32(0x9ab46641) },
	{ SPH_C32(0xa4cc0000), SPH_C32(0xcd939c03), SPH_C32(0xf311cd53),
	  SPH_C32(0xe2647753), SPH_C32(0xef020380), SPH_C32(0x1d913104),
	  SPH_C32(0xd4b9805f), SPH_C32(0x69d6d472) },
	{ SPH_C32(0xff1e0080), SPH_C32(0x889c84ef), SPH_C32(0x31d5a106),
	  SPH_C32(0x1106c560), SPH_C32(0xd6a40380), SPH_C32(0x572662ef),
	  SPH_C32(0x05f78914), SPH_C32(0xdea46059) },
	{ SPH_C32(0x2ace0100), SPH_C32(0x0d3bfe30), SPH_C32(0xa7d61cb3),
	  SPH_C32(0xb3d2a71c), SPH_C32(0xfe3a0300), SPH_C32(0x8746ddd5),
	  SPH_C32(0x65aef7d6), SPH_C32(0x4323ba3e) },
	{ SPH_C32(0x711c0180), SPH_C32(0x4834e6dc), SPH_C32(0x651270e6),
	  SPH_C32(0x40b0152f), SPH_C32(0xc79c0300), SPH_C32(0xcdf18e3e),
	  SPH_C32(0xb4e0fe9d), SPH_C32(0xf4510e15) },
	{ SPH_C32(0x13680100), SPH_C32(0x478caddb), SPH_C32(0x769815f8),
	  SPH_C32(0x04a01337), SPH_C32(0x9c4e0380), SPH_C32(0x88fe96d2),
	  SPH_C32(0x762492c8), SPH_C32(0x0733bc26) },
	{ SPH_C32(0x48ba0180), SPH_C32(0x0283b537), SPH_C32(0xb45c79ad),
	  SPH_C32(0xf7c2a104), SPH_C32(0xa5e80380), SPH_C32(0xc249c539),
	  SPH_C32(0xa76a9b83), SPH_C32(0xb041080d) },
	{ SPH_C32(0x499e0200), SPH_C32(0x0d59ec0d), SPH_C32(0xe0272f7d),
	  SPH_C32(0xa5e7de5a), SPH_C32(0xa7b80200), SPH_C32(0x1f128433),
	  SPH_C32(0x60e5f9f2), SPH_C32(0x9e147576) },
	{ SPH_C32(0x124c0280), SPH_C32(0x4856f4e1), SPH_C32(0x22e34328),
	  SPH_C32(0x56856c69), SPH_C32(0x9e1e0200), SPH_C32(0x55a5d7d8),
	  SPH_C32(0xb1abf0b9), SPH_C32(0x2966c15d) },
	{ SPH_C32(0x70380200), SPH_C32(0x47eebfe6), SPH_C32(0x31692636),
	  SPH_C32(0x12956a71), SPH_C32(0xc5cc0280), SPH_C32(0x10aacf34),
	  SPH_C32(0x736f9cec), SPH_C32(0xda04736e) },
	{ SPH_C32(0x2bea0280), SPH_C32(0x02e1a70a), SPH_C32(0xf3ad4a63),
	  SPH_C32(0xe1f7d842), SPH_C32(0xfc6a0280), SPH_C32(0x5a1d9cdf),
	  SPH_C32(0xa22195a7), SPH_C32(0x6d76c745) },
	{ SPH_C32(0xfe3a0300), SPH_C32(0x8746ddd5), SPH_C32(0x65aef7d6),
	  SPH_C32(0x4323ba3e), SPH_C32(0xd4f40200), SPH_C32(0x8a7d23e5),
	  SPH_C32(0xc278eb65), SPH_C32(0xf0f11d22) },
	{ SPH_C32(0xa5e80380), SPH_C32(0xc249c539), SPH_C32(0xa76a9b83),
	  SPH_C32(0xb041080d), SPH_C32(0xed520200), SPH_C32(0xc0ca700e),
	  SPH_C32(0x1336e22e), SPH_C32(0x4783a909) },
	{ SPH_C32(0xc79c0300), SPH_C32(0xcdf18e3e), SPH_C32(0xb4e0fe9d),
	  SPH_C32(0xf4510e15), SPH_C32(0xb6800280), SPH_C32(0x85c568e2),
	  SPH_C32(0xd1f28e7b), SPH_C32(0xb4e11b3a) },
	{ SPH_C32(0x9c4e0380), SPH_C32(0x88fe96d2), SPH_C32(0x762492c8),
	  SPH_C32(0x0733bc26), SPH_C32(0x8f260280), SPH_C32(0xcf723b09),
	  SPH_C32(0x00bc8730), SPH_C32(0x0393af11) },
	{ SPH_C32(0x3ad20200), SPH_C32(0x98364bdb), SPH_C32(0x42ba3dea),
	  SPH_C32(0xcb02b60e), SPH_C32(0x63500300), SPH_C32(0x0062123d),
	  SPH_C32(0x47f133ce), SPH_C32(0x16357946) },
	{ SPH_C32(0x61000280), SPH_C32(0xdd395337), SPH_C32(0x807e51bf),
	  SPH_C32(0x3860043d), SPH_C32(0x5af60300), SPH_C32(0x4ad541d6),
	  SPH_C32(0x96bf3a85), SPH_C32(0xa147cd6d) },
	{ SPH_C32(0x03740200), SPH_C32(0xd2811830), SPH_C32(0x93f434a1),
	  SPH_C32(0x7c700225), SPH_C32(0x01240380), SPH_C32(0x0fda593a),
	  SPH_C32(0x547b56d0), SPH_C32(0x52257f5e) },
	{ SPH_C32(0x58a60280), SPH_C32(0x978e00dc), SPH_C32(0x513058f4),
	  SPH_C32(0x8f12b016), SPH_C32(0x38820380), SPH_C32(0x456d0ad1),
	  SPH_C32(0x85355f9b), SPH_C32(0xe557cb75) },
	{ SPH_C32(0x8d760300), SPH_C32(0x12297a03), SPH_C32(0xc733e541),
	  SPH_C32(0x2dc6d26a), SPH_C32(0x101c0300), SPH_C32(0x950db5eb),
	  SPH_C32(0xe56c2159), SPH_C32(0x78d01112) },
	{ SPH_C32(0xd6a40380), SPH_C32(0x572662ef), SPH_C32(0x05f78914),
	  SPH_C32(0xdea46059), SPH_C32(0x29ba0300), SPH_C32(0xdfbae600),
	  SPH_C32(0x34222812), SPH_C32(0xcfa2a539) },
	{ SPH_C32(0xb4d00300), SPH_C32(0x589e29e8), SPH_C32(0x167dec0a),
	  SPH_C32(0x9ab46641), SPH_C32(0x72680380), SPH_C32(0x9ab5feec),
	  SPH_C32(0xf6e64447), SPH_C32(0x3cc0170a) },
	{ SPH_C32(0xef020380), SPH_C32(0x1d913104), SPH_C32(0xd4b9805f),
	  SPH_C32(0x69d6d472), SPH_C32(0x4bce0380), SPH_C32(0xd002ad07),
	  SPH_C32(0x27a84d0c), SPH_C32(0x8bb2a321) }
};

static const sph_u32 T256_18[64][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x515c0010), SPH_C32(0x40f372fb), SPH_C32(0xfce72602),
	  SPH_C32(0x71575061), SPH_C32(0x2e390000), SPH_C32(0x64dd6689),
	  SPH_C32(0x3cd406fc), SPH_C32(0xb1f490bc) },
	{ SPH_C32(0x2e390000), SPH_C32(0x64dd6689), SPH_C32(0x3cd406fc),
	  SPH_C32(0xb1f490bc), SPH_C32(0x7f650010), SPH_C32(0x242e1472),
	  SPH_C32(0xc03320fe), SPH_C32(0xc0a3c0dd) },
	{ SPH_C32(0x7f650010), SPH_C32(0x242e1472), SPH_C32(0xc03320fe),
	  SPH_C32(0xc0a3c0dd), SPH_C32(0x515c0010), SPH_C32(0x40f372fb),
	  SPH_C32(0xfce72602), SPH_C32(0x71575061) },
	{ SPH_C32(0xa2b80020), SPH_C32(0x81e7e5f6), SPH_C32(0xf9ce4c04),
	  SPH_C32(0xe2afa0c0), SPH_C32(0x5c720000), SPH_C32(0xc9bacd12),
	  SPH_C32(0x79a90df9), SPH_C32(0x63e92178) },
	{ SPH_C32(0xf3e40030), SPH_C32(0xc114970d), SPH_C32(0x05296a06),
	  SPH_C32(0x93f8f0a1), SPH_C32(0x724b0000), SPH_C32(0xad67ab9b),
	  SPH_C32(0x457d0b05), SPH_C32(0xd21db1c4) },
	{ SPH_C32(0x8c810020), SPH_C32(0xe53a837f), SPH_C32(0xc51a4af8),
	  SPH_C32(0x535b307c), SPH_C32(0x23170010), SPH_C32(0xed94d960),
	  SPH_C32(0xb99a2d07), SPH_C32(0xa34ae1a5) },
	{ SPH_C32(0xdddd0030), SPH_C32(0xa5c9f184), SPH_C32(0x39fd6cfa),
	  SPH_C32(0x220c601d), SPH_C32(0x0d2e0010), SPH_C32(0x8949bfe9),
	  SPH_C32(0x854e2bfb), SPH_C32(0x12be7119) },
	{ SPH_C32(0x5c720000), SPH_C32(0xc9bacd12), SPH_C32(0x79a90df9),
	  SPH_C32(0x63e92178), SPH_C32(0xfeca0020), SPH_C32(0x485d28e4),
	  SPH_C32(0x806741fd), SPH_C32(0x814681b8) },
	{ SPH_C32(0x0d2e0010), SPH_C32(0x8949bfe9), SPH_C32(0x854e2bfb),
	  SPH_C32(0x12be7119), SPH_C32(0xd0f30020), SPH_C32(0x2c804e6d),
	  SPH_C32(0xbcb34701), SPH_C32(0x30b21104) },
	{ SPH_C32(0x724b0000), SPH_C32(0xad67ab9b), SPH_C32(0x457d0b05),
	  SPH_C32(0xd21db1c4), SPH_C32(0x81af0030), SPH_C32(0x6c733c96),
	  SPH_C32(0x40546103), SPH_C32(0x41e54165) },
	{ SPH_C32(0x23170010), SPH_C32(0xed94d960), SPH_C32(0xb99a2d07),
	  SPH_C32(0xa34ae1a5), SPH_C32(0xaf960030), SPH_C32(0x08ae5a1f),
	  SPH_C32(0x7c8067ff), SPH_C32(0xf011d1d9) },
	{ SPH_C32(0xfeca0020), SPH_C32(0x485d28e4), SPH_C32(0x806741fd),
	  SPH_C32(0x814681b8), SPH_C32(0xa2b80020), SPH_C32(0x81e7e5f6),
	  SPH_C32(0xf9ce4c04), SPH_C32(0xe2afa0c0) },
	{ SPH_C32(0xaf960030), SPH_C32(0x08ae5a1f), SPH_C32(0x7c8067ff),
	  SPH_C32(0xf011d1d9), SPH_C32(0x8c810020), SPH_C32(0xe53a837f),
	  SPH_C32(0xc51a4af8), SPH_C32(0x535b307c) },
	{ SPH_C32(0xd0f30020), SPH_C32(0x2c804e6d), SPH_C32(0xbcb34701),
	  SPH_C32(0x30b21104), SPH_C32(0xdddd0030), SPH_C32(0xa5c9f184),
	  SPH_C32(0x39fd6cfa), SPH_C32(0x220c601d) },
	{ SPH_C32(0x81af0030), SPH_C32(0x6c733c96), SPH_C32(0x40546103),
	  SPH_C32(0x41e54165), SPH_C32(0xf3e40030), SPH_C32(0xc114970d),
	  SPH_C32(0x05296a06), SPH_C32(0x93f8f0a1) },
	{ SPH_C32(0x4dce0040), SPH_C32(0x3b5bec7e), SPH_C32(0x36656ba8),
	  SPH_C32(0x23633a05), SPH_C32(0x78ab0000), SPH_C32(0xa0cd5a34),
	  SPH_C32(0x5d5ca0f7), SPH_C32(0x727784cb) },
	{ SPH_C32(0x1c920050), SPH_C32(0x7ba89e85), SPH_C32(0xca824daa),
	  SPH_C32(0x52346a64), SPH_C32(0x56920000), SPH_C32(0xc4103cbd),
	  SPH_C32(0x6188a60b), SPH_C32(0xc3831477) },
	{ SPH_C32(0x63f70040), SPH_C32(0x5f868af7), SPH_C32(0x0ab16d54),
	  SPH_C32(0x9297aab9), SPH_C32(0x07ce0010), SPH_C32(0x84e34e46),
	  SPH_C32(0x9d6f8009), SPH_C32(0xb2d44416) },
	{ SPH_C32(0x32ab0050), SPH_C32(0x1f75f80c), SPH_C32(0xf6564b56),
	  SPH_C32(0xe3c0fad8), SPH_C32(0x29f70010), SPH_C32(0xe03e28cf),
	  SPH_C32(0xa1bb86f5), SPH_C32(0x0320d4aa) },
	{ SPH_C32(0xef760060), SPH_C32(0xbabc0988), SPH_C32(0xcfab27ac),
	  SPH_C32(0xc1cc9ac5), SPH_C32(0x24d90000), SPH_C32(0x69779726),
	  SPH_C32(0x24f5ad0e), SPH_C32(0x119ea5b3) },
	{ SPH_C32(0xbe2a0070), SPH_C32(0xfa4f7b73), SPH_C32(0x334c01ae),
	  SPH_C32(0xb09bcaa4), SPH_C32(0x0ae00000), SPH_C32(0x0daaf1af),
	  SPH_C32(0x1821abf2), SPH_C32(0xa06a350f) },
	{ SPH_C32(0xc14f0060), SPH_C32(0xde616f01), SPH_C32(0xf37f2150),
	  SPH_C32(0x70380a79), SPH_C32(0x5bbc0010), SPH_C32(0x4d598354),
	  SPH_C32(0xe4c68df0), SPH_C32(0xd13d656e) },
	{ SPH_C32(0x90130070), SPH_C32(0x9e921dfa), SPH_C32(0x0f980752),
	  SPH_C32(0x016f5a18), SPH_C32(0x75850010), SPH_C32(0x2984e5dd),
	  SPH_C32(0xd8128b0c), SPH_C32(0x60c9f5d2) },
	{ SPH_C32(0x11bc0040), SPH_C32(0xf2e1216c), SPH_C32(0x4fcc6651),
	  SPH_C32(0x408a1b7d), SPH_C32(0x86610020), SPH_C32(0xe89072d0),
	  SPH_C32(0xdd3be10a), SPH_C32(0xf3310573) },
	{ SPH_C32(0x40e00050), SPH_C32(0xb2125397), SPH_C32(0xb32b4053),
	  SPH_C32(0x31dd4b1c), SPH_C32(0xa8580020), SPH_C32(0x8c4d1459),
	  SPH_C32(0xe1efe7f6), SPH_C32(0x42c595cf) },
	{ SPH_C32(0x3f850040), SPH_C32(0x963c47e5), SPH_C32(0x731860ad),
	  SPH_C32(0xf17e8bc1), SPH_C32(0xf9040030), SPH_C32(0xccbe66a2),
	  SPH_C32(0x1d08c1f4), SPH_C32(0x3392c5ae) },
	{ SPH_C32(0x6ed90050), SPH_C32(0xd6cf351e), SPH_C32(0x8fff46af),
	  SPH_C32(0x8029dba0), SPH_C32(0xd73d0030), SPH_C32(0xa863002b),
	  SPH_C32(0x21dcc708), SPH_C32(0x82665512) },
	{ SPH_C32(0xb3040060), SPH_C32(0x7306c49a), SPH_C32(0xb6022a55),
	  SPH_C32(0xa225bbbd), SPH_C32(0xda130020), SPH_C32(0x212abfc2),
	  SPH_C32(0xa492ecf3), SPH_C32(0x90d8240b) },
	{ SPH_C32(0xe2580070), SPH_C32(0x33f5b661), SPH_C32(0x4ae50c57),
	  SPH_C32(0xd372ebdc), SPH_C32(0xf42a0020), SPH_C32(0x45f7d94b),
	  SPH_C32(0x9846ea0f), SPH_C32(0x212cb4b7) },
	{ SPH_C32(0x9d3d0060), SPH_C32(0x17dba213), SPH_C32(0x8ad62ca9),
	  SPH_C32(0x13d12b01), SPH_C32(0xa5760030), SPH_C32(0x0504abb0),
	  SPH_C32(0x64a1cc0d), SPH_C32(0x507be4d6) },
	{ SPH_C32(0xcc610070), SPH_C32(0x5728d0e8), SPH_C32(0x76310aab),
	  SPH_C32(0x62867b60), SPH_C32(0x8b4f0030), SPH_C32(0x61d9cd39),
	  SPH_C32(0x5875caf1), SPH_C32(0xe18f746a) },
	{ SPH_C32(0x78ab0000), SPH_C32(0xa0cd5a34), SPH_C32(0x5d5ca0f7),
	  SPH_C32(0x727784cb), SPH_C32(0x35650040), SPH_C32(0x9b96b64a),
	  SPH_C32(0x6b39cb5f), SPH_C32(0x5114bece) },
	{ SPH_C32(0x29f70010), SPH_C32(0xe03e28cf), SPH_C32(0xa1bb86f5),
	  SPH_C32(0x0320d4aa), SPH_C32(0x1b5c0040), SPH_C32(0xff4bd0c3),
	  SPH_C32(0x57edcda3), SPH_C32(0xe0e02e72) },
	{ SPH_C32(0x56920000), SPH_C32(0xc4103cbd), SPH_C32(0x6188a60b),
	  SPH_C32(0xc3831477), SPH_C32(0x4a000050), SPH_C32(0xbfb8a238),
	  SPH_C32(0xab0aeba1), SPH_C32(0x91b77e13) },
	{ SPH_C32(0x07ce0010), SPH_C32(0x84e34e46), SPH_C32(0x9d6f8009),
	  SPH_C32(0xb2d44416), SPH_C32(0x64390050), SPH_C32(0xdb65c4b1),
	  SPH_C32(0x97deed5d), SPH_C32(0x2043eeaf) },
	{ SPH_C32(0xda130020), SPH_C32(0x212abfc2), SPH_C32(0xa492ecf3),
	  SPH_C32(0x90d8240b), SPH_C32(0x69170040), SPH_C32(0x522c7b58),
	  SPH_C32(0x1290c6a6), SPH_C32(0x32fd9fb6) },
	{ SPH_C32(0x8b4f0030), SPH_C32(0x61d9cd39), SPH_C32(0x5875caf1),
	  SPH_C32(0xe18f746a), SPH_C32(0x472e0040), SPH_C32(0x36f11dd1),
	  SPH_C32(0x2e44c05a), SPH_C32(0x83090f0a) },
	{ SPH_C32(0xf42a0020), SPH_C32(0x45f7d94b), SPH_C32(0x9846ea0f),
	  SPH_C32(0x212cb4b7), SPH_C32(0x16720050), SPH_C32(0x76026f2a),
	  SPH_C32(0xd2a3e658), SPH_C32(0xf25e5f6b) },
	{ SPH_C32(0xa5760030), SPH_C32(0x0504abb0), SPH_C32(0x64a1cc0d),
	  SPH_C32(0x507be4d6), SPH_C32(0x384b0050), SPH_C32(0x12df09a3),
	  SPH_C32(0xee77e0a4), SPH_C32(0x43aacfd7) },
	{ SPH_C32(0x24d90000), SPH_C32(0x69779726), SPH_C32(0x24f5ad0e),
	  SPH_C32(0x119ea5b3), SPH_C32(0xcbaf0060), SPH_C32(0xd3cb9eae),
	  SPH_C32(0xeb5e8aa2), SPH_C32(0xd0523f76) },
	{ SPH_C32(0x75850010), SPH_C32(0x2984e5dd), SPH_C32(0xd8128b0c),
	  SPH_C32(0x60c9f5d2), SPH_C32(0xe5960060), SPH_C32(0xb716f827),
	  SPH_C32(0xd78a8c5e), SPH_C32(0x61a6afca) },
	{ SPH_C32(0x0ae00000), SPH_C32(0x0daaf1af), SPH_C32(0x1821abf2),
	  SPH_C32(0xa06a350f), SPH_C32(0xb4ca0070), SPH_C32(0xf7e58adc),
	  SPH_C32(0x2b6daa5c), SPH_C32(0x10f1ffab) },
	{ SPH_C32(0x5bbc0010), SPH_C32(0x4d598354), SPH_C32(0xe4c68df0),
	  SPH_C32(0xd13d656e), SPH_C32(0x9af30070), SPH_C32(0x9338ec55),
	  SPH_C32(0x17b9aca0), SPH_C32(0xa1056f17) },
	{ SPH_C32(0x86610020), SPH_C32(0xe89072d0), SPH_C32(0xdd3be10a),
	  SPH_C32(0xf3310573), SPH_C32(0x97dd0060), SPH_C32(0x1a7153bc),
	  SPH_C32(0x92f7875b), SPH_C32(0xb3bb1e0e) },
	{ SPH_C32(0xd73d0030), SPH_C32(0xa863002b), SPH_C32(0x21dcc708),
	  SPH_C32(0x82665512), SPH_C32(0xb9e40060), SPH_C32(0x7eac3535),
	  SPH_C32(0xae2381a7), SPH_C32(0x024f8eb2) },
	{ SPH_C32(0xa8580020), SPH_C32(0x8c4d1459), SPH_C32(0xe1efe7f6),
	  SPH_C32(0x42c595cf), SPH_C32(0xe8b80070), SPH_C32(0x3e5f47ce),
	  SPH_C32(0x52c4a7a5), SPH_C32(0x7318ded3) },
	{ SPH_C32(0xf9040030), SPH_C32(0xccbe66a2), SPH_C32(0x1d08c1f4),
	  SPH_C32(0x3392c5ae), SPH_C32(0xc6810070), SPH_C32(0x5a822147),
	  SPH_C32(0x6e10a159), SPH_C32(0xc2ec4e6f) },
	{ SPH_C32(0x35650040), SPH_C32(0x9b96b64a), SPH_C32(0x6b39cb5f),
	  SPH_C32(0x5114bece), SPH_C32(0x4dce0040), SPH_C32(0x3b5bec7e),
	  SPH_C32(0x36656ba8), SPH_C32(0x23633a05) },
	{ SPH_C32(0x64390050), SPH_C32(0xdb65c4b1), SPH_C32(0x97deed5d),
	  SPH_C32(0x2043eeaf), SPH_C32(0x63f70040), SPH_C32(0x5f868af7),
	  SPH_C32(0x0ab16d54), SPH_C32(0x9297aab9) },
	{ SPH_C32(0x1b5c0040), SPH_C32(0xff4bd0c3), SPH_C32(0x57edcda3),
	  SPH_C32(0xe0e02e72), SPH_C32(0x32ab0050), SPH_C32(0x1f75f80c),
	  SPH_C32(0xf6564b56), SPH_C32(0xe3c0fad8) },
	{ SPH_C32(0x4a000050), SPH_C32(0xbfb8a238), SPH_C32(0xab0aeba1),
	  SPH_C32(0x91b77e13), SPH_C32(0x1c920050), SPH_C32(0x7ba89e85),
	  SPH_C32(0xca824daa), SPH_C32(0x52346a64) },
	{ SPH_C32(0x97dd0060), SPH_C32(0x1a7153bc), SPH_C32(0x92f7875b),
	  SPH_C32(0xb3bb1e0e), SPH_C32(0x11bc0040), SPH_C32(0xf2e1216c),
	  SPH_C32(0x4fcc6651), SPH_C32(0x408a1b7d) },
	{ SPH_C32(0xc6810070), SPH_C32(0x5a822147), SPH_C32(0x6e10a159),
	  SPH_C32(0xc2ec4e6f), SPH_C32(0x3f850040), SPH_C32(0x963c47e5),
	  SPH_C32(0x731860ad), SPH_C32(0xf17e8bc1) },
	{ SPH_C32(0xb9e40060), SPH_C32(0x7eac3535), SPH_C32(0xae2381a7),
	  SPH_C32(0x024f8eb2), SPH_C32(0x6ed90050), SPH_C32(0xd6cf351e),
	  SPH_C32(0x8fff46af), SPH_C32(0x8029dba0) },
	{ SPH_C32(0xe8b80070), SPH_C32(0x3e5f47ce), SPH_C32(0x52c4a7a5),
	  SPH_C32(0x7318ded3), SPH_C32(0x40e00050), SPH_C32(0xb2125397),
	  SPH_C32(0xb32b4053), SPH_C32(0x31dd4b1c) },
	{ SPH_C32(0x69170040), SPH_C32(0x522c7b58), SPH_C32(0x1290c6a6),
	  SPH_C32(0x32fd9fb6), SPH_C32(0xb3040060), SPH_C32(0x7306c49a),
	  SPH_C32(0xb6022a55), SPH_C32(0xa225bbbd) },
	{ SPH_C32(0x384b0050), SPH_C32(0x12df09a3), SPH_C32(0xee77e0a4),
	  SPH_C32(0x43aacfd7), SPH_C32(0x9d3d0060), SPH_C32(0x17dba213),
	  SPH_C32(0x8ad62ca9), SPH_C32(0x13d12b01) },
	{ SPH_C32(0x472e0040), SPH_C32(0x36f11dd1), SPH_C32(0x2e44c05a),
	  SPH_C32(0x83090f0a), SPH_C32(0xcc610070), SPH_C32(0x5728d0e8),
	  SPH_C32(0x76310aab), SPH_C32(0x62867b60) },
	{ SPH_C32(0x16720050), SPH_C32(0x76026f2a), SPH_C32(0xd2a3e658),
	  SPH_C32(0xf25e5f6b), SPH_C32(0xe2580070), SPH_C32(0x33f5b661),
	  SPH_C32(0x4ae50c57), SPH_C32(0xd372ebdc) },
	{ SPH_C32(0xcbaf0060), SPH_C32(0xd3cb9eae), SPH_C32(0xeb5e8aa2),
	  SPH_C32(0xd0523f76), SPH_C32(0xef760060), SPH_C32(0xbabc0988),
	  SPH_C32(0xcfab27ac), SPH_C32(0xc1cc9ac5) },
	{ SPH_C32(0x9af30070), SPH_C32(0x9338ec55), SPH_C32(0x17b9aca0),
	  SPH_C32(0xa1056f17), SPH_C32(0xc14f0060), SPH_C32(0xde616f01),
	  SPH_C32(0xf37f2150), SPH_C32(0x70380a79) },
	{ SPH_C32(0xe5960060), SPH_C32(0xb716f827), SPH_C32(0xd78a8c5e),
	  SPH_C32(0x61a6afca), SPH_C32(0x90130070), SPH_C32(0x9e921dfa),
	  SPH_C32(0x0f980752), SPH_C32(0x016f5a18) },
	{ SPH_C32(0xb4ca0070), SPH_C32(0xf7e58adc), SPH_C32(0x2b6daa5c),
	  SPH_C32(0x10f1ffab), SPH_C32(0xbe2a0070), SPH_C32(0xfa4f7b73),
	  SPH_C32(0x334c01ae), SPH_C32(0xb09bcaa4) }
};

static const sph_u32 T256_24[64][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x88230002), SPH_C32(0x5fe7a7b3), SPH_C32(0x99e585aa),
	  SPH_C32(0x8d75f7f1), SPH_C32(0x51ac0000), SPH_C32(0x25e30f14),
	  SPH_C32(0x79e22a4c), SPH_C32(0x1298bd46) },
	{ SPH_C32(0x51ac0000), SPH_C32(0x25e30f14), SPH_C32(0x79e22a4c),
	  SPH_C32(0x1298bd46), SPH_C32(0xd98f0002), SPH_C32(0x7a04a8a7),
	  SPH_C32(0xe007afe6), SPH_C32(0x9fed4ab7) },
	{ SPH_C32(0xd98f0002), SPH_C32(0x7a04a8a7), SPH_C32(0xe007afe6),
	  SPH_C32(0x9fed4ab7), SPH_C32(0x88230002), SPH_C32(0x5fe7a7b3),
	  SPH_C32(0x99e585aa), SPH_C32(0x8d75f7f1) },
	{ SPH_C32(0xd0080004), SPH_C32(0x8c768f77), SPH_C32(0x9dc5b050),
	  SPH_C32(0xaf4a29da), SPH_C32(0x6ba90000), SPH_C32(0x40ebf9aa),
	  SPH_C32(0x98321c3d), SPH_C32(0x76acc733) },
	{ SPH_C32(0x582b0006), SPH_C32(0xd39128c4), SPH_C32(0x042035fa),
	  SPH_C32(0x223fde2b), SPH_C32(0x3a050000), SPH_C32(0x6508f6be),
	  SPH_C32(0xe1d03671), SPH_C32(0x64347a75) },
	{ SPH_C32(0x81a40004), SPH_C32(0xa9958063), SPH_C32(0xe4279a1c),
	  SPH_C32(0xbdd2949c), SPH_C32(0xb2260002), SPH_C32(0x3aef510d),
	  SPH_C32(0x7835b3db), SPH_C32(0xe9418d84) },
	{ SPH_C32(0x09870006), SPH_C32(0xf67227d0), SPH_C32(0x7dc21fb6),
	  SPH_C32(0x30a7636d), SPH_C32(0xe38a0002), SPH_C32(0x1f0c5e19),
	  SPH_C32(0x01d79997), SPH_C32(0xfbd930c2) },
	{ SPH_C32(0x6ba90000), SPH_C32(0x40ebf9aa), SPH_C32(0x98321c3d),
	  SPH_C32(0x76acc733), SPH_C32(0xbba10004), SPH_C32(0xcc9d76dd),
	  SPH_C32(0x05f7ac6d), SPH_C32(0xd9e6eee9) },
	{ SPH_C32(0xe38a0002), SPH_C32(0x1f0c5e19), SPH_C32(0x01d79997),
	  SPH_C32(0xfbd930c2), SPH_C32(0xea0d0004), SPH_C32(0xe97e79c9),
	  SPH_C32(0x7c158621), SPH_C32(0xcb7e53af) },
	{ SPH_C32(0x3a050000), SPH_C32(0x6508f6be), SPH_C32(0xe1d03671),
	  SPH_C32(0x64347a75), SPH_C32(0x622e0006), SPH_C32(0xb699de7a),
	  SPH_C32(0xe5f0038b), SPH_C32(0x460ba45e) },
	{ SPH_C32(0xb2260002), SPH_C32(0x3aef510d), SPH_C32(0x7835b3db),
	  SPH_C32(0xe9418d84), SPH_C32(0x33820006), SPH_C32(0x937ad16e),
	  SPH_C32(0x9c1229c7), SPH_C32(0x54931918) },
	{ SPH_C32(0xbba10004), SPH_C32(0xcc9d76dd), SPH_C32(0x05f7ac6d),
	  SPH_C32(0xd9e6eee9), SPH_C32(0xd0080004), SPH_C32(0x8c768f77),
	  SPH_C32(0x9dc5b050), SPH_C32(0xaf4a29da) },
	{ SPH_C32(0x33820006), SPH_C32(0x937ad16e), SPH_C32(0x9c1229c7),
	  SPH_C32(0x54931918), SPH_C32(0x81a40004), SPH_C32(0xa9958063),
	  SPH_C32(0xe4279a1c), SPH_C32(0xbdd2949c) },
	{ SPH_C32(0xea0d0004), SPH_C32(0xe97e79c9), SPH_C32(0x7c158621),
	  SPH_C32(0xcb7e53af), SPH_C32(0x09870006), SPH_C32(0xf67227d0),
	  SPH_C32(0x7dc21fb6), SPH_C32(0x30a7636d) },
	{ SPH_C32(0x622e0006), SPH_C32(0xb699de7a), SPH_C32(0xe5f0038b),
	  SPH_C32(0x460ba45e), SPH_C32(0x582b0006), SPH_C32(0xd39128c4),
	  SPH_C32(0x042035fa), SPH_C32(0x223fde2b) },
	{ SPH_C32(0xa8ae0008), SPH_C32(0x2079397d), SPH_C32(0xfe739301),
	  SPH_C32(0xb8a92831), SPH_C32(0x171c0000), SPH_C32(0xb26e3344),
	  SPH_C32(0x9e6a837e), SPH_C32(0x58f8485f) },
	{ SPH_C32(0x208d000a), SPH_C32(0x7f9e9ece), SPH_C32(0x679616ab),
	  SPH_C32(0x35dcdfc0), SPH_C32(0x46b00000), SPH_C32(0x978d3c50),
	  SPH_C32(0xe788a932), SPH_C32(0x4a60f519) },
	{ SPH_C32(0xf9020008), SPH_C32(0x059a3669), SPH_C32(0x8791b94d),
	  SPH_C32(0xaa319577), SPH_C32(0xce930002), SPH_C32(0xc86a9be3),
	  SPH_C32(0x7e6d2c98), SPH_C32(0xc71502e8) },
	{ SPH_C32(0x7121000a), SPH_C32(0x5a7d91da), SPH_C32(0x1e743ce7),
	  SPH_C32(0x27446286), SPH_C32(0x9f3f0002), SPH_C32(0xed8994f7),
	  SPH_C32(0x078f06d4), SPH_C32(0xd58dbfae) },
	{ SPH_C32(0x78a6000c), SPH_C32(0xac0fb60a), SPH_C32(0x63b62351),
	  SPH_C32(0x17e301eb), SPH_C32(0x7cb50000), SPH_C32(0xf285caee),
	  SPH_C32(0x06589f43), SPH_C32(0x2e548f6c) },
	{ SPH_C32(0xf085000e), SPH_C32(0xf3e811b9), SPH_C32(0xfa53a6fb),
	  SPH_C32(0x9a96f61a), SPH_C32(0x2d190000), SPH_C32(0xd766c5fa),
	  SPH_C32(0x7fbab50f), SPH_C32(0x3ccc322a) },
	{ SPH_C32(0x290a000c), SPH_C32(0x89ecb91e), SPH_C32(0x1a54091d),
	  SPH_C32(0x057bbcad), SPH_C32(0xa53a0002), SPH_C32(0x88816249),
	  SPH_C32(0xe65f30a5), SPH_C32(0xb1b9c5db) },
	{ SPH_C32(0xa129000e), SPH_C32(0xd60b1ead), SPH_C32(0x83b18cb7),
	  SPH_C32(0x880e4b5c), SPH_C32(0xf4960002), SPH_C32(0xad626d5d),
	  SPH_C32(0x9fbd1ae9), SPH_C32(0xa321789d) },
	{ SPH_C32(0xc3070008), SPH_C32(0x6092c0d7), SPH_C32(0x66418f3c),
	  SPH_C32(0xce05ef02), SPH_C32(0xacbd0004), SPH_C32(0x7ef34599),
	  SPH_C32(0x9b9d2f13), SPH_C32(0x811ea6b6) },
	{ SPH_C32(0x4b24000a), SPH_C32(0x3f756764), SPH_C32(0xffa40a96),
	  SPH_C32(0x437018f3), SPH_C32(0xfd110004), SPH_C32(0x5b104a8d),
	  SPH_C32(0xe27f055f), SPH_C32(0x93861bf0) },
	{ SPH_C32(0x92ab0008), SPH_C32(0x4571cfc3), SPH_C32(0x1fa3a570),
	  SPH_C32(0xdc9d5244), SPH_C32(0x75320006), SPH_C32(0x04f7ed3e),
	  SPH_C32(0x7b9a80f5), SPH_C32(0x1ef3ec01) },
	{ SPH_C32(0x1a88000a), SPH_C32(0x1a966870), SPH_C32(0x864620da),
	  SPH_C32(0x51e8a5b5), SPH_C32(0x249e0006), SPH_C32(0x2114e22a),
	  SPH_C32(0x0278aab9), SPH_C32(0x0c6b5147) },
	{ SPH_C32(0x130f000c), SPH_C32(0xece44fa0), SPH_C32(0xfb843f6c),
	  SPH_C32(0x614fc6d8), SPH_C32(0xc7140004), SPH_C32(0x3e18bc33),
	  SPH_C32(0x03af332e), SPH_C32(0xf7b26185) },
	{ SPH_C32(0x9b2c000e), SPH_C32(0xb303e813), SPH_C32(0x6261bac6),
	  SPH_C32(0xec3a3129), SPH_C32(0x96b80004), SPH_C32(0x1bfbb327),
	  SPH_C32(0x7a4d1962), SPH_C32(0xe52adcc3) },
	{ SPH_C32(0x42a3000c), SPH_C32(0xc90740b4), SPH_C32(0x82661520),
	  SPH_C32(0x73d77b9e), SPH_C32(0x1e9b0006), SPH_C32(0x441c1494),
	  SPH_C32(0xe3a89cc8), SPH_C32(0x685f2b32) },
	{ SPH_C32(0xca80000e), SPH_C32(0x96e0e707), SPH_C32(0x1b83908a),
	  SPH_C32(0xfea28c6f), SPH_C32(0x4f370006), SPH_C32(0x61ff1b80),
	  SPH_C32(0x9a4ab684), SPH_C32(0x7ac79674) },
	{ SPH_C32(0x171c0000), SPH_C32(0xb26e3344), SPH_C32(0x9e6a837e),
	  SPH_C32(0x58f8485f), SPH_C32(0xbfb20008), SPH_C32(0x92170a39),
	  SPH_C32(0x6019107f), SPH_C32(0xe051606e) },
	{ SPH_C32(0x9f3f0002), SPH_C32(0xed8994f7), SPH_C32(0x078f06d4),
	  SPH_C32(0xd58dbfae), SPH_C32(0xee1e0008), SPH_C32(0xb7f4052d),
	  SPH_C32(0x19fb3a33), SPH_C32(0xf2c9dd28) },
	{ SPH_C32(0x46b00000), SPH_C32(0x978d3c50), SPH_C32(0xe788a932),
	  SPH_C32(0x4a60f519), SPH_C32(0x663d000a), SPH_C32(0xe813a29e),
	  SPH_C32(0x801ebf99), SPH_C32(0x7fbc2ad9) },
	{ SPH_C32(0xce930002), SPH_C32(0xc86a9be3), SPH_C32(0x7e6d2c98),
	  SPH_C32(0xc71502e8), SPH_C32(0x3791000a), SPH_C32(0xcdf0ad8a),
	  SPH_C32(0xf9fc95d5), SPH_C32(0x6d24979f) },
	{ SPH_C32(0xc7140004), SPH_C32(0x3e18bc33), SPH_C32(0x03af332e),
	  SPH_C32(0xf7b26185), SPH_C32(0xd41b0008), SPH_C32(0xd2fcf393),
	  SPH_C32(0xf82b0c42), SPH_C32(0x96fda75d) },
	{ SPH_C32(0x4f370006), SPH_C32(0x61ff1b80), SPH_C32(0x9a4ab684),
	  SPH_C32(0x7ac79674), SPH_C32(0x85b70008), SPH_C32(0xf71ffc87),
	  SPH_C32(0x81c9260e), SPH_C32(0x84651a1b) },
	{ SPH_C32(0x96b80004), SPH_C32(0x1bfbb327), SPH_C32(0x7a4d1962),
	  SPH_C32(0xe52adcc3), SPH_C32(0x0d94000a), SPH_C32(0xa8f85b34),
	  SPH_C32(0x182ca3a4), SPH_C32(0x0910edea) },
	{ SPH_C32(0x1e9b0006), SPH_C32(0x441c1494), SPH_C32(0xe3a89cc8),
	  SPH_C32(0x685f2b32), SPH_C32(0x5c38000a), SPH_C32(0x8d1b5420),
	  SPH_C32(0x61ce89e8), SPH_C32(0x1b8850ac) },
	{ SPH_C32(0x7cb50000), SPH_C32(0xf285caee), SPH_C32(0x06589f43),
	  SPH_C32(0x2e548f6c), SPH_C32(0x0413000c), SPH_C32(0x5e8a7ce4),
	  SPH_C32(0x65eebc12), SPH_C32(0x39b78e87) },
	{ SPH_C32(0xf4960002), SPH_C32(0xad626d5d), SPH_C32(0x9fbd1ae9),
	  SPH_C32(0xa321789d), SPH_C32(0x55bf000c), SPH_C32(0x7b6973f0),
	  SPH_C32(0x1c0c965e), SPH_C32(0x2b2f33c1) },
	{ SPH_C32(0x2d190000), SPH_C32(0xd766c5fa), SPH_C32(0x7fbab50f),
	  SPH_C32(0x3ccc322a), SPH_C32(0xdd9c000e), SPH_C32(0x248ed443),
	  SPH_C32(0x85e913f4), SPH_C32(0xa65ac430) },
	{ SPH_C32(0xa53a0002), SPH_C32(0x88816249), SPH_C32(0xe65f30a5),
	  SPH_C32(0xb1b9c5db), SPH_C32(0x8c30000e), SPH_C32(0x016ddb57),
	  SPH_C32(0xfc0b39b8), SPH_C32(0xb4c27976) },
	{ SPH_C32(0xacbd0004), SPH_C32(0x7ef34599), SPH_C32(0x9b9d2f13),
	  SPH_C32(0x811ea6b6), SPH_C32(0x6fba000c), SPH_C32(0x1e61854e),
	  SPH_C32(0xfddca02f), SPH_C32(0x4f1b49b4) },
	{ SPH_C32(0x249e0006), SPH_C32(0x2114e22a), SPH_C32(0x0278aab9),
	  SPH_C32(0x0c6b5147), SPH_C32(0x3e16000c), SPH_C32(0x3b828a5a),
	  SPH_C32(0x843e8a63), SPH_C32(0x5d83f4f2) },
	{ SPH_C32(0xfd110004), SPH_C32(0x5b104a8d), SPH_C32(0xe27f055f),
	  SPH_C32(0x93861bf0), SPH_C32(0xb635000e), SPH_C32(0x64652de9),
	  SPH_C32(0x1ddb0fc9), SPH_C32(0xd0f60303) },
	{ SPH_C32(0x75320006), SPH_C32(0x04f7ed3e), SPH_C32(0x7b9a80f5),
	  SPH_C32(0x1ef3ec01), SPH_C32(0xe799000e), SPH_C32(0x418622fd),
	  SPH_C32(0x64392585), SPH_C32(0xc26ebe45) },
	{ SPH_C32(0xbfb20008), SPH_C32(0x92170a39), SPH_C32(0x6019107f),
	  SPH_C32(0xe051606e), SPH_C32(0xa8ae0008), SPH_C32(0x2079397d),
	  SPH_C32(0xfe739301), SPH_C32(0xb8a92831) },
	{ SPH_C32(0x3791000a), SPH_C32(0xcdf0ad8a), SPH_C32(0xf9fc95d5),
	  SPH_C32(0x6d24979f), SPH_C32(0xf9020008), SPH_C32(0x059a3669),
	  SPH_C32(0x8791b94d), SPH_C32(0xaa319577) },
	{ SPH_C32(0xee1e0008), SPH_C32(0xb7f4052d), SPH_C32(0x19fb3a33),
	  SPH_C32(0xf2c9dd28), SPH_C32(0x7121000a), SPH_C32(0x5a7d91da),
	  SPH_C32(0x1e743ce7), SPH_C32(0x27446286) },
	{ SPH_C32(0x663d000a), SPH_C32(0xe813a29e), SPH_C32(0x801ebf99),
	  SPH_C32(0x7fbc2ad9), SPH_C32(0x208d000a), SPH_C32(0x7f9e9ece),
	  SPH_C32(0x679616ab), SPH_C32(0x35dcdfc0) },
	{ SPH_C32(0x6fba000c), SPH_C32(0x1e61854e), SPH_C32(0xfddca02f),
	  SPH_C32(0x4f1b49b4), SPH_C32(0xc3070008), SPH_C32(0x6092c0d7),
	  SPH_C32(0x66418f3c), SPH_C32(0xce05ef02) },
	{ SPH_C32(0xe799000e), SPH_C32(0x418622fd), SPH_C32(0x64392585),
	  SPH_C32(0xc26ebe45), SPH_C32(0x92ab0008), SPH_C32(0x4571cfc3),
	  SPH_C32(0x1fa3a570), SPH_C32(0xdc9d5244) },
	{ SPH_C32(0x3e16000c), SPH_C32(0x3b828a5a), SPH_C32(0x843e8a63),
	  SPH_C32(0x5d83f4f2), SPH_C32(0x1a88000a), SPH_C32(0x1a966870),
	  SPH_C32(0x864620da), SPH_C32(0x51e8a5b5) },
	{ SPH_C32(0xb635000e), SPH_C32(0x64652de9), SPH_C32(0x1ddb0fc9),
	  SPH_C32(0xd0f60303), SPH_C32(0x4b24000a), SPH_C32(0x3f756764),
	  SPH_C32(0xffa40a96), SPH_C32(0x437018f3) },
	{ SPH_C32(0xd41b0008), SPH_C32(0xd2fcf393), SPH_C32(0xf82b0c42),
	  SPH_C32(0x96fda75d), SPH_C32(0x130f000c), SPH_C32(0xece44fa0),
	  SPH_C32(0xfb843f6c), SPH_C32(0x614fc6d8) },
	{ SPH_C32(0x5c38000a), SPH_C32(0x8d1b5420), SPH_C32(0x61ce89e8),
	  SPH_C32(0x1b8850ac), SPH_C32(0x42a3000c), SPH_C32(0xc90740b4),
	  SPH_C32(0x82661520), SPH_C32(0x73d77b9e) },
	{ SPH_C32(0x85b70008), SPH_C32(0xf71ffc87), SPH_C32(0x81c9260e),
	  SPH_C32(0x84651a1b), SPH_C32(0xca80000e), SPH_C32(0x96e0e707),
	  SPH_C32(0x1b83908a), SPH_C32(0xfea28c6f) },
	{ SPH_C32(0x0d94000a), SPH_C32(0xa8f85b34), SPH_C32(0x182ca3a4),
	  SPH_C32(0x0910edea), SPH_C32(0x9b2c000e), SPH_C32(0xb303e813),
	  SPH_C32(0x6261bac6), SPH_C32(0xec3a3129) },
	{ SPH_C32(0x0413000c), SPH_C32(0x5e8a7ce4), SPH_C32(0x65eebc12),
	  SPH_C32(0x39b78e87), SPH_C32(0x78a6000c), SPH_C32(0xac0fb60a),
	  SPH_C32(0x63b62351), SPH_C32(0x17e301eb) },
	{ SPH_C32(0x8c30000e), SPH_C32(0x016ddb57), SPH_C32(0xfc0b39b8),
	  SPH_C32(0xb4c27976), SPH_C32(0x290a000c), SPH_C32(0x89ecb91e),
	  SPH_C32(0x1a54091d), SPH_C32(0x057bbcad) },
	{ SPH_C32(0x55bf000c), SPH_C32(0x7b6973f0), SPH_C32(0x1c0c965e),
	  SPH_C32(0x2b2f33c1), SPH_C32(0xa129000e), SPH_C32(0xd60b1ead),
	  SPH_C32(0x83b18cb7), SPH_C32(0x880e4b5c) },
	{ SPH_C32(0xdd9c000e), SPH_C32(0x248ed443), SPH_C32(0x85e913f4),
	  SPH_C32(0xa65ac430), SPH_C32(0xf085000e), SPH_C32(0xf3e811b9),
	  SPH_C32(0xfa53a6fb), SPH_C32(0x9a96f61a) }
};

static const sph_u32 T256_30[4][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xc04e0001), SPH_C32(0x33b9c010), SPH_C32(0xae0ebb05),
	  SPH_C32(0xb5a4c63b), SPH_C32(0xc8f10000), SPH_C32(0x0b2de782),
	  SPH_C32(0x6bf648a4), SPH_C32(0x539cbdbf) },
	{ SPH_C32(0xc8f10000), SPH_C32(0x0b2de782), SPH_C32(0x6bf648a4),
	  SPH_C32(0x539cbdbf), SPH_C32(0x08bf0001), SPH_C32(0x38942792),
	  SPH_C32(0xc5f8f3a1), SPH_C32(0xe6387b84) },
	{ SPH_C32(0x08bf0001), SPH_C32(0x38942792), SPH_C32(0xc5f8f3a1),
	  SPH_C32(0xe6387b84), SPH_C32(0xc04e0001), SPH_C32(0x33b9c010),
	  SPH_C32(0xae0ebb05), SPH_C32(0xb5a4c63b) }
};

#define INPUT_SMALL   do { \
		unsigned acc = buf[0]; \
		const sph_u32 *rp; \
		rp = &T256_0[acc >> 2][0]; \
		m0 = rp[0]; \
		m1 = rp[1]; \
		m2 = rp[2]; \
		m3 = rp[3]; \
		m4 = rp[4]; \
		m5 = rp[5]; \
		m6 = rp[6]; \
		m7 = rp[7]; \
		acc = (acc << 8) | buf[1]; \
		rp = &T256_6[(acc >> 4) & 0x3f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		acc = (acc << 8) | buf[2]; \
		rp = &T256_12[(acc >> 6) & 0x3f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_18[acc & 0x3f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		acc = buf[3]; \
		rp = &T256_24[acc >> 2][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_30[acc & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
	} while (0)

#endif

#if SPH_HAMSI_EXPAND_SMALL == 7

static const sph_u32 T256_0[128][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xcba90000), SPH_C32(0x90273769), SPH_C32(0xbbdcf407),
	  SPH_C32(0xd0f4af61), SPH_C32(0xbf3c1000), SPH_C32(0xca0c7117),
	  SPH_C32(0x3321e92c), SPH_C32(0xce122df3) },
	{ SPH_C32(0xe92a2000), SPH_C32(0xb4578cfc), SPH_C32(0x11fa3a57),
	  SPH_C32(0x3dc90524), SPH_C32(0x97530000), SPH_C32(0x204f6ed3),
	  SPH_C32(0x77b9e80f), SPH_C32(0xa1ec5ec1) },
	{ SPH_C32(0x22832000), SPH_C32(0x2470bb95), SPH_C32(0xaa26ce50),
	  SPH_C32(0xed3daa45), SPH_C32(0x286f1000), SPH_C32(0xea431fc4),
	  SPH_C32(0x44980123), SPH_C32(0x6ffe7332) },
	{ SPH_C32(0x97530000), SPH_C32(0x204f6ed3), SPH_C32(0x77b9e80f),
	  SPH_C32(0xa1ec5ec1), SPH_C32(0x7e792000), SPH_C32(0x9418e22f),
	  SPH_C32(0x6643d258), SPH_C32(0x9c255be5) },
	{ SPH_C32(0x5cfa0000), SPH_C32(0xb06859ba), SPH_C32(0xcc651c08),
	  SPH_C32(0x7118f1a0), SPH_C32(0xc1453000), SPH_C32(0x5e149338),
	  SPH_C32(0x55623b74), SPH_C32(0x52377616) },
	{ SPH_C32(0x7e792000), SPH_C32(0x9418e22f), SPH_C32(0x6643d258),
	  SPH_C32(0x9c255be5), SPH_C32(0xe92a2000), SPH_C32(0xb4578cfc),
	  SPH_C32(0x11fa3a57), SPH_C32(0x3dc90524) },
	{ SPH_C32(0xb5d02000), SPH_C32(0x043fd546), SPH_C32(0xdd9f265f),
	  SPH_C32(0x4cd1f484), SPH_C32(0x56163000), SPH_C32(0x7e5bfdeb),
	  SPH_C32(0x22dbd37b), SPH_C32(0xf3db28d7) },
	{ SPH_C32(0x121b4000), SPH_C32(0x5b17d9e8), SPH_C32(0x8dfacfab),
	  SPH_C32(0xce36cc72), SPH_C32(0xe6570000), SPH_C32(0x4bb33a25),
	  SPH_C32(0x848598ba), SPH_C32(0x1041003e) },
	{ SPH_C32(0xd9b24000), SPH_C32(0xcb30ee81), SPH_C32(0x36263bac),
	  SPH_C32(0x1ec26313), SPH_C32(0x596b1000), SPH_C32(0x81bf4b32),
	  SPH_C32(0xb7a47196), SPH_C32(0xde532dcd) },
	{ SPH_C32(0xfb316000), SPH_C32(0xef405514), SPH_C32(0x9c00f5fc),
	  SPH_C32(0xf3ffc956), SPH_C32(0x71040000), SPH_C32(0x6bfc54f6),
	  SPH_C32(0xf33c70b5), SPH_C32(0xb1ad5eff) },
	{ SPH_C32(0x30986000), SPH_C32(0x7f67627d), SPH_C32(0x27dc01fb),
	  SPH_C32(0x230b6637), SPH_C32(0xce381000), SPH_C32(0xa1f025e1),
	  SPH_C32(0xc01d9999), SPH_C32(0x7fbf730c) },
	{ SPH_C32(0x85484000), SPH_C32(0x7b58b73b), SPH_C32(0xfa4327a4),
	  SPH_C32(0x6fda92b3), SPH_C32(0x982e2000), SPH_C32(0xdfabd80a),
	  SPH_C32(0xe2c64ae2), SPH_C32(0x8c645bdb) },
	{ SPH_C32(0x4ee14000), SPH_C32(0xeb7f8052), SPH_C32(0x419fd3a3),
	  SPH_C32(0xbf2e3dd2), SPH_C32(0x27123000), SPH_C32(0x15a7a91d),
	  SPH_C32(0xd1e7a3ce), SPH_C32(0x42767628) },
	{ SPH_C32(0x6c626000), SPH_C32(0xcf0f3bc7), SPH_C32(0xebb91df3),
	  SPH_C32(0x52139797), SPH_C32(0x0f7d2000), SPH_C32(0xffe4b6d9),
	  SPH_C32(0x957fa2ed), SPH_C32(0x2d88051a) },
	{ SPH_C32(0xa7cb6000), SPH_C32(0x5f280cae), SPH_C32(0x5065e9f4),
	  SPH_C32(0x82e738f6), SPH_C32(0xb0413000), SPH_C32(0x35e8c7ce),
	  SPH_C32(0xa65e4bc1), SPH_C32(0xe39a28e9) },
	{ SPH_C32(0xe6570000), SPH_C32(0x4bb33a25), SPH_C32(0x848598ba),
	  SPH_C32(0x1041003e), SPH_C32(0xf44c4000), SPH_C32(0x10a4e3cd),
	  SPH_C32(0x097f5711), SPH_C32(0xde77cc4c) },
	{ SPH_C32(0x2dfe0000), SPH_C32(0xdb940d4c), SPH_C32(0x3f596cbd),
	  SPH_C32(0xc0b5af5f), SPH_C32(0x4b705000), SPH_C32(0xdaa892da),
	  SPH_C32(0x3a5ebe3d), SPH_C32(0x1065e1bf) },
	{ SPH_C32(0x0f7d2000), SPH_C32(0xffe4b6d9), SPH_C32(0x957fa2ed),
	  SPH_C32(0x2d88051a), SPH_C32(0x631f4000), SPH_C32(0x30eb8d1e),
	  SPH_C32(0x7ec6bf1e), SPH_C32(0x7f9b928d) },
	{ SPH_C32(0xc4d42000), SPH_C32(0x6fc381b0), SPH_C32(0x2ea356ea),
	  SPH_C32(0xfd7caa7b), SPH_C32(0xdc235000), SPH_C32(0xfae7fc09),
	  SPH_C32(0x4de75632), SPH_C32(0xb189bf7e) },
	{ SPH_C32(0x71040000), SPH_C32(0x6bfc54f6), SPH_C32(0xf33c70b5),
	  SPH_C32(0xb1ad5eff), SPH_C32(0x8a356000), SPH_C32(0x84bc01e2),
	  SPH_C32(0x6f3c8549), SPH_C32(0x425297a9) },
	{ SPH_C32(0xbaad0000), SPH_C32(0xfbdb639f), SPH_C32(0x48e084b2),
	  SPH_C32(0x6159f19e), SPH_C32(0x35097000), SPH_C32(0x4eb070f5),
	  SPH_C32(0x5c1d6c65), SPH_C32(0x8c40ba5a) },
	{ SPH_C32(0x982e2000), SPH_C32(0xdfabd80a), SPH_C32(0xe2c64ae2),
	  SPH_C32(0x8c645bdb), SPH_C32(0x1d666000), SPH_C32(0xa4f36f31),
	  SPH_C32(0x18856d46), SPH_C32(0xe3bec968) },
	{ SPH_C32(0x53872000), SPH_C32(0x4f8cef63), SPH_C32(0x591abee5),
	  SPH_C32(0x5c90f4ba), SPH_C32(0xa25a7000), SPH_C32(0x6eff1e26),
	  SPH_C32(0x2ba4846a), SPH_C32(0x2dace49b) },
	{ SPH_C32(0xf44c4000), SPH_C32(0x10a4e3cd), SPH_C32(0x097f5711),
	  SPH_C32(0xde77cc4c), SPH_C32(0x121b4000), SPH_C32(0x5b17d9e8),
	  SPH_C32(0x8dfacfab), SPH_C32(0xce36cc72) },
	{ SPH_C32(0x3fe54000), SPH_C32(0x8083d4a4), SPH_C32(0xb2a3a316),
	  SPH_C32(0x0e83632d), SPH_C32(0xad275000), SPH_C32(0x911ba8ff),
	  SPH_C32(0xbedb2687), SPH_C32(0x0024e181) },
	{ SPH_C32(0x1d666000), SPH_C32(0xa4f36f31), SPH_C32(0x18856d46),
	  SPH_C32(0xe3bec968), SPH_C32(0x85484000), SPH_C32(0x7b58b73b),
	  SPH_C32(0xfa4327a4), SPH_C32(0x6fda92b3) },
	{ SPH_C32(0xd6cf6000), SPH_C32(0x34d45858), SPH_C32(0xa3599941),
	  SPH_C32(0x334a6609), SPH_C32(0x3a745000), SPH_C32(0xb154c62c),
	  SPH_C32(0xc962ce88), SPH_C32(0xa1c8bf40) },
	{ SPH_C32(0x631f4000), SPH_C32(0x30eb8d1e), SPH_C32(0x7ec6bf1e),
	  SPH_C32(0x7f9b928d), SPH_C32(0x6c626000), SPH_C32(0xcf0f3bc7),
	  SPH_C32(0xebb91df3), SPH_C32(0x52139797) },
	{ SPH_C32(0xa8b64000), SPH_C32(0xa0ccba77), SPH_C32(0xc51a4b19),
	  SPH_C32(0xaf6f3dec), SPH_C32(0xd35e7000), SPH_C32(0x05034ad0),
	  SPH_C32(0xd898f4df), SPH_C32(0x9c01ba64) },
	{ SPH_C32(0x8a356000), SPH_C32(0x84bc01e2), SPH_C32(0x6f3c8549),
	  SPH_C32(0x425297a9), SPH_C32(0xfb316000), SPH_C32(0xef405514),
	  SPH_C32(0x9c00f5fc), SPH_C32(0xf3ffc956) },
	{ SPH_C32(0x419c6000), SPH_C32(0x149b368b), SPH_C32(0xd4e0714e),
	  SPH_C32(0x92a638c8), SPH_C32(0x440d7000), SPH_C32(0x254c2403),
	  SPH_C32(0xaf211cd0), SPH_C32(0x3dede4a5) },
	{ SPH_C32(0xe4788000), SPH_C32(0x859673c1), SPH_C32(0xb5fb2452),
	  SPH_C32(0x29cc5edf), SPH_C32(0x045f0000), SPH_C32(0x9c4a93c9),
	  SPH_C32(0x62fc79d0), SPH_C32(0x731ebdc2) },
	{ SPH_C32(0x2fd18000), SPH_C32(0x15b144a8), SPH_C32(0x0e27d055),
	  SPH_C32(0xf938f1be), SPH_C32(0xbb631000), SPH_C32(0x5646e2de),
	  SPH_C32(0x51dd90fc), SPH_C32(0xbd0c9031) },
	{ SPH_C32(0x0d52a000), SPH_C32(0x31c1ff3d), SPH_C32(0xa4011e05),
	  SPH_C32(0x14055bfb), SPH_C32(0x930c0000), SPH_C32(0xbc05fd1a),
	  SPH_C32(0x154591df), SPH_C32(0xd2f2e303) },
	{ SPH_C32(0xc6fba000), SPH_C32(0xa1e6c854), SPH_C32(0x1fddea02),
	  SPH_C32(0xc4f1f49a), SPH_C32(0x2c301000), SPH_C32(0x76098c0d),
	  SPH_C32(0x266478f3), SPH_C32(0x1ce0cef0) },
	{ SPH_C32(0x732b8000), SPH_C32(0xa5d91d12), SPH_C32(0xc242cc5d),
	  SPH_C32(0x8820001e), SPH_C32(0x7a262000), SPH_C32(0x085271e6),
	  SPH_C32(0x04bfab88), SPH_C32(0xef3be627) },
	{ SPH_C32(0xb8828000), SPH_C32(0x35fe2a7b), SPH_C32(0x799e385a),
	  SPH_C32(0x58d4af7f), SPH_C32(0xc51a3000), SPH_C32(0xc25e00f1),
	  SPH_C32(0x379e42a4), SPH_C32(0x2129cbd4) },
	{ SPH_C32(0x9a01a000), SPH_C32(0x118e91ee), SPH_C32(0xd3b8f60a),
	  SPH_C32(0xb5e9053a), SPH_C32(0xed752000), SPH_C32(0x281d1f35),
	  SPH_C32(0x73064387), SPH_C32(0x4ed7b8e6) },
	{ SPH_C32(0x51a8a000), SPH_C32(0x81a9a687), SPH_C32(0x6864020d),
	  SPH_C32(0x651daa5b), SPH_C32(0x52493000), SPH_C32(0xe2116e22),
	  SPH_C32(0x4027aaab), SPH_C32(0x80c59515) },
	{ SPH_C32(0xf663c000), SPH_C32(0xde81aa29), SPH_C32(0x3801ebf9),
	  SPH_C32(0xe7fa92ad), SPH_C32(0xe2080000), SPH_C32(0xd7f9a9ec),
	  SPH_C32(0xe679e16a), SPH_C32(0x635fbdfc) },
	{ SPH_C32(0x3dcac000), SPH_C32(0x4ea69d40), SPH_C32(0x83dd1ffe),
	  SPH_C32(0x370e3dcc), SPH_C32(0x5d341000), SPH_C32(0x1df5d8fb),
	  SPH_C32(0xd5580846), SPH_C32(0xad4d900f) },
	{ SPH_C32(0x1f49e000), SPH_C32(0x6ad626d5), SPH_C32(0x29fbd1ae),
	  SPH_C32(0xda339789), SPH_C32(0x755b0000), SPH_C32(0xf7b6c73f),
	  SPH_C32(0x91c00965), SPH_C32(0xc2b3e33d) },
	{ SPH_C32(0xd4e0e000), SPH_C32(0xfaf111bc), SPH_C32(0x922725a9),
	  SPH_C32(0x0ac738e8), SPH_C32(0xca671000), SPH_C32(0x3dbab628),
	  SPH_C32(0xa2e1e049), SPH_C32(0x0ca1cece) },
	{ SPH_C32(0x6130c000), SPH_C32(0xfecec4fa), SPH_C32(0x4fb803f6),
	  SPH_C32(0x4616cc6c), SPH_C32(0x9c712000), SPH_C32(0x43e14bc3),
	  SPH_C32(0x803a3332), SPH_C32(0xff7ae619) },
	{ SPH_C32(0xaa99c000), SPH_C32(0x6ee9f393), SPH_C32(0xf464f7f1),
	  SPH_C32(0x96e2630d), SPH_C32(0x234d3000), SPH_C32(0x89ed3ad4),
	  SPH_C32(0xb31bda1e), SPH_C32(0x3168cbea) },
	{ SPH_C32(0x881ae000), SPH_C32(0x4a994806), SPH_C32(0x5e4239a1),
	  SPH_C32(0x7bdfc948), SPH_C32(0x0b222000), SPH_C32(0x63ae2510),
	  SPH_C32(0xf783db3d), SPH_C32(0x5e96b8d8) },
	{ SPH_C32(0x43b3e000), SPH_C32(0xdabe7f6f), SPH_C32(0xe59ecda6),
	  SPH_C32(0xab2b6629), SPH_C32(0xb41e3000), SPH_C32(0xa9a25407),
	  SPH_C32(0xc4a23211), SPH_C32(0x9084952b) },
	{ SPH_C32(0x022f8000), SPH_C32(0xce2549e4), SPH_C32(0x317ebce8),
	  SPH_C32(0x398d5ee1), SPH_C32(0xf0134000), SPH_C32(0x8cee7004),
	  SPH_C32(0x6b832ec1), SPH_C32(0xad69718e) },
	{ SPH_C32(0xc9868000), SPH_C32(0x5e027e8d), SPH_C32(0x8aa248ef),
	  SPH_C32(0xe979f180), SPH_C32(0x4f2f5000), SPH_C32(0x46e20113),
	  SPH_C32(0x58a2c7ed), SPH_C32(0x637b5c7d) },
	{ SPH_C32(0xeb05a000), SPH_C32(0x7a72c518), SPH_C32(0x208486bf),
	  SPH_C32(0x04445bc5), SPH_C32(0x67404000), SPH_C32(0xaca11ed7),
	  SPH_C32(0x1c3ac6ce), SPH_C32(0x0c852f4f) },
	{ SPH_C32(0x20aca000), SPH_C32(0xea55f271), SPH_C32(0x9b5872b8),
	  SPH_C32(0xd4b0f4a4), SPH_C32(0xd87c5000), SPH_C32(0x66ad6fc0),
	  SPH_C32(0x2f1b2fe2), SPH_C32(0xc29702bc) },
	{ SPH_C32(0x957c8000), SPH_C32(0xee6a2737), SPH_C32(0x46c754e7),
	  SPH_C32(0x98610020), SPH_C32(0x8e6a6000), SPH_C32(0x18f6922b),
	  SPH_C32(0x0dc0fc99), SPH_C32(0x314c2a6b) },
	{ SPH_C32(0x5ed58000), SPH_C32(0x7e4d105e), SPH_C32(0xfd1ba0e0),
	  SPH_C32(0x4895af41), SPH_C32(0x31567000), SPH_C32(0xd2fae33c),
	  SPH_C32(0x3ee115b5), SPH_C32(0xff5e0798) },
	{ SPH_C32(0x7c56a000), SPH_C32(0x5a3dabcb), SPH_C32(0x573d6eb0),
	  SPH_C32(0xa5a80504), SPH_C32(0x19396000), SPH_C32(0x38b9fcf8),
	  SPH_C32(0x7a791496), SPH_C32(0x90a074aa) },
	{ SPH_C32(0xb7ffa000), SPH_C32(0xca1a9ca2), SPH_C32(0xece19ab7),
	  SPH_C32(0x755caa65), SPH_C32(0xa6057000), SPH_C32(0xf2b58def),
	  SPH_C32(0x4958fdba), SPH_C32(0x5eb25959) },
	{ SPH_C32(0x1034c000), SPH_C32(0x9532900c), SPH_C32(0xbc847343),
	  SPH_C32(0xf7bb9293), SPH_C32(0x16444000), SPH_C32(0xc75d4a21),
	  SPH_C32(0xef06b67b), SPH_C32(0xbd2871b0) },
	{ SPH_C32(0xdb9dc000), SPH_C32(0x0515a765), SPH_C32(0x07588744),
	  SPH_C32(0x274f3df2), SPH_C32(0xa9785000), SPH_C32(0x0d513b36),
	  SPH_C32(0xdc275f57), SPH_C32(0x733a5c43) },
	{ SPH_C32(0xf91ee000), SPH_C32(0x21651cf0), SPH_C32(0xad7e4914),
	  SPH_C32(0xca7297b7), SPH_C32(0x81174000), SPH_C32(0xe71224f2),
	  SPH_C32(0x98bf5e74), SPH_C32(0x1cc42f71) },
	{ SPH_C32(0x32b7e000), SPH_C32(0xb1422b99), SPH_C32(0x16a2bd13),
	  SPH_C32(0x1a8638d6), SPH_C32(0x3e2b5000), SPH_C32(0x2d1e55e5),
	  SPH_C32(0xab9eb758), SPH_C32(0xd2d60282) },
	{ SPH_C32(0x8767c000), SPH_C32(0xb57dfedf), SPH_C32(0xcb3d9b4c),
	  SPH_C32(0x5657cc52), SPH_C32(0x683d6000), SPH_C32(0x5345a80e),
	  SPH_C32(0x89456423), SPH_C32(0x210d2a55) },
	{ SPH_C32(0x4ccec000), SPH_C32(0x255ac9b6), SPH_C32(0x70e16f4b),
	  SPH_C32(0x86a36333), SPH_C32(0xd7017000), SPH_C32(0x9949d919),
	  SPH_C32(0xba648d0f), SPH_C32(0xef1f07a6) },
	{ SPH_C32(0x6e4de000), SPH_C32(0x012a7223), SPH_C32(0xdac7a11b),
	  SPH_C32(0x6b9ec976), SPH_C32(0xff6e6000), SPH_C32(0x730ac6dd),
	  SPH_C32(0xfefc8c2c), SPH_C32(0x80e17494) },
	{ SPH_C32(0xa5e4e000), SPH_C32(0x910d454a), SPH_C32(0x611b551c),
	  SPH_C32(0xbb6a6617), SPH_C32(0x40527000), SPH_C32(0xb906b7ca),
	  SPH_C32(0xcddd6500), SPH_C32(0x4ef35967) },
	{ SPH_C32(0x045f0000), SPH_C32(0x9c4a93c9), SPH_C32(0x62fc79d0),
	  SPH_C32(0x731ebdc2), SPH_C32(0xe0278000), SPH_C32(0x19dce008),
	  SPH_C32(0xd7075d82), SPH_C32(0x5ad2e31d) },
	{ SPH_C32(0xcff60000), SPH_C32(0x0c6da4a0), SPH_C32(0xd9208dd7),
	  SPH_C32(0xa3ea12a3), SPH_C32(0x5f1b9000), SPH_C32(0xd3d0911f),
	  SPH_C32(0xe426b4ae), SPH_C32(0x94c0ceee) },
	{ SPH_C32(0xed752000), SPH_C32(0x281d1f35), SPH_C32(0x73064387),
	  SPH_C32(0x4ed7b8e6), SPH_C32(0x77748000), SPH_C32(0x39938edb),
	  SPH_C32(0xa0beb58d), SPH_C32(0xfb3ebddc) },
	{ SPH_C32(0x26dc2000), SPH_C32(0xb83a285c), SPH_C32(0xc8dab780),
	  SPH_C32(0x9e231787), SPH_C32(0xc8489000), SPH_C32(0xf39fffcc),
	  SPH_C32(0x939f5ca1), SPH_C32(0x352c902f) },
	{ SPH_C32(0x930c0000), SPH_C32(0xbc05fd1a), SPH_C32(0x154591df),
	  SPH_C32(0xd2f2e303), SPH_C32(0x9e5ea000), SPH_C32(0x8dc40227),
	  SPH_C32(0xb1448fda), SPH_C32(0xc6f7b8f8) },
	{ SPH_C32(0x58a50000), SPH_C32(0x2c22ca73), SPH_C32(0xae9965d8),
	  SPH_C32(0x02064c62), SPH_C32(0x2162b000), SPH_C32(0x47c87330),
	  SPH_C32(0x826566f6), SPH_C32(0x08e5950b) },
	{ SPH_C32(0x7a262000), SPH_C32(0x085271e6), SPH_C32(0x04bfab88),
	  SPH_C32(0xef3be627), SPH_C32(0x090da000), SPH_C32(0xad8b6cf4),
	  SPH_C32(0xc6fd67d5), SPH_C32(0x671be639) },
	{ SPH_C32(0xb18f2000), SPH_C32(0x9875468f), SPH_C32(0xbf635f8f),
	  SPH_C32(0x3fcf4946), SPH_C32(0xb631b000), SPH_C32(0x67871de3),
	  SPH_C32(0xf5dc8ef9), SPH_C32(0xa909cbca) },
	{ SPH_C32(0x16444000), SPH_C32(0xc75d4a21), SPH_C32(0xef06b67b),
	  SPH_C32(0xbd2871b0), SPH_C32(0x06708000), SPH_C32(0x526fda2d),
	  SPH_C32(0x5382c538), SPH_C32(0x4a93e323) },
	{ SPH_C32(0xdded4000), SPH_C32(0x577a7d48), SPH_C32(0x54da427c),
	  SPH_C32(0x6ddcded1), SPH_C32(0xb94c9000), SPH_C32(0x9863ab3a),
	  SPH_C32(0x60a32c14), SPH_C32(0x8481ced0) },
	{ SPH_C32(0xff6e6000), SPH_C32(0x730ac6dd), SPH_C32(0xfefc8c2c),
	  SPH_C32(0x80e17494), SPH_C32(0x91238000), SPH_C32(0x7220b4fe),
	  SPH_C32(0x243b2d37), SPH_C32(0xeb7fbde2) },
	{ SPH_C32(0x34c76000), SPH_C32(0xe32df1b4), SPH_C32(0x4520782b),
	  SPH_C32(0x5015dbf5), SPH_C32(0x2e1f9000), SPH_C32(0xb82cc5e9),
	  SPH_C32(0x171ac41b), SPH_C32(0x256d9011) },
	{ SPH_C32(0x81174000), SPH_C32(0xe71224f2), SPH_C32(0x98bf5e74),
	  SPH_C32(0x1cc42f71), SPH_C32(0x7809a000), SPH_C32(0xc6773802),
	  SPH_C32(0x35c11760), SPH_C32(0xd6b6b8c6) },
	{ SPH_C32(0x4abe4000), SPH_C32(0x7735139b), SPH_C32(0x2363aa73),
	  SPH_C32(0xcc308010), SPH_C32(0xc735b000), SPH_C32(0x0c7b4915),
	  SPH_C32(0x06e0fe4c), SPH_C32(0x18a49535) },
	{ SPH_C32(0x683d6000), SPH_C32(0x5345a80e), SPH_C32(0x89456423),
	  SPH_C32(0x210d2a55), SPH_C32(0xef5aa000), SPH_C32(0xe63856d1),
	  SPH_C32(0x4278ff6f), SPH_C32(0x775ae607) },
	{ SPH_C32(0xa3946000), SPH_C32(0xc3629f67), SPH_C32(0x32999024),
	  SPH_C32(0xf1f98534), SPH_C32(0x5066b000), SPH_C32(0x2c3427c6),
	  SPH_C32(0x71591643), SPH_C32(0xb948cbf4) },
	{ SPH_C32(0xe2080000), SPH_C32(0xd7f9a9ec), SPH_C32(0xe679e16a),
	  SPH_C32(0x635fbdfc), SPH_C32(0x146bc000), SPH_C32(0x097803c5),
	  SPH_C32(0xde780a93), SPH_C32(0x84a52f51) },
	{ SPH_C32(0x29a10000), SPH_C32(0x47de9e85), SPH_C32(0x5da5156d),
	  SPH_C32(0xb3ab129d), SPH_C32(0xab57d000), SPH_C32(0xc37472d2),
	  SPH_C32(0xed59e3bf), SPH_C32(0x4ab702a2) },
	{ SPH_C32(0x0b222000), SPH_C32(0x63ae2510), SPH_C32(0xf783db3d),
	  SPH_C32(0x5e96b8d8), SPH_C32(0x8338c000), SPH_C32(0x29376d16),
	  SPH_C32(0xa9c1e29c), SPH_C32(0x25497190) },
	{ SPH_C32(0xc08b2000), SPH_C32(0xf3891279), SPH_C32(0x4c5f2f3a),
	  SPH_C32(0x8e6217b9), SPH_C32(0x3c04d000), SPH_C32(0xe33b1c01),
	  SPH_C32(0x9ae00bb0), SPH_C32(0xeb5b5c63) },
	{ SPH_C32(0x755b0000), SPH_C32(0xf7b6c73f), SPH_C32(0x91c00965),
	  SPH_C32(0xc2b3e33d), SPH_C32(0x6a12e000), SPH_C32(0x9d60e1ea),
	  SPH_C32(0xb83bd8cb), SPH_C32(0x188074b4) },
	{ SPH_C32(0xbef20000), SPH_C32(0x6791f056), SPH_C32(0x2a1cfd62),
	  SPH_C32(0x12474c5c), SPH_C32(0xd52ef000), SPH_C32(0x576c90fd),
	  SPH_C32(0x8b1a31e7), SPH_C32(0xd6925947) },
	{ SPH_C32(0x9c712000), SPH_C32(0x43e14bc3), SPH_C32(0x803a3332),
	  SPH_C32(0xff7ae619), SPH_C32(0xfd41e000), SPH_C32(0xbd2f8f39),
	  SPH_C32(0xcf8230c4), SPH_C32(0xb96c2a75) },
	{ SPH_C32(0x57d82000), SPH_C32(0xd3c67caa), SPH_C32(0x3be6c735),
	  SPH_C32(0x2f8e4978), SPH_C32(0x427df000), SPH_C32(0x7723fe2e),
	  SPH_C32(0xfca3d9e8), SPH_C32(0x777e0786) },
	{ SPH_C32(0xf0134000), SPH_C32(0x8cee7004), SPH_C32(0x6b832ec1),
	  SPH_C32(0xad69718e), SPH_C32(0xf23cc000), SPH_C32(0x42cb39e0),
	  SPH_C32(0x5afd9229), SPH_C32(0x94e42f6f) },
	{ SPH_C32(0x3bba4000), SPH_C32(0x1cc9476d), SPH_C32(0xd05fdac6),
	  SPH_C32(0x7d9ddeef), SPH_C32(0x4d00d000), SPH_C32(0x88c748f7),
	  SPH_C32(0x69dc7b05), SPH_C32(0x5af6029c) },
	{ SPH_C32(0x19396000), SPH_C32(0x38b9fcf8), SPH_C32(0x7a791496),
	  SPH_C32(0x90a074aa), SPH_C32(0x656fc000), SPH_C32(0x62845733),
	  SPH_C32(0x2d447a26), SPH_C32(0x350871ae) },
	{ SPH_C32(0xd2906000), SPH_C32(0xa89ecb91), SPH_C32(0xc1a5e091),
	  SPH_C32(0x4054dbcb), SPH_C32(0xda53d000), SPH_C32(0xa8882624),
	  SPH_C32(0x1e65930a), SPH_C32(0xfb1a5c5d) },
	{ SPH_C32(0x67404000), SPH_C32(0xaca11ed7), SPH_C32(0x1c3ac6ce),
	  SPH_C32(0x0c852f4f), SPH_C32(0x8c45e000), SPH_C32(0xd6d3dbcf),
	  SPH_C32(0x3cbe4071), SPH_C32(0x08c1748a) },
	{ SPH_C32(0xace94000), SPH_C32(0x3c8629be), SPH_C32(0xa7e632c9),
	  SPH_C32(0xdc71802e), SPH_C32(0x3379f000), SPH_C32(0x1cdfaad8),
	  SPH_C32(0x0f9fa95d), SPH_C32(0xc6d35979) },
	{ SPH_C32(0x8e6a6000), SPH_C32(0x18f6922b), SPH_C32(0x0dc0fc99),
	  SPH_C32(0x314c2a6b), SPH_C32(0x1b16e000), SPH_C32(0xf69cb51c),
	  SPH_C32(0x4b07a87e), SPH_C32(0xa92d2a4b) },
	{ SPH_C32(0x45c36000), SPH_C32(0x88d1a542), SPH_C32(0xb61c089e),
	  SPH_C32(0xe1b8850a), SPH_C32(0xa42af000), SPH_C32(0x3c90c40b),
	  SPH_C32(0x78264152), SPH_C32(0x673f07b8) },
	{ SPH_C32(0xe0278000), SPH_C32(0x19dce008), SPH_C32(0xd7075d82),
	  SPH_C32(0x5ad2e31d), SPH_C32(0xe4788000), SPH_C32(0x859673c1),
	  SPH_C32(0xb5fb2452), SPH_C32(0x29cc5edf) },
	{ SPH_C32(0x2b8e8000), SPH_C32(0x89fbd761), SPH_C32(0x6cdba985),
	  SPH_C32(0x8a264c7c), SPH_C32(0x5b449000), SPH_C32(0x4f9a02d6),
	  SPH_C32(0x86dacd7e), SPH_C32(0xe7de732c) },
	{ SPH_C32(0x090da000), SPH_C32(0xad8b6cf4), SPH_C32(0xc6fd67d5),
	  SPH_C32(0x671be639), SPH_C32(0x732b8000), SPH_C32(0xa5d91d12),
	  SPH_C32(0xc242cc5d), SPH_C32(0x8820001e) },
	{ SPH_C32(0xc2a4a000), SPH_C32(0x3dac5b9d), SPH_C32(0x7d2193d2),
	  SPH_C32(0xb7ef4958), SPH_C32(0xcc179000), SPH_C32(0x6fd56c05),
	  SPH_C32(0xf1632571), SPH_C32(0x46322ded) },
	{ SPH_C32(0x77748000), SPH_C32(0x39938edb), SPH_C32(0xa0beb58d),
	  SPH_C32(0xfb3ebddc), SPH_C32(0x9a01a000), SPH_C32(0x118e91ee),
	  SPH_C32(0xd3b8f60a), SPH_C32(0xb5e9053a) },
	{ SPH_C32(0xbcdd8000), SPH_C32(0xa9b4b9b2), SPH_C32(0x1b62418a),
	  SPH_C32(0x2bca12bd), SPH_C32(0x253db000), SPH_C32(0xdb82e0f9),
	  SPH_C32(0xe0991f26), SPH_C32(0x7bfb28c9) },
	{ SPH_C32(0x9e5ea000), SPH_C32(0x8dc40227), SPH_C32(0xb1448fda),
	  SPH_C32(0xc6f7b8f8), SPH_C32(0x0d52a000), SPH_C32(0x31c1ff3d),
	  SPH_C32(0xa4011e05), SPH_C32(0x14055bfb) },
	{ SPH_C32(0x55f7a000), SPH_C32(0x1de3354e), SPH_C32(0x0a987bdd),
	  SPH_C32(0x16031799), SPH_C32(0xb26eb000), SPH_C32(0xfbcd8e2a),
	  SPH_C32(0x9720f729), SPH_C32(0xda177608) },
	{ SPH_C32(0xf23cc000), SPH_C32(0x42cb39e0), SPH_C32(0x5afd9229),
	  SPH_C32(0x94e42f6f), SPH_C32(0x022f8000), SPH_C32(0xce2549e4),
	  SPH_C32(0x317ebce8), SPH_C32(0x398d5ee1) },
	{ SPH_C32(0x3995c000), SPH_C32(0xd2ec0e89), SPH_C32(0xe121662e),
	  SPH_C32(0x4410800e), SPH_C32(0xbd139000), SPH_C32(0x042938f3),
	  SPH_C32(0x025f55c4), SPH_C32(0xf79f7312) },
	{ SPH_C32(0x1b16e000), SPH_C32(0xf69cb51c), SPH_C32(0x4b07a87e),
	  SPH_C32(0xa92d2a4b), SPH_C32(0x957c8000), SPH_C32(0xee6a2737),
	  SPH_C32(0x46c754e7), SPH_C32(0x98610020) },
	{ SPH_C32(0xd0bfe000), SPH_C32(0x66bb8275), SPH_C32(0xf0db5c79),
	  SPH_C32(0x79d9852a), SPH_C32(0x2a409000), SPH_C32(0x24665620),
	  SPH_C32(0x75e6bdcb), SPH_C32(0x56732dd3) },
	{ SPH_C32(0x656fc000), SPH_C32(0x62845733), SPH_C32(0x2d447a26),
	  SPH_C32(0x350871ae), SPH_C32(0x7c56a000), SPH_C32(0x5a3dabcb),
	  SPH_C32(0x573d6eb0), SPH_C32(0xa5a80504) },
	{ SPH_C32(0xaec6c000), SPH_C32(0xf2a3605a), SPH_C32(0x96988e21),
	  SPH_C32(0xe5fcdecf), SPH_C32(0xc36ab000), SPH_C32(0x9031dadc),
	  SPH_C32(0x641c879c), SPH_C32(0x6bba28f7) },
	{ SPH_C32(0x8c45e000), SPH_C32(0xd6d3dbcf), SPH_C32(0x3cbe4071),
	  SPH_C32(0x08c1748a), SPH_C32(0xeb05a000), SPH_C32(0x7a72c518),
	  SPH_C32(0x208486bf), SPH_C32(0x04445bc5) },
	{ SPH_C32(0x47ece000), SPH_C32(0x46f4eca6), SPH_C32(0x8762b476),
	  SPH_C32(0xd835dbeb), SPH_C32(0x5439b000), SPH_C32(0xb07eb40f),
	  SPH_C32(0x13a56f93), SPH_C32(0xca567636) },
	{ SPH_C32(0x06708000), SPH_C32(0x526fda2d), SPH_C32(0x5382c538),
	  SPH_C32(0x4a93e323), SPH_C32(0x1034c000), SPH_C32(0x9532900c),
	  SPH_C32(0xbc847343), SPH_C32(0xf7bb9293) },
	{ SPH_C32(0xcdd98000), SPH_C32(0xc248ed44), SPH_C32(0xe85e313f),
	  SPH_C32(0x9a674c42), SPH_C32(0xaf08d000), SPH_C32(0x5f3ee11b),
	  SPH_C32(0x8fa59a6f), SPH_C32(0x39a9bf60) },
	{ SPH_C32(0xef5aa000), SPH_C32(0xe63856d1), SPH_C32(0x4278ff6f),
	  SPH_C32(0x775ae607), SPH_C32(0x8767c000), SPH_C32(0xb57dfedf),
	  SPH_C32(0xcb3d9b4c), SPH_C32(0x5657cc52) },
	{ SPH_C32(0x24f3a000), SPH_C32(0x761f61b8), SPH_C32(0xf9a40b68),
	  SPH_C32(0xa7ae4966), SPH_C32(0x385bd000), SPH_C32(0x7f718fc8),
	  SPH_C32(0xf81c7260), SPH_C32(0x9845e1a1) },
	{ SPH_C32(0x91238000), SPH_C32(0x7220b4fe), SPH_C32(0x243b2d37),
	  SPH_C32(0xeb7fbde2), SPH_C32(0x6e4de000), SPH_C32(0x012a7223),
	  SPH_C32(0xdac7a11b), SPH_C32(0x6b9ec976) },
	{ SPH_C32(0x5a8a8000), SPH_C32(0xe2078397), SPH_C32(0x9fe7d930),
	  SPH_C32(0x3b8b1283), SPH_C32(0xd171f000), SPH_C32(0xcb260334),
	  SPH_C32(0xe9e64837), SPH_C32(0xa58ce485) },
	{ SPH_C32(0x7809a000), SPH_C32(0xc6773802), SPH_C32(0x35c11760),
	  SPH_C32(0xd6b6b8c6), SPH_C32(0xf91ee000), SPH_C32(0x21651cf0),
	  SPH_C32(0xad7e4914), SPH_C32(0xca7297b7) },
	{ SPH_C32(0xb3a0a000), SPH_C32(0x56500f6b), SPH_C32(0x8e1de367),
	  SPH_C32(0x064217a7), SPH_C32(0x4622f000), SPH_C32(0xeb696de7),
	  SPH_C32(0x9e5fa038), SPH_C32(0x0460ba44) },
	{ SPH_C32(0x146bc000), SPH_C32(0x097803c5), SPH_C32(0xde780a93),
	  SPH_C32(0x84a52f51), SPH_C32(0xf663c000), SPH_C32(0xde81aa29),
	  SPH_C32(0x3801ebf9), SPH_C32(0xe7fa92ad) },
	{ SPH_C32(0xdfc2c000), SPH_C32(0x995f34ac), SPH_C32(0x65a4fe94),
	  SPH_C32(0x54518030), SPH_C32(0x495fd000), SPH_C32(0x148ddb3e),
	  SPH_C32(0x0b2002d5), SPH_C32(0x29e8bf5e) },
	{ SPH_C32(0xfd41e000), SPH_C32(0xbd2f8f39), SPH_C32(0xcf8230c4),
	  SPH_C32(0xb96c2a75), SPH_C32(0x6130c000), SPH_C32(0xfecec4fa),
	  SPH_C32(0x4fb803f6), SPH_C32(0x4616cc6c) },
	{ SPH_C32(0x36e8e000), SPH_C32(0x2d08b850), SPH_C32(0x745ec4c3),
	  SPH_C32(0x69988514), SPH_C32(0xde0cd000), SPH_C32(0x34c2b5ed),
	  SPH_C32(0x7c99eada), SPH_C32(0x8804e19f) },
	{ SPH_C32(0x8338c000), SPH_C32(0x29376d16), SPH_C32(0xa9c1e29c),
	  SPH_C32(0x25497190), SPH_C32(0x881ae000), SPH_C32(0x4a994806),
	  SPH_C32(0x5e4239a1), SPH_C32(0x7bdfc948) },
	{ SPH_C32(0x4891c000), SPH_C32(0xb9105a7f), SPH_C32(0x121d169b),
	  SPH_C32(0xf5bddef1), SPH_C32(0x3726f000), SPH_C32(0x80953911),
	  SPH_C32(0x6d63d08d), SPH_C32(0xb5cde4bb) },
	{ SPH_C32(0x6a12e000), SPH_C32(0x9d60e1ea), SPH_C32(0xb83bd8cb),
	  SPH_C32(0x188074b4), SPH_C32(0x1f49e000), SPH_C32(0x6ad626d5),
	  SPH_C32(0x29fbd1ae), SPH_C32(0xda339789) },
	{ SPH_C32(0xa1bbe000), SPH_C32(0x0d47d683), SPH_C32(0x03e72ccc),
	  SPH_C32(0xc874dbd5), SPH_C32(0xa075f000), SPH_C32(0xa0da57c2),
	  SPH_C32(0x1ada3882), SPH_C32(0x1421ba7a) }
};

static const sph_u32 T256_7[128][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xa7b80200), SPH_C32(0x1f128433), SPH_C32(0x60e5f9f2),
	  SPH_C32(0x9e147576), SPH_C32(0xee260000), SPH_C32(0x124b683e),
	  SPH_C32(0x80c2d68f), SPH_C32(0x3bf3ab2c) },
	{ SPH_C32(0xee260000), SPH_C32(0x124b683e), SPH_C32(0x80c2d68f),
	  SPH_C32(0x3bf3ab2c), SPH_C32(0x499e0200), SPH_C32(0x0d59ec0d),
	  SPH_C32(0xe0272f7d), SPH_C32(0xa5e7de5a) },
	{ SPH_C32(0x499e0200), SPH_C32(0x0d59ec0d), SPH_C32(0xe0272f7d),
	  SPH_C32(0xa5e7de5a), SPH_C32(0xa7b80200), SPH_C32(0x1f128433),
	  SPH_C32(0x60e5f9f2), SPH_C32(0x9e147576) },
	{ SPH_C32(0x8f3e0400), SPH_C32(0x0d9dc877), SPH_C32(0x6fc548e1),
	  SPH_C32(0x898d2cd6), SPH_C32(0x14bd0000), SPH_C32(0x2fba37ff),
	  SPH_C32(0x6a72e5bb), SPH_C32(0x247febe6) },
	{ SPH_C32(0x28860600), SPH_C32(0x128f4c44), SPH_C32(0x0f20b113),
	  SPH_C32(0x179959a0), SPH_C32(0xfa9b0000), SPH_C32(0x3df15fc1),
	  SPH_C32(0xeab03334), SPH_C32(0x1f8c40ca) },
	{ SPH_C32(0x61180400), SPH_C32(0x1fd6a049), SPH_C32(0xef079e6e),
	  SPH_C32(0xb27e87fa), SPH_C32(0x5d230200), SPH_C32(0x22e3dbf2),
	  SPH_C32(0x8a55cac6), SPH_C32(0x819835bc) },
	{ SPH_C32(0xc6a00600), SPH_C32(0x00c4247a), SPH_C32(0x8fe2679c),
	  SPH_C32(0x2c6af28c), SPH_C32(0xb3050200), SPH_C32(0x30a8b3cc),
	  SPH_C32(0x0a971c49), SPH_C32(0xba6b9e90) },
	{ SPH_C32(0x14bd0000), SPH_C32(0x2fba37ff), SPH_C32(0x6a72e5bb),
	  SPH_C32(0x247febe6), SPH_C32(0x9b830400), SPH_C32(0x2227ff88),
	  SPH_C32(0x05b7ad5a), SPH_C32(0xadf2c730) },
	{ SPH_C32(0xb3050200), SPH_C32(0x30a8b3cc), SPH_C32(0x0a971c49),
	  SPH_C32(0xba6b9e90), SPH_C32(0x75a50400), SPH_C32(0x306c97b6),
	  SPH_C32(0x85757bd5), SPH_C32(0x96016c1c) },
	{ SPH_C32(0xfa9b0000), SPH_C32(0x3df15fc1), SPH_C32(0xeab03334),
	  SPH_C32(0x1f8c40ca), SPH_C32(0xd21d0600), SPH_C32(0x2f7e1385),
	  SPH_C32(0xe5908227), SPH_C32(0x0815196a) },
	{ SPH_C32(0x5d230200), SPH_C32(0x22e3dbf2), SPH_C32(0x8a55cac6),
	  SPH_C32(0x819835bc), SPH_C32(0x3c3b0600), SPH_C32(0x3d357bbb),
	  SPH_C32(0x655254a8), SPH_C32(0x33e6b246) },
	{ SPH_C32(0x9b830400), SPH_C32(0x2227ff88), SPH_C32(0x05b7ad5a),
	  SPH_C32(0xadf2c730), SPH_C32(0x8f3e0400), SPH_C32(0x0d9dc877),
	  SPH_C32(0x6fc548e1), SPH_C32(0x898d2cd6) },
	{ SPH_C32(0x3c3b0600), SPH_C32(0x3d357bbb), SPH_C32(0x655254a8),
	  SPH_C32(0x33e6b246), SPH_C32(0x61180400), SPH_C32(0x1fd6a049),
	  SPH_C32(0xef079e6e), SPH_C32(0xb27e87fa) },
	{ SPH_C32(0x75a50400), SPH_C32(0x306c97b6), SPH_C32(0x85757bd5),
	  SPH_C32(0x96016c1c), SPH_C32(0xc6a00600), SPH_C32(0x00c4247a),
	  SPH_C32(0x8fe2679c), SPH_C32(0x2c6af28c) },
	{ SPH_C32(0xd21d0600), SPH_C32(0x2f7e1385), SPH_C32(0xe5908227),
	  SPH_C32(0x0815196a), SPH_C32(0x28860600), SPH_C32(0x128f4c44),
	  SPH_C32(0x0f20b113), SPH_C32(0x179959a0) },
	{ SPH_C32(0xde320800), SPH_C32(0x288350fe), SPH_C32(0x71852ac7),
	  SPH_C32(0xa6bf9f96), SPH_C32(0xe18b0000), SPH_C32(0x5459887d),
	  SPH_C32(0xbf1283d3), SPH_C32(0x1b666a73) },
	{ SPH_C32(0x798a0a00), SPH_C32(0x3791d4cd), SPH_C32(0x1160d335),
	  SPH_C32(0x38abeae0), SPH_C32(0x0fad0000), SPH_C32(0x4612e043),
	  SPH_C32(0x3fd0555c), SPH_C32(0x2095c15f) },
	{ SPH_C32(0x30140800), SPH_C32(0x3ac838c0), SPH_C32(0xf147fc48),
	  SPH_C32(0x9d4c34ba), SPH_C32(0xa8150200), SPH_C32(0x59006470),
	  SPH_C32(0x5f35acae), SPH_C32(0xbe81b429) },
	{ SPH_C32(0x97ac0a00), SPH_C32(0x25dabcf3), SPH_C32(0x91a205ba),
	  SPH_C32(0x035841cc), SPH_C32(0x46330200), SPH_C32(0x4b4b0c4e),
	  SPH_C32(0xdff77a21), SPH_C32(0x85721f05) },
	{ SPH_C32(0x510c0c00), SPH_C32(0x251e9889), SPH_C32(0x1e406226),
	  SPH_C32(0x2f32b340), SPH_C32(0xf5360000), SPH_C32(0x7be3bf82),
	  SPH_C32(0xd5606668), SPH_C32(0x3f198195) },
	{ SPH_C32(0xf6b40e00), SPH_C32(0x3a0c1cba), SPH_C32(0x7ea59bd4),
	  SPH_C32(0xb126c636), SPH_C32(0x1b100000), SPH_C32(0x69a8d7bc),
	  SPH_C32(0x55a2b0e7), SPH_C32(0x04ea2ab9) },
	{ SPH_C32(0xbf2a0c00), SPH_C32(0x3755f0b7), SPH_C32(0x9e82b4a9),
	  SPH_C32(0x14c1186c), SPH_C32(0xbca80200), SPH_C32(0x76ba538f),
	  SPH_C32(0x35474915), SPH_C32(0x9afe5fcf) },
	{ SPH_C32(0x18920e00), SPH_C32(0x28477484), SPH_C32(0xfe674d5b),
	  SPH_C32(0x8ad56d1a), SPH_C32(0x528e0200), SPH_C32(0x64f13bb1),
	  SPH_C32(0xb5859f9a), SPH_C32(0xa10df4e3) },
	{ SPH_C32(0xca8f0800), SPH_C32(0x07396701), SPH_C32(0x1bf7cf7c),
	  SPH_C32(0x82c07470), SPH_C32(0x7a080400), SPH_C32(0x767e77f5),
	  SPH_C32(0xbaa52e89), SPH_C32(0xb694ad43) },
	{ SPH_C32(0x6d370a00), SPH_C32(0x182be332), SPH_C32(0x7b12368e),
	  SPH_C32(0x1cd40106), SPH_C32(0x942e0400), SPH_C32(0x64351fcb),
	  SPH_C32(0x3a67f806), SPH_C32(0x8d67066f) },
	{ SPH_C32(0x24a90800), SPH_C32(0x15720f3f), SPH_C32(0x9b3519f3),
	  SPH_C32(0xb933df5c), SPH_C32(0x33960600), SPH_C32(0x7b279bf8),
	  SPH_C32(0x5a8201f4), SPH_C32(0x13737319) },
	{ SPH_C32(0x83110a00), SPH_C32(0x0a608b0c), SPH_C32(0xfbd0e001),
	  SPH_C32(0x2727aa2a), SPH_C32(0xddb00600), SPH_C32(0x696cf3c6),
	  SPH_C32(0xda40d77b), SPH_C32(0x2880d835) },
	{ SPH_C32(0x45b10c00), SPH_C32(0x0aa4af76), SPH_C32(0x7432879d),
	  SPH_C32(0x0b4d58a6), SPH_C32(0x6eb50400), SPH_C32(0x59c4400a),
	  SPH_C32(0xd0d7cb32), SPH_C32(0x92eb46a5) },
	{ SPH_C32(0xe2090e00), SPH_C32(0x15b62b45), SPH_C32(0x14d77e6f),
	  SPH_C32(0x95592dd0), SPH_C32(0x80930400), SPH_C32(0x4b8f2834),
	  SPH_C32(0x50151dbd), SPH_C32(0xa918ed89) },
	{ SPH_C32(0xab970c00), SPH_C32(0x18efc748), SPH_C32(0xf4f05112),
	  SPH_C32(0x30bef38a), SPH_C32(0x272b0600), SPH_C32(0x549dac07),
	  SPH_C32(0x30f0e44f), SPH_C32(0x370c98ff) },
	{ SPH_C32(0x0c2f0e00), SPH_C32(0x07fd437b), SPH_C32(0x9415a8e0),
	  SPH_C32(0xaeaa86fc), SPH_C32(0xc90d0600), SPH_C32(0x46d6c439),
	  SPH_C32(0xb03232c0), SPH_C32(0x0cff33d3) },
	{ SPH_C32(0xe18b0000), SPH_C32(0x5459887d), SPH_C32(0xbf1283d3),
	  SPH_C32(0x1b666a73), SPH_C32(0x3fb90800), SPH_C32(0x7cdad883),
	  SPH_C32(0xce97a914), SPH_C32(0xbdd9f5e5) },
	{ SPH_C32(0x46330200), SPH_C32(0x4b4b0c4e), SPH_C32(0xdff77a21),
	  SPH_C32(0x85721f05), SPH_C32(0xd19f0800), SPH_C32(0x6e91b0bd),
	  SPH_C32(0x4e557f9b), SPH_C32(0x862a5ec9) },
	{ SPH_C32(0x0fad0000), SPH_C32(0x4612e043), SPH_C32(0x3fd0555c),
	  SPH_C32(0x2095c15f), SPH_C32(0x76270a00), SPH_C32(0x7183348e),
	  SPH_C32(0x2eb08669), SPH_C32(0x183e2bbf) },
	{ SPH_C32(0xa8150200), SPH_C32(0x59006470), SPH_C32(0x5f35acae),
	  SPH_C32(0xbe81b429), SPH_C32(0x98010a00), SPH_C32(0x63c85cb0),
	  SPH_C32(0xae7250e6), SPH_C32(0x23cd8093) },
	{ SPH_C32(0x6eb50400), SPH_C32(0x59c4400a), SPH_C32(0xd0d7cb32),
	  SPH_C32(0x92eb46a5), SPH_C32(0x2b040800), SPH_C32(0x5360ef7c),
	  SPH_C32(0xa4e54caf), SPH_C32(0x99a61e03) },
	{ SPH_C32(0xc90d0600), SPH_C32(0x46d6c439), SPH_C32(0xb03232c0),
	  SPH_C32(0x0cff33d3), SPH_C32(0xc5220800), SPH_C32(0x412b8742),
	  SPH_C32(0x24279a20), SPH_C32(0xa255b52f) },
	{ SPH_C32(0x80930400), SPH_C32(0x4b8f2834), SPH_C32(0x50151dbd),
	  SPH_C32(0xa918ed89), SPH_C32(0x629a0a00), SPH_C32(0x5e390371),
	  SPH_C32(0x44c263d2), SPH_C32(0x3c41c059) },
	{ SPH_C32(0x272b0600), SPH_C32(0x549dac07), SPH_C32(0x30f0e44f),
	  SPH_C32(0x370c98ff), SPH_C32(0x8cbc0a00), SPH_C32(0x4c726b4f),
	  SPH_C32(0xc400b55d), SPH_C32(0x07b26b75) },
	{ SPH_C32(0xf5360000), SPH_C32(0x7be3bf82), SPH_C32(0xd5606668),
	  SPH_C32(0x3f198195), SPH_C32(0xa43a0c00), SPH_C32(0x5efd270b),
	  SPH_C32(0xcb20044e), SPH_C32(0x102b32d5) },
	{ SPH_C32(0x528e0200), SPH_C32(0x64f13bb1), SPH_C32(0xb5859f9a),
	  SPH_C32(0xa10df4e3), SPH_C32(0x4a1c0c00), SPH_C32(0x4cb64f35),
	  SPH_C32(0x4be2d2c1), SPH_C32(0x2bd899f9) },
	{ SPH_C32(0x1b100000), SPH_C32(0x69a8d7bc), SPH_C32(0x55a2b0e7),
	  SPH_C32(0x04ea2ab9), SPH_C32(0xeda40e00), SPH_C32(0x53a4cb06),
	  SPH_C32(0x2b072b33), SPH_C32(0xb5ccec8f) },
	{ SPH_C32(0xbca80200), SPH_C32(0x76ba538f), SPH_C32(0x35474915),
	  SPH_C32(0x9afe5fcf), SPH_C32(0x03820e00), SPH_C32(0x41efa338),
	  SPH_C32(0xabc5fdbc), SPH_C32(0x8e3f47a3) },
	{ SPH_C32(0x7a080400), SPH_C32(0x767e77f5), SPH_C32(0xbaa52e89),
	  SPH_C32(0xb694ad43), SPH_C32(0xb0870c00), SPH_C32(0x714710f4),
	  SPH_C32(0xa152e1f5), SPH_C32(0x3454d933) },
	{ SPH_C32(0xddb00600), SPH_C32(0x696cf3c6), SPH_C32(0xda40d77b),
	  SPH_C32(0x2880d835), SPH_C32(0x5ea10c00), SPH_C32(0x630c78ca),
	  SPH_C32(0x2190377a), SPH_C32(0x0fa7721f) },
	{ SPH_C32(0x942e0400), SPH_C32(0x64351fcb), SPH_C32(0x3a67f806),
	  SPH_C32(0x8d67066f), SPH_C32(0xf9190e00), SPH_C32(0x7c1efcf9),
	  SPH_C32(0x4175ce88), SPH_C32(0x91b30769) },
	{ SPH_C32(0x33960600), SPH_C32(0x7b279bf8), SPH_C32(0x5a8201f4),
	  SPH_C32(0x13737319), SPH_C32(0x173f0e00), SPH_C32(0x6e5594c7),
	  SPH_C32(0xc1b71807), SPH_C32(0xaa40ac45) },
	{ SPH_C32(0x3fb90800), SPH_C32(0x7cdad883), SPH_C32(0xce97a914),
	  SPH_C32(0xbdd9f5e5), SPH_C32(0xde320800), SPH_C32(0x288350fe),
	  SPH_C32(0x71852ac7), SPH_C32(0xa6bf9f96) },
	{ SPH_C32(0x98010a00), SPH_C32(0x63c85cb0), SPH_C32(0xae7250e6),
	  SPH_C32(0x23cd8093), SPH_C32(0x30140800), SPH_C32(0x3ac838c0),
	  SPH_C32(0xf147fc48), SPH_C32(0x9d4c34ba) },
	{ SPH_C32(0xd19f0800), SPH_C32(0x6e91b0bd), SPH_C32(0x4e557f9b),
	  SPH_C32(0x862a5ec9), SPH_C32(0x97ac0a00), SPH_C32(0x25dabcf3),
	  SPH_C32(0x91a205ba), SPH_C32(0x035841cc) },
	{ SPH_C32(0x76270a00), SPH_C32(0x7183348e), SPH_C32(0x2eb08669),
	  SPH_C32(0x183e2bbf), SPH_C32(0x798a0a00), SPH_C32(0x3791d4cd),
	  SPH_C32(0x1160d335), SPH_C32(0x38abeae0) },
	{ SPH_C32(0xb0870c00), SPH_C32(0x714710f4), SPH_C32(0xa152e1f5),
	  SPH_C32(0x3454d933), SPH_C32(0xca8f0800), SPH_C32(0x07396701),
	  SPH_C32(0x1bf7cf7c), SPH_C32(0x82c07470) },
	{ SPH_C32(0x173f0e00), SPH_C32(0x6e5594c7), SPH_C32(0xc1b71807),
	  SPH_C32(0xaa40ac45), SPH_C32(0x24a90800), SPH_C32(0x15720f3f),
	  SPH_C32(0x9b3519f3), SPH_C32(0xb933df5c) },
	{ SPH_C32(0x5ea10c00), SPH_C32(0x630c78ca), SPH_C32(0x2190377a),
	  SPH_C32(0x0fa7721f), SPH_C32(0x83110a00), SPH_C32(0x0a608b0c),
	  SPH_C32(0xfbd0e001), SPH_C32(0x2727aa2a) },
	{ SPH_C32(0xf9190e00), SPH_C32(0x7c1efcf9), SPH_C32(0x4175ce88),
	  SPH_C32(0x91b30769), SPH_C32(0x6d370a00), SPH_C32(0x182be332),
	  SPH_C32(0x7b12368e), SPH_C32(0x1cd40106) },
	{ SPH_C32(0x2b040800), SPH_C32(0x5360ef7c), SPH_C32(0xa4e54caf),
	  SPH_C32(0x99a61e03), SPH_C32(0x45b10c00), SPH_C32(0x0aa4af76),
	  SPH_C32(0x7432879d), SPH_C32(0x0b4d58a6) },
	{ SPH_C32(0x8cbc0a00), SPH_C32(0x4c726b4f), SPH_C32(0xc400b55d),
	  SPH_C32(0x07b26b75), SPH_C32(0xab970c00), SPH_C32(0x18efc748),
	  SPH_C32(0xf4f05112), SPH_C32(0x30bef38a) },
	{ SPH_C32(0xc5220800), SPH_C32(0x412b8742), SPH_C32(0x24279a20),
	  SPH_C32(0xa255b52f), SPH_C32(0x0c2f0e00), SPH_C32(0x07fd437b),
	  SPH_C32(0x9415a8e0), SPH_C32(0xaeaa86fc) },
	{ SPH_C32(0x629a0a00), SPH_C32(0x5e390371), SPH_C32(0x44c263d2),
	  SPH_C32(0x3c41c059), SPH_C32(0xe2090e00), SPH_C32(0x15b62b45),
	  SPH_C32(0x14d77e6f), SPH_C32(0x95592dd0) },
	{ SPH_C32(0xa43a0c00), SPH_C32(0x5efd270b), SPH_C32(0xcb20044e),
	  SPH_C32(0x102b32d5), SPH_C32(0x510c0c00), SPH_C32(0x251e9889),
	  SPH_C32(0x1e406226), SPH_C32(0x2f32b340) },
	{ SPH_C32(0x03820e00), SPH_C32(0x41efa338), SPH_C32(0xabc5fdbc),
	  SPH_C32(0x8e3f47a3), SPH_C32(0xbf2a0c00), SPH_C32(0x3755f0b7),
	  SPH_C32(0x9e82b4a9), SPH_C32(0x14c1186c) },
	{ SPH_C32(0x4a1c0c00), SPH_C32(0x4cb64f35), SPH_C32(0x4be2d2c1),
	  SPH_C32(0x2bd899f9), SPH_C32(0x18920e00), SPH_C32(0x28477484),
	  SPH_C32(0xfe674d5b), SPH_C32(0x8ad56d1a) },
	{ SPH_C32(0xeda40e00), SPH_C32(0x53a4cb06), SPH_C32(0x2b072b33),
	  SPH_C32(0xb5ccec8f), SPH_C32(0xf6b40e00), SPH_C32(0x3a0c1cba),
	  SPH_C32(0x7ea59bd4), SPH_C32(0xb126c636) },
	{ SPH_C32(0x74951000), SPH_C32(0x5a2b467e), SPH_C32(0x88fd1d2b),
	  SPH_C32(0x1ee68292), SPH_C32(0xcba90000), SPH_C32(0x90273769),
	  SPH_C32(0xbbdcf407), SPH_C32(0xd0f4af61) },
	{ SPH_C32(0xd32d1200), SPH_C32(0x4539c24d), SPH_C32(0xe818e4d9),
	  SPH_C32(0x80f2f7e4), SPH_C32(0x258f0000), SPH_C32(0x826c5f57),
	  SPH_C32(0x3b1e2288), SPH_C32(0xeb07044d) },
	{ SPH_C32(0x9ab31000), SPH_C32(0x48602e40), SPH_C32(0x083fcba4),
	  SPH_C32(0x251529be), SPH_C32(0x82370200), SPH_C32(0x9d7edb64),
	  SPH_C32(0x5bfbdb7a), SPH_C32(0x7513713b) },
	{ SPH_C32(0x3d0b1200), SPH_C32(0x5772aa73), SPH_C32(0x68da3256),
	  SPH_C32(0xbb015cc8), SPH_C32(0x6c110200), SPH_C32(0x8f35b35a),
	  SPH_C32(0xdb390df5), SPH_C32(0x4ee0da17) },
	{ SPH_C32(0xfbab1400), SPH_C32(0x57b68e09), SPH_C32(0xe73855ca),
	  SPH_C32(0x976bae44), SPH_C32(0xdf140000), SPH_C32(0xbf9d0096),
	  SPH_C32(0xd1ae11bc), SPH_C32(0xf48b4487) },
	{ SPH_C32(0x5c131600), SPH_C32(0x48a40a3a), SPH_C32(0x87ddac38),
	  SPH_C32(0x097fdb32), SPH_C32(0x31320000), SPH_C32(0xadd668a8),
	  SPH_C32(0x516cc733), SPH_C32(0xcf78efab) },
	{ SPH_C32(0x158d1400), SPH_C32(0x45fde637), SPH_C32(0x67fa8345),
	  SPH_C32(0xac980568), SPH_C32(0x968a0200), SPH_C32(0xb2c4ec9b),
	  SPH_C32(0x31893ec1), SPH_C32(0x516c9add) },
	{ SPH_C32(0xb2351600), SPH_C32(0x5aef6204), SPH_C32(0x071f7ab7),
	  SPH_C32(0x328c701e), SPH_C32(0x78ac0200), SPH_C32(0xa08f84a5),
	  SPH_C32(0xb14be84e), SPH_C32(0x6a9f31f1) },
	{ SPH_C32(0x60281000), SPH_C32(0x75917181), SPH_C32(0xe28ff890),
	  SPH_C32(0x3a996974), SPH_C32(0x502a0400), SPH_C32(0xb200c8e1),
	  SPH_C32(0xbe6b595d), SPH_C32(0x7d066851) },
	{ SPH_C32(0xc7901200), SPH_C32(0x6a83f5b2), SPH_C32(0x826a0162),
	  SPH_C32(0xa48d1c02), SPH_C32(0xbe0c0400), SPH_C32(0xa04ba0df),
	  SPH_C32(0x3ea98fd2), SPH_C32(0x46f5c37d) },
	{ SPH_C32(0x8e0e1000), SPH_C32(0x67da19bf), SPH_C32(0x624d2e1f),
	  SPH_C32(0x016ac258), SPH_C32(0x19b40600), SPH_C32(0xbf5924ec),
	  SPH_C32(0x5e4c7620), SPH_C32(0xd8e1b60b) },
	{ SPH_C32(0x29b61200), SPH_C32(0x78c89d8c), SPH_C32(0x02a8d7ed),
	  SPH_C32(0x9f7eb72e), SPH_C32(0xf7920600), SPH_C32(0xad124cd2),
	  SPH_C32(0xde8ea0af), SPH_C32(0xe3121d27) },
	{ SPH_C32(0xef161400), SPH_C32(0x780cb9f6), SPH_C32(0x8d4ab071),
	  SPH_C32(0xb31445a2), SPH_C32(0x44970400), SPH_C32(0x9dbaff1e),
	  SPH_C32(0xd419bce6), SPH_C32(0x597983b7) },
	{ SPH_C32(0x48ae1600), SPH_C32(0x671e3dc5), SPH_C32(0xedaf4983),
	  SPH_C32(0x2d0030d4), SPH_C32(0xaab10400), SPH_C32(0x8ff19720),
	  SPH_C32(0x54db6a69), SPH_C32(0x628a289b) },
	{ SPH_C32(0x01301400), SPH_C32(0x6a47d1c8), SPH_C32(0x0d8866fe),
	  SPH_C32(0x88e7ee8e), SPH_C32(0x0d090600), SPH_C32(0x90e31313),
	  SPH_C32(0x343e939b), SPH_C32(0xfc9e5ded) },
	{ SPH_C32(0xa6881600), SPH_C32(0x755555fb), SPH_C32(0x6d6d9f0c),
	  SPH_C32(0x16f39bf8), SPH_C32(0xe32f0600), SPH_C32(0x82a87b2d),
	  SPH_C32(0xb4fc4514), SPH_C32(0xc76df6c1) },
	{ SPH_C32(0xaaa71800), SPH_C32(0x72a81680), SPH_C32(0xf97837ec),
	  SPH_C32(0xb8591d04), SPH_C32(0x2a220000), SPH_C32(0xc47ebf14),
	  SPH_C32(0x04ce77d4), SPH_C32(0xcb92c512) },
	{ SPH_C32(0x0d1f1a00), SPH_C32(0x6dba92b3), SPH_C32(0x999dce1e),
	  SPH_C32(0x264d6872), SPH_C32(0xc4040000), SPH_C32(0xd635d72a),
	  SPH_C32(0x840ca15b), SPH_C32(0xf0616e3e) },
	{ SPH_C32(0x44811800), SPH_C32(0x60e37ebe), SPH_C32(0x79bae163),
	  SPH_C32(0x83aab628), SPH_C32(0x63bc0200), SPH_C32(0xc9275319),
	  SPH_C32(0xe4e958a9), SPH_C32(0x6e751b48) },
	{ SPH_C32(0xe3391a00), SPH_C32(0x7ff1fa8d), SPH_C32(0x195f1891),
	  SPH_C32(0x1dbec35e), SPH_C32(0x8d9a0200), SPH_C32(0xdb6c3b27),
	  SPH_C32(0x642b8e26), SPH_C32(0x5586b064) },
	{ SPH_C32(0x25991c00), SPH_C32(0x7f35def7), SPH_C32(0x96bd7f0d),
	  SPH_C32(0x31d431d2), SPH_C32(0x3e9f0000), SPH_C32(0xebc488eb),
	  SPH_C32(0x6ebc926f), SPH_C32(0xefed2ef4) },
	{ SPH_C32(0x82211e00), SPH_C32(0x60275ac4), SPH_C32(0xf65886ff),
	  SPH_C32(0xafc044a4), SPH_C32(0xd0b90000), SPH_C32(0xf98fe0d5),
	  SPH_C32(0xee7e44e0), SPH_C32(0xd41e85d8) },
	{ SPH_C32(0xcbbf1c00), SPH_C32(0x6d7eb6c9), SPH_C32(0x167fa982),
	  SPH_C32(0x0a279afe), SPH_C32(0x77010200), SPH_C32(0xe69d64e6),
	  SPH_C32(0x8e9bbd12), SPH_C32(0x4a0af0ae) },
	{ SPH_C32(0x6c071e00), SPH_C32(0x726c32fa), SPH_C32(0x769a5070),
	  SPH_C32(0x9433ef88), SPH_C32(0x99270200), SPH_C32(0xf4d60cd8),
	  SPH_C32(0x0e596b9d), SPH_C32(0x71f95b82) },
	{ SPH_C32(0xbe1a1800), SPH_C32(0x5d12217f), SPH_C32(0x930ad257),
	  SPH_C32(0x9c26f6e2), SPH_C32(0xb1a10400), SPH_C32(0xe659409c),
	  SPH_C32(0x0179da8e), SPH_C32(0x66600222) },
	{ SPH_C32(0x19a21a00), SPH_C32(0x4200a54c), SPH_C32(0xf3ef2ba5),
	  SPH_C32(0x02328394), SPH_C32(0x5f870400), SPH_C32(0xf41228a2),
	  SPH_C32(0x81bb0c01), SPH_C32(0x5d93a90e) },
	{ SPH_C32(0x503c1800), SPH_C32(0x4f594941), SPH_C32(0x13c804d8),
	  SPH_C32(0xa7d55dce), SPH_C32(0xf83f0600), SPH_C32(0xeb00ac91),
	  SPH_C32(0xe15ef5f3), SPH_C32(0xc387dc78) },
	{ SPH_C32(0xf7841a00), SPH_C32(0x504bcd72), SPH_C32(0x732dfd2a),
	  SPH_C32(0x39c128b8), SPH_C32(0x16190600), SPH_C32(0xf94bc4af),
	  SPH_C32(0x619c237c), SPH_C32(0xf8747754) },
	{ SPH_C32(0x31241c00), SPH_C32(0x508fe908), SPH_C32(0xfccf9ab6),
	  SPH_C32(0x15abda34), SPH_C32(0xa51c0400), SPH_C32(0xc9e37763),
	  SPH_C32(0x6b0b3f35), SPH_C32(0x421fe9c4) },
	{ SPH_C32(0x969c1e00), SPH_C32(0x4f9d6d3b), SPH_C32(0x9c2a6344),
	  SPH_C32(0x8bbfaf42), SPH_C32(0x4b3a0400), SPH_C32(0xdba81f5d),
	  SPH_C32(0xebc9e9ba), SPH_C32(0x79ec42e8) },
	{ SPH_C32(0xdf021c00), SPH_C32(0x42c48136), SPH_C32(0x7c0d4c39),
	  SPH_C32(0x2e587118), SPH_C32(0xec820600), SPH_C32(0xc4ba9b6e),
	  SPH_C32(0x8b2c1048), SPH_C32(0xe7f8379e) },
	{ SPH_C32(0x78ba1e00), SPH_C32(0x5dd60505), SPH_C32(0x1ce8b5cb),
	  SPH_C32(0xb04c046e), SPH_C32(0x02a40600), SPH_C32(0xd6f1f350),
	  SPH_C32(0x0beec6c7), SPH_C32(0xdc0b9cb2) },
	{ SPH_C32(0x951e1000), SPH_C32(0x0e72ce03), SPH_C32(0x37ef9ef8),
	  SPH_C32(0x0580e8e1), SPH_C32(0xf4100800), SPH_C32(0xecfdefea),
	  SPH_C32(0x754b5d13), SPH_C32(0x6d2d5a84) },
	{ SPH_C32(0x32a61200), SPH_C32(0x11604a30), SPH_C32(0x570a670a),
	  SPH_C32(0x9b949d97), SPH_C32(0x1a360800), SPH_C32(0xfeb687d4),
	  SPH_C32(0xf5898b9c), SPH_C32(0x56def1a8) },
	{ SPH_C32(0x7b381000), SPH_C32(0x1c39a63d), SPH_C32(0xb72d4877),
	  SPH_C32(0x3e7343cd), SPH_C32(0xbd8e0a00), SPH_C32(0xe1a403e7),
	  SPH_C32(0x956c726e), SPH_C32(0xc8ca84de) },
	{ SPH_C32(0xdc801200), SPH_C32(0x032b220e), SPH_C32(0xd7c8b185),
	  SPH_C32(0xa06736bb), SPH_C32(0x53a80a00), SPH_C32(0xf3ef6bd9),
	  SPH_C32(0x15aea4e1), SPH_C32(0xf3392ff2) },
	{ SPH_C32(0x1a201400), SPH_C32(0x03ef0674), SPH_C32(0x582ad619),
	  SPH_C32(0x8c0dc437), SPH_C32(0xe0ad0800), SPH_C32(0xc347d815),
	  SPH_C32(0x1f39b8a8), SPH_C32(0x4952b162) },
	{ SPH_C32(0xbd981600), SPH_C32(0x1cfd8247), SPH_C32(0x38cf2feb),
	  SPH_C32(0x1219b141), SPH_C32(0x0e8b0800), SPH_C32(0xd10cb02b),
	  SPH_C32(0x9ffb6e27), SPH_C32(0x72a11a4e) },
	{ SPH_C32(0xf4061400), SPH_C32(0x11a46e4a), SPH_C32(0xd8e80096),
	  SPH_C32(0xb7fe6f1b), SPH_C32(0xa9330a00), SPH_C32(0xce1e3418),
	  SPH_C32(0xff1e97d5), SPH_C32(0xecb56f38) },
	{ SPH_C32(0x53be1600), SPH_C32(0x0eb6ea79), SPH_C32(0xb80df964),
	  SPH_C32(0x29ea1a6d), SPH_C32(0x47150a00), SPH_C32(0xdc555c26),
	  SPH_C32(0x7fdc415a), SPH_C32(0xd746c414) },
	{ SPH_C32(0x81a31000), SPH_C32(0x21c8f9fc), SPH_C32(0x5d9d7b43),
	  SPH_C32(0x21ff0307), SPH_C32(0x6f930c00), SPH_C32(0xceda1062),
	  SPH_C32(0x70fcf049), SPH_C32(0xc0df9db4) },
	{ SPH_C32(0x261b1200), SPH_C32(0x3eda7dcf), SPH_C32(0x3d7882b1),
	  SPH_C32(0xbfeb7671), SPH_C32(0x81b50c00), SPH_C32(0xdc91785c),
	  SPH_C32(0xf03e26c6), SPH_C32(0xfb2c3698) },
	{ SPH_C32(0x6f851000), SPH_C32(0x338391c2), SPH_C32(0xdd5fadcc),
	  SPH_C32(0x1a0ca82b), SPH_C32(0x260d0e00), SPH_C32(0xc383fc6f),
	  SPH_C32(0x90dbdf34), SPH_C32(0x653843ee) },
	{ SPH_C32(0xc83d1200), SPH_C32(0x2c9115f1), SPH_C32(0xbdba543e),
	  SPH_C32(0x8418dd5d), SPH_C32(0xc82b0e00), SPH_C32(0xd1c89451),
	  SPH_C32(0x101909bb), SPH_C32(0x5ecbe8c2) },
	{ SPH_C32(0x0e9d1400), SPH_C32(0x2c55318b), SPH_C32(0x325833a2),
	  SPH_C32(0xa8722fd1), SPH_C32(0x7b2e0c00), SPH_C32(0xe160279d),
	  SPH_C32(0x1a8e15f2), SPH_C32(0xe4a07652) },
	{ SPH_C32(0xa9251600), SPH_C32(0x3347b5b8), SPH_C32(0x52bdca50),
	  SPH_C32(0x36665aa7), SPH_C32(0x95080c00), SPH_C32(0xf32b4fa3),
	  SPH_C32(0x9a4cc37d), SPH_C32(0xdf53dd7e) },
	{ SPH_C32(0xe0bb1400), SPH_C32(0x3e1e59b5), SPH_C32(0xb29ae52d),
	  SPH_C32(0x938184fd), SPH_C32(0x32b00e00), SPH_C32(0xec39cb90),
	  SPH_C32(0xfaa93a8f), SPH_C32(0x4147a808) },
	{ SPH_C32(0x47031600), SPH_C32(0x210cdd86), SPH_C32(0xd27f1cdf),
	  SPH_C32(0x0d95f18b), SPH_C32(0xdc960e00), SPH_C32(0xfe72a3ae),
	  SPH_C32(0x7a6bec00), SPH_C32(0x7ab40324) },
	{ SPH_C32(0x4b2c1800), SPH_C32(0x26f19efd), SPH_C32(0x466ab43f),
	  SPH_C32(0xa33f7777), SPH_C32(0x159b0800), SPH_C32(0xb8a46797),
	  SPH_C32(0xca59dec0), SPH_C32(0x764b30f7) },
	{ SPH_C32(0xec941a00), SPH_C32(0x39e31ace), SPH_C32(0x268f4dcd),
	  SPH_C32(0x3d2b0201), SPH_C32(0xfbbd0800), SPH_C32(0xaaef0fa9),
	  SPH_C32(0x4a9b084f), SPH_C32(0x4db89bdb) },
	{ SPH_C32(0xa50a1800), SPH_C32(0x34baf6c3), SPH_C32(0xc6a862b0),
	  SPH_C32(0x98ccdc5b), SPH_C32(0x5c050a00), SPH_C32(0xb5fd8b9a),
	  SPH_C32(0x2a7ef1bd), SPH_C32(0xd3aceead) },
	{ SPH_C32(0x02b21a00), SPH_C32(0x2ba872f0), SPH_C32(0xa64d9b42),
	  SPH_C32(0x06d8a92d), SPH_C32(0xb2230a00), SPH_C32(0xa7b6e3a4),
	  SPH_C32(0xaabc2732), SPH_C32(0xe85f4581) },
	{ SPH_C32(0xc4121c00), SPH_C32(0x2b6c568a), SPH_C32(0x29affcde),
	  SPH_C32(0x2ab25ba1), SPH_C32(0x01260800), SPH_C32(0x971e5068),
	  SPH_C32(0xa02b3b7b), SPH_C32(0x5234db11) },
	{ SPH_C32(0x63aa1e00), SPH_C32(0x347ed2b9), SPH_C32(0x494a052c),
	  SPH_C32(0xb4a62ed7), SPH_C32(0xef000800), SPH_C32(0x85553856),
	  SPH_C32(0x20e9edf4), SPH_C32(0x69c7703d) },
	{ SPH_C32(0x2a341c00), SPH_C32(0x39273eb4), SPH_C32(0xa96d2a51),
	  SPH_C32(0x1141f08d), SPH_C32(0x48b80a00), SPH_C32(0x9a47bc65),
	  SPH_C32(0x400c1406), SPH_C32(0xf7d3054b) },
	{ SPH_C32(0x8d8c1e00), SPH_C32(0x2635ba87), SPH_C32(0xc988d3a3),
	  SPH_C32(0x8f5585fb), SPH_C32(0xa69e0a00), SPH_C32(0x880cd45b),
	  SPH_C32(0xc0cec289), SPH_C32(0xcc20ae67) },
	{ SPH_C32(0x5f911800), SPH_C32(0x094ba902), SPH_C32(0x2c185184),
	  SPH_C32(0x87409c91), SPH_C32(0x8e180c00), SPH_C32(0x9a83981f),
	  SPH_C32(0xcfee739a), SPH_C32(0xdbb9f7c7) },
	{ SPH_C32(0xf8291a00), SPH_C32(0x16592d31), SPH_C32(0x4cfda876),
	  SPH_C32(0x1954e9e7), SPH_C32(0x603e0c00), SPH_C32(0x88c8f021),
	  SPH_C32(0x4f2ca515), SPH_C32(0xe04a5ceb) },
	{ SPH_C32(0xb1b71800), SPH_C32(0x1b00c13c), SPH_C32(0xacda870b),
	  SPH_C32(0xbcb337bd), SPH_C32(0xc7860e00), SPH_C32(0x97da7412),
	  SPH_C32(0x2fc95ce7), SPH_C32(0x7e5e299d) },
	{ SPH_C32(0x160f1a00), SPH_C32(0x0412450f), SPH_C32(0xcc3f7ef9),
	  SPH_C32(0x22a742cb), SPH_C32(0x29a00e00), SPH_C32(0x85911c2c),
	  SPH_C32(0xaf0b8a68), SPH_C32(0x45ad82b1) },
	{ SPH_C32(0xd0af1c00), SPH_C32(0x04d66175), SPH_C32(0x43dd1965),
	  SPH_C32(0x0ecdb047), SPH_C32(0x9aa50c00), SPH_C32(0xb539afe0),
	  SPH_C32(0xa59c9621), SPH_C32(0xffc61c21) },
	{ SPH_C32(0x77171e00), SPH_C32(0x1bc4e546), SPH_C32(0x2338e097),
	  SPH_C32(0x90d9c531), SPH_C32(0x74830c00), SPH_C32(0xa772c7de),
	  SPH_C32(0x255e40ae), SPH_C32(0xc435b70d) },
	{ SPH_C32(0x3e891c00), SPH_C32(0x169d094b), SPH_C32(0xc31fcfea),
	  SPH_C32(0x353e1b6b), SPH_C32(0xd33b0e00), SPH_C32(0xb86043ed),
	  SPH_C32(0x45bbb95c), SPH_C32(0x5a21c27b) },
	{ SPH_C32(0x99311e00), SPH_C32(0x098f8d78), SPH_C32(0xa3fa3618),
	  SPH_C32(0xab2a6e1d), SPH_C32(0x3d1d0e00), SPH_C32(0xaa2b2bd3),
	  SPH_C32(0xc5796fd3), SPH_C32(0x61d26957) }
};

static const sph_u32 T256_14[128][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x5c720000), SPH_C32(0xc9bacd12), SPH_C32(0x79a90df9),
	  SPH_C32(0x63e92178), SPH_C32(0xfeca0020), SPH_C32(0x485d28e4),
	  SPH_C32(0x806741fd), SPH_C32(0x814681b8) },
	{ SPH_C32(0x4dce0040), SPH_C32(0x3b5bec7e), SPH_C32(0x36656ba8),
	  SPH_C32(0x23633a05), SPH_C32(0x78ab0000), SPH_C32(0xa0cd5a34),
	  SPH_C32(0x5d5ca0f7), SPH_C32(0x727784cb) },
	{ SPH_C32(0x11bc0040), SPH_C32(0xf2e1216c), SPH_C32(0x4fcc6651),
	  SPH_C32(0x408a1b7d), SPH_C32(0x86610020), SPH_C32(0xe89072d0),
	  SPH_C32(0xdd3be10a), SPH_C32(0xf3310573) },
	{ SPH_C32(0x78ab0000), SPH_C32(0xa0cd5a34), SPH_C32(0x5d5ca0f7),
	  SPH_C32(0x727784cb), SPH_C32(0x35650040), SPH_C32(0x9b96b64a),
	  SPH_C32(0x6b39cb5f), SPH_C32(0x5114bece) },
	{ SPH_C32(0x24d90000), SPH_C32(0x69779726), SPH_C32(0x24f5ad0e),
	  SPH_C32(0x119ea5b3), SPH_C32(0xcbaf0060), SPH_C32(0xd3cb9eae),
	  SPH_C32(0xeb5e8aa2), SPH_C32(0xd0523f76) },
	{ SPH_C32(0x35650040), SPH_C32(0x9b96b64a), SPH_C32(0x6b39cb5f),
	  SPH_C32(0x5114bece), SPH_C32(0x4dce0040), SPH_C32(0x3b5bec7e),
	  SPH_C32(0x36656ba8), SPH_C32(0x23633a05) },
	{ SPH_C32(0x69170040), SPH_C32(0x522c7b58), SPH_C32(0x1290c6a6),
	  SPH_C32(0x32fd9fb6), SPH_C32(0xb3040060), SPH_C32(0x7306c49a),
	  SPH_C32(0xb6022a55), SPH_C32(0xa225bbbd) },
	{ SPH_C32(0x5bd20080), SPH_C32(0x450f18ec), SPH_C32(0xc2c46c55),
	  SPH_C32(0xf362b233), SPH_C32(0x39a60000), SPH_C32(0x4ab753eb),
	  SPH_C32(0xd14e094b), SPH_C32(0xb772b42b) },
	{ SPH_C32(0x07a00080), SPH_C32(0x8cb5d5fe), SPH_C32(0xbb6d61ac),
	  SPH_C32(0x908b934b), SPH_C32(0xc76c0020), SPH_C32(0x02ea7b0f),
	  SPH_C32(0x512948b6), SPH_C32(0x36343593) },
	{ SPH_C32(0x161c00c0), SPH_C32(0x7e54f492), SPH_C32(0xf4a107fd),
	  SPH_C32(0xd0018836), SPH_C32(0x410d0000), SPH_C32(0xea7a09df),
	  SPH_C32(0x8c12a9bc), SPH_C32(0xc50530e0) },
	{ SPH_C32(0x4a6e00c0), SPH_C32(0xb7ee3980), SPH_C32(0x8d080a04),
	  SPH_C32(0xb3e8a94e), SPH_C32(0xbfc70020), SPH_C32(0xa227213b),
	  SPH_C32(0x0c75e841), SPH_C32(0x4443b158) },
	{ SPH_C32(0x23790080), SPH_C32(0xe5c242d8), SPH_C32(0x9f98cca2),
	  SPH_C32(0x811536f8), SPH_C32(0x0cc30040), SPH_C32(0xd121e5a1),
	  SPH_C32(0xba77c214), SPH_C32(0xe6660ae5) },
	{ SPH_C32(0x7f0b0080), SPH_C32(0x2c788fca), SPH_C32(0xe631c15b),
	  SPH_C32(0xe2fc1780), SPH_C32(0xf2090060), SPH_C32(0x997ccd45),
	  SPH_C32(0x3a1083e9), SPH_C32(0x67208b5d) },
	{ SPH_C32(0x6eb700c0), SPH_C32(0xde99aea6), SPH_C32(0xa9fda70a),
	  SPH_C32(0xa2760cfd), SPH_C32(0x74680040), SPH_C32(0x71ecbf95),
	  SPH_C32(0xe72b62e3), SPH_C32(0x94118e2e) },
	{ SPH_C32(0x32c500c0), SPH_C32(0x172363b4), SPH_C32(0xd054aaf3),
	  SPH_C32(0xc19f2d85), SPH_C32(0x8aa20060), SPH_C32(0x39b19771),
	  SPH_C32(0x674c231e), SPH_C32(0x15570f96) },
	{ SPH_C32(0x39a60000), SPH_C32(0x4ab753eb), SPH_C32(0xd14e094b),
	  SPH_C32(0xb772b42b), SPH_C32(0x62740080), SPH_C32(0x0fb84b07),
	  SPH_C32(0x138a651e), SPH_C32(0x44100618) },
	{ SPH_C32(0x65d40000), SPH_C32(0x830d9ef9), SPH_C32(0xa8e704b2),
	  SPH_C32(0xd49b9553), SPH_C32(0x9cbe00a0), SPH_C32(0x47e563e3),
	  SPH_C32(0x93ed24e3), SPH_C32(0xc55687a0) },
	{ SPH_C32(0x74680040), SPH_C32(0x71ecbf95), SPH_C32(0xe72b62e3),
	  SPH_C32(0x94118e2e), SPH_C32(0x1adf0080), SPH_C32(0xaf751133),
	  SPH_C32(0x4ed6c5e9), SPH_C32(0x366782d3) },
	{ SPH_C32(0x281a0040), SPH_C32(0xb8567287), SPH_C32(0x9e826f1a),
	  SPH_C32(0xf7f8af56), SPH_C32(0xe41500a0), SPH_C32(0xe72839d7),
	  SPH_C32(0xceb18414), SPH_C32(0xb721036b) },
	{ SPH_C32(0x410d0000), SPH_C32(0xea7a09df), SPH_C32(0x8c12a9bc),
	  SPH_C32(0xc50530e0), SPH_C32(0x571100c0), SPH_C32(0x942efd4d),
	  SPH_C32(0x78b3ae41), SPH_C32(0x1504b8d6) },
	{ SPH_C32(0x1d7f0000), SPH_C32(0x23c0c4cd), SPH_C32(0xf5bba445),
	  SPH_C32(0xa6ec1198), SPH_C32(0xa9db00e0), SPH_C32(0xdc73d5a9),
	  SPH_C32(0xf8d4efbc), SPH_C32(0x9442396e) },
	{ SPH_C32(0x0cc30040), SPH_C32(0xd121e5a1), SPH_C32(0xba77c214),
	  SPH_C32(0xe6660ae5), SPH_C32(0x2fba00c0), SPH_C32(0x34e3a779),
	  SPH_C32(0x25ef0eb6), SPH_C32(0x67733c1d) },
	{ SPH_C32(0x50b10040), SPH_C32(0x189b28b3), SPH_C32(0xc3decfed),
	  SPH_C32(0x858f2b9d), SPH_C32(0xd17000e0), SPH_C32(0x7cbe8f9d),
	  SPH_C32(0xa5884f4b), SPH_C32(0xe635bda5) },
	{ SPH_C32(0x62740080), SPH_C32(0x0fb84b07), SPH_C32(0x138a651e),
	  SPH_C32(0x44100618), SPH_C32(0x5bd20080), SPH_C32(0x450f18ec),
	  SPH_C32(0xc2c46c55), SPH_C32(0xf362b233) },
	{ SPH_C32(0x3e060080), SPH_C32(0xc6028615), SPH_C32(0x6a2368e7),
	  SPH_C32(0x27f92760), SPH_C32(0xa51800a0), SPH_C32(0x0d523008),
	  SPH_C32(0x42a32da8), SPH_C32(0x7224338b) },
	{ SPH_C32(0x2fba00c0), SPH_C32(0x34e3a779), SPH_C32(0x25ef0eb6),
	  SPH_C32(0x67733c1d), SPH_C32(0x23790080), SPH_C32(0xe5c242d8),
	  SPH_C32(0x9f98cca2), SPH_C32(0x811536f8) },
	{ SPH_C32(0x73c800c0), SPH_C32(0xfd596a6b), SPH_C32(0x5c46034f),
	  SPH_C32(0x049a1d65), SPH_C32(0xddb300a0), SPH_C32(0xad9f6a3c),
	  SPH_C32(0x1fff8d5f), SPH_C32(0x0053b740) },
	{ SPH_C32(0x1adf0080), SPH_C32(0xaf751133), SPH_C32(0x4ed6c5e9),
	  SPH_C32(0x366782d3), SPH_C32(0x6eb700c0), SPH_C32(0xde99aea6),
	  SPH_C32(0xa9fda70a), SPH_C32(0xa2760cfd) },
	{ SPH_C32(0x46ad0080), SPH_C32(0x66cfdc21), SPH_C32(0x377fc810),
	  SPH_C32(0x558ea3ab), SPH_C32(0x907d00e0), SPH_C32(0x96c48642),
	  SPH_C32(0x299ae6f7), SPH_C32(0x23308d45) },
	{ SPH_C32(0x571100c0), SPH_C32(0x942efd4d), SPH_C32(0x78b3ae41),
	  SPH_C32(0x1504b8d6), SPH_C32(0x161c00c0), SPH_C32(0x7e54f492),
	  SPH_C32(0xf4a107fd), SPH_C32(0xd0018836) },
	{ SPH_C32(0x0b6300c0), SPH_C32(0x5d94305f), SPH_C32(0x011aa3b8),
	  SPH_C32(0x76ed99ae), SPH_C32(0xe8d600e0), SPH_C32(0x3609dc76),
	  SPH_C32(0x74c64600), SPH_C32(0x5147098e) },
	{ SPH_C32(0xb7a40100), SPH_C32(0x8a1f31d8), SPH_C32(0x8589d8ab),
	  SPH_C32(0xe6c46464), SPH_C32(0x734c0000), SPH_C32(0x956fa7d6),
	  SPH_C32(0xa29d1297), SPH_C32(0x6ee56854) },
	{ SPH_C32(0xebd60100), SPH_C32(0x43a5fcca), SPH_C32(0xfc20d552),
	  SPH_C32(0x852d451c), SPH_C32(0x8d860020), SPH_C32(0xdd328f32),
	  SPH_C32(0x22fa536a), SPH_C32(0xefa3e9ec) },
	{ SPH_C32(0xfa6a0140), SPH_C32(0xb144dda6), SPH_C32(0xb3ecb303),
	  SPH_C32(0xc5a75e61), SPH_C32(0x0be70000), SPH_C32(0x35a2fde2),
	  SPH_C32(0xffc1b260), SPH_C32(0x1c92ec9f) },
	{ SPH_C32(0xa6180140), SPH_C32(0x78fe10b4), SPH_C32(0xca45befa),
	  SPH_C32(0xa64e7f19), SPH_C32(0xf52d0020), SPH_C32(0x7dffd506),
	  SPH_C32(0x7fa6f39d), SPH_C32(0x9dd46d27) },
	{ SPH_C32(0xcf0f0100), SPH_C32(0x2ad26bec), SPH_C32(0xd8d5785c),
	  SPH_C32(0x94b3e0af), SPH_C32(0x46290040), SPH_C32(0x0ef9119c),
	  SPH_C32(0xc9a4d9c8), SPH_C32(0x3ff1d69a) },
	{ SPH_C32(0x937d0100), SPH_C32(0xe368a6fe), SPH_C32(0xa17c75a5),
	  SPH_C32(0xf75ac1d7), SPH_C32(0xb8e30060), SPH_C32(0x46a43978),
	  SPH_C32(0x49c39835), SPH_C32(0xbeb75722) },
	{ SPH_C32(0x82c10140), SPH_C32(0x11898792), SPH_C32(0xeeb013f4),
	  SPH_C32(0xb7d0daaa), SPH_C32(0x3e820040), SPH_C32(0xae344ba8),
	  SPH_C32(0x94f8793f), SPH_C32(0x4d865251) },
	{ SPH_C32(0xdeb30140), SPH_C32(0xd8334a80), SPH_C32(0x97191e0d),
	  SPH_C32(0xd439fbd2), SPH_C32(0xc0480060), SPH_C32(0xe669634c),
	  SPH_C32(0x149f38c2), SPH_C32(0xccc0d3e9) },
	{ SPH_C32(0xec760180), SPH_C32(0xcf102934), SPH_C32(0x474db4fe),
	  SPH_C32(0x15a6d657), SPH_C32(0x4aea0000), SPH_C32(0xdfd8f43d),
	  SPH_C32(0x73d31bdc), SPH_C32(0xd997dc7f) },
	{ SPH_C32(0xb0040180), SPH_C32(0x06aae426), SPH_C32(0x3ee4b907),
	  SPH_C32(0x764ff72f), SPH_C32(0xb4200020), SPH_C32(0x9785dcd9),
	  SPH_C32(0xf3b45a21), SPH_C32(0x58d15dc7) },
	{ SPH_C32(0xa1b801c0), SPH_C32(0xf44bc54a), SPH_C32(0x7128df56),
	  SPH_C32(0x36c5ec52), SPH_C32(0x32410000), SPH_C32(0x7f15ae09),
	  SPH_C32(0x2e8fbb2b), SPH_C32(0xabe058b4) },
	{ SPH_C32(0xfdca01c0), SPH_C32(0x3df10858), SPH_C32(0x0881d2af),
	  SPH_C32(0x552ccd2a), SPH_C32(0xcc8b0020), SPH_C32(0x374886ed),
	  SPH_C32(0xaee8fad6), SPH_C32(0x2aa6d90c) },
	{ SPH_C32(0x94dd0180), SPH_C32(0x6fdd7300), SPH_C32(0x1a111409),
	  SPH_C32(0x67d1529c), SPH_C32(0x7f8f0040), SPH_C32(0x444e4277),
	  SPH_C32(0x18ead083), SPH_C32(0x888362b1) },
	{ SPH_C32(0xc8af0180), SPH_C32(0xa667be12), SPH_C32(0x63b819f0),
	  SPH_C32(0x043873e4), SPH_C32(0x81450060), SPH_C32(0x0c136a93),
	  SPH_C32(0x988d917e), SPH_C32(0x09c5e309) },
	{ SPH_C32(0xd91301c0), SPH_C32(0x54869f7e), SPH_C32(0x2c747fa1),
	  SPH_C32(0x44b26899), SPH_C32(0x07240040), SPH_C32(0xe4831843),
	  SPH_C32(0x45b67074), SPH_C32(0xfaf4e67a) },
	{ SPH_C32(0x856101c0), SPH_C32(0x9d3c526c), SPH_C32(0x55dd7258),
	  SPH_C32(0x275b49e1), SPH_C32(0xf9ee0060), SPH_C32(0xacde30a7),
	  SPH_C32(0xc5d13189), SPH_C32(0x7bb267c2) },
	{ SPH_C32(0x8e020100), SPH_C32(0xc0a86233), SPH_C32(0x54c7d1e0),
	  SPH_C32(0x51b6d04f), SPH_C32(0x11380080), SPH_C32(0x9ad7ecd1),
	  SPH_C32(0xb1177789), SPH_C32(0x2af56e4c) },
	{ SPH_C32(0xd2700100), SPH_C32(0x0912af21), SPH_C32(0x2d6edc19),
	  SPH_C32(0x325ff137), SPH_C32(0xeff200a0), SPH_C32(0xd28ac435),
	  SPH_C32(0x31703674), SPH_C32(0xabb3eff4) },
	{ SPH_C32(0xc3cc0140), SPH_C32(0xfbf38e4d), SPH_C32(0x62a2ba48),
	  SPH_C32(0x72d5ea4a), SPH_C32(0x69930080), SPH_C32(0x3a1ab6e5),
	  SPH_C32(0xec4bd77e), SPH_C32(0x5882ea87) },
	{ SPH_C32(0x9fbe0140), SPH_C32(0x3249435f), SPH_C32(0x1b0bb7b1),
	  SPH_C32(0x113ccb32), SPH_C32(0x975900a0), SPH_C32(0x72479e01),
	  SPH_C32(0x6c2c9683), SPH_C32(0xd9c46b3f) },
	{ SPH_C32(0xf6a90100), SPH_C32(0x60653807), SPH_C32(0x099b7117),
	  SPH_C32(0x23c15484), SPH_C32(0x245d00c0), SPH_C32(0x01415a9b),
	  SPH_C32(0xda2ebcd6), SPH_C32(0x7be1d082) },
	{ SPH_C32(0xaadb0100), SPH_C32(0xa9dff515), SPH_C32(0x70327cee),
	  SPH_C32(0x402875fc), SPH_C32(0xda9700e0), SPH_C32(0x491c727f),
	  SPH_C32(0x5a49fd2b), SPH_C32(0xfaa7513a) },
	{ SPH_C32(0xbb670140), SPH_C32(0x5b3ed479), SPH_C32(0x3ffe1abf),
	  SPH_C32(0x00a26e81), SPH_C32(0x5cf600c0), SPH_C32(0xa18c00af),
	  SPH_C32(0x87721c21), SPH_C32(0x09965449) },
	{ SPH_C32(0xe7150140), SPH_C32(0x9284196b), SPH_C32(0x46571746),
	  SPH_C32(0x634b4ff9), SPH_C32(0xa23c00e0), SPH_C32(0xe9d1284b),
	  SPH_C32(0x07155ddc), SPH_C32(0x88d0d5f1) },
	{ SPH_C32(0xd5d00180), SPH_C32(0x85a77adf), SPH_C32(0x9603bdb5),
	  SPH_C32(0xa2d4627c), SPH_C32(0x289e0080), SPH_C32(0xd060bf3a),
	  SPH_C32(0x60597ec2), SPH_C32(0x9d87da67) },
	{ SPH_C32(0x89a20180), SPH_C32(0x4c1db7cd), SPH_C32(0xefaab04c),
	  SPH_C32(0xc13d4304), SPH_C32(0xd65400a0), SPH_C32(0x983d97de),
	  SPH_C32(0xe03e3f3f), SPH_C32(0x1cc15bdf) },
	{ SPH_C32(0x981e01c0), SPH_C32(0xbefc96a1), SPH_C32(0xa066d61d),
	  SPH_C32(0x81b75879), SPH_C32(0x50350080), SPH_C32(0x70ade50e),
	  SPH_C32(0x3d05de35), SPH_C32(0xeff05eac) },
	{ SPH_C32(0xc46c01c0), SPH_C32(0x77465bb3), SPH_C32(0xd9cfdbe4),
	  SPH_C32(0xe25e7901), SPH_C32(0xaeff00a0), SPH_C32(0x38f0cdea),
	  SPH_C32(0xbd629fc8), SPH_C32(0x6eb6df14) },
	{ SPH_C32(0xad7b0180), SPH_C32(0x256a20eb), SPH_C32(0xcb5f1d42),
	  SPH_C32(0xd0a3e6b7), SPH_C32(0x1dfb00c0), SPH_C32(0x4bf60970),
	  SPH_C32(0x0b60b59d), SPH_C32(0xcc9364a9) },
	{ SPH_C32(0xf1090180), SPH_C32(0xecd0edf9), SPH_C32(0xb2f610bb),
	  SPH_C32(0xb34ac7cf), SPH_C32(0xe33100e0), SPH_C32(0x03ab2194),
	  SPH_C32(0x8b07f460), SPH_C32(0x4dd5e511) },
	{ SPH_C32(0xe0b501c0), SPH_C32(0x1e31cc95), SPH_C32(0xfd3a76ea),
	  SPH_C32(0xf3c0dcb2), SPH_C32(0x655000c0), SPH_C32(0xeb3b5344),
	  SPH_C32(0x563c156a), SPH_C32(0xbee4e062) },
	{ SPH_C32(0xbcc701c0), SPH_C32(0xd78b0187), SPH_C32(0x84937b13),
	  SPH_C32(0x9029fdca), SPH_C32(0x9b9a00e0), SPH_C32(0xa3667ba0),
	  SPH_C32(0xd65b5497), SPH_C32(0x3fa261da) },
	{ SPH_C32(0x734c0000), SPH_C32(0x956fa7d6), SPH_C32(0xa29d1297),
	  SPH_C32(0x6ee56854), SPH_C32(0xc4e80100), SPH_C32(0x1f70960e),
	  SPH_C32(0x2714ca3c), SPH_C32(0x88210c30) },
	{ SPH_C32(0x2f3e0000), SPH_C32(0x5cd56ac4), SPH_C32(0xdb341f6e),
	  SPH_C32(0x0d0c492c), SPH_C32(0x3a220120), SPH_C32(0x572dbeea),
	  SPH_C32(0xa7738bc1), SPH_C32(0x09678d88) },
	{ SPH_C32(0x3e820040), SPH_C32(0xae344ba8), SPH_C32(0x94f8793f),
	  SPH_C32(0x4d865251), SPH_C32(0xbc430100), SPH_C32(0xbfbdcc3a),
	  SPH_C32(0x7a486acb), SPH_C32(0xfa5688fb) },
	{ SPH_C32(0x62f00040), SPH_C32(0x678e86ba), SPH_C32(0xed5174c6),
	  SPH_C32(0x2e6f7329), SPH_C32(0x42890120), SPH_C32(0xf7e0e4de),
	  SPH_C32(0xfa2f2b36), SPH_C32(0x7b100943) },
	{ SPH_C32(0x0be70000), SPH_C32(0x35a2fde2), SPH_C32(0xffc1b260),
	  SPH_C32(0x1c92ec9f), SPH_C32(0xf18d0140), SPH_C32(0x84e62044),
	  SPH_C32(0x4c2d0163), SPH_C32(0xd935b2fe) },
	{ SPH_C32(0x57950000), SPH_C32(0xfc1830f0), SPH_C32(0x8668bf99),
	  SPH_C32(0x7f7bcde7), SPH_C32(0x0f470160), SPH_C32(0xccbb08a0),
	  SPH_C32(0xcc4a409e), SPH_C32(0x58733346) },
	{ SPH_C32(0x46290040), SPH_C32(0x0ef9119c), SPH_C32(0xc9a4d9c8),
	  SPH_C32(0x3ff1d69a), SPH_C32(0x89260140), SPH_C32(0x242b7a70),
	  SPH_C32(0x1171a194), SPH_C32(0xab423635) },
	{ SPH_C32(0x1a5b0040), SPH_C32(0xc743dc8e), SPH_C32(0xb00dd431),
	  SPH_C32(0x5c18f7e2), SPH_C32(0x77ec0160), SPH_C32(0x6c765294),
	  SPH_C32(0x9116e069), SPH_C32(0x2a04b78d) },
	{ SPH_C32(0x289e0080), SPH_C32(0xd060bf3a), SPH_C32(0x60597ec2),
	  SPH_C32(0x9d87da67), SPH_C32(0xfd4e0100), SPH_C32(0x55c7c5e5),
	  SPH_C32(0xf65ac377), SPH_C32(0x3f53b81b) },
	{ SPH_C32(0x74ec0080), SPH_C32(0x19da7228), SPH_C32(0x19f0733b),
	  SPH_C32(0xfe6efb1f), SPH_C32(0x03840120), SPH_C32(0x1d9aed01),
	  SPH_C32(0x763d828a), SPH_C32(0xbe1539a3) },
	{ SPH_C32(0x655000c0), SPH_C32(0xeb3b5344), SPH_C32(0x563c156a),
	  SPH_C32(0xbee4e062), SPH_C32(0x85e50100), SPH_C32(0xf50a9fd1),
	  SPH_C32(0xab066380), SPH_C32(0x4d243cd0) },
	{ SPH_C32(0x392200c0), SPH_C32(0x22819e56), SPH_C32(0x2f951893),
	  SPH_C32(0xdd0dc11a), SPH_C32(0x7b2f0120), SPH_C32(0xbd57b735),
	  SPH_C32(0x2b61227d), SPH_C32(0xcc62bd68) },
	{ SPH_C32(0x50350080), SPH_C32(0x70ade50e), SPH_C32(0x3d05de35),
	  SPH_C32(0xeff05eac), SPH_C32(0xc82b0140), SPH_C32(0xce5173af),
	  SPH_C32(0x9d630828), SPH_C32(0x6e4706d5) },
	{ SPH_C32(0x0c470080), SPH_C32(0xb917281c), SPH_C32(0x44acd3cc),
	  SPH_C32(0x8c197fd4), SPH_C32(0x36e10160), SPH_C32(0x860c5b4b),
	  SPH_C32(0x1d0449d5), SPH_C32(0xef01876d) },
	{ SPH_C32(0x1dfb00c0), SPH_C32(0x4bf60970), SPH_C32(0x0b60b59d),
	  SPH_C32(0xcc9364a9), SPH_C32(0xb0800140), SPH_C32(0x6e9c299b),
	  SPH_C32(0xc03fa8df), SPH_C32(0x1c30821e) },
	{ SPH_C32(0x418900c0), SPH_C32(0x824cc462), SPH_C32(0x72c9b864),
	  SPH_C32(0xaf7a45d1), SPH_C32(0x4e4a0160), SPH_C32(0x26c1017f),
	  SPH_C32(0x4058e922), SPH_C32(0x9d7603a6) },
	{ SPH_C32(0x4aea0000), SPH_C32(0xdfd8f43d), SPH_C32(0x73d31bdc),
	  SPH_C32(0xd997dc7f), SPH_C32(0xa69c0180), SPH_C32(0x10c8dd09),
	  SPH_C32(0x349eaf22), SPH_C32(0xcc310a28) },
	{ SPH_C32(0x16980000), SPH_C32(0x1662392f), SPH_C32(0x0a7a1625),
	  SPH_C32(0xba7efd07), SPH_C32(0x585601a0), SPH_C32(0x5895f5ed),
	  SPH_C32(0xb4f9eedf), SPH_C32(0x4d778b90) },
	{ SPH_C32(0x07240040), SPH_C32(0xe4831843), SPH_C32(0x45b67074),
	  SPH_C32(0xfaf4e67a), SPH_C32(0xde370180), SPH_C32(0xb005873d),
	  SPH_C32(0x69c20fd5), SPH_C32(0xbe468ee3) },
	{ SPH_C32(0x5b560040), SPH_C32(0x2d39d551), SPH_C32(0x3c1f7d8d),
	  SPH_C32(0x991dc702), SPH_C32(0x20fd01a0), SPH_C32(0xf858afd9),
	  SPH_C32(0xe9a54e28), SPH_C32(0x3f000f5b) },
	{ SPH_C32(0x32410000), SPH_C32(0x7f15ae09), SPH_C32(0x2e8fbb2b),
	  SPH_C32(0xabe058b4), SPH_C32(0x93f901c0), SPH_C32(0x8b5e6b43),
	  SPH_C32(0x5fa7647d), SPH_C32(0x9d25b4e6) },
	{ SPH_C32(0x6e330000), SPH_C32(0xb6af631b), SPH_C32(0x5726b6d2),
	  SPH_C32(0xc80979cc), SPH_C32(0x6d3301e0), SPH_C32(0xc30343a7),
	  SPH_C32(0xdfc02580), SPH_C32(0x1c63355e) },
	{ SPH_C32(0x7f8f0040), SPH_C32(0x444e4277), SPH_C32(0x18ead083),
	  SPH_C32(0x888362b1), SPH_C32(0xeb5201c0), SPH_C32(0x2b933177),
	  SPH_C32(0x02fbc48a), SPH_C32(0xef52302d) },
	{ SPH_C32(0x23fd0040), SPH_C32(0x8df48f65), SPH_C32(0x6143dd7a),
	  SPH_C32(0xeb6a43c9), SPH_C32(0x159801e0), SPH_C32(0x63ce1993),
	  SPH_C32(0x829c8577), SPH_C32(0x6e14b195) },
	{ SPH_C32(0x11380080), SPH_C32(0x9ad7ecd1), SPH_C32(0xb1177789),
	  SPH_C32(0x2af56e4c), SPH_C32(0x9f3a0180), SPH_C32(0x5a7f8ee2),
	  SPH_C32(0xe5d0a669), SPH_C32(0x7b43be03) },
	{ SPH_C32(0x4d4a0080), SPH_C32(0x536d21c3), SPH_C32(0xc8be7a70),
	  SPH_C32(0x491c4f34), SPH_C32(0x61f001a0), SPH_C32(0x1222a606),
	  SPH_C32(0x65b7e794), SPH_C32(0xfa053fbb) },
	{ SPH_C32(0x5cf600c0), SPH_C32(0xa18c00af), SPH_C32(0x87721c21),
	  SPH_C32(0x09965449), SPH_C32(0xe7910180), SPH_C32(0xfab2d4d6),
	  SPH_C32(0xb88c069e), SPH_C32(0x09343ac8) },
	{ SPH_C32(0x008400c0), SPH_C32(0x6836cdbd), SPH_C32(0xfedb11d8),
	  SPH_C32(0x6a7f7531), SPH_C32(0x195b01a0), SPH_C32(0xb2effc32),
	  SPH_C32(0x38eb4763), SPH_C32(0x8872bb70) },
	{ SPH_C32(0x69930080), SPH_C32(0x3a1ab6e5), SPH_C32(0xec4bd77e),
	  SPH_C32(0x5882ea87), SPH_C32(0xaa5f01c0), SPH_C32(0xc1e938a8),
	  SPH_C32(0x8ee96d36), SPH_C32(0x2a5700cd) },
	{ SPH_C32(0x35e10080), SPH_C32(0xf3a07bf7), SPH_C32(0x95e2da87),
	  SPH_C32(0x3b6bcbff), SPH_C32(0x549501e0), SPH_C32(0x89b4104c),
	  SPH_C32(0x0e8e2ccb), SPH_C32(0xab118175) },
	{ SPH_C32(0x245d00c0), SPH_C32(0x01415a9b), SPH_C32(0xda2ebcd6),
	  SPH_C32(0x7be1d082), SPH_C32(0xd2f401c0), SPH_C32(0x6124629c),
	  SPH_C32(0xd3b5cdc1), SPH_C32(0x58208406) },
	{ SPH_C32(0x782f00c0), SPH_C32(0xc8fb9789), SPH_C32(0xa387b12f),
	  SPH_C32(0x1808f1fa), SPH_C32(0x2c3e01e0), SPH_C32(0x29794a78),
	  SPH_C32(0x53d28c3c), SPH_C32(0xd96605be) },
	{ SPH_C32(0xc4e80100), SPH_C32(0x1f70960e), SPH_C32(0x2714ca3c),
	  SPH_C32(0x88210c30), SPH_C32(0xb7a40100), SPH_C32(0x8a1f31d8),
	  SPH_C32(0x8589d8ab), SPH_C32(0xe6c46464) },
	{ SPH_C32(0x989a0100), SPH_C32(0xd6ca5b1c), SPH_C32(0x5ebdc7c5),
	  SPH_C32(0xebc82d48), SPH_C32(0x496e0120), SPH_C32(0xc242193c),
	  SPH_C32(0x05ee9956), SPH_C32(0x6782e5dc) },
	{ SPH_C32(0x89260140), SPH_C32(0x242b7a70), SPH_C32(0x1171a194),
	  SPH_C32(0xab423635), SPH_C32(0xcf0f0100), SPH_C32(0x2ad26bec),
	  SPH_C32(0xd8d5785c), SPH_C32(0x94b3e0af) },
	{ SPH_C32(0xd5540140), SPH_C32(0xed91b762), SPH_C32(0x68d8ac6d),
	  SPH_C32(0xc8ab174d), SPH_C32(0x31c50120), SPH_C32(0x628f4308),
	  SPH_C32(0x58b239a1), SPH_C32(0x15f56117) },
	{ SPH_C32(0xbc430100), SPH_C32(0xbfbdcc3a), SPH_C32(0x7a486acb),
	  SPH_C32(0xfa5688fb), SPH_C32(0x82c10140), SPH_C32(0x11898792),
	  SPH_C32(0xeeb013f4), SPH_C32(0xb7d0daaa) },
	{ SPH_C32(0xe0310100), SPH_C32(0x76070128), SPH_C32(0x03e16732),
	  SPH_C32(0x99bfa983), SPH_C32(0x7c0b0160), SPH_C32(0x59d4af76),
	  SPH_C32(0x6ed75209), SPH_C32(0x36965b12) },
	{ SPH_C32(0xf18d0140), SPH_C32(0x84e62044), SPH_C32(0x4c2d0163),
	  SPH_C32(0xd935b2fe), SPH_C32(0xfa6a0140), SPH_C32(0xb144dda6),
	  SPH_C32(0xb3ecb303), SPH_C32(0xc5a75e61) },
	{ SPH_C32(0xadff0140), SPH_C32(0x4d5ced56), SPH_C32(0x35840c9a),
	  SPH_C32(0xbadc9386), SPH_C32(0x04a00160), SPH_C32(0xf919f542),
	  SPH_C32(0x338bf2fe), SPH_C32(0x44e1dfd9) },
	{ SPH_C32(0x9f3a0180), SPH_C32(0x5a7f8ee2), SPH_C32(0xe5d0a669),
	  SPH_C32(0x7b43be03), SPH_C32(0x8e020100), SPH_C32(0xc0a86233),
	  SPH_C32(0x54c7d1e0), SPH_C32(0x51b6d04f) },
	{ SPH_C32(0xc3480180), SPH_C32(0x93c543f0), SPH_C32(0x9c79ab90),
	  SPH_C32(0x18aa9f7b), SPH_C32(0x70c80120), SPH_C32(0x88f54ad7),
	  SPH_C32(0xd4a0901d), SPH_C32(0xd0f051f7) },
	{ SPH_C32(0xd2f401c0), SPH_C32(0x6124629c), SPH_C32(0xd3b5cdc1),
	  SPH_C32(0x58208406), SPH_C32(0xf6a90100), SPH_C32(0x60653807),
	  SPH_C32(0x099b7117), SPH_C32(0x23c15484) },
	{ SPH_C32(0x8e8601c0), SPH_C32(0xa89eaf8e), SPH_C32(0xaa1cc038),
	  SPH_C32(0x3bc9a57e), SPH_C32(0x08630120), SPH_C32(0x283810e3),
	  SPH_C32(0x89fc30ea), SPH_C32(0xa287d53c) },
	{ SPH_C32(0xe7910180), SPH_C32(0xfab2d4d6), SPH_C32(0xb88c069e),
	  SPH_C32(0x09343ac8), SPH_C32(0xbb670140), SPH_C32(0x5b3ed479),
	  SPH_C32(0x3ffe1abf), SPH_C32(0x00a26e81) },
	{ SPH_C32(0xbbe30180), SPH_C32(0x330819c4), SPH_C32(0xc1250b67),
	  SPH_C32(0x6add1bb0), SPH_C32(0x45ad0160), SPH_C32(0x1363fc9d),
	  SPH_C32(0xbf995b42), SPH_C32(0x81e4ef39) },
	{ SPH_C32(0xaa5f01c0), SPH_C32(0xc1e938a8), SPH_C32(0x8ee96d36),
	  SPH_C32(0x2a5700cd), SPH_C32(0xc3cc0140), SPH_C32(0xfbf38e4d),
	  SPH_C32(0x62a2ba48), SPH_C32(0x72d5ea4a) },
	{ SPH_C32(0xf62d01c0), SPH_C32(0x0853f5ba), SPH_C32(0xf74060cf),
	  SPH_C32(0x49be21b5), SPH_C32(0x3d060160), SPH_C32(0xb3aea6a9),
	  SPH_C32(0xe2c5fbb5), SPH_C32(0xf3936bf2) },
	{ SPH_C32(0xfd4e0100), SPH_C32(0x55c7c5e5), SPH_C32(0xf65ac377),
	  SPH_C32(0x3f53b81b), SPH_C32(0xd5d00180), SPH_C32(0x85a77adf),
	  SPH_C32(0x9603bdb5), SPH_C32(0xa2d4627c) },
	{ SPH_C32(0xa13c0100), SPH_C32(0x9c7d08f7), SPH_C32(0x8ff3ce8e),
	  SPH_C32(0x5cba9963), SPH_C32(0x2b1a01a0), SPH_C32(0xcdfa523b),
	  SPH_C32(0x1664fc48), SPH_C32(0x2392e3c4) },
	{ SPH_C32(0xb0800140), SPH_C32(0x6e9c299b), SPH_C32(0xc03fa8df),
	  SPH_C32(0x1c30821e), SPH_C32(0xad7b0180), SPH_C32(0x256a20eb),
	  SPH_C32(0xcb5f1d42), SPH_C32(0xd0a3e6b7) },
	{ SPH_C32(0xecf20140), SPH_C32(0xa726e489), SPH_C32(0xb996a526),
	  SPH_C32(0x7fd9a366), SPH_C32(0x53b101a0), SPH_C32(0x6d37080f),
	  SPH_C32(0x4b385cbf), SPH_C32(0x51e5670f) },
	{ SPH_C32(0x85e50100), SPH_C32(0xf50a9fd1), SPH_C32(0xab066380),
	  SPH_C32(0x4d243cd0), SPH_C32(0xe0b501c0), SPH_C32(0x1e31cc95),
	  SPH_C32(0xfd3a76ea), SPH_C32(0xf3c0dcb2) },
	{ SPH_C32(0xd9970100), SPH_C32(0x3cb052c3), SPH_C32(0xd2af6e79),
	  SPH_C32(0x2ecd1da8), SPH_C32(0x1e7f01e0), SPH_C32(0x566ce471),
	  SPH_C32(0x7d5d3717), SPH_C32(0x72865d0a) },
	{ SPH_C32(0xc82b0140), SPH_C32(0xce5173af), SPH_C32(0x9d630828),
	  SPH_C32(0x6e4706d5), SPH_C32(0x981e01c0), SPH_C32(0xbefc96a1),
	  SPH_C32(0xa066d61d), SPH_C32(0x81b75879) },
	{ SPH_C32(0x94590140), SPH_C32(0x07ebbebd), SPH_C32(0xe4ca05d1),
	  SPH_C32(0x0dae27ad), SPH_C32(0x66d401e0), SPH_C32(0xf6a1be45),
	  SPH_C32(0x200197e0), SPH_C32(0x00f1d9c1) },
	{ SPH_C32(0xa69c0180), SPH_C32(0x10c8dd09), SPH_C32(0x349eaf22),
	  SPH_C32(0xcc310a28), SPH_C32(0xec760180), SPH_C32(0xcf102934),
	  SPH_C32(0x474db4fe), SPH_C32(0x15a6d657) },
	{ SPH_C32(0xfaee0180), SPH_C32(0xd972101b), SPH_C32(0x4d37a2db),
	  SPH_C32(0xafd82b50), SPH_C32(0x12bc01a0), SPH_C32(0x874d01d0),
	  SPH_C32(0xc72af503), SPH_C32(0x94e057ef) },
	{ SPH_C32(0xeb5201c0), SPH_C32(0x2b933177), SPH_C32(0x02fbc48a),
	  SPH_C32(0xef52302d), SPH_C32(0x94dd0180), SPH_C32(0x6fdd7300),
	  SPH_C32(0x1a111409), SPH_C32(0x67d1529c) },
	{ SPH_C32(0xb72001c0), SPH_C32(0xe229fc65), SPH_C32(0x7b52c973),
	  SPH_C32(0x8cbb1155), SPH_C32(0x6a1701a0), SPH_C32(0x27805be4),
	  SPH_C32(0x9a7655f4), SPH_C32(0xe697d324) },
	{ SPH_C32(0xde370180), SPH_C32(0xb005873d), SPH_C32(0x69c20fd5),
	  SPH_C32(0xbe468ee3), SPH_C32(0xd91301c0), SPH_C32(0x54869f7e),
	  SPH_C32(0x2c747fa1), SPH_C32(0x44b26899) },
	{ SPH_C32(0x82450180), SPH_C32(0x79bf4a2f), SPH_C32(0x106b022c),
	  SPH_C32(0xddafaf9b), SPH_C32(0x27d901e0), SPH_C32(0x1cdbb79a),
	  SPH_C32(0xac133e5c), SPH_C32(0xc5f4e921) },
	{ SPH_C32(0x93f901c0), SPH_C32(0x8b5e6b43), SPH_C32(0x5fa7647d),
	  SPH_C32(0x9d25b4e6), SPH_C32(0xa1b801c0), SPH_C32(0xf44bc54a),
	  SPH_C32(0x7128df56), SPH_C32(0x36c5ec52) },
	{ SPH_C32(0xcf8b01c0), SPH_C32(0x42e4a651), SPH_C32(0x260e6984),
	  SPH_C32(0xfecc959e), SPH_C32(0x5f7201e0), SPH_C32(0xbc16edae),
	  SPH_C32(0xf14f9eab), SPH_C32(0xb7836dea) }
};

static const sph_u32 T256_21[128][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xd0080004), SPH_C32(0x8c768f77), SPH_C32(0x9dc5b050),
	  SPH_C32(0xaf4a29da), SPH_C32(0x6ba90000), SPH_C32(0x40ebf9aa),
	  SPH_C32(0x98321c3d), SPH_C32(0x76acc733) },
	{ SPH_C32(0x6ba90000), SPH_C32(0x40ebf9aa), SPH_C32(0x98321c3d),
	  SPH_C32(0x76acc733), SPH_C32(0xbba10004), SPH_C32(0xcc9d76dd),
	  SPH_C32(0x05f7ac6d), SPH_C32(0xd9e6eee9) },
	{ SPH_C32(0xbba10004), SPH_C32(0xcc9d76dd), SPH_C32(0x05f7ac6d),
	  SPH_C32(0xd9e6eee9), SPH_C32(0xd0080004), SPH_C32(0x8c768f77),
	  SPH_C32(0x9dc5b050), SPH_C32(0xaf4a29da) },
	{ SPH_C32(0xa8ae0008), SPH_C32(0x2079397d), SPH_C32(0xfe739301),
	  SPH_C32(0xb8a92831), SPH_C32(0x171c0000), SPH_C32(0xb26e3344),
	  SPH_C32(0x9e6a837e), SPH_C32(0x58f8485f) },
	{ SPH_C32(0x78a6000c), SPH_C32(0xac0fb60a), SPH_C32(0x63b62351),
	  SPH_C32(0x17e301eb), SPH_C32(0x7cb50000), SPH_C32(0xf285caee),
	  SPH_C32(0x06589f43), SPH_C32(0x2e548f6c) },
	{ SPH_C32(0xc3070008), SPH_C32(0x6092c0d7), SPH_C32(0x66418f3c),
	  SPH_C32(0xce05ef02), SPH_C32(0xacbd0004), SPH_C32(0x7ef34599),
	  SPH_C32(0x9b9d2f13), SPH_C32(0x811ea6b6) },
	{ SPH_C32(0x130f000c), SPH_C32(0xece44fa0), SPH_C32(0xfb843f6c),
	  SPH_C32(0x614fc6d8), SPH_C32(0xc7140004), SPH_C32(0x3e18bc33),
	  SPH_C32(0x03af332e), SPH_C32(0xf7b26185) },
	{ SPH_C32(0x171c0000), SPH_C32(0xb26e3344), SPH_C32(0x9e6a837e),
	  SPH_C32(0x58f8485f), SPH_C32(0xbfb20008), SPH_C32(0x92170a39),
	  SPH_C32(0x6019107f), SPH_C32(0xe051606e) },
	{ SPH_C32(0xc7140004), SPH_C32(0x3e18bc33), SPH_C32(0x03af332e),
	  SPH_C32(0xf7b26185), SPH_C32(0xd41b0008), SPH_C32(0xd2fcf393),
	  SPH_C32(0xf82b0c42), SPH_C32(0x96fda75d) },
	{ SPH_C32(0x7cb50000), SPH_C32(0xf285caee), SPH_C32(0x06589f43),
	  SPH_C32(0x2e548f6c), SPH_C32(0x0413000c), SPH_C32(0x5e8a7ce4),
	  SPH_C32(0x65eebc12), SPH_C32(0x39b78e87) },
	{ SPH_C32(0xacbd0004), SPH_C32(0x7ef34599), SPH_C32(0x9b9d2f13),
	  SPH_C32(0x811ea6b6), SPH_C32(0x6fba000c), SPH_C32(0x1e61854e),
	  SPH_C32(0xfddca02f), SPH_C32(0x4f1b49b4) },
	{ SPH_C32(0xbfb20008), SPH_C32(0x92170a39), SPH_C32(0x6019107f),
	  SPH_C32(0xe051606e), SPH_C32(0xa8ae0008), SPH_C32(0x2079397d),
	  SPH_C32(0xfe739301), SPH_C32(0xb8a92831) },
	{ SPH_C32(0x6fba000c), SPH_C32(0x1e61854e), SPH_C32(0xfddca02f),
	  SPH_C32(0x4f1b49b4), SPH_C32(0xc3070008), SPH_C32(0x6092c0d7),
	  SPH_C32(0x66418f3c), SPH_C32(0xce05ef02) },
	{ SPH_C32(0xd41b0008), SPH_C32(0xd2fcf393), SPH_C32(0xf82b0c42),
	  SPH_C32(0x96fda75d), SPH_C32(0x130f000c), SPH_C32(0xece44fa0),
	  SPH_C32(0xfb843f6c), SPH_C32(0x614fc6d8) },
	{ SPH_C32(0x0413000c), SPH_C32(0x5e8a7ce4), SPH_C32(0x65eebc12),
	  SPH_C32(0x39b78e87), SPH_C32(0x78a6000c), SPH_C32(0xac0fb60a),
	  SPH_C32(0x63b62351), SPH_C32(0x17e301eb) },
	{ SPH_C32(0x515c0010), SPH_C32(0x40f372fb), SPH_C32(0xfce72602),
	  SPH_C32(0x71575061), SPH_C32(0x2e390000), SPH_C32(0x64dd6689),
	  SPH_C32(0x3cd406fc), SPH_C32(0xb1f490bc) },
	{ SPH_C32(0x81540014), SPH_C32(0xcc85fd8c), SPH_C32(0x61229652),
	  SPH_C32(0xde1d79bb), SPH_C32(0x45900000), SPH_C32(0x24369f23),
	  SPH_C32(0xa4e61ac1), SPH_C32(0xc758578f) },
	{ SPH_C32(0x3af50010), SPH_C32(0x00188b51), SPH_C32(0x64d53a3f),
	  SPH_C32(0x07fb9752), SPH_C32(0x95980004), SPH_C32(0xa8401054),
	  SPH_C32(0x3923aa91), SPH_C32(0x68127e55) },
	{ SPH_C32(0xeafd0014), SPH_C32(0x8c6e0426), SPH_C32(0xf9108a6f),
	  SPH_C32(0xa8b1be88), SPH_C32(0xfe310004), SPH_C32(0xe8abe9fe),
	  SPH_C32(0xa111b6ac), SPH_C32(0x1ebeb966) },
	{ SPH_C32(0xf9f20018), SPH_C32(0x608a4b86), SPH_C32(0x0294b503),
	  SPH_C32(0xc9fe7850), SPH_C32(0x39250000), SPH_C32(0xd6b355cd),
	  SPH_C32(0xa2be8582), SPH_C32(0xe90cd8e3) },
	{ SPH_C32(0x29fa001c), SPH_C32(0xecfcc4f1), SPH_C32(0x9f510553),
	  SPH_C32(0x66b4518a), SPH_C32(0x528c0000), SPH_C32(0x9658ac67),
	  SPH_C32(0x3a8c99bf), SPH_C32(0x9fa01fd0) },
	{ SPH_C32(0x925b0018), SPH_C32(0x2061b22c), SPH_C32(0x9aa6a93e),
	  SPH_C32(0xbf52bf63), SPH_C32(0x82840004), SPH_C32(0x1a2e2310),
	  SPH_C32(0xa74929ef), SPH_C32(0x30ea360a) },
	{ SPH_C32(0x4253001c), SPH_C32(0xac173d5b), SPH_C32(0x0763196e),
	  SPH_C32(0x101896b9), SPH_C32(0xe92d0004), SPH_C32(0x5ac5daba),
	  SPH_C32(0x3f7b35d2), SPH_C32(0x4646f139) },
	{ SPH_C32(0x46400010), SPH_C32(0xf29d41bf), SPH_C32(0x628da57c),
	  SPH_C32(0x29af183e), SPH_C32(0x918b0008), SPH_C32(0xf6ca6cb0),
	  SPH_C32(0x5ccd1683), SPH_C32(0x51a5f0d2) },
	{ SPH_C32(0x96480014), SPH_C32(0x7eebcec8), SPH_C32(0xff48152c),
	  SPH_C32(0x86e531e4), SPH_C32(0xfa220008), SPH_C32(0xb621951a),
	  SPH_C32(0xc4ff0abe), SPH_C32(0x270937e1) },
	{ SPH_C32(0x2de90010), SPH_C32(0xb276b815), SPH_C32(0xfabfb941),
	  SPH_C32(0x5f03df0d), SPH_C32(0x2a2a000c), SPH_C32(0x3a571a6d),
	  SPH_C32(0x593abaee), SPH_C32(0x88431e3b) },
	{ SPH_C32(0xfde10014), SPH_C32(0x3e003762), SPH_C32(0x677a0911),
	  SPH_C32(0xf049f6d7), SPH_C32(0x4183000c), SPH_C32(0x7abce3c7),
	  SPH_C32(0xc108a6d3), SPH_C32(0xfeefd908) },
	{ SPH_C32(0xeeee0018), SPH_C32(0xd2e478c2), SPH_C32(0x9cfe367d),
	  SPH_C32(0x9106300f), SPH_C32(0x86970008), SPH_C32(0x44a45ff4),
	  SPH_C32(0xc2a795fd), SPH_C32(0x095db88d) },
	{ SPH_C32(0x3ee6001c), SPH_C32(0x5e92f7b5), SPH_C32(0x013b862d),
	  SPH_C32(0x3e4c19d5), SPH_C32(0xed3e0008), SPH_C32(0x044fa65e),
	  SPH_C32(0x5a9589c0), SPH_C32(0x7ff17fbe) },
	{ SPH_C32(0x85470018), SPH_C32(0x920f8168), SPH_C32(0x04cc2a40),
	  SPH_C32(0xe7aaf73c), SPH_C32(0x3d36000c), SPH_C32(0x88392929),
	  SPH_C32(0xc7503990), SPH_C32(0xd0bb5664) },
	{ SPH_C32(0x554f001c), SPH_C32(0x1e790e1f), SPH_C32(0x99099a10),
	  SPH_C32(0x48e0dee6), SPH_C32(0x569f000c), SPH_C32(0xc8d2d083),
	  SPH_C32(0x5f6225ad), SPH_C32(0xa6179157) },
	{ SPH_C32(0x2e390000), SPH_C32(0x64dd6689), SPH_C32(0x3cd406fc),
	  SPH_C32(0xb1f490bc), SPH_C32(0x7f650010), SPH_C32(0x242e1472),
	  SPH_C32(0xc03320fe), SPH_C32(0xc0a3c0dd) },
	{ SPH_C32(0xfe310004), SPH_C32(0xe8abe9fe), SPH_C32(0xa111b6ac),
	  SPH_C32(0x1ebeb966), SPH_C32(0x14cc0010), SPH_C32(0x64c5edd8),
	  SPH_C32(0x58013cc3), SPH_C32(0xb60f07ee) },
	{ SPH_C32(0x45900000), SPH_C32(0x24369f23), SPH_C32(0xa4e61ac1),
	  SPH_C32(0xc758578f), SPH_C32(0xc4c40014), SPH_C32(0xe8b362af),
	  SPH_C32(0xc5c48c93), SPH_C32(0x19452e34) },
	{ SPH_C32(0x95980004), SPH_C32(0xa8401054), SPH_C32(0x3923aa91),
	  SPH_C32(0x68127e55), SPH_C32(0xaf6d0014), SPH_C32(0xa8589b05),
	  SPH_C32(0x5df690ae), SPH_C32(0x6fe9e907) },
	{ SPH_C32(0x86970008), SPH_C32(0x44a45ff4), SPH_C32(0xc2a795fd),
	  SPH_C32(0x095db88d), SPH_C32(0x68790010), SPH_C32(0x96402736),
	  SPH_C32(0x5e59a380), SPH_C32(0x985b8882) },
	{ SPH_C32(0x569f000c), SPH_C32(0xc8d2d083), SPH_C32(0x5f6225ad),
	  SPH_C32(0xa6179157), SPH_C32(0x03d00010), SPH_C32(0xd6abde9c),
	  SPH_C32(0xc66bbfbd), SPH_C32(0xeef74fb1) },
	{ SPH_C32(0xed3e0008), SPH_C32(0x044fa65e), SPH_C32(0x5a9589c0),
	  SPH_C32(0x7ff17fbe), SPH_C32(0xd3d80014), SPH_C32(0x5add51eb),
	  SPH_C32(0x5bae0fed), SPH_C32(0x41bd666b) },
	{ SPH_C32(0x3d36000c), SPH_C32(0x88392929), SPH_C32(0xc7503990),
	  SPH_C32(0xd0bb5664), SPH_C32(0xb8710014), SPH_C32(0x1a36a841),
	  SPH_C32(0xc39c13d0), SPH_C32(0x3711a158) },
	{ SPH_C32(0x39250000), SPH_C32(0xd6b355cd), SPH_C32(0xa2be8582),
	  SPH_C32(0xe90cd8e3), SPH_C32(0xc0d70018), SPH_C32(0xb6391e4b),
	  SPH_C32(0xa02a3081), SPH_C32(0x20f2a0b3) },
	{ SPH_C32(0xe92d0004), SPH_C32(0x5ac5daba), SPH_C32(0x3f7b35d2),
	  SPH_C32(0x4646f139), SPH_C32(0xab7e0018), SPH_C32(0xf6d2e7e1),
	  SPH_C32(0x38182cbc), SPH_C32(0x565e6780) },
	{ SPH_C32(0x528c0000), SPH_C32(0x9658ac67), SPH_C32(0x3a8c99bf),
	  SPH_C32(0x9fa01fd0), SPH_C32(0x7b76001c), SPH_C32(0x7aa46896),
	  SPH_C32(0xa5dd9cec), SPH_C32(0xf9144e5a) },
	{ SPH_C32(0x82840004), SPH_C32(0x1a2e2310), SPH_C32(0xa74929ef),
	  SPH_C32(0x30ea360a), SPH_C32(0x10df001c), SPH_C32(0x3a4f913c),
	  SPH_C32(0x3def80d1), SPH_C32(0x8fb88969) },
	{ SPH_C32(0x918b0008), SPH_C32(0xf6ca6cb0), SPH_C32(0x5ccd1683),
	  SPH_C32(0x51a5f0d2), SPH_C32(0xd7cb0018), SPH_C32(0x04572d0f),
	  SPH_C32(0x3e40b3ff), SPH_C32(0x780ae8ec) },
	{ SPH_C32(0x4183000c), SPH_C32(0x7abce3c7), SPH_C32(0xc108a6d3),
	  SPH_C32(0xfeefd908), SPH_C32(0xbc620018), SPH_C32(0x44bcd4a5),
	  SPH_C32(0xa672afc2), SPH_C32(0x0ea62fdf) },
	{ SPH_C32(0xfa220008), SPH_C32(0xb621951a), SPH_C32(0xc4ff0abe),
	  SPH_C32(0x270937e1), SPH_C32(0x6c6a001c), SPH_C32(0xc8ca5bd2),
	  SPH_C32(0x3bb71f92), SPH_C32(0xa1ec0605) },
	{ SPH_C32(0x2a2a000c), SPH_C32(0x3a571a6d), SPH_C32(0x593abaee),
	  SPH_C32(0x88431e3b), SPH_C32(0x07c3001c), SPH_C32(0x8821a278),
	  SPH_C32(0xa38503af), SPH_C32(0xd740c136) },
	{ SPH_C32(0x7f650010), SPH_C32(0x242e1472), SPH_C32(0xc03320fe),
	  SPH_C32(0xc0a3c0dd), SPH_C32(0x515c0010), SPH_C32(0x40f372fb),
	  SPH_C32(0xfce72602), SPH_C32(0x71575061) },
	{ SPH_C32(0xaf6d0014), SPH_C32(0xa8589b05), SPH_C32(0x5df690ae),
	  SPH_C32(0x6fe9e907), SPH_C32(0x3af50010), SPH_C32(0x00188b51),
	  SPH_C32(0x64d53a3f), SPH_C32(0x07fb9752) },
	{ SPH_C32(0x14cc0010), SPH_C32(0x64c5edd8), SPH_C32(0x58013cc3),
	  SPH_C32(0xb60f07ee), SPH_C32(0xeafd0014), SPH_C32(0x8c6e0426),
	  SPH_C32(0xf9108a6f), SPH_C32(0xa8b1be88) },
	{ SPH_C32(0xc4c40014), SPH_C32(0xe8b362af), SPH_C32(0xc5c48c93),
	  SPH_C32(0x19452e34), SPH_C32(0x81540014), SPH_C32(0xcc85fd8c),
	  SPH_C32(0x61229652), SPH_C32(0xde1d79bb) },
	{ SPH_C32(0xd7cb0018), SPH_C32(0x04572d0f), SPH_C32(0x3e40b3ff),
	  SPH_C32(0x780ae8ec), SPH_C32(0x46400010), SPH_C32(0xf29d41bf),
	  SPH_C32(0x628da57c), SPH_C32(0x29af183e) },
	{ SPH_C32(0x07c3001c), SPH_C32(0x8821a278), SPH_C32(0xa38503af),
	  SPH_C32(0xd740c136), SPH_C32(0x2de90010), SPH_C32(0xb276b815),
	  SPH_C32(0xfabfb941), SPH_C32(0x5f03df0d) },
	{ SPH_C32(0xbc620018), SPH_C32(0x44bcd4a5), SPH_C32(0xa672afc2),
	  SPH_C32(0x0ea62fdf), SPH_C32(0xfde10014), SPH_C32(0x3e003762),
	  SPH_C32(0x677a0911), SPH_C32(0xf049f6d7) },
	{ SPH_C32(0x6c6a001c), SPH_C32(0xc8ca5bd2), SPH_C32(0x3bb71f92),
	  SPH_C32(0xa1ec0605), SPH_C32(0x96480014), SPH_C32(0x7eebcec8),
	  SPH_C32(0xff48152c), SPH_C32(0x86e531e4) },
	{ SPH_C32(0x68790010), SPH_C32(0x96402736), SPH_C32(0x5e59a380),
	  SPH_C32(0x985b8882), SPH_C32(0xeeee0018), SPH_C32(0xd2e478c2),
	  SPH_C32(0x9cfe367d), SPH_C32(0x9106300f) },
	{ SPH_C32(0xb8710014), SPH_C32(0x1a36a841), SPH_C32(0xc39c13d0),
	  SPH_C32(0x3711a158), SPH_C32(0x85470018), SPH_C32(0x920f8168),
	  SPH_C32(0x04cc2a40), SPH_C32(0xe7aaf73c) },
	{ SPH_C32(0x03d00010), SPH_C32(0xd6abde9c), SPH_C32(0xc66bbfbd),
	  SPH_C32(0xeef74fb1), SPH_C32(0x554f001c), SPH_C32(0x1e790e1f),
	  SPH_C32(0x99099a10), SPH_C32(0x48e0dee6) },
	{ SPH_C32(0xd3d80014), SPH_C32(0x5add51eb), SPH_C32(0x5bae0fed),
	  SPH_C32(0x41bd666b), SPH_C32(0x3ee6001c), SPH_C32(0x5e92f7b5),
	  SPH_C32(0x013b862d), SPH_C32(0x3e4c19d5) },
	{ SPH_C32(0xc0d70018), SPH_C32(0xb6391e4b), SPH_C32(0xa02a3081),
	  SPH_C32(0x20f2a0b3), SPH_C32(0xf9f20018), SPH_C32(0x608a4b86),
	  SPH_C32(0x0294b503), SPH_C32(0xc9fe7850) },
	{ SPH_C32(0x10df001c), SPH_C32(0x3a4f913c), SPH_C32(0x3def80d1),
	  SPH_C32(0x8fb88969), SPH_C32(0x925b0018), SPH_C32(0x2061b22c),
	  SPH_C32(0x9aa6a93e), SPH_C32(0xbf52bf63) },
	{ SPH_C32(0xab7e0018), SPH_C32(0xf6d2e7e1), SPH_C32(0x38182cbc),
	  SPH_C32(0x565e6780), SPH_C32(0x4253001c), SPH_C32(0xac173d5b),
	  SPH_C32(0x0763196e), SPH_C32(0x101896b9) },
	{ SPH_C32(0x7b76001c), SPH_C32(0x7aa46896), SPH_C32(0xa5dd9cec),
	  SPH_C32(0xf9144e5a), SPH_C32(0x29fa001c), SPH_C32(0xecfcc4f1),
	  SPH_C32(0x9f510553), SPH_C32(0x66b4518a) },
	{ SPH_C32(0xa2b80020), SPH_C32(0x81e7e5f6), SPH_C32(0xf9ce4c04),
	  SPH_C32(0xe2afa0c0), SPH_C32(0x5c720000), SPH_C32(0xc9bacd12),
	  SPH_C32(0x79a90df9), SPH_C32(0x63e92178) },
	{ SPH_C32(0x72b00024), SPH_C32(0x0d916a81), SPH_C32(0x640bfc54),
	  SPH_C32(0x4de5891a), SPH_C32(0x37db0000), SPH_C32(0x895134b8),
	  SPH_C32(0xe19b11c4), SPH_C32(0x1545e64b) },
	{ SPH_C32(0xc9110020), SPH_C32(0xc10c1c5c), SPH_C32(0x61fc5039),
	  SPH_C32(0x940367f3), SPH_C32(0xe7d30004), SPH_C32(0x0527bbcf),
	  SPH_C32(0x7c5ea194), SPH_C32(0xba0fcf91) },
	{ SPH_C32(0x19190024), SPH_C32(0x4d7a932b), SPH_C32(0xfc39e069),
	  SPH_C32(0x3b494e29), SPH_C32(0x8c7a0004), SPH_C32(0x45cc4265),
	  SPH_C32(0xe46cbda9), SPH_C32(0xcca308a2) },
	{ SPH_C32(0x0a160028), SPH_C32(0xa19edc8b), SPH_C32(0x07bddf05),
	  SPH_C32(0x5a0688f1), SPH_C32(0x4b6e0000), SPH_C32(0x7bd4fe56),
	  SPH_C32(0xe7c38e87), SPH_C32(0x3b116927) },
	{ SPH_C32(0xda1e002c), SPH_C32(0x2de853fc), SPH_C32(0x9a786f55),
	  SPH_C32(0xf54ca12b), SPH_C32(0x20c70000), SPH_C32(0x3b3f07fc),
	  SPH_C32(0x7ff192ba), SPH_C32(0x4dbdae14) },
	{ SPH_C32(0x61bf0028), SPH_C32(0xe1752521), SPH_C32(0x9f8fc338),
	  SPH_C32(0x2caa4fc2), SPH_C32(0xf0cf0004), SPH_C32(0xb749888b),
	  SPH_C32(0xe23422ea), SPH_C32(0xe2f787ce) },
	{ SPH_C32(0xb1b7002c), SPH_C32(0x6d03aa56), SPH_C32(0x024a7368),
	  SPH_C32(0x83e06618), SPH_C32(0x9b660004), SPH_C32(0xf7a27121),
	  SPH_C32(0x7a063ed7), SPH_C32(0x945b40fd) },
	{ SPH_C32(0xb5a40020), SPH_C32(0x3389d6b2), SPH_C32(0x67a4cf7a),
	  SPH_C32(0xba57e89f), SPH_C32(0xe3c00008), SPH_C32(0x5badc72b),
	  SPH_C32(0x19b01d86), SPH_C32(0x83b84116) },
	{ SPH_C32(0x65ac0024), SPH_C32(0xbfff59c5), SPH_C32(0xfa617f2a),
	  SPH_C32(0x151dc145), SPH_C32(0x88690008), SPH_C32(0x1b463e81),
	  SPH_C32(0x818201bb), SPH_C32(0xf5148625) },
	{ SPH_C32(0xde0d0020), SPH_C32(0x73622f18), SPH_C32(0xff96d347),
	  SPH_C32(0xccfb2fac), SPH_C32(0x5861000c), SPH_C32(0x9730b1f6),
	  SPH_C32(0x1c47b1eb), SPH_C32(0x5a5eafff) },
	{ SPH_C32(0x0e050024), SPH_C32(0xff14a06f), SPH_C32(0x62536317),
	  SPH_C32(0x63b10676), SPH_C32(0x33c8000c), SPH_C32(0xd7db485c),
	  SPH_C32(0x8475add6), SPH_C32(0x2cf268cc) },
	{ SPH_C32(0x1d0a0028), SPH_C32(0x13f0efcf), SPH_C32(0x99d75c7b),
	  SPH_C32(0x02fec0ae), SPH_C32(0xf4dc0008), SPH_C32(0xe9c3f46f),
	  SPH_C32(0x87da9ef8), SPH_C32(0xdb400949) },
	{ SPH_C32(0xcd02002c), SPH_C32(0x9f8660b8), SPH_C32(0x0412ec2b),
	  SPH_C32(0xadb4e974), SPH_C32(0x9f750008), SPH_C32(0xa9280dc5),
	  SPH_C32(0x1fe882c5), SPH_C32(0xadecce7a) },
	{ SPH_C32(0x76a30028), SPH_C32(0x531b1665), SPH_C32(0x01e54046),
	  SPH_C32(0x7452079d), SPH_C32(0x4f7d000c), SPH_C32(0x255e82b2),
	  SPH_C32(0x822d3295), SPH_C32(0x02a6e7a0) },
	{ SPH_C32(0xa6ab002c), SPH_C32(0xdf6d9912), SPH_C32(0x9c20f016),
	  SPH_C32(0xdb182e47), SPH_C32(0x24d4000c), SPH_C32(0x65b57b18),
	  SPH_C32(0x1a1f2ea8), SPH_C32(0x740a2093) },
	{ SPH_C32(0xf3e40030), SPH_C32(0xc114970d), SPH_C32(0x05296a06),
	  SPH_C32(0x93f8f0a1), SPH_C32(0x724b0000), SPH_C32(0xad67ab9b),
	  SPH_C32(0x457d0b05), SPH_C32(0xd21db1c4) },
	{ SPH_C32(0x23ec0034), SPH_C32(0x4d62187a), SPH_C32(0x98ecda56),
	  SPH_C32(0x3cb2d97b), SPH_C32(0x19e20000), SPH_C32(0xed8c5231),
	  SPH_C32(0xdd4f1738), SPH_C32(0xa4b176f7) },
	{ SPH_C32(0x984d0030), SPH_C32(0x81ff6ea7), SPH_C32(0x9d1b763b),
	  SPH_C32(0xe5543792), SPH_C32(0xc9ea0004), SPH_C32(0x61fadd46),
	  SPH_C32(0x408aa768), SPH_C32(0x0bfb5f2d) },
	{ SPH_C32(0x48450034), SPH_C32(0x0d89e1d0), SPH_C32(0x00dec66b),
	  SPH_C32(0x4a1e1e48), SPH_C32(0xa2430004), SPH_C32(0x211124ec),
	  SPH_C32(0xd8b8bb55), SPH_C32(0x7d57981e) },
	{ SPH_C32(0x5b4a0038), SPH_C32(0xe16dae70), SPH_C32(0xfb5af907),
	  SPH_C32(0x2b51d890), SPH_C32(0x65570000), SPH_C32(0x1f0998df),
	  SPH_C32(0xdb17887b), SPH_C32(0x8ae5f99b) },
	{ SPH_C32(0x8b42003c), SPH_C32(0x6d1b2107), SPH_C32(0x669f4957),
	  SPH_C32(0x841bf14a), SPH_C32(0x0efe0000), SPH_C32(0x5fe26175),
	  SPH_C32(0x43259446), SPH_C32(0xfc493ea8) },
	{ SPH_C32(0x30e30038), SPH_C32(0xa18657da), SPH_C32(0x6368e53a),
	  SPH_C32(0x5dfd1fa3), SPH_C32(0xdef60004), SPH_C32(0xd394ee02),
	  SPH_C32(0xdee02416), SPH_C32(0x53031772) },
	{ SPH_C32(0xe0eb003c), SPH_C32(0x2df0d8ad), SPH_C32(0xfead556a),
	  SPH_C32(0xf2b73679), SPH_C32(0xb55f0004), SPH_C32(0x937f17a8),
	  SPH_C32(0x46d2382b), SPH_C32(0x25afd041) },
	{ SPH_C32(0xe4f80030), SPH_C32(0x737aa449), SPH_C32(0x9b43e978),
	  SPH_C32(0xcb00b8fe), SPH_C32(0xcdf90008), SPH_C32(0x3f70a1a2),
	  SPH_C32(0x25641b7a), SPH_C32(0x324cd1aa) },
	{ SPH_C32(0x34f00034), SPH_C32(0xff0c2b3e), SPH_C32(0x06865928),
	  SPH_C32(0x644a9124), SPH_C32(0xa6500008), SPH_C32(0x7f9b5808),
	  SPH_C32(0xbd560747), SPH_C32(0x44e01699) },
	{ SPH_C32(0x8f510030), SPH_C32(0x33915de3), SPH_C32(0x0371f545),
	  SPH_C32(0xbdac7fcd), SPH_C32(0x7658000c), SPH_C32(0xf3edd77f),
	  SPH_C32(0x2093b717), SPH_C32(0xebaa3f43) },
	{ SPH_C32(0x5f590034), SPH_C32(0xbfe7d294), SPH_C32(0x9eb44515),
	  SPH_C32(0x12e65617), SPH_C32(0x1df1000c), SPH_C32(0xb3062ed5),
	  SPH_C32(0xb8a1ab2a), SPH_C32(0x9d06f870) },
	{ SPH_C32(0x4c560038), SPH_C32(0x53039d34), SPH_C32(0x65307a79),
	  SPH_C32(0x73a990cf), SPH_C32(0xdae50008), SPH_C32(0x8d1e92e6),
	  SPH_C32(0xbb0e9804), SPH_C32(0x6ab499f5) },
	{ SPH_C32(0x9c5e003c), SPH_C32(0xdf751243), SPH_C32(0xf8f5ca29),
	  SPH_C32(0xdce3b915), SPH_C32(0xb14c0008), SPH_C32(0xcdf56b4c),
	  SPH_C32(0x233c8439), SPH_C32(0x1c185ec6) },
	{ SPH_C32(0x27ff0038), SPH_C32(0x13e8649e), SPH_C32(0xfd026644),
	  SPH_C32(0x050557fc), SPH_C32(0x6144000c), SPH_C32(0x4183e43b),
	  SPH_C32(0xbef93469), SPH_C32(0xb352771c) },
	{ SPH_C32(0xf7f7003c), SPH_C32(0x9f9eebe9), SPH_C32(0x60c7d614),
	  SPH_C32(0xaa4f7e26), SPH_C32(0x0aed000c), SPH_C32(0x01681d91),
	  SPH_C32(0x26cb2854), SPH_C32(0xc5feb02f) },
	{ SPH_C32(0x8c810020), SPH_C32(0xe53a837f), SPH_C32(0xc51a4af8),
	  SPH_C32(0x535b307c), SPH_C32(0x23170010), SPH_C32(0xed94d960),
	  SPH_C32(0xb99a2d07), SPH_C32(0xa34ae1a5) },
	{ SPH_C32(0x5c890024), SPH_C32(0x694c0c08), SPH_C32(0x58dffaa8),
	  SPH_C32(0xfc1119a6), SPH_C32(0x48be0010), SPH_C32(0xad7f20ca),
	  SPH_C32(0x21a8313a), SPH_C32(0xd5e62696) },
	{ SPH_C32(0xe7280020), SPH_C32(0xa5d17ad5), SPH_C32(0x5d2856c5),
	  SPH_C32(0x25f7f74f), SPH_C32(0x98b60014), SPH_C32(0x2109afbd),
	  SPH_C32(0xbc6d816a), SPH_C32(0x7aac0f4c) },
	{ SPH_C32(0x37200024), SPH_C32(0x29a7f5a2), SPH_C32(0xc0ede695),
	  SPH_C32(0x8abdde95), SPH_C32(0xf31f0014), SPH_C32(0x61e25617),
	  SPH_C32(0x245f9d57), SPH_C32(0x0c00c87f) },
	{ SPH_C32(0x242f0028), SPH_C32(0xc543ba02), SPH_C32(0x3b69d9f9),
	  SPH_C32(0xebf2184d), SPH_C32(0x340b0010), SPH_C32(0x5ffaea24),
	  SPH_C32(0x27f0ae79), SPH_C32(0xfbb2a9fa) },
	{ SPH_C32(0xf427002c), SPH_C32(0x49353575), SPH_C32(0xa6ac69a9),
	  SPH_C32(0x44b83197), SPH_C32(0x5fa20010), SPH_C32(0x1f11138e),
	  SPH_C32(0xbfc2b244), SPH_C32(0x8d1e6ec9) },
	{ SPH_C32(0x4f860028), SPH_C32(0x85a843a8), SPH_C32(0xa35bc5c4),
	  SPH_C32(0x9d5edf7e), SPH_C32(0x8faa0014), SPH_C32(0x93679cf9),
	  SPH_C32(0x22070214), SPH_C32(0x22544713) },
	{ SPH_C32(0x9f8e002c), SPH_C32(0x09deccdf), SPH_C32(0x3e9e7594),
	  SPH_C32(0x3214f6a4), SPH_C32(0xe4030014), SPH_C32(0xd38c6553),
	  SPH_C32(0xba351e29), SPH_C32(0x54f88020) },
	{ SPH_C32(0x9b9d0020), SPH_C32(0x5754b03b), SPH_C32(0x5b70c986),
	  SPH_C32(0x0ba37823), SPH_C32(0x9ca50018), SPH_C32(0x7f83d359),
	  SPH_C32(0xd9833d78), SPH_C32(0x431b81cb) },
	{ SPH_C32(0x4b950024), SPH_C32(0xdb223f4c), SPH_C32(0xc6b579d6),
	  SPH_C32(0xa4e951f9), SPH_C32(0xf70c0018), SPH_C32(0x3f682af3),
	  SPH_C32(0x41b12145), SPH_C32(0x35b746f8) },
	{ SPH_C32(0xf0340020), SPH_C32(0x17bf4991), SPH_C32(0xc342d5bb),
	  SPH_C32(0x7d0fbf10), SPH_C32(0x2704001c), SPH_C32(0xb31ea584),
	  SPH_C32(0xdc749115), SPH_C32(0x9afd6f22) },
	{ SPH_C32(0x203c0024), SPH_C32(0x9bc9c6e6), SPH_C32(0x5e8765eb),
	  SPH_C32(0xd24596ca), SPH_C32(0x4cad001c), SPH_C32(0xf3f55c2e),
	  SPH_C32(0x44468d28), SPH_C32(0xec51a811) },
	{ SPH_C32(0x33330028), SPH_C32(0x772d8946), SPH_C32(0xa5035a87),
	  SPH_C32(0xb30a5012), SPH_C32(0x8bb90018), SPH_C32(0xcdede01d),
	  SPH_C32(0x47e9be06), SPH_C32(0x1be3c994) },
	{ SPH_C32(0xe33b002c), SPH_C32(0xfb5b0631), SPH_C32(0x38c6ead7),
	  SPH_C32(0x1c4079c8), SPH_C32(0xe0100018), SPH_C32(0x8d0619b7),
	  SPH_C32(0xdfdba23b), SPH_C32(0x6d4f0ea7) },
	{ SPH_C32(0x589a0028), SPH_C32(0x37c670ec), SPH_C32(0x3d3146ba),
	  SPH_C32(0xc5a69721), SPH_C32(0x3018001c), SPH_C32(0x017096c0),
	  SPH_C32(0x421e126b), SPH_C32(0xc205277d) },
	{ SPH_C32(0x8892002c), SPH_C32(0xbbb0ff9b), SPH_C32(0xa0f4f6ea),
	  SPH_C32(0x6aecbefb), SPH_C32(0x5bb1001c), SPH_C32(0x419b6f6a),
	  SPH_C32(0xda2c0e56), SPH_C32(0xb4a9e04e) },
	{ SPH_C32(0xdddd0030), SPH_C32(0xa5c9f184), SPH_C32(0x39fd6cfa),
	  SPH_C32(0x220c601d), SPH_C32(0x0d2e0010), SPH_C32(0x8949bfe9),
	  SPH_C32(0x854e2bfb), SPH_C32(0x12be7119) },
	{ SPH_C32(0x0dd50034), SPH_C32(0x29bf7ef3), SPH_C32(0xa438dcaa),
	  SPH_C32(0x8d4649c7), SPH_C32(0x66870010), SPH_C32(0xc9a24643),
	  SPH_C32(0x1d7c37c6), SPH_C32(0x6412b62a) },
	{ SPH_C32(0xb6740030), SPH_C32(0xe522082e), SPH_C32(0xa1cf70c7),
	  SPH_C32(0x54a0a72e), SPH_C32(0xb68f0014), SPH_C32(0x45d4c934),
	  SPH_C32(0x80b98796), SPH_C32(0xcb589ff0) },
	{ SPH_C32(0x667c0034), SPH_C32(0x69548759), SPH_C32(0x3c0ac097),
	  SPH_C32(0xfbea8ef4), SPH_C32(0xdd260014), SPH_C32(0x053f309e),
	  SPH_C32(0x188b9bab), SPH_C32(0xbdf458c3) },
	{ SPH_C32(0x75730038), SPH_C32(0x85b0c8f9), SPH_C32(0xc78efffb),
	  SPH_C32(0x9aa5482c), SPH_C32(0x1a320010), SPH_C32(0x3b278cad),
	  SPH_C32(0x1b24a885), SPH_C32(0x4a463946) },
	{ SPH_C32(0xa57b003c), SPH_C32(0x09c6478e), SPH_C32(0x5a4b4fab),
	  SPH_C32(0x35ef61f6), SPH_C32(0x719b0010), SPH_C32(0x7bcc7507),
	  SPH_C32(0x8316b4b8), SPH_C32(0x3ceafe75) },
	{ SPH_C32(0x1eda0038), SPH_C32(0xc55b3153), SPH_C32(0x5fbce3c6),
	  SPH_C32(0xec098f1f), SPH_C32(0xa1930014), SPH_C32(0xf7bafa70),
	  SPH_C32(0x1ed304e8), SPH_C32(0x93a0d7af) },
	{ SPH_C32(0xced2003c), SPH_C32(0x492dbe24), SPH_C32(0xc2795396),
	  SPH_C32(0x4343a6c5), SPH_C32(0xca3a0014), SPH_C32(0xb75103da),
	  SPH_C32(0x86e118d5), SPH_C32(0xe50c109c) },
	{ SPH_C32(0xcac10030), SPH_C32(0x17a7c2c0), SPH_C32(0xa797ef84),
	  SPH_C32(0x7af42842), SPH_C32(0xb29c0018), SPH_C32(0x1b5eb5d0),
	  SPH_C32(0xe5573b84), SPH_C32(0xf2ef1177) },
	{ SPH_C32(0x1ac90034), SPH_C32(0x9bd14db7), SPH_C32(0x3a525fd4),
	  SPH_C32(0xd5be0198), SPH_C32(0xd9350018), SPH_C32(0x5bb54c7a),
	  SPH_C32(0x7d6527b9), SPH_C32(0x8443d644) },
	{ SPH_C32(0xa1680030), SPH_C32(0x574c3b6a), SPH_C32(0x3fa5f3b9),
	  SPH_C32(0x0c58ef71), SPH_C32(0x093d001c), SPH_C32(0xd7c3c30d),
	  SPH_C32(0xe0a097e9), SPH_C32(0x2b09ff9e) },
	{ SPH_C32(0x71600034), SPH_C32(0xdb3ab41d), SPH_C32(0xa26043e9),
	  SPH_C32(0xa312c6ab), SPH_C32(0x6294001c), SPH_C32(0x97283aa7),
	  SPH_C32(0x78928bd4), SPH_C32(0x5da538ad) },
	{ SPH_C32(0x626f0038), SPH_C32(0x37defbbd), SPH_C32(0x59e47c85),
	  SPH_C32(0xc25d0073), SPH_C32(0xa5800018), SPH_C32(0xa9308694),
	  SPH_C32(0x7b3db8fa), SPH_C32(0xaa175928) },
	{ SPH_C32(0xb267003c), SPH_C32(0xbba874ca), SPH_C32(0xc421ccd5),
	  SPH_C32(0x6d1729a9), SPH_C32(0xce290018), SPH_C32(0xe9db7f3e),
	  SPH_C32(0xe30fa4c7), SPH_C32(0xdcbb9e1b) },
	{ SPH_C32(0x09c60038), SPH_C32(0x77350217), SPH_C32(0xc1d660b8),
	  SPH_C32(0xb4f1c740), SPH_C32(0x1e21001c), SPH_C32(0x65adf049),
	  SPH_C32(0x7eca1497), SPH_C32(0x73f1b7c1) },
	{ SPH_C32(0xd9ce003c), SPH_C32(0xfb438d60), SPH_C32(0x5c13d0e8),
	  SPH_C32(0x1bbbee9a), SPH_C32(0x7588001c), SPH_C32(0x254609e3),
	  SPH_C32(0xe6f808aa), SPH_C32(0x055d70f2) }
};

static const sph_u32 T256_28[16][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xc04e0001), SPH_C32(0x33b9c010), SPH_C32(0xae0ebb05),
	  SPH_C32(0xb5a4c63b), SPH_C32(0xc8f10000), SPH_C32(0x0b2de782),
	  SPH_C32(0x6bf648a4), SPH_C32(0x539cbdbf) },
	{ SPH_C32(0xc8f10000), SPH_C32(0x0b2de782), SPH_C32(0x6bf648a4),
	  SPH_C32(0x539cbdbf), SPH_C32(0x08bf0001), SPH_C32(0x38942792),
	  SPH_C32(0xc5f8f3a1), SPH_C32(0xe6387b84) },
	{ SPH_C32(0x08bf0001), SPH_C32(0x38942792), SPH_C32(0xc5f8f3a1),
	  SPH_C32(0xe6387b84), SPH_C32(0xc04e0001), SPH_C32(0x33b9c010),
	  SPH_C32(0xae0ebb05), SPH_C32(0xb5a4c63b) },
	{ SPH_C32(0x88230002), SPH_C32(0x5fe7a7b3), SPH_C32(0x99e585aa),
	  SPH_C32(0x8d75f7f1), SPH_C32(0x51ac0000), SPH_C32(0x25e30f14),
	  SPH_C32(0x79e22a4c), SPH_C32(0x1298bd46) },
	{ SPH_C32(0x486d0003), SPH_C32(0x6c5e67a3), SPH_C32(0x37eb3eaf),
	  SPH_C32(0x38d131ca), SPH_C32(0x995d0000), SPH_C32(0x2ecee896),
	  SPH_C32(0x121462e8), SPH_C32(0x410400f9) },
	{ SPH_C32(0x40d20002), SPH_C32(0x54ca4031), SPH_C32(0xf213cd0e),
	  SPH_C32(0xdee94a4e), SPH_C32(0x59130001), SPH_C32(0x1d772886),
	  SPH_C32(0xbc1ad9ed), SPH_C32(0xf4a0c6c2) },
	{ SPH_C32(0x809c0003), SPH_C32(0x67738021), SPH_C32(0x5c1d760b),
	  SPH_C32(0x6b4d8c75), SPH_C32(0x91e20001), SPH_C32(0x165acf04),
	  SPH_C32(0xd7ec9149), SPH_C32(0xa73c7b7d) },
	{ SPH_C32(0x51ac0000), SPH_C32(0x25e30f14), SPH_C32(0x79e22a4c),
	  SPH_C32(0x1298bd46), SPH_C32(0xd98f0002), SPH_C32(0x7a04a8a7),
	  SPH_C32(0xe007afe6), SPH_C32(0x9fed4ab7) },
	{ SPH_C32(0x91e20001), SPH_C32(0x165acf04), SPH_C32(0xd7ec9149),
	  SPH_C32(0xa73c7b7d), SPH_C32(0x117e0002), SPH_C32(0x71294f25),
	  SPH_C32(0x8bf1e742), SPH_C32(0xcc71f708) },
	{ SPH_C32(0x995d0000), SPH_C32(0x2ecee896), SPH_C32(0x121462e8),
	  SPH_C32(0x410400f9), SPH_C32(0xd1300003), SPH_C32(0x42908f35),
	  SPH_C32(0x25ff5c47), SPH_C32(0x79d53133) },
	{ SPH_C32(0x59130001), SPH_C32(0x1d772886), SPH_C32(0xbc1ad9ed),
	  SPH_C32(0xf4a0c6c2), SPH_C32(0x19c10003), SPH_C32(0x49bd68b7),
	  SPH_C32(0x4e0914e3), SPH_C32(0x2a498c8c) },
	{ SPH_C32(0xd98f0002), SPH_C32(0x7a04a8a7), SPH_C32(0xe007afe6),
	  SPH_C32(0x9fed4ab7), SPH_C32(0x88230002), SPH_C32(0x5fe7a7b3),
	  SPH_C32(0x99e585aa), SPH_C32(0x8d75f7f1) },
	{ SPH_C32(0x19c10003), SPH_C32(0x49bd68b7), SPH_C32(0x4e0914e3),
	  SPH_C32(0x2a498c8c), SPH_C32(0x40d20002), SPH_C32(0x54ca4031),
	  SPH_C32(0xf213cd0e), SPH_C32(0xdee94a4e) },
	{ SPH_C32(0x117e0002), SPH_C32(0x71294f25), SPH_C32(0x8bf1e742),
	  SPH_C32(0xcc71f708), SPH_C32(0x809c0003), SPH_C32(0x67738021),
	  SPH_C32(0x5c1d760b), SPH_C32(0x6b4d8c75) },
	{ SPH_C32(0xd1300003), SPH_C32(0x42908f35), SPH_C32(0x25ff5c47),
	  SPH_C32(0x79d53133), SPH_C32(0x486d0003), SPH_C32(0x6c5e67a3),
	  SPH_C32(0x37eb3eaf), SPH_C32(0x38d131ca) }
};

#define INPUT_SMALL   do { \
		unsigned acc = buf[0]; \
		const sph_u32 *rp; \
		rp = &T256_0[acc >> 1][0]; \
		m0 = rp[0]; \
		m1 = rp[1]; \
		m2 = rp[2]; \
		m3 = rp[3]; \
		m4 = rp[4]; \
		m5 = rp[5]; \
		m6 = rp[6]; \
		m7 = rp[7]; \
		acc = (acc << 8) | buf[1]; \
		rp = &T256_7[(acc >> 2) & 0x7f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		acc = (acc << 8) | buf[2]; \
		rp = &T256_14[(acc >> 3) & 0x7f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		acc = (acc << 8) | buf[3]; \
		rp = &T256_21[(acc >> 4) & 0x7f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		rp = &T256_28[acc & 0x0f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
	} while (0)

#endif

#if SPH_HAMSI_EXPAND_SMALL == 8

static const sph_u32 T256_0[256][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x74951000), SPH_C32(0x5a2b467e), SPH_C32(0x88fd1d2b),
	  SPH_C32(0x1ee68292), SPH_C32(0xcba90000), SPH_C32(0x90273769),
	  SPH_C32(0xbbdcf407), SPH_C32(0xd0f4af61) },
	{ SPH_C32(0xcba90000), SPH_C32(0x90273769), SPH_C32(0xbbdcf407),
	  SPH_C32(0xd0f4af61), SPH_C32(0xbf3c1000), SPH_C32(0xca0c7117),
	  SPH_C32(0x3321e92c), SPH_C32(0xce122df3) },
	{ SPH_C32(0xbf3c1000), SPH_C32(0xca0c7117), SPH_C32(0x3321e92c),
	  SPH_C32(0xce122df3), SPH_C32(0x74951000), SPH_C32(0x5a2b467e),
	  SPH_C32(0x88fd1d2b), SPH_C32(0x1ee68292) },
	{ SPH_C32(0xe92a2000), SPH_C32(0xb4578cfc), SPH_C32(0x11fa3a57),
	  SPH_C32(0x3dc90524), SPH_C32(0x97530000), SPH_C32(0x204f6ed3),
	  SPH_C32(0x77b9e80f), SPH_C32(0xa1ec5ec1) },
	{ SPH_C32(0x9dbf3000), SPH_C32(0xee7cca82), SPH_C32(0x9907277c),
	  SPH_C32(0x232f87b6), SPH_C32(0x5cfa0000), SPH_C32(0xb06859ba),
	  SPH_C32(0xcc651c08), SPH_C32(0x7118f1a0) },
	{ SPH_C32(0x22832000), SPH_C32(0x2470bb95), SPH_C32(0xaa26ce50),
	  SPH_C32(0xed3daa45), SPH_C32(0x286f1000), SPH_C32(0xea431fc4),
	  SPH_C32(0x44980123), SPH_C32(0x6ffe7332) },
	{ SPH_C32(0x56163000), SPH_C32(0x7e5bfdeb), SPH_C32(0x22dbd37b),
	  SPH_C32(0xf3db28d7), SPH_C32(0xe3c61000), SPH_C32(0x7a6428ad),
	  SPH_C32(0xff44f524), SPH_C32(0xbf0adc53) },
	{ SPH_C32(0x97530000), SPH_C32(0x204f6ed3), SPH_C32(0x77b9e80f),
	  SPH_C32(0xa1ec5ec1), SPH_C32(0x7e792000), SPH_C32(0x9418e22f),
	  SPH_C32(0x6643d258), SPH_C32(0x9c255be5) },
	{ SPH_C32(0xe3c61000), SPH_C32(0x7a6428ad), SPH_C32(0xff44f524),
	  SPH_C32(0xbf0adc53), SPH_C32(0xb5d02000), SPH_C32(0x043fd546),
	  SPH_C32(0xdd9f265f), SPH_C32(0x4cd1f484) },
	{ SPH_C32(0x5cfa0000), SPH_C32(0xb06859ba), SPH_C32(0xcc651c08),
	  SPH_C32(0x7118f1a0), SPH_C32(0xc1453000), SPH_C32(0x5e149338),
	  SPH_C32(0x55623b74), SPH_C32(0x52377616) },
	{ SPH_C32(0x286f1000), SPH_C32(0xea431fc4), SPH_C32(0x44980123),
	  SPH_C32(0x6ffe7332), SPH_C32(0x0aec3000), SPH_C32(0xce33a451),
	  SPH_C32(0xeebecf73), SPH_C32(0x82c3d977) },
	{ SPH_C32(0x7e792000), SPH_C32(0x9418e22f), SPH_C32(0x6643d258),
	  SPH_C32(0x9c255be5), SPH_C32(0xe92a2000), SPH_C32(0xb4578cfc),
	  SPH_C32(0x11fa3a57), SPH_C32(0x3dc90524) },
	{ SPH_C32(0x0aec3000), SPH_C32(0xce33a451), SPH_C32(0xeebecf73),
	  SPH_C32(0x82c3d977), SPH_C32(0x22832000), SPH_C32(0x2470bb95),
	  SPH_C32(0xaa26ce50), SPH_C32(0xed3daa45) },
	{ SPH_C32(0xb5d02000), SPH_C32(0x043fd546), SPH_C32(0xdd9f265f),
	  SPH_C32(0x4cd1f484), SPH_C32(0x56163000), SPH_C32(0x7e5bfdeb),
	  SPH_C32(0x22dbd37b), SPH_C32(0xf3db28d7) },
	{ SPH_C32(0xc1453000), SPH_C32(0x5e149338), SPH_C32(0x55623b74),
	  SPH_C32(0x52377616), SPH_C32(0x9dbf3000), SPH_C32(0xee7cca82),
	  SPH_C32(0x9907277c), SPH_C32(0x232f87b6) },
	{ SPH_C32(0x121b4000), SPH_C32(0x5b17d9e8), SPH_C32(0x8dfacfab),
	  SPH_C32(0xce36cc72), SPH_C32(0xe6570000), SPH_C32(0x4bb33a25),
	  SPH_C32(0x848598ba), SPH_C32(0x1041003e) },
	{ SPH_C32(0x668e5000), SPH_C32(0x013c9f96), SPH_C32(0x0507d280),
	  SPH_C32(0xd0d04ee0), SPH_C32(0x2dfe0000), SPH_C32(0xdb940d4c),
	  SPH_C32(0x3f596cbd), SPH_C32(0xc0b5af5f) },
	{ SPH_C32(0xd9b24000), SPH_C32(0xcb30ee81), SPH_C32(0x36263bac),
	  SPH_C32(0x1ec26313), SPH_C32(0x596b1000), SPH_C32(0x81bf4b32),
	  SPH_C32(0xb7a47196), SPH_C32(0xde532dcd) },
	{ SPH_C32(0xad275000), SPH_C32(0x911ba8ff), SPH_C32(0xbedb2687),
	  SPH_C32(0x0024e181), SPH_C32(0x92c21000), SPH_C32(0x11987c5b),
	  SPH_C32(0x0c788591), SPH_C32(0x0ea782ac) },
	{ SPH_C32(0xfb316000), SPH_C32(0xef405514), SPH_C32(0x9c00f5fc),
	  SPH_C32(0xf3ffc956), SPH_C32(0x71040000), SPH_C32(0x6bfc54f6),
	  SPH_C32(0xf33c70b5), SPH_C32(0xb1ad5eff) },
	{ SPH_C32(0x8fa47000), SPH_C32(0xb56b136a), SPH_C32(0x14fde8d7),
	  SPH_C32(0xed194bc4), SPH_C32(0xbaad0000), SPH_C32(0xfbdb639f),
	  SPH_C32(0x48e084b2), SPH_C32(0x6159f19e) },
	{ SPH_C32(0x30986000), SPH_C32(0x7f67627d), SPH_C32(0x27dc01fb),
	  SPH_C32(0x230b6637), SPH_C32(0xce381000), SPH_C32(0xa1f025e1),
	  SPH_C32(0xc01d9999), SPH_C32(0x7fbf730c) },
	{ SPH_C32(0x440d7000), SPH_C32(0x254c2403), SPH_C32(0xaf211cd0),
	  SPH_C32(0x3dede4a5), SPH_C32(0x05911000), SPH_C32(0x31d71288),
	  SPH_C32(0x7bc16d9e), SPH_C32(0xaf4bdc6d) },
	{ SPH_C32(0x85484000), SPH_C32(0x7b58b73b), SPH_C32(0xfa4327a4),
	  SPH_C32(0x6fda92b3), SPH_C32(0x982e2000), SPH_C32(0xdfabd80a),
	  SPH_C32(0xe2c64ae2), SPH_C32(0x8c645bdb) },
	{ SPH_C32(0xf1dd5000), SPH_C32(0x2173f145), SPH_C32(0x72be3a8f),
	  SPH_C32(0x713c1021), SPH_C32(0x53872000), SPH_C32(0x4f8cef63),
	  SPH_C32(0x591abee5), SPH_C32(0x5c90f4ba) },
	{ SPH_C32(0x4ee14000), SPH_C32(0xeb7f8052), SPH_C32(0x419fd3a3),
	  SPH_C32(0xbf2e3dd2), SPH_C32(0x27123000), SPH_C32(0x15a7a91d),
	  SPH_C32(0xd1e7a3ce), SPH_C32(0x42767628) },
	{ SPH_C32(0x3a745000), SPH_C32(0xb154c62c), SPH_C32(0xc962ce88),
	  SPH_C32(0xa1c8bf40), SPH_C32(0xecbb3000), SPH_C32(0x85809e74),
	  SPH_C32(0x6a3b57c9), SPH_C32(0x9282d949) },
	{ SPH_C32(0x6c626000), SPH_C32(0xcf0f3bc7), SPH_C32(0xebb91df3),
	  SPH_C32(0x52139797), SPH_C32(0x0f7d2000), SPH_C32(0xffe4b6d9),
	  SPH_C32(0x957fa2ed), SPH_C32(0x2d88051a) },
	{ SPH_C32(0x18f77000), SPH_C32(0x95247db9), SPH_C32(0x634400d8),
	  SPH_C32(0x4cf51505), SPH_C32(0xc4d42000), SPH_C32(0x6fc381b0),
	  SPH_C32(0x2ea356ea), SPH_C32(0xfd7caa7b) },
	{ SPH_C32(0xa7cb6000), SPH_C32(0x5f280cae), SPH_C32(0x5065e9f4),
	  SPH_C32(0x82e738f6), SPH_C32(0xb0413000), SPH_C32(0x35e8c7ce),
	  SPH_C32(0xa65e4bc1), SPH_C32(0xe39a28e9) },
	{ SPH_C32(0xd35e7000), SPH_C32(0x05034ad0), SPH_C32(0xd898f4df),
	  SPH_C32(0x9c01ba64), SPH_C32(0x7be83000), SPH_C32(0xa5cff0a7),
	  SPH_C32(0x1d82bfc6), SPH_C32(0x336e8788) },
	{ SPH_C32(0xe6570000), SPH_C32(0x4bb33a25), SPH_C32(0x848598ba),
	  SPH_C32(0x1041003e), SPH_C32(0xf44c4000), SPH_C32(0x10a4e3cd),
	  SPH_C32(0x097f5711), SPH_C32(0xde77cc4c) },
	{ SPH_C32(0x92c21000), SPH_C32(0x11987c5b), SPH_C32(0x0c788591),
	  SPH_C32(0x0ea782ac), SPH_C32(0x3fe54000), SPH_C32(0x8083d4a4),
	  SPH_C32(0xb2a3a316), SPH_C32(0x0e83632d) },
	{ SPH_C32(0x2dfe0000), SPH_C32(0xdb940d4c), SPH_C32(0x3f596cbd),
	  SPH_C32(0xc0b5af5f), SPH_C32(0x4b705000), SPH_C32(0xdaa892da),
	  SPH_C32(0x3a5ebe3d), SPH_C32(0x1065e1bf) },
	{ SPH_C32(0x596b1000), SPH_C32(0x81bf4b32), SPH_C32(0xb7a47196),
	  SPH_C32(0xde532dcd), SPH_C32(0x80d95000), SPH_C32(0x4a8fa5b3),
	  SPH_C32(0x81824a3a), SPH_C32(0xc0914ede) },
	{ SPH_C32(0x0f7d2000), SPH_C32(0xffe4b6d9), SPH_C32(0x957fa2ed),
	  SPH_C32(0x2d88051a), SPH_C32(0x631f4000), SPH_C32(0x30eb8d1e),
	  SPH_C32(0x7ec6bf1e), SPH_C32(0x7f9b928d) },
	{ SPH_C32(0x7be83000), SPH_C32(0xa5cff0a7), SPH_C32(0x1d82bfc6),
	  SPH_C32(0x336e8788), SPH_C32(0xa8b64000), SPH_C32(0xa0ccba77),
	  SPH_C32(0xc51a4b19), SPH_C32(0xaf6f3dec) },
	{ SPH_C32(0xc4d42000), SPH_C32(0x6fc381b0), SPH_C32(0x2ea356ea),
	  SPH_C32(0xfd7caa7b), SPH_C32(0xdc235000), SPH_C32(0xfae7fc09),
	  SPH_C32(0x4de75632), SPH_C32(0xb189bf7e) },
	{ SPH_C32(0xb0413000), SPH_C32(0x35e8c7ce), SPH_C32(0xa65e4bc1),
	  SPH_C32(0xe39a28e9), SPH_C32(0x178a5000), SPH_C32(0x6ac0cb60),
	  SPH_C32(0xf63ba235), SPH_C32(0x617d101f) },
	{ SPH_C32(0x71040000), SPH_C32(0x6bfc54f6), SPH_C32(0xf33c70b5),
	  SPH_C32(0xb1ad5eff), SPH_C32(0x8a356000), SPH_C32(0x84bc01e2),
	  SPH_C32(0x6f3c8549), SPH_C32(0x425297a9) },
	{ SPH_C32(0x05911000), SPH_C32(0x31d71288), SPH_C32(0x7bc16d9e),
	  SPH_C32(0xaf4bdc6d), SPH_C32(0x419c6000), SPH_C32(0x149b368b),
	  SPH_C32(0xd4e0714e), SPH_C32(0x92a638c8) },
	{ SPH_C32(0xbaad0000), SPH_C32(0xfbdb639f), SPH_C32(0x48e084b2),
	  SPH_C32(0x6159f19e), SPH_C32(0x35097000), SPH_C32(0x4eb070f5),
	  SPH_C32(0x5c1d6c65), SPH_C32(0x8c40ba5a) },
	{ SPH_C32(0xce381000), SPH_C32(0xa1f025e1), SPH_C32(0xc01d9999),
	  SPH_C32(0x7fbf730c), SPH_C32(0xfea07000), SPH_C32(0xde97479c),
	  SPH_C32(0xe7c19862), SPH_C32(0x5cb4153b) },
	{ SPH_C32(0x982e2000), SPH_C32(0xdfabd80a), SPH_C32(0xe2c64ae2),
	  SPH_C32(0x8c645bdb), SPH_C32(0x1d666000), SPH_C32(0xa4f36f31),
	  SPH_C32(0x18856d46), SPH_C32(0xe3bec968) },
	{ SPH_C32(0xecbb3000), SPH_C32(0x85809e74), SPH_C32(0x6a3b57c9),
	  SPH_C32(0x9282d949), SPH_C32(0xd6cf6000), SPH_C32(0x34d45858),
	  SPH_C32(0xa3599941), SPH_C32(0x334a6609) },
	{ SPH_C32(0x53872000), SPH_C32(0x4f8cef63), SPH_C32(0x591abee5),
	  SPH_C32(0x5c90f4ba), SPH_C32(0xa25a7000), SPH_C32(0x6eff1e26),
	  SPH_C32(0x2ba4846a), SPH_C32(0x2dace49b) },
	{ SPH_C32(0x27123000), SPH_C32(0x15a7a91d), SPH_C32(0xd1e7a3ce),
	  SPH_C32(0x42767628), SPH_C32(0x69f37000), SPH_C32(0xfed8294f),
	  SPH_C32(0x9078706d), SPH_C32(0xfd584bfa) },
	{ SPH_C32(0xf44c4000), SPH_C32(0x10a4e3cd), SPH_C32(0x097f5711),
	  SPH_C32(0xde77cc4c), SPH_C32(0x121b4000), SPH_C32(0x5b17d9e8),
	  SPH_C32(0x8dfacfab), SPH_C32(0xce36cc72) },
	{ SPH_C32(0x80d95000), SPH_C32(0x4a8fa5b3), SPH_C32(0x81824a3a),
	  SPH_C32(0xc0914ede), SPH_C32(0xd9b24000), SPH_C32(0xcb30ee81),
	  SPH_C32(0x36263bac), SPH_C32(0x1ec26313) },
	{ SPH_C32(0x3fe54000), SPH_C32(0x8083d4a4), SPH_C32(0xb2a3a316),
	  SPH_C32(0x0e83632d), SPH_C32(0xad275000), SPH_C32(0x911ba8ff),
	  SPH_C32(0xbedb2687), SPH_C32(0x0024e181) },
	{ SPH_C32(0x4b705000), SPH_C32(0xdaa892da), SPH_C32(0x3a5ebe3d),
	  SPH_C32(0x1065e1bf), SPH_C32(0x668e5000), SPH_C32(0x013c9f96),
	  SPH_C32(0x0507d280), SPH_C32(0xd0d04ee0) },
	{ SPH_C32(0x1d666000), SPH_C32(0xa4f36f31), SPH_C32(0x18856d46),
	  SPH_C32(0xe3bec968), SPH_C32(0x85484000), SPH_C32(0x7b58b73b),
	  SPH_C32(0xfa4327a4), SPH_C32(0x6fda92b3) },
	{ SPH_C32(0x69f37000), SPH_C32(0xfed8294f), SPH_C32(0x9078706d),
	  SPH_C32(0xfd584bfa), SPH_C32(0x4ee14000), SPH_C32(0xeb7f8052),
	  SPH_C32(0x419fd3a3), SPH_C32(0xbf2e3dd2) },
	{ SPH_C32(0xd6cf6000), SPH_C32(0x34d45858), SPH_C32(0xa3599941),
	  SPH_C32(0x334a6609), SPH_C32(0x3a745000), SPH_C32(0xb154c62c),
	  SPH_C32(0xc962ce88), SPH_C32(0xa1c8bf40) },
	{ SPH_C32(0xa25a7000), SPH_C32(0x6eff1e26), SPH_C32(0x2ba4846a),
	  SPH_C32(0x2dace49b), SPH_C32(0xf1dd5000), SPH_C32(0x2173f145),
	  SPH_C32(0x72be3a8f), SPH_C32(0x713c1021) },
	{ SPH_C32(0x631f4000), SPH_C32(0x30eb8d1e), SPH_C32(0x7ec6bf1e),
	  SPH_C32(0x7f9b928d), SPH_C32(0x6c626000), SPH_C32(0xcf0f3bc7),
	  SPH_C32(0xebb91df3), SPH_C32(0x52139797) },
	{ SPH_C32(0x178a5000), SPH_C32(0x6ac0cb60), SPH_C32(0xf63ba235),
	  SPH_C32(0x617d101f), SPH_C32(0xa7cb6000), SPH_C32(0x5f280cae),
	  SPH_C32(0x5065e9f4), SPH_C32(0x82e738f6) },
	{ SPH_C32(0xa8b64000), SPH_C32(0xa0ccba77), SPH_C32(0xc51a4b19),
	  SPH_C32(0xaf6f3dec), SPH_C32(0xd35e7000), SPH_C32(0x05034ad0),
	  SPH_C32(0xd898f4df), SPH_C32(0x9c01ba64) },
	{ SPH_C32(0xdc235000), SPH_C32(0xfae7fc09), SPH_C32(0x4de75632),
	  SPH_C32(0xb189bf7e), SPH_C32(0x18f77000), SPH_C32(0x95247db9),
	  SPH_C32(0x634400d8), SPH_C32(0x4cf51505) },
	{ SPH_C32(0x8a356000), SPH_C32(0x84bc01e2), SPH_C32(0x6f3c8549),
	  SPH_C32(0x425297a9), SPH_C32(0xfb316000), SPH_C32(0xef405514),
	  SPH_C32(0x9c00f5fc), SPH_C32(0xf3ffc956) },
	{ SPH_C32(0xfea07000), SPH_C32(0xde97479c), SPH_C32(0xe7c19862),
	  SPH_C32(0x5cb4153b), SPH_C32(0x30986000), SPH_C32(0x7f67627d),
	  SPH_C32(0x27dc01fb), SPH_C32(0x230b6637) },
	{ SPH_C32(0x419c6000), SPH_C32(0x149b368b), SPH_C32(0xd4e0714e),
	  SPH_C32(0x92a638c8), SPH_C32(0x440d7000), SPH_C32(0x254c2403),
	  SPH_C32(0xaf211cd0), SPH_C32(0x3dede4a5) },
	{ SPH_C32(0x35097000), SPH_C32(0x4eb070f5), SPH_C32(0x5c1d6c65),
	  SPH_C32(0x8c40ba5a), SPH_C32(0x8fa47000), SPH_C32(0xb56b136a),
	  SPH_C32(0x14fde8d7), SPH_C32(0xed194bc4) },
	{ SPH_C32(0xe4788000), SPH_C32(0x859673c1), SPH_C32(0xb5fb2452),
	  SPH_C32(0x29cc5edf), SPH_C32(0x045f0000), SPH_C32(0x9c4a93c9),
	  SPH_C32(0x62fc79d0), SPH_C32(0x731ebdc2) },
	{ SPH_C32(0x90ed9000), SPH_C32(0xdfbd35bf), SPH_C32(0x3d063979),
	  SPH_C32(0x372adc4d), SPH_C32(0xcff60000), SPH_C32(0x0c6da4a0),
	  SPH_C32(0xd9208dd7), SPH_C32(0xa3ea12a3) },
	{ SPH_C32(0x2fd18000), SPH_C32(0x15b144a8), SPH_C32(0x0e27d055),
	  SPH_C32(0xf938f1be), SPH_C32(0xbb631000), SPH_C32(0x5646e2de),
	  SPH_C32(0x51dd90fc), SPH_C32(0xbd0c9031) },
	{ SPH_C32(0x5b449000), SPH_C32(0x4f9a02d6), SPH_C32(0x86dacd7e),
	  SPH_C32(0xe7de732c), SPH_C32(0x70ca1000), SPH_C32(0xc661d5b7),
	  SPH_C32(0xea0164fb), SPH_C32(0x6df83f50) },
	{ SPH_C32(0x0d52a000), SPH_C32(0x31c1ff3d), SPH_C32(0xa4011e05),
	  SPH_C32(0x14055bfb), SPH_C32(0x930c0000), SPH_C32(0xbc05fd1a),
	  SPH_C32(0x154591df), SPH_C32(0xd2f2e303) },
	{ SPH_C32(0x79c7b000), SPH_C32(0x6beab943), SPH_C32(0x2cfc032e),
	  SPH_C32(0x0ae3d969), SPH_C32(0x58a50000), SPH_C32(0x2c22ca73),
	  SPH_C32(0xae9965d8), SPH_C32(0x02064c62) },
	{ SPH_C32(0xc6fba000), SPH_C32(0xa1e6c854), SPH_C32(0x1fddea02),
	  SPH_C32(0xc4f1f49a), SPH_C32(0x2c301000), SPH_C32(0x76098c0d),
	  SPH_C32(0x266478f3), SPH_C32(0x1ce0cef0) },
	{ SPH_C32(0xb26eb000), SPH_C32(0xfbcd8e2a), SPH_C32(0x9720f729),
	  SPH_C32(0xda177608), SPH_C32(0xe7991000), SPH_C32(0xe62ebb64),
	  SPH_C32(0x9db88cf4), SPH_C32(0xcc146191) },
	{ SPH_C32(0x732b8000), SPH_C32(0xa5d91d12), SPH_C32(0xc242cc5d),
	  SPH_C32(0x8820001e), SPH_C32(0x7a262000), SPH_C32(0x085271e6),
	  SPH_C32(0x04bfab88), SPH_C32(0xef3be627) },
	{ SPH_C32(0x07be9000), SPH_C32(0xfff25b6c), SPH_C32(0x4abfd176),
	  SPH_C32(0x96c6828c), SPH_C32(0xb18f2000), SPH_C32(0x9875468f),
	  SPH_C32(0xbf635f8f), SPH_C32(0x3fcf4946) },
	{ SPH_C32(0xb8828000), SPH_C32(0x35fe2a7b), SPH_C32(0x799e385a),
	  SPH_C32(0x58d4af7f), SPH_C32(0xc51a3000), SPH_C32(0xc25e00f1),
	  SPH_C32(0x379e42a4), SPH_C32(0x2129cbd4) },
	{ SPH_C32(0xcc179000), SPH_C32(0x6fd56c05), SPH_C32(0xf1632571),
	  SPH_C32(0x46322ded), SPH_C32(0x0eb33000), SPH_C32(0x52793798),
	  SPH_C32(0x8c42b6a3), SPH_C32(0xf1dd64b5) },
	{ SPH_C32(0x9a01a000), SPH_C32(0x118e91ee), SPH_C32(0xd3b8f60a),
	  SPH_C32(0xb5e9053a), SPH_C32(0xed752000), SPH_C32(0x281d1f35),
	  SPH_C32(0x73064387), SPH_C32(0x4ed7b8e6) },
	{ SPH_C32(0xee94b000), SPH_C32(0x4ba5d790), SPH_C32(0x5b45eb21),
	  SPH_C32(0xab0f87a8), SPH_C32(0x26dc2000), SPH_C32(0xb83a285c),
	  SPH_C32(0xc8dab780), SPH_C32(0x9e231787) },
	{ SPH_C32(0x51a8a000), SPH_C32(0x81a9a687), SPH_C32(0x6864020d),
	  SPH_C32(0x651daa5b), SPH_C32(0x52493000), SPH_C32(0xe2116e22),
	  SPH_C32(0x4027aaab), SPH_C32(0x80c59515) },
	{ SPH_C32(0x253db000), SPH_C32(0xdb82e0f9), SPH_C32(0xe0991f26),
	  SPH_C32(0x7bfb28c9), SPH_C32(0x99e03000), SPH_C32(0x7236594b),
	  SPH_C32(0xfbfb5eac), SPH_C32(0x50313a74) },
	{ SPH_C32(0xf663c000), SPH_C32(0xde81aa29), SPH_C32(0x3801ebf9),
	  SPH_C32(0xe7fa92ad), SPH_C32(0xe2080000), SPH_C32(0xd7f9a9ec),
	  SPH_C32(0xe679e16a), SPH_C32(0x635fbdfc) },
	{ SPH_C32(0x82f6d000), SPH_C32(0x84aaec57), SPH_C32(0xb0fcf6d2),
	  SPH_C32(0xf91c103f), SPH_C32(0x29a10000), SPH_C32(0x47de9e85),
	  SPH_C32(0x5da5156d), SPH_C32(0xb3ab129d) },
	{ SPH_C32(0x3dcac000), SPH_C32(0x4ea69d40), SPH_C32(0x83dd1ffe),
	  SPH_C32(0x370e3dcc), SPH_C32(0x5d341000), SPH_C32(0x1df5d8fb),
	  SPH_C32(0xd5580846), SPH_C32(0xad4d900f) },
	{ SPH_C32(0x495fd000), SPH_C32(0x148ddb3e), SPH_C32(0x0b2002d5),
	  SPH_C32(0x29e8bf5e), SPH_C32(0x969d1000), SPH_C32(0x8dd2ef92),
	  SPH_C32(0x6e84fc41), SPH_C32(0x7db93f6e) },
	{ SPH_C32(0x1f49e000), SPH_C32(0x6ad626d5), SPH_C32(0x29fbd1ae),
	  SPH_C32(0xda339789), SPH_C32(0x755b0000), SPH_C32(0xf7b6c73f),
	  SPH_C32(0x91c00965), SPH_C32(0xc2b3e33d) },
	{ SPH_C32(0x6bdcf000), SPH_C32(0x30fd60ab), SPH_C32(0xa106cc85),
	  SPH_C32(0xc4d5151b), SPH_C32(0xbef20000), SPH_C32(0x6791f056),
	  SPH_C32(0x2a1cfd62), SPH_C32(0x12474c5c) },
	{ SPH_C32(0xd4e0e000), SPH_C32(0xfaf111bc), SPH_C32(0x922725a9),
	  SPH_C32(0x0ac738e8), SPH_C32(0xca671000), SPH_C32(0x3dbab628),
	  SPH_C32(0xa2e1e049), SPH_C32(0x0ca1cece) },
	{ SPH_C32(0xa075f000), SPH_C32(0xa0da57c2), SPH_C32(0x1ada3882),
	  SPH_C32(0x1421ba7a), SPH_C32(0x01ce1000), SPH_C32(0xad9d8141),
	  SPH_C32(0x193d144e), SPH_C32(0xdc5561af) },
	{ SPH_C32(0x6130c000), SPH_C32(0xfecec4fa), SPH_C32(0x4fb803f6),
	  SPH_C32(0x4616cc6c), SPH_C32(0x9c712000), SPH_C32(0x43e14bc3),
	  SPH_C32(0x803a3332), SPH_C32(0xff7ae619) },
	{ SPH_C32(0x15a5d000), SPH_C32(0xa4e58284), SPH_C32(0xc7451edd),
	  SPH_C32(0x58f04efe), SPH_C32(0x57d82000), SPH_C32(0xd3c67caa),
	  SPH_C32(0x3be6c735), SPH_C32(0x2f8e4978) },
	{ SPH_C32(0xaa99c000), SPH_C32(0x6ee9f393), SPH_C32(0xf464f7f1),
	  SPH_C32(0x96e2630d), SPH_C32(0x234d3000), SPH_C32(0x89ed3ad4),
	  SPH_C32(0xb31bda1e), SPH_C32(0x3168cbea) },
	{ SPH_C32(0xde0cd000), SPH_C32(0x34c2b5ed), SPH_C32(0x7c99eada),
	  SPH_C32(0x8804e19f), SPH_C32(0xe8e43000), SPH_C32(0x19ca0dbd),
	  SPH_C32(0x08c72e19), SPH_C32(0xe19c648b) },
	{ SPH_C32(0x881ae000), SPH_C32(0x4a994806), SPH_C32(0x5e4239a1),
	  SPH_C32(0x7bdfc948), SPH_C32(0x0b222000), SPH_C32(0x63ae2510),
	  SPH_C32(0xf783db3d), SPH_C32(0x5e96b8d8) },
	{ SPH_C32(0xfc8ff000), SPH_C32(0x10b20e78), SPH_C32(0xd6bf248a),
	  SPH_C32(0x65394bda), SPH_C32(0xc08b2000), SPH_C32(0xf3891279),
	  SPH_C32(0x4c5f2f3a), SPH_C32(0x8e6217b9) },
	{ SPH_C32(0x43b3e000), SPH_C32(0xdabe7f6f), SPH_C32(0xe59ecda6),
	  SPH_C32(0xab2b6629), SPH_C32(0xb41e3000), SPH_C32(0xa9a25407),
	  SPH_C32(0xc4a23211), SPH_C32(0x9084952b) },
	{ SPH_C32(0x3726f000), SPH_C32(0x80953911), SPH_C32(0x6d63d08d),
	  SPH_C32(0xb5cde4bb), SPH_C32(0x7fb73000), SPH_C32(0x3985636e),
	  SPH_C32(0x7f7ec616), SPH_C32(0x40703a4a) },
	{ SPH_C32(0x022f8000), SPH_C32(0xce2549e4), SPH_C32(0x317ebce8),
	  SPH_C32(0x398d5ee1), SPH_C32(0xf0134000), SPH_C32(0x8cee7004),
	  SPH_C32(0x6b832ec1), SPH_C32(0xad69718e) },
	{ SPH_C32(0x76ba9000), SPH_C32(0x940e0f9a), SPH_C32(0xb983a1c3),
	  SPH_C32(0x276bdc73), SPH_C32(0x3bba4000), SPH_C32(0x1cc9476d),
	  SPH_C32(0xd05fdac6), SPH_C32(0x7d9ddeef) },
	{ SPH_C32(0xc9868000), SPH_C32(0x5e027e8d), SPH_C32(0x8aa248ef),
	  SPH_C32(0xe979f180), SPH_C32(0x4f2f5000), SPH_C32(0x46e20113),
	  SPH_C32(0x58a2c7ed), SPH_C32(0x637b5c7d) },
	{ SPH_C32(0xbd139000), SPH_C32(0x042938f3), SPH_C32(0x025f55c4),
	  SPH_C32(0xf79f7312), SPH_C32(0x84865000), SPH_C32(0xd6c5367a),
	  SPH_C32(0xe37e33ea), SPH_C32(0xb38ff31c) },
	{ SPH_C32(0xeb05a000), SPH_C32(0x7a72c518), SPH_C32(0x208486bf),
	  SPH_C32(0x04445bc5), SPH_C32(0x67404000), SPH_C32(0xaca11ed7),
	  SPH_C32(0x1c3ac6ce), SPH_C32(0x0c852f4f) },
	{ SPH_C32(0x9f90b000), SPH_C32(0x20598366), SPH_C32(0xa8799b94),
	  SPH_C32(0x1aa2d957), SPH_C32(0xace94000), SPH_C32(0x3c8629be),
	  SPH_C32(0xa7e632c9), SPH_C32(0xdc71802e) },
	{ SPH_C32(0x20aca000), SPH_C32(0xea55f271), SPH_C32(0x9b5872b8),
	  SPH_C32(0xd4b0f4a4), SPH_C32(0xd87c5000), SPH_C32(0x66ad6fc0),
	  SPH_C32(0x2f1b2fe2), SPH_C32(0xc29702bc) },
	{ SPH_C32(0x5439b000), SPH_C32(0xb07eb40f), SPH_C32(0x13a56f93),
	  SPH_C32(0xca567636), SPH_C32(0x13d55000), SPH_C32(0xf68a58a9),
	  SPH_C32(0x94c7dbe5), SPH_C32(0x1263addd) },
	{ SPH_C32(0x957c8000), SPH_C32(0xee6a2737), SPH_C32(0x46c754e7),
	  SPH_C32(0x98610020), SPH_C32(0x8e6a6000), SPH_C32(0x18f6922b),
	  SPH_C32(0x0dc0fc99), SPH_C32(0x314c2a6b) },
	{ SPH_C32(0xe1e99000), SPH_C32(0xb4416149), SPH_C32(0xce3a49cc),
	  SPH_C32(0x868782b2), SPH_C32(0x45c36000), SPH_C32(0x88d1a542),
	  SPH_C32(0xb61c089e), SPH_C32(0xe1b8850a) },
	{ SPH_C32(0x5ed58000), SPH_C32(0x7e4d105e), SPH_C32(0xfd1ba0e0),
	  SPH_C32(0x4895af41), SPH_C32(0x31567000), SPH_C32(0xd2fae33c),
	  SPH_C32(0x3ee115b5), SPH_C32(0xff5e0798) },
	{ SPH_C32(0x2a409000), SPH_C32(0x24665620), SPH_C32(0x75e6bdcb),
	  SPH_C32(0x56732dd3), SPH_C32(0xfaff7000), SPH_C32(0x42ddd455),
	  SPH_C32(0x853de1b2), SPH_C32(0x2faaa8f9) },
	{ SPH_C32(0x7c56a000), SPH_C32(0x5a3dabcb), SPH_C32(0x573d6eb0),
	  SPH_C32(0xa5a80504), SPH_C32(0x19396000), SPH_C32(0x38b9fcf8),
	  SPH_C32(0x7a791496), SPH_C32(0x90a074aa) },
	{ SPH_C32(0x08c3b000), SPH_C32(0x0016edb5), SPH_C32(0xdfc0739b),
	  SPH_C32(0xbb4e8796), SPH_C32(0xd2906000), SPH_C32(0xa89ecb91),
	  SPH_C32(0xc1a5e091), SPH_C32(0x4054dbcb) },
	{ SPH_C32(0xb7ffa000), SPH_C32(0xca1a9ca2), SPH_C32(0xece19ab7),
	  SPH_C32(0x755caa65), SPH_C32(0xa6057000), SPH_C32(0xf2b58def),
	  SPH_C32(0x4958fdba), SPH_C32(0x5eb25959) },
	{ SPH_C32(0xc36ab000), SPH_C32(0x9031dadc), SPH_C32(0x641c879c),
	  SPH_C32(0x6bba28f7), SPH_C32(0x6dac7000), SPH_C32(0x6292ba86),
	  SPH_C32(0xf28409bd), SPH_C32(0x8e46f638) },
	{ SPH_C32(0x1034c000), SPH_C32(0x9532900c), SPH_C32(0xbc847343),
	  SPH_C32(0xf7bb9293), SPH_C32(0x16444000), SPH_C32(0xc75d4a21),
	  SPH_C32(0xef06b67b), SPH_C32(0xbd2871b0) },
	{ SPH_C32(0x64a1d000), SPH_C32(0xcf19d672), SPH_C32(0x34796e68),
	  SPH_C32(0xe95d1001), SPH_C32(0xdded4000), SPH_C32(0x577a7d48),
	  SPH_C32(0x54da427c), SPH_C32(0x6ddcded1) },
	{ SPH_C32(0xdb9dc000), SPH_C32(0x0515a765), SPH_C32(0x07588744),
	  SPH_C32(0x274f3df2), SPH_C32(0xa9785000), SPH_C32(0x0d513b36),
	  SPH_C32(0xdc275f57), SPH_C32(0x733a5c43) },
	{ SPH_C32(0xaf08d000), SPH_C32(0x5f3ee11b), SPH_C32(0x8fa59a6f),
	  SPH_C32(0x39a9bf60), SPH_C32(0x62d15000), SPH_C32(0x9d760c5f),
	  SPH_C32(0x67fbab50), SPH_C32(0xa3cef322) },
	{ SPH_C32(0xf91ee000), SPH_C32(0x21651cf0), SPH_C32(0xad7e4914),
	  SPH_C32(0xca7297b7), SPH_C32(0x81174000), SPH_C32(0xe71224f2),
	  SPH_C32(0x98bf5e74), SPH_C32(0x1cc42f71) },
	{ SPH_C32(0x8d8bf000), SPH_C32(0x7b4e5a8e), SPH_C32(0x2583543f),
	  SPH_C32(0xd4941525), SPH_C32(0x4abe4000), SPH_C32(0x7735139b),
	  SPH_C32(0x2363aa73), SPH_C32(0xcc308010) },
	{ SPH_C32(0x32b7e000), SPH_C32(0xb1422b99), SPH_C32(0x16a2bd13),
	  SPH_C32(0x1a8638d6), SPH_C32(0x3e2b5000), SPH_C32(0x2d1e55e5),
	  SPH_C32(0xab9eb758), SPH_C32(0xd2d60282) },
	{ SPH_C32(0x4622f000), SPH_C32(0xeb696de7), SPH_C32(0x9e5fa038),
	  SPH_C32(0x0460ba44), SPH_C32(0xf5825000), SPH_C32(0xbd39628c),
	  SPH_C32(0x1042435f), SPH_C32(0x0222ade3) },
	{ SPH_C32(0x8767c000), SPH_C32(0xb57dfedf), SPH_C32(0xcb3d9b4c),
	  SPH_C32(0x5657cc52), SPH_C32(0x683d6000), SPH_C32(0x5345a80e),
	  SPH_C32(0x89456423), SPH_C32(0x210d2a55) },
	{ SPH_C32(0xf3f2d000), SPH_C32(0xef56b8a1), SPH_C32(0x43c08667),
	  SPH_C32(0x48b14ec0), SPH_C32(0xa3946000), SPH_C32(0xc3629f67),
	  SPH_C32(0x32999024), SPH_C32(0xf1f98534) },
	{ SPH_C32(0x4ccec000), SPH_C32(0x255ac9b6), SPH_C32(0x70e16f4b),
	  SPH_C32(0x86a36333), SPH_C32(0xd7017000), SPH_C32(0x9949d919),
	  SPH_C32(0xba648d0f), SPH_C32(0xef1f07a6) },
	{ SPH_C32(0x385bd000), SPH_C32(0x7f718fc8), SPH_C32(0xf81c7260),
	  SPH_C32(0x9845e1a1), SPH_C32(0x1ca87000), SPH_C32(0x096eee70),
	  SPH_C32(0x01b87908), SPH_C32(0x3feba8c7) },
	{ SPH_C32(0x6e4de000), SPH_C32(0x012a7223), SPH_C32(0xdac7a11b),
	  SPH_C32(0x6b9ec976), SPH_C32(0xff6e6000), SPH_C32(0x730ac6dd),
	  SPH_C32(0xfefc8c2c), SPH_C32(0x80e17494) },
	{ SPH_C32(0x1ad8f000), SPH_C32(0x5b01345d), SPH_C32(0x523abc30),
	  SPH_C32(0x75784be4), SPH_C32(0x34c76000), SPH_C32(0xe32df1b4),
	  SPH_C32(0x4520782b), SPH_C32(0x5015dbf5) },
	{ SPH_C32(0xa5e4e000), SPH_C32(0x910d454a), SPH_C32(0x611b551c),
	  SPH_C32(0xbb6a6617), SPH_C32(0x40527000), SPH_C32(0xb906b7ca),
	  SPH_C32(0xcddd6500), SPH_C32(0x4ef35967) },
	{ SPH_C32(0xd171f000), SPH_C32(0xcb260334), SPH_C32(0xe9e64837),
	  SPH_C32(0xa58ce485), SPH_C32(0x8bfb7000), SPH_C32(0x292180a3),
	  SPH_C32(0x76019107), SPH_C32(0x9e07f606) },
	{ SPH_C32(0x045f0000), SPH_C32(0x9c4a93c9), SPH_C32(0x62fc79d0),
	  SPH_C32(0x731ebdc2), SPH_C32(0xe0278000), SPH_C32(0x19dce008),
	  SPH_C32(0xd7075d82), SPH_C32(0x5ad2e31d) },
	{ SPH_C32(0x70ca1000), SPH_C32(0xc661d5b7), SPH_C32(0xea0164fb),
	  SPH_C32(0x6df83f50), SPH_C32(0x2b8e8000), SPH_C32(0x89fbd761),
	  SPH_C32(0x6cdba985), SPH_C32(0x8a264c7c) },
	{ SPH_C32(0xcff60000), SPH_C32(0x0c6da4a0), SPH_C32(0xd9208dd7),
	  SPH_C32(0xa3ea12a3), SPH_C32(0x5f1b9000), SPH_C32(0xd3d0911f),
	  SPH_C32(0xe426b4ae), SPH_C32(0x94c0ceee) },
	{ SPH_C32(0xbb631000), SPH_C32(0x5646e2de), SPH_C32(0x51dd90fc),
	  SPH_C32(0xbd0c9031), SPH_C32(0x94b29000), SPH_C32(0x43f7a676),
	  SPH_C32(0x5ffa40a9), SPH_C32(0x4434618f) },
	{ SPH_C32(0xed752000), SPH_C32(0x281d1f35), SPH_C32(0x73064387),
	  SPH_C32(0x4ed7b8e6), SPH_C32(0x77748000), SPH_C32(0x39938edb),
	  SPH_C32(0xa0beb58d), SPH_C32(0xfb3ebddc) },
	{ SPH_C32(0x99e03000), SPH_C32(0x7236594b), SPH_C32(0xfbfb5eac),
	  SPH_C32(0x50313a74), SPH_C32(0xbcdd8000), SPH_C32(0xa9b4b9b2),
	  SPH_C32(0x1b62418a), SPH_C32(0x2bca12bd) },
	{ SPH_C32(0x26dc2000), SPH_C32(0xb83a285c), SPH_C32(0xc8dab780),
	  SPH_C32(0x9e231787), SPH_C32(0xc8489000), SPH_C32(0xf39fffcc),
	  SPH_C32(0x939f5ca1), SPH_C32(0x352c902f) },
	{ SPH_C32(0x52493000), SPH_C32(0xe2116e22), SPH_C32(0x4027aaab),
	  SPH_C32(0x80c59515), SPH_C32(0x03e19000), SPH_C32(0x63b8c8a5),
	  SPH_C32(0x2843a8a6), SPH_C32(0xe5d83f4e) },
	{ SPH_C32(0x930c0000), SPH_C32(0xbc05fd1a), SPH_C32(0x154591df),
	  SPH_C32(0xd2f2e303), SPH_C32(0x9e5ea000), SPH_C32(0x8dc40227),
	  SPH_C32(0xb1448fda), SPH_C32(0xc6f7b8f8) },
	{ SPH_C32(0xe7991000), SPH_C32(0xe62ebb64), SPH_C32(0x9db88cf4),
	  SPH_C32(0xcc146191), SPH_C32(0x55f7a000), SPH_C32(0x1de3354e),
	  SPH_C32(0x0a987bdd), SPH_C32(0x16031799) },
	{ SPH_C32(0x58a50000), SPH_C32(0x2c22ca73), SPH_C32(0xae9965d8),
	  SPH_C32(0x02064c62), SPH_C32(0x2162b000), SPH_C32(0x47c87330),
	  SPH_C32(0x826566f6), SPH_C32(0x08e5950b) },
	{ SPH_C32(0x2c301000), SPH_C32(0x76098c0d), SPH_C32(0x266478f3),
	  SPH_C32(0x1ce0cef0), SPH_C32(0xeacbb000), SPH_C32(0xd7ef4459),
	  SPH_C32(0x39b992f1), SPH_C32(0xd8113a6a) },
	{ SPH_C32(0x7a262000), SPH_C32(0x085271e6), SPH_C32(0x04bfab88),
	  SPH_C32(0xef3be627), SPH_C32(0x090da000), SPH_C32(0xad8b6cf4),
	  SPH_C32(0xc6fd67d5), SPH_C32(0x671be639) },
	{ SPH_C32(0x0eb33000), SPH_C32(0x52793798), SPH_C32(0x8c42b6a3),
	  SPH_C32(0xf1dd64b5), SPH_C32(0xc2a4a000), SPH_C32(0x3dac5b9d),
	  SPH_C32(0x7d2193d2), SPH_C32(0xb7ef4958) },
	{ SPH_C32(0xb18f2000), SPH_C32(0x9875468f), SPH_C32(0xbf635f8f),
	  SPH_C32(0x3fcf4946), SPH_C32(0xb631b000), SPH_C32(0x67871de3),
	  SPH_C32(0xf5dc8ef9), SPH_C32(0xa909cbca) },
	{ SPH_C32(0xc51a3000), SPH_C32(0xc25e00f1), SPH_C32(0x379e42a4),
	  SPH_C32(0x2129cbd4), SPH_C32(0x7d98b000), SPH_C32(0xf7a02a8a),
	  SPH_C32(0x4e007afe), SPH_C32(0x79fd64ab) },
	{ SPH_C32(0x16444000), SPH_C32(0xc75d4a21), SPH_C32(0xef06b67b),
	  SPH_C32(0xbd2871b0), SPH_C32(0x06708000), SPH_C32(0x526fda2d),
	  SPH_C32(0x5382c538), SPH_C32(0x4a93e323) },
	{ SPH_C32(0x62d15000), SPH_C32(0x9d760c5f), SPH_C32(0x67fbab50),
	  SPH_C32(0xa3cef322), SPH_C32(0xcdd98000), SPH_C32(0xc248ed44),
	  SPH_C32(0xe85e313f), SPH_C32(0x9a674c42) },
	{ SPH_C32(0xdded4000), SPH_C32(0x577a7d48), SPH_C32(0x54da427c),
	  SPH_C32(0x6ddcded1), SPH_C32(0xb94c9000), SPH_C32(0x9863ab3a),
	  SPH_C32(0x60a32c14), SPH_C32(0x8481ced0) },
	{ SPH_C32(0xa9785000), SPH_C32(0x0d513b36), SPH_C32(0xdc275f57),
	  SPH_C32(0x733a5c43), SPH_C32(0x72e59000), SPH_C32(0x08449c53),
	  SPH_C32(0xdb7fd813), SPH_C32(0x547561b1) },
	{ SPH_C32(0xff6e6000), SPH_C32(0x730ac6dd), SPH_C32(0xfefc8c2c),
	  SPH_C32(0x80e17494), SPH_C32(0x91238000), SPH_C32(0x7220b4fe),
	  SPH_C32(0x243b2d37), SPH_C32(0xeb7fbde2) },
	{ SPH_C32(0x8bfb7000), SPH_C32(0x292180a3), SPH_C32(0x76019107),
	  SPH_C32(0x9e07f606), SPH_C32(0x5a8a8000), SPH_C32(0xe2078397),
	  SPH_C32(0x9fe7d930), SPH_C32(0x3b8b1283) },
	{ SPH_C32(0x34c76000), SPH_C32(0xe32df1b4), SPH_C32(0x4520782b),
	  SPH_C32(0x5015dbf5), SPH_C32(0x2e1f9000), SPH_C32(0xb82cc5e9),
	  SPH_C32(0x171ac41b), SPH_C32(0x256d9011) },
	{ SPH_C32(0x40527000), SPH_C32(0xb906b7ca), SPH_C32(0xcddd6500),
	  SPH_C32(0x4ef35967), SPH_C32(0xe5b69000), SPH_C32(0x280bf280),
	  SPH_C32(0xacc6301c), SPH_C32(0xf5993f70) },
	{ SPH_C32(0x81174000), SPH_C32(0xe71224f2), SPH_C32(0x98bf5e74),
	  SPH_C32(0x1cc42f71), SPH_C32(0x7809a000), SPH_C32(0xc6773802),
	  SPH_C32(0x35c11760), SPH_C32(0xd6b6b8c6) },
	{ SPH_C32(0xf5825000), SPH_C32(0xbd39628c), SPH_C32(0x1042435f),
	  SPH_C32(0x0222ade3), SPH_C32(0xb3a0a000), SPH_C32(0x56500f6b),
	  SPH_C32(0x8e1de367), SPH_C32(0x064217a7) },
	{ SPH_C32(0x4abe4000), SPH_C32(0x7735139b), SPH_C32(0x2363aa73),
	  SPH_C32(0xcc308010), SPH_C32(0xc735b000), SPH_C32(0x0c7b4915),
	  SPH_C32(0x06e0fe4c), SPH_C32(0x18a49535) },
	{ SPH_C32(0x3e2b5000), SPH_C32(0x2d1e55e5), SPH_C32(0xab9eb758),
	  SPH_C32(0xd2d60282), SPH_C32(0x0c9cb000), SPH_C32(0x9c5c7e7c),
	  SPH_C32(0xbd3c0a4b), SPH_C32(0xc8503a54) },
	{ SPH_C32(0x683d6000), SPH_C32(0x5345a80e), SPH_C32(0x89456423),
	  SPH_C32(0x210d2a55), SPH_C32(0xef5aa000), SPH_C32(0xe63856d1),
	  SPH_C32(0x4278ff6f), SPH_C32(0x775ae607) },
	{ SPH_C32(0x1ca87000), SPH_C32(0x096eee70), SPH_C32(0x01b87908),
	  SPH_C32(0x3feba8c7), SPH_C32(0x24f3a000), SPH_C32(0x761f61b8),
	  SPH_C32(0xf9a40b68), SPH_C32(0xa7ae4966) },
	{ SPH_C32(0xa3946000), SPH_C32(0xc3629f67), SPH_C32(0x32999024),
	  SPH_C32(0xf1f98534), SPH_C32(0x5066b000), SPH_C32(0x2c3427c6),
	  SPH_C32(0x71591643), SPH_C32(0xb948cbf4) },
	{ SPH_C32(0xd7017000), SPH_C32(0x9949d919), SPH_C32(0xba648d0f),
	  SPH_C32(0xef1f07a6), SPH_C32(0x9bcfb000), SPH_C32(0xbc1310af),
	  SPH_C32(0xca85e244), SPH_C32(0x69bc6495) },
	{ SPH_C32(0xe2080000), SPH_C32(0xd7f9a9ec), SPH_C32(0xe679e16a),
	  SPH_C32(0x635fbdfc), SPH_C32(0x146bc000), SPH_C32(0x097803c5),
	  SPH_C32(0xde780a93), SPH_C32(0x84a52f51) },
	{ SPH_C32(0x969d1000), SPH_C32(0x8dd2ef92), SPH_C32(0x6e84fc41),
	  SPH_C32(0x7db93f6e), SPH_C32(0xdfc2c000), SPH_C32(0x995f34ac),
	  SPH_C32(0x65a4fe94), SPH_C32(0x54518030) },
	{ SPH_C32(0x29a10000), SPH_C32(0x47de9e85), SPH_C32(0x5da5156d),
	  SPH_C32(0xb3ab129d), SPH_C32(0xab57d000), SPH_C32(0xc37472d2),
	  SPH_C32(0xed59e3bf), SPH_C32(0x4ab702a2) },
	{ SPH_C32(0x5d341000), SPH_C32(0x1df5d8fb), SPH_C32(0xd5580846),
	  SPH_C32(0xad4d900f), SPH_C32(0x60fed000), SPH_C32(0x535345bb),
	  SPH_C32(0x568517b8), SPH_C32(0x9a43adc3) },
	{ SPH_C32(0x0b222000), SPH_C32(0x63ae2510), SPH_C32(0xf783db3d),
	  SPH_C32(0x5e96b8d8), SPH_C32(0x8338c000), SPH_C32(0x29376d16),
	  SPH_C32(0xa9c1e29c), SPH_C32(0x25497190) },
	{ SPH_C32(0x7fb73000), SPH_C32(0x3985636e), SPH_C32(0x7f7ec616),
	  SPH_C32(0x40703a4a), SPH_C32(0x4891c000), SPH_C32(0xb9105a7f),
	  SPH_C32(0x121d169b), SPH_C32(0xf5bddef1) },
	{ SPH_C32(0xc08b2000), SPH_C32(0xf3891279), SPH_C32(0x4c5f2f3a),
	  SPH_C32(0x8e6217b9), SPH_C32(0x3c04d000), SPH_C32(0xe33b1c01),
	  SPH_C32(0x9ae00bb0), SPH_C32(0xeb5b5c63) },
	{ SPH_C32(0xb41e3000), SPH_C32(0xa9a25407), SPH_C32(0xc4a23211),
	  SPH_C32(0x9084952b), SPH_C32(0xf7add000), SPH_C32(0x731c2b68),
	  SPH_C32(0x213cffb7), SPH_C32(0x3baff302) },
	{ SPH_C32(0x755b0000), SPH_C32(0xf7b6c73f), SPH_C32(0x91c00965),
	  SPH_C32(0xc2b3e33d), SPH_C32(0x6a12e000), SPH_C32(0x9d60e1ea),
	  SPH_C32(0xb83bd8cb), SPH_C32(0x188074b4) },
	{ SPH_C32(0x01ce1000), SPH_C32(0xad9d8141), SPH_C32(0x193d144e),
	  SPH_C32(0xdc5561af), SPH_C32(0xa1bbe000), SPH_C32(0x0d47d683),
	  SPH_C32(0x03e72ccc), SPH_C32(0xc874dbd5) },
	{ SPH_C32(0xbef20000), SPH_C32(0x6791f056), SPH_C32(0x2a1cfd62),
	  SPH_C32(0x12474c5c), SPH_C32(0xd52ef000), SPH_C32(0x576c90fd),
	  SPH_C32(0x8b1a31e7), SPH_C32(0xd6925947) },
	{ SPH_C32(0xca671000), SPH_C32(0x3dbab628), SPH_C32(0xa2e1e049),
	  SPH_C32(0x0ca1cece), SPH_C32(0x1e87f000), SPH_C32(0xc74ba794),
	  SPH_C32(0x30c6c5e0), SPH_C32(0x0666f626) },
	{ SPH_C32(0x9c712000), SPH_C32(0x43e14bc3), SPH_C32(0x803a3332),
	  SPH_C32(0xff7ae619), SPH_C32(0xfd41e000), SPH_C32(0xbd2f8f39),
	  SPH_C32(0xcf8230c4), SPH_C32(0xb96c2a75) },
	{ SPH_C32(0xe8e43000), SPH_C32(0x19ca0dbd), SPH_C32(0x08c72e19),
	  SPH_C32(0xe19c648b), SPH_C32(0x36e8e000), SPH_C32(0x2d08b850),
	  SPH_C32(0x745ec4c3), SPH_C32(0x69988514) },
	{ SPH_C32(0x57d82000), SPH_C32(0xd3c67caa), SPH_C32(0x3be6c735),
	  SPH_C32(0x2f8e4978), SPH_C32(0x427df000), SPH_C32(0x7723fe2e),
	  SPH_C32(0xfca3d9e8), SPH_C32(0x777e0786) },
	{ SPH_C32(0x234d3000), SPH_C32(0x89ed3ad4), SPH_C32(0xb31bda1e),
	  SPH_C32(0x3168cbea), SPH_C32(0x89d4f000), SPH_C32(0xe704c947),
	  SPH_C32(0x477f2def), SPH_C32(0xa78aa8e7) },
	{ SPH_C32(0xf0134000), SPH_C32(0x8cee7004), SPH_C32(0x6b832ec1),
	  SPH_C32(0xad69718e), SPH_C32(0xf23cc000), SPH_C32(0x42cb39e0),
	  SPH_C32(0x5afd9229), SPH_C32(0x94e42f6f) },
	{ SPH_C32(0x84865000), SPH_C32(0xd6c5367a), SPH_C32(0xe37e33ea),
	  SPH_C32(0xb38ff31c), SPH_C32(0x3995c000), SPH_C32(0xd2ec0e89),
	  SPH_C32(0xe121662e), SPH_C32(0x4410800e) },
	{ SPH_C32(0x3bba4000), SPH_C32(0x1cc9476d), SPH_C32(0xd05fdac6),
	  SPH_C32(0x7d9ddeef), SPH_C32(0x4d00d000), SPH_C32(0x88c748f7),
	  SPH_C32(0x69dc7b05), SPH_C32(0x5af6029c) },
	{ SPH_C32(0x4f2f5000), SPH_C32(0x46e20113), SPH_C32(0x58a2c7ed),
	  SPH_C32(0x637b5c7d), SPH_C32(0x86a9d000), SPH_C32(0x18e07f9e),
	  SPH_C32(0xd2008f02), SPH_C32(0x8a02adfd) },
	{ SPH_C32(0x19396000), SPH_C32(0x38b9fcf8), SPH_C32(0x7a791496),
	  SPH_C32(0x90a074aa), SPH_C32(0x656fc000), SPH_C32(0x62845733),
	  SPH_C32(0x2d447a26), SPH_C32(0x350871ae) },
	{ SPH_C32(0x6dac7000), SPH_C32(0x6292ba86), SPH_C32(0xf28409bd),
	  SPH_C32(0x8e46f638), SPH_C32(0xaec6c000), SPH_C32(0xf2a3605a),
	  SPH_C32(0x96988e21), SPH_C32(0xe5fcdecf) },
	{ SPH_C32(0xd2906000), SPH_C32(0xa89ecb91), SPH_C32(0xc1a5e091),
	  SPH_C32(0x4054dbcb), SPH_C32(0xda53d000), SPH_C32(0xa8882624),
	  SPH_C32(0x1e65930a), SPH_C32(0xfb1a5c5d) },
	{ SPH_C32(0xa6057000), SPH_C32(0xf2b58def), SPH_C32(0x4958fdba),
	  SPH_C32(0x5eb25959), SPH_C32(0x11fad000), SPH_C32(0x38af114d),
	  SPH_C32(0xa5b9670d), SPH_C32(0x2beef33c) },
	{ SPH_C32(0x67404000), SPH_C32(0xaca11ed7), SPH_C32(0x1c3ac6ce),
	  SPH_C32(0x0c852f4f), SPH_C32(0x8c45e000), SPH_C32(0xd6d3dbcf),
	  SPH_C32(0x3cbe4071), SPH_C32(0x08c1748a) },
	{ SPH_C32(0x13d55000), SPH_C32(0xf68a58a9), SPH_C32(0x94c7dbe5),
	  SPH_C32(0x1263addd), SPH_C32(0x47ece000), SPH_C32(0x46f4eca6),
	  SPH_C32(0x8762b476), SPH_C32(0xd835dbeb) },
	{ SPH_C32(0xace94000), SPH_C32(0x3c8629be), SPH_C32(0xa7e632c9),
	  SPH_C32(0xdc71802e), SPH_C32(0x3379f000), SPH_C32(0x1cdfaad8),
	  SPH_C32(0x0f9fa95d), SPH_C32(0xc6d35979) },
	{ SPH_C32(0xd87c5000), SPH_C32(0x66ad6fc0), SPH_C32(0x2f1b2fe2),
	  SPH_C32(0xc29702bc), SPH_C32(0xf8d0f000), SPH_C32(0x8cf89db1),
	  SPH_C32(0xb4435d5a), SPH_C32(0x1627f618) },
	{ SPH_C32(0x8e6a6000), SPH_C32(0x18f6922b), SPH_C32(0x0dc0fc99),
	  SPH_C32(0x314c2a6b), SPH_C32(0x1b16e000), SPH_C32(0xf69cb51c),
	  SPH_C32(0x4b07a87e), SPH_C32(0xa92d2a4b) },
	{ SPH_C32(0xfaff7000), SPH_C32(0x42ddd455), SPH_C32(0x853de1b2),
	  SPH_C32(0x2faaa8f9), SPH_C32(0xd0bfe000), SPH_C32(0x66bb8275),
	  SPH_C32(0xf0db5c79), SPH_C32(0x79d9852a) },
	{ SPH_C32(0x45c36000), SPH_C32(0x88d1a542), SPH_C32(0xb61c089e),
	  SPH_C32(0xe1b8850a), SPH_C32(0xa42af000), SPH_C32(0x3c90c40b),
	  SPH_C32(0x78264152), SPH_C32(0x673f07b8) },
	{ SPH_C32(0x31567000), SPH_C32(0xd2fae33c), SPH_C32(0x3ee115b5),
	  SPH_C32(0xff5e0798), SPH_C32(0x6f83f000), SPH_C32(0xacb7f362),
	  SPH_C32(0xc3fab555), SPH_C32(0xb7cba8d9) },
	{ SPH_C32(0xe0278000), SPH_C32(0x19dce008), SPH_C32(0xd7075d82),
	  SPH_C32(0x5ad2e31d), SPH_C32(0xe4788000), SPH_C32(0x859673c1),
	  SPH_C32(0xb5fb2452), SPH_C32(0x29cc5edf) },
	{ SPH_C32(0x94b29000), SPH_C32(0x43f7a676), SPH_C32(0x5ffa40a9),
	  SPH_C32(0x4434618f), SPH_C32(0x2fd18000), SPH_C32(0x15b144a8),
	  SPH_C32(0x0e27d055), SPH_C32(0xf938f1be) },
	{ SPH_C32(0x2b8e8000), SPH_C32(0x89fbd761), SPH_C32(0x6cdba985),
	  SPH_C32(0x8a264c7c), SPH_C32(0x5b449000), SPH_C32(0x4f9a02d6),
	  SPH_C32(0x86dacd7e), SPH_C32(0xe7de732c) },
	{ SPH_C32(0x5f1b9000), SPH_C32(0xd3d0911f), SPH_C32(0xe426b4ae),
	  SPH_C32(0x94c0ceee), SPH_C32(0x90ed9000), SPH_C32(0xdfbd35bf),
	  SPH_C32(0x3d063979), SPH_C32(0x372adc4d) },
	{ SPH_C32(0x090da000), SPH_C32(0xad8b6cf4), SPH_C32(0xc6fd67d5),
	  SPH_C32(0x671be639), SPH_C32(0x732b8000), SPH_C32(0xa5d91d12),
	  SPH_C32(0xc242cc5d), SPH_C32(0x8820001e) },
	{ SPH_C32(0x7d98b000), SPH_C32(0xf7a02a8a), SPH_C32(0x4e007afe),
	  SPH_C32(0x79fd64ab), SPH_C32(0xb8828000), SPH_C32(0x35fe2a7b),
	  SPH_C32(0x799e385a), SPH_C32(0x58d4af7f) },
	{ SPH_C32(0xc2a4a000), SPH_C32(0x3dac5b9d), SPH_C32(0x7d2193d2),
	  SPH_C32(0xb7ef4958), SPH_C32(0xcc179000), SPH_C32(0x6fd56c05),
	  SPH_C32(0xf1632571), SPH_C32(0x46322ded) },
	{ SPH_C32(0xb631b000), SPH_C32(0x67871de3), SPH_C32(0xf5dc8ef9),
	  SPH_C32(0xa909cbca), SPH_C32(0x07be9000), SPH_C32(0xfff25b6c),
	  SPH_C32(0x4abfd176), SPH_C32(0x96c6828c) },
	{ SPH_C32(0x77748000), SPH_C32(0x39938edb), SPH_C32(0xa0beb58d),
	  SPH_C32(0xfb3ebddc), SPH_C32(0x9a01a000), SPH_C32(0x118e91ee),
	  SPH_C32(0xd3b8f60a), SPH_C32(0xb5e9053a) },
	{ SPH_C32(0x03e19000), SPH_C32(0x63b8c8a5), SPH_C32(0x2843a8a6),
	  SPH_C32(0xe5d83f4e), SPH_C32(0x51a8a000), SPH_C32(0x81a9a687),
	  SPH_C32(0x6864020d), SPH_C32(0x651daa5b) },
	{ SPH_C32(0xbcdd8000), SPH_C32(0xa9b4b9b2), SPH_C32(0x1b62418a),
	  SPH_C32(0x2bca12bd), SPH_C32(0x253db000), SPH_C32(0xdb82e0f9),
	  SPH_C32(0xe0991f26), SPH_C32(0x7bfb28c9) },
	{ SPH_C32(0xc8489000), SPH_C32(0xf39fffcc), SPH_C32(0x939f5ca1),
	  SPH_C32(0x352c902f), SPH_C32(0xee94b000), SPH_C32(0x4ba5d790),
	  SPH_C32(0x5b45eb21), SPH_C32(0xab0f87a8) },
	{ SPH_C32(0x9e5ea000), SPH_C32(0x8dc40227), SPH_C32(0xb1448fda),
	  SPH_C32(0xc6f7b8f8), SPH_C32(0x0d52a000), SPH_C32(0x31c1ff3d),
	  SPH_C32(0xa4011e05), SPH_C32(0x14055bfb) },
	{ SPH_C32(0xeacbb000), SPH_C32(0xd7ef4459), SPH_C32(0x39b992f1),
	  SPH_C32(0xd8113a6a), SPH_C32(0xc6fba000), SPH_C32(0xa1e6c854),
	  SPH_C32(0x1fddea02), SPH_C32(0xc4f1f49a) },
	{ SPH_C32(0x55f7a000), SPH_C32(0x1de3354e), SPH_C32(0x0a987bdd),
	  SPH_C32(0x16031799), SPH_C32(0xb26eb000), SPH_C32(0xfbcd8e2a),
	  SPH_C32(0x9720f729), SPH_C32(0xda177608) },
	{ SPH_C32(0x2162b000), SPH_C32(0x47c87330), SPH_C32(0x826566f6),
	  SPH_C32(0x08e5950b), SPH_C32(0x79c7b000), SPH_C32(0x6beab943),
	  SPH_C32(0x2cfc032e), SPH_C32(0x0ae3d969) },
	{ SPH_C32(0xf23cc000), SPH_C32(0x42cb39e0), SPH_C32(0x5afd9229),
	  SPH_C32(0x94e42f6f), SPH_C32(0x022f8000), SPH_C32(0xce2549e4),
	  SPH_C32(0x317ebce8), SPH_C32(0x398d5ee1) },
	{ SPH_C32(0x86a9d000), SPH_C32(0x18e07f9e), SPH_C32(0xd2008f02),
	  SPH_C32(0x8a02adfd), SPH_C32(0xc9868000), SPH_C32(0x5e027e8d),
	  SPH_C32(0x8aa248ef), SPH_C32(0xe979f180) },
	{ SPH_C32(0x3995c000), SPH_C32(0xd2ec0e89), SPH_C32(0xe121662e),
	  SPH_C32(0x4410800e), SPH_C32(0xbd139000), SPH_C32(0x042938f3),
	  SPH_C32(0x025f55c4), SPH_C32(0xf79f7312) },
	{ SPH_C32(0x4d00d000), SPH_C32(0x88c748f7), SPH_C32(0x69dc7b05),
	  SPH_C32(0x5af6029c), SPH_C32(0x76ba9000), SPH_C32(0x940e0f9a),
	  SPH_C32(0xb983a1c3), SPH_C32(0x276bdc73) },
	{ SPH_C32(0x1b16e000), SPH_C32(0xf69cb51c), SPH_C32(0x4b07a87e),
	  SPH_C32(0xa92d2a4b), SPH_C32(0x957c8000), SPH_C32(0xee6a2737),
	  SPH_C32(0x46c754e7), SPH_C32(0x98610020) },
	{ SPH_C32(0x6f83f000), SPH_C32(0xacb7f362), SPH_C32(0xc3fab555),
	  SPH_C32(0xb7cba8d9), SPH_C32(0x5ed58000), SPH_C32(0x7e4d105e),
	  SPH_C32(0xfd1ba0e0), SPH_C32(0x4895af41) },
	{ SPH_C32(0xd0bfe000), SPH_C32(0x66bb8275), SPH_C32(0xf0db5c79),
	  SPH_C32(0x79d9852a), SPH_C32(0x2a409000), SPH_C32(0x24665620),
	  SPH_C32(0x75e6bdcb), SPH_C32(0x56732dd3) },
	{ SPH_C32(0xa42af000), SPH_C32(0x3c90c40b), SPH_C32(0x78264152),
	  SPH_C32(0x673f07b8), SPH_C32(0xe1e99000), SPH_C32(0xb4416149),
	  SPH_C32(0xce3a49cc), SPH_C32(0x868782b2) },
	{ SPH_C32(0x656fc000), SPH_C32(0x62845733), SPH_C32(0x2d447a26),
	  SPH_C32(0x350871ae), SPH_C32(0x7c56a000), SPH_C32(0x5a3dabcb),
	  SPH_C32(0x573d6eb0), SPH_C32(0xa5a80504) },
	{ SPH_C32(0x11fad000), SPH_C32(0x38af114d), SPH_C32(0xa5b9670d),
	  SPH_C32(0x2beef33c), SPH_C32(0xb7ffa000), SPH_C32(0xca1a9ca2),
	  SPH_C32(0xece19ab7), SPH_C32(0x755caa65) },
	{ SPH_C32(0xaec6c000), SPH_C32(0xf2a3605a), SPH_C32(0x96988e21),
	  SPH_C32(0xe5fcdecf), SPH_C32(0xc36ab000), SPH_C32(0x9031dadc),
	  SPH_C32(0x641c879c), SPH_C32(0x6bba28f7) },
	{ SPH_C32(0xda53d000), SPH_C32(0xa8882624), SPH_C32(0x1e65930a),
	  SPH_C32(0xfb1a5c5d), SPH_C32(0x08c3b000), SPH_C32(0x0016edb5),
	  SPH_C32(0xdfc0739b), SPH_C32(0xbb4e8796) },
	{ SPH_C32(0x8c45e000), SPH_C32(0xd6d3dbcf), SPH_C32(0x3cbe4071),
	  SPH_C32(0x08c1748a), SPH_C32(0xeb05a000), SPH_C32(0x7a72c518),
	  SPH_C32(0x208486bf), SPH_C32(0x04445bc5) },
	{ SPH_C32(0xf8d0f000), SPH_C32(0x8cf89db1), SPH_C32(0xb4435d5a),
	  SPH_C32(0x1627f618), SPH_C32(0x20aca000), SPH_C32(0xea55f271),
	  SPH_C32(0x9b5872b8), SPH_C32(0xd4b0f4a4) },
	{ SPH_C32(0x47ece000), SPH_C32(0x46f4eca6), SPH_C32(0x8762b476),
	  SPH_C32(0xd835dbeb), SPH_C32(0x5439b000), SPH_C32(0xb07eb40f),
	  SPH_C32(0x13a56f93), SPH_C32(0xca567636) },
	{ SPH_C32(0x3379f000), SPH_C32(0x1cdfaad8), SPH_C32(0x0f9fa95d),
	  SPH_C32(0xc6d35979), SPH_C32(0x9f90b000), SPH_C32(0x20598366),
	  SPH_C32(0xa8799b94), SPH_C32(0x1aa2d957) },
	{ SPH_C32(0x06708000), SPH_C32(0x526fda2d), SPH_C32(0x5382c538),
	  SPH_C32(0x4a93e323), SPH_C32(0x1034c000), SPH_C32(0x9532900c),
	  SPH_C32(0xbc847343), SPH_C32(0xf7bb9293) },
	{ SPH_C32(0x72e59000), SPH_C32(0x08449c53), SPH_C32(0xdb7fd813),
	  SPH_C32(0x547561b1), SPH_C32(0xdb9dc000), SPH_C32(0x0515a765),
	  SPH_C32(0x07588744), SPH_C32(0x274f3df2) },
	{ SPH_C32(0xcdd98000), SPH_C32(0xc248ed44), SPH_C32(0xe85e313f),
	  SPH_C32(0x9a674c42), SPH_C32(0xaf08d000), SPH_C32(0x5f3ee11b),
	  SPH_C32(0x8fa59a6f), SPH_C32(0x39a9bf60) },
	{ SPH_C32(0xb94c9000), SPH_C32(0x9863ab3a), SPH_C32(0x60a32c14),
	  SPH_C32(0x8481ced0), SPH_C32(0x64a1d000), SPH_C32(0xcf19d672),
	  SPH_C32(0x34796e68), SPH_C32(0xe95d1001) },
	{ SPH_C32(0xef5aa000), SPH_C32(0xe63856d1), SPH_C32(0x4278ff6f),
	  SPH_C32(0x775ae607), SPH_C32(0x8767c000), SPH_C32(0xb57dfedf),
	  SPH_C32(0xcb3d9b4c), SPH_C32(0x5657cc52) },
	{ SPH_C32(0x9bcfb000), SPH_C32(0xbc1310af), SPH_C32(0xca85e244),
	  SPH_C32(0x69bc6495), SPH_C32(0x4ccec000), SPH_C32(0x255ac9b6),
	  SPH_C32(0x70e16f4b), SPH_C32(0x86a36333) },
	{ SPH_C32(0x24f3a000), SPH_C32(0x761f61b8), SPH_C32(0xf9a40b68),
	  SPH_C32(0xa7ae4966), SPH_C32(0x385bd000), SPH_C32(0x7f718fc8),
	  SPH_C32(0xf81c7260), SPH_C32(0x9845e1a1) },
	{ SPH_C32(0x5066b000), SPH_C32(0x2c3427c6), SPH_C32(0x71591643),
	  SPH_C32(0xb948cbf4), SPH_C32(0xf3f2d000), SPH_C32(0xef56b8a1),
	  SPH_C32(0x43c08667), SPH_C32(0x48b14ec0) },
	{ SPH_C32(0x91238000), SPH_C32(0x7220b4fe), SPH_C32(0x243b2d37),
	  SPH_C32(0xeb7fbde2), SPH_C32(0x6e4de000), SPH_C32(0x012a7223),
	  SPH_C32(0xdac7a11b), SPH_C32(0x6b9ec976) },
	{ SPH_C32(0xe5b69000), SPH_C32(0x280bf280), SPH_C32(0xacc6301c),
	  SPH_C32(0xf5993f70), SPH_C32(0xa5e4e000), SPH_C32(0x910d454a),
	  SPH_C32(0x611b551c), SPH_C32(0xbb6a6617) },
	{ SPH_C32(0x5a8a8000), SPH_C32(0xe2078397), SPH_C32(0x9fe7d930),
	  SPH_C32(0x3b8b1283), SPH_C32(0xd171f000), SPH_C32(0xcb260334),
	  SPH_C32(0xe9e64837), SPH_C32(0xa58ce485) },
	{ SPH_C32(0x2e1f9000), SPH_C32(0xb82cc5e9), SPH_C32(0x171ac41b),
	  SPH_C32(0x256d9011), SPH_C32(0x1ad8f000), SPH_C32(0x5b01345d),
	  SPH_C32(0x523abc30), SPH_C32(0x75784be4) },
	{ SPH_C32(0x7809a000), SPH_C32(0xc6773802), SPH_C32(0x35c11760),
	  SPH_C32(0xd6b6b8c6), SPH_C32(0xf91ee000), SPH_C32(0x21651cf0),
	  SPH_C32(0xad7e4914), SPH_C32(0xca7297b7) },
	{ SPH_C32(0x0c9cb000), SPH_C32(0x9c5c7e7c), SPH_C32(0xbd3c0a4b),
	  SPH_C32(0xc8503a54), SPH_C32(0x32b7e000), SPH_C32(0xb1422b99),
	  SPH_C32(0x16a2bd13), SPH_C32(0x1a8638d6) },
	{ SPH_C32(0xb3a0a000), SPH_C32(0x56500f6b), SPH_C32(0x8e1de367),
	  SPH_C32(0x064217a7), SPH_C32(0x4622f000), SPH_C32(0xeb696de7),
	  SPH_C32(0x9e5fa038), SPH_C32(0x0460ba44) },
	{ SPH_C32(0xc735b000), SPH_C32(0x0c7b4915), SPH_C32(0x06e0fe4c),
	  SPH_C32(0x18a49535), SPH_C32(0x8d8bf000), SPH_C32(0x7b4e5a8e),
	  SPH_C32(0x2583543f), SPH_C32(0xd4941525) },
	{ SPH_C32(0x146bc000), SPH_C32(0x097803c5), SPH_C32(0xde780a93),
	  SPH_C32(0x84a52f51), SPH_C32(0xf663c000), SPH_C32(0xde81aa29),
	  SPH_C32(0x3801ebf9), SPH_C32(0xe7fa92ad) },
	{ SPH_C32(0x60fed000), SPH_C32(0x535345bb), SPH_C32(0x568517b8),
	  SPH_C32(0x9a43adc3), SPH_C32(0x3dcac000), SPH_C32(0x4ea69d40),
	  SPH_C32(0x83dd1ffe), SPH_C32(0x370e3dcc) },
	{ SPH_C32(0xdfc2c000), SPH_C32(0x995f34ac), SPH_C32(0x65a4fe94),
	  SPH_C32(0x54518030), SPH_C32(0x495fd000), SPH_C32(0x148ddb3e),
	  SPH_C32(0x0b2002d5), SPH_C32(0x29e8bf5e) },
	{ SPH_C32(0xab57d000), SPH_C32(0xc37472d2), SPH_C32(0xed59e3bf),
	  SPH_C32(0x4ab702a2), SPH_C32(0x82f6d000), SPH_C32(0x84aaec57),
	  SPH_C32(0xb0fcf6d2), SPH_C32(0xf91c103f) },
	{ SPH_C32(0xfd41e000), SPH_C32(0xbd2f8f39), SPH_C32(0xcf8230c4),
	  SPH_C32(0xb96c2a75), SPH_C32(0x6130c000), SPH_C32(0xfecec4fa),
	  SPH_C32(0x4fb803f6), SPH_C32(0x4616cc6c) },
	{ SPH_C32(0x89d4f000), SPH_C32(0xe704c947), SPH_C32(0x477f2def),
	  SPH_C32(0xa78aa8e7), SPH_C32(0xaa99c000), SPH_C32(0x6ee9f393),
	  SPH_C32(0xf464f7f1), SPH_C32(0x96e2630d) },
	{ SPH_C32(0x36e8e000), SPH_C32(0x2d08b850), SPH_C32(0x745ec4c3),
	  SPH_C32(0x69988514), SPH_C32(0xde0cd000), SPH_C32(0x34c2b5ed),
	  SPH_C32(0x7c99eada), SPH_C32(0x8804e19f) },
	{ SPH_C32(0x427df000), SPH_C32(0x7723fe2e), SPH_C32(0xfca3d9e8),
	  SPH_C32(0x777e0786), SPH_C32(0x15a5d000), SPH_C32(0xa4e58284),
	  SPH_C32(0xc7451edd), SPH_C32(0x58f04efe) },
	{ SPH_C32(0x8338c000), SPH_C32(0x29376d16), SPH_C32(0xa9c1e29c),
	  SPH_C32(0x25497190), SPH_C32(0x881ae000), SPH_C32(0x4a994806),
	  SPH_C32(0x5e4239a1), SPH_C32(0x7bdfc948) },
	{ SPH_C32(0xf7add000), SPH_C32(0x731c2b68), SPH_C32(0x213cffb7),
	  SPH_C32(0x3baff302), SPH_C32(0x43b3e000), SPH_C32(0xdabe7f6f),
	  SPH_C32(0xe59ecda6), SPH_C32(0xab2b6629) },
	{ SPH_C32(0x4891c000), SPH_C32(0xb9105a7f), SPH_C32(0x121d169b),
	  SPH_C32(0xf5bddef1), SPH_C32(0x3726f000), SPH_C32(0x80953911),
	  SPH_C32(0x6d63d08d), SPH_C32(0xb5cde4bb) },
	{ SPH_C32(0x3c04d000), SPH_C32(0xe33b1c01), SPH_C32(0x9ae00bb0),
	  SPH_C32(0xeb5b5c63), SPH_C32(0xfc8ff000), SPH_C32(0x10b20e78),
	  SPH_C32(0xd6bf248a), SPH_C32(0x65394bda) },
	{ SPH_C32(0x6a12e000), SPH_C32(0x9d60e1ea), SPH_C32(0xb83bd8cb),
	  SPH_C32(0x188074b4), SPH_C32(0x1f49e000), SPH_C32(0x6ad626d5),
	  SPH_C32(0x29fbd1ae), SPH_C32(0xda339789) },
	{ SPH_C32(0x1e87f000), SPH_C32(0xc74ba794), SPH_C32(0x30c6c5e0),
	  SPH_C32(0x0666f626), SPH_C32(0xd4e0e000), SPH_C32(0xfaf111bc),
	  SPH_C32(0x922725a9), SPH_C32(0x0ac738e8) },
	{ SPH_C32(0xa1bbe000), SPH_C32(0x0d47d683), SPH_C32(0x03e72ccc),
	  SPH_C32(0xc874dbd5), SPH_C32(0xa075f000), SPH_C32(0xa0da57c2),
	  SPH_C32(0x1ada3882), SPH_C32(0x1421ba7a) },
	{ SPH_C32(0xd52ef000), SPH_C32(0x576c90fd), SPH_C32(0x8b1a31e7),
	  SPH_C32(0xd6925947), SPH_C32(0x6bdcf000), SPH_C32(0x30fd60ab),
	  SPH_C32(0xa106cc85), SPH_C32(0xc4d5151b) }
};

static const sph_u32 T256_8[256][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xb7a40100), SPH_C32(0x8a1f31d8), SPH_C32(0x8589d8ab),
	  SPH_C32(0xe6c46464), SPH_C32(0x734c0000), SPH_C32(0x956fa7d6),
	  SPH_C32(0xa29d1297), SPH_C32(0x6ee56854) },
	{ SPH_C32(0x734c0000), SPH_C32(0x956fa7d6), SPH_C32(0xa29d1297),
	  SPH_C32(0x6ee56854), SPH_C32(0xc4e80100), SPH_C32(0x1f70960e),
	  SPH_C32(0x2714ca3c), SPH_C32(0x88210c30) },
	{ SPH_C32(0xc4e80100), SPH_C32(0x1f70960e), SPH_C32(0x2714ca3c),
	  SPH_C32(0x88210c30), SPH_C32(0xb7a40100), SPH_C32(0x8a1f31d8),
	  SPH_C32(0x8589d8ab), SPH_C32(0xe6c46464) },
	{ SPH_C32(0xa7b80200), SPH_C32(0x1f128433), SPH_C32(0x60e5f9f2),
	  SPH_C32(0x9e147576), SPH_C32(0xee260000), SPH_C32(0x124b683e),
	  SPH_C32(0x80c2d68f), SPH_C32(0x3bf3ab2c) },
	{ SPH_C32(0x101c0300), SPH_C32(0x950db5eb), SPH_C32(0xe56c2159),
	  SPH_C32(0x78d01112), SPH_C32(0x9d6a0000), SPH_C32(0x8724cfe8),
	  SPH_C32(0x225fc418), SPH_C32(0x5516c378) },
	{ SPH_C32(0xd4f40200), SPH_C32(0x8a7d23e5), SPH_C32(0xc278eb65),
	  SPH_C32(0xf0f11d22), SPH_C32(0x2ace0100), SPH_C32(0x0d3bfe30),
	  SPH_C32(0xa7d61cb3), SPH_C32(0xb3d2a71c) },
	{ SPH_C32(0x63500300), SPH_C32(0x0062123d), SPH_C32(0x47f133ce),
	  SPH_C32(0x16357946), SPH_C32(0x59820100), SPH_C32(0x985459e6),
	  SPH_C32(0x054b0e24), SPH_C32(0xdd37cf48) },
	{ SPH_C32(0xee260000), SPH_C32(0x124b683e), SPH_C32(0x80c2d68f),
	  SPH_C32(0x3bf3ab2c), SPH_C32(0x499e0200), SPH_C32(0x0d59ec0d),
	  SPH_C32(0xe0272f7d), SPH_C32(0xa5e7de5a) },
	{ SPH_C32(0x59820100), SPH_C32(0x985459e6), SPH_C32(0x054b0e24),
	  SPH_C32(0xdd37cf48), SPH_C32(0x3ad20200), SPH_C32(0x98364bdb),
	  SPH_C32(0x42ba3dea), SPH_C32(0xcb02b60e) },
	{ SPH_C32(0x9d6a0000), SPH_C32(0x8724cfe8), SPH_C32(0x225fc418),
	  SPH_C32(0x5516c378), SPH_C32(0x8d760300), SPH_C32(0x12297a03),
	  SPH_C32(0xc733e541), SPH_C32(0x2dc6d26a) },
	{ SPH_C32(0x2ace0100), SPH_C32(0x0d3bfe30), SPH_C32(0xa7d61cb3),
	  SPH_C32(0xb3d2a71c), SPH_C32(0xfe3a0300), SPH_C32(0x8746ddd5),
	  SPH_C32(0x65aef7d6), SPH_C32(0x4323ba3e) },
	{ SPH_C32(0x499e0200), SPH_C32(0x0d59ec0d), SPH_C32(0xe0272f7d),
	  SPH_C32(0xa5e7de5a), SPH_C32(0xa7b80200), SPH_C32(0x1f128433),
	  SPH_C32(0x60e5f9f2), SPH_C32(0x9e147576) },
	{ SPH_C32(0xfe3a0300), SPH_C32(0x8746ddd5), SPH_C32(0x65aef7d6),
	  SPH_C32(0x4323ba3e), SPH_C32(0xd4f40200), SPH_C32(0x8a7d23e5),
	  SPH_C32(0xc278eb65), SPH_C32(0xf0f11d22) },
	{ SPH_C32(0x3ad20200), SPH_C32(0x98364bdb), SPH_C32(0x42ba3dea),
	  SPH_C32(0xcb02b60e), SPH_C32(0x63500300), SPH_C32(0x0062123d),
	  SPH_C32(0x47f133ce), SPH_C32(0x16357946) },
	{ SPH_C32(0x8d760300), SPH_C32(0x12297a03), SPH_C32(0xc733e541),
	  SPH_C32(0x2dc6d26a), SPH_C32(0x101c0300), SPH_C32(0x950db5eb),
	  SPH_C32(0xe56c2159), SPH_C32(0x78d01112) },
	{ SPH_C32(0x8f3e0400), SPH_C32(0x0d9dc877), SPH_C32(0x6fc548e1),
	  SPH_C32(0x898d2cd6), SPH_C32(0x14bd0000), SPH_C32(0x2fba37ff),
	  SPH_C32(0x6a72e5bb), SPH_C32(0x247febe6) },
	{ SPH_C32(0x389a0500), SPH_C32(0x8782f9af), SPH_C32(0xea4c904a),
	  SPH_C32(0x6f4948b2), SPH_C32(0x67f10000), SPH_C32(0xbad59029),
	  SPH_C32(0xc8eff72c), SPH_C32(0x4a9a83b2) },
	{ SPH_C32(0xfc720400), SPH_C32(0x98f26fa1), SPH_C32(0xcd585a76),
	  SPH_C32(0xe7684482), SPH_C32(0xd0550100), SPH_C32(0x30caa1f1),
	  SPH_C32(0x4d662f87), SPH_C32(0xac5ee7d6) },
	{ SPH_C32(0x4bd60500), SPH_C32(0x12ed5e79), SPH_C32(0x48d182dd),
	  SPH_C32(0x01ac20e6), SPH_C32(0xa3190100), SPH_C32(0xa5a50627),
	  SPH_C32(0xeffb3d10), SPH_C32(0xc2bb8f82) },
	{ SPH_C32(0x28860600), SPH_C32(0x128f4c44), SPH_C32(0x0f20b113),
	  SPH_C32(0x179959a0), SPH_C32(0xfa9b0000), SPH_C32(0x3df15fc1),
	  SPH_C32(0xeab03334), SPH_C32(0x1f8c40ca) },
	{ SPH_C32(0x9f220700), SPH_C32(0x98907d9c), SPH_C32(0x8aa969b8),
	  SPH_C32(0xf15d3dc4), SPH_C32(0x89d70000), SPH_C32(0xa89ef817),
	  SPH_C32(0x482d21a3), SPH_C32(0x7169289e) },
	{ SPH_C32(0x5bca0600), SPH_C32(0x87e0eb92), SPH_C32(0xadbda384),
	  SPH_C32(0x797c31f4), SPH_C32(0x3e730100), SPH_C32(0x2281c9cf),
	  SPH_C32(0xcda4f908), SPH_C32(0x97ad4cfa) },
	{ SPH_C32(0xec6e0700), SPH_C32(0x0dffda4a), SPH_C32(0x28347b2f),
	  SPH_C32(0x9fb85590), SPH_C32(0x4d3f0100), SPH_C32(0xb7ee6e19),
	  SPH_C32(0x6f39eb9f), SPH_C32(0xf94824ae) },
	{ SPH_C32(0x61180400), SPH_C32(0x1fd6a049), SPH_C32(0xef079e6e),
	  SPH_C32(0xb27e87fa), SPH_C32(0x5d230200), SPH_C32(0x22e3dbf2),
	  SPH_C32(0x8a55cac6), SPH_C32(0x819835bc) },
	{ SPH_C32(0xd6bc0500), SPH_C32(0x95c99191), SPH_C32(0x6a8e46c5),
	  SPH_C32(0x54bae39e), SPH_C32(0x2e6f0200), SPH_C32(0xb78c7c24),
	  SPH_C32(0x28c8d851), SPH_C32(0xef7d5de8) },
	{ SPH_C32(0x12540400), SPH_C32(0x8ab9079f), SPH_C32(0x4d9a8cf9),
	  SPH_C32(0xdc9befae), SPH_C32(0x99cb0300), SPH_C32(0x3d934dfc),
	  SPH_C32(0xad4100fa), SPH_C32(0x09b9398c) },
	{ SPH_C32(0xa5f00500), SPH_C32(0x00a63647), SPH_C32(0xc8135452),
	  SPH_C32(0x3a5f8bca), SPH_C32(0xea870300), SPH_C32(0xa8fcea2a),
	  SPH_C32(0x0fdc126d), SPH_C32(0x675c51d8) },
	{ SPH_C32(0xc6a00600), SPH_C32(0x00c4247a), SPH_C32(0x8fe2679c),
	  SPH_C32(0x2c6af28c), SPH_C32(0xb3050200), SPH_C32(0x30a8b3cc),
	  SPH_C32(0x0a971c49), SPH_C32(0xba6b9e90) },
	{ SPH_C32(0x71040700), SPH_C32(0x8adb15a2), SPH_C32(0x0a6bbf37),
	  SPH_C32(0xcaae96e8), SPH_C32(0xc0490200), SPH_C32(0xa5c7141a),
	  SPH_C32(0xa80a0ede), SPH_C32(0xd48ef6c4) },
	{ SPH_C32(0xb5ec0600), SPH_C32(0x95ab83ac), SPH_C32(0x2d7f750b),
	  SPH_C32(0x428f9ad8), SPH_C32(0x77ed0300), SPH_C32(0x2fd825c2),
	  SPH_C32(0x2d83d675), SPH_C32(0x324a92a0) },
	{ SPH_C32(0x02480700), SPH_C32(0x1fb4b274), SPH_C32(0xa8f6ada0),
	  SPH_C32(0xa44bfebc), SPH_C32(0x04a10300), SPH_C32(0xbab78214),
	  SPH_C32(0x8f1ec4e2), SPH_C32(0x5caffaf4) },
	{ SPH_C32(0x14bd0000), SPH_C32(0x2fba37ff), SPH_C32(0x6a72e5bb),
	  SPH_C32(0x247febe6), SPH_C32(0x9b830400), SPH_C32(0x2227ff88),
	  SPH_C32(0x05b7ad5a), SPH_C32(0xadf2c730) },
	{ SPH_C32(0xa3190100), SPH_C32(0xa5a50627), SPH_C32(0xeffb3d10),
	  SPH_C32(0xc2bb8f82), SPH_C32(0xe8cf0400), SPH_C32(0xb748585e),
	  SPH_C32(0xa72abfcd), SPH_C32(0xc317af64) },
	{ SPH_C32(0x67f10000), SPH_C32(0xbad59029), SPH_C32(0xc8eff72c),
	  SPH_C32(0x4a9a83b2), SPH_C32(0x5f6b0500), SPH_C32(0x3d576986),
	  SPH_C32(0x22a36766), SPH_C32(0x25d3cb00) },
	{ SPH_C32(0xd0550100), SPH_C32(0x30caa1f1), SPH_C32(0x4d662f87),
	  SPH_C32(0xac5ee7d6), SPH_C32(0x2c270500), SPH_C32(0xa838ce50),
	  SPH_C32(0x803e75f1), SPH_C32(0x4b36a354) },
	{ SPH_C32(0xb3050200), SPH_C32(0x30a8b3cc), SPH_C32(0x0a971c49),
	  SPH_C32(0xba6b9e90), SPH_C32(0x75a50400), SPH_C32(0x306c97b6),
	  SPH_C32(0x85757bd5), SPH_C32(0x96016c1c) },
	{ SPH_C32(0x04a10300), SPH_C32(0xbab78214), SPH_C32(0x8f1ec4e2),
	  SPH_C32(0x5caffaf4), SPH_C32(0x06e90400), SPH_C32(0xa5033060),
	  SPH_C32(0x27e86942), SPH_C32(0xf8e40448) },
	{ SPH_C32(0xc0490200), SPH_C32(0xa5c7141a), SPH_C32(0xa80a0ede),
	  SPH_C32(0xd48ef6c4), SPH_C32(0xb14d0500), SPH_C32(0x2f1c01b8),
	  SPH_C32(0xa261b1e9), SPH_C32(0x1e20602c) },
	{ SPH_C32(0x77ed0300), SPH_C32(0x2fd825c2), SPH_C32(0x2d83d675),
	  SPH_C32(0x324a92a0), SPH_C32(0xc2010500), SPH_C32(0xba73a66e),
	  SPH_C32(0x00fca37e), SPH_C32(0x70c50878) },
	{ SPH_C32(0xfa9b0000), SPH_C32(0x3df15fc1), SPH_C32(0xeab03334),
	  SPH_C32(0x1f8c40ca), SPH_C32(0xd21d0600), SPH_C32(0x2f7e1385),
	  SPH_C32(0xe5908227), SPH_C32(0x0815196a) },
	{ SPH_C32(0x4d3f0100), SPH_C32(0xb7ee6e19), SPH_C32(0x6f39eb9f),
	  SPH_C32(0xf94824ae), SPH_C32(0xa1510600), SPH_C32(0xba11b453),
	  SPH_C32(0x470d90b0), SPH_C32(0x66f0713e) },
	{ SPH_C32(0x89d70000), SPH_C32(0xa89ef817), SPH_C32(0x482d21a3),
	  SPH_C32(0x7169289e), SPH_C32(0x16f50700), SPH_C32(0x300e858b),
	  SPH_C32(0xc284481b), SPH_C32(0x8034155a) },
	{ SPH_C32(0x3e730100), SPH_C32(0x2281c9cf), SPH_C32(0xcda4f908),
	  SPH_C32(0x97ad4cfa), SPH_C32(0x65b90700), SPH_C32(0xa561225d),
	  SPH_C32(0x60195a8c), SPH_C32(0xeed17d0e) },
	{ SPH_C32(0x5d230200), SPH_C32(0x22e3dbf2), SPH_C32(0x8a55cac6),
	  SPH_C32(0x819835bc), SPH_C32(0x3c3b0600), SPH_C32(0x3d357bbb),
	  SPH_C32(0x655254a8), SPH_C32(0x33e6b246) },
	{ SPH_C32(0xea870300), SPH_C32(0xa8fcea2a), SPH_C32(0x0fdc126d),
	  SPH_C32(0x675c51d8), SPH_C32(0x4f770600), SPH_C32(0xa85adc6d),
	  SPH_C32(0xc7cf463f), SPH_C32(0x5d03da12) },
	{ SPH_C32(0x2e6f0200), SPH_C32(0xb78c7c24), SPH_C32(0x28c8d851),
	  SPH_C32(0xef7d5de8), SPH_C32(0xf8d30700), SPH_C32(0x2245edb5),
	  SPH_C32(0x42469e94), SPH_C32(0xbbc7be76) },
	{ SPH_C32(0x99cb0300), SPH_C32(0x3d934dfc), SPH_C32(0xad4100fa),
	  SPH_C32(0x09b9398c), SPH_C32(0x8b9f0700), SPH_C32(0xb72a4a63),
	  SPH_C32(0xe0db8c03), SPH_C32(0xd522d622) },
	{ SPH_C32(0x9b830400), SPH_C32(0x2227ff88), SPH_C32(0x05b7ad5a),
	  SPH_C32(0xadf2c730), SPH_C32(0x8f3e0400), SPH_C32(0x0d9dc877),
	  SPH_C32(0x6fc548e1), SPH_C32(0x898d2cd6) },
	{ SPH_C32(0x2c270500), SPH_C32(0xa838ce50), SPH_C32(0x803e75f1),
	  SPH_C32(0x4b36a354), SPH_C32(0xfc720400), SPH_C32(0x98f26fa1),
	  SPH_C32(0xcd585a76), SPH_C32(0xe7684482) },
	{ SPH_C32(0xe8cf0400), SPH_C32(0xb748585e), SPH_C32(0xa72abfcd),
	  SPH_C32(0xc317af64), SPH_C32(0x4bd60500), SPH_C32(0x12ed5e79),
	  SPH_C32(0x48d182dd), SPH_C32(0x01ac20e6) },
	{ SPH_C32(0x5f6b0500), SPH_C32(0x3d576986), SPH_C32(0x22a36766),
	  SPH_C32(0x25d3cb00), SPH_C32(0x389a0500), SPH_C32(0x8782f9af),
	  SPH_C32(0xea4c904a), SPH_C32(0x6f4948b2) },
	{ SPH_C32(0x3c3b0600), SPH_C32(0x3d357bbb), SPH_C32(0x655254a8),
	  SPH_C32(0x33e6b246), SPH_C32(0x61180400), SPH_C32(0x1fd6a049),
	  SPH_C32(0xef079e6e), SPH_C32(0xb27e87fa) },
	{ SPH_C32(0x8b9f0700), SPH_C32(0xb72a4a63), SPH_C32(0xe0db8c03),
	  SPH_C32(0xd522d622), SPH_C32(0x12540400), SPH_C32(0x8ab9079f),
	  SPH_C32(0x4d9a8cf9), SPH_C32(0xdc9befae) },
	{ SPH_C32(0x4f770600), SPH_C32(0xa85adc6d), SPH_C32(0xc7cf463f),
	  SPH_C32(0x5d03da12), SPH_C32(0xa5f00500), SPH_C32(0x00a63647),
	  SPH_C32(0xc8135452), SPH_C32(0x3a5f8bca) },
	{ SPH_C32(0xf8d30700), SPH_C32(0x2245edb5), SPH_C32(0x42469e94),
	  SPH_C32(0xbbc7be76), SPH_C32(0xd6bc0500), SPH_C32(0x95c99191),
	  SPH_C32(0x6a8e46c5), SPH_C32(0x54bae39e) },
	{ SPH_C32(0x75a50400), SPH_C32(0x306c97b6), SPH_C32(0x85757bd5),
	  SPH_C32(0x96016c1c), SPH_C32(0xc6a00600), SPH_C32(0x00c4247a),
	  SPH_C32(0x8fe2679c), SPH_C32(0x2c6af28c) },
	{ SPH_C32(0xc2010500), SPH_C32(0xba73a66e), SPH_C32(0x00fca37e),
	  SPH_C32(0x70c50878), SPH_C32(0xb5ec0600), SPH_C32(0x95ab83ac),
	  SPH_C32(0x2d7f750b), SPH_C32(0x428f9ad8) },
	{ SPH_C32(0x06e90400), SPH_C32(0xa5033060), SPH_C32(0x27e86942),
	  SPH_C32(0xf8e40448), SPH_C32(0x02480700), SPH_C32(0x1fb4b274),
	  SPH_C32(0xa8f6ada0), SPH_C32(0xa44bfebc) },
	{ SPH_C32(0xb14d0500), SPH_C32(0x2f1c01b8), SPH_C32(0xa261b1e9),
	  SPH_C32(0x1e20602c), SPH_C32(0x71040700), SPH_C32(0x8adb15a2),
	  SPH_C32(0x0a6bbf37), SPH_C32(0xcaae96e8) },
	{ SPH_C32(0xd21d0600), SPH_C32(0x2f7e1385), SPH_C32(0xe5908227),
	  SPH_C32(0x0815196a), SPH_C32(0x28860600), SPH_C32(0x128f4c44),
	  SPH_C32(0x0f20b113), SPH_C32(0x179959a0) },
	{ SPH_C32(0x65b90700), SPH_C32(0xa561225d), SPH_C32(0x60195a8c),
	  SPH_C32(0xeed17d0e), SPH_C32(0x5bca0600), SPH_C32(0x87e0eb92),
	  SPH_C32(0xadbda384), SPH_C32(0x797c31f4) },
	{ SPH_C32(0xa1510600), SPH_C32(0xba11b453), SPH_C32(0x470d90b0),
	  SPH_C32(0x66f0713e), SPH_C32(0xec6e0700), SPH_C32(0x0dffda4a),
	  SPH_C32(0x28347b2f), SPH_C32(0x9fb85590) },
	{ SPH_C32(0x16f50700), SPH_C32(0x300e858b), SPH_C32(0xc284481b),
	  SPH_C32(0x8034155a), SPH_C32(0x9f220700), SPH_C32(0x98907d9c),
	  SPH_C32(0x8aa969b8), SPH_C32(0xf15d3dc4) },
	{ SPH_C32(0xde320800), SPH_C32(0x288350fe), SPH_C32(0x71852ac7),
	  SPH_C32(0xa6bf9f96), SPH_C32(0xe18b0000), SPH_C32(0x5459887d),
	  SPH_C32(0xbf1283d3), SPH_C32(0x1b666a73) },
	{ SPH_C32(0x69960900), SPH_C32(0xa29c6126), SPH_C32(0xf40cf26c),
	  SPH_C32(0x407bfbf2), SPH_C32(0x92c70000), SPH_C32(0xc1362fab),
	  SPH_C32(0x1d8f9144), SPH_C32(0x75830227) },
	{ SPH_C32(0xad7e0800), SPH_C32(0xbdecf728), SPH_C32(0xd3183850),
	  SPH_C32(0xc85af7c2), SPH_C32(0x25630100), SPH_C32(0x4b291e73),
	  SPH_C32(0x980649ef), SPH_C32(0x93476643) },
	{ SPH_C32(0x1ada0900), SPH_C32(0x37f3c6f0), SPH_C32(0x5691e0fb),
	  SPH_C32(0x2e9e93a6), SPH_C32(0x562f0100), SPH_C32(0xde46b9a5),
	  SPH_C32(0x3a9b5b78), SPH_C32(0xfda20e17) },
	{ SPH_C32(0x798a0a00), SPH_C32(0x3791d4cd), SPH_C32(0x1160d335),
	  SPH_C32(0x38abeae0), SPH_C32(0x0fad0000), SPH_C32(0x4612e043),
	  SPH_C32(0x3fd0555c), SPH_C32(0x2095c15f) },
	{ SPH_C32(0xce2e0b00), SPH_C32(0xbd8ee515), SPH_C32(0x94e90b9e),
	  SPH_C32(0xde6f8e84), SPH_C32(0x7ce10000), SPH_C32(0xd37d4795),
	  SPH_C32(0x9d4d47cb), SPH_C32(0x4e70a90b) },
	{ SPH_C32(0x0ac60a00), SPH_C32(0xa2fe731b), SPH_C32(0xb3fdc1a2),
	  SPH_C32(0x564e82b4), SPH_C32(0xcb450100), SPH_C32(0x5962764d),
	  SPH_C32(0x18c49f60), SPH_C32(0xa8b4cd6f) },
	{ SPH_C32(0xbd620b00), SPH_C32(0x28e142c3), SPH_C32(0x36741909),
	  SPH_C32(0xb08ae6d0), SPH_C32(0xb8090100), SPH_C32(0xcc0dd19b),
	  SPH_C32(0xba598df7), SPH_C32(0xc651a53b) },
	{ SPH_C32(0x30140800), SPH_C32(0x3ac838c0), SPH_C32(0xf147fc48),
	  SPH_C32(0x9d4c34ba), SPH_C32(0xa8150200), SPH_C32(0x59006470),
	  SPH_C32(0x5f35acae), SPH_C32(0xbe81b429) },
	{ SPH_C32(0x87b00900), SPH_C32(0xb0d70918), SPH_C32(0x74ce24e3),
	  SPH_C32(0x7b8850de), SPH_C32(0xdb590200), SPH_C32(0xcc6fc3a6),
	  SPH_C32(0xfda8be39), SPH_C32(0xd064dc7d) },
	{ SPH_C32(0x43580800), SPH_C32(0xafa79f16), SPH_C32(0x53daeedf),
	  SPH_C32(0xf3a95cee), SPH_C32(0x6cfd0300), SPH_C32(0x4670f27e),
	  SPH_C32(0x78216692), SPH_C32(0x36a0b819) },
	{ SPH_C32(0xf4fc0900), SPH_C32(0x25b8aece), SPH_C32(0xd6533674),
	  SPH_C32(0x156d388a), SPH_C32(0x1fb10300), SPH_C32(0xd31f55a8),
	  SPH_C32(0xdabc7405), SPH_C32(0x5845d04d) },
	{ SPH_C32(0x97ac0a00), SPH_C32(0x25dabcf3), SPH_C32(0x91a205ba),
	  SPH_C32(0x035841cc), SPH_C32(0x46330200), SPH_C32(0x4b4b0c4e),
	  SPH_C32(0xdff77a21), SPH_C32(0x85721f05) },
	{ SPH_C32(0x20080b00), SPH_C32(0xafc58d2b), SPH_C32(0x142bdd11),
	  SPH_C32(0xe59c25a8), SPH_C32(0x357f0200), SPH_C32(0xde24ab98),
	  SPH_C32(0x7d6a68b6), SPH_C32(0xeb977751) },
	{ SPH_C32(0xe4e00a00), SPH_C32(0xb0b51b25), SPH_C32(0x333f172d),
	  SPH_C32(0x6dbd2998), SPH_C32(0x82db0300), SPH_C32(0x543b9a40),
	  SPH_C32(0xf8e3b01d), SPH_C32(0x0d531335) },
	{ SPH_C32(0x53440b00), SPH_C32(0x3aaa2afd), SPH_C32(0xb6b6cf86),
	  SPH_C32(0x8b794dfc), SPH_C32(0xf1970300), SPH_C32(0xc1543d96),
	  SPH_C32(0x5a7ea28a), SPH_C32(0x63b67b61) },
	{ SPH_C32(0x510c0c00), SPH_C32(0x251e9889), SPH_C32(0x1e406226),
	  SPH_C32(0x2f32b340), SPH_C32(0xf5360000), SPH_C32(0x7be3bf82),
	  SPH_C32(0xd5606668), SPH_C32(0x3f198195) },
	{ SPH_C32(0xe6a80d00), SPH_C32(0xaf01a951), SPH_C32(0x9bc9ba8d),
	  SPH_C32(0xc9f6d724), SPH_C32(0x867a0000), SPH_C32(0xee8c1854),
	  SPH_C32(0x77fd74ff), SPH_C32(0x51fce9c1) },
	{ SPH_C32(0x22400c00), SPH_C32(0xb0713f5f), SPH_C32(0xbcdd70b1),
	  SPH_C32(0x41d7db14), SPH_C32(0x31de0100), SPH_C32(0x6493298c),
	  SPH_C32(0xf274ac54), SPH_C32(0xb7388da5) },
	{ SPH_C32(0x95e40d00), SPH_C32(0x3a6e0e87), SPH_C32(0x3954a81a),
	  SPH_C32(0xa713bf70), SPH_C32(0x42920100), SPH_C32(0xf1fc8e5a),
	  SPH_C32(0x50e9bec3), SPH_C32(0xd9dde5f1) },
	{ SPH_C32(0xf6b40e00), SPH_C32(0x3a0c1cba), SPH_C32(0x7ea59bd4),
	  SPH_C32(0xb126c636), SPH_C32(0x1b100000), SPH_C32(0x69a8d7bc),
	  SPH_C32(0x55a2b0e7), SPH_C32(0x04ea2ab9) },
	{ SPH_C32(0x41100f00), SPH_C32(0xb0132d62), SPH_C32(0xfb2c437f),
	  SPH_C32(0x57e2a252), SPH_C32(0x685c0000), SPH_C32(0xfcc7706a),
	  SPH_C32(0xf73fa270), SPH_C32(0x6a0f42ed) },
	{ SPH_C32(0x85f80e00), SPH_C32(0xaf63bb6c), SPH_C32(0xdc388943),
	  SPH_C32(0xdfc3ae62), SPH_C32(0xdff80100), SPH_C32(0x76d841b2),
	  SPH_C32(0x72b67adb), SPH_C32(0x8ccb2689) },
	{ SPH_C32(0x325c0f00), SPH_C32(0x257c8ab4), SPH_C32(0x59b151e8),
	  SPH_C32(0x3907ca06), SPH_C32(0xacb40100), SPH_C32(0xe3b7e664),
	  SPH_C32(0xd02b684c), SPH_C32(0xe22e4edd) },
	{ SPH_C32(0xbf2a0c00), SPH_C32(0x3755f0b7), SPH_C32(0x9e82b4a9),
	  SPH_C32(0x14c1186c), SPH_C32(0xbca80200), SPH_C32(0x76ba538f),
	  SPH_C32(0x35474915), SPH_C32(0x9afe5fcf) },
	{ SPH_C32(0x088e0d00), SPH_C32(0xbd4ac16f), SPH_C32(0x1b0b6c02),
	  SPH_C32(0xf2057c08), SPH_C32(0xcfe40200), SPH_C32(0xe3d5f459),
	  SPH_C32(0x97da5b82), SPH_C32(0xf41b379b) },
	{ SPH_C32(0xcc660c00), SPH_C32(0xa23a5761), SPH_C32(0x3c1fa63e),
	  SPH_C32(0x7a247038), SPH_C32(0x78400300), SPH_C32(0x69cac581),
	  SPH_C32(0x12538329), SPH_C32(0x12df53ff) },
	{ SPH_C32(0x7bc20d00), SPH_C32(0x282566b9), SPH_C32(0xb9967e95),
	  SPH_C32(0x9ce0145c), SPH_C32(0x0b0c0300), SPH_C32(0xfca56257),
	  SPH_C32(0xb0ce91be), SPH_C32(0x7c3a3bab) },
	{ SPH_C32(0x18920e00), SPH_C32(0x28477484), SPH_C32(0xfe674d5b),
	  SPH_C32(0x8ad56d1a), SPH_C32(0x528e0200), SPH_C32(0x64f13bb1),
	  SPH_C32(0xb5859f9a), SPH_C32(0xa10df4e3) },
	{ SPH_C32(0xaf360f00), SPH_C32(0xa258455c), SPH_C32(0x7bee95f0),
	  SPH_C32(0x6c11097e), SPH_C32(0x21c20200), SPH_C32(0xf19e9c67),
	  SPH_C32(0x17188d0d), SPH_C32(0xcfe89cb7) },
	{ SPH_C32(0x6bde0e00), SPH_C32(0xbd28d352), SPH_C32(0x5cfa5fcc),
	  SPH_C32(0xe430054e), SPH_C32(0x96660300), SPH_C32(0x7b81adbf),
	  SPH_C32(0x929155a6), SPH_C32(0x292cf8d3) },
	{ SPH_C32(0xdc7a0f00), SPH_C32(0x3737e28a), SPH_C32(0xd9738767),
	  SPH_C32(0x02f4612a), SPH_C32(0xe52a0300), SPH_C32(0xeeee0a69),
	  SPH_C32(0x300c4731), SPH_C32(0x47c99087) },
	{ SPH_C32(0xca8f0800), SPH_C32(0x07396701), SPH_C32(0x1bf7cf7c),
	  SPH_C32(0x82c07470), SPH_C32(0x7a080400), SPH_C32(0x767e77f5),
	  SPH_C32(0xbaa52e89), SPH_C32(0xb694ad43) },
	{ SPH_C32(0x7d2b0900), SPH_C32(0x8d2656d9), SPH_C32(0x9e7e17d7),
	  SPH_C32(0x64041014), SPH_C32(0x09440400), SPH_C32(0xe311d023),
	  SPH_C32(0x18383c1e), SPH_C32(0xd871c517) },
	{ SPH_C32(0xb9c30800), SPH_C32(0x9256c0d7), SPH_C32(0xb96addeb),
	  SPH_C32(0xec251c24), SPH_C32(0xbee00500), SPH_C32(0x690ee1fb),
	  SPH_C32(0x9db1e4b5), SPH_C32(0x3eb5a173) },
	{ SPH_C32(0x0e670900), SPH_C32(0x1849f10f), SPH_C32(0x3ce30540),
	  SPH_C32(0x0ae17840), SPH_C32(0xcdac0500), SPH_C32(0xfc61462d),
	  SPH_C32(0x3f2cf622), SPH_C32(0x5050c927) },
	{ SPH_C32(0x6d370a00), SPH_C32(0x182be332), SPH_C32(0x7b12368e),
	  SPH_C32(0x1cd40106), SPH_C32(0x942e0400), SPH_C32(0x64351fcb),
	  SPH_C32(0x3a67f806), SPH_C32(0x8d67066f) },
	{ SPH_C32(0xda930b00), SPH_C32(0x9234d2ea), SPH_C32(0xfe9bee25),
	  SPH_C32(0xfa106562), SPH_C32(0xe7620400), SPH_C32(0xf15ab81d),
	  SPH_C32(0x98faea91), SPH_C32(0xe3826e3b) },
	{ SPH_C32(0x1e7b0a00), SPH_C32(0x8d4444e4), SPH_C32(0xd98f2419),
	  SPH_C32(0x72316952), SPH_C32(0x50c60500), SPH_C32(0x7b4589c5),
	  SPH_C32(0x1d73323a), SPH_C32(0x05460a5f) },
	{ SPH_C32(0xa9df0b00), SPH_C32(0x075b753c), SPH_C32(0x5c06fcb2),
	  SPH_C32(0x94f50d36), SPH_C32(0x238a0500), SPH_C32(0xee2a2e13),
	  SPH_C32(0xbfee20ad), SPH_C32(0x6ba3620b) },
	{ SPH_C32(0x24a90800), SPH_C32(0x15720f3f), SPH_C32(0x9b3519f3),
	  SPH_C32(0xb933df5c), SPH_C32(0x33960600), SPH_C32(0x7b279bf8),
	  SPH_C32(0x5a8201f4), SPH_C32(0x13737319) },
	{ SPH_C32(0x930d0900), SPH_C32(0x9f6d3ee7), SPH_C32(0x1ebcc158),
	  SPH_C32(0x5ff7bb38), SPH_C32(0x40da0600), SPH_C32(0xee483c2e),
	  SPH_C32(0xf81f1363), SPH_C32(0x7d961b4d) },
	{ SPH_C32(0x57e50800), SPH_C32(0x801da8e9), SPH_C32(0x39a80b64),
	  SPH_C32(0xd7d6b708), SPH_C32(0xf77e0700), SPH_C32(0x64570df6),
	  SPH_C32(0x7d96cbc8), SPH_C32(0x9b527f29) },
	{ SPH_C32(0xe0410900), SPH_C32(0x0a029931), SPH_C32(0xbc21d3cf),
	  SPH_C32(0x3112d36c), SPH_C32(0x84320700), SPH_C32(0xf138aa20),
	  SPH_C32(0xdf0bd95f), SPH_C32(0xf5b7177d) },
	{ SPH_C32(0x83110a00), SPH_C32(0x0a608b0c), SPH_C32(0xfbd0e001),
	  SPH_C32(0x2727aa2a), SPH_C32(0xddb00600), SPH_C32(0x696cf3c6),
	  SPH_C32(0xda40d77b), SPH_C32(0x2880d835) },
	{ SPH_C32(0x34b50b00), SPH_C32(0x807fbad4), SPH_C32(0x7e5938aa),
	  SPH_C32(0xc1e3ce4e), SPH_C32(0xaefc0600), SPH_C32(0xfc035410),
	  SPH_C32(0x78ddc5ec), SPH_C32(0x4665b061) },
	{ SPH_C32(0xf05d0a00), SPH_C32(0x9f0f2cda), SPH_C32(0x594df296),
	  SPH_C32(0x49c2c27e), SPH_C32(0x19580700), SPH_C32(0x761c65c8),
	  SPH_C32(0xfd541d47), SPH_C32(0xa0a1d405) },
	{ SPH_C32(0x47f90b00), SPH_C32(0x15101d02), SPH_C32(0xdcc42a3d),
	  SPH_C32(0xaf06a61a), SPH_C32(0x6a140700), SPH_C32(0xe373c21e),
	  SPH_C32(0x5fc90fd0), SPH_C32(0xce44bc51) },
	{ SPH_C32(0x45b10c00), SPH_C32(0x0aa4af76), SPH_C32(0x7432879d),
	  SPH_C32(0x0b4d58a6), SPH_C32(0x6eb50400), SPH_C32(0x59c4400a),
	  SPH_C32(0xd0d7cb32), SPH_C32(0x92eb46a5) },
	{ SPH_C32(0xf2150d00), SPH_C32(0x80bb9eae), SPH_C32(0xf1bb5f36),
	  SPH_C32(0xed893cc2), SPH_C32(0x1df90400), SPH_C32(0xccabe7dc),
	  SPH_C32(0x724ad9a5), SPH_C32(0xfc0e2ef1) },
	{ SPH_C32(0x36fd0c00), SPH_C32(0x9fcb08a0), SPH_C32(0xd6af950a),
	  SPH_C32(0x65a830f2), SPH_C32(0xaa5d0500), SPH_C32(0x46b4d604),
	  SPH_C32(0xf7c3010e), SPH_C32(0x1aca4a95) },
	{ SPH_C32(0x81590d00), SPH_C32(0x15d43978), SPH_C32(0x53264da1),
	  SPH_C32(0x836c5496), SPH_C32(0xd9110500), SPH_C32(0xd3db71d2),
	  SPH_C32(0x555e1399), SPH_C32(0x742f22c1) },
	{ SPH_C32(0xe2090e00), SPH_C32(0x15b62b45), SPH_C32(0x14d77e6f),
	  SPH_C32(0x95592dd0), SPH_C32(0x80930400), SPH_C32(0x4b8f2834),
	  SPH_C32(0x50151dbd), SPH_C32(0xa918ed89) },
	{ SPH_C32(0x55ad0f00), SPH_C32(0x9fa91a9d), SPH_C32(0x915ea6c4),
	  SPH_C32(0x739d49b4), SPH_C32(0xf3df0400), SPH_C32(0xdee08fe2),
	  SPH_C32(0xf2880f2a), SPH_C32(0xc7fd85dd) },
	{ SPH_C32(0x91450e00), SPH_C32(0x80d98c93), SPH_C32(0xb64a6cf8),
	  SPH_C32(0xfbbc4584), SPH_C32(0x447b0500), SPH_C32(0x54ffbe3a),
	  SPH_C32(0x7701d781), SPH_C32(0x2139e1b9) },
	{ SPH_C32(0x26e10f00), SPH_C32(0x0ac6bd4b), SPH_C32(0x33c3b453),
	  SPH_C32(0x1d7821e0), SPH_C32(0x37370500), SPH_C32(0xc19019ec),
	  SPH_C32(0xd59cc516), SPH_C32(0x4fdc89ed) },
	{ SPH_C32(0xab970c00), SPH_C32(0x18efc748), SPH_C32(0xf4f05112),
	  SPH_C32(0x30bef38a), SPH_C32(0x272b0600), SPH_C32(0x549dac07),
	  SPH_C32(0x30f0e44f), SPH_C32(0x370c98ff) },
	{ SPH_C32(0x1c330d00), SPH_C32(0x92f0f690), SPH_C32(0x717989b9),
	  SPH_C32(0xd67a97ee), SPH_C32(0x54670600), SPH_C32(0xc1f20bd1),
	  SPH_C32(0x926df6d8), SPH_C32(0x59e9f0ab) },
	{ SPH_C32(0xd8db0c00), SPH_C32(0x8d80609e), SPH_C32(0x566d4385),
	  SPH_C32(0x5e5b9bde), SPH_C32(0xe3c30700), SPH_C32(0x4bed3a09),
	  SPH_C32(0x17e42e73), SPH_C32(0xbf2d94cf) },
	{ SPH_C32(0x6f7f0d00), SPH_C32(0x079f5146), SPH_C32(0xd3e49b2e),
	  SPH_C32(0xb89fffba), SPH_C32(0x908f0700), SPH_C32(0xde829ddf),
	  SPH_C32(0xb5793ce4), SPH_C32(0xd1c8fc9b) },
	{ SPH_C32(0x0c2f0e00), SPH_C32(0x07fd437b), SPH_C32(0x9415a8e0),
	  SPH_C32(0xaeaa86fc), SPH_C32(0xc90d0600), SPH_C32(0x46d6c439),
	  SPH_C32(0xb03232c0), SPH_C32(0x0cff33d3) },
	{ SPH_C32(0xbb8b0f00), SPH_C32(0x8de272a3), SPH_C32(0x119c704b),
	  SPH_C32(0x486ee298), SPH_C32(0xba410600), SPH_C32(0xd3b963ef),
	  SPH_C32(0x12af2057), SPH_C32(0x621a5b87) },
	{ SPH_C32(0x7f630e00), SPH_C32(0x9292e4ad), SPH_C32(0x3688ba77),
	  SPH_C32(0xc04feea8), SPH_C32(0x0de50700), SPH_C32(0x59a65237),
	  SPH_C32(0x9726f8fc), SPH_C32(0x84de3fe3) },
	{ SPH_C32(0xc8c70f00), SPH_C32(0x188dd575), SPH_C32(0xb30162dc),
	  SPH_C32(0x268b8acc), SPH_C32(0x7ea90700), SPH_C32(0xccc9f5e1),
	  SPH_C32(0x35bbea6b), SPH_C32(0xea3b57b7) },
	{ SPH_C32(0xe18b0000), SPH_C32(0x5459887d), SPH_C32(0xbf1283d3),
	  SPH_C32(0x1b666a73), SPH_C32(0x3fb90800), SPH_C32(0x7cdad883),
	  SPH_C32(0xce97a914), SPH_C32(0xbdd9f5e5) },
	{ SPH_C32(0x562f0100), SPH_C32(0xde46b9a5), SPH_C32(0x3a9b5b78),
	  SPH_C32(0xfda20e17), SPH_C32(0x4cf50800), SPH_C32(0xe9b57f55),
	  SPH_C32(0x6c0abb83), SPH_C32(0xd33c9db1) },
	{ SPH_C32(0x92c70000), SPH_C32(0xc1362fab), SPH_C32(0x1d8f9144),
	  SPH_C32(0x75830227), SPH_C32(0xfb510900), SPH_C32(0x63aa4e8d),
	  SPH_C32(0xe9836328), SPH_C32(0x35f8f9d5) },
	{ SPH_C32(0x25630100), SPH_C32(0x4b291e73), SPH_C32(0x980649ef),
	  SPH_C32(0x93476643), SPH_C32(0x881d0900), SPH_C32(0xf6c5e95b),
	  SPH_C32(0x4b1e71bf), SPH_C32(0x5b1d9181) },
	{ SPH_C32(0x46330200), SPH_C32(0x4b4b0c4e), SPH_C32(0xdff77a21),
	  SPH_C32(0x85721f05), SPH_C32(0xd19f0800), SPH_C32(0x6e91b0bd),
	  SPH_C32(0x4e557f9b), SPH_C32(0x862a5ec9) },
	{ SPH_C32(0xf1970300), SPH_C32(0xc1543d96), SPH_C32(0x5a7ea28a),
	  SPH_C32(0x63b67b61), SPH_C32(0xa2d30800), SPH_C32(0xfbfe176b),
	  SPH_C32(0xecc86d0c), SPH_C32(0xe8cf369d) },
	{ SPH_C32(0x357f0200), SPH_C32(0xde24ab98), SPH_C32(0x7d6a68b6),
	  SPH_C32(0xeb977751), SPH_C32(0x15770900), SPH_C32(0x71e126b3),
	  SPH_C32(0x6941b5a7), SPH_C32(0x0e0b52f9) },
	{ SPH_C32(0x82db0300), SPH_C32(0x543b9a40), SPH_C32(0xf8e3b01d),
	  SPH_C32(0x0d531335), SPH_C32(0x663b0900), SPH_C32(0xe48e8165),
	  SPH_C32(0xcbdca730), SPH_C32(0x60ee3aad) },
	{ SPH_C32(0x0fad0000), SPH_C32(0x4612e043), SPH_C32(0x3fd0555c),
	  SPH_C32(0x2095c15f), SPH_C32(0x76270a00), SPH_C32(0x7183348e),
	  SPH_C32(0x2eb08669), SPH_C32(0x183e2bbf) },
	{ SPH_C32(0xb8090100), SPH_C32(0xcc0dd19b), SPH_C32(0xba598df7),
	  SPH_C32(0xc651a53b), SPH_C32(0x056b0a00), SPH_C32(0xe4ec9358),
	  SPH_C32(0x8c2d94fe), SPH_C32(0x76db43eb) },
	{ SPH_C32(0x7ce10000), SPH_C32(0xd37d4795), SPH_C32(0x9d4d47cb),
	  SPH_C32(0x4e70a90b), SPH_C32(0xb2cf0b00), SPH_C32(0x6ef3a280),
	  SPH_C32(0x09a44c55), SPH_C32(0x901f278f) },
	{ SPH_C32(0xcb450100), SPH_C32(0x5962764d), SPH_C32(0x18c49f60),
	  SPH_C32(0xa8b4cd6f), SPH_C32(0xc1830b00), SPH_C32(0xfb9c0556),
	  SPH_C32(0xab395ec2), SPH_C32(0xfefa4fdb) },
	{ SPH_C32(0xa8150200), SPH_C32(0x59006470), SPH_C32(0x5f35acae),
	  SPH_C32(0xbe81b429), SPH_C32(0x98010a00), SPH_C32(0x63c85cb0),
	  SPH_C32(0xae7250e6), SPH_C32(0x23cd8093) },
	{ SPH_C32(0x1fb10300), SPH_C32(0xd31f55a8), SPH_C32(0xdabc7405),
	  SPH_C32(0x5845d04d), SPH_C32(0xeb4d0a00), SPH_C32(0xf6a7fb66),
	  SPH_C32(0x0cef4271), SPH_C32(0x4d28e8c7) },
	{ SPH_C32(0xdb590200), SPH_C32(0xcc6fc3a6), SPH_C32(0xfda8be39),
	  SPH_C32(0xd064dc7d), SPH_C32(0x5ce90b00), SPH_C32(0x7cb8cabe),
	  SPH_C32(0x89669ada), SPH_C32(0xabec8ca3) },
	{ SPH_C32(0x6cfd0300), SPH_C32(0x4670f27e), SPH_C32(0x78216692),
	  SPH_C32(0x36a0b819), SPH_C32(0x2fa50b00), SPH_C32(0xe9d76d68),
	  SPH_C32(0x2bfb884d), SPH_C32(0xc509e4f7) },
	{ SPH_C32(0x6eb50400), SPH_C32(0x59c4400a), SPH_C32(0xd0d7cb32),
	  SPH_C32(0x92eb46a5), SPH_C32(0x2b040800), SPH_C32(0x5360ef7c),
	  SPH_C32(0xa4e54caf), SPH_C32(0x99a61e03) },
	{ SPH_C32(0xd9110500), SPH_C32(0xd3db71d2), SPH_C32(0x555e1399),
	  SPH_C32(0x742f22c1), SPH_C32(0x58480800), SPH_C32(0xc60f48aa),
	  SPH_C32(0x06785e38), SPH_C32(0xf7437657) },
	{ SPH_C32(0x1df90400), SPH_C32(0xccabe7dc), SPH_C32(0x724ad9a5),
	  SPH_C32(0xfc0e2ef1), SPH_C32(0xefec0900), SPH_C32(0x4c107972),
	  SPH_C32(0x83f18693), SPH_C32(0x11871233) },
	{ SPH_C32(0xaa5d0500), SPH_C32(0x46b4d604), SPH_C32(0xf7c3010e),
	  SPH_C32(0x1aca4a95), SPH_C32(0x9ca00900), SPH_C32(0xd97fdea4),
	  SPH_C32(0x216c9404), SPH_C32(0x7f627a67) },
	{ SPH_C32(0xc90d0600), SPH_C32(0x46d6c439), SPH_C32(0xb03232c0),
	  SPH_C32(0x0cff33d3), SPH_C32(0xc5220800), SPH_C32(0x412b8742),
	  SPH_C32(0x24279a20), SPH_C32(0xa255b52f) },
	{ SPH_C32(0x7ea90700), SPH_C32(0xccc9f5e1), SPH_C32(0x35bbea6b),
	  SPH_C32(0xea3b57b7), SPH_C32(0xb66e0800), SPH_C32(0xd4442094),
	  SPH_C32(0x86ba88b7), SPH_C32(0xccb0dd7b) },
	{ SPH_C32(0xba410600), SPH_C32(0xd3b963ef), SPH_C32(0x12af2057),
	  SPH_C32(0x621a5b87), SPH_C32(0x01ca0900), SPH_C32(0x5e5b114c),
	  SPH_C32(0x0333501c), SPH_C32(0x2a74b91f) },
	{ SPH_C32(0x0de50700), SPH_C32(0x59a65237), SPH_C32(0x9726f8fc),
	  SPH_C32(0x84de3fe3), SPH_C32(0x72860900), SPH_C32(0xcb34b69a),
	  SPH_C32(0xa1ae428b), SPH_C32(0x4491d14b) },
	{ SPH_C32(0x80930400), SPH_C32(0x4b8f2834), SPH_C32(0x50151dbd),
	  SPH_C32(0xa918ed89), SPH_C32(0x629a0a00), SPH_C32(0x5e390371),
	  SPH_C32(0x44c263d2), SPH_C32(0x3c41c059) },
	{ SPH_C32(0x37370500), SPH_C32(0xc19019ec), SPH_C32(0xd59cc516),
	  SPH_C32(0x4fdc89ed), SPH_C32(0x11d60a00), SPH_C32(0xcb56a4a7),
	  SPH_C32(0xe65f7145), SPH_C32(0x52a4a80d) },
	{ SPH_C32(0xf3df0400), SPH_C32(0xdee08fe2), SPH_C32(0xf2880f2a),
	  SPH_C32(0xc7fd85dd), SPH_C32(0xa6720b00), SPH_C32(0x4149957f),
	  SPH_C32(0x63d6a9ee), SPH_C32(0xb460cc69) },
	{ SPH_C32(0x447b0500), SPH_C32(0x54ffbe3a), SPH_C32(0x7701d781),
	  SPH_C32(0x2139e1b9), SPH_C32(0xd53e0b00), SPH_C32(0xd42632a9),
	  SPH_C32(0xc14bbb79), SPH_C32(0xda85a43d) },
	{ SPH_C32(0x272b0600), SPH_C32(0x549dac07), SPH_C32(0x30f0e44f),
	  SPH_C32(0x370c98ff), SPH_C32(0x8cbc0a00), SPH_C32(0x4c726b4f),
	  SPH_C32(0xc400b55d), SPH_C32(0x07b26b75) },
	{ SPH_C32(0x908f0700), SPH_C32(0xde829ddf), SPH_C32(0xb5793ce4),
	  SPH_C32(0xd1c8fc9b), SPH_C32(0xfff00a00), SPH_C32(0xd91dcc99),
	  SPH_C32(0x669da7ca), SPH_C32(0x69570321) },
	{ SPH_C32(0x54670600), SPH_C32(0xc1f20bd1), SPH_C32(0x926df6d8),
	  SPH_C32(0x59e9f0ab), SPH_C32(0x48540b00), SPH_C32(0x5302fd41),
	  SPH_C32(0xe3147f61), SPH_C32(0x8f936745) },
	{ SPH_C32(0xe3c30700), SPH_C32(0x4bed3a09), SPH_C32(0x17e42e73),
	  SPH_C32(0xbf2d94cf), SPH_C32(0x3b180b00), SPH_C32(0xc66d5a97),
	  SPH_C32(0x41896df6), SPH_C32(0xe1760f11) },
	{ SPH_C32(0xf5360000), SPH_C32(0x7be3bf82), SPH_C32(0xd5606668),
	  SPH_C32(0x3f198195), SPH_C32(0xa43a0c00), SPH_C32(0x5efd270b),
	  SPH_C32(0xcb20044e), SPH_C32(0x102b32d5) },
	{ SPH_C32(0x42920100), SPH_C32(0xf1fc8e5a), SPH_C32(0x50e9bec3),
	  SPH_C32(0xd9dde5f1), SPH_C32(0xd7760c00), SPH_C32(0xcb9280dd),
	  SPH_C32(0x69bd16d9), SPH_C32(0x7ece5a81) },
	{ SPH_C32(0x867a0000), SPH_C32(0xee8c1854), SPH_C32(0x77fd74ff),
	  SPH_C32(0x51fce9c1), SPH_C32(0x60d20d00), SPH_C32(0x418db105),
	  SPH_C32(0xec34ce72), SPH_C32(0x980a3ee5) },
	{ SPH_C32(0x31de0100), SPH_C32(0x6493298c), SPH_C32(0xf274ac54),
	  SPH_C32(0xb7388da5), SPH_C32(0x139e0d00), SPH_C32(0xd4e216d3),
	  SPH_C32(0x4ea9dce5), SPH_C32(0xf6ef56b1) },
	{ SPH_C32(0x528e0200), SPH_C32(0x64f13bb1), SPH_C32(0xb5859f9a),
	  SPH_C32(0xa10df4e3), SPH_C32(0x4a1c0c00), SPH_C32(0x4cb64f35),
	  SPH_C32(0x4be2d2c1), SPH_C32(0x2bd899f9) },
	{ SPH_C32(0xe52a0300), SPH_C32(0xeeee0a69), SPH_C32(0x300c4731),
	  SPH_C32(0x47c99087), SPH_C32(0x39500c00), SPH_C32(0xd9d9e8e3),
	  SPH_C32(0xe97fc056), SPH_C32(0x453df1ad) },
	{ SPH_C32(0x21c20200), SPH_C32(0xf19e9c67), SPH_C32(0x17188d0d),
	  SPH_C32(0xcfe89cb7), SPH_C32(0x8ef40d00), SPH_C32(0x53c6d93b),
	  SPH_C32(0x6cf618fd), SPH_C32(0xa3f995c9) },
	{ SPH_C32(0x96660300), SPH_C32(0x7b81adbf), SPH_C32(0x929155a6),
	  SPH_C32(0x292cf8d3), SPH_C32(0xfdb80d00), SPH_C32(0xc6a97eed),
	  SPH_C32(0xce6b0a6a), SPH_C32(0xcd1cfd9d) },
	{ SPH_C32(0x1b100000), SPH_C32(0x69a8d7bc), SPH_C32(0x55a2b0e7),
	  SPH_C32(0x04ea2ab9), SPH_C32(0xeda40e00), SPH_C32(0x53a4cb06),
	  SPH_C32(0x2b072b33), SPH_C32(0xb5ccec8f) },
	{ SPH_C32(0xacb40100), SPH_C32(0xe3b7e664), SPH_C32(0xd02b684c),
	  SPH_C32(0xe22e4edd), SPH_C32(0x9ee80e00), SPH_C32(0xc6cb6cd0),
	  SPH_C32(0x899a39a4), SPH_C32(0xdb2984db) },
	{ SPH_C32(0x685c0000), SPH_C32(0xfcc7706a), SPH_C32(0xf73fa270),
	  SPH_C32(0x6a0f42ed), SPH_C32(0x294c0f00), SPH_C32(0x4cd45d08),
	  SPH_C32(0x0c13e10f), SPH_C32(0x3dede0bf) },
	{ SPH_C32(0xdff80100), SPH_C32(0x76d841b2), SPH_C32(0x72b67adb),
	  SPH_C32(0x8ccb2689), SPH_C32(0x5a000f00), SPH_C32(0xd9bbfade),
	  SPH_C32(0xae8ef398), SPH_C32(0x530888eb) },
	{ SPH_C32(0xbca80200), SPH_C32(0x76ba538f), SPH_C32(0x35474915),
	  SPH_C32(0x9afe5fcf), SPH_C32(0x03820e00), SPH_C32(0x41efa338),
	  SPH_C32(0xabc5fdbc), SPH_C32(0x8e3f47a3) },
	{ SPH_C32(0x0b0c0300), SPH_C32(0xfca56257), SPH_C32(0xb0ce91be),
	  SPH_C32(0x7c3a3bab), SPH_C32(0x70ce0e00), SPH_C32(0xd48004ee),
	  SPH_C32(0x0958ef2b), SPH_C32(0xe0da2ff7) },
	{ SPH_C32(0xcfe40200), SPH_C32(0xe3d5f459), SPH_C32(0x97da5b82),
	  SPH_C32(0xf41b379b), SPH_C32(0xc76a0f00), SPH_C32(0x5e9f3536),
	  SPH_C32(0x8cd13780), SPH_C32(0x061e4b93) },
	{ SPH_C32(0x78400300), SPH_C32(0x69cac581), SPH_C32(0x12538329),
	  SPH_C32(0x12df53ff), SPH_C32(0xb4260f00), SPH_C32(0xcbf092e0),
	  SPH_C32(0x2e4c2517), SPH_C32(0x68fb23c7) },
	{ SPH_C32(0x7a080400), SPH_C32(0x767e77f5), SPH_C32(0xbaa52e89),
	  SPH_C32(0xb694ad43), SPH_C32(0xb0870c00), SPH_C32(0x714710f4),
	  SPH_C32(0xa152e1f5), SPH_C32(0x3454d933) },
	{ SPH_C32(0xcdac0500), SPH_C32(0xfc61462d), SPH_C32(0x3f2cf622),
	  SPH_C32(0x5050c927), SPH_C32(0xc3cb0c00), SPH_C32(0xe428b722),
	  SPH_C32(0x03cff362), SPH_C32(0x5ab1b167) },
	{ SPH_C32(0x09440400), SPH_C32(0xe311d023), SPH_C32(0x18383c1e),
	  SPH_C32(0xd871c517), SPH_C32(0x746f0d00), SPH_C32(0x6e3786fa),
	  SPH_C32(0x86462bc9), SPH_C32(0xbc75d503) },
	{ SPH_C32(0xbee00500), SPH_C32(0x690ee1fb), SPH_C32(0x9db1e4b5),
	  SPH_C32(0x3eb5a173), SPH_C32(0x07230d00), SPH_C32(0xfb58212c),
	  SPH_C32(0x24db395e), SPH_C32(0xd290bd57) },
	{ SPH_C32(0xddb00600), SPH_C32(0x696cf3c6), SPH_C32(0xda40d77b),
	  SPH_C32(0x2880d835), SPH_C32(0x5ea10c00), SPH_C32(0x630c78ca),
	  SPH_C32(0x2190377a), SPH_C32(0x0fa7721f) },
	{ SPH_C32(0x6a140700), SPH_C32(0xe373c21e), SPH_C32(0x5fc90fd0),
	  SPH_C32(0xce44bc51), SPH_C32(0x2ded0c00), SPH_C32(0xf663df1c),
	  SPH_C32(0x830d25ed), SPH_C32(0x61421a4b) },
	{ SPH_C32(0xaefc0600), SPH_C32(0xfc035410), SPH_C32(0x78ddc5ec),
	  SPH_C32(0x4665b061), SPH_C32(0x9a490d00), SPH_C32(0x7c7ceec4),
	  SPH_C32(0x0684fd46), SPH_C32(0x87867e2f) },
	{ SPH_C32(0x19580700), SPH_C32(0x761c65c8), SPH_C32(0xfd541d47),
	  SPH_C32(0xa0a1d405), SPH_C32(0xe9050d00), SPH_C32(0xe9134912),
	  SPH_C32(0xa419efd1), SPH_C32(0xe963167b) },
	{ SPH_C32(0x942e0400), SPH_C32(0x64351fcb), SPH_C32(0x3a67f806),
	  SPH_C32(0x8d67066f), SPH_C32(0xf9190e00), SPH_C32(0x7c1efcf9),
	  SPH_C32(0x4175ce88), SPH_C32(0x91b30769) },
	{ SPH_C32(0x238a0500), SPH_C32(0xee2a2e13), SPH_C32(0xbfee20ad),
	  SPH_C32(0x6ba3620b), SPH_C32(0x8a550e00), SPH_C32(0xe9715b2f),
	  SPH_C32(0xe3e8dc1f), SPH_C32(0xff566f3d) },
	{ SPH_C32(0xe7620400), SPH_C32(0xf15ab81d), SPH_C32(0x98faea91),
	  SPH_C32(0xe3826e3b), SPH_C32(0x3df10f00), SPH_C32(0x636e6af7),
	  SPH_C32(0x666104b4), SPH_C32(0x19920b59) },
	{ SPH_C32(0x50c60500), SPH_C32(0x7b4589c5), SPH_C32(0x1d73323a),
	  SPH_C32(0x05460a5f), SPH_C32(0x4ebd0f00), SPH_C32(0xf601cd21),
	  SPH_C32(0xc4fc1623), SPH_C32(0x7777630d) },
	{ SPH_C32(0x33960600), SPH_C32(0x7b279bf8), SPH_C32(0x5a8201f4),
	  SPH_C32(0x13737319), SPH_C32(0x173f0e00), SPH_C32(0x6e5594c7),
	  SPH_C32(0xc1b71807), SPH_C32(0xaa40ac45) },
	{ SPH_C32(0x84320700), SPH_C32(0xf138aa20), SPH_C32(0xdf0bd95f),
	  SPH_C32(0xf5b7177d), SPH_C32(0x64730e00), SPH_C32(0xfb3a3311),
	  SPH_C32(0x632a0a90), SPH_C32(0xc4a5c411) },
	{ SPH_C32(0x40da0600), SPH_C32(0xee483c2e), SPH_C32(0xf81f1363),
	  SPH_C32(0x7d961b4d), SPH_C32(0xd3d70f00), SPH_C32(0x712502c9),
	  SPH_C32(0xe6a3d23b), SPH_C32(0x2261a075) },
	{ SPH_C32(0xf77e0700), SPH_C32(0x64570df6), SPH_C32(0x7d96cbc8),
	  SPH_C32(0x9b527f29), SPH_C32(0xa09b0f00), SPH_C32(0xe44aa51f),
	  SPH_C32(0x443ec0ac), SPH_C32(0x4c84c821) },
	{ SPH_C32(0x3fb90800), SPH_C32(0x7cdad883), SPH_C32(0xce97a914),
	  SPH_C32(0xbdd9f5e5), SPH_C32(0xde320800), SPH_C32(0x288350fe),
	  SPH_C32(0x71852ac7), SPH_C32(0xa6bf9f96) },
	{ SPH_C32(0x881d0900), SPH_C32(0xf6c5e95b), SPH_C32(0x4b1e71bf),
	  SPH_C32(0x5b1d9181), SPH_C32(0xad7e0800), SPH_C32(0xbdecf728),
	  SPH_C32(0xd3183850), SPH_C32(0xc85af7c2) },
	{ SPH_C32(0x4cf50800), SPH_C32(0xe9b57f55), SPH_C32(0x6c0abb83),
	  SPH_C32(0xd33c9db1), SPH_C32(0x1ada0900), SPH_C32(0x37f3c6f0),
	  SPH_C32(0x5691e0fb), SPH_C32(0x2e9e93a6) },
	{ SPH_C32(0xfb510900), SPH_C32(0x63aa4e8d), SPH_C32(0xe9836328),
	  SPH_C32(0x35f8f9d5), SPH_C32(0x69960900), SPH_C32(0xa29c6126),
	  SPH_C32(0xf40cf26c), SPH_C32(0x407bfbf2) },
	{ SPH_C32(0x98010a00), SPH_C32(0x63c85cb0), SPH_C32(0xae7250e6),
	  SPH_C32(0x23cd8093), SPH_C32(0x30140800), SPH_C32(0x3ac838c0),
	  SPH_C32(0xf147fc48), SPH_C32(0x9d4c34ba) },
	{ SPH_C32(0x2fa50b00), SPH_C32(0xe9d76d68), SPH_C32(0x2bfb884d),
	  SPH_C32(0xc509e4f7), SPH_C32(0x43580800), SPH_C32(0xafa79f16),
	  SPH_C32(0x53daeedf), SPH_C32(0xf3a95cee) },
	{ SPH_C32(0xeb4d0a00), SPH_C32(0xf6a7fb66), SPH_C32(0x0cef4271),
	  SPH_C32(0x4d28e8c7), SPH_C32(0xf4fc0900), SPH_C32(0x25b8aece),
	  SPH_C32(0xd6533674), SPH_C32(0x156d388a) },
	{ SPH_C32(0x5ce90b00), SPH_C32(0x7cb8cabe), SPH_C32(0x89669ada),
	  SPH_C32(0xabec8ca3), SPH_C32(0x87b00900), SPH_C32(0xb0d70918),
	  SPH_C32(0x74ce24e3), SPH_C32(0x7b8850de) },
	{ SPH_C32(0xd19f0800), SPH_C32(0x6e91b0bd), SPH_C32(0x4e557f9b),
	  SPH_C32(0x862a5ec9), SPH_C32(0x97ac0a00), SPH_C32(0x25dabcf3),
	  SPH_C32(0x91a205ba), SPH_C32(0x035841cc) },
	{ SPH_C32(0x663b0900), SPH_C32(0xe48e8165), SPH_C32(0xcbdca730),
	  SPH_C32(0x60ee3aad), SPH_C32(0xe4e00a00), SPH_C32(0xb0b51b25),
	  SPH_C32(0x333f172d), SPH_C32(0x6dbd2998) },
	{ SPH_C32(0xa2d30800), SPH_C32(0xfbfe176b), SPH_C32(0xecc86d0c),
	  SPH_C32(0xe8cf369d), SPH_C32(0x53440b00), SPH_C32(0x3aaa2afd),
	  SPH_C32(0xb6b6cf86), SPH_C32(0x8b794dfc) },
	{ SPH_C32(0x15770900), SPH_C32(0x71e126b3), SPH_C32(0x6941b5a7),
	  SPH_C32(0x0e0b52f9), SPH_C32(0x20080b00), SPH_C32(0xafc58d2b),
	  SPH_C32(0x142bdd11), SPH_C32(0xe59c25a8) },
	{ SPH_C32(0x76270a00), SPH_C32(0x7183348e), SPH_C32(0x2eb08669),
	  SPH_C32(0x183e2bbf), SPH_C32(0x798a0a00), SPH_C32(0x3791d4cd),
	  SPH_C32(0x1160d335), SPH_C32(0x38abeae0) },
	{ SPH_C32(0xc1830b00), SPH_C32(0xfb9c0556), SPH_C32(0xab395ec2),
	  SPH_C32(0xfefa4fdb), SPH_C32(0x0ac60a00), SPH_C32(0xa2fe731b),
	  SPH_C32(0xb3fdc1a2), SPH_C32(0x564e82b4) },
	{ SPH_C32(0x056b0a00), SPH_C32(0xe4ec9358), SPH_C32(0x8c2d94fe),
	  SPH_C32(0x76db43eb), SPH_C32(0xbd620b00), SPH_C32(0x28e142c3),
	  SPH_C32(0x36741909), SPH_C32(0xb08ae6d0) },
	{ SPH_C32(0xb2cf0b00), SPH_C32(0x6ef3a280), SPH_C32(0x09a44c55),
	  SPH_C32(0x901f278f), SPH_C32(0xce2e0b00), SPH_C32(0xbd8ee515),
	  SPH_C32(0x94e90b9e), SPH_C32(0xde6f8e84) },
	{ SPH_C32(0xb0870c00), SPH_C32(0x714710f4), SPH_C32(0xa152e1f5),
	  SPH_C32(0x3454d933), SPH_C32(0xca8f0800), SPH_C32(0x07396701),
	  SPH_C32(0x1bf7cf7c), SPH_C32(0x82c07470) },
	{ SPH_C32(0x07230d00), SPH_C32(0xfb58212c), SPH_C32(0x24db395e),
	  SPH_C32(0xd290bd57), SPH_C32(0xb9c30800), SPH_C32(0x9256c0d7),
	  SPH_C32(0xb96addeb), SPH_C32(0xec251c24) },
	{ SPH_C32(0xc3cb0c00), SPH_C32(0xe428b722), SPH_C32(0x03cff362),
	  SPH_C32(0x5ab1b167), SPH_C32(0x0e670900), SPH_C32(0x1849f10f),
	  SPH_C32(0x3ce30540), SPH_C32(0x0ae17840) },
	{ SPH_C32(0x746f0d00), SPH_C32(0x6e3786fa), SPH_C32(0x86462bc9),
	  SPH_C32(0xbc75d503), SPH_C32(0x7d2b0900), SPH_C32(0x8d2656d9),
	  SPH_C32(0x9e7e17d7), SPH_C32(0x64041014) },
	{ SPH_C32(0x173f0e00), SPH_C32(0x6e5594c7), SPH_C32(0xc1b71807),
	  SPH_C32(0xaa40ac45), SPH_C32(0x24a90800), SPH_C32(0x15720f3f),
	  SPH_C32(0x9b3519f3), SPH_C32(0xb933df5c) },
	{ SPH_C32(0xa09b0f00), SPH_C32(0xe44aa51f), SPH_C32(0x443ec0ac),
	  SPH_C32(0x4c84c821), SPH_C32(0x57e50800), SPH_C32(0x801da8e9),
	  SPH_C32(0x39a80b64), SPH_C32(0xd7d6b708) },
	{ SPH_C32(0x64730e00), SPH_C32(0xfb3a3311), SPH_C32(0x632a0a90),
	  SPH_C32(0xc4a5c411), SPH_C32(0xe0410900), SPH_C32(0x0a029931),
	  SPH_C32(0xbc21d3cf), SPH_C32(0x3112d36c) },
	{ SPH_C32(0xd3d70f00), SPH_C32(0x712502c9), SPH_C32(0xe6a3d23b),
	  SPH_C32(0x2261a075), SPH_C32(0x930d0900), SPH_C32(0x9f6d3ee7),
	  SPH_C32(0x1ebcc158), SPH_C32(0x5ff7bb38) },
	{ SPH_C32(0x5ea10c00), SPH_C32(0x630c78ca), SPH_C32(0x2190377a),
	  SPH_C32(0x0fa7721f), SPH_C32(0x83110a00), SPH_C32(0x0a608b0c),
	  SPH_C32(0xfbd0e001), SPH_C32(0x2727aa2a) },
	{ SPH_C32(0xe9050d00), SPH_C32(0xe9134912), SPH_C32(0xa419efd1),
	  SPH_C32(0xe963167b), SPH_C32(0xf05d0a00), SPH_C32(0x9f0f2cda),
	  SPH_C32(0x594df296), SPH_C32(0x49c2c27e) },
	{ SPH_C32(0x2ded0c00), SPH_C32(0xf663df1c), SPH_C32(0x830d25ed),
	  SPH_C32(0x61421a4b), SPH_C32(0x47f90b00), SPH_C32(0x15101d02),
	  SPH_C32(0xdcc42a3d), SPH_C32(0xaf06a61a) },
	{ SPH_C32(0x9a490d00), SPH_C32(0x7c7ceec4), SPH_C32(0x0684fd46),
	  SPH_C32(0x87867e2f), SPH_C32(0x34b50b00), SPH_C32(0x807fbad4),
	  SPH_C32(0x7e5938aa), SPH_C32(0xc1e3ce4e) },
	{ SPH_C32(0xf9190e00), SPH_C32(0x7c1efcf9), SPH_C32(0x4175ce88),
	  SPH_C32(0x91b30769), SPH_C32(0x6d370a00), SPH_C32(0x182be332),
	  SPH_C32(0x7b12368e), SPH_C32(0x1cd40106) },
	{ SPH_C32(0x4ebd0f00), SPH_C32(0xf601cd21), SPH_C32(0xc4fc1623),
	  SPH_C32(0x7777630d), SPH_C32(0x1e7b0a00), SPH_C32(0x8d4444e4),
	  SPH_C32(0xd98f2419), SPH_C32(0x72316952) },
	{ SPH_C32(0x8a550e00), SPH_C32(0xe9715b2f), SPH_C32(0xe3e8dc1f),
	  SPH_C32(0xff566f3d), SPH_C32(0xa9df0b00), SPH_C32(0x075b753c),
	  SPH_C32(0x5c06fcb2), SPH_C32(0x94f50d36) },
	{ SPH_C32(0x3df10f00), SPH_C32(0x636e6af7), SPH_C32(0x666104b4),
	  SPH_C32(0x19920b59), SPH_C32(0xda930b00), SPH_C32(0x9234d2ea),
	  SPH_C32(0xfe9bee25), SPH_C32(0xfa106562) },
	{ SPH_C32(0x2b040800), SPH_C32(0x5360ef7c), SPH_C32(0xa4e54caf),
	  SPH_C32(0x99a61e03), SPH_C32(0x45b10c00), SPH_C32(0x0aa4af76),
	  SPH_C32(0x7432879d), SPH_C32(0x0b4d58a6) },
	{ SPH_C32(0x9ca00900), SPH_C32(0xd97fdea4), SPH_C32(0x216c9404),
	  SPH_C32(0x7f627a67), SPH_C32(0x36fd0c00), SPH_C32(0x9fcb08a0),
	  SPH_C32(0xd6af950a), SPH_C32(0x65a830f2) },
	{ SPH_C32(0x58480800), SPH_C32(0xc60f48aa), SPH_C32(0x06785e38),
	  SPH_C32(0xf7437657), SPH_C32(0x81590d00), SPH_C32(0x15d43978),
	  SPH_C32(0x53264da1), SPH_C32(0x836c5496) },
	{ SPH_C32(0xefec0900), SPH_C32(0x4c107972), SPH_C32(0x83f18693),
	  SPH_C32(0x11871233), SPH_C32(0xf2150d00), SPH_C32(0x80bb9eae),
	  SPH_C32(0xf1bb5f36), SPH_C32(0xed893cc2) },
	{ SPH_C32(0x8cbc0a00), SPH_C32(0x4c726b4f), SPH_C32(0xc400b55d),
	  SPH_C32(0x07b26b75), SPH_C32(0xab970c00), SPH_C32(0x18efc748),
	  SPH_C32(0xf4f05112), SPH_C32(0x30bef38a) },
	{ SPH_C32(0x3b180b00), SPH_C32(0xc66d5a97), SPH_C32(0x41896df6),
	  SPH_C32(0xe1760f11), SPH_C32(0xd8db0c00), SPH_C32(0x8d80609e),
	  SPH_C32(0x566d4385), SPH_C32(0x5e5b9bde) },
	{ SPH_C32(0xfff00a00), SPH_C32(0xd91dcc99), SPH_C32(0x669da7ca),
	  SPH_C32(0x69570321), SPH_C32(0x6f7f0d00), SPH_C32(0x079f5146),
	  SPH_C32(0xd3e49b2e), SPH_C32(0xb89fffba) },
	{ SPH_C32(0x48540b00), SPH_C32(0x5302fd41), SPH_C32(0xe3147f61),
	  SPH_C32(0x8f936745), SPH_C32(0x1c330d00), SPH_C32(0x92f0f690),
	  SPH_C32(0x717989b9), SPH_C32(0xd67a97ee) },
	{ SPH_C32(0xc5220800), SPH_C32(0x412b8742), SPH_C32(0x24279a20),
	  SPH_C32(0xa255b52f), SPH_C32(0x0c2f0e00), SPH_C32(0x07fd437b),
	  SPH_C32(0x9415a8e0), SPH_C32(0xaeaa86fc) },
	{ SPH_C32(0x72860900), SPH_C32(0xcb34b69a), SPH_C32(0xa1ae428b),
	  SPH_C32(0x4491d14b), SPH_C32(0x7f630e00), SPH_C32(0x9292e4ad),
	  SPH_C32(0x3688ba77), SPH_C32(0xc04feea8) },
	{ SPH_C32(0xb66e0800), SPH_C32(0xd4442094), SPH_C32(0x86ba88b7),
	  SPH_C32(0xccb0dd7b), SPH_C32(0xc8c70f00), SPH_C32(0x188dd575),
	  SPH_C32(0xb30162dc), SPH_C32(0x268b8acc) },
	{ SPH_C32(0x01ca0900), SPH_C32(0x5e5b114c), SPH_C32(0x0333501c),
	  SPH_C32(0x2a74b91f), SPH_C32(0xbb8b0f00), SPH_C32(0x8de272a3),
	  SPH_C32(0x119c704b), SPH_C32(0x486ee298) },
	{ SPH_C32(0x629a0a00), SPH_C32(0x5e390371), SPH_C32(0x44c263d2),
	  SPH_C32(0x3c41c059), SPH_C32(0xe2090e00), SPH_C32(0x15b62b45),
	  SPH_C32(0x14d77e6f), SPH_C32(0x95592dd0) },
	{ SPH_C32(0xd53e0b00), SPH_C32(0xd42632a9), SPH_C32(0xc14bbb79),
	  SPH_C32(0xda85a43d), SPH_C32(0x91450e00), SPH_C32(0x80d98c93),
	  SPH_C32(0xb64a6cf8), SPH_C32(0xfbbc4584) },
	{ SPH_C32(0x11d60a00), SPH_C32(0xcb56a4a7), SPH_C32(0xe65f7145),
	  SPH_C32(0x52a4a80d), SPH_C32(0x26e10f00), SPH_C32(0x0ac6bd4b),
	  SPH_C32(0x33c3b453), SPH_C32(0x1d7821e0) },
	{ SPH_C32(0xa6720b00), SPH_C32(0x4149957f), SPH_C32(0x63d6a9ee),
	  SPH_C32(0xb460cc69), SPH_C32(0x55ad0f00), SPH_C32(0x9fa91a9d),
	  SPH_C32(0x915ea6c4), SPH_C32(0x739d49b4) },
	{ SPH_C32(0xa43a0c00), SPH_C32(0x5efd270b), SPH_C32(0xcb20044e),
	  SPH_C32(0x102b32d5), SPH_C32(0x510c0c00), SPH_C32(0x251e9889),
	  SPH_C32(0x1e406226), SPH_C32(0x2f32b340) },
	{ SPH_C32(0x139e0d00), SPH_C32(0xd4e216d3), SPH_C32(0x4ea9dce5),
	  SPH_C32(0xf6ef56b1), SPH_C32(0x22400c00), SPH_C32(0xb0713f5f),
	  SPH_C32(0xbcdd70b1), SPH_C32(0x41d7db14) },
	{ SPH_C32(0xd7760c00), SPH_C32(0xcb9280dd), SPH_C32(0x69bd16d9),
	  SPH_C32(0x7ece5a81), SPH_C32(0x95e40d00), SPH_C32(0x3a6e0e87),
	  SPH_C32(0x3954a81a), SPH_C32(0xa713bf70) },
	{ SPH_C32(0x60d20d00), SPH_C32(0x418db105), SPH_C32(0xec34ce72),
	  SPH_C32(0x980a3ee5), SPH_C32(0xe6a80d00), SPH_C32(0xaf01a951),
	  SPH_C32(0x9bc9ba8d), SPH_C32(0xc9f6d724) },
	{ SPH_C32(0x03820e00), SPH_C32(0x41efa338), SPH_C32(0xabc5fdbc),
	  SPH_C32(0x8e3f47a3), SPH_C32(0xbf2a0c00), SPH_C32(0x3755f0b7),
	  SPH_C32(0x9e82b4a9), SPH_C32(0x14c1186c) },
	{ SPH_C32(0xb4260f00), SPH_C32(0xcbf092e0), SPH_C32(0x2e4c2517),
	  SPH_C32(0x68fb23c7), SPH_C32(0xcc660c00), SPH_C32(0xa23a5761),
	  SPH_C32(0x3c1fa63e), SPH_C32(0x7a247038) },
	{ SPH_C32(0x70ce0e00), SPH_C32(0xd48004ee), SPH_C32(0x0958ef2b),
	  SPH_C32(0xe0da2ff7), SPH_C32(0x7bc20d00), SPH_C32(0x282566b9),
	  SPH_C32(0xb9967e95), SPH_C32(0x9ce0145c) },
	{ SPH_C32(0xc76a0f00), SPH_C32(0x5e9f3536), SPH_C32(0x8cd13780),
	  SPH_C32(0x061e4b93), SPH_C32(0x088e0d00), SPH_C32(0xbd4ac16f),
	  SPH_C32(0x1b0b6c02), SPH_C32(0xf2057c08) },
	{ SPH_C32(0x4a1c0c00), SPH_C32(0x4cb64f35), SPH_C32(0x4be2d2c1),
	  SPH_C32(0x2bd899f9), SPH_C32(0x18920e00), SPH_C32(0x28477484),
	  SPH_C32(0xfe674d5b), SPH_C32(0x8ad56d1a) },
	{ SPH_C32(0xfdb80d00), SPH_C32(0xc6a97eed), SPH_C32(0xce6b0a6a),
	  SPH_C32(0xcd1cfd9d), SPH_C32(0x6bde0e00), SPH_C32(0xbd28d352),
	  SPH_C32(0x5cfa5fcc), SPH_C32(0xe430054e) },
	{ SPH_C32(0x39500c00), SPH_C32(0xd9d9e8e3), SPH_C32(0xe97fc056),
	  SPH_C32(0x453df1ad), SPH_C32(0xdc7a0f00), SPH_C32(0x3737e28a),
	  SPH_C32(0xd9738767), SPH_C32(0x02f4612a) },
	{ SPH_C32(0x8ef40d00), SPH_C32(0x53c6d93b), SPH_C32(0x6cf618fd),
	  SPH_C32(0xa3f995c9), SPH_C32(0xaf360f00), SPH_C32(0xa258455c),
	  SPH_C32(0x7bee95f0), SPH_C32(0x6c11097e) },
	{ SPH_C32(0xeda40e00), SPH_C32(0x53a4cb06), SPH_C32(0x2b072b33),
	  SPH_C32(0xb5ccec8f), SPH_C32(0xf6b40e00), SPH_C32(0x3a0c1cba),
	  SPH_C32(0x7ea59bd4), SPH_C32(0xb126c636) },
	{ SPH_C32(0x5a000f00), SPH_C32(0xd9bbfade), SPH_C32(0xae8ef398),
	  SPH_C32(0x530888eb), SPH_C32(0x85f80e00), SPH_C32(0xaf63bb6c),
	  SPH_C32(0xdc388943), SPH_C32(0xdfc3ae62) },
	{ SPH_C32(0x9ee80e00), SPH_C32(0xc6cb6cd0), SPH_C32(0x899a39a4),
	  SPH_C32(0xdb2984db), SPH_C32(0x325c0f00), SPH_C32(0x257c8ab4),
	  SPH_C32(0x59b151e8), SPH_C32(0x3907ca06) },
	{ SPH_C32(0x294c0f00), SPH_C32(0x4cd45d08), SPH_C32(0x0c13e10f),
	  SPH_C32(0x3dede0bf), SPH_C32(0x41100f00), SPH_C32(0xb0132d62),
	  SPH_C32(0xfb2c437f), SPH_C32(0x57e2a252) }
};

static const sph_u32 T256_16[256][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0x515c0010), SPH_C32(0x40f372fb), SPH_C32(0xfce72602),
	  SPH_C32(0x71575061), SPH_C32(0x2e390000), SPH_C32(0x64dd6689),
	  SPH_C32(0x3cd406fc), SPH_C32(0xb1f490bc) },
	{ SPH_C32(0x2e390000), SPH_C32(0x64dd6689), SPH_C32(0x3cd406fc),
	  SPH_C32(0xb1f490bc), SPH_C32(0x7f650010), SPH_C32(0x242e1472),
	  SPH_C32(0xc03320fe), SPH_C32(0xc0a3c0dd) },
	{ SPH_C32(0x7f650010), SPH_C32(0x242e1472), SPH_C32(0xc03320fe),
	  SPH_C32(0xc0a3c0dd), SPH_C32(0x515c0010), SPH_C32(0x40f372fb),
	  SPH_C32(0xfce72602), SPH_C32(0x71575061) },
	{ SPH_C32(0xa2b80020), SPH_C32(0x81e7e5f6), SPH_C32(0xf9ce4c04),
	  SPH_C32(0xe2afa0c0), SPH_C32(0x5c720000), SPH_C32(0xc9bacd12),
	  SPH_C32(0x79a90df9), SPH_C32(0x63e92178) },
	{ SPH_C32(0xf3e40030), SPH_C32(0xc114970d), SPH_C32(0x05296a06),
	  SPH_C32(0x93f8f0a1), SPH_C32(0x724b0000), SPH_C32(0xad67ab9b),
	  SPH_C32(0x457d0b05), SPH_C32(0xd21db1c4) },
	{ SPH_C32(0x8c810020), SPH_C32(0xe53a837f), SPH_C32(0xc51a4af8),
	  SPH_C32(0x535b307c), SPH_C32(0x23170010), SPH_C32(0xed94d960),
	  SPH_C32(0xb99a2d07), SPH_C32(0xa34ae1a5) },
	{ SPH_C32(0xdddd0030), SPH_C32(0xa5c9f184), SPH_C32(0x39fd6cfa),
	  SPH_C32(0x220c601d), SPH_C32(0x0d2e0010), SPH_C32(0x8949bfe9),
	  SPH_C32(0x854e2bfb), SPH_C32(0x12be7119) },
	{ SPH_C32(0x5c720000), SPH_C32(0xc9bacd12), SPH_C32(0x79a90df9),
	  SPH_C32(0x63e92178), SPH_C32(0xfeca0020), SPH_C32(0x485d28e4),
	  SPH_C32(0x806741fd), SPH_C32(0x814681b8) },
	{ SPH_C32(0x0d2e0010), SPH_C32(0x8949bfe9), SPH_C32(0x854e2bfb),
	  SPH_C32(0x12be7119), SPH_C32(0xd0f30020), SPH_C32(0x2c804e6d),
	  SPH_C32(0xbcb34701), SPH_C32(0x30b21104) },
	{ SPH_C32(0x724b0000), SPH_C32(0xad67ab9b), SPH_C32(0x457d0b05),
	  SPH_C32(0xd21db1c4), SPH_C32(0x81af0030), SPH_C32(0x6c733c96),
	  SPH_C32(0x40546103), SPH_C32(0x41e54165) },
	{ SPH_C32(0x23170010), SPH_C32(0xed94d960), SPH_C32(0xb99a2d07),
	  SPH_C32(0xa34ae1a5), SPH_C32(0xaf960030), SPH_C32(0x08ae5a1f),
	  SPH_C32(0x7c8067ff), SPH_C32(0xf011d1d9) },
	{ SPH_C32(0xfeca0020), SPH_C32(0x485d28e4), SPH_C32(0x806741fd),
	  SPH_C32(0x814681b8), SPH_C32(0xa2b80020), SPH_C32(0x81e7e5f6),
	  SPH_C32(0xf9ce4c04), SPH_C32(0xe2afa0c0) },
	{ SPH_C32(0xaf960030), SPH_C32(0x08ae5a1f), SPH_C32(0x7c8067ff),
	  SPH_C32(0xf011d1d9), SPH_C32(0x8c810020), SPH_C32(0xe53a837f),
	  SPH_C32(0xc51a4af8), SPH_C32(0x535b307c) },
	{ SPH_C32(0xd0f30020), SPH_C32(0x2c804e6d), SPH_C32(0xbcb34701),
	  SPH_C32(0x30b21104), SPH_C32(0xdddd0030), SPH_C32(0xa5c9f184),
	  SPH_C32(0x39fd6cfa), SPH_C32(0x220c601d) },
	{ SPH_C32(0x81af0030), SPH_C32(0x6c733c96), SPH_C32(0x40546103),
	  SPH_C32(0x41e54165), SPH_C32(0xf3e40030), SPH_C32(0xc114970d),
	  SPH_C32(0x05296a06), SPH_C32(0x93f8f0a1) },
	{ SPH_C32(0x4dce0040), SPH_C32(0x3b5bec7e), SPH_C32(0x36656ba8),
	  SPH_C32(0x23633a05), SPH_C32(0x78ab0000), SPH_C32(0xa0cd5a34),
	  SPH_C32(0x5d5ca0f7), SPH_C32(0x727784cb) },
	{ SPH_C32(0x1c920050), SPH_C32(0x7ba89e85), SPH_C32(0xca824daa),
	  SPH_C32(0x52346a64), SPH_C32(0x56920000), SPH_C32(0xc4103cbd),
	  SPH_C32(0x6188a60b), SPH_C32(0xc3831477) },
	{ SPH_C32(0x63f70040), SPH_C32(0x5f868af7), SPH_C32(0x0ab16d54),
	  SPH_C32(0x9297aab9), SPH_C32(0x07ce0010), SPH_C32(0x84e34e46),
	  SPH_C32(0x9d6f8009), SPH_C32(0xb2d44416) },
	{ SPH_C32(0x32ab0050), SPH_C32(0x1f75f80c), SPH_C32(0xf6564b56),
	  SPH_C32(0xe3c0fad8), SPH_C32(0x29f70010), SPH_C32(0xe03e28cf),
	  SPH_C32(0xa1bb86f5), SPH_C32(0x0320d4aa) },
	{ SPH_C32(0xef760060), SPH_C32(0xbabc0988), SPH_C32(0xcfab27ac),
	  SPH_C32(0xc1cc9ac5), SPH_C32(0x24d90000), SPH_C32(0x69779726),
	  SPH_C32(0x24f5ad0e), SPH_C32(0x119ea5b3) },
	{ SPH_C32(0xbe2a0070), SPH_C32(0xfa4f7b73), SPH_C32(0x334c01ae),
	  SPH_C32(0xb09bcaa4), SPH_C32(0x0ae00000), SPH_C32(0x0daaf1af),
	  SPH_C32(0x1821abf2), SPH_C32(0xa06a350f) },
	{ SPH_C32(0xc14f0060), SPH_C32(0xde616f01), SPH_C32(0xf37f2150),
	  SPH_C32(0x70380a79), SPH_C32(0x5bbc0010), SPH_C32(0x4d598354),
	  SPH_C32(0xe4c68df0), SPH_C32(0xd13d656e) },
	{ SPH_C32(0x90130070), SPH_C32(0x9e921dfa), SPH_C32(0x0f980752),
	  SPH_C32(0x016f5a18), SPH_C32(0x75850010), SPH_C32(0x2984e5dd),
	  SPH_C32(0xd8128b0c), SPH_C32(0x60c9f5d2) },
	{ SPH_C32(0x11bc0040), SPH_C32(0xf2e1216c), SPH_C32(0x4fcc6651),
	  SPH_C32(0x408a1b7d), SPH_C32(0x86610020), SPH_C32(0xe89072d0),
	  SPH_C32(0xdd3be10a), SPH_C32(0xf3310573) },
	{ SPH_C32(0x40e00050), SPH_C32(0xb2125397), SPH_C32(0xb32b4053),
	  SPH_C32(0x31dd4b1c), SPH_C32(0xa8580020), SPH_C32(0x8c4d1459),
	  SPH_C32(0xe1efe7f6), SPH_C32(0x42c595cf) },
	{ SPH_C32(0x3f850040), SPH_C32(0x963c47e5), SPH_C32(0x731860ad),
	  SPH_C32(0xf17e8bc1), SPH_C32(0xf9040030), SPH_C32(0xccbe66a2),
	  SPH_C32(0x1d08c1f4), SPH_C32(0x3392c5ae) },
	{ SPH_C32(0x6ed90050), SPH_C32(0xd6cf351e), SPH_C32(0x8fff46af),
	  SPH_C32(0x8029dba0), SPH_C32(0xd73d0030), SPH_C32(0xa863002b),
	  SPH_C32(0x21dcc708), SPH_C32(0x82665512) },
	{ SPH_C32(0xb3040060), SPH_C32(0x7306c49a), SPH_C32(0xb6022a55),
	  SPH_C32(0xa225bbbd), SPH_C32(0xda130020), SPH_C32(0x212abfc2),
	  SPH_C32(0xa492ecf3), SPH_C32(0x90d8240b) },
	{ SPH_C32(0xe2580070), SPH_C32(0x33f5b661), SPH_C32(0x4ae50c57),
	  SPH_C32(0xd372ebdc), SPH_C32(0xf42a0020), SPH_C32(0x45f7d94b),
	  SPH_C32(0x9846ea0f), SPH_C32(0x212cb4b7) },
	{ SPH_C32(0x9d3d0060), SPH_C32(0x17dba213), SPH_C32(0x8ad62ca9),
	  SPH_C32(0x13d12b01), SPH_C32(0xa5760030), SPH_C32(0x0504abb0),
	  SPH_C32(0x64a1cc0d), SPH_C32(0x507be4d6) },
	{ SPH_C32(0xcc610070), SPH_C32(0x5728d0e8), SPH_C32(0x76310aab),
	  SPH_C32(0x62867b60), SPH_C32(0x8b4f0030), SPH_C32(0x61d9cd39),
	  SPH_C32(0x5875caf1), SPH_C32(0xe18f746a) },
	{ SPH_C32(0x78ab0000), SPH_C32(0xa0cd5a34), SPH_C32(0x5d5ca0f7),
	  SPH_C32(0x727784cb), SPH_C32(0x35650040), SPH_C32(0x9b96b64a),
	  SPH_C32(0x6b39cb5f), SPH_C32(0x5114bece) },
	{ SPH_C32(0x29f70010), SPH_C32(0xe03e28cf), SPH_C32(0xa1bb86f5),
	  SPH_C32(0x0320d4aa), SPH_C32(0x1b5c0040), SPH_C32(0xff4bd0c3),
	  SPH_C32(0x57edcda3), SPH_C32(0xe0e02e72) },
	{ SPH_C32(0x56920000), SPH_C32(0xc4103cbd), SPH_C32(0x6188a60b),
	  SPH_C32(0xc3831477), SPH_C32(0x4a000050), SPH_C32(0xbfb8a238),
	  SPH_C32(0xab0aeba1), SPH_C32(0x91b77e13) },
	{ SPH_C32(0x07ce0010), SPH_C32(0x84e34e46), SPH_C32(0x9d6f8009),
	  SPH_C32(0xb2d44416), SPH_C32(0x64390050), SPH_C32(0xdb65c4b1),
	  SPH_C32(0x97deed5d), SPH_C32(0x2043eeaf) },
	{ SPH_C32(0xda130020), SPH_C32(0x212abfc2), SPH_C32(0xa492ecf3),
	  SPH_C32(0x90d8240b), SPH_C32(0x69170040), SPH_C32(0x522c7b58),
	  SPH_C32(0x1290c6a6), SPH_C32(0x32fd9fb6) },
	{ SPH_C32(0x8b4f0030), SPH_C32(0x61d9cd39), SPH_C32(0x5875caf1),
	  SPH_C32(0xe18f746a), SPH_C32(0x472e0040), SPH_C32(0x36f11dd1),
	  SPH_C32(0x2e44c05a), SPH_C32(0x83090f0a) },
	{ SPH_C32(0xf42a0020), SPH_C32(0x45f7d94b), SPH_C32(0x9846ea0f),
	  SPH_C32(0x212cb4b7), SPH_C32(0x16720050), SPH_C32(0x76026f2a),
	  SPH_C32(0xd2a3e658), SPH_C32(0xf25e5f6b) },
	{ SPH_C32(0xa5760030), SPH_C32(0x0504abb0), SPH_C32(0x64a1cc0d),
	  SPH_C32(0x507be4d6), SPH_C32(0x384b0050), SPH_C32(0x12df09a3),
	  SPH_C32(0xee77e0a4), SPH_C32(0x43aacfd7) },
	{ SPH_C32(0x24d90000), SPH_C32(0x69779726), SPH_C32(0x24f5ad0e),
	  SPH_C32(0x119ea5b3), SPH_C32(0xcbaf0060), SPH_C32(0xd3cb9eae),
	  SPH_C32(0xeb5e8aa2), SPH_C32(0xd0523f76) },
	{ SPH_C32(0x75850010), SPH_C32(0x2984e5dd), SPH_C32(0xd8128b0c),
	  SPH_C32(0x60c9f5d2), SPH_C32(0xe5960060), SPH_C32(0xb716f827),
	  SPH_C32(0xd78a8c5e), SPH_C32(0x61a6afca) },
	{ SPH_C32(0x0ae00000), SPH_C32(0x0daaf1af), SPH_C32(0x1821abf2),
	  SPH_C32(0xa06a350f), SPH_C32(0xb4ca0070), SPH_C32(0xf7e58adc),
	  SPH_C32(0x2b6daa5c), SPH_C32(0x10f1ffab) },
	{ SPH_C32(0x5bbc0010), SPH_C32(0x4d598354), SPH_C32(0xe4c68df0),
	  SPH_C32(0xd13d656e), SPH_C32(0x9af30070), SPH_C32(0x9338ec55),
	  SPH_C32(0x17b9aca0), SPH_C32(0xa1056f17) },
	{ SPH_C32(0x86610020), SPH_C32(0xe89072d0), SPH_C32(0xdd3be10a),
	  SPH_C32(0xf3310573), SPH_C32(0x97dd0060), SPH_C32(0x1a7153bc),
	  SPH_C32(0x92f7875b), SPH_C32(0xb3bb1e0e) },
	{ SPH_C32(0xd73d0030), SPH_C32(0xa863002b), SPH_C32(0x21dcc708),
	  SPH_C32(0x82665512), SPH_C32(0xb9e40060), SPH_C32(0x7eac3535),
	  SPH_C32(0xae2381a7), SPH_C32(0x024f8eb2) },
	{ SPH_C32(0xa8580020), SPH_C32(0x8c4d1459), SPH_C32(0xe1efe7f6),
	  SPH_C32(0x42c595cf), SPH_C32(0xe8b80070), SPH_C32(0x3e5f47ce),
	  SPH_C32(0x52c4a7a5), SPH_C32(0x7318ded3) },
	{ SPH_C32(0xf9040030), SPH_C32(0xccbe66a2), SPH_C32(0x1d08c1f4),
	  SPH_C32(0x3392c5ae), SPH_C32(0xc6810070), SPH_C32(0x5a822147),
	  SPH_C32(0x6e10a159), SPH_C32(0xc2ec4e6f) },
	{ SPH_C32(0x35650040), SPH_C32(0x9b96b64a), SPH_C32(0x6b39cb5f),
	  SPH_C32(0x5114bece), SPH_C32(0x4dce0040), SPH_C32(0x3b5bec7e),
	  SPH_C32(0x36656ba8), SPH_C32(0x23633a05) },
	{ SPH_C32(0x64390050), SPH_C32(0xdb65c4b1), SPH_C32(0x97deed5d),
	  SPH_C32(0x2043eeaf), SPH_C32(0x63f70040), SPH_C32(0x5f868af7),
	  SPH_C32(0x0ab16d54), SPH_C32(0x9297aab9) },
	{ SPH_C32(0x1b5c0040), SPH_C32(0xff4bd0c3), SPH_C32(0x57edcda3),
	  SPH_C32(0xe0e02e72), SPH_C32(0x32ab0050), SPH_C32(0x1f75f80c),
	  SPH_C32(0xf6564b56), SPH_C32(0xe3c0fad8) },
	{ SPH_C32(0x4a000050), SPH_C32(0xbfb8a238), SPH_C32(0xab0aeba1),
	  SPH_C32(0x91b77e13), SPH_C32(0x1c920050), SPH_C32(0x7ba89e85),
	  SPH_C32(0xca824daa), SPH_C32(0x52346a64) },
	{ SPH_C32(0x97dd0060), SPH_C32(0x1a7153bc), SPH_C32(0x92f7875b),
	  SPH_C32(0xb3bb1e0e), SPH_C32(0x11bc0040), SPH_C32(0xf2e1216c),
	  SPH_C32(0x4fcc6651), SPH_C32(0x408a1b7d) },
	{ SPH_C32(0xc6810070), SPH_C32(0x5a822147), SPH_C32(0x6e10a159),
	  SPH_C32(0xc2ec4e6f), SPH_C32(0x3f850040), SPH_C32(0x963c47e5),
	  SPH_C32(0x731860ad), SPH_C32(0xf17e8bc1) },
	{ SPH_C32(0xb9e40060), SPH_C32(0x7eac3535), SPH_C32(0xae2381a7),
	  SPH_C32(0x024f8eb2), SPH_C32(0x6ed90050), SPH_C32(0xd6cf351e),
	  SPH_C32(0x8fff46af), SPH_C32(0x8029dba0) },
	{ SPH_C32(0xe8b80070), SPH_C32(0x3e5f47ce), SPH_C32(0x52c4a7a5),
	  SPH_C32(0x7318ded3), SPH_C32(0x40e00050), SPH_C32(0xb2125397),
	  SPH_C32(0xb32b4053), SPH_C32(0x31dd4b1c) },
	{ SPH_C32(0x69170040), SPH_C32(0x522c7b58), SPH_C32(0x1290c6a6),
	  SPH_C32(0x32fd9fb6), SPH_C32(0xb3040060), SPH_C32(0x7306c49a),
	  SPH_C32(0xb6022a55), SPH_C32(0xa225bbbd) },
	{ SPH_C32(0x384b0050), SPH_C32(0x12df09a3), SPH_C32(0xee77e0a4),
	  SPH_C32(0x43aacfd7), SPH_C32(0x9d3d0060), SPH_C32(0x17dba213),
	  SPH_C32(0x8ad62ca9), SPH_C32(0x13d12b01) },
	{ SPH_C32(0x472e0040), SPH_C32(0x36f11dd1), SPH_C32(0x2e44c05a),
	  SPH_C32(0x83090f0a), SPH_C32(0xcc610070), SPH_C32(0x5728d0e8),
	  SPH_C32(0x76310aab), SPH_C32(0x62867b60) },
	{ SPH_C32(0x16720050), SPH_C32(0x76026f2a), SPH_C32(0xd2a3e658),
	  SPH_C32(0xf25e5f6b), SPH_C32(0xe2580070), SPH_C32(0x33f5b661),
	  SPH_C32(0x4ae50c57), SPH_C32(0xd372ebdc) },
	{ SPH_C32(0xcbaf0060), SPH_C32(0xd3cb9eae), SPH_C32(0xeb5e8aa2),
	  SPH_C32(0xd0523f76), SPH_C32(0xef760060), SPH_C32(0xbabc0988),
	  SPH_C32(0xcfab27ac), SPH_C32(0xc1cc9ac5) },
	{ SPH_C32(0x9af30070), SPH_C32(0x9338ec55), SPH_C32(0x17b9aca0),
	  SPH_C32(0xa1056f17), SPH_C32(0xc14f0060), SPH_C32(0xde616f01),
	  SPH_C32(0xf37f2150), SPH_C32(0x70380a79) },
	{ SPH_C32(0xe5960060), SPH_C32(0xb716f827), SPH_C32(0xd78a8c5e),
	  SPH_C32(0x61a6afca), SPH_C32(0x90130070), SPH_C32(0x9e921dfa),
	  SPH_C32(0x0f980752), SPH_C32(0x016f5a18) },
	{ SPH_C32(0xb4ca0070), SPH_C32(0xf7e58adc), SPH_C32(0x2b6daa5c),
	  SPH_C32(0x10f1ffab), SPH_C32(0xbe2a0070), SPH_C32(0xfa4f7b73),
	  SPH_C32(0x334c01ae), SPH_C32(0xb09bcaa4) },
	{ SPH_C32(0x5bd20080), SPH_C32(0x450f18ec), SPH_C32(0xc2c46c55),
	  SPH_C32(0xf362b233), SPH_C32(0x39a60000), SPH_C32(0x4ab753eb),
	  SPH_C32(0xd14e094b), SPH_C32(0xb772b42b) },
	{ SPH_C32(0x0a8e0090), SPH_C32(0x05fc6a17), SPH_C32(0x3e234a57),
	  SPH_C32(0x8235e252), SPH_C32(0x179f0000), SPH_C32(0x2e6a3562),
	  SPH_C32(0xed9a0fb7), SPH_C32(0x06862497) },
	{ SPH_C32(0x75eb0080), SPH_C32(0x21d27e65), SPH_C32(0xfe106aa9),
	  SPH_C32(0x4296228f), SPH_C32(0x46c30010), SPH_C32(0x6e994799),
	  SPH_C32(0x117d29b5), SPH_C32(0x77d174f6) },
	{ SPH_C32(0x24b70090), SPH_C32(0x61210c9e), SPH_C32(0x02f74cab),
	  SPH_C32(0x33c172ee), SPH_C32(0x68fa0010), SPH_C32(0x0a442110),
	  SPH_C32(0x2da92f49), SPH_C32(0xc625e44a) },
	{ SPH_C32(0xf96a00a0), SPH_C32(0xc4e8fd1a), SPH_C32(0x3b0a2051),
	  SPH_C32(0x11cd12f3), SPH_C32(0x65d40000), SPH_C32(0x830d9ef9),
	  SPH_C32(0xa8e704b2), SPH_C32(0xd49b9553) },
	{ SPH_C32(0xa83600b0), SPH_C32(0x841b8fe1), SPH_C32(0xc7ed0653),
	  SPH_C32(0x609a4292), SPH_C32(0x4bed0000), SPH_C32(0xe7d0f870),
	  SPH_C32(0x9433024e), SPH_C32(0x656f05ef) },
	{ SPH_C32(0xd75300a0), SPH_C32(0xa0359b93), SPH_C32(0x07de26ad),
	  SPH_C32(0xa039824f), SPH_C32(0x1ab10010), SPH_C32(0xa7238a8b),
	  SPH_C32(0x68d4244c), SPH_C32(0x1438558e) },
	{ SPH_C32(0x860f00b0), SPH_C32(0xe0c6e968), SPH_C32(0xfb3900af),
	  SPH_C32(0xd16ed22e), SPH_C32(0x34880010), SPH_C32(0xc3feec02),
	  SPH_C32(0x540022b0), SPH_C32(0xa5ccc532) },
	{ SPH_C32(0x07a00080), SPH_C32(0x8cb5d5fe), SPH_C32(0xbb6d61ac),
	  SPH_C32(0x908b934b), SPH_C32(0xc76c0020), SPH_C32(0x02ea7b0f),
	  SPH_C32(0x512948b6), SPH_C32(0x36343593) },
	{ SPH_C32(0x56fc0090), SPH_C32(0xcc46a705), SPH_C32(0x478a47ae),
	  SPH_C32(0xe1dcc32a), SPH_C32(0xe9550020), SPH_C32(0x66371d86),
	  SPH_C32(0x6dfd4e4a), SPH_C32(0x87c0a52f) },
	{ SPH_C32(0x29990080), SPH_C32(0xe868b377), SPH_C32(0x87b96750),
	  SPH_C32(0x217f03f7), SPH_C32(0xb8090030), SPH_C32(0x26c46f7d),
	  SPH_C32(0x911a6848), SPH_C32(0xf697f54e) },
	{ SPH_C32(0x78c50090), SPH_C32(0xa89bc18c), SPH_C32(0x7b5e4152),
	  SPH_C32(0x50285396), SPH_C32(0x96300030), SPH_C32(0x421909f4),
	  SPH_C32(0xadce6eb4), SPH_C32(0x476365f2) },
	{ SPH_C32(0xa51800a0), SPH_C32(0x0d523008), SPH_C32(0x42a32da8),
	  SPH_C32(0x7224338b), SPH_C32(0x9b1e0020), SPH_C32(0xcb50b61d),
	  SPH_C32(0x2880454f), SPH_C32(0x55dd14eb) },
	{ SPH_C32(0xf44400b0), SPH_C32(0x4da142f3), SPH_C32(0xbe440baa),
	  SPH_C32(0x037363ea), SPH_C32(0xb5270020), SPH_C32(0xaf8dd094),
	  SPH_C32(0x145443b3), SPH_C32(0xe4298457) },
	{ SPH_C32(0x8b2100a0), SPH_C32(0x698f5681), SPH_C32(0x7e772b54),
	  SPH_C32(0xc3d0a337), SPH_C32(0xe47b0030), SPH_C32(0xef7ea26f),
	  SPH_C32(0xe8b365b1), SPH_C32(0x957ed436) },
	{ SPH_C32(0xda7d00b0), SPH_C32(0x297c247a), SPH_C32(0x82900d56),
	  SPH_C32(0xb287f356), SPH_C32(0xca420030), SPH_C32(0x8ba3c4e6),
	  SPH_C32(0xd467634d), SPH_C32(0x248a448a) },
	{ SPH_C32(0x161c00c0), SPH_C32(0x7e54f492), SPH_C32(0xf4a107fd),
	  SPH_C32(0xd0018836), SPH_C32(0x410d0000), SPH_C32(0xea7a09df),
	  SPH_C32(0x8c12a9bc), SPH_C32(0xc50530e0) },
	{ SPH_C32(0x474000d0), SPH_C32(0x3ea78669), SPH_C32(0x084621ff),
	  SPH_C32(0xa156d857), SPH_C32(0x6f340000), SPH_C32(0x8ea76f56),
	  SPH_C32(0xb0c6af40), SPH_C32(0x74f1a05c) },
	{ SPH_C32(0x382500c0), SPH_C32(0x1a89921b), SPH_C32(0xc8750101),
	  SPH_C32(0x61f5188a), SPH_C32(0x3e680010), SPH_C32(0xce541dad),
	  SPH_C32(0x4c218942), SPH_C32(0x05a6f03d) },
	{ SPH_C32(0x697900d0), SPH_C32(0x5a7ae0e0), SPH_C32(0x34922703),
	  SPH_C32(0x10a248eb), SPH_C32(0x10510010), SPH_C32(0xaa897b24),
	  SPH_C32(0x70f58fbe), SPH_C32(0xb4526081) },
	{ SPH_C32(0xb4a400e0), SPH_C32(0xffb31164), SPH_C32(0x0d6f4bf9),
	  SPH_C32(0x32ae28f6), SPH_C32(0x1d7f0000), SPH_C32(0x23c0c4cd),
	  SPH_C32(0xf5bba445), SPH_C32(0xa6ec1198) },
	{ SPH_C32(0xe5f800f0), SPH_C32(0xbf40639f), SPH_C32(0xf1886dfb),
	  SPH_C32(0x43f97897), SPH_C32(0x33460000), SPH_C32(0x471da244),
	  SPH_C32(0xc96fa2b9), SPH_C32(0x17188124) },
	{ SPH_C32(0x9a9d00e0), SPH_C32(0x9b6e77ed), SPH_C32(0x31bb4d05),
	  SPH_C32(0x835ab84a), SPH_C32(0x621a0010), SPH_C32(0x07eed0bf),
	  SPH_C32(0x358884bb), SPH_C32(0x664fd145) },
	{ SPH_C32(0xcbc100f0), SPH_C32(0xdb9d0516), SPH_C32(0xcd5c6b07),
	  SPH_C32(0xf20de82b), SPH_C32(0x4c230010), SPH_C32(0x6333b636),
	  SPH_C32(0x095c8247), SPH_C32(0xd7bb41f9) },
	{ SPH_C32(0x4a6e00c0), SPH_C32(0xb7ee3980), SPH_C32(0x8d080a04),
	  SPH_C32(0xb3e8a94e), SPH_C32(0xbfc70020), SPH_C32(0xa227213b),
	  SPH_C32(0x0c75e841), SPH_C32(0x4443b158) },
	{ SPH_C32(0x1b3200d0), SPH_C32(0xf71d4b7b), SPH_C32(0x71ef2c06),
	  SPH_C32(0xc2bff92f), SPH_C32(0x91fe0020), SPH_C32(0xc6fa47b2),
	  SPH_C32(0x30a1eebd), SPH_C32(0xf5b721e4) },
	{ SPH_C32(0x645700c0), SPH_C32(0xd3335f09), SPH_C32(0xb1dc0cf8),
	  SPH_C32(0x021c39f2), SPH_C32(0xc0a20030), SPH_C32(0x86093549),
	  SPH_C32(0xcc46c8bf), SPH_C32(0x84e07185) },
	{ SPH_C32(0x350b00d0), SPH_C32(0x93c02df2), SPH_C32(0x4d3b2afa),
	  SPH_C32(0x734b6993), SPH_C32(0xee9b0030), SPH_C32(0xe2d453c0),
	  SPH_C32(0xf092ce43), SPH_C32(0x3514e139) },
	{ SPH_C32(0xe8d600e0), SPH_C32(0x3609dc76), SPH_C32(0x74c64600),
	  SPH_C32(0x5147098e), SPH_C32(0xe3b50020), SPH_C32(0x6b9dec29),
	  SPH_C32(0x75dce5b8), SPH_C32(0x27aa9020) },
	{ SPH_C32(0xb98a00f0), SPH_C32(0x76faae8d), SPH_C32(0x88216002),
	  SPH_C32(0x201059ef), SPH_C32(0xcd8c0020), SPH_C32(0x0f408aa0),
	  SPH_C32(0x4908e344), SPH_C32(0x965e009c) },
	{ SPH_C32(0xc6ef00e0), SPH_C32(0x52d4baff), SPH_C32(0x481240fc),
	  SPH_C32(0xe0b39932), SPH_C32(0x9cd00030), SPH_C32(0x4fb3f85b),
	  SPH_C32(0xb5efc546), SPH_C32(0xe70950fd) },
	{ SPH_C32(0x97b300f0), SPH_C32(0x1227c804), SPH_C32(0xb4f566fe),
	  SPH_C32(0x91e4c953), SPH_C32(0xb2e90030), SPH_C32(0x2b6e9ed2),
	  SPH_C32(0x893bc3ba), SPH_C32(0x56fdc041) },
	{ SPH_C32(0x23790080), SPH_C32(0xe5c242d8), SPH_C32(0x9f98cca2),
	  SPH_C32(0x811536f8), SPH_C32(0x0cc30040), SPH_C32(0xd121e5a1),
	  SPH_C32(0xba77c214), SPH_C32(0xe6660ae5) },
	{ SPH_C32(0x72250090), SPH_C32(0xa5313023), SPH_C32(0x637feaa0),
	  SPH_C32(0xf0426699), SPH_C32(0x22fa0040), SPH_C32(0xb5fc8328),
	  SPH_C32(0x86a3c4e8), SPH_C32(0x57929a59) },
	{ SPH_C32(0x0d400080), SPH_C32(0x811f2451), SPH_C32(0xa34cca5e),
	  SPH_C32(0x30e1a644), SPH_C32(0x73a60050), SPH_C32(0xf50ff1d3),
	  SPH_C32(0x7a44e2ea), SPH_C32(0x26c5ca38) },
	{ SPH_C32(0x5c1c0090), SPH_C32(0xc1ec56aa), SPH_C32(0x5fabec5c),
	  SPH_C32(0x41b6f625), SPH_C32(0x5d9f0050), SPH_C32(0x91d2975a),
	  SPH_C32(0x4690e416), SPH_C32(0x97315a84) },
	{ SPH_C32(0x81c100a0), SPH_C32(0x6425a72e), SPH_C32(0x665680a6),
	  SPH_C32(0x63ba9638), SPH_C32(0x50b10040), SPH_C32(0x189b28b3),
	  SPH_C32(0xc3decfed), SPH_C32(0x858f2b9d) },
	{ SPH_C32(0xd09d00b0), SPH_C32(0x24d6d5d5), SPH_C32(0x9ab1a6a4),
	  SPH_C32(0x12edc659), SPH_C32(0x7e880040), SPH_C32(0x7c464e3a),
	  SPH_C32(0xff0ac911), SPH_C32(0x347bbb21) },
	{ SPH_C32(0xaff800a0), SPH_C32(0x00f8c1a7), SPH_C32(0x5a82865a),
	  SPH_C32(0xd24e0684), SPH_C32(0x2fd40050), SPH_C32(0x3cb53cc1),
	  SPH_C32(0x03edef13), SPH_C32(0x452ceb40) },
	{ SPH_C32(0xfea400b0), SPH_C32(0x400bb35c), SPH_C32(0xa665a058),
	  SPH_C32(0xa31956e5), SPH_C32(0x01ed0050), SPH_C32(0x58685a48),
	  SPH_C32(0x3f39e9ef), SPH_C32(0xf4d87bfc) },
	{ SPH_C32(0x7f0b0080), SPH_C32(0x2c788fca), SPH_C32(0xe631c15b),
	  SPH_C32(0xe2fc1780), SPH_C32(0xf2090060), SPH_C32(0x997ccd45),
	  SPH_C32(0x3a1083e9), SPH_C32(0x67208b5d) },
	{ SPH_C32(0x2e570090), SPH_C32(0x6c8bfd31), SPH_C32(0x1ad6e759),
	  SPH_C32(0x93ab47e1), SPH_C32(0xdc300060), SPH_C32(0xfda1abcc),
	  SPH_C32(0x06c48515), SPH_C32(0xd6d41be1) },
	{ SPH_C32(0x51320080), SPH_C32(0x48a5e943), SPH_C32(0xdae5c7a7),
	  SPH_C32(0x5308873c), SPH_C32(0x8d6c0070), SPH_C32(0xbd52d937),
	  SPH_C32(0xfa23a317), SPH_C32(0xa7834b80) },
	{ SPH_C32(0x006e0090), SPH_C32(0x08569bb8), SPH_C32(0x2602e1a5),
	  SPH_C32(0x225fd75d), SPH_C32(0xa3550070), SPH_C32(0xd98fbfbe),
	  SPH_C32(0xc6f7a5eb), SPH_C32(0x1677db3c) },
	{ SPH_C32(0xddb300a0), SPH_C32(0xad9f6a3c), SPH_C32(0x1fff8d5f),
	  SPH_C32(0x0053b740), SPH_C32(0xae7b0060), SPH_C32(0x50c60057),
	  SPH_C32(0x43b98e10), SPH_C32(0x04c9aa25) },
	{ SPH_C32(0x8cef00b0), SPH_C32(0xed6c18c7), SPH_C32(0xe318ab5d),
	  SPH_C32(0x7104e721), SPH_C32(0x80420060), SPH_C32(0x341b66de),
	  SPH_C32(0x7f6d88ec), SPH_C32(0xb53d3a99) },
	{ SPH_C32(0xf38a00a0), SPH_C32(0xc9420cb5), SPH_C32(0x232b8ba3),
	  SPH_C32(0xb1a727fc), SPH_C32(0xd11e0070), SPH_C32(0x74e81425),
	  SPH_C32(0x838aaeee), SPH_C32(0xc46a6af8) },
	{ SPH_C32(0xa2d600b0), SPH_C32(0x89b17e4e), SPH_C32(0xdfccada1),
	  SPH_C32(0xc0f0779d), SPH_C32(0xff270070), SPH_C32(0x103572ac),
	  SPH_C32(0xbf5ea812), SPH_C32(0x759efa44) },
	{ SPH_C32(0x6eb700c0), SPH_C32(0xde99aea6), SPH_C32(0xa9fda70a),
	  SPH_C32(0xa2760cfd), SPH_C32(0x74680040), SPH_C32(0x71ecbf95),
	  SPH_C32(0xe72b62e3), SPH_C32(0x94118e2e) },
	{ SPH_C32(0x3feb00d0), SPH_C32(0x9e6adc5d), SPH_C32(0x551a8108),
	  SPH_C32(0xd3215c9c), SPH_C32(0x5a510040), SPH_C32(0x1531d91c),
	  SPH_C32(0xdbff641f), SPH_C32(0x25e51e92) },
	{ SPH_C32(0x408e00c0), SPH_C32(0xba44c82f), SPH_C32(0x9529a1f6),
	  SPH_C32(0x13829c41), SPH_C32(0x0b0d0050), SPH_C32(0x55c2abe7),
	  SPH_C32(0x2718421d), SPH_C32(0x54b24ef3) },
	{ SPH_C32(0x11d200d0), SPH_C32(0xfab7bad4), SPH_C32(0x69ce87f4),
	  SPH_C32(0x62d5cc20), SPH_C32(0x25340050), SPH_C32(0x311fcd6e),
	  SPH_C32(0x1bcc44e1), SPH_C32(0xe546de4f) },
	{ SPH_C32(0xcc0f00e0), SPH_C32(0x5f7e4b50), SPH_C32(0x5033eb0e),
	  SPH_C32(0x40d9ac3d), SPH_C32(0x281a0040), SPH_C32(0xb8567287),
	  SPH_C32(0x9e826f1a), SPH_C32(0xf7f8af56) },
	{ SPH_C32(0x9d5300f0), SPH_C32(0x1f8d39ab), SPH_C32(0xacd4cd0c),
	  SPH_C32(0x318efc5c), SPH_C32(0x06230040), SPH_C32(0xdc8b140e),
	  SPH_C32(0xa25669e6), SPH_C32(0x460c3fea) },
	{ SPH_C32(0xe23600e0), SPH_C32(0x3ba32dd9), SPH_C32(0x6ce7edf2),
	  SPH_C32(0xf12d3c81), SPH_C32(0x577f0050), SPH_C32(0x9c7866f5),
	  SPH_C32(0x5eb14fe4), SPH_C32(0x375b6f8b) },
	{ SPH_C32(0xb36a00f0), SPH_C32(0x7b505f22), SPH_C32(0x9000cbf0),
	  SPH_C32(0x807a6ce0), SPH_C32(0x79460050), SPH_C32(0xf8a5007c),
	  SPH_C32(0x62654918), SPH_C32(0x86afff37) },
	{ SPH_C32(0x32c500c0), SPH_C32(0x172363b4), SPH_C32(0xd054aaf3),
	  SPH_C32(0xc19f2d85), SPH_C32(0x8aa20060), SPH_C32(0x39b19771),
	  SPH_C32(0x674c231e), SPH_C32(0x15570f96) },
	{ SPH_C32(0x639900d0), SPH_C32(0x57d0114f), SPH_C32(0x2cb38cf1),
	  SPH_C32(0xb0c87de4), SPH_C32(0xa49b0060), SPH_C32(0x5d6cf1f8),
	  SPH_C32(0x5b9825e2), SPH_C32(0xa4a39f2a) },
	{ SPH_C32(0x1cfc00c0), SPH_C32(0x73fe053d), SPH_C32(0xec80ac0f),
	  SPH_C32(0x706bbd39), SPH_C32(0xf5c70070), SPH_C32(0x1d9f8303),
	  SPH_C32(0xa77f03e0), SPH_C32(0xd5f4cf4b) },
	{ SPH_C32(0x4da000d0), SPH_C32(0x330d77c6), SPH_C32(0x10678a0d),
	  SPH_C32(0x013ced58), SPH_C32(0xdbfe0070), SPH_C32(0x7942e58a),
	  SPH_C32(0x9bab051c), SPH_C32(0x64005ff7) },
	{ SPH_C32(0x907d00e0), SPH_C32(0x96c48642), SPH_C32(0x299ae6f7),
	  SPH_C32(0x23308d45), SPH_C32(0xd6d00060), SPH_C32(0xf00b5a63),
	  SPH_C32(0x1ee52ee7), SPH_C32(0x76be2eee) },
	{ SPH_C32(0xc12100f0), SPH_C32(0xd637f4b9), SPH_C32(0xd57dc0f5),
	  SPH_C32(0x5267dd24), SPH_C32(0xf8e90060), SPH_C32(0x94d63cea),
	  SPH_C32(0x2231281b), SPH_C32(0xc74abe52) },
	{ SPH_C32(0xbe4400e0), SPH_C32(0xf219e0cb), SPH_C32(0x154ee00b),
	  SPH_C32(0x92c41df9), SPH_C32(0xa9b50070), SPH_C32(0xd4254e11),
	  SPH_C32(0xded60e19), SPH_C32(0xb61dee33) },
	{ SPH_C32(0xef1800f0), SPH_C32(0xb2ea9230), SPH_C32(0xe9a9c609),
	  SPH_C32(0xe3934d98), SPH_C32(0x878c0070), SPH_C32(0xb0f82898),
	  SPH_C32(0xe20208e5), SPH_C32(0x07e97e8f) },
	{ SPH_C32(0x39a60000), SPH_C32(0x4ab753eb), SPH_C32(0xd14e094b),
	  SPH_C32(0xb772b42b), SPH_C32(0x62740080), SPH_C32(0x0fb84b07),
	  SPH_C32(0x138a651e), SPH_C32(0x44100618) },
	{ SPH_C32(0x68fa0010), SPH_C32(0x0a442110), SPH_C32(0x2da92f49),
	  SPH_C32(0xc625e44a), SPH_C32(0x4c4d0080), SPH_C32(0x6b652d8e),
	  SPH_C32(0x2f5e63e2), SPH_C32(0xf5e496a4) },
	{ SPH_C32(0x179f0000), SPH_C32(0x2e6a3562), SPH_C32(0xed9a0fb7),
	  SPH_C32(0x06862497), SPH_C32(0x1d110090), SPH_C32(0x2b965f75),
	  SPH_C32(0xd3b945e0), SPH_C32(0x84b3c6c5) },
	{ SPH_C32(0x46c30010), SPH_C32(0x6e994799), SPH_C32(0x117d29b5),
	  SPH_C32(0x77d174f6), SPH_C32(0x33280090), SPH_C32(0x4f4b39fc),
	  SPH_C32(0xef6d431c), SPH_C32(0x35475679) },
	{ SPH_C32(0x9b1e0020), SPH_C32(0xcb50b61d), SPH_C32(0x2880454f),
	  SPH_C32(0x55dd14eb), SPH_C32(0x3e060080), SPH_C32(0xc6028615),
	  SPH_C32(0x6a2368e7), SPH_C32(0x27f92760) },
	{ SPH_C32(0xca420030), SPH_C32(0x8ba3c4e6), SPH_C32(0xd467634d),
	  SPH_C32(0x248a448a), SPH_C32(0x103f0080), SPH_C32(0xa2dfe09c),
	  SPH_C32(0x56f76e1b), SPH_C32(0x960db7dc) },
	{ SPH_C32(0xb5270020), SPH_C32(0xaf8dd094), SPH_C32(0x145443b3),
	  SPH_C32(0xe4298457), SPH_C32(0x41630090), SPH_C32(0xe22c9267),
	  SPH_C32(0xaa104819), SPH_C32(0xe75ae7bd) },
	{ SPH_C32(0xe47b0030), SPH_C32(0xef7ea26f), SPH_C32(0xe8b365b1),
	  SPH_C32(0x957ed436), SPH_C32(0x6f5a0090), SPH_C32(0x86f1f4ee),
	  SPH_C32(0x96c44ee5), SPH_C32(0x56ae7701) },
	{ SPH_C32(0x65d40000), SPH_C32(0x830d9ef9), SPH_C32(0xa8e704b2),
	  SPH_C32(0xd49b9553), SPH_C32(0x9cbe00a0), SPH_C32(0x47e563e3),
	  SPH_C32(0x93ed24e3), SPH_C32(0xc55687a0) },
	{ SPH_C32(0x34880010), SPH_C32(0xc3feec02), SPH_C32(0x540022b0),
	  SPH_C32(0xa5ccc532), SPH_C32(0xb28700a0), SPH_C32(0x2338056a),
	  SPH_C32(0xaf39221f), SPH_C32(0x74a2171c) },
	{ SPH_C32(0x4bed0000), SPH_C32(0xe7d0f870), SPH_C32(0x9433024e),
	  SPH_C32(0x656f05ef), SPH_C32(0xe3db00b0), SPH_C32(0x63cb7791),
	  SPH_C32(0x53de041d), SPH_C32(0x05f5477d) },
	{ SPH_C32(0x1ab10010), SPH_C32(0xa7238a8b), SPH_C32(0x68d4244c),
	  SPH_C32(0x1438558e), SPH_C32(0xcde200b0), SPH_C32(0x07161118),
	  SPH_C32(0x6f0a02e1), SPH_C32(0xb401d7c1) },
	{ SPH_C32(0xc76c0020), SPH_C32(0x02ea7b0f), SPH_C32(0x512948b6),
	  SPH_C32(0x36343593), SPH_C32(0xc0cc00a0), SPH_C32(0x8e5faef1),
	  SPH_C32(0xea44291a), SPH_C32(0xa6bfa6d8) },
	{ SPH_C32(0x96300030), SPH_C32(0x421909f4), SPH_C32(0xadce6eb4),
	  SPH_C32(0x476365f2), SPH_C32(0xeef500a0), SPH_C32(0xea82c878),
	  SPH_C32(0xd6902fe6), SPH_C32(0x174b3664) },
	{ SPH_C32(0xe9550020), SPH_C32(0x66371d86), SPH_C32(0x6dfd4e4a),
	  SPH_C32(0x87c0a52f), SPH_C32(0xbfa900b0), SPH_C32(0xaa71ba83),
	  SPH_C32(0x2a7709e4), SPH_C32(0x661c6605) },
	{ SPH_C32(0xb8090030), SPH_C32(0x26c46f7d), SPH_C32(0x911a6848),
	  SPH_C32(0xf697f54e), SPH_C32(0x919000b0), SPH_C32(0xceacdc0a),
	  SPH_C32(0x16a30f18), SPH_C32(0xd7e8f6b9) },
	{ SPH_C32(0x74680040), SPH_C32(0x71ecbf95), SPH_C32(0xe72b62e3),
	  SPH_C32(0x94118e2e), SPH_C32(0x1adf0080), SPH_C32(0xaf751133),
	  SPH_C32(0x4ed6c5e9), SPH_C32(0x366782d3) },
	{ SPH_C32(0x25340050), SPH_C32(0x311fcd6e), SPH_C32(0x1bcc44e1),
	  SPH_C32(0xe546de4f), SPH_C32(0x34e60080), SPH_C32(0xcba877ba),
	  SPH_C32(0x7202c315), SPH_C32(0x8793126f) },
	{ SPH_C32(0x5a510040), SPH_C32(0x1531d91c), SPH_C32(0xdbff641f),
	  SPH_C32(0x25e51e92), SPH_C32(0x65ba0090), SPH_C32(0x8b5b0541),
	  SPH_C32(0x8ee5e517), SPH_C32(0xf6c4420e) },
	{ SPH_C32(0x0b0d0050), SPH_C32(0x55c2abe7), SPH_C32(0x2718421d),
	  SPH_C32(0x54b24ef3), SPH_C32(0x4b830090), SPH_C32(0xef8663c8),
	  SPH_C32(0xb231e3eb), SPH_C32(0x4730d2b2) },
	{ SPH_C32(0xd6d00060), SPH_C32(0xf00b5a63), SPH_C32(0x1ee52ee7),
	  SPH_C32(0x76be2eee), SPH_C32(0x46ad0080), SPH_C32(0x66cfdc21),
	  SPH_C32(0x377fc810), SPH_C32(0x558ea3ab) },
	{ SPH_C32(0x878c0070), SPH_C32(0xb0f82898), SPH_C32(0xe20208e5),
	  SPH_C32(0x07e97e8f), SPH_C32(0x68940080), SPH_C32(0x0212baa8),
	  SPH_C32(0x0babceec), SPH_C32(0xe47a3317) },
	{ SPH_C32(0xf8e90060), SPH_C32(0x94d63cea), SPH_C32(0x2231281b),
	  SPH_C32(0xc74abe52), SPH_C32(0x39c80090), SPH_C32(0x42e1c853),
	  SPH_C32(0xf74ce8ee), SPH_C32(0x952d6376) },
	{ SPH_C32(0xa9b50070), SPH_C32(0xd4254e11), SPH_C32(0xded60e19),
	  SPH_C32(0xb61dee33), SPH_C32(0x17f10090), SPH_C32(0x263caeda),
	  SPH_C32(0xcb98ee12), SPH_C32(0x24d9f3ca) },
	{ SPH_C32(0x281a0040), SPH_C32(0xb8567287), SPH_C32(0x9e826f1a),
	  SPH_C32(0xf7f8af56), SPH_C32(0xe41500a0), SPH_C32(0xe72839d7),
	  SPH_C32(0xceb18414), SPH_C32(0xb721036b) },
	{ SPH_C32(0x79460050), SPH_C32(0xf8a5007c), SPH_C32(0x62654918),
	  SPH_C32(0x86afff37), SPH_C32(0xca2c00a0), SPH_C32(0x83f55f5e),
	  SPH_C32(0xf26582e8), SPH_C32(0x06d593d7) },
	{ SPH_C32(0x06230040), SPH_C32(0xdc8b140e), SPH_C32(0xa25669e6),
	  SPH_C32(0x460c3fea), SPH_C32(0x9b7000b0), SPH_C32(0xc3062da5),
	  SPH_C32(0x0e82a4ea), SPH_C32(0x7782c3b6) },
	{ SPH_C32(0x577f0050), SPH_C32(0x9c7866f5), SPH_C32(0x5eb14fe4),
	  SPH_C32(0x375b6f8b), SPH_C32(0xb54900b0), SPH_C32(0xa7db4b2c),
	  SPH_C32(0x3256a216), SPH_C32(0xc676530a) },
	{ SPH_C32(0x8aa20060), SPH_C32(0x39b19771), SPH_C32(0x674c231e),
	  SPH_C32(0x15570f96), SPH_C32(0xb86700a0), SPH_C32(0x2e92f4c5),
	  SPH_C32(0xb71889ed), SPH_C32(0xd4c82213) },
	{ SPH_C32(0xdbfe0070), SPH_C32(0x7942e58a), SPH_C32(0x9bab051c),
	  SPH_C32(0x64005ff7), SPH_C32(0x965e00a0), SPH_C32(0x4a4f924c),
	  SPH_C32(0x8bcc8f11), SPH_C32(0x653cb2af) },
	{ SPH_C32(0xa49b0060), SPH_C32(0x5d6cf1f8), SPH_C32(0x5b9825e2),
	  SPH_C32(0xa4a39f2a), SPH_C32(0xc70200b0), SPH_C32(0x0abce0b7),
	  SPH_C32(0x772ba913), SPH_C32(0x146be2ce) },
	{ SPH_C32(0xf5c70070), SPH_C32(0x1d9f8303), SPH_C32(0xa77f03e0),
	  SPH_C32(0xd5f4cf4b), SPH_C32(0xe93b00b0), SPH_C32(0x6e61863e),
	  SPH_C32(0x4bffafef), SPH_C32(0xa59f7272) },
	{ SPH_C32(0x410d0000), SPH_C32(0xea7a09df), SPH_C32(0x8c12a9bc),
	  SPH_C32(0xc50530e0), SPH_C32(0x571100c0), SPH_C32(0x942efd4d),
	  SPH_C32(0x78b3ae41), SPH_C32(0x1504b8d6) },
	{ SPH_C32(0x10510010), SPH_C32(0xaa897b24), SPH_C32(0x70f58fbe),
	  SPH_C32(0xb4526081), SPH_C32(0x792800c0), SPH_C32(0xf0f39bc4),
	  SPH_C32(0x4467a8bd), SPH_C32(0xa4f0286a) },
	{ SPH_C32(0x6f340000), SPH_C32(0x8ea76f56), SPH_C32(0xb0c6af40),
	  SPH_C32(0x74f1a05c), SPH_C32(0x287400d0), SPH_C32(0xb000e93f),
	  SPH_C32(0xb8808ebf), SPH_C32(0xd5a7780b) },
	{ SPH_C32(0x3e680010), SPH_C32(0xce541dad), SPH_C32(0x4c218942),
	  SPH_C32(0x05a6f03d), SPH_C32(0x064d00d0), SPH_C32(0xd4dd8fb6),
	  SPH_C32(0x84548843), SPH_C32(0x6453e8b7) },
	{ SPH_C32(0xe3b50020), SPH_C32(0x6b9dec29), SPH_C32(0x75dce5b8),
	  SPH_C32(0x27aa9020), SPH_C32(0x0b6300c0), SPH_C32(0x5d94305f),
	  SPH_C32(0x011aa3b8), SPH_C32(0x76ed99ae) },
	{ SPH_C32(0xb2e90030), SPH_C32(0x2b6e9ed2), SPH_C32(0x893bc3ba),
	  SPH_C32(0x56fdc041), SPH_C32(0x255a00c0), SPH_C32(0x394956d6),
	  SPH_C32(0x3dcea544), SPH_C32(0xc7190912) },
	{ SPH_C32(0xcd8c0020), SPH_C32(0x0f408aa0), SPH_C32(0x4908e344),
	  SPH_C32(0x965e009c), SPH_C32(0x740600d0), SPH_C32(0x79ba242d),
	  SPH_C32(0xc1298346), SPH_C32(0xb64e5973) },
	{ SPH_C32(0x9cd00030), SPH_C32(0x4fb3f85b), SPH_C32(0xb5efc546),
	  SPH_C32(0xe70950fd), SPH_C32(0x5a3f00d0), SPH_C32(0x1d6742a4),
	  SPH_C32(0xfdfd85ba), SPH_C32(0x07bac9cf) },
	{ SPH_C32(0x1d7f0000), SPH_C32(0x23c0c4cd), SPH_C32(0xf5bba445),
	  SPH_C32(0xa6ec1198), SPH_C32(0xa9db00e0), SPH_C32(0xdc73d5a9),
	  SPH_C32(0xf8d4efbc), SPH_C32(0x9442396e) },
	{ SPH_C32(0x4c230010), SPH_C32(0x6333b636), SPH_C32(0x095c8247),
	  SPH_C32(0xd7bb41f9), SPH_C32(0x87e200e0), SPH_C32(0xb8aeb320),
	  SPH_C32(0xc400e940), SPH_C32(0x25b6a9d2) },
	{ SPH_C32(0x33460000), SPH_C32(0x471da244), SPH_C32(0xc96fa2b9),
	  SPH_C32(0x17188124), SPH_C32(0xd6be00f0), SPH_C32(0xf85dc1db),
	  SPH_C32(0x38e7cf42), SPH_C32(0x54e1f9b3) },
	{ SPH_C32(0x621a0010), SPH_C32(0x07eed0bf), SPH_C32(0x358884bb),
	  SPH_C32(0x664fd145), SPH_C32(0xf88700f0), SPH_C32(0x9c80a752),
	  SPH_C32(0x0433c9be), SPH_C32(0xe515690f) },
	{ SPH_C32(0xbfc70020), SPH_C32(0xa227213b), SPH_C32(0x0c75e841),
	  SPH_C32(0x4443b158), SPH_C32(0xf5a900e0), SPH_C32(0x15c918bb),
	  SPH_C32(0x817de245), SPH_C32(0xf7ab1816) },
	{ SPH_C32(0xee9b0030), SPH_C32(0xe2d453c0), SPH_C32(0xf092ce43),
	  SPH_C32(0x3514e139), SPH_C32(0xdb9000e0), SPH_C32(0x71147e32),
	  SPH_C32(0xbda9e4b9), SPH_C32(0x465f88aa) },
	{ SPH_C32(0x91fe0020), SPH_C32(0xc6fa47b2), SPH_C32(0x30a1eebd),
	  SPH_C32(0xf5b721e4), SPH_C32(0x8acc00f0), SPH_C32(0x31e70cc9),
	  SPH_C32(0x414ec2bb), SPH_C32(0x3708d8cb) },
	{ SPH_C32(0xc0a20030), SPH_C32(0x86093549), SPH_C32(0xcc46c8bf),
	  SPH_C32(0x84e07185), SPH_C32(0xa4f500f0), SPH_C32(0x553a6a40),
	  SPH_C32(0x7d9ac447), SPH_C32(0x86fc4877) },
	{ SPH_C32(0x0cc30040), SPH_C32(0xd121e5a1), SPH_C32(0xba77c214),
	  SPH_C32(0xe6660ae5), SPH_C32(0x2fba00c0), SPH_C32(0x34e3a779),
	  SPH_C32(0x25ef0eb6), SPH_C32(0x67733c1d) },
	{ SPH_C32(0x5d9f0050), SPH_C32(0x91d2975a), SPH_C32(0x4690e416),
	  SPH_C32(0x97315a84), SPH_C32(0x018300c0), SPH_C32(0x503ec1f0),
	  SPH_C32(0x193b084a), SPH_C32(0xd687aca1) },
	{ SPH_C32(0x22fa0040), SPH_C32(0xb5fc8328), SPH_C32(0x86a3c4e8),
	  SPH_C32(0x57929a59), SPH_C32(0x50df00d0), SPH_C32(0x10cdb30b),
	  SPH_C32(0xe5dc2e48), SPH_C32(0xa7d0fcc0) },
	{ SPH_C32(0x73a60050), SPH_C32(0xf50ff1d3), SPH_C32(0x7a44e2ea),
	  SPH_C32(0x26c5ca38), SPH_C32(0x7ee600d0), SPH_C32(0x7410d582),
	  SPH_C32(0xd90828b4), SPH_C32(0x16246c7c) },
	{ SPH_C32(0xae7b0060), SPH_C32(0x50c60057), SPH_C32(0x43b98e10),
	  SPH_C32(0x04c9aa25), SPH_C32(0x73c800c0), SPH_C32(0xfd596a6b),
	  SPH_C32(0x5c46034f), SPH_C32(0x049a1d65) },
	{ SPH_C32(0xff270070), SPH_C32(0x103572ac), SPH_C32(0xbf5ea812),
	  SPH_C32(0x759efa44), SPH_C32(0x5df100c0), SPH_C32(0x99840ce2),
	  SPH_C32(0x609205b3), SPH_C32(0xb56e8dd9) },
	{ SPH_C32(0x80420060), SPH_C32(0x341b66de), SPH_C32(0x7f6d88ec),
	  SPH_C32(0xb53d3a99), SPH_C32(0x0cad00d0), SPH_C32(0xd9777e19),
	  SPH_C32(0x9c7523b1), SPH_C32(0xc439ddb8) },
	{ SPH_C32(0xd11e0070), SPH_C32(0x74e81425), SPH_C32(0x838aaeee),
	  SPH_C32(0xc46a6af8), SPH_C32(0x229400d0), SPH_C32(0xbdaa1890),
	  SPH_C32(0xa0a1254d), SPH_C32(0x75cd4d04) },
	{ SPH_C32(0x50b10040), SPH_C32(0x189b28b3), SPH_C32(0xc3decfed),
	  SPH_C32(0x858f2b9d), SPH_C32(0xd17000e0), SPH_C32(0x7cbe8f9d),
	  SPH_C32(0xa5884f4b), SPH_C32(0xe635bda5) },
	{ SPH_C32(0x01ed0050), SPH_C32(0x58685a48), SPH_C32(0x3f39e9ef),
	  SPH_C32(0xf4d87bfc), SPH_C32(0xff4900e0), SPH_C32(0x1863e914),
	  SPH_C32(0x995c49b7), SPH_C32(0x57c12d19) },
	{ SPH_C32(0x7e880040), SPH_C32(0x7c464e3a), SPH_C32(0xff0ac911),
	  SPH_C32(0x347bbb21), SPH_C32(0xae1500f0), SPH_C32(0x58909bef),
	  SPH_C32(0x65bb6fb5), SPH_C32(0x26967d78) },
	{ SPH_C32(0x2fd40050), SPH_C32(0x3cb53cc1), SPH_C32(0x03edef13),
	  SPH_C32(0x452ceb40), SPH_C32(0x802c00f0), SPH_C32(0x3c4dfd66),
	  SPH_C32(0x596f6949), SPH_C32(0x9762edc4) },
	{ SPH_C32(0xf2090060), SPH_C32(0x997ccd45), SPH_C32(0x3a1083e9),
	  SPH_C32(0x67208b5d), SPH_C32(0x8d0200e0), SPH_C32(0xb504428f),
	  SPH_C32(0xdc2142b2), SPH_C32(0x85dc9cdd) },
	{ SPH_C32(0xa3550070), SPH_C32(0xd98fbfbe), SPH_C32(0xc6f7a5eb),
	  SPH_C32(0x1677db3c), SPH_C32(0xa33b00e0), SPH_C32(0xd1d92406),
	  SPH_C32(0xe0f5444e), SPH_C32(0x34280c61) },
	{ SPH_C32(0xdc300060), SPH_C32(0xfda1abcc), SPH_C32(0x06c48515),
	  SPH_C32(0xd6d41be1), SPH_C32(0xf26700f0), SPH_C32(0x912a56fd),
	  SPH_C32(0x1c12624c), SPH_C32(0x457f5c00) },
	{ SPH_C32(0x8d6c0070), SPH_C32(0xbd52d937), SPH_C32(0xfa23a317),
	  SPH_C32(0xa7834b80), SPH_C32(0xdc5e00f0), SPH_C32(0xf5f73074),
	  SPH_C32(0x20c664b0), SPH_C32(0xf48bccbc) },
	{ SPH_C32(0x62740080), SPH_C32(0x0fb84b07), SPH_C32(0x138a651e),
	  SPH_C32(0x44100618), SPH_C32(0x5bd20080), SPH_C32(0x450f18ec),
	  SPH_C32(0xc2c46c55), SPH_C32(0xf362b233) },
	{ SPH_C32(0x33280090), SPH_C32(0x4f4b39fc), SPH_C32(0xef6d431c),
	  SPH_C32(0x35475679), SPH_C32(0x75eb0080), SPH_C32(0x21d27e65),
	  SPH_C32(0xfe106aa9), SPH_C32(0x4296228f) },
	{ SPH_C32(0x4c4d0080), SPH_C32(0x6b652d8e), SPH_C32(0x2f5e63e2),
	  SPH_C32(0xf5e496a4), SPH_C32(0x24b70090), SPH_C32(0x61210c9e),
	  SPH_C32(0x02f74cab), SPH_C32(0x33c172ee) },
	{ SPH_C32(0x1d110090), SPH_C32(0x2b965f75), SPH_C32(0xd3b945e0),
	  SPH_C32(0x84b3c6c5), SPH_C32(0x0a8e0090), SPH_C32(0x05fc6a17),
	  SPH_C32(0x3e234a57), SPH_C32(0x8235e252) },
	{ SPH_C32(0xc0cc00a0), SPH_C32(0x8e5faef1), SPH_C32(0xea44291a),
	  SPH_C32(0xa6bfa6d8), SPH_C32(0x07a00080), SPH_C32(0x8cb5d5fe),
	  SPH_C32(0xbb6d61ac), SPH_C32(0x908b934b) },
	{ SPH_C32(0x919000b0), SPH_C32(0xceacdc0a), SPH_C32(0x16a30f18),
	  SPH_C32(0xd7e8f6b9), SPH_C32(0x29990080), SPH_C32(0xe868b377),
	  SPH_C32(0x87b96750), SPH_C32(0x217f03f7) },
	{ SPH_C32(0xeef500a0), SPH_C32(0xea82c878), SPH_C32(0xd6902fe6),
	  SPH_C32(0x174b3664), SPH_C32(0x78c50090), SPH_C32(0xa89bc18c),
	  SPH_C32(0x7b5e4152), SPH_C32(0x50285396) },
	{ SPH_C32(0xbfa900b0), SPH_C32(0xaa71ba83), SPH_C32(0x2a7709e4),
	  SPH_C32(0x661c6605), SPH_C32(0x56fc0090), SPH_C32(0xcc46a705),
	  SPH_C32(0x478a47ae), SPH_C32(0xe1dcc32a) },
	{ SPH_C32(0x3e060080), SPH_C32(0xc6028615), SPH_C32(0x6a2368e7),
	  SPH_C32(0x27f92760), SPH_C32(0xa51800a0), SPH_C32(0x0d523008),
	  SPH_C32(0x42a32da8), SPH_C32(0x7224338b) },
	{ SPH_C32(0x6f5a0090), SPH_C32(0x86f1f4ee), SPH_C32(0x96c44ee5),
	  SPH_C32(0x56ae7701), SPH_C32(0x8b2100a0), SPH_C32(0x698f5681),
	  SPH_C32(0x7e772b54), SPH_C32(0xc3d0a337) },
	{ SPH_C32(0x103f0080), SPH_C32(0xa2dfe09c), SPH_C32(0x56f76e1b),
	  SPH_C32(0x960db7dc), SPH_C32(0xda7d00b0), SPH_C32(0x297c247a),
	  SPH_C32(0x82900d56), SPH_C32(0xb287f356) },
	{ SPH_C32(0x41630090), SPH_C32(0xe22c9267), SPH_C32(0xaa104819),
	  SPH_C32(0xe75ae7bd), SPH_C32(0xf44400b0), SPH_C32(0x4da142f3),
	  SPH_C32(0xbe440baa), SPH_C32(0x037363ea) },
	{ SPH_C32(0x9cbe00a0), SPH_C32(0x47e563e3), SPH_C32(0x93ed24e3),
	  SPH_C32(0xc55687a0), SPH_C32(0xf96a00a0), SPH_C32(0xc4e8fd1a),
	  SPH_C32(0x3b0a2051), SPH_C32(0x11cd12f3) },
	{ SPH_C32(0xcde200b0), SPH_C32(0x07161118), SPH_C32(0x6f0a02e1),
	  SPH_C32(0xb401d7c1), SPH_C32(0xd75300a0), SPH_C32(0xa0359b93),
	  SPH_C32(0x07de26ad), SPH_C32(0xa039824f) },
	{ SPH_C32(0xb28700a0), SPH_C32(0x2338056a), SPH_C32(0xaf39221f),
	  SPH_C32(0x74a2171c), SPH_C32(0x860f00b0), SPH_C32(0xe0c6e968),
	  SPH_C32(0xfb3900af), SPH_C32(0xd16ed22e) },
	{ SPH_C32(0xe3db00b0), SPH_C32(0x63cb7791), SPH_C32(0x53de041d),
	  SPH_C32(0x05f5477d), SPH_C32(0xa83600b0), SPH_C32(0x841b8fe1),
	  SPH_C32(0xc7ed0653), SPH_C32(0x609a4292) },
	{ SPH_C32(0x2fba00c0), SPH_C32(0x34e3a779), SPH_C32(0x25ef0eb6),
	  SPH_C32(0x67733c1d), SPH_C32(0x23790080), SPH_C32(0xe5c242d8),
	  SPH_C32(0x9f98cca2), SPH_C32(0x811536f8) },
	{ SPH_C32(0x7ee600d0), SPH_C32(0x7410d582), SPH_C32(0xd90828b4),
	  SPH_C32(0x16246c7c), SPH_C32(0x0d400080), SPH_C32(0x811f2451),
	  SPH_C32(0xa34cca5e), SPH_C32(0x30e1a644) },
	{ SPH_C32(0x018300c0), SPH_C32(0x503ec1f0), SPH_C32(0x193b084a),
	  SPH_C32(0xd687aca1), SPH_C32(0x5c1c0090), SPH_C32(0xc1ec56aa),
	  SPH_C32(0x5fabec5c), SPH_C32(0x41b6f625) },
	{ SPH_C32(0x50df00d0), SPH_C32(0x10cdb30b), SPH_C32(0xe5dc2e48),
	  SPH_C32(0xa7d0fcc0), SPH_C32(0x72250090), SPH_C32(0xa5313023),
	  SPH_C32(0x637feaa0), SPH_C32(0xf0426699) },
	{ SPH_C32(0x8d0200e0), SPH_C32(0xb504428f), SPH_C32(0xdc2142b2),
	  SPH_C32(0x85dc9cdd), SPH_C32(0x7f0b0080), SPH_C32(0x2c788fca),
	  SPH_C32(0xe631c15b), SPH_C32(0xe2fc1780) },
	{ SPH_C32(0xdc5e00f0), SPH_C32(0xf5f73074), SPH_C32(0x20c664b0),
	  SPH_C32(0xf48bccbc), SPH_C32(0x51320080), SPH_C32(0x48a5e943),
	  SPH_C32(0xdae5c7a7), SPH_C32(0x5308873c) },
	{ SPH_C32(0xa33b00e0), SPH_C32(0xd1d92406), SPH_C32(0xe0f5444e),
	  SPH_C32(0x34280c61), SPH_C32(0x006e0090), SPH_C32(0x08569bb8),
	  SPH_C32(0x2602e1a5), SPH_C32(0x225fd75d) },
	{ SPH_C32(0xf26700f0), SPH_C32(0x912a56fd), SPH_C32(0x1c12624c),
	  SPH_C32(0x457f5c00), SPH_C32(0x2e570090), SPH_C32(0x6c8bfd31),
	  SPH_C32(0x1ad6e759), SPH_C32(0x93ab47e1) },
	{ SPH_C32(0x73c800c0), SPH_C32(0xfd596a6b), SPH_C32(0x5c46034f),
	  SPH_C32(0x049a1d65), SPH_C32(0xddb300a0), SPH_C32(0xad9f6a3c),
	  SPH_C32(0x1fff8d5f), SPH_C32(0x0053b740) },
	{ SPH_C32(0x229400d0), SPH_C32(0xbdaa1890), SPH_C32(0xa0a1254d),
	  SPH_C32(0x75cd4d04), SPH_C32(0xf38a00a0), SPH_C32(0xc9420cb5),
	  SPH_C32(0x232b8ba3), SPH_C32(0xb1a727fc) },
	{ SPH_C32(0x5df100c0), SPH_C32(0x99840ce2), SPH_C32(0x609205b3),
	  SPH_C32(0xb56e8dd9), SPH_C32(0xa2d600b0), SPH_C32(0x89b17e4e),
	  SPH_C32(0xdfccada1), SPH_C32(0xc0f0779d) },
	{ SPH_C32(0x0cad00d0), SPH_C32(0xd9777e19), SPH_C32(0x9c7523b1),
	  SPH_C32(0xc439ddb8), SPH_C32(0x8cef00b0), SPH_C32(0xed6c18c7),
	  SPH_C32(0xe318ab5d), SPH_C32(0x7104e721) },
	{ SPH_C32(0xd17000e0), SPH_C32(0x7cbe8f9d), SPH_C32(0xa5884f4b),
	  SPH_C32(0xe635bda5), SPH_C32(0x81c100a0), SPH_C32(0x6425a72e),
	  SPH_C32(0x665680a6), SPH_C32(0x63ba9638) },
	{ SPH_C32(0x802c00f0), SPH_C32(0x3c4dfd66), SPH_C32(0x596f6949),
	  SPH_C32(0x9762edc4), SPH_C32(0xaff800a0), SPH_C32(0x00f8c1a7),
	  SPH_C32(0x5a82865a), SPH_C32(0xd24e0684) },
	{ SPH_C32(0xff4900e0), SPH_C32(0x1863e914), SPH_C32(0x995c49b7),
	  SPH_C32(0x57c12d19), SPH_C32(0xfea400b0), SPH_C32(0x400bb35c),
	  SPH_C32(0xa665a058), SPH_C32(0xa31956e5) },
	{ SPH_C32(0xae1500f0), SPH_C32(0x58909bef), SPH_C32(0x65bb6fb5),
	  SPH_C32(0x26967d78), SPH_C32(0xd09d00b0), SPH_C32(0x24d6d5d5),
	  SPH_C32(0x9ab1a6a4), SPH_C32(0x12edc659) },
	{ SPH_C32(0x1adf0080), SPH_C32(0xaf751133), SPH_C32(0x4ed6c5e9),
	  SPH_C32(0x366782d3), SPH_C32(0x6eb700c0), SPH_C32(0xde99aea6),
	  SPH_C32(0xa9fda70a), SPH_C32(0xa2760cfd) },
	{ SPH_C32(0x4b830090), SPH_C32(0xef8663c8), SPH_C32(0xb231e3eb),
	  SPH_C32(0x4730d2b2), SPH_C32(0x408e00c0), SPH_C32(0xba44c82f),
	  SPH_C32(0x9529a1f6), SPH_C32(0x13829c41) },
	{ SPH_C32(0x34e60080), SPH_C32(0xcba877ba), SPH_C32(0x7202c315),
	  SPH_C32(0x8793126f), SPH_C32(0x11d200d0), SPH_C32(0xfab7bad4),
	  SPH_C32(0x69ce87f4), SPH_C32(0x62d5cc20) },
	{ SPH_C32(0x65ba0090), SPH_C32(0x8b5b0541), SPH_C32(0x8ee5e517),
	  SPH_C32(0xf6c4420e), SPH_C32(0x3feb00d0), SPH_C32(0x9e6adc5d),
	  SPH_C32(0x551a8108), SPH_C32(0xd3215c9c) },
	{ SPH_C32(0xb86700a0), SPH_C32(0x2e92f4c5), SPH_C32(0xb71889ed),
	  SPH_C32(0xd4c82213), SPH_C32(0x32c500c0), SPH_C32(0x172363b4),
	  SPH_C32(0xd054aaf3), SPH_C32(0xc19f2d85) },
	{ SPH_C32(0xe93b00b0), SPH_C32(0x6e61863e), SPH_C32(0x4bffafef),
	  SPH_C32(0xa59f7272), SPH_C32(0x1cfc00c0), SPH_C32(0x73fe053d),
	  SPH_C32(0xec80ac0f), SPH_C32(0x706bbd39) },
	{ SPH_C32(0x965e00a0), SPH_C32(0x4a4f924c), SPH_C32(0x8bcc8f11),
	  SPH_C32(0x653cb2af), SPH_C32(0x4da000d0), SPH_C32(0x330d77c6),
	  SPH_C32(0x10678a0d), SPH_C32(0x013ced58) },
	{ SPH_C32(0xc70200b0), SPH_C32(0x0abce0b7), SPH_C32(0x772ba913),
	  SPH_C32(0x146be2ce), SPH_C32(0x639900d0), SPH_C32(0x57d0114f),
	  SPH_C32(0x2cb38cf1), SPH_C32(0xb0c87de4) },
	{ SPH_C32(0x46ad0080), SPH_C32(0x66cfdc21), SPH_C32(0x377fc810),
	  SPH_C32(0x558ea3ab), SPH_C32(0x907d00e0), SPH_C32(0x96c48642),
	  SPH_C32(0x299ae6f7), SPH_C32(0x23308d45) },
	{ SPH_C32(0x17f10090), SPH_C32(0x263caeda), SPH_C32(0xcb98ee12),
	  SPH_C32(0x24d9f3ca), SPH_C32(0xbe4400e0), SPH_C32(0xf219e0cb),
	  SPH_C32(0x154ee00b), SPH_C32(0x92c41df9) },
	{ SPH_C32(0x68940080), SPH_C32(0x0212baa8), SPH_C32(0x0babceec),
	  SPH_C32(0xe47a3317), SPH_C32(0xef1800f0), SPH_C32(0xb2ea9230),
	  SPH_C32(0xe9a9c609), SPH_C32(0xe3934d98) },
	{ SPH_C32(0x39c80090), SPH_C32(0x42e1c853), SPH_C32(0xf74ce8ee),
	  SPH_C32(0x952d6376), SPH_C32(0xc12100f0), SPH_C32(0xd637f4b9),
	  SPH_C32(0xd57dc0f5), SPH_C32(0x5267dd24) },
	{ SPH_C32(0xe41500a0), SPH_C32(0xe72839d7), SPH_C32(0xceb18414),
	  SPH_C32(0xb721036b), SPH_C32(0xcc0f00e0), SPH_C32(0x5f7e4b50),
	  SPH_C32(0x5033eb0e), SPH_C32(0x40d9ac3d) },
	{ SPH_C32(0xb54900b0), SPH_C32(0xa7db4b2c), SPH_C32(0x3256a216),
	  SPH_C32(0xc676530a), SPH_C32(0xe23600e0), SPH_C32(0x3ba32dd9),
	  SPH_C32(0x6ce7edf2), SPH_C32(0xf12d3c81) },
	{ SPH_C32(0xca2c00a0), SPH_C32(0x83f55f5e), SPH_C32(0xf26582e8),
	  SPH_C32(0x06d593d7), SPH_C32(0xb36a00f0), SPH_C32(0x7b505f22),
	  SPH_C32(0x9000cbf0), SPH_C32(0x807a6ce0) },
	{ SPH_C32(0x9b7000b0), SPH_C32(0xc3062da5), SPH_C32(0x0e82a4ea),
	  SPH_C32(0x7782c3b6), SPH_C32(0x9d5300f0), SPH_C32(0x1f8d39ab),
	  SPH_C32(0xacd4cd0c), SPH_C32(0x318efc5c) },
	{ SPH_C32(0x571100c0), SPH_C32(0x942efd4d), SPH_C32(0x78b3ae41),
	  SPH_C32(0x1504b8d6), SPH_C32(0x161c00c0), SPH_C32(0x7e54f492),
	  SPH_C32(0xf4a107fd), SPH_C32(0xd0018836) },
	{ SPH_C32(0x064d00d0), SPH_C32(0xd4dd8fb6), SPH_C32(0x84548843),
	  SPH_C32(0x6453e8b7), SPH_C32(0x382500c0), SPH_C32(0x1a89921b),
	  SPH_C32(0xc8750101), SPH_C32(0x61f5188a) },
	{ SPH_C32(0x792800c0), SPH_C32(0xf0f39bc4), SPH_C32(0x4467a8bd),
	  SPH_C32(0xa4f0286a), SPH_C32(0x697900d0), SPH_C32(0x5a7ae0e0),
	  SPH_C32(0x34922703), SPH_C32(0x10a248eb) },
	{ SPH_C32(0x287400d0), SPH_C32(0xb000e93f), SPH_C32(0xb8808ebf),
	  SPH_C32(0xd5a7780b), SPH_C32(0x474000d0), SPH_C32(0x3ea78669),
	  SPH_C32(0x084621ff), SPH_C32(0xa156d857) },
	{ SPH_C32(0xf5a900e0), SPH_C32(0x15c918bb), SPH_C32(0x817de245),
	  SPH_C32(0xf7ab1816), SPH_C32(0x4a6e00c0), SPH_C32(0xb7ee3980),
	  SPH_C32(0x8d080a04), SPH_C32(0xb3e8a94e) },
	{ SPH_C32(0xa4f500f0), SPH_C32(0x553a6a40), SPH_C32(0x7d9ac447),
	  SPH_C32(0x86fc4877), SPH_C32(0x645700c0), SPH_C32(0xd3335f09),
	  SPH_C32(0xb1dc0cf8), SPH_C32(0x021c39f2) },
	{ SPH_C32(0xdb9000e0), SPH_C32(0x71147e32), SPH_C32(0xbda9e4b9),
	  SPH_C32(0x465f88aa), SPH_C32(0x350b00d0), SPH_C32(0x93c02df2),
	  SPH_C32(0x4d3b2afa), SPH_C32(0x734b6993) },
	{ SPH_C32(0x8acc00f0), SPH_C32(0x31e70cc9), SPH_C32(0x414ec2bb),
	  SPH_C32(0x3708d8cb), SPH_C32(0x1b3200d0), SPH_C32(0xf71d4b7b),
	  SPH_C32(0x71ef2c06), SPH_C32(0xc2bff92f) },
	{ SPH_C32(0x0b6300c0), SPH_C32(0x5d94305f), SPH_C32(0x011aa3b8),
	  SPH_C32(0x76ed99ae), SPH_C32(0xe8d600e0), SPH_C32(0x3609dc76),
	  SPH_C32(0x74c64600), SPH_C32(0x5147098e) },
	{ SPH_C32(0x5a3f00d0), SPH_C32(0x1d6742a4), SPH_C32(0xfdfd85ba),
	  SPH_C32(0x07bac9cf), SPH_C32(0xc6ef00e0), SPH_C32(0x52d4baff),
	  SPH_C32(0x481240fc), SPH_C32(0xe0b39932) },
	{ SPH_C32(0x255a00c0), SPH_C32(0x394956d6), SPH_C32(0x3dcea544),
	  SPH_C32(0xc7190912), SPH_C32(0x97b300f0), SPH_C32(0x1227c804),
	  SPH_C32(0xb4f566fe), SPH_C32(0x91e4c953) },
	{ SPH_C32(0x740600d0), SPH_C32(0x79ba242d), SPH_C32(0xc1298346),
	  SPH_C32(0xb64e5973), SPH_C32(0xb98a00f0), SPH_C32(0x76faae8d),
	  SPH_C32(0x88216002), SPH_C32(0x201059ef) },
	{ SPH_C32(0xa9db00e0), SPH_C32(0xdc73d5a9), SPH_C32(0xf8d4efbc),
	  SPH_C32(0x9442396e), SPH_C32(0xb4a400e0), SPH_C32(0xffb31164),
	  SPH_C32(0x0d6f4bf9), SPH_C32(0x32ae28f6) },
	{ SPH_C32(0xf88700f0), SPH_C32(0x9c80a752), SPH_C32(0x0433c9be),
	  SPH_C32(0xe515690f), SPH_C32(0x9a9d00e0), SPH_C32(0x9b6e77ed),
	  SPH_C32(0x31bb4d05), SPH_C32(0x835ab84a) },
	{ SPH_C32(0x87e200e0), SPH_C32(0xb8aeb320), SPH_C32(0xc400e940),
	  SPH_C32(0x25b6a9d2), SPH_C32(0xcbc100f0), SPH_C32(0xdb9d0516),
	  SPH_C32(0xcd5c6b07), SPH_C32(0xf20de82b) },
	{ SPH_C32(0xd6be00f0), SPH_C32(0xf85dc1db), SPH_C32(0x38e7cf42),
	  SPH_C32(0x54e1f9b3), SPH_C32(0xe5f800f0), SPH_C32(0xbf40639f),
	  SPH_C32(0xf1886dfb), SPH_C32(0x43f97897) }
};

static const sph_u32 T256_24[256][8] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000) },
	{ SPH_C32(0xc04e0001), SPH_C32(0x33b9c010), SPH_C32(0xae0ebb05),
	  SPH_C32(0xb5a4c63b), SPH_C32(0xc8f10000), SPH_C32(0x0b2de782),
	  SPH_C32(0x6bf648a4), SPH_C32(0x539cbdbf) },
	{ SPH_C32(0xc8f10000), SPH_C32(0x0b2de782), SPH_C32(0x6bf648a4),
	  SPH_C32(0x539cbdbf), SPH_C32(0x08bf0001), SPH_C32(0x38942792),
	  SPH_C32(0xc5f8f3a1), SPH_C32(0xe6387b84) },
	{ SPH_C32(0x08bf0001), SPH_C32(0x38942792), SPH_C32(0xc5f8f3a1),
	  SPH_C32(0xe6387b84), SPH_C32(0xc04e0001), SPH_C32(0x33b9c010),
	  SPH_C32(0xae0ebb05), SPH_C32(0xb5a4c63b) },
	{ SPH_C32(0x88230002), SPH_C32(0x5fe7a7b3), SPH_C32(0x99e585aa),
	  SPH_C32(0x8d75f7f1), SPH_C32(0x51ac0000), SPH_C32(0x25e30f14),
	  SPH_C32(0x79e22a4c), SPH_C32(0x1298bd46) },
	{ SPH_C32(0x486d0003), SPH_C32(0x6c5e67a3), SPH_C32(0x37eb3eaf),
	  SPH_C32(0x38d131ca), SPH_C32(0x995d0000), SPH_C32(0x2ecee896),
	  SPH_C32(0x121462e8), SPH_C32(0x410400f9) },
	{ SPH_C32(0x40d20002), SPH_C32(0x54ca4031), SPH_C32(0xf213cd0e),
	  SPH_C32(0xdee94a4e), SPH_C32(0x59130001), SPH_C32(0x1d772886),
	  SPH_C32(0xbc1ad9ed), SPH_C32(0xf4a0c6c2) },
	{ SPH_C32(0x809c0003), SPH_C32(0x67738021), SPH_C32(0x5c1d760b),
	  SPH_C32(0x6b4d8c75), SPH_C32(0x91e20001), SPH_C32(0x165acf04),
	  SPH_C32(0xd7ec9149), SPH_C32(0xa73c7b7d) },
	{ SPH_C32(0x51ac0000), SPH_C32(0x25e30f14), SPH_C32(0x79e22a4c),
	  SPH_C32(0x1298bd46), SPH_C32(0xd98f0002), SPH_C32(0x7a04a8a7),
	  SPH_C32(0xe007afe6), SPH_C32(0x9fed4ab7) },
	{ SPH_C32(0x91e20001), SPH_C32(0x165acf04), SPH_C32(0xd7ec9149),
	  SPH_C32(0xa73c7b7d), SPH_C32(0x117e0002), SPH_C32(0x71294f25),
	  SPH_C32(0x8bf1e742), SPH_C32(0xcc71f708) },
	{ SPH_C32(0x995d0000), SPH_C32(0x2ecee896), SPH_C32(0x121462e8),
	  SPH_C32(0x410400f9), SPH_C32(0xd1300003), SPH_C32(0x42908f35),
	  SPH_C32(0x25ff5c47), SPH_C32(0x79d53133) },
	{ SPH_C32(0x59130001), SPH_C32(0x1d772886), SPH_C32(0xbc1ad9ed),
	  SPH_C32(0xf4a0c6c2), SPH_C32(0x19c10003), SPH_C32(0x49bd68b7),
	  SPH_C32(0x4e0914e3), SPH_C32(0x2a498c8c) },
	{ SPH_C32(0xd98f0002), SPH_C32(0x7a04a8a7), SPH_C32(0xe007afe6),
	  SPH_C32(0x9fed4ab7), SPH_C32(0x88230002), SPH_C32(0x5fe7a7b3),
	  SPH_C32(0x99e585aa), SPH_C32(0x8d75f7f1) },
	{ SPH_C32(0x19c10003), SPH_C32(0x49bd68b7), SPH_C32(0x4e0914e3),
	  SPH_C32(0x2a498c8c), SPH_C32(0x40d20002), SPH_C32(0x54ca4031),
	  SPH_C32(0xf213cd0e), SPH_C32(0xdee94a4e) },
	{ SPH_C32(0x117e0002), SPH_C32(0x71294f25), SPH_C32(0x8bf1e742),
	  SPH_C32(0xcc71f708), SPH_C32(0x809c0003), SPH_C32(0x67738021),
	  SPH_C32(0x5c1d760b), SPH_C32(0x6b4d8c75) },
	{ SPH_C32(0xd1300003), SPH_C32(0x42908f35), SPH_C32(0x25ff5c47),
	  SPH_C32(0x79d53133), SPH_C32(0x486d0003), SPH_C32(0x6c5e67a3),
	  SPH_C32(0x37eb3eaf), SPH_C32(0x38d131ca) },
	{ SPH_C32(0xd0080004), SPH_C32(0x8c768f77), SPH_C32(0x9dc5b050),
	  SPH_C32(0xaf4a29da), SPH_C32(0x6ba90000), SPH_C32(0x40ebf9aa),
	  SPH_C32(0x98321c3d), SPH_C32(0x76acc733) },
	{ SPH_C32(0x10460005), SPH_C32(0xbfcf4f67), SPH_C32(0x33cb0b55),
	  SPH_C32(0x1aeeefe1), SPH_C32(0xa3580000), SPH_C32(0x4bc61e28),
	  SPH_C32(0xf3c45499), SPH_C32(0x25307a8c) },
	{ SPH_C32(0x18f90004), SPH_C32(0x875b68f5), SPH_C32(0xf633f8f4),
	  SPH_C32(0xfcd69465), SPH_C32(0x63160001), SPH_C32(0x787fde38),
	  SPH_C32(0x5dcaef9c), SPH_C32(0x9094bcb7) },
	{ SPH_C32(0xd8b70005), SPH_C32(0xb4e2a8e5), SPH_C32(0x583d43f1),
	  SPH_C32(0x4972525e), SPH_C32(0xabe70001), SPH_C32(0x735239ba),
	  SPH_C32(0x363ca738), SPH_C32(0xc3080108) },
	{ SPH_C32(0x582b0006), SPH_C32(0xd39128c4), SPH_C32(0x042035fa),
	  SPH_C32(0x223fde2b), SPH_C32(0x3a050000), SPH_C32(0x6508f6be),
	  SPH_C32(0xe1d03671), SPH_C32(0x64347a75) },
	{ SPH_C32(0x98650007), SPH_C32(0xe028e8d4), SPH_C32(0xaa2e8eff),
	  SPH_C32(0x979b1810), SPH_C32(0xf2f40000), SPH_C32(0x6e25113c),
	  SPH_C32(0x8a267ed5), SPH_C32(0x37a8c7ca) },
	{ SPH_C32(0x90da0006), SPH_C32(0xd8bccf46), SPH_C32(0x6fd67d5e),
	  SPH_C32(0x71a36394), SPH_C32(0x32ba0001), SPH_C32(0x5d9cd12c),
	  SPH_C32(0x2428c5d0), SPH_C32(0x820c01f1) },
	{ SPH_C32(0x50940007), SPH_C32(0xeb050f56), SPH_C32(0xc1d8c65b),
	  SPH_C32(0xc407a5af), SPH_C32(0xfa4b0001), SPH_C32(0x56b136ae),
	  SPH_C32(0x4fde8d74), SPH_C32(0xd190bc4e) },
	{ SPH_C32(0x81a40004), SPH_C32(0xa9958063), SPH_C32(0xe4279a1c),
	  SPH_C32(0xbdd2949c), SPH_C32(0xb2260002), SPH_C32(0x3aef510d),
	  SPH_C32(0x7835b3db), SPH_C32(0xe9418d84) },
	{ SPH_C32(0x41ea0005), SPH_C32(0x9a2c4073), SPH_C32(0x4a292119),
	  SPH_C32(0x087652a7), SPH_C32(0x7ad70002), SPH_C32(0x31c2b68f),
	  SPH_C32(0x13c3fb7f), SPH_C32(0xbadd303b) },
	{ SPH_C32(0x49550004), SPH_C32(0xa2b867e1), SPH_C32(0x8fd1d2b8),
	  SPH_C32(0xee4e2923), SPH_C32(0xba990003), SPH_C32(0x027b769f),
	  SPH_C32(0xbdcd407a), SPH_C32(0x0f79f600) },
	{ SPH_C32(0x891b0005), SPH_C32(0x9101a7f1), SPH_C32(0x21df69bd),
	  SPH_C32(0x5beaef18), SPH_C32(0x72680003), SPH_C32(0x0956911d),
	  SPH_C32(0xd63b08de), SPH_C32(0x5ce54bbf) },
	{ SPH_C32(0x09870006), SPH_C32(0xf67227d0), SPH_C32(0x7dc21fb6),
	  SPH_C32(0x30a7636d), SPH_C32(0xe38a0002), SPH_C32(0x1f0c5e19),
	  SPH_C32(0x01d79997), SPH_C32(0xfbd930c2) },
	{ SPH_C32(0xc9c90007), SPH_C32(0xc5cbe7c0), SPH_C32(0xd3cca4b3),
	  SPH_C32(0x8503a556), SPH_C32(0x2b7b0002), SPH_C32(0x1421b99b),
	  SPH_C32(0x6a21d133), SPH_C32(0xa8458d7d) },
	{ SPH_C32(0xc1760006), SPH_C32(0xfd5fc052), SPH_C32(0x16345712),
	  SPH_C32(0x633bded2), SPH_C32(0xeb350003), SPH_C32(0x2798798b),
	  SPH_C32(0xc42f6a36), SPH_C32(0x1de14b46) },
	{ SPH_C32(0x01380007), SPH_C32(0xcee60042), SPH_C32(0xb83aec17),
	  SPH_C32(0xd69f18e9), SPH_C32(0x23c40003), SPH_C32(0x2cb59e09),
	  SPH_C32(0xafd92292), SPH_C32(0x4e7df6f9) },
	{ SPH_C32(0x6ba90000), SPH_C32(0x40ebf9aa), SPH_C32(0x98321c3d),
	  SPH_C32(0x76acc733), SPH_C32(0xbba10004), SPH_C32(0xcc9d76dd),
	  SPH_C32(0x05f7ac6d), SPH_C32(0xd9e6eee9) },
	{ SPH_C32(0xabe70001), SPH_C32(0x735239ba), SPH_C32(0x363ca738),
	  SPH_C32(0xc3080108), SPH_C32(0x73500004), SPH_C32(0xc7b0915f),
	  SPH_C32(0x6e01e4c9), SPH_C32(0x8a7a5356) },
	{ SPH_C32(0xa3580000), SPH_C32(0x4bc61e28), SPH_C32(0xf3c45499),
	  SPH_C32(0x25307a8c), SPH_C32(0xb31e0005), SPH_C32(0xf409514f),
	  SPH_C32(0xc00f5fcc), SPH_C32(0x3fde956d) },
	{ SPH_C32(0x63160001), SPH_C32(0x787fde38), SPH_C32(0x5dcaef9c),
	  SPH_C32(0x9094bcb7), SPH_C32(0x7bef0005), SPH_C32(0xff24b6cd),
	  SPH_C32(0xabf91768), SPH_C32(0x6c4228d2) },
	{ SPH_C32(0xe38a0002), SPH_C32(0x1f0c5e19), SPH_C32(0x01d79997),
	  SPH_C32(0xfbd930c2), SPH_C32(0xea0d0004), SPH_C32(0xe97e79c9),
	  SPH_C32(0x7c158621), SPH_C32(0xcb7e53af) },
	{ SPH_C32(0x23c40003), SPH_C32(0x2cb59e09), SPH_C32(0xafd92292),
	  SPH_C32(0x4e7df6f9), SPH_C32(0x22fc0004), SPH_C32(0xe2539e4b),
	  SPH_C32(0x17e3ce85), SPH_C32(0x98e2ee10) },
	{ SPH_C32(0x2b7b0002), SPH_C32(0x1421b99b), SPH_C32(0x6a21d133),
	  SPH_C32(0xa8458d7d), SPH_C32(0xe2b20005), SPH_C32(0xd1ea5e5b),
	  SPH_C32(0xb9ed7580), SPH_C32(0x2d46282b) },
	{ SPH_C32(0xeb350003), SPH_C32(0x2798798b), SPH_C32(0xc42f6a36),
	  SPH_C32(0x1de14b46), SPH_C32(0x2a430005), SPH_C32(0xdac7b9d9),
	  SPH_C32(0xd21b3d24), SPH_C32(0x7eda9594) },
	{ SPH_C32(0x3a050000), SPH_C32(0x6508f6be), SPH_C32(0xe1d03671),
	  SPH_C32(0x64347a75), SPH_C32(0x622e0006), SPH_C32(0xb699de7a),
	  SPH_C32(0xe5f0038b), SPH_C32(0x460ba45e) },
	{ SPH_C32(0xfa4b0001), SPH_C32(0x56b136ae), SPH_C32(0x4fde8d74),
	  SPH_C32(0xd190bc4e), SPH_C32(0xaadf0006), SPH_C32(0xbdb439f8),
	  SPH_C32(0x8e064b2f), SPH_C32(0x159719e1) },
	{ SPH_C32(0xf2f40000), SPH_C32(0x6e25113c), SPH_C32(0x8a267ed5),
	  SPH_C32(0x37a8c7ca), SPH_C32(0x6a910007), SPH_C32(0x8e0df9e8),
	  SPH_C32(0x2008f02a), SPH_C32(0xa033dfda) },
	{ SPH_C32(0x32ba0001), SPH_C32(0x5d9cd12c), SPH_C32(0x2428c5d0),
	  SPH_C32(0x820c01f1), SPH_C32(0xa2600007), SPH_C32(0x85201e6a),
	  SPH_C32(0x4bfeb88e), SPH_C32(0xf3af6265) },
	{ SPH_C32(0xb2260002), SPH_C32(0x3aef510d), SPH_C32(0x7835b3db),
	  SPH_C32(0xe9418d84), SPH_C32(0x33820006), SPH_C32(0x937ad16e),
	  SPH_C32(0x9c1229c7), SPH_C32(0x54931918) },
	{ SPH_C32(0x72680003), SPH_C32(0x0956911d), SPH_C32(0xd63b08de),
	  SPH_C32(0x5ce54bbf), SPH_C32(0xfb730006), SPH_C32(0x985736ec),
	  SPH_C32(0xf7e46163), SPH_C32(0x070fa4a7) },
	{ SPH_C32(0x7ad70002), SPH_C32(0x31c2b68f), SPH_C32(0x13c3fb7f),
	  SPH_C32(0xbadd303b), SPH_C32(0x3b3d0007), SPH_C32(0xabeef6fc),
	  SPH_C32(0x59eada66), SPH_C32(0xb2ab629c) },
	{ SPH_C32(0xba990003), SPH_C32(0x027b769f), SPH_C32(0xbdcd407a),
	  SPH_C32(0x0f79f600), SPH_C32(0xf3cc0007), SPH_C32(0xa0c3117e),
	  SPH_C32(0x321c92c2), SPH_C32(0xe137df23) },
	{ SPH_C32(0xbba10004), SPH_C32(0xcc9d76dd), SPH_C32(0x05f7ac6d),
	  SPH_C32(0xd9e6eee9), SPH_C32(0xd0080004), SPH_C32(0x8c768f77),
	  SPH_C32(0x9dc5b050), SPH_C32(0xaf4a29da) },
	{ SPH_C32(0x7bef0005), SPH_C32(0xff24b6cd), SPH_C32(0xabf91768),
	  SPH_C32(0x6c4228d2), SPH_C32(0x18f90004), SPH_C32(0x875b68f5),
	  SPH_C32(0xf633f8f4), SPH_C32(0xfcd69465) },
	{ SPH_C32(0x73500004), SPH_C32(0xc7b0915f), SPH_C32(0x6e01e4c9),
	  SPH_C32(0x8a7a5356), SPH_C32(0xd8b70005), SPH_C32(0xb4e2a8e5),
	  SPH_C32(0x583d43f1), SPH_C32(0x4972525e) },
	{ SPH_C32(0xb31e0005), SPH_C32(0xf409514f), SPH_C32(0xc00f5fcc),
	  SPH_C32(0x3fde956d), SPH_C32(0x10460005), SPH_C32(0xbfcf4f67),
	  SPH_C32(0x33cb0b55), SPH_C32(0x1aeeefe1) },
	{ SPH_C32(0x33820006), SPH_C32(0x937ad16e), SPH_C32(0x9c1229c7),
	  SPH_C32(0x54931918), SPH_C32(0x81a40004), SPH_C32(0xa9958063),
	  SPH_C32(0xe4279a1c), SPH_C32(0xbdd2949c) },
	{ SPH_C32(0xf3cc0007), SPH_C32(0xa0c3117e), SPH_C32(0x321c92c2),
	  SPH_C32(0xe137df23), SPH_C32(0x49550004), SPH_C32(0xa2b867e1),
	  SPH_C32(0x8fd1d2b8), SPH_C32(0xee4e2923) },
	{ SPH_C32(0xfb730006), SPH_C32(0x985736ec), SPH_C32(0xf7e46163),
	  SPH_C32(0x070fa4a7), SPH_C32(0x891b0005), SPH_C32(0x9101a7f1),
	  SPH_C32(0x21df69bd), SPH_C32(0x5beaef18) },
	{ SPH_C32(0x3b3d0007), SPH_C32(0xabeef6fc), SPH_C32(0x59eada66),
	  SPH_C32(0xb2ab629c), SPH_C32(0x41ea0005), SPH_C32(0x9a2c4073),
	  SPH_C32(0x4a292119), SPH_C32(0x087652a7) },
	{ SPH_C32(0xea0d0004), SPH_C32(0xe97e79c9), SPH_C32(0x7c158621),
	  SPH_C32(0xcb7e53af), SPH_C32(0x09870006), SPH_C32(0xf67227d0),
	  SPH_C32(0x7dc21fb6), SPH_C32(0x30a7636d) },
	{ SPH_C32(0x2a430005), SPH_C32(0xdac7b9d9), SPH_C32(0xd21b3d24),
	  SPH_C32(0x7eda9594), SPH_C32(0xc1760006), SPH_C32(0xfd5fc052),
	  SPH_C32(0x16345712), SPH_C32(0x633bded2) },
	{ SPH_C32(0x22fc0004), SPH_C32(0xe2539e4b), SPH_C32(0x17e3ce85),
	  SPH_C32(0x98e2ee10), SPH_C32(0x01380007), SPH_C32(0xcee60042),
	  SPH_C32(0xb83aec17), SPH_C32(0xd69f18e9) },
	{ SPH_C32(0xe2b20005), SPH_C32(0xd1ea5e5b), SPH_C32(0xb9ed7580),
	  SPH_C32(0x2d46282b), SPH_C32(0xc9c90007), SPH_C32(0xc5cbe7c0),
	  SPH_C32(0xd3cca4b3), SPH_C32(0x8503a556) },
	{ SPH_C32(0x622e0006), SPH_C32(0xb699de7a), SPH_C32(0xe5f0038b),
	  SPH_C32(0x460ba45e), SPH_C32(0x582b0006), SPH_C32(0xd39128c4),
	  SPH_C32(0x042035fa), SPH_C32(0x223fde2b) },
	{ SPH_C32(0xa2600007), SPH_C32(0x85201e6a), SPH_C32(0x4bfeb88e),
	  SPH_C32(0xf3af6265), SPH_C32(0x90da0006), SPH_C32(0xd8bccf46),
	  SPH_C32(0x6fd67d5e), SPH_C32(0x71a36394) },
	{ SPH_C32(0xaadf0006), SPH_C32(0xbdb439f8), SPH_C32(0x8e064b2f),
	  SPH_C32(0x159719e1), SPH_C32(0x50940007), SPH_C32(0xeb050f56),
	  SPH_C32(0xc1d8c65b), SPH_C32(0xc407a5af) },
	{ SPH_C32(0x6a910007), SPH_C32(0x8e0df9e8), SPH_C32(0x2008f02a),
	  SPH_C32(0xa033dfda), SPH_C32(0x98650007), SPH_C32(0xe028e8d4),
	  SPH_C32(0xaa2e8eff), SPH_C32(0x979b1810) },
	{ SPH_C32(0xa8ae0008), SPH_C32(0x2079397d), SPH_C32(0xfe739301),
	  SPH_C32(0xb8a92831), SPH_C32(0x171c0000), SPH_C32(0xb26e3344),
	  SPH_C32(0x9e6a837e), SPH_C32(0x58f8485f) },
	{ SPH_C32(0x68e00009), SPH_C32(0x13c0f96d), SPH_C32(0x507d2804),
	  SPH_C32(0x0d0dee0a), SPH_C32(0xdfed0000), SPH_C32(0xb943d4c6),
	  SPH_C32(0xf59ccbda), SPH_C32(0x0b64f5e0) },
	{ SPH_C32(0x605f0008), SPH_C32(0x2b54deff), SPH_C32(0x9585dba5),
	  SPH_C32(0xeb35958e), SPH_C32(0x1fa30001), SPH_C32(0x8afa14d6),
	  SPH_C32(0x5b9270df), SPH_C32(0xbec033db) },
	{ SPH_C32(0xa0110009), SPH_C32(0x18ed1eef), SPH_C32(0x3b8b60a0),
	  SPH_C32(0x5e9153b5), SPH_C32(0xd7520001), SPH_C32(0x81d7f354),
	  SPH_C32(0x3064387b), SPH_C32(0xed5c8e64) },
	{ SPH_C32(0x208d000a), SPH_C32(0x7f9e9ece), SPH_C32(0x679616ab),
	  SPH_C32(0x35dcdfc0), SPH_C32(0x46b00000), SPH_C32(0x978d3c50),
	  SPH_C32(0xe788a932), SPH_C32(0x4a60f519) },
	{ SPH_C32(0xe0c3000b), SPH_C32(0x4c275ede), SPH_C32(0xc998adae),
	  SPH_C32(0x807819fb), SPH_C32(0x8e410000), SPH_C32(0x9ca0dbd2),
	  SPH_C32(0x8c7ee196), SPH_C32(0x19fc48a6) },
	{ SPH_C32(0xe87c000a), SPH_C32(0x74b3794c), SPH_C32(0x0c605e0f),
	  SPH_C32(0x6640627f), SPH_C32(0x4e0f0001), SPH_C32(0xaf191bc2),
	  SPH_C32(0x22705a93), SPH_C32(0xac588e9d) },
	{ SPH_C32(0x2832000b), SPH_C32(0x470ab95c), SPH_C32(0xa26ee50a),
	  SPH_C32(0xd3e4a444), SPH_C32(0x86fe0001), SPH_C32(0xa434fc40),
	  SPH_C32(0x49861237), SPH_C32(0xffc43322) },
	{ SPH_C32(0xf9020008), SPH_C32(0x059a3669), SPH_C32(0x8791b94d),
	  SPH_C32(0xaa319577), SPH_C32(0xce930002), SPH_C32(0xc86a9be3),
	  SPH_C32(0x7e6d2c98), SPH_C32(0xc71502e8) },
	{ SPH_C32(0x394c0009), SPH_C32(0x3623f679), SPH_C32(0x299f0248),
	  SPH_C32(0x1f95534c), SPH_C32(0x06620002), SPH_C32(0xc3477c61),
	  SPH_C32(0x159b643c), SPH_C32(0x9489bf57) },
	{ SPH_C32(0x31f30008), SPH_C32(0x0eb7d1eb), SPH_C32(0xec67f1e9),
	  SPH_C32(0xf9ad28c8), SPH_C32(0xc62c0003), SPH_C32(0xf0febc71),
	  SPH_C32(0xbb95df39), SPH_C32(0x212d796c) },
	{ SPH_C32(0xf1bd0009), SPH_C32(0x3d0e11fb), SPH_C32(0x42694aec),
	  SPH_C32(0x4c09eef3), SPH_C32(0x0edd0003), SPH_C32(0xfbd35bf3),
	  SPH_C32(0xd063979d), SPH_C32(0x72b1c4d3) },
	{ SPH_C32(0x7121000a), SPH_C32(0x5a7d91da), SPH_C32(0x1e743ce7),
	  SPH_C32(0x27446286), SPH_C32(0x9f3f0002), SPH_C32(0xed8994f7),
	  SPH_C32(0x078f06d4), SPH_C32(0xd58dbfae) },
	{ SPH_C32(0xb16f000b), SPH_C32(0x69c451ca), SPH_C32(0xb07a87e2),
	  SPH_C32(0x92e0a4bd), SPH_C32(0x57ce0002), SPH_C32(0xe6a47375),
	  SPH_C32(0x6c794e70), SPH_C32(0x86110211) },
	{ SPH_C32(0xb9d0000a), SPH_C32(0x51507658), SPH_C32(0x75827443),
	  SPH_C32(0x74d8df39), SPH_C32(0x97800003), SPH_C32(0xd51db365),
	  SPH_C32(0xc277f575), SPH_C32(0x33b5c42a) },
	{ SPH_C32(0x799e000b), SPH_C32(0x62e9b648), SPH_C32(0xdb8ccf46),
	  SPH_C32(0xc17c1902), SPH_C32(0x5f710003), SPH_C32(0xde3054e7),
	  SPH_C32(0xa981bdd1), SPH_C32(0x60297995) },
	{ SPH_C32(0x78a6000c), SPH_C32(0xac0fb60a), SPH_C32(0x63b62351),
	  SPH_C32(0x17e301eb), SPH_C32(0x7cb50000), SPH_C32(0xf285caee),
	  SPH_C32(0x06589f43), SPH_C32(0x2e548f6c) },
	{ SPH_C32(0xb8e8000d), SPH_C32(0x9fb6761a), SPH_C32(0xcdb89854),
	  SPH_C32(0xa247c7d0), SPH_C32(0xb4440000), SPH_C32(0xf9a82d6c),
	  SPH_C32(0x6daed7e7), SPH_C32(0x7dc832d3) },
	{ SPH_C32(0xb057000c), SPH_C32(0xa7225188), SPH_C32(0x08406bf5),
	  SPH_C32(0x447fbc54), SPH_C32(0x740a0001), SPH_C32(0xca11ed7c),
	  SPH_C32(0xc3a06ce2), SPH_C32(0xc86cf4e8) },
	{ SPH_C32(0x7019000d), SPH_C32(0x949b9198), SPH_C32(0xa64ed0f0),
	  SPH_C32(0xf1db7a6f), SPH_C32(0xbcfb0001), SPH_C32(0xc13c0afe),
	  SPH_C32(0xa8562446), SPH_C32(0x9bf04957) },
	{ SPH_C32(0xf085000e), SPH_C32(0xf3e811b9), SPH_C32(0xfa53a6fb),
	  SPH_C32(0x9a96f61a), SPH_C32(0x2d190000), SPH_C32(0xd766c5fa),
	  SPH_C32(0x7fbab50f), SPH_C32(0x3ccc322a) },
	{ SPH_C32(0x30cb000f), SPH_C32(0xc051d1a9), SPH_C32(0x545d1dfe),
	  SPH_C32(0x2f323021), SPH_C32(0xe5e80000), SPH_C32(0xdc4b2278),
	  SPH_C32(0x144cfdab), SPH_C32(0x6f508f95) },
	{ SPH_C32(0x3874000e), SPH_C32(0xf8c5f63b), SPH_C32(0x91a5ee5f),
	  SPH_C32(0xc90a4ba5), SPH_C32(0x25a60001), SPH_C32(0xeff2e268),
	  SPH_C32(0xba4246ae), SPH_C32(0xdaf449ae) },
	{ SPH_C32(0xf83a000f), SPH_C32(0xcb7c362b), SPH_C32(0x3fab555a),
	  SPH_C32(0x7cae8d9e), SPH_C32(0xed570001), SPH_C32(0xe4df05ea),
	  SPH_C32(0xd1b40e0a), SPH_C32(0x8968f411) },
	{ SPH_C32(0x290a000c), SPH_C32(0x89ecb91e), SPH_C32(0x1a54091d),
	  SPH_C32(0x057bbcad), SPH_C32(0xa53a0002), SPH_C32(0x88816249),
	  SPH_C32(0xe65f30a5), SPH_C32(0xb1b9c5db) },
	{ SPH_C32(0xe944000d), SPH_C32(0xba55790e), SPH_C32(0xb45ab218),
	  SPH_C32(0xb0df7a96), SPH_C32(0x6dcb0002), SPH_C32(0x83ac85cb),
	  SPH_C32(0x8da97801), SPH_C32(0xe2257864) },
	{ SPH_C32(0xe1fb000c), SPH_C32(0x82c15e9c), SPH_C32(0x71a241b9),
	  SPH_C32(0x56e70112), SPH_C32(0xad850003), SPH_C32(0xb01545db),
	  SPH_C32(0x23a7c304), SPH_C32(0x5781be5f) },
	{ SPH_C32(0x21b5000d), SPH_C32(0xb1789e8c), SPH_C32(0xdfacfabc),
	  SPH_C32(0xe343c729), SPH_C32(0x65740003), SPH_C32(0xbb38a259),
	  SPH_C32(0x48518ba0), SPH_C32(0x041d03e0) },
	{ SPH_C32(0xa129000e), SPH_C32(0xd60b1ead), SPH_C32(0x83b18cb7),
	  SPH_C32(0x880e4b5c), SPH_C32(0xf4960002), SPH_C32(0xad626d5d),
	  SPH_C32(0x9fbd1ae9), SPH_C32(0xa321789d) },
	{ SPH_C32(0x6167000f), SPH_C32(0xe5b2debd), SPH_C32(0x2dbf37b2),
	  SPH_C32(0x3daa8d67), SPH_C32(0x3c670002), SPH_C32(0xa64f8adf),
	  SPH_C32(0xf44b524d), SPH_C32(0xf0bdc522) },
	{ SPH_C32(0x69d8000e), SPH_C32(0xdd26f92f), SPH_C32(0xe847c413),
	  SPH_C32(0xdb92f6e3), SPH_C32(0xfc290003), SPH_C32(0x95f64acf),
	  SPH_C32(0x5a45e948), SPH_C32(0x45190319) },
	{ SPH_C32(0xa996000f), SPH_C32(0xee9f393f), SPH_C32(0x46497f16),
	  SPH_C32(0x6e3630d8), SPH_C32(0x34d80003), SPH_C32(0x9edbad4d),
	  SPH_C32(0x31b3a1ec), SPH_C32(0x1685bea6) },
	{ SPH_C32(0xc3070008), SPH_C32(0x6092c0d7), SPH_C32(0x66418f3c),
	  SPH_C32(0xce05ef02), SPH_C32(0xacbd0004), SPH_C32(0x7ef34599),
	  SPH_C32(0x9b9d2f13), SPH_C32(0x811ea6b6) },
	{ SPH_C32(0x03490009), SPH_C32(0x532b00c7), SPH_C32(0xc84f3439),
	  SPH_C32(0x7ba12939), SPH_C32(0x644c0004), SPH_C32(0x75dea21b),
	  SPH_C32(0xf06b67b7), SPH_C32(0xd2821b09) },
	{ SPH_C32(0x0bf60008), SPH_C32(0x6bbf2755), SPH_C32(0x0db7c798),
	  SPH_C32(0x9d9952bd), SPH_C32(0xa4020005), SPH_C32(0x4667620b),
	  SPH_C32(0x5e65dcb2), SPH_C32(0x6726dd32) },
	{ SPH_C32(0xcbb80009), SPH_C32(0x5806e745), SPH_C32(0xa3b97c9d),
	  SPH_C32(0x283d9486), SPH_C32(0x6cf30005), SPH_C32(0x4d4a8589),
	  SPH_C32(0x35939416), SPH_C32(0x34ba608d) },
	{ SPH_C32(0x4b24000a), SPH_C32(0x3f756764), SPH_C32(0xffa40a96),
	  SPH_C32(0x437018f3), SPH_C32(0xfd110004), SPH_C32(0x5b104a8d),
	  SPH_C32(0xe27f055f), SPH_C32(0x93861bf0) },
	{ SPH_C32(0x8b6a000b), SPH_C32(0x0ccca774), SPH_C32(0x51aab193),
	  SPH_C32(0xf6d4dec8), SPH_C32(0x35e00004), SPH_C32(0x503dad0f),
	  SPH_C32(0x89894dfb), SPH_C32(0xc01aa64f) },
	{ SPH_C32(0x83d5000a), SPH_C32(0x345880e6), SPH_C32(0x94524232),
	  SPH_C32(0x10eca54c), SPH_C32(0xf5ae0005), SPH_C32(0x63846d1f),
	  SPH_C32(0x2787f6fe), SPH_C32(0x75be6074) },
	{ SPH_C32(0x439b000b), SPH_C32(0x07e140f6), SPH_C32(0x3a5cf937),
	  SPH_C32(0xa5486377), SPH_C32(0x3d5f0005), SPH_C32(0x68a98a9d),
	  SPH_C32(0x4c71be5a), SPH_C32(0x2622ddcb) },
	{ SPH_C32(0x92ab0008), SPH_C32(0x4571cfc3), SPH_C32(0x1fa3a570),
	  SPH_C32(0xdc9d5244), SPH_C32(0x75320006), SPH_C32(0x04f7ed3e),
	  SPH_C32(0x7b9a80f5), SPH_C32(0x1ef3ec01) },
	{ SPH_C32(0x52e50009), SPH_C32(0x76c80fd3), SPH_C32(0xb1ad1e75),
	  SPH_C32(0x6939947f), SPH_C32(0xbdc30006), SPH_C32(0x0fda0abc),
	  SPH_C32(0x106cc851), SPH_C32(0x4d6f51be) },
	{ SPH_C32(0x5a5a0008), SPH_C32(0x4e5c2841), SPH_C32(0x7455edd4),
	  SPH_C32(0x8f01effb), SPH_C32(0x7d8d0007), SPH_C32(0x3c63caac),
	  SPH_C32(0xbe627354), SPH_C32(0xf8cb9785) },
	{ SPH_C32(0x9a140009), SPH_C32(0x7de5e851), SPH_C32(0xda5b56d1),
	  SPH_C32(0x3aa529c0), SPH_C32(0xb57c0007), SPH_C32(0x374e2d2e),
	  SPH_C32(0xd5943bf0), SPH_C32(0xab572a3a) },
	{ SPH_C32(0x1a88000a), SPH_C32(0x1a966870), SPH_C32(0x864620da),
	  SPH_C32(0x51e8a5b5), SPH_C32(0x249e0006), SPH_C32(0x2114e22a),
	  SPH_C32(0x0278aab9), SPH_C32(0x0c6b5147) },
	{ SPH_C32(0xdac6000b), SPH_C32(0x292fa860), SPH_C32(0x28489bdf),
	  SPH_C32(0xe44c638e), SPH_C32(0xec6f0006), SPH_C32(0x2a3905a8),
	  SPH_C32(0x698ee21d), SPH_C32(0x5ff7ecf8) },
	{ SPH_C32(0xd279000a), SPH_C32(0x11bb8ff2), SPH_C32(0xedb0687e),
	  SPH_C32(0x0274180a), SPH_C32(0x2c210007), SPH_C32(0x1980c5b8),
	  SPH_C32(0xc7805918), SPH_C32(0xea532ac3) },
	{ SPH_C32(0x1237000b), SPH_C32(0x22024fe2), SPH_C32(0x43bed37b),
	  SPH_C32(0xb7d0de31), SPH_C32(0xe4d00007), SPH_C32(0x12ad223a),
	  SPH_C32(0xac7611bc), SPH_C32(0xb9cf977c) },
	{ SPH_C32(0x130f000c), SPH_C32(0xece44fa0), SPH_C32(0xfb843f6c),
	  SPH_C32(0x614fc6d8), SPH_C32(0xc7140004), SPH_C32(0x3e18bc33),
	  SPH_C32(0x03af332e), SPH_C32(0xf7b26185) },
	{ SPH_C32(0xd341000d), SPH_C32(0xdf5d8fb0), SPH_C32(0x558a8469),
	  SPH_C32(0xd4eb00e3), SPH_C32(0x0fe50004), SPH_C32(0x35355bb1),
	  SPH_C32(0x68597b8a), SPH_C32(0xa42edc3a) },
	{ SPH_C32(0xdbfe000c), SPH_C32(0xe7c9a822), SPH_C32(0x907277c8),
	  SPH_C32(0x32d37b67), SPH_C32(0xcfab0005), SPH_C32(0x068c9ba1),
	  SPH_C32(0xc657c08f), SPH_C32(0x118a1a01) },
	{ SPH_C32(0x1bb0000d), SPH_C32(0xd4706832), SPH_C32(0x3e7ccccd),
	  SPH_C32(0x8777bd5c), SPH_C32(0x075a0005), SPH_C32(0x0da17c23),
	  SPH_C32(0xada1882b), SPH_C32(0x4216a7be) },
	{ SPH_C32(0x9b2c000e), SPH_C32(0xb303e813), SPH_C32(0x6261bac6),
	  SPH_C32(0xec3a3129), SPH_C32(0x96b80004), SPH_C32(0x1bfbb327),
	  SPH_C32(0x7a4d1962), SPH_C32(0xe52adcc3) },
	{ SPH_C32(0x5b62000f), SPH_C32(0x80ba2803), SPH_C32(0xcc6f01c3),
	  SPH_C32(0x599ef712), SPH_C32(0x5e490004), SPH_C32(0x10d654a5),
	  SPH_C32(0x11bb51c6), SPH_C32(0xb6b6617c) },
	{ SPH_C32(0x53dd000e), SPH_C32(0xb82e0f91), SPH_C32(0x0997f262),
	  SPH_C32(0xbfa68c96), SPH_C32(0x9e070005), SPH_C32(0x236f94b5),
	  SPH_C32(0xbfb5eac3), SPH_C32(0x0312a747) },
	{ SPH_C32(0x9393000f), SPH_C32(0x8b97cf81), SPH_C32(0xa7994967),
	  SPH_C32(0x0a024aad), SPH_C32(0x56f60005), SPH_C32(0x28427337),
	  SPH_C32(0xd443a267), SPH_C32(0x508e1af8) },
	{ SPH_C32(0x42a3000c), SPH_C32(0xc90740b4), SPH_C32(0x82661520),
	  SPH_C32(0x73d77b9e), SPH_C32(0x1e9b0006), SPH_C32(0x441c1494),
	  SPH_C32(0xe3a89cc8), SPH_C32(0x685f2b32) },
	{ SPH_C32(0x82ed000d), SPH_C32(0xfabe80a4), SPH_C32(0x2c68ae25),
	  SPH_C32(0xc673bda5), SPH_C32(0xd66a0006), SPH_C32(0x4f31f316),
	  SPH_C32(0x885ed46c), SPH_C32(0x3bc3968d) },
	{ SPH_C32(0x8a52000c), SPH_C32(0xc22aa736), SPH_C32(0xe9905d84),
	  SPH_C32(0x204bc621), SPH_C32(0x16240007), SPH_C32(0x7c883306),
	  SPH_C32(0x26506f69), SPH_C32(0x8e6750b6) },
	{ SPH_C32(0x4a1c000d), SPH_C32(0xf1936726), SPH_C32(0x479ee681),
	  SPH_C32(0x95ef001a), SPH_C32(0xded50007), SPH_C32(0x77a5d484),
	  SPH_C32(0x4da627cd), SPH_C32(0xddfbed09) },
	{ SPH_C32(0xca80000e), SPH_C32(0x96e0e707), SPH_C32(0x1b83908a),
	  SPH_C32(0xfea28c6f), SPH_C32(0x4f370006), SPH_C32(0x61ff1b80),
	  SPH_C32(0x9a4ab684), SPH_C32(0x7ac79674) },
	{ SPH_C32(0x0ace000f), SPH_C32(0xa5592717), SPH_C32(0xb58d2b8f),
	  SPH_C32(0x4b064a54), SPH_C32(0x87c60006), SPH_C32(0x6ad2fc02),
	  SPH_C32(0xf1bcfe20), SPH_C32(0x295b2bcb) },
	{ SPH_C32(0x0271000e), SPH_C32(0x9dcd0085), SPH_C32(0x7075d82e),
	  SPH_C32(0xad3e31d0), SPH_C32(0x47880007), SPH_C32(0x596b3c12),
	  SPH_C32(0x5fb24525), SPH_C32(0x9cffedf0) },
	{ SPH_C32(0xc23f000f), SPH_C32(0xae74c095), SPH_C32(0xde7b632b),
	  SPH_C32(0x189af7eb), SPH_C32(0x8f790007), SPH_C32(0x5246db90),
	  SPH_C32(0x34440d81), SPH_C32(0xcf63504f) },
	{ SPH_C32(0x171c0000), SPH_C32(0xb26e3344), SPH_C32(0x9e6a837e),
	  SPH_C32(0x58f8485f), SPH_C32(0xbfb20008), SPH_C32(0x92170a39),
	  SPH_C32(0x6019107f), SPH_C32(0xe051606e) },
	{ SPH_C32(0xd7520001), SPH_C32(0x81d7f354), SPH_C32(0x3064387b),
	  SPH_C32(0xed5c8e64), SPH_C32(0x77430008), SPH_C32(0x993aedbb),
	  SPH_C32(0x0bef58db), SPH_C32(0xb3cdddd1) },
	{ SPH_C32(0xdfed0000), SPH_C32(0xb943d4c6), SPH_C32(0xf59ccbda),
	  SPH_C32(0x0b64f5e0), SPH_C32(0xb70d0009), SPH_C32(0xaa832dab),
	  SPH_C32(0xa5e1e3de), SPH_C32(0x06691bea) },
	{ SPH_C32(0x1fa30001), SPH_C32(0x8afa14d6), SPH_C32(0x5b9270df),
	  SPH_C32(0xbec033db), SPH_C32(0x7ffc0009), SPH_C32(0xa1aeca29),
	  SPH_C32(0xce17ab7a), SPH_C32(0x55f5a655) },
	{ SPH_C32(0x9f3f0002), SPH_C32(0xed8994f7), SPH_C32(0x078f06d4),
	  SPH_C32(0xd58dbfae), SPH_C32(0xee1e0008), SPH_C32(0xb7f4052d),
	  SPH_C32(0x19fb3a33), SPH_C32(0xf2c9dd28) },
	{ SPH_C32(0x5f710003), SPH_C32(0xde3054e7), SPH_C32(0xa981bdd1),
	  SPH_C32(0x60297995), SPH_C32(0x26ef0008), SPH_C32(0xbcd9e2af),
	  SPH_C32(0x720d7297), SPH_C32(0xa1556097) },
	{ SPH_C32(0x57ce0002), SPH_C32(0xe6a47375), SPH_C32(0x6c794e70),
	  SPH_C32(0x86110211), SPH_C32(0xe6a10009), SPH_C32(0x8f6022bf),
	  SPH_C32(0xdc03c992), SPH_C32(0x14f1a6ac) },
	{ SPH_C32(0x97800003), SPH_C32(0xd51db365), SPH_C32(0xc277f575),
	  SPH_C32(0x33b5c42a), SPH_C32(0x2e500009), SPH_C32(0x844dc53d),
	  SPH_C32(0xb7f58136), SPH_C32(0x476d1b13) },
	{ SPH_C32(0x46b00000), SPH_C32(0x978d3c50), SPH_C32(0xe788a932),
	  SPH_C32(0x4a60f519), SPH_C32(0x663d000a), SPH_C32(0xe813a29e),
	  SPH_C32(0x801ebf99), SPH_C32(0x7fbc2ad9) },
	{ SPH_C32(0x86fe0001), SPH_C32(0xa434fc40), SPH_C32(0x49861237),
	  SPH_C32(0xffc43322), SPH_C32(0xaecc000a), SPH_C32(0xe33e451c),
	  SPH_C32(0xebe8f73d), SPH_C32(0x2c209766) },
	{ SPH_C32(0x8e410000), SPH_C32(0x9ca0dbd2), SPH_C32(0x8c7ee196),
	  SPH_C32(0x19fc48a6), SPH_C32(0x6e82000b), SPH_C32(0xd087850c),
	  SPH_C32(0x45e64c38), SPH_C32(0x9984515d) },
	{ SPH_C32(0x4e0f0001), SPH_C32(0xaf191bc2), SPH_C32(0x22705a93),
	  SPH_C32(0xac588e9d), SPH_C32(0xa673000b), SPH_C32(0xdbaa628e),
	  SPH_C32(0x2e10049c), SPH_C32(0xca18ece2) },
	{ SPH_C32(0xce930002), SPH_C32(0xc86a9be3), SPH_C32(0x7e6d2c98),
	  SPH_C32(0xc71502e8), SPH_C32(0x3791000a), SPH_C32(0xcdf0ad8a),
	  SPH_C32(0xf9fc95d5), SPH_C32(0x6d24979f) },
	{ SPH_C32(0x0edd0003), SPH_C32(0xfbd35bf3), SPH_C32(0xd063979d),
	  SPH_C32(0x72b1c4d3), SPH_C32(0xff60000a), SPH_C32(0xc6dd4a08),
	  SPH_C32(0x920add71), SPH_C32(0x3eb82a20) },
	{ SPH_C32(0x06620002), SPH_C32(0xc3477c61), SPH_C32(0x159b643c),
	  SPH_C32(0x9489bf57), SPH_C32(0x3f2e000b), SPH_C32(0xf5648a18),
	  SPH_C32(0x3c046674), SPH_C32(0x8b1cec1b) },
	{ SPH_C32(0xc62c0003), SPH_C32(0xf0febc71), SPH_C32(0xbb95df39),
	  SPH_C32(0x212d796c), SPH_C32(0xf7df000b), SPH_C32(0xfe496d9a),
	  SPH_C32(0x57f22ed0), SPH_C32(0xd88051a4) },
	{ SPH_C32(0xc7140004), SPH_C32(0x3e18bc33), SPH_C32(0x03af332e),
	  SPH_C32(0xf7b26185), SPH_C32(0xd41b0008), SPH_C32(0xd2fcf393),
	  SPH_C32(0xf82b0c42), SPH_C32(0x96fda75d) },
	{ SPH_C32(0x075a0005), SPH_C32(0x0da17c23), SPH_C32(0xada1882b),
	  SPH_C32(0x4216a7be), SPH_C32(0x1cea0008), SPH_C32(0xd9d11411),
	  SPH_C32(0x93dd44e6), SPH_C32(0xc5611ae2) },
	{ SPH_C32(0x0fe50004), SPH_C32(0x35355bb1), SPH_C32(0x68597b8a),
	  SPH_C32(0xa42edc3a), SPH_C32(0xdca40009), SPH_C32(0xea68d401),
	  SPH_C32(0x3dd3ffe3), SPH_C32(0x70c5dcd9) },
	{ SPH_C32(0xcfab0005), SPH_C32(0x068c9ba1), SPH_C32(0xc657c08f),
	  SPH_C32(0x118a1a01), SPH_C32(0x14550009), SPH_C32(0xe1453383),
	  SPH_C32(0x5625b747), SPH_C32(0x23596166) },
	{ SPH_C32(0x4f370006), SPH_C32(0x61ff1b80), SPH_C32(0x9a4ab684),
	  SPH_C32(0x7ac79674), SPH_C32(0x85b70008), SPH_C32(0xf71ffc87),
	  SPH_C32(0x81c9260e), SPH_C32(0x84651a1b) },
	{ SPH_C32(0x8f790007), SPH_C32(0x5246db90), SPH_C32(0x34440d81),
	  SPH_C32(0xcf63504f), SPH_C32(0x4d460008), SPH_C32(0xfc321b05),
	  SPH_C32(0xea3f6eaa), SPH_C32(0xd7f9a7a4) },
	{ SPH_C32(0x87c60006), SPH_C32(0x6ad2fc02), SPH_C32(0xf1bcfe20),
	  SPH_C32(0x295b2bcb), SPH_C32(0x8d080009), SPH_C32(0xcf8bdb15),
	  SPH_C32(0x4431d5af), SPH_C32(0x625d619f) },
	{ SPH_C32(0x47880007), SPH_C32(0x596b3c12), SPH_C32(0x5fb24525),
	  SPH_C32(0x9cffedf0), SPH_C32(0x45f90009), SPH_C32(0xc4a63c97),
	  SPH_C32(0x2fc79d0b), SPH_C32(0x31c1dc20) },
	{ SPH_C32(0x96b80004), SPH_C32(0x1bfbb327), SPH_C32(0x7a4d1962),
	  SPH_C32(0xe52adcc3), SPH_C32(0x0d94000a), SPH_C32(0xa8f85b34),
	  SPH_C32(0x182ca3a4), SPH_C32(0x0910edea) },
	{ SPH_C32(0x56f60005), SPH_C32(0x28427337), SPH_C32(0xd443a267),
	  SPH_C32(0x508e1af8), SPH_C32(0xc565000a), SPH_C32(0xa3d5bcb6),
	  SPH_C32(0x73daeb00), SPH_C32(0x5a8c5055) },
	{ SPH_C32(0x5e490004), SPH_C32(0x10d654a5), SPH_C32(0x11bb51c6),
	  SPH_C32(0xb6b6617c), SPH_C32(0x052b000b), SPH_C32(0x906c7ca6),
	  SPH_C32(0xddd45005), SPH_C32(0xef28966e) },
	{ SPH_C32(0x9e070005), SPH_C32(0x236f94b5), SPH_C32(0xbfb5eac3),
	  SPH_C32(0x0312a747), SPH_C32(0xcdda000b), SPH_C32(0x9b419b24),
	  SPH_C32(0xb62218a1), SPH_C32(0xbcb42bd1) },
	{ SPH_C32(0x1e9b0006), SPH_C32(0x441c1494), SPH_C32(0xe3a89cc8),
	  SPH_C32(0x685f2b32), SPH_C32(0x5c38000a), SPH_C32(0x8d1b5420),
	  SPH_C32(0x61ce89e8), SPH_C32(0x1b8850ac) },
	{ SPH_C32(0xded50007), SPH_C32(0x77a5d484), SPH_C32(0x4da627cd),
	  SPH_C32(0xddfbed09), SPH_C32(0x94c9000a), SPH_C32(0x8636b3a2),
	  SPH_C32(0x0a38c14c), SPH_C32(0x4814ed13) },
	{ SPH_C32(0xd66a0006), SPH_C32(0x4f31f316), SPH_C32(0x885ed46c),
	  SPH_C32(0x3bc3968d), SPH_C32(0x5487000b), SPH_C32(0xb58f73b2),
	  SPH_C32(0xa4367a49), SPH_C32(0xfdb02b28) },
	{ SPH_C32(0x16240007), SPH_C32(0x7c883306), SPH_C32(0x26506f69),
	  SPH_C32(0x8e6750b6), SPH_C32(0x9c76000b), SPH_C32(0xbea29430),
	  SPH_C32(0xcfc032ed), SPH_C32(0xae2c9697) },
	{ SPH_C32(0x7cb50000), SPH_C32(0xf285caee), SPH_C32(0x06589f43),
	  SPH_C32(0x2e548f6c), SPH_C32(0x0413000c), SPH_C32(0x5e8a7ce4),
	  SPH_C32(0x65eebc12), SPH_C32(0x39b78e87) },
	{ SPH_C32(0xbcfb0001), SPH_C32(0xc13c0afe), SPH_C32(0xa8562446),
	  SPH_C32(0x9bf04957), SPH_C32(0xcce2000c), SPH_C32(0x55a79b66),
	  SPH_C32(0x0e18f4b6), SPH_C32(0x6a2b3338) },
	{ SPH_C32(0xb4440000), SPH_C32(0xf9a82d6c), SPH_C32(0x6daed7e7),
	  SPH_C32(0x7dc832d3), SPH_C32(0x0cac000d), SPH_C32(0x661e5b76),
	  SPH_C32(0xa0164fb3), SPH_C32(0xdf8ff503) },
	{ SPH_C32(0x740a0001), SPH_C32(0xca11ed7c), SPH_C32(0xc3a06ce2),
	  SPH_C32(0xc86cf4e8), SPH_C32(0xc45d000d), SPH_C32(0x6d33bcf4),
	  SPH_C32(0xcbe00717), SPH_C32(0x8c1348bc) },
	{ SPH_C32(0xf4960002), SPH_C32(0xad626d5d), SPH_C32(0x9fbd1ae9),
	  SPH_C32(0xa321789d), SPH_C32(0x55bf000c), SPH_C32(0x7b6973f0),
	  SPH_C32(0x1c0c965e), SPH_C32(0x2b2f33c1) },
	{ SPH_C32(0x34d80003), SPH_C32(0x9edbad4d), SPH_C32(0x31b3a1ec),
	  SPH_C32(0x1685bea6), SPH_C32(0x9d4e000c), SPH_C32(0x70449472),
	  SPH_C32(0x77fadefa), SPH_C32(0x78b38e7e) },
	{ SPH_C32(0x3c670002), SPH_C32(0xa64f8adf), SPH_C32(0xf44b524d),
	  SPH_C32(0xf0bdc522), SPH_C32(0x5d00000d), SPH_C32(0x43fd5462),
	  SPH_C32(0xd9f465ff), SPH_C32(0xcd174845) },
	{ SPH_C32(0xfc290003), SPH_C32(0x95f64acf), SPH_C32(0x5a45e948),
	  SPH_C32(0x45190319), SPH_C32(0x95f1000d), SPH_C32(0x48d0b3e0),
	  SPH_C32(0xb2022d5b), SPH_C32(0x9e8bf5fa) },
	{ SPH_C32(0x2d190000), SPH_C32(0xd766c5fa), SPH_C32(0x7fbab50f),
	  SPH_C32(0x3ccc322a), SPH_C32(0xdd9c000e), SPH_C32(0x248ed443),
	  SPH_C32(0x85e913f4), SPH_C32(0xa65ac430) },
	{ SPH_C32(0xed570001), SPH_C32(0xe4df05ea), SPH_C32(0xd1b40e0a),
	  SPH_C32(0x8968f411), SPH_C32(0x156d000e), SPH_C32(0x2fa333c1),
	  SPH_C32(0xee1f5b50), SPH_C32(0xf5c6798f) },
	{ SPH_C32(0xe5e80000), SPH_C32(0xdc4b2278), SPH_C32(0x144cfdab),
	  SPH_C32(0x6f508f95), SPH_C32(0xd523000f), SPH_C32(0x1c1af3d1),
	  SPH_C32(0x4011e055), SPH_C32(0x4062bfb4) },
	{ SPH_C32(0x25a60001), SPH_C32(0xeff2e268), SPH_C32(0xba4246ae),
	  SPH_C32(0xdaf449ae), SPH_C32(0x1dd2000f), SPH_C32(0x17371453),
	  SPH_C32(0x2be7a8f1), SPH_C32(0x13fe020b) },
	{ SPH_C32(0xa53a0002), SPH_C32(0x88816249), SPH_C32(0xe65f30a5),
	  SPH_C32(0xb1b9c5db), SPH_C32(0x8c30000e), SPH_C32(0x016ddb57),
	  SPH_C32(0xfc0b39b8), SPH_C32(0xb4c27976) },
	{ SPH_C32(0x65740003), SPH_C32(0xbb38a259), SPH_C32(0x48518ba0),
	  SPH_C32(0x041d03e0), SPH_C32(0x44c1000e), SPH_C32(0x0a403cd5),
	  SPH_C32(0x97fd711c), SPH_C32(0xe75ec4c9) },
	{ SPH_C32(0x6dcb0002), SPH_C32(0x83ac85cb), SPH_C32(0x8da97801),
	  SPH_C32(0xe2257864), SPH_C32(0x848f000f), SPH_C32(0x39f9fcc5),
	  SPH_C32(0x39f3ca19), SPH_C32(0x52fa02f2) },
	{ SPH_C32(0xad850003), SPH_C32(0xb01545db), SPH_C32(0x23a7c304),
	  SPH_C32(0x5781be5f), SPH_C32(0x4c7e000f), SPH_C32(0x32d41b47),
	  SPH_C32(0x520582bd), SPH_C32(0x0166bf4d) },
	{ SPH_C32(0xacbd0004), SPH_C32(0x7ef34599), SPH_C32(0x9b9d2f13),
	  SPH_C32(0x811ea6b6), SPH_C32(0x6fba000c), SPH_C32(0x1e61854e),
	  SPH_C32(0xfddca02f), SPH_C32(0x4f1b49b4) },
	{ SPH_C32(0x6cf30005), SPH_C32(0x4d4a8589), SPH_C32(0x35939416),
	  SPH_C32(0x34ba608d), SPH_C32(0xa74b000c), SPH_C32(0x154c62cc),
	  SPH_C32(0x962ae88b), SPH_C32(0x1c87f40b) },
	{ SPH_C32(0x644c0004), SPH_C32(0x75dea21b), SPH_C32(0xf06b67b7),
	  SPH_C32(0xd2821b09), SPH_C32(0x6705000d), SPH_C32(0x26f5a2dc),
	  SPH_C32(0x3824538e), SPH_C32(0xa9233230) },
	{ SPH_C32(0xa4020005), SPH_C32(0x4667620b), SPH_C32(0x5e65dcb2),
	  SPH_C32(0x6726dd32), SPH_C32(0xaff4000d), SPH_C32(0x2dd8455e),
	  SPH_C32(0x53d21b2a), SPH_C32(0xfabf8f8f) },
	{ SPH_C32(0x249e0006), SPH_C32(0x2114e22a), SPH_C32(0x0278aab9),
	  SPH_C32(0x0c6b5147), SPH_C32(0x3e16000c), SPH_C32(0x3b828a5a),
	  SPH_C32(0x843e8a63), SPH_C32(0x5d83f4f2) },
	{ SPH_C32(0xe4d00007), SPH_C32(0x12ad223a), SPH_C32(0xac7611bc),
	  SPH_C32(0xb9cf977c), SPH_C32(0xf6e7000c), SPH_C32(0x30af6dd8),
	  SPH_C32(0xefc8c2c7), SPH_C32(0x0e1f494d) },
	{ SPH_C32(0xec6f0006), SPH_C32(0x2a3905a8), SPH_C32(0x698ee21d),
	  SPH_C32(0x5ff7ecf8), SPH_C32(0x36a9000d), SPH_C32(0x0316adc8),
	  SPH_C32(0x41c679c2), SPH_C32(0xbbbb8f76) },
	{ SPH_C32(0x2c210007), SPH_C32(0x1980c5b8), SPH_C32(0xc7805918),
	  SPH_C32(0xea532ac3), SPH_C32(0xfe58000d), SPH_C32(0x083b4a4a),
	  SPH_C32(0x2a303166), SPH_C32(0xe82732c9) },
	{ SPH_C32(0xfd110004), SPH_C32(0x5b104a8d), SPH_C32(0xe27f055f),
	  SPH_C32(0x93861bf0), SPH_C32(0xb635000e), SPH_C32(0x64652de9),
	  SPH_C32(0x1ddb0fc9), SPH_C32(0xd0f60303) },
	{ SPH_C32(0x3d5f0005), SPH_C32(0x68a98a9d), SPH_C32(0x4c71be5a),
	  SPH_C32(0x2622ddcb), SPH_C32(0x7ec4000e), SPH_C32(0x6f48ca6b),
	  SPH_C32(0x762d476d), SPH_C32(0x836abebc) },
	{ SPH_C32(0x35e00004), SPH_C32(0x503dad0f), SPH_C32(0x89894dfb),
	  SPH_C32(0xc01aa64f), SPH_C32(0xbe8a000f), SPH_C32(0x5cf10a7b),
	  SPH_C32(0xd823fc68), SPH_C32(0x36ce7887) },
	{ SPH_C32(0xf5ae0005), SPH_C32(0x63846d1f), SPH_C32(0x2787f6fe),
	  SPH_C32(0x75be6074), SPH_C32(0x767b000f), SPH_C32(0x57dcedf9),
	  SPH_C32(0xb3d5b4cc), SPH_C32(0x6552c538) },
	{ SPH_C32(0x75320006), SPH_C32(0x04f7ed3e), SPH_C32(0x7b9a80f5),
	  SPH_C32(0x1ef3ec01), SPH_C32(0xe799000e), SPH_C32(0x418622fd),
	  SPH_C32(0x64392585), SPH_C32(0xc26ebe45) },
	{ SPH_C32(0xb57c0007), SPH_C32(0x374e2d2e), SPH_C32(0xd5943bf0),
	  SPH_C32(0xab572a3a), SPH_C32(0x2f68000e), SPH_C32(0x4aabc57f),
	  SPH_C32(0x0fcf6d21), SPH_C32(0x91f203fa) },
	{ SPH_C32(0xbdc30006), SPH_C32(0x0fda0abc), SPH_C32(0x106cc851),
	  SPH_C32(0x4d6f51be), SPH_C32(0xef26000f), SPH_C32(0x7912056f),
	  SPH_C32(0xa1c1d624), SPH_C32(0x2456c5c1) },
	{ SPH_C32(0x7d8d0007), SPH_C32(0x3c63caac), SPH_C32(0xbe627354),
	  SPH_C32(0xf8cb9785), SPH_C32(0x27d7000f), SPH_C32(0x723fe2ed),
	  SPH_C32(0xca379e80), SPH_C32(0x77ca787e) },
	{ SPH_C32(0xbfb20008), SPH_C32(0x92170a39), SPH_C32(0x6019107f),
	  SPH_C32(0xe051606e), SPH_C32(0xa8ae0008), SPH_C32(0x2079397d),
	  SPH_C32(0xfe739301), SPH_C32(0xb8a92831) },
	{ SPH_C32(0x7ffc0009), SPH_C32(0xa1aeca29), SPH_C32(0xce17ab7a),
	  SPH_C32(0x55f5a655), SPH_C32(0x605f0008), SPH_C32(0x2b54deff),
	  SPH_C32(0x9585dba5), SPH_C32(0xeb35958e) },
	{ SPH_C32(0x77430008), SPH_C32(0x993aedbb), SPH_C32(0x0bef58db),
	  SPH_C32(0xb3cdddd1), SPH_C32(0xa0110009), SPH_C32(0x18ed1eef),
	  SPH_C32(0x3b8b60a0), SPH_C32(0x5e9153b5) },
	{ SPH_C32(0xb70d0009), SPH_C32(0xaa832dab), SPH_C32(0xa5e1e3de),
	  SPH_C32(0x06691bea), SPH_C32(0x68e00009), SPH_C32(0x13c0f96d),
	  SPH_C32(0x507d2804), SPH_C32(0x0d0dee0a) },
	{ SPH_C32(0x3791000a), SPH_C32(0xcdf0ad8a), SPH_C32(0xf9fc95d5),
	  SPH_C32(0x6d24979f), SPH_C32(0xf9020008), SPH_C32(0x059a3669),
	  SPH_C32(0x8791b94d), SPH_C32(0xaa319577) },
	{ SPH_C32(0xf7df000b), SPH_C32(0xfe496d9a), SPH_C32(0x57f22ed0),
	  SPH_C32(0xd88051a4), SPH_C32(0x31f30008), SPH_C32(0x0eb7d1eb),
	  SPH_C32(0xec67f1e9), SPH_C32(0xf9ad28c8) },
	{ SPH_C32(0xff60000a), SPH_C32(0xc6dd4a08), SPH_C32(0x920add71),
	  SPH_C32(0x3eb82a20), SPH_C32(0xf1bd0009), SPH_C32(0x3d0e11fb),
	  SPH_C32(0x42694aec), SPH_C32(0x4c09eef3) },
	{ SPH_C32(0x3f2e000b), SPH_C32(0xf5648a18), SPH_C32(0x3c046674),
	  SPH_C32(0x8b1cec1b), SPH_C32(0x394c0009), SPH_C32(0x3623f679),
	  SPH_C32(0x299f0248), SPH_C32(0x1f95534c) },
	{ SPH_C32(0xee1e0008), SPH_C32(0xb7f4052d), SPH_C32(0x19fb3a33),
	  SPH_C32(0xf2c9dd28), SPH_C32(0x7121000a), SPH_C32(0x5a7d91da),
	  SPH_C32(0x1e743ce7), SPH_C32(0x27446286) },
	{ SPH_C32(0x2e500009), SPH_C32(0x844dc53d), SPH_C32(0xb7f58136),
	  SPH_C32(0x476d1b13), SPH_C32(0xb9d0000a), SPH_C32(0x51507658),
	  SPH_C32(0x75827443), SPH_C32(0x74d8df39) },
	{ SPH_C32(0x26ef0008), SPH_C32(0xbcd9e2af), SPH_C32(0x720d7297),
	  SPH_C32(0xa1556097), SPH_C32(0x799e000b), SPH_C32(0x62e9b648),
	  SPH_C32(0xdb8ccf46), SPH_C32(0xc17c1902) },
	{ SPH_C32(0xe6a10009), SPH_C32(0x8f6022bf), SPH_C32(0xdc03c992),
	  SPH_C32(0x14f1a6ac), SPH_C32(0xb16f000b), SPH_C32(0x69c451ca),
	  SPH_C32(0xb07a87e2), SPH_C32(0x92e0a4bd) },
	{ SPH_C32(0x663d000a), SPH_C32(0xe813a29e), SPH_C32(0x801ebf99),
	  SPH_C32(0x7fbc2ad9), SPH_C32(0x208d000a), SPH_C32(0x7f9e9ece),
	  SPH_C32(0x679616ab), SPH_C32(0x35dcdfc0) },
	{ SPH_C32(0xa673000b), SPH_C32(0xdbaa628e), SPH_C32(0x2e10049c),
	  SPH_C32(0xca18ece2), SPH_C32(0xe87c000a), SPH_C32(0x74b3794c),
	  SPH_C32(0x0c605e0f), SPH_C32(0x6640627f) },
	{ SPH_C32(0xaecc000a), SPH_C32(0xe33e451c), SPH_C32(0xebe8f73d),
	  SPH_C32(0x2c209766), SPH_C32(0x2832000b), SPH_C32(0x470ab95c),
	  SPH_C32(0xa26ee50a), SPH_C32(0xd3e4a444) },
	{ SPH_C32(0x6e82000b), SPH_C32(0xd087850c), SPH_C32(0x45e64c38),
	  SPH_C32(0x9984515d), SPH_C32(0xe0c3000b), SPH_C32(0x4c275ede),
	  SPH_C32(0xc998adae), SPH_C32(0x807819fb) },
	{ SPH_C32(0x6fba000c), SPH_C32(0x1e61854e), SPH_C32(0xfddca02f),
	  SPH_C32(0x4f1b49b4), SPH_C32(0xc3070008), SPH_C32(0x6092c0d7),
	  SPH_C32(0x66418f3c), SPH_C32(0xce05ef02) },
	{ SPH_C32(0xaff4000d), SPH_C32(0x2dd8455e), SPH_C32(0x53d21b2a),
	  SPH_C32(0xfabf8f8f), SPH_C32(0x0bf60008), SPH_C32(0x6bbf2755),
	  SPH_C32(0x0db7c798), SPH_C32(0x9d9952bd) },
	{ SPH_C32(0xa74b000c), SPH_C32(0x154c62cc), SPH_C32(0x962ae88b),
	  SPH_C32(0x1c87f40b), SPH_C32(0xcbb80009), SPH_C32(0x5806e745),
	  SPH_C32(0xa3b97c9d), SPH_C32(0x283d9486) },
	{ SPH_C32(0x6705000d), SPH_C32(0x26f5a2dc), SPH_C32(0x3824538e),
	  SPH_C32(0xa9233230), SPH_C32(0x03490009), SPH_C32(0x532b00c7),
	  SPH_C32(0xc84f3439), SPH_C32(0x7ba12939) },
	{ SPH_C32(0xe799000e), SPH_C32(0x418622fd), SPH_C32(0x64392585),
	  SPH_C32(0xc26ebe45), SPH_C32(0x92ab0008), SPH_C32(0x4571cfc3),
	  SPH_C32(0x1fa3a570), SPH_C32(0xdc9d5244) },
	{ SPH_C32(0x27d7000f), SPH_C32(0x723fe2ed), SPH_C32(0xca379e80),
	  SPH_C32(0x77ca787e), SPH_C32(0x5a5a0008), SPH_C32(0x4e5c2841),
	  SPH_C32(0x7455edd4), SPH_C32(0x8f01effb) },
	{ SPH_C32(0x2f68000e), SPH_C32(0x4aabc57f), SPH_C32(0x0fcf6d21),
	  SPH_C32(0x91f203fa), SPH_C32(0x9a140009), SPH_C32(0x7de5e851),
	  SPH_C32(0xda5b56d1), SPH_C32(0x3aa529c0) },
	{ SPH_C32(0xef26000f), SPH_C32(0x7912056f), SPH_C32(0xa1c1d624),
	  SPH_C32(0x2456c5c1), SPH_C32(0x52e50009), SPH_C32(0x76c80fd3),
	  SPH_C32(0xb1ad1e75), SPH_C32(0x6939947f) },
	{ SPH_C32(0x3e16000c), SPH_C32(0x3b828a5a), SPH_C32(0x843e8a63),
	  SPH_C32(0x5d83f4f2), SPH_C32(0x1a88000a), SPH_C32(0x1a966870),
	  SPH_C32(0x864620da), SPH_C32(0x51e8a5b5) },
	{ SPH_C32(0xfe58000d), SPH_C32(0x083b4a4a), SPH_C32(0x2a303166),
	  SPH_C32(0xe82732c9), SPH_C32(0xd279000a), SPH_C32(0x11bb8ff2),
	  SPH_C32(0xedb0687e), SPH_C32(0x0274180a) },
	{ SPH_C32(0xf6e7000c), SPH_C32(0x30af6dd8), SPH_C32(0xefc8c2c7),
	  SPH_C32(0x0e1f494d), SPH_C32(0x1237000b), SPH_C32(0x22024fe2),
	  SPH_C32(0x43bed37b), SPH_C32(0xb7d0de31) },
	{ SPH_C32(0x36a9000d), SPH_C32(0x0316adc8), SPH_C32(0x41c679c2),
	  SPH_C32(0xbbbb8f76), SPH_C32(0xdac6000b), SPH_C32(0x292fa860),
	  SPH_C32(0x28489bdf), SPH_C32(0xe44c638e) },
	{ SPH_C32(0xb635000e), SPH_C32(0x64652de9), SPH_C32(0x1ddb0fc9),
	  SPH_C32(0xd0f60303), SPH_C32(0x4b24000a), SPH_C32(0x3f756764),
	  SPH_C32(0xffa40a96), SPH_C32(0x437018f3) },
	{ SPH_C32(0x767b000f), SPH_C32(0x57dcedf9), SPH_C32(0xb3d5b4cc),
	  SPH_C32(0x6552c538), SPH_C32(0x83d5000a), SPH_C32(0x345880e6),
	  SPH_C32(0x94524232), SPH_C32(0x10eca54c) },
	{ SPH_C32(0x7ec4000e), SPH_C32(0x6f48ca6b), SPH_C32(0x762d476d),
	  SPH_C32(0x836abebc), SPH_C32(0x439b000b), SPH_C32(0x07e140f6),
	  SPH_C32(0x3a5cf937), SPH_C32(0xa5486377) },
	{ SPH_C32(0xbe8a000f), SPH_C32(0x5cf10a7b), SPH_C32(0xd823fc68),
	  SPH_C32(0x36ce7887), SPH_C32(0x8b6a000b), SPH_C32(0x0ccca774),
	  SPH_C32(0x51aab193), SPH_C32(0xf6d4dec8) },
	{ SPH_C32(0xd41b0008), SPH_C32(0xd2fcf393), SPH_C32(0xf82b0c42),
	  SPH_C32(0x96fda75d), SPH_C32(0x130f000c), SPH_C32(0xece44fa0),
	  SPH_C32(0xfb843f6c), SPH_C32(0x614fc6d8) },
	{ SPH_C32(0x14550009), SPH_C32(0xe1453383), SPH_C32(0x5625b747),
	  SPH_C32(0x23596166), SPH_C32(0xdbfe000c), SPH_C32(0xe7c9a822),
	  SPH_C32(0x907277c8), SPH_C32(0x32d37b67) },
	{ SPH_C32(0x1cea0008), SPH_C32(0xd9d11411), SPH_C32(0x93dd44e6),
	  SPH_C32(0xc5611ae2), SPH_C32(0x1bb0000d), SPH_C32(0xd4706832),
	  SPH_C32(0x3e7ccccd), SPH_C32(0x8777bd5c) },
	{ SPH_C32(0xdca40009), SPH_C32(0xea68d401), SPH_C32(0x3dd3ffe3),
	  SPH_C32(0x70c5dcd9), SPH_C32(0xd341000d), SPH_C32(0xdf5d8fb0),
	  SPH_C32(0x558a8469), SPH_C32(0xd4eb00e3) },
	{ SPH_C32(0x5c38000a), SPH_C32(0x8d1b5420), SPH_C32(0x61ce89e8),
	  SPH_C32(0x1b8850ac), SPH_C32(0x42a3000c), SPH_C32(0xc90740b4),
	  SPH_C32(0x82661520), SPH_C32(0x73d77b9e) },
	{ SPH_C32(0x9c76000b), SPH_C32(0xbea29430), SPH_C32(0xcfc032ed),
	  SPH_C32(0xae2c9697), SPH_C32(0x8a52000c), SPH_C32(0xc22aa736),
	  SPH_C32(0xe9905d84), SPH_C32(0x204bc621) },
	{ SPH_C32(0x94c9000a), SPH_C32(0x8636b3a2), SPH_C32(0x0a38c14c),
	  SPH_C32(0x4814ed13), SPH_C32(0x4a1c000d), SPH_C32(0xf1936726),
	  SPH_C32(0x479ee681), SPH_C32(0x95ef001a) },
	{ SPH_C32(0x5487000b), SPH_C32(0xb58f73b2), SPH_C32(0xa4367a49),
	  SPH_C32(0xfdb02b28), SPH_C32(0x82ed000d), SPH_C32(0xfabe80a4),
	  SPH_C32(0x2c68ae25), SPH_C32(0xc673bda5) },
	{ SPH_C32(0x85b70008), SPH_C32(0xf71ffc87), SPH_C32(0x81c9260e),
	  SPH_C32(0x84651a1b), SPH_C32(0xca80000e), SPH_C32(0x96e0e707),
	  SPH_C32(0x1b83908a), SPH_C32(0xfea28c6f) },
	{ SPH_C32(0x45f90009), SPH_C32(0xc4a63c97), SPH_C32(0x2fc79d0b),
	  SPH_C32(0x31c1dc20), SPH_C32(0x0271000e), SPH_C32(0x9dcd0085),
	  SPH_C32(0x7075d82e), SPH_C32(0xad3e31d0) },
	{ SPH_C32(0x4d460008), SPH_C32(0xfc321b05), SPH_C32(0xea3f6eaa),
	  SPH_C32(0xd7f9a7a4), SPH_C32(0xc23f000f), SPH_C32(0xae74c095),
	  SPH_C32(0xde7b632b), SPH_C32(0x189af7eb) },
	{ SPH_C32(0x8d080009), SPH_C32(0xcf8bdb15), SPH_C32(0x4431d5af),
	  SPH_C32(0x625d619f), SPH_C32(0x0ace000f), SPH_C32(0xa5592717),
	  SPH_C32(0xb58d2b8f), SPH_C32(0x4b064a54) },
	{ SPH_C32(0x0d94000a), SPH_C32(0xa8f85b34), SPH_C32(0x182ca3a4),
	  SPH_C32(0x0910edea), SPH_C32(0x9b2c000e), SPH_C32(0xb303e813),
	  SPH_C32(0x6261bac6), SPH_C32(0xec3a3129) },
	{ SPH_C32(0xcdda000b), SPH_C32(0x9b419b24), SPH_C32(0xb62218a1),
	  SPH_C32(0xbcb42bd1), SPH_C32(0x53dd000e), SPH_C32(0xb82e0f91),
	  SPH_C32(0x0997f262), SPH_C32(0xbfa68c96) },
	{ SPH_C32(0xc565000a), SPH_C32(0xa3d5bcb6), SPH_C32(0x73daeb00),
	  SPH_C32(0x5a8c5055), SPH_C32(0x9393000f), SPH_C32(0x8b97cf81),
	  SPH_C32(0xa7994967), SPH_C32(0x0a024aad) },
	{ SPH_C32(0x052b000b), SPH_C32(0x906c7ca6), SPH_C32(0xddd45005),
	  SPH_C32(0xef28966e), SPH_C32(0x5b62000f), SPH_C32(0x80ba2803),
	  SPH_C32(0xcc6f01c3), SPH_C32(0x599ef712) },
	{ SPH_C32(0x0413000c), SPH_C32(0x5e8a7ce4), SPH_C32(0x65eebc12),
	  SPH_C32(0x39b78e87), SPH_C32(0x78a6000c), SPH_C32(0xac0fb60a),
	  SPH_C32(0x63b62351), SPH_C32(0x17e301eb) },
	{ SPH_C32(0xc45d000d), SPH_C32(0x6d33bcf4), SPH_C32(0xcbe00717),
	  SPH_C32(0x8c1348bc), SPH_C32(0xb057000c), SPH_C32(0xa7225188),
	  SPH_C32(0x08406bf5), SPH_C32(0x447fbc54) },
	{ SPH_C32(0xcce2000c), SPH_C32(0x55a79b66), SPH_C32(0x0e18f4b6),
	  SPH_C32(0x6a2b3338), SPH_C32(0x7019000d), SPH_C32(0x949b9198),
	  SPH_C32(0xa64ed0f0), SPH_C32(0xf1db7a6f) },
	{ SPH_C32(0x0cac000d), SPH_C32(0x661e5b76), SPH_C32(0xa0164fb3),
	  SPH_C32(0xdf8ff503), SPH_C32(0xb8e8000d), SPH_C32(0x9fb6761a),
	  SPH_C32(0xcdb89854), SPH_C32(0xa247c7d0) },
	{ SPH_C32(0x8c30000e), SPH_C32(0x016ddb57), SPH_C32(0xfc0b39b8),
	  SPH_C32(0xb4c27976), SPH_C32(0x290a000c), SPH_C32(0x89ecb91e),
	  SPH_C32(0x1a54091d), SPH_C32(0x057bbcad) },
	{ SPH_C32(0x4c7e000f), SPH_C32(0x32d41b47), SPH_C32(0x520582bd),
	  SPH_C32(0x0166bf4d), SPH_C32(0xe1fb000c), SPH_C32(0x82c15e9c),
	  SPH_C32(0x71a241b9), SPH_C32(0x56e70112) },
	{ SPH_C32(0x44c1000e), SPH_C32(0x0a403cd5), SPH_C32(0x97fd711c),
	  SPH_C32(0xe75ec4c9), SPH_C32(0x21b5000d), SPH_C32(0xb1789e8c),
	  SPH_C32(0xdfacfabc), SPH_C32(0xe343c729) },
	{ SPH_C32(0x848f000f), SPH_C32(0x39f9fcc5), SPH_C32(0x39f3ca19),
	  SPH_C32(0x52fa02f2), SPH_C32(0xe944000d), SPH_C32(0xba55790e),
	  SPH_C32(0xb45ab218), SPH_C32(0xb0df7a96) },
	{ SPH_C32(0x55bf000c), SPH_C32(0x7b6973f0), SPH_C32(0x1c0c965e),
	  SPH_C32(0x2b2f33c1), SPH_C32(0xa129000e), SPH_C32(0xd60b1ead),
	  SPH_C32(0x83b18cb7), SPH_C32(0x880e4b5c) },
	{ SPH_C32(0x95f1000d), SPH_C32(0x48d0b3e0), SPH_C32(0xb2022d5b),
	  SPH_C32(0x9e8bf5fa), SPH_C32(0x69d8000e), SPH_C32(0xdd26f92f),
	  SPH_C32(0xe847c413), SPH_C32(0xdb92f6e3) },
	{ SPH_C32(0x9d4e000c), SPH_C32(0x70449472), SPH_C32(0x77fadefa),
	  SPH_C32(0x78b38e7e), SPH_C32(0xa996000f), SPH_C32(0xee9f393f),
	  SPH_C32(0x46497f16), SPH_C32(0x6e3630d8) },
	{ SPH_C32(0x5d00000d), SPH_C32(0x43fd5462), SPH_C32(0xd9f465ff),
	  SPH_C32(0xcd174845), SPH_C32(0x6167000f), SPH_C32(0xe5b2debd),
	  SPH_C32(0x2dbf37b2), SPH_C32(0x3daa8d67) },
	{ SPH_C32(0xdd9c000e), SPH_C32(0x248ed443), SPH_C32(0x85e913f4),
	  SPH_C32(0xa65ac430), SPH_C32(0xf085000e), SPH_C32(0xf3e811b9),
	  SPH_C32(0xfa53a6fb), SPH_C32(0x9a96f61a) },
	{ SPH_C32(0x1dd2000f), SPH_C32(0x17371453), SPH_C32(0x2be7a8f1),
	  SPH_C32(0x13fe020b), SPH_C32(0x3874000e), SPH_C32(0xf8c5f63b),
	  SPH_C32(0x91a5ee5f), SPH_C32(0xc90a4ba5) },
	{ SPH_C32(0x156d000e), SPH_C32(0x2fa333c1), SPH_C32(0xee1f5b50),
	  SPH_C32(0xf5c6798f), SPH_C32(0xf83a000f), SPH_C32(0xcb7c362b),
	  SPH_C32(0x3fab555a), SPH_C32(0x7cae8d9e) },
	{ SPH_C32(0xd523000f), SPH_C32(0x1c1af3d1), SPH_C32(0x4011e055),
	  SPH_C32(0x4062bfb4), SPH_C32(0x30cb000f), SPH_C32(0xc051d1a9),
	  SPH_C32(0x545d1dfe), SPH_C32(0x2f323021) }
};

#define INPUT_SMALL   do { \
		unsigned acc = buf[0]; \
		const sph_u32 *rp; \
		rp = &T256_0[acc][0]; \
		m0 = rp[0]; \
		m1 = rp[1]; \
		m2 = rp[2]; \
		m3 = rp[3]; \
		m4 = rp[4]; \
		m5 = rp[5]; \
		m6 = rp[6]; \
		m7 = rp[7]; \
		acc = buf[1]; \
		rp = &T256_8[acc][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		acc = buf[2]; \
		rp = &T256_16[acc][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		acc = buf[3]; \
		rp = &T256_24[acc][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
	} while (0)

#endif

#if SPH_HAMSI_EXPAND_BIG == 1

/* Note: this table lists bits within each byte from least
   siginificant to most significant. */
static const sph_u32 T512[64][16] = {
	{ SPH_C32(0xef0b0270), SPH_C32(0x3afd0000), SPH_C32(0x5dae0000),
	  SPH_C32(0x69490000), SPH_C32(0x9b0f3c06), SPH_C32(0x4405b5f9),
	  SPH_C32(0x66140a51), SPH_C32(0x924f5d0a), SPH_C32(0xc96b0030),
	  SPH_C32(0xe7250000), SPH_C32(0x2f840000), SPH_C32(0x264f0000),
	  SPH_C32(0x08695bf9), SPH_C32(0x6dfcf137), SPH_C32(0x509f6984),
	  SPH_C32(0x9e69af68) },
	{ SPH_C32(0xc96b0030), SPH_C32(0xe7250000), SPH_C32(0x2f840000),
	  SPH_C32(0x264f0000), SPH_C32(0x08695bf9), SPH_C32(0x6dfcf137),
	  SPH_C32(0x509f6984), SPH_C32(0x9e69af68), SPH_C32(0x26600240),
	  SPH_C32(0xddd80000), SPH_C32(0x722a0000), SPH_C32(0x4f060000),
	  SPH_C32(0x936667ff), SPH_C32(0x29f944ce), SPH_C32(0x368b63d5),
	  SPH_C32(0x0c26f262) },
	{ SPH_C32(0x145a3c00), SPH_C32(0xb9e90000), SPH_C32(0x61270000),
	  SPH_C32(0xf1610000), SPH_C32(0xce613d6c), SPH_C32(0xb0493d78),
	  SPH_C32(0x47a96720), SPH_C32(0xe18e24c5), SPH_C32(0x23671400),
	  SPH_C32(0xc8b90000), SPH_C32(0xf4c70000), SPH_C32(0xfb750000),
	  SPH_C32(0x73cd2465), SPH_C32(0xf8a6a549), SPH_C32(0x02c40a3f),
	  SPH_C32(0xdc24e61f) },
	{ SPH_C32(0x23671400), SPH_C32(0xc8b90000), SPH_C32(0xf4c70000),
	  SPH_C32(0xfb750000), SPH_C32(0x73cd2465), SPH_C32(0xf8a6a549),
	  SPH_C32(0x02c40a3f), SPH_C32(0xdc24e61f), SPH_C32(0x373d2800),
	  SPH_C32(0x71500000), SPH_C32(0x95e00000), SPH_C32(0x0a140000),
	  SPH_C32(0xbdac1909), SPH_C32(0x48ef9831), SPH_C32(0x456d6d1f),
	  SPH_C32(0x3daac2da) },
	{ SPH_C32(0x54285c00), SPH_C32(0xeaed0000), SPH_C32(0xc5d60000),
	  SPH_C32(0xa1c50000), SPH_C32(0xb3a26770), SPH_C32(0x94a5c4e1),
	  SPH_C32(0x6bb0419d), SPH_C32(0x551b3782), SPH_C32(0x9cbb1800),
	  SPH_C32(0xb0d30000), SPH_C32(0x92510000), SPH_C32(0xed930000),
	  SPH_C32(0x593a4345), SPH_C32(0xe114d5f4), SPH_C32(0x430633da),
	  SPH_C32(0x78cace29) },
	{ SPH_C32(0x9cbb1800), SPH_C32(0xb0d30000), SPH_C32(0x92510000),
	  SPH_C32(0xed930000), SPH_C32(0x593a4345), SPH_C32(0xe114d5f4),
	  SPH_C32(0x430633da), SPH_C32(0x78cace29), SPH_C32(0xc8934400),
	  SPH_C32(0x5a3e0000), SPH_C32(0x57870000), SPH_C32(0x4c560000),
	  SPH_C32(0xea982435), SPH_C32(0x75b11115), SPH_C32(0x28b67247),
	  SPH_C32(0x2dd1f9ab) },
	{ SPH_C32(0x29449c00), SPH_C32(0x64e70000), SPH_C32(0xf24b0000),
	  SPH_C32(0xc2f30000), SPH_C32(0x0ede4e8f), SPH_C32(0x56c23745),
	  SPH_C32(0xf3e04259), SPH_C32(0x8d0d9ec4), SPH_C32(0x466d0c00),
	  SPH_C32(0x08620000), SPH_C32(0xdd5d0000), SPH_C32(0xbadd0000),
	  SPH_C32(0x6a927942), SPH_C32(0x441f2b93), SPH_C32(0x218ace6f),
	  SPH_C32(0xbf2c0be2) },
	{ SPH_C32(0x466d0c00), SPH_C32(0x08620000), SPH_C32(0xdd5d0000),
	  SPH_C32(0xbadd0000), SPH_C32(0x6a927942), SPH_C32(0x441f2b93),
	  SPH_C32(0x218ace6f), SPH_C32(0xbf2c0be2), SPH_C32(0x6f299000),
	  SPH_C32(0x6c850000), SPH_C32(0x2f160000), SPH_C32(0x782e0000),
	  SPH_C32(0x644c37cd), SPH_C32(0x12dd1cd6), SPH_C32(0xd26a8c36),
	  SPH_C32(0x32219526) },
	{ SPH_C32(0xf6800005), SPH_C32(0x3443c000), SPH_C32(0x24070000),
	  SPH_C32(0x8f3d0000), SPH_C32(0x21373bfb), SPH_C32(0x0ab8d5ae),
	  SPH_C32(0xcdc58b19), SPH_C32(0xd795ba31), SPH_C32(0xa67f0001),
	  SPH_C32(0x71378000), SPH_C32(0x19fc0000), SPH_C32(0x96db0000),
	  SPH_C32(0x3a8b6dfd), SPH_C32(0xebcaaef3), SPH_C32(0x2c6d478f),
	  SPH_C32(0xac8e6c88) },
	{ SPH_C32(0xa67f0001), SPH_C32(0x71378000), SPH_C32(0x19fc0000),
	  SPH_C32(0x96db0000), SPH_C32(0x3a8b6dfd), SPH_C32(0xebcaaef3),
	  SPH_C32(0x2c6d478f), SPH_C32(0xac8e6c88), SPH_C32(0x50ff0004),
	  SPH_C32(0x45744000), SPH_C32(0x3dfb0000), SPH_C32(0x19e60000),
	  SPH_C32(0x1bbc5606), SPH_C32(0xe1727b5d), SPH_C32(0xe1a8cc96),
	  SPH_C32(0x7b1bd6b9) },
	{ SPH_C32(0xf7750009), SPH_C32(0xcf3cc000), SPH_C32(0xc3d60000),
	  SPH_C32(0x04920000), SPH_C32(0x029519a9), SPH_C32(0xf8e836ba),
	  SPH_C32(0x7a87f14e), SPH_C32(0x9e16981a), SPH_C32(0xd46a0000),
	  SPH_C32(0x8dc8c000), SPH_C32(0xa5af0000), SPH_C32(0x4a290000),
	  SPH_C32(0xfc4e427a), SPH_C32(0xc9b4866c), SPH_C32(0x98369604),
	  SPH_C32(0xf746c320) },
	{ SPH_C32(0xd46a0000), SPH_C32(0x8dc8c000), SPH_C32(0xa5af0000),
	  SPH_C32(0x4a290000), SPH_C32(0xfc4e427a), SPH_C32(0xc9b4866c),
	  SPH_C32(0x98369604), SPH_C32(0xf746c320), SPH_C32(0x231f0009),
	  SPH_C32(0x42f40000), SPH_C32(0x66790000), SPH_C32(0x4ebb0000),
	  SPH_C32(0xfedb5bd3), SPH_C32(0x315cb0d6), SPH_C32(0xe2b1674a),
	  SPH_C32(0x69505b3a) },
	{ SPH_C32(0x774400f0), SPH_C32(0xf15a0000), SPH_C32(0xf5b20000),
	  SPH_C32(0x34140000), SPH_C32(0x89377e8c), SPH_C32(0x5a8bec25),
	  SPH_C32(0x0bc3cd1e), SPH_C32(0xcf3775cb), SPH_C32(0xf46c0050),
	  SPH_C32(0x96180000), SPH_C32(0x14a50000), SPH_C32(0x031f0000),
	  SPH_C32(0x42947eb8), SPH_C32(0x66bf7e19), SPH_C32(0x9ca470d2),
	  SPH_C32(0x8a341574) },
	{ SPH_C32(0xf46c0050), SPH_C32(0x96180000), SPH_C32(0x14a50000),
	  SPH_C32(0x031f0000), SPH_C32(0x42947eb8), SPH_C32(0x66bf7e19),
	  SPH_C32(0x9ca470d2), SPH_C32(0x8a341574), SPH_C32(0x832800a0),
	  SPH_C32(0x67420000), SPH_C32(0xe1170000), SPH_C32(0x370b0000),
	  SPH_C32(0xcba30034), SPH_C32(0x3c34923c), SPH_C32(0x9767bdcc),
	  SPH_C32(0x450360bf) },
	{ SPH_C32(0xe8870170), SPH_C32(0x9d720000), SPH_C32(0x12db0000),
	  SPH_C32(0xd4220000), SPH_C32(0xf2886b27), SPH_C32(0xa921e543),
	  SPH_C32(0x4ef8b518), SPH_C32(0x618813b1), SPH_C32(0xb4370060),
	  SPH_C32(0x0c4c0000), SPH_C32(0x56c20000), SPH_C32(0x5cae0000),
	  SPH_C32(0x94541f3f), SPH_C32(0x3b3ef825), SPH_C32(0x1b365f3d),
	  SPH_C32(0xf3d45758) },
	{ SPH_C32(0xb4370060), SPH_C32(0x0c4c0000), SPH_C32(0x56c20000),
	  SPH_C32(0x5cae0000), SPH_C32(0x94541f3f), SPH_C32(0x3b3ef825),
	  SPH_C32(0x1b365f3d), SPH_C32(0xf3d45758), SPH_C32(0x5cb00110),
	  SPH_C32(0x913e0000), SPH_C32(0x44190000), SPH_C32(0x888c0000),
	  SPH_C32(0x66dc7418), SPH_C32(0x921f1d66), SPH_C32(0x55ceea25),
	  SPH_C32(0x925c44e9) },
	{ SPH_C32(0x0c720000), SPH_C32(0x49e50f00), SPH_C32(0x42790000),
	  SPH_C32(0x5cea0000), SPH_C32(0x33aa301a), SPH_C32(0x15822514),
	  SPH_C32(0x95a34b7b), SPH_C32(0xb44b0090), SPH_C32(0xfe220000),
	  SPH_C32(0xa7580500), SPH_C32(0x25d10000), SPH_C32(0xf7600000),
	  SPH_C32(0x893178da), SPH_C32(0x1fd4f860), SPH_C32(0x4ed0a315),
	  SPH_C32(0xa123ff9f) },
	{ SPH_C32(0xfe220000), SPH_C32(0xa7580500), SPH_C32(0x25d10000),
	  SPH_C32(0xf7600000), SPH_C32(0x893178da), SPH_C32(0x1fd4f860),
	  SPH_C32(0x4ed0a315), SPH_C32(0xa123ff9f), SPH_C32(0xf2500000),
	  SPH_C32(0xeebd0a00), SPH_C32(0x67a80000), SPH_C32(0xab8a0000),
	  SPH_C32(0xba9b48c0), SPH_C32(0x0a56dd74), SPH_C32(0xdb73e86e),
	  SPH_C32(0x1568ff0f) },
	{ SPH_C32(0x45180000), SPH_C32(0xa5b51700), SPH_C32(0xf96a0000),
	  SPH_C32(0x3b480000), SPH_C32(0x1ecc142c), SPH_C32(0x231395d6),
	  SPH_C32(0x16bca6b0), SPH_C32(0xdf33f4df), SPH_C32(0xb83d0000),
	  SPH_C32(0x16710600), SPH_C32(0x379a0000), SPH_C32(0xf5b10000),
	  SPH_C32(0x228161ac), SPH_C32(0xae48f145), SPH_C32(0x66241616),
	  SPH_C32(0xc5c1eb3e) },
	{ SPH_C32(0xb83d0000), SPH_C32(0x16710600), SPH_C32(0x379a0000),
	  SPH_C32(0xf5b10000), SPH_C32(0x228161ac), SPH_C32(0xae48f145),
	  SPH_C32(0x66241616), SPH_C32(0xc5c1eb3e), SPH_C32(0xfd250000),
	  SPH_C32(0xb3c41100), SPH_C32(0xcef00000), SPH_C32(0xcef90000),
	  SPH_C32(0x3c4d7580), SPH_C32(0x8d5b6493), SPH_C32(0x7098b0a6),
	  SPH_C32(0x1af21fe1) },
	{ SPH_C32(0x75a40000), SPH_C32(0xc28b2700), SPH_C32(0x94a40000),
	  SPH_C32(0x90f50000), SPH_C32(0xfb7857e0), SPH_C32(0x49ce0bae),
	  SPH_C32(0x1767c483), SPH_C32(0xaedf667e), SPH_C32(0xd1660000),
	  SPH_C32(0x1bbc0300), SPH_C32(0x9eec0000), SPH_C32(0xf6940000),
	  SPH_C32(0x03024527), SPH_C32(0xcf70fcf2), SPH_C32(0xb4431b17),
	  SPH_C32(0x857f3c2b) },
	{ SPH_C32(0xd1660000), SPH_C32(0x1bbc0300), SPH_C32(0x9eec0000),
	  SPH_C32(0xf6940000), SPH_C32(0x03024527), SPH_C32(0xcf70fcf2),
	  SPH_C32(0xb4431b17), SPH_C32(0x857f3c2b), SPH_C32(0xa4c20000),
	  SPH_C32(0xd9372400), SPH_C32(0x0a480000), SPH_C32(0x66610000),
	  SPH_C32(0xf87a12c7), SPH_C32(0x86bef75c), SPH_C32(0xa324df94),
	  SPH_C32(0x2ba05a55) },
	{ SPH_C32(0x75c90003), SPH_C32(0x0e10c000), SPH_C32(0xd1200000),
	  SPH_C32(0xbaea0000), SPH_C32(0x8bc42f3e), SPH_C32(0x8758b757),
	  SPH_C32(0xbb28761d), SPH_C32(0x00b72e2b), SPH_C32(0xeecf0001),
	  SPH_C32(0x6f564000), SPH_C32(0xf33e0000), SPH_C32(0xa79e0000),
	  SPH_C32(0xbdb57219), SPH_C32(0xb711ebc5), SPH_C32(0x4a3b40ba),
	  SPH_C32(0xfeabf254) },
	{ SPH_C32(0xeecf0001), SPH_C32(0x6f564000), SPH_C32(0xf33e0000),
	  SPH_C32(0xa79e0000), SPH_C32(0xbdb57219), SPH_C32(0xb711ebc5),
	  SPH_C32(0x4a3b40ba), SPH_C32(0xfeabf254), SPH_C32(0x9b060002),
	  SPH_C32(0x61468000), SPH_C32(0x221e0000), SPH_C32(0x1d740000),
	  SPH_C32(0x36715d27), SPH_C32(0x30495c92), SPH_C32(0xf11336a7),
	  SPH_C32(0xfe1cdc7f) },
	{ SPH_C32(0x86790000), SPH_C32(0x3f390002), SPH_C32(0xe19ae000),
	  SPH_C32(0x98560000), SPH_C32(0x9565670e), SPH_C32(0x4e88c8ea),
	  SPH_C32(0xd3dd4944), SPH_C32(0x161ddab9), SPH_C32(0x30b70000),
	  SPH_C32(0xe5d00000), SPH_C32(0xf4f46000), SPH_C32(0x42c40000),
	  SPH_C32(0x63b83d6a), SPH_C32(0x78ba9460), SPH_C32(0x21afa1ea),
	  SPH_C32(0xb0a51834) },
	{ SPH_C32(0x30b70000), SPH_C32(0xe5d00000), SPH_C32(0xf4f46000),
	  SPH_C32(0x42c40000), SPH_C32(0x63b83d6a), SPH_C32(0x78ba9460),
	  SPH_C32(0x21afa1ea), SPH_C32(0xb0a51834), SPH_C32(0xb6ce0000),
	  SPH_C32(0xdae90002), SPH_C32(0x156e8000), SPH_C32(0xda920000),
	  SPH_C32(0xf6dd5a64), SPH_C32(0x36325c8a), SPH_C32(0xf272e8ae),
	  SPH_C32(0xa6b8c28d) },
	{ SPH_C32(0x14190000), SPH_C32(0x23ca003c), SPH_C32(0x50df0000),
	  SPH_C32(0x44b60000), SPH_C32(0x1b6c67b0), SPH_C32(0x3cf3ac75),
	  SPH_C32(0x61e610b0), SPH_C32(0xdbcadb80), SPH_C32(0xe3430000),
	  SPH_C32(0x3a4e0014), SPH_C32(0xf2c60000), SPH_C32(0xaa4e0000),
	  SPH_C32(0xdb1e42a6), SPH_C32(0x256bbe15), SPH_C32(0x123db156),
	  SPH_C32(0x3a4e99d7) },
	{ SPH_C32(0xe3430000), SPH_C32(0x3a4e0014), SPH_C32(0xf2c60000),
	  SPH_C32(0xaa4e0000), SPH_C32(0xdb1e42a6), SPH_C32(0x256bbe15),
	  SPH_C32(0x123db156), SPH_C32(0x3a4e99d7), SPH_C32(0xf75a0000),
	  SPH_C32(0x19840028), SPH_C32(0xa2190000), SPH_C32(0xeef80000),
	  SPH_C32(0xc0722516), SPH_C32(0x19981260), SPH_C32(0x73dba1e6),
	  SPH_C32(0xe1844257) },
	{ SPH_C32(0x54500000), SPH_C32(0x0671005c), SPH_C32(0x25ae0000),
	  SPH_C32(0x6a1e0000), SPH_C32(0x2ea54edf), SPH_C32(0x664e8512),
	  SPH_C32(0xbfba18c3), SPH_C32(0x7e715d17), SPH_C32(0xbc8d0000),
	  SPH_C32(0xfc3b0018), SPH_C32(0x19830000), SPH_C32(0xd10b0000),
	  SPH_C32(0xae1878c4), SPH_C32(0x42a69856), SPH_C32(0x0012da37),
	  SPH_C32(0x2c3b504e) },
	{ SPH_C32(0xbc8d0000), SPH_C32(0xfc3b0018), SPH_C32(0x19830000),
	  SPH_C32(0xd10b0000), SPH_C32(0xae1878c4), SPH_C32(0x42a69856),
	  SPH_C32(0x0012da37), SPH_C32(0x2c3b504e), SPH_C32(0xe8dd0000),
	  SPH_C32(0xfa4a0044), SPH_C32(0x3c2d0000), SPH_C32(0xbb150000),
	  SPH_C32(0x80bd361b), SPH_C32(0x24e81d44), SPH_C32(0xbfa8c2f4),
	  SPH_C32(0x524a0d59) },
	{ SPH_C32(0x69510000), SPH_C32(0xd4e1009c), SPH_C32(0xc3230000),
	  SPH_C32(0xac2f0000), SPH_C32(0xe4950bae), SPH_C32(0xcea415dc),
	  SPH_C32(0x87ec287c), SPH_C32(0xbce1a3ce), SPH_C32(0xc6730000),
	  SPH_C32(0xaf8d000c), SPH_C32(0xa4c10000), SPH_C32(0x218d0000),
	  SPH_C32(0x23111587), SPH_C32(0x7913512f), SPH_C32(0x1d28ac88),
	  SPH_C32(0x378dd173) },
	{ SPH_C32(0xc6730000), SPH_C32(0xaf8d000c), SPH_C32(0xa4c10000),
	  SPH_C32(0x218d0000), SPH_C32(0x23111587), SPH_C32(0x7913512f),
	  SPH_C32(0x1d28ac88), SPH_C32(0x378dd173), SPH_C32(0xaf220000),
	  SPH_C32(0x7b6c0090), SPH_C32(0x67e20000), SPH_C32(0x8da20000),
	  SPH_C32(0xc7841e29), SPH_C32(0xb7b744f3), SPH_C32(0x9ac484f4),
	  SPH_C32(0x8b6c72bd) },
	{ SPH_C32(0xcc140000), SPH_C32(0xa5630000), SPH_C32(0x5ab90780),
	  SPH_C32(0x3b500000), SPH_C32(0x4bd013ff), SPH_C32(0x879b3418),
	  SPH_C32(0x694348c1), SPH_C32(0xca5a87fe), SPH_C32(0x819e0000),
	  SPH_C32(0xec570000), SPH_C32(0x66320280), SPH_C32(0x95f30000),
	  SPH_C32(0x5da92802), SPH_C32(0x48f43cbc), SPH_C32(0xe65aa22d),
	  SPH_C32(0x8e67b7fa) },
	{ SPH_C32(0x819e0000), SPH_C32(0xec570000), SPH_C32(0x66320280),
	  SPH_C32(0x95f30000), SPH_C32(0x5da92802), SPH_C32(0x48f43cbc),
	  SPH_C32(0xe65aa22d), SPH_C32(0x8e67b7fa), SPH_C32(0x4d8a0000),
	  SPH_C32(0x49340000), SPH_C32(0x3c8b0500), SPH_C32(0xaea30000),
	  SPH_C32(0x16793bfd), SPH_C32(0xcf6f08a4), SPH_C32(0x8f19eaec),
	  SPH_C32(0x443d3004) },
	{ SPH_C32(0x78230000), SPH_C32(0x12fc0000), SPH_C32(0xa93a0b80),
	  SPH_C32(0x90a50000), SPH_C32(0x713e2879), SPH_C32(0x7ee98924),
	  SPH_C32(0xf08ca062), SPH_C32(0x636f8bab), SPH_C32(0x02af0000),
	  SPH_C32(0xb7280000), SPH_C32(0xba1c0300), SPH_C32(0x56980000),
	  SPH_C32(0xba8d45d3), SPH_C32(0x8048c667), SPH_C32(0xa95c149a),
	  SPH_C32(0xf4f6ea7b) },
	{ SPH_C32(0x02af0000), SPH_C32(0xb7280000), SPH_C32(0xba1c0300),
	  SPH_C32(0x56980000), SPH_C32(0xba8d45d3), SPH_C32(0x8048c667),
	  SPH_C32(0xa95c149a), SPH_C32(0xf4f6ea7b), SPH_C32(0x7a8c0000),
	  SPH_C32(0xa5d40000), SPH_C32(0x13260880), SPH_C32(0xc63d0000),
	  SPH_C32(0xcbb36daa), SPH_C32(0xfea14f43), SPH_C32(0x59d0b4f8),
	  SPH_C32(0x979961d0) },
	{ SPH_C32(0xac480000), SPH_C32(0x1ba60000), SPH_C32(0x45fb1380),
	  SPH_C32(0x03430000), SPH_C32(0x5a85316a), SPH_C32(0x1fb250b6),
	  SPH_C32(0xfe72c7fe), SPH_C32(0x91e478f6), SPH_C32(0x1e4e0000),
	  SPH_C32(0xdecf0000), SPH_C32(0x6df80180), SPH_C32(0x77240000),
	  SPH_C32(0xec47079e), SPH_C32(0xf4a0694e), SPH_C32(0xcda31812),
	  SPH_C32(0x98aa496e) },
	{ SPH_C32(0x1e4e0000), SPH_C32(0xdecf0000), SPH_C32(0x6df80180),
	  SPH_C32(0x77240000), SPH_C32(0xec47079e), SPH_C32(0xf4a0694e),
	  SPH_C32(0xcda31812), SPH_C32(0x98aa496e), SPH_C32(0xb2060000),
	  SPH_C32(0xc5690000), SPH_C32(0x28031200), SPH_C32(0x74670000),
	  SPH_C32(0xb6c236f4), SPH_C32(0xeb1239f8), SPH_C32(0x33d1dfec),
	  SPH_C32(0x094e3198) },
	{ SPH_C32(0xaec30000), SPH_C32(0x9c4f0001), SPH_C32(0x79d1e000),
	  SPH_C32(0x2c150000), SPH_C32(0x45cc75b3), SPH_C32(0x6650b736),
	  SPH_C32(0xab92f78f), SPH_C32(0xa312567b), SPH_C32(0xdb250000),
	  SPH_C32(0x09290000), SPH_C32(0x49aac000), SPH_C32(0x81e10000),
	  SPH_C32(0xcafe6b59), SPH_C32(0x42793431), SPH_C32(0x43566b76),
	  SPH_C32(0xe86cba2e) },
	{ SPH_C32(0xdb250000), SPH_C32(0x09290000), SPH_C32(0x49aac000),
	  SPH_C32(0x81e10000), SPH_C32(0xcafe6b59), SPH_C32(0x42793431),
	  SPH_C32(0x43566b76), SPH_C32(0xe86cba2e), SPH_C32(0x75e60000),
	  SPH_C32(0x95660001), SPH_C32(0x307b2000), SPH_C32(0xadf40000),
	  SPH_C32(0x8f321eea), SPH_C32(0x24298307), SPH_C32(0xe8c49cf9),
	  SPH_C32(0x4b7eec55) },
	{ SPH_C32(0x58430000), SPH_C32(0x807e0000), SPH_C32(0x78330001),
	  SPH_C32(0xc66b3800), SPH_C32(0xe7375cdc), SPH_C32(0x79ad3fdd),
	  SPH_C32(0xac73fe6f), SPH_C32(0x3a4479b1), SPH_C32(0x1d5a0000),
	  SPH_C32(0x2b720000), SPH_C32(0x488d0000), SPH_C32(0xaf611800),
	  SPH_C32(0x25cb2ec5), SPH_C32(0xc879bfd0), SPH_C32(0x81a20429),
	  SPH_C32(0x1e7536a6) },
	{ SPH_C32(0x1d5a0000), SPH_C32(0x2b720000), SPH_C32(0x488d0000),
	  SPH_C32(0xaf611800), SPH_C32(0x25cb2ec5), SPH_C32(0xc879bfd0),
	  SPH_C32(0x81a20429), SPH_C32(0x1e7536a6), SPH_C32(0x45190000),
	  SPH_C32(0xab0c0000), SPH_C32(0x30be0001), SPH_C32(0x690a2000),
	  SPH_C32(0xc2fc7219), SPH_C32(0xb1d4800d), SPH_C32(0x2dd1fa46),
	  SPH_C32(0x24314f17) },
	{ SPH_C32(0xa53b0000), SPH_C32(0x14260000), SPH_C32(0x4e30001e),
	  SPH_C32(0x7cae0000), SPH_C32(0x8f9e0dd5), SPH_C32(0x78dfaa3d),
	  SPH_C32(0xf73168d8), SPH_C32(0x0b1b4946), SPH_C32(0x07ed0000),
	  SPH_C32(0xb2500000), SPH_C32(0x8774000a), SPH_C32(0x970d0000),
	  SPH_C32(0x437223ae), SPH_C32(0x48c76ea4), SPH_C32(0xf4786222),
	  SPH_C32(0x9075b1ce) },
	{ SPH_C32(0x07ed0000), SPH_C32(0xb2500000), SPH_C32(0x8774000a),
	  SPH_C32(0x970d0000), SPH_C32(0x437223ae), SPH_C32(0x48c76ea4),
	  SPH_C32(0xf4786222), SPH_C32(0x9075b1ce), SPH_C32(0xa2d60000),
	  SPH_C32(0xa6760000), SPH_C32(0xc9440014), SPH_C32(0xeba30000),
	  SPH_C32(0xccec2e7b), SPH_C32(0x3018c499), SPH_C32(0x03490afa),
	  SPH_C32(0x9b6ef888) },
	{ SPH_C32(0x88980000), SPH_C32(0x1f940000), SPH_C32(0x7fcf002e),
	  SPH_C32(0xfb4e0000), SPH_C32(0xf158079a), SPH_C32(0x61ae9167),
	  SPH_C32(0xa895706c), SPH_C32(0xe6107494), SPH_C32(0x0bc20000),
	  SPH_C32(0xdb630000), SPH_C32(0x7e88000c), SPH_C32(0x15860000),
	  SPH_C32(0x91fd48f3), SPH_C32(0x7581bb43), SPH_C32(0xf460449e),
	  SPH_C32(0xd8b61463) },
	{ SPH_C32(0x0bc20000), SPH_C32(0xdb630000), SPH_C32(0x7e88000c),
	  SPH_C32(0x15860000), SPH_C32(0x91fd48f3), SPH_C32(0x7581bb43),
	  SPH_C32(0xf460449e), SPH_C32(0xd8b61463), SPH_C32(0x835a0000),
	  SPH_C32(0xc4f70000), SPH_C32(0x01470022), SPH_C32(0xeec80000),
	  SPH_C32(0x60a54f69), SPH_C32(0x142f2a24), SPH_C32(0x5cf534f2),
	  SPH_C32(0x3ea660f7) },
	{ SPH_C32(0x52500000), SPH_C32(0x29540000), SPH_C32(0x6a61004e),
	  SPH_C32(0xf0ff0000), SPH_C32(0x9a317eec), SPH_C32(0x452341ce),
	  SPH_C32(0xcf568fe5), SPH_C32(0x5303130f), SPH_C32(0x538d0000),
	  SPH_C32(0xa9fc0000), SPH_C32(0x9ef70006), SPH_C32(0x56ff0000),
	  SPH_C32(0x0ae4004e), SPH_C32(0x92c5cdf9), SPH_C32(0xa9444018),
	  SPH_C32(0x7f975691) },
	{ SPH_C32(0x538d0000), SPH_C32(0xa9fc0000), SPH_C32(0x9ef70006),
	  SPH_C32(0x56ff0000), SPH_C32(0x0ae4004e), SPH_C32(0x92c5cdf9),
	  SPH_C32(0xa9444018), SPH_C32(0x7f975691), SPH_C32(0x01dd0000),
	  SPH_C32(0x80a80000), SPH_C32(0xf4960048), SPH_C32(0xa6000000),
	  SPH_C32(0x90d57ea2), SPH_C32(0xd7e68c37), SPH_C32(0x6612cffd),
	  SPH_C32(0x2c94459e) },
	{ SPH_C32(0xe6280000), SPH_C32(0x4c4b0000), SPH_C32(0xa8550000),
	  SPH_C32(0xd3d002e0), SPH_C32(0xd86130b8), SPH_C32(0x98a7b0da),
	  SPH_C32(0x289506b4), SPH_C32(0xd75a4897), SPH_C32(0xf0c50000),
	  SPH_C32(0x59230000), SPH_C32(0x45820000), SPH_C32(0xe18d00c0),
	  SPH_C32(0x3b6d0631), SPH_C32(0xc2ed5699), SPH_C32(0xcbe0fe1c),
	  SPH_C32(0x56a7b19f) },
	{ SPH_C32(0xf0c50000), SPH_C32(0x59230000), SPH_C32(0x45820000),
	  SPH_C32(0xe18d00c0), SPH_C32(0x3b6d0631), SPH_C32(0xc2ed5699),
	  SPH_C32(0xcbe0fe1c), SPH_C32(0x56a7b19f), SPH_C32(0x16ed0000),
	  SPH_C32(0x15680000), SPH_C32(0xedd70000), SPH_C32(0x325d0220),
	  SPH_C32(0xe30c3689), SPH_C32(0x5a4ae643), SPH_C32(0xe375f8a8),
	  SPH_C32(0x81fdf908) },
	{ SPH_C32(0xb4310000), SPH_C32(0x77330000), SPH_C32(0xb15d0000),
	  SPH_C32(0x7fd004e0), SPH_C32(0x78a26138), SPH_C32(0xd116c35d),
	  SPH_C32(0xd256d489), SPH_C32(0x4e6f74de), SPH_C32(0xe3060000),
	  SPH_C32(0xbdc10000), SPH_C32(0x87130000), SPH_C32(0xbff20060),
	  SPH_C32(0x2eba0a1a), SPH_C32(0x8db53751), SPH_C32(0x73c5ab06),
	  SPH_C32(0x5bd61539) },
	{ SPH_C32(0xe3060000), SPH_C32(0xbdc10000), SPH_C32(0x87130000),
	  SPH_C32(0xbff20060), SPH_C32(0x2eba0a1a), SPH_C32(0x8db53751),
	  SPH_C32(0x73c5ab06), SPH_C32(0x5bd61539), SPH_C32(0x57370000),
	  SPH_C32(0xcaf20000), SPH_C32(0x364e0000), SPH_C32(0xc0220480),
	  SPH_C32(0x56186b22), SPH_C32(0x5ca3f40c), SPH_C32(0xa1937f8f),
	  SPH_C32(0x15b961e7) },
	{ SPH_C32(0x02f20000), SPH_C32(0xa2810000), SPH_C32(0x873f0000),
	  SPH_C32(0xe36c7800), SPH_C32(0x1e1d74ef), SPH_C32(0x073d2bd6),
	  SPH_C32(0xc4c23237), SPH_C32(0x7f32259e), SPH_C32(0xbadd0000),
	  SPH_C32(0x13ad0000), SPH_C32(0xb7e70000), SPH_C32(0xf7282800),
	  SPH_C32(0xdf45144d), SPH_C32(0x361ac33a), SPH_C32(0xea5a8d14),
	  SPH_C32(0x2a2c18f0) },
	{ SPH_C32(0xbadd0000), SPH_C32(0x13ad0000), SPH_C32(0xb7e70000),
	  SPH_C32(0xf7282800), SPH_C32(0xdf45144d), SPH_C32(0x361ac33a),
	  SPH_C32(0xea5a8d14), SPH_C32(0x2a2c18f0), SPH_C32(0xb82f0000),
	  SPH_C32(0xb12c0000), SPH_C32(0x30d80000), SPH_C32(0x14445000),
	  SPH_C32(0xc15860a2), SPH_C32(0x3127e8ec), SPH_C32(0x2e98bf23),
	  SPH_C32(0x551e3d6e) },
	{ SPH_C32(0x1e6c0000), SPH_C32(0xc4420000), SPH_C32(0x8a2e0000),
	  SPH_C32(0xbcb6b800), SPH_C32(0x2c4413b6), SPH_C32(0x8bfdd3da),
	  SPH_C32(0x6a0c1bc8), SPH_C32(0xb99dc2eb), SPH_C32(0x92560000),
	  SPH_C32(0x1eda0000), SPH_C32(0xea510000), SPH_C32(0xe8b13000),
	  SPH_C32(0xa93556a5), SPH_C32(0xebfb6199), SPH_C32(0xb15c2254),
	  SPH_C32(0x33c5244f) },
	{ SPH_C32(0x92560000), SPH_C32(0x1eda0000), SPH_C32(0xea510000),
	  SPH_C32(0xe8b13000), SPH_C32(0xa93556a5), SPH_C32(0xebfb6199),
	  SPH_C32(0xb15c2254), SPH_C32(0x33c5244f), SPH_C32(0x8c3a0000),
	  SPH_C32(0xda980000), SPH_C32(0x607f0000), SPH_C32(0x54078800),
	  SPH_C32(0x85714513), SPH_C32(0x6006b243), SPH_C32(0xdb50399c),
	  SPH_C32(0x8a58e6a4) },
	{ SPH_C32(0x033d0000), SPH_C32(0x08b30000), SPH_C32(0xf33a0000),
	  SPH_C32(0x3ac20007), SPH_C32(0x51298a50), SPH_C32(0x6b6e661f),
	  SPH_C32(0x0ea5cfe3), SPH_C32(0xe6da7ffe), SPH_C32(0xa8da0000),
	  SPH_C32(0x96be0000), SPH_C32(0x5c1d0000), SPH_C32(0x07da0002),
	  SPH_C32(0x7d669583), SPH_C32(0x1f98708a), SPH_C32(0xbb668808),
	  SPH_C32(0xda878000) },
	{ SPH_C32(0xa8da0000), SPH_C32(0x96be0000), SPH_C32(0x5c1d0000),
	  SPH_C32(0x07da0002), SPH_C32(0x7d669583), SPH_C32(0x1f98708a),
	  SPH_C32(0xbb668808), SPH_C32(0xda878000), SPH_C32(0xabe70000),
	  SPH_C32(0x9e0d0000), SPH_C32(0xaf270000), SPH_C32(0x3d180005),
	  SPH_C32(0x2c4f1fd3), SPH_C32(0x74f61695), SPH_C32(0xb5c347eb),
	  SPH_C32(0x3c5dfffe) },
	{ SPH_C32(0x01930000), SPH_C32(0xe7820000), SPH_C32(0xedfb0000),
	  SPH_C32(0xcf0c000b), SPH_C32(0x8dd08d58), SPH_C32(0xbca3b42e),
	  SPH_C32(0x063661e1), SPH_C32(0x536f9e7b), SPH_C32(0x92280000),
	  SPH_C32(0xdc850000), SPH_C32(0x57fa0000), SPH_C32(0x56dc0003),
	  SPH_C32(0xbae92316), SPH_C32(0x5aefa30c), SPH_C32(0x90cef752),
	  SPH_C32(0x7b1675d7) },
	{ SPH_C32(0x92280000), SPH_C32(0xdc850000), SPH_C32(0x57fa0000),
	  SPH_C32(0x56dc0003), SPH_C32(0xbae92316), SPH_C32(0x5aefa30c),
	  SPH_C32(0x90cef752), SPH_C32(0x7b1675d7), SPH_C32(0x93bb0000),
	  SPH_C32(0x3b070000), SPH_C32(0xba010000), SPH_C32(0x99d00008),
	  SPH_C32(0x3739ae4e), SPH_C32(0xe64c1722), SPH_C32(0x96f896b3),
	  SPH_C32(0x2879ebac) },
	{ SPH_C32(0x5fa80000), SPH_C32(0x56030000), SPH_C32(0x43ae0000),
	  SPH_C32(0x64f30013), SPH_C32(0x257e86bf), SPH_C32(0x1311944e),
	  SPH_C32(0x541e95bf), SPH_C32(0x8ea4db69), SPH_C32(0x00440000),
	  SPH_C32(0x7f480000), SPH_C32(0xda7c0000), SPH_C32(0x2a230001),
	  SPH_C32(0x3badc9cc), SPH_C32(0xa9b69c87), SPH_C32(0x030a9e60),
	  SPH_C32(0xbe0a679e) },
	{ SPH_C32(0x00440000), SPH_C32(0x7f480000), SPH_C32(0xda7c0000),
	  SPH_C32(0x2a230001), SPH_C32(0x3badc9cc), SPH_C32(0xa9b69c87),
	  SPH_C32(0x030a9e60), SPH_C32(0xbe0a679e), SPH_C32(0x5fec0000),
	  SPH_C32(0x294b0000), SPH_C32(0x99d20000), SPH_C32(0x4ed00012),
	  SPH_C32(0x1ed34f73), SPH_C32(0xbaa708c9), SPH_C32(0x57140bdf),
	  SPH_C32(0x30aebcf7) },
	{ SPH_C32(0xee930000), SPH_C32(0xd6070000), SPH_C32(0x92c10000),
	  SPH_C32(0x2b9801e0), SPH_C32(0x9451287c), SPH_C32(0x3b6cfb57),
	  SPH_C32(0x45312374), SPH_C32(0x201f6a64), SPH_C32(0x7b280000),
	  SPH_C32(0x57420000), SPH_C32(0xa9e50000), SPH_C32(0x634300a0),
	  SPH_C32(0x9edb442f), SPH_C32(0x6d9995bb), SPH_C32(0x27f83b03),
	  SPH_C32(0xc7ff60f0) },
	{ SPH_C32(0x7b280000), SPH_C32(0x57420000), SPH_C32(0xa9e50000),
	  SPH_C32(0x634300a0), SPH_C32(0x9edb442f), SPH_C32(0x6d9995bb),
	  SPH_C32(0x27f83b03), SPH_C32(0xc7ff60f0), SPH_C32(0x95bb0000),
	  SPH_C32(0x81450000), SPH_C32(0x3b240000), SPH_C32(0x48db0140),
	  SPH_C32(0x0a8a6c53), SPH_C32(0x56f56eec), SPH_C32(0x62c91877),
	  SPH_C32(0xe7e00a94) }
};

#define INPUT_BIG   do { \
		const sph_u32 *tp = &T512[0][0]; \
		unsigned u, v; \
		m0 = 0; \
		m1 = 0; \
		m2 = 0; \
		m3 = 0; \
		m4 = 0; \
		m5 = 0; \
		m6 = 0; \
		m7 = 0; \
		m8 = 0; \
		m9 = 0; \
		mA = 0; \
		mB = 0; \
		mC = 0; \
		mD = 0; \
		mE = 0; \
		mF = 0; \
		for (u = 0; u < 8; u ++) { \
			unsigned db = buf[u]; \
			for (v = 0; v < 8; v ++, db >>= 1) { \
				sph_u32 dm = SPH_T32(-(sph_u32)(db & 1)); \
				m0 ^= dm & *tp ++; \
				m1 ^= dm & *tp ++; \
				m2 ^= dm & *tp ++; \
				m3 ^= dm & *tp ++; \
				m4 ^= dm & *tp ++; \
				m5 ^= dm & *tp ++; \
				m6 ^= dm & *tp ++; \
				m7 ^= dm & *tp ++; \
				m8 ^= dm & *tp ++; \
				m9 ^= dm & *tp ++; \
				mA ^= dm & *tp ++; \
				mB ^= dm & *tp ++; \
				mC ^= dm & *tp ++; \
				mD ^= dm & *tp ++; \
				mE ^= dm & *tp ++; \
				mF ^= dm & *tp ++; \
			} \
		} \
	} while (0)

#endif

#if SPH_HAMSI_EXPAND_BIG == 2

static const sph_u32 T512_0[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x29449c00), SPH_C32(0x64e70000), SPH_C32(0xf24b0000),
	  SPH_C32(0xc2f30000), SPH_C32(0x0ede4e8f), SPH_C32(0x56c23745),
	  SPH_C32(0xf3e04259), SPH_C32(0x8d0d9ec4), SPH_C32(0x466d0c00),
	  SPH_C32(0x08620000), SPH_C32(0xdd5d0000), SPH_C32(0xbadd0000),
	  SPH_C32(0x6a927942), SPH_C32(0x441f2b93), SPH_C32(0x218ace6f),
	  SPH_C32(0xbf2c0be2) },
	{ SPH_C32(0x466d0c00), SPH_C32(0x08620000), SPH_C32(0xdd5d0000),
	  SPH_C32(0xbadd0000), SPH_C32(0x6a927942), SPH_C32(0x441f2b93),
	  SPH_C32(0x218ace6f), SPH_C32(0xbf2c0be2), SPH_C32(0x6f299000),
	  SPH_C32(0x6c850000), SPH_C32(0x2f160000), SPH_C32(0x782e0000),
	  SPH_C32(0x644c37cd), SPH_C32(0x12dd1cd6), SPH_C32(0xd26a8c36),
	  SPH_C32(0x32219526) },
	{ SPH_C32(0x6f299000), SPH_C32(0x6c850000), SPH_C32(0x2f160000),
	  SPH_C32(0x782e0000), SPH_C32(0x644c37cd), SPH_C32(0x12dd1cd6),
	  SPH_C32(0xd26a8c36), SPH_C32(0x32219526), SPH_C32(0x29449c00),
	  SPH_C32(0x64e70000), SPH_C32(0xf24b0000), SPH_C32(0xc2f30000),
	  SPH_C32(0x0ede4e8f), SPH_C32(0x56c23745), SPH_C32(0xf3e04259),
	  SPH_C32(0x8d0d9ec4) }
};

static const sph_u32 T512_2[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x54285c00), SPH_C32(0xeaed0000), SPH_C32(0xc5d60000),
	  SPH_C32(0xa1c50000), SPH_C32(0xb3a26770), SPH_C32(0x94a5c4e1),
	  SPH_C32(0x6bb0419d), SPH_C32(0x551b3782), SPH_C32(0x9cbb1800),
	  SPH_C32(0xb0d30000), SPH_C32(0x92510000), SPH_C32(0xed930000),
	  SPH_C32(0x593a4345), SPH_C32(0xe114d5f4), SPH_C32(0x430633da),
	  SPH_C32(0x78cace29) },
	{ SPH_C32(0x9cbb1800), SPH_C32(0xb0d30000), SPH_C32(0x92510000),
	  SPH_C32(0xed930000), SPH_C32(0x593a4345), SPH_C32(0xe114d5f4),
	  SPH_C32(0x430633da), SPH_C32(0x78cace29), SPH_C32(0xc8934400),
	  SPH_C32(0x5a3e0000), SPH_C32(0x57870000), SPH_C32(0x4c560000),
	  SPH_C32(0xea982435), SPH_C32(0x75b11115), SPH_C32(0x28b67247),
	  SPH_C32(0x2dd1f9ab) },
	{ SPH_C32(0xc8934400), SPH_C32(0x5a3e0000), SPH_C32(0x57870000),
	  SPH_C32(0x4c560000), SPH_C32(0xea982435), SPH_C32(0x75b11115),
	  SPH_C32(0x28b67247), SPH_C32(0x2dd1f9ab), SPH_C32(0x54285c00),
	  SPH_C32(0xeaed0000), SPH_C32(0xc5d60000), SPH_C32(0xa1c50000),
	  SPH_C32(0xb3a26770), SPH_C32(0x94a5c4e1), SPH_C32(0x6bb0419d),
	  SPH_C32(0x551b3782) }
};

static const sph_u32 T512_4[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x145a3c00), SPH_C32(0xb9e90000), SPH_C32(0x61270000),
	  SPH_C32(0xf1610000), SPH_C32(0xce613d6c), SPH_C32(0xb0493d78),
	  SPH_C32(0x47a96720), SPH_C32(0xe18e24c5), SPH_C32(0x23671400),
	  SPH_C32(0xc8b90000), SPH_C32(0xf4c70000), SPH_C32(0xfb750000),
	  SPH_C32(0x73cd2465), SPH_C32(0xf8a6a549), SPH_C32(0x02c40a3f),
	  SPH_C32(0xdc24e61f) },
	{ SPH_C32(0x23671400), SPH_C32(0xc8b90000), SPH_C32(0xf4c70000),
	  SPH_C32(0xfb750000), SPH_C32(0x73cd2465), SPH_C32(0xf8a6a549),
	  SPH_C32(0x02c40a3f), SPH_C32(0xdc24e61f), SPH_C32(0x373d2800),
	  SPH_C32(0x71500000), SPH_C32(0x95e00000), SPH_C32(0x0a140000),
	  SPH_C32(0xbdac1909), SPH_C32(0x48ef9831), SPH_C32(0x456d6d1f),
	  SPH_C32(0x3daac2da) },
	{ SPH_C32(0x373d2800), SPH_C32(0x71500000), SPH_C32(0x95e00000),
	  SPH_C32(0x0a140000), SPH_C32(0xbdac1909), SPH_C32(0x48ef9831),
	  SPH_C32(0x456d6d1f), SPH_C32(0x3daac2da), SPH_C32(0x145a3c00),
	  SPH_C32(0xb9e90000), SPH_C32(0x61270000), SPH_C32(0xf1610000),
	  SPH_C32(0xce613d6c), SPH_C32(0xb0493d78), SPH_C32(0x47a96720),
	  SPH_C32(0xe18e24c5) }
};

static const sph_u32 T512_6[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xef0b0270), SPH_C32(0x3afd0000), SPH_C32(0x5dae0000),
	  SPH_C32(0x69490000), SPH_C32(0x9b0f3c06), SPH_C32(0x4405b5f9),
	  SPH_C32(0x66140a51), SPH_C32(0x924f5d0a), SPH_C32(0xc96b0030),
	  SPH_C32(0xe7250000), SPH_C32(0x2f840000), SPH_C32(0x264f0000),
	  SPH_C32(0x08695bf9), SPH_C32(0x6dfcf137), SPH_C32(0x509f6984),
	  SPH_C32(0x9e69af68) },
	{ SPH_C32(0xc96b0030), SPH_C32(0xe7250000), SPH_C32(0x2f840000),
	  SPH_C32(0x264f0000), SPH_C32(0x08695bf9), SPH_C32(0x6dfcf137),
	  SPH_C32(0x509f6984), SPH_C32(0x9e69af68), SPH_C32(0x26600240),
	  SPH_C32(0xddd80000), SPH_C32(0x722a0000), SPH_C32(0x4f060000),
	  SPH_C32(0x936667ff), SPH_C32(0x29f944ce), SPH_C32(0x368b63d5),
	  SPH_C32(0x0c26f262) },
	{ SPH_C32(0x26600240), SPH_C32(0xddd80000), SPH_C32(0x722a0000),
	  SPH_C32(0x4f060000), SPH_C32(0x936667ff), SPH_C32(0x29f944ce),
	  SPH_C32(0x368b63d5), SPH_C32(0x0c26f262), SPH_C32(0xef0b0270),
	  SPH_C32(0x3afd0000), SPH_C32(0x5dae0000), SPH_C32(0x69490000),
	  SPH_C32(0x9b0f3c06), SPH_C32(0x4405b5f9), SPH_C32(0x66140a51),
	  SPH_C32(0x924f5d0a) }
};

static const sph_u32 T512_8[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xe8870170), SPH_C32(0x9d720000), SPH_C32(0x12db0000),
	  SPH_C32(0xd4220000), SPH_C32(0xf2886b27), SPH_C32(0xa921e543),
	  SPH_C32(0x4ef8b518), SPH_C32(0x618813b1), SPH_C32(0xb4370060),
	  SPH_C32(0x0c4c0000), SPH_C32(0x56c20000), SPH_C32(0x5cae0000),
	  SPH_C32(0x94541f3f), SPH_C32(0x3b3ef825), SPH_C32(0x1b365f3d),
	  SPH_C32(0xf3d45758) },
	{ SPH_C32(0xb4370060), SPH_C32(0x0c4c0000), SPH_C32(0x56c20000),
	  SPH_C32(0x5cae0000), SPH_C32(0x94541f3f), SPH_C32(0x3b3ef825),
	  SPH_C32(0x1b365f3d), SPH_C32(0xf3d45758), SPH_C32(0x5cb00110),
	  SPH_C32(0x913e0000), SPH_C32(0x44190000), SPH_C32(0x888c0000),
	  SPH_C32(0x66dc7418), SPH_C32(0x921f1d66), SPH_C32(0x55ceea25),
	  SPH_C32(0x925c44e9) },
	{ SPH_C32(0x5cb00110), SPH_C32(0x913e0000), SPH_C32(0x44190000),
	  SPH_C32(0x888c0000), SPH_C32(0x66dc7418), SPH_C32(0x921f1d66),
	  SPH_C32(0x55ceea25), SPH_C32(0x925c44e9), SPH_C32(0xe8870170),
	  SPH_C32(0x9d720000), SPH_C32(0x12db0000), SPH_C32(0xd4220000),
	  SPH_C32(0xf2886b27), SPH_C32(0xa921e543), SPH_C32(0x4ef8b518),
	  SPH_C32(0x618813b1) }
};

static const sph_u32 T512_10[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x774400f0), SPH_C32(0xf15a0000), SPH_C32(0xf5b20000),
	  SPH_C32(0x34140000), SPH_C32(0x89377e8c), SPH_C32(0x5a8bec25),
	  SPH_C32(0x0bc3cd1e), SPH_C32(0xcf3775cb), SPH_C32(0xf46c0050),
	  SPH_C32(0x96180000), SPH_C32(0x14a50000), SPH_C32(0x031f0000),
	  SPH_C32(0x42947eb8), SPH_C32(0x66bf7e19), SPH_C32(0x9ca470d2),
	  SPH_C32(0x8a341574) },
	{ SPH_C32(0xf46c0050), SPH_C32(0x96180000), SPH_C32(0x14a50000),
	  SPH_C32(0x031f0000), SPH_C32(0x42947eb8), SPH_C32(0x66bf7e19),
	  SPH_C32(0x9ca470d2), SPH_C32(0x8a341574), SPH_C32(0x832800a0),
	  SPH_C32(0x67420000), SPH_C32(0xe1170000), SPH_C32(0x370b0000),
	  SPH_C32(0xcba30034), SPH_C32(0x3c34923c), SPH_C32(0x9767bdcc),
	  SPH_C32(0x450360bf) },
	{ SPH_C32(0x832800a0), SPH_C32(0x67420000), SPH_C32(0xe1170000),
	  SPH_C32(0x370b0000), SPH_C32(0xcba30034), SPH_C32(0x3c34923c),
	  SPH_C32(0x9767bdcc), SPH_C32(0x450360bf), SPH_C32(0x774400f0),
	  SPH_C32(0xf15a0000), SPH_C32(0xf5b20000), SPH_C32(0x34140000),
	  SPH_C32(0x89377e8c), SPH_C32(0x5a8bec25), SPH_C32(0x0bc3cd1e),
	  SPH_C32(0xcf3775cb) }
};

static const sph_u32 T512_12[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xf7750009), SPH_C32(0xcf3cc000), SPH_C32(0xc3d60000),
	  SPH_C32(0x04920000), SPH_C32(0x029519a9), SPH_C32(0xf8e836ba),
	  SPH_C32(0x7a87f14e), SPH_C32(0x9e16981a), SPH_C32(0xd46a0000),
	  SPH_C32(0x8dc8c000), SPH_C32(0xa5af0000), SPH_C32(0x4a290000),
	  SPH_C32(0xfc4e427a), SPH_C32(0xc9b4866c), SPH_C32(0x98369604),
	  SPH_C32(0xf746c320) },
	{ SPH_C32(0xd46a0000), SPH_C32(0x8dc8c000), SPH_C32(0xa5af0000),
	  SPH_C32(0x4a290000), SPH_C32(0xfc4e427a), SPH_C32(0xc9b4866c),
	  SPH_C32(0x98369604), SPH_C32(0xf746c320), SPH_C32(0x231f0009),
	  SPH_C32(0x42f40000), SPH_C32(0x66790000), SPH_C32(0x4ebb0000),
	  SPH_C32(0xfedb5bd3), SPH_C32(0x315cb0d6), SPH_C32(0xe2b1674a),
	  SPH_C32(0x69505b3a) },
	{ SPH_C32(0x231f0009), SPH_C32(0x42f40000), SPH_C32(0x66790000),
	  SPH_C32(0x4ebb0000), SPH_C32(0xfedb5bd3), SPH_C32(0x315cb0d6),
	  SPH_C32(0xe2b1674a), SPH_C32(0x69505b3a), SPH_C32(0xf7750009),
	  SPH_C32(0xcf3cc000), SPH_C32(0xc3d60000), SPH_C32(0x04920000),
	  SPH_C32(0x029519a9), SPH_C32(0xf8e836ba), SPH_C32(0x7a87f14e),
	  SPH_C32(0x9e16981a) }
};

static const sph_u32 T512_14[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xf6800005), SPH_C32(0x3443c000), SPH_C32(0x24070000),
	  SPH_C32(0x8f3d0000), SPH_C32(0x21373bfb), SPH_C32(0x0ab8d5ae),
	  SPH_C32(0xcdc58b19), SPH_C32(0xd795ba31), SPH_C32(0xa67f0001),
	  SPH_C32(0x71378000), SPH_C32(0x19fc0000), SPH_C32(0x96db0000),
	  SPH_C32(0x3a8b6dfd), SPH_C32(0xebcaaef3), SPH_C32(0x2c6d478f),
	  SPH_C32(0xac8e6c88) },
	{ SPH_C32(0xa67f0001), SPH_C32(0x71378000), SPH_C32(0x19fc0000),
	  SPH_C32(0x96db0000), SPH_C32(0x3a8b6dfd), SPH_C32(0xebcaaef3),
	  SPH_C32(0x2c6d478f), SPH_C32(0xac8e6c88), SPH_C32(0x50ff0004),
	  SPH_C32(0x45744000), SPH_C32(0x3dfb0000), SPH_C32(0x19e60000),
	  SPH_C32(0x1bbc5606), SPH_C32(0xe1727b5d), SPH_C32(0xe1a8cc96),
	  SPH_C32(0x7b1bd6b9) },
	{ SPH_C32(0x50ff0004), SPH_C32(0x45744000), SPH_C32(0x3dfb0000),
	  SPH_C32(0x19e60000), SPH_C32(0x1bbc5606), SPH_C32(0xe1727b5d),
	  SPH_C32(0xe1a8cc96), SPH_C32(0x7b1bd6b9), SPH_C32(0xf6800005),
	  SPH_C32(0x3443c000), SPH_C32(0x24070000), SPH_C32(0x8f3d0000),
	  SPH_C32(0x21373bfb), SPH_C32(0x0ab8d5ae), SPH_C32(0xcdc58b19),
	  SPH_C32(0xd795ba31) }
};

static const sph_u32 T512_16[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x75c90003), SPH_C32(0x0e10c000), SPH_C32(0xd1200000),
	  SPH_C32(0xbaea0000), SPH_C32(0x8bc42f3e), SPH_C32(0x8758b757),
	  SPH_C32(0xbb28761d), SPH_C32(0x00b72e2b), SPH_C32(0xeecf0001),
	  SPH_C32(0x6f564000), SPH_C32(0xf33e0000), SPH_C32(0xa79e0000),
	  SPH_C32(0xbdb57219), SPH_C32(0xb711ebc5), SPH_C32(0x4a3b40ba),
	  SPH_C32(0xfeabf254) },
	{ SPH_C32(0xeecf0001), SPH_C32(0x6f564000), SPH_C32(0xf33e0000),
	  SPH_C32(0xa79e0000), SPH_C32(0xbdb57219), SPH_C32(0xb711ebc5),
	  SPH_C32(0x4a3b40ba), SPH_C32(0xfeabf254), SPH_C32(0x9b060002),
	  SPH_C32(0x61468000), SPH_C32(0x221e0000), SPH_C32(0x1d740000),
	  SPH_C32(0x36715d27), SPH_C32(0x30495c92), SPH_C32(0xf11336a7),
	  SPH_C32(0xfe1cdc7f) },
	{ SPH_C32(0x9b060002), SPH_C32(0x61468000), SPH_C32(0x221e0000),
	  SPH_C32(0x1d740000), SPH_C32(0x36715d27), SPH_C32(0x30495c92),
	  SPH_C32(0xf11336a7), SPH_C32(0xfe1cdc7f), SPH_C32(0x75c90003),
	  SPH_C32(0x0e10c000), SPH_C32(0xd1200000), SPH_C32(0xbaea0000),
	  SPH_C32(0x8bc42f3e), SPH_C32(0x8758b757), SPH_C32(0xbb28761d),
	  SPH_C32(0x00b72e2b) }
};

static const sph_u32 T512_18[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x75a40000), SPH_C32(0xc28b2700), SPH_C32(0x94a40000),
	  SPH_C32(0x90f50000), SPH_C32(0xfb7857e0), SPH_C32(0x49ce0bae),
	  SPH_C32(0x1767c483), SPH_C32(0xaedf667e), SPH_C32(0xd1660000),
	  SPH_C32(0x1bbc0300), SPH_C32(0x9eec0000), SPH_C32(0xf6940000),
	  SPH_C32(0x03024527), SPH_C32(0xcf70fcf2), SPH_C32(0xb4431b17),
	  SPH_C32(0x857f3c2b) },
	{ SPH_C32(0xd1660000), SPH_C32(0x1bbc0300), SPH_C32(0x9eec0000),
	  SPH_C32(0xf6940000), SPH_C32(0x03024527), SPH_C32(0xcf70fcf2),
	  SPH_C32(0xb4431b17), SPH_C32(0x857f3c2b), SPH_C32(0xa4c20000),
	  SPH_C32(0xd9372400), SPH_C32(0x0a480000), SPH_C32(0x66610000),
	  SPH_C32(0xf87a12c7), SPH_C32(0x86bef75c), SPH_C32(0xa324df94),
	  SPH_C32(0x2ba05a55) },
	{ SPH_C32(0xa4c20000), SPH_C32(0xd9372400), SPH_C32(0x0a480000),
	  SPH_C32(0x66610000), SPH_C32(0xf87a12c7), SPH_C32(0x86bef75c),
	  SPH_C32(0xa324df94), SPH_C32(0x2ba05a55), SPH_C32(0x75a40000),
	  SPH_C32(0xc28b2700), SPH_C32(0x94a40000), SPH_C32(0x90f50000),
	  SPH_C32(0xfb7857e0), SPH_C32(0x49ce0bae), SPH_C32(0x1767c483),
	  SPH_C32(0xaedf667e) }
};

static const sph_u32 T512_20[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x45180000), SPH_C32(0xa5b51700), SPH_C32(0xf96a0000),
	  SPH_C32(0x3b480000), SPH_C32(0x1ecc142c), SPH_C32(0x231395d6),
	  SPH_C32(0x16bca6b0), SPH_C32(0xdf33f4df), SPH_C32(0xb83d0000),
	  SPH_C32(0x16710600), SPH_C32(0x379a0000), SPH_C32(0xf5b10000),
	  SPH_C32(0x228161ac), SPH_C32(0xae48f145), SPH_C32(0x66241616),
	  SPH_C32(0xc5c1eb3e) },
	{ SPH_C32(0xb83d0000), SPH_C32(0x16710600), SPH_C32(0x379a0000),
	  SPH_C32(0xf5b10000), SPH_C32(0x228161ac), SPH_C32(0xae48f145),
	  SPH_C32(0x66241616), SPH_C32(0xc5c1eb3e), SPH_C32(0xfd250000),
	  SPH_C32(0xb3c41100), SPH_C32(0xcef00000), SPH_C32(0xcef90000),
	  SPH_C32(0x3c4d7580), SPH_C32(0x8d5b6493), SPH_C32(0x7098b0a6),
	  SPH_C32(0x1af21fe1) },
	{ SPH_C32(0xfd250000), SPH_C32(0xb3c41100), SPH_C32(0xcef00000),
	  SPH_C32(0xcef90000), SPH_C32(0x3c4d7580), SPH_C32(0x8d5b6493),
	  SPH_C32(0x7098b0a6), SPH_C32(0x1af21fe1), SPH_C32(0x45180000),
	  SPH_C32(0xa5b51700), SPH_C32(0xf96a0000), SPH_C32(0x3b480000),
	  SPH_C32(0x1ecc142c), SPH_C32(0x231395d6), SPH_C32(0x16bca6b0),
	  SPH_C32(0xdf33f4df) }
};

static const sph_u32 T512_22[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x0c720000), SPH_C32(0x49e50f00), SPH_C32(0x42790000),
	  SPH_C32(0x5cea0000), SPH_C32(0x33aa301a), SPH_C32(0x15822514),
	  SPH_C32(0x95a34b7b), SPH_C32(0xb44b0090), SPH_C32(0xfe220000),
	  SPH_C32(0xa7580500), SPH_C32(0x25d10000), SPH_C32(0xf7600000),
	  SPH_C32(0x893178da), SPH_C32(0x1fd4f860), SPH_C32(0x4ed0a315),
	  SPH_C32(0xa123ff9f) },
	{ SPH_C32(0xfe220000), SPH_C32(0xa7580500), SPH_C32(0x25d10000),
	  SPH_C32(0xf7600000), SPH_C32(0x893178da), SPH_C32(0x1fd4f860),
	  SPH_C32(0x4ed0a315), SPH_C32(0xa123ff9f), SPH_C32(0xf2500000),
	  SPH_C32(0xeebd0a00), SPH_C32(0x67a80000), SPH_C32(0xab8a0000),
	  SPH_C32(0xba9b48c0), SPH_C32(0x0a56dd74), SPH_C32(0xdb73e86e),
	  SPH_C32(0x1568ff0f) },
	{ SPH_C32(0xf2500000), SPH_C32(0xeebd0a00), SPH_C32(0x67a80000),
	  SPH_C32(0xab8a0000), SPH_C32(0xba9b48c0), SPH_C32(0x0a56dd74),
	  SPH_C32(0xdb73e86e), SPH_C32(0x1568ff0f), SPH_C32(0x0c720000),
	  SPH_C32(0x49e50f00), SPH_C32(0x42790000), SPH_C32(0x5cea0000),
	  SPH_C32(0x33aa301a), SPH_C32(0x15822514), SPH_C32(0x95a34b7b),
	  SPH_C32(0xb44b0090) }
};

static const sph_u32 T512_24[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x69510000), SPH_C32(0xd4e1009c), SPH_C32(0xc3230000),
	  SPH_C32(0xac2f0000), SPH_C32(0xe4950bae), SPH_C32(0xcea415dc),
	  SPH_C32(0x87ec287c), SPH_C32(0xbce1a3ce), SPH_C32(0xc6730000),
	  SPH_C32(0xaf8d000c), SPH_C32(0xa4c10000), SPH_C32(0x218d0000),
	  SPH_C32(0x23111587), SPH_C32(0x7913512f), SPH_C32(0x1d28ac88),
	  SPH_C32(0x378dd173) },
	{ SPH_C32(0xc6730000), SPH_C32(0xaf8d000c), SPH_C32(0xa4c10000),
	  SPH_C32(0x218d0000), SPH_C32(0x23111587), SPH_C32(0x7913512f),
	  SPH_C32(0x1d28ac88), SPH_C32(0x378dd173), SPH_C32(0xaf220000),
	  SPH_C32(0x7b6c0090), SPH_C32(0x67e20000), SPH_C32(0x8da20000),
	  SPH_C32(0xc7841e29), SPH_C32(0xb7b744f3), SPH_C32(0x9ac484f4),
	  SPH_C32(0x8b6c72bd) },
	{ SPH_C32(0xaf220000), SPH_C32(0x7b6c0090), SPH_C32(0x67e20000),
	  SPH_C32(0x8da20000), SPH_C32(0xc7841e29), SPH_C32(0xb7b744f3),
	  SPH_C32(0x9ac484f4), SPH_C32(0x8b6c72bd), SPH_C32(0x69510000),
	  SPH_C32(0xd4e1009c), SPH_C32(0xc3230000), SPH_C32(0xac2f0000),
	  SPH_C32(0xe4950bae), SPH_C32(0xcea415dc), SPH_C32(0x87ec287c),
	  SPH_C32(0xbce1a3ce) }
};

static const sph_u32 T512_26[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x54500000), SPH_C32(0x0671005c), SPH_C32(0x25ae0000),
	  SPH_C32(0x6a1e0000), SPH_C32(0x2ea54edf), SPH_C32(0x664e8512),
	  SPH_C32(0xbfba18c3), SPH_C32(0x7e715d17), SPH_C32(0xbc8d0000),
	  SPH_C32(0xfc3b0018), SPH_C32(0x19830000), SPH_C32(0xd10b0000),
	  SPH_C32(0xae1878c4), SPH_C32(0x42a69856), SPH_C32(0x0012da37),
	  SPH_C32(0x2c3b504e) },
	{ SPH_C32(0xbc8d0000), SPH_C32(0xfc3b0018), SPH_C32(0x19830000),
	  SPH_C32(0xd10b0000), SPH_C32(0xae1878c4), SPH_C32(0x42a69856),
	  SPH_C32(0x0012da37), SPH_C32(0x2c3b504e), SPH_C32(0xe8dd0000),
	  SPH_C32(0xfa4a0044), SPH_C32(0x3c2d0000), SPH_C32(0xbb150000),
	  SPH_C32(0x80bd361b), SPH_C32(0x24e81d44), SPH_C32(0xbfa8c2f4),
	  SPH_C32(0x524a0d59) },
	{ SPH_C32(0xe8dd0000), SPH_C32(0xfa4a0044), SPH_C32(0x3c2d0000),
	  SPH_C32(0xbb150000), SPH_C32(0x80bd361b), SPH_C32(0x24e81d44),
	  SPH_C32(0xbfa8c2f4), SPH_C32(0x524a0d59), SPH_C32(0x54500000),
	  SPH_C32(0x0671005c), SPH_C32(0x25ae0000), SPH_C32(0x6a1e0000),
	  SPH_C32(0x2ea54edf), SPH_C32(0x664e8512), SPH_C32(0xbfba18c3),
	  SPH_C32(0x7e715d17) }
};

static const sph_u32 T512_28[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x14190000), SPH_C32(0x23ca003c), SPH_C32(0x50df0000),
	  SPH_C32(0x44b60000), SPH_C32(0x1b6c67b0), SPH_C32(0x3cf3ac75),
	  SPH_C32(0x61e610b0), SPH_C32(0xdbcadb80), SPH_C32(0xe3430000),
	  SPH_C32(0x3a4e0014), SPH_C32(0xf2c60000), SPH_C32(0xaa4e0000),
	  SPH_C32(0xdb1e42a6), SPH_C32(0x256bbe15), SPH_C32(0x123db156),
	  SPH_C32(0x3a4e99d7) },
	{ SPH_C32(0xe3430000), SPH_C32(0x3a4e0014), SPH_C32(0xf2c60000),
	  SPH_C32(0xaa4e0000), SPH_C32(0xdb1e42a6), SPH_C32(0x256bbe15),
	  SPH_C32(0x123db156), SPH_C32(0x3a4e99d7), SPH_C32(0xf75a0000),
	  SPH_C32(0x19840028), SPH_C32(0xa2190000), SPH_C32(0xeef80000),
	  SPH_C32(0xc0722516), SPH_C32(0x19981260), SPH_C32(0x73dba1e6),
	  SPH_C32(0xe1844257) },
	{ SPH_C32(0xf75a0000), SPH_C32(0x19840028), SPH_C32(0xa2190000),
	  SPH_C32(0xeef80000), SPH_C32(0xc0722516), SPH_C32(0x19981260),
	  SPH_C32(0x73dba1e6), SPH_C32(0xe1844257), SPH_C32(0x14190000),
	  SPH_C32(0x23ca003c), SPH_C32(0x50df0000), SPH_C32(0x44b60000),
	  SPH_C32(0x1b6c67b0), SPH_C32(0x3cf3ac75), SPH_C32(0x61e610b0),
	  SPH_C32(0xdbcadb80) }
};

static const sph_u32 T512_30[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x86790000), SPH_C32(0x3f390002), SPH_C32(0xe19ae000),
	  SPH_C32(0x98560000), SPH_C32(0x9565670e), SPH_C32(0x4e88c8ea),
	  SPH_C32(0xd3dd4944), SPH_C32(0x161ddab9), SPH_C32(0x30b70000),
	  SPH_C32(0xe5d00000), SPH_C32(0xf4f46000), SPH_C32(0x42c40000),
	  SPH_C32(0x63b83d6a), SPH_C32(0x78ba9460), SPH_C32(0x21afa1ea),
	  SPH_C32(0xb0a51834) },
	{ SPH_C32(0x30b70000), SPH_C32(0xe5d00000), SPH_C32(0xf4f46000),
	  SPH_C32(0x42c40000), SPH_C32(0x63b83d6a), SPH_C32(0x78ba9460),
	  SPH_C32(0x21afa1ea), SPH_C32(0xb0a51834), SPH_C32(0xb6ce0000),
	  SPH_C32(0xdae90002), SPH_C32(0x156e8000), SPH_C32(0xda920000),
	  SPH_C32(0xf6dd5a64), SPH_C32(0x36325c8a), SPH_C32(0xf272e8ae),
	  SPH_C32(0xa6b8c28d) },
	{ SPH_C32(0xb6ce0000), SPH_C32(0xdae90002), SPH_C32(0x156e8000),
	  SPH_C32(0xda920000), SPH_C32(0xf6dd5a64), SPH_C32(0x36325c8a),
	  SPH_C32(0xf272e8ae), SPH_C32(0xa6b8c28d), SPH_C32(0x86790000),
	  SPH_C32(0x3f390002), SPH_C32(0xe19ae000), SPH_C32(0x98560000),
	  SPH_C32(0x9565670e), SPH_C32(0x4e88c8ea), SPH_C32(0xd3dd4944),
	  SPH_C32(0x161ddab9) }
};

static const sph_u32 T512_32[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xaec30000), SPH_C32(0x9c4f0001), SPH_C32(0x79d1e000),
	  SPH_C32(0x2c150000), SPH_C32(0x45cc75b3), SPH_C32(0x6650b736),
	  SPH_C32(0xab92f78f), SPH_C32(0xa312567b), SPH_C32(0xdb250000),
	  SPH_C32(0x09290000), SPH_C32(0x49aac000), SPH_C32(0x81e10000),
	  SPH_C32(0xcafe6b59), SPH_C32(0x42793431), SPH_C32(0x43566b76),
	  SPH_C32(0xe86cba2e) },
	{ SPH_C32(0xdb250000), SPH_C32(0x09290000), SPH_C32(0x49aac000),
	  SPH_C32(0x81e10000), SPH_C32(0xcafe6b59), SPH_C32(0x42793431),
	  SPH_C32(0x43566b76), SPH_C32(0xe86cba2e), SPH_C32(0x75e60000),
	  SPH_C32(0x95660001), SPH_C32(0x307b2000), SPH_C32(0xadf40000),
	  SPH_C32(0x8f321eea), SPH_C32(0x24298307), SPH_C32(0xe8c49cf9),
	  SPH_C32(0x4b7eec55) },
	{ SPH_C32(0x75e60000), SPH_C32(0x95660001), SPH_C32(0x307b2000),
	  SPH_C32(0xadf40000), SPH_C32(0x8f321eea), SPH_C32(0x24298307),
	  SPH_C32(0xe8c49cf9), SPH_C32(0x4b7eec55), SPH_C32(0xaec30000),
	  SPH_C32(0x9c4f0001), SPH_C32(0x79d1e000), SPH_C32(0x2c150000),
	  SPH_C32(0x45cc75b3), SPH_C32(0x6650b736), SPH_C32(0xab92f78f),
	  SPH_C32(0xa312567b) }
};

static const sph_u32 T512_34[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xac480000), SPH_C32(0x1ba60000), SPH_C32(0x45fb1380),
	  SPH_C32(0x03430000), SPH_C32(0x5a85316a), SPH_C32(0x1fb250b6),
	  SPH_C32(0xfe72c7fe), SPH_C32(0x91e478f6), SPH_C32(0x1e4e0000),
	  SPH_C32(0xdecf0000), SPH_C32(0x6df80180), SPH_C32(0x77240000),
	  SPH_C32(0xec47079e), SPH_C32(0xf4a0694e), SPH_C32(0xcda31812),
	  SPH_C32(0x98aa496e) },
	{ SPH_C32(0x1e4e0000), SPH_C32(0xdecf0000), SPH_C32(0x6df80180),
	  SPH_C32(0x77240000), SPH_C32(0xec47079e), SPH_C32(0xf4a0694e),
	  SPH_C32(0xcda31812), SPH_C32(0x98aa496e), SPH_C32(0xb2060000),
	  SPH_C32(0xc5690000), SPH_C32(0x28031200), SPH_C32(0x74670000),
	  SPH_C32(0xb6c236f4), SPH_C32(0xeb1239f8), SPH_C32(0x33d1dfec),
	  SPH_C32(0x094e3198) },
	{ SPH_C32(0xb2060000), SPH_C32(0xc5690000), SPH_C32(0x28031200),
	  SPH_C32(0x74670000), SPH_C32(0xb6c236f4), SPH_C32(0xeb1239f8),
	  SPH_C32(0x33d1dfec), SPH_C32(0x094e3198), SPH_C32(0xac480000),
	  SPH_C32(0x1ba60000), SPH_C32(0x45fb1380), SPH_C32(0x03430000),
	  SPH_C32(0x5a85316a), SPH_C32(0x1fb250b6), SPH_C32(0xfe72c7fe),
	  SPH_C32(0x91e478f6) }
};

static const sph_u32 T512_36[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x78230000), SPH_C32(0x12fc0000), SPH_C32(0xa93a0b80),
	  SPH_C32(0x90a50000), SPH_C32(0x713e2879), SPH_C32(0x7ee98924),
	  SPH_C32(0xf08ca062), SPH_C32(0x636f8bab), SPH_C32(0x02af0000),
	  SPH_C32(0xb7280000), SPH_C32(0xba1c0300), SPH_C32(0x56980000),
	  SPH_C32(0xba8d45d3), SPH_C32(0x8048c667), SPH_C32(0xa95c149a),
	  SPH_C32(0xf4f6ea7b) },
	{ SPH_C32(0x02af0000), SPH_C32(0xb7280000), SPH_C32(0xba1c0300),
	  SPH_C32(0x56980000), SPH_C32(0xba8d45d3), SPH_C32(0x8048c667),
	  SPH_C32(0xa95c149a), SPH_C32(0xf4f6ea7b), SPH_C32(0x7a8c0000),
	  SPH_C32(0xa5d40000), SPH_C32(0x13260880), SPH_C32(0xc63d0000),
	  SPH_C32(0xcbb36daa), SPH_C32(0xfea14f43), SPH_C32(0x59d0b4f8),
	  SPH_C32(0x979961d0) },
	{ SPH_C32(0x7a8c0000), SPH_C32(0xa5d40000), SPH_C32(0x13260880),
	  SPH_C32(0xc63d0000), SPH_C32(0xcbb36daa), SPH_C32(0xfea14f43),
	  SPH_C32(0x59d0b4f8), SPH_C32(0x979961d0), SPH_C32(0x78230000),
	  SPH_C32(0x12fc0000), SPH_C32(0xa93a0b80), SPH_C32(0x90a50000),
	  SPH_C32(0x713e2879), SPH_C32(0x7ee98924), SPH_C32(0xf08ca062),
	  SPH_C32(0x636f8bab) }
};

static const sph_u32 T512_38[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xcc140000), SPH_C32(0xa5630000), SPH_C32(0x5ab90780),
	  SPH_C32(0x3b500000), SPH_C32(0x4bd013ff), SPH_C32(0x879b3418),
	  SPH_C32(0x694348c1), SPH_C32(0xca5a87fe), SPH_C32(0x819e0000),
	  SPH_C32(0xec570000), SPH_C32(0x66320280), SPH_C32(0x95f30000),
	  SPH_C32(0x5da92802), SPH_C32(0x48f43cbc), SPH_C32(0xe65aa22d),
	  SPH_C32(0x8e67b7fa) },
	{ SPH_C32(0x819e0000), SPH_C32(0xec570000), SPH_C32(0x66320280),
	  SPH_C32(0x95f30000), SPH_C32(0x5da92802), SPH_C32(0x48f43cbc),
	  SPH_C32(0xe65aa22d), SPH_C32(0x8e67b7fa), SPH_C32(0x4d8a0000),
	  SPH_C32(0x49340000), SPH_C32(0x3c8b0500), SPH_C32(0xaea30000),
	  SPH_C32(0x16793bfd), SPH_C32(0xcf6f08a4), SPH_C32(0x8f19eaec),
	  SPH_C32(0x443d3004) },
	{ SPH_C32(0x4d8a0000), SPH_C32(0x49340000), SPH_C32(0x3c8b0500),
	  SPH_C32(0xaea30000), SPH_C32(0x16793bfd), SPH_C32(0xcf6f08a4),
	  SPH_C32(0x8f19eaec), SPH_C32(0x443d3004), SPH_C32(0xcc140000),
	  SPH_C32(0xa5630000), SPH_C32(0x5ab90780), SPH_C32(0x3b500000),
	  SPH_C32(0x4bd013ff), SPH_C32(0x879b3418), SPH_C32(0x694348c1),
	  SPH_C32(0xca5a87fe) }
};

static const sph_u32 T512_40[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x52500000), SPH_C32(0x29540000), SPH_C32(0x6a61004e),
	  SPH_C32(0xf0ff0000), SPH_C32(0x9a317eec), SPH_C32(0x452341ce),
	  SPH_C32(0xcf568fe5), SPH_C32(0x5303130f), SPH_C32(0x538d0000),
	  SPH_C32(0xa9fc0000), SPH_C32(0x9ef70006), SPH_C32(0x56ff0000),
	  SPH_C32(0x0ae4004e), SPH_C32(0x92c5cdf9), SPH_C32(0xa9444018),
	  SPH_C32(0x7f975691) },
	{ SPH_C32(0x538d0000), SPH_C32(0xa9fc0000), SPH_C32(0x9ef70006),
	  SPH_C32(0x56ff0000), SPH_C32(0x0ae4004e), SPH_C32(0x92c5cdf9),
	  SPH_C32(0xa9444018), SPH_C32(0x7f975691), SPH_C32(0x01dd0000),
	  SPH_C32(0x80a80000), SPH_C32(0xf4960048), SPH_C32(0xa6000000),
	  SPH_C32(0x90d57ea2), SPH_C32(0xd7e68c37), SPH_C32(0x6612cffd),
	  SPH_C32(0x2c94459e) },
	{ SPH_C32(0x01dd0000), SPH_C32(0x80a80000), SPH_C32(0xf4960048),
	  SPH_C32(0xa6000000), SPH_C32(0x90d57ea2), SPH_C32(0xd7e68c37),
	  SPH_C32(0x6612cffd), SPH_C32(0x2c94459e), SPH_C32(0x52500000),
	  SPH_C32(0x29540000), SPH_C32(0x6a61004e), SPH_C32(0xf0ff0000),
	  SPH_C32(0x9a317eec), SPH_C32(0x452341ce), SPH_C32(0xcf568fe5),
	  SPH_C32(0x5303130f) }
};

static const sph_u32 T512_42[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x88980000), SPH_C32(0x1f940000), SPH_C32(0x7fcf002e),
	  SPH_C32(0xfb4e0000), SPH_C32(0xf158079a), SPH_C32(0x61ae9167),
	  SPH_C32(0xa895706c), SPH_C32(0xe6107494), SPH_C32(0x0bc20000),
	  SPH_C32(0xdb630000), SPH_C32(0x7e88000c), SPH_C32(0x15860000),
	  SPH_C32(0x91fd48f3), SPH_C32(0x7581bb43), SPH_C32(0xf460449e),
	  SPH_C32(0xd8b61463) },
	{ SPH_C32(0x0bc20000), SPH_C32(0xdb630000), SPH_C32(0x7e88000c),
	  SPH_C32(0x15860000), SPH_C32(0x91fd48f3), SPH_C32(0x7581bb43),
	  SPH_C32(0xf460449e), SPH_C32(0xd8b61463), SPH_C32(0x835a0000),
	  SPH_C32(0xc4f70000), SPH_C32(0x01470022), SPH_C32(0xeec80000),
	  SPH_C32(0x60a54f69), SPH_C32(0x142f2a24), SPH_C32(0x5cf534f2),
	  SPH_C32(0x3ea660f7) },
	{ SPH_C32(0x835a0000), SPH_C32(0xc4f70000), SPH_C32(0x01470022),
	  SPH_C32(0xeec80000), SPH_C32(0x60a54f69), SPH_C32(0x142f2a24),
	  SPH_C32(0x5cf534f2), SPH_C32(0x3ea660f7), SPH_C32(0x88980000),
	  SPH_C32(0x1f940000), SPH_C32(0x7fcf002e), SPH_C32(0xfb4e0000),
	  SPH_C32(0xf158079a), SPH_C32(0x61ae9167), SPH_C32(0xa895706c),
	  SPH_C32(0xe6107494) }
};

static const sph_u32 T512_44[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xa53b0000), SPH_C32(0x14260000), SPH_C32(0x4e30001e),
	  SPH_C32(0x7cae0000), SPH_C32(0x8f9e0dd5), SPH_C32(0x78dfaa3d),
	  SPH_C32(0xf73168d8), SPH_C32(0x0b1b4946), SPH_C32(0x07ed0000),
	  SPH_C32(0xb2500000), SPH_C32(0x8774000a), SPH_C32(0x970d0000),
	  SPH_C32(0x437223ae), SPH_C32(0x48c76ea4), SPH_C32(0xf4786222),
	  SPH_C32(0x9075b1ce) },
	{ SPH_C32(0x07ed0000), SPH_C32(0xb2500000), SPH_C32(0x8774000a),
	  SPH_C32(0x970d0000), SPH_C32(0x437223ae), SPH_C32(0x48c76ea4),
	  SPH_C32(0xf4786222), SPH_C32(0x9075b1ce), SPH_C32(0xa2d60000),
	  SPH_C32(0xa6760000), SPH_C32(0xc9440014), SPH_C32(0xeba30000),
	  SPH_C32(0xccec2e7b), SPH_C32(0x3018c499), SPH_C32(0x03490afa),
	  SPH_C32(0x9b6ef888) },
	{ SPH_C32(0xa2d60000), SPH_C32(0xa6760000), SPH_C32(0xc9440014),
	  SPH_C32(0xeba30000), SPH_C32(0xccec2e7b), SPH_C32(0x3018c499),
	  SPH_C32(0x03490afa), SPH_C32(0x9b6ef888), SPH_C32(0xa53b0000),
	  SPH_C32(0x14260000), SPH_C32(0x4e30001e), SPH_C32(0x7cae0000),
	  SPH_C32(0x8f9e0dd5), SPH_C32(0x78dfaa3d), SPH_C32(0xf73168d8),
	  SPH_C32(0x0b1b4946) }
};

static const sph_u32 T512_46[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x58430000), SPH_C32(0x807e0000), SPH_C32(0x78330001),
	  SPH_C32(0xc66b3800), SPH_C32(0xe7375cdc), SPH_C32(0x79ad3fdd),
	  SPH_C32(0xac73fe6f), SPH_C32(0x3a4479b1), SPH_C32(0x1d5a0000),
	  SPH_C32(0x2b720000), SPH_C32(0x488d0000), SPH_C32(0xaf611800),
	  SPH_C32(0x25cb2ec5), SPH_C32(0xc879bfd0), SPH_C32(0x81a20429),
	  SPH_C32(0x1e7536a6) },
	{ SPH_C32(0x1d5a0000), SPH_C32(0x2b720000), SPH_C32(0x488d0000),
	  SPH_C32(0xaf611800), SPH_C32(0x25cb2ec5), SPH_C32(0xc879bfd0),
	  SPH_C32(0x81a20429), SPH_C32(0x1e7536a6), SPH_C32(0x45190000),
	  SPH_C32(0xab0c0000), SPH_C32(0x30be0001), SPH_C32(0x690a2000),
	  SPH_C32(0xc2fc7219), SPH_C32(0xb1d4800d), SPH_C32(0x2dd1fa46),
	  SPH_C32(0x24314f17) },
	{ SPH_C32(0x45190000), SPH_C32(0xab0c0000), SPH_C32(0x30be0001),
	  SPH_C32(0x690a2000), SPH_C32(0xc2fc7219), SPH_C32(0xb1d4800d),
	  SPH_C32(0x2dd1fa46), SPH_C32(0x24314f17), SPH_C32(0x58430000),
	  SPH_C32(0x807e0000), SPH_C32(0x78330001), SPH_C32(0xc66b3800),
	  SPH_C32(0xe7375cdc), SPH_C32(0x79ad3fdd), SPH_C32(0xac73fe6f),
	  SPH_C32(0x3a4479b1) }
};

static const sph_u32 T512_48[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x1e6c0000), SPH_C32(0xc4420000), SPH_C32(0x8a2e0000),
	  SPH_C32(0xbcb6b800), SPH_C32(0x2c4413b6), SPH_C32(0x8bfdd3da),
	  SPH_C32(0x6a0c1bc8), SPH_C32(0xb99dc2eb), SPH_C32(0x92560000),
	  SPH_C32(0x1eda0000), SPH_C32(0xea510000), SPH_C32(0xe8b13000),
	  SPH_C32(0xa93556a5), SPH_C32(0xebfb6199), SPH_C32(0xb15c2254),
	  SPH_C32(0x33c5244f) },
	{ SPH_C32(0x92560000), SPH_C32(0x1eda0000), SPH_C32(0xea510000),
	  SPH_C32(0xe8b13000), SPH_C32(0xa93556a5), SPH_C32(0xebfb6199),
	  SPH_C32(0xb15c2254), SPH_C32(0x33c5244f), SPH_C32(0x8c3a0000),
	  SPH_C32(0xda980000), SPH_C32(0x607f0000), SPH_C32(0x54078800),
	  SPH_C32(0x85714513), SPH_C32(0x6006b243), SPH_C32(0xdb50399c),
	  SPH_C32(0x8a58e6a4) },
	{ SPH_C32(0x8c3a0000), SPH_C32(0xda980000), SPH_C32(0x607f0000),
	  SPH_C32(0x54078800), SPH_C32(0x85714513), SPH_C32(0x6006b243),
	  SPH_C32(0xdb50399c), SPH_C32(0x8a58e6a4), SPH_C32(0x1e6c0000),
	  SPH_C32(0xc4420000), SPH_C32(0x8a2e0000), SPH_C32(0xbcb6b800),
	  SPH_C32(0x2c4413b6), SPH_C32(0x8bfdd3da), SPH_C32(0x6a0c1bc8),
	  SPH_C32(0xb99dc2eb) }
};

static const sph_u32 T512_50[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x02f20000), SPH_C32(0xa2810000), SPH_C32(0x873f0000),
	  SPH_C32(0xe36c7800), SPH_C32(0x1e1d74ef), SPH_C32(0x073d2bd6),
	  SPH_C32(0xc4c23237), SPH_C32(0x7f32259e), SPH_C32(0xbadd0000),
	  SPH_C32(0x13ad0000), SPH_C32(0xb7e70000), SPH_C32(0xf7282800),
	  SPH_C32(0xdf45144d), SPH_C32(0x361ac33a), SPH_C32(0xea5a8d14),
	  SPH_C32(0x2a2c18f0) },
	{ SPH_C32(0xbadd0000), SPH_C32(0x13ad0000), SPH_C32(0xb7e70000),
	  SPH_C32(0xf7282800), SPH_C32(0xdf45144d), SPH_C32(0x361ac33a),
	  SPH_C32(0xea5a8d14), SPH_C32(0x2a2c18f0), SPH_C32(0xb82f0000),
	  SPH_C32(0xb12c0000), SPH_C32(0x30d80000), SPH_C32(0x14445000),
	  SPH_C32(0xc15860a2), SPH_C32(0x3127e8ec), SPH_C32(0x2e98bf23),
	  SPH_C32(0x551e3d6e) },
	{ SPH_C32(0xb82f0000), SPH_C32(0xb12c0000), SPH_C32(0x30d80000),
	  SPH_C32(0x14445000), SPH_C32(0xc15860a2), SPH_C32(0x3127e8ec),
	  SPH_C32(0x2e98bf23), SPH_C32(0x551e3d6e), SPH_C32(0x02f20000),
	  SPH_C32(0xa2810000), SPH_C32(0x873f0000), SPH_C32(0xe36c7800),
	  SPH_C32(0x1e1d74ef), SPH_C32(0x073d2bd6), SPH_C32(0xc4c23237),
	  SPH_C32(0x7f32259e) }
};

static const sph_u32 T512_52[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xb4310000), SPH_C32(0x77330000), SPH_C32(0xb15d0000),
	  SPH_C32(0x7fd004e0), SPH_C32(0x78a26138), SPH_C32(0xd116c35d),
	  SPH_C32(0xd256d489), SPH_C32(0x4e6f74de), SPH_C32(0xe3060000),
	  SPH_C32(0xbdc10000), SPH_C32(0x87130000), SPH_C32(0xbff20060),
	  SPH_C32(0x2eba0a1a), SPH_C32(0x8db53751), SPH_C32(0x73c5ab06),
	  SPH_C32(0x5bd61539) },
	{ SPH_C32(0xe3060000), SPH_C32(0xbdc10000), SPH_C32(0x87130000),
	  SPH_C32(0xbff20060), SPH_C32(0x2eba0a1a), SPH_C32(0x8db53751),
	  SPH_C32(0x73c5ab06), SPH_C32(0x5bd61539), SPH_C32(0x57370000),
	  SPH_C32(0xcaf20000), SPH_C32(0x364e0000), SPH_C32(0xc0220480),
	  SPH_C32(0x56186b22), SPH_C32(0x5ca3f40c), SPH_C32(0xa1937f8f),
	  SPH_C32(0x15b961e7) },
	{ SPH_C32(0x57370000), SPH_C32(0xcaf20000), SPH_C32(0x364e0000),
	  SPH_C32(0xc0220480), SPH_C32(0x56186b22), SPH_C32(0x5ca3f40c),
	  SPH_C32(0xa1937f8f), SPH_C32(0x15b961e7), SPH_C32(0xb4310000),
	  SPH_C32(0x77330000), SPH_C32(0xb15d0000), SPH_C32(0x7fd004e0),
	  SPH_C32(0x78a26138), SPH_C32(0xd116c35d), SPH_C32(0xd256d489),
	  SPH_C32(0x4e6f74de) }
};

static const sph_u32 T512_54[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xe6280000), SPH_C32(0x4c4b0000), SPH_C32(0xa8550000),
	  SPH_C32(0xd3d002e0), SPH_C32(0xd86130b8), SPH_C32(0x98a7b0da),
	  SPH_C32(0x289506b4), SPH_C32(0xd75a4897), SPH_C32(0xf0c50000),
	  SPH_C32(0x59230000), SPH_C32(0x45820000), SPH_C32(0xe18d00c0),
	  SPH_C32(0x3b6d0631), SPH_C32(0xc2ed5699), SPH_C32(0xcbe0fe1c),
	  SPH_C32(0x56a7b19f) },
	{ SPH_C32(0xf0c50000), SPH_C32(0x59230000), SPH_C32(0x45820000),
	  SPH_C32(0xe18d00c0), SPH_C32(0x3b6d0631), SPH_C32(0xc2ed5699),
	  SPH_C32(0xcbe0fe1c), SPH_C32(0x56a7b19f), SPH_C32(0x16ed0000),
	  SPH_C32(0x15680000), SPH_C32(0xedd70000), SPH_C32(0x325d0220),
	  SPH_C32(0xe30c3689), SPH_C32(0x5a4ae643), SPH_C32(0xe375f8a8),
	  SPH_C32(0x81fdf908) },
	{ SPH_C32(0x16ed0000), SPH_C32(0x15680000), SPH_C32(0xedd70000),
	  SPH_C32(0x325d0220), SPH_C32(0xe30c3689), SPH_C32(0x5a4ae643),
	  SPH_C32(0xe375f8a8), SPH_C32(0x81fdf908), SPH_C32(0xe6280000),
	  SPH_C32(0x4c4b0000), SPH_C32(0xa8550000), SPH_C32(0xd3d002e0),
	  SPH_C32(0xd86130b8), SPH_C32(0x98a7b0da), SPH_C32(0x289506b4),
	  SPH_C32(0xd75a4897) }
};

static const sph_u32 T512_56[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xee930000), SPH_C32(0xd6070000), SPH_C32(0x92c10000),
	  SPH_C32(0x2b9801e0), SPH_C32(0x9451287c), SPH_C32(0x3b6cfb57),
	  SPH_C32(0x45312374), SPH_C32(0x201f6a64), SPH_C32(0x7b280000),
	  SPH_C32(0x57420000), SPH_C32(0xa9e50000), SPH_C32(0x634300a0),
	  SPH_C32(0x9edb442f), SPH_C32(0x6d9995bb), SPH_C32(0x27f83b03),
	  SPH_C32(0xc7ff60f0) },
	{ SPH_C32(0x7b280000), SPH_C32(0x57420000), SPH_C32(0xa9e50000),
	  SPH_C32(0x634300a0), SPH_C32(0x9edb442f), SPH_C32(0x6d9995bb),
	  SPH_C32(0x27f83b03), SPH_C32(0xc7ff60f0), SPH_C32(0x95bb0000),
	  SPH_C32(0x81450000), SPH_C32(0x3b240000), SPH_C32(0x48db0140),
	  SPH_C32(0x0a8a6c53), SPH_C32(0x56f56eec), SPH_C32(0x62c91877),
	  SPH_C32(0xe7e00a94) },
	{ SPH_C32(0x95bb0000), SPH_C32(0x81450000), SPH_C32(0x3b240000),
	  SPH_C32(0x48db0140), SPH_C32(0x0a8a6c53), SPH_C32(0x56f56eec),
	  SPH_C32(0x62c91877), SPH_C32(0xe7e00a94), SPH_C32(0xee930000),
	  SPH_C32(0xd6070000), SPH_C32(0x92c10000), SPH_C32(0x2b9801e0),
	  SPH_C32(0x9451287c), SPH_C32(0x3b6cfb57), SPH_C32(0x45312374),
	  SPH_C32(0x201f6a64) }
};

static const sph_u32 T512_58[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x5fa80000), SPH_C32(0x56030000), SPH_C32(0x43ae0000),
	  SPH_C32(0x64f30013), SPH_C32(0x257e86bf), SPH_C32(0x1311944e),
	  SPH_C32(0x541e95bf), SPH_C32(0x8ea4db69), SPH_C32(0x00440000),
	  SPH_C32(0x7f480000), SPH_C32(0xda7c0000), SPH_C32(0x2a230001),
	  SPH_C32(0x3badc9cc), SPH_C32(0xa9b69c87), SPH_C32(0x030a9e60),
	  SPH_C32(0xbe0a679e) },
	{ SPH_C32(0x00440000), SPH_C32(0x7f480000), SPH_C32(0xda7c0000),
	  SPH_C32(0x2a230001), SPH_C32(0x3badc9cc), SPH_C32(0xa9b69c87),
	  SPH_C32(0x030a9e60), SPH_C32(0xbe0a679e), SPH_C32(0x5fec0000),
	  SPH_C32(0x294b0000), SPH_C32(0x99d20000), SPH_C32(0x4ed00012),
	  SPH_C32(0x1ed34f73), SPH_C32(0xbaa708c9), SPH_C32(0x57140bdf),
	  SPH_C32(0x30aebcf7) },
	{ SPH_C32(0x5fec0000), SPH_C32(0x294b0000), SPH_C32(0x99d20000),
	  SPH_C32(0x4ed00012), SPH_C32(0x1ed34f73), SPH_C32(0xbaa708c9),
	  SPH_C32(0x57140bdf), SPH_C32(0x30aebcf7), SPH_C32(0x5fa80000),
	  SPH_C32(0x56030000), SPH_C32(0x43ae0000), SPH_C32(0x64f30013),
	  SPH_C32(0x257e86bf), SPH_C32(0x1311944e), SPH_C32(0x541e95bf),
	  SPH_C32(0x8ea4db69) }
};

static const sph_u32 T512_60[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x01930000), SPH_C32(0xe7820000), SPH_C32(0xedfb0000),
	  SPH_C32(0xcf0c000b), SPH_C32(0x8dd08d58), SPH_C32(0xbca3b42e),
	  SPH_C32(0x063661e1), SPH_C32(0x536f9e7b), SPH_C32(0x92280000),
	  SPH_C32(0xdc850000), SPH_C32(0x57fa0000), SPH_C32(0x56dc0003),
	  SPH_C32(0xbae92316), SPH_C32(0x5aefa30c), SPH_C32(0x90cef752),
	  SPH_C32(0x7b1675d7) },
	{ SPH_C32(0x92280000), SPH_C32(0xdc850000), SPH_C32(0x57fa0000),
	  SPH_C32(0x56dc0003), SPH_C32(0xbae92316), SPH_C32(0x5aefa30c),
	  SPH_C32(0x90cef752), SPH_C32(0x7b1675d7), SPH_C32(0x93bb0000),
	  SPH_C32(0x3b070000), SPH_C32(0xba010000), SPH_C32(0x99d00008),
	  SPH_C32(0x3739ae4e), SPH_C32(0xe64c1722), SPH_C32(0x96f896b3),
	  SPH_C32(0x2879ebac) },
	{ SPH_C32(0x93bb0000), SPH_C32(0x3b070000), SPH_C32(0xba010000),
	  SPH_C32(0x99d00008), SPH_C32(0x3739ae4e), SPH_C32(0xe64c1722),
	  SPH_C32(0x96f896b3), SPH_C32(0x2879ebac), SPH_C32(0x01930000),
	  SPH_C32(0xe7820000), SPH_C32(0xedfb0000), SPH_C32(0xcf0c000b),
	  SPH_C32(0x8dd08d58), SPH_C32(0xbca3b42e), SPH_C32(0x063661e1),
	  SPH_C32(0x536f9e7b) }
};

static const sph_u32 T512_62[4][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x033d0000), SPH_C32(0x08b30000), SPH_C32(0xf33a0000),
	  SPH_C32(0x3ac20007), SPH_C32(0x51298a50), SPH_C32(0x6b6e661f),
	  SPH_C32(0x0ea5cfe3), SPH_C32(0xe6da7ffe), SPH_C32(0xa8da0000),
	  SPH_C32(0x96be0000), SPH_C32(0x5c1d0000), SPH_C32(0x07da0002),
	  SPH_C32(0x7d669583), SPH_C32(0x1f98708a), SPH_C32(0xbb668808),
	  SPH_C32(0xda878000) },
	{ SPH_C32(0xa8da0000), SPH_C32(0x96be0000), SPH_C32(0x5c1d0000),
	  SPH_C32(0x07da0002), SPH_C32(0x7d669583), SPH_C32(0x1f98708a),
	  SPH_C32(0xbb668808), SPH_C32(0xda878000), SPH_C32(0xabe70000),
	  SPH_C32(0x9e0d0000), SPH_C32(0xaf270000), SPH_C32(0x3d180005),
	  SPH_C32(0x2c4f1fd3), SPH_C32(0x74f61695), SPH_C32(0xb5c347eb),
	  SPH_C32(0x3c5dfffe) },
	{ SPH_C32(0xabe70000), SPH_C32(0x9e0d0000), SPH_C32(0xaf270000),
	  SPH_C32(0x3d180005), SPH_C32(0x2c4f1fd3), SPH_C32(0x74f61695),
	  SPH_C32(0xb5c347eb), SPH_C32(0x3c5dfffe), SPH_C32(0x033d0000),
	  SPH_C32(0x08b30000), SPH_C32(0xf33a0000), SPH_C32(0x3ac20007),
	  SPH_C32(0x51298a50), SPH_C32(0x6b6e661f), SPH_C32(0x0ea5cfe3),
	  SPH_C32(0xe6da7ffe) }
};

#define INPUT_BIG   do { \
		unsigned acc = buf[0]; \
		const sph_u32 *rp; \
		rp = &T512_0[acc >> 6][0]; \
		m0 = rp[0]; \
		m1 = rp[1]; \
		m2 = rp[2]; \
		m3 = rp[3]; \
		m4 = rp[4]; \
		m5 = rp[5]; \
		m6 = rp[6]; \
		m7 = rp[7]; \
		m8 = rp[8]; \
		m9 = rp[9]; \
		mA = rp[10]; \
		mB = rp[11]; \
		mC = rp[12]; \
		mD = rp[13]; \
		mE = rp[14]; \
		mF = rp[15]; \
		rp = &T512_2[(acc >> 4) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_4[(acc >> 2) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_6[acc & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = buf[1]; \
		rp = &T512_8[acc >> 6][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_10[(acc >> 4) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_12[(acc >> 2) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_14[acc & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = buf[2]; \
		rp = &T512_16[acc >> 6][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_18[(acc >> 4) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_20[(acc >> 2) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_22[acc & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = buf[3]; \
		rp = &T512_24[acc >> 6][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_26[(acc >> 4) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_28[(acc >> 2) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_30[acc & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = buf[4]; \
		rp = &T512_32[acc >> 6][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_34[(acc >> 4) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_36[(acc >> 2) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_38[acc & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = buf[5]; \
		rp = &T512_40[acc >> 6][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_42[(acc >> 4) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_44[(acc >> 2) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_46[acc & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = buf[6]; \
		rp = &T512_48[acc >> 6][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_50[(acc >> 4) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_52[(acc >> 2) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_54[acc & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = buf[7]; \
		rp = &T512_56[acc >> 6][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_58[(acc >> 4) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_60[(acc >> 2) & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_62[acc & 0x03][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
	} while (0)

#endif

#if SPH_HAMSI_EXPAND_BIG == 3

static const sph_u32 T512_0[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x9cbb1800), SPH_C32(0xb0d30000), SPH_C32(0x92510000),
	  SPH_C32(0xed930000), SPH_C32(0x593a4345), SPH_C32(0xe114d5f4),
	  SPH_C32(0x430633da), SPH_C32(0x78cace29), SPH_C32(0xc8934400),
	  SPH_C32(0x5a3e0000), SPH_C32(0x57870000), SPH_C32(0x4c560000),
	  SPH_C32(0xea982435), SPH_C32(0x75b11115), SPH_C32(0x28b67247),
	  SPH_C32(0x2dd1f9ab) },
	{ SPH_C32(0x29449c00), SPH_C32(0x64e70000), SPH_C32(0xf24b0000),
	  SPH_C32(0xc2f30000), SPH_C32(0x0ede4e8f), SPH_C32(0x56c23745),
	  SPH_C32(0xf3e04259), SPH_C32(0x8d0d9ec4), SPH_C32(0x466d0c00),
	  SPH_C32(0x08620000), SPH_C32(0xdd5d0000), SPH_C32(0xbadd0000),
	  SPH_C32(0x6a927942), SPH_C32(0x441f2b93), SPH_C32(0x218ace6f),
	  SPH_C32(0xbf2c0be2) },
	{ SPH_C32(0xb5ff8400), SPH_C32(0xd4340000), SPH_C32(0x601a0000),
	  SPH_C32(0x2f600000), SPH_C32(0x57e40dca), SPH_C32(0xb7d6e2b1),
	  SPH_C32(0xb0e67183), SPH_C32(0xf5c750ed), SPH_C32(0x8efe4800),
	  SPH_C32(0x525c0000), SPH_C32(0x8ada0000), SPH_C32(0xf68b0000),
	  SPH_C32(0x800a5d77), SPH_C32(0x31ae3a86), SPH_C32(0x093cbc28),
	  SPH_C32(0x92fdf249) },
	{ SPH_C32(0x466d0c00), SPH_C32(0x08620000), SPH_C32(0xdd5d0000),
	  SPH_C32(0xbadd0000), SPH_C32(0x6a927942), SPH_C32(0x441f2b93),
	  SPH_C32(0x218ace6f), SPH_C32(0xbf2c0be2), SPH_C32(0x6f299000),
	  SPH_C32(0x6c850000), SPH_C32(0x2f160000), SPH_C32(0x782e0000),
	  SPH_C32(0x644c37cd), SPH_C32(0x12dd1cd6), SPH_C32(0xd26a8c36),
	  SPH_C32(0x32219526) },
	{ SPH_C32(0xdad61400), SPH_C32(0xb8b10000), SPH_C32(0x4f0c0000),
	  SPH_C32(0x574e0000), SPH_C32(0x33a83a07), SPH_C32(0xa50bfe67),
	  SPH_C32(0x628cfdb5), SPH_C32(0xc7e6c5cb), SPH_C32(0xa7bad400),
	  SPH_C32(0x36bb0000), SPH_C32(0x78910000), SPH_C32(0x34780000),
	  SPH_C32(0x8ed413f8), SPH_C32(0x676c0dc3), SPH_C32(0xfadcfe71),
	  SPH_C32(0x1ff06c8d) },
	{ SPH_C32(0x6f299000), SPH_C32(0x6c850000), SPH_C32(0x2f160000),
	  SPH_C32(0x782e0000), SPH_C32(0x644c37cd), SPH_C32(0x12dd1cd6),
	  SPH_C32(0xd26a8c36), SPH_C32(0x32219526), SPH_C32(0x29449c00),
	  SPH_C32(0x64e70000), SPH_C32(0xf24b0000), SPH_C32(0xc2f30000),
	  SPH_C32(0x0ede4e8f), SPH_C32(0x56c23745), SPH_C32(0xf3e04259),
	  SPH_C32(0x8d0d9ec4) },
	{ SPH_C32(0xf3928800), SPH_C32(0xdc560000), SPH_C32(0xbd470000),
	  SPH_C32(0x95bd0000), SPH_C32(0x3d767488), SPH_C32(0xf3c9c922),
	  SPH_C32(0x916cbfec), SPH_C32(0x4aeb5b0f), SPH_C32(0xe1d7d800),
	  SPH_C32(0x3ed90000), SPH_C32(0xa5cc0000), SPH_C32(0x8ea50000),
	  SPH_C32(0xe4466aba), SPH_C32(0x23732650), SPH_C32(0xdb56301e),
	  SPH_C32(0xa0dc676f) }
};

static const sph_u32 T512_3[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x145a3c00), SPH_C32(0xb9e90000), SPH_C32(0x61270000),
	  SPH_C32(0xf1610000), SPH_C32(0xce613d6c), SPH_C32(0xb0493d78),
	  SPH_C32(0x47a96720), SPH_C32(0xe18e24c5), SPH_C32(0x23671400),
	  SPH_C32(0xc8b90000), SPH_C32(0xf4c70000), SPH_C32(0xfb750000),
	  SPH_C32(0x73cd2465), SPH_C32(0xf8a6a549), SPH_C32(0x02c40a3f),
	  SPH_C32(0xdc24e61f) },
	{ SPH_C32(0x23671400), SPH_C32(0xc8b90000), SPH_C32(0xf4c70000),
	  SPH_C32(0xfb750000), SPH_C32(0x73cd2465), SPH_C32(0xf8a6a549),
	  SPH_C32(0x02c40a3f), SPH_C32(0xdc24e61f), SPH_C32(0x373d2800),
	  SPH_C32(0x71500000), SPH_C32(0x95e00000), SPH_C32(0x0a140000),
	  SPH_C32(0xbdac1909), SPH_C32(0x48ef9831), SPH_C32(0x456d6d1f),
	  SPH_C32(0x3daac2da) },
	{ SPH_C32(0x373d2800), SPH_C32(0x71500000), SPH_C32(0x95e00000),
	  SPH_C32(0x0a140000), SPH_C32(0xbdac1909), SPH_C32(0x48ef9831),
	  SPH_C32(0x456d6d1f), SPH_C32(0x3daac2da), SPH_C32(0x145a3c00),
	  SPH_C32(0xb9e90000), SPH_C32(0x61270000), SPH_C32(0xf1610000),
	  SPH_C32(0xce613d6c), SPH_C32(0xb0493d78), SPH_C32(0x47a96720),
	  SPH_C32(0xe18e24c5) },
	{ SPH_C32(0x54285c00), SPH_C32(0xeaed0000), SPH_C32(0xc5d60000),
	  SPH_C32(0xa1c50000), SPH_C32(0xb3a26770), SPH_C32(0x94a5c4e1),
	  SPH_C32(0x6bb0419d), SPH_C32(0x551b3782), SPH_C32(0x9cbb1800),
	  SPH_C32(0xb0d30000), SPH_C32(0x92510000), SPH_C32(0xed930000),
	  SPH_C32(0x593a4345), SPH_C32(0xe114d5f4), SPH_C32(0x430633da),
	  SPH_C32(0x78cace29) },
	{ SPH_C32(0x40726000), SPH_C32(0x53040000), SPH_C32(0xa4f10000),
	  SPH_C32(0x50a40000), SPH_C32(0x7dc35a1c), SPH_C32(0x24ecf999),
	  SPH_C32(0x2c1926bd), SPH_C32(0xb4951347), SPH_C32(0xbfdc0c00),
	  SPH_C32(0x786a0000), SPH_C32(0x66960000), SPH_C32(0x16e60000),
	  SPH_C32(0x2af76720), SPH_C32(0x19b270bd), SPH_C32(0x41c239e5),
	  SPH_C32(0xa4ee2836) },
	{ SPH_C32(0x774f4800), SPH_C32(0x22540000), SPH_C32(0x31110000),
	  SPH_C32(0x5ab00000), SPH_C32(0xc06f4315), SPH_C32(0x6c0361a8),
	  SPH_C32(0x69744ba2), SPH_C32(0x893fd19d), SPH_C32(0xab863000),
	  SPH_C32(0xc1830000), SPH_C32(0x07b10000), SPH_C32(0xe7870000),
	  SPH_C32(0xe4965a4c), SPH_C32(0xa9fb4dc5), SPH_C32(0x066b5ec5),
	  SPH_C32(0x45600cf3) },
	{ SPH_C32(0x63157400), SPH_C32(0x9bbd0000), SPH_C32(0x50360000),
	  SPH_C32(0xabd10000), SPH_C32(0x0e0e7e79), SPH_C32(0xdc4a5cd0),
	  SPH_C32(0x2edd2c82), SPH_C32(0x68b1f558), SPH_C32(0x88e12400),
	  SPH_C32(0x093a0000), SPH_C32(0xf3760000), SPH_C32(0x1cf20000),
	  SPH_C32(0x975b7e29), SPH_C32(0x515de88c), SPH_C32(0x04af54fa),
	  SPH_C32(0x9944eaec) }
};

static const sph_u32 T512_6[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xb4370060), SPH_C32(0x0c4c0000), SPH_C32(0x56c20000),
	  SPH_C32(0x5cae0000), SPH_C32(0x94541f3f), SPH_C32(0x3b3ef825),
	  SPH_C32(0x1b365f3d), SPH_C32(0xf3d45758), SPH_C32(0x5cb00110),
	  SPH_C32(0x913e0000), SPH_C32(0x44190000), SPH_C32(0x888c0000),
	  SPH_C32(0x66dc7418), SPH_C32(0x921f1d66), SPH_C32(0x55ceea25),
	  SPH_C32(0x925c44e9) },
	{ SPH_C32(0xef0b0270), SPH_C32(0x3afd0000), SPH_C32(0x5dae0000),
	  SPH_C32(0x69490000), SPH_C32(0x9b0f3c06), SPH_C32(0x4405b5f9),
	  SPH_C32(0x66140a51), SPH_C32(0x924f5d0a), SPH_C32(0xc96b0030),
	  SPH_C32(0xe7250000), SPH_C32(0x2f840000), SPH_C32(0x264f0000),
	  SPH_C32(0x08695bf9), SPH_C32(0x6dfcf137), SPH_C32(0x509f6984),
	  SPH_C32(0x9e69af68) },
	{ SPH_C32(0x5b3c0210), SPH_C32(0x36b10000), SPH_C32(0x0b6c0000),
	  SPH_C32(0x35e70000), SPH_C32(0x0f5b2339), SPH_C32(0x7f3b4ddc),
	  SPH_C32(0x7d22556c), SPH_C32(0x619b0a52), SPH_C32(0x95db0120),
	  SPH_C32(0x761b0000), SPH_C32(0x6b9d0000), SPH_C32(0xaec30000),
	  SPH_C32(0x6eb52fe1), SPH_C32(0xffe3ec51), SPH_C32(0x055183a1),
	  SPH_C32(0x0c35eb81) },
	{ SPH_C32(0xc96b0030), SPH_C32(0xe7250000), SPH_C32(0x2f840000),
	  SPH_C32(0x264f0000), SPH_C32(0x08695bf9), SPH_C32(0x6dfcf137),
	  SPH_C32(0x509f6984), SPH_C32(0x9e69af68), SPH_C32(0x26600240),
	  SPH_C32(0xddd80000), SPH_C32(0x722a0000), SPH_C32(0x4f060000),
	  SPH_C32(0x936667ff), SPH_C32(0x29f944ce), SPH_C32(0x368b63d5),
	  SPH_C32(0x0c26f262) },
	{ SPH_C32(0x7d5c0050), SPH_C32(0xeb690000), SPH_C32(0x79460000),
	  SPH_C32(0x7ae10000), SPH_C32(0x9c3d44c6), SPH_C32(0x56c20912),
	  SPH_C32(0x4ba936b9), SPH_C32(0x6dbdf830), SPH_C32(0x7ad00350),
	  SPH_C32(0x4ce60000), SPH_C32(0x36330000), SPH_C32(0xc78a0000),
	  SPH_C32(0xf5ba13e7), SPH_C32(0xbbe659a8), SPH_C32(0x634589f0),
	  SPH_C32(0x9e7ab68b) },
	{ SPH_C32(0x26600240), SPH_C32(0xddd80000), SPH_C32(0x722a0000),
	  SPH_C32(0x4f060000), SPH_C32(0x936667ff), SPH_C32(0x29f944ce),
	  SPH_C32(0x368b63d5), SPH_C32(0x0c26f262), SPH_C32(0xef0b0270),
	  SPH_C32(0x3afd0000), SPH_C32(0x5dae0000), SPH_C32(0x69490000),
	  SPH_C32(0x9b0f3c06), SPH_C32(0x4405b5f9), SPH_C32(0x66140a51),
	  SPH_C32(0x924f5d0a) },
	{ SPH_C32(0x92570220), SPH_C32(0xd1940000), SPH_C32(0x24e80000),
	  SPH_C32(0x13a80000), SPH_C32(0x073278c0), SPH_C32(0x12c7bceb),
	  SPH_C32(0x2dbd3ce8), SPH_C32(0xfff2a53a), SPH_C32(0xb3bb0360),
	  SPH_C32(0xabc30000), SPH_C32(0x19b70000), SPH_C32(0xe1c50000),
	  SPH_C32(0xfdd3481e), SPH_C32(0xd61aa89f), SPH_C32(0x33dae074),
	  SPH_C32(0x001319e3) }
};

static const sph_u32 T512_9[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x774400f0), SPH_C32(0xf15a0000), SPH_C32(0xf5b20000),
	  SPH_C32(0x34140000), SPH_C32(0x89377e8c), SPH_C32(0x5a8bec25),
	  SPH_C32(0x0bc3cd1e), SPH_C32(0xcf3775cb), SPH_C32(0xf46c0050),
	  SPH_C32(0x96180000), SPH_C32(0x14a50000), SPH_C32(0x031f0000),
	  SPH_C32(0x42947eb8), SPH_C32(0x66bf7e19), SPH_C32(0x9ca470d2),
	  SPH_C32(0x8a341574) },
	{ SPH_C32(0xf46c0050), SPH_C32(0x96180000), SPH_C32(0x14a50000),
	  SPH_C32(0x031f0000), SPH_C32(0x42947eb8), SPH_C32(0x66bf7e19),
	  SPH_C32(0x9ca470d2), SPH_C32(0x8a341574), SPH_C32(0x832800a0),
	  SPH_C32(0x67420000), SPH_C32(0xe1170000), SPH_C32(0x370b0000),
	  SPH_C32(0xcba30034), SPH_C32(0x3c34923c), SPH_C32(0x9767bdcc),
	  SPH_C32(0x450360bf) },
	{ SPH_C32(0x832800a0), SPH_C32(0x67420000), SPH_C32(0xe1170000),
	  SPH_C32(0x370b0000), SPH_C32(0xcba30034), SPH_C32(0x3c34923c),
	  SPH_C32(0x9767bdcc), SPH_C32(0x450360bf), SPH_C32(0x774400f0),
	  SPH_C32(0xf15a0000), SPH_C32(0xf5b20000), SPH_C32(0x34140000),
	  SPH_C32(0x89377e8c), SPH_C32(0x5a8bec25), SPH_C32(0x0bc3cd1e),
	  SPH_C32(0xcf3775cb) },
	{ SPH_C32(0xe8870170), SPH_C32(0x9d720000), SPH_C32(0x12db0000),
	  SPH_C32(0xd4220000), SPH_C32(0xf2886b27), SPH_C32(0xa921e543),
	  SPH_C32(0x4ef8b518), SPH_C32(0x618813b1), SPH_C32(0xb4370060),
	  SPH_C32(0x0c4c0000), SPH_C32(0x56c20000), SPH_C32(0x5cae0000),
	  SPH_C32(0x94541f3f), SPH_C32(0x3b3ef825), SPH_C32(0x1b365f3d),
	  SPH_C32(0xf3d45758) },
	{ SPH_C32(0x9fc30180), SPH_C32(0x6c280000), SPH_C32(0xe7690000),
	  SPH_C32(0xe0360000), SPH_C32(0x7bbf15ab), SPH_C32(0xf3aa0966),
	  SPH_C32(0x453b7806), SPH_C32(0xaebf667a), SPH_C32(0x405b0030),
	  SPH_C32(0x9a540000), SPH_C32(0x42670000), SPH_C32(0x5fb10000),
	  SPH_C32(0xd6c06187), SPH_C32(0x5d81863c), SPH_C32(0x87922fef),
	  SPH_C32(0x79e0422c) },
	{ SPH_C32(0x1ceb0120), SPH_C32(0x0b6a0000), SPH_C32(0x067e0000),
	  SPH_C32(0xd73d0000), SPH_C32(0xb01c159f), SPH_C32(0xcf9e9b5a),
	  SPH_C32(0xd25cc5ca), SPH_C32(0xebbc06c5), SPH_C32(0x371f00c0),
	  SPH_C32(0x6b0e0000), SPH_C32(0xb7d50000), SPH_C32(0x6ba50000),
	  SPH_C32(0x5ff71f0b), SPH_C32(0x070a6a19), SPH_C32(0x8c51e2f1),
	  SPH_C32(0xb6d737e7) },
	{ SPH_C32(0x6baf01d0), SPH_C32(0xfa300000), SPH_C32(0xf3cc0000),
	  SPH_C32(0xe3290000), SPH_C32(0x392b6b13), SPH_C32(0x9515777f),
	  SPH_C32(0xd99f08d4), SPH_C32(0x248b730e), SPH_C32(0xc3730090),
	  SPH_C32(0xfd160000), SPH_C32(0xa3700000), SPH_C32(0x68ba0000),
	  SPH_C32(0x1d6361b3), SPH_C32(0x61b51400), SPH_C32(0x10f59223),
	  SPH_C32(0x3ce32293) }
};

static const sph_u32 T512_12[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xa67f0001), SPH_C32(0x71378000), SPH_C32(0x19fc0000),
	  SPH_C32(0x96db0000), SPH_C32(0x3a8b6dfd), SPH_C32(0xebcaaef3),
	  SPH_C32(0x2c6d478f), SPH_C32(0xac8e6c88), SPH_C32(0x50ff0004),
	  SPH_C32(0x45744000), SPH_C32(0x3dfb0000), SPH_C32(0x19e60000),
	  SPH_C32(0x1bbc5606), SPH_C32(0xe1727b5d), SPH_C32(0xe1a8cc96),
	  SPH_C32(0x7b1bd6b9) },
	{ SPH_C32(0xf7750009), SPH_C32(0xcf3cc000), SPH_C32(0xc3d60000),
	  SPH_C32(0x04920000), SPH_C32(0x029519a9), SPH_C32(0xf8e836ba),
	  SPH_C32(0x7a87f14e), SPH_C32(0x9e16981a), SPH_C32(0xd46a0000),
	  SPH_C32(0x8dc8c000), SPH_C32(0xa5af0000), SPH_C32(0x4a290000),
	  SPH_C32(0xfc4e427a), SPH_C32(0xc9b4866c), SPH_C32(0x98369604),
	  SPH_C32(0xf746c320) },
	{ SPH_C32(0x510a0008), SPH_C32(0xbe0b4000), SPH_C32(0xda2a0000),
	  SPH_C32(0x92490000), SPH_C32(0x381e7454), SPH_C32(0x13229849),
	  SPH_C32(0x56eab6c1), SPH_C32(0x3298f492), SPH_C32(0x84950004),
	  SPH_C32(0xc8bc8000), SPH_C32(0x98540000), SPH_C32(0x53cf0000),
	  SPH_C32(0xe7f2147c), SPH_C32(0x28c6fd31), SPH_C32(0x799e5a92),
	  SPH_C32(0x8c5d1599) },
	{ SPH_C32(0xd46a0000), SPH_C32(0x8dc8c000), SPH_C32(0xa5af0000),
	  SPH_C32(0x4a290000), SPH_C32(0xfc4e427a), SPH_C32(0xc9b4866c),
	  SPH_C32(0x98369604), SPH_C32(0xf746c320), SPH_C32(0x231f0009),
	  SPH_C32(0x42f40000), SPH_C32(0x66790000), SPH_C32(0x4ebb0000),
	  SPH_C32(0xfedb5bd3), SPH_C32(0x315cb0d6), SPH_C32(0xe2b1674a),
	  SPH_C32(0x69505b3a) },
	{ SPH_C32(0x72150001), SPH_C32(0xfcff4000), SPH_C32(0xbc530000),
	  SPH_C32(0xdcf20000), SPH_C32(0xc6c52f87), SPH_C32(0x227e289f),
	  SPH_C32(0xb45bd18b), SPH_C32(0x5bc8afa8), SPH_C32(0x73e0000d),
	  SPH_C32(0x07804000), SPH_C32(0x5b820000), SPH_C32(0x575d0000),
	  SPH_C32(0xe5670dd5), SPH_C32(0xd02ecb8b), SPH_C32(0x0319abdc),
	  SPH_C32(0x124b8d83) },
	{ SPH_C32(0x231f0009), SPH_C32(0x42f40000), SPH_C32(0x66790000),
	  SPH_C32(0x4ebb0000), SPH_C32(0xfedb5bd3), SPH_C32(0x315cb0d6),
	  SPH_C32(0xe2b1674a), SPH_C32(0x69505b3a), SPH_C32(0xf7750009),
	  SPH_C32(0xcf3cc000), SPH_C32(0xc3d60000), SPH_C32(0x04920000),
	  SPH_C32(0x029519a9), SPH_C32(0xf8e836ba), SPH_C32(0x7a87f14e),
	  SPH_C32(0x9e16981a) },
	{ SPH_C32(0x85600008), SPH_C32(0x33c38000), SPH_C32(0x7f850000),
	  SPH_C32(0xd8600000), SPH_C32(0xc450362e), SPH_C32(0xda961e25),
	  SPH_C32(0xcedc20c5), SPH_C32(0xc5de37b2), SPH_C32(0xa78a000d),
	  SPH_C32(0x8a488000), SPH_C32(0xfe2d0000), SPH_C32(0x1d740000),
	  SPH_C32(0x19294faf), SPH_C32(0x199a4de7), SPH_C32(0x9b2f3dd8),
	  SPH_C32(0xe50d4ea3) }
};

static const sph_u32 T512_15[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x75c90003), SPH_C32(0x0e10c000), SPH_C32(0xd1200000),
	  SPH_C32(0xbaea0000), SPH_C32(0x8bc42f3e), SPH_C32(0x8758b757),
	  SPH_C32(0xbb28761d), SPH_C32(0x00b72e2b), SPH_C32(0xeecf0001),
	  SPH_C32(0x6f564000), SPH_C32(0xf33e0000), SPH_C32(0xa79e0000),
	  SPH_C32(0xbdb57219), SPH_C32(0xb711ebc5), SPH_C32(0x4a3b40ba),
	  SPH_C32(0xfeabf254) },
	{ SPH_C32(0xeecf0001), SPH_C32(0x6f564000), SPH_C32(0xf33e0000),
	  SPH_C32(0xa79e0000), SPH_C32(0xbdb57219), SPH_C32(0xb711ebc5),
	  SPH_C32(0x4a3b40ba), SPH_C32(0xfeabf254), SPH_C32(0x9b060002),
	  SPH_C32(0x61468000), SPH_C32(0x221e0000), SPH_C32(0x1d740000),
	  SPH_C32(0x36715d27), SPH_C32(0x30495c92), SPH_C32(0xf11336a7),
	  SPH_C32(0xfe1cdc7f) },
	{ SPH_C32(0x9b060002), SPH_C32(0x61468000), SPH_C32(0x221e0000),
	  SPH_C32(0x1d740000), SPH_C32(0x36715d27), SPH_C32(0x30495c92),
	  SPH_C32(0xf11336a7), SPH_C32(0xfe1cdc7f), SPH_C32(0x75c90003),
	  SPH_C32(0x0e10c000), SPH_C32(0xd1200000), SPH_C32(0xbaea0000),
	  SPH_C32(0x8bc42f3e), SPH_C32(0x8758b757), SPH_C32(0xbb28761d),
	  SPH_C32(0x00b72e2b) },
	{ SPH_C32(0xf6800005), SPH_C32(0x3443c000), SPH_C32(0x24070000),
	  SPH_C32(0x8f3d0000), SPH_C32(0x21373bfb), SPH_C32(0x0ab8d5ae),
	  SPH_C32(0xcdc58b19), SPH_C32(0xd795ba31), SPH_C32(0xa67f0001),
	  SPH_C32(0x71378000), SPH_C32(0x19fc0000), SPH_C32(0x96db0000),
	  SPH_C32(0x3a8b6dfd), SPH_C32(0xebcaaef3), SPH_C32(0x2c6d478f),
	  SPH_C32(0xac8e6c88) },
	{ SPH_C32(0x83490006), SPH_C32(0x3a530000), SPH_C32(0xf5270000),
	  SPH_C32(0x35d70000), SPH_C32(0xaaf314c5), SPH_C32(0x8de062f9),
	  SPH_C32(0x76edfd04), SPH_C32(0xd722941a), SPH_C32(0x48b00000),
	  SPH_C32(0x1e61c000), SPH_C32(0xeac20000), SPH_C32(0x31450000),
	  SPH_C32(0x873e1fe4), SPH_C32(0x5cdb4536), SPH_C32(0x66560735),
	  SPH_C32(0x52259edc) },
	{ SPH_C32(0x184f0004), SPH_C32(0x5b158000), SPH_C32(0xd7390000),
	  SPH_C32(0x28a30000), SPH_C32(0x9c8249e2), SPH_C32(0xbda93e6b),
	  SPH_C32(0x87fecba3), SPH_C32(0x293e4865), SPH_C32(0x3d790003),
	  SPH_C32(0x10710000), SPH_C32(0x3be20000), SPH_C32(0x8baf0000),
	  SPH_C32(0x0cfa30da), SPH_C32(0xdb83f261), SPH_C32(0xdd7e7128),
	  SPH_C32(0x5292b0f7) },
	{ SPH_C32(0x6d860007), SPH_C32(0x55054000), SPH_C32(0x06190000),
	  SPH_C32(0x92490000), SPH_C32(0x174666dc), SPH_C32(0x3af1893c),
	  SPH_C32(0x3cd6bdbe), SPH_C32(0x2989664e), SPH_C32(0xd3b60002),
	  SPH_C32(0x7f274000), SPH_C32(0xc8dc0000), SPH_C32(0x2c310000),
	  SPH_C32(0xb14f42c3), SPH_C32(0x6c9219a4), SPH_C32(0x97453192),
	  SPH_C32(0xac3942a3) }
};

static const sph_u32 T512_18[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xb83d0000), SPH_C32(0x16710600), SPH_C32(0x379a0000),
	  SPH_C32(0xf5b10000), SPH_C32(0x228161ac), SPH_C32(0xae48f145),
	  SPH_C32(0x66241616), SPH_C32(0xc5c1eb3e), SPH_C32(0xfd250000),
	  SPH_C32(0xb3c41100), SPH_C32(0xcef00000), SPH_C32(0xcef90000),
	  SPH_C32(0x3c4d7580), SPH_C32(0x8d5b6493), SPH_C32(0x7098b0a6),
	  SPH_C32(0x1af21fe1) },
	{ SPH_C32(0x75a40000), SPH_C32(0xc28b2700), SPH_C32(0x94a40000),
	  SPH_C32(0x90f50000), SPH_C32(0xfb7857e0), SPH_C32(0x49ce0bae),
	  SPH_C32(0x1767c483), SPH_C32(0xaedf667e), SPH_C32(0xd1660000),
	  SPH_C32(0x1bbc0300), SPH_C32(0x9eec0000), SPH_C32(0xf6940000),
	  SPH_C32(0x03024527), SPH_C32(0xcf70fcf2), SPH_C32(0xb4431b17),
	  SPH_C32(0x857f3c2b) },
	{ SPH_C32(0xcd990000), SPH_C32(0xd4fa2100), SPH_C32(0xa33e0000),
	  SPH_C32(0x65440000), SPH_C32(0xd9f9364c), SPH_C32(0xe786faeb),
	  SPH_C32(0x7143d295), SPH_C32(0x6b1e8d40), SPH_C32(0x2c430000),
	  SPH_C32(0xa8781200), SPH_C32(0x501c0000), SPH_C32(0x386d0000),
	  SPH_C32(0x3f4f30a7), SPH_C32(0x422b9861), SPH_C32(0xc4dbabb1),
	  SPH_C32(0x9f8d23ca) },
	{ SPH_C32(0xd1660000), SPH_C32(0x1bbc0300), SPH_C32(0x9eec0000),
	  SPH_C32(0xf6940000), SPH_C32(0x03024527), SPH_C32(0xcf70fcf2),
	  SPH_C32(0xb4431b17), SPH_C32(0x857f3c2b), SPH_C32(0xa4c20000),
	  SPH_C32(0xd9372400), SPH_C32(0x0a480000), SPH_C32(0x66610000),
	  SPH_C32(0xf87a12c7), SPH_C32(0x86bef75c), SPH_C32(0xa324df94),
	  SPH_C32(0x2ba05a55) },
	{ SPH_C32(0x695b0000), SPH_C32(0x0dcd0500), SPH_C32(0xa9760000),
	  SPH_C32(0x03250000), SPH_C32(0x2183248b), SPH_C32(0x61380db7),
	  SPH_C32(0xd2670d01), SPH_C32(0x40bed715), SPH_C32(0x59e70000),
	  SPH_C32(0x6af33500), SPH_C32(0xc4b80000), SPH_C32(0xa8980000),
	  SPH_C32(0xc4376747), SPH_C32(0x0be593cf), SPH_C32(0xd3bc6f32),
	  SPH_C32(0x315245b4) },
	{ SPH_C32(0xa4c20000), SPH_C32(0xd9372400), SPH_C32(0x0a480000),
	  SPH_C32(0x66610000), SPH_C32(0xf87a12c7), SPH_C32(0x86bef75c),
	  SPH_C32(0xa324df94), SPH_C32(0x2ba05a55), SPH_C32(0x75a40000),
	  SPH_C32(0xc28b2700), SPH_C32(0x94a40000), SPH_C32(0x90f50000),
	  SPH_C32(0xfb7857e0), SPH_C32(0x49ce0bae), SPH_C32(0x1767c483),
	  SPH_C32(0xaedf667e) },
	{ SPH_C32(0x1cff0000), SPH_C32(0xcf462200), SPH_C32(0x3dd20000),
	  SPH_C32(0x93d00000), SPH_C32(0xdafb736b), SPH_C32(0x28f60619),
	  SPH_C32(0xc500c982), SPH_C32(0xee61b16b), SPH_C32(0x88810000),
	  SPH_C32(0x714f3600), SPH_C32(0x5a540000), SPH_C32(0x5e0c0000),
	  SPH_C32(0xc7352260), SPH_C32(0xc4956f3d), SPH_C32(0x67ff7425),
	  SPH_C32(0xb42d799f) }
};

static const sph_u32 T512_21[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x0c720000), SPH_C32(0x49e50f00), SPH_C32(0x42790000),
	  SPH_C32(0x5cea0000), SPH_C32(0x33aa301a), SPH_C32(0x15822514),
	  SPH_C32(0x95a34b7b), SPH_C32(0xb44b0090), SPH_C32(0xfe220000),
	  SPH_C32(0xa7580500), SPH_C32(0x25d10000), SPH_C32(0xf7600000),
	  SPH_C32(0x893178da), SPH_C32(0x1fd4f860), SPH_C32(0x4ed0a315),
	  SPH_C32(0xa123ff9f) },
	{ SPH_C32(0xfe220000), SPH_C32(0xa7580500), SPH_C32(0x25d10000),
	  SPH_C32(0xf7600000), SPH_C32(0x893178da), SPH_C32(0x1fd4f860),
	  SPH_C32(0x4ed0a315), SPH_C32(0xa123ff9f), SPH_C32(0xf2500000),
	  SPH_C32(0xeebd0a00), SPH_C32(0x67a80000), SPH_C32(0xab8a0000),
	  SPH_C32(0xba9b48c0), SPH_C32(0x0a56dd74), SPH_C32(0xdb73e86e),
	  SPH_C32(0x1568ff0f) },
	{ SPH_C32(0xf2500000), SPH_C32(0xeebd0a00), SPH_C32(0x67a80000),
	  SPH_C32(0xab8a0000), SPH_C32(0xba9b48c0), SPH_C32(0x0a56dd74),
	  SPH_C32(0xdb73e86e), SPH_C32(0x1568ff0f), SPH_C32(0x0c720000),
	  SPH_C32(0x49e50f00), SPH_C32(0x42790000), SPH_C32(0x5cea0000),
	  SPH_C32(0x33aa301a), SPH_C32(0x15822514), SPH_C32(0x95a34b7b),
	  SPH_C32(0xb44b0090) },
	{ SPH_C32(0x45180000), SPH_C32(0xa5b51700), SPH_C32(0xf96a0000),
	  SPH_C32(0x3b480000), SPH_C32(0x1ecc142c), SPH_C32(0x231395d6),
	  SPH_C32(0x16bca6b0), SPH_C32(0xdf33f4df), SPH_C32(0xb83d0000),
	  SPH_C32(0x16710600), SPH_C32(0x379a0000), SPH_C32(0xf5b10000),
	  SPH_C32(0x228161ac), SPH_C32(0xae48f145), SPH_C32(0x66241616),
	  SPH_C32(0xc5c1eb3e) },
	{ SPH_C32(0x496a0000), SPH_C32(0xec501800), SPH_C32(0xbb130000),
	  SPH_C32(0x67a20000), SPH_C32(0x2d662436), SPH_C32(0x3691b0c2),
	  SPH_C32(0x831fedcb), SPH_C32(0x6b78f44f), SPH_C32(0x461f0000),
	  SPH_C32(0xb1290300), SPH_C32(0x124b0000), SPH_C32(0x02d10000),
	  SPH_C32(0xabb01976), SPH_C32(0xb19c0925), SPH_C32(0x28f4b503),
	  SPH_C32(0x64e214a1) },
	{ SPH_C32(0xbb3a0000), SPH_C32(0x02ed1200), SPH_C32(0xdcbb0000),
	  SPH_C32(0xcc280000), SPH_C32(0x97fd6cf6), SPH_C32(0x3cc76db6),
	  SPH_C32(0x586c05a5), SPH_C32(0x7e100b40), SPH_C32(0x4a6d0000),
	  SPH_C32(0xf8cc0c00), SPH_C32(0x50320000), SPH_C32(0x5e3b0000),
	  SPH_C32(0x981a296c), SPH_C32(0xa41e2c31), SPH_C32(0xbd57fe78),
	  SPH_C32(0xd0a91431) },
	{ SPH_C32(0xb7480000), SPH_C32(0x4b081d00), SPH_C32(0x9ec20000),
	  SPH_C32(0x90c20000), SPH_C32(0xa4575cec), SPH_C32(0x294548a2),
	  SPH_C32(0xcdcf4ede), SPH_C32(0xca5b0bd0), SPH_C32(0xb44f0000),
	  SPH_C32(0x5f940900), SPH_C32(0x75e30000), SPH_C32(0xa95b0000),
	  SPH_C32(0x112b51b6), SPH_C32(0xbbcad451), SPH_C32(0xf3875d6d),
	  SPH_C32(0x718aebae) }
};

static const sph_u32 T512_24[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xbc8d0000), SPH_C32(0xfc3b0018), SPH_C32(0x19830000),
	  SPH_C32(0xd10b0000), SPH_C32(0xae1878c4), SPH_C32(0x42a69856),
	  SPH_C32(0x0012da37), SPH_C32(0x2c3b504e), SPH_C32(0xe8dd0000),
	  SPH_C32(0xfa4a0044), SPH_C32(0x3c2d0000), SPH_C32(0xbb150000),
	  SPH_C32(0x80bd361b), SPH_C32(0x24e81d44), SPH_C32(0xbfa8c2f4),
	  SPH_C32(0x524a0d59) },
	{ SPH_C32(0x69510000), SPH_C32(0xd4e1009c), SPH_C32(0xc3230000),
	  SPH_C32(0xac2f0000), SPH_C32(0xe4950bae), SPH_C32(0xcea415dc),
	  SPH_C32(0x87ec287c), SPH_C32(0xbce1a3ce), SPH_C32(0xc6730000),
	  SPH_C32(0xaf8d000c), SPH_C32(0xa4c10000), SPH_C32(0x218d0000),
	  SPH_C32(0x23111587), SPH_C32(0x7913512f), SPH_C32(0x1d28ac88),
	  SPH_C32(0x378dd173) },
	{ SPH_C32(0xd5dc0000), SPH_C32(0x28da0084), SPH_C32(0xdaa00000),
	  SPH_C32(0x7d240000), SPH_C32(0x4a8d736a), SPH_C32(0x8c028d8a),
	  SPH_C32(0x87fef24b), SPH_C32(0x90daf380), SPH_C32(0x2eae0000),
	  SPH_C32(0x55c70048), SPH_C32(0x98ec0000), SPH_C32(0x9a980000),
	  SPH_C32(0xa3ac239c), SPH_C32(0x5dfb4c6b), SPH_C32(0xa2806e7c),
	  SPH_C32(0x65c7dc2a) },
	{ SPH_C32(0xc6730000), SPH_C32(0xaf8d000c), SPH_C32(0xa4c10000),
	  SPH_C32(0x218d0000), SPH_C32(0x23111587), SPH_C32(0x7913512f),
	  SPH_C32(0x1d28ac88), SPH_C32(0x378dd173), SPH_C32(0xaf220000),
	  SPH_C32(0x7b6c0090), SPH_C32(0x67e20000), SPH_C32(0x8da20000),
	  SPH_C32(0xc7841e29), SPH_C32(0xb7b744f3), SPH_C32(0x9ac484f4),
	  SPH_C32(0x8b6c72bd) },
	{ SPH_C32(0x7afe0000), SPH_C32(0x53b60014), SPH_C32(0xbd420000),
	  SPH_C32(0xf0860000), SPH_C32(0x8d096d43), SPH_C32(0x3bb5c979),
	  SPH_C32(0x1d3a76bf), SPH_C32(0x1bb6813d), SPH_C32(0x47ff0000),
	  SPH_C32(0x812600d4), SPH_C32(0x5bcf0000), SPH_C32(0x36b70000),
	  SPH_C32(0x47392832), SPH_C32(0x935f59b7), SPH_C32(0x256c4600),
	  SPH_C32(0xd9267fe4) },
	{ SPH_C32(0xaf220000), SPH_C32(0x7b6c0090), SPH_C32(0x67e20000),
	  SPH_C32(0x8da20000), SPH_C32(0xc7841e29), SPH_C32(0xb7b744f3),
	  SPH_C32(0x9ac484f4), SPH_C32(0x8b6c72bd), SPH_C32(0x69510000),
	  SPH_C32(0xd4e1009c), SPH_C32(0xc3230000), SPH_C32(0xac2f0000),
	  SPH_C32(0xe4950bae), SPH_C32(0xcea415dc), SPH_C32(0x87ec287c),
	  SPH_C32(0xbce1a3ce) },
	{ SPH_C32(0x13af0000), SPH_C32(0x87570088), SPH_C32(0x7e610000),
	  SPH_C32(0x5ca90000), SPH_C32(0x699c66ed), SPH_C32(0xf511dca5),
	  SPH_C32(0x9ad65ec3), SPH_C32(0xa75722f3), SPH_C32(0x818c0000),
	  SPH_C32(0x2eab00d8), SPH_C32(0xff0e0000), SPH_C32(0x173a0000),
	  SPH_C32(0x64283db5), SPH_C32(0xea4c0898), SPH_C32(0x3844ea88),
	  SPH_C32(0xeeabae97) }
};

static const sph_u32 T512_27[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x14190000), SPH_C32(0x23ca003c), SPH_C32(0x50df0000),
	  SPH_C32(0x44b60000), SPH_C32(0x1b6c67b0), SPH_C32(0x3cf3ac75),
	  SPH_C32(0x61e610b0), SPH_C32(0xdbcadb80), SPH_C32(0xe3430000),
	  SPH_C32(0x3a4e0014), SPH_C32(0xf2c60000), SPH_C32(0xaa4e0000),
	  SPH_C32(0xdb1e42a6), SPH_C32(0x256bbe15), SPH_C32(0x123db156),
	  SPH_C32(0x3a4e99d7) },
	{ SPH_C32(0xe3430000), SPH_C32(0x3a4e0014), SPH_C32(0xf2c60000),
	  SPH_C32(0xaa4e0000), SPH_C32(0xdb1e42a6), SPH_C32(0x256bbe15),
	  SPH_C32(0x123db156), SPH_C32(0x3a4e99d7), SPH_C32(0xf75a0000),
	  SPH_C32(0x19840028), SPH_C32(0xa2190000), SPH_C32(0xeef80000),
	  SPH_C32(0xc0722516), SPH_C32(0x19981260), SPH_C32(0x73dba1e6),
	  SPH_C32(0xe1844257) },
	{ SPH_C32(0xf75a0000), SPH_C32(0x19840028), SPH_C32(0xa2190000),
	  SPH_C32(0xeef80000), SPH_C32(0xc0722516), SPH_C32(0x19981260),
	  SPH_C32(0x73dba1e6), SPH_C32(0xe1844257), SPH_C32(0x14190000),
	  SPH_C32(0x23ca003c), SPH_C32(0x50df0000), SPH_C32(0x44b60000),
	  SPH_C32(0x1b6c67b0), SPH_C32(0x3cf3ac75), SPH_C32(0x61e610b0),
	  SPH_C32(0xdbcadb80) },
	{ SPH_C32(0x54500000), SPH_C32(0x0671005c), SPH_C32(0x25ae0000),
	  SPH_C32(0x6a1e0000), SPH_C32(0x2ea54edf), SPH_C32(0x664e8512),
	  SPH_C32(0xbfba18c3), SPH_C32(0x7e715d17), SPH_C32(0xbc8d0000),
	  SPH_C32(0xfc3b0018), SPH_C32(0x19830000), SPH_C32(0xd10b0000),
	  SPH_C32(0xae1878c4), SPH_C32(0x42a69856), SPH_C32(0x0012da37),
	  SPH_C32(0x2c3b504e) },
	{ SPH_C32(0x40490000), SPH_C32(0x25bb0060), SPH_C32(0x75710000),
	  SPH_C32(0x2ea80000), SPH_C32(0x35c9296f), SPH_C32(0x5abd2967),
	  SPH_C32(0xde5c0873), SPH_C32(0xa5bb8697), SPH_C32(0x5fce0000),
	  SPH_C32(0xc675000c), SPH_C32(0xeb450000), SPH_C32(0x7b450000),
	  SPH_C32(0x75063a62), SPH_C32(0x67cd2643), SPH_C32(0x122f6b61),
	  SPH_C32(0x1675c999) },
	{ SPH_C32(0xb7130000), SPH_C32(0x3c3f0048), SPH_C32(0xd7680000),
	  SPH_C32(0xc0500000), SPH_C32(0xf5bb0c79), SPH_C32(0x43253b07),
	  SPH_C32(0xad87a995), SPH_C32(0x443fc4c0), SPH_C32(0x4bd70000),
	  SPH_C32(0xe5bf0030), SPH_C32(0xbb9a0000), SPH_C32(0x3ff30000),
	  SPH_C32(0x6e6a5dd2), SPH_C32(0x5b3e8a36), SPH_C32(0x73c97bd1),
	  SPH_C32(0xcdbf1219) },
	{ SPH_C32(0xa30a0000), SPH_C32(0x1ff50074), SPH_C32(0x87b70000),
	  SPH_C32(0x84e60000), SPH_C32(0xeed76bc9), SPH_C32(0x7fd69772),
	  SPH_C32(0xcc61b925), SPH_C32(0x9ff51f40), SPH_C32(0xa8940000),
	  SPH_C32(0xdff10024), SPH_C32(0x495c0000), SPH_C32(0x95bd0000),
	  SPH_C32(0xb5741f74), SPH_C32(0x7e553423), SPH_C32(0x61f4ca87),
	  SPH_C32(0xf7f18bce) }
};

static const sph_u32 T512_30[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xdb250000), SPH_C32(0x09290000), SPH_C32(0x49aac000),
	  SPH_C32(0x81e10000), SPH_C32(0xcafe6b59), SPH_C32(0x42793431),
	  SPH_C32(0x43566b76), SPH_C32(0xe86cba2e), SPH_C32(0x75e60000),
	  SPH_C32(0x95660001), SPH_C32(0x307b2000), SPH_C32(0xadf40000),
	  SPH_C32(0x8f321eea), SPH_C32(0x24298307), SPH_C32(0xe8c49cf9),
	  SPH_C32(0x4b7eec55) },
	{ SPH_C32(0x86790000), SPH_C32(0x3f390002), SPH_C32(0xe19ae000),
	  SPH_C32(0x98560000), SPH_C32(0x9565670e), SPH_C32(0x4e88c8ea),
	  SPH_C32(0xd3dd4944), SPH_C32(0x161ddab9), SPH_C32(0x30b70000),
	  SPH_C32(0xe5d00000), SPH_C32(0xf4f46000), SPH_C32(0x42c40000),
	  SPH_C32(0x63b83d6a), SPH_C32(0x78ba9460), SPH_C32(0x21afa1ea),
	  SPH_C32(0xb0a51834) },
	{ SPH_C32(0x5d5c0000), SPH_C32(0x36100002), SPH_C32(0xa8302000),
	  SPH_C32(0x19b70000), SPH_C32(0x5f9b0c57), SPH_C32(0x0cf1fcdb),
	  SPH_C32(0x908b2232), SPH_C32(0xfe716097), SPH_C32(0x45510000),
	  SPH_C32(0x70b60001), SPH_C32(0xc48f4000), SPH_C32(0xef300000),
	  SPH_C32(0xec8a2380), SPH_C32(0x5c931767), SPH_C32(0xc96b3d13),
	  SPH_C32(0xfbdbf461) },
	{ SPH_C32(0x30b70000), SPH_C32(0xe5d00000), SPH_C32(0xf4f46000),
	  SPH_C32(0x42c40000), SPH_C32(0x63b83d6a), SPH_C32(0x78ba9460),
	  SPH_C32(0x21afa1ea), SPH_C32(0xb0a51834), SPH_C32(0xb6ce0000),
	  SPH_C32(0xdae90002), SPH_C32(0x156e8000), SPH_C32(0xda920000),
	  SPH_C32(0xf6dd5a64), SPH_C32(0x36325c8a), SPH_C32(0xf272e8ae),
	  SPH_C32(0xa6b8c28d) },
	{ SPH_C32(0xeb920000), SPH_C32(0xecf90000), SPH_C32(0xbd5ea000),
	  SPH_C32(0xc3250000), SPH_C32(0xa9465633), SPH_C32(0x3ac3a051),
	  SPH_C32(0x62f9ca9c), SPH_C32(0x58c9a21a), SPH_C32(0xc3280000),
	  SPH_C32(0x4f8f0003), SPH_C32(0x2515a000), SPH_C32(0x77660000),
	  SPH_C32(0x79ef448e), SPH_C32(0x121bdf8d), SPH_C32(0x1ab67457),
	  SPH_C32(0xedc62ed8) },
	{ SPH_C32(0xb6ce0000), SPH_C32(0xdae90002), SPH_C32(0x156e8000),
	  SPH_C32(0xda920000), SPH_C32(0xf6dd5a64), SPH_C32(0x36325c8a),
	  SPH_C32(0xf272e8ae), SPH_C32(0xa6b8c28d), SPH_C32(0x86790000),
	  SPH_C32(0x3f390002), SPH_C32(0xe19ae000), SPH_C32(0x98560000),
	  SPH_C32(0x9565670e), SPH_C32(0x4e88c8ea), SPH_C32(0xd3dd4944),
	  SPH_C32(0x161ddab9) },
	{ SPH_C32(0x6deb0000), SPH_C32(0xd3c00002), SPH_C32(0x5cc44000),
	  SPH_C32(0x5b730000), SPH_C32(0x3c23313d), SPH_C32(0x744b68bb),
	  SPH_C32(0xb12483d8), SPH_C32(0x4ed478a3), SPH_C32(0xf39f0000),
	  SPH_C32(0xaa5f0003), SPH_C32(0xd1e1c000), SPH_C32(0x35a20000),
	  SPH_C32(0x1a5779e4), SPH_C32(0x6aa14bed), SPH_C32(0x3b19d5bd),
	  SPH_C32(0x5d6336ec) }
};

static const sph_u32 T512_33[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xac480000), SPH_C32(0x1ba60000), SPH_C32(0x45fb1380),
	  SPH_C32(0x03430000), SPH_C32(0x5a85316a), SPH_C32(0x1fb250b6),
	  SPH_C32(0xfe72c7fe), SPH_C32(0x91e478f6), SPH_C32(0x1e4e0000),
	  SPH_C32(0xdecf0000), SPH_C32(0x6df80180), SPH_C32(0x77240000),
	  SPH_C32(0xec47079e), SPH_C32(0xf4a0694e), SPH_C32(0xcda31812),
	  SPH_C32(0x98aa496e) },
	{ SPH_C32(0x1e4e0000), SPH_C32(0xdecf0000), SPH_C32(0x6df80180),
	  SPH_C32(0x77240000), SPH_C32(0xec47079e), SPH_C32(0xf4a0694e),
	  SPH_C32(0xcda31812), SPH_C32(0x98aa496e), SPH_C32(0xb2060000),
	  SPH_C32(0xc5690000), SPH_C32(0x28031200), SPH_C32(0x74670000),
	  SPH_C32(0xb6c236f4), SPH_C32(0xeb1239f8), SPH_C32(0x33d1dfec),
	  SPH_C32(0x094e3198) },
	{ SPH_C32(0xb2060000), SPH_C32(0xc5690000), SPH_C32(0x28031200),
	  SPH_C32(0x74670000), SPH_C32(0xb6c236f4), SPH_C32(0xeb1239f8),
	  SPH_C32(0x33d1dfec), SPH_C32(0x094e3198), SPH_C32(0xac480000),
	  SPH_C32(0x1ba60000), SPH_C32(0x45fb1380), SPH_C32(0x03430000),
	  SPH_C32(0x5a85316a), SPH_C32(0x1fb250b6), SPH_C32(0xfe72c7fe),
	  SPH_C32(0x91e478f6) },
	{ SPH_C32(0xaec30000), SPH_C32(0x9c4f0001), SPH_C32(0x79d1e000),
	  SPH_C32(0x2c150000), SPH_C32(0x45cc75b3), SPH_C32(0x6650b736),
	  SPH_C32(0xab92f78f), SPH_C32(0xa312567b), SPH_C32(0xdb250000),
	  SPH_C32(0x09290000), SPH_C32(0x49aac000), SPH_C32(0x81e10000),
	  SPH_C32(0xcafe6b59), SPH_C32(0x42793431), SPH_C32(0x43566b76),
	  SPH_C32(0xe86cba2e) },
	{ SPH_C32(0x028b0000), SPH_C32(0x87e90001), SPH_C32(0x3c2af380),
	  SPH_C32(0x2f560000), SPH_C32(0x1f4944d9), SPH_C32(0x79e2e780),
	  SPH_C32(0x55e03071), SPH_C32(0x32f62e8d), SPH_C32(0xc56b0000),
	  SPH_C32(0xd7e60000), SPH_C32(0x2452c180), SPH_C32(0xf6c50000),
	  SPH_C32(0x26b96cc7), SPH_C32(0xb6d95d7f), SPH_C32(0x8ef57364),
	  SPH_C32(0x70c6f340) },
	{ SPH_C32(0xb08d0000), SPH_C32(0x42800001), SPH_C32(0x1429e180),
	  SPH_C32(0x5b310000), SPH_C32(0xa98b722d), SPH_C32(0x92f0de78),
	  SPH_C32(0x6631ef9d), SPH_C32(0x3bb81f15), SPH_C32(0x69230000),
	  SPH_C32(0xcc400000), SPH_C32(0x61a9d200), SPH_C32(0xf5860000),
	  SPH_C32(0x7c3c5dad), SPH_C32(0xa96b0dc9), SPH_C32(0x7087b49a),
	  SPH_C32(0xe1228bb6) },
	{ SPH_C32(0x1cc50000), SPH_C32(0x59260001), SPH_C32(0x51d2f200),
	  SPH_C32(0x58720000), SPH_C32(0xf30e4347), SPH_C32(0x8d428ece),
	  SPH_C32(0x98432863), SPH_C32(0xaa5c67e3), SPH_C32(0x776d0000),
	  SPH_C32(0x128f0000), SPH_C32(0x0c51d380), SPH_C32(0x82a20000),
	  SPH_C32(0x907b5a33), SPH_C32(0x5dcb6487), SPH_C32(0xbd24ac88),
	  SPH_C32(0x7988c2d8) }
};

static const sph_u32 T512_36[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x819e0000), SPH_C32(0xec570000), SPH_C32(0x66320280),
	  SPH_C32(0x95f30000), SPH_C32(0x5da92802), SPH_C32(0x48f43cbc),
	  SPH_C32(0xe65aa22d), SPH_C32(0x8e67b7fa), SPH_C32(0x4d8a0000),
	  SPH_C32(0x49340000), SPH_C32(0x3c8b0500), SPH_C32(0xaea30000),
	  SPH_C32(0x16793bfd), SPH_C32(0xcf6f08a4), SPH_C32(0x8f19eaec),
	  SPH_C32(0x443d3004) },
	{ SPH_C32(0x78230000), SPH_C32(0x12fc0000), SPH_C32(0xa93a0b80),
	  SPH_C32(0x90a50000), SPH_C32(0x713e2879), SPH_C32(0x7ee98924),
	  SPH_C32(0xf08ca062), SPH_C32(0x636f8bab), SPH_C32(0x02af0000),
	  SPH_C32(0xb7280000), SPH_C32(0xba1c0300), SPH_C32(0x56980000),
	  SPH_C32(0xba8d45d3), SPH_C32(0x8048c667), SPH_C32(0xa95c149a),
	  SPH_C32(0xf4f6ea7b) },
	{ SPH_C32(0xf9bd0000), SPH_C32(0xfeab0000), SPH_C32(0xcf080900),
	  SPH_C32(0x05560000), SPH_C32(0x2c97007b), SPH_C32(0x361db598),
	  SPH_C32(0x16d6024f), SPH_C32(0xed083c51), SPH_C32(0x4f250000),
	  SPH_C32(0xfe1c0000), SPH_C32(0x86970600), SPH_C32(0xf83b0000),
	  SPH_C32(0xacf47e2e), SPH_C32(0x4f27cec3), SPH_C32(0x2645fe76),
	  SPH_C32(0xb0cbda7f) },
	{ SPH_C32(0x02af0000), SPH_C32(0xb7280000), SPH_C32(0xba1c0300),
	  SPH_C32(0x56980000), SPH_C32(0xba8d45d3), SPH_C32(0x8048c667),
	  SPH_C32(0xa95c149a), SPH_C32(0xf4f6ea7b), SPH_C32(0x7a8c0000),
	  SPH_C32(0xa5d40000), SPH_C32(0x13260880), SPH_C32(0xc63d0000),
	  SPH_C32(0xcbb36daa), SPH_C32(0xfea14f43), SPH_C32(0x59d0b4f8),
	  SPH_C32(0x979961d0) },
	{ SPH_C32(0x83310000), SPH_C32(0x5b7f0000), SPH_C32(0xdc2e0180),
	  SPH_C32(0xc36b0000), SPH_C32(0xe7246dd1), SPH_C32(0xc8bcfadb),
	  SPH_C32(0x4f06b6b7), SPH_C32(0x7a915d81), SPH_C32(0x37060000),
	  SPH_C32(0xece00000), SPH_C32(0x2fad0d80), SPH_C32(0x689e0000),
	  SPH_C32(0xddca5657), SPH_C32(0x31ce47e7), SPH_C32(0xd6c95e14),
	  SPH_C32(0xd3a451d4) },
	{ SPH_C32(0x7a8c0000), SPH_C32(0xa5d40000), SPH_C32(0x13260880),
	  SPH_C32(0xc63d0000), SPH_C32(0xcbb36daa), SPH_C32(0xfea14f43),
	  SPH_C32(0x59d0b4f8), SPH_C32(0x979961d0), SPH_C32(0x78230000),
	  SPH_C32(0x12fc0000), SPH_C32(0xa93a0b80), SPH_C32(0x90a50000),
	  SPH_C32(0x713e2879), SPH_C32(0x7ee98924), SPH_C32(0xf08ca062),
	  SPH_C32(0x636f8bab) },
	{ SPH_C32(0xfb120000), SPH_C32(0x49830000), SPH_C32(0x75140a00),
	  SPH_C32(0x53ce0000), SPH_C32(0x961a45a8), SPH_C32(0xb65573ff),
	  SPH_C32(0xbf8a16d5), SPH_C32(0x19fed62a), SPH_C32(0x35a90000),
	  SPH_C32(0x5bc80000), SPH_C32(0x95b10e80), SPH_C32(0x3e060000),
	  SPH_C32(0x67471384), SPH_C32(0xb1868180), SPH_C32(0x7f954a8e),
	  SPH_C32(0x2752bbaf) }
};

static const sph_u32 T512_39[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x52500000), SPH_C32(0x29540000), SPH_C32(0x6a61004e),
	  SPH_C32(0xf0ff0000), SPH_C32(0x9a317eec), SPH_C32(0x452341ce),
	  SPH_C32(0xcf568fe5), SPH_C32(0x5303130f), SPH_C32(0x538d0000),
	  SPH_C32(0xa9fc0000), SPH_C32(0x9ef70006), SPH_C32(0x56ff0000),
	  SPH_C32(0x0ae4004e), SPH_C32(0x92c5cdf9), SPH_C32(0xa9444018),
	  SPH_C32(0x7f975691) },
	{ SPH_C32(0x538d0000), SPH_C32(0xa9fc0000), SPH_C32(0x9ef70006),
	  SPH_C32(0x56ff0000), SPH_C32(0x0ae4004e), SPH_C32(0x92c5cdf9),
	  SPH_C32(0xa9444018), SPH_C32(0x7f975691), SPH_C32(0x01dd0000),
	  SPH_C32(0x80a80000), SPH_C32(0xf4960048), SPH_C32(0xa6000000),
	  SPH_C32(0x90d57ea2), SPH_C32(0xd7e68c37), SPH_C32(0x6612cffd),
	  SPH_C32(0x2c94459e) },
	{ SPH_C32(0x01dd0000), SPH_C32(0x80a80000), SPH_C32(0xf4960048),
	  SPH_C32(0xa6000000), SPH_C32(0x90d57ea2), SPH_C32(0xd7e68c37),
	  SPH_C32(0x6612cffd), SPH_C32(0x2c94459e), SPH_C32(0x52500000),
	  SPH_C32(0x29540000), SPH_C32(0x6a61004e), SPH_C32(0xf0ff0000),
	  SPH_C32(0x9a317eec), SPH_C32(0x452341ce), SPH_C32(0xcf568fe5),
	  SPH_C32(0x5303130f) },
	{ SPH_C32(0xcc140000), SPH_C32(0xa5630000), SPH_C32(0x5ab90780),
	  SPH_C32(0x3b500000), SPH_C32(0x4bd013ff), SPH_C32(0x879b3418),
	  SPH_C32(0x694348c1), SPH_C32(0xca5a87fe), SPH_C32(0x819e0000),
	  SPH_C32(0xec570000), SPH_C32(0x66320280), SPH_C32(0x95f30000),
	  SPH_C32(0x5da92802), SPH_C32(0x48f43cbc), SPH_C32(0xe65aa22d),
	  SPH_C32(0x8e67b7fa) },
	{ SPH_C32(0x9e440000), SPH_C32(0x8c370000), SPH_C32(0x30d807ce),
	  SPH_C32(0xcbaf0000), SPH_C32(0xd1e16d13), SPH_C32(0xc2b875d6),
	  SPH_C32(0xa615c724), SPH_C32(0x995994f1), SPH_C32(0xd2130000),
	  SPH_C32(0x45ab0000), SPH_C32(0xf8c50286), SPH_C32(0xc30c0000),
	  SPH_C32(0x574d284c), SPH_C32(0xda31f145), SPH_C32(0x4f1ee235),
	  SPH_C32(0xf1f0e16b) },
	{ SPH_C32(0x9f990000), SPH_C32(0x0c9f0000), SPH_C32(0xc44e0786),
	  SPH_C32(0x6daf0000), SPH_C32(0x413413b1), SPH_C32(0x155ef9e1),
	  SPH_C32(0xc00708d9), SPH_C32(0xb5cdd16f), SPH_C32(0x80430000),
	  SPH_C32(0x6cff0000), SPH_C32(0x92a402c8), SPH_C32(0x33f30000),
	  SPH_C32(0xcd7c56a0), SPH_C32(0x9f12b08b), SPH_C32(0x80486dd0),
	  SPH_C32(0xa2f3f264) },
	{ SPH_C32(0xcdc90000), SPH_C32(0x25cb0000), SPH_C32(0xae2f07c8),
	  SPH_C32(0x9d500000), SPH_C32(0xdb056d5d), SPH_C32(0x507db82f),
	  SPH_C32(0x0f51873c), SPH_C32(0xe6cec260), SPH_C32(0xd3ce0000),
	  SPH_C32(0xc5030000), SPH_C32(0x0c5302ce), SPH_C32(0x650c0000),
	  SPH_C32(0xc79856ee), SPH_C32(0x0dd77d72), SPH_C32(0x290c2dc8),
	  SPH_C32(0xdd64a4f5) }
};

static const sph_u32 T512_42[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x07ed0000), SPH_C32(0xb2500000), SPH_C32(0x8774000a),
	  SPH_C32(0x970d0000), SPH_C32(0x437223ae), SPH_C32(0x48c76ea4),
	  SPH_C32(0xf4786222), SPH_C32(0x9075b1ce), SPH_C32(0xa2d60000),
	  SPH_C32(0xa6760000), SPH_C32(0xc9440014), SPH_C32(0xeba30000),
	  SPH_C32(0xccec2e7b), SPH_C32(0x3018c499), SPH_C32(0x03490afa),
	  SPH_C32(0x9b6ef888) },
	{ SPH_C32(0x88980000), SPH_C32(0x1f940000), SPH_C32(0x7fcf002e),
	  SPH_C32(0xfb4e0000), SPH_C32(0xf158079a), SPH_C32(0x61ae9167),
	  SPH_C32(0xa895706c), SPH_C32(0xe6107494), SPH_C32(0x0bc20000),
	  SPH_C32(0xdb630000), SPH_C32(0x7e88000c), SPH_C32(0x15860000),
	  SPH_C32(0x91fd48f3), SPH_C32(0x7581bb43), SPH_C32(0xf460449e),
	  SPH_C32(0xd8b61463) },
	{ SPH_C32(0x8f750000), SPH_C32(0xadc40000), SPH_C32(0xf8bb0024),
	  SPH_C32(0x6c430000), SPH_C32(0xb22a2434), SPH_C32(0x2969ffc3),
	  SPH_C32(0x5ced124e), SPH_C32(0x7665c55a), SPH_C32(0xa9140000),
	  SPH_C32(0x7d150000), SPH_C32(0xb7cc0018), SPH_C32(0xfe250000),
	  SPH_C32(0x5d116688), SPH_C32(0x45997fda), SPH_C32(0xf7294e64),
	  SPH_C32(0x43d8eceb) },
	{ SPH_C32(0x0bc20000), SPH_C32(0xdb630000), SPH_C32(0x7e88000c),
	  SPH_C32(0x15860000), SPH_C32(0x91fd48f3), SPH_C32(0x7581bb43),
	  SPH_C32(0xf460449e), SPH_C32(0xd8b61463), SPH_C32(0x835a0000),
	  SPH_C32(0xc4f70000), SPH_C32(0x01470022), SPH_C32(0xeec80000),
	  SPH_C32(0x60a54f69), SPH_C32(0x142f2a24), SPH_C32(0x5cf534f2),
	  SPH_C32(0x3ea660f7) },
	{ SPH_C32(0x0c2f0000), SPH_C32(0x69330000), SPH_C32(0xf9fc0006),
	  SPH_C32(0x828b0000), SPH_C32(0xd28f6b5d), SPH_C32(0x3d46d5e7),
	  SPH_C32(0x001826bc), SPH_C32(0x48c3a5ad), SPH_C32(0x218c0000),
	  SPH_C32(0x62810000), SPH_C32(0xc8030036), SPH_C32(0x056b0000),
	  SPH_C32(0xac496112), SPH_C32(0x2437eebd), SPH_C32(0x5fbc3e08),
	  SPH_C32(0xa5c8987f) },
	{ SPH_C32(0x835a0000), SPH_C32(0xc4f70000), SPH_C32(0x01470022),
	  SPH_C32(0xeec80000), SPH_C32(0x60a54f69), SPH_C32(0x142f2a24),
	  SPH_C32(0x5cf534f2), SPH_C32(0x3ea660f7), SPH_C32(0x88980000),
	  SPH_C32(0x1f940000), SPH_C32(0x7fcf002e), SPH_C32(0xfb4e0000),
	  SPH_C32(0xf158079a), SPH_C32(0x61ae9167), SPH_C32(0xa895706c),
	  SPH_C32(0xe6107494) },
	{ SPH_C32(0x84b70000), SPH_C32(0x76a70000), SPH_C32(0x86330028),
	  SPH_C32(0x79c50000), SPH_C32(0x23d76cc7), SPH_C32(0x5ce84480),
	  SPH_C32(0xa88d56d0), SPH_C32(0xaed3d139), SPH_C32(0x2a4e0000),
	  SPH_C32(0xb9e20000), SPH_C32(0xb68b003a), SPH_C32(0x10ed0000),
	  SPH_C32(0x3db429e1), SPH_C32(0x51b655fe), SPH_C32(0xabdc7a96),
	  SPH_C32(0x7d7e8c1c) }
};

static const sph_u32 T512_45[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x58430000), SPH_C32(0x807e0000), SPH_C32(0x78330001),
	  SPH_C32(0xc66b3800), SPH_C32(0xe7375cdc), SPH_C32(0x79ad3fdd),
	  SPH_C32(0xac73fe6f), SPH_C32(0x3a4479b1), SPH_C32(0x1d5a0000),
	  SPH_C32(0x2b720000), SPH_C32(0x488d0000), SPH_C32(0xaf611800),
	  SPH_C32(0x25cb2ec5), SPH_C32(0xc879bfd0), SPH_C32(0x81a20429),
	  SPH_C32(0x1e7536a6) },
	{ SPH_C32(0x1d5a0000), SPH_C32(0x2b720000), SPH_C32(0x488d0000),
	  SPH_C32(0xaf611800), SPH_C32(0x25cb2ec5), SPH_C32(0xc879bfd0),
	  SPH_C32(0x81a20429), SPH_C32(0x1e7536a6), SPH_C32(0x45190000),
	  SPH_C32(0xab0c0000), SPH_C32(0x30be0001), SPH_C32(0x690a2000),
	  SPH_C32(0xc2fc7219), SPH_C32(0xb1d4800d), SPH_C32(0x2dd1fa46),
	  SPH_C32(0x24314f17) },
	{ SPH_C32(0x45190000), SPH_C32(0xab0c0000), SPH_C32(0x30be0001),
	  SPH_C32(0x690a2000), SPH_C32(0xc2fc7219), SPH_C32(0xb1d4800d),
	  SPH_C32(0x2dd1fa46), SPH_C32(0x24314f17), SPH_C32(0x58430000),
	  SPH_C32(0x807e0000), SPH_C32(0x78330001), SPH_C32(0xc66b3800),
	  SPH_C32(0xe7375cdc), SPH_C32(0x79ad3fdd), SPH_C32(0xac73fe6f),
	  SPH_C32(0x3a4479b1) },
	{ SPH_C32(0xa53b0000), SPH_C32(0x14260000), SPH_C32(0x4e30001e),
	  SPH_C32(0x7cae0000), SPH_C32(0x8f9e0dd5), SPH_C32(0x78dfaa3d),
	  SPH_C32(0xf73168d8), SPH_C32(0x0b1b4946), SPH_C32(0x07ed0000),
	  SPH_C32(0xb2500000), SPH_C32(0x8774000a), SPH_C32(0x970d0000),
	  SPH_C32(0x437223ae), SPH_C32(0x48c76ea4), SPH_C32(0xf4786222),
	  SPH_C32(0x9075b1ce) },
	{ SPH_C32(0xfd780000), SPH_C32(0x94580000), SPH_C32(0x3603001f),
	  SPH_C32(0xbac53800), SPH_C32(0x68a95109), SPH_C32(0x017295e0),
	  SPH_C32(0x5b4296b7), SPH_C32(0x315f30f7), SPH_C32(0x1ab70000),
	  SPH_C32(0x99220000), SPH_C32(0xcff9000a), SPH_C32(0x386c1800),
	  SPH_C32(0x66b90d6b), SPH_C32(0x80bed174), SPH_C32(0x75da660b),
	  SPH_C32(0x8e008768) },
	{ SPH_C32(0xb8610000), SPH_C32(0x3f540000), SPH_C32(0x06bd001e),
	  SPH_C32(0xd3cf1800), SPH_C32(0xaa552310), SPH_C32(0xb0a615ed),
	  SPH_C32(0x76936cf1), SPH_C32(0x156e7fe0), SPH_C32(0x42f40000),
	  SPH_C32(0x195c0000), SPH_C32(0xb7ca000b), SPH_C32(0xfe072000),
	  SPH_C32(0x818e51b7), SPH_C32(0xf913eea9), SPH_C32(0xd9a99864),
	  SPH_C32(0xb444fed9) },
	{ SPH_C32(0xe0220000), SPH_C32(0xbf2a0000), SPH_C32(0x7e8e001f),
	  SPH_C32(0x15a42000), SPH_C32(0x4d627fcc), SPH_C32(0xc90b2a30),
	  SPH_C32(0xdae0929e), SPH_C32(0x2f2a0651), SPH_C32(0x5fae0000),
	  SPH_C32(0x322e0000), SPH_C32(0xff47000b), SPH_C32(0x51663800),
	  SPH_C32(0xa4457f72), SPH_C32(0x316a5179), SPH_C32(0x580b9c4d),
	  SPH_C32(0xaa31c87f) }
};

static const sph_u32 T512_48[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xbadd0000), SPH_C32(0x13ad0000), SPH_C32(0xb7e70000),
	  SPH_C32(0xf7282800), SPH_C32(0xdf45144d), SPH_C32(0x361ac33a),
	  SPH_C32(0xea5a8d14), SPH_C32(0x2a2c18f0), SPH_C32(0xb82f0000),
	  SPH_C32(0xb12c0000), SPH_C32(0x30d80000), SPH_C32(0x14445000),
	  SPH_C32(0xc15860a2), SPH_C32(0x3127e8ec), SPH_C32(0x2e98bf23),
	  SPH_C32(0x551e3d6e) },
	{ SPH_C32(0x1e6c0000), SPH_C32(0xc4420000), SPH_C32(0x8a2e0000),
	  SPH_C32(0xbcb6b800), SPH_C32(0x2c4413b6), SPH_C32(0x8bfdd3da),
	  SPH_C32(0x6a0c1bc8), SPH_C32(0xb99dc2eb), SPH_C32(0x92560000),
	  SPH_C32(0x1eda0000), SPH_C32(0xea510000), SPH_C32(0xe8b13000),
	  SPH_C32(0xa93556a5), SPH_C32(0xebfb6199), SPH_C32(0xb15c2254),
	  SPH_C32(0x33c5244f) },
	{ SPH_C32(0xa4b10000), SPH_C32(0xd7ef0000), SPH_C32(0x3dc90000),
	  SPH_C32(0x4b9e9000), SPH_C32(0xf30107fb), SPH_C32(0xbde710e0),
	  SPH_C32(0x805696dc), SPH_C32(0x93b1da1b), SPH_C32(0x2a790000),
	  SPH_C32(0xaff60000), SPH_C32(0xda890000), SPH_C32(0xfcf56000),
	  SPH_C32(0x686d3607), SPH_C32(0xdadc8975), SPH_C32(0x9fc49d77),
	  SPH_C32(0x66db1921) },
	{ SPH_C32(0x92560000), SPH_C32(0x1eda0000), SPH_C32(0xea510000),
	  SPH_C32(0xe8b13000), SPH_C32(0xa93556a5), SPH_C32(0xebfb6199),
	  SPH_C32(0xb15c2254), SPH_C32(0x33c5244f), SPH_C32(0x8c3a0000),
	  SPH_C32(0xda980000), SPH_C32(0x607f0000), SPH_C32(0x54078800),
	  SPH_C32(0x85714513), SPH_C32(0x6006b243), SPH_C32(0xdb50399c),
	  SPH_C32(0x8a58e6a4) },
	{ SPH_C32(0x288b0000), SPH_C32(0x0d770000), SPH_C32(0x5db60000),
	  SPH_C32(0x1f991800), SPH_C32(0x767042e8), SPH_C32(0xdde1a2a3),
	  SPH_C32(0x5b06af40), SPH_C32(0x19e93cbf), SPH_C32(0x34150000),
	  SPH_C32(0x6bb40000), SPH_C32(0x50a70000), SPH_C32(0x4043d800),
	  SPH_C32(0x442925b1), SPH_C32(0x51215aaf), SPH_C32(0xf5c886bf),
	  SPH_C32(0xdf46dbca) },
	{ SPH_C32(0x8c3a0000), SPH_C32(0xda980000), SPH_C32(0x607f0000),
	  SPH_C32(0x54078800), SPH_C32(0x85714513), SPH_C32(0x6006b243),
	  SPH_C32(0xdb50399c), SPH_C32(0x8a58e6a4), SPH_C32(0x1e6c0000),
	  SPH_C32(0xc4420000), SPH_C32(0x8a2e0000), SPH_C32(0xbcb6b800),
	  SPH_C32(0x2c4413b6), SPH_C32(0x8bfdd3da), SPH_C32(0x6a0c1bc8),
	  SPH_C32(0xb99dc2eb) },
	{ SPH_C32(0x36e70000), SPH_C32(0xc9350000), SPH_C32(0xd7980000),
	  SPH_C32(0xa32fa000), SPH_C32(0x5a34515e), SPH_C32(0x561c7179),
	  SPH_C32(0x310ab488), SPH_C32(0xa074fe54), SPH_C32(0xa6430000),
	  SPH_C32(0x756e0000), SPH_C32(0xbaf60000), SPH_C32(0xa8f2e800),
	  SPH_C32(0xed1c7314), SPH_C32(0xbada3b36), SPH_C32(0x4494a4eb),
	  SPH_C32(0xec83ff85) }
};

static const sph_u32 T512_51[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xb4310000), SPH_C32(0x77330000), SPH_C32(0xb15d0000),
	  SPH_C32(0x7fd004e0), SPH_C32(0x78a26138), SPH_C32(0xd116c35d),
	  SPH_C32(0xd256d489), SPH_C32(0x4e6f74de), SPH_C32(0xe3060000),
	  SPH_C32(0xbdc10000), SPH_C32(0x87130000), SPH_C32(0xbff20060),
	  SPH_C32(0x2eba0a1a), SPH_C32(0x8db53751), SPH_C32(0x73c5ab06),
	  SPH_C32(0x5bd61539) },
	{ SPH_C32(0xe3060000), SPH_C32(0xbdc10000), SPH_C32(0x87130000),
	  SPH_C32(0xbff20060), SPH_C32(0x2eba0a1a), SPH_C32(0x8db53751),
	  SPH_C32(0x73c5ab06), SPH_C32(0x5bd61539), SPH_C32(0x57370000),
	  SPH_C32(0xcaf20000), SPH_C32(0x364e0000), SPH_C32(0xc0220480),
	  SPH_C32(0x56186b22), SPH_C32(0x5ca3f40c), SPH_C32(0xa1937f8f),
	  SPH_C32(0x15b961e7) },
	{ SPH_C32(0x57370000), SPH_C32(0xcaf20000), SPH_C32(0x364e0000),
	  SPH_C32(0xc0220480), SPH_C32(0x56186b22), SPH_C32(0x5ca3f40c),
	  SPH_C32(0xa1937f8f), SPH_C32(0x15b961e7), SPH_C32(0xb4310000),
	  SPH_C32(0x77330000), SPH_C32(0xb15d0000), SPH_C32(0x7fd004e0),
	  SPH_C32(0x78a26138), SPH_C32(0xd116c35d), SPH_C32(0xd256d489),
	  SPH_C32(0x4e6f74de) },
	{ SPH_C32(0x02f20000), SPH_C32(0xa2810000), SPH_C32(0x873f0000),
	  SPH_C32(0xe36c7800), SPH_C32(0x1e1d74ef), SPH_C32(0x073d2bd6),
	  SPH_C32(0xc4c23237), SPH_C32(0x7f32259e), SPH_C32(0xbadd0000),
	  SPH_C32(0x13ad0000), SPH_C32(0xb7e70000), SPH_C32(0xf7282800),
	  SPH_C32(0xdf45144d), SPH_C32(0x361ac33a), SPH_C32(0xea5a8d14),
	  SPH_C32(0x2a2c18f0) },
	{ SPH_C32(0xb6c30000), SPH_C32(0xd5b20000), SPH_C32(0x36620000),
	  SPH_C32(0x9cbc7ce0), SPH_C32(0x66bf15d7), SPH_C32(0xd62be88b),
	  SPH_C32(0x1694e6be), SPH_C32(0x315d5140), SPH_C32(0x59db0000),
	  SPH_C32(0xae6c0000), SPH_C32(0x30f40000), SPH_C32(0x48da2860),
	  SPH_C32(0xf1ff1e57), SPH_C32(0xbbaff46b), SPH_C32(0x999f2612),
	  SPH_C32(0x71fa0dc9) },
	{ SPH_C32(0xe1f40000), SPH_C32(0x1f400000), SPH_C32(0x002c0000),
	  SPH_C32(0x5c9e7860), SPH_C32(0x30a77ef5), SPH_C32(0x8a881c87),
	  SPH_C32(0xb7079931), SPH_C32(0x24e430a7), SPH_C32(0xedea0000),
	  SPH_C32(0xd95f0000), SPH_C32(0x81a90000), SPH_C32(0x370a2c80),
	  SPH_C32(0x895d7f6f), SPH_C32(0x6ab93736), SPH_C32(0x4bc9f29b),
	  SPH_C32(0x3f957917) },
	{ SPH_C32(0x55c50000), SPH_C32(0x68730000), SPH_C32(0xb1710000),
	  SPH_C32(0x234e7c80), SPH_C32(0x48051fcd), SPH_C32(0x5b9edfda),
	  SPH_C32(0x65514db8), SPH_C32(0x6a8b4479), SPH_C32(0x0eec0000),
	  SPH_C32(0x649e0000), SPH_C32(0x06ba0000), SPH_C32(0x88f82ce0),
	  SPH_C32(0xa7e77575), SPH_C32(0xe70c0067), SPH_C32(0x380c599d),
	  SPH_C32(0x64436c2e) }
};

static const sph_u32 T512_54[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x7b280000), SPH_C32(0x57420000), SPH_C32(0xa9e50000),
	  SPH_C32(0x634300a0), SPH_C32(0x9edb442f), SPH_C32(0x6d9995bb),
	  SPH_C32(0x27f83b03), SPH_C32(0xc7ff60f0), SPH_C32(0x95bb0000),
	  SPH_C32(0x81450000), SPH_C32(0x3b240000), SPH_C32(0x48db0140),
	  SPH_C32(0x0a8a6c53), SPH_C32(0x56f56eec), SPH_C32(0x62c91877),
	  SPH_C32(0xe7e00a94) },
	{ SPH_C32(0xe6280000), SPH_C32(0x4c4b0000), SPH_C32(0xa8550000),
	  SPH_C32(0xd3d002e0), SPH_C32(0xd86130b8), SPH_C32(0x98a7b0da),
	  SPH_C32(0x289506b4), SPH_C32(0xd75a4897), SPH_C32(0xf0c50000),
	  SPH_C32(0x59230000), SPH_C32(0x45820000), SPH_C32(0xe18d00c0),
	  SPH_C32(0x3b6d0631), SPH_C32(0xc2ed5699), SPH_C32(0xcbe0fe1c),
	  SPH_C32(0x56a7b19f) },
	{ SPH_C32(0x9d000000), SPH_C32(0x1b090000), SPH_C32(0x01b00000),
	  SPH_C32(0xb0930240), SPH_C32(0x46ba7497), SPH_C32(0xf53e2561),
	  SPH_C32(0x0f6d3db7), SPH_C32(0x10a52867), SPH_C32(0x657e0000),
	  SPH_C32(0xd8660000), SPH_C32(0x7ea60000), SPH_C32(0xa9560180),
	  SPH_C32(0x31e76a62), SPH_C32(0x94183875), SPH_C32(0xa929e66b),
	  SPH_C32(0xb147bb0b) },
	{ SPH_C32(0xf0c50000), SPH_C32(0x59230000), SPH_C32(0x45820000),
	  SPH_C32(0xe18d00c0), SPH_C32(0x3b6d0631), SPH_C32(0xc2ed5699),
	  SPH_C32(0xcbe0fe1c), SPH_C32(0x56a7b19f), SPH_C32(0x16ed0000),
	  SPH_C32(0x15680000), SPH_C32(0xedd70000), SPH_C32(0x325d0220),
	  SPH_C32(0xe30c3689), SPH_C32(0x5a4ae643), SPH_C32(0xe375f8a8),
	  SPH_C32(0x81fdf908) },
	{ SPH_C32(0x8bed0000), SPH_C32(0x0e610000), SPH_C32(0xec670000),
	  SPH_C32(0x82ce0060), SPH_C32(0xa5b6421e), SPH_C32(0xaf74c322),
	  SPH_C32(0xec18c51f), SPH_C32(0x9158d16f), SPH_C32(0x83560000),
	  SPH_C32(0x942d0000), SPH_C32(0xd6f30000), SPH_C32(0x7a860360),
	  SPH_C32(0xe9865ada), SPH_C32(0x0cbf88af), SPH_C32(0x81bce0df),
	  SPH_C32(0x661df39c) },
	{ SPH_C32(0x16ed0000), SPH_C32(0x15680000), SPH_C32(0xedd70000),
	  SPH_C32(0x325d0220), SPH_C32(0xe30c3689), SPH_C32(0x5a4ae643),
	  SPH_C32(0xe375f8a8), SPH_C32(0x81fdf908), SPH_C32(0xe6280000),
	  SPH_C32(0x4c4b0000), SPH_C32(0xa8550000), SPH_C32(0xd3d002e0),
	  SPH_C32(0xd86130b8), SPH_C32(0x98a7b0da), SPH_C32(0x289506b4),
	  SPH_C32(0xd75a4897) },
	{ SPH_C32(0x6dc50000), SPH_C32(0x422a0000), SPH_C32(0x44320000),
	  SPH_C32(0x511e0280), SPH_C32(0x7dd772a6), SPH_C32(0x37d373f8),
	  SPH_C32(0xc48dc3ab), SPH_C32(0x460299f8), SPH_C32(0x73930000),
	  SPH_C32(0xcd0e0000), SPH_C32(0x93710000), SPH_C32(0x9b0b03a0),
	  SPH_C32(0xd2eb5ceb), SPH_C32(0xce52de36), SPH_C32(0x4a5c1ec3),
	  SPH_C32(0x30ba4203) }
};

static const sph_u32 T512_57[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x5fa80000), SPH_C32(0x56030000), SPH_C32(0x43ae0000),
	  SPH_C32(0x64f30013), SPH_C32(0x257e86bf), SPH_C32(0x1311944e),
	  SPH_C32(0x541e95bf), SPH_C32(0x8ea4db69), SPH_C32(0x00440000),
	  SPH_C32(0x7f480000), SPH_C32(0xda7c0000), SPH_C32(0x2a230001),
	  SPH_C32(0x3badc9cc), SPH_C32(0xa9b69c87), SPH_C32(0x030a9e60),
	  SPH_C32(0xbe0a679e) },
	{ SPH_C32(0x00440000), SPH_C32(0x7f480000), SPH_C32(0xda7c0000),
	  SPH_C32(0x2a230001), SPH_C32(0x3badc9cc), SPH_C32(0xa9b69c87),
	  SPH_C32(0x030a9e60), SPH_C32(0xbe0a679e), SPH_C32(0x5fec0000),
	  SPH_C32(0x294b0000), SPH_C32(0x99d20000), SPH_C32(0x4ed00012),
	  SPH_C32(0x1ed34f73), SPH_C32(0xbaa708c9), SPH_C32(0x57140bdf),
	  SPH_C32(0x30aebcf7) },
	{ SPH_C32(0x5fec0000), SPH_C32(0x294b0000), SPH_C32(0x99d20000),
	  SPH_C32(0x4ed00012), SPH_C32(0x1ed34f73), SPH_C32(0xbaa708c9),
	  SPH_C32(0x57140bdf), SPH_C32(0x30aebcf7), SPH_C32(0x5fa80000),
	  SPH_C32(0x56030000), SPH_C32(0x43ae0000), SPH_C32(0x64f30013),
	  SPH_C32(0x257e86bf), SPH_C32(0x1311944e), SPH_C32(0x541e95bf),
	  SPH_C32(0x8ea4db69) },
	{ SPH_C32(0xee930000), SPH_C32(0xd6070000), SPH_C32(0x92c10000),
	  SPH_C32(0x2b9801e0), SPH_C32(0x9451287c), SPH_C32(0x3b6cfb57),
	  SPH_C32(0x45312374), SPH_C32(0x201f6a64), SPH_C32(0x7b280000),
	  SPH_C32(0x57420000), SPH_C32(0xa9e50000), SPH_C32(0x634300a0),
	  SPH_C32(0x9edb442f), SPH_C32(0x6d9995bb), SPH_C32(0x27f83b03),
	  SPH_C32(0xc7ff60f0) },
	{ SPH_C32(0xb13b0000), SPH_C32(0x80040000), SPH_C32(0xd16f0000),
	  SPH_C32(0x4f6b01f3), SPH_C32(0xb12faec3), SPH_C32(0x287d6f19),
	  SPH_C32(0x112fb6cb), SPH_C32(0xaebbb10d), SPH_C32(0x7b6c0000),
	  SPH_C32(0x280a0000), SPH_C32(0x73990000), SPH_C32(0x496000a1),
	  SPH_C32(0xa5768de3), SPH_C32(0xc42f093c), SPH_C32(0x24f2a563),
	  SPH_C32(0x79f5076e) },
	{ SPH_C32(0xeed70000), SPH_C32(0xa94f0000), SPH_C32(0x48bd0000),
	  SPH_C32(0x01bb01e1), SPH_C32(0xaffce1b0), SPH_C32(0x92da67d0),
	  SPH_C32(0x463bbd14), SPH_C32(0x9e150dfa), SPH_C32(0x24c40000),
	  SPH_C32(0x7e090000), SPH_C32(0x30370000), SPH_C32(0x2d9300b2),
	  SPH_C32(0x80080b5c), SPH_C32(0xd73e9d72), SPH_C32(0x70ec30dc),
	  SPH_C32(0xf751dc07) },
	{ SPH_C32(0xb17f0000), SPH_C32(0xff4c0000), SPH_C32(0x0b130000),
	  SPH_C32(0x654801f2), SPH_C32(0x8a82670f), SPH_C32(0x81cbf39e),
	  SPH_C32(0x122528ab), SPH_C32(0x10b1d693), SPH_C32(0x24800000),
	  SPH_C32(0x01410000), SPH_C32(0xea4b0000), SPH_C32(0x07b000b3),
	  SPH_C32(0xbba5c290), SPH_C32(0x7e8801f5), SPH_C32(0x73e6aebc),
	  SPH_C32(0x495bbb99) }
};

static const sph_u32 T512_60[8][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xa8da0000), SPH_C32(0x96be0000), SPH_C32(0x5c1d0000),
	  SPH_C32(0x07da0002), SPH_C32(0x7d669583), SPH_C32(0x1f98708a),
	  SPH_C32(0xbb668808), SPH_C32(0xda878000), SPH_C32(0xabe70000),
	  SPH_C32(0x9e0d0000), SPH_C32(0xaf270000), SPH_C32(0x3d180005),
	  SPH_C32(0x2c4f1fd3), SPH_C32(0x74f61695), SPH_C32(0xb5c347eb),
	  SPH_C32(0x3c5dfffe) },
	{ SPH_C32(0x01930000), SPH_C32(0xe7820000), SPH_C32(0xedfb0000),
	  SPH_C32(0xcf0c000b), SPH_C32(0x8dd08d58), SPH_C32(0xbca3b42e),
	  SPH_C32(0x063661e1), SPH_C32(0x536f9e7b), SPH_C32(0x92280000),
	  SPH_C32(0xdc850000), SPH_C32(0x57fa0000), SPH_C32(0x56dc0003),
	  SPH_C32(0xbae92316), SPH_C32(0x5aefa30c), SPH_C32(0x90cef752),
	  SPH_C32(0x7b1675d7) },
	{ SPH_C32(0xa9490000), SPH_C32(0x713c0000), SPH_C32(0xb1e60000),
	  SPH_C32(0xc8d60009), SPH_C32(0xf0b618db), SPH_C32(0xa33bc4a4),
	  SPH_C32(0xbd50e9e9), SPH_C32(0x89e81e7b), SPH_C32(0x39cf0000),
	  SPH_C32(0x42880000), SPH_C32(0xf8dd0000), SPH_C32(0x6bc40006),
	  SPH_C32(0x96a63cc5), SPH_C32(0x2e19b599), SPH_C32(0x250db0b9),
	  SPH_C32(0x474b8a29) },
	{ SPH_C32(0x92280000), SPH_C32(0xdc850000), SPH_C32(0x57fa0000),
	  SPH_C32(0x56dc0003), SPH_C32(0xbae92316), SPH_C32(0x5aefa30c),
	  SPH_C32(0x90cef752), SPH_C32(0x7b1675d7), SPH_C32(0x93bb0000),
	  SPH_C32(0x3b070000), SPH_C32(0xba010000), SPH_C32(0x99d00008),
	  SPH_C32(0x3739ae4e), SPH_C32(0xe64c1722), SPH_C32(0x96f896b3),
	  SPH_C32(0x2879ebac) },
	{ SPH_C32(0x3af20000), SPH_C32(0x4a3b0000), SPH_C32(0x0be70000),
	  SPH_C32(0x51060001), SPH_C32(0xc78fb695), SPH_C32(0x4577d386),
	  SPH_C32(0x2ba87f5a), SPH_C32(0xa191f5d7), SPH_C32(0x385c0000),
	  SPH_C32(0xa50a0000), SPH_C32(0x15260000), SPH_C32(0xa4c8000d),
	  SPH_C32(0x1b76b19d), SPH_C32(0x92ba01b7), SPH_C32(0x233bd158),
	  SPH_C32(0x14241452) },
	{ SPH_C32(0x93bb0000), SPH_C32(0x3b070000), SPH_C32(0xba010000),
	  SPH_C32(0x99d00008), SPH_C32(0x3739ae4e), SPH_C32(0xe64c1722),
	  SPH_C32(0x96f896b3), SPH_C32(0x2879ebac), SPH_C32(0x01930000),
	  SPH_C32(0xe7820000), SPH_C32(0xedfb0000), SPH_C32(0xcf0c000b),
	  SPH_C32(0x8dd08d58), SPH_C32(0xbca3b42e), SPH_C32(0x063661e1),
	  SPH_C32(0x536f9e7b) },
	{ SPH_C32(0x3b610000), SPH_C32(0xadb90000), SPH_C32(0xe61c0000),
	  SPH_C32(0x9e0a000a), SPH_C32(0x4a5f3bcd), SPH_C32(0xf9d467a8),
	  SPH_C32(0x2d9e1ebb), SPH_C32(0xf2fe6bac), SPH_C32(0xaa740000),
	  SPH_C32(0x798f0000), SPH_C32(0x42dc0000), SPH_C32(0xf214000e),
	  SPH_C32(0xa19f928b), SPH_C32(0xc855a2bb), SPH_C32(0xb3f5260a),
	  SPH_C32(0x6f326185) }
};

static const sph_u32 T512_63[2][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x033d0000), SPH_C32(0x08b30000), SPH_C32(0xf33a0000),
	  SPH_C32(0x3ac20007), SPH_C32(0x51298a50), SPH_C32(0x6b6e661f),
	  SPH_C32(0x0ea5cfe3), SPH_C32(0xe6da7ffe), SPH_C32(0xa8da0000),
	  SPH_C32(0x96be0000), SPH_C32(0x5c1d0000), SPH_C32(0x07da0002),
	  SPH_C32(0x7d669583), SPH_C32(0x1f98708a), SPH_C32(0xbb668808),
	  SPH_C32(0xda878000) }
};

#define INPUT_BIG   do { \
		unsigned acc = buf[0]; \
		const sph_u32 *rp; \
		rp = &T512_0[acc >> 5][0]; \
		m0 = rp[0]; \
		m1 = rp[1]; \
		m2 = rp[2]; \
		m3 = rp[3]; \
		m4 = rp[4]; \
		m5 = rp[5]; \
		m6 = rp[6]; \
		m7 = rp[7]; \
		m8 = rp[8]; \
		m9 = rp[9]; \
		mA = rp[10]; \
		mB = rp[11]; \
		mC = rp[12]; \
		mD = rp[13]; \
		mE = rp[14]; \
		mF = rp[15]; \
		rp = &T512_3[(acc >> 2) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = (acc << 8) | buf[1]; \
		rp = &T512_6[(acc >> 7) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_9[(acc >> 4) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_12[(acc >> 1) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = (acc << 8) | buf[2]; \
		rp = &T512_15[(acc >> 6) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_18[(acc >> 3) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_21[acc & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = buf[3]; \
		rp = &T512_24[acc >> 5][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_27[(acc >> 2) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = (acc << 8) | buf[4]; \
		rp = &T512_30[(acc >> 7) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_33[(acc >> 4) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_36[(acc >> 1) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = (acc << 8) | buf[5]; \
		rp = &T512_39[(acc >> 6) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_42[(acc >> 3) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_45[acc & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = buf[6]; \
		rp = &T512_48[acc >> 5][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_51[(acc >> 2) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = (acc << 8) | buf[7]; \
		rp = &T512_54[(acc >> 7) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_57[(acc >> 4) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_60[(acc >> 1) & 0x07][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_63[acc & 0x01][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
	} while (0)

#endif

#if SPH_HAMSI_EXPAND_BIG == 4

static const sph_u32 T512_0[16][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x54285c00), SPH_C32(0xeaed0000), SPH_C32(0xc5d60000),
	  SPH_C32(0xa1c50000), SPH_C32(0xb3a26770), SPH_C32(0x94a5c4e1),
	  SPH_C32(0x6bb0419d), SPH_C32(0x551b3782), SPH_C32(0x9cbb1800),
	  SPH_C32(0xb0d30000), SPH_C32(0x92510000), SPH_C32(0xed930000),
	  SPH_C32(0x593a4345), SPH_C32(0xe114d5f4), SPH_C32(0x430633da),
	  SPH_C32(0x78cace29) },
	{ SPH_C32(0x9cbb1800), SPH_C32(0xb0d30000), SPH_C32(0x92510000),
	  SPH_C32(0xed930000), SPH_C32(0x593a4345), SPH_C32(0xe114d5f4),
	  SPH_C32(0x430633da), SPH_C32(0x78cace29), SPH_C32(0xc8934400),
	  SPH_C32(0x5a3e0000), SPH_C32(0x57870000), SPH_C32(0x4c560000),
	  SPH_C32(0xea982435), SPH_C32(0x75b11115), SPH_C32(0x28b67247),
	  SPH_C32(0x2dd1f9ab) },
	{ SPH_C32(0xc8934400), SPH_C32(0x5a3e0000), SPH_C32(0x57870000),
	  SPH_C32(0x4c560000), SPH_C32(0xea982435), SPH_C32(0x75b11115),
	  SPH_C32(0x28b67247), SPH_C32(0x2dd1f9ab), SPH_C32(0x54285c00),
	  SPH_C32(0xeaed0000), SPH_C32(0xc5d60000), SPH_C32(0xa1c50000),
	  SPH_C32(0xb3a26770), SPH_C32(0x94a5c4e1), SPH_C32(0x6bb0419d),
	  SPH_C32(0x551b3782) },
	{ SPH_C32(0x29449c00), SPH_C32(0x64e70000), SPH_C32(0xf24b0000),
	  SPH_C32(0xc2f30000), SPH_C32(0x0ede4e8f), SPH_C32(0x56c23745),
	  SPH_C32(0xf3e04259), SPH_C32(0x8d0d9ec4), SPH_C32(0x466d0c00),
	  SPH_C32(0x08620000), SPH_C32(0xdd5d0000), SPH_C32(0xbadd0000),
	  SPH_C32(0x6a927942), SPH_C32(0x441f2b93), SPH_C32(0x218ace6f),
	  SPH_C32(0xbf2c0be2) },
	{ SPH_C32(0x7d6cc000), SPH_C32(0x8e0a0000), SPH_C32(0x379d0000),
	  SPH_C32(0x63360000), SPH_C32(0xbd7c29ff), SPH_C32(0xc267f3a4),
	  SPH_C32(0x985003c4), SPH_C32(0xd816a946), SPH_C32(0xdad61400),
	  SPH_C32(0xb8b10000), SPH_C32(0x4f0c0000), SPH_C32(0x574e0000),
	  SPH_C32(0x33a83a07), SPH_C32(0xa50bfe67), SPH_C32(0x628cfdb5),
	  SPH_C32(0xc7e6c5cb) },
	{ SPH_C32(0xb5ff8400), SPH_C32(0xd4340000), SPH_C32(0x601a0000),
	  SPH_C32(0x2f600000), SPH_C32(0x57e40dca), SPH_C32(0xb7d6e2b1),
	  SPH_C32(0xb0e67183), SPH_C32(0xf5c750ed), SPH_C32(0x8efe4800),
	  SPH_C32(0x525c0000), SPH_C32(0x8ada0000), SPH_C32(0xf68b0000),
	  SPH_C32(0x800a5d77), SPH_C32(0x31ae3a86), SPH_C32(0x093cbc28),
	  SPH_C32(0x92fdf249) },
	{ SPH_C32(0xe1d7d800), SPH_C32(0x3ed90000), SPH_C32(0xa5cc0000),
	  SPH_C32(0x8ea50000), SPH_C32(0xe4466aba), SPH_C32(0x23732650),
	  SPH_C32(0xdb56301e), SPH_C32(0xa0dc676f), SPH_C32(0x12455000),
	  SPH_C32(0xe28f0000), SPH_C32(0x188b0000), SPH_C32(0x1b180000),
	  SPH_C32(0xd9301e32), SPH_C32(0xd0baef72), SPH_C32(0x4a3a8ff2),
	  SPH_C32(0xea373c60) },
	{ SPH_C32(0x466d0c00), SPH_C32(0x08620000), SPH_C32(0xdd5d0000),
	  SPH_C32(0xbadd0000), SPH_C32(0x6a927942), SPH_C32(0x441f2b93),
	  SPH_C32(0x218ace6f), SPH_C32(0xbf2c0be2), SPH_C32(0x6f299000),
	  SPH_C32(0x6c850000), SPH_C32(0x2f160000), SPH_C32(0x782e0000),
	  SPH_C32(0x644c37cd), SPH_C32(0x12dd1cd6), SPH_C32(0xd26a8c36),
	  SPH_C32(0x32219526) },
	{ SPH_C32(0x12455000), SPH_C32(0xe28f0000), SPH_C32(0x188b0000),
	  SPH_C32(0x1b180000), SPH_C32(0xd9301e32), SPH_C32(0xd0baef72),
	  SPH_C32(0x4a3a8ff2), SPH_C32(0xea373c60), SPH_C32(0xf3928800),
	  SPH_C32(0xdc560000), SPH_C32(0xbd470000), SPH_C32(0x95bd0000),
	  SPH_C32(0x3d767488), SPH_C32(0xf3c9c922), SPH_C32(0x916cbfec),
	  SPH_C32(0x4aeb5b0f) },
	{ SPH_C32(0xdad61400), SPH_C32(0xb8b10000), SPH_C32(0x4f0c0000),
	  SPH_C32(0x574e0000), SPH_C32(0x33a83a07), SPH_C32(0xa50bfe67),
	  SPH_C32(0x628cfdb5), SPH_C32(0xc7e6c5cb), SPH_C32(0xa7bad400),
	  SPH_C32(0x36bb0000), SPH_C32(0x78910000), SPH_C32(0x34780000),
	  SPH_C32(0x8ed413f8), SPH_C32(0x676c0dc3), SPH_C32(0xfadcfe71),
	  SPH_C32(0x1ff06c8d) },
	{ SPH_C32(0x8efe4800), SPH_C32(0x525c0000), SPH_C32(0x8ada0000),
	  SPH_C32(0xf68b0000), SPH_C32(0x800a5d77), SPH_C32(0x31ae3a86),
	  SPH_C32(0x093cbc28), SPH_C32(0x92fdf249), SPH_C32(0x3b01cc00),
	  SPH_C32(0x86680000), SPH_C32(0xeac00000), SPH_C32(0xd9eb0000),
	  SPH_C32(0xd7ee50bd), SPH_C32(0x8678d837), SPH_C32(0xb9dacdab),
	  SPH_C32(0x673aa2a4) },
	{ SPH_C32(0x6f299000), SPH_C32(0x6c850000), SPH_C32(0x2f160000),
	  SPH_C32(0x782e0000), SPH_C32(0x644c37cd), SPH_C32(0x12dd1cd6),
	  SPH_C32(0xd26a8c36), SPH_C32(0x32219526), SPH_C32(0x29449c00),
	  SPH_C32(0x64e70000), SPH_C32(0xf24b0000), SPH_C32(0xc2f30000),
	  SPH_C32(0x0ede4e8f), SPH_C32(0x56c23745), SPH_C32(0xf3e04259),
	  SPH_C32(0x8d0d9ec4) },
	{ SPH_C32(0x3b01cc00), SPH_C32(0x86680000), SPH_C32(0xeac00000),
	  SPH_C32(0xd9eb0000), SPH_C32(0xd7ee50bd), SPH_C32(0x8678d837),
	  SPH_C32(0xb9dacdab), SPH_C32(0x673aa2a4), SPH_C32(0xb5ff8400),
	  SPH_C32(0xd4340000), SPH_C32(0x601a0000), SPH_C32(0x2f600000),
	  SPH_C32(0x57e40dca), SPH_C32(0xb7d6e2b1), SPH_C32(0xb0e67183),
	  SPH_C32(0xf5c750ed) },
	{ SPH_C32(0xf3928800), SPH_C32(0xdc560000), SPH_C32(0xbd470000),
	  SPH_C32(0x95bd0000), SPH_C32(0x3d767488), SPH_C32(0xf3c9c922),
	  SPH_C32(0x916cbfec), SPH_C32(0x4aeb5b0f), SPH_C32(0xe1d7d800),
	  SPH_C32(0x3ed90000), SPH_C32(0xa5cc0000), SPH_C32(0x8ea50000),
	  SPH_C32(0xe4466aba), SPH_C32(0x23732650), SPH_C32(0xdb56301e),
	  SPH_C32(0xa0dc676f) },
	{ SPH_C32(0xa7bad400), SPH_C32(0x36bb0000), SPH_C32(0x78910000),
	  SPH_C32(0x34780000), SPH_C32(0x8ed413f8), SPH_C32(0x676c0dc3),
	  SPH_C32(0xfadcfe71), SPH_C32(0x1ff06c8d), SPH_C32(0x7d6cc000),
	  SPH_C32(0x8e0a0000), SPH_C32(0x379d0000), SPH_C32(0x63360000),
	  SPH_C32(0xbd7c29ff), SPH_C32(0xc267f3a4), SPH_C32(0x985003c4),
	  SPH_C32(0xd816a946) }
};

static const sph_u32 T512_4[16][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xef0b0270), SPH_C32(0x3afd0000), SPH_C32(0x5dae0000),
	  SPH_C32(0x69490000), SPH_C32(0x9b0f3c06), SPH_C32(0x4405b5f9),
	  SPH_C32(0x66140a51), SPH_C32(0x924f5d0a), SPH_C32(0xc96b0030),
	  SPH_C32(0xe7250000), SPH_C32(0x2f840000), SPH_C32(0x264f0000),
	  SPH_C32(0x08695bf9), SPH_C32(0x6dfcf137), SPH_C32(0x509f6984),
	  SPH_C32(0x9e69af68) },
	{ SPH_C32(0xc96b0030), SPH_C32(0xe7250000), SPH_C32(0x2f840000),
	  SPH_C32(0x264f0000), SPH_C32(0x08695bf9), SPH_C32(0x6dfcf137),
	  SPH_C32(0x509f6984), SPH_C32(0x9e69af68), SPH_C32(0x26600240),
	  SPH_C32(0xddd80000), SPH_C32(0x722a0000), SPH_C32(0x4f060000),
	  SPH_C32(0x936667ff), SPH_C32(0x29f944ce), SPH_C32(0x368b63d5),
	  SPH_C32(0x0c26f262) },
	{ SPH_C32(0x26600240), SPH_C32(0xddd80000), SPH_C32(0x722a0000),
	  SPH_C32(0x4f060000), SPH_C32(0x936667ff), SPH_C32(0x29f944ce),
	  SPH_C32(0x368b63d5), SPH_C32(0x0c26f262), SPH_C32(0xef0b0270),
	  SPH_C32(0x3afd0000), SPH_C32(0x5dae0000), SPH_C32(0x69490000),
	  SPH_C32(0x9b0f3c06), SPH_C32(0x4405b5f9), SPH_C32(0x66140a51),
	  SPH_C32(0x924f5d0a) },
	{ SPH_C32(0x145a3c00), SPH_C32(0xb9e90000), SPH_C32(0x61270000),
	  SPH_C32(0xf1610000), SPH_C32(0xce613d6c), SPH_C32(0xb0493d78),
	  SPH_C32(0x47a96720), SPH_C32(0xe18e24c5), SPH_C32(0x23671400),
	  SPH_C32(0xc8b90000), SPH_C32(0xf4c70000), SPH_C32(0xfb750000),
	  SPH_C32(0x73cd2465), SPH_C32(0xf8a6a549), SPH_C32(0x02c40a3f),
	  SPH_C32(0xdc24e61f) },
	{ SPH_C32(0xfb513e70), SPH_C32(0x83140000), SPH_C32(0x3c890000),
	  SPH_C32(0x98280000), SPH_C32(0x556e016a), SPH_C32(0xf44c8881),
	  SPH_C32(0x21bd6d71), SPH_C32(0x73c179cf), SPH_C32(0xea0c1430),
	  SPH_C32(0x2f9c0000), SPH_C32(0xdb430000), SPH_C32(0xdd3a0000),
	  SPH_C32(0x7ba47f9c), SPH_C32(0x955a547e), SPH_C32(0x525b63bb),
	  SPH_C32(0x424d4977) },
	{ SPH_C32(0xdd313c30), SPH_C32(0x5ecc0000), SPH_C32(0x4ea30000),
	  SPH_C32(0xd72e0000), SPH_C32(0xc6086695), SPH_C32(0xddb5cc4f),
	  SPH_C32(0x17360ea4), SPH_C32(0x7fe78bad), SPH_C32(0x05071640),
	  SPH_C32(0x15610000), SPH_C32(0x86ed0000), SPH_C32(0xb4730000),
	  SPH_C32(0xe0ab439a), SPH_C32(0xd15fe187), SPH_C32(0x344f69ea),
	  SPH_C32(0xd002147d) },
	{ SPH_C32(0x323a3e40), SPH_C32(0x64310000), SPH_C32(0x130d0000),
	  SPH_C32(0xbe670000), SPH_C32(0x5d075a93), SPH_C32(0x99b079b6),
	  SPH_C32(0x712204f5), SPH_C32(0xeda8d6a7), SPH_C32(0xcc6c1670),
	  SPH_C32(0xf2440000), SPH_C32(0xa9690000), SPH_C32(0x923c0000),
	  SPH_C32(0xe8c21863), SPH_C32(0xbca310b0), SPH_C32(0x64d0006e),
	  SPH_C32(0x4e6bbb15) },
	{ SPH_C32(0x23671400), SPH_C32(0xc8b90000), SPH_C32(0xf4c70000),
	  SPH_C32(0xfb750000), SPH_C32(0x73cd2465), SPH_C32(0xf8a6a549),
	  SPH_C32(0x02c40a3f), SPH_C32(0xdc24e61f), SPH_C32(0x373d2800),
	  SPH_C32(0x71500000), SPH_C32(0x95e00000), SPH_C32(0x0a140000),
	  SPH_C32(0xbdac1909), SPH_C32(0x48ef9831), SPH_C32(0x456d6d1f),
	  SPH_C32(0x3daac2da) },
	{ SPH_C32(0xcc6c1670), SPH_C32(0xf2440000), SPH_C32(0xa9690000),
	  SPH_C32(0x923c0000), SPH_C32(0xe8c21863), SPH_C32(0xbca310b0),
	  SPH_C32(0x64d0006e), SPH_C32(0x4e6bbb15), SPH_C32(0xfe562830),
	  SPH_C32(0x96750000), SPH_C32(0xba640000), SPH_C32(0x2c5b0000),
	  SPH_C32(0xb5c542f0), SPH_C32(0x25136906), SPH_C32(0x15f2049b),
	  SPH_C32(0xa3c36db2) },
	{ SPH_C32(0xea0c1430), SPH_C32(0x2f9c0000), SPH_C32(0xdb430000),
	  SPH_C32(0xdd3a0000), SPH_C32(0x7ba47f9c), SPH_C32(0x955a547e),
	  SPH_C32(0x525b63bb), SPH_C32(0x424d4977), SPH_C32(0x115d2a40),
	  SPH_C32(0xac880000), SPH_C32(0xe7ca0000), SPH_C32(0x45120000),
	  SPH_C32(0x2eca7ef6), SPH_C32(0x6116dcff), SPH_C32(0x73e60eca),
	  SPH_C32(0x318c30b8) },
	{ SPH_C32(0x05071640), SPH_C32(0x15610000), SPH_C32(0x86ed0000),
	  SPH_C32(0xb4730000), SPH_C32(0xe0ab439a), SPH_C32(0xd15fe187),
	  SPH_C32(0x344f69ea), SPH_C32(0xd002147d), SPH_C32(0xd8362a70),
	  SPH_C32(0x4bad0000), SPH_C32(0xc84e0000), SPH_C32(0x635d0000),
	  SPH_C32(0x26a3250f), SPH_C32(0x0cea2dc8), SPH_C32(0x2379674e),
	  SPH_C32(0xafe59fd0) },
	{ SPH_C32(0x373d2800), SPH_C32(0x71500000), SPH_C32(0x95e00000),
	  SPH_C32(0x0a140000), SPH_C32(0xbdac1909), SPH_C32(0x48ef9831),
	  SPH_C32(0x456d6d1f), SPH_C32(0x3daac2da), SPH_C32(0x145a3c00),
	  SPH_C32(0xb9e90000), SPH_C32(0x61270000), SPH_C32(0xf1610000),
	  SPH_C32(0xce613d6c), SPH_C32(0xb0493d78), SPH_C32(0x47a96720),
	  SPH_C32(0xe18e24c5) },
	{ SPH_C32(0xd8362a70), SPH_C32(0x4bad0000), SPH_C32(0xc84e0000),
	  SPH_C32(0x635d0000), SPH_C32(0x26a3250f), SPH_C32(0x0cea2dc8),
	  SPH_C32(0x2379674e), SPH_C32(0xafe59fd0), SPH_C32(0xdd313c30),
	  SPH_C32(0x5ecc0000), SPH_C32(0x4ea30000), SPH_C32(0xd72e0000),
	  SPH_C32(0xc6086695), SPH_C32(0xddb5cc4f), SPH_C32(0x17360ea4),
	  SPH_C32(0x7fe78bad) },
	{ SPH_C32(0xfe562830), SPH_C32(0x96750000), SPH_C32(0xba640000),
	  SPH_C32(0x2c5b0000), SPH_C32(0xb5c542f0), SPH_C32(0x25136906),
	  SPH_C32(0x15f2049b), SPH_C32(0xa3c36db2), SPH_C32(0x323a3e40),
	  SPH_C32(0x64310000), SPH_C32(0x130d0000), SPH_C32(0xbe670000),
	  SPH_C32(0x5d075a93), SPH_C32(0x99b079b6), SPH_C32(0x712204f5),
	  SPH_C32(0xeda8d6a7) },
	{ SPH_C32(0x115d2a40), SPH_C32(0xac880000), SPH_C32(0xe7ca0000),
	  SPH_C32(0x45120000), SPH_C32(0x2eca7ef6), SPH_C32(0x6116dcff),
	  SPH_C32(0x73e60eca), SPH_C32(0x318c30b8), SPH_C32(0xfb513e70),
	  SPH_C32(0x83140000), SPH_C32(0x3c890000), SPH_C32(0x98280000),
	  SPH_C32(0x556e016a), SPH_C32(0xf44c8881), SPH_C32(0x21bd6d71),
	  SPH_C32(0x73c179cf) }
};

static const sph_u32 T512_8[16][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x774400f0), SPH_C32(0xf15a0000), SPH_C32(0xf5b20000),
	  SPH_C32(0x34140000), SPH_C32(0x89377e8c), SPH_C32(0x5a8bec25),
	  SPH_C32(0x0bc3cd1e), SPH_C32(0xcf3775cb), SPH_C32(0xf46c0050),
	  SPH_C32(0x96180000), SPH_C32(0x14a50000), SPH_C32(0x031f0000),
	  SPH_C32(0x42947eb8), SPH_C32(0x66bf7e19), SPH_C32(0x9ca470d2),
	  SPH_C32(0x8a341574) },
	{ SPH_C32(0xf46c0050), SPH_C32(0x96180000), SPH_C32(0x14a50000),
	  SPH_C32(0x031f0000), SPH_C32(0x42947eb8), SPH_C32(0x66bf7e19),
	  SPH_C32(0x9ca470d2), SPH_C32(0x8a341574), SPH_C32(0x832800a0),
	  SPH_C32(0x67420000), SPH_C32(0xe1170000), SPH_C32(0x370b0000),
	  SPH_C32(0xcba30034), SPH_C32(0x3c34923c), SPH_C32(0x9767bdcc),
	  SPH_C32(0x450360bf) },
	{ SPH_C32(0x832800a0), SPH_C32(0x67420000), SPH_C32(0xe1170000),
	  SPH_C32(0x370b0000), SPH_C32(0xcba30034), SPH_C32(0x3c34923c),
	  SPH_C32(0x9767bdcc), SPH_C32(0x450360bf), SPH_C32(0x774400f0),
	  SPH_C32(0xf15a0000), SPH_C32(0xf5b20000), SPH_C32(0x34140000),
	  SPH_C32(0x89377e8c), SPH_C32(0x5a8bec25), SPH_C32(0x0bc3cd1e),
	  SPH_C32(0xcf3775cb) },
	{ SPH_C32(0xe8870170), SPH_C32(0x9d720000), SPH_C32(0x12db0000),
	  SPH_C32(0xd4220000), SPH_C32(0xf2886b27), SPH_C32(0xa921e543),
	  SPH_C32(0x4ef8b518), SPH_C32(0x618813b1), SPH_C32(0xb4370060),
	  SPH_C32(0x0c4c0000), SPH_C32(0x56c20000), SPH_C32(0x5cae0000),
	  SPH_C32(0x94541f3f), SPH_C32(0x3b3ef825), SPH_C32(0x1b365f3d),
	  SPH_C32(0xf3d45758) },
	{ SPH_C32(0x9fc30180), SPH_C32(0x6c280000), SPH_C32(0xe7690000),
	  SPH_C32(0xe0360000), SPH_C32(0x7bbf15ab), SPH_C32(0xf3aa0966),
	  SPH_C32(0x453b7806), SPH_C32(0xaebf667a), SPH_C32(0x405b0030),
	  SPH_C32(0x9a540000), SPH_C32(0x42670000), SPH_C32(0x5fb10000),
	  SPH_C32(0xd6c06187), SPH_C32(0x5d81863c), SPH_C32(0x87922fef),
	  SPH_C32(0x79e0422c) },
	{ SPH_C32(0x1ceb0120), SPH_C32(0x0b6a0000), SPH_C32(0x067e0000),
	  SPH_C32(0xd73d0000), SPH_C32(0xb01c159f), SPH_C32(0xcf9e9b5a),
	  SPH_C32(0xd25cc5ca), SPH_C32(0xebbc06c5), SPH_C32(0x371f00c0),
	  SPH_C32(0x6b0e0000), SPH_C32(0xb7d50000), SPH_C32(0x6ba50000),
	  SPH_C32(0x5ff71f0b), SPH_C32(0x070a6a19), SPH_C32(0x8c51e2f1),
	  SPH_C32(0xb6d737e7) },
	{ SPH_C32(0x6baf01d0), SPH_C32(0xfa300000), SPH_C32(0xf3cc0000),
	  SPH_C32(0xe3290000), SPH_C32(0x392b6b13), SPH_C32(0x9515777f),
	  SPH_C32(0xd99f08d4), SPH_C32(0x248b730e), SPH_C32(0xc3730090),
	  SPH_C32(0xfd160000), SPH_C32(0xa3700000), SPH_C32(0x68ba0000),
	  SPH_C32(0x1d6361b3), SPH_C32(0x61b51400), SPH_C32(0x10f59223),
	  SPH_C32(0x3ce32293) },
	{ SPH_C32(0xb4370060), SPH_C32(0x0c4c0000), SPH_C32(0x56c20000),
	  SPH_C32(0x5cae0000), SPH_C32(0x94541f3f), SPH_C32(0x3b3ef825),
	  SPH_C32(0x1b365f3d), SPH_C32(0xf3d45758), SPH_C32(0x5cb00110),
	  SPH_C32(0x913e0000), SPH_C32(0x44190000), SPH_C32(0x888c0000),
	  SPH_C32(0x66dc7418), SPH_C32(0x921f1d66), SPH_C32(0x55ceea25),
	  SPH_C32(0x925c44e9) },
	{ SPH_C32(0xc3730090), SPH_C32(0xfd160000), SPH_C32(0xa3700000),
	  SPH_C32(0x68ba0000), SPH_C32(0x1d6361b3), SPH_C32(0x61b51400),
	  SPH_C32(0x10f59223), SPH_C32(0x3ce32293), SPH_C32(0xa8dc0140),
	  SPH_C32(0x07260000), SPH_C32(0x50bc0000), SPH_C32(0x8b930000),
	  SPH_C32(0x24480aa0), SPH_C32(0xf4a0637f), SPH_C32(0xc96a9af7),
	  SPH_C32(0x1868519d) },
	{ SPH_C32(0x405b0030), SPH_C32(0x9a540000), SPH_C32(0x42670000),
	  SPH_C32(0x5fb10000), SPH_C32(0xd6c06187), SPH_C32(0x5d81863c),
	  SPH_C32(0x87922fef), SPH_C32(0x79e0422c), SPH_C32(0xdf9801b0),
	  SPH_C32(0xf67c0000), SPH_C32(0xa50e0000), SPH_C32(0xbf870000),
	  SPH_C32(0xad7f742c), SPH_C32(0xae2b8f5a), SPH_C32(0xc2a957e9),
	  SPH_C32(0xd75f2456) },
	{ SPH_C32(0x371f00c0), SPH_C32(0x6b0e0000), SPH_C32(0xb7d50000),
	  SPH_C32(0x6ba50000), SPH_C32(0x5ff71f0b), SPH_C32(0x070a6a19),
	  SPH_C32(0x8c51e2f1), SPH_C32(0xb6d737e7), SPH_C32(0x2bf401e0),
	  SPH_C32(0x60640000), SPH_C32(0xb1ab0000), SPH_C32(0xbc980000),
	  SPH_C32(0xefeb0a94), SPH_C32(0xc894f143), SPH_C32(0x5e0d273b),
	  SPH_C32(0x5d6b3122) },
	{ SPH_C32(0x5cb00110), SPH_C32(0x913e0000), SPH_C32(0x44190000),
	  SPH_C32(0x888c0000), SPH_C32(0x66dc7418), SPH_C32(0x921f1d66),
	  SPH_C32(0x55ceea25), SPH_C32(0x925c44e9), SPH_C32(0xe8870170),
	  SPH_C32(0x9d720000), SPH_C32(0x12db0000), SPH_C32(0xd4220000),
	  SPH_C32(0xf2886b27), SPH_C32(0xa921e543), SPH_C32(0x4ef8b518),
	  SPH_C32(0x618813b1) },
	{ SPH_C32(0x2bf401e0), SPH_C32(0x60640000), SPH_C32(0xb1ab0000),
	  SPH_C32(0xbc980000), SPH_C32(0xefeb0a94), SPH_C32(0xc894f143),
	  SPH_C32(0x5e0d273b), SPH_C32(0x5d6b3122), SPH_C32(0x1ceb0120),
	  SPH_C32(0x0b6a0000), SPH_C32(0x067e0000), SPH_C32(0xd73d0000),
	  SPH_C32(0xb01c159f), SPH_C32(0xcf9e9b5a), SPH_C32(0xd25cc5ca),
	  SPH_C32(0xebbc06c5) },
	{ SPH_C32(0xa8dc0140), SPH_C32(0x07260000), SPH_C32(0x50bc0000),
	  SPH_C32(0x8b930000), SPH_C32(0x24480aa0), SPH_C32(0xf4a0637f),
	  SPH_C32(0xc96a9af7), SPH_C32(0x1868519d), SPH_C32(0x6baf01d0),
	  SPH_C32(0xfa300000), SPH_C32(0xf3cc0000), SPH_C32(0xe3290000),
	  SPH_C32(0x392b6b13), SPH_C32(0x9515777f), SPH_C32(0xd99f08d4),
	  SPH_C32(0x248b730e) },
	{ SPH_C32(0xdf9801b0), SPH_C32(0xf67c0000), SPH_C32(0xa50e0000),
	  SPH_C32(0xbf870000), SPH_C32(0xad7f742c), SPH_C32(0xae2b8f5a),
	  SPH_C32(0xc2a957e9), SPH_C32(0xd75f2456), SPH_C32(0x9fc30180),
	  SPH_C32(0x6c280000), SPH_C32(0xe7690000), SPH_C32(0xe0360000),
	  SPH_C32(0x7bbf15ab), SPH_C32(0xf3aa0966), SPH_C32(0x453b7806),
	  SPH_C32(0xaebf667a) }
};

static const sph_u32 T512_12[16][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xf6800005), SPH_C32(0x3443c000), SPH_C32(0x24070000),
	  SPH_C32(0x8f3d0000), SPH_C32(0x21373bfb), SPH_C32(0x0ab8d5ae),
	  SPH_C32(0xcdc58b19), SPH_C32(0xd795ba31), SPH_C32(0xa67f0001),
	  SPH_C32(0x71378000), SPH_C32(0x19fc0000), SPH_C32(0x96db0000),
	  SPH_C32(0x3a8b6dfd), SPH_C32(0xebcaaef3), SPH_C32(0x2c6d478f),
	  SPH_C32(0xac8e6c88) },
	{ SPH_C32(0xa67f0001), SPH_C32(0x71378000), SPH_C32(0x19fc0000),
	  SPH_C32(0x96db0000), SPH_C32(0x3a8b6dfd), SPH_C32(0xebcaaef3),
	  SPH_C32(0x2c6d478f), SPH_C32(0xac8e6c88), SPH_C32(0x50ff0004),
	  SPH_C32(0x45744000), SPH_C32(0x3dfb0000), SPH_C32(0x19e60000),
	  SPH_C32(0x1bbc5606), SPH_C32(0xe1727b5d), SPH_C32(0xe1a8cc96),
	  SPH_C32(0x7b1bd6b9) },
	{ SPH_C32(0x50ff0004), SPH_C32(0x45744000), SPH_C32(0x3dfb0000),
	  SPH_C32(0x19e60000), SPH_C32(0x1bbc5606), SPH_C32(0xe1727b5d),
	  SPH_C32(0xe1a8cc96), SPH_C32(0x7b1bd6b9), SPH_C32(0xf6800005),
	  SPH_C32(0x3443c000), SPH_C32(0x24070000), SPH_C32(0x8f3d0000),
	  SPH_C32(0x21373bfb), SPH_C32(0x0ab8d5ae), SPH_C32(0xcdc58b19),
	  SPH_C32(0xd795ba31) },
	{ SPH_C32(0xf7750009), SPH_C32(0xcf3cc000), SPH_C32(0xc3d60000),
	  SPH_C32(0x04920000), SPH_C32(0x029519a9), SPH_C32(0xf8e836ba),
	  SPH_C32(0x7a87f14e), SPH_C32(0x9e16981a), SPH_C32(0xd46a0000),
	  SPH_C32(0x8dc8c000), SPH_C32(0xa5af0000), SPH_C32(0x4a290000),
	  SPH_C32(0xfc4e427a), SPH_C32(0xc9b4866c), SPH_C32(0x98369604),
	  SPH_C32(0xf746c320) },
	{ SPH_C32(0x01f5000c), SPH_C32(0xfb7f0000), SPH_C32(0xe7d10000),
	  SPH_C32(0x8baf0000), SPH_C32(0x23a22252), SPH_C32(0xf250e314),
	  SPH_C32(0xb7427a57), SPH_C32(0x4983222b), SPH_C32(0x72150001),
	  SPH_C32(0xfcff4000), SPH_C32(0xbc530000), SPH_C32(0xdcf20000),
	  SPH_C32(0xc6c52f87), SPH_C32(0x227e289f), SPH_C32(0xb45bd18b),
	  SPH_C32(0x5bc8afa8) },
	{ SPH_C32(0x510a0008), SPH_C32(0xbe0b4000), SPH_C32(0xda2a0000),
	  SPH_C32(0x92490000), SPH_C32(0x381e7454), SPH_C32(0x13229849),
	  SPH_C32(0x56eab6c1), SPH_C32(0x3298f492), SPH_C32(0x84950004),
	  SPH_C32(0xc8bc8000), SPH_C32(0x98540000), SPH_C32(0x53cf0000),
	  SPH_C32(0xe7f2147c), SPH_C32(0x28c6fd31), SPH_C32(0x799e5a92),
	  SPH_C32(0x8c5d1599) },
	{ SPH_C32(0xa78a000d), SPH_C32(0x8a488000), SPH_C32(0xfe2d0000),
	  SPH_C32(0x1d740000), SPH_C32(0x19294faf), SPH_C32(0x199a4de7),
	  SPH_C32(0x9b2f3dd8), SPH_C32(0xe50d4ea3), SPH_C32(0x22ea0005),
	  SPH_C32(0xb98b0000), SPH_C32(0x81a80000), SPH_C32(0xc5140000),
	  SPH_C32(0xdd797981), SPH_C32(0xc30c53c2), SPH_C32(0x55f31d1d),
	  SPH_C32(0x20d37911) },
	{ SPH_C32(0xd46a0000), SPH_C32(0x8dc8c000), SPH_C32(0xa5af0000),
	  SPH_C32(0x4a290000), SPH_C32(0xfc4e427a), SPH_C32(0xc9b4866c),
	  SPH_C32(0x98369604), SPH_C32(0xf746c320), SPH_C32(0x231f0009),
	  SPH_C32(0x42f40000), SPH_C32(0x66790000), SPH_C32(0x4ebb0000),
	  SPH_C32(0xfedb5bd3), SPH_C32(0x315cb0d6), SPH_C32(0xe2b1674a),
	  SPH_C32(0x69505b3a) },
	{ SPH_C32(0x22ea0005), SPH_C32(0xb98b0000), SPH_C32(0x81a80000),
	  SPH_C32(0xc5140000), SPH_C32(0xdd797981), SPH_C32(0xc30c53c2),
	  SPH_C32(0x55f31d1d), SPH_C32(0x20d37911), SPH_C32(0x85600008),
	  SPH_C32(0x33c38000), SPH_C32(0x7f850000), SPH_C32(0xd8600000),
	  SPH_C32(0xc450362e), SPH_C32(0xda961e25), SPH_C32(0xcedc20c5),
	  SPH_C32(0xc5de37b2) },
	{ SPH_C32(0x72150001), SPH_C32(0xfcff4000), SPH_C32(0xbc530000),
	  SPH_C32(0xdcf20000), SPH_C32(0xc6c52f87), SPH_C32(0x227e289f),
	  SPH_C32(0xb45bd18b), SPH_C32(0x5bc8afa8), SPH_C32(0x73e0000d),
	  SPH_C32(0x07804000), SPH_C32(0x5b820000), SPH_C32(0x575d0000),
	  SPH_C32(0xe5670dd5), SPH_C32(0xd02ecb8b), SPH_C32(0x0319abdc),
	  SPH_C32(0x124b8d83) },
	{ SPH_C32(0x84950004), SPH_C32(0xc8bc8000), SPH_C32(0x98540000),
	  SPH_C32(0x53cf0000), SPH_C32(0xe7f2147c), SPH_C32(0x28c6fd31),
	  SPH_C32(0x799e5a92), SPH_C32(0x8c5d1599), SPH_C32(0xd59f000c),
	  SPH_C32(0x76b7c000), SPH_C32(0x427e0000), SPH_C32(0xc1860000),
	  SPH_C32(0xdfec6028), SPH_C32(0x3be46578), SPH_C32(0x2f74ec53),
	  SPH_C32(0xbec5e10b) },
	{ SPH_C32(0x231f0009), SPH_C32(0x42f40000), SPH_C32(0x66790000),
	  SPH_C32(0x4ebb0000), SPH_C32(0xfedb5bd3), SPH_C32(0x315cb0d6),
	  SPH_C32(0xe2b1674a), SPH_C32(0x69505b3a), SPH_C32(0xf7750009),
	  SPH_C32(0xcf3cc000), SPH_C32(0xc3d60000), SPH_C32(0x04920000),
	  SPH_C32(0x029519a9), SPH_C32(0xf8e836ba), SPH_C32(0x7a87f14e),
	  SPH_C32(0x9e16981a) },
	{ SPH_C32(0xd59f000c), SPH_C32(0x76b7c000), SPH_C32(0x427e0000),
	  SPH_C32(0xc1860000), SPH_C32(0xdfec6028), SPH_C32(0x3be46578),
	  SPH_C32(0x2f74ec53), SPH_C32(0xbec5e10b), SPH_C32(0x510a0008),
	  SPH_C32(0xbe0b4000), SPH_C32(0xda2a0000), SPH_C32(0x92490000),
	  SPH_C32(0x381e7454), SPH_C32(0x13229849), SPH_C32(0x56eab6c1),
	  SPH_C32(0x3298f492) },
	{ SPH_C32(0x85600008), SPH_C32(0x33c38000), SPH_C32(0x7f850000),
	  SPH_C32(0xd8600000), SPH_C32(0xc450362e), SPH_C32(0xda961e25),
	  SPH_C32(0xcedc20c5), SPH_C32(0xc5de37b2), SPH_C32(0xa78a000d),
	  SPH_C32(0x8a488000), SPH_C32(0xfe2d0000), SPH_C32(0x1d740000),
	  SPH_C32(0x19294faf), SPH_C32(0x199a4de7), SPH_C32(0x9b2f3dd8),
	  SPH_C32(0xe50d4ea3) },
	{ SPH_C32(0x73e0000d), SPH_C32(0x07804000), SPH_C32(0x5b820000),
	  SPH_C32(0x575d0000), SPH_C32(0xe5670dd5), SPH_C32(0xd02ecb8b),
	  SPH_C32(0x0319abdc), SPH_C32(0x124b8d83), SPH_C32(0x01f5000c),
	  SPH_C32(0xfb7f0000), SPH_C32(0xe7d10000), SPH_C32(0x8baf0000),
	  SPH_C32(0x23a22252), SPH_C32(0xf250e314), SPH_C32(0xb7427a57),
	  SPH_C32(0x4983222b) }
};

static const sph_u32 T512_16[16][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x75a40000), SPH_C32(0xc28b2700), SPH_C32(0x94a40000),
	  SPH_C32(0x90f50000), SPH_C32(0xfb7857e0), SPH_C32(0x49ce0bae),
	  SPH_C32(0x1767c483), SPH_C32(0xaedf667e), SPH_C32(0xd1660000),
	  SPH_C32(0x1bbc0300), SPH_C32(0x9eec0000), SPH_C32(0xf6940000),
	  SPH_C32(0x03024527), SPH_C32(0xcf70fcf2), SPH_C32(0xb4431b17),
	  SPH_C32(0x857f3c2b) },
	{ SPH_C32(0xd1660000), SPH_C32(0x1bbc0300), SPH_C32(0x9eec0000),
	  SPH_C32(0xf6940000), SPH_C32(0x03024527), SPH_C32(0xcf70fcf2),
	  SPH_C32(0xb4431b17), SPH_C32(0x857f3c2b), SPH_C32(0xa4c20000),
	  SPH_C32(0xd9372400), SPH_C32(0x0a480000), SPH_C32(0x66610000),
	  SPH_C32(0xf87a12c7), SPH_C32(0x86bef75c), SPH_C32(0xa324df94),
	  SPH_C32(0x2ba05a55) },
	{ SPH_C32(0xa4c20000), SPH_C32(0xd9372400), SPH_C32(0x0a480000),
	  SPH_C32(0x66610000), SPH_C32(0xf87a12c7), SPH_C32(0x86bef75c),
	  SPH_C32(0xa324df94), SPH_C32(0x2ba05a55), SPH_C32(0x75a40000),
	  SPH_C32(0xc28b2700), SPH_C32(0x94a40000), SPH_C32(0x90f50000),
	  SPH_C32(0xfb7857e0), SPH_C32(0x49ce0bae), SPH_C32(0x1767c483),
	  SPH_C32(0xaedf667e) },
	{ SPH_C32(0x75c90003), SPH_C32(0x0e10c000), SPH_C32(0xd1200000),
	  SPH_C32(0xbaea0000), SPH_C32(0x8bc42f3e), SPH_C32(0x8758b757),
	  SPH_C32(0xbb28761d), SPH_C32(0x00b72e2b), SPH_C32(0xeecf0001),
	  SPH_C32(0x6f564000), SPH_C32(0xf33e0000), SPH_C32(0xa79e0000),
	  SPH_C32(0xbdb57219), SPH_C32(0xb711ebc5), SPH_C32(0x4a3b40ba),
	  SPH_C32(0xfeabf254) },
	{ SPH_C32(0x006d0003), SPH_C32(0xcc9be700), SPH_C32(0x45840000),
	  SPH_C32(0x2a1f0000), SPH_C32(0x70bc78de), SPH_C32(0xce96bcf9),
	  SPH_C32(0xac4fb29e), SPH_C32(0xae684855), SPH_C32(0x3fa90001),
	  SPH_C32(0x74ea4300), SPH_C32(0x6dd20000), SPH_C32(0x510a0000),
	  SPH_C32(0xbeb7373e), SPH_C32(0x78611737), SPH_C32(0xfe785bad),
	  SPH_C32(0x7bd4ce7f) },
	{ SPH_C32(0xa4af0003), SPH_C32(0x15acc300), SPH_C32(0x4fcc0000),
	  SPH_C32(0x4c7e0000), SPH_C32(0x88c66a19), SPH_C32(0x48284ba5),
	  SPH_C32(0x0f6b6d0a), SPH_C32(0x85c81200), SPH_C32(0x4a0d0001),
	  SPH_C32(0xb6616400), SPH_C32(0xf9760000), SPH_C32(0xc1ff0000),
	  SPH_C32(0x45cf60de), SPH_C32(0x31af1c99), SPH_C32(0xe91f9f2e),
	  SPH_C32(0xd50ba801) },
	{ SPH_C32(0xd10b0003), SPH_C32(0xd727e400), SPH_C32(0xdb680000),
	  SPH_C32(0xdc8b0000), SPH_C32(0x73be3df9), SPH_C32(0x01e6400b),
	  SPH_C32(0x180ca989), SPH_C32(0x2b17747e), SPH_C32(0x9b6b0001),
	  SPH_C32(0xaddd6700), SPH_C32(0x679a0000), SPH_C32(0x376b0000),
	  SPH_C32(0x46cd25f9), SPH_C32(0xfedfe06b), SPH_C32(0x5d5c8439),
	  SPH_C32(0x5074942a) },
	{ SPH_C32(0xeecf0001), SPH_C32(0x6f564000), SPH_C32(0xf33e0000),
	  SPH_C32(0xa79e0000), SPH_C32(0xbdb57219), SPH_C32(0xb711ebc5),
	  SPH_C32(0x4a3b40ba), SPH_C32(0xfeabf254), SPH_C32(0x9b060002),
	  SPH_C32(0x61468000), SPH_C32(0x221e0000), SPH_C32(0x1d740000),
	  SPH_C32(0x36715d27), SPH_C32(0x30495c92), SPH_C32(0xf11336a7),
	  SPH_C32(0xfe1cdc7f) },
	{ SPH_C32(0x9b6b0001), SPH_C32(0xaddd6700), SPH_C32(0x679a0000),
	  SPH_C32(0x376b0000), SPH_C32(0x46cd25f9), SPH_C32(0xfedfe06b),
	  SPH_C32(0x5d5c8439), SPH_C32(0x5074942a), SPH_C32(0x4a600002),
	  SPH_C32(0x7afa8300), SPH_C32(0xbcf20000), SPH_C32(0xebe00000),
	  SPH_C32(0x35731800), SPH_C32(0xff39a060), SPH_C32(0x45502db0),
	  SPH_C32(0x7b63e054) },
	{ SPH_C32(0x3fa90001), SPH_C32(0x74ea4300), SPH_C32(0x6dd20000),
	  SPH_C32(0x510a0000), SPH_C32(0xbeb7373e), SPH_C32(0x78611737),
	  SPH_C32(0xfe785bad), SPH_C32(0x7bd4ce7f), SPH_C32(0x3fc40002),
	  SPH_C32(0xb871a400), SPH_C32(0x28560000), SPH_C32(0x7b150000),
	  SPH_C32(0xce0b4fe0), SPH_C32(0xb6f7abce), SPH_C32(0x5237e933),
	  SPH_C32(0xd5bc862a) },
	{ SPH_C32(0x4a0d0001), SPH_C32(0xb6616400), SPH_C32(0xf9760000),
	  SPH_C32(0xc1ff0000), SPH_C32(0x45cf60de), SPH_C32(0x31af1c99),
	  SPH_C32(0xe91f9f2e), SPH_C32(0xd50ba801), SPH_C32(0xeea20002),
	  SPH_C32(0xa3cda700), SPH_C32(0xb6ba0000), SPH_C32(0x8d810000),
	  SPH_C32(0xcd090ac7), SPH_C32(0x7987573c), SPH_C32(0xe674f224),
	  SPH_C32(0x50c3ba01) },
	{ SPH_C32(0x9b060002), SPH_C32(0x61468000), SPH_C32(0x221e0000),
	  SPH_C32(0x1d740000), SPH_C32(0x36715d27), SPH_C32(0x30495c92),
	  SPH_C32(0xf11336a7), SPH_C32(0xfe1cdc7f), SPH_C32(0x75c90003),
	  SPH_C32(0x0e10c000), SPH_C32(0xd1200000), SPH_C32(0xbaea0000),
	  SPH_C32(0x8bc42f3e), SPH_C32(0x8758b757), SPH_C32(0xbb28761d),
	  SPH_C32(0x00b72e2b) },
	{ SPH_C32(0xeea20002), SPH_C32(0xa3cda700), SPH_C32(0xb6ba0000),
	  SPH_C32(0x8d810000), SPH_C32(0xcd090ac7), SPH_C32(0x7987573c),
	  SPH_C32(0xe674f224), SPH_C32(0x50c3ba01), SPH_C32(0xa4af0003),
	  SPH_C32(0x15acc300), SPH_C32(0x4fcc0000), SPH_C32(0x4c7e0000),
	  SPH_C32(0x88c66a19), SPH_C32(0x48284ba5), SPH_C32(0x0f6b6d0a),
	  SPH_C32(0x85c81200) },
	{ SPH_C32(0x4a600002), SPH_C32(0x7afa8300), SPH_C32(0xbcf20000),
	  SPH_C32(0xebe00000), SPH_C32(0x35731800), SPH_C32(0xff39a060),
	  SPH_C32(0x45502db0), SPH_C32(0x7b63e054), SPH_C32(0xd10b0003),
	  SPH_C32(0xd727e400), SPH_C32(0xdb680000), SPH_C32(0xdc8b0000),
	  SPH_C32(0x73be3df9), SPH_C32(0x01e6400b), SPH_C32(0x180ca989),
	  SPH_C32(0x2b17747e) },
	{ SPH_C32(0x3fc40002), SPH_C32(0xb871a400), SPH_C32(0x28560000),
	  SPH_C32(0x7b150000), SPH_C32(0xce0b4fe0), SPH_C32(0xb6f7abce),
	  SPH_C32(0x5237e933), SPH_C32(0xd5bc862a), SPH_C32(0x006d0003),
	  SPH_C32(0xcc9be700), SPH_C32(0x45840000), SPH_C32(0x2a1f0000),
	  SPH_C32(0x70bc78de), SPH_C32(0xce96bcf9), SPH_C32(0xac4fb29e),
	  SPH_C32(0xae684855) }
};

static const sph_u32 T512_20[16][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x0c720000), SPH_C32(0x49e50f00), SPH_C32(0x42790000),
	  SPH_C32(0x5cea0000), SPH_C32(0x33aa301a), SPH_C32(0x15822514),
	  SPH_C32(0x95a34b7b), SPH_C32(0xb44b0090), SPH_C32(0xfe220000),
	  SPH_C32(0xa7580500), SPH_C32(0x25d10000), SPH_C32(0xf7600000),
	  SPH_C32(0x893178da), SPH_C32(0x1fd4f860), SPH_C32(0x4ed0a315),
	  SPH_C32(0xa123ff9f) },
	{ SPH_C32(0xfe220000), SPH_C32(0xa7580500), SPH_C32(0x25d10000),
	  SPH_C32(0xf7600000), SPH_C32(0x893178da), SPH_C32(0x1fd4f860),
	  SPH_C32(0x4ed0a315), SPH_C32(0xa123ff9f), SPH_C32(0xf2500000),
	  SPH_C32(0xeebd0a00), SPH_C32(0x67a80000), SPH_C32(0xab8a0000),
	  SPH_C32(0xba9b48c0), SPH_C32(0x0a56dd74), SPH_C32(0xdb73e86e),
	  SPH_C32(0x1568ff0f) },
	{ SPH_C32(0xf2500000), SPH_C32(0xeebd0a00), SPH_C32(0x67a80000),
	  SPH_C32(0xab8a0000), SPH_C32(0xba9b48c0), SPH_C32(0x0a56dd74),
	  SPH_C32(0xdb73e86e), SPH_C32(0x1568ff0f), SPH_C32(0x0c720000),
	  SPH_C32(0x49e50f00), SPH_C32(0x42790000), SPH_C32(0x5cea0000),
	  SPH_C32(0x33aa301a), SPH_C32(0x15822514), SPH_C32(0x95a34b7b),
	  SPH_C32(0xb44b0090) },
	{ SPH_C32(0x45180000), SPH_C32(0xa5b51700), SPH_C32(0xf96a0000),
	  SPH_C32(0x3b480000), SPH_C32(0x1ecc142c), SPH_C32(0x231395d6),
	  SPH_C32(0x16bca6b0), SPH_C32(0xdf33f4df), SPH_C32(0xb83d0000),
	  SPH_C32(0x16710600), SPH_C32(0x379a0000), SPH_C32(0xf5b10000),
	  SPH_C32(0x228161ac), SPH_C32(0xae48f145), SPH_C32(0x66241616),
	  SPH_C32(0xc5c1eb3e) },
	{ SPH_C32(0x496a0000), SPH_C32(0xec501800), SPH_C32(0xbb130000),
	  SPH_C32(0x67a20000), SPH_C32(0x2d662436), SPH_C32(0x3691b0c2),
	  SPH_C32(0x831fedcb), SPH_C32(0x6b78f44f), SPH_C32(0x461f0000),
	  SPH_C32(0xb1290300), SPH_C32(0x124b0000), SPH_C32(0x02d10000),
	  SPH_C32(0xabb01976), SPH_C32(0xb19c0925), SPH_C32(0x28f4b503),
	  SPH_C32(0x64e214a1) },
	{ SPH_C32(0xbb3a0000), SPH_C32(0x02ed1200), SPH_C32(0xdcbb0000),
	  SPH_C32(0xcc280000), SPH_C32(0x97fd6cf6), SPH_C32(0x3cc76db6),
	  SPH_C32(0x586c05a5), SPH_C32(0x7e100b40), SPH_C32(0x4a6d0000),
	  SPH_C32(0xf8cc0c00), SPH_C32(0x50320000), SPH_C32(0x5e3b0000),
	  SPH_C32(0x981a296c), SPH_C32(0xa41e2c31), SPH_C32(0xbd57fe78),
	  SPH_C32(0xd0a91431) },
	{ SPH_C32(0xb7480000), SPH_C32(0x4b081d00), SPH_C32(0x9ec20000),
	  SPH_C32(0x90c20000), SPH_C32(0xa4575cec), SPH_C32(0x294548a2),
	  SPH_C32(0xcdcf4ede), SPH_C32(0xca5b0bd0), SPH_C32(0xb44f0000),
	  SPH_C32(0x5f940900), SPH_C32(0x75e30000), SPH_C32(0xa95b0000),
	  SPH_C32(0x112b51b6), SPH_C32(0xbbcad451), SPH_C32(0xf3875d6d),
	  SPH_C32(0x718aebae) },
	{ SPH_C32(0xb83d0000), SPH_C32(0x16710600), SPH_C32(0x379a0000),
	  SPH_C32(0xf5b10000), SPH_C32(0x228161ac), SPH_C32(0xae48f145),
	  SPH_C32(0x66241616), SPH_C32(0xc5c1eb3e), SPH_C32(0xfd250000),
	  SPH_C32(0xb3c41100), SPH_C32(0xcef00000), SPH_C32(0xcef90000),
	  SPH_C32(0x3c4d7580), SPH_C32(0x8d5b6493), SPH_C32(0x7098b0a6),
	  SPH_C32(0x1af21fe1) },
	{ SPH_C32(0xb44f0000), SPH_C32(0x5f940900), SPH_C32(0x75e30000),
	  SPH_C32(0xa95b0000), SPH_C32(0x112b51b6), SPH_C32(0xbbcad451),
	  SPH_C32(0xf3875d6d), SPH_C32(0x718aebae), SPH_C32(0x03070000),
	  SPH_C32(0x149c1400), SPH_C32(0xeb210000), SPH_C32(0x39990000),
	  SPH_C32(0xb57c0d5a), SPH_C32(0x928f9cf3), SPH_C32(0x3e4813b3),
	  SPH_C32(0xbbd1e07e) },
	{ SPH_C32(0x461f0000), SPH_C32(0xb1290300), SPH_C32(0x124b0000),
	  SPH_C32(0x02d10000), SPH_C32(0xabb01976), SPH_C32(0xb19c0925),
	  SPH_C32(0x28f4b503), SPH_C32(0x64e214a1), SPH_C32(0x0f750000),
	  SPH_C32(0x5d791b00), SPH_C32(0xa9580000), SPH_C32(0x65730000),
	  SPH_C32(0x86d63d40), SPH_C32(0x870db9e7), SPH_C32(0xabeb58c8),
	  SPH_C32(0x0f9ae0ee) },
	{ SPH_C32(0x4a6d0000), SPH_C32(0xf8cc0c00), SPH_C32(0x50320000),
	  SPH_C32(0x5e3b0000), SPH_C32(0x981a296c), SPH_C32(0xa41e2c31),
	  SPH_C32(0xbd57fe78), SPH_C32(0xd0a91431), SPH_C32(0xf1570000),
	  SPH_C32(0xfa211e00), SPH_C32(0x8c890000), SPH_C32(0x92130000),
	  SPH_C32(0x0fe7459a), SPH_C32(0x98d94187), SPH_C32(0xe53bfbdd),
	  SPH_C32(0xaeb91f71) },
	{ SPH_C32(0xfd250000), SPH_C32(0xb3c41100), SPH_C32(0xcef00000),
	  SPH_C32(0xcef90000), SPH_C32(0x3c4d7580), SPH_C32(0x8d5b6493),
	  SPH_C32(0x7098b0a6), SPH_C32(0x1af21fe1), SPH_C32(0x45180000),
	  SPH_C32(0xa5b51700), SPH_C32(0xf96a0000), SPH_C32(0x3b480000),
	  SPH_C32(0x1ecc142c), SPH_C32(0x231395d6), SPH_C32(0x16bca6b0),
	  SPH_C32(0xdf33f4df) },
	{ SPH_C32(0xf1570000), SPH_C32(0xfa211e00), SPH_C32(0x8c890000),
	  SPH_C32(0x92130000), SPH_C32(0x0fe7459a), SPH_C32(0x98d94187),
	  SPH_C32(0xe53bfbdd), SPH_C32(0xaeb91f71), SPH_C32(0xbb3a0000),
	  SPH_C32(0x02ed1200), SPH_C32(0xdcbb0000), SPH_C32(0xcc280000),
	  SPH_C32(0x97fd6cf6), SPH_C32(0x3cc76db6), SPH_C32(0x586c05a5),
	  SPH_C32(0x7e100b40) },
	{ SPH_C32(0x03070000), SPH_C32(0x149c1400), SPH_C32(0xeb210000),
	  SPH_C32(0x39990000), SPH_C32(0xb57c0d5a), SPH_C32(0x928f9cf3),
	  SPH_C32(0x3e4813b3), SPH_C32(0xbbd1e07e), SPH_C32(0xb7480000),
	  SPH_C32(0x4b081d00), SPH_C32(0x9ec20000), SPH_C32(0x90c20000),
	  SPH_C32(0xa4575cec), SPH_C32(0x294548a2), SPH_C32(0xcdcf4ede),
	  SPH_C32(0xca5b0bd0) },
	{ SPH_C32(0x0f750000), SPH_C32(0x5d791b00), SPH_C32(0xa9580000),
	  SPH_C32(0x65730000), SPH_C32(0x86d63d40), SPH_C32(0x870db9e7),
	  SPH_C32(0xabeb58c8), SPH_C32(0x0f9ae0ee), SPH_C32(0x496a0000),
	  SPH_C32(0xec501800), SPH_C32(0xbb130000), SPH_C32(0x67a20000),
	  SPH_C32(0x2d662436), SPH_C32(0x3691b0c2), SPH_C32(0x831fedcb),
	  SPH_C32(0x6b78f44f) }
};

static const sph_u32 T512_24[16][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x54500000), SPH_C32(0x0671005c), SPH_C32(0x25ae0000),
	  SPH_C32(0x6a1e0000), SPH_C32(0x2ea54edf), SPH_C32(0x664e8512),
	  SPH_C32(0xbfba18c3), SPH_C32(0x7e715d17), SPH_C32(0xbc8d0000),
	  SPH_C32(0xfc3b0018), SPH_C32(0x19830000), SPH_C32(0xd10b0000),
	  SPH_C32(0xae1878c4), SPH_C32(0x42a69856), SPH_C32(0x0012da37),
	  SPH_C32(0x2c3b504e) },
	{ SPH_C32(0xbc8d0000), SPH_C32(0xfc3b0018), SPH_C32(0x19830000),
	  SPH_C32(0xd10b0000), SPH_C32(0xae1878c4), SPH_C32(0x42a69856),
	  SPH_C32(0x0012da37), SPH_C32(0x2c3b504e), SPH_C32(0xe8dd0000),
	  SPH_C32(0xfa4a0044), SPH_C32(0x3c2d0000), SPH_C32(0xbb150000),
	  SPH_C32(0x80bd361b), SPH_C32(0x24e81d44), SPH_C32(0xbfa8c2f4),
	  SPH_C32(0x524a0d59) },
	{ SPH_C32(0xe8dd0000), SPH_C32(0xfa4a0044), SPH_C32(0x3c2d0000),
	  SPH_C32(0xbb150000), SPH_C32(0x80bd361b), SPH_C32(0x24e81d44),
	  SPH_C32(0xbfa8c2f4), SPH_C32(0x524a0d59), SPH_C32(0x54500000),
	  SPH_C32(0x0671005c), SPH_C32(0x25ae0000), SPH_C32(0x6a1e0000),
	  SPH_C32(0x2ea54edf), SPH_C32(0x664e8512), SPH_C32(0xbfba18c3),
	  SPH_C32(0x7e715d17) },
	{ SPH_C32(0x69510000), SPH_C32(0xd4e1009c), SPH_C32(0xc3230000),
	  SPH_C32(0xac2f0000), SPH_C32(0xe4950bae), SPH_C32(0xcea415dc),
	  SPH_C32(0x87ec287c), SPH_C32(0xbce1a3ce), SPH_C32(0xc6730000),
	  SPH_C32(0xaf8d000c), SPH_C32(0xa4c10000), SPH_C32(0x218d0000),
	  SPH_C32(0x23111587), SPH_C32(0x7913512f), SPH_C32(0x1d28ac88),
	  SPH_C32(0x378dd173) },
	{ SPH_C32(0x3d010000), SPH_C32(0xd29000c0), SPH_C32(0xe68d0000),
	  SPH_C32(0xc6310000), SPH_C32(0xca304571), SPH_C32(0xa8ea90ce),
	  SPH_C32(0x385630bf), SPH_C32(0xc290fed9), SPH_C32(0x7afe0000),
	  SPH_C32(0x53b60014), SPH_C32(0xbd420000), SPH_C32(0xf0860000),
	  SPH_C32(0x8d096d43), SPH_C32(0x3bb5c979), SPH_C32(0x1d3a76bf),
	  SPH_C32(0x1bb6813d) },
	{ SPH_C32(0xd5dc0000), SPH_C32(0x28da0084), SPH_C32(0xdaa00000),
	  SPH_C32(0x7d240000), SPH_C32(0x4a8d736a), SPH_C32(0x8c028d8a),
	  SPH_C32(0x87fef24b), SPH_C32(0x90daf380), SPH_C32(0x2eae0000),
	  SPH_C32(0x55c70048), SPH_C32(0x98ec0000), SPH_C32(0x9a980000),
	  SPH_C32(0xa3ac239c), SPH_C32(0x5dfb4c6b), SPH_C32(0xa2806e7c),
	  SPH_C32(0x65c7dc2a) },
	{ SPH_C32(0x818c0000), SPH_C32(0x2eab00d8), SPH_C32(0xff0e0000),
	  SPH_C32(0x173a0000), SPH_C32(0x64283db5), SPH_C32(0xea4c0898),
	  SPH_C32(0x3844ea88), SPH_C32(0xeeabae97), SPH_C32(0x92230000),
	  SPH_C32(0xa9fc0050), SPH_C32(0x816f0000), SPH_C32(0x4b930000),
	  SPH_C32(0x0db45b58), SPH_C32(0x1f5dd43d), SPH_C32(0xa292b44b),
	  SPH_C32(0x49fc8c64) },
	{ SPH_C32(0xc6730000), SPH_C32(0xaf8d000c), SPH_C32(0xa4c10000),
	  SPH_C32(0x218d0000), SPH_C32(0x23111587), SPH_C32(0x7913512f),
	  SPH_C32(0x1d28ac88), SPH_C32(0x378dd173), SPH_C32(0xaf220000),
	  SPH_C32(0x7b6c0090), SPH_C32(0x67e20000), SPH_C32(0x8da20000),
	  SPH_C32(0xc7841e29), SPH_C32(0xb7b744f3), SPH_C32(0x9ac484f4),
	  SPH_C32(0x8b6c72bd) },
	{ SPH_C32(0x92230000), SPH_C32(0xa9fc0050), SPH_C32(0x816f0000),
	  SPH_C32(0x4b930000), SPH_C32(0x0db45b58), SPH_C32(0x1f5dd43d),
	  SPH_C32(0xa292b44b), SPH_C32(0x49fc8c64), SPH_C32(0x13af0000),
	  SPH_C32(0x87570088), SPH_C32(0x7e610000), SPH_C32(0x5ca90000),
	  SPH_C32(0x699c66ed), SPH_C32(0xf511dca5), SPH_C32(0x9ad65ec3),
	  SPH_C32(0xa75722f3) },
	{ SPH_C32(0x7afe0000), SPH_C32(0x53b60014), SPH_C32(0xbd420000),
	  SPH_C32(0xf0860000), SPH_C32(0x8d096d43), SPH_C32(0x3bb5c979),
	  SPH_C32(0x1d3a76bf), SPH_C32(0x1bb6813d), SPH_C32(0x47ff0000),
	  SPH_C32(0x812600d4), SPH_C32(0x5bcf0000), SPH_C32(0x36b70000),
	  SPH_C32(0x47392832), SPH_C32(0x935f59b7), SPH_C32(0x256c4600),
	  SPH_C32(0xd9267fe4) },
	{ SPH_C32(0x2eae0000), SPH_C32(0x55c70048), SPH_C32(0x98ec0000),
	  SPH_C32(0x9a980000), SPH_C32(0xa3ac239c), SPH_C32(0x5dfb4c6b),
	  SPH_C32(0xa2806e7c), SPH_C32(0x65c7dc2a), SPH_C32(0xfb720000),
	  SPH_C32(0x7d1d00cc), SPH_C32(0x424c0000), SPH_C32(0xe7bc0000),
	  SPH_C32(0xe92150f6), SPH_C32(0xd1f9c1e1), SPH_C32(0x257e9c37),
	  SPH_C32(0xf51d2faa) },
	{ SPH_C32(0xaf220000), SPH_C32(0x7b6c0090), SPH_C32(0x67e20000),
	  SPH_C32(0x8da20000), SPH_C32(0xc7841e29), SPH_C32(0xb7b744f3),
	  SPH_C32(0x9ac484f4), SPH_C32(0x8b6c72bd), SPH_C32(0x69510000),
	  SPH_C32(0xd4e1009c), SPH_C32(0xc3230000), SPH_C32(0xac2f0000),
	  SPH_C32(0xe4950bae), SPH_C32(0xcea415dc), SPH_C32(0x87ec287c),
	  SPH_C32(0xbce1a3ce) },
	{ SPH_C32(0xfb720000), SPH_C32(0x7d1d00cc), SPH_C32(0x424c0000),
	  SPH_C32(0xe7bc0000), SPH_C32(0xe92150f6), SPH_C32(0xd1f9c1e1),
	  SPH_C32(0x257e9c37), SPH_C32(0xf51d2faa), SPH_C32(0xd5dc0000),
	  SPH_C32(0x28da0084), SPH_C32(0xdaa00000), SPH_C32(0x7d240000),
	  SPH_C32(0x4a8d736a), SPH_C32(0x8c028d8a), SPH_C32(0x87fef24b),
	  SPH_C32(0x90daf380) },
	{ SPH_C32(0x13af0000), SPH_C32(0x87570088), SPH_C32(0x7e610000),
	  SPH_C32(0x5ca90000), SPH_C32(0x699c66ed), SPH_C32(0xf511dca5),
	  SPH_C32(0x9ad65ec3), SPH_C32(0xa75722f3), SPH_C32(0x818c0000),
	  SPH_C32(0x2eab00d8), SPH_C32(0xff0e0000), SPH_C32(0x173a0000),
	  SPH_C32(0x64283db5), SPH_C32(0xea4c0898), SPH_C32(0x3844ea88),
	  SPH_C32(0xeeabae97) },
	{ SPH_C32(0x47ff0000), SPH_C32(0x812600d4), SPH_C32(0x5bcf0000),
	  SPH_C32(0x36b70000), SPH_C32(0x47392832), SPH_C32(0x935f59b7),
	  SPH_C32(0x256c4600), SPH_C32(0xd9267fe4), SPH_C32(0x3d010000),
	  SPH_C32(0xd29000c0), SPH_C32(0xe68d0000), SPH_C32(0xc6310000),
	  SPH_C32(0xca304571), SPH_C32(0xa8ea90ce), SPH_C32(0x385630bf),
	  SPH_C32(0xc290fed9) }
};

static const sph_u32 T512_28[16][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x86790000), SPH_C32(0x3f390002), SPH_C32(0xe19ae000),
	  SPH_C32(0x98560000), SPH_C32(0x9565670e), SPH_C32(0x4e88c8ea),
	  SPH_C32(0xd3dd4944), SPH_C32(0x161ddab9), SPH_C32(0x30b70000),
	  SPH_C32(0xe5d00000), SPH_C32(0xf4f46000), SPH_C32(0x42c40000),
	  SPH_C32(0x63b83d6a), SPH_C32(0x78ba9460), SPH_C32(0x21afa1ea),
	  SPH_C32(0xb0a51834) },
	{ SPH_C32(0x30b70000), SPH_C32(0xe5d00000), SPH_C32(0xf4f46000),
	  SPH_C32(0x42c40000), SPH_C32(0x63b83d6a), SPH_C32(0x78ba9460),
	  SPH_C32(0x21afa1ea), SPH_C32(0xb0a51834), SPH_C32(0xb6ce0000),
	  SPH_C32(0xdae90002), SPH_C32(0x156e8000), SPH_C32(0xda920000),
	  SPH_C32(0xf6dd5a64), SPH_C32(0x36325c8a), SPH_C32(0xf272e8ae),
	  SPH_C32(0xa6b8c28d) },
	{ SPH_C32(0xb6ce0000), SPH_C32(0xdae90002), SPH_C32(0x156e8000),
	  SPH_C32(0xda920000), SPH_C32(0xf6dd5a64), SPH_C32(0x36325c8a),
	  SPH_C32(0xf272e8ae), SPH_C32(0xa6b8c28d), SPH_C32(0x86790000),
	  SPH_C32(0x3f390002), SPH_C32(0xe19ae000), SPH_C32(0x98560000),
	  SPH_C32(0x9565670e), SPH_C32(0x4e88c8ea), SPH_C32(0xd3dd4944),
	  SPH_C32(0x161ddab9) },
	{ SPH_C32(0x14190000), SPH_C32(0x23ca003c), SPH_C32(0x50df0000),
	  SPH_C32(0x44b60000), SPH_C32(0x1b6c67b0), SPH_C32(0x3cf3ac75),
	  SPH_C32(0x61e610b0), SPH_C32(0xdbcadb80), SPH_C32(0xe3430000),
	  SPH_C32(0x3a4e0014), SPH_C32(0xf2c60000), SPH_C32(0xaa4e0000),
	  SPH_C32(0xdb1e42a6), SPH_C32(0x256bbe15), SPH_C32(0x123db156),
	  SPH_C32(0x3a4e99d7) },
	{ SPH_C32(0x92600000), SPH_C32(0x1cf3003e), SPH_C32(0xb145e000),
	  SPH_C32(0xdce00000), SPH_C32(0x8e0900be), SPH_C32(0x727b649f),
	  SPH_C32(0xb23b59f4), SPH_C32(0xcdd70139), SPH_C32(0xd3f40000),
	  SPH_C32(0xdf9e0014), SPH_C32(0x06326000), SPH_C32(0xe88a0000),
	  SPH_C32(0xb8a67fcc), SPH_C32(0x5dd12a75), SPH_C32(0x339210bc),
	  SPH_C32(0x8aeb81e3) },
	{ SPH_C32(0x24ae0000), SPH_C32(0xc61a003c), SPH_C32(0xa42b6000),
	  SPH_C32(0x06720000), SPH_C32(0x78d45ada), SPH_C32(0x44493815),
	  SPH_C32(0x4049b15a), SPH_C32(0x6b6fc3b4), SPH_C32(0x558d0000),
	  SPH_C32(0xe0a70016), SPH_C32(0xe7a88000), SPH_C32(0x70dc0000),
	  SPH_C32(0x2dc318c2), SPH_C32(0x1359e29f), SPH_C32(0xe04f59f8),
	  SPH_C32(0x9cf65b5a) },
	{ SPH_C32(0xa2d70000), SPH_C32(0xf923003e), SPH_C32(0x45b18000),
	  SPH_C32(0x9e240000), SPH_C32(0xedb13dd4), SPH_C32(0x0ac1f0ff),
	  SPH_C32(0x9394f81e), SPH_C32(0x7d72190d), SPH_C32(0x653a0000),
	  SPH_C32(0x05770016), SPH_C32(0x135ce000), SPH_C32(0x32180000),
	  SPH_C32(0x4e7b25a8), SPH_C32(0x6be376ff), SPH_C32(0xc1e0f812),
	  SPH_C32(0x2c53436e) },
	{ SPH_C32(0xe3430000), SPH_C32(0x3a4e0014), SPH_C32(0xf2c60000),
	  SPH_C32(0xaa4e0000), SPH_C32(0xdb1e42a6), SPH_C32(0x256bbe15),
	  SPH_C32(0x123db156), SPH_C32(0x3a4e99d7), SPH_C32(0xf75a0000),
	  SPH_C32(0x19840028), SPH_C32(0xa2190000), SPH_C32(0xeef80000),
	  SPH_C32(0xc0722516), SPH_C32(0x19981260), SPH_C32(0x73dba1e6),
	  SPH_C32(0xe1844257) },
	{ SPH_C32(0x653a0000), SPH_C32(0x05770016), SPH_C32(0x135ce000),
	  SPH_C32(0x32180000), SPH_C32(0x4e7b25a8), SPH_C32(0x6be376ff),
	  SPH_C32(0xc1e0f812), SPH_C32(0x2c53436e), SPH_C32(0xc7ed0000),
	  SPH_C32(0xfc540028), SPH_C32(0x56ed6000), SPH_C32(0xac3c0000),
	  SPH_C32(0xa3ca187c), SPH_C32(0x61228600), SPH_C32(0x5274000c),
	  SPH_C32(0x51215a63) },
	{ SPH_C32(0xd3f40000), SPH_C32(0xdf9e0014), SPH_C32(0x06326000),
	  SPH_C32(0xe88a0000), SPH_C32(0xb8a67fcc), SPH_C32(0x5dd12a75),
	  SPH_C32(0x339210bc), SPH_C32(0x8aeb81e3), SPH_C32(0x41940000),
	  SPH_C32(0xc36d002a), SPH_C32(0xb7778000), SPH_C32(0x346a0000),
	  SPH_C32(0x36af7f72), SPH_C32(0x2faa4eea), SPH_C32(0x81a94948),
	  SPH_C32(0x473c80da) },
	{ SPH_C32(0x558d0000), SPH_C32(0xe0a70016), SPH_C32(0xe7a88000),
	  SPH_C32(0x70dc0000), SPH_C32(0x2dc318c2), SPH_C32(0x1359e29f),
	  SPH_C32(0xe04f59f8), SPH_C32(0x9cf65b5a), SPH_C32(0x71230000),
	  SPH_C32(0x26bd002a), SPH_C32(0x4383e000), SPH_C32(0x76ae0000),
	  SPH_C32(0x55174218), SPH_C32(0x5710da8a), SPH_C32(0xa006e8a2),
	  SPH_C32(0xf79998ee) },
	{ SPH_C32(0xf75a0000), SPH_C32(0x19840028), SPH_C32(0xa2190000),
	  SPH_C32(0xeef80000), SPH_C32(0xc0722516), SPH_C32(0x19981260),
	  SPH_C32(0x73dba1e6), SPH_C32(0xe1844257), SPH_C32(0x14190000),
	  SPH_C32(0x23ca003c), SPH_C32(0x50df0000), SPH_C32(0x44b60000),
	  SPH_C32(0x1b6c67b0), SPH_C32(0x3cf3ac75), SPH_C32(0x61e610b0),
	  SPH_C32(0xdbcadb80) },
	{ SPH_C32(0x71230000), SPH_C32(0x26bd002a), SPH_C32(0x4383e000),
	  SPH_C32(0x76ae0000), SPH_C32(0x55174218), SPH_C32(0x5710da8a),
	  SPH_C32(0xa006e8a2), SPH_C32(0xf79998ee), SPH_C32(0x24ae0000),
	  SPH_C32(0xc61a003c), SPH_C32(0xa42b6000), SPH_C32(0x06720000),
	  SPH_C32(0x78d45ada), SPH_C32(0x44493815), SPH_C32(0x4049b15a),
	  SPH_C32(0x6b6fc3b4) },
	{ SPH_C32(0xc7ed0000), SPH_C32(0xfc540028), SPH_C32(0x56ed6000),
	  SPH_C32(0xac3c0000), SPH_C32(0xa3ca187c), SPH_C32(0x61228600),
	  SPH_C32(0x5274000c), SPH_C32(0x51215a63), SPH_C32(0xa2d70000),
	  SPH_C32(0xf923003e), SPH_C32(0x45b18000), SPH_C32(0x9e240000),
	  SPH_C32(0xedb13dd4), SPH_C32(0x0ac1f0ff), SPH_C32(0x9394f81e),
	  SPH_C32(0x7d72190d) },
	{ SPH_C32(0x41940000), SPH_C32(0xc36d002a), SPH_C32(0xb7778000),
	  SPH_C32(0x346a0000), SPH_C32(0x36af7f72), SPH_C32(0x2faa4eea),
	  SPH_C32(0x81a94948), SPH_C32(0x473c80da), SPH_C32(0x92600000),
	  SPH_C32(0x1cf3003e), SPH_C32(0xb145e000), SPH_C32(0xdce00000),
	  SPH_C32(0x8e0900be), SPH_C32(0x727b649f), SPH_C32(0xb23b59f4),
	  SPH_C32(0xcdd70139) }
};

static const sph_u32 T512_32[16][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xac480000), SPH_C32(0x1ba60000), SPH_C32(0x45fb1380),
	  SPH_C32(0x03430000), SPH_C32(0x5a85316a), SPH_C32(0x1fb250b6),
	  SPH_C32(0xfe72c7fe), SPH_C32(0x91e478f6), SPH_C32(0x1e4e0000),
	  SPH_C32(0xdecf0000), SPH_C32(0x6df80180), SPH_C32(0x77240000),
	  SPH_C32(0xec47079e), SPH_C32(0xf4a0694e), SPH_C32(0xcda31812),
	  SPH_C32(0x98aa496e) },
	{ SPH_C32(0x1e4e0000), SPH_C32(0xdecf0000), SPH_C32(0x6df80180),
	  SPH_C32(0x77240000), SPH_C32(0xec47079e), SPH_C32(0xf4a0694e),
	  SPH_C32(0xcda31812), SPH_C32(0x98aa496e), SPH_C32(0xb2060000),
	  SPH_C32(0xc5690000), SPH_C32(0x28031200), SPH_C32(0x74670000),
	  SPH_C32(0xb6c236f4), SPH_C32(0xeb1239f8), SPH_C32(0x33d1dfec),
	  SPH_C32(0x094e3198) },
	{ SPH_C32(0xb2060000), SPH_C32(0xc5690000), SPH_C32(0x28031200),
	  SPH_C32(0x74670000), SPH_C32(0xb6c236f4), SPH_C32(0xeb1239f8),
	  SPH_C32(0x33d1dfec), SPH_C32(0x094e3198), SPH_C32(0xac480000),
	  SPH_C32(0x1ba60000), SPH_C32(0x45fb1380), SPH_C32(0x03430000),
	  SPH_C32(0x5a85316a), SPH_C32(0x1fb250b6), SPH_C32(0xfe72c7fe),
	  SPH_C32(0x91e478f6) },
	{ SPH_C32(0xaec30000), SPH_C32(0x9c4f0001), SPH_C32(0x79d1e000),
	  SPH_C32(0x2c150000), SPH_C32(0x45cc75b3), SPH_C32(0x6650b736),
	  SPH_C32(0xab92f78f), SPH_C32(0xa312567b), SPH_C32(0xdb250000),
	  SPH_C32(0x09290000), SPH_C32(0x49aac000), SPH_C32(0x81e10000),
	  SPH_C32(0xcafe6b59), SPH_C32(0x42793431), SPH_C32(0x43566b76),
	  SPH_C32(0xe86cba2e) },
	{ SPH_C32(0x028b0000), SPH_C32(0x87e90001), SPH_C32(0x3c2af380),
	  SPH_C32(0x2f560000), SPH_C32(0x1f4944d9), SPH_C32(0x79e2e780),
	  SPH_C32(0x55e03071), SPH_C32(0x32f62e8d), SPH_C32(0xc56b0000),
	  SPH_C32(0xd7e60000), SPH_C32(0x2452c180), SPH_C32(0xf6c50000),
	  SPH_C32(0x26b96cc7), SPH_C32(0xb6d95d7f), SPH_C32(0x8ef57364),
	  SPH_C32(0x70c6f340) },
	{ SPH_C32(0xb08d0000), SPH_C32(0x42800001), SPH_C32(0x1429e180),
	  SPH_C32(0x5b310000), SPH_C32(0xa98b722d), SPH_C32(0x92f0de78),
	  SPH_C32(0x6631ef9d), SPH_C32(0x3bb81f15), SPH_C32(0x69230000),
	  SPH_C32(0xcc400000), SPH_C32(0x61a9d200), SPH_C32(0xf5860000),
	  SPH_C32(0x7c3c5dad), SPH_C32(0xa96b0dc9), SPH_C32(0x7087b49a),
	  SPH_C32(0xe1228bb6) },
	{ SPH_C32(0x1cc50000), SPH_C32(0x59260001), SPH_C32(0x51d2f200),
	  SPH_C32(0x58720000), SPH_C32(0xf30e4347), SPH_C32(0x8d428ece),
	  SPH_C32(0x98432863), SPH_C32(0xaa5c67e3), SPH_C32(0x776d0000),
	  SPH_C32(0x128f0000), SPH_C32(0x0c51d380), SPH_C32(0x82a20000),
	  SPH_C32(0x907b5a33), SPH_C32(0x5dcb6487), SPH_C32(0xbd24ac88),
	  SPH_C32(0x7988c2d8) },
	{ SPH_C32(0xdb250000), SPH_C32(0x09290000), SPH_C32(0x49aac000),
	  SPH_C32(0x81e10000), SPH_C32(0xcafe6b59), SPH_C32(0x42793431),
	  SPH_C32(0x43566b76), SPH_C32(0xe86cba2e), SPH_C32(0x75e60000),
	  SPH_C32(0x95660001), SPH_C32(0x307b2000), SPH_C32(0xadf40000),
	  SPH_C32(0x8f321eea), SPH_C32(0x24298307), SPH_C32(0xe8c49cf9),
	  SPH_C32(0x4b7eec55) },
	{ SPH_C32(0x776d0000), SPH_C32(0x128f0000), SPH_C32(0x0c51d380),
	  SPH_C32(0x82a20000), SPH_C32(0x907b5a33), SPH_C32(0x5dcb6487),
	  SPH_C32(0xbd24ac88), SPH_C32(0x7988c2d8), SPH_C32(0x6ba80000),
	  SPH_C32(0x4ba90001), SPH_C32(0x5d832180), SPH_C32(0xdad00000),
	  SPH_C32(0x63751974), SPH_C32(0xd089ea49), SPH_C32(0x256784eb),
	  SPH_C32(0xd3d4a53b) },
	{ SPH_C32(0xc56b0000), SPH_C32(0xd7e60000), SPH_C32(0x2452c180),
	  SPH_C32(0xf6c50000), SPH_C32(0x26b96cc7), SPH_C32(0xb6d95d7f),
	  SPH_C32(0x8ef57364), SPH_C32(0x70c6f340), SPH_C32(0xc7e00000),
	  SPH_C32(0x500f0001), SPH_C32(0x18783200), SPH_C32(0xd9930000),
	  SPH_C32(0x39f0281e), SPH_C32(0xcf3bbaff), SPH_C32(0xdb154315),
	  SPH_C32(0x4230ddcd) },
	{ SPH_C32(0x69230000), SPH_C32(0xcc400000), SPH_C32(0x61a9d200),
	  SPH_C32(0xf5860000), SPH_C32(0x7c3c5dad), SPH_C32(0xa96b0dc9),
	  SPH_C32(0x7087b49a), SPH_C32(0xe1228bb6), SPH_C32(0xd9ae0000),
	  SPH_C32(0x8ec00001), SPH_C32(0x75803380), SPH_C32(0xaeb70000),
	  SPH_C32(0xd5b72f80), SPH_C32(0x3b9bd3b1), SPH_C32(0x16b65b07),
	  SPH_C32(0xda9a94a3) },
	{ SPH_C32(0x75e60000), SPH_C32(0x95660001), SPH_C32(0x307b2000),
	  SPH_C32(0xadf40000), SPH_C32(0x8f321eea), SPH_C32(0x24298307),
	  SPH_C32(0xe8c49cf9), SPH_C32(0x4b7eec55), SPH_C32(0xaec30000),
	  SPH_C32(0x9c4f0001), SPH_C32(0x79d1e000), SPH_C32(0x2c150000),
	  SPH_C32(0x45cc75b3), SPH_C32(0x6650b736), SPH_C32(0xab92f78f),
	  SPH_C32(0xa312567b) },
	{ SPH_C32(0xd9ae0000), SPH_C32(0x8ec00001), SPH_C32(0x75803380),
	  SPH_C32(0xaeb70000), SPH_C32(0xd5b72f80), SPH_C32(0x3b9bd3b1),
	  SPH_C32(0x16b65b07), SPH_C32(0xda9a94a3), SPH_C32(0xb08d0000),
	  SPH_C32(0x42800001), SPH_C32(0x1429e180), SPH_C32(0x5b310000),
	  SPH_C32(0xa98b722d), SPH_C32(0x92f0de78), SPH_C32(0x6631ef9d),
	  SPH_C32(0x3bb81f15) },
	{ SPH_C32(0x6ba80000), SPH_C32(0x4ba90001), SPH_C32(0x5d832180),
	  SPH_C32(0xdad00000), SPH_C32(0x63751974), SPH_C32(0xd089ea49),
	  SPH_C32(0x256784eb), SPH_C32(0xd3d4a53b), SPH_C32(0x1cc50000),
	  SPH_C32(0x59260001), SPH_C32(0x51d2f200), SPH_C32(0x58720000),
	  SPH_C32(0xf30e4347), SPH_C32(0x8d428ece), SPH_C32(0x98432863),
	  SPH_C32(0xaa5c67e3) },
	{ SPH_C32(0xc7e00000), SPH_C32(0x500f0001), SPH_C32(0x18783200),
	  SPH_C32(0xd9930000), SPH_C32(0x39f0281e), SPH_C32(0xcf3bbaff),
	  SPH_C32(0xdb154315), SPH_C32(0x4230ddcd), SPH_C32(0x028b0000),
	  SPH_C32(0x87e90001), SPH_C32(0x3c2af380), SPH_C32(0x2f560000),
	  SPH_C32(0x1f4944d9), SPH_C32(0x79e2e780), SPH_C32(0x55e03071),
	  SPH_C32(0x32f62e8d) }
};

static const sph_u32 T512_36[16][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xcc140000), SPH_C32(0xa5630000), SPH_C32(0x5ab90780),
	  SPH_C32(0x3b500000), SPH_C32(0x4bd013ff), SPH_C32(0x879b3418),
	  SPH_C32(0x694348c1), SPH_C32(0xca5a87fe), SPH_C32(0x819e0000),
	  SPH_C32(0xec570000), SPH_C32(0x66320280), SPH_C32(0x95f30000),
	  SPH_C32(0x5da92802), SPH_C32(0x48f43cbc), SPH_C32(0xe65aa22d),
	  SPH_C32(0x8e67b7fa) },
	{ SPH_C32(0x819e0000), SPH_C32(0xec570000), SPH_C32(0x66320280),
	  SPH_C32(0x95f30000), SPH_C32(0x5da92802), SPH_C32(0x48f43cbc),
	  SPH_C32(0xe65aa22d), SPH_C32(0x8e67b7fa), SPH_C32(0x4d8a0000),
	  SPH_C32(0x49340000), SPH_C32(0x3c8b0500), SPH_C32(0xaea30000),
	  SPH_C32(0x16793bfd), SPH_C32(0xcf6f08a4), SPH_C32(0x8f19eaec),
	  SPH_C32(0x443d3004) },
	{ SPH_C32(0x4d8a0000), SPH_C32(0x49340000), SPH_C32(0x3c8b0500),
	  SPH_C32(0xaea30000), SPH_C32(0x16793bfd), SPH_C32(0xcf6f08a4),
	  SPH_C32(0x8f19eaec), SPH_C32(0x443d3004), SPH_C32(0xcc140000),
	  SPH_C32(0xa5630000), SPH_C32(0x5ab90780), SPH_C32(0x3b500000),
	  SPH_C32(0x4bd013ff), SPH_C32(0x879b3418), SPH_C32(0x694348c1),
	  SPH_C32(0xca5a87fe) },
	{ SPH_C32(0x78230000), SPH_C32(0x12fc0000), SPH_C32(0xa93a0b80),
	  SPH_C32(0x90a50000), SPH_C32(0x713e2879), SPH_C32(0x7ee98924),
	  SPH_C32(0xf08ca062), SPH_C32(0x636f8bab), SPH_C32(0x02af0000),
	  SPH_C32(0xb7280000), SPH_C32(0xba1c0300), SPH_C32(0x56980000),
	  SPH_C32(0xba8d45d3), SPH_C32(0x8048c667), SPH_C32(0xa95c149a),
	  SPH_C32(0xf4f6ea7b) },
	{ SPH_C32(0xb4370000), SPH_C32(0xb79f0000), SPH_C32(0xf3830c00),
	  SPH_C32(0xabf50000), SPH_C32(0x3aee3b86), SPH_C32(0xf972bd3c),
	  SPH_C32(0x99cfe8a3), SPH_C32(0xa9350c55), SPH_C32(0x83310000),
	  SPH_C32(0x5b7f0000), SPH_C32(0xdc2e0180), SPH_C32(0xc36b0000),
	  SPH_C32(0xe7246dd1), SPH_C32(0xc8bcfadb), SPH_C32(0x4f06b6b7),
	  SPH_C32(0x7a915d81) },
	{ SPH_C32(0xf9bd0000), SPH_C32(0xfeab0000), SPH_C32(0xcf080900),
	  SPH_C32(0x05560000), SPH_C32(0x2c97007b), SPH_C32(0x361db598),
	  SPH_C32(0x16d6024f), SPH_C32(0xed083c51), SPH_C32(0x4f250000),
	  SPH_C32(0xfe1c0000), SPH_C32(0x86970600), SPH_C32(0xf83b0000),
	  SPH_C32(0xacf47e2e), SPH_C32(0x4f27cec3), SPH_C32(0x2645fe76),
	  SPH_C32(0xb0cbda7f) },
	{ SPH_C32(0x35a90000), SPH_C32(0x5bc80000), SPH_C32(0x95b10e80),
	  SPH_C32(0x3e060000), SPH_C32(0x67471384), SPH_C32(0xb1868180),
	  SPH_C32(0x7f954a8e), SPH_C32(0x2752bbaf), SPH_C32(0xcebb0000),
	  SPH_C32(0x124b0000), SPH_C32(0xe0a50480), SPH_C32(0x6dc80000),
	  SPH_C32(0xf15d562c), SPH_C32(0x07d3f27f), SPH_C32(0xc01f5c5b),
	  SPH_C32(0x3eac6d85) },
	{ SPH_C32(0x02af0000), SPH_C32(0xb7280000), SPH_C32(0xba1c0300),
	  SPH_C32(0x56980000), SPH_C32(0xba8d45d3), SPH_C32(0x8048c667),
	  SPH_C32(0xa95c149a), SPH_C32(0xf4f6ea7b), SPH_C32(0x7a8c0000),
	  SPH_C32(0xa5d40000), SPH_C32(0x13260880), SPH_C32(0xc63d0000),
	  SPH_C32(0xcbb36daa), SPH_C32(0xfea14f43), SPH_C32(0x59d0b4f8),
	  SPH_C32(0x979961d0) },
	{ SPH_C32(0xcebb0000), SPH_C32(0x124b0000), SPH_C32(0xe0a50480),
	  SPH_C32(0x6dc80000), SPH_C32(0xf15d562c), SPH_C32(0x07d3f27f),
	  SPH_C32(0xc01f5c5b), SPH_C32(0x3eac6d85), SPH_C32(0xfb120000),
	  SPH_C32(0x49830000), SPH_C32(0x75140a00), SPH_C32(0x53ce0000),
	  SPH_C32(0x961a45a8), SPH_C32(0xb65573ff), SPH_C32(0xbf8a16d5),
	  SPH_C32(0x19fed62a) },
	{ SPH_C32(0x83310000), SPH_C32(0x5b7f0000), SPH_C32(0xdc2e0180),
	  SPH_C32(0xc36b0000), SPH_C32(0xe7246dd1), SPH_C32(0xc8bcfadb),
	  SPH_C32(0x4f06b6b7), SPH_C32(0x7a915d81), SPH_C32(0x37060000),
	  SPH_C32(0xece00000), SPH_C32(0x2fad0d80), SPH_C32(0x689e0000),
	  SPH_C32(0xddca5657), SPH_C32(0x31ce47e7), SPH_C32(0xd6c95e14),
	  SPH_C32(0xd3a451d4) },
	{ SPH_C32(0x4f250000), SPH_C32(0xfe1c0000), SPH_C32(0x86970600),
	  SPH_C32(0xf83b0000), SPH_C32(0xacf47e2e), SPH_C32(0x4f27cec3),
	  SPH_C32(0x2645fe76), SPH_C32(0xb0cbda7f), SPH_C32(0xb6980000),
	  SPH_C32(0x00b70000), SPH_C32(0x499f0f00), SPH_C32(0xfd6d0000),
	  SPH_C32(0x80637e55), SPH_C32(0x793a7b5b), SPH_C32(0x3093fc39),
	  SPH_C32(0x5dc3e62e) },
	{ SPH_C32(0x7a8c0000), SPH_C32(0xa5d40000), SPH_C32(0x13260880),
	  SPH_C32(0xc63d0000), SPH_C32(0xcbb36daa), SPH_C32(0xfea14f43),
	  SPH_C32(0x59d0b4f8), SPH_C32(0x979961d0), SPH_C32(0x78230000),
	  SPH_C32(0x12fc0000), SPH_C32(0xa93a0b80), SPH_C32(0x90a50000),
	  SPH_C32(0x713e2879), SPH_C32(0x7ee98924), SPH_C32(0xf08ca062),
	  SPH_C32(0x636f8bab) },
	{ SPH_C32(0xb6980000), SPH_C32(0x00b70000), SPH_C32(0x499f0f00),
	  SPH_C32(0xfd6d0000), SPH_C32(0x80637e55), SPH_C32(0x793a7b5b),
	  SPH_C32(0x3093fc39), SPH_C32(0x5dc3e62e), SPH_C32(0xf9bd0000),
	  SPH_C32(0xfeab0000), SPH_C32(0xcf080900), SPH_C32(0x05560000),
	  SPH_C32(0x2c97007b), SPH_C32(0x361db598), SPH_C32(0x16d6024f),
	  SPH_C32(0xed083c51) },
	{ SPH_C32(0xfb120000), SPH_C32(0x49830000), SPH_C32(0x75140a00),
	  SPH_C32(0x53ce0000), SPH_C32(0x961a45a8), SPH_C32(0xb65573ff),
	  SPH_C32(0xbf8a16d5), SPH_C32(0x19fed62a), SPH_C32(0x35a90000),
	  SPH_C32(0x5bc80000), SPH_C32(0x95b10e80), SPH_C32(0x3e060000),
	  SPH_C32(0x67471384), SPH_C32(0xb1868180), SPH_C32(0x7f954a8e),
	  SPH_C32(0x2752bbaf) },
	{ SPH_C32(0x37060000), SPH_C32(0xece00000), SPH_C32(0x2fad0d80),
	  SPH_C32(0x689e0000), SPH_C32(0xddca5657), SPH_C32(0x31ce47e7),
	  SPH_C32(0xd6c95e14), SPH_C32(0xd3a451d4), SPH_C32(0xb4370000),
	  SPH_C32(0xb79f0000), SPH_C32(0xf3830c00), SPH_C32(0xabf50000),
	  SPH_C32(0x3aee3b86), SPH_C32(0xf972bd3c), SPH_C32(0x99cfe8a3),
	  SPH_C32(0xa9350c55) }
};

static const sph_u32 T512_40[16][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x88980000), SPH_C32(0x1f940000), SPH_C32(0x7fcf002e),
	  SPH_C32(0xfb4e0000), SPH_C32(0xf158079a), SPH_C32(0x61ae9167),
	  SPH_C32(0xa895706c), SPH_C32(0xe6107494), SPH_C32(0x0bc20000),
	  SPH_C32(0xdb630000), SPH_C32(0x7e88000c), SPH_C32(0x15860000),
	  SPH_C32(0x91fd48f3), SPH_C32(0x7581bb43), SPH_C32(0xf460449e),
	  SPH_C32(0xd8b61463) },
	{ SPH_C32(0x0bc20000), SPH_C32(0xdb630000), SPH_C32(0x7e88000c),
	  SPH_C32(0x15860000), SPH_C32(0x91fd48f3), SPH_C32(0x7581bb43),
	  SPH_C32(0xf460449e), SPH_C32(0xd8b61463), SPH_C32(0x835a0000),
	  SPH_C32(0xc4f70000), SPH_C32(0x01470022), SPH_C32(0xeec80000),
	  SPH_C32(0x60a54f69), SPH_C32(0x142f2a24), SPH_C32(0x5cf534f2),
	  SPH_C32(0x3ea660f7) },
	{ SPH_C32(0x835a0000), SPH_C32(0xc4f70000), SPH_C32(0x01470022),
	  SPH_C32(0xeec80000), SPH_C32(0x60a54f69), SPH_C32(0x142f2a24),
	  SPH_C32(0x5cf534f2), SPH_C32(0x3ea660f7), SPH_C32(0x88980000),
	  SPH_C32(0x1f940000), SPH_C32(0x7fcf002e), SPH_C32(0xfb4e0000),
	  SPH_C32(0xf158079a), SPH_C32(0x61ae9167), SPH_C32(0xa895706c),
	  SPH_C32(0xe6107494) },
	{ SPH_C32(0x52500000), SPH_C32(0x29540000), SPH_C32(0x6a61004e),
	  SPH_C32(0xf0ff0000), SPH_C32(0x9a317eec), SPH_C32(0x452341ce),
	  SPH_C32(0xcf568fe5), SPH_C32(0x5303130f), SPH_C32(0x538d0000),
	  SPH_C32(0xa9fc0000), SPH_C32(0x9ef70006), SPH_C32(0x56ff0000),
	  SPH_C32(0x0ae4004e), SPH_C32(0x92c5cdf9), SPH_C32(0xa9444018),
	  SPH_C32(0x7f975691) },
	{ SPH_C32(0xdac80000), SPH_C32(0x36c00000), SPH_C32(0x15ae0060),
	  SPH_C32(0x0bb10000), SPH_C32(0x6b697976), SPH_C32(0x248dd0a9),
	  SPH_C32(0x67c3ff89), SPH_C32(0xb513679b), SPH_C32(0x584f0000),
	  SPH_C32(0x729f0000), SPH_C32(0xe07f000a), SPH_C32(0x43790000),
	  SPH_C32(0x9b1948bd), SPH_C32(0xe74476ba), SPH_C32(0x5d240486),
	  SPH_C32(0xa72142f2) },
	{ SPH_C32(0x59920000), SPH_C32(0xf2370000), SPH_C32(0x14e90042),
	  SPH_C32(0xe5790000), SPH_C32(0x0bcc361f), SPH_C32(0x30a2fa8d),
	  SPH_C32(0x3b36cb7b), SPH_C32(0x8bb5076c), SPH_C32(0xd0d70000),
	  SPH_C32(0x6d0b0000), SPH_C32(0x9fb00024), SPH_C32(0xb8370000),
	  SPH_C32(0x6a414f27), SPH_C32(0x86eae7dd), SPH_C32(0xf5b174ea),
	  SPH_C32(0x41313666) },
	{ SPH_C32(0xd10a0000), SPH_C32(0xeda30000), SPH_C32(0x6b26006c),
	  SPH_C32(0x1e370000), SPH_C32(0xfa943185), SPH_C32(0x510c6bea),
	  SPH_C32(0x93a3bb17), SPH_C32(0x6da573f8), SPH_C32(0xdb150000),
	  SPH_C32(0xb6680000), SPH_C32(0xe1380028), SPH_C32(0xadb10000),
	  SPH_C32(0xfbbc07d4), SPH_C32(0xf36b5c9e), SPH_C32(0x01d13074),
	  SPH_C32(0x99872205) },
	{ SPH_C32(0x538d0000), SPH_C32(0xa9fc0000), SPH_C32(0x9ef70006),
	  SPH_C32(0x56ff0000), SPH_C32(0x0ae4004e), SPH_C32(0x92c5cdf9),
	  SPH_C32(0xa9444018), SPH_C32(0x7f975691), SPH_C32(0x01dd0000),
	  SPH_C32(0x80a80000), SPH_C32(0xf4960048), SPH_C32(0xa6000000),
	  SPH_C32(0x90d57ea2), SPH_C32(0xd7e68c37), SPH_C32(0x6612cffd),
	  SPH_C32(0x2c94459e) },
	{ SPH_C32(0xdb150000), SPH_C32(0xb6680000), SPH_C32(0xe1380028),
	  SPH_C32(0xadb10000), SPH_C32(0xfbbc07d4), SPH_C32(0xf36b5c9e),
	  SPH_C32(0x01d13074), SPH_C32(0x99872205), SPH_C32(0x0a1f0000),
	  SPH_C32(0x5bcb0000), SPH_C32(0x8a1e0044), SPH_C32(0xb3860000),
	  SPH_C32(0x01283651), SPH_C32(0xa2673774), SPH_C32(0x92728b63),
	  SPH_C32(0xf42251fd) },
	{ SPH_C32(0x584f0000), SPH_C32(0x729f0000), SPH_C32(0xe07f000a),
	  SPH_C32(0x43790000), SPH_C32(0x9b1948bd), SPH_C32(0xe74476ba),
	  SPH_C32(0x5d240486), SPH_C32(0xa72142f2), SPH_C32(0x82870000),
	  SPH_C32(0x445f0000), SPH_C32(0xf5d1006a), SPH_C32(0x48c80000),
	  SPH_C32(0xf07031cb), SPH_C32(0xc3c9a613), SPH_C32(0x3ae7fb0f),
	  SPH_C32(0x12322569) },
	{ SPH_C32(0xd0d70000), SPH_C32(0x6d0b0000), SPH_C32(0x9fb00024),
	  SPH_C32(0xb8370000), SPH_C32(0x6a414f27), SPH_C32(0x86eae7dd),
	  SPH_C32(0xf5b174ea), SPH_C32(0x41313666), SPH_C32(0x89450000),
	  SPH_C32(0x9f3c0000), SPH_C32(0x8b590066), SPH_C32(0x5d4e0000),
	  SPH_C32(0x618d7938), SPH_C32(0xb6481d50), SPH_C32(0xce87bf91),
	  SPH_C32(0xca84310a) },
	{ SPH_C32(0x01dd0000), SPH_C32(0x80a80000), SPH_C32(0xf4960048),
	  SPH_C32(0xa6000000), SPH_C32(0x90d57ea2), SPH_C32(0xd7e68c37),
	  SPH_C32(0x6612cffd), SPH_C32(0x2c94459e), SPH_C32(0x52500000),
	  SPH_C32(0x29540000), SPH_C32(0x6a61004e), SPH_C32(0xf0ff0000),
	  SPH_C32(0x9a317eec), SPH_C32(0x452341ce), SPH_C32(0xcf568fe5),
	  SPH_C32(0x5303130f) },
	{ SPH_C32(0x89450000), SPH_C32(0x9f3c0000), SPH_C32(0x8b590066),
	  SPH_C32(0x5d4e0000), SPH_C32(0x618d7938), SPH_C32(0xb6481d50),
	  SPH_C32(0xce87bf91), SPH_C32(0xca84310a), SPH_C32(0x59920000),
	  SPH_C32(0xf2370000), SPH_C32(0x14e90042), SPH_C32(0xe5790000),
	  SPH_C32(0x0bcc361f), SPH_C32(0x30a2fa8d), SPH_C32(0x3b36cb7b),
	  SPH_C32(0x8bb5076c) },
	{ SPH_C32(0x0a1f0000), SPH_C32(0x5bcb0000), SPH_C32(0x8a1e0044),
	  SPH_C32(0xb3860000), SPH_C32(0x01283651), SPH_C32(0xa2673774),
	  SPH_C32(0x92728b63), SPH_C32(0xf42251fd), SPH_C32(0xd10a0000),
	  SPH_C32(0xeda30000), SPH_C32(0x6b26006c), SPH_C32(0x1e370000),
	  SPH_C32(0xfa943185), SPH_C32(0x510c6bea), SPH_C32(0x93a3bb17),
	  SPH_C32(0x6da573f8) },
	{ SPH_C32(0x82870000), SPH_C32(0x445f0000), SPH_C32(0xf5d1006a),
	  SPH_C32(0x48c80000), SPH_C32(0xf07031cb), SPH_C32(0xc3c9a613),
	  SPH_C32(0x3ae7fb0f), SPH_C32(0x12322569), SPH_C32(0xdac80000),
	  SPH_C32(0x36c00000), SPH_C32(0x15ae0060), SPH_C32(0x0bb10000),
	  SPH_C32(0x6b697976), SPH_C32(0x248dd0a9), SPH_C32(0x67c3ff89),
	  SPH_C32(0xb513679b) }
};

static const sph_u32 T512_44[16][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x58430000), SPH_C32(0x807e0000), SPH_C32(0x78330001),
	  SPH_C32(0xc66b3800), SPH_C32(0xe7375cdc), SPH_C32(0x79ad3fdd),
	  SPH_C32(0xac73fe6f), SPH_C32(0x3a4479b1), SPH_C32(0x1d5a0000),
	  SPH_C32(0x2b720000), SPH_C32(0x488d0000), SPH_C32(0xaf611800),
	  SPH_C32(0x25cb2ec5), SPH_C32(0xc879bfd0), SPH_C32(0x81a20429),
	  SPH_C32(0x1e7536a6) },
	{ SPH_C32(0x1d5a0000), SPH_C32(0x2b720000), SPH_C32(0x488d0000),
	  SPH_C32(0xaf611800), SPH_C32(0x25cb2ec5), SPH_C32(0xc879bfd0),
	  SPH_C32(0x81a20429), SPH_C32(0x1e7536a6), SPH_C32(0x45190000),
	  SPH_C32(0xab0c0000), SPH_C32(0x30be0001), SPH_C32(0x690a2000),
	  SPH_C32(0xc2fc7219), SPH_C32(0xb1d4800d), SPH_C32(0x2dd1fa46),
	  SPH_C32(0x24314f17) },
	{ SPH_C32(0x45190000), SPH_C32(0xab0c0000), SPH_C32(0x30be0001),
	  SPH_C32(0x690a2000), SPH_C32(0xc2fc7219), SPH_C32(0xb1d4800d),
	  SPH_C32(0x2dd1fa46), SPH_C32(0x24314f17), SPH_C32(0x58430000),
	  SPH_C32(0x807e0000), SPH_C32(0x78330001), SPH_C32(0xc66b3800),
	  SPH_C32(0xe7375cdc), SPH_C32(0x79ad3fdd), SPH_C32(0xac73fe6f),
	  SPH_C32(0x3a4479b1) },
	{ SPH_C32(0xa53b0000), SPH_C32(0x14260000), SPH_C32(0x4e30001e),
	  SPH_C32(0x7cae0000), SPH_C32(0x8f9e0dd5), SPH_C32(0x78dfaa3d),
	  SPH_C32(0xf73168d8), SPH_C32(0x0b1b4946), SPH_C32(0x07ed0000),
	  SPH_C32(0xb2500000), SPH_C32(0x8774000a), SPH_C32(0x970d0000),
	  SPH_C32(0x437223ae), SPH_C32(0x48c76ea4), SPH_C32(0xf4786222),
	  SPH_C32(0x9075b1ce) },
	{ SPH_C32(0xfd780000), SPH_C32(0x94580000), SPH_C32(0x3603001f),
	  SPH_C32(0xbac53800), SPH_C32(0x68a95109), SPH_C32(0x017295e0),
	  SPH_C32(0x5b4296b7), SPH_C32(0x315f30f7), SPH_C32(0x1ab70000),
	  SPH_C32(0x99220000), SPH_C32(0xcff9000a), SPH_C32(0x386c1800),
	  SPH_C32(0x66b90d6b), SPH_C32(0x80bed174), SPH_C32(0x75da660b),
	  SPH_C32(0x8e008768) },
	{ SPH_C32(0xb8610000), SPH_C32(0x3f540000), SPH_C32(0x06bd001e),
	  SPH_C32(0xd3cf1800), SPH_C32(0xaa552310), SPH_C32(0xb0a615ed),
	  SPH_C32(0x76936cf1), SPH_C32(0x156e7fe0), SPH_C32(0x42f40000),
	  SPH_C32(0x195c0000), SPH_C32(0xb7ca000b), SPH_C32(0xfe072000),
	  SPH_C32(0x818e51b7), SPH_C32(0xf913eea9), SPH_C32(0xd9a99864),
	  SPH_C32(0xb444fed9) },
	{ SPH_C32(0xe0220000), SPH_C32(0xbf2a0000), SPH_C32(0x7e8e001f),
	  SPH_C32(0x15a42000), SPH_C32(0x4d627fcc), SPH_C32(0xc90b2a30),
	  SPH_C32(0xdae0929e), SPH_C32(0x2f2a0651), SPH_C32(0x5fae0000),
	  SPH_C32(0x322e0000), SPH_C32(0xff47000b), SPH_C32(0x51663800),
	  SPH_C32(0xa4457f72), SPH_C32(0x316a5179), SPH_C32(0x580b9c4d),
	  SPH_C32(0xaa31c87f) },
	{ SPH_C32(0x07ed0000), SPH_C32(0xb2500000), SPH_C32(0x8774000a),
	  SPH_C32(0x970d0000), SPH_C32(0x437223ae), SPH_C32(0x48c76ea4),
	  SPH_C32(0xf4786222), SPH_C32(0x9075b1ce), SPH_C32(0xa2d60000),
	  SPH_C32(0xa6760000), SPH_C32(0xc9440014), SPH_C32(0xeba30000),
	  SPH_C32(0xccec2e7b), SPH_C32(0x3018c499), SPH_C32(0x03490afa),
	  SPH_C32(0x9b6ef888) },
	{ SPH_C32(0x5fae0000), SPH_C32(0x322e0000), SPH_C32(0xff47000b),
	  SPH_C32(0x51663800), SPH_C32(0xa4457f72), SPH_C32(0x316a5179),
	  SPH_C32(0x580b9c4d), SPH_C32(0xaa31c87f), SPH_C32(0xbf8c0000),
	  SPH_C32(0x8d040000), SPH_C32(0x81c90014), SPH_C32(0x44c21800),
	  SPH_C32(0xe92700be), SPH_C32(0xf8617b49), SPH_C32(0x82eb0ed3),
	  SPH_C32(0x851bce2e) },
	{ SPH_C32(0x1ab70000), SPH_C32(0x99220000), SPH_C32(0xcff9000a),
	  SPH_C32(0x386c1800), SPH_C32(0x66b90d6b), SPH_C32(0x80bed174),
	  SPH_C32(0x75da660b), SPH_C32(0x8e008768), SPH_C32(0xe7cf0000),
	  SPH_C32(0x0d7a0000), SPH_C32(0xf9fa0015), SPH_C32(0x82a92000),
	  SPH_C32(0x0e105c62), SPH_C32(0x81cc4494), SPH_C32(0x2e98f0bc),
	  SPH_C32(0xbf5fb79f) },
	{ SPH_C32(0x42f40000), SPH_C32(0x195c0000), SPH_C32(0xb7ca000b),
	  SPH_C32(0xfe072000), SPH_C32(0x818e51b7), SPH_C32(0xf913eea9),
	  SPH_C32(0xd9a99864), SPH_C32(0xb444fed9), SPH_C32(0xfa950000),
	  SPH_C32(0x26080000), SPH_C32(0xb1770015), SPH_C32(0x2dc83800),
	  SPH_C32(0x2bdb72a7), SPH_C32(0x49b5fb44), SPH_C32(0xaf3af495),
	  SPH_C32(0xa12a8139) },
	{ SPH_C32(0xa2d60000), SPH_C32(0xa6760000), SPH_C32(0xc9440014),
	  SPH_C32(0xeba30000), SPH_C32(0xccec2e7b), SPH_C32(0x3018c499),
	  SPH_C32(0x03490afa), SPH_C32(0x9b6ef888), SPH_C32(0xa53b0000),
	  SPH_C32(0x14260000), SPH_C32(0x4e30001e), SPH_C32(0x7cae0000),
	  SPH_C32(0x8f9e0dd5), SPH_C32(0x78dfaa3d), SPH_C32(0xf73168d8),
	  SPH_C32(0x0b1b4946) },
	{ SPH_C32(0xfa950000), SPH_C32(0x26080000), SPH_C32(0xb1770015),
	  SPH_C32(0x2dc83800), SPH_C32(0x2bdb72a7), SPH_C32(0x49b5fb44),
	  SPH_C32(0xaf3af495), SPH_C32(0xa12a8139), SPH_C32(0xb8610000),
	  SPH_C32(0x3f540000), SPH_C32(0x06bd001e), SPH_C32(0xd3cf1800),
	  SPH_C32(0xaa552310), SPH_C32(0xb0a615ed), SPH_C32(0x76936cf1),
	  SPH_C32(0x156e7fe0) },
	{ SPH_C32(0xbf8c0000), SPH_C32(0x8d040000), SPH_C32(0x81c90014),
	  SPH_C32(0x44c21800), SPH_C32(0xe92700be), SPH_C32(0xf8617b49),
	  SPH_C32(0x82eb0ed3), SPH_C32(0x851bce2e), SPH_C32(0xe0220000),
	  SPH_C32(0xbf2a0000), SPH_C32(0x7e8e001f), SPH_C32(0x15a42000),
	  SPH_C32(0x4d627fcc), SPH_C32(0xc90b2a30), SPH_C32(0xdae0929e),
	  SPH_C32(0x2f2a0651) },
	{ SPH_C32(0xe7cf0000), SPH_C32(0x0d7a0000), SPH_C32(0xf9fa0015),
	  SPH_C32(0x82a92000), SPH_C32(0x0e105c62), SPH_C32(0x81cc4494),
	  SPH_C32(0x2e98f0bc), SPH_C32(0xbf5fb79f), SPH_C32(0xfd780000),
	  SPH_C32(0x94580000), SPH_C32(0x3603001f), SPH_C32(0xbac53800),
	  SPH_C32(0x68a95109), SPH_C32(0x017295e0), SPH_C32(0x5b4296b7),
	  SPH_C32(0x315f30f7) }
};

static const sph_u32 T512_48[16][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x02f20000), SPH_C32(0xa2810000), SPH_C32(0x873f0000),
	  SPH_C32(0xe36c7800), SPH_C32(0x1e1d74ef), SPH_C32(0x073d2bd6),
	  SPH_C32(0xc4c23237), SPH_C32(0x7f32259e), SPH_C32(0xbadd0000),
	  SPH_C32(0x13ad0000), SPH_C32(0xb7e70000), SPH_C32(0xf7282800),
	  SPH_C32(0xdf45144d), SPH_C32(0x361ac33a), SPH_C32(0xea5a8d14),
	  SPH_C32(0x2a2c18f0) },
	{ SPH_C32(0xbadd0000), SPH_C32(0x13ad0000), SPH_C32(0xb7e70000),
	  SPH_C32(0xf7282800), SPH_C32(0xdf45144d), SPH_C32(0x361ac33a),
	  SPH_C32(0xea5a8d14), SPH_C32(0x2a2c18f0), SPH_C32(0xb82f0000),
	  SPH_C32(0xb12c0000), SPH_C32(0x30d80000), SPH_C32(0x14445000),
	  SPH_C32(0xc15860a2), SPH_C32(0x3127e8ec), SPH_C32(0x2e98bf23),
	  SPH_C32(0x551e3d6e) },
	{ SPH_C32(0xb82f0000), SPH_C32(0xb12c0000), SPH_C32(0x30d80000),
	  SPH_C32(0x14445000), SPH_C32(0xc15860a2), SPH_C32(0x3127e8ec),
	  SPH_C32(0x2e98bf23), SPH_C32(0x551e3d6e), SPH_C32(0x02f20000),
	  SPH_C32(0xa2810000), SPH_C32(0x873f0000), SPH_C32(0xe36c7800),
	  SPH_C32(0x1e1d74ef), SPH_C32(0x073d2bd6), SPH_C32(0xc4c23237),
	  SPH_C32(0x7f32259e) },
	{ SPH_C32(0x1e6c0000), SPH_C32(0xc4420000), SPH_C32(0x8a2e0000),
	  SPH_C32(0xbcb6b800), SPH_C32(0x2c4413b6), SPH_C32(0x8bfdd3da),
	  SPH_C32(0x6a0c1bc8), SPH_C32(0xb99dc2eb), SPH_C32(0x92560000),
	  SPH_C32(0x1eda0000), SPH_C32(0xea510000), SPH_C32(0xe8b13000),
	  SPH_C32(0xa93556a5), SPH_C32(0xebfb6199), SPH_C32(0xb15c2254),
	  SPH_C32(0x33c5244f) },
	{ SPH_C32(0x1c9e0000), SPH_C32(0x66c30000), SPH_C32(0x0d110000),
	  SPH_C32(0x5fdac000), SPH_C32(0x32596759), SPH_C32(0x8cc0f80c),
	  SPH_C32(0xaece29ff), SPH_C32(0xc6afe775), SPH_C32(0x288b0000),
	  SPH_C32(0x0d770000), SPH_C32(0x5db60000), SPH_C32(0x1f991800),
	  SPH_C32(0x767042e8), SPH_C32(0xdde1a2a3), SPH_C32(0x5b06af40),
	  SPH_C32(0x19e93cbf) },
	{ SPH_C32(0xa4b10000), SPH_C32(0xd7ef0000), SPH_C32(0x3dc90000),
	  SPH_C32(0x4b9e9000), SPH_C32(0xf30107fb), SPH_C32(0xbde710e0),
	  SPH_C32(0x805696dc), SPH_C32(0x93b1da1b), SPH_C32(0x2a790000),
	  SPH_C32(0xaff60000), SPH_C32(0xda890000), SPH_C32(0xfcf56000),
	  SPH_C32(0x686d3607), SPH_C32(0xdadc8975), SPH_C32(0x9fc49d77),
	  SPH_C32(0x66db1921) },
	{ SPH_C32(0xa6430000), SPH_C32(0x756e0000), SPH_C32(0xbaf60000),
	  SPH_C32(0xa8f2e800), SPH_C32(0xed1c7314), SPH_C32(0xbada3b36),
	  SPH_C32(0x4494a4eb), SPH_C32(0xec83ff85), SPH_C32(0x90a40000),
	  SPH_C32(0xbc5b0000), SPH_C32(0x6d6e0000), SPH_C32(0x0bdd4800),
	  SPH_C32(0xb728224a), SPH_C32(0xecc64a4f), SPH_C32(0x759e1063),
	  SPH_C32(0x4cf701d1) },
	{ SPH_C32(0x92560000), SPH_C32(0x1eda0000), SPH_C32(0xea510000),
	  SPH_C32(0xe8b13000), SPH_C32(0xa93556a5), SPH_C32(0xebfb6199),
	  SPH_C32(0xb15c2254), SPH_C32(0x33c5244f), SPH_C32(0x8c3a0000),
	  SPH_C32(0xda980000), SPH_C32(0x607f0000), SPH_C32(0x54078800),
	  SPH_C32(0x85714513), SPH_C32(0x6006b243), SPH_C32(0xdb50399c),
	  SPH_C32(0x8a58e6a4) },
	{ SPH_C32(0x90a40000), SPH_C32(0xbc5b0000), SPH_C32(0x6d6e0000),
	  SPH_C32(0x0bdd4800), SPH_C32(0xb728224a), SPH_C32(0xecc64a4f),
	  SPH_C32(0x759e1063), SPH_C32(0x4cf701d1), SPH_C32(0x36e70000),
	  SPH_C32(0xc9350000), SPH_C32(0xd7980000), SPH_C32(0xa32fa000),
	  SPH_C32(0x5a34515e), SPH_C32(0x561c7179), SPH_C32(0x310ab488),
	  SPH_C32(0xa074fe54) },
	{ SPH_C32(0x288b0000), SPH_C32(0x0d770000), SPH_C32(0x5db60000),
	  SPH_C32(0x1f991800), SPH_C32(0x767042e8), SPH_C32(0xdde1a2a3),
	  SPH_C32(0x5b06af40), SPH_C32(0x19e93cbf), SPH_C32(0x34150000),
	  SPH_C32(0x6bb40000), SPH_C32(0x50a70000), SPH_C32(0x4043d800),
	  SPH_C32(0x442925b1), SPH_C32(0x51215aaf), SPH_C32(0xf5c886bf),
	  SPH_C32(0xdf46dbca) },
	{ SPH_C32(0x2a790000), SPH_C32(0xaff60000), SPH_C32(0xda890000),
	  SPH_C32(0xfcf56000), SPH_C32(0x686d3607), SPH_C32(0xdadc8975),
	  SPH_C32(0x9fc49d77), SPH_C32(0x66db1921), SPH_C32(0x8ec80000),
	  SPH_C32(0x78190000), SPH_C32(0xe7400000), SPH_C32(0xb76bf000),
	  SPH_C32(0x9b6c31fc), SPH_C32(0x673b9995), SPH_C32(0x1f920bab),
	  SPH_C32(0xf56ac33a) },
	{ SPH_C32(0x8c3a0000), SPH_C32(0xda980000), SPH_C32(0x607f0000),
	  SPH_C32(0x54078800), SPH_C32(0x85714513), SPH_C32(0x6006b243),
	  SPH_C32(0xdb50399c), SPH_C32(0x8a58e6a4), SPH_C32(0x1e6c0000),
	  SPH_C32(0xc4420000), SPH_C32(0x8a2e0000), SPH_C32(0xbcb6b800),
	  SPH_C32(0x2c4413b6), SPH_C32(0x8bfdd3da), SPH_C32(0x6a0c1bc8),
	  SPH_C32(0xb99dc2eb) },
	{ SPH_C32(0x8ec80000), SPH_C32(0x78190000), SPH_C32(0xe7400000),
	  SPH_C32(0xb76bf000), SPH_C32(0x9b6c31fc), SPH_C32(0x673b9995),
	  SPH_C32(0x1f920bab), SPH_C32(0xf56ac33a), SPH_C32(0xa4b10000),
	  SPH_C32(0xd7ef0000), SPH_C32(0x3dc90000), SPH_C32(0x4b9e9000),
	  SPH_C32(0xf30107fb), SPH_C32(0xbde710e0), SPH_C32(0x805696dc),
	  SPH_C32(0x93b1da1b) },
	{ SPH_C32(0x36e70000), SPH_C32(0xc9350000), SPH_C32(0xd7980000),
	  SPH_C32(0xa32fa000), SPH_C32(0x5a34515e), SPH_C32(0x561c7179),
	  SPH_C32(0x310ab488), SPH_C32(0xa074fe54), SPH_C32(0xa6430000),
	  SPH_C32(0x756e0000), SPH_C32(0xbaf60000), SPH_C32(0xa8f2e800),
	  SPH_C32(0xed1c7314), SPH_C32(0xbada3b36), SPH_C32(0x4494a4eb),
	  SPH_C32(0xec83ff85) },
	{ SPH_C32(0x34150000), SPH_C32(0x6bb40000), SPH_C32(0x50a70000),
	  SPH_C32(0x4043d800), SPH_C32(0x442925b1), SPH_C32(0x51215aaf),
	  SPH_C32(0xf5c886bf), SPH_C32(0xdf46dbca), SPH_C32(0x1c9e0000),
	  SPH_C32(0x66c30000), SPH_C32(0x0d110000), SPH_C32(0x5fdac000),
	  SPH_C32(0x32596759), SPH_C32(0x8cc0f80c), SPH_C32(0xaece29ff),
	  SPH_C32(0xc6afe775) }
};

static const sph_u32 T512_52[16][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xe6280000), SPH_C32(0x4c4b0000), SPH_C32(0xa8550000),
	  SPH_C32(0xd3d002e0), SPH_C32(0xd86130b8), SPH_C32(0x98a7b0da),
	  SPH_C32(0x289506b4), SPH_C32(0xd75a4897), SPH_C32(0xf0c50000),
	  SPH_C32(0x59230000), SPH_C32(0x45820000), SPH_C32(0xe18d00c0),
	  SPH_C32(0x3b6d0631), SPH_C32(0xc2ed5699), SPH_C32(0xcbe0fe1c),
	  SPH_C32(0x56a7b19f) },
	{ SPH_C32(0xf0c50000), SPH_C32(0x59230000), SPH_C32(0x45820000),
	  SPH_C32(0xe18d00c0), SPH_C32(0x3b6d0631), SPH_C32(0xc2ed5699),
	  SPH_C32(0xcbe0fe1c), SPH_C32(0x56a7b19f), SPH_C32(0x16ed0000),
	  SPH_C32(0x15680000), SPH_C32(0xedd70000), SPH_C32(0x325d0220),
	  SPH_C32(0xe30c3689), SPH_C32(0x5a4ae643), SPH_C32(0xe375f8a8),
	  SPH_C32(0x81fdf908) },
	{ SPH_C32(0x16ed0000), SPH_C32(0x15680000), SPH_C32(0xedd70000),
	  SPH_C32(0x325d0220), SPH_C32(0xe30c3689), SPH_C32(0x5a4ae643),
	  SPH_C32(0xe375f8a8), SPH_C32(0x81fdf908), SPH_C32(0xe6280000),
	  SPH_C32(0x4c4b0000), SPH_C32(0xa8550000), SPH_C32(0xd3d002e0),
	  SPH_C32(0xd86130b8), SPH_C32(0x98a7b0da), SPH_C32(0x289506b4),
	  SPH_C32(0xd75a4897) },
	{ SPH_C32(0xb4310000), SPH_C32(0x77330000), SPH_C32(0xb15d0000),
	  SPH_C32(0x7fd004e0), SPH_C32(0x78a26138), SPH_C32(0xd116c35d),
	  SPH_C32(0xd256d489), SPH_C32(0x4e6f74de), SPH_C32(0xe3060000),
	  SPH_C32(0xbdc10000), SPH_C32(0x87130000), SPH_C32(0xbff20060),
	  SPH_C32(0x2eba0a1a), SPH_C32(0x8db53751), SPH_C32(0x73c5ab06),
	  SPH_C32(0x5bd61539) },
	{ SPH_C32(0x52190000), SPH_C32(0x3b780000), SPH_C32(0x19080000),
	  SPH_C32(0xac000600), SPH_C32(0xa0c35180), SPH_C32(0x49b17387),
	  SPH_C32(0xfac3d23d), SPH_C32(0x99353c49), SPH_C32(0x13c30000),
	  SPH_C32(0xe4e20000), SPH_C32(0xc2910000), SPH_C32(0x5e7f00a0),
	  SPH_C32(0x15d70c2b), SPH_C32(0x4f5861c8), SPH_C32(0xb825551a),
	  SPH_C32(0x0d71a4a6) },
	{ SPH_C32(0x44f40000), SPH_C32(0x2e100000), SPH_C32(0xf4df0000),
	  SPH_C32(0x9e5d0420), SPH_C32(0x43cf6709), SPH_C32(0x13fb95c4),
	  SPH_C32(0x19b62a95), SPH_C32(0x18c8c541), SPH_C32(0xf5eb0000),
	  SPH_C32(0xa8a90000), SPH_C32(0x6ac40000), SPH_C32(0x8daf0240),
	  SPH_C32(0xcdb63c93), SPH_C32(0xd7ffd112), SPH_C32(0x90b053ae),
	  SPH_C32(0xda2bec31) },
	{ SPH_C32(0xa2dc0000), SPH_C32(0x625b0000), SPH_C32(0x5c8a0000),
	  SPH_C32(0x4d8d06c0), SPH_C32(0x9bae57b1), SPH_C32(0x8b5c251e),
	  SPH_C32(0x31232c21), SPH_C32(0xcf928dd6), SPH_C32(0x052e0000),
	  SPH_C32(0xf18a0000), SPH_C32(0x2f460000), SPH_C32(0x6c220280),
	  SPH_C32(0xf6db3aa2), SPH_C32(0x1512878b), SPH_C32(0x5b50adb2),
	  SPH_C32(0x8c8c5dae) },
	{ SPH_C32(0xe3060000), SPH_C32(0xbdc10000), SPH_C32(0x87130000),
	  SPH_C32(0xbff20060), SPH_C32(0x2eba0a1a), SPH_C32(0x8db53751),
	  SPH_C32(0x73c5ab06), SPH_C32(0x5bd61539), SPH_C32(0x57370000),
	  SPH_C32(0xcaf20000), SPH_C32(0x364e0000), SPH_C32(0xc0220480),
	  SPH_C32(0x56186b22), SPH_C32(0x5ca3f40c), SPH_C32(0xa1937f8f),
	  SPH_C32(0x15b961e7) },
	{ SPH_C32(0x052e0000), SPH_C32(0xf18a0000), SPH_C32(0x2f460000),
	  SPH_C32(0x6c220280), SPH_C32(0xf6db3aa2), SPH_C32(0x1512878b),
	  SPH_C32(0x5b50adb2), SPH_C32(0x8c8c5dae), SPH_C32(0xa7f20000),
	  SPH_C32(0x93d10000), SPH_C32(0x73cc0000), SPH_C32(0x21af0440),
	  SPH_C32(0x6d756d13), SPH_C32(0x9e4ea295), SPH_C32(0x6a738193),
	  SPH_C32(0x431ed078) },
	{ SPH_C32(0x13c30000), SPH_C32(0xe4e20000), SPH_C32(0xc2910000),
	  SPH_C32(0x5e7f00a0), SPH_C32(0x15d70c2b), SPH_C32(0x4f5861c8),
	  SPH_C32(0xb825551a), SPH_C32(0x0d71a4a6), SPH_C32(0x41da0000),
	  SPH_C32(0xdf9a0000), SPH_C32(0xdb990000), SPH_C32(0xf27f06a0),
	  SPH_C32(0xb5145dab), SPH_C32(0x06e9124f), SPH_C32(0x42e68727),
	  SPH_C32(0x944498ef) },
	{ SPH_C32(0xf5eb0000), SPH_C32(0xa8a90000), SPH_C32(0x6ac40000),
	  SPH_C32(0x8daf0240), SPH_C32(0xcdb63c93), SPH_C32(0xd7ffd112),
	  SPH_C32(0x90b053ae), SPH_C32(0xda2bec31), SPH_C32(0xb11f0000),
	  SPH_C32(0x86b90000), SPH_C32(0x9e1b0000), SPH_C32(0x13f20660),
	  SPH_C32(0x8e795b9a), SPH_C32(0xc40444d6), SPH_C32(0x8906793b),
	  SPH_C32(0xc2e32970) },
	{ SPH_C32(0x57370000), SPH_C32(0xcaf20000), SPH_C32(0x364e0000),
	  SPH_C32(0xc0220480), SPH_C32(0x56186b22), SPH_C32(0x5ca3f40c),
	  SPH_C32(0xa1937f8f), SPH_C32(0x15b961e7), SPH_C32(0xb4310000),
	  SPH_C32(0x77330000), SPH_C32(0xb15d0000), SPH_C32(0x7fd004e0),
	  SPH_C32(0x78a26138), SPH_C32(0xd116c35d), SPH_C32(0xd256d489),
	  SPH_C32(0x4e6f74de) },
	{ SPH_C32(0xb11f0000), SPH_C32(0x86b90000), SPH_C32(0x9e1b0000),
	  SPH_C32(0x13f20660), SPH_C32(0x8e795b9a), SPH_C32(0xc40444d6),
	  SPH_C32(0x8906793b), SPH_C32(0xc2e32970), SPH_C32(0x44f40000),
	  SPH_C32(0x2e100000), SPH_C32(0xf4df0000), SPH_C32(0x9e5d0420),
	  SPH_C32(0x43cf6709), SPH_C32(0x13fb95c4), SPH_C32(0x19b62a95),
	  SPH_C32(0x18c8c541) },
	{ SPH_C32(0xa7f20000), SPH_C32(0x93d10000), SPH_C32(0x73cc0000),
	  SPH_C32(0x21af0440), SPH_C32(0x6d756d13), SPH_C32(0x9e4ea295),
	  SPH_C32(0x6a738193), SPH_C32(0x431ed078), SPH_C32(0xa2dc0000),
	  SPH_C32(0x625b0000), SPH_C32(0x5c8a0000), SPH_C32(0x4d8d06c0),
	  SPH_C32(0x9bae57b1), SPH_C32(0x8b5c251e), SPH_C32(0x31232c21),
	  SPH_C32(0xcf928dd6) },
	{ SPH_C32(0x41da0000), SPH_C32(0xdf9a0000), SPH_C32(0xdb990000),
	  SPH_C32(0xf27f06a0), SPH_C32(0xb5145dab), SPH_C32(0x06e9124f),
	  SPH_C32(0x42e68727), SPH_C32(0x944498ef), SPH_C32(0x52190000),
	  SPH_C32(0x3b780000), SPH_C32(0x19080000), SPH_C32(0xac000600),
	  SPH_C32(0xa0c35180), SPH_C32(0x49b17387), SPH_C32(0xfac3d23d),
	  SPH_C32(0x99353c49) }
};

static const sph_u32 T512_56[16][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x5fa80000), SPH_C32(0x56030000), SPH_C32(0x43ae0000),
	  SPH_C32(0x64f30013), SPH_C32(0x257e86bf), SPH_C32(0x1311944e),
	  SPH_C32(0x541e95bf), SPH_C32(0x8ea4db69), SPH_C32(0x00440000),
	  SPH_C32(0x7f480000), SPH_C32(0xda7c0000), SPH_C32(0x2a230001),
	  SPH_C32(0x3badc9cc), SPH_C32(0xa9b69c87), SPH_C32(0x030a9e60),
	  SPH_C32(0xbe0a679e) },
	{ SPH_C32(0x00440000), SPH_C32(0x7f480000), SPH_C32(0xda7c0000),
	  SPH_C32(0x2a230001), SPH_C32(0x3badc9cc), SPH_C32(0xa9b69c87),
	  SPH_C32(0x030a9e60), SPH_C32(0xbe0a679e), SPH_C32(0x5fec0000),
	  SPH_C32(0x294b0000), SPH_C32(0x99d20000), SPH_C32(0x4ed00012),
	  SPH_C32(0x1ed34f73), SPH_C32(0xbaa708c9), SPH_C32(0x57140bdf),
	  SPH_C32(0x30aebcf7) },
	{ SPH_C32(0x5fec0000), SPH_C32(0x294b0000), SPH_C32(0x99d20000),
	  SPH_C32(0x4ed00012), SPH_C32(0x1ed34f73), SPH_C32(0xbaa708c9),
	  SPH_C32(0x57140bdf), SPH_C32(0x30aebcf7), SPH_C32(0x5fa80000),
	  SPH_C32(0x56030000), SPH_C32(0x43ae0000), SPH_C32(0x64f30013),
	  SPH_C32(0x257e86bf), SPH_C32(0x1311944e), SPH_C32(0x541e95bf),
	  SPH_C32(0x8ea4db69) },
	{ SPH_C32(0xee930000), SPH_C32(0xd6070000), SPH_C32(0x92c10000),
	  SPH_C32(0x2b9801e0), SPH_C32(0x9451287c), SPH_C32(0x3b6cfb57),
	  SPH_C32(0x45312374), SPH_C32(0x201f6a64), SPH_C32(0x7b280000),
	  SPH_C32(0x57420000), SPH_C32(0xa9e50000), SPH_C32(0x634300a0),
	  SPH_C32(0x9edb442f), SPH_C32(0x6d9995bb), SPH_C32(0x27f83b03),
	  SPH_C32(0xc7ff60f0) },
	{ SPH_C32(0xb13b0000), SPH_C32(0x80040000), SPH_C32(0xd16f0000),
	  SPH_C32(0x4f6b01f3), SPH_C32(0xb12faec3), SPH_C32(0x287d6f19),
	  SPH_C32(0x112fb6cb), SPH_C32(0xaebbb10d), SPH_C32(0x7b6c0000),
	  SPH_C32(0x280a0000), SPH_C32(0x73990000), SPH_C32(0x496000a1),
	  SPH_C32(0xa5768de3), SPH_C32(0xc42f093c), SPH_C32(0x24f2a563),
	  SPH_C32(0x79f5076e) },
	{ SPH_C32(0xeed70000), SPH_C32(0xa94f0000), SPH_C32(0x48bd0000),
	  SPH_C32(0x01bb01e1), SPH_C32(0xaffce1b0), SPH_C32(0x92da67d0),
	  SPH_C32(0x463bbd14), SPH_C32(0x9e150dfa), SPH_C32(0x24c40000),
	  SPH_C32(0x7e090000), SPH_C32(0x30370000), SPH_C32(0x2d9300b2),
	  SPH_C32(0x80080b5c), SPH_C32(0xd73e9d72), SPH_C32(0x70ec30dc),
	  SPH_C32(0xf751dc07) },
	{ SPH_C32(0xb17f0000), SPH_C32(0xff4c0000), SPH_C32(0x0b130000),
	  SPH_C32(0x654801f2), SPH_C32(0x8a82670f), SPH_C32(0x81cbf39e),
	  SPH_C32(0x122528ab), SPH_C32(0x10b1d693), SPH_C32(0x24800000),
	  SPH_C32(0x01410000), SPH_C32(0xea4b0000), SPH_C32(0x07b000b3),
	  SPH_C32(0xbba5c290), SPH_C32(0x7e8801f5), SPH_C32(0x73e6aebc),
	  SPH_C32(0x495bbb99) },
	{ SPH_C32(0x7b280000), SPH_C32(0x57420000), SPH_C32(0xa9e50000),
	  SPH_C32(0x634300a0), SPH_C32(0x9edb442f), SPH_C32(0x6d9995bb),
	  SPH_C32(0x27f83b03), SPH_C32(0xc7ff60f0), SPH_C32(0x95bb0000),
	  SPH_C32(0x81450000), SPH_C32(0x3b240000), SPH_C32(0x48db0140),
	  SPH_C32(0x0a8a6c53), SPH_C32(0x56f56eec), SPH_C32(0x62c91877),
	  SPH_C32(0xe7e00a94) },
	{ SPH_C32(0x24800000), SPH_C32(0x01410000), SPH_C32(0xea4b0000),
	  SPH_C32(0x07b000b3), SPH_C32(0xbba5c290), SPH_C32(0x7e8801f5),
	  SPH_C32(0x73e6aebc), SPH_C32(0x495bbb99), SPH_C32(0x95ff0000),
	  SPH_C32(0xfe0d0000), SPH_C32(0xe1580000), SPH_C32(0x62f80141),
	  SPH_C32(0x3127a59f), SPH_C32(0xff43f26b), SPH_C32(0x61c38617),
	  SPH_C32(0x59ea6d0a) },
	{ SPH_C32(0x7b6c0000), SPH_C32(0x280a0000), SPH_C32(0x73990000),
	  SPH_C32(0x496000a1), SPH_C32(0xa5768de3), SPH_C32(0xc42f093c),
	  SPH_C32(0x24f2a563), SPH_C32(0x79f5076e), SPH_C32(0xca570000),
	  SPH_C32(0xa80e0000), SPH_C32(0xa2f60000), SPH_C32(0x060b0152),
	  SPH_C32(0x14592320), SPH_C32(0xec526625), SPH_C32(0x35dd13a8),
	  SPH_C32(0xd74eb663) },
	{ SPH_C32(0x24c40000), SPH_C32(0x7e090000), SPH_C32(0x30370000),
	  SPH_C32(0x2d9300b2), SPH_C32(0x80080b5c), SPH_C32(0xd73e9d72),
	  SPH_C32(0x70ec30dc), SPH_C32(0xf751dc07), SPH_C32(0xca130000),
	  SPH_C32(0xd7460000), SPH_C32(0x788a0000), SPH_C32(0x2c280153),
	  SPH_C32(0x2ff4eaec), SPH_C32(0x45e4faa2), SPH_C32(0x36d78dc8),
	  SPH_C32(0x6944d1fd) },
	{ SPH_C32(0x95bb0000), SPH_C32(0x81450000), SPH_C32(0x3b240000),
	  SPH_C32(0x48db0140), SPH_C32(0x0a8a6c53), SPH_C32(0x56f56eec),
	  SPH_C32(0x62c91877), SPH_C32(0xe7e00a94), SPH_C32(0xee930000),
	  SPH_C32(0xd6070000), SPH_C32(0x92c10000), SPH_C32(0x2b9801e0),
	  SPH_C32(0x9451287c), SPH_C32(0x3b6cfb57), SPH_C32(0x45312374),
	  SPH_C32(0x201f6a64) },
	{ SPH_C32(0xca130000), SPH_C32(0xd7460000), SPH_C32(0x788a0000),
	  SPH_C32(0x2c280153), SPH_C32(0x2ff4eaec), SPH_C32(0x45e4faa2),
	  SPH_C32(0x36d78dc8), SPH_C32(0x6944d1fd), SPH_C32(0xeed70000),
	  SPH_C32(0xa94f0000), SPH_C32(0x48bd0000), SPH_C32(0x01bb01e1),
	  SPH_C32(0xaffce1b0), SPH_C32(0x92da67d0), SPH_C32(0x463bbd14),
	  SPH_C32(0x9e150dfa) },
	{ SPH_C32(0x95ff0000), SPH_C32(0xfe0d0000), SPH_C32(0xe1580000),
	  SPH_C32(0x62f80141), SPH_C32(0x3127a59f), SPH_C32(0xff43f26b),
	  SPH_C32(0x61c38617), SPH_C32(0x59ea6d0a), SPH_C32(0xb17f0000),
	  SPH_C32(0xff4c0000), SPH_C32(0x0b130000), SPH_C32(0x654801f2),
	  SPH_C32(0x8a82670f), SPH_C32(0x81cbf39e), SPH_C32(0x122528ab),
	  SPH_C32(0x10b1d693) },
	{ SPH_C32(0xca570000), SPH_C32(0xa80e0000), SPH_C32(0xa2f60000),
	  SPH_C32(0x060b0152), SPH_C32(0x14592320), SPH_C32(0xec526625),
	  SPH_C32(0x35dd13a8), SPH_C32(0xd74eb663), SPH_C32(0xb13b0000),
	  SPH_C32(0x80040000), SPH_C32(0xd16f0000), SPH_C32(0x4f6b01f3),
	  SPH_C32(0xb12faec3), SPH_C32(0x287d6f19), SPH_C32(0x112fb6cb),
	  SPH_C32(0xaebbb10d) }
};

static const sph_u32 T512_60[16][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x033d0000), SPH_C32(0x08b30000), SPH_C32(0xf33a0000),
	  SPH_C32(0x3ac20007), SPH_C32(0x51298a50), SPH_C32(0x6b6e661f),
	  SPH_C32(0x0ea5cfe3), SPH_C32(0xe6da7ffe), SPH_C32(0xa8da0000),
	  SPH_C32(0x96be0000), SPH_C32(0x5c1d0000), SPH_C32(0x07da0002),
	  SPH_C32(0x7d669583), SPH_C32(0x1f98708a), SPH_C32(0xbb668808),
	  SPH_C32(0xda878000) },
	{ SPH_C32(0xa8da0000), SPH_C32(0x96be0000), SPH_C32(0x5c1d0000),
	  SPH_C32(0x07da0002), SPH_C32(0x7d669583), SPH_C32(0x1f98708a),
	  SPH_C32(0xbb668808), SPH_C32(0xda878000), SPH_C32(0xabe70000),
	  SPH_C32(0x9e0d0000), SPH_C32(0xaf270000), SPH_C32(0x3d180005),
	  SPH_C32(0x2c4f1fd3), SPH_C32(0x74f61695), SPH_C32(0xb5c347eb),
	  SPH_C32(0x3c5dfffe) },
	{ SPH_C32(0xabe70000), SPH_C32(0x9e0d0000), SPH_C32(0xaf270000),
	  SPH_C32(0x3d180005), SPH_C32(0x2c4f1fd3), SPH_C32(0x74f61695),
	  SPH_C32(0xb5c347eb), SPH_C32(0x3c5dfffe), SPH_C32(0x033d0000),
	  SPH_C32(0x08b30000), SPH_C32(0xf33a0000), SPH_C32(0x3ac20007),
	  SPH_C32(0x51298a50), SPH_C32(0x6b6e661f), SPH_C32(0x0ea5cfe3),
	  SPH_C32(0xe6da7ffe) },
	{ SPH_C32(0x01930000), SPH_C32(0xe7820000), SPH_C32(0xedfb0000),
	  SPH_C32(0xcf0c000b), SPH_C32(0x8dd08d58), SPH_C32(0xbca3b42e),
	  SPH_C32(0x063661e1), SPH_C32(0x536f9e7b), SPH_C32(0x92280000),
	  SPH_C32(0xdc850000), SPH_C32(0x57fa0000), SPH_C32(0x56dc0003),
	  SPH_C32(0xbae92316), SPH_C32(0x5aefa30c), SPH_C32(0x90cef752),
	  SPH_C32(0x7b1675d7) },
	{ SPH_C32(0x02ae0000), SPH_C32(0xef310000), SPH_C32(0x1ec10000),
	  SPH_C32(0xf5ce000c), SPH_C32(0xdcf90708), SPH_C32(0xd7cdd231),
	  SPH_C32(0x0893ae02), SPH_C32(0xb5b5e185), SPH_C32(0x3af20000),
	  SPH_C32(0x4a3b0000), SPH_C32(0x0be70000), SPH_C32(0x51060001),
	  SPH_C32(0xc78fb695), SPH_C32(0x4577d386), SPH_C32(0x2ba87f5a),
	  SPH_C32(0xa191f5d7) },
	{ SPH_C32(0xa9490000), SPH_C32(0x713c0000), SPH_C32(0xb1e60000),
	  SPH_C32(0xc8d60009), SPH_C32(0xf0b618db), SPH_C32(0xa33bc4a4),
	  SPH_C32(0xbd50e9e9), SPH_C32(0x89e81e7b), SPH_C32(0x39cf0000),
	  SPH_C32(0x42880000), SPH_C32(0xf8dd0000), SPH_C32(0x6bc40006),
	  SPH_C32(0x96a63cc5), SPH_C32(0x2e19b599), SPH_C32(0x250db0b9),
	  SPH_C32(0x474b8a29) },
	{ SPH_C32(0xaa740000), SPH_C32(0x798f0000), SPH_C32(0x42dc0000),
	  SPH_C32(0xf214000e), SPH_C32(0xa19f928b), SPH_C32(0xc855a2bb),
	  SPH_C32(0xb3f5260a), SPH_C32(0x6f326185), SPH_C32(0x91150000),
	  SPH_C32(0xd4360000), SPH_C32(0xa4c00000), SPH_C32(0x6c1e0004),
	  SPH_C32(0xebc0a946), SPH_C32(0x3181c513), SPH_C32(0x9e6b38b1),
	  SPH_C32(0x9dcc0a29) },
	{ SPH_C32(0x92280000), SPH_C32(0xdc850000), SPH_C32(0x57fa0000),
	  SPH_C32(0x56dc0003), SPH_C32(0xbae92316), SPH_C32(0x5aefa30c),
	  SPH_C32(0x90cef752), SPH_C32(0x7b1675d7), SPH_C32(0x93bb0000),
	  SPH_C32(0x3b070000), SPH_C32(0xba010000), SPH_C32(0x99d00008),
	  SPH_C32(0x3739ae4e), SPH_C32(0xe64c1722), SPH_C32(0x96f896b3),
	  SPH_C32(0x2879ebac) },
	{ SPH_C32(0x91150000), SPH_C32(0xd4360000), SPH_C32(0xa4c00000),
	  SPH_C32(0x6c1e0004), SPH_C32(0xebc0a946), SPH_C32(0x3181c513),
	  SPH_C32(0x9e6b38b1), SPH_C32(0x9dcc0a29), SPH_C32(0x3b610000),
	  SPH_C32(0xadb90000), SPH_C32(0xe61c0000), SPH_C32(0x9e0a000a),
	  SPH_C32(0x4a5f3bcd), SPH_C32(0xf9d467a8), SPH_C32(0x2d9e1ebb),
	  SPH_C32(0xf2fe6bac) },
	{ SPH_C32(0x3af20000), SPH_C32(0x4a3b0000), SPH_C32(0x0be70000),
	  SPH_C32(0x51060001), SPH_C32(0xc78fb695), SPH_C32(0x4577d386),
	  SPH_C32(0x2ba87f5a), SPH_C32(0xa191f5d7), SPH_C32(0x385c0000),
	  SPH_C32(0xa50a0000), SPH_C32(0x15260000), SPH_C32(0xa4c8000d),
	  SPH_C32(0x1b76b19d), SPH_C32(0x92ba01b7), SPH_C32(0x233bd158),
	  SPH_C32(0x14241452) },
	{ SPH_C32(0x39cf0000), SPH_C32(0x42880000), SPH_C32(0xf8dd0000),
	  SPH_C32(0x6bc40006), SPH_C32(0x96a63cc5), SPH_C32(0x2e19b599),
	  SPH_C32(0x250db0b9), SPH_C32(0x474b8a29), SPH_C32(0x90860000),
	  SPH_C32(0x33b40000), SPH_C32(0x493b0000), SPH_C32(0xa312000f),
	  SPH_C32(0x6610241e), SPH_C32(0x8d22713d), SPH_C32(0x985d5950),
	  SPH_C32(0xcea39452) },
	{ SPH_C32(0x93bb0000), SPH_C32(0x3b070000), SPH_C32(0xba010000),
	  SPH_C32(0x99d00008), SPH_C32(0x3739ae4e), SPH_C32(0xe64c1722),
	  SPH_C32(0x96f896b3), SPH_C32(0x2879ebac), SPH_C32(0x01930000),
	  SPH_C32(0xe7820000), SPH_C32(0xedfb0000), SPH_C32(0xcf0c000b),
	  SPH_C32(0x8dd08d58), SPH_C32(0xbca3b42e), SPH_C32(0x063661e1),
	  SPH_C32(0x536f9e7b) },
	{ SPH_C32(0x90860000), SPH_C32(0x33b40000), SPH_C32(0x493b0000),
	  SPH_C32(0xa312000f), SPH_C32(0x6610241e), SPH_C32(0x8d22713d),
	  SPH_C32(0x985d5950), SPH_C32(0xcea39452), SPH_C32(0xa9490000),
	  SPH_C32(0x713c0000), SPH_C32(0xb1e60000), SPH_C32(0xc8d60009),
	  SPH_C32(0xf0b618db), SPH_C32(0xa33bc4a4), SPH_C32(0xbd50e9e9),
	  SPH_C32(0x89e81e7b) },
	{ SPH_C32(0x3b610000), SPH_C32(0xadb90000), SPH_C32(0xe61c0000),
	  SPH_C32(0x9e0a000a), SPH_C32(0x4a5f3bcd), SPH_C32(0xf9d467a8),
	  SPH_C32(0x2d9e1ebb), SPH_C32(0xf2fe6bac), SPH_C32(0xaa740000),
	  SPH_C32(0x798f0000), SPH_C32(0x42dc0000), SPH_C32(0xf214000e),
	  SPH_C32(0xa19f928b), SPH_C32(0xc855a2bb), SPH_C32(0xb3f5260a),
	  SPH_C32(0x6f326185) },
	{ SPH_C32(0x385c0000), SPH_C32(0xa50a0000), SPH_C32(0x15260000),
	  SPH_C32(0xa4c8000d), SPH_C32(0x1b76b19d), SPH_C32(0x92ba01b7),
	  SPH_C32(0x233bd158), SPH_C32(0x14241452), SPH_C32(0x02ae0000),
	  SPH_C32(0xef310000), SPH_C32(0x1ec10000), SPH_C32(0xf5ce000c),
	  SPH_C32(0xdcf90708), SPH_C32(0xd7cdd231), SPH_C32(0x0893ae02),
	  SPH_C32(0xb5b5e185) }
};

#define INPUT_BIG   do { \
		unsigned acc = buf[0]; \
		const sph_u32 *rp; \
		rp = &T512_0[acc >> 4][0]; \
		m0 = rp[0]; \
		m1 = rp[1]; \
		m2 = rp[2]; \
		m3 = rp[3]; \
		m4 = rp[4]; \
		m5 = rp[5]; \
		m6 = rp[6]; \
		m7 = rp[7]; \
		m8 = rp[8]; \
		m9 = rp[9]; \
		mA = rp[10]; \
		mB = rp[11]; \
		mC = rp[12]; \
		mD = rp[13]; \
		mE = rp[14]; \
		mF = rp[15]; \
		rp = &T512_4[acc & 0x0f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = buf[1]; \
		rp = &T512_8[acc >> 4][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_12[acc & 0x0f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = buf[2]; \
		rp = &T512_16[acc >> 4][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_20[acc & 0x0f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = buf[3]; \
		rp = &T512_24[acc >> 4][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_28[acc & 0x0f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = buf[4]; \
		rp = &T512_32[acc >> 4][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_36[acc & 0x0f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = buf[5]; \
		rp = &T512_40[acc >> 4][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_44[acc & 0x0f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = buf[6]; \
		rp = &T512_48[acc >> 4][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_52[acc & 0x0f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = buf[7]; \
		rp = &T512_56[acc >> 4][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_60[acc & 0x0f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
	} while (0)

#endif

#if SPH_HAMSI_EXPAND_BIG == 5

static const sph_u32 T512_0[32][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x23671400), SPH_C32(0xc8b90000), SPH_C32(0xf4c70000),
	  SPH_C32(0xfb750000), SPH_C32(0x73cd2465), SPH_C32(0xf8a6a549),
	  SPH_C32(0x02c40a3f), SPH_C32(0xdc24e61f), SPH_C32(0x373d2800),
	  SPH_C32(0x71500000), SPH_C32(0x95e00000), SPH_C32(0x0a140000),
	  SPH_C32(0xbdac1909), SPH_C32(0x48ef9831), SPH_C32(0x456d6d1f),
	  SPH_C32(0x3daac2da) },
	{ SPH_C32(0x54285c00), SPH_C32(0xeaed0000), SPH_C32(0xc5d60000),
	  SPH_C32(0xa1c50000), SPH_C32(0xb3a26770), SPH_C32(0x94a5c4e1),
	  SPH_C32(0x6bb0419d), SPH_C32(0x551b3782), SPH_C32(0x9cbb1800),
	  SPH_C32(0xb0d30000), SPH_C32(0x92510000), SPH_C32(0xed930000),
	  SPH_C32(0x593a4345), SPH_C32(0xe114d5f4), SPH_C32(0x430633da),
	  SPH_C32(0x78cace29) },
	{ SPH_C32(0x774f4800), SPH_C32(0x22540000), SPH_C32(0x31110000),
	  SPH_C32(0x5ab00000), SPH_C32(0xc06f4315), SPH_C32(0x6c0361a8),
	  SPH_C32(0x69744ba2), SPH_C32(0x893fd19d), SPH_C32(0xab863000),
	  SPH_C32(0xc1830000), SPH_C32(0x07b10000), SPH_C32(0xe7870000),
	  SPH_C32(0xe4965a4c), SPH_C32(0xa9fb4dc5), SPH_C32(0x066b5ec5),
	  SPH_C32(0x45600cf3) },
	{ SPH_C32(0x9cbb1800), SPH_C32(0xb0d30000), SPH_C32(0x92510000),
	  SPH_C32(0xed930000), SPH_C32(0x593a4345), SPH_C32(0xe114d5f4),
	  SPH_C32(0x430633da), SPH_C32(0x78cace29), SPH_C32(0xc8934400),
	  SPH_C32(0x5a3e0000), SPH_C32(0x57870000), SPH_C32(0x4c560000),
	  SPH_C32(0xea982435), SPH_C32(0x75b11115), SPH_C32(0x28b67247),
	  SPH_C32(0x2dd1f9ab) },
	{ SPH_C32(0xbfdc0c00), SPH_C32(0x786a0000), SPH_C32(0x66960000),
	  SPH_C32(0x16e60000), SPH_C32(0x2af76720), SPH_C32(0x19b270bd),
	  SPH_C32(0x41c239e5), SPH_C32(0xa4ee2836), SPH_C32(0xffae6c00),
	  SPH_C32(0x2b6e0000), SPH_C32(0xc2670000), SPH_C32(0x46420000),
	  SPH_C32(0x57343d3c), SPH_C32(0x3d5e8924), SPH_C32(0x6ddb1f58),
	  SPH_C32(0x107b3b71) },
	{ SPH_C32(0xc8934400), SPH_C32(0x5a3e0000), SPH_C32(0x57870000),
	  SPH_C32(0x4c560000), SPH_C32(0xea982435), SPH_C32(0x75b11115),
	  SPH_C32(0x28b67247), SPH_C32(0x2dd1f9ab), SPH_C32(0x54285c00),
	  SPH_C32(0xeaed0000), SPH_C32(0xc5d60000), SPH_C32(0xa1c50000),
	  SPH_C32(0xb3a26770), SPH_C32(0x94a5c4e1), SPH_C32(0x6bb0419d),
	  SPH_C32(0x551b3782) },
	{ SPH_C32(0xebf45000), SPH_C32(0x92870000), SPH_C32(0xa3400000),
	  SPH_C32(0xb7230000), SPH_C32(0x99550050), SPH_C32(0x8d17b45c),
	  SPH_C32(0x2a727878), SPH_C32(0xf1f51fb4), SPH_C32(0x63157400),
	  SPH_C32(0x9bbd0000), SPH_C32(0x50360000), SPH_C32(0xabd10000),
	  SPH_C32(0x0e0e7e79), SPH_C32(0xdc4a5cd0), SPH_C32(0x2edd2c82),
	  SPH_C32(0x68b1f558) },
	{ SPH_C32(0x29449c00), SPH_C32(0x64e70000), SPH_C32(0xf24b0000),
	  SPH_C32(0xc2f30000), SPH_C32(0x0ede4e8f), SPH_C32(0x56c23745),
	  SPH_C32(0xf3e04259), SPH_C32(0x8d0d9ec4), SPH_C32(0x466d0c00),
	  SPH_C32(0x08620000), SPH_C32(0xdd5d0000), SPH_C32(0xbadd0000),
	  SPH_C32(0x6a927942), SPH_C32(0x441f2b93), SPH_C32(0x218ace6f),
	  SPH_C32(0xbf2c0be2) },
	{ SPH_C32(0x0a238800), SPH_C32(0xac5e0000), SPH_C32(0x068c0000),
	  SPH_C32(0x39860000), SPH_C32(0x7d136aea), SPH_C32(0xae64920c),
	  SPH_C32(0xf1244866), SPH_C32(0x512978db), SPH_C32(0x71502400),
	  SPH_C32(0x79320000), SPH_C32(0x48bd0000), SPH_C32(0xb0c90000),
	  SPH_C32(0xd73e604b), SPH_C32(0x0cf0b3a2), SPH_C32(0x64e7a370),
	  SPH_C32(0x8286c938) },
	{ SPH_C32(0x7d6cc000), SPH_C32(0x8e0a0000), SPH_C32(0x379d0000),
	  SPH_C32(0x63360000), SPH_C32(0xbd7c29ff), SPH_C32(0xc267f3a4),
	  SPH_C32(0x985003c4), SPH_C32(0xd816a946), SPH_C32(0xdad61400),
	  SPH_C32(0xb8b10000), SPH_C32(0x4f0c0000), SPH_C32(0x574e0000),
	  SPH_C32(0x33a83a07), SPH_C32(0xa50bfe67), SPH_C32(0x628cfdb5),
	  SPH_C32(0xc7e6c5cb) },
	{ SPH_C32(0x5e0bd400), SPH_C32(0x46b30000), SPH_C32(0xc35a0000),
	  SPH_C32(0x98430000), SPH_C32(0xceb10d9a), SPH_C32(0x3ac156ed),
	  SPH_C32(0x9a9409fb), SPH_C32(0x04324f59), SPH_C32(0xedeb3c00),
	  SPH_C32(0xc9e10000), SPH_C32(0xdaec0000), SPH_C32(0x5d5a0000),
	  SPH_C32(0x8e04230e), SPH_C32(0xede46656), SPH_C32(0x27e190aa),
	  SPH_C32(0xfa4c0711) },
	{ SPH_C32(0xb5ff8400), SPH_C32(0xd4340000), SPH_C32(0x601a0000),
	  SPH_C32(0x2f600000), SPH_C32(0x57e40dca), SPH_C32(0xb7d6e2b1),
	  SPH_C32(0xb0e67183), SPH_C32(0xf5c750ed), SPH_C32(0x8efe4800),
	  SPH_C32(0x525c0000), SPH_C32(0x8ada0000), SPH_C32(0xf68b0000),
	  SPH_C32(0x800a5d77), SPH_C32(0x31ae3a86), SPH_C32(0x093cbc28),
	  SPH_C32(0x92fdf249) },
	{ SPH_C32(0x96989000), SPH_C32(0x1c8d0000), SPH_C32(0x94dd0000),
	  SPH_C32(0xd4150000), SPH_C32(0x242929af), SPH_C32(0x4f7047f8),
	  SPH_C32(0xb2227bbc), SPH_C32(0x29e3b6f2), SPH_C32(0xb9c36000),
	  SPH_C32(0x230c0000), SPH_C32(0x1f3a0000), SPH_C32(0xfc9f0000),
	  SPH_C32(0x3da6447e), SPH_C32(0x7941a2b7), SPH_C32(0x4c51d137),
	  SPH_C32(0xaf573093) },
	{ SPH_C32(0xe1d7d800), SPH_C32(0x3ed90000), SPH_C32(0xa5cc0000),
	  SPH_C32(0x8ea50000), SPH_C32(0xe4466aba), SPH_C32(0x23732650),
	  SPH_C32(0xdb56301e), SPH_C32(0xa0dc676f), SPH_C32(0x12455000),
	  SPH_C32(0xe28f0000), SPH_C32(0x188b0000), SPH_C32(0x1b180000),
	  SPH_C32(0xd9301e32), SPH_C32(0xd0baef72), SPH_C32(0x4a3a8ff2),
	  SPH_C32(0xea373c60) },
	{ SPH_C32(0xc2b0cc00), SPH_C32(0xf6600000), SPH_C32(0x510b0000),
	  SPH_C32(0x75d00000), SPH_C32(0x978b4edf), SPH_C32(0xdbd58319),
	  SPH_C32(0xd9923a21), SPH_C32(0x7cf88170), SPH_C32(0x25787800),
	  SPH_C32(0x93df0000), SPH_C32(0x8d6b0000), SPH_C32(0x110c0000),
	  SPH_C32(0x649c073b), SPH_C32(0x98557743), SPH_C32(0x0f57e2ed),
	  SPH_C32(0xd79dfeba) },
	{ SPH_C32(0x466d0c00), SPH_C32(0x08620000), SPH_C32(0xdd5d0000),
	  SPH_C32(0xbadd0000), SPH_C32(0x6a927942), SPH_C32(0x441f2b93),
	  SPH_C32(0x218ace6f), SPH_C32(0xbf2c0be2), SPH_C32(0x6f299000),
	  SPH_C32(0x6c850000), SPH_C32(0x2f160000), SPH_C32(0x782e0000),
	  SPH_C32(0x644c37cd), SPH_C32(0x12dd1cd6), SPH_C32(0xd26a8c36),
	  SPH_C32(0x32219526) },
	{ SPH_C32(0x650a1800), SPH_C32(0xc0db0000), SPH_C32(0x299a0000),
	  SPH_C32(0x41a80000), SPH_C32(0x195f5d27), SPH_C32(0xbcb98eda),
	  SPH_C32(0x234ec450), SPH_C32(0x6308edfd), SPH_C32(0x5814b800),
	  SPH_C32(0x1dd50000), SPH_C32(0xbaf60000), SPH_C32(0x723a0000),
	  SPH_C32(0xd9e02ec4), SPH_C32(0x5a3284e7), SPH_C32(0x9707e129),
	  SPH_C32(0x0f8b57fc) },
	{ SPH_C32(0x12455000), SPH_C32(0xe28f0000), SPH_C32(0x188b0000),
	  SPH_C32(0x1b180000), SPH_C32(0xd9301e32), SPH_C32(0xd0baef72),
	  SPH_C32(0x4a3a8ff2), SPH_C32(0xea373c60), SPH_C32(0xf3928800),
	  SPH_C32(0xdc560000), SPH_C32(0xbd470000), SPH_C32(0x95bd0000),
	  SPH_C32(0x3d767488), SPH_C32(0xf3c9c922), SPH_C32(0x916cbfec),
	  SPH_C32(0x4aeb5b0f) },
	{ SPH_C32(0x31224400), SPH_C32(0x2a360000), SPH_C32(0xec4c0000),
	  SPH_C32(0xe06d0000), SPH_C32(0xaafd3a57), SPH_C32(0x281c4a3b),
	  SPH_C32(0x48fe85cd), SPH_C32(0x3613da7f), SPH_C32(0xc4afa000),
	  SPH_C32(0xad060000), SPH_C32(0x28a70000), SPH_C32(0x9fa90000),
	  SPH_C32(0x80da6d81), SPH_C32(0xbb265113), SPH_C32(0xd401d2f3),
	  SPH_C32(0x774199d5) },
	{ SPH_C32(0xdad61400), SPH_C32(0xb8b10000), SPH_C32(0x4f0c0000),
	  SPH_C32(0x574e0000), SPH_C32(0x33a83a07), SPH_C32(0xa50bfe67),
	  SPH_C32(0x628cfdb5), SPH_C32(0xc7e6c5cb), SPH_C32(0xa7bad400),
	  SPH_C32(0x36bb0000), SPH_C32(0x78910000), SPH_C32(0x34780000),
	  SPH_C32(0x8ed413f8), SPH_C32(0x676c0dc3), SPH_C32(0xfadcfe71),
	  SPH_C32(0x1ff06c8d) },
	{ SPH_C32(0xf9b10000), SPH_C32(0x70080000), SPH_C32(0xbbcb0000),
	  SPH_C32(0xac3b0000), SPH_C32(0x40651e62), SPH_C32(0x5dad5b2e),
	  SPH_C32(0x6048f78a), SPH_C32(0x1bc223d4), SPH_C32(0x9087fc00),
	  SPH_C32(0x47eb0000), SPH_C32(0xed710000), SPH_C32(0x3e6c0000),
	  SPH_C32(0x33780af1), SPH_C32(0x2f8395f2), SPH_C32(0xbfb1936e),
	  SPH_C32(0x225aae57) },
	{ SPH_C32(0x8efe4800), SPH_C32(0x525c0000), SPH_C32(0x8ada0000),
	  SPH_C32(0xf68b0000), SPH_C32(0x800a5d77), SPH_C32(0x31ae3a86),
	  SPH_C32(0x093cbc28), SPH_C32(0x92fdf249), SPH_C32(0x3b01cc00),
	  SPH_C32(0x86680000), SPH_C32(0xeac00000), SPH_C32(0xd9eb0000),
	  SPH_C32(0xd7ee50bd), SPH_C32(0x8678d837), SPH_C32(0xb9dacdab),
	  SPH_C32(0x673aa2a4) },
	{ SPH_C32(0xad995c00), SPH_C32(0x9ae50000), SPH_C32(0x7e1d0000),
	  SPH_C32(0x0dfe0000), SPH_C32(0xf3c77912), SPH_C32(0xc9089fcf),
	  SPH_C32(0x0bf8b617), SPH_C32(0x4ed91456), SPH_C32(0x0c3ce400),
	  SPH_C32(0xf7380000), SPH_C32(0x7f200000), SPH_C32(0xd3ff0000),
	  SPH_C32(0x6a4249b4), SPH_C32(0xce974006), SPH_C32(0xfcb7a0b4),
	  SPH_C32(0x5a90607e) },
	{ SPH_C32(0x6f299000), SPH_C32(0x6c850000), SPH_C32(0x2f160000),
	  SPH_C32(0x782e0000), SPH_C32(0x644c37cd), SPH_C32(0x12dd1cd6),
	  SPH_C32(0xd26a8c36), SPH_C32(0x32219526), SPH_C32(0x29449c00),
	  SPH_C32(0x64e70000), SPH_C32(0xf24b0000), SPH_C32(0xc2f30000),
	  SPH_C32(0x0ede4e8f), SPH_C32(0x56c23745), SPH_C32(0xf3e04259),
	  SPH_C32(0x8d0d9ec4) },
	{ SPH_C32(0x4c4e8400), SPH_C32(0xa43c0000), SPH_C32(0xdbd10000),
	  SPH_C32(0x835b0000), SPH_C32(0x178113a8), SPH_C32(0xea7bb99f),
	  SPH_C32(0xd0ae8609), SPH_C32(0xee057339), SPH_C32(0x1e79b400),
	  SPH_C32(0x15b70000), SPH_C32(0x67ab0000), SPH_C32(0xc8e70000),
	  SPH_C32(0xb3725786), SPH_C32(0x1e2daf74), SPH_C32(0xb68d2f46),
	  SPH_C32(0xb0a75c1e) },
	{ SPH_C32(0x3b01cc00), SPH_C32(0x86680000), SPH_C32(0xeac00000),
	  SPH_C32(0xd9eb0000), SPH_C32(0xd7ee50bd), SPH_C32(0x8678d837),
	  SPH_C32(0xb9dacdab), SPH_C32(0x673aa2a4), SPH_C32(0xb5ff8400),
	  SPH_C32(0xd4340000), SPH_C32(0x601a0000), SPH_C32(0x2f600000),
	  SPH_C32(0x57e40dca), SPH_C32(0xb7d6e2b1), SPH_C32(0xb0e67183),
	  SPH_C32(0xf5c750ed) },
	{ SPH_C32(0x1866d800), SPH_C32(0x4ed10000), SPH_C32(0x1e070000),
	  SPH_C32(0x229e0000), SPH_C32(0xa42374d8), SPH_C32(0x7ede7d7e),
	  SPH_C32(0xbb1ec794), SPH_C32(0xbb1e44bb), SPH_C32(0x82c2ac00),
	  SPH_C32(0xa5640000), SPH_C32(0xf5fa0000), SPH_C32(0x25740000),
	  SPH_C32(0xea4814c3), SPH_C32(0xff397a80), SPH_C32(0xf58b1c9c),
	  SPH_C32(0xc86d9237) },
	{ SPH_C32(0xf3928800), SPH_C32(0xdc560000), SPH_C32(0xbd470000),
	  SPH_C32(0x95bd0000), SPH_C32(0x3d767488), SPH_C32(0xf3c9c922),
	  SPH_C32(0x916cbfec), SPH_C32(0x4aeb5b0f), SPH_C32(0xe1d7d800),
	  SPH_C32(0x3ed90000), SPH_C32(0xa5cc0000), SPH_C32(0x8ea50000),
	  SPH_C32(0xe4466aba), SPH_C32(0x23732650), SPH_C32(0xdb56301e),
	  SPH_C32(0xa0dc676f) },
	{ SPH_C32(0xd0f59c00), SPH_C32(0x14ef0000), SPH_C32(0x49800000),
	  SPH_C32(0x6ec80000), SPH_C32(0x4ebb50ed), SPH_C32(0x0b6f6c6b),
	  SPH_C32(0x93a8b5d3), SPH_C32(0x96cfbd10), SPH_C32(0xd6eaf000),
	  SPH_C32(0x4f890000), SPH_C32(0x302c0000), SPH_C32(0x84b10000),
	  SPH_C32(0x59ea73b3), SPH_C32(0x6b9cbe61), SPH_C32(0x9e3b5d01),
	  SPH_C32(0x9d76a5b5) },
	{ SPH_C32(0xa7bad400), SPH_C32(0x36bb0000), SPH_C32(0x78910000),
	  SPH_C32(0x34780000), SPH_C32(0x8ed413f8), SPH_C32(0x676c0dc3),
	  SPH_C32(0xfadcfe71), SPH_C32(0x1ff06c8d), SPH_C32(0x7d6cc000),
	  SPH_C32(0x8e0a0000), SPH_C32(0x379d0000), SPH_C32(0x63360000),
	  SPH_C32(0xbd7c29ff), SPH_C32(0xc267f3a4), SPH_C32(0x985003c4),
	  SPH_C32(0xd816a946) },
	{ SPH_C32(0x84ddc000), SPH_C32(0xfe020000), SPH_C32(0x8c560000),
	  SPH_C32(0xcf0d0000), SPH_C32(0xfd19379d), SPH_C32(0x9fcaa88a),
	  SPH_C32(0xf818f44e), SPH_C32(0xc3d48a92), SPH_C32(0x4a51e800),
	  SPH_C32(0xff5a0000), SPH_C32(0xa27d0000), SPH_C32(0x69220000),
	  SPH_C32(0x00d030f6), SPH_C32(0x8a886b95), SPH_C32(0xdd3d6edb),
	  SPH_C32(0xe5bc6b9c) }
};

static const sph_u32 T512_5[32][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xe8870170), SPH_C32(0x9d720000), SPH_C32(0x12db0000),
	  SPH_C32(0xd4220000), SPH_C32(0xf2886b27), SPH_C32(0xa921e543),
	  SPH_C32(0x4ef8b518), SPH_C32(0x618813b1), SPH_C32(0xb4370060),
	  SPH_C32(0x0c4c0000), SPH_C32(0x56c20000), SPH_C32(0x5cae0000),
	  SPH_C32(0x94541f3f), SPH_C32(0x3b3ef825), SPH_C32(0x1b365f3d),
	  SPH_C32(0xf3d45758) },
	{ SPH_C32(0xb4370060), SPH_C32(0x0c4c0000), SPH_C32(0x56c20000),
	  SPH_C32(0x5cae0000), SPH_C32(0x94541f3f), SPH_C32(0x3b3ef825),
	  SPH_C32(0x1b365f3d), SPH_C32(0xf3d45758), SPH_C32(0x5cb00110),
	  SPH_C32(0x913e0000), SPH_C32(0x44190000), SPH_C32(0x888c0000),
	  SPH_C32(0x66dc7418), SPH_C32(0x921f1d66), SPH_C32(0x55ceea25),
	  SPH_C32(0x925c44e9) },
	{ SPH_C32(0x5cb00110), SPH_C32(0x913e0000), SPH_C32(0x44190000),
	  SPH_C32(0x888c0000), SPH_C32(0x66dc7418), SPH_C32(0x921f1d66),
	  SPH_C32(0x55ceea25), SPH_C32(0x925c44e9), SPH_C32(0xe8870170),
	  SPH_C32(0x9d720000), SPH_C32(0x12db0000), SPH_C32(0xd4220000),
	  SPH_C32(0xf2886b27), SPH_C32(0xa921e543), SPH_C32(0x4ef8b518),
	  SPH_C32(0x618813b1) },
	{ SPH_C32(0xef0b0270), SPH_C32(0x3afd0000), SPH_C32(0x5dae0000),
	  SPH_C32(0x69490000), SPH_C32(0x9b0f3c06), SPH_C32(0x4405b5f9),
	  SPH_C32(0x66140a51), SPH_C32(0x924f5d0a), SPH_C32(0xc96b0030),
	  SPH_C32(0xe7250000), SPH_C32(0x2f840000), SPH_C32(0x264f0000),
	  SPH_C32(0x08695bf9), SPH_C32(0x6dfcf137), SPH_C32(0x509f6984),
	  SPH_C32(0x9e69af68) },
	{ SPH_C32(0x078c0300), SPH_C32(0xa78f0000), SPH_C32(0x4f750000),
	  SPH_C32(0xbd6b0000), SPH_C32(0x69875721), SPH_C32(0xed2450ba),
	  SPH_C32(0x28ecbf49), SPH_C32(0xf3c74ebb), SPH_C32(0x7d5c0050),
	  SPH_C32(0xeb690000), SPH_C32(0x79460000), SPH_C32(0x7ae10000),
	  SPH_C32(0x9c3d44c6), SPH_C32(0x56c20912), SPH_C32(0x4ba936b9),
	  SPH_C32(0x6dbdf830) },
	{ SPH_C32(0x5b3c0210), SPH_C32(0x36b10000), SPH_C32(0x0b6c0000),
	  SPH_C32(0x35e70000), SPH_C32(0x0f5b2339), SPH_C32(0x7f3b4ddc),
	  SPH_C32(0x7d22556c), SPH_C32(0x619b0a52), SPH_C32(0x95db0120),
	  SPH_C32(0x761b0000), SPH_C32(0x6b9d0000), SPH_C32(0xaec30000),
	  SPH_C32(0x6eb52fe1), SPH_C32(0xffe3ec51), SPH_C32(0x055183a1),
	  SPH_C32(0x0c35eb81) },
	{ SPH_C32(0xb3bb0360), SPH_C32(0xabc30000), SPH_C32(0x19b70000),
	  SPH_C32(0xe1c50000), SPH_C32(0xfdd3481e), SPH_C32(0xd61aa89f),
	  SPH_C32(0x33dae074), SPH_C32(0x001319e3), SPH_C32(0x21ec0140),
	  SPH_C32(0x7a570000), SPH_C32(0x3d5f0000), SPH_C32(0xf26d0000),
	  SPH_C32(0xfae130de), SPH_C32(0xc4dd1474), SPH_C32(0x1e67dc9c),
	  SPH_C32(0xffe1bcd9) },
	{ SPH_C32(0xc96b0030), SPH_C32(0xe7250000), SPH_C32(0x2f840000),
	  SPH_C32(0x264f0000), SPH_C32(0x08695bf9), SPH_C32(0x6dfcf137),
	  SPH_C32(0x509f6984), SPH_C32(0x9e69af68), SPH_C32(0x26600240),
	  SPH_C32(0xddd80000), SPH_C32(0x722a0000), SPH_C32(0x4f060000),
	  SPH_C32(0x936667ff), SPH_C32(0x29f944ce), SPH_C32(0x368b63d5),
	  SPH_C32(0x0c26f262) },
	{ SPH_C32(0x21ec0140), SPH_C32(0x7a570000), SPH_C32(0x3d5f0000),
	  SPH_C32(0xf26d0000), SPH_C32(0xfae130de), SPH_C32(0xc4dd1474),
	  SPH_C32(0x1e67dc9c), SPH_C32(0xffe1bcd9), SPH_C32(0x92570220),
	  SPH_C32(0xd1940000), SPH_C32(0x24e80000), SPH_C32(0x13a80000),
	  SPH_C32(0x073278c0), SPH_C32(0x12c7bceb), SPH_C32(0x2dbd3ce8),
	  SPH_C32(0xfff2a53a) },
	{ SPH_C32(0x7d5c0050), SPH_C32(0xeb690000), SPH_C32(0x79460000),
	  SPH_C32(0x7ae10000), SPH_C32(0x9c3d44c6), SPH_C32(0x56c20912),
	  SPH_C32(0x4ba936b9), SPH_C32(0x6dbdf830), SPH_C32(0x7ad00350),
	  SPH_C32(0x4ce60000), SPH_C32(0x36330000), SPH_C32(0xc78a0000),
	  SPH_C32(0xf5ba13e7), SPH_C32(0xbbe659a8), SPH_C32(0x634589f0),
	  SPH_C32(0x9e7ab68b) },
	{ SPH_C32(0x95db0120), SPH_C32(0x761b0000), SPH_C32(0x6b9d0000),
	  SPH_C32(0xaec30000), SPH_C32(0x6eb52fe1), SPH_C32(0xffe3ec51),
	  SPH_C32(0x055183a1), SPH_C32(0x0c35eb81), SPH_C32(0xcee70330),
	  SPH_C32(0x40aa0000), SPH_C32(0x60f10000), SPH_C32(0x9b240000),
	  SPH_C32(0x61ee0cd8), SPH_C32(0x80d8a18d), SPH_C32(0x7873d6cd),
	  SPH_C32(0x6daee1d3) },
	{ SPH_C32(0x26600240), SPH_C32(0xddd80000), SPH_C32(0x722a0000),
	  SPH_C32(0x4f060000), SPH_C32(0x936667ff), SPH_C32(0x29f944ce),
	  SPH_C32(0x368b63d5), SPH_C32(0x0c26f262), SPH_C32(0xef0b0270),
	  SPH_C32(0x3afd0000), SPH_C32(0x5dae0000), SPH_C32(0x69490000),
	  SPH_C32(0x9b0f3c06), SPH_C32(0x4405b5f9), SPH_C32(0x66140a51),
	  SPH_C32(0x924f5d0a) },
	{ SPH_C32(0xcee70330), SPH_C32(0x40aa0000), SPH_C32(0x60f10000),
	  SPH_C32(0x9b240000), SPH_C32(0x61ee0cd8), SPH_C32(0x80d8a18d),
	  SPH_C32(0x7873d6cd), SPH_C32(0x6daee1d3), SPH_C32(0x5b3c0210),
	  SPH_C32(0x36b10000), SPH_C32(0x0b6c0000), SPH_C32(0x35e70000),
	  SPH_C32(0x0f5b2339), SPH_C32(0x7f3b4ddc), SPH_C32(0x7d22556c),
	  SPH_C32(0x619b0a52) },
	{ SPH_C32(0x92570220), SPH_C32(0xd1940000), SPH_C32(0x24e80000),
	  SPH_C32(0x13a80000), SPH_C32(0x073278c0), SPH_C32(0x12c7bceb),
	  SPH_C32(0x2dbd3ce8), SPH_C32(0xfff2a53a), SPH_C32(0xb3bb0360),
	  SPH_C32(0xabc30000), SPH_C32(0x19b70000), SPH_C32(0xe1c50000),
	  SPH_C32(0xfdd3481e), SPH_C32(0xd61aa89f), SPH_C32(0x33dae074),
	  SPH_C32(0x001319e3) },
	{ SPH_C32(0x7ad00350), SPH_C32(0x4ce60000), SPH_C32(0x36330000),
	  SPH_C32(0xc78a0000), SPH_C32(0xf5ba13e7), SPH_C32(0xbbe659a8),
	  SPH_C32(0x634589f0), SPH_C32(0x9e7ab68b), SPH_C32(0x078c0300),
	  SPH_C32(0xa78f0000), SPH_C32(0x4f750000), SPH_C32(0xbd6b0000),
	  SPH_C32(0x69875721), SPH_C32(0xed2450ba), SPH_C32(0x28ecbf49),
	  SPH_C32(0xf3c74ebb) },
	{ SPH_C32(0x145a3c00), SPH_C32(0xb9e90000), SPH_C32(0x61270000),
	  SPH_C32(0xf1610000), SPH_C32(0xce613d6c), SPH_C32(0xb0493d78),
	  SPH_C32(0x47a96720), SPH_C32(0xe18e24c5), SPH_C32(0x23671400),
	  SPH_C32(0xc8b90000), SPH_C32(0xf4c70000), SPH_C32(0xfb750000),
	  SPH_C32(0x73cd2465), SPH_C32(0xf8a6a549), SPH_C32(0x02c40a3f),
	  SPH_C32(0xdc24e61f) },
	{ SPH_C32(0xfcdd3d70), SPH_C32(0x249b0000), SPH_C32(0x73fc0000),
	  SPH_C32(0x25430000), SPH_C32(0x3ce9564b), SPH_C32(0x1968d83b),
	  SPH_C32(0x0951d238), SPH_C32(0x80063774), SPH_C32(0x97501460),
	  SPH_C32(0xc4f50000), SPH_C32(0xa2050000), SPH_C32(0xa7db0000),
	  SPH_C32(0xe7993b5a), SPH_C32(0xc3985d6c), SPH_C32(0x19f25502),
	  SPH_C32(0x2ff0b147) },
	{ SPH_C32(0xa06d3c60), SPH_C32(0xb5a50000), SPH_C32(0x37e50000),
	  SPH_C32(0xadcf0000), SPH_C32(0x5a352253), SPH_C32(0x8b77c55d),
	  SPH_C32(0x5c9f381d), SPH_C32(0x125a739d), SPH_C32(0x7fd71510),
	  SPH_C32(0x59870000), SPH_C32(0xb0de0000), SPH_C32(0x73f90000),
	  SPH_C32(0x1511507d), SPH_C32(0x6ab9b82f), SPH_C32(0x570ae01a),
	  SPH_C32(0x4e78a2f6) },
	{ SPH_C32(0x48ea3d10), SPH_C32(0x28d70000), SPH_C32(0x253e0000),
	  SPH_C32(0x79ed0000), SPH_C32(0xa8bd4974), SPH_C32(0x2256201e),
	  SPH_C32(0x12678d05), SPH_C32(0x73d2602c), SPH_C32(0xcbe01570),
	  SPH_C32(0x55cb0000), SPH_C32(0xe61c0000), SPH_C32(0x2f570000),
	  SPH_C32(0x81454f42), SPH_C32(0x5187400a), SPH_C32(0x4c3cbf27),
	  SPH_C32(0xbdacf5ae) },
	{ SPH_C32(0xfb513e70), SPH_C32(0x83140000), SPH_C32(0x3c890000),
	  SPH_C32(0x98280000), SPH_C32(0x556e016a), SPH_C32(0xf44c8881),
	  SPH_C32(0x21bd6d71), SPH_C32(0x73c179cf), SPH_C32(0xea0c1430),
	  SPH_C32(0x2f9c0000), SPH_C32(0xdb430000), SPH_C32(0xdd3a0000),
	  SPH_C32(0x7ba47f9c), SPH_C32(0x955a547e), SPH_C32(0x525b63bb),
	  SPH_C32(0x424d4977) },
	{ SPH_C32(0x13d63f00), SPH_C32(0x1e660000), SPH_C32(0x2e520000),
	  SPH_C32(0x4c0a0000), SPH_C32(0xa7e66a4d), SPH_C32(0x5d6d6dc2),
	  SPH_C32(0x6f45d869), SPH_C32(0x12496a7e), SPH_C32(0x5e3b1450),
	  SPH_C32(0x23d00000), SPH_C32(0x8d810000), SPH_C32(0x81940000),
	  SPH_C32(0xeff060a3), SPH_C32(0xae64ac5b), SPH_C32(0x496d3c86),
	  SPH_C32(0xb1991e2f) },
	{ SPH_C32(0x4f663e10), SPH_C32(0x8f580000), SPH_C32(0x6a4b0000),
	  SPH_C32(0xc4860000), SPH_C32(0xc13a1e55), SPH_C32(0xcf7270a4),
	  SPH_C32(0x3a8b324c), SPH_C32(0x80152e97), SPH_C32(0xb6bc1520),
	  SPH_C32(0xbea20000), SPH_C32(0x9f5a0000), SPH_C32(0x55b60000),
	  SPH_C32(0x1d780b84), SPH_C32(0x07454918), SPH_C32(0x0795899e),
	  SPH_C32(0xd0110d9e) },
	{ SPH_C32(0xa7e13f60), SPH_C32(0x122a0000), SPH_C32(0x78900000),
	  SPH_C32(0x10a40000), SPH_C32(0x33b27572), SPH_C32(0x665395e7),
	  SPH_C32(0x74738754), SPH_C32(0xe19d3d26), SPH_C32(0x028b1540),
	  SPH_C32(0xb2ee0000), SPH_C32(0xc9980000), SPH_C32(0x09180000),
	  SPH_C32(0x892c14bb), SPH_C32(0x3c7bb13d), SPH_C32(0x1ca3d6a3),
	  SPH_C32(0x23c55ac6) },
	{ SPH_C32(0xdd313c30), SPH_C32(0x5ecc0000), SPH_C32(0x4ea30000),
	  SPH_C32(0xd72e0000), SPH_C32(0xc6086695), SPH_C32(0xddb5cc4f),
	  SPH_C32(0x17360ea4), SPH_C32(0x7fe78bad), SPH_C32(0x05071640),
	  SPH_C32(0x15610000), SPH_C32(0x86ed0000), SPH_C32(0xb4730000),
	  SPH_C32(0xe0ab439a), SPH_C32(0xd15fe187), SPH_C32(0x344f69ea),
	  SPH_C32(0xd002147d) },
	{ SPH_C32(0x35b63d40), SPH_C32(0xc3be0000), SPH_C32(0x5c780000),
	  SPH_C32(0x030c0000), SPH_C32(0x34800db2), SPH_C32(0x7494290c),
	  SPH_C32(0x59cebbbc), SPH_C32(0x1e6f981c), SPH_C32(0xb1301620),
	  SPH_C32(0x192d0000), SPH_C32(0xd02f0000), SPH_C32(0xe8dd0000),
	  SPH_C32(0x74ff5ca5), SPH_C32(0xea6119a2), SPH_C32(0x2f7936d7),
	  SPH_C32(0x23d64325) },
	{ SPH_C32(0x69063c50), SPH_C32(0x52800000), SPH_C32(0x18610000),
	  SPH_C32(0x8b800000), SPH_C32(0x525c79aa), SPH_C32(0xe68b346a),
	  SPH_C32(0x0c005199), SPH_C32(0x8c33dcf5), SPH_C32(0x59b71750),
	  SPH_C32(0x845f0000), SPH_C32(0xc2f40000), SPH_C32(0x3cff0000),
	  SPH_C32(0x86773782), SPH_C32(0x4340fce1), SPH_C32(0x618183cf),
	  SPH_C32(0x425e5094) },
	{ SPH_C32(0x81813d20), SPH_C32(0xcff20000), SPH_C32(0x0aba0000),
	  SPH_C32(0x5fa20000), SPH_C32(0xa0d4128d), SPH_C32(0x4faad129),
	  SPH_C32(0x42f8e481), SPH_C32(0xedbbcf44), SPH_C32(0xed801730),
	  SPH_C32(0x88130000), SPH_C32(0x94360000), SPH_C32(0x60510000),
	  SPH_C32(0x122328bd), SPH_C32(0x787e04c4), SPH_C32(0x7ab7dcf2),
	  SPH_C32(0xb18a07cc) },
	{ SPH_C32(0x323a3e40), SPH_C32(0x64310000), SPH_C32(0x130d0000),
	  SPH_C32(0xbe670000), SPH_C32(0x5d075a93), SPH_C32(0x99b079b6),
	  SPH_C32(0x712204f5), SPH_C32(0xeda8d6a7), SPH_C32(0xcc6c1670),
	  SPH_C32(0xf2440000), SPH_C32(0xa9690000), SPH_C32(0x923c0000),
	  SPH_C32(0xe8c21863), SPH_C32(0xbca310b0), SPH_C32(0x64d0006e),
	  SPH_C32(0x4e6bbb15) },
	{ SPH_C32(0xdabd3f30), SPH_C32(0xf9430000), SPH_C32(0x01d60000),
	  SPH_C32(0x6a450000), SPH_C32(0xaf8f31b4), SPH_C32(0x30919cf5),
	  SPH_C32(0x3fdab1ed), SPH_C32(0x8c20c516), SPH_C32(0x785b1610),
	  SPH_C32(0xfe080000), SPH_C32(0xffab0000), SPH_C32(0xce920000),
	  SPH_C32(0x7c96075c), SPH_C32(0x879de895), SPH_C32(0x7fe65f53),
	  SPH_C32(0xbdbfec4d) },
	{ SPH_C32(0x860d3e20), SPH_C32(0x687d0000), SPH_C32(0x45cf0000),
	  SPH_C32(0xe2c90000), SPH_C32(0xc95345ac), SPH_C32(0xa28e8193),
	  SPH_C32(0x6a145bc8), SPH_C32(0x1e7c81ff), SPH_C32(0x90dc1760),
	  SPH_C32(0x637a0000), SPH_C32(0xed700000), SPH_C32(0x1ab00000),
	  SPH_C32(0x8e1e6c7b), SPH_C32(0x2ebc0dd6), SPH_C32(0x311eea4b),
	  SPH_C32(0xdc37fffc) },
	{ SPH_C32(0x6e8a3f50), SPH_C32(0xf50f0000), SPH_C32(0x57140000),
	  SPH_C32(0x36eb0000), SPH_C32(0x3bdb2e8b), SPH_C32(0x0baf64d0),
	  SPH_C32(0x24eceed0), SPH_C32(0x7ff4924e), SPH_C32(0x24eb1700),
	  SPH_C32(0x6f360000), SPH_C32(0xbbb20000), SPH_C32(0x461e0000),
	  SPH_C32(0x1a4a7344), SPH_C32(0x1582f5f3), SPH_C32(0x2a28b576),
	  SPH_C32(0x2fe3a8a4) }
};

static const sph_u32 T512_10[32][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xa67f0001), SPH_C32(0x71378000), SPH_C32(0x19fc0000),
	  SPH_C32(0x96db0000), SPH_C32(0x3a8b6dfd), SPH_C32(0xebcaaef3),
	  SPH_C32(0x2c6d478f), SPH_C32(0xac8e6c88), SPH_C32(0x50ff0004),
	  SPH_C32(0x45744000), SPH_C32(0x3dfb0000), SPH_C32(0x19e60000),
	  SPH_C32(0x1bbc5606), SPH_C32(0xe1727b5d), SPH_C32(0xe1a8cc96),
	  SPH_C32(0x7b1bd6b9) },
	{ SPH_C32(0xf7750009), SPH_C32(0xcf3cc000), SPH_C32(0xc3d60000),
	  SPH_C32(0x04920000), SPH_C32(0x029519a9), SPH_C32(0xf8e836ba),
	  SPH_C32(0x7a87f14e), SPH_C32(0x9e16981a), SPH_C32(0xd46a0000),
	  SPH_C32(0x8dc8c000), SPH_C32(0xa5af0000), SPH_C32(0x4a290000),
	  SPH_C32(0xfc4e427a), SPH_C32(0xc9b4866c), SPH_C32(0x98369604),
	  SPH_C32(0xf746c320) },
	{ SPH_C32(0x510a0008), SPH_C32(0xbe0b4000), SPH_C32(0xda2a0000),
	  SPH_C32(0x92490000), SPH_C32(0x381e7454), SPH_C32(0x13229849),
	  SPH_C32(0x56eab6c1), SPH_C32(0x3298f492), SPH_C32(0x84950004),
	  SPH_C32(0xc8bc8000), SPH_C32(0x98540000), SPH_C32(0x53cf0000),
	  SPH_C32(0xe7f2147c), SPH_C32(0x28c6fd31), SPH_C32(0x799e5a92),
	  SPH_C32(0x8c5d1599) },
	{ SPH_C32(0xd46a0000), SPH_C32(0x8dc8c000), SPH_C32(0xa5af0000),
	  SPH_C32(0x4a290000), SPH_C32(0xfc4e427a), SPH_C32(0xc9b4866c),
	  SPH_C32(0x98369604), SPH_C32(0xf746c320), SPH_C32(0x231f0009),
	  SPH_C32(0x42f40000), SPH_C32(0x66790000), SPH_C32(0x4ebb0000),
	  SPH_C32(0xfedb5bd3), SPH_C32(0x315cb0d6), SPH_C32(0xe2b1674a),
	  SPH_C32(0x69505b3a) },
	{ SPH_C32(0x72150001), SPH_C32(0xfcff4000), SPH_C32(0xbc530000),
	  SPH_C32(0xdcf20000), SPH_C32(0xc6c52f87), SPH_C32(0x227e289f),
	  SPH_C32(0xb45bd18b), SPH_C32(0x5bc8afa8), SPH_C32(0x73e0000d),
	  SPH_C32(0x07804000), SPH_C32(0x5b820000), SPH_C32(0x575d0000),
	  SPH_C32(0xe5670dd5), SPH_C32(0xd02ecb8b), SPH_C32(0x0319abdc),
	  SPH_C32(0x124b8d83) },
	{ SPH_C32(0x231f0009), SPH_C32(0x42f40000), SPH_C32(0x66790000),
	  SPH_C32(0x4ebb0000), SPH_C32(0xfedb5bd3), SPH_C32(0x315cb0d6),
	  SPH_C32(0xe2b1674a), SPH_C32(0x69505b3a), SPH_C32(0xf7750009),
	  SPH_C32(0xcf3cc000), SPH_C32(0xc3d60000), SPH_C32(0x04920000),
	  SPH_C32(0x029519a9), SPH_C32(0xf8e836ba), SPH_C32(0x7a87f14e),
	  SPH_C32(0x9e16981a) },
	{ SPH_C32(0x85600008), SPH_C32(0x33c38000), SPH_C32(0x7f850000),
	  SPH_C32(0xd8600000), SPH_C32(0xc450362e), SPH_C32(0xda961e25),
	  SPH_C32(0xcedc20c5), SPH_C32(0xc5de37b2), SPH_C32(0xa78a000d),
	  SPH_C32(0x8a488000), SPH_C32(0xfe2d0000), SPH_C32(0x1d740000),
	  SPH_C32(0x19294faf), SPH_C32(0x199a4de7), SPH_C32(0x9b2f3dd8),
	  SPH_C32(0xe50d4ea3) },
	{ SPH_C32(0x774400f0), SPH_C32(0xf15a0000), SPH_C32(0xf5b20000),
	  SPH_C32(0x34140000), SPH_C32(0x89377e8c), SPH_C32(0x5a8bec25),
	  SPH_C32(0x0bc3cd1e), SPH_C32(0xcf3775cb), SPH_C32(0xf46c0050),
	  SPH_C32(0x96180000), SPH_C32(0x14a50000), SPH_C32(0x031f0000),
	  SPH_C32(0x42947eb8), SPH_C32(0x66bf7e19), SPH_C32(0x9ca470d2),
	  SPH_C32(0x8a341574) },
	{ SPH_C32(0xd13b00f1), SPH_C32(0x806d8000), SPH_C32(0xec4e0000),
	  SPH_C32(0xa2cf0000), SPH_C32(0xb3bc1371), SPH_C32(0xb14142d6),
	  SPH_C32(0x27ae8a91), SPH_C32(0x63b91943), SPH_C32(0xa4930054),
	  SPH_C32(0xd36c4000), SPH_C32(0x295e0000), SPH_C32(0x1af90000),
	  SPH_C32(0x592828be), SPH_C32(0x87cd0544), SPH_C32(0x7d0cbc44),
	  SPH_C32(0xf12fc3cd) },
	{ SPH_C32(0x803100f9), SPH_C32(0x3e66c000), SPH_C32(0x36640000),
	  SPH_C32(0x30860000), SPH_C32(0x8ba26725), SPH_C32(0xa263da9f),
	  SPH_C32(0x71443c50), SPH_C32(0x5121edd1), SPH_C32(0x20060050),
	  SPH_C32(0x1bd0c000), SPH_C32(0xb10a0000), SPH_C32(0x49360000),
	  SPH_C32(0xbeda3cc2), SPH_C32(0xaf0bf875), SPH_C32(0x0492e6d6),
	  SPH_C32(0x7d72d654) },
	{ SPH_C32(0x264e00f8), SPH_C32(0x4f514000), SPH_C32(0x2f980000),
	  SPH_C32(0xa65d0000), SPH_C32(0xb1290ad8), SPH_C32(0x49a9746c),
	  SPH_C32(0x5d297bdf), SPH_C32(0xfdaf8159), SPH_C32(0x70f90054),
	  SPH_C32(0x5ea48000), SPH_C32(0x8cf10000), SPH_C32(0x50d00000),
	  SPH_C32(0xa5666ac4), SPH_C32(0x4e798328), SPH_C32(0xe53a2a40),
	  SPH_C32(0x066900ed) },
	{ SPH_C32(0xa32e00f0), SPH_C32(0x7c92c000), SPH_C32(0x501d0000),
	  SPH_C32(0x7e3d0000), SPH_C32(0x75793cf6), SPH_C32(0x933f6a49),
	  SPH_C32(0x93f55b1a), SPH_C32(0x3871b6eb), SPH_C32(0xd7730059),
	  SPH_C32(0xd4ec0000), SPH_C32(0x72dc0000), SPH_C32(0x4da40000),
	  SPH_C32(0xbc4f256b), SPH_C32(0x57e3cecf), SPH_C32(0x7e151798),
	  SPH_C32(0xe3644e4e) },
	{ SPH_C32(0x055100f1), SPH_C32(0x0da54000), SPH_C32(0x49e10000),
	  SPH_C32(0xe8e60000), SPH_C32(0x4ff2510b), SPH_C32(0x78f5c4ba),
	  SPH_C32(0xbf981c95), SPH_C32(0x94ffda63), SPH_C32(0x878c005d),
	  SPH_C32(0x91984000), SPH_C32(0x4f270000), SPH_C32(0x54420000),
	  SPH_C32(0xa7f3736d), SPH_C32(0xb691b592), SPH_C32(0x9fbddb0e),
	  SPH_C32(0x987f98f7) },
	{ SPH_C32(0x545b00f9), SPH_C32(0xb3ae0000), SPH_C32(0x93cb0000),
	  SPH_C32(0x7aaf0000), SPH_C32(0x77ec255f), SPH_C32(0x6bd75cf3),
	  SPH_C32(0xe972aa54), SPH_C32(0xa6672ef1), SPH_C32(0x03190059),
	  SPH_C32(0x5924c000), SPH_C32(0xd7730000), SPH_C32(0x078d0000),
	  SPH_C32(0x40016711), SPH_C32(0x9e5748a3), SPH_C32(0xe623819c),
	  SPH_C32(0x14228d6e) },
	{ SPH_C32(0xf22400f8), SPH_C32(0xc2998000), SPH_C32(0x8a370000),
	  SPH_C32(0xec740000), SPH_C32(0x4d6748a2), SPH_C32(0x801df200),
	  SPH_C32(0xc51feddb), SPH_C32(0x0ae94279), SPH_C32(0x53e6005d),
	  SPH_C32(0x1c508000), SPH_C32(0xea880000), SPH_C32(0x1e6b0000),
	  SPH_C32(0x5bbd3117), SPH_C32(0x7f2533fe), SPH_C32(0x078b4d0a),
	  SPH_C32(0x6f395bd7) },
	{ SPH_C32(0xf46c0050), SPH_C32(0x96180000), SPH_C32(0x14a50000),
	  SPH_C32(0x031f0000), SPH_C32(0x42947eb8), SPH_C32(0x66bf7e19),
	  SPH_C32(0x9ca470d2), SPH_C32(0x8a341574), SPH_C32(0x832800a0),
	  SPH_C32(0x67420000), SPH_C32(0xe1170000), SPH_C32(0x370b0000),
	  SPH_C32(0xcba30034), SPH_C32(0x3c34923c), SPH_C32(0x9767bdcc),
	  SPH_C32(0x450360bf) },
	{ SPH_C32(0x52130051), SPH_C32(0xe72f8000), SPH_C32(0x0d590000),
	  SPH_C32(0x95c40000), SPH_C32(0x781f1345), SPH_C32(0x8d75d0ea),
	  SPH_C32(0xb0c9375d), SPH_C32(0x26ba79fc), SPH_C32(0xd3d700a4),
	  SPH_C32(0x22364000), SPH_C32(0xdcec0000), SPH_C32(0x2eed0000),
	  SPH_C32(0xd01f5632), SPH_C32(0xdd46e961), SPH_C32(0x76cf715a),
	  SPH_C32(0x3e18b606) },
	{ SPH_C32(0x03190059), SPH_C32(0x5924c000), SPH_C32(0xd7730000),
	  SPH_C32(0x078d0000), SPH_C32(0x40016711), SPH_C32(0x9e5748a3),
	  SPH_C32(0xe623819c), SPH_C32(0x14228d6e), SPH_C32(0x574200a0),
	  SPH_C32(0xea8ac000), SPH_C32(0x44b80000), SPH_C32(0x7d220000),
	  SPH_C32(0x37ed424e), SPH_C32(0xf5801450), SPH_C32(0x0f512bc8),
	  SPH_C32(0xb245a39f) },
	{ SPH_C32(0xa5660058), SPH_C32(0x28134000), SPH_C32(0xce8f0000),
	  SPH_C32(0x91560000), SPH_C32(0x7a8a0aec), SPH_C32(0x759de650),
	  SPH_C32(0xca4ec613), SPH_C32(0xb8ace1e6), SPH_C32(0x07bd00a4),
	  SPH_C32(0xaffe8000), SPH_C32(0x79430000), SPH_C32(0x64c40000),
	  SPH_C32(0x2c511448), SPH_C32(0x14f26f0d), SPH_C32(0xeef9e75e),
	  SPH_C32(0xc95e7526) },
	{ SPH_C32(0x20060050), SPH_C32(0x1bd0c000), SPH_C32(0xb10a0000),
	  SPH_C32(0x49360000), SPH_C32(0xbeda3cc2), SPH_C32(0xaf0bf875),
	  SPH_C32(0x0492e6d6), SPH_C32(0x7d72d654), SPH_C32(0xa03700a9),
	  SPH_C32(0x25b60000), SPH_C32(0x876e0000), SPH_C32(0x79b00000),
	  SPH_C32(0x35785be7), SPH_C32(0x0d6822ea), SPH_C32(0x75d6da86),
	  SPH_C32(0x2c533b85) },
	{ SPH_C32(0x86790051), SPH_C32(0x6ae74000), SPH_C32(0xa8f60000),
	  SPH_C32(0xdfed0000), SPH_C32(0x8451513f), SPH_C32(0x44c15686),
	  SPH_C32(0x28ffa159), SPH_C32(0xd1fcbadc), SPH_C32(0xf0c800ad),
	  SPH_C32(0x60c24000), SPH_C32(0xba950000), SPH_C32(0x60560000),
	  SPH_C32(0x2ec40de1), SPH_C32(0xec1a59b7), SPH_C32(0x947e1610),
	  SPH_C32(0x5748ed3c) },
	{ SPH_C32(0xd7730059), SPH_C32(0xd4ec0000), SPH_C32(0x72dc0000),
	  SPH_C32(0x4da40000), SPH_C32(0xbc4f256b), SPH_C32(0x57e3cecf),
	  SPH_C32(0x7e151798), SPH_C32(0xe3644e4e), SPH_C32(0x745d00a9),
	  SPH_C32(0xa87ec000), SPH_C32(0x22c10000), SPH_C32(0x33990000),
	  SPH_C32(0xc936199d), SPH_C32(0xc4dca486), SPH_C32(0xede04c82),
	  SPH_C32(0xdb15f8a5) },
	{ SPH_C32(0x710c0058), SPH_C32(0xa5db8000), SPH_C32(0x6b200000),
	  SPH_C32(0xdb7f0000), SPH_C32(0x86c44896), SPH_C32(0xbc29603c),
	  SPH_C32(0x52785017), SPH_C32(0x4fea22c6), SPH_C32(0x24a200ad),
	  SPH_C32(0xed0a8000), SPH_C32(0x1f3a0000), SPH_C32(0x2a7f0000),
	  SPH_C32(0xd28a4f9b), SPH_C32(0x25aedfdb), SPH_C32(0x0c488014),
	  SPH_C32(0xa00e2e1c) },
	{ SPH_C32(0x832800a0), SPH_C32(0x67420000), SPH_C32(0xe1170000),
	  SPH_C32(0x370b0000), SPH_C32(0xcba30034), SPH_C32(0x3c34923c),
	  SPH_C32(0x9767bdcc), SPH_C32(0x450360bf), SPH_C32(0x774400f0),
	  SPH_C32(0xf15a0000), SPH_C32(0xf5b20000), SPH_C32(0x34140000),
	  SPH_C32(0x89377e8c), SPH_C32(0x5a8bec25), SPH_C32(0x0bc3cd1e),
	  SPH_C32(0xcf3775cb) },
	{ SPH_C32(0x255700a1), SPH_C32(0x16758000), SPH_C32(0xf8eb0000),
	  SPH_C32(0xa1d00000), SPH_C32(0xf1286dc9), SPH_C32(0xd7fe3ccf),
	  SPH_C32(0xbb0afa43), SPH_C32(0xe98d0c37), SPH_C32(0x27bb00f4),
	  SPH_C32(0xb42e4000), SPH_C32(0xc8490000), SPH_C32(0x2df20000),
	  SPH_C32(0x928b288a), SPH_C32(0xbbf99778), SPH_C32(0xea6b0188),
	  SPH_C32(0xb42ca372) },
	{ SPH_C32(0x745d00a9), SPH_C32(0xa87ec000), SPH_C32(0x22c10000),
	  SPH_C32(0x33990000), SPH_C32(0xc936199d), SPH_C32(0xc4dca486),
	  SPH_C32(0xede04c82), SPH_C32(0xdb15f8a5), SPH_C32(0xa32e00f0),
	  SPH_C32(0x7c92c000), SPH_C32(0x501d0000), SPH_C32(0x7e3d0000),
	  SPH_C32(0x75793cf6), SPH_C32(0x933f6a49), SPH_C32(0x93f55b1a),
	  SPH_C32(0x3871b6eb) },
	{ SPH_C32(0xd22200a8), SPH_C32(0xd9494000), SPH_C32(0x3b3d0000),
	  SPH_C32(0xa5420000), SPH_C32(0xf3bd7460), SPH_C32(0x2f160a75),
	  SPH_C32(0xc18d0b0d), SPH_C32(0x779b942d), SPH_C32(0xf3d100f4),
	  SPH_C32(0x39e68000), SPH_C32(0x6de60000), SPH_C32(0x67db0000),
	  SPH_C32(0x6ec56af0), SPH_C32(0x724d1114), SPH_C32(0x725d978c),
	  SPH_C32(0x436a6052) },
	{ SPH_C32(0x574200a0), SPH_C32(0xea8ac000), SPH_C32(0x44b80000),
	  SPH_C32(0x7d220000), SPH_C32(0x37ed424e), SPH_C32(0xf5801450),
	  SPH_C32(0x0f512bc8), SPH_C32(0xb245a39f), SPH_C32(0x545b00f9),
	  SPH_C32(0xb3ae0000), SPH_C32(0x93cb0000), SPH_C32(0x7aaf0000),
	  SPH_C32(0x77ec255f), SPH_C32(0x6bd75cf3), SPH_C32(0xe972aa54),
	  SPH_C32(0xa6672ef1) },
	{ SPH_C32(0xf13d00a1), SPH_C32(0x9bbd4000), SPH_C32(0x5d440000),
	  SPH_C32(0xebf90000), SPH_C32(0x0d662fb3), SPH_C32(0x1e4abaa3),
	  SPH_C32(0x233c6c47), SPH_C32(0x1ecbcf17), SPH_C32(0x04a400fd),
	  SPH_C32(0xf6da4000), SPH_C32(0xae300000), SPH_C32(0x63490000),
	  SPH_C32(0x6c507359), SPH_C32(0x8aa527ae), SPH_C32(0x08da66c2),
	  SPH_C32(0xdd7cf848) },
	{ SPH_C32(0xa03700a9), SPH_C32(0x25b60000), SPH_C32(0x876e0000),
	  SPH_C32(0x79b00000), SPH_C32(0x35785be7), SPH_C32(0x0d6822ea),
	  SPH_C32(0x75d6da86), SPH_C32(0x2c533b85), SPH_C32(0x803100f9),
	  SPH_C32(0x3e66c000), SPH_C32(0x36640000), SPH_C32(0x30860000),
	  SPH_C32(0x8ba26725), SPH_C32(0xa263da9f), SPH_C32(0x71443c50),
	  SPH_C32(0x5121edd1) },
	{ SPH_C32(0x064800a8), SPH_C32(0x54818000), SPH_C32(0x9e920000),
	  SPH_C32(0xef6b0000), SPH_C32(0x0ff3361a), SPH_C32(0xe6a28c19),
	  SPH_C32(0x59bb9d09), SPH_C32(0x80dd570d), SPH_C32(0xd0ce00fd),
	  SPH_C32(0x7b128000), SPH_C32(0x0b9f0000), SPH_C32(0x29600000),
	  SPH_C32(0x901e3123), SPH_C32(0x4311a1c2), SPH_C32(0x90ecf0c6),
	  SPH_C32(0x2a3a3b68) }
};

static const sph_u32 T512_15[32][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x75a40000), SPH_C32(0xc28b2700), SPH_C32(0x94a40000),
	  SPH_C32(0x90f50000), SPH_C32(0xfb7857e0), SPH_C32(0x49ce0bae),
	  SPH_C32(0x1767c483), SPH_C32(0xaedf667e), SPH_C32(0xd1660000),
	  SPH_C32(0x1bbc0300), SPH_C32(0x9eec0000), SPH_C32(0xf6940000),
	  SPH_C32(0x03024527), SPH_C32(0xcf70fcf2), SPH_C32(0xb4431b17),
	  SPH_C32(0x857f3c2b) },
	{ SPH_C32(0xd1660000), SPH_C32(0x1bbc0300), SPH_C32(0x9eec0000),
	  SPH_C32(0xf6940000), SPH_C32(0x03024527), SPH_C32(0xcf70fcf2),
	  SPH_C32(0xb4431b17), SPH_C32(0x857f3c2b), SPH_C32(0xa4c20000),
	  SPH_C32(0xd9372400), SPH_C32(0x0a480000), SPH_C32(0x66610000),
	  SPH_C32(0xf87a12c7), SPH_C32(0x86bef75c), SPH_C32(0xa324df94),
	  SPH_C32(0x2ba05a55) },
	{ SPH_C32(0xa4c20000), SPH_C32(0xd9372400), SPH_C32(0x0a480000),
	  SPH_C32(0x66610000), SPH_C32(0xf87a12c7), SPH_C32(0x86bef75c),
	  SPH_C32(0xa324df94), SPH_C32(0x2ba05a55), SPH_C32(0x75a40000),
	  SPH_C32(0xc28b2700), SPH_C32(0x94a40000), SPH_C32(0x90f50000),
	  SPH_C32(0xfb7857e0), SPH_C32(0x49ce0bae), SPH_C32(0x1767c483),
	  SPH_C32(0xaedf667e) },
	{ SPH_C32(0x75c90003), SPH_C32(0x0e10c000), SPH_C32(0xd1200000),
	  SPH_C32(0xbaea0000), SPH_C32(0x8bc42f3e), SPH_C32(0x8758b757),
	  SPH_C32(0xbb28761d), SPH_C32(0x00b72e2b), SPH_C32(0xeecf0001),
	  SPH_C32(0x6f564000), SPH_C32(0xf33e0000), SPH_C32(0xa79e0000),
	  SPH_C32(0xbdb57219), SPH_C32(0xb711ebc5), SPH_C32(0x4a3b40ba),
	  SPH_C32(0xfeabf254) },
	{ SPH_C32(0x006d0003), SPH_C32(0xcc9be700), SPH_C32(0x45840000),
	  SPH_C32(0x2a1f0000), SPH_C32(0x70bc78de), SPH_C32(0xce96bcf9),
	  SPH_C32(0xac4fb29e), SPH_C32(0xae684855), SPH_C32(0x3fa90001),
	  SPH_C32(0x74ea4300), SPH_C32(0x6dd20000), SPH_C32(0x510a0000),
	  SPH_C32(0xbeb7373e), SPH_C32(0x78611737), SPH_C32(0xfe785bad),
	  SPH_C32(0x7bd4ce7f) },
	{ SPH_C32(0xa4af0003), SPH_C32(0x15acc300), SPH_C32(0x4fcc0000),
	  SPH_C32(0x4c7e0000), SPH_C32(0x88c66a19), SPH_C32(0x48284ba5),
	  SPH_C32(0x0f6b6d0a), SPH_C32(0x85c81200), SPH_C32(0x4a0d0001),
	  SPH_C32(0xb6616400), SPH_C32(0xf9760000), SPH_C32(0xc1ff0000),
	  SPH_C32(0x45cf60de), SPH_C32(0x31af1c99), SPH_C32(0xe91f9f2e),
	  SPH_C32(0xd50ba801) },
	{ SPH_C32(0xd10b0003), SPH_C32(0xd727e400), SPH_C32(0xdb680000),
	  SPH_C32(0xdc8b0000), SPH_C32(0x73be3df9), SPH_C32(0x01e6400b),
	  SPH_C32(0x180ca989), SPH_C32(0x2b17747e), SPH_C32(0x9b6b0001),
	  SPH_C32(0xaddd6700), SPH_C32(0x679a0000), SPH_C32(0x376b0000),
	  SPH_C32(0x46cd25f9), SPH_C32(0xfedfe06b), SPH_C32(0x5d5c8439),
	  SPH_C32(0x5074942a) },
	{ SPH_C32(0xeecf0001), SPH_C32(0x6f564000), SPH_C32(0xf33e0000),
	  SPH_C32(0xa79e0000), SPH_C32(0xbdb57219), SPH_C32(0xb711ebc5),
	  SPH_C32(0x4a3b40ba), SPH_C32(0xfeabf254), SPH_C32(0x9b060002),
	  SPH_C32(0x61468000), SPH_C32(0x221e0000), SPH_C32(0x1d740000),
	  SPH_C32(0x36715d27), SPH_C32(0x30495c92), SPH_C32(0xf11336a7),
	  SPH_C32(0xfe1cdc7f) },
	{ SPH_C32(0x9b6b0001), SPH_C32(0xaddd6700), SPH_C32(0x679a0000),
	  SPH_C32(0x376b0000), SPH_C32(0x46cd25f9), SPH_C32(0xfedfe06b),
	  SPH_C32(0x5d5c8439), SPH_C32(0x5074942a), SPH_C32(0x4a600002),
	  SPH_C32(0x7afa8300), SPH_C32(0xbcf20000), SPH_C32(0xebe00000),
	  SPH_C32(0x35731800), SPH_C32(0xff39a060), SPH_C32(0x45502db0),
	  SPH_C32(0x7b63e054) },
	{ SPH_C32(0x3fa90001), SPH_C32(0x74ea4300), SPH_C32(0x6dd20000),
	  SPH_C32(0x510a0000), SPH_C32(0xbeb7373e), SPH_C32(0x78611737),
	  SPH_C32(0xfe785bad), SPH_C32(0x7bd4ce7f), SPH_C32(0x3fc40002),
	  SPH_C32(0xb871a400), SPH_C32(0x28560000), SPH_C32(0x7b150000),
	  SPH_C32(0xce0b4fe0), SPH_C32(0xb6f7abce), SPH_C32(0x5237e933),
	  SPH_C32(0xd5bc862a) },
	{ SPH_C32(0x4a0d0001), SPH_C32(0xb6616400), SPH_C32(0xf9760000),
	  SPH_C32(0xc1ff0000), SPH_C32(0x45cf60de), SPH_C32(0x31af1c99),
	  SPH_C32(0xe91f9f2e), SPH_C32(0xd50ba801), SPH_C32(0xeea20002),
	  SPH_C32(0xa3cda700), SPH_C32(0xb6ba0000), SPH_C32(0x8d810000),
	  SPH_C32(0xcd090ac7), SPH_C32(0x7987573c), SPH_C32(0xe674f224),
	  SPH_C32(0x50c3ba01) },
	{ SPH_C32(0x9b060002), SPH_C32(0x61468000), SPH_C32(0x221e0000),
	  SPH_C32(0x1d740000), SPH_C32(0x36715d27), SPH_C32(0x30495c92),
	  SPH_C32(0xf11336a7), SPH_C32(0xfe1cdc7f), SPH_C32(0x75c90003),
	  SPH_C32(0x0e10c000), SPH_C32(0xd1200000), SPH_C32(0xbaea0000),
	  SPH_C32(0x8bc42f3e), SPH_C32(0x8758b757), SPH_C32(0xbb28761d),
	  SPH_C32(0x00b72e2b) },
	{ SPH_C32(0xeea20002), SPH_C32(0xa3cda700), SPH_C32(0xb6ba0000),
	  SPH_C32(0x8d810000), SPH_C32(0xcd090ac7), SPH_C32(0x7987573c),
	  SPH_C32(0xe674f224), SPH_C32(0x50c3ba01), SPH_C32(0xa4af0003),
	  SPH_C32(0x15acc300), SPH_C32(0x4fcc0000), SPH_C32(0x4c7e0000),
	  SPH_C32(0x88c66a19), SPH_C32(0x48284ba5), SPH_C32(0x0f6b6d0a),
	  SPH_C32(0x85c81200) },
	{ SPH_C32(0x4a600002), SPH_C32(0x7afa8300), SPH_C32(0xbcf20000),
	  SPH_C32(0xebe00000), SPH_C32(0x35731800), SPH_C32(0xff39a060),
	  SPH_C32(0x45502db0), SPH_C32(0x7b63e054), SPH_C32(0xd10b0003),
	  SPH_C32(0xd727e400), SPH_C32(0xdb680000), SPH_C32(0xdc8b0000),
	  SPH_C32(0x73be3df9), SPH_C32(0x01e6400b), SPH_C32(0x180ca989),
	  SPH_C32(0x2b17747e) },
	{ SPH_C32(0x3fc40002), SPH_C32(0xb871a400), SPH_C32(0x28560000),
	  SPH_C32(0x7b150000), SPH_C32(0xce0b4fe0), SPH_C32(0xb6f7abce),
	  SPH_C32(0x5237e933), SPH_C32(0xd5bc862a), SPH_C32(0x006d0003),
	  SPH_C32(0xcc9be700), SPH_C32(0x45840000), SPH_C32(0x2a1f0000),
	  SPH_C32(0x70bc78de), SPH_C32(0xce96bcf9), SPH_C32(0xac4fb29e),
	  SPH_C32(0xae684855) },
	{ SPH_C32(0xf6800005), SPH_C32(0x3443c000), SPH_C32(0x24070000),
	  SPH_C32(0x8f3d0000), SPH_C32(0x21373bfb), SPH_C32(0x0ab8d5ae),
	  SPH_C32(0xcdc58b19), SPH_C32(0xd795ba31), SPH_C32(0xa67f0001),
	  SPH_C32(0x71378000), SPH_C32(0x19fc0000), SPH_C32(0x96db0000),
	  SPH_C32(0x3a8b6dfd), SPH_C32(0xebcaaef3), SPH_C32(0x2c6d478f),
	  SPH_C32(0xac8e6c88) },
	{ SPH_C32(0x83240005), SPH_C32(0xf6c8e700), SPH_C32(0xb0a30000),
	  SPH_C32(0x1fc80000), SPH_C32(0xda4f6c1b), SPH_C32(0x4376de00),
	  SPH_C32(0xdaa24f9a), SPH_C32(0x794adc4f), SPH_C32(0x77190001),
	  SPH_C32(0x6a8b8300), SPH_C32(0x87100000), SPH_C32(0x604f0000),
	  SPH_C32(0x398928da), SPH_C32(0x24ba5201), SPH_C32(0x982e5c98),
	  SPH_C32(0x29f150a3) },
	{ SPH_C32(0x27e60005), SPH_C32(0x2fffc300), SPH_C32(0xbaeb0000),
	  SPH_C32(0x79a90000), SPH_C32(0x22357edc), SPH_C32(0xc5c8295c),
	  SPH_C32(0x7986900e), SPH_C32(0x52ea861a), SPH_C32(0x02bd0001),
	  SPH_C32(0xa800a400), SPH_C32(0x13b40000), SPH_C32(0xf0ba0000),
	  SPH_C32(0xc2f17f3a), SPH_C32(0x6d7459af), SPH_C32(0x8f49981b),
	  SPH_C32(0x872e36dd) },
	{ SPH_C32(0x52420005), SPH_C32(0xed74e400), SPH_C32(0x2e4f0000),
	  SPH_C32(0xe95c0000), SPH_C32(0xd94d293c), SPH_C32(0x8c0622f2),
	  SPH_C32(0x6ee1548d), SPH_C32(0xfc35e064), SPH_C32(0xd3db0001),
	  SPH_C32(0xb3bca700), SPH_C32(0x8d580000), SPH_C32(0x062e0000),
	  SPH_C32(0xc1f33a1d), SPH_C32(0xa204a55d), SPH_C32(0x3b0a830c),
	  SPH_C32(0x02510af6) },
	{ SPH_C32(0x83490006), SPH_C32(0x3a530000), SPH_C32(0xf5270000),
	  SPH_C32(0x35d70000), SPH_C32(0xaaf314c5), SPH_C32(0x8de062f9),
	  SPH_C32(0x76edfd04), SPH_C32(0xd722941a), SPH_C32(0x48b00000),
	  SPH_C32(0x1e61c000), SPH_C32(0xeac20000), SPH_C32(0x31450000),
	  SPH_C32(0x873e1fe4), SPH_C32(0x5cdb4536), SPH_C32(0x66560735),
	  SPH_C32(0x52259edc) },
	{ SPH_C32(0xf6ed0006), SPH_C32(0xf8d82700), SPH_C32(0x61830000),
	  SPH_C32(0xa5220000), SPH_C32(0x518b4325), SPH_C32(0xc42e6957),
	  SPH_C32(0x618a3987), SPH_C32(0x79fdf264), SPH_C32(0x99d60000),
	  SPH_C32(0x05ddc300), SPH_C32(0x742e0000), SPH_C32(0xc7d10000),
	  SPH_C32(0x843c5ac3), SPH_C32(0x93abb9c4), SPH_C32(0xd2151c22),
	  SPH_C32(0xd75aa2f7) },
	{ SPH_C32(0x522f0006), SPH_C32(0x21ef0300), SPH_C32(0x6bcb0000),
	  SPH_C32(0xc3430000), SPH_C32(0xa9f151e2), SPH_C32(0x42909e0b),
	  SPH_C32(0xc2aee613), SPH_C32(0x525da831), SPH_C32(0xec720000),
	  SPH_C32(0xc756e400), SPH_C32(0xe08a0000), SPH_C32(0x57240000),
	  SPH_C32(0x7f440d23), SPH_C32(0xda65b26a), SPH_C32(0xc572d8a1),
	  SPH_C32(0x7985c489) },
	{ SPH_C32(0x278b0006), SPH_C32(0xe3642400), SPH_C32(0xff6f0000),
	  SPH_C32(0x53b60000), SPH_C32(0x52890602), SPH_C32(0x0b5e95a5),
	  SPH_C32(0xd5c92290), SPH_C32(0xfc82ce4f), SPH_C32(0x3d140000),
	  SPH_C32(0xdceae700), SPH_C32(0x7e660000), SPH_C32(0xa1b00000),
	  SPH_C32(0x7c464804), SPH_C32(0x15154e98), SPH_C32(0x7131c3b6),
	  SPH_C32(0xfcfaf8a2) },
	{ SPH_C32(0x184f0004), SPH_C32(0x5b158000), SPH_C32(0xd7390000),
	  SPH_C32(0x28a30000), SPH_C32(0x9c8249e2), SPH_C32(0xbda93e6b),
	  SPH_C32(0x87fecba3), SPH_C32(0x293e4865), SPH_C32(0x3d790003),
	  SPH_C32(0x10710000), SPH_C32(0x3be20000), SPH_C32(0x8baf0000),
	  SPH_C32(0x0cfa30da), SPH_C32(0xdb83f261), SPH_C32(0xdd7e7128),
	  SPH_C32(0x5292b0f7) },
	{ SPH_C32(0x6deb0004), SPH_C32(0x999ea700), SPH_C32(0x439d0000),
	  SPH_C32(0xb8560000), SPH_C32(0x67fa1e02), SPH_C32(0xf46735c5),
	  SPH_C32(0x90990f20), SPH_C32(0x87e12e1b), SPH_C32(0xec1f0003),
	  SPH_C32(0x0bcd0300), SPH_C32(0xa50e0000), SPH_C32(0x7d3b0000),
	  SPH_C32(0x0ff875fd), SPH_C32(0x14f30e93), SPH_C32(0x693d6a3f),
	  SPH_C32(0xd7ed8cdc) },
	{ SPH_C32(0xc9290004), SPH_C32(0x40a98300), SPH_C32(0x49d50000),
	  SPH_C32(0xde370000), SPH_C32(0x9f800cc5), SPH_C32(0x72d9c299),
	  SPH_C32(0x33bdd0b4), SPH_C32(0xac41744e), SPH_C32(0x99bb0003),
	  SPH_C32(0xc9462400), SPH_C32(0x31aa0000), SPH_C32(0xedce0000),
	  SPH_C32(0xf480221d), SPH_C32(0x5d3d053d), SPH_C32(0x7e5aaebc),
	  SPH_C32(0x7932eaa2) },
	{ SPH_C32(0xbc8d0004), SPH_C32(0x8222a400), SPH_C32(0xdd710000),
	  SPH_C32(0x4ec20000), SPH_C32(0x64f85b25), SPH_C32(0x3b17c937),
	  SPH_C32(0x24da1437), SPH_C32(0x029e1230), SPH_C32(0x48dd0003),
	  SPH_C32(0xd2fa2700), SPH_C32(0xaf460000), SPH_C32(0x1b5a0000),
	  SPH_C32(0xf782673a), SPH_C32(0x924df9cf), SPH_C32(0xca19b5ab),
	  SPH_C32(0xfc4dd689) },
	{ SPH_C32(0x6d860007), SPH_C32(0x55054000), SPH_C32(0x06190000),
	  SPH_C32(0x92490000), SPH_C32(0x174666dc), SPH_C32(0x3af1893c),
	  SPH_C32(0x3cd6bdbe), SPH_C32(0x2989664e), SPH_C32(0xd3b60002),
	  SPH_C32(0x7f274000), SPH_C32(0xc8dc0000), SPH_C32(0x2c310000),
	  SPH_C32(0xb14f42c3), SPH_C32(0x6c9219a4), SPH_C32(0x97453192),
	  SPH_C32(0xac3942a3) },
	{ SPH_C32(0x18220007), SPH_C32(0x978e6700), SPH_C32(0x92bd0000),
	  SPH_C32(0x02bc0000), SPH_C32(0xec3e313c), SPH_C32(0x733f8292),
	  SPH_C32(0x2bb1793d), SPH_C32(0x87560030), SPH_C32(0x02d00002),
	  SPH_C32(0x649b4300), SPH_C32(0x56300000), SPH_C32(0xdaa50000),
	  SPH_C32(0xb24d07e4), SPH_C32(0xa3e2e556), SPH_C32(0x23062a85),
	  SPH_C32(0x29467e88) },
	{ SPH_C32(0xbce00007), SPH_C32(0x4eb94300), SPH_C32(0x98f50000),
	  SPH_C32(0x64dd0000), SPH_C32(0x144423fb), SPH_C32(0xf58175ce),
	  SPH_C32(0x8895a6a9), SPH_C32(0xacf65a65), SPH_C32(0x77740002),
	  SPH_C32(0xa6106400), SPH_C32(0xc2940000), SPH_C32(0x4a500000),
	  SPH_C32(0x49355004), SPH_C32(0xea2ceef8), SPH_C32(0x3461ee06),
	  SPH_C32(0x879918f6) },
	{ SPH_C32(0xc9440007), SPH_C32(0x8c326400), SPH_C32(0x0c510000),
	  SPH_C32(0xf4280000), SPH_C32(0xef3c741b), SPH_C32(0xbc4f7e60),
	  SPH_C32(0x9ff2622a), SPH_C32(0x02293c1b), SPH_C32(0xa6120002),
	  SPH_C32(0xbdac6700), SPH_C32(0x5c780000), SPH_C32(0xbcc40000),
	  SPH_C32(0x4a371523), SPH_C32(0x255c120a), SPH_C32(0x8022f511),
	  SPH_C32(0x02e624dd) }
};

static const sph_u32 T512_20[32][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xc6730000), SPH_C32(0xaf8d000c), SPH_C32(0xa4c10000),
	  SPH_C32(0x218d0000), SPH_C32(0x23111587), SPH_C32(0x7913512f),
	  SPH_C32(0x1d28ac88), SPH_C32(0x378dd173), SPH_C32(0xaf220000),
	  SPH_C32(0x7b6c0090), SPH_C32(0x67e20000), SPH_C32(0x8da20000),
	  SPH_C32(0xc7841e29), SPH_C32(0xb7b744f3), SPH_C32(0x9ac484f4),
	  SPH_C32(0x8b6c72bd) },
	{ SPH_C32(0x0c720000), SPH_C32(0x49e50f00), SPH_C32(0x42790000),
	  SPH_C32(0x5cea0000), SPH_C32(0x33aa301a), SPH_C32(0x15822514),
	  SPH_C32(0x95a34b7b), SPH_C32(0xb44b0090), SPH_C32(0xfe220000),
	  SPH_C32(0xa7580500), SPH_C32(0x25d10000), SPH_C32(0xf7600000),
	  SPH_C32(0x893178da), SPH_C32(0x1fd4f860), SPH_C32(0x4ed0a315),
	  SPH_C32(0xa123ff9f) },
	{ SPH_C32(0xca010000), SPH_C32(0xe6680f0c), SPH_C32(0xe6b80000),
	  SPH_C32(0x7d670000), SPH_C32(0x10bb259d), SPH_C32(0x6c91743b),
	  SPH_C32(0x888be7f3), SPH_C32(0x83c6d1e3), SPH_C32(0x51000000),
	  SPH_C32(0xdc340590), SPH_C32(0x42330000), SPH_C32(0x7ac20000),
	  SPH_C32(0x4eb566f3), SPH_C32(0xa863bc93), SPH_C32(0xd41427e1),
	  SPH_C32(0x2a4f8d22) },
	{ SPH_C32(0xfe220000), SPH_C32(0xa7580500), SPH_C32(0x25d10000),
	  SPH_C32(0xf7600000), SPH_C32(0x893178da), SPH_C32(0x1fd4f860),
	  SPH_C32(0x4ed0a315), SPH_C32(0xa123ff9f), SPH_C32(0xf2500000),
	  SPH_C32(0xeebd0a00), SPH_C32(0x67a80000), SPH_C32(0xab8a0000),
	  SPH_C32(0xba9b48c0), SPH_C32(0x0a56dd74), SPH_C32(0xdb73e86e),
	  SPH_C32(0x1568ff0f) },
	{ SPH_C32(0x38510000), SPH_C32(0x08d5050c), SPH_C32(0x81100000),
	  SPH_C32(0xd6ed0000), SPH_C32(0xaa206d5d), SPH_C32(0x66c7a94f),
	  SPH_C32(0x53f80f9d), SPH_C32(0x96ae2eec), SPH_C32(0x5d720000),
	  SPH_C32(0x95d10a90), SPH_C32(0x004a0000), SPH_C32(0x26280000),
	  SPH_C32(0x7d1f56e9), SPH_C32(0xbde19987), SPH_C32(0x41b76c9a),
	  SPH_C32(0x9e048db2) },
	{ SPH_C32(0xf2500000), SPH_C32(0xeebd0a00), SPH_C32(0x67a80000),
	  SPH_C32(0xab8a0000), SPH_C32(0xba9b48c0), SPH_C32(0x0a56dd74),
	  SPH_C32(0xdb73e86e), SPH_C32(0x1568ff0f), SPH_C32(0x0c720000),
	  SPH_C32(0x49e50f00), SPH_C32(0x42790000), SPH_C32(0x5cea0000),
	  SPH_C32(0x33aa301a), SPH_C32(0x15822514), SPH_C32(0x95a34b7b),
	  SPH_C32(0xb44b0090) },
	{ SPH_C32(0x34230000), SPH_C32(0x41300a0c), SPH_C32(0xc3690000),
	  SPH_C32(0x8a070000), SPH_C32(0x998a5d47), SPH_C32(0x73458c5b),
	  SPH_C32(0xc65b44e6), SPH_C32(0x22e52e7c), SPH_C32(0xa3500000),
	  SPH_C32(0x32890f90), SPH_C32(0x259b0000), SPH_C32(0xd1480000),
	  SPH_C32(0xf42e2e33), SPH_C32(0xa23561e7), SPH_C32(0x0f67cf8f),
	  SPH_C32(0x3f27722d) },
	{ SPH_C32(0x45180000), SPH_C32(0xa5b51700), SPH_C32(0xf96a0000),
	  SPH_C32(0x3b480000), SPH_C32(0x1ecc142c), SPH_C32(0x231395d6),
	  SPH_C32(0x16bca6b0), SPH_C32(0xdf33f4df), SPH_C32(0xb83d0000),
	  SPH_C32(0x16710600), SPH_C32(0x379a0000), SPH_C32(0xf5b10000),
	  SPH_C32(0x228161ac), SPH_C32(0xae48f145), SPH_C32(0x66241616),
	  SPH_C32(0xc5c1eb3e) },
	{ SPH_C32(0x836b0000), SPH_C32(0x0a38170c), SPH_C32(0x5dab0000),
	  SPH_C32(0x1ac50000), SPH_C32(0x3ddd01ab), SPH_C32(0x5a00c4f9),
	  SPH_C32(0x0b940a38), SPH_C32(0xe8be25ac), SPH_C32(0x171f0000),
	  SPH_C32(0x6d1d0690), SPH_C32(0x50780000), SPH_C32(0x78130000),
	  SPH_C32(0xe5057f85), SPH_C32(0x19ffb5b6), SPH_C32(0xfce092e2),
	  SPH_C32(0x4ead9983) },
	{ SPH_C32(0x496a0000), SPH_C32(0xec501800), SPH_C32(0xbb130000),
	  SPH_C32(0x67a20000), SPH_C32(0x2d662436), SPH_C32(0x3691b0c2),
	  SPH_C32(0x831fedcb), SPH_C32(0x6b78f44f), SPH_C32(0x461f0000),
	  SPH_C32(0xb1290300), SPH_C32(0x124b0000), SPH_C32(0x02d10000),
	  SPH_C32(0xabb01976), SPH_C32(0xb19c0925), SPH_C32(0x28f4b503),
	  SPH_C32(0x64e214a1) },
	{ SPH_C32(0x8f190000), SPH_C32(0x43dd180c), SPH_C32(0x1fd20000),
	  SPH_C32(0x462f0000), SPH_C32(0x0e7731b1), SPH_C32(0x4f82e1ed),
	  SPH_C32(0x9e374143), SPH_C32(0x5cf5253c), SPH_C32(0xe93d0000),
	  SPH_C32(0xca450390), SPH_C32(0x75a90000), SPH_C32(0x8f730000),
	  SPH_C32(0x6c34075f), SPH_C32(0x062b4dd6), SPH_C32(0xb23031f7),
	  SPH_C32(0xef8e661c) },
	{ SPH_C32(0xbb3a0000), SPH_C32(0x02ed1200), SPH_C32(0xdcbb0000),
	  SPH_C32(0xcc280000), SPH_C32(0x97fd6cf6), SPH_C32(0x3cc76db6),
	  SPH_C32(0x586c05a5), SPH_C32(0x7e100b40), SPH_C32(0x4a6d0000),
	  SPH_C32(0xf8cc0c00), SPH_C32(0x50320000), SPH_C32(0x5e3b0000),
	  SPH_C32(0x981a296c), SPH_C32(0xa41e2c31), SPH_C32(0xbd57fe78),
	  SPH_C32(0xd0a91431) },
	{ SPH_C32(0x7d490000), SPH_C32(0xad60120c), SPH_C32(0x787a0000),
	  SPH_C32(0xeda50000), SPH_C32(0xb4ec7971), SPH_C32(0x45d43c99),
	  SPH_C32(0x4544a92d), SPH_C32(0x499dda33), SPH_C32(0xe54f0000),
	  SPH_C32(0x83a00c90), SPH_C32(0x37d00000), SPH_C32(0xd3990000),
	  SPH_C32(0x5f9e3745), SPH_C32(0x13a968c2), SPH_C32(0x27937a8c),
	  SPH_C32(0x5bc5668c) },
	{ SPH_C32(0xb7480000), SPH_C32(0x4b081d00), SPH_C32(0x9ec20000),
	  SPH_C32(0x90c20000), SPH_C32(0xa4575cec), SPH_C32(0x294548a2),
	  SPH_C32(0xcdcf4ede), SPH_C32(0xca5b0bd0), SPH_C32(0xb44f0000),
	  SPH_C32(0x5f940900), SPH_C32(0x75e30000), SPH_C32(0xa95b0000),
	  SPH_C32(0x112b51b6), SPH_C32(0xbbcad451), SPH_C32(0xf3875d6d),
	  SPH_C32(0x718aebae) },
	{ SPH_C32(0x713b0000), SPH_C32(0xe4851d0c), SPH_C32(0x3a030000),
	  SPH_C32(0xb14f0000), SPH_C32(0x8746496b), SPH_C32(0x5056198d),
	  SPH_C32(0xd0e7e256), SPH_C32(0xfdd6daa3), SPH_C32(0x1b6d0000),
	  SPH_C32(0x24f80990), SPH_C32(0x12010000), SPH_C32(0x24f90000),
	  SPH_C32(0xd6af4f9f), SPH_C32(0x0c7d90a2), SPH_C32(0x6943d999),
	  SPH_C32(0xfae69913) },
	{ SPH_C32(0xb83d0000), SPH_C32(0x16710600), SPH_C32(0x379a0000),
	  SPH_C32(0xf5b10000), SPH_C32(0x228161ac), SPH_C32(0xae48f145),
	  SPH_C32(0x66241616), SPH_C32(0xc5c1eb3e), SPH_C32(0xfd250000),
	  SPH_C32(0xb3c41100), SPH_C32(0xcef00000), SPH_C32(0xcef90000),
	  SPH_C32(0x3c4d7580), SPH_C32(0x8d5b6493), SPH_C32(0x7098b0a6),
	  SPH_C32(0x1af21fe1) },
	{ SPH_C32(0x7e4e0000), SPH_C32(0xb9fc060c), SPH_C32(0x935b0000),
	  SPH_C32(0xd43c0000), SPH_C32(0x0190742b), SPH_C32(0xd75ba06a),
	  SPH_C32(0x7b0cba9e), SPH_C32(0xf24c3a4d), SPH_C32(0x52070000),
	  SPH_C32(0xc8a81190), SPH_C32(0xa9120000), SPH_C32(0x435b0000),
	  SPH_C32(0xfbc96ba9), SPH_C32(0x3aec2060), SPH_C32(0xea5c3452),
	  SPH_C32(0x919e6d5c) },
	{ SPH_C32(0xb44f0000), SPH_C32(0x5f940900), SPH_C32(0x75e30000),
	  SPH_C32(0xa95b0000), SPH_C32(0x112b51b6), SPH_C32(0xbbcad451),
	  SPH_C32(0xf3875d6d), SPH_C32(0x718aebae), SPH_C32(0x03070000),
	  SPH_C32(0x149c1400), SPH_C32(0xeb210000), SPH_C32(0x39990000),
	  SPH_C32(0xb57c0d5a), SPH_C32(0x928f9cf3), SPH_C32(0x3e4813b3),
	  SPH_C32(0xbbd1e07e) },
	{ SPH_C32(0x723c0000), SPH_C32(0xf019090c), SPH_C32(0xd1220000),
	  SPH_C32(0x88d60000), SPH_C32(0x323a4431), SPH_C32(0xc2d9857e),
	  SPH_C32(0xeeaff1e5), SPH_C32(0x46073add), SPH_C32(0xac250000),
	  SPH_C32(0x6ff01490), SPH_C32(0x8cc30000), SPH_C32(0xb43b0000),
	  SPH_C32(0x72f81373), SPH_C32(0x2538d800), SPH_C32(0xa48c9747),
	  SPH_C32(0x30bd92c3) },
	{ SPH_C32(0x461f0000), SPH_C32(0xb1290300), SPH_C32(0x124b0000),
	  SPH_C32(0x02d10000), SPH_C32(0xabb01976), SPH_C32(0xb19c0925),
	  SPH_C32(0x28f4b503), SPH_C32(0x64e214a1), SPH_C32(0x0f750000),
	  SPH_C32(0x5d791b00), SPH_C32(0xa9580000), SPH_C32(0x65730000),
	  SPH_C32(0x86d63d40), SPH_C32(0x870db9e7), SPH_C32(0xabeb58c8),
	  SPH_C32(0x0f9ae0ee) },
	{ SPH_C32(0x806c0000), SPH_C32(0x1ea4030c), SPH_C32(0xb68a0000),
	  SPH_C32(0x235c0000), SPH_C32(0x88a10cf1), SPH_C32(0xc88f580a),
	  SPH_C32(0x35dc198b), SPH_C32(0x536fc5d2), SPH_C32(0xa0570000),
	  SPH_C32(0x26151b90), SPH_C32(0xceba0000), SPH_C32(0xe8d10000),
	  SPH_C32(0x41522369), SPH_C32(0x30bafd14), SPH_C32(0x312fdc3c),
	  SPH_C32(0x84f69253) },
	{ SPH_C32(0x4a6d0000), SPH_C32(0xf8cc0c00), SPH_C32(0x50320000),
	  SPH_C32(0x5e3b0000), SPH_C32(0x981a296c), SPH_C32(0xa41e2c31),
	  SPH_C32(0xbd57fe78), SPH_C32(0xd0a91431), SPH_C32(0xf1570000),
	  SPH_C32(0xfa211e00), SPH_C32(0x8c890000), SPH_C32(0x92130000),
	  SPH_C32(0x0fe7459a), SPH_C32(0x98d94187), SPH_C32(0xe53bfbdd),
	  SPH_C32(0xaeb91f71) },
	{ SPH_C32(0x8c1e0000), SPH_C32(0x57410c0c), SPH_C32(0xf4f30000),
	  SPH_C32(0x7fb60000), SPH_C32(0xbb0b3ceb), SPH_C32(0xdd0d7d1e),
	  SPH_C32(0xa07f52f0), SPH_C32(0xe724c542), SPH_C32(0x5e750000),
	  SPH_C32(0x814d1e90), SPH_C32(0xeb6b0000), SPH_C32(0x1fb10000),
	  SPH_C32(0xc8635bb3), SPH_C32(0x2f6e0574), SPH_C32(0x7fff7f29),
	  SPH_C32(0x25d56dcc) },
	{ SPH_C32(0xfd250000), SPH_C32(0xb3c41100), SPH_C32(0xcef00000),
	  SPH_C32(0xcef90000), SPH_C32(0x3c4d7580), SPH_C32(0x8d5b6493),
	  SPH_C32(0x7098b0a6), SPH_C32(0x1af21fe1), SPH_C32(0x45180000),
	  SPH_C32(0xa5b51700), SPH_C32(0xf96a0000), SPH_C32(0x3b480000),
	  SPH_C32(0x1ecc142c), SPH_C32(0x231395d6), SPH_C32(0x16bca6b0),
	  SPH_C32(0xdf33f4df) },
	{ SPH_C32(0x3b560000), SPH_C32(0x1c49110c), SPH_C32(0x6a310000),
	  SPH_C32(0xef740000), SPH_C32(0x1f5c6007), SPH_C32(0xf44835bc),
	  SPH_C32(0x6db01c2e), SPH_C32(0x2d7fce92), SPH_C32(0xea3a0000),
	  SPH_C32(0xded91790), SPH_C32(0x9e880000), SPH_C32(0xb6ea0000),
	  SPH_C32(0xd9480a05), SPH_C32(0x94a4d125), SPH_C32(0x8c782244),
	  SPH_C32(0x545f8662) },
	{ SPH_C32(0xf1570000), SPH_C32(0xfa211e00), SPH_C32(0x8c890000),
	  SPH_C32(0x92130000), SPH_C32(0x0fe7459a), SPH_C32(0x98d94187),
	  SPH_C32(0xe53bfbdd), SPH_C32(0xaeb91f71), SPH_C32(0xbb3a0000),
	  SPH_C32(0x02ed1200), SPH_C32(0xdcbb0000), SPH_C32(0xcc280000),
	  SPH_C32(0x97fd6cf6), SPH_C32(0x3cc76db6), SPH_C32(0x586c05a5),
	  SPH_C32(0x7e100b40) },
	{ SPH_C32(0x37240000), SPH_C32(0x55ac1e0c), SPH_C32(0x28480000),
	  SPH_C32(0xb39e0000), SPH_C32(0x2cf6501d), SPH_C32(0xe1ca10a8),
	  SPH_C32(0xf8135755), SPH_C32(0x9934ce02), SPH_C32(0x14180000),
	  SPH_C32(0x79811290), SPH_C32(0xbb590000), SPH_C32(0x418a0000),
	  SPH_C32(0x507972df), SPH_C32(0x8b702945), SPH_C32(0xc2a88151),
	  SPH_C32(0xf57c79fd) },
	{ SPH_C32(0x03070000), SPH_C32(0x149c1400), SPH_C32(0xeb210000),
	  SPH_C32(0x39990000), SPH_C32(0xb57c0d5a), SPH_C32(0x928f9cf3),
	  SPH_C32(0x3e4813b3), SPH_C32(0xbbd1e07e), SPH_C32(0xb7480000),
	  SPH_C32(0x4b081d00), SPH_C32(0x9ec20000), SPH_C32(0x90c20000),
	  SPH_C32(0xa4575cec), SPH_C32(0x294548a2), SPH_C32(0xcdcf4ede),
	  SPH_C32(0xca5b0bd0) },
	{ SPH_C32(0xc5740000), SPH_C32(0xbb11140c), SPH_C32(0x4fe00000),
	  SPH_C32(0x18140000), SPH_C32(0x966d18dd), SPH_C32(0xeb9ccddc),
	  SPH_C32(0x2360bf3b), SPH_C32(0x8c5c310d), SPH_C32(0x186a0000),
	  SPH_C32(0x30641d90), SPH_C32(0xf9200000), SPH_C32(0x1d600000),
	  SPH_C32(0x63d342c5), SPH_C32(0x9ef20c51), SPH_C32(0x570bca2a),
	  SPH_C32(0x4137796d) },
	{ SPH_C32(0x0f750000), SPH_C32(0x5d791b00), SPH_C32(0xa9580000),
	  SPH_C32(0x65730000), SPH_C32(0x86d63d40), SPH_C32(0x870db9e7),
	  SPH_C32(0xabeb58c8), SPH_C32(0x0f9ae0ee), SPH_C32(0x496a0000),
	  SPH_C32(0xec501800), SPH_C32(0xbb130000), SPH_C32(0x67a20000),
	  SPH_C32(0x2d662436), SPH_C32(0x3691b0c2), SPH_C32(0x831fedcb),
	  SPH_C32(0x6b78f44f) },
	{ SPH_C32(0xc9060000), SPH_C32(0xf2f41b0c), SPH_C32(0x0d990000),
	  SPH_C32(0x44fe0000), SPH_C32(0xa5c728c7), SPH_C32(0xfe1ee8c8),
	  SPH_C32(0xb6c3f440), SPH_C32(0x3817319d), SPH_C32(0xe6480000),
	  SPH_C32(0x973c1890), SPH_C32(0xdcf10000), SPH_C32(0xea000000),
	  SPH_C32(0xeae23a1f), SPH_C32(0x8126f431), SPH_C32(0x19db693f),
	  SPH_C32(0xe01486f2) }
};

static const sph_u32 T512_25[32][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x14190000), SPH_C32(0x23ca003c), SPH_C32(0x50df0000),
	  SPH_C32(0x44b60000), SPH_C32(0x1b6c67b0), SPH_C32(0x3cf3ac75),
	  SPH_C32(0x61e610b0), SPH_C32(0xdbcadb80), SPH_C32(0xe3430000),
	  SPH_C32(0x3a4e0014), SPH_C32(0xf2c60000), SPH_C32(0xaa4e0000),
	  SPH_C32(0xdb1e42a6), SPH_C32(0x256bbe15), SPH_C32(0x123db156),
	  SPH_C32(0x3a4e99d7) },
	{ SPH_C32(0xe3430000), SPH_C32(0x3a4e0014), SPH_C32(0xf2c60000),
	  SPH_C32(0xaa4e0000), SPH_C32(0xdb1e42a6), SPH_C32(0x256bbe15),
	  SPH_C32(0x123db156), SPH_C32(0x3a4e99d7), SPH_C32(0xf75a0000),
	  SPH_C32(0x19840028), SPH_C32(0xa2190000), SPH_C32(0xeef80000),
	  SPH_C32(0xc0722516), SPH_C32(0x19981260), SPH_C32(0x73dba1e6),
	  SPH_C32(0xe1844257) },
	{ SPH_C32(0xf75a0000), SPH_C32(0x19840028), SPH_C32(0xa2190000),
	  SPH_C32(0xeef80000), SPH_C32(0xc0722516), SPH_C32(0x19981260),
	  SPH_C32(0x73dba1e6), SPH_C32(0xe1844257), SPH_C32(0x14190000),
	  SPH_C32(0x23ca003c), SPH_C32(0x50df0000), SPH_C32(0x44b60000),
	  SPH_C32(0x1b6c67b0), SPH_C32(0x3cf3ac75), SPH_C32(0x61e610b0),
	  SPH_C32(0xdbcadb80) },
	{ SPH_C32(0x54500000), SPH_C32(0x0671005c), SPH_C32(0x25ae0000),
	  SPH_C32(0x6a1e0000), SPH_C32(0x2ea54edf), SPH_C32(0x664e8512),
	  SPH_C32(0xbfba18c3), SPH_C32(0x7e715d17), SPH_C32(0xbc8d0000),
	  SPH_C32(0xfc3b0018), SPH_C32(0x19830000), SPH_C32(0xd10b0000),
	  SPH_C32(0xae1878c4), SPH_C32(0x42a69856), SPH_C32(0x0012da37),
	  SPH_C32(0x2c3b504e) },
	{ SPH_C32(0x40490000), SPH_C32(0x25bb0060), SPH_C32(0x75710000),
	  SPH_C32(0x2ea80000), SPH_C32(0x35c9296f), SPH_C32(0x5abd2967),
	  SPH_C32(0xde5c0873), SPH_C32(0xa5bb8697), SPH_C32(0x5fce0000),
	  SPH_C32(0xc675000c), SPH_C32(0xeb450000), SPH_C32(0x7b450000),
	  SPH_C32(0x75063a62), SPH_C32(0x67cd2643), SPH_C32(0x122f6b61),
	  SPH_C32(0x1675c999) },
	{ SPH_C32(0xb7130000), SPH_C32(0x3c3f0048), SPH_C32(0xd7680000),
	  SPH_C32(0xc0500000), SPH_C32(0xf5bb0c79), SPH_C32(0x43253b07),
	  SPH_C32(0xad87a995), SPH_C32(0x443fc4c0), SPH_C32(0x4bd70000),
	  SPH_C32(0xe5bf0030), SPH_C32(0xbb9a0000), SPH_C32(0x3ff30000),
	  SPH_C32(0x6e6a5dd2), SPH_C32(0x5b3e8a36), SPH_C32(0x73c97bd1),
	  SPH_C32(0xcdbf1219) },
	{ SPH_C32(0xa30a0000), SPH_C32(0x1ff50074), SPH_C32(0x87b70000),
	  SPH_C32(0x84e60000), SPH_C32(0xeed76bc9), SPH_C32(0x7fd69772),
	  SPH_C32(0xcc61b925), SPH_C32(0x9ff51f40), SPH_C32(0xa8940000),
	  SPH_C32(0xdff10024), SPH_C32(0x495c0000), SPH_C32(0x95bd0000),
	  SPH_C32(0xb5741f74), SPH_C32(0x7e553423), SPH_C32(0x61f4ca87),
	  SPH_C32(0xf7f18bce) },
	{ SPH_C32(0xbc8d0000), SPH_C32(0xfc3b0018), SPH_C32(0x19830000),
	  SPH_C32(0xd10b0000), SPH_C32(0xae1878c4), SPH_C32(0x42a69856),
	  SPH_C32(0x0012da37), SPH_C32(0x2c3b504e), SPH_C32(0xe8dd0000),
	  SPH_C32(0xfa4a0044), SPH_C32(0x3c2d0000), SPH_C32(0xbb150000),
	  SPH_C32(0x80bd361b), SPH_C32(0x24e81d44), SPH_C32(0xbfa8c2f4),
	  SPH_C32(0x524a0d59) },
	{ SPH_C32(0xa8940000), SPH_C32(0xdff10024), SPH_C32(0x495c0000),
	  SPH_C32(0x95bd0000), SPH_C32(0xb5741f74), SPH_C32(0x7e553423),
	  SPH_C32(0x61f4ca87), SPH_C32(0xf7f18bce), SPH_C32(0x0b9e0000),
	  SPH_C32(0xc0040050), SPH_C32(0xceeb0000), SPH_C32(0x115b0000),
	  SPH_C32(0x5ba374bd), SPH_C32(0x0183a351), SPH_C32(0xad9573a2),
	  SPH_C32(0x6804948e) },
	{ SPH_C32(0x5fce0000), SPH_C32(0xc675000c), SPH_C32(0xeb450000),
	  SPH_C32(0x7b450000), SPH_C32(0x75063a62), SPH_C32(0x67cd2643),
	  SPH_C32(0x122f6b61), SPH_C32(0x1675c999), SPH_C32(0x1f870000),
	  SPH_C32(0xe3ce006c), SPH_C32(0x9e340000), SPH_C32(0x55ed0000),
	  SPH_C32(0x40cf130d), SPH_C32(0x3d700f24), SPH_C32(0xcc736312),
	  SPH_C32(0xb3ce4f0e) },
	{ SPH_C32(0x4bd70000), SPH_C32(0xe5bf0030), SPH_C32(0xbb9a0000),
	  SPH_C32(0x3ff30000), SPH_C32(0x6e6a5dd2), SPH_C32(0x5b3e8a36),
	  SPH_C32(0x73c97bd1), SPH_C32(0xcdbf1219), SPH_C32(0xfcc40000),
	  SPH_C32(0xd9800078), SPH_C32(0x6cf20000), SPH_C32(0xffa30000),
	  SPH_C32(0x9bd151ab), SPH_C32(0x181bb131), SPH_C32(0xde4ed244),
	  SPH_C32(0x8980d6d9) },
	{ SPH_C32(0xe8dd0000), SPH_C32(0xfa4a0044), SPH_C32(0x3c2d0000),
	  SPH_C32(0xbb150000), SPH_C32(0x80bd361b), SPH_C32(0x24e81d44),
	  SPH_C32(0xbfa8c2f4), SPH_C32(0x524a0d59), SPH_C32(0x54500000),
	  SPH_C32(0x0671005c), SPH_C32(0x25ae0000), SPH_C32(0x6a1e0000),
	  SPH_C32(0x2ea54edf), SPH_C32(0x664e8512), SPH_C32(0xbfba18c3),
	  SPH_C32(0x7e715d17) },
	{ SPH_C32(0xfcc40000), SPH_C32(0xd9800078), SPH_C32(0x6cf20000),
	  SPH_C32(0xffa30000), SPH_C32(0x9bd151ab), SPH_C32(0x181bb131),
	  SPH_C32(0xde4ed244), SPH_C32(0x8980d6d9), SPH_C32(0xb7130000),
	  SPH_C32(0x3c3f0048), SPH_C32(0xd7680000), SPH_C32(0xc0500000),
	  SPH_C32(0xf5bb0c79), SPH_C32(0x43253b07), SPH_C32(0xad87a995),
	  SPH_C32(0x443fc4c0) },
	{ SPH_C32(0x0b9e0000), SPH_C32(0xc0040050), SPH_C32(0xceeb0000),
	  SPH_C32(0x115b0000), SPH_C32(0x5ba374bd), SPH_C32(0x0183a351),
	  SPH_C32(0xad9573a2), SPH_C32(0x6804948e), SPH_C32(0xa30a0000),
	  SPH_C32(0x1ff50074), SPH_C32(0x87b70000), SPH_C32(0x84e60000),
	  SPH_C32(0xeed76bc9), SPH_C32(0x7fd69772), SPH_C32(0xcc61b925),
	  SPH_C32(0x9ff51f40) },
	{ SPH_C32(0x1f870000), SPH_C32(0xe3ce006c), SPH_C32(0x9e340000),
	  SPH_C32(0x55ed0000), SPH_C32(0x40cf130d), SPH_C32(0x3d700f24),
	  SPH_C32(0xcc736312), SPH_C32(0xb3ce4f0e), SPH_C32(0x40490000),
	  SPH_C32(0x25bb0060), SPH_C32(0x75710000), SPH_C32(0x2ea80000),
	  SPH_C32(0x35c9296f), SPH_C32(0x5abd2967), SPH_C32(0xde5c0873),
	  SPH_C32(0xa5bb8697) },
	{ SPH_C32(0x69510000), SPH_C32(0xd4e1009c), SPH_C32(0xc3230000),
	  SPH_C32(0xac2f0000), SPH_C32(0xe4950bae), SPH_C32(0xcea415dc),
	  SPH_C32(0x87ec287c), SPH_C32(0xbce1a3ce), SPH_C32(0xc6730000),
	  SPH_C32(0xaf8d000c), SPH_C32(0xa4c10000), SPH_C32(0x218d0000),
	  SPH_C32(0x23111587), SPH_C32(0x7913512f), SPH_C32(0x1d28ac88),
	  SPH_C32(0x378dd173) },
	{ SPH_C32(0x7d480000), SPH_C32(0xf72b00a0), SPH_C32(0x93fc0000),
	  SPH_C32(0xe8990000), SPH_C32(0xfff96c1e), SPH_C32(0xf257b9a9),
	  SPH_C32(0xe60a38cc), SPH_C32(0x672b784e), SPH_C32(0x25300000),
	  SPH_C32(0x95c30018), SPH_C32(0x56070000), SPH_C32(0x8bc30000),
	  SPH_C32(0xf80f5721), SPH_C32(0x5c78ef3a), SPH_C32(0x0f151dde),
	  SPH_C32(0x0dc348a4) },
	{ SPH_C32(0x8a120000), SPH_C32(0xeeaf0088), SPH_C32(0x31e50000),
	  SPH_C32(0x06610000), SPH_C32(0x3f8b4908), SPH_C32(0xebcfabc9),
	  SPH_C32(0x95d1992a), SPH_C32(0x86af3a19), SPH_C32(0x31290000),
	  SPH_C32(0xb6090024), SPH_C32(0x06d80000), SPH_C32(0xcf750000),
	  SPH_C32(0xe3633091), SPH_C32(0x608b434f), SPH_C32(0x6ef30d6e),
	  SPH_C32(0xd6099324) },
	{ SPH_C32(0x9e0b0000), SPH_C32(0xcd6500b4), SPH_C32(0x613a0000),
	  SPH_C32(0x42d70000), SPH_C32(0x24e72eb8), SPH_C32(0xd73c07bc),
	  SPH_C32(0xf437899a), SPH_C32(0x5d65e199), SPH_C32(0xd26a0000),
	  SPH_C32(0x8c470030), SPH_C32(0xf41e0000), SPH_C32(0x653b0000),
	  SPH_C32(0x387d7237), SPH_C32(0x45e0fd5a), SPH_C32(0x7ccebc38),
	  SPH_C32(0xec470af3) },
	{ SPH_C32(0x3d010000), SPH_C32(0xd29000c0), SPH_C32(0xe68d0000),
	  SPH_C32(0xc6310000), SPH_C32(0xca304571), SPH_C32(0xa8ea90ce),
	  SPH_C32(0x385630bf), SPH_C32(0xc290fed9), SPH_C32(0x7afe0000),
	  SPH_C32(0x53b60014), SPH_C32(0xbd420000), SPH_C32(0xf0860000),
	  SPH_C32(0x8d096d43), SPH_C32(0x3bb5c979), SPH_C32(0x1d3a76bf),
	  SPH_C32(0x1bb6813d) },
	{ SPH_C32(0x29180000), SPH_C32(0xf15a00fc), SPH_C32(0xb6520000),
	  SPH_C32(0x82870000), SPH_C32(0xd15c22c1), SPH_C32(0x94193cbb),
	  SPH_C32(0x59b0200f), SPH_C32(0x195a2559), SPH_C32(0x99bd0000),
	  SPH_C32(0x69f80000), SPH_C32(0x4f840000), SPH_C32(0x5ac80000),
	  SPH_C32(0x56172fe5), SPH_C32(0x1ede776c), SPH_C32(0x0f07c7e9),
	  SPH_C32(0x21f818ea) },
	{ SPH_C32(0xde420000), SPH_C32(0xe8de00d4), SPH_C32(0x144b0000),
	  SPH_C32(0x6c7f0000), SPH_C32(0x112e07d7), SPH_C32(0x8d812edb),
	  SPH_C32(0x2a6b81e9), SPH_C32(0xf8de670e), SPH_C32(0x8da40000),
	  SPH_C32(0x4a32003c), SPH_C32(0x1f5b0000), SPH_C32(0x1e7e0000),
	  SPH_C32(0x4d7b4855), SPH_C32(0x222ddb19), SPH_C32(0x6ee1d759),
	  SPH_C32(0xfa32c36a) },
	{ SPH_C32(0xca5b0000), SPH_C32(0xcb1400e8), SPH_C32(0x44940000),
	  SPH_C32(0x28c90000), SPH_C32(0x0a426067), SPH_C32(0xb17282ae),
	  SPH_C32(0x4b8d9159), SPH_C32(0x2314bc8e), SPH_C32(0x6ee70000),
	  SPH_C32(0x707c0028), SPH_C32(0xed9d0000), SPH_C32(0xb4300000),
	  SPH_C32(0x96650af3), SPH_C32(0x0746650c), SPH_C32(0x7cdc660f),
	  SPH_C32(0xc07c5abd) },
	{ SPH_C32(0xd5dc0000), SPH_C32(0x28da0084), SPH_C32(0xdaa00000),
	  SPH_C32(0x7d240000), SPH_C32(0x4a8d736a), SPH_C32(0x8c028d8a),
	  SPH_C32(0x87fef24b), SPH_C32(0x90daf380), SPH_C32(0x2eae0000),
	  SPH_C32(0x55c70048), SPH_C32(0x98ec0000), SPH_C32(0x9a980000),
	  SPH_C32(0xa3ac239c), SPH_C32(0x5dfb4c6b), SPH_C32(0xa2806e7c),
	  SPH_C32(0x65c7dc2a) },
	{ SPH_C32(0xc1c50000), SPH_C32(0x0b1000b8), SPH_C32(0x8a7f0000),
	  SPH_C32(0x39920000), SPH_C32(0x51e114da), SPH_C32(0xb0f121ff),
	  SPH_C32(0xe618e2fb), SPH_C32(0x4b102800), SPH_C32(0xcded0000),
	  SPH_C32(0x6f89005c), SPH_C32(0x6a2a0000), SPH_C32(0x30d60000),
	  SPH_C32(0x78b2613a), SPH_C32(0x7890f27e), SPH_C32(0xb0bddf2a),
	  SPH_C32(0x5f8945fd) },
	{ SPH_C32(0x369f0000), SPH_C32(0x12940090), SPH_C32(0x28660000),
	  SPH_C32(0xd76a0000), SPH_C32(0x919331cc), SPH_C32(0xa969339f),
	  SPH_C32(0x95c3431d), SPH_C32(0xaa946a57), SPH_C32(0xd9f40000),
	  SPH_C32(0x4c430060), SPH_C32(0x3af50000), SPH_C32(0x74600000),
	  SPH_C32(0x63de068a), SPH_C32(0x44635e0b), SPH_C32(0xd15bcf9a),
	  SPH_C32(0x84439e7d) },
	{ SPH_C32(0x22860000), SPH_C32(0x315e00ac), SPH_C32(0x78b90000),
	  SPH_C32(0x93dc0000), SPH_C32(0x8aff567c), SPH_C32(0x959a9fea),
	  SPH_C32(0xf42553ad), SPH_C32(0x715eb1d7), SPH_C32(0x3ab70000),
	  SPH_C32(0x760d0074), SPH_C32(0xc8330000), SPH_C32(0xde2e0000),
	  SPH_C32(0xb8c0442c), SPH_C32(0x6108e01e), SPH_C32(0xc3667ecc),
	  SPH_C32(0xbe0d07aa) },
	{ SPH_C32(0x818c0000), SPH_C32(0x2eab00d8), SPH_C32(0xff0e0000),
	  SPH_C32(0x173a0000), SPH_C32(0x64283db5), SPH_C32(0xea4c0898),
	  SPH_C32(0x3844ea88), SPH_C32(0xeeabae97), SPH_C32(0x92230000),
	  SPH_C32(0xa9fc0050), SPH_C32(0x816f0000), SPH_C32(0x4b930000),
	  SPH_C32(0x0db45b58), SPH_C32(0x1f5dd43d), SPH_C32(0xa292b44b),
	  SPH_C32(0x49fc8c64) },
	{ SPH_C32(0x95950000), SPH_C32(0x0d6100e4), SPH_C32(0xafd10000),
	  SPH_C32(0x538c0000), SPH_C32(0x7f445a05), SPH_C32(0xd6bfa4ed),
	  SPH_C32(0x59a2fa38), SPH_C32(0x35617517), SPH_C32(0x71600000),
	  SPH_C32(0x93b20044), SPH_C32(0x73a90000), SPH_C32(0xe1dd0000),
	  SPH_C32(0xd6aa19fe), SPH_C32(0x3a366a28), SPH_C32(0xb0af051d),
	  SPH_C32(0x73b215b3) },
	{ SPH_C32(0x62cf0000), SPH_C32(0x14e500cc), SPH_C32(0x0dc80000),
	  SPH_C32(0xbd740000), SPH_C32(0xbf367f13), SPH_C32(0xcf27b68d),
	  SPH_C32(0x2a795bde), SPH_C32(0xd4e53740), SPH_C32(0x65790000),
	  SPH_C32(0xb0780078), SPH_C32(0x23760000), SPH_C32(0xa56b0000),
	  SPH_C32(0xcdc67e4e), SPH_C32(0x06c5c65d), SPH_C32(0xd14915ad),
	  SPH_C32(0xa878ce33) },
	{ SPH_C32(0x76d60000), SPH_C32(0x372f00f0), SPH_C32(0x5d170000),
	  SPH_C32(0xf9c20000), SPH_C32(0xa45a18a3), SPH_C32(0xf3d41af8),
	  SPH_C32(0x4b9f4b6e), SPH_C32(0x0f2fecc0), SPH_C32(0x863a0000),
	  SPH_C32(0x8a36006c), SPH_C32(0xd1b00000), SPH_C32(0x0f250000),
	  SPH_C32(0x16d83ce8), SPH_C32(0x23ae7848), SPH_C32(0xc374a4fb),
	  SPH_C32(0x923657e4) }
};

static const sph_u32 T512_30[32][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x1e4e0000), SPH_C32(0xdecf0000), SPH_C32(0x6df80180),
	  SPH_C32(0x77240000), SPH_C32(0xec47079e), SPH_C32(0xf4a0694e),
	  SPH_C32(0xcda31812), SPH_C32(0x98aa496e), SPH_C32(0xb2060000),
	  SPH_C32(0xc5690000), SPH_C32(0x28031200), SPH_C32(0x74670000),
	  SPH_C32(0xb6c236f4), SPH_C32(0xeb1239f8), SPH_C32(0x33d1dfec),
	  SPH_C32(0x094e3198) },
	{ SPH_C32(0xaec30000), SPH_C32(0x9c4f0001), SPH_C32(0x79d1e000),
	  SPH_C32(0x2c150000), SPH_C32(0x45cc75b3), SPH_C32(0x6650b736),
	  SPH_C32(0xab92f78f), SPH_C32(0xa312567b), SPH_C32(0xdb250000),
	  SPH_C32(0x09290000), SPH_C32(0x49aac000), SPH_C32(0x81e10000),
	  SPH_C32(0xcafe6b59), SPH_C32(0x42793431), SPH_C32(0x43566b76),
	  SPH_C32(0xe86cba2e) },
	{ SPH_C32(0xb08d0000), SPH_C32(0x42800001), SPH_C32(0x1429e180),
	  SPH_C32(0x5b310000), SPH_C32(0xa98b722d), SPH_C32(0x92f0de78),
	  SPH_C32(0x6631ef9d), SPH_C32(0x3bb81f15), SPH_C32(0x69230000),
	  SPH_C32(0xcc400000), SPH_C32(0x61a9d200), SPH_C32(0xf5860000),
	  SPH_C32(0x7c3c5dad), SPH_C32(0xa96b0dc9), SPH_C32(0x7087b49a),
	  SPH_C32(0xe1228bb6) },
	{ SPH_C32(0xdb250000), SPH_C32(0x09290000), SPH_C32(0x49aac000),
	  SPH_C32(0x81e10000), SPH_C32(0xcafe6b59), SPH_C32(0x42793431),
	  SPH_C32(0x43566b76), SPH_C32(0xe86cba2e), SPH_C32(0x75e60000),
	  SPH_C32(0x95660001), SPH_C32(0x307b2000), SPH_C32(0xadf40000),
	  SPH_C32(0x8f321eea), SPH_C32(0x24298307), SPH_C32(0xe8c49cf9),
	  SPH_C32(0x4b7eec55) },
	{ SPH_C32(0xc56b0000), SPH_C32(0xd7e60000), SPH_C32(0x2452c180),
	  SPH_C32(0xf6c50000), SPH_C32(0x26b96cc7), SPH_C32(0xb6d95d7f),
	  SPH_C32(0x8ef57364), SPH_C32(0x70c6f340), SPH_C32(0xc7e00000),
	  SPH_C32(0x500f0001), SPH_C32(0x18783200), SPH_C32(0xd9930000),
	  SPH_C32(0x39f0281e), SPH_C32(0xcf3bbaff), SPH_C32(0xdb154315),
	  SPH_C32(0x4230ddcd) },
	{ SPH_C32(0x75e60000), SPH_C32(0x95660001), SPH_C32(0x307b2000),
	  SPH_C32(0xadf40000), SPH_C32(0x8f321eea), SPH_C32(0x24298307),
	  SPH_C32(0xe8c49cf9), SPH_C32(0x4b7eec55), SPH_C32(0xaec30000),
	  SPH_C32(0x9c4f0001), SPH_C32(0x79d1e000), SPH_C32(0x2c150000),
	  SPH_C32(0x45cc75b3), SPH_C32(0x6650b736), SPH_C32(0xab92f78f),
	  SPH_C32(0xa312567b) },
	{ SPH_C32(0x6ba80000), SPH_C32(0x4ba90001), SPH_C32(0x5d832180),
	  SPH_C32(0xdad00000), SPH_C32(0x63751974), SPH_C32(0xd089ea49),
	  SPH_C32(0x256784eb), SPH_C32(0xd3d4a53b), SPH_C32(0x1cc50000),
	  SPH_C32(0x59260001), SPH_C32(0x51d2f200), SPH_C32(0x58720000),
	  SPH_C32(0xf30e4347), SPH_C32(0x8d428ece), SPH_C32(0x98432863),
	  SPH_C32(0xaa5c67e3) },
	{ SPH_C32(0x86790000), SPH_C32(0x3f390002), SPH_C32(0xe19ae000),
	  SPH_C32(0x98560000), SPH_C32(0x9565670e), SPH_C32(0x4e88c8ea),
	  SPH_C32(0xd3dd4944), SPH_C32(0x161ddab9), SPH_C32(0x30b70000),
	  SPH_C32(0xe5d00000), SPH_C32(0xf4f46000), SPH_C32(0x42c40000),
	  SPH_C32(0x63b83d6a), SPH_C32(0x78ba9460), SPH_C32(0x21afa1ea),
	  SPH_C32(0xb0a51834) },
	{ SPH_C32(0x98370000), SPH_C32(0xe1f60002), SPH_C32(0x8c62e180),
	  SPH_C32(0xef720000), SPH_C32(0x79226090), SPH_C32(0xba28a1a4),
	  SPH_C32(0x1e7e5156), SPH_C32(0x8eb793d7), SPH_C32(0x82b10000),
	  SPH_C32(0x20b90000), SPH_C32(0xdcf77200), SPH_C32(0x36a30000),
	  SPH_C32(0xd57a0b9e), SPH_C32(0x93a8ad98), SPH_C32(0x127e7e06),
	  SPH_C32(0xb9eb29ac) },
	{ SPH_C32(0x28ba0000), SPH_C32(0xa3760003), SPH_C32(0x984b0000),
	  SPH_C32(0xb4430000), SPH_C32(0xd0a912bd), SPH_C32(0x28d87fdc),
	  SPH_C32(0x784fbecb), SPH_C32(0xb50f8cc2), SPH_C32(0xeb920000),
	  SPH_C32(0xecf90000), SPH_C32(0xbd5ea000), SPH_C32(0xc3250000),
	  SPH_C32(0xa9465633), SPH_C32(0x3ac3a051), SPH_C32(0x62f9ca9c),
	  SPH_C32(0x58c9a21a) },
	{ SPH_C32(0x36f40000), SPH_C32(0x7db90003), SPH_C32(0xf5b30180),
	  SPH_C32(0xc3670000), SPH_C32(0x3cee1523), SPH_C32(0xdc781692),
	  SPH_C32(0xb5eca6d9), SPH_C32(0x2da5c5ac), SPH_C32(0x59940000),
	  SPH_C32(0x29900000), SPH_C32(0x955db200), SPH_C32(0xb7420000),
	  SPH_C32(0x1f8460c7), SPH_C32(0xd1d199a9), SPH_C32(0x51281570),
	  SPH_C32(0x51879382) },
	{ SPH_C32(0x5d5c0000), SPH_C32(0x36100002), SPH_C32(0xa8302000),
	  SPH_C32(0x19b70000), SPH_C32(0x5f9b0c57), SPH_C32(0x0cf1fcdb),
	  SPH_C32(0x908b2232), SPH_C32(0xfe716097), SPH_C32(0x45510000),
	  SPH_C32(0x70b60001), SPH_C32(0xc48f4000), SPH_C32(0xef300000),
	  SPH_C32(0xec8a2380), SPH_C32(0x5c931767), SPH_C32(0xc96b3d13),
	  SPH_C32(0xfbdbf461) },
	{ SPH_C32(0x43120000), SPH_C32(0xe8df0002), SPH_C32(0xc5c82180),
	  SPH_C32(0x6e930000), SPH_C32(0xb3dc0bc9), SPH_C32(0xf8519595),
	  SPH_C32(0x5d283a20), SPH_C32(0x66db29f9), SPH_C32(0xf7570000),
	  SPH_C32(0xb5df0001), SPH_C32(0xec8c5200), SPH_C32(0x9b570000),
	  SPH_C32(0x5a481574), SPH_C32(0xb7812e9f), SPH_C32(0xfabae2ff),
	  SPH_C32(0xf295c5f9) },
	{ SPH_C32(0xf39f0000), SPH_C32(0xaa5f0003), SPH_C32(0xd1e1c000),
	  SPH_C32(0x35a20000), SPH_C32(0x1a5779e4), SPH_C32(0x6aa14bed),
	  SPH_C32(0x3b19d5bd), SPH_C32(0x5d6336ec), SPH_C32(0x9e740000),
	  SPH_C32(0x799f0001), SPH_C32(0x8d258000), SPH_C32(0x6ed10000),
	  SPH_C32(0x267448d9), SPH_C32(0x1eea2356), SPH_C32(0x8a3d5665),
	  SPH_C32(0x13b74e4f) },
	{ SPH_C32(0xedd10000), SPH_C32(0x74900003), SPH_C32(0xbc19c180),
	  SPH_C32(0x42860000), SPH_C32(0xf6107e7a), SPH_C32(0x9e0122a3),
	  SPH_C32(0xf6bacdaf), SPH_C32(0xc5c97f82), SPH_C32(0x2c720000),
	  SPH_C32(0xbcf60001), SPH_C32(0xa5269200), SPH_C32(0x1ab60000),
	  SPH_C32(0x90b67e2d), SPH_C32(0xf5f81aae), SPH_C32(0xb9ec8989),
	  SPH_C32(0x1af97fd7) },
	{ SPH_C32(0x30b70000), SPH_C32(0xe5d00000), SPH_C32(0xf4f46000),
	  SPH_C32(0x42c40000), SPH_C32(0x63b83d6a), SPH_C32(0x78ba9460),
	  SPH_C32(0x21afa1ea), SPH_C32(0xb0a51834), SPH_C32(0xb6ce0000),
	  SPH_C32(0xdae90002), SPH_C32(0x156e8000), SPH_C32(0xda920000),
	  SPH_C32(0xf6dd5a64), SPH_C32(0x36325c8a), SPH_C32(0xf272e8ae),
	  SPH_C32(0xa6b8c28d) },
	{ SPH_C32(0x2ef90000), SPH_C32(0x3b1f0000), SPH_C32(0x990c6180),
	  SPH_C32(0x35e00000), SPH_C32(0x8fff3af4), SPH_C32(0x8c1afd2e),
	  SPH_C32(0xec0cb9f8), SPH_C32(0x280f515a), SPH_C32(0x04c80000),
	  SPH_C32(0x1f800002), SPH_C32(0x3d6d9200), SPH_C32(0xaef50000),
	  SPH_C32(0x401f6c90), SPH_C32(0xdd206572), SPH_C32(0xc1a33742),
	  SPH_C32(0xaff6f315) },
	{ SPH_C32(0x9e740000), SPH_C32(0x799f0001), SPH_C32(0x8d258000),
	  SPH_C32(0x6ed10000), SPH_C32(0x267448d9), SPH_C32(0x1eea2356),
	  SPH_C32(0x8a3d5665), SPH_C32(0x13b74e4f), SPH_C32(0x6deb0000),
	  SPH_C32(0xd3c00002), SPH_C32(0x5cc44000), SPH_C32(0x5b730000),
	  SPH_C32(0x3c23313d), SPH_C32(0x744b68bb), SPH_C32(0xb12483d8),
	  SPH_C32(0x4ed478a3) },
	{ SPH_C32(0x803a0000), SPH_C32(0xa7500001), SPH_C32(0xe0dd8180),
	  SPH_C32(0x19f50000), SPH_C32(0xca334f47), SPH_C32(0xea4a4a18),
	  SPH_C32(0x479e4e77), SPH_C32(0x8b1d0721), SPH_C32(0xdfed0000),
	  SPH_C32(0x16a90002), SPH_C32(0x74c75200), SPH_C32(0x2f140000),
	  SPH_C32(0x8ae107c9), SPH_C32(0x9f595143), SPH_C32(0x82f55c34),
	  SPH_C32(0x479a493b) },
	{ SPH_C32(0xeb920000), SPH_C32(0xecf90000), SPH_C32(0xbd5ea000),
	  SPH_C32(0xc3250000), SPH_C32(0xa9465633), SPH_C32(0x3ac3a051),
	  SPH_C32(0x62f9ca9c), SPH_C32(0x58c9a21a), SPH_C32(0xc3280000),
	  SPH_C32(0x4f8f0003), SPH_C32(0x2515a000), SPH_C32(0x77660000),
	  SPH_C32(0x79ef448e), SPH_C32(0x121bdf8d), SPH_C32(0x1ab67457),
	  SPH_C32(0xedc62ed8) },
	{ SPH_C32(0xf5dc0000), SPH_C32(0x32360000), SPH_C32(0xd0a6a180),
	  SPH_C32(0xb4010000), SPH_C32(0x450151ad), SPH_C32(0xce63c91f),
	  SPH_C32(0xaf5ad28e), SPH_C32(0xc063eb74), SPH_C32(0x712e0000),
	  SPH_C32(0x8ae60003), SPH_C32(0x0d16b200), SPH_C32(0x03010000),
	  SPH_C32(0xcf2d727a), SPH_C32(0xf909e675), SPH_C32(0x2967abbb),
	  SPH_C32(0xe4881f40) },
	{ SPH_C32(0x45510000), SPH_C32(0x70b60001), SPH_C32(0xc48f4000),
	  SPH_C32(0xef300000), SPH_C32(0xec8a2380), SPH_C32(0x5c931767),
	  SPH_C32(0xc96b3d13), SPH_C32(0xfbdbf461), SPH_C32(0x180d0000),
	  SPH_C32(0x46a60003), SPH_C32(0x6cbf6000), SPH_C32(0xf6870000),
	  SPH_C32(0xb3112fd7), SPH_C32(0x5062ebbc), SPH_C32(0x59e01f21),
	  SPH_C32(0x05aa94f6) },
	{ SPH_C32(0x5b1f0000), SPH_C32(0xae790001), SPH_C32(0xa9774180),
	  SPH_C32(0x98140000), SPH_C32(0x00cd241e), SPH_C32(0xa8337e29),
	  SPH_C32(0x04c82501), SPH_C32(0x6371bd0f), SPH_C32(0xaa0b0000),
	  SPH_C32(0x83cf0003), SPH_C32(0x44bc7200), SPH_C32(0x82e00000),
	  SPH_C32(0x05d31923), SPH_C32(0xbb70d244), SPH_C32(0x6a31c0cd),
	  SPH_C32(0x0ce4a56e) },
	{ SPH_C32(0xb6ce0000), SPH_C32(0xdae90002), SPH_C32(0x156e8000),
	  SPH_C32(0xda920000), SPH_C32(0xf6dd5a64), SPH_C32(0x36325c8a),
	  SPH_C32(0xf272e8ae), SPH_C32(0xa6b8c28d), SPH_C32(0x86790000),
	  SPH_C32(0x3f390002), SPH_C32(0xe19ae000), SPH_C32(0x98560000),
	  SPH_C32(0x9565670e), SPH_C32(0x4e88c8ea), SPH_C32(0xd3dd4944),
	  SPH_C32(0x161ddab9) },
	{ SPH_C32(0xa8800000), SPH_C32(0x04260002), SPH_C32(0x78968180),
	  SPH_C32(0xadb60000), SPH_C32(0x1a9a5dfa), SPH_C32(0xc29235c4),
	  SPH_C32(0x3fd1f0bc), SPH_C32(0x3e128be3), SPH_C32(0x347f0000),
	  SPH_C32(0xfa500002), SPH_C32(0xc999f200), SPH_C32(0xec310000),
	  SPH_C32(0x23a751fa), SPH_C32(0xa59af112), SPH_C32(0xe00c96a8),
	  SPH_C32(0x1f53eb21) },
	{ SPH_C32(0x180d0000), SPH_C32(0x46a60003), SPH_C32(0x6cbf6000),
	  SPH_C32(0xf6870000), SPH_C32(0xb3112fd7), SPH_C32(0x5062ebbc),
	  SPH_C32(0x59e01f21), SPH_C32(0x05aa94f6), SPH_C32(0x5d5c0000),
	  SPH_C32(0x36100002), SPH_C32(0xa8302000), SPH_C32(0x19b70000),
	  SPH_C32(0x5f9b0c57), SPH_C32(0x0cf1fcdb), SPH_C32(0x908b2232),
	  SPH_C32(0xfe716097) },
	{ SPH_C32(0x06430000), SPH_C32(0x98690003), SPH_C32(0x01476180),
	  SPH_C32(0x81a30000), SPH_C32(0x5f562849), SPH_C32(0xa4c282f2),
	  SPH_C32(0x94430733), SPH_C32(0x9d00dd98), SPH_C32(0xef5a0000),
	  SPH_C32(0xf3790002), SPH_C32(0x80333200), SPH_C32(0x6dd00000),
	  SPH_C32(0xe9593aa3), SPH_C32(0xe7e3c523), SPH_C32(0xa35afdde),
	  SPH_C32(0xf73f510f) },
	{ SPH_C32(0x6deb0000), SPH_C32(0xd3c00002), SPH_C32(0x5cc44000),
	  SPH_C32(0x5b730000), SPH_C32(0x3c23313d), SPH_C32(0x744b68bb),
	  SPH_C32(0xb12483d8), SPH_C32(0x4ed478a3), SPH_C32(0xf39f0000),
	  SPH_C32(0xaa5f0003), SPH_C32(0xd1e1c000), SPH_C32(0x35a20000),
	  SPH_C32(0x1a5779e4), SPH_C32(0x6aa14bed), SPH_C32(0x3b19d5bd),
	  SPH_C32(0x5d6336ec) },
	{ SPH_C32(0x73a50000), SPH_C32(0x0d0f0002), SPH_C32(0x313c4180),
	  SPH_C32(0x2c570000), SPH_C32(0xd06436a3), SPH_C32(0x80eb01f5),
	  SPH_C32(0x7c879bca), SPH_C32(0xd67e31cd), SPH_C32(0x41990000),
	  SPH_C32(0x6f360003), SPH_C32(0xf9e2d200), SPH_C32(0x41c50000),
	  SPH_C32(0xac954f10), SPH_C32(0x81b37215), SPH_C32(0x08c80a51),
	  SPH_C32(0x542d0774) },
	{ SPH_C32(0xc3280000), SPH_C32(0x4f8f0003), SPH_C32(0x2515a000),
	  SPH_C32(0x77660000), SPH_C32(0x79ef448e), SPH_C32(0x121bdf8d),
	  SPH_C32(0x1ab67457), SPH_C32(0xedc62ed8), SPH_C32(0x28ba0000),
	  SPH_C32(0xa3760003), SPH_C32(0x984b0000), SPH_C32(0xb4430000),
	  SPH_C32(0xd0a912bd), SPH_C32(0x28d87fdc), SPH_C32(0x784fbecb),
	  SPH_C32(0xb50f8cc2) },
	{ SPH_C32(0xdd660000), SPH_C32(0x91400003), SPH_C32(0x48eda180),
	  SPH_C32(0x00420000), SPH_C32(0x95a84310), SPH_C32(0xe6bbb6c3),
	  SPH_C32(0xd7156c45), SPH_C32(0x756c67b6), SPH_C32(0x9abc0000),
	  SPH_C32(0x661f0003), SPH_C32(0xb0481200), SPH_C32(0xc0240000),
	  SPH_C32(0x666b2449), SPH_C32(0xc3ca4624), SPH_C32(0x4b9e6127),
	  SPH_C32(0xbc41bd5a) }
};

static const sph_u32 T512_35[32][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xcc140000), SPH_C32(0xa5630000), SPH_C32(0x5ab90780),
	  SPH_C32(0x3b500000), SPH_C32(0x4bd013ff), SPH_C32(0x879b3418),
	  SPH_C32(0x694348c1), SPH_C32(0xca5a87fe), SPH_C32(0x819e0000),
	  SPH_C32(0xec570000), SPH_C32(0x66320280), SPH_C32(0x95f30000),
	  SPH_C32(0x5da92802), SPH_C32(0x48f43cbc), SPH_C32(0xe65aa22d),
	  SPH_C32(0x8e67b7fa) },
	{ SPH_C32(0x819e0000), SPH_C32(0xec570000), SPH_C32(0x66320280),
	  SPH_C32(0x95f30000), SPH_C32(0x5da92802), SPH_C32(0x48f43cbc),
	  SPH_C32(0xe65aa22d), SPH_C32(0x8e67b7fa), SPH_C32(0x4d8a0000),
	  SPH_C32(0x49340000), SPH_C32(0x3c8b0500), SPH_C32(0xaea30000),
	  SPH_C32(0x16793bfd), SPH_C32(0xcf6f08a4), SPH_C32(0x8f19eaec),
	  SPH_C32(0x443d3004) },
	{ SPH_C32(0x4d8a0000), SPH_C32(0x49340000), SPH_C32(0x3c8b0500),
	  SPH_C32(0xaea30000), SPH_C32(0x16793bfd), SPH_C32(0xcf6f08a4),
	  SPH_C32(0x8f19eaec), SPH_C32(0x443d3004), SPH_C32(0xcc140000),
	  SPH_C32(0xa5630000), SPH_C32(0x5ab90780), SPH_C32(0x3b500000),
	  SPH_C32(0x4bd013ff), SPH_C32(0x879b3418), SPH_C32(0x694348c1),
	  SPH_C32(0xca5a87fe) },
	{ SPH_C32(0x78230000), SPH_C32(0x12fc0000), SPH_C32(0xa93a0b80),
	  SPH_C32(0x90a50000), SPH_C32(0x713e2879), SPH_C32(0x7ee98924),
	  SPH_C32(0xf08ca062), SPH_C32(0x636f8bab), SPH_C32(0x02af0000),
	  SPH_C32(0xb7280000), SPH_C32(0xba1c0300), SPH_C32(0x56980000),
	  SPH_C32(0xba8d45d3), SPH_C32(0x8048c667), SPH_C32(0xa95c149a),
	  SPH_C32(0xf4f6ea7b) },
	{ SPH_C32(0xb4370000), SPH_C32(0xb79f0000), SPH_C32(0xf3830c00),
	  SPH_C32(0xabf50000), SPH_C32(0x3aee3b86), SPH_C32(0xf972bd3c),
	  SPH_C32(0x99cfe8a3), SPH_C32(0xa9350c55), SPH_C32(0x83310000),
	  SPH_C32(0x5b7f0000), SPH_C32(0xdc2e0180), SPH_C32(0xc36b0000),
	  SPH_C32(0xe7246dd1), SPH_C32(0xc8bcfadb), SPH_C32(0x4f06b6b7),
	  SPH_C32(0x7a915d81) },
	{ SPH_C32(0xf9bd0000), SPH_C32(0xfeab0000), SPH_C32(0xcf080900),
	  SPH_C32(0x05560000), SPH_C32(0x2c97007b), SPH_C32(0x361db598),
	  SPH_C32(0x16d6024f), SPH_C32(0xed083c51), SPH_C32(0x4f250000),
	  SPH_C32(0xfe1c0000), SPH_C32(0x86970600), SPH_C32(0xf83b0000),
	  SPH_C32(0xacf47e2e), SPH_C32(0x4f27cec3), SPH_C32(0x2645fe76),
	  SPH_C32(0xb0cbda7f) },
	{ SPH_C32(0x35a90000), SPH_C32(0x5bc80000), SPH_C32(0x95b10e80),
	  SPH_C32(0x3e060000), SPH_C32(0x67471384), SPH_C32(0xb1868180),
	  SPH_C32(0x7f954a8e), SPH_C32(0x2752bbaf), SPH_C32(0xcebb0000),
	  SPH_C32(0x124b0000), SPH_C32(0xe0a50480), SPH_C32(0x6dc80000),
	  SPH_C32(0xf15d562c), SPH_C32(0x07d3f27f), SPH_C32(0xc01f5c5b),
	  SPH_C32(0x3eac6d85) },
	{ SPH_C32(0x02af0000), SPH_C32(0xb7280000), SPH_C32(0xba1c0300),
	  SPH_C32(0x56980000), SPH_C32(0xba8d45d3), SPH_C32(0x8048c667),
	  SPH_C32(0xa95c149a), SPH_C32(0xf4f6ea7b), SPH_C32(0x7a8c0000),
	  SPH_C32(0xa5d40000), SPH_C32(0x13260880), SPH_C32(0xc63d0000),
	  SPH_C32(0xcbb36daa), SPH_C32(0xfea14f43), SPH_C32(0x59d0b4f8),
	  SPH_C32(0x979961d0) },
	{ SPH_C32(0xcebb0000), SPH_C32(0x124b0000), SPH_C32(0xe0a50480),
	  SPH_C32(0x6dc80000), SPH_C32(0xf15d562c), SPH_C32(0x07d3f27f),
	  SPH_C32(0xc01f5c5b), SPH_C32(0x3eac6d85), SPH_C32(0xfb120000),
	  SPH_C32(0x49830000), SPH_C32(0x75140a00), SPH_C32(0x53ce0000),
	  SPH_C32(0x961a45a8), SPH_C32(0xb65573ff), SPH_C32(0xbf8a16d5),
	  SPH_C32(0x19fed62a) },
	{ SPH_C32(0x83310000), SPH_C32(0x5b7f0000), SPH_C32(0xdc2e0180),
	  SPH_C32(0xc36b0000), SPH_C32(0xe7246dd1), SPH_C32(0xc8bcfadb),
	  SPH_C32(0x4f06b6b7), SPH_C32(0x7a915d81), SPH_C32(0x37060000),
	  SPH_C32(0xece00000), SPH_C32(0x2fad0d80), SPH_C32(0x689e0000),
	  SPH_C32(0xddca5657), SPH_C32(0x31ce47e7), SPH_C32(0xd6c95e14),
	  SPH_C32(0xd3a451d4) },
	{ SPH_C32(0x4f250000), SPH_C32(0xfe1c0000), SPH_C32(0x86970600),
	  SPH_C32(0xf83b0000), SPH_C32(0xacf47e2e), SPH_C32(0x4f27cec3),
	  SPH_C32(0x2645fe76), SPH_C32(0xb0cbda7f), SPH_C32(0xb6980000),
	  SPH_C32(0x00b70000), SPH_C32(0x499f0f00), SPH_C32(0xfd6d0000),
	  SPH_C32(0x80637e55), SPH_C32(0x793a7b5b), SPH_C32(0x3093fc39),
	  SPH_C32(0x5dc3e62e) },
	{ SPH_C32(0x7a8c0000), SPH_C32(0xa5d40000), SPH_C32(0x13260880),
	  SPH_C32(0xc63d0000), SPH_C32(0xcbb36daa), SPH_C32(0xfea14f43),
	  SPH_C32(0x59d0b4f8), SPH_C32(0x979961d0), SPH_C32(0x78230000),
	  SPH_C32(0x12fc0000), SPH_C32(0xa93a0b80), SPH_C32(0x90a50000),
	  SPH_C32(0x713e2879), SPH_C32(0x7ee98924), SPH_C32(0xf08ca062),
	  SPH_C32(0x636f8bab) },
	{ SPH_C32(0xb6980000), SPH_C32(0x00b70000), SPH_C32(0x499f0f00),
	  SPH_C32(0xfd6d0000), SPH_C32(0x80637e55), SPH_C32(0x793a7b5b),
	  SPH_C32(0x3093fc39), SPH_C32(0x5dc3e62e), SPH_C32(0xf9bd0000),
	  SPH_C32(0xfeab0000), SPH_C32(0xcf080900), SPH_C32(0x05560000),
	  SPH_C32(0x2c97007b), SPH_C32(0x361db598), SPH_C32(0x16d6024f),
	  SPH_C32(0xed083c51) },
	{ SPH_C32(0xfb120000), SPH_C32(0x49830000), SPH_C32(0x75140a00),
	  SPH_C32(0x53ce0000), SPH_C32(0x961a45a8), SPH_C32(0xb65573ff),
	  SPH_C32(0xbf8a16d5), SPH_C32(0x19fed62a), SPH_C32(0x35a90000),
	  SPH_C32(0x5bc80000), SPH_C32(0x95b10e80), SPH_C32(0x3e060000),
	  SPH_C32(0x67471384), SPH_C32(0xb1868180), SPH_C32(0x7f954a8e),
	  SPH_C32(0x2752bbaf) },
	{ SPH_C32(0x37060000), SPH_C32(0xece00000), SPH_C32(0x2fad0d80),
	  SPH_C32(0x689e0000), SPH_C32(0xddca5657), SPH_C32(0x31ce47e7),
	  SPH_C32(0xd6c95e14), SPH_C32(0xd3a451d4), SPH_C32(0xb4370000),
	  SPH_C32(0xb79f0000), SPH_C32(0xf3830c00), SPH_C32(0xabf50000),
	  SPH_C32(0x3aee3b86), SPH_C32(0xf972bd3c), SPH_C32(0x99cfe8a3),
	  SPH_C32(0xa9350c55) },
	{ SPH_C32(0xac480000), SPH_C32(0x1ba60000), SPH_C32(0x45fb1380),
	  SPH_C32(0x03430000), SPH_C32(0x5a85316a), SPH_C32(0x1fb250b6),
	  SPH_C32(0xfe72c7fe), SPH_C32(0x91e478f6), SPH_C32(0x1e4e0000),
	  SPH_C32(0xdecf0000), SPH_C32(0x6df80180), SPH_C32(0x77240000),
	  SPH_C32(0xec47079e), SPH_C32(0xf4a0694e), SPH_C32(0xcda31812),
	  SPH_C32(0x98aa496e) },
	{ SPH_C32(0x605c0000), SPH_C32(0xbec50000), SPH_C32(0x1f421400),
	  SPH_C32(0x38130000), SPH_C32(0x11552295), SPH_C32(0x982964ae),
	  SPH_C32(0x97318f3f), SPH_C32(0x5bbeff08), SPH_C32(0x9fd00000),
	  SPH_C32(0x32980000), SPH_C32(0x0bca0300), SPH_C32(0xe2d70000),
	  SPH_C32(0xb1ee2f9c), SPH_C32(0xbc5455f2), SPH_C32(0x2bf9ba3f),
	  SPH_C32(0x16cdfe94) },
	{ SPH_C32(0x2dd60000), SPH_C32(0xf7f10000), SPH_C32(0x23c91100),
	  SPH_C32(0x96b00000), SPH_C32(0x072c1968), SPH_C32(0x57466c0a),
	  SPH_C32(0x182865d3), SPH_C32(0x1f83cf0c), SPH_C32(0x53c40000),
	  SPH_C32(0x97fb0000), SPH_C32(0x51730480), SPH_C32(0xd9870000),
	  SPH_C32(0xfa3e3c63), SPH_C32(0x3bcf61ea), SPH_C32(0x42baf2fe),
	  SPH_C32(0xdc97796a) },
	{ SPH_C32(0xe1c20000), SPH_C32(0x52920000), SPH_C32(0x79701680),
	  SPH_C32(0xade00000), SPH_C32(0x4cfc0a97), SPH_C32(0xd0dd5812),
	  SPH_C32(0x716b2d12), SPH_C32(0xd5d948f2), SPH_C32(0xd25a0000),
	  SPH_C32(0x7bac0000), SPH_C32(0x37410600), SPH_C32(0x4c740000),
	  SPH_C32(0xa7971461), SPH_C32(0x733b5d56), SPH_C32(0xa4e050d3),
	  SPH_C32(0x52f0ce90) },
	{ SPH_C32(0xd46b0000), SPH_C32(0x095a0000), SPH_C32(0xecc11800),
	  SPH_C32(0x93e60000), SPH_C32(0x2bbb1913), SPH_C32(0x615bd992),
	  SPH_C32(0x0efe679c), SPH_C32(0xf28bf35d), SPH_C32(0x1ce10000),
	  SPH_C32(0x69e70000), SPH_C32(0xd7e40280), SPH_C32(0x21bc0000),
	  SPH_C32(0x56ca424d), SPH_C32(0x74e8af29), SPH_C32(0x64ff0c88),
	  SPH_C32(0x6c5ca315) },
	{ SPH_C32(0x187f0000), SPH_C32(0xac390000), SPH_C32(0xb6781f80),
	  SPH_C32(0xa8b60000), SPH_C32(0x606b0aec), SPH_C32(0xe6c0ed8a),
	  SPH_C32(0x67bd2f5d), SPH_C32(0x38d174a3), SPH_C32(0x9d7f0000),
	  SPH_C32(0x85b00000), SPH_C32(0xb1d60000), SPH_C32(0xb44f0000),
	  SPH_C32(0x0b636a4f), SPH_C32(0x3c1c9395), SPH_C32(0x82a5aea5),
	  SPH_C32(0xe23b14ef) },
	{ SPH_C32(0x55f50000), SPH_C32(0xe50d0000), SPH_C32(0x8af31a80),
	  SPH_C32(0x06150000), SPH_C32(0x76123111), SPH_C32(0x29afe52e),
	  SPH_C32(0xe8a4c5b1), SPH_C32(0x7cec44a7), SPH_C32(0x516b0000),
	  SPH_C32(0x20d30000), SPH_C32(0xeb6f0780), SPH_C32(0x8f1f0000),
	  SPH_C32(0x40b379b0), SPH_C32(0xbb87a78d), SPH_C32(0xebe6e664),
	  SPH_C32(0x28619311) },
	{ SPH_C32(0x99e10000), SPH_C32(0x406e0000), SPH_C32(0xd04a1d00),
	  SPH_C32(0x3d450000), SPH_C32(0x3dc222ee), SPH_C32(0xae34d136),
	  SPH_C32(0x81e78d70), SPH_C32(0xb6b6c359), SPH_C32(0xd0f50000),
	  SPH_C32(0xcc840000), SPH_C32(0x8d5d0500), SPH_C32(0x1aec0000),
	  SPH_C32(0x1d1a51b2), SPH_C32(0xf3739b31), SPH_C32(0x0dbc4449),
	  SPH_C32(0xa60624eb) },
	{ SPH_C32(0xaee70000), SPH_C32(0xac8e0000), SPH_C32(0xffe71080),
	  SPH_C32(0x55db0000), SPH_C32(0xe00874b9), SPH_C32(0x9ffa96d1),
	  SPH_C32(0x572ed364), SPH_C32(0x6512928d), SPH_C32(0x64c20000),
	  SPH_C32(0x7b1b0000), SPH_C32(0x7ede0900), SPH_C32(0xb1190000),
	  SPH_C32(0x27f46a34), SPH_C32(0x0a01260d), SPH_C32(0x9473acea),
	  SPH_C32(0x0f3328be) },
	{ SPH_C32(0x62f30000), SPH_C32(0x09ed0000), SPH_C32(0xa55e1700),
	  SPH_C32(0x6e8b0000), SPH_C32(0xabd86746), SPH_C32(0x1861a2c9),
	  SPH_C32(0x3e6d9ba5), SPH_C32(0xaf481573), SPH_C32(0xe55c0000),
	  SPH_C32(0x974c0000), SPH_C32(0x18ec0b80), SPH_C32(0x24ea0000),
	  SPH_C32(0x7a5d4236), SPH_C32(0x42f51ab1), SPH_C32(0x72290ec7),
	  SPH_C32(0x81549f44) },
	{ SPH_C32(0x2f790000), SPH_C32(0x40d90000), SPH_C32(0x99d51200),
	  SPH_C32(0xc0280000), SPH_C32(0xbda15cbb), SPH_C32(0xd70eaa6d),
	  SPH_C32(0xb1747149), SPH_C32(0xeb752577), SPH_C32(0x29480000),
	  SPH_C32(0x322f0000), SPH_C32(0x42550c00), SPH_C32(0x1fba0000),
	  SPH_C32(0x318d51c9), SPH_C32(0xc56e2ea9), SPH_C32(0x1b6a4606),
	  SPH_C32(0x4b0e18ba) },
	{ SPH_C32(0xe36d0000), SPH_C32(0xe5ba0000), SPH_C32(0xc36c1580),
	  SPH_C32(0xfb780000), SPH_C32(0xf6714f44), SPH_C32(0x50959e75),
	  SPH_C32(0xd8373988), SPH_C32(0x212fa289), SPH_C32(0xa8d60000),
	  SPH_C32(0xde780000), SPH_C32(0x24670e80), SPH_C32(0x8a490000),
	  SPH_C32(0x6c2479cb), SPH_C32(0x8d9a1215), SPH_C32(0xfd30e42b),
	  SPH_C32(0xc569af40) },
	{ SPH_C32(0xd6c40000), SPH_C32(0xbe720000), SPH_C32(0x56dd1b00),
	  SPH_C32(0xc57e0000), SPH_C32(0x91365cc0), SPH_C32(0xe1131ff5),
	  SPH_C32(0xa7a27306), SPH_C32(0x067d1926), SPH_C32(0x666d0000),
	  SPH_C32(0xcc330000), SPH_C32(0xc4c20a00), SPH_C32(0xe7810000),
	  SPH_C32(0x9d792fe7), SPH_C32(0x8a49e06a), SPH_C32(0x3d2fb870),
	  SPH_C32(0xfbc5c2c5) },
	{ SPH_C32(0x1ad00000), SPH_C32(0x1b110000), SPH_C32(0x0c641c80),
	  SPH_C32(0xfe2e0000), SPH_C32(0xdae64f3f), SPH_C32(0x66882bed),
	  SPH_C32(0xcee13bc7), SPH_C32(0xcc279ed8), SPH_C32(0xe7f30000),
	  SPH_C32(0x20640000), SPH_C32(0xa2f00880), SPH_C32(0x72720000),
	  SPH_C32(0xc0d007e5), SPH_C32(0xc2bddcd6), SPH_C32(0xdb751a5d),
	  SPH_C32(0x75a2753f) },
	{ SPH_C32(0x575a0000), SPH_C32(0x52250000), SPH_C32(0x30ef1980),
	  SPH_C32(0x508d0000), SPH_C32(0xcc9f74c2), SPH_C32(0xa9e72349),
	  SPH_C32(0x41f8d12b), SPH_C32(0x881aaedc), SPH_C32(0x2be70000),
	  SPH_C32(0x85070000), SPH_C32(0xf8490f00), SPH_C32(0x49220000),
	  SPH_C32(0x8b00141a), SPH_C32(0x4526e8ce), SPH_C32(0xb236529c),
	  SPH_C32(0xbff8f2c1) },
	{ SPH_C32(0x9b4e0000), SPH_C32(0xf7460000), SPH_C32(0x6a561e00),
	  SPH_C32(0x6bdd0000), SPH_C32(0x874f673d), SPH_C32(0x2e7c1751),
	  SPH_C32(0x28bb99ea), SPH_C32(0x42402922), SPH_C32(0xaa790000),
	  SPH_C32(0x69500000), SPH_C32(0x9e7b0d80), SPH_C32(0xdcd10000),
	  SPH_C32(0xd6a93c18), SPH_C32(0x0dd2d472), SPH_C32(0x546cf0b1),
	  SPH_C32(0x319f453b) }
};

static const sph_u32 T512_40[32][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x07ed0000), SPH_C32(0xb2500000), SPH_C32(0x8774000a),
	  SPH_C32(0x970d0000), SPH_C32(0x437223ae), SPH_C32(0x48c76ea4),
	  SPH_C32(0xf4786222), SPH_C32(0x9075b1ce), SPH_C32(0xa2d60000),
	  SPH_C32(0xa6760000), SPH_C32(0xc9440014), SPH_C32(0xeba30000),
	  SPH_C32(0xccec2e7b), SPH_C32(0x3018c499), SPH_C32(0x03490afa),
	  SPH_C32(0x9b6ef888) },
	{ SPH_C32(0x88980000), SPH_C32(0x1f940000), SPH_C32(0x7fcf002e),
	  SPH_C32(0xfb4e0000), SPH_C32(0xf158079a), SPH_C32(0x61ae9167),
	  SPH_C32(0xa895706c), SPH_C32(0xe6107494), SPH_C32(0x0bc20000),
	  SPH_C32(0xdb630000), SPH_C32(0x7e88000c), SPH_C32(0x15860000),
	  SPH_C32(0x91fd48f3), SPH_C32(0x7581bb43), SPH_C32(0xf460449e),
	  SPH_C32(0xd8b61463) },
	{ SPH_C32(0x8f750000), SPH_C32(0xadc40000), SPH_C32(0xf8bb0024),
	  SPH_C32(0x6c430000), SPH_C32(0xb22a2434), SPH_C32(0x2969ffc3),
	  SPH_C32(0x5ced124e), SPH_C32(0x7665c55a), SPH_C32(0xa9140000),
	  SPH_C32(0x7d150000), SPH_C32(0xb7cc0018), SPH_C32(0xfe250000),
	  SPH_C32(0x5d116688), SPH_C32(0x45997fda), SPH_C32(0xf7294e64),
	  SPH_C32(0x43d8eceb) },
	{ SPH_C32(0x0bc20000), SPH_C32(0xdb630000), SPH_C32(0x7e88000c),
	  SPH_C32(0x15860000), SPH_C32(0x91fd48f3), SPH_C32(0x7581bb43),
	  SPH_C32(0xf460449e), SPH_C32(0xd8b61463), SPH_C32(0x835a0000),
	  SPH_C32(0xc4f70000), SPH_C32(0x01470022), SPH_C32(0xeec80000),
	  SPH_C32(0x60a54f69), SPH_C32(0x142f2a24), SPH_C32(0x5cf534f2),
	  SPH_C32(0x3ea660f7) },
	{ SPH_C32(0x0c2f0000), SPH_C32(0x69330000), SPH_C32(0xf9fc0006),
	  SPH_C32(0x828b0000), SPH_C32(0xd28f6b5d), SPH_C32(0x3d46d5e7),
	  SPH_C32(0x001826bc), SPH_C32(0x48c3a5ad), SPH_C32(0x218c0000),
	  SPH_C32(0x62810000), SPH_C32(0xc8030036), SPH_C32(0x056b0000),
	  SPH_C32(0xac496112), SPH_C32(0x2437eebd), SPH_C32(0x5fbc3e08),
	  SPH_C32(0xa5c8987f) },
	{ SPH_C32(0x835a0000), SPH_C32(0xc4f70000), SPH_C32(0x01470022),
	  SPH_C32(0xeec80000), SPH_C32(0x60a54f69), SPH_C32(0x142f2a24),
	  SPH_C32(0x5cf534f2), SPH_C32(0x3ea660f7), SPH_C32(0x88980000),
	  SPH_C32(0x1f940000), SPH_C32(0x7fcf002e), SPH_C32(0xfb4e0000),
	  SPH_C32(0xf158079a), SPH_C32(0x61ae9167), SPH_C32(0xa895706c),
	  SPH_C32(0xe6107494) },
	{ SPH_C32(0x84b70000), SPH_C32(0x76a70000), SPH_C32(0x86330028),
	  SPH_C32(0x79c50000), SPH_C32(0x23d76cc7), SPH_C32(0x5ce84480),
	  SPH_C32(0xa88d56d0), SPH_C32(0xaed3d139), SPH_C32(0x2a4e0000),
	  SPH_C32(0xb9e20000), SPH_C32(0xb68b003a), SPH_C32(0x10ed0000),
	  SPH_C32(0x3db429e1), SPH_C32(0x51b655fe), SPH_C32(0xabdc7a96),
	  SPH_C32(0x7d7e8c1c) },
	{ SPH_C32(0x52500000), SPH_C32(0x29540000), SPH_C32(0x6a61004e),
	  SPH_C32(0xf0ff0000), SPH_C32(0x9a317eec), SPH_C32(0x452341ce),
	  SPH_C32(0xcf568fe5), SPH_C32(0x5303130f), SPH_C32(0x538d0000),
	  SPH_C32(0xa9fc0000), SPH_C32(0x9ef70006), SPH_C32(0x56ff0000),
	  SPH_C32(0x0ae4004e), SPH_C32(0x92c5cdf9), SPH_C32(0xa9444018),
	  SPH_C32(0x7f975691) },
	{ SPH_C32(0x55bd0000), SPH_C32(0x9b040000), SPH_C32(0xed150044),
	  SPH_C32(0x67f20000), SPH_C32(0xd9435d42), SPH_C32(0x0de42f6a),
	  SPH_C32(0x3b2eedc7), SPH_C32(0xc376a2c1), SPH_C32(0xf15b0000),
	  SPH_C32(0x0f8a0000), SPH_C32(0x57b30012), SPH_C32(0xbd5c0000),
	  SPH_C32(0xc6082e35), SPH_C32(0xa2dd0960), SPH_C32(0xaa0d4ae2),
	  SPH_C32(0xe4f9ae19) },
	{ SPH_C32(0xdac80000), SPH_C32(0x36c00000), SPH_C32(0x15ae0060),
	  SPH_C32(0x0bb10000), SPH_C32(0x6b697976), SPH_C32(0x248dd0a9),
	  SPH_C32(0x67c3ff89), SPH_C32(0xb513679b), SPH_C32(0x584f0000),
	  SPH_C32(0x729f0000), SPH_C32(0xe07f000a), SPH_C32(0x43790000),
	  SPH_C32(0x9b1948bd), SPH_C32(0xe74476ba), SPH_C32(0x5d240486),
	  SPH_C32(0xa72142f2) },
	{ SPH_C32(0xdd250000), SPH_C32(0x84900000), SPH_C32(0x92da006a),
	  SPH_C32(0x9cbc0000), SPH_C32(0x281b5ad8), SPH_C32(0x6c4abe0d),
	  SPH_C32(0x93bb9dab), SPH_C32(0x2566d655), SPH_C32(0xfa990000),
	  SPH_C32(0xd4e90000), SPH_C32(0x293b001e), SPH_C32(0xa8da0000),
	  SPH_C32(0x57f566c6), SPH_C32(0xd75cb223), SPH_C32(0x5e6d0e7c),
	  SPH_C32(0x3c4fba7a) },
	{ SPH_C32(0x59920000), SPH_C32(0xf2370000), SPH_C32(0x14e90042),
	  SPH_C32(0xe5790000), SPH_C32(0x0bcc361f), SPH_C32(0x30a2fa8d),
	  SPH_C32(0x3b36cb7b), SPH_C32(0x8bb5076c), SPH_C32(0xd0d70000),
	  SPH_C32(0x6d0b0000), SPH_C32(0x9fb00024), SPH_C32(0xb8370000),
	  SPH_C32(0x6a414f27), SPH_C32(0x86eae7dd), SPH_C32(0xf5b174ea),
	  SPH_C32(0x41313666) },
	{ SPH_C32(0x5e7f0000), SPH_C32(0x40670000), SPH_C32(0x939d0048),
	  SPH_C32(0x72740000), SPH_C32(0x48be15b1), SPH_C32(0x78659429),
	  SPH_C32(0xcf4ea959), SPH_C32(0x1bc0b6a2), SPH_C32(0x72010000),
	  SPH_C32(0xcb7d0000), SPH_C32(0x56f40030), SPH_C32(0x53940000),
	  SPH_C32(0xa6ad615c), SPH_C32(0xb6f22344), SPH_C32(0xf6f87e10),
	  SPH_C32(0xda5fceee) },
	{ SPH_C32(0xd10a0000), SPH_C32(0xeda30000), SPH_C32(0x6b26006c),
	  SPH_C32(0x1e370000), SPH_C32(0xfa943185), SPH_C32(0x510c6bea),
	  SPH_C32(0x93a3bb17), SPH_C32(0x6da573f8), SPH_C32(0xdb150000),
	  SPH_C32(0xb6680000), SPH_C32(0xe1380028), SPH_C32(0xadb10000),
	  SPH_C32(0xfbbc07d4), SPH_C32(0xf36b5c9e), SPH_C32(0x01d13074),
	  SPH_C32(0x99872205) },
	{ SPH_C32(0xd6e70000), SPH_C32(0x5ff30000), SPH_C32(0xec520066),
	  SPH_C32(0x893a0000), SPH_C32(0xb9e6122b), SPH_C32(0x19cb054e),
	  SPH_C32(0x67dbd935), SPH_C32(0xfdd0c236), SPH_C32(0x79c30000),
	  SPH_C32(0x101e0000), SPH_C32(0x287c003c), SPH_C32(0x46120000),
	  SPH_C32(0x375029af), SPH_C32(0xc3739807), SPH_C32(0x02983a8e),
	  SPH_C32(0x02e9da8d) },
	{ SPH_C32(0x538d0000), SPH_C32(0xa9fc0000), SPH_C32(0x9ef70006),
	  SPH_C32(0x56ff0000), SPH_C32(0x0ae4004e), SPH_C32(0x92c5cdf9),
	  SPH_C32(0xa9444018), SPH_C32(0x7f975691), SPH_C32(0x01dd0000),
	  SPH_C32(0x80a80000), SPH_C32(0xf4960048), SPH_C32(0xa6000000),
	  SPH_C32(0x90d57ea2), SPH_C32(0xd7e68c37), SPH_C32(0x6612cffd),
	  SPH_C32(0x2c94459e) },
	{ SPH_C32(0x54600000), SPH_C32(0x1bac0000), SPH_C32(0x1983000c),
	  SPH_C32(0xc1f20000), SPH_C32(0x499623e0), SPH_C32(0xda02a35d),
	  SPH_C32(0x5d3c223a), SPH_C32(0xefe2e75f), SPH_C32(0xa30b0000),
	  SPH_C32(0x26de0000), SPH_C32(0x3dd2005c), SPH_C32(0x4da30000),
	  SPH_C32(0x5c3950d9), SPH_C32(0xe7fe48ae), SPH_C32(0x655bc507),
	  SPH_C32(0xb7fabd16) },
	{ SPH_C32(0xdb150000), SPH_C32(0xb6680000), SPH_C32(0xe1380028),
	  SPH_C32(0xadb10000), SPH_C32(0xfbbc07d4), SPH_C32(0xf36b5c9e),
	  SPH_C32(0x01d13074), SPH_C32(0x99872205), SPH_C32(0x0a1f0000),
	  SPH_C32(0x5bcb0000), SPH_C32(0x8a1e0044), SPH_C32(0xb3860000),
	  SPH_C32(0x01283651), SPH_C32(0xa2673774), SPH_C32(0x92728b63),
	  SPH_C32(0xf42251fd) },
	{ SPH_C32(0xdcf80000), SPH_C32(0x04380000), SPH_C32(0x664c0022),
	  SPH_C32(0x3abc0000), SPH_C32(0xb8ce247a), SPH_C32(0xbbac323a),
	  SPH_C32(0xf5a95256), SPH_C32(0x09f293cb), SPH_C32(0xa8c90000),
	  SPH_C32(0xfdbd0000), SPH_C32(0x435a0050), SPH_C32(0x58250000),
	  SPH_C32(0xcdc4182a), SPH_C32(0x927ff3ed), SPH_C32(0x913b8199),
	  SPH_C32(0x6f4ca975) },
	{ SPH_C32(0x584f0000), SPH_C32(0x729f0000), SPH_C32(0xe07f000a),
	  SPH_C32(0x43790000), SPH_C32(0x9b1948bd), SPH_C32(0xe74476ba),
	  SPH_C32(0x5d240486), SPH_C32(0xa72142f2), SPH_C32(0x82870000),
	  SPH_C32(0x445f0000), SPH_C32(0xf5d1006a), SPH_C32(0x48c80000),
	  SPH_C32(0xf07031cb), SPH_C32(0xc3c9a613), SPH_C32(0x3ae7fb0f),
	  SPH_C32(0x12322569) },
	{ SPH_C32(0x5fa20000), SPH_C32(0xc0cf0000), SPH_C32(0x670b0000),
	  SPH_C32(0xd4740000), SPH_C32(0xd86b6b13), SPH_C32(0xaf83181e),
	  SPH_C32(0xa95c66a4), SPH_C32(0x3754f33c), SPH_C32(0x20510000),
	  SPH_C32(0xe2290000), SPH_C32(0x3c95007e), SPH_C32(0xa36b0000),
	  SPH_C32(0x3c9c1fb0), SPH_C32(0xf3d1628a), SPH_C32(0x39aef1f5),
	  SPH_C32(0x895cdde1) },
	{ SPH_C32(0xd0d70000), SPH_C32(0x6d0b0000), SPH_C32(0x9fb00024),
	  SPH_C32(0xb8370000), SPH_C32(0x6a414f27), SPH_C32(0x86eae7dd),
	  SPH_C32(0xf5b174ea), SPH_C32(0x41313666), SPH_C32(0x89450000),
	  SPH_C32(0x9f3c0000), SPH_C32(0x8b590066), SPH_C32(0x5d4e0000),
	  SPH_C32(0x618d7938), SPH_C32(0xb6481d50), SPH_C32(0xce87bf91),
	  SPH_C32(0xca84310a) },
	{ SPH_C32(0xd73a0000), SPH_C32(0xdf5b0000), SPH_C32(0x18c4002e),
	  SPH_C32(0x2f3a0000), SPH_C32(0x29336c89), SPH_C32(0xce2d8979),
	  SPH_C32(0x01c916c8), SPH_C32(0xd14487a8), SPH_C32(0x2b930000),
	  SPH_C32(0x394a0000), SPH_C32(0x421d0072), SPH_C32(0xb6ed0000),
	  SPH_C32(0xad615743), SPH_C32(0x8650d9c9), SPH_C32(0xcdceb56b),
	  SPH_C32(0x51eac982) },
	{ SPH_C32(0x01dd0000), SPH_C32(0x80a80000), SPH_C32(0xf4960048),
	  SPH_C32(0xa6000000), SPH_C32(0x90d57ea2), SPH_C32(0xd7e68c37),
	  SPH_C32(0x6612cffd), SPH_C32(0x2c94459e), SPH_C32(0x52500000),
	  SPH_C32(0x29540000), SPH_C32(0x6a61004e), SPH_C32(0xf0ff0000),
	  SPH_C32(0x9a317eec), SPH_C32(0x452341ce), SPH_C32(0xcf568fe5),
	  SPH_C32(0x5303130f) },
	{ SPH_C32(0x06300000), SPH_C32(0x32f80000), SPH_C32(0x73e20042),
	  SPH_C32(0x310d0000), SPH_C32(0xd3a75d0c), SPH_C32(0x9f21e293),
	  SPH_C32(0x926aaddf), SPH_C32(0xbce1f450), SPH_C32(0xf0860000),
	  SPH_C32(0x8f220000), SPH_C32(0xa325005a), SPH_C32(0x1b5c0000),
	  SPH_C32(0x56dd5097), SPH_C32(0x753b8557), SPH_C32(0xcc1f851f),
	  SPH_C32(0xc86deb87) },
	{ SPH_C32(0x89450000), SPH_C32(0x9f3c0000), SPH_C32(0x8b590066),
	  SPH_C32(0x5d4e0000), SPH_C32(0x618d7938), SPH_C32(0xb6481d50),
	  SPH_C32(0xce87bf91), SPH_C32(0xca84310a), SPH_C32(0x59920000),
	  SPH_C32(0xf2370000), SPH_C32(0x14e90042), SPH_C32(0xe5790000),
	  SPH_C32(0x0bcc361f), SPH_C32(0x30a2fa8d), SPH_C32(0x3b36cb7b),
	  SPH_C32(0x8bb5076c) },
	{ SPH_C32(0x8ea80000), SPH_C32(0x2d6c0000), SPH_C32(0x0c2d006c),
	  SPH_C32(0xca430000), SPH_C32(0x22ff5a96), SPH_C32(0xfe8f73f4),
	  SPH_C32(0x3affddb3), SPH_C32(0x5af180c4), SPH_C32(0xfb440000),
	  SPH_C32(0x54410000), SPH_C32(0xddad0056), SPH_C32(0x0eda0000),
	  SPH_C32(0xc7201864), SPH_C32(0x00ba3e14), SPH_C32(0x387fc181),
	  SPH_C32(0x10dbffe4) },
	{ SPH_C32(0x0a1f0000), SPH_C32(0x5bcb0000), SPH_C32(0x8a1e0044),
	  SPH_C32(0xb3860000), SPH_C32(0x01283651), SPH_C32(0xa2673774),
	  SPH_C32(0x92728b63), SPH_C32(0xf42251fd), SPH_C32(0xd10a0000),
	  SPH_C32(0xeda30000), SPH_C32(0x6b26006c), SPH_C32(0x1e370000),
	  SPH_C32(0xfa943185), SPH_C32(0x510c6bea), SPH_C32(0x93a3bb17),
	  SPH_C32(0x6da573f8) },
	{ SPH_C32(0x0df20000), SPH_C32(0xe99b0000), SPH_C32(0x0d6a004e),
	  SPH_C32(0x248b0000), SPH_C32(0x425a15ff), SPH_C32(0xeaa059d0),
	  SPH_C32(0x660ae941), SPH_C32(0x6457e033), SPH_C32(0x73dc0000),
	  SPH_C32(0x4bd50000), SPH_C32(0xa2620078), SPH_C32(0xf5940000),
	  SPH_C32(0x36781ffe), SPH_C32(0x6114af73), SPH_C32(0x90eab1ed),
	  SPH_C32(0xf6cb8b70) },
	{ SPH_C32(0x82870000), SPH_C32(0x445f0000), SPH_C32(0xf5d1006a),
	  SPH_C32(0x48c80000), SPH_C32(0xf07031cb), SPH_C32(0xc3c9a613),
	  SPH_C32(0x3ae7fb0f), SPH_C32(0x12322569), SPH_C32(0xdac80000),
	  SPH_C32(0x36c00000), SPH_C32(0x15ae0060), SPH_C32(0x0bb10000),
	  SPH_C32(0x6b697976), SPH_C32(0x248dd0a9), SPH_C32(0x67c3ff89),
	  SPH_C32(0xb513679b) },
	{ SPH_C32(0x856a0000), SPH_C32(0xf60f0000), SPH_C32(0x72a50060),
	  SPH_C32(0xdfc50000), SPH_C32(0xb3021265), SPH_C32(0x8b0ec8b7),
	  SPH_C32(0xce9f992d), SPH_C32(0x824794a7), SPH_C32(0x781e0000),
	  SPH_C32(0x90b60000), SPH_C32(0xdcea0074), SPH_C32(0xe0120000),
	  SPH_C32(0xa785570d), SPH_C32(0x14951430), SPH_C32(0x648af573),
	  SPH_C32(0x2e7d9f13) }
};

static const sph_u32 T512_45[32][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x1e6c0000), SPH_C32(0xc4420000), SPH_C32(0x8a2e0000),
	  SPH_C32(0xbcb6b800), SPH_C32(0x2c4413b6), SPH_C32(0x8bfdd3da),
	  SPH_C32(0x6a0c1bc8), SPH_C32(0xb99dc2eb), SPH_C32(0x92560000),
	  SPH_C32(0x1eda0000), SPH_C32(0xea510000), SPH_C32(0xe8b13000),
	  SPH_C32(0xa93556a5), SPH_C32(0xebfb6199), SPH_C32(0xb15c2254),
	  SPH_C32(0x33c5244f) },
	{ SPH_C32(0x92560000), SPH_C32(0x1eda0000), SPH_C32(0xea510000),
	  SPH_C32(0xe8b13000), SPH_C32(0xa93556a5), SPH_C32(0xebfb6199),
	  SPH_C32(0xb15c2254), SPH_C32(0x33c5244f), SPH_C32(0x8c3a0000),
	  SPH_C32(0xda980000), SPH_C32(0x607f0000), SPH_C32(0x54078800),
	  SPH_C32(0x85714513), SPH_C32(0x6006b243), SPH_C32(0xdb50399c),
	  SPH_C32(0x8a58e6a4) },
	{ SPH_C32(0x8c3a0000), SPH_C32(0xda980000), SPH_C32(0x607f0000),
	  SPH_C32(0x54078800), SPH_C32(0x85714513), SPH_C32(0x6006b243),
	  SPH_C32(0xdb50399c), SPH_C32(0x8a58e6a4), SPH_C32(0x1e6c0000),
	  SPH_C32(0xc4420000), SPH_C32(0x8a2e0000), SPH_C32(0xbcb6b800),
	  SPH_C32(0x2c4413b6), SPH_C32(0x8bfdd3da), SPH_C32(0x6a0c1bc8),
	  SPH_C32(0xb99dc2eb) },
	{ SPH_C32(0x58430000), SPH_C32(0x807e0000), SPH_C32(0x78330001),
	  SPH_C32(0xc66b3800), SPH_C32(0xe7375cdc), SPH_C32(0x79ad3fdd),
	  SPH_C32(0xac73fe6f), SPH_C32(0x3a4479b1), SPH_C32(0x1d5a0000),
	  SPH_C32(0x2b720000), SPH_C32(0x488d0000), SPH_C32(0xaf611800),
	  SPH_C32(0x25cb2ec5), SPH_C32(0xc879bfd0), SPH_C32(0x81a20429),
	  SPH_C32(0x1e7536a6) },
	{ SPH_C32(0x462f0000), SPH_C32(0x443c0000), SPH_C32(0xf21d0001),
	  SPH_C32(0x7add8000), SPH_C32(0xcb734f6a), SPH_C32(0xf250ec07),
	  SPH_C32(0xc67fe5a7), SPH_C32(0x83d9bb5a), SPH_C32(0x8f0c0000),
	  SPH_C32(0x35a80000), SPH_C32(0xa2dc0000), SPH_C32(0x47d02800),
	  SPH_C32(0x8cfe7860), SPH_C32(0x2382de49), SPH_C32(0x30fe267d),
	  SPH_C32(0x2db012e9) },
	{ SPH_C32(0xca150000), SPH_C32(0x9ea40000), SPH_C32(0x92620001),
	  SPH_C32(0x2eda0800), SPH_C32(0x4e020a79), SPH_C32(0x92565e44),
	  SPH_C32(0x1d2fdc3b), SPH_C32(0x09815dfe), SPH_C32(0x91600000),
	  SPH_C32(0xf1ea0000), SPH_C32(0x28f20000), SPH_C32(0xfb669000),
	  SPH_C32(0xa0ba6bd6), SPH_C32(0xa87f0d93), SPH_C32(0x5af23db5),
	  SPH_C32(0x942dd002) },
	{ SPH_C32(0xd4790000), SPH_C32(0x5ae60000), SPH_C32(0x184c0001),
	  SPH_C32(0x926cb000), SPH_C32(0x624619cf), SPH_C32(0x19ab8d9e),
	  SPH_C32(0x7723c7f3), SPH_C32(0xb01c9f15), SPH_C32(0x03360000),
	  SPH_C32(0xef300000), SPH_C32(0xc2a30000), SPH_C32(0x13d7a000),
	  SPH_C32(0x098f3d73), SPH_C32(0x43846c0a), SPH_C32(0xebae1fe1),
	  SPH_C32(0xa7e8f44d) },
	{ SPH_C32(0x1d5a0000), SPH_C32(0x2b720000), SPH_C32(0x488d0000),
	  SPH_C32(0xaf611800), SPH_C32(0x25cb2ec5), SPH_C32(0xc879bfd0),
	  SPH_C32(0x81a20429), SPH_C32(0x1e7536a6), SPH_C32(0x45190000),
	  SPH_C32(0xab0c0000), SPH_C32(0x30be0001), SPH_C32(0x690a2000),
	  SPH_C32(0xc2fc7219), SPH_C32(0xb1d4800d), SPH_C32(0x2dd1fa46),
	  SPH_C32(0x24314f17) },
	{ SPH_C32(0x03360000), SPH_C32(0xef300000), SPH_C32(0xc2a30000),
	  SPH_C32(0x13d7a000), SPH_C32(0x098f3d73), SPH_C32(0x43846c0a),
	  SPH_C32(0xebae1fe1), SPH_C32(0xa7e8f44d), SPH_C32(0xd74f0000),
	  SPH_C32(0xb5d60000), SPH_C32(0xdaef0001), SPH_C32(0x81bb1000),
	  SPH_C32(0x6bc924bc), SPH_C32(0x5a2fe194), SPH_C32(0x9c8dd812),
	  SPH_C32(0x17f46b58) },
	{ SPH_C32(0x8f0c0000), SPH_C32(0x35a80000), SPH_C32(0xa2dc0000),
	  SPH_C32(0x47d02800), SPH_C32(0x8cfe7860), SPH_C32(0x2382de49),
	  SPH_C32(0x30fe267d), SPH_C32(0x2db012e9), SPH_C32(0xc9230000),
	  SPH_C32(0x71940000), SPH_C32(0x50c10001), SPH_C32(0x3d0da800),
	  SPH_C32(0x478d370a), SPH_C32(0xd1d2324e), SPH_C32(0xf681c3da),
	  SPH_C32(0xae69a9b3) },
	{ SPH_C32(0x91600000), SPH_C32(0xf1ea0000), SPH_C32(0x28f20000),
	  SPH_C32(0xfb669000), SPH_C32(0xa0ba6bd6), SPH_C32(0xa87f0d93),
	  SPH_C32(0x5af23db5), SPH_C32(0x942dd002), SPH_C32(0x5b750000),
	  SPH_C32(0x6f4e0000), SPH_C32(0xba900001), SPH_C32(0xd5bc9800),
	  SPH_C32(0xeeb861af), SPH_C32(0x3a2953d7), SPH_C32(0x47dde18e),
	  SPH_C32(0x9dac8dfc) },
	{ SPH_C32(0x45190000), SPH_C32(0xab0c0000), SPH_C32(0x30be0001),
	  SPH_C32(0x690a2000), SPH_C32(0xc2fc7219), SPH_C32(0xb1d4800d),
	  SPH_C32(0x2dd1fa46), SPH_C32(0x24314f17), SPH_C32(0x58430000),
	  SPH_C32(0x807e0000), SPH_C32(0x78330001), SPH_C32(0xc66b3800),
	  SPH_C32(0xe7375cdc), SPH_C32(0x79ad3fdd), SPH_C32(0xac73fe6f),
	  SPH_C32(0x3a4479b1) },
	{ SPH_C32(0x5b750000), SPH_C32(0x6f4e0000), SPH_C32(0xba900001),
	  SPH_C32(0xd5bc9800), SPH_C32(0xeeb861af), SPH_C32(0x3a2953d7),
	  SPH_C32(0x47dde18e), SPH_C32(0x9dac8dfc), SPH_C32(0xca150000),
	  SPH_C32(0x9ea40000), SPH_C32(0x92620001), SPH_C32(0x2eda0800),
	  SPH_C32(0x4e020a79), SPH_C32(0x92565e44), SPH_C32(0x1d2fdc3b),
	  SPH_C32(0x09815dfe) },
	{ SPH_C32(0xd74f0000), SPH_C32(0xb5d60000), SPH_C32(0xdaef0001),
	  SPH_C32(0x81bb1000), SPH_C32(0x6bc924bc), SPH_C32(0x5a2fe194),
	  SPH_C32(0x9c8dd812), SPH_C32(0x17f46b58), SPH_C32(0xd4790000),
	  SPH_C32(0x5ae60000), SPH_C32(0x184c0001), SPH_C32(0x926cb000),
	  SPH_C32(0x624619cf), SPH_C32(0x19ab8d9e), SPH_C32(0x7723c7f3),
	  SPH_C32(0xb01c9f15) },
	{ SPH_C32(0xc9230000), SPH_C32(0x71940000), SPH_C32(0x50c10001),
	  SPH_C32(0x3d0da800), SPH_C32(0x478d370a), SPH_C32(0xd1d2324e),
	  SPH_C32(0xf681c3da), SPH_C32(0xae69a9b3), SPH_C32(0x462f0000),
	  SPH_C32(0x443c0000), SPH_C32(0xf21d0001), SPH_C32(0x7add8000),
	  SPH_C32(0xcb734f6a), SPH_C32(0xf250ec07), SPH_C32(0xc67fe5a7),
	  SPH_C32(0x83d9bb5a) },
	{ SPH_C32(0xa53b0000), SPH_C32(0x14260000), SPH_C32(0x4e30001e),
	  SPH_C32(0x7cae0000), SPH_C32(0x8f9e0dd5), SPH_C32(0x78dfaa3d),
	  SPH_C32(0xf73168d8), SPH_C32(0x0b1b4946), SPH_C32(0x07ed0000),
	  SPH_C32(0xb2500000), SPH_C32(0x8774000a), SPH_C32(0x970d0000),
	  SPH_C32(0x437223ae), SPH_C32(0x48c76ea4), SPH_C32(0xf4786222),
	  SPH_C32(0x9075b1ce) },
	{ SPH_C32(0xbb570000), SPH_C32(0xd0640000), SPH_C32(0xc41e001e),
	  SPH_C32(0xc018b800), SPH_C32(0xa3da1e63), SPH_C32(0xf32279e7),
	  SPH_C32(0x9d3d7310), SPH_C32(0xb2868bad), SPH_C32(0x95bb0000),
	  SPH_C32(0xac8a0000), SPH_C32(0x6d25000a), SPH_C32(0x7fbc3000),
	  SPH_C32(0xea47750b), SPH_C32(0xa33c0f3d), SPH_C32(0x45244076),
	  SPH_C32(0xa3b09581) },
	{ SPH_C32(0x376d0000), SPH_C32(0x0afc0000), SPH_C32(0xa461001e),
	  SPH_C32(0x941f3000), SPH_C32(0x26ab5b70), SPH_C32(0x9324cba4),
	  SPH_C32(0x466d4a8c), SPH_C32(0x38de6d09), SPH_C32(0x8bd70000),
	  SPH_C32(0x68c80000), SPH_C32(0xe70b000a), SPH_C32(0xc30a8800),
	  SPH_C32(0xc60366bd), SPH_C32(0x28c1dce7), SPH_C32(0x2f285bbe),
	  SPH_C32(0x1a2d576a) },
	{ SPH_C32(0x29010000), SPH_C32(0xcebe0000), SPH_C32(0x2e4f001e),
	  SPH_C32(0x28a98800), SPH_C32(0x0aef48c6), SPH_C32(0x18d9187e),
	  SPH_C32(0x2c615144), SPH_C32(0x8143afe2), SPH_C32(0x19810000),
	  SPH_C32(0x76120000), SPH_C32(0x0d5a000a), SPH_C32(0x2bbbb800),
	  SPH_C32(0x6f363018), SPH_C32(0xc33abd7e), SPH_C32(0x9e7479ea),
	  SPH_C32(0x29e87325) },
	{ SPH_C32(0xfd780000), SPH_C32(0x94580000), SPH_C32(0x3603001f),
	  SPH_C32(0xbac53800), SPH_C32(0x68a95109), SPH_C32(0x017295e0),
	  SPH_C32(0x5b4296b7), SPH_C32(0x315f30f7), SPH_C32(0x1ab70000),
	  SPH_C32(0x99220000), SPH_C32(0xcff9000a), SPH_C32(0x386c1800),
	  SPH_C32(0x66b90d6b), SPH_C32(0x80bed174), SPH_C32(0x75da660b),
	  SPH_C32(0x8e008768) },
	{ SPH_C32(0xe3140000), SPH_C32(0x501a0000), SPH_C32(0xbc2d001f),
	  SPH_C32(0x06738000), SPH_C32(0x44ed42bf), SPH_C32(0x8a8f463a),
	  SPH_C32(0x314e8d7f), SPH_C32(0x88c2f21c), SPH_C32(0x88e10000),
	  SPH_C32(0x87f80000), SPH_C32(0x25a8000a), SPH_C32(0xd0dd2800),
	  SPH_C32(0xcf8c5bce), SPH_C32(0x6b45b0ed), SPH_C32(0xc486445f),
	  SPH_C32(0xbdc5a327) },
	{ SPH_C32(0x6f2e0000), SPH_C32(0x8a820000), SPH_C32(0xdc52001f),
	  SPH_C32(0x52740800), SPH_C32(0xc19c07ac), SPH_C32(0xea89f479),
	  SPH_C32(0xea1eb4e3), SPH_C32(0x029a14b8), SPH_C32(0x968d0000),
	  SPH_C32(0x43ba0000), SPH_C32(0xaf86000a), SPH_C32(0x6c6b9000),
	  SPH_C32(0xe3c84878), SPH_C32(0xe0b86337), SPH_C32(0xae8a5f97),
	  SPH_C32(0x045861cc) },
	{ SPH_C32(0x71420000), SPH_C32(0x4ec00000), SPH_C32(0x567c001f),
	  SPH_C32(0xeec2b000), SPH_C32(0xedd8141a), SPH_C32(0x617427a3),
	  SPH_C32(0x8012af2b), SPH_C32(0xbb07d653), SPH_C32(0x04db0000),
	  SPH_C32(0x5d600000), SPH_C32(0x45d7000a), SPH_C32(0x84daa000),
	  SPH_C32(0x4afd1edd), SPH_C32(0x0b4302ae), SPH_C32(0x1fd67dc3),
	  SPH_C32(0x379d4583) },
	{ SPH_C32(0xb8610000), SPH_C32(0x3f540000), SPH_C32(0x06bd001e),
	  SPH_C32(0xd3cf1800), SPH_C32(0xaa552310), SPH_C32(0xb0a615ed),
	  SPH_C32(0x76936cf1), SPH_C32(0x156e7fe0), SPH_C32(0x42f40000),
	  SPH_C32(0x195c0000), SPH_C32(0xb7ca000b), SPH_C32(0xfe072000),
	  SPH_C32(0x818e51b7), SPH_C32(0xf913eea9), SPH_C32(0xd9a99864),
	  SPH_C32(0xb444fed9) },
	{ SPH_C32(0xa60d0000), SPH_C32(0xfb160000), SPH_C32(0x8c93001e),
	  SPH_C32(0x6f79a000), SPH_C32(0x861130a6), SPH_C32(0x3b5bc637),
	  SPH_C32(0x1c9f7739), SPH_C32(0xacf3bd0b), SPH_C32(0xd0a20000),
	  SPH_C32(0x07860000), SPH_C32(0x5d9b000b), SPH_C32(0x16b61000),
	  SPH_C32(0x28bb0712), SPH_C32(0x12e88f30), SPH_C32(0x68f5ba30),
	  SPH_C32(0x8781da96) },
	{ SPH_C32(0x2a370000), SPH_C32(0x218e0000), SPH_C32(0xecec001e),
	  SPH_C32(0x3b7e2800), SPH_C32(0x036075b5), SPH_C32(0x5b5d7474),
	  SPH_C32(0xc7cf4ea5), SPH_C32(0x26ab5baf), SPH_C32(0xcece0000),
	  SPH_C32(0xc3c40000), SPH_C32(0xd7b5000b), SPH_C32(0xaa00a800),
	  SPH_C32(0x04ff14a4), SPH_C32(0x99155cea), SPH_C32(0x02f9a1f8),
	  SPH_C32(0x3e1c187d) },
	{ SPH_C32(0x345b0000), SPH_C32(0xe5cc0000), SPH_C32(0x66c2001e),
	  SPH_C32(0x87c89000), SPH_C32(0x2f246603), SPH_C32(0xd0a0a7ae),
	  SPH_C32(0xadc3556d), SPH_C32(0x9f369944), SPH_C32(0x5c980000),
	  SPH_C32(0xdd1e0000), SPH_C32(0x3de4000b), SPH_C32(0x42b19800),
	  SPH_C32(0xadca4201), SPH_C32(0x72ee3d73), SPH_C32(0xb3a583ac),
	  SPH_C32(0x0dd93c32) },
	{ SPH_C32(0xe0220000), SPH_C32(0xbf2a0000), SPH_C32(0x7e8e001f),
	  SPH_C32(0x15a42000), SPH_C32(0x4d627fcc), SPH_C32(0xc90b2a30),
	  SPH_C32(0xdae0929e), SPH_C32(0x2f2a0651), SPH_C32(0x5fae0000),
	  SPH_C32(0x322e0000), SPH_C32(0xff47000b), SPH_C32(0x51663800),
	  SPH_C32(0xa4457f72), SPH_C32(0x316a5179), SPH_C32(0x580b9c4d),
	  SPH_C32(0xaa31c87f) },
	{ SPH_C32(0xfe4e0000), SPH_C32(0x7b680000), SPH_C32(0xf4a0001f),
	  SPH_C32(0xa9129800), SPH_C32(0x61266c7a), SPH_C32(0x42f6f9ea),
	  SPH_C32(0xb0ec8956), SPH_C32(0x96b7c4ba), SPH_C32(0xcdf80000),
	  SPH_C32(0x2cf40000), SPH_C32(0x1516000b), SPH_C32(0xb9d70800),
	  SPH_C32(0x0d7029d7), SPH_C32(0xda9130e0), SPH_C32(0xe957be19),
	  SPH_C32(0x99f4ec30) },
	{ SPH_C32(0x72740000), SPH_C32(0xa1f00000), SPH_C32(0x94df001f),
	  SPH_C32(0xfd151000), SPH_C32(0xe4572969), SPH_C32(0x22f04ba9),
	  SPH_C32(0x6bbcb0ca), SPH_C32(0x1cef221e), SPH_C32(0xd3940000),
	  SPH_C32(0xe8b60000), SPH_C32(0x9f38000b), SPH_C32(0x0561b000),
	  SPH_C32(0x21343a61), SPH_C32(0x516ce33a), SPH_C32(0x835ba5d1),
	  SPH_C32(0x20692edb) },
	{ SPH_C32(0x6c180000), SPH_C32(0x65b20000), SPH_C32(0x1ef1001f),
	  SPH_C32(0x41a3a800), SPH_C32(0xc8133adf), SPH_C32(0xa90d9873),
	  SPH_C32(0x01b0ab02), SPH_C32(0xa572e0f5), SPH_C32(0x41c20000),
	  SPH_C32(0xf66c0000), SPH_C32(0x7569000b), SPH_C32(0xedd08000),
	  SPH_C32(0x88016cc4), SPH_C32(0xba9782a3), SPH_C32(0x32078785),
	  SPH_C32(0x13ac0a94) }
};

static const sph_u32 T512_50[32][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xf0c50000), SPH_C32(0x59230000), SPH_C32(0x45820000),
	  SPH_C32(0xe18d00c0), SPH_C32(0x3b6d0631), SPH_C32(0xc2ed5699),
	  SPH_C32(0xcbe0fe1c), SPH_C32(0x56a7b19f), SPH_C32(0x16ed0000),
	  SPH_C32(0x15680000), SPH_C32(0xedd70000), SPH_C32(0x325d0220),
	  SPH_C32(0xe30c3689), SPH_C32(0x5a4ae643), SPH_C32(0xe375f8a8),
	  SPH_C32(0x81fdf908) },
	{ SPH_C32(0xb4310000), SPH_C32(0x77330000), SPH_C32(0xb15d0000),
	  SPH_C32(0x7fd004e0), SPH_C32(0x78a26138), SPH_C32(0xd116c35d),
	  SPH_C32(0xd256d489), SPH_C32(0x4e6f74de), SPH_C32(0xe3060000),
	  SPH_C32(0xbdc10000), SPH_C32(0x87130000), SPH_C32(0xbff20060),
	  SPH_C32(0x2eba0a1a), SPH_C32(0x8db53751), SPH_C32(0x73c5ab06),
	  SPH_C32(0x5bd61539) },
	{ SPH_C32(0x44f40000), SPH_C32(0x2e100000), SPH_C32(0xf4df0000),
	  SPH_C32(0x9e5d0420), SPH_C32(0x43cf6709), SPH_C32(0x13fb95c4),
	  SPH_C32(0x19b62a95), SPH_C32(0x18c8c541), SPH_C32(0xf5eb0000),
	  SPH_C32(0xa8a90000), SPH_C32(0x6ac40000), SPH_C32(0x8daf0240),
	  SPH_C32(0xcdb63c93), SPH_C32(0xd7ffd112), SPH_C32(0x90b053ae),
	  SPH_C32(0xda2bec31) },
	{ SPH_C32(0xe3060000), SPH_C32(0xbdc10000), SPH_C32(0x87130000),
	  SPH_C32(0xbff20060), SPH_C32(0x2eba0a1a), SPH_C32(0x8db53751),
	  SPH_C32(0x73c5ab06), SPH_C32(0x5bd61539), SPH_C32(0x57370000),
	  SPH_C32(0xcaf20000), SPH_C32(0x364e0000), SPH_C32(0xc0220480),
	  SPH_C32(0x56186b22), SPH_C32(0x5ca3f40c), SPH_C32(0xa1937f8f),
	  SPH_C32(0x15b961e7) },
	{ SPH_C32(0x13c30000), SPH_C32(0xe4e20000), SPH_C32(0xc2910000),
	  SPH_C32(0x5e7f00a0), SPH_C32(0x15d70c2b), SPH_C32(0x4f5861c8),
	  SPH_C32(0xb825551a), SPH_C32(0x0d71a4a6), SPH_C32(0x41da0000),
	  SPH_C32(0xdf9a0000), SPH_C32(0xdb990000), SPH_C32(0xf27f06a0),
	  SPH_C32(0xb5145dab), SPH_C32(0x06e9124f), SPH_C32(0x42e68727),
	  SPH_C32(0x944498ef) },
	{ SPH_C32(0x57370000), SPH_C32(0xcaf20000), SPH_C32(0x364e0000),
	  SPH_C32(0xc0220480), SPH_C32(0x56186b22), SPH_C32(0x5ca3f40c),
	  SPH_C32(0xa1937f8f), SPH_C32(0x15b961e7), SPH_C32(0xb4310000),
	  SPH_C32(0x77330000), SPH_C32(0xb15d0000), SPH_C32(0x7fd004e0),
	  SPH_C32(0x78a26138), SPH_C32(0xd116c35d), SPH_C32(0xd256d489),
	  SPH_C32(0x4e6f74de) },
	{ SPH_C32(0xa7f20000), SPH_C32(0x93d10000), SPH_C32(0x73cc0000),
	  SPH_C32(0x21af0440), SPH_C32(0x6d756d13), SPH_C32(0x9e4ea295),
	  SPH_C32(0x6a738193), SPH_C32(0x431ed078), SPH_C32(0xa2dc0000),
	  SPH_C32(0x625b0000), SPH_C32(0x5c8a0000), SPH_C32(0x4d8d06c0),
	  SPH_C32(0x9bae57b1), SPH_C32(0x8b5c251e), SPH_C32(0x31232c21),
	  SPH_C32(0xcf928dd6) },
	{ SPH_C32(0x02f20000), SPH_C32(0xa2810000), SPH_C32(0x873f0000),
	  SPH_C32(0xe36c7800), SPH_C32(0x1e1d74ef), SPH_C32(0x073d2bd6),
	  SPH_C32(0xc4c23237), SPH_C32(0x7f32259e), SPH_C32(0xbadd0000),
	  SPH_C32(0x13ad0000), SPH_C32(0xb7e70000), SPH_C32(0xf7282800),
	  SPH_C32(0xdf45144d), SPH_C32(0x361ac33a), SPH_C32(0xea5a8d14),
	  SPH_C32(0x2a2c18f0) },
	{ SPH_C32(0xf2370000), SPH_C32(0xfba20000), SPH_C32(0xc2bd0000),
	  SPH_C32(0x02e178c0), SPH_C32(0x257072de), SPH_C32(0xc5d07d4f),
	  SPH_C32(0x0f22cc2b), SPH_C32(0x29959401), SPH_C32(0xac300000),
	  SPH_C32(0x06c50000), SPH_C32(0x5a300000), SPH_C32(0xc5752a20),
	  SPH_C32(0x3c4922c4), SPH_C32(0x6c502579), SPH_C32(0x092f75bc),
	  SPH_C32(0xabd1e1f8) },
	{ SPH_C32(0xb6c30000), SPH_C32(0xd5b20000), SPH_C32(0x36620000),
	  SPH_C32(0x9cbc7ce0), SPH_C32(0x66bf15d7), SPH_C32(0xd62be88b),
	  SPH_C32(0x1694e6be), SPH_C32(0x315d5140), SPH_C32(0x59db0000),
	  SPH_C32(0xae6c0000), SPH_C32(0x30f40000), SPH_C32(0x48da2860),
	  SPH_C32(0xf1ff1e57), SPH_C32(0xbbaff46b), SPH_C32(0x999f2612),
	  SPH_C32(0x71fa0dc9) },
	{ SPH_C32(0x46060000), SPH_C32(0x8c910000), SPH_C32(0x73e00000),
	  SPH_C32(0x7d317c20), SPH_C32(0x5dd213e6), SPH_C32(0x14c6be12),
	  SPH_C32(0xdd7418a2), SPH_C32(0x67fae0df), SPH_C32(0x4f360000),
	  SPH_C32(0xbb040000), SPH_C32(0xdd230000), SPH_C32(0x7a872a40),
	  SPH_C32(0x12f328de), SPH_C32(0xe1e51228), SPH_C32(0x7aeadeba),
	  SPH_C32(0xf007f4c1) },
	{ SPH_C32(0xe1f40000), SPH_C32(0x1f400000), SPH_C32(0x002c0000),
	  SPH_C32(0x5c9e7860), SPH_C32(0x30a77ef5), SPH_C32(0x8a881c87),
	  SPH_C32(0xb7079931), SPH_C32(0x24e430a7), SPH_C32(0xedea0000),
	  SPH_C32(0xd95f0000), SPH_C32(0x81a90000), SPH_C32(0x370a2c80),
	  SPH_C32(0x895d7f6f), SPH_C32(0x6ab93736), SPH_C32(0x4bc9f29b),
	  SPH_C32(0x3f957917) },
	{ SPH_C32(0x11310000), SPH_C32(0x46630000), SPH_C32(0x45ae0000),
	  SPH_C32(0xbd1378a0), SPH_C32(0x0bca78c4), SPH_C32(0x48654a1e),
	  SPH_C32(0x7ce7672d), SPH_C32(0x72438138), SPH_C32(0xfb070000),
	  SPH_C32(0xcc370000), SPH_C32(0x6c7e0000), SPH_C32(0x05572ea0),
	  SPH_C32(0x6a5149e6), SPH_C32(0x30f3d175), SPH_C32(0xa8bc0a33),
	  SPH_C32(0xbe68801f) },
	{ SPH_C32(0x55c50000), SPH_C32(0x68730000), SPH_C32(0xb1710000),
	  SPH_C32(0x234e7c80), SPH_C32(0x48051fcd), SPH_C32(0x5b9edfda),
	  SPH_C32(0x65514db8), SPH_C32(0x6a8b4479), SPH_C32(0x0eec0000),
	  SPH_C32(0x649e0000), SPH_C32(0x06ba0000), SPH_C32(0x88f82ce0),
	  SPH_C32(0xa7e77575), SPH_C32(0xe70c0067), SPH_C32(0x380c599d),
	  SPH_C32(0x64436c2e) },
	{ SPH_C32(0xa5000000), SPH_C32(0x31500000), SPH_C32(0xf4f30000),
	  SPH_C32(0xc2c37c40), SPH_C32(0x736819fc), SPH_C32(0x99738943),
	  SPH_C32(0xaeb1b3a4), SPH_C32(0x3c2cf5e6), SPH_C32(0x18010000),
	  SPH_C32(0x71f60000), SPH_C32(0xeb6d0000), SPH_C32(0xbaa52ec0),
	  SPH_C32(0x44eb43fc), SPH_C32(0xbd46e624), SPH_C32(0xdb79a135),
	  SPH_C32(0xe5be9526) },
	{ SPH_C32(0xbadd0000), SPH_C32(0x13ad0000), SPH_C32(0xb7e70000),
	  SPH_C32(0xf7282800), SPH_C32(0xdf45144d), SPH_C32(0x361ac33a),
	  SPH_C32(0xea5a8d14), SPH_C32(0x2a2c18f0), SPH_C32(0xb82f0000),
	  SPH_C32(0xb12c0000), SPH_C32(0x30d80000), SPH_C32(0x14445000),
	  SPH_C32(0xc15860a2), SPH_C32(0x3127e8ec), SPH_C32(0x2e98bf23),
	  SPH_C32(0x551e3d6e) },
	{ SPH_C32(0x4a180000), SPH_C32(0x4a8e0000), SPH_C32(0xf2650000),
	  SPH_C32(0x16a528c0), SPH_C32(0xe428127c), SPH_C32(0xf4f795a3),
	  SPH_C32(0x21ba7308), SPH_C32(0x7c8ba96f), SPH_C32(0xaec20000),
	  SPH_C32(0xa4440000), SPH_C32(0xdd0f0000), SPH_C32(0x26195220),
	  SPH_C32(0x2254562b), SPH_C32(0x6b6d0eaf), SPH_C32(0xcded478b),
	  SPH_C32(0xd4e3c466) },
	{ SPH_C32(0x0eec0000), SPH_C32(0x649e0000), SPH_C32(0x06ba0000),
	  SPH_C32(0x88f82ce0), SPH_C32(0xa7e77575), SPH_C32(0xe70c0067),
	  SPH_C32(0x380c599d), SPH_C32(0x64436c2e), SPH_C32(0x5b290000),
	  SPH_C32(0x0ced0000), SPH_C32(0xb7cb0000), SPH_C32(0xabb65060),
	  SPH_C32(0xefe26ab8), SPH_C32(0xbc92dfbd), SPH_C32(0x5d5d1425),
	  SPH_C32(0x0ec82857) },
	{ SPH_C32(0xfe290000), SPH_C32(0x3dbd0000), SPH_C32(0x43380000),
	  SPH_C32(0x69752c20), SPH_C32(0x9c8a7344), SPH_C32(0x25e156fe),
	  SPH_C32(0xf3eca781), SPH_C32(0x32e4ddb1), SPH_C32(0x4dc40000),
	  SPH_C32(0x19850000), SPH_C32(0x5a1c0000), SPH_C32(0x99eb5240),
	  SPH_C32(0x0cee5c31), SPH_C32(0xe6d839fe), SPH_C32(0xbe28ec8d),
	  SPH_C32(0x8f35d15f) },
	{ SPH_C32(0x59db0000), SPH_C32(0xae6c0000), SPH_C32(0x30f40000),
	  SPH_C32(0x48da2860), SPH_C32(0xf1ff1e57), SPH_C32(0xbbaff46b),
	  SPH_C32(0x999f2612), SPH_C32(0x71fa0dc9), SPH_C32(0xef180000),
	  SPH_C32(0x7bde0000), SPH_C32(0x06960000), SPH_C32(0xd4665480),
	  SPH_C32(0x97400b80), SPH_C32(0x6d841ce0), SPH_C32(0x8f0bc0ac),
	  SPH_C32(0x40a75c89) },
	{ SPH_C32(0xa91e0000), SPH_C32(0xf74f0000), SPH_C32(0x75760000),
	  SPH_C32(0xa95728a0), SPH_C32(0xca921866), SPH_C32(0x7942a2f2),
	  SPH_C32(0x527fd80e), SPH_C32(0x275dbc56), SPH_C32(0xf9f50000),
	  SPH_C32(0x6eb60000), SPH_C32(0xeb410000), SPH_C32(0xe63b56a0),
	  SPH_C32(0x744c3d09), SPH_C32(0x37cefaa3), SPH_C32(0x6c7e3804),
	  SPH_C32(0xc15aa581) },
	{ SPH_C32(0xedea0000), SPH_C32(0xd95f0000), SPH_C32(0x81a90000),
	  SPH_C32(0x370a2c80), SPH_C32(0x895d7f6f), SPH_C32(0x6ab93736),
	  SPH_C32(0x4bc9f29b), SPH_C32(0x3f957917), SPH_C32(0x0c1e0000),
	  SPH_C32(0xc61f0000), SPH_C32(0x81850000), SPH_C32(0x6b9454e0),
	  SPH_C32(0xb9fa019a), SPH_C32(0xe0312bb1), SPH_C32(0xfcce6baa),
	  SPH_C32(0x1b7149b0) },
	{ SPH_C32(0x1d2f0000), SPH_C32(0x807c0000), SPH_C32(0xc42b0000),
	  SPH_C32(0xd6872c40), SPH_C32(0xb230795e), SPH_C32(0xa85461af),
	  SPH_C32(0x80290c87), SPH_C32(0x6932c888), SPH_C32(0x1af30000),
	  SPH_C32(0xd3770000), SPH_C32(0x6c520000), SPH_C32(0x59c956c0),
	  SPH_C32(0x5af63713), SPH_C32(0xba7bcdf2), SPH_C32(0x1fbb9302),
	  SPH_C32(0x9a8cb0b8) },
	{ SPH_C32(0xb82f0000), SPH_C32(0xb12c0000), SPH_C32(0x30d80000),
	  SPH_C32(0x14445000), SPH_C32(0xc15860a2), SPH_C32(0x3127e8ec),
	  SPH_C32(0x2e98bf23), SPH_C32(0x551e3d6e), SPH_C32(0x02f20000),
	  SPH_C32(0xa2810000), SPH_C32(0x873f0000), SPH_C32(0xe36c7800),
	  SPH_C32(0x1e1d74ef), SPH_C32(0x073d2bd6), SPH_C32(0xc4c23237),
	  SPH_C32(0x7f32259e) },
	{ SPH_C32(0x48ea0000), SPH_C32(0xe80f0000), SPH_C32(0x755a0000),
	  SPH_C32(0xf5c950c0), SPH_C32(0xfa356693), SPH_C32(0xf3cabe75),
	  SPH_C32(0xe578413f), SPH_C32(0x03b98cf1), SPH_C32(0x141f0000),
	  SPH_C32(0xb7e90000), SPH_C32(0x6ae80000), SPH_C32(0xd1317a20),
	  SPH_C32(0xfd114266), SPH_C32(0x5d77cd95), SPH_C32(0x27b7ca9f),
	  SPH_C32(0xfecfdc96) },
	{ SPH_C32(0x0c1e0000), SPH_C32(0xc61f0000), SPH_C32(0x81850000),
	  SPH_C32(0x6b9454e0), SPH_C32(0xb9fa019a), SPH_C32(0xe0312bb1),
	  SPH_C32(0xfcce6baa), SPH_C32(0x1b7149b0), SPH_C32(0xe1f40000),
	  SPH_C32(0x1f400000), SPH_C32(0x002c0000), SPH_C32(0x5c9e7860),
	  SPH_C32(0x30a77ef5), SPH_C32(0x8a881c87), SPH_C32(0xb7079931),
	  SPH_C32(0x24e430a7) },
	{ SPH_C32(0xfcdb0000), SPH_C32(0x9f3c0000), SPH_C32(0xc4070000),
	  SPH_C32(0x8a195420), SPH_C32(0x829707ab), SPH_C32(0x22dc7d28),
	  SPH_C32(0x372e95b6), SPH_C32(0x4dd6f82f), SPH_C32(0xf7190000),
	  SPH_C32(0x0a280000), SPH_C32(0xedfb0000), SPH_C32(0x6ec37a40),
	  SPH_C32(0xd3ab487c), SPH_C32(0xd0c2fac4), SPH_C32(0x54726199),
	  SPH_C32(0xa519c9af) },
	{ SPH_C32(0x5b290000), SPH_C32(0x0ced0000), SPH_C32(0xb7cb0000),
	  SPH_C32(0xabb65060), SPH_C32(0xefe26ab8), SPH_C32(0xbc92dfbd),
	  SPH_C32(0x5d5d1425), SPH_C32(0x0ec82857), SPH_C32(0x55c50000),
	  SPH_C32(0x68730000), SPH_C32(0xb1710000), SPH_C32(0x234e7c80),
	  SPH_C32(0x48051fcd), SPH_C32(0x5b9edfda), SPH_C32(0x65514db8),
	  SPH_C32(0x6a8b4479) },
	{ SPH_C32(0xabec0000), SPH_C32(0x55ce0000), SPH_C32(0xf2490000),
	  SPH_C32(0x4a3b50a0), SPH_C32(0xd48f6c89), SPH_C32(0x7e7f8924),
	  SPH_C32(0x96bdea39), SPH_C32(0x586f99c8), SPH_C32(0x43280000),
	  SPH_C32(0x7d1b0000), SPH_C32(0x5ca60000), SPH_C32(0x11137ea0),
	  SPH_C32(0xab092944), SPH_C32(0x01d43999), SPH_C32(0x8624b510),
	  SPH_C32(0xeb76bd71) },
	{ SPH_C32(0xef180000), SPH_C32(0x7bde0000), SPH_C32(0x06960000),
	  SPH_C32(0xd4665480), SPH_C32(0x97400b80), SPH_C32(0x6d841ce0),
	  SPH_C32(0x8f0bc0ac), SPH_C32(0x40a75c89), SPH_C32(0xb6c30000),
	  SPH_C32(0xd5b20000), SPH_C32(0x36620000), SPH_C32(0x9cbc7ce0),
	  SPH_C32(0x66bf15d7), SPH_C32(0xd62be88b), SPH_C32(0x1694e6be),
	  SPH_C32(0x315d5140) },
	{ SPH_C32(0x1fdd0000), SPH_C32(0x22fd0000), SPH_C32(0x43140000),
	  SPH_C32(0x35eb5440), SPH_C32(0xac2d0db1), SPH_C32(0xaf694a79),
	  SPH_C32(0x44eb3eb0), SPH_C32(0x1600ed16), SPH_C32(0xa02e0000),
	  SPH_C32(0xc0da0000), SPH_C32(0xdbb50000), SPH_C32(0xaee17ec0),
	  SPH_C32(0x85b3235e), SPH_C32(0x8c610ec8), SPH_C32(0xf5e11e16),
	  SPH_C32(0xb0a0a848) }
};

static const sph_u32 T512_55[32][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x5fa80000), SPH_C32(0x56030000), SPH_C32(0x43ae0000),
	  SPH_C32(0x64f30013), SPH_C32(0x257e86bf), SPH_C32(0x1311944e),
	  SPH_C32(0x541e95bf), SPH_C32(0x8ea4db69), SPH_C32(0x00440000),
	  SPH_C32(0x7f480000), SPH_C32(0xda7c0000), SPH_C32(0x2a230001),
	  SPH_C32(0x3badc9cc), SPH_C32(0xa9b69c87), SPH_C32(0x030a9e60),
	  SPH_C32(0xbe0a679e) },
	{ SPH_C32(0x00440000), SPH_C32(0x7f480000), SPH_C32(0xda7c0000),
	  SPH_C32(0x2a230001), SPH_C32(0x3badc9cc), SPH_C32(0xa9b69c87),
	  SPH_C32(0x030a9e60), SPH_C32(0xbe0a679e), SPH_C32(0x5fec0000),
	  SPH_C32(0x294b0000), SPH_C32(0x99d20000), SPH_C32(0x4ed00012),
	  SPH_C32(0x1ed34f73), SPH_C32(0xbaa708c9), SPH_C32(0x57140bdf),
	  SPH_C32(0x30aebcf7) },
	{ SPH_C32(0x5fec0000), SPH_C32(0x294b0000), SPH_C32(0x99d20000),
	  SPH_C32(0x4ed00012), SPH_C32(0x1ed34f73), SPH_C32(0xbaa708c9),
	  SPH_C32(0x57140bdf), SPH_C32(0x30aebcf7), SPH_C32(0x5fa80000),
	  SPH_C32(0x56030000), SPH_C32(0x43ae0000), SPH_C32(0x64f30013),
	  SPH_C32(0x257e86bf), SPH_C32(0x1311944e), SPH_C32(0x541e95bf),
	  SPH_C32(0x8ea4db69) },
	{ SPH_C32(0xee930000), SPH_C32(0xd6070000), SPH_C32(0x92c10000),
	  SPH_C32(0x2b9801e0), SPH_C32(0x9451287c), SPH_C32(0x3b6cfb57),
	  SPH_C32(0x45312374), SPH_C32(0x201f6a64), SPH_C32(0x7b280000),
	  SPH_C32(0x57420000), SPH_C32(0xa9e50000), SPH_C32(0x634300a0),
	  SPH_C32(0x9edb442f), SPH_C32(0x6d9995bb), SPH_C32(0x27f83b03),
	  SPH_C32(0xc7ff60f0) },
	{ SPH_C32(0xb13b0000), SPH_C32(0x80040000), SPH_C32(0xd16f0000),
	  SPH_C32(0x4f6b01f3), SPH_C32(0xb12faec3), SPH_C32(0x287d6f19),
	  SPH_C32(0x112fb6cb), SPH_C32(0xaebbb10d), SPH_C32(0x7b6c0000),
	  SPH_C32(0x280a0000), SPH_C32(0x73990000), SPH_C32(0x496000a1),
	  SPH_C32(0xa5768de3), SPH_C32(0xc42f093c), SPH_C32(0x24f2a563),
	  SPH_C32(0x79f5076e) },
	{ SPH_C32(0xeed70000), SPH_C32(0xa94f0000), SPH_C32(0x48bd0000),
	  SPH_C32(0x01bb01e1), SPH_C32(0xaffce1b0), SPH_C32(0x92da67d0),
	  SPH_C32(0x463bbd14), SPH_C32(0x9e150dfa), SPH_C32(0x24c40000),
	  SPH_C32(0x7e090000), SPH_C32(0x30370000), SPH_C32(0x2d9300b2),
	  SPH_C32(0x80080b5c), SPH_C32(0xd73e9d72), SPH_C32(0x70ec30dc),
	  SPH_C32(0xf751dc07) },
	{ SPH_C32(0xb17f0000), SPH_C32(0xff4c0000), SPH_C32(0x0b130000),
	  SPH_C32(0x654801f2), SPH_C32(0x8a82670f), SPH_C32(0x81cbf39e),
	  SPH_C32(0x122528ab), SPH_C32(0x10b1d693), SPH_C32(0x24800000),
	  SPH_C32(0x01410000), SPH_C32(0xea4b0000), SPH_C32(0x07b000b3),
	  SPH_C32(0xbba5c290), SPH_C32(0x7e8801f5), SPH_C32(0x73e6aebc),
	  SPH_C32(0x495bbb99) },
	{ SPH_C32(0x7b280000), SPH_C32(0x57420000), SPH_C32(0xa9e50000),
	  SPH_C32(0x634300a0), SPH_C32(0x9edb442f), SPH_C32(0x6d9995bb),
	  SPH_C32(0x27f83b03), SPH_C32(0xc7ff60f0), SPH_C32(0x95bb0000),
	  SPH_C32(0x81450000), SPH_C32(0x3b240000), SPH_C32(0x48db0140),
	  SPH_C32(0x0a8a6c53), SPH_C32(0x56f56eec), SPH_C32(0x62c91877),
	  SPH_C32(0xe7e00a94) },
	{ SPH_C32(0x24800000), SPH_C32(0x01410000), SPH_C32(0xea4b0000),
	  SPH_C32(0x07b000b3), SPH_C32(0xbba5c290), SPH_C32(0x7e8801f5),
	  SPH_C32(0x73e6aebc), SPH_C32(0x495bbb99), SPH_C32(0x95ff0000),
	  SPH_C32(0xfe0d0000), SPH_C32(0xe1580000), SPH_C32(0x62f80141),
	  SPH_C32(0x3127a59f), SPH_C32(0xff43f26b), SPH_C32(0x61c38617),
	  SPH_C32(0x59ea6d0a) },
	{ SPH_C32(0x7b6c0000), SPH_C32(0x280a0000), SPH_C32(0x73990000),
	  SPH_C32(0x496000a1), SPH_C32(0xa5768de3), SPH_C32(0xc42f093c),
	  SPH_C32(0x24f2a563), SPH_C32(0x79f5076e), SPH_C32(0xca570000),
	  SPH_C32(0xa80e0000), SPH_C32(0xa2f60000), SPH_C32(0x060b0152),
	  SPH_C32(0x14592320), SPH_C32(0xec526625), SPH_C32(0x35dd13a8),
	  SPH_C32(0xd74eb663) },
	{ SPH_C32(0x24c40000), SPH_C32(0x7e090000), SPH_C32(0x30370000),
	  SPH_C32(0x2d9300b2), SPH_C32(0x80080b5c), SPH_C32(0xd73e9d72),
	  SPH_C32(0x70ec30dc), SPH_C32(0xf751dc07), SPH_C32(0xca130000),
	  SPH_C32(0xd7460000), SPH_C32(0x788a0000), SPH_C32(0x2c280153),
	  SPH_C32(0x2ff4eaec), SPH_C32(0x45e4faa2), SPH_C32(0x36d78dc8),
	  SPH_C32(0x6944d1fd) },
	{ SPH_C32(0x95bb0000), SPH_C32(0x81450000), SPH_C32(0x3b240000),
	  SPH_C32(0x48db0140), SPH_C32(0x0a8a6c53), SPH_C32(0x56f56eec),
	  SPH_C32(0x62c91877), SPH_C32(0xe7e00a94), SPH_C32(0xee930000),
	  SPH_C32(0xd6070000), SPH_C32(0x92c10000), SPH_C32(0x2b9801e0),
	  SPH_C32(0x9451287c), SPH_C32(0x3b6cfb57), SPH_C32(0x45312374),
	  SPH_C32(0x201f6a64) },
	{ SPH_C32(0xca130000), SPH_C32(0xd7460000), SPH_C32(0x788a0000),
	  SPH_C32(0x2c280153), SPH_C32(0x2ff4eaec), SPH_C32(0x45e4faa2),
	  SPH_C32(0x36d78dc8), SPH_C32(0x6944d1fd), SPH_C32(0xeed70000),
	  SPH_C32(0xa94f0000), SPH_C32(0x48bd0000), SPH_C32(0x01bb01e1),
	  SPH_C32(0xaffce1b0), SPH_C32(0x92da67d0), SPH_C32(0x463bbd14),
	  SPH_C32(0x9e150dfa) },
	{ SPH_C32(0x95ff0000), SPH_C32(0xfe0d0000), SPH_C32(0xe1580000),
	  SPH_C32(0x62f80141), SPH_C32(0x3127a59f), SPH_C32(0xff43f26b),
	  SPH_C32(0x61c38617), SPH_C32(0x59ea6d0a), SPH_C32(0xb17f0000),
	  SPH_C32(0xff4c0000), SPH_C32(0x0b130000), SPH_C32(0x654801f2),
	  SPH_C32(0x8a82670f), SPH_C32(0x81cbf39e), SPH_C32(0x122528ab),
	  SPH_C32(0x10b1d693) },
	{ SPH_C32(0xca570000), SPH_C32(0xa80e0000), SPH_C32(0xa2f60000),
	  SPH_C32(0x060b0152), SPH_C32(0x14592320), SPH_C32(0xec526625),
	  SPH_C32(0x35dd13a8), SPH_C32(0xd74eb663), SPH_C32(0xb13b0000),
	  SPH_C32(0x80040000), SPH_C32(0xd16f0000), SPH_C32(0x4f6b01f3),
	  SPH_C32(0xb12faec3), SPH_C32(0x287d6f19), SPH_C32(0x112fb6cb),
	  SPH_C32(0xaebbb10d) },
	{ SPH_C32(0xe6280000), SPH_C32(0x4c4b0000), SPH_C32(0xa8550000),
	  SPH_C32(0xd3d002e0), SPH_C32(0xd86130b8), SPH_C32(0x98a7b0da),
	  SPH_C32(0x289506b4), SPH_C32(0xd75a4897), SPH_C32(0xf0c50000),
	  SPH_C32(0x59230000), SPH_C32(0x45820000), SPH_C32(0xe18d00c0),
	  SPH_C32(0x3b6d0631), SPH_C32(0xc2ed5699), SPH_C32(0xcbe0fe1c),
	  SPH_C32(0x56a7b19f) },
	{ SPH_C32(0xb9800000), SPH_C32(0x1a480000), SPH_C32(0xebfb0000),
	  SPH_C32(0xb72302f3), SPH_C32(0xfd1fb607), SPH_C32(0x8bb62494),
	  SPH_C32(0x7c8b930b), SPH_C32(0x59fe93fe), SPH_C32(0xf0810000),
	  SPH_C32(0x266b0000), SPH_C32(0x9ffe0000), SPH_C32(0xcbae00c1),
	  SPH_C32(0x00c0cffd), SPH_C32(0x6b5bca1e), SPH_C32(0xc8ea607c),
	  SPH_C32(0xe8add601) },
	{ SPH_C32(0xe66c0000), SPH_C32(0x33030000), SPH_C32(0x72290000),
	  SPH_C32(0xf9f302e1), SPH_C32(0xe3ccf974), SPH_C32(0x31112c5d),
	  SPH_C32(0x2b9f98d4), SPH_C32(0x69502f09), SPH_C32(0xaf290000),
	  SPH_C32(0x70680000), SPH_C32(0xdc500000), SPH_C32(0xaf5d00d2),
	  SPH_C32(0x25be4942), SPH_C32(0x784a5e50), SPH_C32(0x9cf4f5c3),
	  SPH_C32(0x66090d68) },
	{ SPH_C32(0xb9c40000), SPH_C32(0x65000000), SPH_C32(0x31870000),
	  SPH_C32(0x9d0002f2), SPH_C32(0xc6b27fcb), SPH_C32(0x2200b813),
	  SPH_C32(0x7f810d6b), SPH_C32(0xe7f4f460), SPH_C32(0xaf6d0000),
	  SPH_C32(0x0f200000), SPH_C32(0x062c0000), SPH_C32(0x857e00d3),
	  SPH_C32(0x1e13808e), SPH_C32(0xd1fcc2d7), SPH_C32(0x9ffe6ba3),
	  SPH_C32(0xd8036af6) },
	{ SPH_C32(0x08bb0000), SPH_C32(0x9a4c0000), SPH_C32(0x3a940000),
	  SPH_C32(0xf8480300), SPH_C32(0x4c3018c4), SPH_C32(0xa3cb4b8d),
	  SPH_C32(0x6da425c0), SPH_C32(0xf74522f3), SPH_C32(0x8bed0000),
	  SPH_C32(0x0e610000), SPH_C32(0xec670000), SPH_C32(0x82ce0060),
	  SPH_C32(0xa5b6421e), SPH_C32(0xaf74c322), SPH_C32(0xec18c51f),
	  SPH_C32(0x9158d16f) },
	{ SPH_C32(0x57130000), SPH_C32(0xcc4f0000), SPH_C32(0x793a0000),
	  SPH_C32(0x9cbb0313), SPH_C32(0x694e9e7b), SPH_C32(0xb0dadfc3),
	  SPH_C32(0x39bab07f), SPH_C32(0x79e1f99a), SPH_C32(0x8ba90000),
	  SPH_C32(0x71290000), SPH_C32(0x361b0000), SPH_C32(0xa8ed0061),
	  SPH_C32(0x9e1b8bd2), SPH_C32(0x06c25fa5), SPH_C32(0xef125b7f),
	  SPH_C32(0x2f52b6f1) },
	{ SPH_C32(0x08ff0000), SPH_C32(0xe5040000), SPH_C32(0xe0e80000),
	  SPH_C32(0xd26b0301), SPH_C32(0x779dd108), SPH_C32(0x0a7dd70a),
	  SPH_C32(0x6eaebba0), SPH_C32(0x494f456d), SPH_C32(0xd4010000),
	  SPH_C32(0x272a0000), SPH_C32(0x75b50000), SPH_C32(0xcc1e0072),
	  SPH_C32(0xbb650d6d), SPH_C32(0x15d3cbeb), SPH_C32(0xbb0ccec0),
	  SPH_C32(0xa1f66d98) },
	{ SPH_C32(0x57570000), SPH_C32(0xb3070000), SPH_C32(0xa3460000),
	  SPH_C32(0xb6980312), SPH_C32(0x52e357b7), SPH_C32(0x196c4344),
	  SPH_C32(0x3ab02e1f), SPH_C32(0xc7eb9e04), SPH_C32(0xd4450000),
	  SPH_C32(0x58620000), SPH_C32(0xafc90000), SPH_C32(0xe63d0073),
	  SPH_C32(0x80c8c4a1), SPH_C32(0xbc65576c), SPH_C32(0xb80650a0),
	  SPH_C32(0x1ffc0a06) },
	{ SPH_C32(0x9d000000), SPH_C32(0x1b090000), SPH_C32(0x01b00000),
	  SPH_C32(0xb0930240), SPH_C32(0x46ba7497), SPH_C32(0xf53e2561),
	  SPH_C32(0x0f6d3db7), SPH_C32(0x10a52867), SPH_C32(0x657e0000),
	  SPH_C32(0xd8660000), SPH_C32(0x7ea60000), SPH_C32(0xa9560180),
	  SPH_C32(0x31e76a62), SPH_C32(0x94183875), SPH_C32(0xa929e66b),
	  SPH_C32(0xb147bb0b) },
	{ SPH_C32(0xc2a80000), SPH_C32(0x4d0a0000), SPH_C32(0x421e0000),
	  SPH_C32(0xd4600253), SPH_C32(0x63c4f228), SPH_C32(0xe62fb12f),
	  SPH_C32(0x5b73a808), SPH_C32(0x9e01f30e), SPH_C32(0x653a0000),
	  SPH_C32(0xa72e0000), SPH_C32(0xa4da0000), SPH_C32(0x83750181),
	  SPH_C32(0x0a4aa3ae), SPH_C32(0x3daea4f2), SPH_C32(0xaa23780b),
	  SPH_C32(0x0f4ddc95) },
	{ SPH_C32(0x9d440000), SPH_C32(0x64410000), SPH_C32(0xdbcc0000),
	  SPH_C32(0x9ab00241), SPH_C32(0x7d17bd5b), SPH_C32(0x5c88b9e6),
	  SPH_C32(0x0c67a3d7), SPH_C32(0xaeaf4ff9), SPH_C32(0x3a920000),
	  SPH_C32(0xf12d0000), SPH_C32(0xe7740000), SPH_C32(0xe7860192),
	  SPH_C32(0x2f342511), SPH_C32(0x2ebf30bc), SPH_C32(0xfe3dedb4),
	  SPH_C32(0x81e907fc) },
	{ SPH_C32(0xc2ec0000), SPH_C32(0x32420000), SPH_C32(0x98620000),
	  SPH_C32(0xfe430252), SPH_C32(0x58693be4), SPH_C32(0x4f992da8),
	  SPH_C32(0x58793668), SPH_C32(0x200b9490), SPH_C32(0x3ad60000),
	  SPH_C32(0x8e650000), SPH_C32(0x3d080000), SPH_C32(0xcda50193),
	  SPH_C32(0x1499ecdd), SPH_C32(0x8709ac3b), SPH_C32(0xfd3773d4),
	  SPH_C32(0x3fe36062) },
	{ SPH_C32(0x73930000), SPH_C32(0xcd0e0000), SPH_C32(0x93710000),
	  SPH_C32(0x9b0b03a0), SPH_C32(0xd2eb5ceb), SPH_C32(0xce52de36),
	  SPH_C32(0x4a5c1ec3), SPH_C32(0x30ba4203), SPH_C32(0x1e560000),
	  SPH_C32(0x8f240000), SPH_C32(0xd7430000), SPH_C32(0xca150120),
	  SPH_C32(0xaf3c2e4d), SPH_C32(0xf981adce), SPH_C32(0x8ed1dd68),
	  SPH_C32(0x76b8dbfb) },
	{ SPH_C32(0x2c3b0000), SPH_C32(0x9b0d0000), SPH_C32(0xd0df0000),
	  SPH_C32(0xfff803b3), SPH_C32(0xf795da54), SPH_C32(0xdd434a78),
	  SPH_C32(0x1e428b7c), SPH_C32(0xbe1e996a), SPH_C32(0x1e120000),
	  SPH_C32(0xf06c0000), SPH_C32(0x0d3f0000), SPH_C32(0xe0360121),
	  SPH_C32(0x9491e781), SPH_C32(0x50373149), SPH_C32(0x8ddb4308),
	  SPH_C32(0xc8b2bc65) },
	{ SPH_C32(0x73d70000), SPH_C32(0xb2460000), SPH_C32(0x490d0000),
	  SPH_C32(0xb12803a1), SPH_C32(0xe9469527), SPH_C32(0x67e442b1),
	  SPH_C32(0x495680a3), SPH_C32(0x8eb0259d), SPH_C32(0x41ba0000),
	  SPH_C32(0xa66f0000), SPH_C32(0x4e910000), SPH_C32(0x84c50132),
	  SPH_C32(0xb1ef613e), SPH_C32(0x4326a507), SPH_C32(0xd9c5d6b7),
	  SPH_C32(0x4616670c) },
	{ SPH_C32(0x2c7f0000), SPH_C32(0xe4450000), SPH_C32(0x0aa30000),
	  SPH_C32(0xd5db03b2), SPH_C32(0xcc381398), SPH_C32(0x74f5d6ff),
	  SPH_C32(0x1d48151c), SPH_C32(0x0014fef4), SPH_C32(0x41fe0000),
	  SPH_C32(0xd9270000), SPH_C32(0x94ed0000), SPH_C32(0xaee60133),
	  SPH_C32(0x8a42a8f2), SPH_C32(0xea903980), SPH_C32(0xdacf48d7),
	  SPH_C32(0xf81c0092) }
};

static const sph_u32 T512_60[16][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x033d0000), SPH_C32(0x08b30000), SPH_C32(0xf33a0000),
	  SPH_C32(0x3ac20007), SPH_C32(0x51298a50), SPH_C32(0x6b6e661f),
	  SPH_C32(0x0ea5cfe3), SPH_C32(0xe6da7ffe), SPH_C32(0xa8da0000),
	  SPH_C32(0x96be0000), SPH_C32(0x5c1d0000), SPH_C32(0x07da0002),
	  SPH_C32(0x7d669583), SPH_C32(0x1f98708a), SPH_C32(0xbb668808),
	  SPH_C32(0xda878000) },
	{ SPH_C32(0xa8da0000), SPH_C32(0x96be0000), SPH_C32(0x5c1d0000),
	  SPH_C32(0x07da0002), SPH_C32(0x7d669583), SPH_C32(0x1f98708a),
	  SPH_C32(0xbb668808), SPH_C32(0xda878000), SPH_C32(0xabe70000),
	  SPH_C32(0x9e0d0000), SPH_C32(0xaf270000), SPH_C32(0x3d180005),
	  SPH_C32(0x2c4f1fd3), SPH_C32(0x74f61695), SPH_C32(0xb5c347eb),
	  SPH_C32(0x3c5dfffe) },
	{ SPH_C32(0xabe70000), SPH_C32(0x9e0d0000), SPH_C32(0xaf270000),
	  SPH_C32(0x3d180005), SPH_C32(0x2c4f1fd3), SPH_C32(0x74f61695),
	  SPH_C32(0xb5c347eb), SPH_C32(0x3c5dfffe), SPH_C32(0x033d0000),
	  SPH_C32(0x08b30000), SPH_C32(0xf33a0000), SPH_C32(0x3ac20007),
	  SPH_C32(0x51298a50), SPH_C32(0x6b6e661f), SPH_C32(0x0ea5cfe3),
	  SPH_C32(0xe6da7ffe) },
	{ SPH_C32(0x01930000), SPH_C32(0xe7820000), SPH_C32(0xedfb0000),
	  SPH_C32(0xcf0c000b), SPH_C32(0x8dd08d58), SPH_C32(0xbca3b42e),
	  SPH_C32(0x063661e1), SPH_C32(0x536f9e7b), SPH_C32(0x92280000),
	  SPH_C32(0xdc850000), SPH_C32(0x57fa0000), SPH_C32(0x56dc0003),
	  SPH_C32(0xbae92316), SPH_C32(0x5aefa30c), SPH_C32(0x90cef752),
	  SPH_C32(0x7b1675d7) },
	{ SPH_C32(0x02ae0000), SPH_C32(0xef310000), SPH_C32(0x1ec10000),
	  SPH_C32(0xf5ce000c), SPH_C32(0xdcf90708), SPH_C32(0xd7cdd231),
	  SPH_C32(0x0893ae02), SPH_C32(0xb5b5e185), SPH_C32(0x3af20000),
	  SPH_C32(0x4a3b0000), SPH_C32(0x0be70000), SPH_C32(0x51060001),
	  SPH_C32(0xc78fb695), SPH_C32(0x4577d386), SPH_C32(0x2ba87f5a),
	  SPH_C32(0xa191f5d7) },
	{ SPH_C32(0xa9490000), SPH_C32(0x713c0000), SPH_C32(0xb1e60000),
	  SPH_C32(0xc8d60009), SPH_C32(0xf0b618db), SPH_C32(0xa33bc4a4),
	  SPH_C32(0xbd50e9e9), SPH_C32(0x89e81e7b), SPH_C32(0x39cf0000),
	  SPH_C32(0x42880000), SPH_C32(0xf8dd0000), SPH_C32(0x6bc40006),
	  SPH_C32(0x96a63cc5), SPH_C32(0x2e19b599), SPH_C32(0x250db0b9),
	  SPH_C32(0x474b8a29) },
	{ SPH_C32(0xaa740000), SPH_C32(0x798f0000), SPH_C32(0x42dc0000),
	  SPH_C32(0xf214000e), SPH_C32(0xa19f928b), SPH_C32(0xc855a2bb),
	  SPH_C32(0xb3f5260a), SPH_C32(0x6f326185), SPH_C32(0x91150000),
	  SPH_C32(0xd4360000), SPH_C32(0xa4c00000), SPH_C32(0x6c1e0004),
	  SPH_C32(0xebc0a946), SPH_C32(0x3181c513), SPH_C32(0x9e6b38b1),
	  SPH_C32(0x9dcc0a29) },
	{ SPH_C32(0x92280000), SPH_C32(0xdc850000), SPH_C32(0x57fa0000),
	  SPH_C32(0x56dc0003), SPH_C32(0xbae92316), SPH_C32(0x5aefa30c),
	  SPH_C32(0x90cef752), SPH_C32(0x7b1675d7), SPH_C32(0x93bb0000),
	  SPH_C32(0x3b070000), SPH_C32(0xba010000), SPH_C32(0x99d00008),
	  SPH_C32(0x3739ae4e), SPH_C32(0xe64c1722), SPH_C32(0x96f896b3),
	  SPH_C32(0x2879ebac) },
	{ SPH_C32(0x91150000), SPH_C32(0xd4360000), SPH_C32(0xa4c00000),
	  SPH_C32(0x6c1e0004), SPH_C32(0xebc0a946), SPH_C32(0x3181c513),
	  SPH_C32(0x9e6b38b1), SPH_C32(0x9dcc0a29), SPH_C32(0x3b610000),
	  SPH_C32(0xadb90000), SPH_C32(0xe61c0000), SPH_C32(0x9e0a000a),
	  SPH_C32(0x4a5f3bcd), SPH_C32(0xf9d467a8), SPH_C32(0x2d9e1ebb),
	  SPH_C32(0xf2fe6bac) },
	{ SPH_C32(0x3af20000), SPH_C32(0x4a3b0000), SPH_C32(0x0be70000),
	  SPH_C32(0x51060001), SPH_C32(0xc78fb695), SPH_C32(0x4577d386),
	  SPH_C32(0x2ba87f5a), SPH_C32(0xa191f5d7), SPH_C32(0x385c0000),
	  SPH_C32(0xa50a0000), SPH_C32(0x15260000), SPH_C32(0xa4c8000d),
	  SPH_C32(0x1b76b19d), SPH_C32(0x92ba01b7), SPH_C32(0x233bd158),
	  SPH_C32(0x14241452) },
	{ SPH_C32(0x39cf0000), SPH_C32(0x42880000), SPH_C32(0xf8dd0000),
	  SPH_C32(0x6bc40006), SPH_C32(0x96a63cc5), SPH_C32(0x2e19b599),
	  SPH_C32(0x250db0b9), SPH_C32(0x474b8a29), SPH_C32(0x90860000),
	  SPH_C32(0x33b40000), SPH_C32(0x493b0000), SPH_C32(0xa312000f),
	  SPH_C32(0x6610241e), SPH_C32(0x8d22713d), SPH_C32(0x985d5950),
	  SPH_C32(0xcea39452) },
	{ SPH_C32(0x93bb0000), SPH_C32(0x3b070000), SPH_C32(0xba010000),
	  SPH_C32(0x99d00008), SPH_C32(0x3739ae4e), SPH_C32(0xe64c1722),
	  SPH_C32(0x96f896b3), SPH_C32(0x2879ebac), SPH_C32(0x01930000),
	  SPH_C32(0xe7820000), SPH_C32(0xedfb0000), SPH_C32(0xcf0c000b),
	  SPH_C32(0x8dd08d58), SPH_C32(0xbca3b42e), SPH_C32(0x063661e1),
	  SPH_C32(0x536f9e7b) },
	{ SPH_C32(0x90860000), SPH_C32(0x33b40000), SPH_C32(0x493b0000),
	  SPH_C32(0xa312000f), SPH_C32(0x6610241e), SPH_C32(0x8d22713d),
	  SPH_C32(0x985d5950), SPH_C32(0xcea39452), SPH_C32(0xa9490000),
	  SPH_C32(0x713c0000), SPH_C32(0xb1e60000), SPH_C32(0xc8d60009),
	  SPH_C32(0xf0b618db), SPH_C32(0xa33bc4a4), SPH_C32(0xbd50e9e9),
	  SPH_C32(0x89e81e7b) },
	{ SPH_C32(0x3b610000), SPH_C32(0xadb90000), SPH_C32(0xe61c0000),
	  SPH_C32(0x9e0a000a), SPH_C32(0x4a5f3bcd), SPH_C32(0xf9d467a8),
	  SPH_C32(0x2d9e1ebb), SPH_C32(0xf2fe6bac), SPH_C32(0xaa740000),
	  SPH_C32(0x798f0000), SPH_C32(0x42dc0000), SPH_C32(0xf214000e),
	  SPH_C32(0xa19f928b), SPH_C32(0xc855a2bb), SPH_C32(0xb3f5260a),
	  SPH_C32(0x6f326185) },
	{ SPH_C32(0x385c0000), SPH_C32(0xa50a0000), SPH_C32(0x15260000),
	  SPH_C32(0xa4c8000d), SPH_C32(0x1b76b19d), SPH_C32(0x92ba01b7),
	  SPH_C32(0x233bd158), SPH_C32(0x14241452), SPH_C32(0x02ae0000),
	  SPH_C32(0xef310000), SPH_C32(0x1ec10000), SPH_C32(0xf5ce000c),
	  SPH_C32(0xdcf90708), SPH_C32(0xd7cdd231), SPH_C32(0x0893ae02),
	  SPH_C32(0xb5b5e185) }
};

#define INPUT_BIG   do { \
		unsigned acc = buf[0]; \
		const sph_u32 *rp; \
		rp = &T512_0[acc >> 3][0]; \
		m0 = rp[0]; \
		m1 = rp[1]; \
		m2 = rp[2]; \
		m3 = rp[3]; \
		m4 = rp[4]; \
		m5 = rp[5]; \
		m6 = rp[6]; \
		m7 = rp[7]; \
		m8 = rp[8]; \
		m9 = rp[9]; \
		mA = rp[10]; \
		mB = rp[11]; \
		mC = rp[12]; \
		mD = rp[13]; \
		mE = rp[14]; \
		mF = rp[15]; \
		acc = (acc << 8) | buf[1]; \
		rp = &T512_5[(acc >> 6) & 0x1f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_10[(acc >> 1) & 0x1f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = (acc << 8) | buf[2]; \
		rp = &T512_15[(acc >> 4) & 0x1f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = (acc << 8) | buf[3]; \
		rp = &T512_20[(acc >> 7) & 0x1f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_25[(acc >> 2) & 0x1f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = (acc << 8) | buf[4]; \
		rp = &T512_30[(acc >> 5) & 0x1f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_35[acc & 0x1f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = buf[5]; \
		rp = &T512_40[acc >> 3][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = (acc << 8) | buf[6]; \
		rp = &T512_45[(acc >> 6) & 0x1f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_50[(acc >> 1) & 0x1f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		acc = (acc << 8) | buf[7]; \
		rp = &T512_55[(acc >> 4) & 0x1f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
		rp = &T512_60[acc & 0x0f][0]; \
		m0 ^= rp[0]; \
		m1 ^= rp[1]; \
		m2 ^= rp[2]; \
		m3 ^= rp[3]; \
		m4 ^= rp[4]; \
		m5 ^= rp[5]; \
		m6 ^= rp[6]; \
		m7 ^= rp[7]; \
		m8 ^= rp[8]; \
		m9 ^= rp[9]; \
		mA ^= rp[10]; \
		mB ^= rp[11]; \
		mC ^= rp[12]; \
		mD ^= rp[13]; \
		mE ^= rp[14]; \
		mF ^= rp[15]; \
	} while (0)

#endif

#if SPH_HAMSI_EXPAND_BIG == 6

static const sph_u32 T512_0[64][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x145a3c00), SPH_C32(0xb9e90000), SPH_C32(0x61270000),
	  SPH_C32(0xf1610000), SPH_C32(0xce613d6c), SPH_C32(0xb0493d78),
	  SPH_C32(0x47a96720), SPH_C32(0xe18e24c5), SPH_C32(0x23671400),
	  SPH_C32(0xc8b90000), SPH_C32(0xf4c70000), SPH_C32(0xfb750000),
	  SPH_C32(0x73cd2465), SPH_C32(0xf8a6a549), SPH_C32(0x02c40a3f),
	  SPH_C32(0xdc24e61f) },
	{ SPH_C32(0x23671400), SPH_C32(0xc8b90000), SPH_C32(0xf4c70000),
	  SPH_C32(0xfb750000), SPH_C32(0x73cd2465), SPH_C32(0xf8a6a549),
	  SPH_C32(0x02c40a3f), SPH_C32(0xdc24e61f), SPH_C32(0x373d2800),
	  SPH_C32(0x71500000), SPH_C32(0x95e00000), SPH_C32(0x0a140000),
	  SPH_C32(0xbdac1909), SPH_C32(0x48ef9831), SPH_C32(0x456d6d1f),
	  SPH_C32(0x3daac2da) },
	{ SPH_C32(0x373d2800), SPH_C32(0x71500000), SPH_C32(0x95e00000),
	  SPH_C32(0x0a140000), SPH_C32(0xbdac1909), SPH_C32(0x48ef9831),
	  SPH_C32(0x456d6d1f), SPH_C32(0x3daac2da), SPH_C32(0x145a3c00),
	  SPH_C32(0xb9e90000), SPH_C32(0x61270000), SPH_C32(0xf1610000),
	  SPH_C32(0xce613d6c), SPH_C32(0xb0493d78), SPH_C32(0x47a96720),
	  SPH_C32(0xe18e24c5) },
	{ SPH_C32(0x54285c00), SPH_C32(0xeaed0000), SPH_C32(0xc5d60000),
	  SPH_C32(0xa1c50000), SPH_C32(0xb3a26770), SPH_C32(0x94a5c4e1),
	  SPH_C32(0x6bb0419d), SPH_C32(0x551b3782), SPH_C32(0x9cbb1800),
	  SPH_C32(0xb0d30000), SPH_C32(0x92510000), SPH_C32(0xed930000),
	  SPH_C32(0x593a4345), SPH_C32(0xe114d5f4), SPH_C32(0x430633da),
	  SPH_C32(0x78cace29) },
	{ SPH_C32(0x40726000), SPH_C32(0x53040000), SPH_C32(0xa4f10000),
	  SPH_C32(0x50a40000), SPH_C32(0x7dc35a1c), SPH_C32(0x24ecf999),
	  SPH_C32(0x2c1926bd), SPH_C32(0xb4951347), SPH_C32(0xbfdc0c00),
	  SPH_C32(0x786a0000), SPH_C32(0x66960000), SPH_C32(0x16e60000),
	  SPH_C32(0x2af76720), SPH_C32(0x19b270bd), SPH_C32(0x41c239e5),
	  SPH_C32(0xa4ee2836) },
	{ SPH_C32(0x774f4800), SPH_C32(0x22540000), SPH_C32(0x31110000),
	  SPH_C32(0x5ab00000), SPH_C32(0xc06f4315), SPH_C32(0x6c0361a8),
	  SPH_C32(0x69744ba2), SPH_C32(0x893fd19d), SPH_C32(0xab863000),
	  SPH_C32(0xc1830000), SPH_C32(0x07b10000), SPH_C32(0xe7870000),
	  SPH_C32(0xe4965a4c), SPH_C32(0xa9fb4dc5), SPH_C32(0x066b5ec5),
	  SPH_C32(0x45600cf3) },
	{ SPH_C32(0x63157400), SPH_C32(0x9bbd0000), SPH_C32(0x50360000),
	  SPH_C32(0xabd10000), SPH_C32(0x0e0e7e79), SPH_C32(0xdc4a5cd0),
	  SPH_C32(0x2edd2c82), SPH_C32(0x68b1f558), SPH_C32(0x88e12400),
	  SPH_C32(0x093a0000), SPH_C32(0xf3760000), SPH_C32(0x1cf20000),
	  SPH_C32(0x975b7e29), SPH_C32(0x515de88c), SPH_C32(0x04af54fa),
	  SPH_C32(0x9944eaec) },
	{ SPH_C32(0x9cbb1800), SPH_C32(0xb0d30000), SPH_C32(0x92510000),
	  SPH_C32(0xed930000), SPH_C32(0x593a4345), SPH_C32(0xe114d5f4),
	  SPH_C32(0x430633da), SPH_C32(0x78cace29), SPH_C32(0xc8934400),
	  SPH_C32(0x5a3e0000), SPH_C32(0x57870000), SPH_C32(0x4c560000),
	  SPH_C32(0xea982435), SPH_C32(0x75b11115), SPH_C32(0x28b67247),
	  SPH_C32(0x2dd1f9ab) },
	{ SPH_C32(0x88e12400), SPH_C32(0x093a0000), SPH_C32(0xf3760000),
	  SPH_C32(0x1cf20000), SPH_C32(0x975b7e29), SPH_C32(0x515de88c),
	  SPH_C32(0x04af54fa), SPH_C32(0x9944eaec), SPH_C32(0xebf45000),
	  SPH_C32(0x92870000), SPH_C32(0xa3400000), SPH_C32(0xb7230000),
	  SPH_C32(0x99550050), SPH_C32(0x8d17b45c), SPH_C32(0x2a727878),
	  SPH_C32(0xf1f51fb4) },
	{ SPH_C32(0xbfdc0c00), SPH_C32(0x786a0000), SPH_C32(0x66960000),
	  SPH_C32(0x16e60000), SPH_C32(0x2af76720), SPH_C32(0x19b270bd),
	  SPH_C32(0x41c239e5), SPH_C32(0xa4ee2836), SPH_C32(0xffae6c00),
	  SPH_C32(0x2b6e0000), SPH_C32(0xc2670000), SPH_C32(0x46420000),
	  SPH_C32(0x57343d3c), SPH_C32(0x3d5e8924), SPH_C32(0x6ddb1f58),
	  SPH_C32(0x107b3b71) },
	{ SPH_C32(0xab863000), SPH_C32(0xc1830000), SPH_C32(0x07b10000),
	  SPH_C32(0xe7870000), SPH_C32(0xe4965a4c), SPH_C32(0xa9fb4dc5),
	  SPH_C32(0x066b5ec5), SPH_C32(0x45600cf3), SPH_C32(0xdcc97800),
	  SPH_C32(0xe3d70000), SPH_C32(0x36a00000), SPH_C32(0xbd370000),
	  SPH_C32(0x24f91959), SPH_C32(0xc5f82c6d), SPH_C32(0x6f1f1567),
	  SPH_C32(0xcc5fdd6e) },
	{ SPH_C32(0xc8934400), SPH_C32(0x5a3e0000), SPH_C32(0x57870000),
	  SPH_C32(0x4c560000), SPH_C32(0xea982435), SPH_C32(0x75b11115),
	  SPH_C32(0x28b67247), SPH_C32(0x2dd1f9ab), SPH_C32(0x54285c00),
	  SPH_C32(0xeaed0000), SPH_C32(0xc5d60000), SPH_C32(0xa1c50000),
	  SPH_C32(0xb3a26770), SPH_C32(0x94a5c4e1), SPH_C32(0x6bb0419d),
	  SPH_C32(0x551b3782) },
	{ SPH_C32(0xdcc97800), SPH_C32(0xe3d70000), SPH_C32(0x36a00000),
	  SPH_C32(0xbd370000), SPH_C32(0x24f91959), SPH_C32(0xc5f82c6d),
	  SPH_C32(0x6f1f1567), SPH_C32(0xcc5fdd6e), SPH_C32(0x774f4800),
	  SPH_C32(0x22540000), SPH_C32(0x31110000), SPH_C32(0x5ab00000),
	  SPH_C32(0xc06f4315), SPH_C32(0x6c0361a8), SPH_C32(0x69744ba2),
	  SPH_C32(0x893fd19d) },
	{ SPH_C32(0xebf45000), SPH_C32(0x92870000), SPH_C32(0xa3400000),
	  SPH_C32(0xb7230000), SPH_C32(0x99550050), SPH_C32(0x8d17b45c),
	  SPH_C32(0x2a727878), SPH_C32(0xf1f51fb4), SPH_C32(0x63157400),
	  SPH_C32(0x9bbd0000), SPH_C32(0x50360000), SPH_C32(0xabd10000),
	  SPH_C32(0x0e0e7e79), SPH_C32(0xdc4a5cd0), SPH_C32(0x2edd2c82),
	  SPH_C32(0x68b1f558) },
	{ SPH_C32(0xffae6c00), SPH_C32(0x2b6e0000), SPH_C32(0xc2670000),
	  SPH_C32(0x46420000), SPH_C32(0x57343d3c), SPH_C32(0x3d5e8924),
	  SPH_C32(0x6ddb1f58), SPH_C32(0x107b3b71), SPH_C32(0x40726000),
	  SPH_C32(0x53040000), SPH_C32(0xa4f10000), SPH_C32(0x50a40000),
	  SPH_C32(0x7dc35a1c), SPH_C32(0x24ecf999), SPH_C32(0x2c1926bd),
	  SPH_C32(0xb4951347) },
	{ SPH_C32(0x29449c00), SPH_C32(0x64e70000), SPH_C32(0xf24b0000),
	  SPH_C32(0xc2f30000), SPH_C32(0x0ede4e8f), SPH_C32(0x56c23745),
	  SPH_C32(0xf3e04259), SPH_C32(0x8d0d9ec4), SPH_C32(0x466d0c00),
	  SPH_C32(0x08620000), SPH_C32(0xdd5d0000), SPH_C32(0xbadd0000),
	  SPH_C32(0x6a927942), SPH_C32(0x441f2b93), SPH_C32(0x218ace6f),
	  SPH_C32(0xbf2c0be2) },
	{ SPH_C32(0x3d1ea000), SPH_C32(0xdd0e0000), SPH_C32(0x936c0000),
	  SPH_C32(0x33920000), SPH_C32(0xc0bf73e3), SPH_C32(0xe68b0a3d),
	  SPH_C32(0xb4492579), SPH_C32(0x6c83ba01), SPH_C32(0x650a1800),
	  SPH_C32(0xc0db0000), SPH_C32(0x299a0000), SPH_C32(0x41a80000),
	  SPH_C32(0x195f5d27), SPH_C32(0xbcb98eda), SPH_C32(0x234ec450),
	  SPH_C32(0x6308edfd) },
	{ SPH_C32(0x0a238800), SPH_C32(0xac5e0000), SPH_C32(0x068c0000),
	  SPH_C32(0x39860000), SPH_C32(0x7d136aea), SPH_C32(0xae64920c),
	  SPH_C32(0xf1244866), SPH_C32(0x512978db), SPH_C32(0x71502400),
	  SPH_C32(0x79320000), SPH_C32(0x48bd0000), SPH_C32(0xb0c90000),
	  SPH_C32(0xd73e604b), SPH_C32(0x0cf0b3a2), SPH_C32(0x64e7a370),
	  SPH_C32(0x8286c938) },
	{ SPH_C32(0x1e79b400), SPH_C32(0x15b70000), SPH_C32(0x67ab0000),
	  SPH_C32(0xc8e70000), SPH_C32(0xb3725786), SPH_C32(0x1e2daf74),
	  SPH_C32(0xb68d2f46), SPH_C32(0xb0a75c1e), SPH_C32(0x52373000),
	  SPH_C32(0xb18b0000), SPH_C32(0xbc7a0000), SPH_C32(0x4bbc0000),
	  SPH_C32(0xa4f3442e), SPH_C32(0xf45616eb), SPH_C32(0x6623a94f),
	  SPH_C32(0x5ea22f27) },
	{ SPH_C32(0x7d6cc000), SPH_C32(0x8e0a0000), SPH_C32(0x379d0000),
	  SPH_C32(0x63360000), SPH_C32(0xbd7c29ff), SPH_C32(0xc267f3a4),
	  SPH_C32(0x985003c4), SPH_C32(0xd816a946), SPH_C32(0xdad61400),
	  SPH_C32(0xb8b10000), SPH_C32(0x4f0c0000), SPH_C32(0x574e0000),
	  SPH_C32(0x33a83a07), SPH_C32(0xa50bfe67), SPH_C32(0x628cfdb5),
	  SPH_C32(0xc7e6c5cb) },
	{ SPH_C32(0x6936fc00), SPH_C32(0x37e30000), SPH_C32(0x56ba0000),
	  SPH_C32(0x92570000), SPH_C32(0x731d1493), SPH_C32(0x722ecedc),
	  SPH_C32(0xdff964e4), SPH_C32(0x39988d83), SPH_C32(0xf9b10000),
	  SPH_C32(0x70080000), SPH_C32(0xbbcb0000), SPH_C32(0xac3b0000),
	  SPH_C32(0x40651e62), SPH_C32(0x5dad5b2e), SPH_C32(0x6048f78a),
	  SPH_C32(0x1bc223d4) },
	{ SPH_C32(0x5e0bd400), SPH_C32(0x46b30000), SPH_C32(0xc35a0000),
	  SPH_C32(0x98430000), SPH_C32(0xceb10d9a), SPH_C32(0x3ac156ed),
	  SPH_C32(0x9a9409fb), SPH_C32(0x04324f59), SPH_C32(0xedeb3c00),
	  SPH_C32(0xc9e10000), SPH_C32(0xdaec0000), SPH_C32(0x5d5a0000),
	  SPH_C32(0x8e04230e), SPH_C32(0xede46656), SPH_C32(0x27e190aa),
	  SPH_C32(0xfa4c0711) },
	{ SPH_C32(0x4a51e800), SPH_C32(0xff5a0000), SPH_C32(0xa27d0000),
	  SPH_C32(0x69220000), SPH_C32(0x00d030f6), SPH_C32(0x8a886b95),
	  SPH_C32(0xdd3d6edb), SPH_C32(0xe5bc6b9c), SPH_C32(0xce8c2800),
	  SPH_C32(0x01580000), SPH_C32(0x2e2b0000), SPH_C32(0xa62f0000),
	  SPH_C32(0xfdc9076b), SPH_C32(0x1542c31f), SPH_C32(0x25259a95),
	  SPH_C32(0x2668e10e) },
	{ SPH_C32(0xb5ff8400), SPH_C32(0xd4340000), SPH_C32(0x601a0000),
	  SPH_C32(0x2f600000), SPH_C32(0x57e40dca), SPH_C32(0xb7d6e2b1),
	  SPH_C32(0xb0e67183), SPH_C32(0xf5c750ed), SPH_C32(0x8efe4800),
	  SPH_C32(0x525c0000), SPH_C32(0x8ada0000), SPH_C32(0xf68b0000),
	  SPH_C32(0x800a5d77), SPH_C32(0x31ae3a86), SPH_C32(0x093cbc28),
	  SPH_C32(0x92fdf249) },
	{ SPH_C32(0xa1a5b800), SPH_C32(0x6ddd0000), SPH_C32(0x013d0000),
	  SPH_C32(0xde010000), SPH_C32(0x998530a6), SPH_C32(0x079fdfc9),
	  SPH_C32(0xf74f16a3), SPH_C32(0x14497428), SPH_C32(0xad995c00),
	  SPH_C32(0x9ae50000), SPH_C32(0x7e1d0000), SPH_C32(0x0dfe0000),
	  SPH_C32(0xf3c77912), SPH_C32(0xc9089fcf), SPH_C32(0x0bf8b617),
	  SPH_C32(0x4ed91456) },
	{ SPH_C32(0x96989000), SPH_C32(0x1c8d0000), SPH_C32(0x94dd0000),
	  SPH_C32(0xd4150000), SPH_C32(0x242929af), SPH_C32(0x4f7047f8),
	  SPH_C32(0xb2227bbc), SPH_C32(0x29e3b6f2), SPH_C32(0xb9c36000),
	  SPH_C32(0x230c0000), SPH_C32(0x1f3a0000), SPH_C32(0xfc9f0000),
	  SPH_C32(0x3da6447e), SPH_C32(0x7941a2b7), SPH_C32(0x4c51d137),
	  SPH_C32(0xaf573093) },
	{ SPH_C32(0x82c2ac00), SPH_C32(0xa5640000), SPH_C32(0xf5fa0000),
	  SPH_C32(0x25740000), SPH_C32(0xea4814c3), SPH_C32(0xff397a80),
	  SPH_C32(0xf58b1c9c), SPH_C32(0xc86d9237), SPH_C32(0x9aa47400),
	  SPH_C32(0xebb50000), SPH_C32(0xebfd0000), SPH_C32(0x07ea0000),
	  SPH_C32(0x4e6b601b), SPH_C32(0x81e707fe), SPH_C32(0x4e95db08),
	  SPH_C32(0x7373d68c) },
	{ SPH_C32(0xe1d7d800), SPH_C32(0x3ed90000), SPH_C32(0xa5cc0000),
	  SPH_C32(0x8ea50000), SPH_C32(0xe4466aba), SPH_C32(0x23732650),
	  SPH_C32(0xdb56301e), SPH_C32(0xa0dc676f), SPH_C32(0x12455000),
	  SPH_C32(0xe28f0000), SPH_C32(0x188b0000), SPH_C32(0x1b180000),
	  SPH_C32(0xd9301e32), SPH_C32(0xd0baef72), SPH_C32(0x4a3a8ff2),
	  SPH_C32(0xea373c60) },
	{ SPH_C32(0xf58de400), SPH_C32(0x87300000), SPH_C32(0xc4eb0000),
	  SPH_C32(0x7fc40000), SPH_C32(0x2a2757d6), SPH_C32(0x933a1b28),
	  SPH_C32(0x9cff573e), SPH_C32(0x415243aa), SPH_C32(0x31224400),
	  SPH_C32(0x2a360000), SPH_C32(0xec4c0000), SPH_C32(0xe06d0000),
	  SPH_C32(0xaafd3a57), SPH_C32(0x281c4a3b), SPH_C32(0x48fe85cd),
	  SPH_C32(0x3613da7f) },
	{ SPH_C32(0xc2b0cc00), SPH_C32(0xf6600000), SPH_C32(0x510b0000),
	  SPH_C32(0x75d00000), SPH_C32(0x978b4edf), SPH_C32(0xdbd58319),
	  SPH_C32(0xd9923a21), SPH_C32(0x7cf88170), SPH_C32(0x25787800),
	  SPH_C32(0x93df0000), SPH_C32(0x8d6b0000), SPH_C32(0x110c0000),
	  SPH_C32(0x649c073b), SPH_C32(0x98557743), SPH_C32(0x0f57e2ed),
	  SPH_C32(0xd79dfeba) },
	{ SPH_C32(0xd6eaf000), SPH_C32(0x4f890000), SPH_C32(0x302c0000),
	  SPH_C32(0x84b10000), SPH_C32(0x59ea73b3), SPH_C32(0x6b9cbe61),
	  SPH_C32(0x9e3b5d01), SPH_C32(0x9d76a5b5), SPH_C32(0x061f6c00),
	  SPH_C32(0x5b660000), SPH_C32(0x79ac0000), SPH_C32(0xea790000),
	  SPH_C32(0x1751235e), SPH_C32(0x60f3d20a), SPH_C32(0x0d93e8d2),
	  SPH_C32(0x0bb918a5) },
	{ SPH_C32(0x466d0c00), SPH_C32(0x08620000), SPH_C32(0xdd5d0000),
	  SPH_C32(0xbadd0000), SPH_C32(0x6a927942), SPH_C32(0x441f2b93),
	  SPH_C32(0x218ace6f), SPH_C32(0xbf2c0be2), SPH_C32(0x6f299000),
	  SPH_C32(0x6c850000), SPH_C32(0x2f160000), SPH_C32(0x782e0000),
	  SPH_C32(0x644c37cd), SPH_C32(0x12dd1cd6), SPH_C32(0xd26a8c36),
	  SPH_C32(0x32219526) },
	{ SPH_C32(0x52373000), SPH_C32(0xb18b0000), SPH_C32(0xbc7a0000),
	  SPH_C32(0x4bbc0000), SPH_C32(0xa4f3442e), SPH_C32(0xf45616eb),
	  SPH_C32(0x6623a94f), SPH_C32(0x5ea22f27), SPH_C32(0x4c4e8400),
	  SPH_C32(0xa43c0000), SPH_C32(0xdbd10000), SPH_C32(0x835b0000),
	  SPH_C32(0x178113a8), SPH_C32(0xea7bb99f), SPH_C32(0xd0ae8609),
	  SPH_C32(0xee057339) },
	{ SPH_C32(0x650a1800), SPH_C32(0xc0db0000), SPH_C32(0x299a0000),
	  SPH_C32(0x41a80000), SPH_C32(0x195f5d27), SPH_C32(0xbcb98eda),
	  SPH_C32(0x234ec450), SPH_C32(0x6308edfd), SPH_C32(0x5814b800),
	  SPH_C32(0x1dd50000), SPH_C32(0xbaf60000), SPH_C32(0x723a0000),
	  SPH_C32(0xd9e02ec4), SPH_C32(0x5a3284e7), SPH_C32(0x9707e129),
	  SPH_C32(0x0f8b57fc) },
	{ SPH_C32(0x71502400), SPH_C32(0x79320000), SPH_C32(0x48bd0000),
	  SPH_C32(0xb0c90000), SPH_C32(0xd73e604b), SPH_C32(0x0cf0b3a2),
	  SPH_C32(0x64e7a370), SPH_C32(0x8286c938), SPH_C32(0x7b73ac00),
	  SPH_C32(0xd56c0000), SPH_C32(0x4e310000), SPH_C32(0x894f0000),
	  SPH_C32(0xaa2d0aa1), SPH_C32(0xa29421ae), SPH_C32(0x95c3eb16),
	  SPH_C32(0xd3afb1e3) },
	{ SPH_C32(0x12455000), SPH_C32(0xe28f0000), SPH_C32(0x188b0000),
	  SPH_C32(0x1b180000), SPH_C32(0xd9301e32), SPH_C32(0xd0baef72),
	  SPH_C32(0x4a3a8ff2), SPH_C32(0xea373c60), SPH_C32(0xf3928800),
	  SPH_C32(0xdc560000), SPH_C32(0xbd470000), SPH_C32(0x95bd0000),
	  SPH_C32(0x3d767488), SPH_C32(0xf3c9c922), SPH_C32(0x916cbfec),
	  SPH_C32(0x4aeb5b0f) },
	{ SPH_C32(0x061f6c00), SPH_C32(0x5b660000), SPH_C32(0x79ac0000),
	  SPH_C32(0xea790000), SPH_C32(0x1751235e), SPH_C32(0x60f3d20a),
	  SPH_C32(0x0d93e8d2), SPH_C32(0x0bb918a5), SPH_C32(0xd0f59c00),
	  SPH_C32(0x14ef0000), SPH_C32(0x49800000), SPH_C32(0x6ec80000),
	  SPH_C32(0x4ebb50ed), SPH_C32(0x0b6f6c6b), SPH_C32(0x93a8b5d3),
	  SPH_C32(0x96cfbd10) },
	{ SPH_C32(0x31224400), SPH_C32(0x2a360000), SPH_C32(0xec4c0000),
	  SPH_C32(0xe06d0000), SPH_C32(0xaafd3a57), SPH_C32(0x281c4a3b),
	  SPH_C32(0x48fe85cd), SPH_C32(0x3613da7f), SPH_C32(0xc4afa000),
	  SPH_C32(0xad060000), SPH_C32(0x28a70000), SPH_C32(0x9fa90000),
	  SPH_C32(0x80da6d81), SPH_C32(0xbb265113), SPH_C32(0xd401d2f3),
	  SPH_C32(0x774199d5) },
	{ SPH_C32(0x25787800), SPH_C32(0x93df0000), SPH_C32(0x8d6b0000),
	  SPH_C32(0x110c0000), SPH_C32(0x649c073b), SPH_C32(0x98557743),
	  SPH_C32(0x0f57e2ed), SPH_C32(0xd79dfeba), SPH_C32(0xe7c8b400),
	  SPH_C32(0x65bf0000), SPH_C32(0xdc600000), SPH_C32(0x64dc0000),
	  SPH_C32(0xf31749e4), SPH_C32(0x4380f45a), SPH_C32(0xd6c5d8cc),
	  SPH_C32(0xab657fca) },
	{ SPH_C32(0xdad61400), SPH_C32(0xb8b10000), SPH_C32(0x4f0c0000),
	  SPH_C32(0x574e0000), SPH_C32(0x33a83a07), SPH_C32(0xa50bfe67),
	  SPH_C32(0x628cfdb5), SPH_C32(0xc7e6c5cb), SPH_C32(0xa7bad400),
	  SPH_C32(0x36bb0000), SPH_C32(0x78910000), SPH_C32(0x34780000),
	  SPH_C32(0x8ed413f8), SPH_C32(0x676c0dc3), SPH_C32(0xfadcfe71),
	  SPH_C32(0x1ff06c8d) },
	{ SPH_C32(0xce8c2800), SPH_C32(0x01580000), SPH_C32(0x2e2b0000),
	  SPH_C32(0xa62f0000), SPH_C32(0xfdc9076b), SPH_C32(0x1542c31f),
	  SPH_C32(0x25259a95), SPH_C32(0x2668e10e), SPH_C32(0x84ddc000),
	  SPH_C32(0xfe020000), SPH_C32(0x8c560000), SPH_C32(0xcf0d0000),
	  SPH_C32(0xfd19379d), SPH_C32(0x9fcaa88a), SPH_C32(0xf818f44e),
	  SPH_C32(0xc3d48a92) },
	{ SPH_C32(0xf9b10000), SPH_C32(0x70080000), SPH_C32(0xbbcb0000),
	  SPH_C32(0xac3b0000), SPH_C32(0x40651e62), SPH_C32(0x5dad5b2e),
	  SPH_C32(0x6048f78a), SPH_C32(0x1bc223d4), SPH_C32(0x9087fc00),
	  SPH_C32(0x47eb0000), SPH_C32(0xed710000), SPH_C32(0x3e6c0000),
	  SPH_C32(0x33780af1), SPH_C32(0x2f8395f2), SPH_C32(0xbfb1936e),
	  SPH_C32(0x225aae57) },
	{ SPH_C32(0xedeb3c00), SPH_C32(0xc9e10000), SPH_C32(0xdaec0000),
	  SPH_C32(0x5d5a0000), SPH_C32(0x8e04230e), SPH_C32(0xede46656),
	  SPH_C32(0x27e190aa), SPH_C32(0xfa4c0711), SPH_C32(0xb3e0e800),
	  SPH_C32(0x8f520000), SPH_C32(0x19b60000), SPH_C32(0xc5190000),
	  SPH_C32(0x40b52e94), SPH_C32(0xd72530bb), SPH_C32(0xbd759951),
	  SPH_C32(0xfe7e4848) },
	{ SPH_C32(0x8efe4800), SPH_C32(0x525c0000), SPH_C32(0x8ada0000),
	  SPH_C32(0xf68b0000), SPH_C32(0x800a5d77), SPH_C32(0x31ae3a86),
	  SPH_C32(0x093cbc28), SPH_C32(0x92fdf249), SPH_C32(0x3b01cc00),
	  SPH_C32(0x86680000), SPH_C32(0xeac00000), SPH_C32(0xd9eb0000),
	  SPH_C32(0xd7ee50bd), SPH_C32(0x8678d837), SPH_C32(0xb9dacdab),
	  SPH_C32(0x673aa2a4) },
	{ SPH_C32(0x9aa47400), SPH_C32(0xebb50000), SPH_C32(0xebfd0000),
	  SPH_C32(0x07ea0000), SPH_C32(0x4e6b601b), SPH_C32(0x81e707fe),
	  SPH_C32(0x4e95db08), SPH_C32(0x7373d68c), SPH_C32(0x1866d800),
	  SPH_C32(0x4ed10000), SPH_C32(0x1e070000), SPH_C32(0x229e0000),
	  SPH_C32(0xa42374d8), SPH_C32(0x7ede7d7e), SPH_C32(0xbb1ec794),
	  SPH_C32(0xbb1e44bb) },
	{ SPH_C32(0xad995c00), SPH_C32(0x9ae50000), SPH_C32(0x7e1d0000),
	  SPH_C32(0x0dfe0000), SPH_C32(0xf3c77912), SPH_C32(0xc9089fcf),
	  SPH_C32(0x0bf8b617), SPH_C32(0x4ed91456), SPH_C32(0x0c3ce400),
	  SPH_C32(0xf7380000), SPH_C32(0x7f200000), SPH_C32(0xd3ff0000),
	  SPH_C32(0x6a4249b4), SPH_C32(0xce974006), SPH_C32(0xfcb7a0b4),
	  SPH_C32(0x5a90607e) },
	{ SPH_C32(0xb9c36000), SPH_C32(0x230c0000), SPH_C32(0x1f3a0000),
	  SPH_C32(0xfc9f0000), SPH_C32(0x3da6447e), SPH_C32(0x7941a2b7),
	  SPH_C32(0x4c51d137), SPH_C32(0xaf573093), SPH_C32(0x2f5bf000),
	  SPH_C32(0x3f810000), SPH_C32(0x8be70000), SPH_C32(0x288a0000),
	  SPH_C32(0x198f6dd1), SPH_C32(0x3631e54f), SPH_C32(0xfe73aa8b),
	  SPH_C32(0x86b48661) },
	{ SPH_C32(0x6f299000), SPH_C32(0x6c850000), SPH_C32(0x2f160000),
	  SPH_C32(0x782e0000), SPH_C32(0x644c37cd), SPH_C32(0x12dd1cd6),
	  SPH_C32(0xd26a8c36), SPH_C32(0x32219526), SPH_C32(0x29449c00),
	  SPH_C32(0x64e70000), SPH_C32(0xf24b0000), SPH_C32(0xc2f30000),
	  SPH_C32(0x0ede4e8f), SPH_C32(0x56c23745), SPH_C32(0xf3e04259),
	  SPH_C32(0x8d0d9ec4) },
	{ SPH_C32(0x7b73ac00), SPH_C32(0xd56c0000), SPH_C32(0x4e310000),
	  SPH_C32(0x894f0000), SPH_C32(0xaa2d0aa1), SPH_C32(0xa29421ae),
	  SPH_C32(0x95c3eb16), SPH_C32(0xd3afb1e3), SPH_C32(0x0a238800),
	  SPH_C32(0xac5e0000), SPH_C32(0x068c0000), SPH_C32(0x39860000),
	  SPH_C32(0x7d136aea), SPH_C32(0xae64920c), SPH_C32(0xf1244866),
	  SPH_C32(0x512978db) },
	{ SPH_C32(0x4c4e8400), SPH_C32(0xa43c0000), SPH_C32(0xdbd10000),
	  SPH_C32(0x835b0000), SPH_C32(0x178113a8), SPH_C32(0xea7bb99f),
	  SPH_C32(0xd0ae8609), SPH_C32(0xee057339), SPH_C32(0x1e79b400),
	  SPH_C32(0x15b70000), SPH_C32(0x67ab0000), SPH_C32(0xc8e70000),
	  SPH_C32(0xb3725786), SPH_C32(0x1e2daf74), SPH_C32(0xb68d2f46),
	  SPH_C32(0xb0a75c1e) },
	{ SPH_C32(0x5814b800), SPH_C32(0x1dd50000), SPH_C32(0xbaf60000),
	  SPH_C32(0x723a0000), SPH_C32(0xd9e02ec4), SPH_C32(0x5a3284e7),
	  SPH_C32(0x9707e129), SPH_C32(0x0f8b57fc), SPH_C32(0x3d1ea000),
	  SPH_C32(0xdd0e0000), SPH_C32(0x936c0000), SPH_C32(0x33920000),
	  SPH_C32(0xc0bf73e3), SPH_C32(0xe68b0a3d), SPH_C32(0xb4492579),
	  SPH_C32(0x6c83ba01) },
	{ SPH_C32(0x3b01cc00), SPH_C32(0x86680000), SPH_C32(0xeac00000),
	  SPH_C32(0xd9eb0000), SPH_C32(0xd7ee50bd), SPH_C32(0x8678d837),
	  SPH_C32(0xb9dacdab), SPH_C32(0x673aa2a4), SPH_C32(0xb5ff8400),
	  SPH_C32(0xd4340000), SPH_C32(0x601a0000), SPH_C32(0x2f600000),
	  SPH_C32(0x57e40dca), SPH_C32(0xb7d6e2b1), SPH_C32(0xb0e67183),
	  SPH_C32(0xf5c750ed) },
	{ SPH_C32(0x2f5bf000), SPH_C32(0x3f810000), SPH_C32(0x8be70000),
	  SPH_C32(0x288a0000), SPH_C32(0x198f6dd1), SPH_C32(0x3631e54f),
	  SPH_C32(0xfe73aa8b), SPH_C32(0x86b48661), SPH_C32(0x96989000),
	  SPH_C32(0x1c8d0000), SPH_C32(0x94dd0000), SPH_C32(0xd4150000),
	  SPH_C32(0x242929af), SPH_C32(0x4f7047f8), SPH_C32(0xb2227bbc),
	  SPH_C32(0x29e3b6f2) },
	{ SPH_C32(0x1866d800), SPH_C32(0x4ed10000), SPH_C32(0x1e070000),
	  SPH_C32(0x229e0000), SPH_C32(0xa42374d8), SPH_C32(0x7ede7d7e),
	  SPH_C32(0xbb1ec794), SPH_C32(0xbb1e44bb), SPH_C32(0x82c2ac00),
	  SPH_C32(0xa5640000), SPH_C32(0xf5fa0000), SPH_C32(0x25740000),
	  SPH_C32(0xea4814c3), SPH_C32(0xff397a80), SPH_C32(0xf58b1c9c),
	  SPH_C32(0xc86d9237) },
	{ SPH_C32(0x0c3ce400), SPH_C32(0xf7380000), SPH_C32(0x7f200000),
	  SPH_C32(0xd3ff0000), SPH_C32(0x6a4249b4), SPH_C32(0xce974006),
	  SPH_C32(0xfcb7a0b4), SPH_C32(0x5a90607e), SPH_C32(0xa1a5b800),
	  SPH_C32(0x6ddd0000), SPH_C32(0x013d0000), SPH_C32(0xde010000),
	  SPH_C32(0x998530a6), SPH_C32(0x079fdfc9), SPH_C32(0xf74f16a3),
	  SPH_C32(0x14497428) },
	{ SPH_C32(0xf3928800), SPH_C32(0xdc560000), SPH_C32(0xbd470000),
	  SPH_C32(0x95bd0000), SPH_C32(0x3d767488), SPH_C32(0xf3c9c922),
	  SPH_C32(0x916cbfec), SPH_C32(0x4aeb5b0f), SPH_C32(0xe1d7d800),
	  SPH_C32(0x3ed90000), SPH_C32(0xa5cc0000), SPH_C32(0x8ea50000),
	  SPH_C32(0xe4466aba), SPH_C32(0x23732650), SPH_C32(0xdb56301e),
	  SPH_C32(0xa0dc676f) },
	{ SPH_C32(0xe7c8b400), SPH_C32(0x65bf0000), SPH_C32(0xdc600000),
	  SPH_C32(0x64dc0000), SPH_C32(0xf31749e4), SPH_C32(0x4380f45a),
	  SPH_C32(0xd6c5d8cc), SPH_C32(0xab657fca), SPH_C32(0xc2b0cc00),
	  SPH_C32(0xf6600000), SPH_C32(0x510b0000), SPH_C32(0x75d00000),
	  SPH_C32(0x978b4edf), SPH_C32(0xdbd58319), SPH_C32(0xd9923a21),
	  SPH_C32(0x7cf88170) },
	{ SPH_C32(0xd0f59c00), SPH_C32(0x14ef0000), SPH_C32(0x49800000),
	  SPH_C32(0x6ec80000), SPH_C32(0x4ebb50ed), SPH_C32(0x0b6f6c6b),
	  SPH_C32(0x93a8b5d3), SPH_C32(0x96cfbd10), SPH_C32(0xd6eaf000),
	  SPH_C32(0x4f890000), SPH_C32(0x302c0000), SPH_C32(0x84b10000),
	  SPH_C32(0x59ea73b3), SPH_C32(0x6b9cbe61), SPH_C32(0x9e3b5d01),
	  SPH_C32(0x9d76a5b5) },
	{ SPH_C32(0xc4afa000), SPH_C32(0xad060000), SPH_C32(0x28a70000),
	  SPH_C32(0x9fa90000), SPH_C32(0x80da6d81), SPH_C32(0xbb265113),
	  SPH_C32(0xd401d2f3), SPH_C32(0x774199d5), SPH_C32(0xf58de400),
	  SPH_C32(0x87300000), SPH_C32(0xc4eb0000), SPH_C32(0x7fc40000),
	  SPH_C32(0x2a2757d6), SPH_C32(0x933a1b28), SPH_C32(0x9cff573e),
	  SPH_C32(0x415243aa) },
	{ SPH_C32(0xa7bad400), SPH_C32(0x36bb0000), SPH_C32(0x78910000),
	  SPH_C32(0x34780000), SPH_C32(0x8ed413f8), SPH_C32(0x676c0dc3),
	  SPH_C32(0xfadcfe71), SPH_C32(0x1ff06c8d), SPH_C32(0x7d6cc000),
	  SPH_C32(0x8e0a0000), SPH_C32(0x379d0000), SPH_C32(0x63360000),
	  SPH_C32(0xbd7c29ff), SPH_C32(0xc267f3a4), SPH_C32(0x985003c4),
	  SPH_C32(0xd816a946) },
	{ SPH_C32(0xb3e0e800), SPH_C32(0x8f520000), SPH_C32(0x19b60000),
	  SPH_C32(0xc5190000), SPH_C32(0x40b52e94), SPH_C32(0xd72530bb),
	  SPH_C32(0xbd759951), SPH_C32(0xfe7e4848), SPH_C32(0x5e0bd400),
	  SPH_C32(0x46b30000), SPH_C32(0xc35a0000), SPH_C32(0x98430000),
	  SPH_C32(0xceb10d9a), SPH_C32(0x3ac156ed), SPH_C32(0x9a9409fb),
	  SPH_C32(0x04324f59) },
	{ SPH_C32(0x84ddc000), SPH_C32(0xfe020000), SPH_C32(0x8c560000),
	  SPH_C32(0xcf0d0000), SPH_C32(0xfd19379d), SPH_C32(0x9fcaa88a),
	  SPH_C32(0xf818f44e), SPH_C32(0xc3d48a92), SPH_C32(0x4a51e800),
	  SPH_C32(0xff5a0000), SPH_C32(0xa27d0000), SPH_C32(0x69220000),
	  SPH_C32(0x00d030f6), SPH_C32(0x8a886b95), SPH_C32(0xdd3d6edb),
	  SPH_C32(0xe5bc6b9c) },
	{ SPH_C32(0x9087fc00), SPH_C32(0x47eb0000), SPH_C32(0xed710000),
	  SPH_C32(0x3e6c0000), SPH_C32(0x33780af1), SPH_C32(0x2f8395f2),
	  SPH_C32(0xbfb1936e), SPH_C32(0x225aae57), SPH_C32(0x6936fc00),
	  SPH_C32(0x37e30000), SPH_C32(0x56ba0000), SPH_C32(0x92570000),
	  SPH_C32(0x731d1493), SPH_C32(0x722ecedc), SPH_C32(0xdff964e4),
	  SPH_C32(0x39988d83) }
};

static const sph_u32 T512_6[64][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x774400f0), SPH_C32(0xf15a0000), SPH_C32(0xf5b20000),
	  SPH_C32(0x34140000), SPH_C32(0x89377e8c), SPH_C32(0x5a8bec25),
	  SPH_C32(0x0bc3cd1e), SPH_C32(0xcf3775cb), SPH_C32(0xf46c0050),
	  SPH_C32(0x96180000), SPH_C32(0x14a50000), SPH_C32(0x031f0000),
	  SPH_C32(0x42947eb8), SPH_C32(0x66bf7e19), SPH_C32(0x9ca470d2),
	  SPH_C32(0x8a341574) },
	{ SPH_C32(0xf46c0050), SPH_C32(0x96180000), SPH_C32(0x14a50000),
	  SPH_C32(0x031f0000), SPH_C32(0x42947eb8), SPH_C32(0x66bf7e19),
	  SPH_C32(0x9ca470d2), SPH_C32(0x8a341574), SPH_C32(0x832800a0),
	  SPH_C32(0x67420000), SPH_C32(0xe1170000), SPH_C32(0x370b0000),
	  SPH_C32(0xcba30034), SPH_C32(0x3c34923c), SPH_C32(0x9767bdcc),
	  SPH_C32(0x450360bf) },
	{ SPH_C32(0x832800a0), SPH_C32(0x67420000), SPH_C32(0xe1170000),
	  SPH_C32(0x370b0000), SPH_C32(0xcba30034), SPH_C32(0x3c34923c),
	  SPH_C32(0x9767bdcc), SPH_C32(0x450360bf), SPH_C32(0x774400f0),
	  SPH_C32(0xf15a0000), SPH_C32(0xf5b20000), SPH_C32(0x34140000),
	  SPH_C32(0x89377e8c), SPH_C32(0x5a8bec25), SPH_C32(0x0bc3cd1e),
	  SPH_C32(0xcf3775cb) },
	{ SPH_C32(0xe8870170), SPH_C32(0x9d720000), SPH_C32(0x12db0000),
	  SPH_C32(0xd4220000), SPH_C32(0xf2886b27), SPH_C32(0xa921e543),
	  SPH_C32(0x4ef8b518), SPH_C32(0x618813b1), SPH_C32(0xb4370060),
	  SPH_C32(0x0c4c0000), SPH_C32(0x56c20000), SPH_C32(0x5cae0000),
	  SPH_C32(0x94541f3f), SPH_C32(0x3b3ef825), SPH_C32(0x1b365f3d),
	  SPH_C32(0xf3d45758) },
	{ SPH_C32(0x9fc30180), SPH_C32(0x6c280000), SPH_C32(0xe7690000),
	  SPH_C32(0xe0360000), SPH_C32(0x7bbf15ab), SPH_C32(0xf3aa0966),
	  SPH_C32(0x453b7806), SPH_C32(0xaebf667a), SPH_C32(0x405b0030),
	  SPH_C32(0x9a540000), SPH_C32(0x42670000), SPH_C32(0x5fb10000),
	  SPH_C32(0xd6c06187), SPH_C32(0x5d81863c), SPH_C32(0x87922fef),
	  SPH_C32(0x79e0422c) },
	{ SPH_C32(0x1ceb0120), SPH_C32(0x0b6a0000), SPH_C32(0x067e0000),
	  SPH_C32(0xd73d0000), SPH_C32(0xb01c159f), SPH_C32(0xcf9e9b5a),
	  SPH_C32(0xd25cc5ca), SPH_C32(0xebbc06c5), SPH_C32(0x371f00c0),
	  SPH_C32(0x6b0e0000), SPH_C32(0xb7d50000), SPH_C32(0x6ba50000),
	  SPH_C32(0x5ff71f0b), SPH_C32(0x070a6a19), SPH_C32(0x8c51e2f1),
	  SPH_C32(0xb6d737e7) },
	{ SPH_C32(0x6baf01d0), SPH_C32(0xfa300000), SPH_C32(0xf3cc0000),
	  SPH_C32(0xe3290000), SPH_C32(0x392b6b13), SPH_C32(0x9515777f),
	  SPH_C32(0xd99f08d4), SPH_C32(0x248b730e), SPH_C32(0xc3730090),
	  SPH_C32(0xfd160000), SPH_C32(0xa3700000), SPH_C32(0x68ba0000),
	  SPH_C32(0x1d6361b3), SPH_C32(0x61b51400), SPH_C32(0x10f59223),
	  SPH_C32(0x3ce32293) },
	{ SPH_C32(0xb4370060), SPH_C32(0x0c4c0000), SPH_C32(0x56c20000),
	  SPH_C32(0x5cae0000), SPH_C32(0x94541f3f), SPH_C32(0x3b3ef825),
	  SPH_C32(0x1b365f3d), SPH_C32(0xf3d45758), SPH_C32(0x5cb00110),
	  SPH_C32(0x913e0000), SPH_C32(0x44190000), SPH_C32(0x888c0000),
	  SPH_C32(0x66dc7418), SPH_C32(0x921f1d66), SPH_C32(0x55ceea25),
	  SPH_C32(0x925c44e9) },
	{ SPH_C32(0xc3730090), SPH_C32(0xfd160000), SPH_C32(0xa3700000),
	  SPH_C32(0x68ba0000), SPH_C32(0x1d6361b3), SPH_C32(0x61b51400),
	  SPH_C32(0x10f59223), SPH_C32(0x3ce32293), SPH_C32(0xa8dc0140),
	  SPH_C32(0x07260000), SPH_C32(0x50bc0000), SPH_C32(0x8b930000),
	  SPH_C32(0x24480aa0), SPH_C32(0xf4a0637f), SPH_C32(0xc96a9af7),
	  SPH_C32(0x1868519d) },
	{ SPH_C32(0x405b0030), SPH_C32(0x9a540000), SPH_C32(0x42670000),
	  SPH_C32(0x5fb10000), SPH_C32(0xd6c06187), SPH_C32(0x5d81863c),
	  SPH_C32(0x87922fef), SPH_C32(0x79e0422c), SPH_C32(0xdf9801b0),
	  SPH_C32(0xf67c0000), SPH_C32(0xa50e0000), SPH_C32(0xbf870000),
	  SPH_C32(0xad7f742c), SPH_C32(0xae2b8f5a), SPH_C32(0xc2a957e9),
	  SPH_C32(0xd75f2456) },
	{ SPH_C32(0x371f00c0), SPH_C32(0x6b0e0000), SPH_C32(0xb7d50000),
	  SPH_C32(0x6ba50000), SPH_C32(0x5ff71f0b), SPH_C32(0x070a6a19),
	  SPH_C32(0x8c51e2f1), SPH_C32(0xb6d737e7), SPH_C32(0x2bf401e0),
	  SPH_C32(0x60640000), SPH_C32(0xb1ab0000), SPH_C32(0xbc980000),
	  SPH_C32(0xefeb0a94), SPH_C32(0xc894f143), SPH_C32(0x5e0d273b),
	  SPH_C32(0x5d6b3122) },
	{ SPH_C32(0x5cb00110), SPH_C32(0x913e0000), SPH_C32(0x44190000),
	  SPH_C32(0x888c0000), SPH_C32(0x66dc7418), SPH_C32(0x921f1d66),
	  SPH_C32(0x55ceea25), SPH_C32(0x925c44e9), SPH_C32(0xe8870170),
	  SPH_C32(0x9d720000), SPH_C32(0x12db0000), SPH_C32(0xd4220000),
	  SPH_C32(0xf2886b27), SPH_C32(0xa921e543), SPH_C32(0x4ef8b518),
	  SPH_C32(0x618813b1) },
	{ SPH_C32(0x2bf401e0), SPH_C32(0x60640000), SPH_C32(0xb1ab0000),
	  SPH_C32(0xbc980000), SPH_C32(0xefeb0a94), SPH_C32(0xc894f143),
	  SPH_C32(0x5e0d273b), SPH_C32(0x5d6b3122), SPH_C32(0x1ceb0120),
	  SPH_C32(0x0b6a0000), SPH_C32(0x067e0000), SPH_C32(0xd73d0000),
	  SPH_C32(0xb01c159f), SPH_C32(0xcf9e9b5a), SPH_C32(0xd25cc5ca),
	  SPH_C32(0xebbc06c5) },
	{ SPH_C32(0xa8dc0140), SPH_C32(0x07260000), SPH_C32(0x50bc0000),
	  SPH_C32(0x8b930000), SPH_C32(0x24480aa0), SPH_C32(0xf4a0637f),
	  SPH_C32(0xc96a9af7), SPH_C32(0x1868519d), SPH_C32(0x6baf01d0),
	  SPH_C32(0xfa300000), SPH_C32(0xf3cc0000), SPH_C32(0xe3290000),
	  SPH_C32(0x392b6b13), SPH_C32(0x9515777f), SPH_C32(0xd99f08d4),
	  SPH_C32(0x248b730e) },
	{ SPH_C32(0xdf9801b0), SPH_C32(0xf67c0000), SPH_C32(0xa50e0000),
	  SPH_C32(0xbf870000), SPH_C32(0xad7f742c), SPH_C32(0xae2b8f5a),
	  SPH_C32(0xc2a957e9), SPH_C32(0xd75f2456), SPH_C32(0x9fc30180),
	  SPH_C32(0x6c280000), SPH_C32(0xe7690000), SPH_C32(0xe0360000),
	  SPH_C32(0x7bbf15ab), SPH_C32(0xf3aa0966), SPH_C32(0x453b7806),
	  SPH_C32(0xaebf667a) },
	{ SPH_C32(0xef0b0270), SPH_C32(0x3afd0000), SPH_C32(0x5dae0000),
	  SPH_C32(0x69490000), SPH_C32(0x9b0f3c06), SPH_C32(0x4405b5f9),
	  SPH_C32(0x66140a51), SPH_C32(0x924f5d0a), SPH_C32(0xc96b0030),
	  SPH_C32(0xe7250000), SPH_C32(0x2f840000), SPH_C32(0x264f0000),
	  SPH_C32(0x08695bf9), SPH_C32(0x6dfcf137), SPH_C32(0x509f6984),
	  SPH_C32(0x9e69af68) },
	{ SPH_C32(0x984f0280), SPH_C32(0xcba70000), SPH_C32(0xa81c0000),
	  SPH_C32(0x5d5d0000), SPH_C32(0x1238428a), SPH_C32(0x1e8e59dc),
	  SPH_C32(0x6dd7c74f), SPH_C32(0x5d7828c1), SPH_C32(0x3d070060),
	  SPH_C32(0x713d0000), SPH_C32(0x3b210000), SPH_C32(0x25500000),
	  SPH_C32(0x4afd2541), SPH_C32(0x0b438f2e), SPH_C32(0xcc3b1956),
	  SPH_C32(0x145dba1c) },
	{ SPH_C32(0x1b670220), SPH_C32(0xace50000), SPH_C32(0x490b0000),
	  SPH_C32(0x6a560000), SPH_C32(0xd99b42be), SPH_C32(0x22bacbe0),
	  SPH_C32(0xfab07a83), SPH_C32(0x187b487e), SPH_C32(0x4a430090),
	  SPH_C32(0x80670000), SPH_C32(0xce930000), SPH_C32(0x11440000),
	  SPH_C32(0xc3ca5bcd), SPH_C32(0x51c8630b), SPH_C32(0xc7f8d448),
	  SPH_C32(0xdb6acfd7) },
	{ SPH_C32(0x6c2302d0), SPH_C32(0x5dbf0000), SPH_C32(0xbcb90000),
	  SPH_C32(0x5e420000), SPH_C32(0x50ac3c32), SPH_C32(0x783127c5),
	  SPH_C32(0xf173b79d), SPH_C32(0xd74c3db5), SPH_C32(0xbe2f00c0),
	  SPH_C32(0x167f0000), SPH_C32(0xda360000), SPH_C32(0x125b0000),
	  SPH_C32(0x815e2575), SPH_C32(0x37771d12), SPH_C32(0x5b5ca49a),
	  SPH_C32(0x515edaa3) },
	{ SPH_C32(0x078c0300), SPH_C32(0xa78f0000), SPH_C32(0x4f750000),
	  SPH_C32(0xbd6b0000), SPH_C32(0x69875721), SPH_C32(0xed2450ba),
	  SPH_C32(0x28ecbf49), SPH_C32(0xf3c74ebb), SPH_C32(0x7d5c0050),
	  SPH_C32(0xeb690000), SPH_C32(0x79460000), SPH_C32(0x7ae10000),
	  SPH_C32(0x9c3d44c6), SPH_C32(0x56c20912), SPH_C32(0x4ba936b9),
	  SPH_C32(0x6dbdf830) },
	{ SPH_C32(0x70c803f0), SPH_C32(0x56d50000), SPH_C32(0xbac70000),
	  SPH_C32(0x897f0000), SPH_C32(0xe0b029ad), SPH_C32(0xb7afbc9f),
	  SPH_C32(0x232f7257), SPH_C32(0x3cf03b70), SPH_C32(0x89300000),
	  SPH_C32(0x7d710000), SPH_C32(0x6de30000), SPH_C32(0x79fe0000),
	  SPH_C32(0xdea93a7e), SPH_C32(0x307d770b), SPH_C32(0xd70d466b),
	  SPH_C32(0xe789ed44) },
	{ SPH_C32(0xf3e00350), SPH_C32(0x31970000), SPH_C32(0x5bd00000),
	  SPH_C32(0xbe740000), SPH_C32(0x2b132999), SPH_C32(0x8b9b2ea3),
	  SPH_C32(0xb448cf9b), SPH_C32(0x79f35bcf), SPH_C32(0xfe7400f0),
	  SPH_C32(0x8c2b0000), SPH_C32(0x98510000), SPH_C32(0x4dea0000),
	  SPH_C32(0x579e44f2), SPH_C32(0x6af69b2e), SPH_C32(0xdcce8b75),
	  SPH_C32(0x28be988f) },
	{ SPH_C32(0x84a403a0), SPH_C32(0xc0cd0000), SPH_C32(0xae620000),
	  SPH_C32(0x8a600000), SPH_C32(0xa2245715), SPH_C32(0xd110c286),
	  SPH_C32(0xbf8b0285), SPH_C32(0xb6c42e04), SPH_C32(0x0a1800a0),
	  SPH_C32(0x1a330000), SPH_C32(0x8cf40000), SPH_C32(0x4ef50000),
	  SPH_C32(0x150a3a4a), SPH_C32(0x0c49e537), SPH_C32(0x406afba7),
	  SPH_C32(0xa28a8dfb) },
	{ SPH_C32(0x5b3c0210), SPH_C32(0x36b10000), SPH_C32(0x0b6c0000),
	  SPH_C32(0x35e70000), SPH_C32(0x0f5b2339), SPH_C32(0x7f3b4ddc),
	  SPH_C32(0x7d22556c), SPH_C32(0x619b0a52), SPH_C32(0x95db0120),
	  SPH_C32(0x761b0000), SPH_C32(0x6b9d0000), SPH_C32(0xaec30000),
	  SPH_C32(0x6eb52fe1), SPH_C32(0xffe3ec51), SPH_C32(0x055183a1),
	  SPH_C32(0x0c35eb81) },
	{ SPH_C32(0x2c7802e0), SPH_C32(0xc7eb0000), SPH_C32(0xfede0000),
	  SPH_C32(0x01f30000), SPH_C32(0x866c5db5), SPH_C32(0x25b0a1f9),
	  SPH_C32(0x76e19872), SPH_C32(0xaeac7f99), SPH_C32(0x61b70170),
	  SPH_C32(0xe0030000), SPH_C32(0x7f380000), SPH_C32(0xaddc0000),
	  SPH_C32(0x2c215159), SPH_C32(0x995c9248), SPH_C32(0x99f5f373),
	  SPH_C32(0x8601fef5) },
	{ SPH_C32(0xaf500240), SPH_C32(0xa0a90000), SPH_C32(0x1fc90000),
	  SPH_C32(0x36f80000), SPH_C32(0x4dcf5d81), SPH_C32(0x198433c5),
	  SPH_C32(0xe18625be), SPH_C32(0xebaf1f26), SPH_C32(0x16f30180),
	  SPH_C32(0x11590000), SPH_C32(0x8a8a0000), SPH_C32(0x99c80000),
	  SPH_C32(0xa5162fd5), SPH_C32(0xc3d77e6d), SPH_C32(0x92363e6d),
	  SPH_C32(0x49368b3e) },
	{ SPH_C32(0xd81402b0), SPH_C32(0x51f30000), SPH_C32(0xea7b0000),
	  SPH_C32(0x02ec0000), SPH_C32(0xc4f8230d), SPH_C32(0x430fdfe0),
	  SPH_C32(0xea45e8a0), SPH_C32(0x24986aed), SPH_C32(0xe29f01d0),
	  SPH_C32(0x87410000), SPH_C32(0x9e2f0000), SPH_C32(0x9ad70000),
	  SPH_C32(0xe782516d), SPH_C32(0xa5680074), SPH_C32(0x0e924ebf),
	  SPH_C32(0xc3029e4a) },
	{ SPH_C32(0xb3bb0360), SPH_C32(0xabc30000), SPH_C32(0x19b70000),
	  SPH_C32(0xe1c50000), SPH_C32(0xfdd3481e), SPH_C32(0xd61aa89f),
	  SPH_C32(0x33dae074), SPH_C32(0x001319e3), SPH_C32(0x21ec0140),
	  SPH_C32(0x7a570000), SPH_C32(0x3d5f0000), SPH_C32(0xf26d0000),
	  SPH_C32(0xfae130de), SPH_C32(0xc4dd1474), SPH_C32(0x1e67dc9c),
	  SPH_C32(0xffe1bcd9) },
	{ SPH_C32(0xc4ff0390), SPH_C32(0x5a990000), SPH_C32(0xec050000),
	  SPH_C32(0xd5d10000), SPH_C32(0x74e43692), SPH_C32(0x8c9144ba),
	  SPH_C32(0x38192d6a), SPH_C32(0xcf246c28), SPH_C32(0xd5800110),
	  SPH_C32(0xec4f0000), SPH_C32(0x29fa0000), SPH_C32(0xf1720000),
	  SPH_C32(0xb8754e66), SPH_C32(0xa2626a6d), SPH_C32(0x82c3ac4e),
	  SPH_C32(0x75d5a9ad) },
	{ SPH_C32(0x47d70330), SPH_C32(0x3ddb0000), SPH_C32(0x0d120000),
	  SPH_C32(0xe2da0000), SPH_C32(0xbf4736a6), SPH_C32(0xb0a5d686),
	  SPH_C32(0xaf7e90a6), SPH_C32(0x8a270c97), SPH_C32(0xa2c401e0),
	  SPH_C32(0x1d150000), SPH_C32(0xdc480000), SPH_C32(0xc5660000),
	  SPH_C32(0x314230ea), SPH_C32(0xf8e98648), SPH_C32(0x89006150),
	  SPH_C32(0xbae2dc66) },
	{ SPH_C32(0x309303c0), SPH_C32(0xcc810000), SPH_C32(0xf8a00000),
	  SPH_C32(0xd6ce0000), SPH_C32(0x3670482a), SPH_C32(0xea2e3aa3),
	  SPH_C32(0xa4bd5db8), SPH_C32(0x4510795c), SPH_C32(0x56a801b0),
	  SPH_C32(0x8b0d0000), SPH_C32(0xc8ed0000), SPH_C32(0xc6790000),
	  SPH_C32(0x73d64e52), SPH_C32(0x9e56f851), SPH_C32(0x15a41182),
	  SPH_C32(0x30d6c912) },
	{ SPH_C32(0xc96b0030), SPH_C32(0xe7250000), SPH_C32(0x2f840000),
	  SPH_C32(0x264f0000), SPH_C32(0x08695bf9), SPH_C32(0x6dfcf137),
	  SPH_C32(0x509f6984), SPH_C32(0x9e69af68), SPH_C32(0x26600240),
	  SPH_C32(0xddd80000), SPH_C32(0x722a0000), SPH_C32(0x4f060000),
	  SPH_C32(0x936667ff), SPH_C32(0x29f944ce), SPH_C32(0x368b63d5),
	  SPH_C32(0x0c26f262) },
	{ SPH_C32(0xbe2f00c0), SPH_C32(0x167f0000), SPH_C32(0xda360000),
	  SPH_C32(0x125b0000), SPH_C32(0x815e2575), SPH_C32(0x37771d12),
	  SPH_C32(0x5b5ca49a), SPH_C32(0x515edaa3), SPH_C32(0xd20c0210),
	  SPH_C32(0x4bc00000), SPH_C32(0x668f0000), SPH_C32(0x4c190000),
	  SPH_C32(0xd1f21947), SPH_C32(0x4f463ad7), SPH_C32(0xaa2f1307),
	  SPH_C32(0x8612e716) },
	{ SPH_C32(0x3d070060), SPH_C32(0x713d0000), SPH_C32(0x3b210000),
	  SPH_C32(0x25500000), SPH_C32(0x4afd2541), SPH_C32(0x0b438f2e),
	  SPH_C32(0xcc3b1956), SPH_C32(0x145dba1c), SPH_C32(0xa54802e0),
	  SPH_C32(0xba9a0000), SPH_C32(0x933d0000), SPH_C32(0x780d0000),
	  SPH_C32(0x58c567cb), SPH_C32(0x15cdd6f2), SPH_C32(0xa1ecde19),
	  SPH_C32(0x492592dd) },
	{ SPH_C32(0x4a430090), SPH_C32(0x80670000), SPH_C32(0xce930000),
	  SPH_C32(0x11440000), SPH_C32(0xc3ca5bcd), SPH_C32(0x51c8630b),
	  SPH_C32(0xc7f8d448), SPH_C32(0xdb6acfd7), SPH_C32(0x512402b0),
	  SPH_C32(0x2c820000), SPH_C32(0x87980000), SPH_C32(0x7b120000),
	  SPH_C32(0x1a511973), SPH_C32(0x7372a8eb), SPH_C32(0x3d48aecb),
	  SPH_C32(0xc31187a9) },
	{ SPH_C32(0x21ec0140), SPH_C32(0x7a570000), SPH_C32(0x3d5f0000),
	  SPH_C32(0xf26d0000), SPH_C32(0xfae130de), SPH_C32(0xc4dd1474),
	  SPH_C32(0x1e67dc9c), SPH_C32(0xffe1bcd9), SPH_C32(0x92570220),
	  SPH_C32(0xd1940000), SPH_C32(0x24e80000), SPH_C32(0x13a80000),
	  SPH_C32(0x073278c0), SPH_C32(0x12c7bceb), SPH_C32(0x2dbd3ce8),
	  SPH_C32(0xfff2a53a) },
	{ SPH_C32(0x56a801b0), SPH_C32(0x8b0d0000), SPH_C32(0xc8ed0000),
	  SPH_C32(0xc6790000), SPH_C32(0x73d64e52), SPH_C32(0x9e56f851),
	  SPH_C32(0x15a41182), SPH_C32(0x30d6c912), SPH_C32(0x663b0270),
	  SPH_C32(0x478c0000), SPH_C32(0x304d0000), SPH_C32(0x10b70000),
	  SPH_C32(0x45a60678), SPH_C32(0x7478c2f2), SPH_C32(0xb1194c3a),
	  SPH_C32(0x75c6b04e) },
	{ SPH_C32(0xd5800110), SPH_C32(0xec4f0000), SPH_C32(0x29fa0000),
	  SPH_C32(0xf1720000), SPH_C32(0xb8754e66), SPH_C32(0xa2626a6d),
	  SPH_C32(0x82c3ac4e), SPH_C32(0x75d5a9ad), SPH_C32(0x117f0280),
	  SPH_C32(0xb6d60000), SPH_C32(0xc5ff0000), SPH_C32(0x24a30000),
	  SPH_C32(0xcc9178f4), SPH_C32(0x2ef32ed7), SPH_C32(0xbada8124),
	  SPH_C32(0xbaf1c585) },
	{ SPH_C32(0xa2c401e0), SPH_C32(0x1d150000), SPH_C32(0xdc480000),
	  SPH_C32(0xc5660000), SPH_C32(0x314230ea), SPH_C32(0xf8e98648),
	  SPH_C32(0x89006150), SPH_C32(0xbae2dc66), SPH_C32(0xe51302d0),
	  SPH_C32(0x20ce0000), SPH_C32(0xd15a0000), SPH_C32(0x27bc0000),
	  SPH_C32(0x8e05064c), SPH_C32(0x484c50ce), SPH_C32(0x267ef1f6),
	  SPH_C32(0x30c5d0f1) },
	{ SPH_C32(0x7d5c0050), SPH_C32(0xeb690000), SPH_C32(0x79460000),
	  SPH_C32(0x7ae10000), SPH_C32(0x9c3d44c6), SPH_C32(0x56c20912),
	  SPH_C32(0x4ba936b9), SPH_C32(0x6dbdf830), SPH_C32(0x7ad00350),
	  SPH_C32(0x4ce60000), SPH_C32(0x36330000), SPH_C32(0xc78a0000),
	  SPH_C32(0xf5ba13e7), SPH_C32(0xbbe659a8), SPH_C32(0x634589f0),
	  SPH_C32(0x9e7ab68b) },
	{ SPH_C32(0x0a1800a0), SPH_C32(0x1a330000), SPH_C32(0x8cf40000),
	  SPH_C32(0x4ef50000), SPH_C32(0x150a3a4a), SPH_C32(0x0c49e537),
	  SPH_C32(0x406afba7), SPH_C32(0xa28a8dfb), SPH_C32(0x8ebc0300),
	  SPH_C32(0xdafe0000), SPH_C32(0x22960000), SPH_C32(0xc4950000),
	  SPH_C32(0xb72e6d5f), SPH_C32(0xdd5927b1), SPH_C32(0xffe1f922),
	  SPH_C32(0x144ea3ff) },
	{ SPH_C32(0x89300000), SPH_C32(0x7d710000), SPH_C32(0x6de30000),
	  SPH_C32(0x79fe0000), SPH_C32(0xdea93a7e), SPH_C32(0x307d770b),
	  SPH_C32(0xd70d466b), SPH_C32(0xe789ed44), SPH_C32(0xf9f803f0),
	  SPH_C32(0x2ba40000), SPH_C32(0xd7240000), SPH_C32(0xf0810000),
	  SPH_C32(0x3e1913d3), SPH_C32(0x87d2cb94), SPH_C32(0xf422343c),
	  SPH_C32(0xdb79d634) },
	{ SPH_C32(0xfe7400f0), SPH_C32(0x8c2b0000), SPH_C32(0x98510000),
	  SPH_C32(0x4dea0000), SPH_C32(0x579e44f2), SPH_C32(0x6af69b2e),
	  SPH_C32(0xdcce8b75), SPH_C32(0x28be988f), SPH_C32(0x0d9403a0),
	  SPH_C32(0xbdbc0000), SPH_C32(0xc3810000), SPH_C32(0xf39e0000),
	  SPH_C32(0x7c8d6d6b), SPH_C32(0xe16db58d), SPH_C32(0x688644ee),
	  SPH_C32(0x514dc340) },
	{ SPH_C32(0x95db0120), SPH_C32(0x761b0000), SPH_C32(0x6b9d0000),
	  SPH_C32(0xaec30000), SPH_C32(0x6eb52fe1), SPH_C32(0xffe3ec51),
	  SPH_C32(0x055183a1), SPH_C32(0x0c35eb81), SPH_C32(0xcee70330),
	  SPH_C32(0x40aa0000), SPH_C32(0x60f10000), SPH_C32(0x9b240000),
	  SPH_C32(0x61ee0cd8), SPH_C32(0x80d8a18d), SPH_C32(0x7873d6cd),
	  SPH_C32(0x6daee1d3) },
	{ SPH_C32(0xe29f01d0), SPH_C32(0x87410000), SPH_C32(0x9e2f0000),
	  SPH_C32(0x9ad70000), SPH_C32(0xe782516d), SPH_C32(0xa5680074),
	  SPH_C32(0x0e924ebf), SPH_C32(0xc3029e4a), SPH_C32(0x3a8b0360),
	  SPH_C32(0xd6b20000), SPH_C32(0x74540000), SPH_C32(0x983b0000),
	  SPH_C32(0x237a7260), SPH_C32(0xe667df94), SPH_C32(0xe4d7a61f),
	  SPH_C32(0xe79af4a7) },
	{ SPH_C32(0x61b70170), SPH_C32(0xe0030000), SPH_C32(0x7f380000),
	  SPH_C32(0xaddc0000), SPH_C32(0x2c215159), SPH_C32(0x995c9248),
	  SPH_C32(0x99f5f373), SPH_C32(0x8601fef5), SPH_C32(0x4dcf0390),
	  SPH_C32(0x27e80000), SPH_C32(0x81e60000), SPH_C32(0xac2f0000),
	  SPH_C32(0xaa4d0cec), SPH_C32(0xbcec33b1), SPH_C32(0xef146b01),
	  SPH_C32(0x28ad816c) },
	{ SPH_C32(0x16f30180), SPH_C32(0x11590000), SPH_C32(0x8a8a0000),
	  SPH_C32(0x99c80000), SPH_C32(0xa5162fd5), SPH_C32(0xc3d77e6d),
	  SPH_C32(0x92363e6d), SPH_C32(0x49368b3e), SPH_C32(0xb9a303c0),
	  SPH_C32(0xb1f00000), SPH_C32(0x95430000), SPH_C32(0xaf300000),
	  SPH_C32(0xe8d97254), SPH_C32(0xda534da8), SPH_C32(0x73b01bd3),
	  SPH_C32(0xa2999418) },
	{ SPH_C32(0x26600240), SPH_C32(0xddd80000), SPH_C32(0x722a0000),
	  SPH_C32(0x4f060000), SPH_C32(0x936667ff), SPH_C32(0x29f944ce),
	  SPH_C32(0x368b63d5), SPH_C32(0x0c26f262), SPH_C32(0xef0b0270),
	  SPH_C32(0x3afd0000), SPH_C32(0x5dae0000), SPH_C32(0x69490000),
	  SPH_C32(0x9b0f3c06), SPH_C32(0x4405b5f9), SPH_C32(0x66140a51),
	  SPH_C32(0x924f5d0a) },
	{ SPH_C32(0x512402b0), SPH_C32(0x2c820000), SPH_C32(0x87980000),
	  SPH_C32(0x7b120000), SPH_C32(0x1a511973), SPH_C32(0x7372a8eb),
	  SPH_C32(0x3d48aecb), SPH_C32(0xc31187a9), SPH_C32(0x1b670220),
	  SPH_C32(0xace50000), SPH_C32(0x490b0000), SPH_C32(0x6a560000),
	  SPH_C32(0xd99b42be), SPH_C32(0x22bacbe0), SPH_C32(0xfab07a83),
	  SPH_C32(0x187b487e) },
	{ SPH_C32(0xd20c0210), SPH_C32(0x4bc00000), SPH_C32(0x668f0000),
	  SPH_C32(0x4c190000), SPH_C32(0xd1f21947), SPH_C32(0x4f463ad7),
	  SPH_C32(0xaa2f1307), SPH_C32(0x8612e716), SPH_C32(0x6c2302d0),
	  SPH_C32(0x5dbf0000), SPH_C32(0xbcb90000), SPH_C32(0x5e420000),
	  SPH_C32(0x50ac3c32), SPH_C32(0x783127c5), SPH_C32(0xf173b79d),
	  SPH_C32(0xd74c3db5) },
	{ SPH_C32(0xa54802e0), SPH_C32(0xba9a0000), SPH_C32(0x933d0000),
	  SPH_C32(0x780d0000), SPH_C32(0x58c567cb), SPH_C32(0x15cdd6f2),
	  SPH_C32(0xa1ecde19), SPH_C32(0x492592dd), SPH_C32(0x984f0280),
	  SPH_C32(0xcba70000), SPH_C32(0xa81c0000), SPH_C32(0x5d5d0000),
	  SPH_C32(0x1238428a), SPH_C32(0x1e8e59dc), SPH_C32(0x6dd7c74f),
	  SPH_C32(0x5d7828c1) },
	{ SPH_C32(0xcee70330), SPH_C32(0x40aa0000), SPH_C32(0x60f10000),
	  SPH_C32(0x9b240000), SPH_C32(0x61ee0cd8), SPH_C32(0x80d8a18d),
	  SPH_C32(0x7873d6cd), SPH_C32(0x6daee1d3), SPH_C32(0x5b3c0210),
	  SPH_C32(0x36b10000), SPH_C32(0x0b6c0000), SPH_C32(0x35e70000),
	  SPH_C32(0x0f5b2339), SPH_C32(0x7f3b4ddc), SPH_C32(0x7d22556c),
	  SPH_C32(0x619b0a52) },
	{ SPH_C32(0xb9a303c0), SPH_C32(0xb1f00000), SPH_C32(0x95430000),
	  SPH_C32(0xaf300000), SPH_C32(0xe8d97254), SPH_C32(0xda534da8),
	  SPH_C32(0x73b01bd3), SPH_C32(0xa2999418), SPH_C32(0xaf500240),
	  SPH_C32(0xa0a90000), SPH_C32(0x1fc90000), SPH_C32(0x36f80000),
	  SPH_C32(0x4dcf5d81), SPH_C32(0x198433c5), SPH_C32(0xe18625be),
	  SPH_C32(0xebaf1f26) },
	{ SPH_C32(0x3a8b0360), SPH_C32(0xd6b20000), SPH_C32(0x74540000),
	  SPH_C32(0x983b0000), SPH_C32(0x237a7260), SPH_C32(0xe667df94),
	  SPH_C32(0xe4d7a61f), SPH_C32(0xe79af4a7), SPH_C32(0xd81402b0),
	  SPH_C32(0x51f30000), SPH_C32(0xea7b0000), SPH_C32(0x02ec0000),
	  SPH_C32(0xc4f8230d), SPH_C32(0x430fdfe0), SPH_C32(0xea45e8a0),
	  SPH_C32(0x24986aed) },
	{ SPH_C32(0x4dcf0390), SPH_C32(0x27e80000), SPH_C32(0x81e60000),
	  SPH_C32(0xac2f0000), SPH_C32(0xaa4d0cec), SPH_C32(0xbcec33b1),
	  SPH_C32(0xef146b01), SPH_C32(0x28ad816c), SPH_C32(0x2c7802e0),
	  SPH_C32(0xc7eb0000), SPH_C32(0xfede0000), SPH_C32(0x01f30000),
	  SPH_C32(0x866c5db5), SPH_C32(0x25b0a1f9), SPH_C32(0x76e19872),
	  SPH_C32(0xaeac7f99) },
	{ SPH_C32(0x92570220), SPH_C32(0xd1940000), SPH_C32(0x24e80000),
	  SPH_C32(0x13a80000), SPH_C32(0x073278c0), SPH_C32(0x12c7bceb),
	  SPH_C32(0x2dbd3ce8), SPH_C32(0xfff2a53a), SPH_C32(0xb3bb0360),
	  SPH_C32(0xabc30000), SPH_C32(0x19b70000), SPH_C32(0xe1c50000),
	  SPH_C32(0xfdd3481e), SPH_C32(0xd61aa89f), SPH_C32(0x33dae074),
	  SPH_C32(0x001319e3) },
	{ SPH_C32(0xe51302d0), SPH_C32(0x20ce0000), SPH_C32(0xd15a0000),
	  SPH_C32(0x27bc0000), SPH_C32(0x8e05064c), SPH_C32(0x484c50ce),
	  SPH_C32(0x267ef1f6), SPH_C32(0x30c5d0f1), SPH_C32(0x47d70330),
	  SPH_C32(0x3ddb0000), SPH_C32(0x0d120000), SPH_C32(0xe2da0000),
	  SPH_C32(0xbf4736a6), SPH_C32(0xb0a5d686), SPH_C32(0xaf7e90a6),
	  SPH_C32(0x8a270c97) },
	{ SPH_C32(0x663b0270), SPH_C32(0x478c0000), SPH_C32(0x304d0000),
	  SPH_C32(0x10b70000), SPH_C32(0x45a60678), SPH_C32(0x7478c2f2),
	  SPH_C32(0xb1194c3a), SPH_C32(0x75c6b04e), SPH_C32(0x309303c0),
	  SPH_C32(0xcc810000), SPH_C32(0xf8a00000), SPH_C32(0xd6ce0000),
	  SPH_C32(0x3670482a), SPH_C32(0xea2e3aa3), SPH_C32(0xa4bd5db8),
	  SPH_C32(0x4510795c) },
	{ SPH_C32(0x117f0280), SPH_C32(0xb6d60000), SPH_C32(0xc5ff0000),
	  SPH_C32(0x24a30000), SPH_C32(0xcc9178f4), SPH_C32(0x2ef32ed7),
	  SPH_C32(0xbada8124), SPH_C32(0xbaf1c585), SPH_C32(0xc4ff0390),
	  SPH_C32(0x5a990000), SPH_C32(0xec050000), SPH_C32(0xd5d10000),
	  SPH_C32(0x74e43692), SPH_C32(0x8c9144ba), SPH_C32(0x38192d6a),
	  SPH_C32(0xcf246c28) },
	{ SPH_C32(0x7ad00350), SPH_C32(0x4ce60000), SPH_C32(0x36330000),
	  SPH_C32(0xc78a0000), SPH_C32(0xf5ba13e7), SPH_C32(0xbbe659a8),
	  SPH_C32(0x634589f0), SPH_C32(0x9e7ab68b), SPH_C32(0x078c0300),
	  SPH_C32(0xa78f0000), SPH_C32(0x4f750000), SPH_C32(0xbd6b0000),
	  SPH_C32(0x69875721), SPH_C32(0xed2450ba), SPH_C32(0x28ecbf49),
	  SPH_C32(0xf3c74ebb) },
	{ SPH_C32(0x0d9403a0), SPH_C32(0xbdbc0000), SPH_C32(0xc3810000),
	  SPH_C32(0xf39e0000), SPH_C32(0x7c8d6d6b), SPH_C32(0xe16db58d),
	  SPH_C32(0x688644ee), SPH_C32(0x514dc340), SPH_C32(0xf3e00350),
	  SPH_C32(0x31970000), SPH_C32(0x5bd00000), SPH_C32(0xbe740000),
	  SPH_C32(0x2b132999), SPH_C32(0x8b9b2ea3), SPH_C32(0xb448cf9b),
	  SPH_C32(0x79f35bcf) },
	{ SPH_C32(0x8ebc0300), SPH_C32(0xdafe0000), SPH_C32(0x22960000),
	  SPH_C32(0xc4950000), SPH_C32(0xb72e6d5f), SPH_C32(0xdd5927b1),
	  SPH_C32(0xffe1f922), SPH_C32(0x144ea3ff), SPH_C32(0x84a403a0),
	  SPH_C32(0xc0cd0000), SPH_C32(0xae620000), SPH_C32(0x8a600000),
	  SPH_C32(0xa2245715), SPH_C32(0xd110c286), SPH_C32(0xbf8b0285),
	  SPH_C32(0xb6c42e04) },
	{ SPH_C32(0xf9f803f0), SPH_C32(0x2ba40000), SPH_C32(0xd7240000),
	  SPH_C32(0xf0810000), SPH_C32(0x3e1913d3), SPH_C32(0x87d2cb94),
	  SPH_C32(0xf422343c), SPH_C32(0xdb79d634), SPH_C32(0x70c803f0),
	  SPH_C32(0x56d50000), SPH_C32(0xbac70000), SPH_C32(0x897f0000),
	  SPH_C32(0xe0b029ad), SPH_C32(0xb7afbc9f), SPH_C32(0x232f7257),
	  SPH_C32(0x3cf03b70) }
};

static const sph_u32 T512_12[64][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x75c90003), SPH_C32(0x0e10c000), SPH_C32(0xd1200000),
	  SPH_C32(0xbaea0000), SPH_C32(0x8bc42f3e), SPH_C32(0x8758b757),
	  SPH_C32(0xbb28761d), SPH_C32(0x00b72e2b), SPH_C32(0xeecf0001),
	  SPH_C32(0x6f564000), SPH_C32(0xf33e0000), SPH_C32(0xa79e0000),
	  SPH_C32(0xbdb57219), SPH_C32(0xb711ebc5), SPH_C32(0x4a3b40ba),
	  SPH_C32(0xfeabf254) },
	{ SPH_C32(0xeecf0001), SPH_C32(0x6f564000), SPH_C32(0xf33e0000),
	  SPH_C32(0xa79e0000), SPH_C32(0xbdb57219), SPH_C32(0xb711ebc5),
	  SPH_C32(0x4a3b40ba), SPH_C32(0xfeabf254), SPH_C32(0x9b060002),
	  SPH_C32(0x61468000), SPH_C32(0x221e0000), SPH_C32(0x1d740000),
	  SPH_C32(0x36715d27), SPH_C32(0x30495c92), SPH_C32(0xf11336a7),
	  SPH_C32(0xfe1cdc7f) },
	{ SPH_C32(0x9b060002), SPH_C32(0x61468000), SPH_C32(0x221e0000),
	  SPH_C32(0x1d740000), SPH_C32(0x36715d27), SPH_C32(0x30495c92),
	  SPH_C32(0xf11336a7), SPH_C32(0xfe1cdc7f), SPH_C32(0x75c90003),
	  SPH_C32(0x0e10c000), SPH_C32(0xd1200000), SPH_C32(0xbaea0000),
	  SPH_C32(0x8bc42f3e), SPH_C32(0x8758b757), SPH_C32(0xbb28761d),
	  SPH_C32(0x00b72e2b) },
	{ SPH_C32(0xf6800005), SPH_C32(0x3443c000), SPH_C32(0x24070000),
	  SPH_C32(0x8f3d0000), SPH_C32(0x21373bfb), SPH_C32(0x0ab8d5ae),
	  SPH_C32(0xcdc58b19), SPH_C32(0xd795ba31), SPH_C32(0xa67f0001),
	  SPH_C32(0x71378000), SPH_C32(0x19fc0000), SPH_C32(0x96db0000),
	  SPH_C32(0x3a8b6dfd), SPH_C32(0xebcaaef3), SPH_C32(0x2c6d478f),
	  SPH_C32(0xac8e6c88) },
	{ SPH_C32(0x83490006), SPH_C32(0x3a530000), SPH_C32(0xf5270000),
	  SPH_C32(0x35d70000), SPH_C32(0xaaf314c5), SPH_C32(0x8de062f9),
	  SPH_C32(0x76edfd04), SPH_C32(0xd722941a), SPH_C32(0x48b00000),
	  SPH_C32(0x1e61c000), SPH_C32(0xeac20000), SPH_C32(0x31450000),
	  SPH_C32(0x873e1fe4), SPH_C32(0x5cdb4536), SPH_C32(0x66560735),
	  SPH_C32(0x52259edc) },
	{ SPH_C32(0x184f0004), SPH_C32(0x5b158000), SPH_C32(0xd7390000),
	  SPH_C32(0x28a30000), SPH_C32(0x9c8249e2), SPH_C32(0xbda93e6b),
	  SPH_C32(0x87fecba3), SPH_C32(0x293e4865), SPH_C32(0x3d790003),
	  SPH_C32(0x10710000), SPH_C32(0x3be20000), SPH_C32(0x8baf0000),
	  SPH_C32(0x0cfa30da), SPH_C32(0xdb83f261), SPH_C32(0xdd7e7128),
	  SPH_C32(0x5292b0f7) },
	{ SPH_C32(0x6d860007), SPH_C32(0x55054000), SPH_C32(0x06190000),
	  SPH_C32(0x92490000), SPH_C32(0x174666dc), SPH_C32(0x3af1893c),
	  SPH_C32(0x3cd6bdbe), SPH_C32(0x2989664e), SPH_C32(0xd3b60002),
	  SPH_C32(0x7f274000), SPH_C32(0xc8dc0000), SPH_C32(0x2c310000),
	  SPH_C32(0xb14f42c3), SPH_C32(0x6c9219a4), SPH_C32(0x97453192),
	  SPH_C32(0xac3942a3) },
	{ SPH_C32(0xa67f0001), SPH_C32(0x71378000), SPH_C32(0x19fc0000),
	  SPH_C32(0x96db0000), SPH_C32(0x3a8b6dfd), SPH_C32(0xebcaaef3),
	  SPH_C32(0x2c6d478f), SPH_C32(0xac8e6c88), SPH_C32(0x50ff0004),
	  SPH_C32(0x45744000), SPH_C32(0x3dfb0000), SPH_C32(0x19e60000),
	  SPH_C32(0x1bbc5606), SPH_C32(0xe1727b5d), SPH_C32(0xe1a8cc96),
	  SPH_C32(0x7b1bd6b9) },
	{ SPH_C32(0xd3b60002), SPH_C32(0x7f274000), SPH_C32(0xc8dc0000),
	  SPH_C32(0x2c310000), SPH_C32(0xb14f42c3), SPH_C32(0x6c9219a4),
	  SPH_C32(0x97453192), SPH_C32(0xac3942a3), SPH_C32(0xbe300005),
	  SPH_C32(0x2a220000), SPH_C32(0xcec50000), SPH_C32(0xbe780000),
	  SPH_C32(0xa609241f), SPH_C32(0x56639098), SPH_C32(0xab938c2c),
	  SPH_C32(0x85b024ed) },
	{ SPH_C32(0x48b00000), SPH_C32(0x1e61c000), SPH_C32(0xeac20000),
	  SPH_C32(0x31450000), SPH_C32(0x873e1fe4), SPH_C32(0x5cdb4536),
	  SPH_C32(0x66560735), SPH_C32(0x52259edc), SPH_C32(0xcbf90006),
	  SPH_C32(0x2432c000), SPH_C32(0x1fe50000), SPH_C32(0x04920000),
	  SPH_C32(0x2dcd0b21), SPH_C32(0xd13b27cf), SPH_C32(0x10bbfa31),
	  SPH_C32(0x85070ac6) },
	{ SPH_C32(0x3d790003), SPH_C32(0x10710000), SPH_C32(0x3be20000),
	  SPH_C32(0x8baf0000), SPH_C32(0x0cfa30da), SPH_C32(0xdb83f261),
	  SPH_C32(0xdd7e7128), SPH_C32(0x5292b0f7), SPH_C32(0x25360007),
	  SPH_C32(0x4b648000), SPH_C32(0xecdb0000), SPH_C32(0xa30c0000),
	  SPH_C32(0x90787938), SPH_C32(0x662acc0a), SPH_C32(0x5a80ba8b),
	  SPH_C32(0x7bacf892) },
	{ SPH_C32(0x50ff0004), SPH_C32(0x45744000), SPH_C32(0x3dfb0000),
	  SPH_C32(0x19e60000), SPH_C32(0x1bbc5606), SPH_C32(0xe1727b5d),
	  SPH_C32(0xe1a8cc96), SPH_C32(0x7b1bd6b9), SPH_C32(0xf6800005),
	  SPH_C32(0x3443c000), SPH_C32(0x24070000), SPH_C32(0x8f3d0000),
	  SPH_C32(0x21373bfb), SPH_C32(0x0ab8d5ae), SPH_C32(0xcdc58b19),
	  SPH_C32(0xd795ba31) },
	{ SPH_C32(0x25360007), SPH_C32(0x4b648000), SPH_C32(0xecdb0000),
	  SPH_C32(0xa30c0000), SPH_C32(0x90787938), SPH_C32(0x662acc0a),
	  SPH_C32(0x5a80ba8b), SPH_C32(0x7bacf892), SPH_C32(0x184f0004),
	  SPH_C32(0x5b158000), SPH_C32(0xd7390000), SPH_C32(0x28a30000),
	  SPH_C32(0x9c8249e2), SPH_C32(0xbda93e6b), SPH_C32(0x87fecba3),
	  SPH_C32(0x293e4865) },
	{ SPH_C32(0xbe300005), SPH_C32(0x2a220000), SPH_C32(0xcec50000),
	  SPH_C32(0xbe780000), SPH_C32(0xa609241f), SPH_C32(0x56639098),
	  SPH_C32(0xab938c2c), SPH_C32(0x85b024ed), SPH_C32(0x6d860007),
	  SPH_C32(0x55054000), SPH_C32(0x06190000), SPH_C32(0x92490000),
	  SPH_C32(0x174666dc), SPH_C32(0x3af1893c), SPH_C32(0x3cd6bdbe),
	  SPH_C32(0x2989664e) },
	{ SPH_C32(0xcbf90006), SPH_C32(0x2432c000), SPH_C32(0x1fe50000),
	  SPH_C32(0x04920000), SPH_C32(0x2dcd0b21), SPH_C32(0xd13b27cf),
	  SPH_C32(0x10bbfa31), SPH_C32(0x85070ac6), SPH_C32(0x83490006),
	  SPH_C32(0x3a530000), SPH_C32(0xf5270000), SPH_C32(0x35d70000),
	  SPH_C32(0xaaf314c5), SPH_C32(0x8de062f9), SPH_C32(0x76edfd04),
	  SPH_C32(0xd722941a) },
	{ SPH_C32(0xf7750009), SPH_C32(0xcf3cc000), SPH_C32(0xc3d60000),
	  SPH_C32(0x04920000), SPH_C32(0x029519a9), SPH_C32(0xf8e836ba),
	  SPH_C32(0x7a87f14e), SPH_C32(0x9e16981a), SPH_C32(0xd46a0000),
	  SPH_C32(0x8dc8c000), SPH_C32(0xa5af0000), SPH_C32(0x4a290000),
	  SPH_C32(0xfc4e427a), SPH_C32(0xc9b4866c), SPH_C32(0x98369604),
	  SPH_C32(0xf746c320) },
	{ SPH_C32(0x82bc000a), SPH_C32(0xc12c0000), SPH_C32(0x12f60000),
	  SPH_C32(0xbe780000), SPH_C32(0x89513697), SPH_C32(0x7fb081ed),
	  SPH_C32(0xc1af8753), SPH_C32(0x9ea1b631), SPH_C32(0x3aa50001),
	  SPH_C32(0xe29e8000), SPH_C32(0x56910000), SPH_C32(0xedb70000),
	  SPH_C32(0x41fb3063), SPH_C32(0x7ea56da9), SPH_C32(0xd20dd6be),
	  SPH_C32(0x09ed3174) },
	{ SPH_C32(0x19ba0008), SPH_C32(0xa06a8000), SPH_C32(0x30e80000),
	  SPH_C32(0xa30c0000), SPH_C32(0xbf206bb0), SPH_C32(0x4ff9dd7f),
	  SPH_C32(0x30bcb1f4), SPH_C32(0x60bd6a4e), SPH_C32(0x4f6c0002),
	  SPH_C32(0xec8e4000), SPH_C32(0x87b10000), SPH_C32(0x575d0000),
	  SPH_C32(0xca3f1f5d), SPH_C32(0xf9fddafe), SPH_C32(0x6925a0a3),
	  SPH_C32(0x095a1f5f) },
	{ SPH_C32(0x6c73000b), SPH_C32(0xae7a4000), SPH_C32(0xe1c80000),
	  SPH_C32(0x19e60000), SPH_C32(0x34e4448e), SPH_C32(0xc8a16a28),
	  SPH_C32(0x8b94c7e9), SPH_C32(0x600a4465), SPH_C32(0xa1a30003),
	  SPH_C32(0x83d80000), SPH_C32(0x748f0000), SPH_C32(0xf0c30000),
	  SPH_C32(0x778a6d44), SPH_C32(0x4eec313b), SPH_C32(0x231ee019),
	  SPH_C32(0xf7f1ed0b) },
	{ SPH_C32(0x01f5000c), SPH_C32(0xfb7f0000), SPH_C32(0xe7d10000),
	  SPH_C32(0x8baf0000), SPH_C32(0x23a22252), SPH_C32(0xf250e314),
	  SPH_C32(0xb7427a57), SPH_C32(0x4983222b), SPH_C32(0x72150001),
	  SPH_C32(0xfcff4000), SPH_C32(0xbc530000), SPH_C32(0xdcf20000),
	  SPH_C32(0xc6c52f87), SPH_C32(0x227e289f), SPH_C32(0xb45bd18b),
	  SPH_C32(0x5bc8afa8) },
	{ SPH_C32(0x743c000f), SPH_C32(0xf56fc000), SPH_C32(0x36f10000),
	  SPH_C32(0x31450000), SPH_C32(0xa8660d6c), SPH_C32(0x75085443),
	  SPH_C32(0x0c6a0c4a), SPH_C32(0x49340c00), SPH_C32(0x9cda0000),
	  SPH_C32(0x93a90000), SPH_C32(0x4f6d0000), SPH_C32(0x7b6c0000),
	  SPH_C32(0x7b705d9e), SPH_C32(0x956fc35a), SPH_C32(0xfe609131),
	  SPH_C32(0xa5635dfc) },
	{ SPH_C32(0xef3a000d), SPH_C32(0x94294000), SPH_C32(0x14ef0000),
	  SPH_C32(0x2c310000), SPH_C32(0x9e17504b), SPH_C32(0x454108d1),
	  SPH_C32(0xfd793aed), SPH_C32(0xb728d07f), SPH_C32(0xe9130003),
	  SPH_C32(0x9db9c000), SPH_C32(0x9e4d0000), SPH_C32(0xc1860000),
	  SPH_C32(0xf0b472a0), SPH_C32(0x1237740d), SPH_C32(0x4548e72c),
	  SPH_C32(0xa5d473d7) },
	{ SPH_C32(0x9af3000e), SPH_C32(0x9a398000), SPH_C32(0xc5cf0000),
	  SPH_C32(0x96db0000), SPH_C32(0x15d37f75), SPH_C32(0xc219bf86),
	  SPH_C32(0x46514cf0), SPH_C32(0xb79ffe54), SPH_C32(0x07dc0002),
	  SPH_C32(0xf2ef8000), SPH_C32(0x6d730000), SPH_C32(0x66180000),
	  SPH_C32(0x4d0100b9), SPH_C32(0xa5269fc8), SPH_C32(0x0f73a796),
	  SPH_C32(0x5b7f8183) },
	{ SPH_C32(0x510a0008), SPH_C32(0xbe0b4000), SPH_C32(0xda2a0000),
	  SPH_C32(0x92490000), SPH_C32(0x381e7454), SPH_C32(0x13229849),
	  SPH_C32(0x56eab6c1), SPH_C32(0x3298f492), SPH_C32(0x84950004),
	  SPH_C32(0xc8bc8000), SPH_C32(0x98540000), SPH_C32(0x53cf0000),
	  SPH_C32(0xe7f2147c), SPH_C32(0x28c6fd31), SPH_C32(0x799e5a92),
	  SPH_C32(0x8c5d1599) },
	{ SPH_C32(0x24c3000b), SPH_C32(0xb01b8000), SPH_C32(0x0b0a0000),
	  SPH_C32(0x28a30000), SPH_C32(0xb3da5b6a), SPH_C32(0x947a2f1e),
	  SPH_C32(0xedc2c0dc), SPH_C32(0x322fdab9), SPH_C32(0x6a5a0005),
	  SPH_C32(0xa7eac000), SPH_C32(0x6b6a0000), SPH_C32(0xf4510000),
	  SPH_C32(0x5a476665), SPH_C32(0x9fd716f4), SPH_C32(0x33a51a28),
	  SPH_C32(0x72f6e7cd) },
	{ SPH_C32(0xbfc50009), SPH_C32(0xd15d0000), SPH_C32(0x29140000),
	  SPH_C32(0x35d70000), SPH_C32(0x85ab064d), SPH_C32(0xa433738c),
	  SPH_C32(0x1cd1f67b), SPH_C32(0xcc3306c6), SPH_C32(0x1f930006),
	  SPH_C32(0xa9fa0000), SPH_C32(0xba4a0000), SPH_C32(0x4ebb0000),
	  SPH_C32(0xd183495b), SPH_C32(0x188fa1a3), SPH_C32(0x888d6c35),
	  SPH_C32(0x7241c9e6) },
	{ SPH_C32(0xca0c000a), SPH_C32(0xdf4dc000), SPH_C32(0xf8340000),
	  SPH_C32(0x8f3d0000), SPH_C32(0x0e6f2973), SPH_C32(0x236bc4db),
	  SPH_C32(0xa7f98066), SPH_C32(0xcc8428ed), SPH_C32(0xf15c0007),
	  SPH_C32(0xc6ac4000), SPH_C32(0x49740000), SPH_C32(0xe9250000),
	  SPH_C32(0x6c363b42), SPH_C32(0xaf9e4a66), SPH_C32(0xc2b62c8f),
	  SPH_C32(0x8cea3bb2) },
	{ SPH_C32(0xa78a000d), SPH_C32(0x8a488000), SPH_C32(0xfe2d0000),
	  SPH_C32(0x1d740000), SPH_C32(0x19294faf), SPH_C32(0x199a4de7),
	  SPH_C32(0x9b2f3dd8), SPH_C32(0xe50d4ea3), SPH_C32(0x22ea0005),
	  SPH_C32(0xb98b0000), SPH_C32(0x81a80000), SPH_C32(0xc5140000),
	  SPH_C32(0xdd797981), SPH_C32(0xc30c53c2), SPH_C32(0x55f31d1d),
	  SPH_C32(0x20d37911) },
	{ SPH_C32(0xd243000e), SPH_C32(0x84584000), SPH_C32(0x2f0d0000),
	  SPH_C32(0xa79e0000), SPH_C32(0x92ed6091), SPH_C32(0x9ec2fab0),
	  SPH_C32(0x20074bc5), SPH_C32(0xe5ba6088), SPH_C32(0xcc250004),
	  SPH_C32(0xd6dd4000), SPH_C32(0x72960000), SPH_C32(0x628a0000),
	  SPH_C32(0x60cc0b98), SPH_C32(0x741db807), SPH_C32(0x1fc85da7),
	  SPH_C32(0xde788b45) },
	{ SPH_C32(0x4945000c), SPH_C32(0xe51ec000), SPH_C32(0x0d130000),
	  SPH_C32(0xbaea0000), SPH_C32(0xa49c3db6), SPH_C32(0xae8ba622),
	  SPH_C32(0xd1147d62), SPH_C32(0x1ba6bcf7), SPH_C32(0xb9ec0007),
	  SPH_C32(0xd8cd8000), SPH_C32(0xa3b60000), SPH_C32(0xd8600000),
	  SPH_C32(0xeb0824a6), SPH_C32(0xf3450f50), SPH_C32(0xa4e02bba),
	  SPH_C32(0xdecfa56e) },
	{ SPH_C32(0x3c8c000f), SPH_C32(0xeb0e0000), SPH_C32(0xdc330000),
	  SPH_C32(0x00000000), SPH_C32(0x2f581288), SPH_C32(0x29d31175),
	  SPH_C32(0x6a3c0b7f), SPH_C32(0x1b1192dc), SPH_C32(0x57230006),
	  SPH_C32(0xb79bc000), SPH_C32(0x50880000), SPH_C32(0x7ffe0000),
	  SPH_C32(0x56bd56bf), SPH_C32(0x4454e495), SPH_C32(0xeedb6b00),
	  SPH_C32(0x2064573a) },
	{ SPH_C32(0xd46a0000), SPH_C32(0x8dc8c000), SPH_C32(0xa5af0000),
	  SPH_C32(0x4a290000), SPH_C32(0xfc4e427a), SPH_C32(0xc9b4866c),
	  SPH_C32(0x98369604), SPH_C32(0xf746c320), SPH_C32(0x231f0009),
	  SPH_C32(0x42f40000), SPH_C32(0x66790000), SPH_C32(0x4ebb0000),
	  SPH_C32(0xfedb5bd3), SPH_C32(0x315cb0d6), SPH_C32(0xe2b1674a),
	  SPH_C32(0x69505b3a) },
	{ SPH_C32(0xa1a30003), SPH_C32(0x83d80000), SPH_C32(0x748f0000),
	  SPH_C32(0xf0c30000), SPH_C32(0x778a6d44), SPH_C32(0x4eec313b),
	  SPH_C32(0x231ee019), SPH_C32(0xf7f1ed0b), SPH_C32(0xcdd00008),
	  SPH_C32(0x2da24000), SPH_C32(0x95470000), SPH_C32(0xe9250000),
	  SPH_C32(0x436e29ca), SPH_C32(0x864d5b13), SPH_C32(0xa88a27f0),
	  SPH_C32(0x97fba96e) },
	{ SPH_C32(0x3aa50001), SPH_C32(0xe29e8000), SPH_C32(0x56910000),
	  SPH_C32(0xedb70000), SPH_C32(0x41fb3063), SPH_C32(0x7ea56da9),
	  SPH_C32(0xd20dd6be), SPH_C32(0x09ed3174), SPH_C32(0xb819000b),
	  SPH_C32(0x23b28000), SPH_C32(0x44670000), SPH_C32(0x53cf0000),
	  SPH_C32(0xc8aa06f4), SPH_C32(0x0115ec44), SPH_C32(0x13a251ed),
	  SPH_C32(0x974c8745) },
	{ SPH_C32(0x4f6c0002), SPH_C32(0xec8e4000), SPH_C32(0x87b10000),
	  SPH_C32(0x575d0000), SPH_C32(0xca3f1f5d), SPH_C32(0xf9fddafe),
	  SPH_C32(0x6925a0a3), SPH_C32(0x095a1f5f), SPH_C32(0x56d6000a),
	  SPH_C32(0x4ce4c000), SPH_C32(0xb7590000), SPH_C32(0xf4510000),
	  SPH_C32(0x751f74ed), SPH_C32(0xb6040781), SPH_C32(0x59991157),
	  SPH_C32(0x69e77511) },
	{ SPH_C32(0x22ea0005), SPH_C32(0xb98b0000), SPH_C32(0x81a80000),
	  SPH_C32(0xc5140000), SPH_C32(0xdd797981), SPH_C32(0xc30c53c2),
	  SPH_C32(0x55f31d1d), SPH_C32(0x20d37911), SPH_C32(0x85600008),
	  SPH_C32(0x33c38000), SPH_C32(0x7f850000), SPH_C32(0xd8600000),
	  SPH_C32(0xc450362e), SPH_C32(0xda961e25), SPH_C32(0xcedc20c5),
	  SPH_C32(0xc5de37b2) },
	{ SPH_C32(0x57230006), SPH_C32(0xb79bc000), SPH_C32(0x50880000),
	  SPH_C32(0x7ffe0000), SPH_C32(0x56bd56bf), SPH_C32(0x4454e495),
	  SPH_C32(0xeedb6b00), SPH_C32(0x2064573a), SPH_C32(0x6baf0009),
	  SPH_C32(0x5c95c000), SPH_C32(0x8cbb0000), SPH_C32(0x7ffe0000),
	  SPH_C32(0x79e54437), SPH_C32(0x6d87f5e0), SPH_C32(0x84e7607f),
	  SPH_C32(0x3b75c5e6) },
	{ SPH_C32(0xcc250004), SPH_C32(0xd6dd4000), SPH_C32(0x72960000),
	  SPH_C32(0x628a0000), SPH_C32(0x60cc0b98), SPH_C32(0x741db807),
	  SPH_C32(0x1fc85da7), SPH_C32(0xde788b45), SPH_C32(0x1e66000a),
	  SPH_C32(0x52850000), SPH_C32(0x5d9b0000), SPH_C32(0xc5140000),
	  SPH_C32(0xf2216b09), SPH_C32(0xeadf42b7), SPH_C32(0x3fcf1662),
	  SPH_C32(0x3bc2ebcd) },
	{ SPH_C32(0xb9ec0007), SPH_C32(0xd8cd8000), SPH_C32(0xa3b60000),
	  SPH_C32(0xd8600000), SPH_C32(0xeb0824a6), SPH_C32(0xf3450f50),
	  SPH_C32(0xa4e02bba), SPH_C32(0xdecfa56e), SPH_C32(0xf0a9000b),
	  SPH_C32(0x3dd34000), SPH_C32(0xaea50000), SPH_C32(0x628a0000),
	  SPH_C32(0x4f941910), SPH_C32(0x5dcea972), SPH_C32(0x75f456d8),
	  SPH_C32(0xc5691999) },
	{ SPH_C32(0x72150001), SPH_C32(0xfcff4000), SPH_C32(0xbc530000),
	  SPH_C32(0xdcf20000), SPH_C32(0xc6c52f87), SPH_C32(0x227e289f),
	  SPH_C32(0xb45bd18b), SPH_C32(0x5bc8afa8), SPH_C32(0x73e0000d),
	  SPH_C32(0x07804000), SPH_C32(0x5b820000), SPH_C32(0x575d0000),
	  SPH_C32(0xe5670dd5), SPH_C32(0xd02ecb8b), SPH_C32(0x0319abdc),
	  SPH_C32(0x124b8d83) },
	{ SPH_C32(0x07dc0002), SPH_C32(0xf2ef8000), SPH_C32(0x6d730000),
	  SPH_C32(0x66180000), SPH_C32(0x4d0100b9), SPH_C32(0xa5269fc8),
	  SPH_C32(0x0f73a796), SPH_C32(0x5b7f8183), SPH_C32(0x9d2f000c),
	  SPH_C32(0x68d60000), SPH_C32(0xa8bc0000), SPH_C32(0xf0c30000),
	  SPH_C32(0x58d27fcc), SPH_C32(0x673f204e), SPH_C32(0x4922eb66),
	  SPH_C32(0xece07fd7) },
	{ SPH_C32(0x9cda0000), SPH_C32(0x93a90000), SPH_C32(0x4f6d0000),
	  SPH_C32(0x7b6c0000), SPH_C32(0x7b705d9e), SPH_C32(0x956fc35a),
	  SPH_C32(0xfe609131), SPH_C32(0xa5635dfc), SPH_C32(0xe8e6000f),
	  SPH_C32(0x66c6c000), SPH_C32(0x799c0000), SPH_C32(0x4a290000),
	  SPH_C32(0xd31650f2), SPH_C32(0xe0679719), SPH_C32(0xf20a9d7b),
	  SPH_C32(0xec5751fc) },
	{ SPH_C32(0xe9130003), SPH_C32(0x9db9c000), SPH_C32(0x9e4d0000),
	  SPH_C32(0xc1860000), SPH_C32(0xf0b472a0), SPH_C32(0x1237740d),
	  SPH_C32(0x4548e72c), SPH_C32(0xa5d473d7), SPH_C32(0x0629000e),
	  SPH_C32(0x09908000), SPH_C32(0x8aa20000), SPH_C32(0xedb70000),
	  SPH_C32(0x6ea322eb), SPH_C32(0x57767cdc), SPH_C32(0xb831ddc1),
	  SPH_C32(0x12fca3a8) },
	{ SPH_C32(0x84950004), SPH_C32(0xc8bc8000), SPH_C32(0x98540000),
	  SPH_C32(0x53cf0000), SPH_C32(0xe7f2147c), SPH_C32(0x28c6fd31),
	  SPH_C32(0x799e5a92), SPH_C32(0x8c5d1599), SPH_C32(0xd59f000c),
	  SPH_C32(0x76b7c000), SPH_C32(0x427e0000), SPH_C32(0xc1860000),
	  SPH_C32(0xdfec6028), SPH_C32(0x3be46578), SPH_C32(0x2f74ec53),
	  SPH_C32(0xbec5e10b) },
	{ SPH_C32(0xf15c0007), SPH_C32(0xc6ac4000), SPH_C32(0x49740000),
	  SPH_C32(0xe9250000), SPH_C32(0x6c363b42), SPH_C32(0xaf9e4a66),
	  SPH_C32(0xc2b62c8f), SPH_C32(0x8cea3bb2), SPH_C32(0x3b50000d),
	  SPH_C32(0x19e18000), SPH_C32(0xb1400000), SPH_C32(0x66180000),
	  SPH_C32(0x62591231), SPH_C32(0x8cf58ebd), SPH_C32(0x654face9),
	  SPH_C32(0x406e135f) },
	{ SPH_C32(0x6a5a0005), SPH_C32(0xa7eac000), SPH_C32(0x6b6a0000),
	  SPH_C32(0xf4510000), SPH_C32(0x5a476665), SPH_C32(0x9fd716f4),
	  SPH_C32(0x33a51a28), SPH_C32(0x72f6e7cd), SPH_C32(0x4e99000e),
	  SPH_C32(0x17f14000), SPH_C32(0x60600000), SPH_C32(0xdcf20000),
	  SPH_C32(0xe99d3d0f), SPH_C32(0x0bad39ea), SPH_C32(0xde67daf4),
	  SPH_C32(0x40d93d74) },
	{ SPH_C32(0x1f930006), SPH_C32(0xa9fa0000), SPH_C32(0xba4a0000),
	  SPH_C32(0x4ebb0000), SPH_C32(0xd183495b), SPH_C32(0x188fa1a3),
	  SPH_C32(0x888d6c35), SPH_C32(0x7241c9e6), SPH_C32(0xa056000f),
	  SPH_C32(0x78a70000), SPH_C32(0x935e0000), SPH_C32(0x7b6c0000),
	  SPH_C32(0x54284f16), SPH_C32(0xbcbcd22f), SPH_C32(0x945c9a4e),
	  SPH_C32(0xbe72cf20) },
	{ SPH_C32(0x231f0009), SPH_C32(0x42f40000), SPH_C32(0x66790000),
	  SPH_C32(0x4ebb0000), SPH_C32(0xfedb5bd3), SPH_C32(0x315cb0d6),
	  SPH_C32(0xe2b1674a), SPH_C32(0x69505b3a), SPH_C32(0xf7750009),
	  SPH_C32(0xcf3cc000), SPH_C32(0xc3d60000), SPH_C32(0x04920000),
	  SPH_C32(0x029519a9), SPH_C32(0xf8e836ba), SPH_C32(0x7a87f14e),
	  SPH_C32(0x9e16981a) },
	{ SPH_C32(0x56d6000a), SPH_C32(0x4ce4c000), SPH_C32(0xb7590000),
	  SPH_C32(0xf4510000), SPH_C32(0x751f74ed), SPH_C32(0xb6040781),
	  SPH_C32(0x59991157), SPH_C32(0x69e77511), SPH_C32(0x19ba0008),
	  SPH_C32(0xa06a8000), SPH_C32(0x30e80000), SPH_C32(0xa30c0000),
	  SPH_C32(0xbf206bb0), SPH_C32(0x4ff9dd7f), SPH_C32(0x30bcb1f4),
	  SPH_C32(0x60bd6a4e) },
	{ SPH_C32(0xcdd00008), SPH_C32(0x2da24000), SPH_C32(0x95470000),
	  SPH_C32(0xe9250000), SPH_C32(0x436e29ca), SPH_C32(0x864d5b13),
	  SPH_C32(0xa88a27f0), SPH_C32(0x97fba96e), SPH_C32(0x6c73000b),
	  SPH_C32(0xae7a4000), SPH_C32(0xe1c80000), SPH_C32(0x19e60000),
	  SPH_C32(0x34e4448e), SPH_C32(0xc8a16a28), SPH_C32(0x8b94c7e9),
	  SPH_C32(0x600a4465) },
	{ SPH_C32(0xb819000b), SPH_C32(0x23b28000), SPH_C32(0x44670000),
	  SPH_C32(0x53cf0000), SPH_C32(0xc8aa06f4), SPH_C32(0x0115ec44),
	  SPH_C32(0x13a251ed), SPH_C32(0x974c8745), SPH_C32(0x82bc000a),
	  SPH_C32(0xc12c0000), SPH_C32(0x12f60000), SPH_C32(0xbe780000),
	  SPH_C32(0x89513697), SPH_C32(0x7fb081ed), SPH_C32(0xc1af8753),
	  SPH_C32(0x9ea1b631) },
	{ SPH_C32(0xd59f000c), SPH_C32(0x76b7c000), SPH_C32(0x427e0000),
	  SPH_C32(0xc1860000), SPH_C32(0xdfec6028), SPH_C32(0x3be46578),
	  SPH_C32(0x2f74ec53), SPH_C32(0xbec5e10b), SPH_C32(0x510a0008),
	  SPH_C32(0xbe0b4000), SPH_C32(0xda2a0000), SPH_C32(0x92490000),
	  SPH_C32(0x381e7454), SPH_C32(0x13229849), SPH_C32(0x56eab6c1),
	  SPH_C32(0x3298f492) },
	{ SPH_C32(0xa056000f), SPH_C32(0x78a70000), SPH_C32(0x935e0000),
	  SPH_C32(0x7b6c0000), SPH_C32(0x54284f16), SPH_C32(0xbcbcd22f),
	  SPH_C32(0x945c9a4e), SPH_C32(0xbe72cf20), SPH_C32(0xbfc50009),
	  SPH_C32(0xd15d0000), SPH_C32(0x29140000), SPH_C32(0x35d70000),
	  SPH_C32(0x85ab064d), SPH_C32(0xa433738c), SPH_C32(0x1cd1f67b),
	  SPH_C32(0xcc3306c6) },
	{ SPH_C32(0x3b50000d), SPH_C32(0x19e18000), SPH_C32(0xb1400000),
	  SPH_C32(0x66180000), SPH_C32(0x62591231), SPH_C32(0x8cf58ebd),
	  SPH_C32(0x654face9), SPH_C32(0x406e135f), SPH_C32(0xca0c000a),
	  SPH_C32(0xdf4dc000), SPH_C32(0xf8340000), SPH_C32(0x8f3d0000),
	  SPH_C32(0x0e6f2973), SPH_C32(0x236bc4db), SPH_C32(0xa7f98066),
	  SPH_C32(0xcc8428ed) },
	{ SPH_C32(0x4e99000e), SPH_C32(0x17f14000), SPH_C32(0x60600000),
	  SPH_C32(0xdcf20000), SPH_C32(0xe99d3d0f), SPH_C32(0x0bad39ea),
	  SPH_C32(0xde67daf4), SPH_C32(0x40d93d74), SPH_C32(0x24c3000b),
	  SPH_C32(0xb01b8000), SPH_C32(0x0b0a0000), SPH_C32(0x28a30000),
	  SPH_C32(0xb3da5b6a), SPH_C32(0x947a2f1e), SPH_C32(0xedc2c0dc),
	  SPH_C32(0x322fdab9) },
	{ SPH_C32(0x85600008), SPH_C32(0x33c38000), SPH_C32(0x7f850000),
	  SPH_C32(0xd8600000), SPH_C32(0xc450362e), SPH_C32(0xda961e25),
	  SPH_C32(0xcedc20c5), SPH_C32(0xc5de37b2), SPH_C32(0xa78a000d),
	  SPH_C32(0x8a488000), SPH_C32(0xfe2d0000), SPH_C32(0x1d740000),
	  SPH_C32(0x19294faf), SPH_C32(0x199a4de7), SPH_C32(0x9b2f3dd8),
	  SPH_C32(0xe50d4ea3) },
	{ SPH_C32(0xf0a9000b), SPH_C32(0x3dd34000), SPH_C32(0xaea50000),
	  SPH_C32(0x628a0000), SPH_C32(0x4f941910), SPH_C32(0x5dcea972),
	  SPH_C32(0x75f456d8), SPH_C32(0xc5691999), SPH_C32(0x4945000c),
	  SPH_C32(0xe51ec000), SPH_C32(0x0d130000), SPH_C32(0xbaea0000),
	  SPH_C32(0xa49c3db6), SPH_C32(0xae8ba622), SPH_C32(0xd1147d62),
	  SPH_C32(0x1ba6bcf7) },
	{ SPH_C32(0x6baf0009), SPH_C32(0x5c95c000), SPH_C32(0x8cbb0000),
	  SPH_C32(0x7ffe0000), SPH_C32(0x79e54437), SPH_C32(0x6d87f5e0),
	  SPH_C32(0x84e7607f), SPH_C32(0x3b75c5e6), SPH_C32(0x3c8c000f),
	  SPH_C32(0xeb0e0000), SPH_C32(0xdc330000), SPH_C32(0x00000000),
	  SPH_C32(0x2f581288), SPH_C32(0x29d31175), SPH_C32(0x6a3c0b7f),
	  SPH_C32(0x1b1192dc) },
	{ SPH_C32(0x1e66000a), SPH_C32(0x52850000), SPH_C32(0x5d9b0000),
	  SPH_C32(0xc5140000), SPH_C32(0xf2216b09), SPH_C32(0xeadf42b7),
	  SPH_C32(0x3fcf1662), SPH_C32(0x3bc2ebcd), SPH_C32(0xd243000e),
	  SPH_C32(0x84584000), SPH_C32(0x2f0d0000), SPH_C32(0xa79e0000),
	  SPH_C32(0x92ed6091), SPH_C32(0x9ec2fab0), SPH_C32(0x20074bc5),
	  SPH_C32(0xe5ba6088) },
	{ SPH_C32(0x73e0000d), SPH_C32(0x07804000), SPH_C32(0x5b820000),
	  SPH_C32(0x575d0000), SPH_C32(0xe5670dd5), SPH_C32(0xd02ecb8b),
	  SPH_C32(0x0319abdc), SPH_C32(0x124b8d83), SPH_C32(0x01f5000c),
	  SPH_C32(0xfb7f0000), SPH_C32(0xe7d10000), SPH_C32(0x8baf0000),
	  SPH_C32(0x23a22252), SPH_C32(0xf250e314), SPH_C32(0xb7427a57),
	  SPH_C32(0x4983222b) },
	{ SPH_C32(0x0629000e), SPH_C32(0x09908000), SPH_C32(0x8aa20000),
	  SPH_C32(0xedb70000), SPH_C32(0x6ea322eb), SPH_C32(0x57767cdc),
	  SPH_C32(0xb831ddc1), SPH_C32(0x12fca3a8), SPH_C32(0xef3a000d),
	  SPH_C32(0x94294000), SPH_C32(0x14ef0000), SPH_C32(0x2c310000),
	  SPH_C32(0x9e17504b), SPH_C32(0x454108d1), SPH_C32(0xfd793aed),
	  SPH_C32(0xb728d07f) },
	{ SPH_C32(0x9d2f000c), SPH_C32(0x68d60000), SPH_C32(0xa8bc0000),
	  SPH_C32(0xf0c30000), SPH_C32(0x58d27fcc), SPH_C32(0x673f204e),
	  SPH_C32(0x4922eb66), SPH_C32(0xece07fd7), SPH_C32(0x9af3000e),
	  SPH_C32(0x9a398000), SPH_C32(0xc5cf0000), SPH_C32(0x96db0000),
	  SPH_C32(0x15d37f75), SPH_C32(0xc219bf86), SPH_C32(0x46514cf0),
	  SPH_C32(0xb79ffe54) },
	{ SPH_C32(0xe8e6000f), SPH_C32(0x66c6c000), SPH_C32(0x799c0000),
	  SPH_C32(0x4a290000), SPH_C32(0xd31650f2), SPH_C32(0xe0679719),
	  SPH_C32(0xf20a9d7b), SPH_C32(0xec5751fc), SPH_C32(0x743c000f),
	  SPH_C32(0xf56fc000), SPH_C32(0x36f10000), SPH_C32(0x31450000),
	  SPH_C32(0xa8660d6c), SPH_C32(0x75085443), SPH_C32(0x0c6a0c4a),
	  SPH_C32(0x49340c00) }
};

static const sph_u32 T512_18[64][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x0c720000), SPH_C32(0x49e50f00), SPH_C32(0x42790000),
	  SPH_C32(0x5cea0000), SPH_C32(0x33aa301a), SPH_C32(0x15822514),
	  SPH_C32(0x95a34b7b), SPH_C32(0xb44b0090), SPH_C32(0xfe220000),
	  SPH_C32(0xa7580500), SPH_C32(0x25d10000), SPH_C32(0xf7600000),
	  SPH_C32(0x893178da), SPH_C32(0x1fd4f860), SPH_C32(0x4ed0a315),
	  SPH_C32(0xa123ff9f) },
	{ SPH_C32(0xfe220000), SPH_C32(0xa7580500), SPH_C32(0x25d10000),
	  SPH_C32(0xf7600000), SPH_C32(0x893178da), SPH_C32(0x1fd4f860),
	  SPH_C32(0x4ed0a315), SPH_C32(0xa123ff9f), SPH_C32(0xf2500000),
	  SPH_C32(0xeebd0a00), SPH_C32(0x67a80000), SPH_C32(0xab8a0000),
	  SPH_C32(0xba9b48c0), SPH_C32(0x0a56dd74), SPH_C32(0xdb73e86e),
	  SPH_C32(0x1568ff0f) },
	{ SPH_C32(0xf2500000), SPH_C32(0xeebd0a00), SPH_C32(0x67a80000),
	  SPH_C32(0xab8a0000), SPH_C32(0xba9b48c0), SPH_C32(0x0a56dd74),
	  SPH_C32(0xdb73e86e), SPH_C32(0x1568ff0f), SPH_C32(0x0c720000),
	  SPH_C32(0x49e50f00), SPH_C32(0x42790000), SPH_C32(0x5cea0000),
	  SPH_C32(0x33aa301a), SPH_C32(0x15822514), SPH_C32(0x95a34b7b),
	  SPH_C32(0xb44b0090) },
	{ SPH_C32(0x45180000), SPH_C32(0xa5b51700), SPH_C32(0xf96a0000),
	  SPH_C32(0x3b480000), SPH_C32(0x1ecc142c), SPH_C32(0x231395d6),
	  SPH_C32(0x16bca6b0), SPH_C32(0xdf33f4df), SPH_C32(0xb83d0000),
	  SPH_C32(0x16710600), SPH_C32(0x379a0000), SPH_C32(0xf5b10000),
	  SPH_C32(0x228161ac), SPH_C32(0xae48f145), SPH_C32(0x66241616),
	  SPH_C32(0xc5c1eb3e) },
	{ SPH_C32(0x496a0000), SPH_C32(0xec501800), SPH_C32(0xbb130000),
	  SPH_C32(0x67a20000), SPH_C32(0x2d662436), SPH_C32(0x3691b0c2),
	  SPH_C32(0x831fedcb), SPH_C32(0x6b78f44f), SPH_C32(0x461f0000),
	  SPH_C32(0xb1290300), SPH_C32(0x124b0000), SPH_C32(0x02d10000),
	  SPH_C32(0xabb01976), SPH_C32(0xb19c0925), SPH_C32(0x28f4b503),
	  SPH_C32(0x64e214a1) },
	{ SPH_C32(0xbb3a0000), SPH_C32(0x02ed1200), SPH_C32(0xdcbb0000),
	  SPH_C32(0xcc280000), SPH_C32(0x97fd6cf6), SPH_C32(0x3cc76db6),
	  SPH_C32(0x586c05a5), SPH_C32(0x7e100b40), SPH_C32(0x4a6d0000),
	  SPH_C32(0xf8cc0c00), SPH_C32(0x50320000), SPH_C32(0x5e3b0000),
	  SPH_C32(0x981a296c), SPH_C32(0xa41e2c31), SPH_C32(0xbd57fe78),
	  SPH_C32(0xd0a91431) },
	{ SPH_C32(0xb7480000), SPH_C32(0x4b081d00), SPH_C32(0x9ec20000),
	  SPH_C32(0x90c20000), SPH_C32(0xa4575cec), SPH_C32(0x294548a2),
	  SPH_C32(0xcdcf4ede), SPH_C32(0xca5b0bd0), SPH_C32(0xb44f0000),
	  SPH_C32(0x5f940900), SPH_C32(0x75e30000), SPH_C32(0xa95b0000),
	  SPH_C32(0x112b51b6), SPH_C32(0xbbcad451), SPH_C32(0xf3875d6d),
	  SPH_C32(0x718aebae) },
	{ SPH_C32(0xb83d0000), SPH_C32(0x16710600), SPH_C32(0x379a0000),
	  SPH_C32(0xf5b10000), SPH_C32(0x228161ac), SPH_C32(0xae48f145),
	  SPH_C32(0x66241616), SPH_C32(0xc5c1eb3e), SPH_C32(0xfd250000),
	  SPH_C32(0xb3c41100), SPH_C32(0xcef00000), SPH_C32(0xcef90000),
	  SPH_C32(0x3c4d7580), SPH_C32(0x8d5b6493), SPH_C32(0x7098b0a6),
	  SPH_C32(0x1af21fe1) },
	{ SPH_C32(0xb44f0000), SPH_C32(0x5f940900), SPH_C32(0x75e30000),
	  SPH_C32(0xa95b0000), SPH_C32(0x112b51b6), SPH_C32(0xbbcad451),
	  SPH_C32(0xf3875d6d), SPH_C32(0x718aebae), SPH_C32(0x03070000),
	  SPH_C32(0x149c1400), SPH_C32(0xeb210000), SPH_C32(0x39990000),
	  SPH_C32(0xb57c0d5a), SPH_C32(0x928f9cf3), SPH_C32(0x3e4813b3),
	  SPH_C32(0xbbd1e07e) },
	{ SPH_C32(0x461f0000), SPH_C32(0xb1290300), SPH_C32(0x124b0000),
	  SPH_C32(0x02d10000), SPH_C32(0xabb01976), SPH_C32(0xb19c0925),
	  SPH_C32(0x28f4b503), SPH_C32(0x64e214a1), SPH_C32(0x0f750000),
	  SPH_C32(0x5d791b00), SPH_C32(0xa9580000), SPH_C32(0x65730000),
	  SPH_C32(0x86d63d40), SPH_C32(0x870db9e7), SPH_C32(0xabeb58c8),
	  SPH_C32(0x0f9ae0ee) },
	{ SPH_C32(0x4a6d0000), SPH_C32(0xf8cc0c00), SPH_C32(0x50320000),
	  SPH_C32(0x5e3b0000), SPH_C32(0x981a296c), SPH_C32(0xa41e2c31),
	  SPH_C32(0xbd57fe78), SPH_C32(0xd0a91431), SPH_C32(0xf1570000),
	  SPH_C32(0xfa211e00), SPH_C32(0x8c890000), SPH_C32(0x92130000),
	  SPH_C32(0x0fe7459a), SPH_C32(0x98d94187), SPH_C32(0xe53bfbdd),
	  SPH_C32(0xaeb91f71) },
	{ SPH_C32(0xfd250000), SPH_C32(0xb3c41100), SPH_C32(0xcef00000),
	  SPH_C32(0xcef90000), SPH_C32(0x3c4d7580), SPH_C32(0x8d5b6493),
	  SPH_C32(0x7098b0a6), SPH_C32(0x1af21fe1), SPH_C32(0x45180000),
	  SPH_C32(0xa5b51700), SPH_C32(0xf96a0000), SPH_C32(0x3b480000),
	  SPH_C32(0x1ecc142c), SPH_C32(0x231395d6), SPH_C32(0x16bca6b0),
	  SPH_C32(0xdf33f4df) },
	{ SPH_C32(0xf1570000), SPH_C32(0xfa211e00), SPH_C32(0x8c890000),
	  SPH_C32(0x92130000), SPH_C32(0x0fe7459a), SPH_C32(0x98d94187),
	  SPH_C32(0xe53bfbdd), SPH_C32(0xaeb91f71), SPH_C32(0xbb3a0000),
	  SPH_C32(0x02ed1200), SPH_C32(0xdcbb0000), SPH_C32(0xcc280000),
	  SPH_C32(0x97fd6cf6), SPH_C32(0x3cc76db6), SPH_C32(0x586c05a5),
	  SPH_C32(0x7e100b40) },
	{ SPH_C32(0x03070000), SPH_C32(0x149c1400), SPH_C32(0xeb210000),
	  SPH_C32(0x39990000), SPH_C32(0xb57c0d5a), SPH_C32(0x928f9cf3),
	  SPH_C32(0x3e4813b3), SPH_C32(0xbbd1e07e), SPH_C32(0xb7480000),
	  SPH_C32(0x4b081d00), SPH_C32(0x9ec20000), SPH_C32(0x90c20000),
	  SPH_C32(0xa4575cec), SPH_C32(0x294548a2), SPH_C32(0xcdcf4ede),
	  SPH_C32(0xca5b0bd0) },
	{ SPH_C32(0x0f750000), SPH_C32(0x5d791b00), SPH_C32(0xa9580000),
	  SPH_C32(0x65730000), SPH_C32(0x86d63d40), SPH_C32(0x870db9e7),
	  SPH_C32(0xabeb58c8), SPH_C32(0x0f9ae0ee), SPH_C32(0x496a0000),
	  SPH_C32(0xec501800), SPH_C32(0xbb130000), SPH_C32(0x67a20000),
	  SPH_C32(0x2d662436), SPH_C32(0x3691b0c2), SPH_C32(0x831fedcb),
	  SPH_C32(0x6b78f44f) },
	{ SPH_C32(0x75a40000), SPH_C32(0xc28b2700), SPH_C32(0x94a40000),
	  SPH_C32(0x90f50000), SPH_C32(0xfb7857e0), SPH_C32(0x49ce0bae),
	  SPH_C32(0x1767c483), SPH_C32(0xaedf667e), SPH_C32(0xd1660000),
	  SPH_C32(0x1bbc0300), SPH_C32(0x9eec0000), SPH_C32(0xf6940000),
	  SPH_C32(0x03024527), SPH_C32(0xcf70fcf2), SPH_C32(0xb4431b17),
	  SPH_C32(0x857f3c2b) },
	{ SPH_C32(0x79d60000), SPH_C32(0x8b6e2800), SPH_C32(0xd6dd0000),
	  SPH_C32(0xcc1f0000), SPH_C32(0xc8d267fa), SPH_C32(0x5c4c2eba),
	  SPH_C32(0x82c48ff8), SPH_C32(0x1a9466ee), SPH_C32(0x2f440000),
	  SPH_C32(0xbce40600), SPH_C32(0xbb3d0000), SPH_C32(0x01f40000),
	  SPH_C32(0x8a333dfd), SPH_C32(0xd0a40492), SPH_C32(0xfa93b802),
	  SPH_C32(0x245cc3b4) },
	{ SPH_C32(0x8b860000), SPH_C32(0x65d32200), SPH_C32(0xb1750000),
	  SPH_C32(0x67950000), SPH_C32(0x72492f3a), SPH_C32(0x561af3ce),
	  SPH_C32(0x59b76796), SPH_C32(0x0ffc99e1), SPH_C32(0x23360000),
	  SPH_C32(0xf5010900), SPH_C32(0xf9440000), SPH_C32(0x5d1e0000),
	  SPH_C32(0xb9990de7), SPH_C32(0xc5262186), SPH_C32(0x6f30f379),
	  SPH_C32(0x9017c324) },
	{ SPH_C32(0x87f40000), SPH_C32(0x2c362d00), SPH_C32(0xf30c0000),
	  SPH_C32(0x3b7f0000), SPH_C32(0x41e31f20), SPH_C32(0x4398d6da),
	  SPH_C32(0xcc142ced), SPH_C32(0xbbb79971), SPH_C32(0xdd140000),
	  SPH_C32(0x52590c00), SPH_C32(0xdc950000), SPH_C32(0xaa7e0000),
	  SPH_C32(0x30a8753d), SPH_C32(0xdaf2d9e6), SPH_C32(0x21e0506c),
	  SPH_C32(0x31343cbb) },
	{ SPH_C32(0x30bc0000), SPH_C32(0x673e3000), SPH_C32(0x6dce0000),
	  SPH_C32(0xabbd0000), SPH_C32(0xe5b443cc), SPH_C32(0x6add9e78),
	  SPH_C32(0x01db6233), SPH_C32(0x71ec92a1), SPH_C32(0x695b0000),
	  SPH_C32(0x0dcd0500), SPH_C32(0xa9760000), SPH_C32(0x03250000),
	  SPH_C32(0x2183248b), SPH_C32(0x61380db7), SPH_C32(0xd2670d01),
	  SPH_C32(0x40bed715) },
	{ SPH_C32(0x3cce0000), SPH_C32(0x2edb3f00), SPH_C32(0x2fb70000),
	  SPH_C32(0xf7570000), SPH_C32(0xd61e73d6), SPH_C32(0x7f5fbb6c),
	  SPH_C32(0x94782948), SPH_C32(0xc5a79231), SPH_C32(0x97790000),
	  SPH_C32(0xaa950000), SPH_C32(0x8ca70000), SPH_C32(0xf4450000),
	  SPH_C32(0xa8b25c51), SPH_C32(0x7eecf5d7), SPH_C32(0x9cb7ae14),
	  SPH_C32(0xe19d288a) },
	{ SPH_C32(0xce9e0000), SPH_C32(0xc0663500), SPH_C32(0x481f0000),
	  SPH_C32(0x5cdd0000), SPH_C32(0x6c853b16), SPH_C32(0x75096618),
	  SPH_C32(0x4f0bc126), SPH_C32(0xd0cf6d3e), SPH_C32(0x9b0b0000),
	  SPH_C32(0xe3700f00), SPH_C32(0xcede0000), SPH_C32(0xa8af0000),
	  SPH_C32(0x9b186c4b), SPH_C32(0x6b6ed0c3), SPH_C32(0x0914e56f),
	  SPH_C32(0x55d6281a) },
	{ SPH_C32(0xc2ec0000), SPH_C32(0x89833a00), SPH_C32(0x0a660000),
	  SPH_C32(0x00370000), SPH_C32(0x5f2f0b0c), SPH_C32(0x608b430c),
	  SPH_C32(0xdaa88a5d), SPH_C32(0x64846dae), SPH_C32(0x65290000),
	  SPH_C32(0x44280a00), SPH_C32(0xeb0f0000), SPH_C32(0x5fcf0000),
	  SPH_C32(0x12291491), SPH_C32(0x74ba28a3), SPH_C32(0x47c4467a),
	  SPH_C32(0xf4f5d785) },
	{ SPH_C32(0xcd990000), SPH_C32(0xd4fa2100), SPH_C32(0xa33e0000),
	  SPH_C32(0x65440000), SPH_C32(0xd9f9364c), SPH_C32(0xe786faeb),
	  SPH_C32(0x7143d295), SPH_C32(0x6b1e8d40), SPH_C32(0x2c430000),
	  SPH_C32(0xa8781200), SPH_C32(0x501c0000), SPH_C32(0x386d0000),
	  SPH_C32(0x3f4f30a7), SPH_C32(0x422b9861), SPH_C32(0xc4dbabb1),
	  SPH_C32(0x9f8d23ca) },
	{ SPH_C32(0xc1eb0000), SPH_C32(0x9d1f2e00), SPH_C32(0xe1470000),
	  SPH_C32(0x39ae0000), SPH_C32(0xea530656), SPH_C32(0xf204dfff),
	  SPH_C32(0xe4e099ee), SPH_C32(0xdf558dd0), SPH_C32(0xd2610000),
	  SPH_C32(0x0f201700), SPH_C32(0x75cd0000), SPH_C32(0xcf0d0000),
	  SPH_C32(0xb67e487d), SPH_C32(0x5dff6001), SPH_C32(0x8a0b08a4),
	  SPH_C32(0x3eaedc55) },
	{ SPH_C32(0x33bb0000), SPH_C32(0x73a22400), SPH_C32(0x86ef0000),
	  SPH_C32(0x92240000), SPH_C32(0x50c84e96), SPH_C32(0xf852028b),
	  SPH_C32(0x3f937180), SPH_C32(0xca3d72df), SPH_C32(0xde130000),
	  SPH_C32(0x46c51800), SPH_C32(0x37b40000), SPH_C32(0x93e70000),
	  SPH_C32(0x85d47867), SPH_C32(0x487d4515), SPH_C32(0x1fa843df),
	  SPH_C32(0x8ae5dcc5) },
	{ SPH_C32(0x3fc90000), SPH_C32(0x3a472b00), SPH_C32(0xc4960000),
	  SPH_C32(0xcece0000), SPH_C32(0x63627e8c), SPH_C32(0xedd0279f),
	  SPH_C32(0xaa303afb), SPH_C32(0x7e76724f), SPH_C32(0x20310000),
	  SPH_C32(0xe19d1d00), SPH_C32(0x12650000), SPH_C32(0x64870000),
	  SPH_C32(0x0ce500bd), SPH_C32(0x57a9bd75), SPH_C32(0x5178e0ca),
	  SPH_C32(0x2bc6235a) },
	{ SPH_C32(0x88810000), SPH_C32(0x714f3600), SPH_C32(0x5a540000),
	  SPH_C32(0x5e0c0000), SPH_C32(0xc7352260), SPH_C32(0xc4956f3d),
	  SPH_C32(0x67ff7425), SPH_C32(0xb42d799f), SPH_C32(0x947e0000),
	  SPH_C32(0xbe091400), SPH_C32(0x67860000), SPH_C32(0xcddc0000),
	  SPH_C32(0x1dce510b), SPH_C32(0xec636924), SPH_C32(0xa2ffbda7),
	  SPH_C32(0x5a4cc8f4) },
	{ SPH_C32(0x84f30000), SPH_C32(0x38aa3900), SPH_C32(0x182d0000),
	  SPH_C32(0x02e60000), SPH_C32(0xf49f127a), SPH_C32(0xd1174a29),
	  SPH_C32(0xf25c3f5e), SPH_C32(0x0066790f), SPH_C32(0x6a5c0000),
	  SPH_C32(0x19511100), SPH_C32(0x42570000), SPH_C32(0x3abc0000),
	  SPH_C32(0x94ff29d1), SPH_C32(0xf3b79144), SPH_C32(0xec2f1eb2),
	  SPH_C32(0xfb6f376b) },
	{ SPH_C32(0x76a30000), SPH_C32(0xd6173300), SPH_C32(0x7f850000),
	  SPH_C32(0xa96c0000), SPH_C32(0x4e045aba), SPH_C32(0xdb41975d),
	  SPH_C32(0x292fd730), SPH_C32(0x150e8600), SPH_C32(0x662e0000),
	  SPH_C32(0x50b41e00), SPH_C32(0x002e0000), SPH_C32(0x66560000),
	  SPH_C32(0xa75519cb), SPH_C32(0xe635b450), SPH_C32(0x798c55c9),
	  SPH_C32(0x4f2437fb) },
	{ SPH_C32(0x7ad10000), SPH_C32(0x9ff23c00), SPH_C32(0x3dfc0000),
	  SPH_C32(0xf5860000), SPH_C32(0x7dae6aa0), SPH_C32(0xcec3b249),
	  SPH_C32(0xbc8c9c4b), SPH_C32(0xa1458690), SPH_C32(0x980c0000),
	  SPH_C32(0xf7ec1b00), SPH_C32(0x25ff0000), SPH_C32(0x91360000),
	  SPH_C32(0x2e646111), SPH_C32(0xf9e14c30), SPH_C32(0x375cf6dc),
	  SPH_C32(0xee07c864) },
	{ SPH_C32(0xd1660000), SPH_C32(0x1bbc0300), SPH_C32(0x9eec0000),
	  SPH_C32(0xf6940000), SPH_C32(0x03024527), SPH_C32(0xcf70fcf2),
	  SPH_C32(0xb4431b17), SPH_C32(0x857f3c2b), SPH_C32(0xa4c20000),
	  SPH_C32(0xd9372400), SPH_C32(0x0a480000), SPH_C32(0x66610000),
	  SPH_C32(0xf87a12c7), SPH_C32(0x86bef75c), SPH_C32(0xa324df94),
	  SPH_C32(0x2ba05a55) },
	{ SPH_C32(0xdd140000), SPH_C32(0x52590c00), SPH_C32(0xdc950000),
	  SPH_C32(0xaa7e0000), SPH_C32(0x30a8753d), SPH_C32(0xdaf2d9e6),
	  SPH_C32(0x21e0506c), SPH_C32(0x31343cbb), SPH_C32(0x5ae00000),
	  SPH_C32(0x7e6f2100), SPH_C32(0x2f990000), SPH_C32(0x91010000),
	  SPH_C32(0x714b6a1d), SPH_C32(0x996a0f3c), SPH_C32(0xedf47c81),
	  SPH_C32(0x8a83a5ca) },
	{ SPH_C32(0x2f440000), SPH_C32(0xbce40600), SPH_C32(0xbb3d0000),
	  SPH_C32(0x01f40000), SPH_C32(0x8a333dfd), SPH_C32(0xd0a40492),
	  SPH_C32(0xfa93b802), SPH_C32(0x245cc3b4), SPH_C32(0x56920000),
	  SPH_C32(0x378a2e00), SPH_C32(0x6de00000), SPH_C32(0xcdeb0000),
	  SPH_C32(0x42e15a07), SPH_C32(0x8ce82a28), SPH_C32(0x785737fa),
	  SPH_C32(0x3ec8a55a) },
	{ SPH_C32(0x23360000), SPH_C32(0xf5010900), SPH_C32(0xf9440000),
	  SPH_C32(0x5d1e0000), SPH_C32(0xb9990de7), SPH_C32(0xc5262186),
	  SPH_C32(0x6f30f379), SPH_C32(0x9017c324), SPH_C32(0xa8b00000),
	  SPH_C32(0x90d22b00), SPH_C32(0x48310000), SPH_C32(0x3a8b0000),
	  SPH_C32(0xcbd022dd), SPH_C32(0x933cd248), SPH_C32(0x368794ef),
	  SPH_C32(0x9feb5ac5) },
	{ SPH_C32(0x947e0000), SPH_C32(0xbe091400), SPH_C32(0x67860000),
	  SPH_C32(0xcddc0000), SPH_C32(0x1dce510b), SPH_C32(0xec636924),
	  SPH_C32(0xa2ffbda7), SPH_C32(0x5a4cc8f4), SPH_C32(0x1cff0000),
	  SPH_C32(0xcf462200), SPH_C32(0x3dd20000), SPH_C32(0x93d00000),
	  SPH_C32(0xdafb736b), SPH_C32(0x28f60619), SPH_C32(0xc500c982),
	  SPH_C32(0xee61b16b) },
	{ SPH_C32(0x980c0000), SPH_C32(0xf7ec1b00), SPH_C32(0x25ff0000),
	  SPH_C32(0x91360000), SPH_C32(0x2e646111), SPH_C32(0xf9e14c30),
	  SPH_C32(0x375cf6dc), SPH_C32(0xee07c864), SPH_C32(0xe2dd0000),
	  SPH_C32(0x681e2700), SPH_C32(0x18030000), SPH_C32(0x64b00000),
	  SPH_C32(0x53ca0bb1), SPH_C32(0x3722fe79), SPH_C32(0x8bd06a97),
	  SPH_C32(0x4f424ef4) },
	{ SPH_C32(0x6a5c0000), SPH_C32(0x19511100), SPH_C32(0x42570000),
	  SPH_C32(0x3abc0000), SPH_C32(0x94ff29d1), SPH_C32(0xf3b79144),
	  SPH_C32(0xec2f1eb2), SPH_C32(0xfb6f376b), SPH_C32(0xeeaf0000),
	  SPH_C32(0x21fb2800), SPH_C32(0x5a7a0000), SPH_C32(0x385a0000),
	  SPH_C32(0x60603bab), SPH_C32(0x22a0db6d), SPH_C32(0x1e7321ec),
	  SPH_C32(0xfb094e64) },
	{ SPH_C32(0x662e0000), SPH_C32(0x50b41e00), SPH_C32(0x002e0000),
	  SPH_C32(0x66560000), SPH_C32(0xa75519cb), SPH_C32(0xe635b450),
	  SPH_C32(0x798c55c9), SPH_C32(0x4f2437fb), SPH_C32(0x108d0000),
	  SPH_C32(0x86a32d00), SPH_C32(0x7fab0000), SPH_C32(0xcf3a0000),
	  SPH_C32(0xe9514371), SPH_C32(0x3d74230d), SPH_C32(0x50a382f9),
	  SPH_C32(0x5a2ab1fb) },
	{ SPH_C32(0x695b0000), SPH_C32(0x0dcd0500), SPH_C32(0xa9760000),
	  SPH_C32(0x03250000), SPH_C32(0x2183248b), SPH_C32(0x61380db7),
	  SPH_C32(0xd2670d01), SPH_C32(0x40bed715), SPH_C32(0x59e70000),
	  SPH_C32(0x6af33500), SPH_C32(0xc4b80000), SPH_C32(0xa8980000),
	  SPH_C32(0xc4376747), SPH_C32(0x0be593cf), SPH_C32(0xd3bc6f32),
	  SPH_C32(0x315245b4) },
	{ SPH_C32(0x65290000), SPH_C32(0x44280a00), SPH_C32(0xeb0f0000),
	  SPH_C32(0x5fcf0000), SPH_C32(0x12291491), SPH_C32(0x74ba28a3),
	  SPH_C32(0x47c4467a), SPH_C32(0xf4f5d785), SPH_C32(0xa7c50000),
	  SPH_C32(0xcdab3000), SPH_C32(0xe1690000), SPH_C32(0x5ff80000),
	  SPH_C32(0x4d061f9d), SPH_C32(0x14316baf), SPH_C32(0x9d6ccc27),
	  SPH_C32(0x9071ba2b) },
	{ SPH_C32(0x97790000), SPH_C32(0xaa950000), SPH_C32(0x8ca70000),
	  SPH_C32(0xf4450000), SPH_C32(0xa8b25c51), SPH_C32(0x7eecf5d7),
	  SPH_C32(0x9cb7ae14), SPH_C32(0xe19d288a), SPH_C32(0xabb70000),
	  SPH_C32(0x844e3f00), SPH_C32(0xa3100000), SPH_C32(0x03120000),
	  SPH_C32(0x7eac2f87), SPH_C32(0x01b34ebb), SPH_C32(0x08cf875c),
	  SPH_C32(0x243ababb) },
	{ SPH_C32(0x9b0b0000), SPH_C32(0xe3700f00), SPH_C32(0xcede0000),
	  SPH_C32(0xa8af0000), SPH_C32(0x9b186c4b), SPH_C32(0x6b6ed0c3),
	  SPH_C32(0x0914e56f), SPH_C32(0x55d6281a), SPH_C32(0x55950000),
	  SPH_C32(0x23163a00), SPH_C32(0x86c10000), SPH_C32(0xf4720000),
	  SPH_C32(0xf79d575d), SPH_C32(0x1e67b6db), SPH_C32(0x461f2449),
	  SPH_C32(0x85194524) },
	{ SPH_C32(0x2c430000), SPH_C32(0xa8781200), SPH_C32(0x501c0000),
	  SPH_C32(0x386d0000), SPH_C32(0x3f4f30a7), SPH_C32(0x422b9861),
	  SPH_C32(0xc4dbabb1), SPH_C32(0x9f8d23ca), SPH_C32(0xe1da0000),
	  SPH_C32(0x7c823300), SPH_C32(0xf3220000), SPH_C32(0x5d290000),
	  SPH_C32(0xe6b606eb), SPH_C32(0xa5ad628a), SPH_C32(0xb5987924),
	  SPH_C32(0xf493ae8a) },
	{ SPH_C32(0x20310000), SPH_C32(0xe19d1d00), SPH_C32(0x12650000),
	  SPH_C32(0x64870000), SPH_C32(0x0ce500bd), SPH_C32(0x57a9bd75),
	  SPH_C32(0x5178e0ca), SPH_C32(0x2bc6235a), SPH_C32(0x1ff80000),
	  SPH_C32(0xdbda3600), SPH_C32(0xd6f30000), SPH_C32(0xaa490000),
	  SPH_C32(0x6f877e31), SPH_C32(0xba799aea), SPH_C32(0xfb48da31),
	  SPH_C32(0x55b05115) },
	{ SPH_C32(0xd2610000), SPH_C32(0x0f201700), SPH_C32(0x75cd0000),
	  SPH_C32(0xcf0d0000), SPH_C32(0xb67e487d), SPH_C32(0x5dff6001),
	  SPH_C32(0x8a0b08a4), SPH_C32(0x3eaedc55), SPH_C32(0x138a0000),
	  SPH_C32(0x923f3900), SPH_C32(0x948a0000), SPH_C32(0xf6a30000),
	  SPH_C32(0x5c2d4e2b), SPH_C32(0xaffbbffe), SPH_C32(0x6eeb914a),
	  SPH_C32(0xe1fb5185) },
	{ SPH_C32(0xde130000), SPH_C32(0x46c51800), SPH_C32(0x37b40000),
	  SPH_C32(0x93e70000), SPH_C32(0x85d47867), SPH_C32(0x487d4515),
	  SPH_C32(0x1fa843df), SPH_C32(0x8ae5dcc5), SPH_C32(0xeda80000),
	  SPH_C32(0x35673c00), SPH_C32(0xb15b0000), SPH_C32(0x01c30000),
	  SPH_C32(0xd51c36f1), SPH_C32(0xb02f479e), SPH_C32(0x203b325f),
	  SPH_C32(0x40d8ae1a) },
	{ SPH_C32(0xa4c20000), SPH_C32(0xd9372400), SPH_C32(0x0a480000),
	  SPH_C32(0x66610000), SPH_C32(0xf87a12c7), SPH_C32(0x86bef75c),
	  SPH_C32(0xa324df94), SPH_C32(0x2ba05a55), SPH_C32(0x75a40000),
	  SPH_C32(0xc28b2700), SPH_C32(0x94a40000), SPH_C32(0x90f50000),
	  SPH_C32(0xfb7857e0), SPH_C32(0x49ce0bae), SPH_C32(0x1767c483),
	  SPH_C32(0xaedf667e) },
	{ SPH_C32(0xa8b00000), SPH_C32(0x90d22b00), SPH_C32(0x48310000),
	  SPH_C32(0x3a8b0000), SPH_C32(0xcbd022dd), SPH_C32(0x933cd248),
	  SPH_C32(0x368794ef), SPH_C32(0x9feb5ac5), SPH_C32(0x8b860000),
	  SPH_C32(0x65d32200), SPH_C32(0xb1750000), SPH_C32(0x67950000),
	  SPH_C32(0x72492f3a), SPH_C32(0x561af3ce), SPH_C32(0x59b76796),
	  SPH_C32(0x0ffc99e1) },
	{ SPH_C32(0x5ae00000), SPH_C32(0x7e6f2100), SPH_C32(0x2f990000),
	  SPH_C32(0x91010000), SPH_C32(0x714b6a1d), SPH_C32(0x996a0f3c),
	  SPH_C32(0xedf47c81), SPH_C32(0x8a83a5ca), SPH_C32(0x87f40000),
	  SPH_C32(0x2c362d00), SPH_C32(0xf30c0000), SPH_C32(0x3b7f0000),
	  SPH_C32(0x41e31f20), SPH_C32(0x4398d6da), SPH_C32(0xcc142ced),
	  SPH_C32(0xbbb79971) },
	{ SPH_C32(0x56920000), SPH_C32(0x378a2e00), SPH_C32(0x6de00000),
	  SPH_C32(0xcdeb0000), SPH_C32(0x42e15a07), SPH_C32(0x8ce82a28),
	  SPH_C32(0x785737fa), SPH_C32(0x3ec8a55a), SPH_C32(0x79d60000),
	  SPH_C32(0x8b6e2800), SPH_C32(0xd6dd0000), SPH_C32(0xcc1f0000),
	  SPH_C32(0xc8d267fa), SPH_C32(0x5c4c2eba), SPH_C32(0x82c48ff8),
	  SPH_C32(0x1a9466ee) },
	{ SPH_C32(0xe1da0000), SPH_C32(0x7c823300), SPH_C32(0xf3220000),
	  SPH_C32(0x5d290000), SPH_C32(0xe6b606eb), SPH_C32(0xa5ad628a),
	  SPH_C32(0xb5987924), SPH_C32(0xf493ae8a), SPH_C32(0xcd990000),
	  SPH_C32(0xd4fa2100), SPH_C32(0xa33e0000), SPH_C32(0x65440000),
	  SPH_C32(0xd9f9364c), SPH_C32(0xe786faeb), SPH_C32(0x7143d295),
	  SPH_C32(0x6b1e8d40) },
	{ SPH_C32(0xeda80000), SPH_C32(0x35673c00), SPH_C32(0xb15b0000),
	  SPH_C32(0x01c30000), SPH_C32(0xd51c36f1), SPH_C32(0xb02f479e),
	  SPH_C32(0x203b325f), SPH_C32(0x40d8ae1a), SPH_C32(0x33bb0000),
	  SPH_C32(0x73a22400), SPH_C32(0x86ef0000), SPH_C32(0x92240000),
	  SPH_C32(0x50c84e96), SPH_C32(0xf852028b), SPH_C32(0x3f937180),
	  SPH_C32(0xca3d72df) },
	{ SPH_C32(0x1ff80000), SPH_C32(0xdbda3600), SPH_C32(0xd6f30000),
	  SPH_C32(0xaa490000), SPH_C32(0x6f877e31), SPH_C32(0xba799aea),
	  SPH_C32(0xfb48da31), SPH_C32(0x55b05115), SPH_C32(0x3fc90000),
	  SPH_C32(0x3a472b00), SPH_C32(0xc4960000), SPH_C32(0xcece0000),
	  SPH_C32(0x63627e8c), SPH_C32(0xedd0279f), SPH_C32(0xaa303afb),
	  SPH_C32(0x7e76724f) },
	{ SPH_C32(0x138a0000), SPH_C32(0x923f3900), SPH_C32(0x948a0000),
	  SPH_C32(0xf6a30000), SPH_C32(0x5c2d4e2b), SPH_C32(0xaffbbffe),
	  SPH_C32(0x6eeb914a), SPH_C32(0xe1fb5185), SPH_C32(0xc1eb0000),
	  SPH_C32(0x9d1f2e00), SPH_C32(0xe1470000), SPH_C32(0x39ae0000),
	  SPH_C32(0xea530656), SPH_C32(0xf204dfff), SPH_C32(0xe4e099ee),
	  SPH_C32(0xdf558dd0) },
	{ SPH_C32(0x1cff0000), SPH_C32(0xcf462200), SPH_C32(0x3dd20000),
	  SPH_C32(0x93d00000), SPH_C32(0xdafb736b), SPH_C32(0x28f60619),
	  SPH_C32(0xc500c982), SPH_C32(0xee61b16b), SPH_C32(0x88810000),
	  SPH_C32(0x714f3600), SPH_C32(0x5a540000), SPH_C32(0x5e0c0000),
	  SPH_C32(0xc7352260), SPH_C32(0xc4956f3d), SPH_C32(0x67ff7425),
	  SPH_C32(0xb42d799f) },
	{ SPH_C32(0x108d0000), SPH_C32(0x86a32d00), SPH_C32(0x7fab0000),
	  SPH_C32(0xcf3a0000), SPH_C32(0xe9514371), SPH_C32(0x3d74230d),
	  SPH_C32(0x50a382f9), SPH_C32(0x5a2ab1fb), SPH_C32(0x76a30000),
	  SPH_C32(0xd6173300), SPH_C32(0x7f850000), SPH_C32(0xa96c0000),
	  SPH_C32(0x4e045aba), SPH_C32(0xdb41975d), SPH_C32(0x292fd730),
	  SPH_C32(0x150e8600) },
	{ SPH_C32(0xe2dd0000), SPH_C32(0x681e2700), SPH_C32(0x18030000),
	  SPH_C32(0x64b00000), SPH_C32(0x53ca0bb1), SPH_C32(0x3722fe79),
	  SPH_C32(0x8bd06a97), SPH_C32(0x4f424ef4), SPH_C32(0x7ad10000),
	  SPH_C32(0x9ff23c00), SPH_C32(0x3dfc0000), SPH_C32(0xf5860000),
	  SPH_C32(0x7dae6aa0), SPH_C32(0xcec3b249), SPH_C32(0xbc8c9c4b),
	  SPH_C32(0xa1458690) },
	{ SPH_C32(0xeeaf0000), SPH_C32(0x21fb2800), SPH_C32(0x5a7a0000),
	  SPH_C32(0x385a0000), SPH_C32(0x60603bab), SPH_C32(0x22a0db6d),
	  SPH_C32(0x1e7321ec), SPH_C32(0xfb094e64), SPH_C32(0x84f30000),
	  SPH_C32(0x38aa3900), SPH_C32(0x182d0000), SPH_C32(0x02e60000),
	  SPH_C32(0xf49f127a), SPH_C32(0xd1174a29), SPH_C32(0xf25c3f5e),
	  SPH_C32(0x0066790f) },
	{ SPH_C32(0x59e70000), SPH_C32(0x6af33500), SPH_C32(0xc4b80000),
	  SPH_C32(0xa8980000), SPH_C32(0xc4376747), SPH_C32(0x0be593cf),
	  SPH_C32(0xd3bc6f32), SPH_C32(0x315245b4), SPH_C32(0x30bc0000),
	  SPH_C32(0x673e3000), SPH_C32(0x6dce0000), SPH_C32(0xabbd0000),
	  SPH_C32(0xe5b443cc), SPH_C32(0x6add9e78), SPH_C32(0x01db6233),
	  SPH_C32(0x71ec92a1) },
	{ SPH_C32(0x55950000), SPH_C32(0x23163a00), SPH_C32(0x86c10000),
	  SPH_C32(0xf4720000), SPH_C32(0xf79d575d), SPH_C32(0x1e67b6db),
	  SPH_C32(0x461f2449), SPH_C32(0x85194524), SPH_C32(0xce9e0000),
	  SPH_C32(0xc0663500), SPH_C32(0x481f0000), SPH_C32(0x5cdd0000),
	  SPH_C32(0x6c853b16), SPH_C32(0x75096618), SPH_C32(0x4f0bc126),
	  SPH_C32(0xd0cf6d3e) },
	{ SPH_C32(0xa7c50000), SPH_C32(0xcdab3000), SPH_C32(0xe1690000),
	  SPH_C32(0x5ff80000), SPH_C32(0x4d061f9d), SPH_C32(0x14316baf),
	  SPH_C32(0x9d6ccc27), SPH_C32(0x9071ba2b), SPH_C32(0xc2ec0000),
	  SPH_C32(0x89833a00), SPH_C32(0x0a660000), SPH_C32(0x00370000),
	  SPH_C32(0x5f2f0b0c), SPH_C32(0x608b430c), SPH_C32(0xdaa88a5d),
	  SPH_C32(0x64846dae) },
	{ SPH_C32(0xabb70000), SPH_C32(0x844e3f00), SPH_C32(0xa3100000),
	  SPH_C32(0x03120000), SPH_C32(0x7eac2f87), SPH_C32(0x01b34ebb),
	  SPH_C32(0x08cf875c), SPH_C32(0x243ababb), SPH_C32(0x3cce0000),
	  SPH_C32(0x2edb3f00), SPH_C32(0x2fb70000), SPH_C32(0xf7570000),
	  SPH_C32(0xd61e73d6), SPH_C32(0x7f5fbb6c), SPH_C32(0x94782948),
	  SPH_C32(0xc5a79231) }
};

static const sph_u32 T512_24[64][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x14190000), SPH_C32(0x23ca003c), SPH_C32(0x50df0000),
	  SPH_C32(0x44b60000), SPH_C32(0x1b6c67b0), SPH_C32(0x3cf3ac75),
	  SPH_C32(0x61e610b0), SPH_C32(0xdbcadb80), SPH_C32(0xe3430000),
	  SPH_C32(0x3a4e0014), SPH_C32(0xf2c60000), SPH_C32(0xaa4e0000),
	  SPH_C32(0xdb1e42a6), SPH_C32(0x256bbe15), SPH_C32(0x123db156),
	  SPH_C32(0x3a4e99d7) },
	{ SPH_C32(0xe3430000), SPH_C32(0x3a4e0014), SPH_C32(0xf2c60000),
	  SPH_C32(0xaa4e0000), SPH_C32(0xdb1e42a6), SPH_C32(0x256bbe15),
	  SPH_C32(0x123db156), SPH_C32(0x3a4e99d7), SPH_C32(0xf75a0000),
	  SPH_C32(0x19840028), SPH_C32(0xa2190000), SPH_C32(0xeef80000),
	  SPH_C32(0xc0722516), SPH_C32(0x19981260), SPH_C32(0x73dba1e6),
	  SPH_C32(0xe1844257) },
	{ SPH_C32(0xf75a0000), SPH_C32(0x19840028), SPH_C32(0xa2190000),
	  SPH_C32(0xeef80000), SPH_C32(0xc0722516), SPH_C32(0x19981260),
	  SPH_C32(0x73dba1e6), SPH_C32(0xe1844257), SPH_C32(0x14190000),
	  SPH_C32(0x23ca003c), SPH_C32(0x50df0000), SPH_C32(0x44b60000),
	  SPH_C32(0x1b6c67b0), SPH_C32(0x3cf3ac75), SPH_C32(0x61e610b0),
	  SPH_C32(0xdbcadb80) },
	{ SPH_C32(0x54500000), SPH_C32(0x0671005c), SPH_C32(0x25ae0000),
	  SPH_C32(0x6a1e0000), SPH_C32(0x2ea54edf), SPH_C32(0x664e8512),
	  SPH_C32(0xbfba18c3), SPH_C32(0x7e715d17), SPH_C32(0xbc8d0000),
	  SPH_C32(0xfc3b0018), SPH_C32(0x19830000), SPH_C32(0xd10b0000),
	  SPH_C32(0xae1878c4), SPH_C32(0x42a69856), SPH_C32(0x0012da37),
	  SPH_C32(0x2c3b504e) },
	{ SPH_C32(0x40490000), SPH_C32(0x25bb0060), SPH_C32(0x75710000),
	  SPH_C32(0x2ea80000), SPH_C32(0x35c9296f), SPH_C32(0x5abd2967),
	  SPH_C32(0xde5c0873), SPH_C32(0xa5bb8697), SPH_C32(0x5fce0000),
	  SPH_C32(0xc675000c), SPH_C32(0xeb450000), SPH_C32(0x7b450000),
	  SPH_C32(0x75063a62), SPH_C32(0x67cd2643), SPH_C32(0x122f6b61),
	  SPH_C32(0x1675c999) },
	{ SPH_C32(0xb7130000), SPH_C32(0x3c3f0048), SPH_C32(0xd7680000),
	  SPH_C32(0xc0500000), SPH_C32(0xf5bb0c79), SPH_C32(0x43253b07),
	  SPH_C32(0xad87a995), SPH_C32(0x443fc4c0), SPH_C32(0x4bd70000),
	  SPH_C32(0xe5bf0030), SPH_C32(0xbb9a0000), SPH_C32(0x3ff30000),
	  SPH_C32(0x6e6a5dd2), SPH_C32(0x5b3e8a36), SPH_C32(0x73c97bd1),
	  SPH_C32(0xcdbf1219) },
	{ SPH_C32(0xa30a0000), SPH_C32(0x1ff50074), SPH_C32(0x87b70000),
	  SPH_C32(0x84e60000), SPH_C32(0xeed76bc9), SPH_C32(0x7fd69772),
	  SPH_C32(0xcc61b925), SPH_C32(0x9ff51f40), SPH_C32(0xa8940000),
	  SPH_C32(0xdff10024), SPH_C32(0x495c0000), SPH_C32(0x95bd0000),
	  SPH_C32(0xb5741f74), SPH_C32(0x7e553423), SPH_C32(0x61f4ca87),
	  SPH_C32(0xf7f18bce) },
	{ SPH_C32(0xbc8d0000), SPH_C32(0xfc3b0018), SPH_C32(0x19830000),
	  SPH_C32(0xd10b0000), SPH_C32(0xae1878c4), SPH_C32(0x42a69856),
	  SPH_C32(0x0012da37), SPH_C32(0x2c3b504e), SPH_C32(0xe8dd0000),
	  SPH_C32(0xfa4a0044), SPH_C32(0x3c2d0000), SPH_C32(0xbb150000),
	  SPH_C32(0x80bd361b), SPH_C32(0x24e81d44), SPH_C32(0xbfa8c2f4),
	  SPH_C32(0x524a0d59) },
	{ SPH_C32(0xa8940000), SPH_C32(0xdff10024), SPH_C32(0x495c0000),
	  SPH_C32(0x95bd0000), SPH_C32(0xb5741f74), SPH_C32(0x7e553423),
	  SPH_C32(0x61f4ca87), SPH_C32(0xf7f18bce), SPH_C32(0x0b9e0000),
	  SPH_C32(0xc0040050), SPH_C32(0xceeb0000), SPH_C32(0x115b0000),
	  SPH_C32(0x5ba374bd), SPH_C32(0x0183a351), SPH_C32(0xad9573a2),
	  SPH_C32(0x6804948e) },
	{ SPH_C32(0x5fce0000), SPH_C32(0xc675000c), SPH_C32(0xeb450000),
	  SPH_C32(0x7b450000), SPH_C32(0x75063a62), SPH_C32(0x67cd2643),
	  SPH_C32(0x122f6b61), SPH_C32(0x1675c999), SPH_C32(0x1f870000),
	  SPH_C32(0xe3ce006c), SPH_C32(0x9e340000), SPH_C32(0x55ed0000),
	  SPH_C32(0x40cf130d), SPH_C32(0x3d700f24), SPH_C32(0xcc736312),
	  SPH_C32(0xb3ce4f0e) },
	{ SPH_C32(0x4bd70000), SPH_C32(0xe5bf0030), SPH_C32(0xbb9a0000),
	  SPH_C32(0x3ff30000), SPH_C32(0x6e6a5dd2), SPH_C32(0x5b3e8a36),
	  SPH_C32(0x73c97bd1), SPH_C32(0xcdbf1219), SPH_C32(0xfcc40000),
	  SPH_C32(0xd9800078), SPH_C32(0x6cf20000), SPH_C32(0xffa30000),
	  SPH_C32(0x9bd151ab), SPH_C32(0x181bb131), SPH_C32(0xde4ed244),
	  SPH_C32(0x8980d6d9) },
	{ SPH_C32(0xe8dd0000), SPH_C32(0xfa4a0044), SPH_C32(0x3c2d0000),
	  SPH_C32(0xbb150000), SPH_C32(0x80bd361b), SPH_C32(0x24e81d44),
	  SPH_C32(0xbfa8c2f4), SPH_C32(0x524a0d59), SPH_C32(0x54500000),
	  SPH_C32(0x0671005c), SPH_C32(0x25ae0000), SPH_C32(0x6a1e0000),
	  SPH_C32(0x2ea54edf), SPH_C32(0x664e8512), SPH_C32(0xbfba18c3),
	  SPH_C32(0x7e715d17) },
	{ SPH_C32(0xfcc40000), SPH_C32(0xd9800078), SPH_C32(0x6cf20000),
	  SPH_C32(0xffa30000), SPH_C32(0x9bd151ab), SPH_C32(0x181bb131),
	  SPH_C32(0xde4ed244), SPH_C32(0x8980d6d9), SPH_C32(0xb7130000),
	  SPH_C32(0x3c3f0048), SPH_C32(0xd7680000), SPH_C32(0xc0500000),
	  SPH_C32(0xf5bb0c79), SPH_C32(0x43253b07), SPH_C32(0xad87a995),
	  SPH_C32(0x443fc4c0) },
	{ SPH_C32(0x0b9e0000), SPH_C32(0xc0040050), SPH_C32(0xceeb0000),
	  SPH_C32(0x115b0000), SPH_C32(0x5ba374bd), SPH_C32(0x0183a351),
	  SPH_C32(0xad9573a2), SPH_C32(0x6804948e), SPH_C32(0xa30a0000),
	  SPH_C32(0x1ff50074), SPH_C32(0x87b70000), SPH_C32(0x84e60000),
	  SPH_C32(0xeed76bc9), SPH_C32(0x7fd69772), SPH_C32(0xcc61b925),
	  SPH_C32(0x9ff51f40) },
	{ SPH_C32(0x1f870000), SPH_C32(0xe3ce006c), SPH_C32(0x9e340000),
	  SPH_C32(0x55ed0000), SPH_C32(0x40cf130d), SPH_C32(0x3d700f24),
	  SPH_C32(0xcc736312), SPH_C32(0xb3ce4f0e), SPH_C32(0x40490000),
	  SPH_C32(0x25bb0060), SPH_C32(0x75710000), SPH_C32(0x2ea80000),
	  SPH_C32(0x35c9296f), SPH_C32(0x5abd2967), SPH_C32(0xde5c0873),
	  SPH_C32(0xa5bb8697) },
	{ SPH_C32(0x69510000), SPH_C32(0xd4e1009c), SPH_C32(0xc3230000),
	  SPH_C32(0xac2f0000), SPH_C32(0xe4950bae), SPH_C32(0xcea415dc),
	  SPH_C32(0x87ec287c), SPH_C32(0xbce1a3ce), SPH_C32(0xc6730000),
	  SPH_C32(0xaf8d000c), SPH_C32(0xa4c10000), SPH_C32(0x218d0000),
	  SPH_C32(0x23111587), SPH_C32(0x7913512f), SPH_C32(0x1d28ac88),
	  SPH_C32(0x378dd173) },
	{ SPH_C32(0x7d480000), SPH_C32(0xf72b00a0), SPH_C32(0x93fc0000),
	  SPH_C32(0xe8990000), SPH_C32(0xfff96c1e), SPH_C32(0xf257b9a9),
	  SPH_C32(0xe60a38cc), SPH_C32(0x672b784e), SPH_C32(0x25300000),
	  SPH_C32(0x95c30018), SPH_C32(0x56070000), SPH_C32(0x8bc30000),
	  SPH_C32(0xf80f5721), SPH_C32(0x5c78ef3a), SPH_C32(0x0f151dde),
	  SPH_C32(0x0dc348a4) },
	{ SPH_C32(0x8a120000), SPH_C32(0xeeaf0088), SPH_C32(0x31e50000),
	  SPH_C32(0x06610000), SPH_C32(0x3f8b4908), SPH_C32(0xebcfabc9),
	  SPH_C32(0x95d1992a), SPH_C32(0x86af3a19), SPH_C32(0x31290000),
	  SPH_C32(0xb6090024), SPH_C32(0x06d80000), SPH_C32(0xcf750000),
	  SPH_C32(0xe3633091), SPH_C32(0x608b434f), SPH_C32(0x6ef30d6e),
	  SPH_C32(0xd6099324) },
	{ SPH_C32(0x9e0b0000), SPH_C32(0xcd6500b4), SPH_C32(0x613a0000),
	  SPH_C32(0x42d70000), SPH_C32(0x24e72eb8), SPH_C32(0xd73c07bc),
	  SPH_C32(0xf437899a), SPH_C32(0x5d65e199), SPH_C32(0xd26a0000),
	  SPH_C32(0x8c470030), SPH_C32(0xf41e0000), SPH_C32(0x653b0000),
	  SPH_C32(0x387d7237), SPH_C32(0x45e0fd5a), SPH_C32(0x7ccebc38),
	  SPH_C32(0xec470af3) },
	{ SPH_C32(0x3d010000), SPH_C32(0xd29000c0), SPH_C32(0xe68d0000),
	  SPH_C32(0xc6310000), SPH_C32(0xca304571), SPH_C32(0xa8ea90ce),
	  SPH_C32(0x385630bf), SPH_C32(0xc290fed9), SPH_C32(0x7afe0000),
	  SPH_C32(0x53b60014), SPH_C32(0xbd420000), SPH_C32(0xf0860000),
	  SPH_C32(0x8d096d43), SPH_C32(0x3bb5c979), SPH_C32(0x1d3a76bf),
	  SPH_C32(0x1bb6813d) },
	{ SPH_C32(0x29180000), SPH_C32(0xf15a00fc), SPH_C32(0xb6520000),
	  SPH_C32(0x82870000), SPH_C32(0xd15c22c1), SPH_C32(0x94193cbb),
	  SPH_C32(0x59b0200f), SPH_C32(0x195a2559), SPH_C32(0x99bd0000),
	  SPH_C32(0x69f80000), SPH_C32(0x4f840000), SPH_C32(0x5ac80000),
	  SPH_C32(0x56172fe5), SPH_C32(0x1ede776c), SPH_C32(0x0f07c7e9),
	  SPH_C32(0x21f818ea) },
	{ SPH_C32(0xde420000), SPH_C32(0xe8de00d4), SPH_C32(0x144b0000),
	  SPH_C32(0x6c7f0000), SPH_C32(0x112e07d7), SPH_C32(0x8d812edb),
	  SPH_C32(0x2a6b81e9), SPH_C32(0xf8de670e), SPH_C32(0x8da40000),
	  SPH_C32(0x4a32003c), SPH_C32(0x1f5b0000), SPH_C32(0x1e7e0000),
	  SPH_C32(0x4d7b4855), SPH_C32(0x222ddb19), SPH_C32(0x6ee1d759),
	  SPH_C32(0xfa32c36a) },
	{ SPH_C32(0xca5b0000), SPH_C32(0xcb1400e8), SPH_C32(0x44940000),
	  SPH_C32(0x28c90000), SPH_C32(0x0a426067), SPH_C32(0xb17282ae),
	  SPH_C32(0x4b8d9159), SPH_C32(0x2314bc8e), SPH_C32(0x6ee70000),
	  SPH_C32(0x707c0028), SPH_C32(0xed9d0000), SPH_C32(0xb4300000),
	  SPH_C32(0x96650af3), SPH_C32(0x0746650c), SPH_C32(0x7cdc660f),
	  SPH_C32(0xc07c5abd) },
	{ SPH_C32(0xd5dc0000), SPH_C32(0x28da0084), SPH_C32(0xdaa00000),
	  SPH_C32(0x7d240000), SPH_C32(0x4a8d736a), SPH_C32(0x8c028d8a),
	  SPH_C32(0x87fef24b), SPH_C32(0x90daf380), SPH_C32(0x2eae0000),
	  SPH_C32(0x55c70048), SPH_C32(0x98ec0000), SPH_C32(0x9a980000),
	  SPH_C32(0xa3ac239c), SPH_C32(0x5dfb4c6b), SPH_C32(0xa2806e7c),
	  SPH_C32(0x65c7dc2a) },
	{ SPH_C32(0xc1c50000), SPH_C32(0x0b1000b8), SPH_C32(0x8a7f0000),
	  SPH_C32(0x39920000), SPH_C32(0x51e114da), SPH_C32(0xb0f121ff),
	  SPH_C32(0xe618e2fb), SPH_C32(0x4b102800), SPH_C32(0xcded0000),
	  SPH_C32(0x6f89005c), SPH_C32(0x6a2a0000), SPH_C32(0x30d60000),
	  SPH_C32(0x78b2613a), SPH_C32(0x7890f27e), SPH_C32(0xb0bddf2a),
	  SPH_C32(0x5f8945fd) },
	{ SPH_C32(0x369f0000), SPH_C32(0x12940090), SPH_C32(0x28660000),
	  SPH_C32(0xd76a0000), SPH_C32(0x919331cc), SPH_C32(0xa969339f),
	  SPH_C32(0x95c3431d), SPH_C32(0xaa946a57), SPH_C32(0xd9f40000),
	  SPH_C32(0x4c430060), SPH_C32(0x3af50000), SPH_C32(0x74600000),
	  SPH_C32(0x63de068a), SPH_C32(0x44635e0b), SPH_C32(0xd15bcf9a),
	  SPH_C32(0x84439e7d) },
	{ SPH_C32(0x22860000), SPH_C32(0x315e00ac), SPH_C32(0x78b90000),
	  SPH_C32(0x93dc0000), SPH_C32(0x8aff567c), SPH_C32(0x959a9fea),
	  SPH_C32(0xf42553ad), SPH_C32(0x715eb1d7), SPH_C32(0x3ab70000),
	  SPH_C32(0x760d0074), SPH_C32(0xc8330000), SPH_C32(0xde2e0000),
	  SPH_C32(0xb8c0442c), SPH_C32(0x6108e01e), SPH_C32(0xc3667ecc),
	  SPH_C32(0xbe0d07aa) },
	{ SPH_C32(0x818c0000), SPH_C32(0x2eab00d8), SPH_C32(0xff0e0000),
	  SPH_C32(0x173a0000), SPH_C32(0x64283db5), SPH_C32(0xea4c0898),
	  SPH_C32(0x3844ea88), SPH_C32(0xeeabae97), SPH_C32(0x92230000),
	  SPH_C32(0xa9fc0050), SPH_C32(0x816f0000), SPH_C32(0x4b930000),
	  SPH_C32(0x0db45b58), SPH_C32(0x1f5dd43d), SPH_C32(0xa292b44b),
	  SPH_C32(0x49fc8c64) },
	{ SPH_C32(0x95950000), SPH_C32(0x0d6100e4), SPH_C32(0xafd10000),
	  SPH_C32(0x538c0000), SPH_C32(0x7f445a05), SPH_C32(0xd6bfa4ed),
	  SPH_C32(0x59a2fa38), SPH_C32(0x35617517), SPH_C32(0x71600000),
	  SPH_C32(0x93b20044), SPH_C32(0x73a90000), SPH_C32(0xe1dd0000),
	  SPH_C32(0xd6aa19fe), SPH_C32(0x3a366a28), SPH_C32(0xb0af051d),
	  SPH_C32(0x73b215b3) },
	{ SPH_C32(0x62cf0000), SPH_C32(0x14e500cc), SPH_C32(0x0dc80000),
	  SPH_C32(0xbd740000), SPH_C32(0xbf367f13), SPH_C32(0xcf27b68d),
	  SPH_C32(0x2a795bde), SPH_C32(0xd4e53740), SPH_C32(0x65790000),
	  SPH_C32(0xb0780078), SPH_C32(0x23760000), SPH_C32(0xa56b0000),
	  SPH_C32(0xcdc67e4e), SPH_C32(0x06c5c65d), SPH_C32(0xd14915ad),
	  SPH_C32(0xa878ce33) },
	{ SPH_C32(0x76d60000), SPH_C32(0x372f00f0), SPH_C32(0x5d170000),
	  SPH_C32(0xf9c20000), SPH_C32(0xa45a18a3), SPH_C32(0xf3d41af8),
	  SPH_C32(0x4b9f4b6e), SPH_C32(0x0f2fecc0), SPH_C32(0x863a0000),
	  SPH_C32(0x8a36006c), SPH_C32(0xd1b00000), SPH_C32(0x0f250000),
	  SPH_C32(0x16d83ce8), SPH_C32(0x23ae7848), SPH_C32(0xc374a4fb),
	  SPH_C32(0x923657e4) },
	{ SPH_C32(0xc6730000), SPH_C32(0xaf8d000c), SPH_C32(0xa4c10000),
	  SPH_C32(0x218d0000), SPH_C32(0x23111587), SPH_C32(0x7913512f),
	  SPH_C32(0x1d28ac88), SPH_C32(0x378dd173), SPH_C32(0xaf220000),
	  SPH_C32(0x7b6c0090), SPH_C32(0x67e20000), SPH_C32(0x8da20000),
	  SPH_C32(0xc7841e29), SPH_C32(0xb7b744f3), SPH_C32(0x9ac484f4),
	  SPH_C32(0x8b6c72bd) },
	{ SPH_C32(0xd26a0000), SPH_C32(0x8c470030), SPH_C32(0xf41e0000),
	  SPH_C32(0x653b0000), SPH_C32(0x387d7237), SPH_C32(0x45e0fd5a),
	  SPH_C32(0x7ccebc38), SPH_C32(0xec470af3), SPH_C32(0x4c610000),
	  SPH_C32(0x41220084), SPH_C32(0x95240000), SPH_C32(0x27ec0000),
	  SPH_C32(0x1c9a5c8f), SPH_C32(0x92dcfae6), SPH_C32(0x88f935a2),
	  SPH_C32(0xb122eb6a) },
	{ SPH_C32(0x25300000), SPH_C32(0x95c30018), SPH_C32(0x56070000),
	  SPH_C32(0x8bc30000), SPH_C32(0xf80f5721), SPH_C32(0x5c78ef3a),
	  SPH_C32(0x0f151dde), SPH_C32(0x0dc348a4), SPH_C32(0x58780000),
	  SPH_C32(0x62e800b8), SPH_C32(0xc5fb0000), SPH_C32(0x635a0000),
	  SPH_C32(0x07f63b3f), SPH_C32(0xae2f5693), SPH_C32(0xe91f2512),
	  SPH_C32(0x6ae830ea) },
	{ SPH_C32(0x31290000), SPH_C32(0xb6090024), SPH_C32(0x06d80000),
	  SPH_C32(0xcf750000), SPH_C32(0xe3633091), SPH_C32(0x608b434f),
	  SPH_C32(0x6ef30d6e), SPH_C32(0xd6099324), SPH_C32(0xbb3b0000),
	  SPH_C32(0x58a600ac), SPH_C32(0x373d0000), SPH_C32(0xc9140000),
	  SPH_C32(0xdce87999), SPH_C32(0x8b44e886), SPH_C32(0xfb229444),
	  SPH_C32(0x50a6a93d) },
	{ SPH_C32(0x92230000), SPH_C32(0xa9fc0050), SPH_C32(0x816f0000),
	  SPH_C32(0x4b930000), SPH_C32(0x0db45b58), SPH_C32(0x1f5dd43d),
	  SPH_C32(0xa292b44b), SPH_C32(0x49fc8c64), SPH_C32(0x13af0000),
	  SPH_C32(0x87570088), SPH_C32(0x7e610000), SPH_C32(0x5ca90000),
	  SPH_C32(0x699c66ed), SPH_C32(0xf511dca5), SPH_C32(0x9ad65ec3),
	  SPH_C32(0xa75722f3) },
	{ SPH_C32(0x863a0000), SPH_C32(0x8a36006c), SPH_C32(0xd1b00000),
	  SPH_C32(0x0f250000), SPH_C32(0x16d83ce8), SPH_C32(0x23ae7848),
	  SPH_C32(0xc374a4fb), SPH_C32(0x923657e4), SPH_C32(0xf0ec0000),
	  SPH_C32(0xbd19009c), SPH_C32(0x8ca70000), SPH_C32(0xf6e70000),
	  SPH_C32(0xb282244b), SPH_C32(0xd07a62b0), SPH_C32(0x88ebef95),
	  SPH_C32(0x9d19bb24) },
	{ SPH_C32(0x71600000), SPH_C32(0x93b20044), SPH_C32(0x73a90000),
	  SPH_C32(0xe1dd0000), SPH_C32(0xd6aa19fe), SPH_C32(0x3a366a28),
	  SPH_C32(0xb0af051d), SPH_C32(0x73b215b3), SPH_C32(0xe4f50000),
	  SPH_C32(0x9ed300a0), SPH_C32(0xdc780000), SPH_C32(0xb2510000),
	  SPH_C32(0xa9ee43fb), SPH_C32(0xec89cec5), SPH_C32(0xe90dff25),
	  SPH_C32(0x46d360a4) },
	{ SPH_C32(0x65790000), SPH_C32(0xb0780078), SPH_C32(0x23760000),
	  SPH_C32(0xa56b0000), SPH_C32(0xcdc67e4e), SPH_C32(0x06c5c65d),
	  SPH_C32(0xd14915ad), SPH_C32(0xa878ce33), SPH_C32(0x07b60000),
	  SPH_C32(0xa49d00b4), SPH_C32(0x2ebe0000), SPH_C32(0x181f0000),
	  SPH_C32(0x72f0015d), SPH_C32(0xc9e270d0), SPH_C32(0xfb304e73),
	  SPH_C32(0x7c9df973) },
	{ SPH_C32(0x7afe0000), SPH_C32(0x53b60014), SPH_C32(0xbd420000),
	  SPH_C32(0xf0860000), SPH_C32(0x8d096d43), SPH_C32(0x3bb5c979),
	  SPH_C32(0x1d3a76bf), SPH_C32(0x1bb6813d), SPH_C32(0x47ff0000),
	  SPH_C32(0x812600d4), SPH_C32(0x5bcf0000), SPH_C32(0x36b70000),
	  SPH_C32(0x47392832), SPH_C32(0x935f59b7), SPH_C32(0x256c4600),
	  SPH_C32(0xd9267fe4) },
	{ SPH_C32(0x6ee70000), SPH_C32(0x707c0028), SPH_C32(0xed9d0000),
	  SPH_C32(0xb4300000), SPH_C32(0x96650af3), SPH_C32(0x0746650c),
	  SPH_C32(0x7cdc660f), SPH_C32(0xc07c5abd), SPH_C32(0xa4bc0000),
	  SPH_C32(0xbb6800c0), SPH_C32(0xa9090000), SPH_C32(0x9cf90000),
	  SPH_C32(0x9c276a94), SPH_C32(0xb634e7a2), SPH_C32(0x3751f756),
	  SPH_C32(0xe368e633) },
	{ SPH_C32(0x99bd0000), SPH_C32(0x69f80000), SPH_C32(0x4f840000),
	  SPH_C32(0x5ac80000), SPH_C32(0x56172fe5), SPH_C32(0x1ede776c),
	  SPH_C32(0x0f07c7e9), SPH_C32(0x21f818ea), SPH_C32(0xb0a50000),
	  SPH_C32(0x98a200fc), SPH_C32(0xf9d60000), SPH_C32(0xd84f0000),
	  SPH_C32(0x874b0d24), SPH_C32(0x8ac74bd7), SPH_C32(0x56b7e7e6),
	  SPH_C32(0x38a23db3) },
	{ SPH_C32(0x8da40000), SPH_C32(0x4a32003c), SPH_C32(0x1f5b0000),
	  SPH_C32(0x1e7e0000), SPH_C32(0x4d7b4855), SPH_C32(0x222ddb19),
	  SPH_C32(0x6ee1d759), SPH_C32(0xfa32c36a), SPH_C32(0x53e60000),
	  SPH_C32(0xa2ec00e8), SPH_C32(0x0b100000), SPH_C32(0x72010000),
	  SPH_C32(0x5c554f82), SPH_C32(0xafacf5c2), SPH_C32(0x448a56b0),
	  SPH_C32(0x02eca464) },
	{ SPH_C32(0x2eae0000), SPH_C32(0x55c70048), SPH_C32(0x98ec0000),
	  SPH_C32(0x9a980000), SPH_C32(0xa3ac239c), SPH_C32(0x5dfb4c6b),
	  SPH_C32(0xa2806e7c), SPH_C32(0x65c7dc2a), SPH_C32(0xfb720000),
	  SPH_C32(0x7d1d00cc), SPH_C32(0x424c0000), SPH_C32(0xe7bc0000),
	  SPH_C32(0xe92150f6), SPH_C32(0xd1f9c1e1), SPH_C32(0x257e9c37),
	  SPH_C32(0xf51d2faa) },
	{ SPH_C32(0x3ab70000), SPH_C32(0x760d0074), SPH_C32(0xc8330000),
	  SPH_C32(0xde2e0000), SPH_C32(0xb8c0442c), SPH_C32(0x6108e01e),
	  SPH_C32(0xc3667ecc), SPH_C32(0xbe0d07aa), SPH_C32(0x18310000),
	  SPH_C32(0x475300d8), SPH_C32(0xb08a0000), SPH_C32(0x4df20000),
	  SPH_C32(0x323f1250), SPH_C32(0xf4927ff4), SPH_C32(0x37432d61),
	  SPH_C32(0xcf53b67d) },
	{ SPH_C32(0xcded0000), SPH_C32(0x6f89005c), SPH_C32(0x6a2a0000),
	  SPH_C32(0x30d60000), SPH_C32(0x78b2613a), SPH_C32(0x7890f27e),
	  SPH_C32(0xb0bddf2a), SPH_C32(0x5f8945fd), SPH_C32(0x0c280000),
	  SPH_C32(0x649900e4), SPH_C32(0xe0550000), SPH_C32(0x09440000),
	  SPH_C32(0x295375e0), SPH_C32(0xc861d381), SPH_C32(0x56a53dd1),
	  SPH_C32(0x14996dfd) },
	{ SPH_C32(0xd9f40000), SPH_C32(0x4c430060), SPH_C32(0x3af50000),
	  SPH_C32(0x74600000), SPH_C32(0x63de068a), SPH_C32(0x44635e0b),
	  SPH_C32(0xd15bcf9a), SPH_C32(0x84439e7d), SPH_C32(0xef6b0000),
	  SPH_C32(0x5ed700f0), SPH_C32(0x12930000), SPH_C32(0xa30a0000),
	  SPH_C32(0xf24d3746), SPH_C32(0xed0a6d94), SPH_C32(0x44988c87),
	  SPH_C32(0x2ed7f42a) },
	{ SPH_C32(0xaf220000), SPH_C32(0x7b6c0090), SPH_C32(0x67e20000),
	  SPH_C32(0x8da20000), SPH_C32(0xc7841e29), SPH_C32(0xb7b744f3),
	  SPH_C32(0x9ac484f4), SPH_C32(0x8b6c72bd), SPH_C32(0x69510000),
	  SPH_C32(0xd4e1009c), SPH_C32(0xc3230000), SPH_C32(0xac2f0000),
	  SPH_C32(0xe4950bae), SPH_C32(0xcea415dc), SPH_C32(0x87ec287c),
	  SPH_C32(0xbce1a3ce) },
	{ SPH_C32(0xbb3b0000), SPH_C32(0x58a600ac), SPH_C32(0x373d0000),
	  SPH_C32(0xc9140000), SPH_C32(0xdce87999), SPH_C32(0x8b44e886),
	  SPH_C32(0xfb229444), SPH_C32(0x50a6a93d), SPH_C32(0x8a120000),
	  SPH_C32(0xeeaf0088), SPH_C32(0x31e50000), SPH_C32(0x06610000),
	  SPH_C32(0x3f8b4908), SPH_C32(0xebcfabc9), SPH_C32(0x95d1992a),
	  SPH_C32(0x86af3a19) },
	{ SPH_C32(0x4c610000), SPH_C32(0x41220084), SPH_C32(0x95240000),
	  SPH_C32(0x27ec0000), SPH_C32(0x1c9a5c8f), SPH_C32(0x92dcfae6),
	  SPH_C32(0x88f935a2), SPH_C32(0xb122eb6a), SPH_C32(0x9e0b0000),
	  SPH_C32(0xcd6500b4), SPH_C32(0x613a0000), SPH_C32(0x42d70000),
	  SPH_C32(0x24e72eb8), SPH_C32(0xd73c07bc), SPH_C32(0xf437899a),
	  SPH_C32(0x5d65e199) },
	{ SPH_C32(0x58780000), SPH_C32(0x62e800b8), SPH_C32(0xc5fb0000),
	  SPH_C32(0x635a0000), SPH_C32(0x07f63b3f), SPH_C32(0xae2f5693),
	  SPH_C32(0xe91f2512), SPH_C32(0x6ae830ea), SPH_C32(0x7d480000),
	  SPH_C32(0xf72b00a0), SPH_C32(0x93fc0000), SPH_C32(0xe8990000),
	  SPH_C32(0xfff96c1e), SPH_C32(0xf257b9a9), SPH_C32(0xe60a38cc),
	  SPH_C32(0x672b784e) },
	{ SPH_C32(0xfb720000), SPH_C32(0x7d1d00cc), SPH_C32(0x424c0000),
	  SPH_C32(0xe7bc0000), SPH_C32(0xe92150f6), SPH_C32(0xd1f9c1e1),
	  SPH_C32(0x257e9c37), SPH_C32(0xf51d2faa), SPH_C32(0xd5dc0000),
	  SPH_C32(0x28da0084), SPH_C32(0xdaa00000), SPH_C32(0x7d240000),
	  SPH_C32(0x4a8d736a), SPH_C32(0x8c028d8a), SPH_C32(0x87fef24b),
	  SPH_C32(0x90daf380) },
	{ SPH_C32(0xef6b0000), SPH_C32(0x5ed700f0), SPH_C32(0x12930000),
	  SPH_C32(0xa30a0000), SPH_C32(0xf24d3746), SPH_C32(0xed0a6d94),
	  SPH_C32(0x44988c87), SPH_C32(0x2ed7f42a), SPH_C32(0x369f0000),
	  SPH_C32(0x12940090), SPH_C32(0x28660000), SPH_C32(0xd76a0000),
	  SPH_C32(0x919331cc), SPH_C32(0xa969339f), SPH_C32(0x95c3431d),
	  SPH_C32(0xaa946a57) },
	{ SPH_C32(0x18310000), SPH_C32(0x475300d8), SPH_C32(0xb08a0000),
	  SPH_C32(0x4df20000), SPH_C32(0x323f1250), SPH_C32(0xf4927ff4),
	  SPH_C32(0x37432d61), SPH_C32(0xcf53b67d), SPH_C32(0x22860000),
	  SPH_C32(0x315e00ac), SPH_C32(0x78b90000), SPH_C32(0x93dc0000),
	  SPH_C32(0x8aff567c), SPH_C32(0x959a9fea), SPH_C32(0xf42553ad),
	  SPH_C32(0x715eb1d7) },
	{ SPH_C32(0x0c280000), SPH_C32(0x649900e4), SPH_C32(0xe0550000),
	  SPH_C32(0x09440000), SPH_C32(0x295375e0), SPH_C32(0xc861d381),
	  SPH_C32(0x56a53dd1), SPH_C32(0x14996dfd), SPH_C32(0xc1c50000),
	  SPH_C32(0x0b1000b8), SPH_C32(0x8a7f0000), SPH_C32(0x39920000),
	  SPH_C32(0x51e114da), SPH_C32(0xb0f121ff), SPH_C32(0xe618e2fb),
	  SPH_C32(0x4b102800) },
	{ SPH_C32(0x13af0000), SPH_C32(0x87570088), SPH_C32(0x7e610000),
	  SPH_C32(0x5ca90000), SPH_C32(0x699c66ed), SPH_C32(0xf511dca5),
	  SPH_C32(0x9ad65ec3), SPH_C32(0xa75722f3), SPH_C32(0x818c0000),
	  SPH_C32(0x2eab00d8), SPH_C32(0xff0e0000), SPH_C32(0x173a0000),
	  SPH_C32(0x64283db5), SPH_C32(0xea4c0898), SPH_C32(0x3844ea88),
	  SPH_C32(0xeeabae97) },
	{ SPH_C32(0x07b60000), SPH_C32(0xa49d00b4), SPH_C32(0x2ebe0000),
	  SPH_C32(0x181f0000), SPH_C32(0x72f0015d), SPH_C32(0xc9e270d0),
	  SPH_C32(0xfb304e73), SPH_C32(0x7c9df973), SPH_C32(0x62cf0000),
	  SPH_C32(0x14e500cc), SPH_C32(0x0dc80000), SPH_C32(0xbd740000),
	  SPH_C32(0xbf367f13), SPH_C32(0xcf27b68d), SPH_C32(0x2a795bde),
	  SPH_C32(0xd4e53740) },
	{ SPH_C32(0xf0ec0000), SPH_C32(0xbd19009c), SPH_C32(0x8ca70000),
	  SPH_C32(0xf6e70000), SPH_C32(0xb282244b), SPH_C32(0xd07a62b0),
	  SPH_C32(0x88ebef95), SPH_C32(0x9d19bb24), SPH_C32(0x76d60000),
	  SPH_C32(0x372f00f0), SPH_C32(0x5d170000), SPH_C32(0xf9c20000),
	  SPH_C32(0xa45a18a3), SPH_C32(0xf3d41af8), SPH_C32(0x4b9f4b6e),
	  SPH_C32(0x0f2fecc0) },
	{ SPH_C32(0xe4f50000), SPH_C32(0x9ed300a0), SPH_C32(0xdc780000),
	  SPH_C32(0xb2510000), SPH_C32(0xa9ee43fb), SPH_C32(0xec89cec5),
	  SPH_C32(0xe90dff25), SPH_C32(0x46d360a4), SPH_C32(0x95950000),
	  SPH_C32(0x0d6100e4), SPH_C32(0xafd10000), SPH_C32(0x538c0000),
	  SPH_C32(0x7f445a05), SPH_C32(0xd6bfa4ed), SPH_C32(0x59a2fa38),
	  SPH_C32(0x35617517) },
	{ SPH_C32(0x47ff0000), SPH_C32(0x812600d4), SPH_C32(0x5bcf0000),
	  SPH_C32(0x36b70000), SPH_C32(0x47392832), SPH_C32(0x935f59b7),
	  SPH_C32(0x256c4600), SPH_C32(0xd9267fe4), SPH_C32(0x3d010000),
	  SPH_C32(0xd29000c0), SPH_C32(0xe68d0000), SPH_C32(0xc6310000),
	  SPH_C32(0xca304571), SPH_C32(0xa8ea90ce), SPH_C32(0x385630bf),
	  SPH_C32(0xc290fed9) },
	{ SPH_C32(0x53e60000), SPH_C32(0xa2ec00e8), SPH_C32(0x0b100000),
	  SPH_C32(0x72010000), SPH_C32(0x5c554f82), SPH_C32(0xafacf5c2),
	  SPH_C32(0x448a56b0), SPH_C32(0x02eca464), SPH_C32(0xde420000),
	  SPH_C32(0xe8de00d4), SPH_C32(0x144b0000), SPH_C32(0x6c7f0000),
	  SPH_C32(0x112e07d7), SPH_C32(0x8d812edb), SPH_C32(0x2a6b81e9),
	  SPH_C32(0xf8de670e) },
	{ SPH_C32(0xa4bc0000), SPH_C32(0xbb6800c0), SPH_C32(0xa9090000),
	  SPH_C32(0x9cf90000), SPH_C32(0x9c276a94), SPH_C32(0xb634e7a2),
	  SPH_C32(0x3751f756), SPH_C32(0xe368e633), SPH_C32(0xca5b0000),
	  SPH_C32(0xcb1400e8), SPH_C32(0x44940000), SPH_C32(0x28c90000),
	  SPH_C32(0x0a426067), SPH_C32(0xb17282ae), SPH_C32(0x4b8d9159),
	  SPH_C32(0x2314bc8e) },
	{ SPH_C32(0xb0a50000), SPH_C32(0x98a200fc), SPH_C32(0xf9d60000),
	  SPH_C32(0xd84f0000), SPH_C32(0x874b0d24), SPH_C32(0x8ac74bd7),
	  SPH_C32(0x56b7e7e6), SPH_C32(0x38a23db3), SPH_C32(0x29180000),
	  SPH_C32(0xf15a00fc), SPH_C32(0xb6520000), SPH_C32(0x82870000),
	  SPH_C32(0xd15c22c1), SPH_C32(0x94193cbb), SPH_C32(0x59b0200f),
	  SPH_C32(0x195a2559) }
};

static const sph_u32 T512_30[64][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0xac480000), SPH_C32(0x1ba60000), SPH_C32(0x45fb1380),
	  SPH_C32(0x03430000), SPH_C32(0x5a85316a), SPH_C32(0x1fb250b6),
	  SPH_C32(0xfe72c7fe), SPH_C32(0x91e478f6), SPH_C32(0x1e4e0000),
	  SPH_C32(0xdecf0000), SPH_C32(0x6df80180), SPH_C32(0x77240000),
	  SPH_C32(0xec47079e), SPH_C32(0xf4a0694e), SPH_C32(0xcda31812),
	  SPH_C32(0x98aa496e) },
	{ SPH_C32(0x1e4e0000), SPH_C32(0xdecf0000), SPH_C32(0x6df80180),
	  SPH_C32(0x77240000), SPH_C32(0xec47079e), SPH_C32(0xf4a0694e),
	  SPH_C32(0xcda31812), SPH_C32(0x98aa496e), SPH_C32(0xb2060000),
	  SPH_C32(0xc5690000), SPH_C32(0x28031200), SPH_C32(0x74670000),
	  SPH_C32(0xb6c236f4), SPH_C32(0xeb1239f8), SPH_C32(0x33d1dfec),
	  SPH_C32(0x094e3198) },
	{ SPH_C32(0xb2060000), SPH_C32(0xc5690000), SPH_C32(0x28031200),
	  SPH_C32(0x74670000), SPH_C32(0xb6c236f4), SPH_C32(0xeb1239f8),
	  SPH_C32(0x33d1dfec), SPH_C32(0x094e3198), SPH_C32(0xac480000),
	  SPH_C32(0x1ba60000), SPH_C32(0x45fb1380), SPH_C32(0x03430000),
	  SPH_C32(0x5a85316a), SPH_C32(0x1fb250b6), SPH_C32(0xfe72c7fe),
	  SPH_C32(0x91e478f6) },
	{ SPH_C32(0xaec30000), SPH_C32(0x9c4f0001), SPH_C32(0x79d1e000),
	  SPH_C32(0x2c150000), SPH_C32(0x45cc75b3), SPH_C32(0x6650b736),
	  SPH_C32(0xab92f78f), SPH_C32(0xa312567b), SPH_C32(0xdb250000),
	  SPH_C32(0x09290000), SPH_C32(0x49aac000), SPH_C32(0x81e10000),
	  SPH_C32(0xcafe6b59), SPH_C32(0x42793431), SPH_C32(0x43566b76),
	  SPH_C32(0xe86cba2e) },
	{ SPH_C32(0x028b0000), SPH_C32(0x87e90001), SPH_C32(0x3c2af380),
	  SPH_C32(0x2f560000), SPH_C32(0x1f4944d9), SPH_C32(0x79e2e780),
	  SPH_C32(0x55e03071), SPH_C32(0x32f62e8d), SPH_C32(0xc56b0000),
	  SPH_C32(0xd7e60000), SPH_C32(0x2452c180), SPH_C32(0xf6c50000),
	  SPH_C32(0x26b96cc7), SPH_C32(0xb6d95d7f), SPH_C32(0x8ef57364),
	  SPH_C32(0x70c6f340) },
	{ SPH_C32(0xb08d0000), SPH_C32(0x42800001), SPH_C32(0x1429e180),
	  SPH_C32(0x5b310000), SPH_C32(0xa98b722d), SPH_C32(0x92f0de78),
	  SPH_C32(0x6631ef9d), SPH_C32(0x3bb81f15), SPH_C32(0x69230000),
	  SPH_C32(0xcc400000), SPH_C32(0x61a9d200), SPH_C32(0xf5860000),
	  SPH_C32(0x7c3c5dad), SPH_C32(0xa96b0dc9), SPH_C32(0x7087b49a),
	  SPH_C32(0xe1228bb6) },
	{ SPH_C32(0x1cc50000), SPH_C32(0x59260001), SPH_C32(0x51d2f200),
	  SPH_C32(0x58720000), SPH_C32(0xf30e4347), SPH_C32(0x8d428ece),
	  SPH_C32(0x98432863), SPH_C32(0xaa5c67e3), SPH_C32(0x776d0000),
	  SPH_C32(0x128f0000), SPH_C32(0x0c51d380), SPH_C32(0x82a20000),
	  SPH_C32(0x907b5a33), SPH_C32(0x5dcb6487), SPH_C32(0xbd24ac88),
	  SPH_C32(0x7988c2d8) },
	{ SPH_C32(0xdb250000), SPH_C32(0x09290000), SPH_C32(0x49aac000),
	  SPH_C32(0x81e10000), SPH_C32(0xcafe6b59), SPH_C32(0x42793431),
	  SPH_C32(0x43566b76), SPH_C32(0xe86cba2e), SPH_C32(0x75e60000),
	  SPH_C32(0x95660001), SPH_C32(0x307b2000), SPH_C32(0xadf40000),
	  SPH_C32(0x8f321eea), SPH_C32(0x24298307), SPH_C32(0xe8c49cf9),
	  SPH_C32(0x4b7eec55) },
	{ SPH_C32(0x776d0000), SPH_C32(0x128f0000), SPH_C32(0x0c51d380),
	  SPH_C32(0x82a20000), SPH_C32(0x907b5a33), SPH_C32(0x5dcb6487),
	  SPH_C32(0xbd24ac88), SPH_C32(0x7988c2d8), SPH_C32(0x6ba80000),
	  SPH_C32(0x4ba90001), SPH_C32(0x5d832180), SPH_C32(0xdad00000),
	  SPH_C32(0x63751974), SPH_C32(0xd089ea49), SPH_C32(0x256784eb),
	  SPH_C32(0xd3d4a53b) },
	{ SPH_C32(0xc56b0000), SPH_C32(0xd7e60000), SPH_C32(0x2452c180),
	  SPH_C32(0xf6c50000), SPH_C32(0x26b96cc7), SPH_C32(0xb6d95d7f),
	  SPH_C32(0x8ef57364), SPH_C32(0x70c6f340), SPH_C32(0xc7e00000),
	  SPH_C32(0x500f0001), SPH_C32(0x18783200), SPH_C32(0xd9930000),
	  SPH_C32(0x39f0281e), SPH_C32(0xcf3bbaff), SPH_C32(0xdb154315),
	  SPH_C32(0x4230ddcd) },
	{ SPH_C32(0x69230000), SPH_C32(0xcc400000), SPH_C32(0x61a9d200),
	  SPH_C32(0xf5860000), SPH_C32(0x7c3c5dad), SPH_C32(0xa96b0dc9),
	  SPH_C32(0x7087b49a), SPH_C32(0xe1228bb6), SPH_C32(0xd9ae0000),
	  SPH_C32(0x8ec00001), SPH_C32(0x75803380), SPH_C32(0xaeb70000),
	  SPH_C32(0xd5b72f80), SPH_C32(0x3b9bd3b1), SPH_C32(0x16b65b07),
	  SPH_C32(0xda9a94a3) },
	{ SPH_C32(0x75e60000), SPH_C32(0x95660001), SPH_C32(0x307b2000),
	  SPH_C32(0xadf40000), SPH_C32(0x8f321eea), SPH_C32(0x24298307),
	  SPH_C32(0xe8c49cf9), SPH_C32(0x4b7eec55), SPH_C32(0xaec30000),
	  SPH_C32(0x9c4f0001), SPH_C32(0x79d1e000), SPH_C32(0x2c150000),
	  SPH_C32(0x45cc75b3), SPH_C32(0x6650b736), SPH_C32(0xab92f78f),
	  SPH_C32(0xa312567b) },
	{ SPH_C32(0xd9ae0000), SPH_C32(0x8ec00001), SPH_C32(0x75803380),
	  SPH_C32(0xaeb70000), SPH_C32(0xd5b72f80), SPH_C32(0x3b9bd3b1),
	  SPH_C32(0x16b65b07), SPH_C32(0xda9a94a3), SPH_C32(0xb08d0000),
	  SPH_C32(0x42800001), SPH_C32(0x1429e180), SPH_C32(0x5b310000),
	  SPH_C32(0xa98b722d), SPH_C32(0x92f0de78), SPH_C32(0x6631ef9d),
	  SPH_C32(0x3bb81f15) },
	{ SPH_C32(0x6ba80000), SPH_C32(0x4ba90001), SPH_C32(0x5d832180),
	  SPH_C32(0xdad00000), SPH_C32(0x63751974), SPH_C32(0xd089ea49),
	  SPH_C32(0x256784eb), SPH_C32(0xd3d4a53b), SPH_C32(0x1cc50000),
	  SPH_C32(0x59260001), SPH_C32(0x51d2f200), SPH_C32(0x58720000),
	  SPH_C32(0xf30e4347), SPH_C32(0x8d428ece), SPH_C32(0x98432863),
	  SPH_C32(0xaa5c67e3) },
	{ SPH_C32(0xc7e00000), SPH_C32(0x500f0001), SPH_C32(0x18783200),
	  SPH_C32(0xd9930000), SPH_C32(0x39f0281e), SPH_C32(0xcf3bbaff),
	  SPH_C32(0xdb154315), SPH_C32(0x4230ddcd), SPH_C32(0x028b0000),
	  SPH_C32(0x87e90001), SPH_C32(0x3c2af380), SPH_C32(0x2f560000),
	  SPH_C32(0x1f4944d9), SPH_C32(0x79e2e780), SPH_C32(0x55e03071),
	  SPH_C32(0x32f62e8d) },
	{ SPH_C32(0x86790000), SPH_C32(0x3f390002), SPH_C32(0xe19ae000),
	  SPH_C32(0x98560000), SPH_C32(0x9565670e), SPH_C32(0x4e88c8ea),
	  SPH_C32(0xd3dd4944), SPH_C32(0x161ddab9), SPH_C32(0x30b70000),
	  SPH_C32(0xe5d00000), SPH_C32(0xf4f46000), SPH_C32(0x42c40000),
	  SPH_C32(0x63b83d6a), SPH_C32(0x78ba9460), SPH_C32(0x21afa1ea),
	  SPH_C32(0xb0a51834) },
	{ SPH_C32(0x2a310000), SPH_C32(0x249f0002), SPH_C32(0xa461f380),
	  SPH_C32(0x9b150000), SPH_C32(0xcfe05664), SPH_C32(0x513a985c),
	  SPH_C32(0x2daf8eba), SPH_C32(0x87f9a24f), SPH_C32(0x2ef90000),
	  SPH_C32(0x3b1f0000), SPH_C32(0x990c6180), SPH_C32(0x35e00000),
	  SPH_C32(0x8fff3af4), SPH_C32(0x8c1afd2e), SPH_C32(0xec0cb9f8),
	  SPH_C32(0x280f515a) },
	{ SPH_C32(0x98370000), SPH_C32(0xe1f60002), SPH_C32(0x8c62e180),
	  SPH_C32(0xef720000), SPH_C32(0x79226090), SPH_C32(0xba28a1a4),
	  SPH_C32(0x1e7e5156), SPH_C32(0x8eb793d7), SPH_C32(0x82b10000),
	  SPH_C32(0x20b90000), SPH_C32(0xdcf77200), SPH_C32(0x36a30000),
	  SPH_C32(0xd57a0b9e), SPH_C32(0x93a8ad98), SPH_C32(0x127e7e06),
	  SPH_C32(0xb9eb29ac) },
	{ SPH_C32(0x347f0000), SPH_C32(0xfa500002), SPH_C32(0xc999f200),
	  SPH_C32(0xec310000), SPH_C32(0x23a751fa), SPH_C32(0xa59af112),
	  SPH_C32(0xe00c96a8), SPH_C32(0x1f53eb21), SPH_C32(0x9cff0000),
	  SPH_C32(0xfe760000), SPH_C32(0xb10f7380), SPH_C32(0x41870000),
	  SPH_C32(0x393d0c00), SPH_C32(0x6708c4d6), SPH_C32(0xdfdd6614),
	  SPH_C32(0x214160c2) },
	{ SPH_C32(0x28ba0000), SPH_C32(0xa3760003), SPH_C32(0x984b0000),
	  SPH_C32(0xb4430000), SPH_C32(0xd0a912bd), SPH_C32(0x28d87fdc),
	  SPH_C32(0x784fbecb), SPH_C32(0xb50f8cc2), SPH_C32(0xeb920000),
	  SPH_C32(0xecf90000), SPH_C32(0xbd5ea000), SPH_C32(0xc3250000),
	  SPH_C32(0xa9465633), SPH_C32(0x3ac3a051), SPH_C32(0x62f9ca9c),
	  SPH_C32(0x58c9a21a) },
	{ SPH_C32(0x84f20000), SPH_C32(0xb8d00003), SPH_C32(0xddb01380),
	  SPH_C32(0xb7000000), SPH_C32(0x8a2c23d7), SPH_C32(0x376a2f6a),
	  SPH_C32(0x863d7935), SPH_C32(0x24ebf434), SPH_C32(0xf5dc0000),
	  SPH_C32(0x32360000), SPH_C32(0xd0a6a180), SPH_C32(0xb4010000),
	  SPH_C32(0x450151ad), SPH_C32(0xce63c91f), SPH_C32(0xaf5ad28e),
	  SPH_C32(0xc063eb74) },
	{ SPH_C32(0x36f40000), SPH_C32(0x7db90003), SPH_C32(0xf5b30180),
	  SPH_C32(0xc3670000), SPH_C32(0x3cee1523), SPH_C32(0xdc781692),
	  SPH_C32(0xb5eca6d9), SPH_C32(0x2da5c5ac), SPH_C32(0x59940000),
	  SPH_C32(0x29900000), SPH_C32(0x955db200), SPH_C32(0xb7420000),
	  SPH_C32(0x1f8460c7), SPH_C32(0xd1d199a9), SPH_C32(0x51281570),
	  SPH_C32(0x51879382) },
	{ SPH_C32(0x9abc0000), SPH_C32(0x661f0003), SPH_C32(0xb0481200),
	  SPH_C32(0xc0240000), SPH_C32(0x666b2449), SPH_C32(0xc3ca4624),
	  SPH_C32(0x4b9e6127), SPH_C32(0xbc41bd5a), SPH_C32(0x47da0000),
	  SPH_C32(0xf75f0000), SPH_C32(0xf8a5b380), SPH_C32(0xc0660000),
	  SPH_C32(0xf3c36759), SPH_C32(0x2571f0e7), SPH_C32(0x9c8b0d62),
	  SPH_C32(0xc92ddaec) },
	{ SPH_C32(0x5d5c0000), SPH_C32(0x36100002), SPH_C32(0xa8302000),
	  SPH_C32(0x19b70000), SPH_C32(0x5f9b0c57), SPH_C32(0x0cf1fcdb),
	  SPH_C32(0x908b2232), SPH_C32(0xfe716097), SPH_C32(0x45510000),
	  SPH_C32(0x70b60001), SPH_C32(0xc48f4000), SPH_C32(0xef300000),
	  SPH_C32(0xec8a2380), SPH_C32(0x5c931767), SPH_C32(0xc96b3d13),
	  SPH_C32(0xfbdbf461) },
	{ SPH_C32(0xf1140000), SPH_C32(0x2db60002), SPH_C32(0xedcb3380),
	  SPH_C32(0x1af40000), SPH_C32(0x051e3d3d), SPH_C32(0x1343ac6d),
	  SPH_C32(0x6ef9e5cc), SPH_C32(0x6f951861), SPH_C32(0x5b1f0000),
	  SPH_C32(0xae790001), SPH_C32(0xa9774180), SPH_C32(0x98140000),
	  SPH_C32(0x00cd241e), SPH_C32(0xa8337e29), SPH_C32(0x04c82501),
	  SPH_C32(0x6371bd0f) },
	{ SPH_C32(0x43120000), SPH_C32(0xe8df0002), SPH_C32(0xc5c82180),
	  SPH_C32(0x6e930000), SPH_C32(0xb3dc0bc9), SPH_C32(0xf8519595),
	  SPH_C32(0x5d283a20), SPH_C32(0x66db29f9), SPH_C32(0xf7570000),
	  SPH_C32(0xb5df0001), SPH_C32(0xec8c5200), SPH_C32(0x9b570000),
	  SPH_C32(0x5a481574), SPH_C32(0xb7812e9f), SPH_C32(0xfabae2ff),
	  SPH_C32(0xf295c5f9) },
	{ SPH_C32(0xef5a0000), SPH_C32(0xf3790002), SPH_C32(0x80333200),
	  SPH_C32(0x6dd00000), SPH_C32(0xe9593aa3), SPH_C32(0xe7e3c523),
	  SPH_C32(0xa35afdde), SPH_C32(0xf73f510f), SPH_C32(0xe9190000),
	  SPH_C32(0x6b100001), SPH_C32(0x81745380), SPH_C32(0xec730000),
	  SPH_C32(0xb60f12ea), SPH_C32(0x432147d1), SPH_C32(0x3719faed),
	  SPH_C32(0x6a3f8c97) },
	{ SPH_C32(0xf39f0000), SPH_C32(0xaa5f0003), SPH_C32(0xd1e1c000),
	  SPH_C32(0x35a20000), SPH_C32(0x1a5779e4), SPH_C32(0x6aa14bed),
	  SPH_C32(0x3b19d5bd), SPH_C32(0x5d6336ec), SPH_C32(0x9e740000),
	  SPH_C32(0x799f0001), SPH_C32(0x8d258000), SPH_C32(0x6ed10000),
	  SPH_C32(0x267448d9), SPH_C32(0x1eea2356), SPH_C32(0x8a3d5665),
	  SPH_C32(0x13b74e4f) },
	{ SPH_C32(0x5fd70000), SPH_C32(0xb1f90003), SPH_C32(0x941ad380),
	  SPH_C32(0x36e10000), SPH_C32(0x40d2488e), SPH_C32(0x75131b5b),
	  SPH_C32(0xc56b1243), SPH_C32(0xcc874e1a), SPH_C32(0x803a0000),
	  SPH_C32(0xa7500001), SPH_C32(0xe0dd8180), SPH_C32(0x19f50000),
	  SPH_C32(0xca334f47), SPH_C32(0xea4a4a18), SPH_C32(0x479e4e77),
	  SPH_C32(0x8b1d0721) },
	{ SPH_C32(0xedd10000), SPH_C32(0x74900003), SPH_C32(0xbc19c180),
	  SPH_C32(0x42860000), SPH_C32(0xf6107e7a), SPH_C32(0x9e0122a3),
	  SPH_C32(0xf6bacdaf), SPH_C32(0xc5c97f82), SPH_C32(0x2c720000),
	  SPH_C32(0xbcf60001), SPH_C32(0xa5269200), SPH_C32(0x1ab60000),
	  SPH_C32(0x90b67e2d), SPH_C32(0xf5f81aae), SPH_C32(0xb9ec8989),
	  SPH_C32(0x1af97fd7) },
	{ SPH_C32(0x41990000), SPH_C32(0x6f360003), SPH_C32(0xf9e2d200),
	  SPH_C32(0x41c50000), SPH_C32(0xac954f10), SPH_C32(0x81b37215),
	  SPH_C32(0x08c80a51), SPH_C32(0x542d0774), SPH_C32(0x323c0000),
	  SPH_C32(0x62390001), SPH_C32(0xc8de9380), SPH_C32(0x6d920000),
	  SPH_C32(0x7cf179b3), SPH_C32(0x015873e0), SPH_C32(0x744f919b),
	  SPH_C32(0x825336b9) },
	{ SPH_C32(0x30b70000), SPH_C32(0xe5d00000), SPH_C32(0xf4f46000),
	  SPH_C32(0x42c40000), SPH_C32(0x63b83d6a), SPH_C32(0x78ba9460),
	  SPH_C32(0x21afa1ea), SPH_C32(0xb0a51834), SPH_C32(0xb6ce0000),
	  SPH_C32(0xdae90002), SPH_C32(0x156e8000), SPH_C32(0xda920000),
	  SPH_C32(0xf6dd5a64), SPH_C32(0x36325c8a), SPH_C32(0xf272e8ae),
	  SPH_C32(0xa6b8c28d) },
	{ SPH_C32(0x9cff0000), SPH_C32(0xfe760000), SPH_C32(0xb10f7380),
	  SPH_C32(0x41870000), SPH_C32(0x393d0c00), SPH_C32(0x6708c4d6),
	  SPH_C32(0xdfdd6614), SPH_C32(0x214160c2), SPH_C32(0xa8800000),
	  SPH_C32(0x04260002), SPH_C32(0x78968180), SPH_C32(0xadb60000),
	  SPH_C32(0x1a9a5dfa), SPH_C32(0xc29235c4), SPH_C32(0x3fd1f0bc),
	  SPH_C32(0x3e128be3) },
	{ SPH_C32(0x2ef90000), SPH_C32(0x3b1f0000), SPH_C32(0x990c6180),
	  SPH_C32(0x35e00000), SPH_C32(0x8fff3af4), SPH_C32(0x8c1afd2e),
	  SPH_C32(0xec0cb9f8), SPH_C32(0x280f515a), SPH_C32(0x04c80000),
	  SPH_C32(0x1f800002), SPH_C32(0x3d6d9200), SPH_C32(0xaef50000),
	  SPH_C32(0x401f6c90), SPH_C32(0xdd206572), SPH_C32(0xc1a33742),
	  SPH_C32(0xaff6f315) },
	{ SPH_C32(0x82b10000), SPH_C32(0x20b90000), SPH_C32(0xdcf77200),
	  SPH_C32(0x36a30000), SPH_C32(0xd57a0b9e), SPH_C32(0x93a8ad98),
	  SPH_C32(0x127e7e06), SPH_C32(0xb9eb29ac), SPH_C32(0x1a860000),
	  SPH_C32(0xc14f0002), SPH_C32(0x50959380), SPH_C32(0xd9d10000),
	  SPH_C32(0xac586b0e), SPH_C32(0x29800c3c), SPH_C32(0x0c002f50),
	  SPH_C32(0x375cba7b) },
	{ SPH_C32(0x9e740000), SPH_C32(0x799f0001), SPH_C32(0x8d258000),
	  SPH_C32(0x6ed10000), SPH_C32(0x267448d9), SPH_C32(0x1eea2356),
	  SPH_C32(0x8a3d5665), SPH_C32(0x13b74e4f), SPH_C32(0x6deb0000),
	  SPH_C32(0xd3c00002), SPH_C32(0x5cc44000), SPH_C32(0x5b730000),
	  SPH_C32(0x3c23313d), SPH_C32(0x744b68bb), SPH_C32(0xb12483d8),
	  SPH_C32(0x4ed478a3) },
	{ SPH_C32(0x323c0000), SPH_C32(0x62390001), SPH_C32(0xc8de9380),
	  SPH_C32(0x6d920000), SPH_C32(0x7cf179b3), SPH_C32(0x015873e0),
	  SPH_C32(0x744f919b), SPH_C32(0x825336b9), SPH_C32(0x73a50000),
	  SPH_C32(0x0d0f0002), SPH_C32(0x313c4180), SPH_C32(0x2c570000),
	  SPH_C32(0xd06436a3), SPH_C32(0x80eb01f5), SPH_C32(0x7c879bca),
	  SPH_C32(0xd67e31cd) },
	{ SPH_C32(0x803a0000), SPH_C32(0xa7500001), SPH_C32(0xe0dd8180),
	  SPH_C32(0x19f50000), SPH_C32(0xca334f47), SPH_C32(0xea4a4a18),
	  SPH_C32(0x479e4e77), SPH_C32(0x8b1d0721), SPH_C32(0xdfed0000),
	  SPH_C32(0x16a90002), SPH_C32(0x74c75200), SPH_C32(0x2f140000),
	  SPH_C32(0x8ae107c9), SPH_C32(0x9f595143), SPH_C32(0x82f55c34),
	  SPH_C32(0x479a493b) },
	{ SPH_C32(0x2c720000), SPH_C32(0xbcf60001), SPH_C32(0xa5269200),
	  SPH_C32(0x1ab60000), SPH_C32(0x90b67e2d), SPH_C32(0xf5f81aae),
	  SPH_C32(0xb9ec8989), SPH_C32(0x1af97fd7), SPH_C32(0xc1a30000),
	  SPH_C32(0xc8660002), SPH_C32(0x193f5380), SPH_C32(0x58300000),
	  SPH_C32(0x66a60057), SPH_C32(0x6bf9380d), SPH_C32(0x4f564426),
	  SPH_C32(0xdf300055) },
	{ SPH_C32(0xeb920000), SPH_C32(0xecf90000), SPH_C32(0xbd5ea000),
	  SPH_C32(0xc3250000), SPH_C32(0xa9465633), SPH_C32(0x3ac3a051),
	  SPH_C32(0x62f9ca9c), SPH_C32(0x58c9a21a), SPH_C32(0xc3280000),
	  SPH_C32(0x4f8f0003), SPH_C32(0x2515a000), SPH_C32(0x77660000),
	  SPH_C32(0x79ef448e), SPH_C32(0x121bdf8d), SPH_C32(0x1ab67457),
	  SPH_C32(0xedc62ed8) },
	{ SPH_C32(0x47da0000), SPH_C32(0xf75f0000), SPH_C32(0xf8a5b380),
	  SPH_C32(0xc0660000), SPH_C32(0xf3c36759), SPH_C32(0x2571f0e7),
	  SPH_C32(0x9c8b0d62), SPH_C32(0xc92ddaec), SPH_C32(0xdd660000),
	  SPH_C32(0x91400003), SPH_C32(0x48eda180), SPH_C32(0x00420000),
	  SPH_C32(0x95a84310), SPH_C32(0xe6bbb6c3), SPH_C32(0xd7156c45),
	  SPH_C32(0x756c67b6) },
	{ SPH_C32(0xf5dc0000), SPH_C32(0x32360000), SPH_C32(0xd0a6a180),
	  SPH_C32(0xb4010000), SPH_C32(0x450151ad), SPH_C32(0xce63c91f),
	  SPH_C32(0xaf5ad28e), SPH_C32(0xc063eb74), SPH_C32(0x712e0000),
	  SPH_C32(0x8ae60003), SPH_C32(0x0d16b200), SPH_C32(0x03010000),
	  SPH_C32(0xcf2d727a), SPH_C32(0xf909e675), SPH_C32(0x2967abbb),
	  SPH_C32(0xe4881f40) },
	{ SPH_C32(0x59940000), SPH_C32(0x29900000), SPH_C32(0x955db200),
	  SPH_C32(0xb7420000), SPH_C32(0x1f8460c7), SPH_C32(0xd1d199a9),
	  SPH_C32(0x51281570), SPH_C32(0x51879382), SPH_C32(0x6f600000),
	  SPH_C32(0x54290003), SPH_C32(0x60eeb380), SPH_C32(0x74250000),
	  SPH_C32(0x236a75e4), SPH_C32(0x0da98f3b), SPH_C32(0xe4c4b3a9),
	  SPH_C32(0x7c22562e) },
	{ SPH_C32(0x45510000), SPH_C32(0x70b60001), SPH_C32(0xc48f4000),
	  SPH_C32(0xef300000), SPH_C32(0xec8a2380), SPH_C32(0x5c931767),
	  SPH_C32(0xc96b3d13), SPH_C32(0xfbdbf461), SPH_C32(0x180d0000),
	  SPH_C32(0x46a60003), SPH_C32(0x6cbf6000), SPH_C32(0xf6870000),
	  SPH_C32(0xb3112fd7), SPH_C32(0x5062ebbc), SPH_C32(0x59e01f21),
	  SPH_C32(0x05aa94f6) },
	{ SPH_C32(0xe9190000), SPH_C32(0x6b100001), SPH_C32(0x81745380),
	  SPH_C32(0xec730000), SPH_C32(0xb60f12ea), SPH_C32(0x432147d1),
	  SPH_C32(0x3719faed), SPH_C32(0x6a3f8c97), SPH_C32(0x06430000),
	  SPH_C32(0x98690003), SPH_C32(0x01476180), SPH_C32(0x81a30000),
	  SPH_C32(0x5f562849), SPH_C32(0xa4c282f2), SPH_C32(0x94430733),
	  SPH_C32(0x9d00dd98) },
	{ SPH_C32(0x5b1f0000), SPH_C32(0xae790001), SPH_C32(0xa9774180),
	  SPH_C32(0x98140000), SPH_C32(0x00cd241e), SPH_C32(0xa8337e29),
	  SPH_C32(0x04c82501), SPH_C32(0x6371bd0f), SPH_C32(0xaa0b0000),
	  SPH_C32(0x83cf0003), SPH_C32(0x44bc7200), SPH_C32(0x82e00000),
	  SPH_C32(0x05d31923), SPH_C32(0xbb70d244), SPH_C32(0x6a31c0cd),
	  SPH_C32(0x0ce4a56e) },
	{ SPH_C32(0xf7570000), SPH_C32(0xb5df0001), SPH_C32(0xec8c5200),
	  SPH_C32(0x9b570000), SPH_C32(0x5a481574), SPH_C32(0xb7812e9f),
	  SPH_C32(0xfabae2ff), SPH_C32(0xf295c5f9), SPH_C32(0xb4450000),
	  SPH_C32(0x5d000003), SPH_C32(0x29447380), SPH_C32(0xf5c40000),
	  SPH_C32(0xe9941ebd), SPH_C32(0x4fd0bb0a), SPH_C32(0xa792d8df),
	  SPH_C32(0x944eec00) },
	{ SPH_C32(0xb6ce0000), SPH_C32(0xdae90002), SPH_C32(0x156e8000),
	  SPH_C32(0xda920000), SPH_C32(0xf6dd5a64), SPH_C32(0x36325c8a),
	  SPH_C32(0xf272e8ae), SPH_C32(0xa6b8c28d), SPH_C32(0x86790000),
	  SPH_C32(0x3f390002), SPH_C32(0xe19ae000), SPH_C32(0x98560000),
	  SPH_C32(0x9565670e), SPH_C32(0x4e88c8ea), SPH_C32(0xd3dd4944),
	  SPH_C32(0x161ddab9) },
	{ SPH_C32(0x1a860000), SPH_C32(0xc14f0002), SPH_C32(0x50959380),
	  SPH_C32(0xd9d10000), SPH_C32(0xac586b0e), SPH_C32(0x29800c3c),
	  SPH_C32(0x0c002f50), SPH_C32(0x375cba7b), SPH_C32(0x98370000),
	  SPH_C32(0xe1f60002), SPH_C32(0x8c62e180), SPH_C32(0xef720000),
	  SPH_C32(0x79226090), SPH_C32(0xba28a1a4), SPH_C32(0x1e7e5156),
	  SPH_C32(0x8eb793d7) },
	{ SPH_C32(0xa8800000), SPH_C32(0x04260002), SPH_C32(0x78968180),
	  SPH_C32(0xadb60000), SPH_C32(0x1a9a5dfa), SPH_C32(0xc29235c4),
	  SPH_C32(0x3fd1f0bc), SPH_C32(0x3e128be3), SPH_C32(0x347f0000),
	  SPH_C32(0xfa500002), SPH_C32(0xc999f200), SPH_C32(0xec310000),
	  SPH_C32(0x23a751fa), SPH_C32(0xa59af112), SPH_C32(0xe00c96a8),
	  SPH_C32(0x1f53eb21) },
	{ SPH_C32(0x04c80000), SPH_C32(0x1f800002), SPH_C32(0x3d6d9200),
	  SPH_C32(0xaef50000), SPH_C32(0x401f6c90), SPH_C32(0xdd206572),
	  SPH_C32(0xc1a33742), SPH_C32(0xaff6f315), SPH_C32(0x2a310000),
	  SPH_C32(0x249f0002), SPH_C32(0xa461f380), SPH_C32(0x9b150000),
	  SPH_C32(0xcfe05664), SPH_C32(0x513a985c), SPH_C32(0x2daf8eba),
	  SPH_C32(0x87f9a24f) },
	{ SPH_C32(0x180d0000), SPH_C32(0x46a60003), SPH_C32(0x6cbf6000),
	  SPH_C32(0xf6870000), SPH_C32(0xb3112fd7), SPH_C32(0x5062ebbc),
	  SPH_C32(0x59e01f21), SPH_C32(0x05aa94f6), SPH_C32(0x5d5c0000),
	  SPH_C32(0x36100002), SPH_C32(0xa8302000), SPH_C32(0x19b70000),
	  SPH_C32(0x5f9b0c57), SPH_C32(0x0cf1fcdb), SPH_C32(0x908b2232),
	  SPH_C32(0xfe716097) },
	{ SPH_C32(0xb4450000), SPH_C32(0x5d000003), SPH_C32(0x29447380),
	  SPH_C32(0xf5c40000), SPH_C32(0xe9941ebd), SPH_C32(0x4fd0bb0a),
	  SPH_C32(0xa792d8df), SPH_C32(0x944eec00), SPH_C32(0x43120000),
	  SPH_C32(0xe8df0002), SPH_C32(0xc5c82180), SPH_C32(0x6e930000),
	  SPH_C32(0xb3dc0bc9), SPH_C32(0xf8519595), SPH_C32(0x5d283a20),
	  SPH_C32(0x66db29f9) },
	{ SPH_C32(0x06430000), SPH_C32(0x98690003), SPH_C32(0x01476180),
	  SPH_C32(0x81a30000), SPH_C32(0x5f562849), SPH_C32(0xa4c282f2),
	  SPH_C32(0x94430733), SPH_C32(0x9d00dd98), SPH_C32(0xef5a0000),
	  SPH_C32(0xf3790002), SPH_C32(0x80333200), SPH_C32(0x6dd00000),
	  SPH_C32(0xe9593aa3), SPH_C32(0xe7e3c523), SPH_C32(0xa35afdde),
	  SPH_C32(0xf73f510f) },
	{ SPH_C32(0xaa0b0000), SPH_C32(0x83cf0003), SPH_C32(0x44bc7200),
	  SPH_C32(0x82e00000), SPH_C32(0x05d31923), SPH_C32(0xbb70d244),
	  SPH_C32(0x6a31c0cd), SPH_C32(0x0ce4a56e), SPH_C32(0xf1140000),
	  SPH_C32(0x2db60002), SPH_C32(0xedcb3380), SPH_C32(0x1af40000),
	  SPH_C32(0x051e3d3d), SPH_C32(0x1343ac6d), SPH_C32(0x6ef9e5cc),
	  SPH_C32(0x6f951861) },
	{ SPH_C32(0x6deb0000), SPH_C32(0xd3c00002), SPH_C32(0x5cc44000),
	  SPH_C32(0x5b730000), SPH_C32(0x3c23313d), SPH_C32(0x744b68bb),
	  SPH_C32(0xb12483d8), SPH_C32(0x4ed478a3), SPH_C32(0xf39f0000),
	  SPH_C32(0xaa5f0003), SPH_C32(0xd1e1c000), SPH_C32(0x35a20000),
	  SPH_C32(0x1a5779e4), SPH_C32(0x6aa14bed), SPH_C32(0x3b19d5bd),
	  SPH_C32(0x5d6336ec) },
	{ SPH_C32(0xc1a30000), SPH_C32(0xc8660002), SPH_C32(0x193f5380),
	  SPH_C32(0x58300000), SPH_C32(0x66a60057), SPH_C32(0x6bf9380d),
	  SPH_C32(0x4f564426), SPH_C32(0xdf300055), SPH_C32(0xedd10000),
	  SPH_C32(0x74900003), SPH_C32(0xbc19c180), SPH_C32(0x42860000),
	  SPH_C32(0xf6107e7a), SPH_C32(0x9e0122a3), SPH_C32(0xf6bacdaf),
	  SPH_C32(0xc5c97f82) },
	{ SPH_C32(0x73a50000), SPH_C32(0x0d0f0002), SPH_C32(0x313c4180),
	  SPH_C32(0x2c570000), SPH_C32(0xd06436a3), SPH_C32(0x80eb01f5),
	  SPH_C32(0x7c879bca), SPH_C32(0xd67e31cd), SPH_C32(0x41990000),
	  SPH_C32(0x6f360003), SPH_C32(0xf9e2d200), SPH_C32(0x41c50000),
	  SPH_C32(0xac954f10), SPH_C32(0x81b37215), SPH_C32(0x08c80a51),
	  SPH_C32(0x542d0774) },
	{ SPH_C32(0xdfed0000), SPH_C32(0x16a90002), SPH_C32(0x74c75200),
	  SPH_C32(0x2f140000), SPH_C32(0x8ae107c9), SPH_C32(0x9f595143),
	  SPH_C32(0x82f55c34), SPH_C32(0x479a493b), SPH_C32(0x5fd70000),
	  SPH_C32(0xb1f90003), SPH_C32(0x941ad380), SPH_C32(0x36e10000),
	  SPH_C32(0x40d2488e), SPH_C32(0x75131b5b), SPH_C32(0xc56b1243),
	  SPH_C32(0xcc874e1a) },
	{ SPH_C32(0xc3280000), SPH_C32(0x4f8f0003), SPH_C32(0x2515a000),
	  SPH_C32(0x77660000), SPH_C32(0x79ef448e), SPH_C32(0x121bdf8d),
	  SPH_C32(0x1ab67457), SPH_C32(0xedc62ed8), SPH_C32(0x28ba0000),
	  SPH_C32(0xa3760003), SPH_C32(0x984b0000), SPH_C32(0xb4430000),
	  SPH_C32(0xd0a912bd), SPH_C32(0x28d87fdc), SPH_C32(0x784fbecb),
	  SPH_C32(0xb50f8cc2) },
	{ SPH_C32(0x6f600000), SPH_C32(0x54290003), SPH_C32(0x60eeb380),
	  SPH_C32(0x74250000), SPH_C32(0x236a75e4), SPH_C32(0x0da98f3b),
	  SPH_C32(0xe4c4b3a9), SPH_C32(0x7c22562e), SPH_C32(0x36f40000),
	  SPH_C32(0x7db90003), SPH_C32(0xf5b30180), SPH_C32(0xc3670000),
	  SPH_C32(0x3cee1523), SPH_C32(0xdc781692), SPH_C32(0xb5eca6d9),
	  SPH_C32(0x2da5c5ac) },
	{ SPH_C32(0xdd660000), SPH_C32(0x91400003), SPH_C32(0x48eda180),
	  SPH_C32(0x00420000), SPH_C32(0x95a84310), SPH_C32(0xe6bbb6c3),
	  SPH_C32(0xd7156c45), SPH_C32(0x756c67b6), SPH_C32(0x9abc0000),
	  SPH_C32(0x661f0003), SPH_C32(0xb0481200), SPH_C32(0xc0240000),
	  SPH_C32(0x666b2449), SPH_C32(0xc3ca4624), SPH_C32(0x4b9e6127),
	  SPH_C32(0xbc41bd5a) },
	{ SPH_C32(0x712e0000), SPH_C32(0x8ae60003), SPH_C32(0x0d16b200),
	  SPH_C32(0x03010000), SPH_C32(0xcf2d727a), SPH_C32(0xf909e675),
	  SPH_C32(0x2967abbb), SPH_C32(0xe4881f40), SPH_C32(0x84f20000),
	  SPH_C32(0xb8d00003), SPH_C32(0xddb01380), SPH_C32(0xb7000000),
	  SPH_C32(0x8a2c23d7), SPH_C32(0x376a2f6a), SPH_C32(0x863d7935),
	  SPH_C32(0x24ebf434) }
};

static const sph_u32 T512_36[64][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x52500000), SPH_C32(0x29540000), SPH_C32(0x6a61004e),
	  SPH_C32(0xf0ff0000), SPH_C32(0x9a317eec), SPH_C32(0x452341ce),
	  SPH_C32(0xcf568fe5), SPH_C32(0x5303130f), SPH_C32(0x538d0000),
	  SPH_C32(0xa9fc0000), SPH_C32(0x9ef70006), SPH_C32(0x56ff0000),
	  SPH_C32(0x0ae4004e), SPH_C32(0x92c5cdf9), SPH_C32(0xa9444018),
	  SPH_C32(0x7f975691) },
	{ SPH_C32(0x538d0000), SPH_C32(0xa9fc0000), SPH_C32(0x9ef70006),
	  SPH_C32(0x56ff0000), SPH_C32(0x0ae4004e), SPH_C32(0x92c5cdf9),
	  SPH_C32(0xa9444018), SPH_C32(0x7f975691), SPH_C32(0x01dd0000),
	  SPH_C32(0x80a80000), SPH_C32(0xf4960048), SPH_C32(0xa6000000),
	  SPH_C32(0x90d57ea2), SPH_C32(0xd7e68c37), SPH_C32(0x6612cffd),
	  SPH_C32(0x2c94459e) },
	{ SPH_C32(0x01dd0000), SPH_C32(0x80a80000), SPH_C32(0xf4960048),
	  SPH_C32(0xa6000000), SPH_C32(0x90d57ea2), SPH_C32(0xd7e68c37),
	  SPH_C32(0x6612cffd), SPH_C32(0x2c94459e), SPH_C32(0x52500000),
	  SPH_C32(0x29540000), SPH_C32(0x6a61004e), SPH_C32(0xf0ff0000),
	  SPH_C32(0x9a317eec), SPH_C32(0x452341ce), SPH_C32(0xcf568fe5),
	  SPH_C32(0x5303130f) },
	{ SPH_C32(0xcc140000), SPH_C32(0xa5630000), SPH_C32(0x5ab90780),
	  SPH_C32(0x3b500000), SPH_C32(0x4bd013ff), SPH_C32(0x879b3418),
	  SPH_C32(0x694348c1), SPH_C32(0xca5a87fe), SPH_C32(0x819e0000),
	  SPH_C32(0xec570000), SPH_C32(0x66320280), SPH_C32(0x95f30000),
	  SPH_C32(0x5da92802), SPH_C32(0x48f43cbc), SPH_C32(0xe65aa22d),
	  SPH_C32(0x8e67b7fa) },
	{ SPH_C32(0x9e440000), SPH_C32(0x8c370000), SPH_C32(0x30d807ce),
	  SPH_C32(0xcbaf0000), SPH_C32(0xd1e16d13), SPH_C32(0xc2b875d6),
	  SPH_C32(0xa615c724), SPH_C32(0x995994f1), SPH_C32(0xd2130000),
	  SPH_C32(0x45ab0000), SPH_C32(0xf8c50286), SPH_C32(0xc30c0000),
	  SPH_C32(0x574d284c), SPH_C32(0xda31f145), SPH_C32(0x4f1ee235),
	  SPH_C32(0xf1f0e16b) },
	{ SPH_C32(0x9f990000), SPH_C32(0x0c9f0000), SPH_C32(0xc44e0786),
	  SPH_C32(0x6daf0000), SPH_C32(0x413413b1), SPH_C32(0x155ef9e1),
	  SPH_C32(0xc00708d9), SPH_C32(0xb5cdd16f), SPH_C32(0x80430000),
	  SPH_C32(0x6cff0000), SPH_C32(0x92a402c8), SPH_C32(0x33f30000),
	  SPH_C32(0xcd7c56a0), SPH_C32(0x9f12b08b), SPH_C32(0x80486dd0),
	  SPH_C32(0xa2f3f264) },
	{ SPH_C32(0xcdc90000), SPH_C32(0x25cb0000), SPH_C32(0xae2f07c8),
	  SPH_C32(0x9d500000), SPH_C32(0xdb056d5d), SPH_C32(0x507db82f),
	  SPH_C32(0x0f51873c), SPH_C32(0xe6cec260), SPH_C32(0xd3ce0000),
	  SPH_C32(0xc5030000), SPH_C32(0x0c5302ce), SPH_C32(0x650c0000),
	  SPH_C32(0xc79856ee), SPH_C32(0x0dd77d72), SPH_C32(0x290c2dc8),
	  SPH_C32(0xdd64a4f5) },
	{ SPH_C32(0x819e0000), SPH_C32(0xec570000), SPH_C32(0x66320280),
	  SPH_C32(0x95f30000), SPH_C32(0x5da92802), SPH_C32(0x48f43cbc),
	  SPH_C32(0xe65aa22d), SPH_C32(0x8e67b7fa), SPH_C32(0x4d8a0000),
	  SPH_C32(0x49340000), SPH_C32(0x3c8b0500), SPH_C32(0xaea30000),
	  SPH_C32(0x16793bfd), SPH_C32(0xcf6f08a4), SPH_C32(0x8f19eaec),
	  SPH_C32(0x443d3004) },
	{ SPH_C32(0xd3ce0000), SPH_C32(0xc5030000), SPH_C32(0x0c5302ce),
	  SPH_C32(0x650c0000), SPH_C32(0xc79856ee), SPH_C32(0x0dd77d72),
	  SPH_C32(0x290c2dc8), SPH_C32(0xdd64a4f5), SPH_C32(0x1e070000),
	  SPH_C32(0xe0c80000), SPH_C32(0xa27c0506), SPH_C32(0xf85c0000),
	  SPH_C32(0x1c9d3bb3), SPH_C32(0x5daac55d), SPH_C32(0x265daaf4),
	  SPH_C32(0x3baa6695) },
	{ SPH_C32(0xd2130000), SPH_C32(0x45ab0000), SPH_C32(0xf8c50286),
	  SPH_C32(0xc30c0000), SPH_C32(0x574d284c), SPH_C32(0xda31f145),
	  SPH_C32(0x4f1ee235), SPH_C32(0xf1f0e16b), SPH_C32(0x4c570000),
	  SPH_C32(0xc99c0000), SPH_C32(0xc81d0548), SPH_C32(0x08a30000),
	  SPH_C32(0x86ac455f), SPH_C32(0x18898493), SPH_C32(0xe90b2511),
	  SPH_C32(0x68a9759a) },
	{ SPH_C32(0x80430000), SPH_C32(0x6cff0000), SPH_C32(0x92a402c8),
	  SPH_C32(0x33f30000), SPH_C32(0xcd7c56a0), SPH_C32(0x9f12b08b),
	  SPH_C32(0x80486dd0), SPH_C32(0xa2f3f264), SPH_C32(0x1fda0000),
	  SPH_C32(0x60600000), SPH_C32(0x56ea054e), SPH_C32(0x5e5c0000),
	  SPH_C32(0x8c484511), SPH_C32(0x8a4c496a), SPH_C32(0x404f6509),
	  SPH_C32(0x173e230b) },
	{ SPH_C32(0x4d8a0000), SPH_C32(0x49340000), SPH_C32(0x3c8b0500),
	  SPH_C32(0xaea30000), SPH_C32(0x16793bfd), SPH_C32(0xcf6f08a4),
	  SPH_C32(0x8f19eaec), SPH_C32(0x443d3004), SPH_C32(0xcc140000),
	  SPH_C32(0xa5630000), SPH_C32(0x5ab90780), SPH_C32(0x3b500000),
	  SPH_C32(0x4bd013ff), SPH_C32(0x879b3418), SPH_C32(0x694348c1),
	  SPH_C32(0xca5a87fe) },
	{ SPH_C32(0x1fda0000), SPH_C32(0x60600000), SPH_C32(0x56ea054e),
	  SPH_C32(0x5e5c0000), SPH_C32(0x8c484511), SPH_C32(0x8a4c496a),
	  SPH_C32(0x404f6509), SPH_C32(0x173e230b), SPH_C32(0x9f990000),
	  SPH_C32(0x0c9f0000), SPH_C32(0xc44e0786), SPH_C32(0x6daf0000),
	  SPH_C32(0x413413b1), SPH_C32(0x155ef9e1), SPH_C32(0xc00708d9),
	  SPH_C32(0xb5cdd16f) },
	{ SPH_C32(0x1e070000), SPH_C32(0xe0c80000), SPH_C32(0xa27c0506),
	  SPH_C32(0xf85c0000), SPH_C32(0x1c9d3bb3), SPH_C32(0x5daac55d),
	  SPH_C32(0x265daaf4), SPH_C32(0x3baa6695), SPH_C32(0xcdc90000),
	  SPH_C32(0x25cb0000), SPH_C32(0xae2f07c8), SPH_C32(0x9d500000),
	  SPH_C32(0xdb056d5d), SPH_C32(0x507db82f), SPH_C32(0x0f51873c),
	  SPH_C32(0xe6cec260) },
	{ SPH_C32(0x4c570000), SPH_C32(0xc99c0000), SPH_C32(0xc81d0548),
	  SPH_C32(0x08a30000), SPH_C32(0x86ac455f), SPH_C32(0x18898493),
	  SPH_C32(0xe90b2511), SPH_C32(0x68a9759a), SPH_C32(0x9e440000),
	  SPH_C32(0x8c370000), SPH_C32(0x30d807ce), SPH_C32(0xcbaf0000),
	  SPH_C32(0xd1e16d13), SPH_C32(0xc2b875d6), SPH_C32(0xa615c724),
	  SPH_C32(0x995994f1) },
	{ SPH_C32(0x78230000), SPH_C32(0x12fc0000), SPH_C32(0xa93a0b80),
	  SPH_C32(0x90a50000), SPH_C32(0x713e2879), SPH_C32(0x7ee98924),
	  SPH_C32(0xf08ca062), SPH_C32(0x636f8bab), SPH_C32(0x02af0000),
	  SPH_C32(0xb7280000), SPH_C32(0xba1c0300), SPH_C32(0x56980000),
	  SPH_C32(0xba8d45d3), SPH_C32(0x8048c667), SPH_C32(0xa95c149a),
	  SPH_C32(0xf4f6ea7b) },
	{ SPH_C32(0x2a730000), SPH_C32(0x3ba80000), SPH_C32(0xc35b0bce),
	  SPH_C32(0x605a0000), SPH_C32(0xeb0f5695), SPH_C32(0x3bcac8ea),
	  SPH_C32(0x3fda2f87), SPH_C32(0x306c98a4), SPH_C32(0x51220000),
	  SPH_C32(0x1ed40000), SPH_C32(0x24eb0306), SPH_C32(0x00670000),
	  SPH_C32(0xb069459d), SPH_C32(0x128d0b9e), SPH_C32(0x00185482),
	  SPH_C32(0x8b61bcea) },
	{ SPH_C32(0x2bae0000), SPH_C32(0xbb000000), SPH_C32(0x37cd0b86),
	  SPH_C32(0xc65a0000), SPH_C32(0x7bda2837), SPH_C32(0xec2c44dd),
	  SPH_C32(0x59c8e07a), SPH_C32(0x1cf8dd3a), SPH_C32(0x03720000),
	  SPH_C32(0x37800000), SPH_C32(0x4e8a0348), SPH_C32(0xf0980000),
	  SPH_C32(0x2a583b71), SPH_C32(0x57ae4a50), SPH_C32(0xcf4edb67),
	  SPH_C32(0xd862afe5) },
	{ SPH_C32(0x79fe0000), SPH_C32(0x92540000), SPH_C32(0x5dac0bc8),
	  SPH_C32(0x36a50000), SPH_C32(0xe1eb56db), SPH_C32(0xa90f0513),
	  SPH_C32(0x969e6f9f), SPH_C32(0x4ffbce35), SPH_C32(0x50ff0000),
	  SPH_C32(0x9e7c0000), SPH_C32(0xd07d034e), SPH_C32(0xa6670000),
	  SPH_C32(0x20bc3b3f), SPH_C32(0xc56b87a9), SPH_C32(0x660a9b7f),
	  SPH_C32(0xa7f5f974) },
	{ SPH_C32(0xb4370000), SPH_C32(0xb79f0000), SPH_C32(0xf3830c00),
	  SPH_C32(0xabf50000), SPH_C32(0x3aee3b86), SPH_C32(0xf972bd3c),
	  SPH_C32(0x99cfe8a3), SPH_C32(0xa9350c55), SPH_C32(0x83310000),
	  SPH_C32(0x5b7f0000), SPH_C32(0xdc2e0180), SPH_C32(0xc36b0000),
	  SPH_C32(0xe7246dd1), SPH_C32(0xc8bcfadb), SPH_C32(0x4f06b6b7),
	  SPH_C32(0x7a915d81) },
	{ SPH_C32(0xe6670000), SPH_C32(0x9ecb0000), SPH_C32(0x99e20c4e),
	  SPH_C32(0x5b0a0000), SPH_C32(0xa0df456a), SPH_C32(0xbc51fcf2),
	  SPH_C32(0x56996746), SPH_C32(0xfa361f5a), SPH_C32(0xd0bc0000),
	  SPH_C32(0xf2830000), SPH_C32(0x42d90186), SPH_C32(0x95940000),
	  SPH_C32(0xedc06d9f), SPH_C32(0x5a793722), SPH_C32(0xe642f6af),
	  SPH_C32(0x05060b10) },
	{ SPH_C32(0xe7ba0000), SPH_C32(0x1e630000), SPH_C32(0x6d740c06),
	  SPH_C32(0xfd0a0000), SPH_C32(0x300a3bc8), SPH_C32(0x6bb770c5),
	  SPH_C32(0x308ba8bb), SPH_C32(0xd6a25ac4), SPH_C32(0x82ec0000),
	  SPH_C32(0xdbd70000), SPH_C32(0x28b801c8), SPH_C32(0x656b0000),
	  SPH_C32(0x77f11373), SPH_C32(0x1f5a76ec), SPH_C32(0x2914794a),
	  SPH_C32(0x5605181f) },
	{ SPH_C32(0xb5ea0000), SPH_C32(0x37370000), SPH_C32(0x07150c48),
	  SPH_C32(0x0df50000), SPH_C32(0xaa3b4524), SPH_C32(0x2e94310b),
	  SPH_C32(0xffdd275e), SPH_C32(0x85a149cb), SPH_C32(0xd1610000),
	  SPH_C32(0x722b0000), SPH_C32(0xb64f01ce), SPH_C32(0x33940000),
	  SPH_C32(0x7d15133d), SPH_C32(0x8d9fbb15), SPH_C32(0x80503952),
	  SPH_C32(0x29924e8e) },
	{ SPH_C32(0xf9bd0000), SPH_C32(0xfeab0000), SPH_C32(0xcf080900),
	  SPH_C32(0x05560000), SPH_C32(0x2c97007b), SPH_C32(0x361db598),
	  SPH_C32(0x16d6024f), SPH_C32(0xed083c51), SPH_C32(0x4f250000),
	  SPH_C32(0xfe1c0000), SPH_C32(0x86970600), SPH_C32(0xf83b0000),
	  SPH_C32(0xacf47e2e), SPH_C32(0x4f27cec3), SPH_C32(0x2645fe76),
	  SPH_C32(0xb0cbda7f) },
	{ SPH_C32(0xabed0000), SPH_C32(0xd7ff0000), SPH_C32(0xa569094e),
	  SPH_C32(0xf5a90000), SPH_C32(0xb6a67e97), SPH_C32(0x733ef456),
	  SPH_C32(0xd9808daa), SPH_C32(0xbe0b2f5e), SPH_C32(0x1ca80000),
	  SPH_C32(0x57e00000), SPH_C32(0x18600606), SPH_C32(0xaec40000),
	  SPH_C32(0xa6107e60), SPH_C32(0xdde2033a), SPH_C32(0x8f01be6e),
	  SPH_C32(0xcf5c8cee) },
	{ SPH_C32(0xaa300000), SPH_C32(0x57570000), SPH_C32(0x51ff0906),
	  SPH_C32(0x53a90000), SPH_C32(0x26730035), SPH_C32(0xa4d87861),
	  SPH_C32(0xbf924257), SPH_C32(0x929f6ac0), SPH_C32(0x4ef80000),
	  SPH_C32(0x7eb40000), SPH_C32(0x72010648), SPH_C32(0x5e3b0000),
	  SPH_C32(0x3c21008c), SPH_C32(0x98c142f4), SPH_C32(0x4057318b),
	  SPH_C32(0x9c5f9fe1) },
	{ SPH_C32(0xf8600000), SPH_C32(0x7e030000), SPH_C32(0x3b9e0948),
	  SPH_C32(0xa3560000), SPH_C32(0xbc427ed9), SPH_C32(0xe1fb39af),
	  SPH_C32(0x70c4cdb2), SPH_C32(0xc19c79cf), SPH_C32(0x1d750000),
	  SPH_C32(0xd7480000), SPH_C32(0xecf6064e), SPH_C32(0x08c40000),
	  SPH_C32(0x36c500c2), SPH_C32(0x0a048f0d), SPH_C32(0xe9137193),
	  SPH_C32(0xe3c8c970) },
	{ SPH_C32(0x35a90000), SPH_C32(0x5bc80000), SPH_C32(0x95b10e80),
	  SPH_C32(0x3e060000), SPH_C32(0x67471384), SPH_C32(0xb1868180),
	  SPH_C32(0x7f954a8e), SPH_C32(0x2752bbaf), SPH_C32(0xcebb0000),
	  SPH_C32(0x124b0000), SPH_C32(0xe0a50480), SPH_C32(0x6dc80000),
	  SPH_C32(0xf15d562c), SPH_C32(0x07d3f27f), SPH_C32(0xc01f5c5b),
	  SPH_C32(0x3eac6d85) },
	{ SPH_C32(0x67f90000), SPH_C32(0x729c0000), SPH_C32(0xffd00ece),
	  SPH_C32(0xcef90000), SPH_C32(0xfd766d68), SPH_C32(0xf4a5c04e),
	  SPH_C32(0xb0c3c56b), SPH_C32(0x7451a8a0), SPH_C32(0x9d360000),
	  SPH_C32(0xbbb70000), SPH_C32(0x7e520486), SPH_C32(0x3b370000),
	  SPH_C32(0xfbb95662), SPH_C32(0x95163f86), SPH_C32(0x695b1c43),
	  SPH_C32(0x413b3b14) },
	{ SPH_C32(0x66240000), SPH_C32(0xf2340000), SPH_C32(0x0b460e86),
	  SPH_C32(0x68f90000), SPH_C32(0x6da313ca), SPH_C32(0x23434c79),
	  SPH_C32(0xd6d10a96), SPH_C32(0x58c5ed3e), SPH_C32(0xcf660000),
	  SPH_C32(0x92e30000), SPH_C32(0x143304c8), SPH_C32(0xcbc80000),
	  SPH_C32(0x6188288e), SPH_C32(0xd0357e48), SPH_C32(0xa60d93a6),
	  SPH_C32(0x1238281b) },
	{ SPH_C32(0x34740000), SPH_C32(0xdb600000), SPH_C32(0x61270ec8),
	  SPH_C32(0x98060000), SPH_C32(0xf7926d26), SPH_C32(0x66600db7),
	  SPH_C32(0x19878573), SPH_C32(0x0bc6fe31), SPH_C32(0x9ceb0000),
	  SPH_C32(0x3b1f0000), SPH_C32(0x8ac404ce), SPH_C32(0x9d370000),
	  SPH_C32(0x6b6c28c0), SPH_C32(0x42f0b3b1), SPH_C32(0x0f49d3be),
	  SPH_C32(0x6daf7e8a) },
	{ SPH_C32(0x02af0000), SPH_C32(0xb7280000), SPH_C32(0xba1c0300),
	  SPH_C32(0x56980000), SPH_C32(0xba8d45d3), SPH_C32(0x8048c667),
	  SPH_C32(0xa95c149a), SPH_C32(0xf4f6ea7b), SPH_C32(0x7a8c0000),
	  SPH_C32(0xa5d40000), SPH_C32(0x13260880), SPH_C32(0xc63d0000),
	  SPH_C32(0xcbb36daa), SPH_C32(0xfea14f43), SPH_C32(0x59d0b4f8),
	  SPH_C32(0x979961d0) },
	{ SPH_C32(0x50ff0000), SPH_C32(0x9e7c0000), SPH_C32(0xd07d034e),
	  SPH_C32(0xa6670000), SPH_C32(0x20bc3b3f), SPH_C32(0xc56b87a9),
	  SPH_C32(0x660a9b7f), SPH_C32(0xa7f5f974), SPH_C32(0x29010000),
	  SPH_C32(0x0c280000), SPH_C32(0x8dd10886), SPH_C32(0x90c20000),
	  SPH_C32(0xc1576de4), SPH_C32(0x6c6482ba), SPH_C32(0xf094f4e0),
	  SPH_C32(0xe80e3741) },
	{ SPH_C32(0x51220000), SPH_C32(0x1ed40000), SPH_C32(0x24eb0306),
	  SPH_C32(0x00670000), SPH_C32(0xb069459d), SPH_C32(0x128d0b9e),
	  SPH_C32(0x00185482), SPH_C32(0x8b61bcea), SPH_C32(0x7b510000),
	  SPH_C32(0x257c0000), SPH_C32(0xe7b008c8), SPH_C32(0x603d0000),
	  SPH_C32(0x5b661308), SPH_C32(0x2947c374), SPH_C32(0x3fc27b05),
	  SPH_C32(0xbb0d244e) },
	{ SPH_C32(0x03720000), SPH_C32(0x37800000), SPH_C32(0x4e8a0348),
	  SPH_C32(0xf0980000), SPH_C32(0x2a583b71), SPH_C32(0x57ae4a50),
	  SPH_C32(0xcf4edb67), SPH_C32(0xd862afe5), SPH_C32(0x28dc0000),
	  SPH_C32(0x8c800000), SPH_C32(0x794708ce), SPH_C32(0x36c20000),
	  SPH_C32(0x51821346), SPH_C32(0xbb820e8d), SPH_C32(0x96863b1d),
	  SPH_C32(0xc49a72df) },
	{ SPH_C32(0xcebb0000), SPH_C32(0x124b0000), SPH_C32(0xe0a50480),
	  SPH_C32(0x6dc80000), SPH_C32(0xf15d562c), SPH_C32(0x07d3f27f),
	  SPH_C32(0xc01f5c5b), SPH_C32(0x3eac6d85), SPH_C32(0xfb120000),
	  SPH_C32(0x49830000), SPH_C32(0x75140a00), SPH_C32(0x53ce0000),
	  SPH_C32(0x961a45a8), SPH_C32(0xb65573ff), SPH_C32(0xbf8a16d5),
	  SPH_C32(0x19fed62a) },
	{ SPH_C32(0x9ceb0000), SPH_C32(0x3b1f0000), SPH_C32(0x8ac404ce),
	  SPH_C32(0x9d370000), SPH_C32(0x6b6c28c0), SPH_C32(0x42f0b3b1),
	  SPH_C32(0x0f49d3be), SPH_C32(0x6daf7e8a), SPH_C32(0xa89f0000),
	  SPH_C32(0xe07f0000), SPH_C32(0xebe30a06), SPH_C32(0x05310000),
	  SPH_C32(0x9cfe45e6), SPH_C32(0x2490be06), SPH_C32(0x16ce56cd),
	  SPH_C32(0x666980bb) },
	{ SPH_C32(0x9d360000), SPH_C32(0xbbb70000), SPH_C32(0x7e520486),
	  SPH_C32(0x3b370000), SPH_C32(0xfbb95662), SPH_C32(0x95163f86),
	  SPH_C32(0x695b1c43), SPH_C32(0x413b3b14), SPH_C32(0xfacf0000),
	  SPH_C32(0xc92b0000), SPH_C32(0x81820a48), SPH_C32(0xf5ce0000),
	  SPH_C32(0x06cf3b0a), SPH_C32(0x61b3ffc8), SPH_C32(0xd998d928),
	  SPH_C32(0x356a93b4) },
	{ SPH_C32(0xcf660000), SPH_C32(0x92e30000), SPH_C32(0x143304c8),
	  SPH_C32(0xcbc80000), SPH_C32(0x6188288e), SPH_C32(0xd0357e48),
	  SPH_C32(0xa60d93a6), SPH_C32(0x1238281b), SPH_C32(0xa9420000),
	  SPH_C32(0x60d70000), SPH_C32(0x1f750a4e), SPH_C32(0xa3310000),
	  SPH_C32(0x0c2b3b44), SPH_C32(0xf3763231), SPH_C32(0x70dc9930),
	  SPH_C32(0x4afdc525) },
	{ SPH_C32(0x83310000), SPH_C32(0x5b7f0000), SPH_C32(0xdc2e0180),
	  SPH_C32(0xc36b0000), SPH_C32(0xe7246dd1), SPH_C32(0xc8bcfadb),
	  SPH_C32(0x4f06b6b7), SPH_C32(0x7a915d81), SPH_C32(0x37060000),
	  SPH_C32(0xece00000), SPH_C32(0x2fad0d80), SPH_C32(0x689e0000),
	  SPH_C32(0xddca5657), SPH_C32(0x31ce47e7), SPH_C32(0xd6c95e14),
	  SPH_C32(0xd3a451d4) },
	{ SPH_C32(0xd1610000), SPH_C32(0x722b0000), SPH_C32(0xb64f01ce),
	  SPH_C32(0x33940000), SPH_C32(0x7d15133d), SPH_C32(0x8d9fbb15),
	  SPH_C32(0x80503952), SPH_C32(0x29924e8e), SPH_C32(0x648b0000),
	  SPH_C32(0x451c0000), SPH_C32(0xb15a0d86), SPH_C32(0x3e610000),
	  SPH_C32(0xd72e5619), SPH_C32(0xa30b8a1e), SPH_C32(0x7f8d1e0c),
	  SPH_C32(0xac330745) },
	{ SPH_C32(0xd0bc0000), SPH_C32(0xf2830000), SPH_C32(0x42d90186),
	  SPH_C32(0x95940000), SPH_C32(0xedc06d9f), SPH_C32(0x5a793722),
	  SPH_C32(0xe642f6af), SPH_C32(0x05060b10), SPH_C32(0x36db0000),
	  SPH_C32(0x6c480000), SPH_C32(0xdb3b0dc8), SPH_C32(0xce9e0000),
	  SPH_C32(0x4d1f28f5), SPH_C32(0xe628cbd0), SPH_C32(0xb0db91e9),
	  SPH_C32(0xff30144a) },
	{ SPH_C32(0x82ec0000), SPH_C32(0xdbd70000), SPH_C32(0x28b801c8),
	  SPH_C32(0x656b0000), SPH_C32(0x77f11373), SPH_C32(0x1f5a76ec),
	  SPH_C32(0x2914794a), SPH_C32(0x5605181f), SPH_C32(0x65560000),
	  SPH_C32(0xc5b40000), SPH_C32(0x45cc0dce), SPH_C32(0x98610000),
	  SPH_C32(0x47fb28bb), SPH_C32(0x74ed0629), SPH_C32(0x199fd1f1),
	  SPH_C32(0x80a742db) },
	{ SPH_C32(0x4f250000), SPH_C32(0xfe1c0000), SPH_C32(0x86970600),
	  SPH_C32(0xf83b0000), SPH_C32(0xacf47e2e), SPH_C32(0x4f27cec3),
	  SPH_C32(0x2645fe76), SPH_C32(0xb0cbda7f), SPH_C32(0xb6980000),
	  SPH_C32(0x00b70000), SPH_C32(0x499f0f00), SPH_C32(0xfd6d0000),
	  SPH_C32(0x80637e55), SPH_C32(0x793a7b5b), SPH_C32(0x3093fc39),
	  SPH_C32(0x5dc3e62e) },
	{ SPH_C32(0x1d750000), SPH_C32(0xd7480000), SPH_C32(0xecf6064e),
	  SPH_C32(0x08c40000), SPH_C32(0x36c500c2), SPH_C32(0x0a048f0d),
	  SPH_C32(0xe9137193), SPH_C32(0xe3c8c970), SPH_C32(0xe5150000),
	  SPH_C32(0xa94b0000), SPH_C32(0xd7680f06), SPH_C32(0xab920000),
	  SPH_C32(0x8a877e1b), SPH_C32(0xebffb6a2), SPH_C32(0x99d7bc21),
	  SPH_C32(0x2254b0bf) },
	{ SPH_C32(0x1ca80000), SPH_C32(0x57e00000), SPH_C32(0x18600606),
	  SPH_C32(0xaec40000), SPH_C32(0xa6107e60), SPH_C32(0xdde2033a),
	  SPH_C32(0x8f01be6e), SPH_C32(0xcf5c8cee), SPH_C32(0xb7450000),
	  SPH_C32(0x801f0000), SPH_C32(0xbd090f48), SPH_C32(0x5b6d0000),
	  SPH_C32(0x10b600f7), SPH_C32(0xaedcf76c), SPH_C32(0x568133c4),
	  SPH_C32(0x7157a3b0) },
	{ SPH_C32(0x4ef80000), SPH_C32(0x7eb40000), SPH_C32(0x72010648),
	  SPH_C32(0x5e3b0000), SPH_C32(0x3c21008c), SPH_C32(0x98c142f4),
	  SPH_C32(0x4057318b), SPH_C32(0x9c5f9fe1), SPH_C32(0xe4c80000),
	  SPH_C32(0x29e30000), SPH_C32(0x23fe0f4e), SPH_C32(0x0d920000),
	  SPH_C32(0x1a5200b9), SPH_C32(0x3c193a95), SPH_C32(0xffc573dc),
	  SPH_C32(0x0ec0f521) },
	{ SPH_C32(0x7a8c0000), SPH_C32(0xa5d40000), SPH_C32(0x13260880),
	  SPH_C32(0xc63d0000), SPH_C32(0xcbb36daa), SPH_C32(0xfea14f43),
	  SPH_C32(0x59d0b4f8), SPH_C32(0x979961d0), SPH_C32(0x78230000),
	  SPH_C32(0x12fc0000), SPH_C32(0xa93a0b80), SPH_C32(0x90a50000),
	  SPH_C32(0x713e2879), SPH_C32(0x7ee98924), SPH_C32(0xf08ca062),
	  SPH_C32(0x636f8bab) },
	{ SPH_C32(0x28dc0000), SPH_C32(0x8c800000), SPH_C32(0x794708ce),
	  SPH_C32(0x36c20000), SPH_C32(0x51821346), SPH_C32(0xbb820e8d),
	  SPH_C32(0x96863b1d), SPH_C32(0xc49a72df), SPH_C32(0x2bae0000),
	  SPH_C32(0xbb000000), SPH_C32(0x37cd0b86), SPH_C32(0xc65a0000),
	  SPH_C32(0x7bda2837), SPH_C32(0xec2c44dd), SPH_C32(0x59c8e07a),
	  SPH_C32(0x1cf8dd3a) },
	{ SPH_C32(0x29010000), SPH_C32(0x0c280000), SPH_C32(0x8dd10886),
	  SPH_C32(0x90c20000), SPH_C32(0xc1576de4), SPH_C32(0x6c6482ba),
	  SPH_C32(0xf094f4e0), SPH_C32(0xe80e3741), SPH_C32(0x79fe0000),
	  SPH_C32(0x92540000), SPH_C32(0x5dac0bc8), SPH_C32(0x36a50000),
	  SPH_C32(0xe1eb56db), SPH_C32(0xa90f0513), SPH_C32(0x969e6f9f),
	  SPH_C32(0x4ffbce35) },
	{ SPH_C32(0x7b510000), SPH_C32(0x257c0000), SPH_C32(0xe7b008c8),
	  SPH_C32(0x603d0000), SPH_C32(0x5b661308), SPH_C32(0x2947c374),
	  SPH_C32(0x3fc27b05), SPH_C32(0xbb0d244e), SPH_C32(0x2a730000),
	  SPH_C32(0x3ba80000), SPH_C32(0xc35b0bce), SPH_C32(0x605a0000),
	  SPH_C32(0xeb0f5695), SPH_C32(0x3bcac8ea), SPH_C32(0x3fda2f87),
	  SPH_C32(0x306c98a4) },
	{ SPH_C32(0xb6980000), SPH_C32(0x00b70000), SPH_C32(0x499f0f00),
	  SPH_C32(0xfd6d0000), SPH_C32(0x80637e55), SPH_C32(0x793a7b5b),
	  SPH_C32(0x3093fc39), SPH_C32(0x5dc3e62e), SPH_C32(0xf9bd0000),
	  SPH_C32(0xfeab0000), SPH_C32(0xcf080900), SPH_C32(0x05560000),
	  SPH_C32(0x2c97007b), SPH_C32(0x361db598), SPH_C32(0x16d6024f),
	  SPH_C32(0xed083c51) },
	{ SPH_C32(0xe4c80000), SPH_C32(0x29e30000), SPH_C32(0x23fe0f4e),
	  SPH_C32(0x0d920000), SPH_C32(0x1a5200b9), SPH_C32(0x3c193a95),
	  SPH_C32(0xffc573dc), SPH_C32(0x0ec0f521), SPH_C32(0xaa300000),
	  SPH_C32(0x57570000), SPH_C32(0x51ff0906), SPH_C32(0x53a90000),
	  SPH_C32(0x26730035), SPH_C32(0xa4d87861), SPH_C32(0xbf924257),
	  SPH_C32(0x929f6ac0) },
	{ SPH_C32(0xe5150000), SPH_C32(0xa94b0000), SPH_C32(0xd7680f06),
	  SPH_C32(0xab920000), SPH_C32(0x8a877e1b), SPH_C32(0xebffb6a2),
	  SPH_C32(0x99d7bc21), SPH_C32(0x2254b0bf), SPH_C32(0xf8600000),
	  SPH_C32(0x7e030000), SPH_C32(0x3b9e0948), SPH_C32(0xa3560000),
	  SPH_C32(0xbc427ed9), SPH_C32(0xe1fb39af), SPH_C32(0x70c4cdb2),
	  SPH_C32(0xc19c79cf) },
	{ SPH_C32(0xb7450000), SPH_C32(0x801f0000), SPH_C32(0xbd090f48),
	  SPH_C32(0x5b6d0000), SPH_C32(0x10b600f7), SPH_C32(0xaedcf76c),
	  SPH_C32(0x568133c4), SPH_C32(0x7157a3b0), SPH_C32(0xabed0000),
	  SPH_C32(0xd7ff0000), SPH_C32(0xa569094e), SPH_C32(0xf5a90000),
	  SPH_C32(0xb6a67e97), SPH_C32(0x733ef456), SPH_C32(0xd9808daa),
	  SPH_C32(0xbe0b2f5e) },
	{ SPH_C32(0xfb120000), SPH_C32(0x49830000), SPH_C32(0x75140a00),
	  SPH_C32(0x53ce0000), SPH_C32(0x961a45a8), SPH_C32(0xb65573ff),
	  SPH_C32(0xbf8a16d5), SPH_C32(0x19fed62a), SPH_C32(0x35a90000),
	  SPH_C32(0x5bc80000), SPH_C32(0x95b10e80), SPH_C32(0x3e060000),
	  SPH_C32(0x67471384), SPH_C32(0xb1868180), SPH_C32(0x7f954a8e),
	  SPH_C32(0x2752bbaf) },
	{ SPH_C32(0xa9420000), SPH_C32(0x60d70000), SPH_C32(0x1f750a4e),
	  SPH_C32(0xa3310000), SPH_C32(0x0c2b3b44), SPH_C32(0xf3763231),
	  SPH_C32(0x70dc9930), SPH_C32(0x4afdc525), SPH_C32(0x66240000),
	  SPH_C32(0xf2340000), SPH_C32(0x0b460e86), SPH_C32(0x68f90000),
	  SPH_C32(0x6da313ca), SPH_C32(0x23434c79), SPH_C32(0xd6d10a96),
	  SPH_C32(0x58c5ed3e) },
	{ SPH_C32(0xa89f0000), SPH_C32(0xe07f0000), SPH_C32(0xebe30a06),
	  SPH_C32(0x05310000), SPH_C32(0x9cfe45e6), SPH_C32(0x2490be06),
	  SPH_C32(0x16ce56cd), SPH_C32(0x666980bb), SPH_C32(0x34740000),
	  SPH_C32(0xdb600000), SPH_C32(0x61270ec8), SPH_C32(0x98060000),
	  SPH_C32(0xf7926d26), SPH_C32(0x66600db7), SPH_C32(0x19878573),
	  SPH_C32(0x0bc6fe31) },
	{ SPH_C32(0xfacf0000), SPH_C32(0xc92b0000), SPH_C32(0x81820a48),
	  SPH_C32(0xf5ce0000), SPH_C32(0x06cf3b0a), SPH_C32(0x61b3ffc8),
	  SPH_C32(0xd998d928), SPH_C32(0x356a93b4), SPH_C32(0x67f90000),
	  SPH_C32(0x729c0000), SPH_C32(0xffd00ece), SPH_C32(0xcef90000),
	  SPH_C32(0xfd766d68), SPH_C32(0xf4a5c04e), SPH_C32(0xb0c3c56b),
	  SPH_C32(0x7451a8a0) },
	{ SPH_C32(0x37060000), SPH_C32(0xece00000), SPH_C32(0x2fad0d80),
	  SPH_C32(0x689e0000), SPH_C32(0xddca5657), SPH_C32(0x31ce47e7),
	  SPH_C32(0xd6c95e14), SPH_C32(0xd3a451d4), SPH_C32(0xb4370000),
	  SPH_C32(0xb79f0000), SPH_C32(0xf3830c00), SPH_C32(0xabf50000),
	  SPH_C32(0x3aee3b86), SPH_C32(0xf972bd3c), SPH_C32(0x99cfe8a3),
	  SPH_C32(0xa9350c55) },
	{ SPH_C32(0x65560000), SPH_C32(0xc5b40000), SPH_C32(0x45cc0dce),
	  SPH_C32(0x98610000), SPH_C32(0x47fb28bb), SPH_C32(0x74ed0629),
	  SPH_C32(0x199fd1f1), SPH_C32(0x80a742db), SPH_C32(0xe7ba0000),
	  SPH_C32(0x1e630000), SPH_C32(0x6d740c06), SPH_C32(0xfd0a0000),
	  SPH_C32(0x300a3bc8), SPH_C32(0x6bb770c5), SPH_C32(0x308ba8bb),
	  SPH_C32(0xd6a25ac4) },
	{ SPH_C32(0x648b0000), SPH_C32(0x451c0000), SPH_C32(0xb15a0d86),
	  SPH_C32(0x3e610000), SPH_C32(0xd72e5619), SPH_C32(0xa30b8a1e),
	  SPH_C32(0x7f8d1e0c), SPH_C32(0xac330745), SPH_C32(0xb5ea0000),
	  SPH_C32(0x37370000), SPH_C32(0x07150c48), SPH_C32(0x0df50000),
	  SPH_C32(0xaa3b4524), SPH_C32(0x2e94310b), SPH_C32(0xffdd275e),
	  SPH_C32(0x85a149cb) },
	{ SPH_C32(0x36db0000), SPH_C32(0x6c480000), SPH_C32(0xdb3b0dc8),
	  SPH_C32(0xce9e0000), SPH_C32(0x4d1f28f5), SPH_C32(0xe628cbd0),
	  SPH_C32(0xb0db91e9), SPH_C32(0xff30144a), SPH_C32(0xe6670000),
	  SPH_C32(0x9ecb0000), SPH_C32(0x99e20c4e), SPH_C32(0x5b0a0000),
	  SPH_C32(0xa0df456a), SPH_C32(0xbc51fcf2), SPH_C32(0x56996746),
	  SPH_C32(0xfa361f5a) }
};

static const sph_u32 T512_42[64][16] = {
	{ SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000), SPH_C32(0x00000000), SPH_C32(0x00000000),
	  SPH_C32(0x00000000) },
	{ SPH_C32(0x58430000), SPH_C32(0x807e0000), SPH_C32(0x78330001),
	  SPH_C32(0xc66b3800), SPH_C32(0xe7375cdc), SPH_C32(0x79ad3fdd),
	  SPH_C32(0xac73fe6f), SPH_C32(0x3a4479b1), SPH_C32(0x1d5a0000),
	  SPH_C32(0x2b720000), SPH_C32(0x488d0000), SPH_C32(0xaf611800),
	  SPH_C32(0x25cb2ec5), SPH_C32(0xc879bfd0), SPH_C32(0x81a20429),
	  SPH_C32(0x1e7536a6) },
	{ SPH_C32(0x1d5a0000), SPH_C32(0x2b720000), SPH_C32(0x488d0000),
	  SPH_C32(0xaf611800), SPH_C32(0x25cb2ec5), SPH_C32(0xc879bfd0),
	  SPH_C32(0x81a20429), SPH_C32(0x1e7536a6), SPH_C32(0x45190000),
	  SPH_C32(0xab0c0000), SPH_C32(0x30be0001), SPH_C32(0x690a2000),
	  SPH_C32(0xc2fc7219), SPH_C32(0xb1d4800d), SPH_C32(0x2dd1fa46),
	  SPH_C32(0x24314f17) },
	{ SPH_C32(0x45190000), SPH_C32(0xab0c0000), SPH_C32(0x30be0001),
	  SPH_C32(0x690a2000), SPH_C32(0xc2fc7219), SPH_C32(0xb1d4800d),
	  SPH_C32(0x2dd1fa46), SPH_C32(0x24314f17), SPH_C32(0x58430000),
	  SPH_C32(0x807e0000), SPH_C32(0x78330001), SPH_C32(0xc66b3800),
	  SPH_C32(0xe7375cdc), SPH_C32(0x79ad3fdd), SPH_C32(0xac73fe6f),
	  SPH_C32(0x3a4479b1) },
	{ SPH_C32(0xa53b0000), SPH_C32(0x14260000), SPH_C32(0x4e30001e),
	  SPH_C32(0x7cae0000), SPH_C32(0x8f9e0dd5), SPH_C32(0x78dfaa3d),
	  SPH_C32(0xf73168d8), SPH_C32(0x0b1b4946), SPH_C32(0x07ed0000),
	  SPH_C32(0xb2500000), SPH_C32(0x8774000a), SPH_C32(0x970d0000),
	  SPH_C32(0x437223ae), SPH_C32(0x48c76ea4), SPH_C32(0xf4786222),
	  SPH_C32(0x9075b1ce) },
	{ SPH_C32(0xfd780000), SPH_C32(0x94580000), SPH_C32(0x3603001f),
	  SPH_C32(0xbac53800), SPH_C32(0x68a95109), SPH_C32(0x017295e0),
	  SPH_C32(0x5b4296b7), SPH_C32(0x315f30f7), SPH_C32(0x1ab70000),
	  SPH_C32(0x99220000), SPH_C32(0xcff9000a), SPH_C32(0x386c1800),
	  SPH_C32(0x66b90d6b), SPH_C32(0x80bed174), SPH_C32(0x75da660b),
	  SPH_C32(0x8e008768) },
	{ SPH_C32(0xb8610000), SPH_C32(0x3f540000), SPH_C32(0x06bd001e),
	  SPH_C32(0xd3cf1800), SPH_C32(0xaa552310), SPH_C32(0xb0a615ed),
	  SPH_C32(0x76936cf1), SPH_C32(0x156e7fe0), SPH_C32(0x42f40000),
	  SPH_C32(0x195c0000), SPH_C32(0xb7ca000b), SPH_C32(0xfe072000),
	  SPH_C32(0x818e51b7), SPH_C32(0xf913eea9), SPH_C32(0xd9a99864),
	  SPH_C32(0xb444fed9) },
	{ SPH_C32(0xe0220000), SPH_C32(0xbf2a0000), SPH_C32(0x7e8e001f),
	  SPH_C32(0x15a42000), SPH_C32(0x4d627fcc), SPH_C32(0xc90b2a30),
	  SPH_C32(0xdae0929e), SPH_C32(0x2f2a0651), SPH_C32(0x5fae0000),
	  SPH_C32(0x322e0000), SPH_C32(0xff47000b), SPH_C32(0x51663800),
	  SPH_C32(0xa4457f72), SPH_C32(0x316a5179), SPH_C32(0x580b9c4d),
	  SPH_C32(0xaa31c87f) },
	{ SPH_C32(0x07ed0000), SPH_C32(0xb2500000), SPH_C32(0x8774000a),
	  SPH_C32(0x970d0000), SPH_C32(0x437223ae), SPH_C32(0x48c76ea4),
	  SPH_C32(0xf4786222), SPH_C32(0x9075b1ce), SPH_C32(0xa2d60000),
	  SPH_C32(0xa6760000), SPH_C32(0xc9440014), SPH_C32(0xeba30000),
	  SPH_C32(0xccec2e7b), SPH_C32(0x3018c499), SPH_C32(0x03490afa),
	  SPH_C32(0x9b6ef888) },
	{ SPH_C32(0x5fae0000), SPH_C32(0x322e0000), SPH_C32(0xff47000b),
	  SPH_C32(0x51663800), SPH_C32(0xa4457f72), SPH_C32(0x316a5179),
	  SPH_C32(0x580b9c4d), SPH_C32(0xaa31c87f), SPH_C32(0xbf8c0000),
	  SPH_C32(0x8d040000), SPH_C32(0x81c90014), SPH_C32(0x44c21800),
	  SPH_C32(0xe92700be), SPH_C32(0xf8617b49), SPH_C32(0x82eb0ed3),
	  SPH_C32(0x851bce2e) },
	{ SPH_C32(0x1ab70000), SPH_C32(0x99220000), SPH_C32(0xcff9000a),
	  SPH_C32(0x386c1800), SPH_C32(0x66b90d6b), SPH_C32(0x80bed174),
	  SPH_C32(0x75da660b), SPH_C32(0x8e008768), SPH_C32(0xe7cf0000),
	  SPH_C32(0x0d7a0000), SPH_C32(0xf9fa0015), SPH_C32(0x82a92000)