//////////////////////////////////////////////////////////////////////////////
//
// 		  Copyright (C) 1996,1997  Matthew Doar  doar@pobox.com
// 
// Permission to use, copy, and distribute this software and its documentation 
// for any purpose with or without fee is hereby granted, provided that the 
// above copyright notice appears with all copies and that both that copyright 
// notice and this permission notice appear in supporting documentation. 
// 
// Permission to modify the software is granted, but not the right to 
// distribute the modified code. Modifications are to be distributed as 
// patches to the released version. 
// 
// This software is provided "as is" without express or implied warranty. 
//
//////////////////////////////////////////////////////////////////////////////

// tiers_prims.cc

#include <iostream.h>
#include <math.h>	// for random
#include <limits.h>	// for ULONG_MAX

#ifndef _TIERS_HH
#include "tiers.hh"
#endif

extern "C" 
{
  long random(void);
}



////////////////////////////////////////////////////////////////////////
// Model::Prims
//
// Calculate a minimum spanning tree (MST) using Prim's Algorithm
// (always add the lowest cost edge which doesn't cause a loop)
// Reference: p.234 "Algorithms", R. Sedgewick, Addison-Wesley, 1983.
////////////////////////////////////////////////////////////////////////
bool
Model::Prims(NodesAndEdges *aNetwork, unsigned long int R)
{
  bool ret = true;
  unsigned long int NumNodes = aNetwork->NumNodes();

  // aNetwork->edges is an NumNodes x NumNodes array [from][to] of the
  // potential edges in the network.
  // When the algorithm has finished, aNetwork->newedges will contain a
  // NumNodes x MaxNumEdges array with edges from a node *not* sorted in order
  // of cost, but with all the ones in the MST first. If there are X edges in
  // the MST from a node, then if R > X, R-X edges are added to the array
  // The R-X extra edges are also sorted in order and are used by the 
  // redundancy functions later on. If R <= X, then no extra edges are added.

  // Note that adding an edge to the MST implies adding it as an edge from two
  // nodes.
  // Note also that the edge (i,i) and its values are undefined.

  // Implementation Notes:
  //
  // A more space efficient representation of the edges would be an array of 
  // lists of edges

  Edge	*edges = aNetwork->edges; // an alias
  unsigned long int MaxNumEdges = aNetwork->MaxNumEdges();
  aNetwork->newedges = new Edge[NumNodes*MaxNumEdges];
  Edge *newedges = aNetwork->newedges; // an alias
  // assign space for the count of edges from each node 
  if (!newedges || !aNetwork->newEdgeList(NumNodes))
	{
	  cout << "Prims: failed to allocate space for edge count or result" << endl;
	  return false;
	}

  // Create the Singly Connected Node set, i.e. nodes which are connected
  // together in a single component of the graph (not nodes with a single 
  // connection!)
  // Also initialize the NumEdges and newedges arrays
  bool *ScnSet = new bool[NumNodes];
  for (unsigned long int i = 0; i < NumNodes; i++)
	{
	  ScnSet[i] = false;
	  aNetwork->NumEdges(i, 0);
	  for (unsigned long int j = 0; j < MaxNumEdges ; j++)
		{
		  if (i != j)
			{
			  // Set all the edges inactive in the MST array
			  newedges[i*NumNodes+j].start = 0;
			  newedges[i*NumNodes+j].end = 0;
			  newedges[i*NumNodes+j].delay = ULONG_MAX; 
			  newedges[i*NumNodes+j].bw = 0;
			  newedges[i*NumNodes+j].state = Model::INACTIVE;	
			}
		}
	}

  // Chose a node at random to start. 
  // and add it to the set of singly connected nodes (ScnSet)
  ScnSet[random() % NumNodes] = true;
  unsigned long int ScnSetNumNodes = 1;

  //
  // PART I
  // Repeat until all the nodes are in the set of singly connected nodes
  //
  while (ScnSetNumNodes < NumNodes)
	{
	  // These two variables will be the edge found in each pass of Prim's alg.
	  unsigned long int ClosestFrom = NumNodes;	// set to an impossible value
	  unsigned long int ClosestTo = NumNodes;
	  unsigned long int MinimumCost = ULONG_MAX;	// largest value possible

	  // Find the shortest edge out of ScnSet
	  // Bidirectionality is handled by the ScnSet forcing one direction
	  for (unsigned long int from = 0; from < NumNodes; from++)
		{
		  // If the node is already in the ScnSet, look for an edge to the 
		  // closest node which isn't in the ScnSet
		  if (ScnSet[from])
			{
			  for (unsigned long int to = 0; to < NumNodes; to++)
				{
				  // If an edge not in the ScnSet and closer has been found
				  // record it as the best so far.
				  // This explicitly uses delay for the metric
				  unsigned long int cost = (edges[from*NumNodes+to]).delay;
				  if ((cost < MinimumCost) && (!ScnSet[to]) && (from != to))
					{
					  MinimumCost = cost;
					  ClosestFrom = from;
					  ClosestTo = to;
					}
				} // to
			} // from in ScnSet
		} // from

	  // Add the closest edge to the MST and mark it in the old array
	  // No need to sort since we chose the minimum here. An MST property
	  unsigned long int index = ClosestFrom*NumNodes+ClosestTo;
	  unsigned long int newindex = ClosestFrom*MaxNumEdges + 
		(aNetwork->NumEdges(ClosestFrom));

	  edges[index].state = Model::ACTIVE;
	  newedges[newindex].start = edges[index].start;
	  newedges[newindex].end = edges[index].end;
	  newedges[newindex].delay = edges[index].delay;
	  newedges[newindex].bw = edges[index].bw;
	  newedges[newindex].state = Model::ACTIVE;
	  //  and increase the count of the number of edges in the MST
	  aNetwork->IncNumEdges(ClosestFrom);

	  // Also add the edge as an edge from the `to' node
	  // This is unsorted so makes all of it unsorted. Sort it later
	  index = ClosestTo*NumNodes+ClosestFrom;
	  edges[index].state = Model::ACTIVE;
	  newindex = ClosestTo*MaxNumEdges + (aNetwork->NumEdges(ClosestTo));
	  newedges[newindex].start = edges[index].start;
	  newedges[newindex].end = edges[index].end;
	  newedges[newindex].delay = edges[index].delay;
	  newedges[newindex].bw = edges[index].bw;
	  newedges[newindex].state = Model::ACTIVE;

	  //  and increase the count of the number of edges in the MST
	  aNetwork->IncNumEdges(ClosestTo);

	  // Increase the number of nodes which are singly connected
	  ScnSetNumNodes++;
	  ScnSet[ClosestTo] = true;

	} // while MST not complete

  //
  // PART II
  // Add the edges which might be needed for redundancy later on
  //
  for (unsigned long int from = 0; from < NumNodes; from++)
	{
	  // Number of edges from the node 'from'
	  unsigned long int NumMstEdges = aNetwork->NumEdges(from);

	  // See how many edges to add and if there are any spare edges anyway
	  // Note that NumMstEdges can be greater than R and both are unsigned
	  if (R > NumMstEdges)
		{
		  unsigned long int Rnum = R - NumMstEdges; // positive num
		  // Check there are spare edges within the network to add
		  if ((NumNodes-NumMstEdges-1) > Rnum)
			{
			  // Add the lowest cost Rnum edges which are free
			  while (Rnum)
				{
				  unsigned long int ClosestTo = NumNodes;
				  unsigned long int MinimumCost = ULONG_MAX;
				  for (unsigned long int to = 0; to < NumNodes; to++)
					{
					  unsigned long int index = from*NumNodes+to;
					  // This explicitly uses delay for the metric
					  unsigned long int cost = edges[index].delay;
					  if ((cost < MinimumCost) && 
						  (edges[index].state == Model::INACTIVE) &&
						  (from != to))
						{
						  MinimumCost = cost;
						  ClosestTo = to;
						}
					}

				  // Add the edge found into the new edge list
				  // No need to sort since we choose the minimum here
				  unsigned long int index = from*NumNodes+ClosestTo;
				  unsigned long int newindex = from*MaxNumEdges +
					(aNetwork->NumEdges(from));
				  // so the edge which was found is not seen again
				  edges[index].state = Model::ACTIVE; 

				  newedges[newindex].start = edges[index].start;
				  newedges[newindex].end = edges[index].end;
				  newedges[newindex].delay = edges[index].delay;
				  newedges[newindex].bw = edges[index].bw;
				  newedges[newindex].state = Model::INACTIVE; // not part of the MST
				  //  and increase the count of the number of edges from 'from'
				  aNetwork->IncNumEdges(from);

				  // Also add the edge as an edge from the `to' node
				  // This is unsorted so makes all of it unsorted. Sort later
				  index = ClosestTo*NumNodes+from;
				  newindex = ClosestTo*MaxNumEdges + 
					(aNetwork->NumEdges(ClosestTo));
				  newedges[newindex].start = edges[index].start;
				  newedges[newindex].end = edges[index].end;
				  newedges[newindex].delay = edges[index].delay;
				  newedges[newindex].bw = edges[index].bw;
				  newedges[newindex].state = Model::INACTIVE; // not part of the MST
				  //  and increase the count of the number of edges from 'to'
				  aNetwork->IncNumEdges(ClosestTo);

				  // decrement the number of edges still to be found
				  Rnum--;

				} // still edges to be added
			} // there are edges to be added
		} // edges need to be added
	} // for all from nodes

  // 
  // PART III
  // Sort the MST edges in newedges in increasing order of cost
  //
  // Since the values of R are expected to be small, the effort of sorting
  // all the edges by cost may not be worth it. So, the edges are left unsorted
  // here. This implies that the ith lowest cost edge will be found later by
  // scanning through NumEdges(node) edges successively. This is  easy if 
  // successive values of i are used.

  // Tidy up 
  delete [] edges;	// the full connectivity matrix is no longer needed
  edges = 0;

  delete [] ScnSet;

  return ret;
}

// end of file

