//////////////////////////////////////////////////////////////////////////////
//
// 		  Copyright (C) 1996,1997  Matthew Doar  doar@pobox.com
// 
// Permission to use, copy, and distribute this software and its documentation 
// for any purpose with or without fee is hereby granted, provided that the 
// above copyright notice appears with all copies and that both that copyright 
// notice and this permission notice appear in supporting documentation. 
// 
// Permission to modify the software is granted, but not the right to 
// distribute the modified code. Modifications are to be distributed as 
// patches to the released version. 
// 
// This software is provided "as is" without express or implied warranty. 
//
//////////////////////////////////////////////////////////////////////////////

// tiers_red_inter.cc

#include <iostream.h>
#include <limits.h>	// for ULONG_MAX

#ifndef _TIERS_HH
#include "tiers.hh"
#endif

extern "C" 
{
  long random(void);
}


////////////////////////////////////////////////////////////////////////
// Model::ConnectMans
//
// Connect each MAN to the WAN with a unit length 
// Add the WAN connection point coordinates to all the nodes in the MAN
// Also add the internetwork redundancy from MANs to WAN
////////////////////////////////////////////////////////////////////////
bool Model::ConnectMans()
{
  bool ret = true;

  unsigned long int NumWanNodes = WanList[0].NumNodes();
  unsigned long int MaxNumWanEdges = WanList[0].MaxNumEdges();
  Edge *wanedges = WanList[0].newedges;	// an alias
  Node *wannodes = WanList[0].nodes;	// an alias

  // Step through each MAN, connecting the MAN to the WAN and vice versa
  for (unsigned long int network = 0; 
	   network < WanList[0].NumNetworks();
	   network++)
	{
	  // Select a node at random from the WAN as an attachment point for a MAN
	  // Duplicates are permitted. 
	  unsigned long int wangateway = random() % NumWanNodes;

	  unsigned long int NumManNodes = ManList[network].NumNodes();
	  unsigned long int MaxNumManEdges = ManList[network].MaxNumEdges();

	  // Choose the index of the first node to be connected
	  unsigned long int gateway = random() % NumManNodes;

	  if (!AddInternetworkEdge(ManList, WanList,
							   gateway, wangateway, 
							   MaxNumManEdges, network)) 
		{
		  cerr << "ConnectMans:: Failed to add internetwork edge to MAN and WAN" << endl;
		  return false;
		}

	  // Update the coordinates of all the nodes in the MAN
	  if (!RepositionNetwork(&(ManList[network]), 
							 &(WanList[0].nodes[wangateway]),	
							   MAN))
		{
		  cerr << "ConnectMans:: Failed to reposition MAN" << endl;
		  return false;
		}


	  // Add edges for internetwork redundancy
	  // Add an edge from the MAN to the closest nodes to the original 
	  // wangateway. Increase the probability that once a node has been 
	  // chosen as a connection point in a MAN to the WAN, that it will be 
	  // chosen again, with p = 1/2.
  	  unsigned long int R = RMW-1;
	  while (R)
		{
		  // Choose the (possibly new) gateway if != 0
		  if ((random() % 2))
			{
			  gateway = random() % NumManNodes;
			}

		  // Find the new wangateway - node closest to current wangateway
		  // by  non INACTIVE edge.
		  // Could have used a full connectivity array here if it still existed
		  // The closest node cannot be self since no edge exists to self
		  unsigned long int NumWanEdges = WanList[0].NumEdges(wangateway);
		  unsigned long int EdgeClosestTo = NumWanEdges;
		  unsigned long int MinimumCost = ULONG_MAX;

		  for (unsigned long int i = 0; 
			   i < NumWanEdges; 
			   i++)
			{
			  // This explicitly uses delay for the metric
			  unsigned long int index = wangateway*MaxNumWanEdges+i;
			  unsigned long int cost = wanedges[index].delay;

			  // Make sure not to use the MAN edge (always closest to the WAN)
			  if ((cost < MinimumCost) && 
				  (wanedges[index].state != INACTIVE) &&
				  (wanedges[index].state < Model::INTERNETWORK))
				{
				  MinimumCost = cost;
				  EdgeClosestTo = i;
				}
			}
		  // There may only be one node in the WAN
		  if ((MinimumCost == ULONG_MAX) && (WanList[0].NumNodes() > 1))
			{ 
			  cerr << "ConnectMans:: WAN edge list corrupt" << endl;
			  return false;
			}

		  // Find the correct WAN index for ClosestTo.end
		  unsigned long int sought_node_num = 
			wanedges[wangateway*MaxNumWanEdges+EdgeClosestTo].end;
		  wangateway = 0;
		  while(wannodes[wangateway].number != 
				sought_node_num) 
			{
			  wangateway++;
			}
		  if (wangateway == NumWanNodes)
			{ 
			  cerr << "ConnectMans:: WAN node list corrupt" << endl;
			  return false;
			}

		  if (!AddInternetworkEdge(ManList, WanList,
								   gateway, wangateway, 
								   MaxNumManEdges, network,
								   Model::RED_INTERNETWORK)) 
			 {
			   cerr << "ConnectMans:: Failed to add redundant internetwork edge to MAN and WAN" << endl;
			   return false;
			 }

		  
		  // And decrement the number of redundant internetwork edges 
		  // still to be added
		  R--;

		}// while R
	} // network

  return ret;
}



////////////////////////////////////////////////////////////////////////
// Model::ConnectLans
//
// Connect each LAN to a MAN using the star of the LAN as the connection point
////////////////////////////////////////////////////////////////////////
bool Model::ConnectLans()
{
  bool ret = true;
  unsigned long int lanindex = 0;

  // Step through each MAN, connecting the LANs to the MAN
  for (unsigned long int man = 0; man < WanList[0].NumNetworks(); man++)
	{
	  unsigned long int NumLans = ManList[man].NumNetworks();
	  unsigned long int NumManNodes = ManList[man].NumNodes();
	  unsigned long int MaxNumManEdges = ManList[man].MaxNumEdges();
	  Node *mannodes = ManList[man].nodes;		// an alias
	  Edge *manedges = ManList[man].newedges;	// an alias

	  // Attach the predetermined number of LANs
	  unsigned long int UpperLimit = lanindex + NumLans;
	  for (lanindex = lanindex; lanindex < UpperLimit; lanindex++)
		{
		  // Select a node at random from the MAN to be an attachment point 
		  // for the LAN. Duplicates are permitted.
		  unsigned long int mangateway = random() % NumManNodes;

		  Edge *lanedges = LanList[lanindex].edges;	// an alias
		  Node *lannodes = LanList[lanindex].nodes;	// an alias
		  unsigned long int NumLanNodes = LanList[lanindex].NumNodes();

		  // Choose the index of the first node to be connected
		  // Always use node 0, the centre of the star
		  //
		  // Add the edge to the LAN
		  LanList[lanindex].IncNumEdges(0);
		  // Extra LAN edges go at the end of the edges array
		  // This is the main connecting edge, so its index is NumLanNodes+0
		  unsigned long int index = NumLanNodes; 
		  CostTableEntry anEntry = theCostTable[(LAN*3)+MAN];

		  lanedges[index].start = lannodes[0].number;
		  lanedges[index].end = mannodes[mangateway].number;
		  lanedges[index].delay = anEntry.d1 + anEntry.d2; // unit length
		  lanedges[index].bw = anEntry.bw;
		  lanedges[index].state = Model::INTERNETWORK;

		  // Add the edge to the MAN
		  ManList[man].IncNumEdges(mangateway);
		  index = mangateway*MaxNumManEdges + (ManList[man].NumEdges(mangateway)) - 1;
		  anEntry = theCostTable[(MAN*3)+LAN];

		  manedges[index].start = mannodes[mangateway].number;
		  manedges[index].end = lannodes[0].number;
		  manedges[index].delay = anEntry.d1 + anEntry.d2; // unit length
		  manedges[index].bw = anEntry.bw;
		  manedges[index].state = Model::INTERNETWORK;

		  // Update the coordinates of all the nodes in the LAN
		  if (!RepositionNetwork(&(LanList[lanindex]), 
								 &(mannodes[mangateway]),	
								 LAN))
			{
			  cerr << "ConnectLans:: Failed to reposition LAN" << endl;
			  return false;
			}

		  
		  // Add edges for internetwork redundancy
		  // Add an edge from the LAN to the closest nodes to the original 
		  // mangateway.
		  unsigned long int R = RLM-1;
		  while (R)
			{

			  // Find the new mangateway - node closest to current mangateway
			  // by  non INACTIVE edge.
			  // The closest node cannot be self since no edge exists to self
			  unsigned long int NumManEdges = ManList[man].NumEdges(mangateway);
			  unsigned long int EdgeClosestTo = NumManEdges;
			  unsigned long int MinimumCost = ULONG_MAX;

			  for (unsigned long int i = 0; 
				   i < NumManEdges; 
				   i++)
				{
				  // This explicitly uses delay for the metric
				  unsigned long int index = mangateway*MaxNumManEdges+i;
				  unsigned long int cost = manedges[index].delay;

				  // Make sure not to use the LAN edge 
				  if ((cost < MinimumCost) && 
					  (manedges[index].state != INACTIVE) &&
					  (manedges[index].state < Model::INTERNETWORK))
					{
					  MinimumCost = cost;
					  EdgeClosestTo = i;
					}
				}
			  if (MinimumCost == ULONG_MAX)
				{ 
				  cerr << "ConnectLans:: MAN edge list corrupt" << endl;
				  return false;
				}

			  // Find the correct MAN index for ClosestTo.end
			  unsigned long int sought_node_num = 
				manedges[mangateway*MaxNumManEdges+EdgeClosestTo].end;
			  mangateway = 0;
			  while(mannodes[mangateway].number != 
					sought_node_num) 
				{
				  mangateway++;
				}
			  if (mangateway == NumManNodes)
				{ 
				  cerr << "ConnectLans:: MAN node list corrupt" << endl;
				  return false;
				}

#if 0
			  // TODO debug this function for LANs if it is to be used
			  if (!AddInternetworkEdge(LanList, ManList,
									   0, mangateway, 
									   MaxNumLanEdges, network,
									   Model::RED_INTERNETWORK)) 
				{
				  cerr << "ConnectLans:: Failed to add redundant internetwork edge to LAN and MAN" << endl;
				  return false;
				}
#endif

			  // Add the edge to the LAN, using node 0, the centre of the star
			  LanList[lanindex].IncNumEdges(0);
			  // Extra LAN edges go at the end of the edges array
			  unsigned long int index =  LanList[lanindex].NumEdges(0);
			  CostTableEntry anEntry = theCostTable[(LAN*3)+MAN];

			  lanedges[index].start = lannodes[0].number;
			  lanedges[index].end = mannodes[mangateway].number;
			  lanedges[index].delay = anEntry.d1 + anEntry.d2; // unit length
			  lanedges[index].bw = anEntry.bw;
			  lanedges[index].state = Model::RED_INTERNETWORK;

			  // Add the edge to the MAN
			  ManList[man].IncNumEdges(mangateway);
			  index = mangateway*MaxNumManEdges + 
				(ManList[man].NumEdges(mangateway)) - 1;
			  anEntry = theCostTable[(MAN*3)+LAN];

			  manedges[index].start = mannodes[mangateway].number;
			  manedges[index].end = lannodes[0].number;
			  manedges[index].delay = anEntry.d1 + anEntry.d2; // unit length
			  manedges[index].bw = anEntry.bw;
			  manedges[index].state = Model::RED_INTERNETWORK;
		  
			  // And decrement the number of redundant internetwork edges 
			  // still to be added
			  R--;

			}// while R

		} // lan
	} // man

  return ret;
}



////////////////////////////////////////////////////////////////////////
// Model::AddInternetworkEdge
//
// Add an edge to a subnetwork (e.g. MAN) and to a supernetwork (e.g. WAN), 
// connecting a node in each
// TODO still uses MAN/WAN variables in spite of the name, so not a general 
// function yet
// subgateway: index of node in subnetwork
// supergateway: index of node in supernetwork
////////////////////////////////////////////////////////////////////////
bool
Model::AddInternetworkEdge(NodesAndEdges *subnetwork,
						   NodesAndEdges *supernetwork,
						   unsigned long int subgateway,
						   unsigned long int supergateway,	
						   unsigned long int MaxNumSubEdges,	
						   unsigned long int subnetwork_index,
						   Model::EdgeState aState)
{
  bool ret = true;

  Edge *wanedges = WanList[0].newedges;	// an alias
  Node *wannodes = WanList[0].nodes;	// an alias
  Edge *manedges = ManList[subnetwork_index].newedges;	// an alias
  Node *mannodes = ManList[subnetwork_index].nodes;	// an alias
  unsigned long int MaxNumWanEdges = WanList[0].MaxNumEdges();

  // Add the edge to the MAN
  ManList[subnetwork_index].IncNumEdges(subgateway);
  unsigned long int index = subgateway*MaxNumSubEdges +
	(ManList[subnetwork_index].NumEdges(subgateway)) - 1 ;
  CostTableEntry anEntry = theCostTable[(MAN*3)+WAN];

  manedges[index].start = mannodes[subgateway].number;
  manedges[index].end = wannodes[supergateway].number;
  manedges[index].delay = anEntry.d1 + anEntry.d2; // unit length
  manedges[index].bw = anEntry.bw;
  manedges[index].state = aState;

  // Add the edge to the WAN
  WanList[0].IncNumEdges(supergateway);
  index = supergateway*MaxNumWanEdges + (WanList[0].NumEdges(supergateway))-1;
  anEntry = theCostTable[(WAN*3)+MAN];

  wanedges[index].start = wannodes[supergateway].number;
  wanedges[index].end = mannodes[subgateway].number;
  wanedges[index].delay = anEntry.d1 + anEntry.d2; // unit length
  wanedges[index].bw = anEntry.bw;
  wanedges[index].state = aState;

  return ret;
}



////////////////////////////////////////////////////////////////////////
// Model::RepositionNetwork
//
// Add offsets to the base coordinates of all the nodes in the Network
// Also scale each node's coordinates to their final values
////////////////////////////////////////////////////////////////////////
bool
Model::RepositionNetwork(NodesAndEdges *Network, 
						 Node *Base,
						 NodeType type)

{
  bool ret = true;

  Node *nodes = Network->nodes;	// an alias
  unsigned long int offsetX = Base->x;
  unsigned long int offsetY = Base->y;

  const unsigned long int SpreadDistance = 1;

  unsigned long int subnetwork_scale = 0;
  unsigned long int supernetwork_scale = 0;
  switch(type)
	{
	case MAN:
	  {
		subnetwork_scale = MAN_SCALE;
		supernetwork_scale = WAN_SCALE;
		break;
	  }
	case LAN:
	  {
		subnetwork_scale = LAN_SCALE;
		supernetwork_scale = 1;	// MAN nodes have already been scaled
		break;
	  }
	case WAN:
	default:
	  {
		cerr << "RepositionNetwork:: Unknown or illegal network type" << endl;
		return false;
		break;
	  }
	}

  // Make the subnetwork offset depend upon the which quadrant of the 
  // supernetwork the Base node came from
  // May take the coordinate over the edge of GRID
  if (offsetX > GRID/2) 
	{
	  offsetX += SpreadDistance;
	}
  else
	{
	  // Don't let the offset go negative since it is unsigned
	  if (offsetX >= SpreadDistance)
		{
		  offsetX -= SpreadDistance;
		}
	}

  if (offsetY > GRID/2) 
	{
	  offsetY += SpreadDistance;
	}
  else
	{
	  // Don't let the offset go negative since it is unsigned
	  if (offsetY >= SpreadDistance)
		{
		  offsetY -= SpreadDistance;
		}
	}

  // Scale the offset appropriately
  offsetX *= supernetwork_scale;
  offsetY *= supernetwork_scale;

  for (unsigned long int i = 0; i < Network->NumNodes(); i++)
  {
	nodes[i].x = (nodes[i].x * subnetwork_scale) + offsetX;
	nodes[i].y = (nodes[i].y * subnetwork_scale) + offsetY;
  }

  return ret;
}

// end of file

