// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/FastJets.hh"
#include "Rivet/Projections/LeptonFinder.hh"
#include "Rivet/Projections/MissingMomentum.hh"
#include "Rivet/Projections/PromptFinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief Charm-hadron production
  class BELLE_2005_I686014 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BELLE_2005_I686014);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // projections
      declare(UnstableParticles(),"UFS");

      // histos
      for (double eVal : allowedEnergies()) {

        const string en = toString(round(eVal/MeV));
        if (isCompatibleWithSqrtS(eVal, 1e-3))  _sqs = en;
        bool ih = en!="10520"s;

        for (size_t ix=0; ix<7; ++ix) {
          book(_r[en][ix], 2-ih, 1, ix+1);
          book(_h[en][ix], 3+ih, 1, ix+1);
        }
        book(_c[en],"TMP/wgt"+en);
      }
      raiseBeamErrorIf(_sqs.empty());
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // unstable particles
      const UnstableParticles& ufs = apply<UnstableParticles>(event, "UFS");
      if (_sqs=="10580"s && ufs.particles(Cuts::pid==300553).size()!=1)  vetoEvent;

      _c[_sqs]->fill();
      for (const Particle & p : ufs.particles(Cuts::abspid==411 ||
                                              Cuts::abspid==421 ||
                                              Cuts::abspid==431 ||
                                              Cuts::abspid==413 ||
                                              Cuts::abspid==423 ||
                                              Cuts::abspid==4122 )) {
        double pmax = sqrt(0.25*sqr(sqrtS())-sqr(p.mass()));
        double xp = p.mom().p3().mod()/pmax;
        if (p.abspid()==421) {
          _r[_sqs][0]->fill(0.5);
          _h[_sqs][0]->fill(xp);
        }
        else if (p.abspid()==411) {
          _r[_sqs][1]->fill(0.5);
          _h[_sqs][1]->fill(xp);
        }
        else if (p.abspid()==431) {
          _r[_sqs][2]->fill(0.5);
          _h[_sqs][2]->fill(xp);
        }
        else if (p.abspid()==4122) {
          _r[_sqs][3]->fill(0.5);
          _h[_sqs][3]->fill(xp);
        }
        else if (p.abspid()==413) {
          _r[_sqs][4]->fill(0.5);
          _h[_sqs][4]->fill(xp);
          _r[_sqs][5]->fill(0.5);
          _h[_sqs][5]->fill(xp);
        }
        else if (p.abspid()==423) {
          _r[_sqs][6]->fill(0.5);
          _h[_sqs][6]->fill(xp);
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal/MeV));
        if (en == "10520"s) {
          scale(_r[en], crossSection()/picobarn/sumOfWeights());
        }
        else {
          scale(_r[en], 0.5/ *_c[en]);
        }
        scale(_h[en], crossSection()/nanobarn/sumOfWeights());
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,map<int,Histo1DPtr>> _h;
    map<string,map<int,Histo1DPtr>> _r;
    map<string,CounterPtr> _c;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(BELLE_2005_I686014);

}
