// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief D_s asymmetry
  class LHCB_2018_I1674916 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(LHCB_2018_I1674916);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // projection
      declare(UnstableParticles(), "UFS");

      for (double eVal : allowedEnergies()) {
        const int en = int(eVal+0.5);
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;

        for (size_t ix=0; ix<2; ++ix) {
          book(_h[en+ix], {2.,3.,3.5,4.5});
          for (auto& b : _h[en+ix]->bins()) {
            book(b,"TMP/h_"+toString(ix+1)+"_"+toString(b.index())+"_"+toString(en), refData(2,1,b.index()));
          }
        }
      }
      raiseBeamErrorIf(_sqs==0);
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const UnstableParticles& ufs = apply<UnstableParticles>(event, "UFS");
      for (const Particle& p : ufs.particles(Cuts::abspid==431) ) {
        if (p.fromBottom()) continue;
        const double pT = p.perp();
        const double y  = p.absrap();
        const bool anti = p.pid()<0;
        _h[_sqs+anti]->fill(y,pT);
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {

      for (double eVal : allowedEnergies()) {
        const int en = int(eVal+0.5);
        int ih = 2+(en==8000);
        // asymmetry
        Estimate1DPtr tmp;
        for (const auto& b : _h[en+0]->bins()) {
          book(tmp, ih, 1, b.index());
          asymm(b, _h[en+1]->bin(b.index()), tmp);
        }
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<int,Histo1DGroupPtr> _h;
    int _sqs = 0;
    /// @}


  };


  RIVET_DECLARE_PLUGIN(LHCB_2018_I1674916);

}
