; AleoBFT Library
;
; Copyright (C) 2024 Provable Inc.
;
; License: See the LICENSE file distributed with this library.
;
; Authors: Alessandro Coglio (www.alessandrocoglio.info)
;          Eric McCarthy (bendyarm on GitHub)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "ALEOBFT-STATIC")

(include-book "invariant-no-self-messages")

(local (include-book "kestrel/built-ins/disable" :dir :system))
(local (acl2::disable-most-builtin-logic-defuns))
(local (acl2::disable-builtin-rewrite-rules-for-defaults))
(set-induction-depth-limit 0)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ invariant-no-self-buffer
  :parents (correctness)
  :short "Invariant that the certificates in the buffer
          are from different validators."
  :long
  (xdoc::topstring
   (xdoc::p
    "This builds upon the "
    (xdoc::seetopic "invariant-no-self-messages"
                    "invariant that messages are never self-addressed")
    ". Certificates are put into the buffer
     when messages are received.
     Therefore, the authors of the certificates
     are never the destination of the message,
     i.e. the validator that owns the buffer in question."))
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define validator-buffer-not-self-p ((val addressp)
                                     (vstate validator-statep))
  :returns (yes/no booleanp)
  :short "Definition of the invariant at the validator level:
          the buffer has no certificates authored by the validator."
  :long
  (xdoc::topstring
   (xdoc::p
    "That is, retrieving from the buffer
     all the certificates authored by the validator
     yields the empty set @('nil')."))
  (equal (certificates-with-author val (validator-state->buffer vstate))
         nil))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define validators-buffer-not-self-p ((vstates validators-statep))
  :returns (yes/no booleanp)
  :short "Definition of the invariant at the validator map level:
          every correct validator in the map
          satisfies the validator-level invariant."
  :long
  (xdoc::topstring
   (xdoc::p
    "This lifts the invariant from single validators to validator maps.
     There is no requirement on faulty validators.")
   (xdoc::p
    "This definition and accompanying theorem
     could be automatically generated by a suitable macro."))
  (b* (((when (omap::emptyp vstates)) t)
       ((mv val vstate) (omap::head vstates))
       ((unless (or (not vstate)
                    (validator-buffer-not-self-p val vstate)))
        nil))
    (validators-buffer-not-self-p (omap::tail vstates)))
  ///

  (defrule validator-buffer-not-self-p-of-lookup
    (implies (and (validators-buffer-not-self-p vstates)
                  (omap::assoc val vstates)
                  (omap::lookup val vstates))
             (validator-buffer-not-self-p val (omap::lookup val vstates)))
    :induct t
    :enable omap::lookup)

  (defrule validators-buffer-not-self-p-of-update
    (implies (and (validators-buffer-not-self-p vstates)
                  (validator-buffer-not-self-p val vstate)
                  (addressp val))
             (validators-buffer-not-self-p
              (omap::update val vstate vstates)))
    :induct t
    :enable (omap::update
             omap::head
             omap::tail)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define system-buffer-not-self-p ((systate system-statep))
  :returns (yes/no booleanp)
  :short "Definition of the invariant at the system level:
          every correct validator in the system
          satisfies the validator-level invariant."
  :long
  (xdoc::topstring
   (xdoc::p
    "This lifts the invariant from the validator map to the system state.")
   (xdoc::p
    "This definition and accompanying theorems
     could be automatically generated by a suitable macro."))
  (validators-buffer-not-self-p (system-state->validators systate))
  :hooks (:fix)
  ///

  (defrule system-buffer-not-self-p-of-update-network-state
    (equal (system-buffer-not-self-p (update-network-state network systate))
           (system-buffer-not-self-p systate))
    :enable update-network-state)

  (defrule system-buffer-not-self-p-of-update-validator-state
    (implies (and (system-buffer-not-self-p systate)
                  (validator-buffer-not-self-p val vstate)
                  (addressp val)
                  (validator-statep vstate))
             (system-buffer-not-self-p
              (update-validator-state val vstate systate)))
    :enable update-validator-state)

  (defrule validator-buffer-not-self-p-of-get-validator-state
    (implies (and (system-buffer-not-self-p systate)
                  (set::in val (correct-addresses systate)))
             (validator-buffer-not-self-p val
                                          (get-validator-state val systate)))
    :enable (get-validator-state
             correct-addresses
             in-of-correct-addresses-loop)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defrule validator-buffer-not-self-p-of-validator-init
  :short "Establishment of the invariant at the validator level:
          the invariant holds on the initial validator state."
  :long
  (xdoc::topstring
   (xdoc::p
    "The initial buffer is empty, so the proof is trivial."))
  (validator-buffer-not-self-p val (validator-init))
  :enable (validator-buffer-not-self-p
           validator-init))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defrule validators-buffer-not-self-p-when-validators-state-initp
  :short "Establishment of the invariant at the validator map level:
          the invariant holds on any initial validator map."
  :long
  (xdoc::topstring
   (xdoc::p
    "This just follows structurally from
     the theorem on a single validator."))
  (implies (validators-state-initp vstates)
           (validators-buffer-not-self-p vstates))
  :induct t
  :enable (validators-buffer-not-self-p
           validators-state-initp))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defrule system-buffer-not-self-p-when-system-state-initp
  :short "Establishment of the invariant at the system level:
          the invariant holds on any initial system state."
  :long
  (xdoc::topstring
   (xdoc::p
    "This just follows from the theorem on the validator map.
     The network is not involved in the invariant."))
  (implies (system-state-initp systate)
           (system-buffer-not-self-p systate))
  :enable (system-state-initp
           system-buffer-not-self-p))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection system-buffer-not-self-p-of-add-endorsed
  :short "Preservation of the invariant by
          the addition of endorsed author-round pairs."
  :long
  (xdoc::topstring
   (xdoc::p
    "A @('create-certificate') event makes use of
     @(tsee add-endorsed),
     which does not modify any buffer,
     but we need to have a theorem about that,
     which is proved here."))

  (defrule validator-buffer-not-self-p-of-add-endorsed-val
    (implies (validator-buffer-not-self-p val vstate)
             (validator-buffer-not-self-p
              val (add-endorsed-val author round vstate)))
    :enable (validator-buffer-not-self-p
             add-endorsed-val))

  (defrule system-buffer-not-self-p-of-add-endorsed
    (implies (and (system-buffer-not-self-p systate)
                  (address-setp endorsers)
                  (set::subset endorsers (all-addresses systate)))
             (system-buffer-not-self-p
              (add-endorsed endorsers author round systate)))
    :induct t
    :enable (add-endorsed
             set::expensive-rules)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection system-buffer-not-self-p-of-create-certificate-next
  :short "Preservation of the invariant by @('create-certificate') events."
  :long
  (xdoc::topstring
   (xdoc::p
    "This kind of event does not modify any buffer.
     The proof makes use of the theorem about @(tsee add-endorsed),
     proved earlier."))

  (defrule validator-buffer-not-self-p-of-create-certificate-next-val
    (implies (validator-buffer-not-self-p val vstate)
             (validator-buffer-not-self-p
              val (create-certificate-next-val cert vstate)))
    :enable (validator-buffer-not-self-p
             create-certificate-next-val))

  (defrule system-buffer-not-self-p-of-create-certificate-next
    (implies (and (system-buffer-not-self-p systate)
                  (create-certificate-possiblep cert systate))
             (system-buffer-not-self-p
              (create-certificate-next cert systate)))
    :enable (create-certificate-possiblep
             create-certificate-next)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection system-buffer-not-self-p-of-receive-certificate-next
  :short "Preservation of the invariant by @('receive-certificate') events."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is one of the the only two kinds of events that modify buffers,
     precisely the buffer of the validator
     whose address is the destination of the message.
     In order to prove that the added new certificate
     is not authored by the validator,
     we need the invariant that
     messages in the network are not self-addressed.
     So that invariant, which has been separately proved,
     is added as a hypothesis to this theorem."))

  (defrule validator-buffer-not-self-p-of-receive-certificate-next-val
    (implies (and (validator-buffer-not-self-p val vstate)
                  (certificatep cert)
                  (addressp val)
                  (not (equal val (certificate->author cert))))
             (validator-buffer-not-self-p
              val (receive-certificate-next-val cert vstate)))
    :enable (validator-buffer-not-self-p
             receive-certificate-next-val))

  (defrule system-buffer-not-self-p-of-receive-certificate-next
    (implies (and (system-buffer-not-self-p systate)
                  (system-messages-not-self-p systate)
                  (receive-certificate-possiblep msg systate))
             (system-buffer-not-self-p
              (receive-certificate-next msg systate)))
    :enable (receive-certificate-possiblep
             receive-certificate-next
             system-messages-not-self-p
             message-not-self-p
             get-network-state)
    :use (:instance message-set-not-self-p-element
                    (msgs (system-state->network systate)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection system-buffer-not-self-p-of-store-certificate-next
  :short "Preservation of the invariant by @('store-certificate') events."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is one of the only two kinds of events that modify buffers,
     but it removes a certificate from a buffer,
     so the proof is easy."))

  (defrule validator-buffer-not-self-p-of-store-certificate-next-val
    (implies (and (validator-buffer-not-self-p val vstate)
                  (addressp val))
             (validator-buffer-not-self-p
              val (store-certificate-next-val cert vstate)))
    :enable (validator-buffer-not-self-p
             store-certificate-next-val))

  (defrule system-buffer-not-self-p-of-store-certificate-next
    (implies (and (system-buffer-not-self-p systate)
                  (store-certificate-possiblep cert val systate))
             (system-buffer-not-self-p
              (store-certificate-next cert val systate)))
    :enable (store-certificate-possiblep
             store-certificate-next)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection system-buffer-not-self-p-of-advance-round-next
  :short "Preservation of the invariant by @('advance-round') events."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is easy to prove,
     because that kind of events does not modify buffers."))

  (defrule validator-buffer-not-self-p-of-advance-round-next-val
    (implies (validator-buffer-not-self-p val vstate)
             (validator-buffer-not-self-p
              val (advance-round-next-val round vstate)))
    :enable (validator-buffer-not-self-p
             advance-round-next-val))

  (defrule system-buffer-not-self-p-of-advance-round-next
    (implies (and (system-buffer-not-self-p systate)
                  (advance-round-possiblep val systate))
             (system-buffer-not-self-p
              (advance-round-next val systate)))
    :enable (advance-round-possiblep
             advance-round-next)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection system-buffer-not-self-p-of-commit-anchors-next
  :short "Preservation of the invariant by @('commit-anchors') events."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is easy to prove,
     because that kind of events does not modify buffers."))

  (defrule validator-buffer-not-self-p-of-commit-anchors-next-val
    (implies (validator-buffer-not-self-p val vstate)
             (validator-buffer-not-self-p
              val (commit-anchors-next-val vals vstate)))
    :enable (validator-buffer-not-self-p
             commit-anchors-next-val))

  (defrule system-buffer-not-self-p-of-commit-anchors-next
    (implies (and (system-buffer-not-self-p systate)
                  (commit-anchors-possiblep val systate))
             (system-buffer-not-self-p
              (commit-anchors-next val systate)))
    :enable (commit-anchors-possiblep
             commit-anchors-next)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection system-buffer-not-self-p-of-timer-expires-next
  :short "Preservation of the invariant by @('timer-expires') events."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is easy to prove,
     because that kind of events does not modify buffers."))

  (defrule validator-buffer-not-self-p-of-timer-expires-next-val
    (implies (validator-buffer-not-self-p val vstate)
             (validator-buffer-not-self-p
              val (timer-expires-next-val vstate)))
    :enable (validator-buffer-not-self-p
             timer-expires-next-val))

  (defrule system-buffer-not-self-p-of-timer-expires-next
    (implies (and (system-buffer-not-self-p systate)
                  (timer-expires-possiblep val systate))
             (system-buffer-not-self-p (timer-expires-next val systate)))
    :enable (timer-expires-possiblep
             timer-expires-next)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defrule system-buffer-not-self-p-of-event-next
  :short "Preservation of the invariant by all events."
  :long
  (xdoc::topstring
   (xdoc::p
    "This follows from the theorems about the various kinds of events.
     The invariant about non-self-addressed messages,
     used in one of those theorems,
     propagates here."))
  (implies (and (system-buffer-not-self-p systate)
                (event-possiblep event systate)
                (system-messages-not-self-p systate))
           (system-buffer-not-self-p
            (event-next event systate)))
  :enable (event-possiblep event-next))
