/*
 * STAN -- Stream Analyser -- http://www.roqe.org/stan
 * Copyright (c) 2001-2004 Konrad Rieck <kr@roqe.org> 
 * All rights reserved.
 * ---
 * $Id: bintree.c,v 1.8 2003/12/10 20:29:29 kr Exp $
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    ``This product includes software developed by Konrad Rieck.''
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * ---
 *
 * Implementation of a binary tree. This data structure is rather simple.
 * Nodes in the tree represent runs of bits. Each node also carries the
 * count of true and false runs. The delete and rotate operations have been
 * skipped since they aren't necessary and I already did all that annoying
 * stuff in treap.c
 */
 
#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>

#include <stan.h>
#include <treap.h>
#include <imath.h>
#include <bintree.h>

bnode_t *bintree;

void bfree(bnode_t * bnode)
{
   dprintf(STRONG, ("bfree: freeing bnode 0x%p\n", bnode));
   free(bnode);
}

void btraverse(bnode_t ** bnode, void (*func) (bnode_t *), int order)
{
   if (*bnode) {
      switch (order) {
      case preorder:
	 func(*bnode);
	 btraverse(&(*bnode)->left, func, order);
	 btraverse(&(*bnode)->right, func, order);
	 break;
      case inorder:
	 btraverse(&(*bnode)->left, func, order);
	 func(*bnode);
	 btraverse(&(*bnode)->right, func, order);
	 break;
      case postorder:
	 btraverse(&(*bnode)->left, func, order);
	 btraverse(&(*bnode)->right, func, order);
	 func(*bnode);
	 break;
      }
   }
}

void init_bintree()
{
   bintree = NULL;
}

void free_bintree()
{
   btraverse(&bintree, bfree, postorder);
}

bnode_t **blocate(bnode_t ** bnode, size_t run)
{
   dprintf(STRONG, ("blocate: bnode 0x%p\n", bnode));

   if (!*bnode) {
      return bnode;
   }

   dprintf(STRONG, ("blocate: bnode 0x%p exists, checking run\n", bnode));

   if ((*bnode)->run > run)
      return blocate(&(*bnode)->left, run);
   else if ((*bnode)->run < run)
      return blocate(&(*bnode)->right, run);

   dprintf(STRONG, ("blocate: bnode 0x%p matches run\n", bnode));
   return bnode;
}

size_t bhigh(bnode_t ** bnode)
{
   if (!*bnode || !(*bnode)->parent)
      return 0;
   return bhigh(&(*bnode)->parent) + 1;
}

void bprint(bnode_t * bnode)
{
   size_t i;
   for (i = bhigh(&bnode); i > 0; i--)
      printf("     ");
   printf("(%d:[%u/%u])\n", bnode->run, bnode->count_false,
	  bnode->count_true);
}

void binsert(bnode_t ** bnode, size_t run, int bit)
{
   bnode_t **parent = NULL;

   while (*bnode) {
      parent = &(*bnode);
      if ((*bnode)->run > run)
	 bnode = &(*bnode)->left;
      else if ((*bnode)->run < run)
	 bnode = &(*bnode)->right;
      else
	 break;
   }

   if (!*bnode) {
      dprintf(STRONG,
	      ("binsert: allocating memory for bnode 0x%p\n", bnode));
      dprintf(STRONG,
	      ("binsert: setting parent 0x%p for bnode 0x%p\n",
	       parent, bnode));
      *bnode = (bnode_t *) calloc(1, sizeof(bnode_t));
      (*bnode)->run = run;
      if (bit)
	 (*bnode)->count_true = 1;
      else
	 (*bnode)->count_false = 1;

      if (parent) {
	 (*bnode)->parent = *parent;
      } else {
	 (*bnode)->parent = NULL;
      }
   } else {
      dprintf(STRONG, ("binsert: increment count for bnode 0x%p\n", bnode));

      if (bit)
	 (*bnode)->count_true++;
      else
	 (*bnode)->count_false++;
   }
}

size_t bdepth(bnode_t ** bnode)
{
   if (!*bnode || (!(*bnode)->left && !(*bnode)->right))
      return 0;
   return max(bdepth(&(*bnode)->left), bdepth(&(*bnode)->right)) + 1;
}

bnode_t **bmax(bnode_t ** bnode)
{
   if (!*bnode || !(*bnode)->right)
      return bnode;
   return bmax(&(*bnode)->right);
}

bnode_t **bmin(bnode_t ** bnode)
{
   if (!*bnode || !(*bnode)->left)
      return bnode;
   return bmin(&(*bnode)->left);
}

bnode_t **bsuccessor(bnode_t ** bnode)
{
   bnode_t **parent;

   if (!*bnode)
      return bnode;

   if ((*bnode)->right)
      return bmin(&(*bnode)->right);

   if (!(*bnode)->parent)
      return bnode;

   parent = &(*bnode)->parent;
   while (*parent && *bnode == (*parent)->right) {
      bnode = parent;
      parent = &(*bnode)->parent;
   }
   return parent;
}

bnode_t **bpredecessor(bnode_t ** bnode)
{
   bnode_t **parent;

   if (!*bnode)
      return bnode;

   if ((*bnode)->left)
      return bmax(&(*bnode)->left);

   if (!(*bnode)->parent)
      return bnode;

   parent = &(*bnode)->parent;
   while (*parent && *bnode == (*parent)->left) {
      bnode = parent;
      parent = &(*bnode)->parent;
   }
   return parent;
}

size_t bsize_bnodes(bnode_t ** bnode, int bit)
{
   size_t add = 0;

   if (!*bnode)
      return 0;

   if ((bit && (*bnode)->count_true) || (!bit && (*bnode)->count_false))
      add = 1;

   return bsize_bnodes(&(*bnode)->left, bit) +
       bsize_bnodes(&(*bnode)->right, bit) + add;
}

size_t bsize_total(bnode_t ** bnode, int bit)
{
   size_t add;

   if (!*bnode)
      return 0;

   if (bit)
      add = (*bnode)->count_true;
   else
      add = (*bnode)->count_false;

   return bsize_total(&(*bnode)->left, bit) +
       bsize_total(&(*bnode)->right, bit) + add;
}
