package sfxclient

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net"
	"net/http"
	"net/http/httptest"
	"strings"
	"testing"
	"time"

	"github.com/golang/protobuf/proto"
	"github.com/signalfx/com_signalfx_metrics_protobuf"
	"github.com/signalfx/golib/v3/datapoint"
	"github.com/signalfx/golib/v3/errors"
	"github.com/signalfx/golib/v3/event"
	"github.com/signalfx/golib/v3/log"
	"github.com/signalfx/golib/v3/pointer"
	"github.com/signalfx/golib/v3/trace"
	sapmpb "github.com/signalfx/sapm-proto/gen"
	. "github.com/smartystreets/goconvey/convey"
)

type errReader struct {
	shouldBlock chan struct{}
}

var errReadErr = errors.New("read bad")

func (e *errReader) Read(_ []byte) (n int, err error) {
	if e.shouldBlock != nil {
		<-e.shouldBlock
	}
	return 0, errReadErr
}

func TestHelperFunctions(t *testing.T) {
	Convey("Just helpers", t, func() {
		Convey("mapToDimensions should filter empty", func() {
			So(len(mapToDimensions(map[string]string{"": "hi"})), ShouldEqual, 0)
		})
	})
}

func TestCounter(t *testing.T) {
	Convey("A counter should create", t, func() {
		c := Counter("counter", nil, 10)
		So(c.Value.String(), ShouldEqual, "10")
	})
}

// GoEventSource returns a set of events that can be tested
var GoEventSource = &goEvents{}

type goEvents struct{}

func (g *goEvents) Events() []*event.Event {
	dims := map[string]string{
		"instance": "global_stats",
		"stattype": "golang_sys",
	}
	return []*event.Event{
		event.NewWithProperties("Alloc", event.COLLECTD, dims, make(map[string]interface{}), time.Time{}),
		event.New("TotalAlloc", event.COLLECTD, dims, time.Time{}),
		event.New("Sys", event.COLLECTD, dims, time.Time{}),
		event.New("Lookups", event.COLLECTD, dims, time.Time{}),
		event.New("Mallocs", event.COLLECTD, dims, time.Time{}),
		event.New("Frees", event.COLLECTD, dims, time.Time{}),
		event.New("HeapAlloc", event.ALERT, dims, time.Time{}),
		event.New("HeapSys", event.AUDIT, dims, time.Time{}),
		event.New("HeapIdle", event.EXCEPTION, dims, time.Time{}),
		event.New("HeapInuse", event.SERVICEDISCOVERY, dims, time.Time{}),
		event.New("HeapReleased", event.JOB, dims, time.Time{}),
		event.New("HeapObjects", event.USERDEFINED, dims, time.Time{}),
		event.New("StackInuse", event.COLLECTD, dims, time.Time{}),
		event.New("StackSys", event.COLLECTD, dims, time.Time{}),
		event.New("MSpanInuse", event.COLLECTD, dims, time.Time{}),
		event.New("MSpanSys", event.COLLECTD, dims, time.Time{}),
		event.New("MCacheInuse", event.COLLECTD, dims, time.Time{}),
		event.New("MCacheSys", event.COLLECTD, dims, time.Time{}),
		event.New("BuckHashSys", event.COLLECTD, dims, time.Time{}),
		event.New("GCSys", event.COLLECTD, dims, time.Time{}),
		event.New("OtherSys", event.COLLECTD, dims, time.Time{}),
		event.New("NextGC", event.COLLECTD, dims, time.Time{}),
		event.New("LastGC", event.COLLECTD, dims, time.Time{}),
		event.New("PauseTotalNs", event.COLLECTD, dims, time.Time{}),
		event.New("NumGC", event.COLLECTD, dims, time.Time{}),

		event.New("GOMAXPROCS", event.COLLECTD, dims, time.Time{}),
		event.New("process.uptime.ns", event.COLLECTD, dims, time.Time{}),
		event.New("num_cpu", event.COLLECTD, dims, time.Time{}),

		event.New("num_cgo_call", event.COLLECTD, dims, time.Time{}),

		event.New("num_goroutine", event.COLLECTD, dims, time.Time{}),
	}
}

func TestGoEventSource(t *testing.T) {
	Convey("go events should fetch", t, func() {
		So(len(GoEventSource.Events()), ShouldEqual, 30)
	})
}

func ExampleHTTPSink() {
	sink := NewHTTPSink()
	sink.AuthToken = "ABCDEFG"
	ctx := context.Background()
	err := sink.AddDatapoints(ctx, []*datapoint.Datapoint{
		// Sending a gauge with the value 1.2
		GaugeF("a.gauge", nil, 1.2),
		// Sending a cumulative counter with dimensions
		Cumulative("a.counter", map[string]string{"type": "dev"}, 100),
	})
	if err != nil {
		panic(err)
	}
}

func TestHTTPDatapointSink(t *testing.T) {
	Convey("A default sink", t, func() {
		s := NewHTTPSink()
		count := 0
		s.ResponseCallback = func(resp *http.Response, responseBody []byte) {
			count++
		}
		s.AdditionalHeaders = map[string]string{"key": "value"}
		s.DisableCompression = true
		ctx := context.Background()
		dps := GoMetricsSource.Datapoints()
		Convey("should timeout", func() {
			s.Client.Timeout = time.Millisecond
			So(s.AddDatapoints(ctx, dps), ShouldNotBeNil)
		})
		Convey("should not try dead contexts", func() {
			var ctx = context.Background()
			ctx, can := context.WithCancel(ctx)
			can()
			So(errors.Details(s.AddDatapoints(ctx, dps)), ShouldContainSubstring, "context already closed")
			Convey("but empty points should always work", func() {
				So(s.AddDatapoints(ctx, []*datapoint.Datapoint{}), ShouldBeNil)
			})
		})
		Convey("should check failure to encode", func() {
			s.protoMarshaler = func(pb proto.Message) ([]byte, error) {
				return nil, errors.New("failure to encode")
			}
			So(errors.Details(s.AddDatapoints(ctx, dps)), ShouldContainSubstring, "failure to encode")
		})
		Convey("should check invalid endpoints", func() {
			s.DatapointEndpoint = "%gh&%ij"
			So(errors.Details(s.AddDatapoints(ctx, dps)), ShouldContainSubstring, "cannot parse new HTTP request to")
		})
		Convey("reading the full body should be checked", func() {
			resp := &http.Response{
				Body: ioutil.NopCloser(&errReader{}),
			}
			So(errors.Tail(s.handleResponse(resp, datapointAndEventResponseValidator)).Error(), ShouldEqual, "cannot fully read response body: read bad: map[]")
		})
		Convey("with a test endpoint", func() {
			retString := `"OK"`
			retCode := http.StatusOK
			var blockResponse chan struct{}
			var cancelCallback func()
			seenBodyPoints := &com_signalfx_metrics_protobuf.DataPointUploadMessage{}
			handler := http.HandlerFunc(func(rw http.ResponseWriter, req *http.Request) {
				bodyBytes := bytes.Buffer{}
				_, err := io.Copy(&bodyBytes, req.Body)
				log.IfErr(log.Panic, err)
				log.IfErr(log.Panic, req.Body.Close())
				log.IfErr(log.Panic, proto.Unmarshal(bodyBytes.Bytes(), seenBodyPoints))
				rw.WriteHeader(retCode)
				errors.PanicIfErrWrite(io.WriteString(rw, retString))
				if blockResponse != nil {
					if cancelCallback != nil {
						cancelCallback()
					}
					select {
					case <-req.Context().Done():
					case <-blockResponse:
					}
				}
			})

			// Note: Using httptest created some strange race conditions around their use of wait group, so
			//       I'm creating my own listener here that I close in Reset()
			l, err := net.Listen("tcp", "127.0.0.1:0")
			So(err, ShouldBeNil)
			server := http.Server{
				Handler: handler,
			}
			serverDone := make(chan struct{})
			go func() {
				if err := server.Serve(l); err == nil {
					panic("I expect serve to eventually error")
				}
				close(serverDone)
			}()
			s.DatapointEndpoint = "http://" + l.Addr().String()
			Convey("Send should normally work", func() {
				So(s.AddDatapoints(ctx, dps), ShouldBeNil)
			})
			Convey("Send should statusAccepted work", func() {
				retCode = http.StatusAccepted
				So(s.AddDatapoints(ctx, dps), ShouldBeNil)
			})
			Convey("should send timestamps", func() {
				dps = dps[0:1]
				now := time.Now()
				dps[0].Timestamp = now
				So(s.AddDatapoints(ctx, dps), ShouldBeNil)
				So(*seenBodyPoints.Datapoints[0].Timestamp, ShouldEqual, now.UnixNano()/time.Millisecond.Nanoseconds())
			})
			Convey("Floats should work", func() {
				dps[0].Value = datapoint.NewFloatValue(1.0)
				dps = dps[0:1]
				So(s.AddDatapoints(ctx, dps), ShouldBeNil)
				So(len(seenBodyPoints.Datapoints), ShouldEqual, 1)
				So(*seenBodyPoints.Datapoints[0].Value.DoubleValue, ShouldEqual, 1.0)
			})
			Convey("Strings should work", func() {
				dps[0].Value = datapoint.NewStringValue("hi")
				dps = dps[0:1]
				So(s.AddDatapoints(ctx, dps), ShouldBeNil)
				So(len(seenBodyPoints.Datapoints), ShouldEqual, 1)
				So(*seenBodyPoints.Datapoints[0].Value.StrValue, ShouldEqual, "hi")
			})
			Convey("empty key filtering should happen", func() {
				dps[0].Dimensions = map[string]string{"": "hi"}
				dps = dps[0:1]
				So(s.AddDatapoints(ctx, dps), ShouldBeNil)
				So(len(seenBodyPoints.Datapoints[0].Dimensions), ShouldEqual, 0)
			})
			Convey("invalid rune filtering should happen", func() {
				dps[0].Dimensions = map[string]string{"hi.bob": "hi"}
				dps = dps[0:1]
				So(s.AddDatapoints(ctx, dps), ShouldBeNil)
				So(*seenBodyPoints.Datapoints[0].Dimensions[0].Key, ShouldEqual, "hi_bob")
			})
			Convey("Invalid datapoints should panic", func() {
				dps[0].MetricType = datapoint.MetricType(1001)
				So(func() { log.IfErr(log.Panic, s.AddDatapoints(ctx, dps)) }, ShouldPanic)
			})
			Convey("return code should be checked", func() {
				retCode = http.StatusNotAcceptable
				So(errors.Details(s.AddDatapoints(ctx, dps)), ShouldContainSubstring, "invalid status code")
			})
			Convey("return string should be checked", func() {
				retString = `"nope"`
				So(errors.Details(s.AddDatapoints(ctx, dps)), ShouldContainSubstring, "invalid response body")
				retString = `INVALID_JSON`
				So(errors.Details(s.AddDatapoints(ctx, dps)), ShouldContainSubstring, "cannot unmarshal response body")
				So(count, ShouldEqual, 2)
			})
			Convey("context cancel should work", func() {
				blockResponse = make(chan struct{})
				ctx, cancelCallback = context.WithCancel(ctx)
				s := errors.Details(s.AddDatapoints(ctx, dps))
				if !strings.Contains(s, "canceled") && !strings.Contains(s, "closed") {
					t.Errorf("Bad error string %s", s)
				}
				cancelCallback()
			})
			Convey("timeouts should work", func() {
				blockResponse = make(chan struct{})
				s.Client.Timeout = time.Millisecond * 10
				So(errors.Details(s.AddDatapoints(ctx, dps)), ShouldContainSubstring, "Client.Timeout")
			})
			Reset(func() {
				if blockResponse != nil {
					close(blockResponse)
				}
				So(l.Close(), ShouldBeNil)
				<-serverDone
			})
		})
	})
}

func BenchmarkHTTPSinkCreate(b *testing.B) {
	for i := 0; i < b.N; i++ {
		ec := NewHTTPSink()
		ec.AuthToken = "abcdefg"
	}
}

func BenchmarkHTTPSinkAddIndividualDatapoints(b *testing.B) {
	points := GoMetricsSource.Datapoints()
	sink := NewHTTPSink()
	ctx := context.Background()
	l := len(points)
	for i := 0; i < b.N; i++ {
		for j := 0; j < l; j++ {
			var dp = make([]*datapoint.Datapoint, 0)
			dp = append(dp, points[j])
			_ = sink.AddDatapoints(ctx, dp)
		}
	}
}

func BenchmarkHTTPSinkAddSeveralDatapoints(b *testing.B) {
	points := GoMetricsSource.Datapoints()
	sink := NewHTTPSink()
	ctx := context.Background()
	for i := 0; i < b.N; i++ {
		_ = sink.AddDatapoints(ctx, points)
	}
}

func BenchmarkHTTPSinkAddIndividualEvents(b *testing.B) {
	events := GoEventSource.Events()
	sink := NewHTTPSink()
	ctx := context.Background()
	l := len(events)
	for i := 0; i < b.N; i++ {
		for j := 0; j < l; j++ {
			var ev = make([]*event.Event, 0)
			ev = append(ev, events[j])
			_ = sink.AddEvents(ctx, ev)
		}
	}
}

func BenchmarkHTTPSinkAddSeveralEvents(b *testing.B) {
	events := GoEventSource.Events()
	sink := NewHTTPSink()
	ctx := context.Background()
	for i := 0; i < b.N; i++ {
		_ = sink.AddEvents(ctx, events)
	}
}

func TestHTTPEventSink(t *testing.T) {
	Convey("A default event sink", t, func() {
		s := NewHTTPSink()
		s.DisableCompression = true
		ctx := context.Background()
		events := GoEventSource.Events()
		Convey("should timeout", func() {
			s.Client.Timeout = time.Millisecond
			So(s.AddEvents(ctx, events), ShouldNotBeNil)
		})
		Convey("should not try dead contexts", func() {
			var ctx = context.Background()
			ctx, can := context.WithCancel(ctx)
			can()
			So(errors.Details(s.AddEvents(ctx, events)), ShouldContainSubstring, "context already closed")
			Convey("but empty events should always work", func() {
				So(s.AddEvents(ctx, []*event.Event{}), ShouldBeNil)
			})
		})
		Convey("should check failure to encode", func() {
			s.protoMarshaler = func(pb proto.Message) ([]byte, error) {
				return nil, errors.New("failure to encode")
			}
			So(errors.Details(s.AddEvents(ctx, events)), ShouldContainSubstring, "failure to encode")
		})
		Convey("should check invalid endpoints", func() {
			s.EventEndpoint = "%gh&%ij"
			So(errors.Details(s.AddEvents(ctx, events)), ShouldContainSubstring, "cannot parse new HTTP request to")
		})
		Convey("reading the full body should be checked", func() {
			resp := &http.Response{
				Body: ioutil.NopCloser(&errReader{}),
			}
			So(errors.Tail(s.handleResponse(resp, datapointAndEventResponseValidator)).Error(), ShouldEqual, "cannot fully read response body: read bad: map[]")
		})
		Convey("with a test endpoint", func() {
			retString := `"OK"`
			retCode := http.StatusOK
			var blockResponse chan struct{}
			var cancelCallback func()
			seenBodyEvents := &com_signalfx_metrics_protobuf.EventUploadMessage{}
			handler := http.HandlerFunc(func(rw http.ResponseWriter, req *http.Request) {
				bodyBytes := bytes.Buffer{}
				_, err := io.Copy(&bodyBytes, req.Body)
				log.IfErr(log.Panic, err)
				log.IfErr(log.Panic, req.Body.Close())
				log.IfErr(log.Panic, proto.Unmarshal(bodyBytes.Bytes(), seenBodyEvents))
				rw.WriteHeader(retCode)
				errors.PanicIfErrWrite(io.WriteString(rw, retString))
				if blockResponse != nil {
					if cancelCallback != nil {
						cancelCallback()
					}
					select {
					case <-req.Context().Done():
					case <-blockResponse:
					}
				}
			})

			// Note: Using httptest created some strange race conditions around their use of wait group, so
			//       I'm creating my own listener here that I close in Reset()
			l, err := net.Listen("tcp", "127.0.0.1:0")
			So(err, ShouldBeNil)
			server := http.Server{
				Handler: handler,
			}
			serverDone := make(chan struct{})
			go func() {
				if err := server.Serve(l); err == nil {
					panic("I expect serve to eventually error")
				}
				close(serverDone)
			}()
			s.EventEndpoint = "http://" + l.Addr().String()
			Convey("Send should normally work", func() {
				So(s.AddEvents(ctx, events), ShouldBeNil)
			})
			Convey("should send timestamps", func() {
				events = events[0:1]
				now := time.Now()
				events[0].Timestamp = now
				So(s.AddEvents(ctx, events), ShouldBeNil)
				So(*seenBodyEvents.Events[0].Timestamp, ShouldEqual, now.UnixNano()/time.Millisecond.Nanoseconds())
			})
			Convey("Should send properties for event", func() {
				events[0].Properties["name"] = "jack"
				events = events[0:1]
				So(s.AddEvents(ctx, events), ShouldBeNil)
				So(len(seenBodyEvents.Events), ShouldEqual, 1)
				So(*seenBodyEvents.Events[0].Properties[0].Key, ShouldEqual, "name")
			})
			Convey("All property types should send", func() {
				events[0].Properties["name"] = "jack"
				events[0].Properties["age"] = 33
				events[0].Properties["awesome"] = true
				events[0].Properties["extra"] = int64(123)
				events[0].Properties["ratio"] = 1.0
				events[0].Properties["unused"] = func() {}
				events = events[0:1]
				So(s.AddEvents(ctx, events), ShouldBeNil)
				So(len(seenBodyEvents.Events), ShouldEqual, 1)
				So(len(seenBodyEvents.Events[0].Properties), ShouldEqual, 5)
			})
			Convey("empty key filtering should happen", func() {
				events[0].Dimensions = map[string]string{"": "hi"}
				events = events[0:1]
				So(s.AddEvents(ctx, events), ShouldBeNil)
				So(len(seenBodyEvents.Events[0].Dimensions), ShouldEqual, 0)
			})
			Convey("invalid rune filtering should happen", func() {
				events[0].Dimensions = map[string]string{"hi.bob": "hi"}
				events = events[0:1]
				So(s.AddEvents(ctx, events), ShouldBeNil)
				So(*seenBodyEvents.Events[0].Dimensions[0].Key, ShouldEqual, "hi_bob")
			})
			Convey("Invalid events should panic", func() {
				events[0].Category = event.Category(999999)
				So(func() { log.IfErr(log.Panic, s.AddEvents(ctx, events)) }, ShouldPanic)
			})
			Convey("return code should be checked", func() {
				retCode = http.StatusNotAcceptable
				So(errors.Details(s.AddEvents(ctx, events)), ShouldContainSubstring, "invalid status code")
			})
			Convey("return string should be checked", func() {
				retString = `"nope"`
				So(errors.Details(s.AddEvents(ctx, events)), ShouldContainSubstring, "invalid response body")
				retString = `INVALID_JSON`
				So(errors.Details(s.AddEvents(ctx, events)), ShouldContainSubstring, "cannot unmarshal response body")
			})
			Convey("context cancel should work", func() {
				blockResponse = make(chan struct{})
				ctx, cancelCallback = context.WithCancel(ctx)
				s := errors.Details(s.AddEvents(ctx, events))
				if !strings.Contains(s, "canceled") && !strings.Contains(s, "closed") {
					t.Errorf("Bad error string %s", s)
				}
				cancelCallback()
			})
			Convey("timeouts should work", func() {
				blockResponse = make(chan struct{})
				s.Client.Timeout = time.Millisecond * 10
				So(errors.Details(s.AddEvents(ctx, events)), ShouldContainSubstring, "Client.Timeout")
			})
			Reset(func() {
				if blockResponse != nil {
					close(blockResponse)
				}
				So(l.Close(), ShouldBeNil)
				<-serverDone
			})
		})
	})
}

func TestHTTPTraceSink(t *testing.T) {
	Convey("A default trace sink", t, func() {
		s := NewHTTPSink()
		ctx := context.Background()
		Convey("should be configured to use JSON", func() {
			So(s.contentTypeHeader, ShouldEqual, contentTypeHeaderJSON)
			So(s.traceMarshal, ShouldEqual, jsonMarshal)
		})
		var traces []*trace.Span
		err := json.Unmarshal([]byte(trace.ValidJSON), &traces)
		So(err, ShouldBeNil)
		Convey("should timeout", func() {
			s.Client.Timeout = time.Millisecond
			So(s.AddSpans(ctx, traces), ShouldNotBeNil)
		})
		Convey("should not try dead contexts", func() {
			var ctx = context.Background()
			ctx, can := context.WithCancel(ctx)
			can()
			So(errors.Details(s.AddSpans(ctx, traces)), ShouldContainSubstring, "context already closed")
			Convey("but empty traces should always work", func() {
				So(s.AddSpans(ctx, []*trace.Span{}), ShouldBeNil)
			})
		})
		Convey("should check failure to encode", func() {
			s.traceMarshal = func(v []*trace.Span) ([]byte, error) {
				return nil, errors.New("failure to encode")
			}
			So(errors.Details(s.AddSpans(ctx, traces)), ShouldContainSubstring, "failure to encode")
		})
		Convey("should check invalid endpoints", func() {
			s.TraceEndpoint = "%gh&%ij"
			err := s.AddSpans(ctx, traces)
			fmt.Println(err)
			So(errors.Details(err), ShouldContainSubstring, "cannot parse new HTTP request to")
		})
		Convey("reading the full body should be checked", func() {
			resp := &http.Response{
				Body: ioutil.NopCloser(&errReader{}),
			}
			So(errors.Tail(s.handleResponse(resp, spanResponseValidator)).Error(), ShouldEqual, "cannot fully read response body: read bad: map[]")
		})

		Convey("with a test endpoint", func() {
			retString := `{"invalid":{}, "valid":1}`
			var blockResponse chan struct{}
			var cancelCallback func()
			seenSpans := []*trace.Span{}
			handler := http.HandlerFunc(func(rw http.ResponseWriter, req *http.Request) {
				bodyBytes := bytes.Buffer{}
				_, err := io.Copy(&bodyBytes, req.Body)
				log.IfErr(log.Panic, err)
				log.IfErr(log.Panic, req.Body.Close())
				log.IfErr(log.Panic, json.Unmarshal(bodyBytes.Bytes(), &seenSpans))
				rw.WriteHeader(http.StatusOK)
				errors.PanicIfErrWrite(io.WriteString(rw, retString))
				if blockResponse != nil {
					if cancelCallback != nil {
						cancelCallback()
					}
					select {
					case <-req.Cancel:
					case <-blockResponse:
					}
				}
			})

			// Note: Using httptest created some strange race conditions around their use of wait group, so
			//       I'm creating my own listener here that I close in Reset()
			l, err := net.Listen("tcp", "127.0.0.1:0")
			So(err, ShouldBeNil)
			server := http.Server{
				Handler: handler,
			}
			serverDone := make(chan struct{})
			go func() {
				if err := server.Serve(l); err == nil {
					panic("I expect serve to eventually error")
				}
				close(serverDone)
			}()
			s.TraceEndpoint = "http://" + l.Addr().String()
			Convey("Send should normally work", func() {
				So(s.AddSpans(ctx, traces), ShouldBeNil)
			})
			Convey("should error on invalid spans", func() {
				retString = `{"invalid":{"invalidSpanID":["abcdef1234567890"]}, "valid":0}`
				err := s.AddSpans(ctx, traces)
				So(errors.Details(err), ShouldContainSubstring, `"invalidSpanID":["abcdef1234567890"]`)
			})
			Convey("should error on invalid json response", func() {
				retString = `"OK"`
				err := s.AddSpans(ctx, traces)
				So(err, ShouldBeNil)
			})
			Reset(func() {
				if blockResponse != nil {
					close(blockResponse)
				}
				So(l.Close(), ShouldBeNil)
				<-serverDone
			})
		})
	})
}

func TestHTTPTraceZipkinSink(t *testing.T) {
	Convey("A trace sink with Zipkin option", t, func() {
		s := NewHTTPSink(WithZipkinTraceExporter())
		Convey("should export to SAPM endpoint and use SAPM marshaller", func() {
			So(s.contentTypeHeader, ShouldEqual, contentTypeHeaderJSON)
			So(s.traceMarshal, ShouldEqual, jsonMarshal)
		})
	})
}

func TestHTTPTraceSAPMSink(t *testing.T) {
	Convey("A trace sink with SAPM option", t, func() {
		s := NewHTTPSink(WithSAPMTraceExporter())
		Convey("should export to SAPM endpoint and use SAPM marshaller", func() {
			So(s.contentTypeHeader, ShouldEqual, contentTypeHeaderSAPM)
			So(s.traceMarshal, ShouldEqual, sapmMarshal)
		})
	})
}

func TestSAPMMarshal(t *testing.T) {
	Convey("A SAPM marshaller", t, func() {
		traces := []*trace.Span{
			{
				TraceID: "fa281a8955571a3a",
				ID:      "acdfec5be6328c3a",
				Name:    pointer.String("get"),
			},
		}
		marshalled, err := sapmMarshal(traces)
		Convey("should marshal traces", func() {
			So(err, ShouldBeNil)
			psr := sapmpb.PostSpansRequest{}
			err = proto.Unmarshal(marshalled, &psr)
			Convey("which should unmarshal to SAPM PostSpansRequest", func() {
				So(err, ShouldBeNil)
				So(psr, ShouldNotBeNil)
				Convey("and the request should have expected data", func() {
					So(len(psr.Batches), ShouldEqual, 1)
					So(len(psr.Batches[0].Spans), ShouldEqual, 1)
					span := psr.Batches[0].Spans[0]
					So(span.SpanID.String(), ShouldEqual, traces[0].ID)
					So(span.TraceID.String(), ShouldEqual, traces[0].TraceID)
					So(span.OperationName, ShouldEqual, *traces[0].Name)
				})
			})
		})

	})
}

const longTraceExample = `[{"traceId":"05d68b91f746bc97","id":"05d68b91f746bc97","name":"signalflowsocket.onwebsockettext","timestamp":1530215261156007,"duration":19242,"annotations":[{"timestamp":1530215261156007,"value":"sr","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"timestamp":1530215261175249,"value":"ss","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}}],"binaryAnnotations":[{"key":"ca","value":true,"endpoint":{"serviceName":"","port":53018}},{"key":"component","value":"jetty-websocket","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"peer.hostname","value":"localhost","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"request.type","value":"execute","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"source","value":"signalboost-rest-lab13--ccaa","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}}]},{"traceId":"05d68b91f746bc97","id":"56863874f14c0c6a","name":"signalflowsocket.execute","parentId":"05d68b91f746bc97","timestamp":1530215261156144,"duration":136233,"binaryAnnotations":[{"key":"orgId","value":"CMtLSYXAEJQ","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"source","value":"signalboost-rest-lab13--ccaa","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"userId","value":"CxWkjSnAAAE","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}}]},{"traceId":"05d68b91f746bc97","id":"715a70c7752246e3","name":"analyticshelper.execute","parentId":"56863874f14c0c6a","timestamp":1530215261175310,"duration":114996,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"source","value":"signalboost-rest-lab13--ccaa","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}}]},{"traceId":"05d68b91f746bc97","id":"bcfc5bad6946f87c","name":"analyticshelper.execute.getsubscribedchannel","parentId":"715a70c7752246e3","timestamp":1530215261175336,"duration":28,"binaryAnnotations":[{"key":"channelId","value":"DgzNrMEAEAE","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"source","value":"signalboost-rest-lab13--ccaa","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}}]},{"traceId":"05d68b91f746bc97","id":"bfe4e6103bec9ab1","name":"analyticshelper.execute.getstreamcontroller","parentId":"715a70c7752246e3","timestamp":1530215261184728,"duration":2508,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"source","value":"signalboost-rest-lab13--ccaa","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}}]},{"traceId":"05d68b91f746bc97","id":"c384b9688b4c925f","name":"execute","parentId":"715a70c7752246e3","timestamp":1530215261268388,"duration":21862,"annotations":[{"timestamp":1530215261268388,"value":"cs","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"timestamp":1530215261290250,"value":"cr","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"message.name","value":"execute","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"message.seqid","value":"1","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}},{"key":"source","value":"signalboost-rest-lab13--ccaa","endpoint":{"serviceName":"signalboost","ipv4":"10.2.6.231"}}]},{"traceId":"05d68b91f746bc97","id":"24335b115ea7632b","name":"execute","parentId":"c384b9688b4c925f","timestamp":1530215261269054,"duration":21119,"annotations":[{"timestamp":1530215261269054,"value":"sr","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}},{"timestamp":1530215261290173,"value":"ss","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}},{"key":"message.name","value":"execute","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}},{"key":"message.seqid","value":"1","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}},{"key":"source","value":"analytics28","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}}]},{"traceId":"05d68b91f746bc97","id":"aee27bf36ca25a66","name":"_execute","parentId":"24335b115ea7632b","timestamp":1530215261269960,"duration":20061,"annotations":[{"timestamp":1530215261269960,"value":"cs","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}},{"timestamp":1530215261290021,"value":"cr","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}},{"key":"message.name","value":"_execute","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}},{"key":"message.seqid","value":"359","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}},{"key":"source","value":"analytics28","endpoint":{"serviceName":"analytics","ipv4":"10.2.8.193"}}]},{"traceId":"05d68b91f746bc97","id":"501fb91e840b9de3","name":"_execute","parentId":"aee27bf36ca25a66","timestamp":1530215261270037,"duration":19395,"annotations":[{"timestamp":1530215261270037,"value":"sr","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"timestamp":1530215261289432,"value":"ss","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.name","value":"_execute","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.seqid","value":"359","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"3e0b07efd15a3e8d","name":"analytics.job.start","parentId":"501fb91e840b9de3","timestamp":1530215261270948,"duration":523683,"binaryAnnotations":[{"key":"jobId","value":"DgzNrMEAEAE","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"orgId","value":"CMtLSYXAEJQ","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"state","value":"RUNNING","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"userId","value":"CxWkjSnAAAE","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"3eca78f81d05e584","name":"analytics.jobfactory.create","parentId":"3e0b07efd15a3e8d","timestamp":1530215261270972,"duration":4069,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"457b54d92d47fe0d","name":"analytics.estimation.resolution","parentId":"3e0b07efd15a3e8d","timestamp":1530215261287024,"duration":205184,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"6d8c9b31473b8c4f","name":"analytics.estimation.maxdelay","parentId":"3e0b07efd15a3e8d","timestamp":1530215261289378,"duration":181824,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"92c9c07be55a60cf","name":"analytics.job.prepare","parentId":"3e0b07efd15a3e8d","timestamp":1530215261492406,"duration":1146,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"0731d3882b17e4fc","name":"analytics.job.scheduler.create","parentId":"3e0b07efd15a3e8d","timestamp":1530215261493599,"duration":71,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"13d9c22f068af536","name":"analytics.job.jmxexport","parentId":"3e0b07efd15a3e8d","timestamp":1530215261493702,"duration":289380,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"b784c25a2b2820f5","name":"sampleddataresolutionlimiter","parentId":"457b54d92d47fe0d","timestamp":1530215261350393,"duration":136615,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"2fb6a673aa291468","name":"mtscountresolutionlimiter","parentId":"457b54d92d47fe0d","timestamp":1530215261487093,"annotations":[{"timestamp":1530215263269792,"value":"brave.flush","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}],"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"1944dc8cb9bfc25e","name":"blocklatencyresolutionlimiter","parentId":"457b54d92d47fe0d","timestamp":1530215261491502,"duration":127,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"ae5642ded7a9ade4","name":"resolutionmultipleresolutionlimiter","parentId":"457b54d92d47fe0d","timestamp":1530215261491718,"duration":78,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"c553e8be1775cf74","name":"timeshiftoffsetresolutionlimiter","parentId":"457b54d92d47fe0d","timestamp":1530215261491852,"duration":47,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"db6a32f8a24ce198","name":"lag-estimation.pipeline","parentId":"6d8c9b31473b8c4f","timestamp":1530215261362476,"duration":108445,"binaryAnnotations":[{"key":"fetch","value":"00-FETCH","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"find","value":"00-FIND","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"daa9f7a9c7d3a930","name":"resolution-estimation.pipeline","parentId":"b784c25a2b2820f5","timestamp":1530215261361553,"duration":125290,"binaryAnnotations":[{"key":"dataStartTimeMs","value":"1530215260000","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"fetch","value":"00-FETCH","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"find","value":"00-FIND","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"latencyMs","value":"3599000","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"b54b878e0b125c8e","name":"lag-estimation.pipeline.metadata","parentId":"db6a32f8a24ce198","timestamp":1530215261362487,"duration":19229,"binaryAnnotations":[{"key":"num.timeseries","value":"11","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"0fc3b0ce19caf080","name":"lag-estimation.pipeline.tsdb","parentId":"db6a32f8a24ce198","timestamp":1530215261381758,"duration":31906,"binaryAnnotations":[{"key":"num.timeseries","value":"11","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"083e313b7999eeba","name":"resolution-estimation.pipeline.metadata","parentId":"daa9f7a9c7d3a930","timestamp":1530215261361564,"duration":20130,"binaryAnnotations":[{"key":"num.results","value":"11","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"6fc0afad16b79a23","name":"tsdb.async.query","parentId":"daa9f7a9c7d3a930","timestamp":1530215261381785,"duration":6473,"binaryAnnotations":[{"key":"num.timeseries","value":"11","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"resolution","value":"3600000","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"timerange.end","value":"1530212400000","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"3770a1c54428b008","name":"tsdb.async.query","parentId":"daa9f7a9c7d3a930","timestamp":1530215261390271,"duration":60875,"binaryAnnotations":[{"key":"num.timeseries","value":"11","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"resolution","value":"1000","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"4a3f495568245d73","name":"tsdb.async.query","parentId":"0fc3b0ce19caf080","timestamp":1530215261381772,"duration":31831,"binaryAnnotations":[{"key":"num.timeseries","value":"11","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"resolution","value":"1000","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"timerange.end","value":"1530215261382","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"timerange.start","value":"1530211661382","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"4ccce5bc9a4c7e67","name":"queryemittingmtswithcache","parentId":"083e313b7999eeba","timestamp":1530215261361640,"duration":8050,"binaryAnnotations":[{"key":"cache.hit","value":"true","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"db.statement","value":"QueryClauseQuery [query=QueryStringQuery [queryString=sf_metric:/sftest\\.chart\\.load\\.4.*/, lowercaseExpandedTerms=false, boost=-1.0], clause=AndClause [clauses=[TermClause [name=sf_organizationID, value=CMtLSYXAEJQ, cache=null], TermClause [name=sf_type, value=MetricTimeSeries, cache=null], OrClause [clauses=[TermClause [name=sf_isActive, value=true, cache=null], AndClause [clauses=[TermClause [name=sf_isActive, value=false, cache=null], RangeClause [name=sf_lastActiveMs, from=1530201600000, to=null, includeLower=true, includeUpper=true, cache=false]], cache=null]], cache=null], NotClause [clause=OrClause [clauses=[ExistsClause [name=sf_detectorDerived], ExistsClause [name=sf_uiAnalyticsDerivedMetric]], cache=null], cache=null], OrClause [clauses=[TermClause [name=sf_backfilledBeforeCreation, value=true, cache=null], RangeClause [name=sf_createdOnMs, from=null, to=9191836015574400000, includeLower=true, includeUpper=true, cache=false]], cache=null]], cache=null]]","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"num.results","value":"11","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"orgId","value":"CMtLSYXAEJQ","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"timerange.end","value":"9191836015574400000","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"timerange.start","value":"1530201600000","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"f64bafb19ca6bc09","name":"registerpercolation","parentId":"083e313b7999eeba","timestamp":1530215261369965,"duration":11285,"annotations":[{"timestamp":1530215261369965,"value":"cs","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"timestamp":1530215261381250,"value":"cr","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.name","value":"registerPercolation","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.seqid","value":"2","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"962797291615ff09","name":"gettimeseriesdata","parentId":"6fc0afad16b79a23","timestamp":1530215261381876,"duration":5930,"annotations":[{"timestamp":1530215261381876,"value":"cs","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"timestamp":1530215261387806,"value":"cr","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"dc90d76e24718e48","name":"gettimeseriesdata","parentId":"6fc0afad16b79a23","timestamp":1530215261381930,"duration":5780,"annotations":[{"timestamp":1530215261381930,"value":"cs","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"timestamp":1530215261387710,"value":"cr","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"1302ff1c6aae9914","name":"gettimeseriesdata","parentId":"6fc0afad16b79a23","timestamp":1530215261381979,"duration":5908,"annotations":[{"timestamp":1530215261381979,"value":"cs","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"timestamp":1530215261387887,"value":"cr","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"a8d8d514f09dcb3a","name":"gettimeseriesdata","parentId":"6fc0afad16b79a23","timestamp":1530215261382041,"duration":6268,"annotations":[{"timestamp":1530215261382041,"value":"cs","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"timestamp":1530215261388309,"value":"cr","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"b2b043e08ecb92fe","name":"gettimeseriesdata","parentId":"6fc0afad16b79a23","timestamp":1530215261382089,"duration":5942,"annotations":[{"timestamp":1530215261382089,"value":"cs","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"timestamp":1530215261388031,"value":"cr","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"60a1f48242a91e82","name":"gettimeseriesdata","parentId":"3770a1c54428b008","timestamp":1530215261390436,"duration":31279,"annotations":[{"timestamp":1530215261390436,"value":"cs","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"timestamp":1530215261421715,"value":"cr","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"d965a236ac1864f1","name":"gettimeseriesdata","parentId":"3770a1c54428b008","timestamp":1530215261390534,"duration":24572,"annotations":[{"timestamp":1530215261390534,"value":"cs","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"timestamp":1530215261415106,"value":"cr","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"794afb9aa95ea6e4","name":"gettimeseriesdata","parentId":"3770a1c54428b008","timestamp":1530215261390626,"duration":60581,"annotations":[{"timestamp":1530215261390626,"value":"cs","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"timestamp":1530215261451207,"value":"cr","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"84979baf76faa1dd","name":"gettimeseriesdata","parentId":"3770a1c54428b008","timestamp":1530215261390678,"duration":19177,"annotations":[{"timestamp":1530215261390678,"value":"cs","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"timestamp":1530215261409855,"value":"cr","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"3380fe810ec23ea2","name":"gettimeseriesdata","parentId":"3770a1c54428b008","timestamp":1530215261390791,"duration":39894,"annotations":[{"timestamp":1530215261390791,"value":"cs","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"timestamp":1530215261430685,"value":"cr","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"ba55cfd7c3dab7d3","name":"gettimeseriesdata","parentId":"4a3f495568245d73","timestamp":1530215261381944,"duration":11169,"annotations":[{"timestamp":1530215261381944,"value":"cs","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"timestamp":1530215261393113,"value":"cr","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"b587e8355d5efdd6","name":"gettimeseriesdata","parentId":"4a3f495568245d73","timestamp":1530215261381987,"duration":10011,"annotations":[{"timestamp":1530215261381987,"value":"cs","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"timestamp":1530215261391998,"value":"cr","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"2a44cb58e7efcd52","name":"gettimeseriesdata","parentId":"4a3f495568245d73","timestamp":1530215261382026,"duration":31638,"annotations":[{"timestamp":1530215261382026,"value":"cs","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"timestamp":1530215261413664,"value":"cr","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"a9133c9c11ec4edd","name":"gettimeseriesdata","parentId":"4a3f495568245d73","timestamp":1530215261382057,"duration":4140,"annotations":[{"timestamp":1530215261382057,"value":"cs","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"timestamp":1530215261386197,"value":"cr","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"a4e2ce576ad9a613","name":"gettimeseriesdata","parentId":"4a3f495568245d73","timestamp":1530215261382091,"duration":9785,"annotations":[{"timestamp":1530215261382091,"value":"cs","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"timestamp":1530215261391876,"value":"cr","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"706001f7f1e00191","name":"getresult","parentId":"4ccce5bc9a4c7e67","timestamp":1530215261361705,"duration":7899,"binaryAnnotations":[{"key":"db.type","value":"metabase","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"num.results","value":"11","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}},{"key":"source","value":"analytics30","endpoint":{"serviceName":"analytics","ipv4":"10.2.4.112"}}]},{"traceId":"05d68b91f746bc97","id":"20210ae8cc44825f","name":"registerpercolation","parentId":"f64bafb19ca6bc09","timestamp":1530215261372018,"duration":9458,"annotations":[{"timestamp":1530215261372018,"value":"sr","endpoint":{"serviceName":"matt","ipv4":"10.2.15.176"}},{"timestamp":1530215261381476,"value":"ss","endpoint":{"serviceName":"matt","ipv4":"10.2.15.176"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"matt","ipv4":"10.2.15.176"}},{"key":"message.name","value":"registerPercolation","endpoint":{"serviceName":"matt","ipv4":"10.2.15.176"}},{"key":"message.seqid","value":"2","endpoint":{"serviceName":"matt","ipv4":"10.2.15.176"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"matt","ipv4":"10.2.15.176"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"matt","ipv4":"10.2.15.176"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"matt","ipv4":"10.2.15.176"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"matt","ipv4":"10.2.15.176"}},{"key":"source","value":"matt-1","endpoint":{"serviceName":"matt","ipv4":"10.2.15.176"}}]},{"traceId":"05d68b91f746bc97","id":"a6a9783bb6bc22fe","name":"gettimeseriesdata","parentId":"962797291615ff09","timestamp":1530215261382010,"duration":4572,"annotations":[{"timestamp":1530215261382010,"value":"sr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"timestamp":1530215261386582,"value":"ss","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"returnedBlocks","value":"12","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"d78645d399992b6f","name":"gettimeseriesdata","parentId":"dc90d76e24718e48","timestamp":1530215261383013,"duration":4391,"annotations":[{"timestamp":1530215261383013,"value":"sr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"timestamp":1530215261387404,"value":"ss","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"returnedBlocks","value":"6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"69ac059028b20fe2","name":"gettimeseriesdata","parentId":"1302ff1c6aae9914","timestamp":1530215261383011,"duration":4145,"annotations":[{"timestamp":1530215261383011,"value":"sr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"timestamp":1530215261387156,"value":"ss","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"returnedBlocks","value":"6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"b20eeb05c3d7f1e5","name":"gettimeseriesdata","parentId":"a8d8d514f09dcb3a","timestamp":1530215261383007,"duration":4080,"annotations":[{"timestamp":1530215261383007,"value":"sr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"timestamp":1530215261387087,"value":"ss","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"returnedBlocks","value":"3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"source","value":"agamemnon-7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}]},{"traceId":"05d68b91f746bc97","id":"2b86020bb44eb44f","name":"gettimeseriesdata","parentId":"b2b043e08ecb92fe","timestamp":1530215261383004,"duration":4030,"annotations":[{"timestamp":1530215261383004,"value":"sr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"timestamp":1530215261387034,"value":"ss","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"returnedBlocks","value":"6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"b80a9416378da990","name":"gettimeseriesdata","parentId":"60a1f48242a91e82","timestamp":1530215261392005,"duration":29632,"annotations":[{"timestamp":1530215261392005,"value":"sr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"timestamp":1530215261421637,"value":"ss","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"returnedBlocks","value":"124","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"c05c02f989be7738","name":"gettimeseriesdata","parentId":"d965a236ac1864f1","timestamp":1530215261392007,"duration":21016,"annotations":[{"timestamp":1530215261392007,"value":"sr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"timestamp":1530215261413023,"value":"ss","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"returnedBlocks","value":"62","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"ae0d2ae4ed2148fa","name":"gettimeseriesdata","parentId":"794afb9aa95ea6e4","timestamp":1530215261438012,"duration":12157,"annotations":[{"timestamp":1530215261438012,"value":"sr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"timestamp":1530215261450169,"value":"ss","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"returnedBlocks","value":"62","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"932a831816389b90","name":"gettimeseriesdata","parentId":"84979baf76faa1dd","timestamp":1530215261391006,"duration":17851,"annotations":[{"timestamp":1530215261391006,"value":"sr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"timestamp":1530215261408857,"value":"ss","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"returnedBlocks","value":"31","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"source","value":"agamemnon-7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}]},{"traceId":"05d68b91f746bc97","id":"69fb3e454c1c00a9","name":"gettimeseriesdata","parentId":"3380fe810ec23ea2","timestamp":1530215261392008,"duration":36248,"annotations":[{"timestamp":1530215261392008,"value":"sr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"timestamp":1530215261428256,"value":"ss","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"returnedBlocks","value":"62","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"effa43d65a0a6f7e","name":"gettimeseriesdata","parentId":"ba55cfd7c3dab7d3","timestamp":1530215261383008,"duration":9834,"annotations":[{"timestamp":1530215261383008,"value":"sr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"timestamp":1530215261392842,"value":"ss","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"returnedBlocks","value":"124","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"2727b346135ea30c","name":"gettimeseriesdata","parentId":"b587e8355d5efdd6","timestamp":1530215261383005,"duration":7430,"annotations":[{"timestamp":1530215261383005,"value":"sr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"timestamp":1530215261390435,"value":"ss","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"returnedBlocks","value":"62","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"708953cf50cac8b6","name":"gettimeseriesdata","parentId":"2a44cb58e7efcd52","timestamp":1530215261383012,"duration":30106,"annotations":[{"timestamp":1530215261383012,"value":"sr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"timestamp":1530215261413118,"value":"ss","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"returnedBlocks","value":"62","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"0adb245af2aa7a8a","name":"gettimeseriesdata","parentId":"a9133c9c11ec4edd","timestamp":1530215261383011,"duration":3289,"annotations":[{"timestamp":1530215261383011,"value":"sr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"timestamp":1530215261386300,"value":"ss","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"returnedBlocks","value":"31","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"source","value":"agamemnon-7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}]},{"traceId":"05d68b91f746bc97","id":"d8443f295dadef13","name":"gettimeseriesdata","parentId":"a4e2ce576ad9a613","timestamp":1530215261383011,"duration":7751,"annotations":[{"timestamp":1530215261383011,"value":"sr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"timestamp":1530215261390762,"value":"ss","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.name","value":"getTimeSeriesData","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.seqid","value":"0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"returnedBlocks","value":"62","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"05aba2076b8f8a02","name":"engine.read","parentId":"a6a9783bb6bc22fe","timestamp":1530215261382023,"duration":4296,"binaryAnnotations":[{"key":"num.timeseries","value":"4","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"resolution","value":"3600000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530212400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"38ce85ba85c6e7bf","name":"encode","parentId":"a6a9783bb6bc22fe","timestamp":1530215261386345,"duration":155,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"adf62bda750c3a95","name":"engine.read","parentId":"d78645d399992b6f","timestamp":1530215261383037,"duration":4043,"binaryAnnotations":[{"key":"num.timeseries","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"resolution","value":"3600000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.end","value":"1530212400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"24135016f7fe5d3f","name":"encode","parentId":"d78645d399992b6f","timestamp":1530215261387123,"duration":146,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"fa72203d812d410f","name":"engine.read","parentId":"69ac059028b20fe2","timestamp":1530215261383031,"duration":3858,"binaryAnnotations":[{"key":"num.timeseries","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"resolution","value":"3600000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.end","value":"1530212400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"fc617f968b3ecd8f","name":"encode","parentId":"69ac059028b20fe2","timestamp":1530215261386918,"duration":129,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"65964c3d6e4e4053","name":"engine.read","parentId":"b20eeb05c3d7f1e5","timestamp":1530215261383023,"duration":3898,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"resolution","value":"3600000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"source","value":"agamemnon-7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"timerange.end","value":"1530212400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}]},{"traceId":"05d68b91f746bc97","id":"2955c84d7a6c8b87","name":"encode","parentId":"b20eeb05c3d7f1e5","timestamp":1530215261386943,"duration":60,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"source","value":"agamemnon-7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}]},{"traceId":"05d68b91f746bc97","id":"e6eda942696c77bf","name":"engine.read","parentId":"2b86020bb44eb44f","timestamp":1530215261383017,"duration":3824,"binaryAnnotations":[{"key":"num.timeseries","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"resolution","value":"3600000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.end","value":"1530212400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"d7e6975635db9578","name":"encode","parentId":"2b86020bb44eb44f","timestamp":1530215261386868,"duration":79,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"f4a8b428e6133da2","name":"engine.read","parentId":"b80a9416378da990","timestamp":1530215261392015,"duration":25586,"binaryAnnotations":[{"key":"num.timeseries","value":"4","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"resolution","value":"1000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"7776767ceb70f08a","name":"encode","parentId":"b80a9416378da990","timestamp":1530215261417623,"duration":2938,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"2ac3d45d382edd98","name":"engine.read","parentId":"c05c02f989be7738","timestamp":1530215261392020,"duration":17466,"binaryAnnotations":[{"key":"num.timeseries","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"resolution","value":"1000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"6afe008a7f644813","name":"encode","parentId":"c05c02f989be7738","timestamp":1530215261409525,"duration":2577,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"b3da50000fb6288f","name":"engine.read","parentId":"ae0d2ae4ed2148fa","timestamp":1530215261438032,"duration":4037,"binaryAnnotations":[{"key":"num.timeseries","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"resolution","value":"1000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"114acab30410702f","name":"encode","parentId":"ae0d2ae4ed2148fa","timestamp":1530215261442090,"duration":3443,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"7be3b11e485c09c4","name":"engine.read","parentId":"932a831816389b90","timestamp":1530215261391044,"duration":16810,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"resolution","value":"1000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"source","value":"agamemnon-7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}]},{"traceId":"05d68b91f746bc97","id":"7201b96f2fc658e3","name":"encode","parentId":"932a831816389b90","timestamp":1530215261407881,"duration":826,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"source","value":"agamemnon-7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}]},{"traceId":"05d68b91f746bc97","id":"a75cc6b26de02da1","name":"engine.read","parentId":"69fb3e454c1c00a9","timestamp":1530215261392019,"duration":33216,"binaryAnnotations":[{"key":"num.timeseries","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"resolution","value":"1000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"fa418ac900bdeac8","name":"encode","parentId":"69fb3e454c1c00a9","timestamp":1530215261425261,"duration":2146,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"4ac138c6bc12eb36","name":"engine.read","parentId":"effa43d65a0a6f7e","timestamp":1530215261383019,"duration":5239,"binaryAnnotations":[{"key":"num.timeseries","value":"4","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"resolution","value":"1000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"b76187db0e532ea6","name":"encode","parentId":"effa43d65a0a6f7e","timestamp":1530215261388277,"duration":3733,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"29975287ec32129b","name":"engine.read","parentId":"2727b346135ea30c","timestamp":1530215261383019,"duration":5165,"binaryAnnotations":[{"key":"num.timeseries","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"resolution","value":"1000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"28b5112d11a99cca","name":"encode","parentId":"2727b346135ea30c","timestamp":1530215261388240,"duration":1869,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"5bf3ab21b42d069d","name":"engine.read","parentId":"708953cf50cac8b6","timestamp":1530215261383033,"duration":4947,"binaryAnnotations":[{"key":"num.timeseries","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"resolution","value":"1000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"fe6af78e15ef3b6b","name":"encode","parentId":"708953cf50cac8b6","timestamp":1530215261388032,"duration":24808,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"c3080e9b6536f53c","name":"engine.read","parentId":"0adb245af2aa7a8a","timestamp":1530215261383026,"duration":2353,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"resolution","value":"1000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"source","value":"agamemnon-7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}]},{"traceId":"05d68b91f746bc97","id":"bb19d1d0318180ae","name":"encode","parentId":"0adb245af2aa7a8a","timestamp":1530215261385405,"duration":775,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"source","value":"agamemnon-7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}]},{"traceId":"05d68b91f746bc97","id":"1f7d5315f967d0a6","name":"engine.read","parentId":"d8443f295dadef13","timestamp":1530215261383044,"duration":5394,"binaryAnnotations":[{"key":"num.timeseries","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"resolution","value":"1000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"d1c2c58b20d5481a","name":"encode","parentId":"d8443f295dadef13","timestamp":1530215261388463,"duration":2044,"binaryAnnotations":[{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"921d1d165bc42b57","name":"memorytiersegment.read","parentId":"05aba2076b8f8a02","timestamp":1530215261382111,"duration":2567,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530212400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"2082b1b77e9d234b","name":"memorytiersegment.read","parentId":"05aba2076b8f8a02","timestamp":1530215261382134,"duration":1316,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530212400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"074872a7abb1cc1c","name":"memorytiersegment.read","parentId":"05aba2076b8f8a02","timestamp":1530215261382157,"duration":4119,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530212400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"cb78814fe1803483","name":"memorytiersegment.read","parentId":"05aba2076b8f8a02","timestamp":1530215261382177,"duration":864,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530212400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"6dd0ec401c8b31e8","name":"memorytiersegment.read","parentId":"adf62bda750c3a95","timestamp":1530215261383165,"duration":3858,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.end","value":"1530212400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"7ed120062539be5f","name":"memorytiersegment.read","parentId":"adf62bda750c3a95","timestamp":1530215261383183,"duration":3466,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.end","value":"1530212400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"5109dfce33114006","name":"memorytiersegment.read","parentId":"fa72203d812d410f","timestamp":1530215261383143,"duration":3700,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.end","value":"1530212400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"e414b1bd757a95db","name":"memorytiersegment.read","parentId":"fa72203d812d410f","timestamp":1530215261383253,"duration":3149,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.end","value":"1530212400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"a6e41c23ff6a7037","name":"memorytiersegment.read","parentId":"65964c3d6e4e4053","timestamp":1530215261383094,"duration":3787,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"source","value":"agamemnon-7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"timerange.end","value":"1530212400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}]},{"traceId":"05d68b91f746bc97","id":"8e541d299f6a6702","name":"memorytiersegment.read","parentId":"e6eda942696c77bf","timestamp":1530215261383112,"duration":3675,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.end","value":"1530212400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"303285d14ab5d529","name":"memorytiersegment.read","parentId":"e6eda942696c77bf","timestamp":1530215261383141,"duration":2713,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.end","value":"1530212400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"79f3385db373b8ad","name":"memorytiersegment.read","parentId":"f4a8b428e6133da2","timestamp":1530215261392157,"duration":23237,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"c97920ff9dab3b9d","name":"memorytiersegment.read","parentId":"f4a8b428e6133da2","timestamp":1530215261392185,"duration":25364,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"173ae2918635814a","name":"memorytiersegment.read","parentId":"f4a8b428e6133da2","timestamp":1530215261392274,"duration":24083,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"adf9d9e162227bc4","name":"memorytiersegment.read","parentId":"f4a8b428e6133da2","timestamp":1530215261392285,"duration":24553,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"b017a811d919386f","name":"memorytiersegment.read","parentId":"2ac3d45d382edd98","timestamp":1530215261392104,"duration":17323,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"1973ea1262488e90","name":"memorytiersegment.read","parentId":"2ac3d45d382edd98","timestamp":1530215261392118,"duration":16415,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"28a9905b16136a6f","name":"memorytiersegment.read","parentId":"b3da50000fb6288f","timestamp":1530215261438133,"duration":3793,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"e5aa4e5d603740a8","name":"memorytiersegment.read","parentId":"b3da50000fb6288f","timestamp":1530215261438568,"duration":3324,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"788a19fcd687a29a","name":"memorytiersegment.read","parentId":"7be3b11e485c09c4","timestamp":1530215261391106,"duration":16700,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"source","value":"agamemnon-7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}]},{"traceId":"05d68b91f746bc97","id":"332d5bcec3f539fa","name":"memorytiersegment.read","parentId":"a75cc6b26de02da1","timestamp":1530215261392086,"duration":32566,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"8ba74c38f5158861","name":"memorytiersegment.read","parentId":"a75cc6b26de02da1","timestamp":1530215261392104,"duration":33070,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"4fc9156864cc0304","name":"memorytiersegment.read","parentId":"4ac138c6bc12eb36","timestamp":1530215261383093,"duration":5099,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"dac90982a7e490ae","name":"memorytiersegment.read","parentId":"4ac138c6bc12eb36","timestamp":1530215261383120,"duration":4413,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"fc11e35565f00cfa","name":"memorytiersegment.read","parentId":"4ac138c6bc12eb36","timestamp":1530215261383154,"duration":5010,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"0f9267cb15ebfcd2","name":"memorytiersegment.read","parentId":"4ac138c6bc12eb36","timestamp":1530215261383314,"duration":2018,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"debfffe6fdeca0ed","name":"memorytiersegment.read","parentId":"29975287ec32129b","timestamp":1530215261383119,"duration":5010,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"82e58d9c7b5b1983","name":"memorytiersegment.read","parentId":"29975287ec32129b","timestamp":1530215261383143,"duration":4034,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"e6f784b6c2d20c4d","name":"memorytiersegment.read","parentId":"5bf3ab21b42d069d","timestamp":1530215261383163,"duration":3634,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"8fc73442f7775015","name":"memorytiersegment.read","parentId":"5bf3ab21b42d069d","timestamp":1530215261383191,"duration":4706,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"583b13080be36b7c","name":"memorytiersegment.read","parentId":"c3080e9b6536f53c","timestamp":1530215261383102,"duration":2230,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"source","value":"agamemnon-7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}]},{"traceId":"05d68b91f746bc97","id":"983cfe8e81675bf4","name":"memorytiersegment.read","parentId":"1f7d5315f967d0a6","timestamp":1530215261383134,"duration":4963,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"aa0eb9f4cc829400","name":"memorytiersegment.read","parentId":"1f7d5315f967d0a6","timestamp":1530215261383213,"duration":5156,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.end","value":"1530215261000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"c652e594cd1db2c0","name":"memorytier.get","parentId":"921d1d165bc42b57","timestamp":1530215261382185,"duration":2455,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530230400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"7834667e45dce002","name":"memorytier.get","parentId":"2082b1b77e9d234b","timestamp":1530215261382220,"duration":1215,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530230400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"d42549922a6d1fec","name":"memorytier.get","parentId":"074872a7abb1cc1c","timestamp":1530215261382260,"duration":3991,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530230400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"4958ad169a986b8b","name":"memorytier.get","parentId":"cb78814fe1803483","timestamp":1530215261382278,"duration":742,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530230400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"653706da0221016d","name":"memorytier.get","parentId":"6dd0ec401c8b31e8","timestamp":1530215261383255,"duration":3756,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.end","value":"1530230400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"f612cc35cd2c0622","name":"memorytier.get","parentId":"7ed120062539be5f","timestamp":1530215261383276,"duration":3360,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.end","value":"1530230400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"a3511994e5a7a6ce","name":"memorytier.get","parentId":"5109dfce33114006","timestamp":1530215261383214,"duration":3009,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.end","value":"1530230400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"3b87e9aa5a889bfc","name":"memorytier.get","parentId":"e414b1bd757a95db","timestamp":1530215261383492,"duration":2731,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.end","value":"1530230400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"88213a1f65eb8804","name":"memorytier.get","parentId":"a6e41c23ff6a7037","timestamp":1530215261383148,"duration":3715,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"source","value":"agamemnon-7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"timerange.end","value":"1530230400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}]},{"traceId":"05d68b91f746bc97","id":"6c73a3ad0b31e01e","name":"memorytier.get","parentId":"8e541d299f6a6702","timestamp":1530215261383185,"duration":3561,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.end","value":"1530230400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"02b81444a42c5031","name":"memorytier.get","parentId":"303285d14ab5d529","timestamp":1530215261383216,"duration":2597,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.end","value":"1530230400000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.start","value":"1530126000000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"90cde3509e2d10ed","name":"memorytier.get","parentId":"79f3385db373b8ad","timestamp":1530215261392220,"duration":22395,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"ff2f63fe87a23046","name":"memorytier.get","parentId":"c97920ff9dab3b9d","timestamp":1530215261392272,"duration":24810,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"707db38e61e8cc3b","name":"memorytier.get","parentId":"173ae2918635814a","timestamp":1530215261392349,"duration":23523,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"2ada108e0c197800","name":"memorytier.get","parentId":"adf9d9e162227bc4","timestamp":1530215261392358,"duration":24038,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"b26cde1634a3056f","name":"memorytier.get","parentId":"b017a811d919386f","timestamp":1530215261392169,"duration":16749,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"ba857d9b6b45371a","name":"memorytier.get","parentId":"1973ea1262488e90","timestamp":1530215261392187,"duration":15848,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"c9fc49076f6f4160","name":"memorytier.get","parentId":"28a9905b16136a6f","timestamp":1530215261438260,"duration":1873,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"042adab6ea885687","name":"memorytier.get","parentId":"e5aa4e5d603740a8","timestamp":1530215261439447,"duration":1570,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"1190630343ba1b86","name":"memorytier.get","parentId":"788a19fcd687a29a","timestamp":1530215261391156,"duration":16183,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"source","value":"agamemnon-7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}]},{"traceId":"05d68b91f746bc97","id":"3d3edca6188039e8","name":"memorytier.get","parentId":"332d5bcec3f539fa","timestamp":1530215261392153,"duration":31815,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"858781abb823835d","name":"memorytier.get","parentId":"8ba74c38f5158861","timestamp":1530215261392169,"duration":31998,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"2701e268222bc247","name":"memorytier.get","parentId":"4fc9156864cc0304","timestamp":1530215261383180,"duration":4565,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"93c4d49442eb0d53","name":"memorytier.get","parentId":"dac90982a7e490ae","timestamp":1530215261383187,"duration":3887,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"a05272cb7d8a6306","name":"memorytier.get","parentId":"fc11e35565f00cfa","timestamp":1530215261383222,"duration":4486,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"4e30faa67aefbe3f","name":"memorytier.get","parentId":"0f9267cb15ebfcd2","timestamp":1530215261383382,"duration":1491,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"7022bd5157722994","name":"memorytier.get","parentId":"debfffe6fdeca0ed","timestamp":1530215261383226,"duration":4411,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"7a6b2d2f2b6749ad","name":"memorytier.get","parentId":"82e58d9c7b5b1983","timestamp":1530215261383255,"duration":3364,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"92e46035465dce14","name":"memorytier.get","parentId":"e6f784b6c2d20c4d","timestamp":1530215261383430,"duration":2810,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"32ec6bbc60a109f5","name":"memorytier.get","parentId":"8fc73442f7775015","timestamp":1530215261383472,"duration":2909,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"6be52e9444b274f6","name":"memorytier.get","parentId":"583b13080be36b7c","timestamp":1530215261383163,"duration":1711,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"source","value":"agamemnon-7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}]},{"traceId":"05d68b91f746bc97","id":"fbf3f30a748b6afb","name":"memorytier.get","parentId":"983cfe8e81675bf4","timestamp":1530215261383197,"duration":4258,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"a63e9f417dd1d8c5","name":"memorytier.get","parentId":"aa0eb9f4cc829400","timestamp":1530215261383282,"duration":4420,"binaryAnnotations":[{"key":"num.timeseries","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.end","value":"1530215280000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"timerange.start","value":"1530211661000","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"4ed122bc24e21321","name":"gettimeseriesblocks","parentId":"c652e594cd1db2c0","timestamp":1530215261382249,"duration":2336,"annotations":[{"timestamp":1530215261382249,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"timestamp":1530215261384585,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.seqid","value":"4","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"b79ab7d2990389b0","name":"gettimeseriesblocks","parentId":"7834667e45dce002","timestamp":1530215261382266,"duration":1114,"annotations":[{"timestamp":1530215261382266,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"timestamp":1530215261383380,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.seqid","value":"3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"2d36c6a2cf6a0dfb","name":"gettimeseriesblocks","parentId":"d42549922a6d1fec","timestamp":1530215261382326,"duration":3880,"annotations":[{"timestamp":1530215261382326,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"timestamp":1530215261386206,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.seqid","value":"3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"1ca452338a71794f","name":"gettimeseriesblocks","parentId":"4958ad169a986b8b","timestamp":1530215261382342,"duration":614,"annotations":[{"timestamp":1530215261382342,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"timestamp":1530215261382956,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.seqid","value":"4","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"6b24373d9cba475f","name":"gettimeseriesblocks","parentId":"653706da0221016d","timestamp":1530215261383421,"duration":3487,"annotations":[{"timestamp":1530215261383421,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"timestamp":1530215261386908,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.seqid","value":"6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"b5691b00e918ab29","name":"gettimeseriesblocks","parentId":"f612cc35cd2c0622","timestamp":1530215261383377,"duration":3165,"annotations":[{"timestamp":1530215261383377,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"timestamp":1530215261386542,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.seqid","value":"9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"ad098656bde9360c","name":"gettimeseriesblocks","parentId":"a3511994e5a7a6ce","timestamp":1530215261383319,"duration":2834,"annotations":[{"timestamp":1530215261383319,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"timestamp":1530215261386153,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.seqid","value":"7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"928ae9b40be64446","name":"gettimeseriesblocks","parentId":"3b87e9aa5a889bfc","timestamp":1530215261383558,"duration":2595,"annotations":[{"timestamp":1530215261383558,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"timestamp":1530215261386153,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.seqid","value":"5","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"71e9378c4deb6ab7","name":"gettimeseriesblocks","parentId":"88213a1f65eb8804","timestamp":1530215261383197,"duration":3599,"annotations":[{"timestamp":1530215261383197,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"timestamp":1530215261386796,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"message.seqid","value":"6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"source","value":"agamemnon-7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}]},{"traceId":"05d68b91f746bc97","id":"e0405f1bcb939098","name":"gettimeseriesblocks","parentId":"6c73a3ad0b31e01e","timestamp":1530215261383229,"duration":3461,"annotations":[{"timestamp":1530215261383229,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"timestamp":1530215261386690,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.seqid","value":"8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"42fc787245d727bb","name":"gettimeseriesblocks","parentId":"02b81444a42c5031","timestamp":1530215261383260,"duration":2486,"annotations":[{"timestamp":1530215261383260,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"timestamp":1530215261385746,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.seqid","value":"4","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"77cb3b4bda6135c3","name":"gettimeseriesblocks","parentId":"90cde3509e2d10ed","timestamp":1530215261408554,"duration":6006,"annotations":[{"timestamp":1530215261408554,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"timestamp":1530215261414560,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.seqid","value":"6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"9fea56f2fff34f84","name":"gettimeseriesblocks","parentId":"ff2f63fe87a23046","timestamp":1530215261410126,"duration":6903,"annotations":[{"timestamp":1530215261410126,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"timestamp":1530215261417029,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.seqid","value":"5","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"5b0be2f756fffb3f","name":"gettimeseriesblocks","parentId":"707db38e61e8cc3b","timestamp":1530215261410724,"duration":5085,"annotations":[{"timestamp":1530215261410724,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"timestamp":1530215261415809,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.seqid","value":"6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"3ce8bcbff9e60da2","name":"gettimeseriesblocks","parentId":"2ada108e0c197800","timestamp":1530215261414930,"duration":1395,"annotations":[{"timestamp":1530215261414930,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"timestamp":1530215261416325,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.seqid","value":"7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"f9ad9bbea7a11377","name":"gettimeseriesblocks","parentId":"b26cde1634a3056f","timestamp":1530215261397531,"duration":11314,"annotations":[{"timestamp":1530215261397531,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"timestamp":1530215261408845,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.seqid","value":"7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"751808f6de635e3c","name":"gettimeseriesblocks","parentId":"ba857d9b6b45371a","timestamp":1530215261395173,"duration":12771,"annotations":[{"timestamp":1530215261395173,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"timestamp":1530215261407944,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.seqid","value":"8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"0a98e7054dba6805","name":"gettimeseriesblocks","parentId":"c9fc49076f6f4160","timestamp":1530215261438323,"duration":1481,"annotations":[{"timestamp":1530215261438323,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"timestamp":1530215261439804,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.seqid","value":"7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"b8cd43396b3f0ac9","name":"gettimeseriesblocks","parentId":"042adab6ea885687","timestamp":1530215261439815,"duration":1139,"annotations":[{"timestamp":1530215261439815,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"timestamp":1530215261440954,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.seqid","value":"8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"f99c9d1388de97da","name":"gettimeseriesblocks","parentId":"1190630343ba1b86","timestamp":1530215261391194,"duration":16084,"annotations":[{"timestamp":1530215261391194,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"timestamp":1530215261407278,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"message.seqid","value":"7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"source","value":"agamemnon-7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}]},{"traceId":"05d68b91f746bc97","id":"f99fda5cf623bfd8","name":"gettimeseriesblocks","parentId":"3d3edca6188039e8","timestamp":1530215261415019,"duration":8887,"annotations":[{"timestamp":1530215261415019,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"timestamp":1530215261423906,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.seqid","value":"4","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"c061064f0af5ddb7","name":"gettimeseriesblocks","parentId":"858781abb823835d","timestamp":1530215261419030,"duration":5084,"annotations":[{"timestamp":1530215261419030,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"timestamp":1530215261424114,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.seqid","value":"8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"b0a0aea5f6f3a217","name":"gettimeseriesblocks","parentId":"2701e268222bc247","timestamp":1530215261383232,"duration":4450,"annotations":[{"timestamp":1530215261383232,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"timestamp":1530215261387682,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.seqid","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"2e246bc456e99455","name":"gettimeseriesblocks","parentId":"93c4d49442eb0d53","timestamp":1530215261383255,"duration":3761,"annotations":[{"timestamp":1530215261383255,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"timestamp":1530215261387016,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.seqid","value":"3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"ef3a0907aea28567","name":"gettimeseriesblocks","parentId":"a05272cb7d8a6306","timestamp":1530215261383282,"duration":4359,"annotations":[{"timestamp":1530215261383282,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"timestamp":1530215261387641,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.seqid","value":"3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"370f8f9eac81f776","name":"gettimeseriesblocks","parentId":"4e30faa67aefbe3f","timestamp":1530215261383426,"duration":1394,"annotations":[{"timestamp":1530215261383426,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"timestamp":1530215261384820,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.seqid","value":"6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}},{"key":"source","value":"agamemnon-3","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.14.171"}}]},{"traceId":"05d68b91f746bc97","id":"5f82bccb81e3f730","name":"gettimeseriesblocks","parentId":"7022bd5157722994","timestamp":1530215261383292,"duration":4289,"annotations":[{"timestamp":1530215261383292,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"timestamp":1530215261387581,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.seqid","value":"7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"c10eb603850ac5df","name":"gettimeseriesblocks","parentId":"7a6b2d2f2b6749ad","timestamp":1530215261383322,"duration":3235,"annotations":[{"timestamp":1530215261383322,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"timestamp":1530215261386557,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.seqid","value":"6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}},{"key":"source","value":"agamemnon-9","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.78.243"}}]},{"traceId":"05d68b91f746bc97","id":"7c2761d1b852a16d","name":"gettimeseriesblocks","parentId":"92e46035465dce14","timestamp":1530215261383494,"duration":2696,"annotations":[{"timestamp":1530215261383494,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"timestamp":1530215261386190,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.seqid","value":"2","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"44972e986c9955d9","name":"gettimeseriesblocks","parentId":"32ec6bbc60a109f5","timestamp":1530215261383542,"duration":2753,"annotations":[{"timestamp":1530215261383542,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"timestamp":1530215261386295,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.seqid","value":"14","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}},{"key":"source","value":"agamemnon-6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.4.15"}}]},{"traceId":"05d68b91f746bc97","id":"c334e9c01fe40ec7","name":"gettimeseriesblocks","parentId":"6be52e9444b274f6","timestamp":1530215261383207,"duration":1603,"annotations":[{"timestamp":1530215261383207,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"timestamp":1530215261384810,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"message.seqid","value":"6","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}},{"key":"source","value":"agamemnon-7","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.37.17"}}]},{"traceId":"05d68b91f746bc97","id":"613a3265135852ce","name":"gettimeseriesblocks","parentId":"fbf3f30a748b6afb","timestamp":1530215261383263,"duration":4143,"annotations":[{"timestamp":1530215261383263,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"timestamp":1530215261387406,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.seqid","value":"5","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"6892bdb3b5f94f45","name":"gettimeseriesblocks","parentId":"a63e9f417dd1d8c5","timestamp":1530215261383327,"duration":4315,"annotations":[{"timestamp":1530215261383327,"value":"cs","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"timestamp":1530215261387642,"value":"cr","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.seqid","value":"4","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"message.type","value":"1","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}},{"key":"source","value":"agamemnon-8","endpoint":{"serviceName":"agamemnon","ipv4":"10.2.65.191"}}]},{"traceId":"05d68b91f746bc97","id":"5f2998f527b4cb0b","name":"gettimeseriesblocks","parentId":"4ed122bc24e21321","timestamp":1530215261383004,"duration":64,"annotations":[{"timestamp":1530215261383004,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"timestamp":1530215261383068,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.seqid","value":"4","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"source","value":"quantizer-34","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}]},{"traceId":"05d68b91f746bc97","id":"6198501ae01a0004","name":"gettimeseriesblocks","parentId":"b79ab7d2990389b0","timestamp":1530215261383009,"duration":151,"annotations":[{"timestamp":1530215261383009,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"timestamp":1530215261383160,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.seqid","value":"3","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"source","value":"quantizer-34","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}]},{"traceId":"05d68b91f746bc97","id":"214a8fd6bf2e0312","name":"gettimeseriesblocks","parentId":"2d36c6a2cf6a0dfb","timestamp":1530215261386003,"duration":84,"annotations":[{"timestamp":1530215261386003,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"timestamp":1530215261386087,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.seqid","value":"3","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"source","value":"quantizer-34","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}]},{"traceId":"05d68b91f746bc97","id":"0dc0bd16ddc102da","name":"gettimeseriesblocks","parentId":"1ca452338a71794f","timestamp":1530215261383010,"duration":91,"annotations":[{"timestamp":1530215261383010,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"timestamp":1530215261383101,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.seqid","value":"4","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"source","value":"quantizer-33","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}}]},{"traceId":"05d68b91f746bc97","id":"17db73e82e8a64dd","name":"gettimeseriesblocks","parentId":"6b24373d9cba475f","timestamp":1530215261385004,"duration":62,"annotations":[{"timestamp":1530215261385004,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"timestamp":1530215261385066,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.seqid","value":"6","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"source","value":"quantizer-32","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}]},{"traceId":"05d68b91f746bc97","id":"25fb341945af665b","name":"gettimeseriesblocks","parentId":"b5691b00e918ab29","timestamp":1530215261385003,"duration":79,"annotations":[{"timestamp":1530215261385003,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"timestamp":1530215261385082,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.seqid","value":"9","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"source","value":"quantizer-34","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}]},{"traceId":"05d68b91f746bc97","id":"17d9b026e42ed327","name":"gettimeseriesblocks","parentId":"ad098656bde9360c","timestamp":1530215261385004,"duration":64,"annotations":[{"timestamp":1530215261385004,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"timestamp":1530215261385068,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.seqid","value":"7","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"source","value":"quantizer-33","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}}]},{"traceId":"05d68b91f746bc97","id":"8d0435218e3259b0","name":"gettimeseriesblocks","parentId":"928ae9b40be64446","timestamp":1530215261384004,"duration":62,"annotations":[{"timestamp":1530215261384004,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"timestamp":1530215261384066,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.seqid","value":"5","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"source","value":"quantizer-33","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}}]},{"traceId":"05d68b91f746bc97","id":"4ed52dfd2b2df09d","name":"gettimeseriesblocks","parentId":"71e9378c4deb6ab7","timestamp":1530215261385003,"duration":63,"annotations":[{"timestamp":1530215261385003,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"timestamp":1530215261385066,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.seqid","value":"6","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"source","value":"quantizer-32","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}]},{"traceId":"05d68b91f746bc97","id":"8625fe2f701af571","name":"gettimeseriesblocks","parentId":"e0405f1bcb939098","timestamp":1530215261385004,"duration":71,"annotations":[{"timestamp":1530215261385004,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"timestamp":1530215261385075,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.seqid","value":"8","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"source","value":"quantizer-32","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}]},{"traceId":"05d68b91f746bc97","id":"c0d7886b5a705367","name":"gettimeseriesblocks","parentId":"42fc787245d727bb","timestamp":1530215261384004,"duration":66,"annotations":[{"timestamp":1530215261384004,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"timestamp":1530215261384070,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.seqid","value":"4","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"source","value":"quantizer-32","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}]},{"traceId":"05d68b91f746bc97","id":"22e457529ab040fb","name":"gettimeseriesblocks","parentId":"77cb3b4bda6135c3","timestamp":1530215261411007,"duration":676,"annotations":[{"timestamp":1530215261411007,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"timestamp":1530215261411683,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.seqid","value":"6","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"source","value":"quantizer-34","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}]},{"traceId":"05d68b91f746bc97","id":"455ec0173d685d30","name":"gettimeseriesblocks","parentId":"9fea56f2fff34f84","timestamp":1530215261414004,"duration":452,"annotations":[{"timestamp":1530215261414004,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"timestamp":1530215261414456,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.seqid","value":"5","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"source","value":"quantizer-34","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}]},{"traceId":"05d68b91f746bc97","id":"e9082f8691986dd7","name":"gettimeseriesblocks","parentId":"5b0be2f756fffb3f","timestamp":1530215261413004,"duration":495,"annotations":[{"timestamp":1530215261413004,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"timestamp":1530215261413499,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.seqid","value":"6","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"source","value":"quantizer-34","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}]},{"traceId":"05d68b91f746bc97","id":"b8d58ae2007e6d41","name":"gettimeseriesblocks","parentId":"3ce8bcbff9e60da2","timestamp":1530215261415010,"duration":683,"annotations":[{"timestamp":1530215261415010,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"timestamp":1530215261415693,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.seqid","value":"7","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"source","value":"quantizer-33","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}}]},{"traceId":"05d68b91f746bc97","id":"4ace3cde71c91f8e","name":"gettimeseriesblocks","parentId":"f9ad9bbea7a11377","timestamp":1530215261402005,"duration":684,"annotations":[{"timestamp":1530215261402005,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"timestamp":1530215261402689,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.seqid","value":"7","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"source","value":"quantizer-32","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}]},{"traceId":"05d68b91f746bc97","id":"35e2b1e8caba2e3a","name":"gettimeseriesblocks","parentId":"751808f6de635e3c","timestamp":1530215261400007,"duration":629,"annotations":[{"timestamp":1530215261400007,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"timestamp":1530215261400636,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.seqid","value":"8","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"source","value":"quantizer-34","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}]},{"traceId":"05d68b91f746bc97","id":"c96f74439bc9ca03","name":"gettimeseriesblocks","parentId":"0a98e7054dba6805","timestamp":1530215261438005,"duration":596,"annotations":[{"timestamp":1530215261438005,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"timestamp":1530215261438601,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.seqid","value":"7","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"source","value":"quantizer-33","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}}]},{"traceId":"05d68b91f746bc97","id":"453f80e75b380977","name":"gettimeseriesblocks","parentId":"b8cd43396b3f0ac9","timestamp":1530215261440004,"duration":484,"annotations":[{"timestamp":1530215261440004,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"timestamp":1530215261440488,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.seqid","value":"8","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"source","value":"quantizer-33","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}}]},{"traceId":"05d68b91f746bc97","id":"ce68a2803229d64e","name":"gettimeseriesblocks","parentId":"f99c9d1388de97da","timestamp":1530215261400008,"duration":728,"annotations":[{"timestamp":1530215261400008,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"timestamp":1530215261400736,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.seqid","value":"7","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"source","value":"quantizer-32","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}]},{"traceId":"05d68b91f746bc97","id":"49d833f138ef21c7","name":"gettimeseriesblocks","parentId":"f99fda5cf623bfd8","timestamp":1530215261419008,"duration":713,"annotations":[{"timestamp":1530215261419008,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"timestamp":1530215261419721,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.seqid","value":"4","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"source","value":"quantizer-32","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}]},{"traceId":"05d68b91f746bc97","id":"e39ff05f3c87f58b","name":"gettimeseriesblocks","parentId":"c061064f0af5ddb7","timestamp":1530215261421006,"duration":913,"annotations":[{"timestamp":1530215261421006,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"timestamp":1530215261421919,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.seqid","value":"8","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"source","value":"quantizer-32","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}]},{"traceId":"05d68b91f746bc97","id":"6ed48bfa84e38bc9","name":"gettimeseriesblocks","parentId":"b0a0aea5f6f3a217","timestamp":1530215261385003,"duration":574,"annotations":[{"timestamp":1530215261385003,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"timestamp":1530215261385577,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.seqid","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"source","value":"quantizer-34","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}]},{"traceId":"05d68b91f746bc97","id":"bf5b0653d9825412","name":"gettimeseriesblocks","parentId":"2e246bc456e99455","timestamp":1530215261384031,"duration":524,"annotations":[{"timestamp":1530215261384031,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"timestamp":1530215261384555,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.seqid","value":"3","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"source","value":"quantizer-34","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}]},{"traceId":"05d68b91f746bc97","id":"6409d9c7d72e9004","name":"gettimeseriesblocks","parentId":"ef3a0907aea28567","timestamp":1530215261383004,"duration":524,"annotations":[{"timestamp":1530215261383004,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"timestamp":1530215261383528,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.seqid","value":"3","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"source","value":"quantizer-34","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}]},{"traceId":"05d68b91f746bc97","id":"0d7ffb80ce17aaa1","name":"gettimeseriesblocks","parentId":"370f8f9eac81f776","timestamp":1530215261383004,"duration":654,"annotations":[{"timestamp":1530215261383004,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"timestamp":1530215261383658,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.seqid","value":"6","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"source","value":"quantizer-33","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}}]},{"traceId":"05d68b91f746bc97","id":"426ba36a670c2bcf","name":"gettimeseriesblocks","parentId":"5f82bccb81e3f730","timestamp":1530215261384005,"duration":585,"annotations":[{"timestamp":1530215261384005,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"timestamp":1530215261384590,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.seqid","value":"7","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"source","value":"quantizer-32","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}]},{"traceId":"05d68b91f746bc97","id":"1c0b8de0525cb7b0","name":"gettimeseriesblocks","parentId":"c10eb603850ac5df","timestamp":1530215261385004,"duration":488,"annotations":[{"timestamp":1530215261385004,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"timestamp":1530215261385492,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.seqid","value":"6","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}},{"key":"source","value":"quantizer-34","endpoint":{"serviceName":"quantizer","ipv4":"10.2.66.243"}}]},{"traceId":"05d68b91f746bc97","id":"277958ac95d755c9","name":"gettimeseriesblocks","parentId":"7c2761d1b852a16d","timestamp":1530215261384004,"duration":519,"annotations":[{"timestamp":1530215261384004,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"timestamp":1530215261384523,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.seqid","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"source","value":"quantizer-33","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}}]},{"traceId":"05d68b91f746bc97","id":"8bcaf1d2944cb070","name":"gettimeseriesblocks","parentId":"44972e986c9955d9","timestamp":1530215261385004,"duration":487,"annotations":[{"timestamp":1530215261385004,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"timestamp":1530215261385491,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.seqid","value":"14","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}},{"key":"source","value":"quantizer-33","endpoint":{"serviceName":"quantizer","ipv4":"10.2.36.150"}}]},{"traceId":"05d68b91f746bc97","id":"786eb27a74d4c0f7","name":"gettimeseriesblocks","parentId":"c334e9c01fe40ec7","timestamp":1530215261383009,"duration":517,"annotations":[{"timestamp":1530215261383009,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"timestamp":1530215261383526,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.seqid","value":"6","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"source","value":"quantizer-32","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}]},{"traceId":"05d68b91f746bc97","id":"b1aada8e334fad31","name":"gettimeseriesblocks","parentId":"613a3265135852ce","timestamp":1530215261385004,"duration":499,"annotations":[{"timestamp":1530215261385004,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"timestamp":1530215261385503,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.seqid","value":"5","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"source","value":"quantizer-32","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}]},{"traceId":"05d68b91f746bc97","id":"aa763d24b704ae83","name":"gettimeseriesblocks","parentId":"6892bdb3b5f94f45","timestamp":1530215261386005,"duration":658,"annotations":[{"timestamp":1530215261386005,"value":"sr","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"timestamp":1530215261386663,"value":"ss","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}],"binaryAnnotations":[{"key":"component","value":"java-thrift","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.name","value":"getTimeSeriesBlocks","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.seqid","value":"4","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"message.type","value":"2","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"realm","value":"lab0","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_namedTokenId","value":"DWMVK55AIAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"sf_organizationID","value":"BqDQY5OAAAA","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}},{"key":"source","value":"quantizer-32","endpoint":{"serviceName":"quantizer","ipv4":"10.2.10.36"}}]}]`

func BenchmarkHTTPSink_AddSpans(b *testing.B) {
	bytes := int64(0)
	var tt []*trace.Span
	json.Unmarshal([]byte(longTraceExample), &tt)

	b.ReportAllocs()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		bb, _ := jsonMarshal(tt)
		bytes += int64(len(bb))
	}
	b.SetBytes(bytes)
}

var GoSpanSource = &goSpans{}

type goSpans struct{}

func (g *goSpans) Spans() []*trace.Span {
	var tt []*trace.Span
	json.Unmarshal([]byte(longTraceExample), &tt)
	return tt
}

type contextPut string

func TestSetHeaders(t *testing.T) {
	Convey("test headers", t, func() {
		h := &HTTPSink{AuthToken: "foo", UserAgent: "agent"}
		req := httptest.NewRequest("post", "/v2/datapoint", nil)
		ctx := context.WithValue(context.Background(), XDebugID, "foo")
		ctx = context.WithValue(ctx, XTracingDebug, "foo")
		ctx = context.WithValue(ctx, XTracingID, "bar")
		h.setHeadersOnBottom(ctx, req, "application/json", true)
		So(req.Header.Get(string(XDebugID)), ShouldEqual, "foo")
		So(req.Header.Get(string(XTracingDebug)), ShouldEqual, "foo")
		So(req.Header.Get(string(XTracingID)), ShouldEqual, "bar")
	})
	Convey("test setTokenHeader", t, func() {
		h := &HTTPSink{AuthToken: "foo", UserAgent: "agent"}
		req := httptest.NewRequest("post", "/v2/datapoint", nil)
		// nolint: golint
		ctx := context.WithValue(context.Background(), TokenHeaderName, "bar")
		h.setTokenHeader(ctx, req)
		So(req.Header.Get(string(TokenHeaderName)), ShouldEqual, "bar")
	})
}
