// CD_GUI.cc for BBCD - a CD player for X11 / BlackBox
// Copyright (c) 2002 Bertrand Duret <bertrand.duret at libertysurf.fr>
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "CD_GUI.hh"
#include "CD_Ctrl.hh"

#include <sys/types.h>
#include <sys/select.h>
#include <unistd.h>

#include <X11/Xlib.h>
#include <X11/Xresource.h>

#include <string>
#include <iostream>
#include <strstream>

#include "Image.hh"

PlayBDrawer::PlayBDrawer(Display* d, GC* _gc, GC* _gca,
                         BTexture* tr, BTexture* tp)
  : ButtonDrawer(d, _gc, _gca, tr, tp), playing(false)
{ }

void PlayBDrawer::draw(Button* b, bool isPressed, bool forced) const {
  ButtonDrawer::draw(b, isPressed, forced);

  XPoint p[3];
  p[0].x = b->getWidth() / 2 + 2; p[0].y = b->getHeight() / 2;
  p[1].x = -4; p[1].y = -2;
  p[2].x = 0; p[2].y = 4;

  XFillPolygon(display, b->getXWindow(), (playing ? *gca : *gc), p, 3, Convex,
      CoordModePrevious);
}



PauseBDrawer::PauseBDrawer(Display* d, GC* _gc, GC* _gca,
                           BTexture* tr, BTexture* tp)
  : ButtonDrawer(d, _gc, _gca, tr, tp), blinkState(false)
{ }

void PauseBDrawer::draw(Button* b, bool p, bool f) const {
  ButtonDrawer::draw(b, p, f);

  GC curGC = blinkState ? *gc : *gca;

  XFillRectangle(display, b->getXWindow(), curGC,
      b->getWidth() / 2 - 3, b->getHeight() / 2 - 2, 2, 4);
  XFillRectangle(display, b->getXWindow(), curGC,
      b->getWidth() / 2 + 1, b->getHeight() / 2 - 2, 2, 4);
}




StopBDrawer::StopBDrawer(Display* d, GC* _gc, GC* _gca,
                         BTexture* tr, BTexture* tp)
: ButtonDrawer(d, _gc, _gca, tr, tp)
{ }

void StopBDrawer::draw(Button* b, bool isPressed, bool forced) const {
  ButtonDrawer::draw(b, isPressed, forced);

  XPoint p[4];
  p[0].x = b->getWidth() / 2 - 2; p[0].y = b->getHeight() / 2 - 2;
  p[1].x = 0; p[1].y = 4;
  p[2].x = 4; p[2].y = 0;
  p[3].x = 0; p[3].y = -4;

  XFillPolygon(display, b->getXWindow(), *gc, p, 4, Convex,
      CoordModePrevious);
}



ForwardBDrawer::ForwardBDrawer(Display* d, GC* _gc, GC* _gca,
                               BTexture* tr, BTexture* tp)
  : ButtonDrawer(d, _gc, _gca, tr, tp)
{ }

void ForwardBDrawer::draw(Button* b, bool isPressed, bool forced) const {
  ButtonDrawer::draw(b, isPressed, forced);

  XPoint p[3];
  p[0].x = b->getWidth() / 2; p[0].y = b->getHeight() / 2;
  p[1].x = -4; p[1].y = -2;
  p[2].x = 0; p[2].y = 4;

  XFillPolygon(display, b->getXWindow(), *gc, p, 3, Convex,
      CoordModePrevious);
  p[0].x += 4;
  XFillPolygon(display, b->getXWindow(), *gc, p, 3, Convex,
      CoordModePrevious);
}




BackwardBDrawer::BackwardBDrawer(Display* d, GC* _gc, GC* _gca,
                               BTexture* tr, BTexture* tp)
  : ButtonDrawer(d, _gc, _gca, tr, tp)
{ }

void BackwardBDrawer::draw(Button* b, bool isPressed, bool forced) const {
  ButtonDrawer::draw(b, isPressed, forced);

  XPoint p[3];
  p[0].x = b->getWidth() / 2 - 4; p[0].y = b->getHeight() / 2;
  p[1].x = 4; p[1].y = -2;
  p[2].x = 0; p[2].y = 4;

  XFillPolygon(display, b->getXWindow(), *gc, p, 3, Convex,
      CoordModePrevious);
  p[0].x += 4;
  XFillPolygon(display, b->getXWindow(), *gc, p, 3, Convex,
      CoordModePrevious);
}





CD_GUI::CD_GUI(Option* opt)
  throw (CD_GUI::UnableToOpenDisplay)
  : display(XOpenDisplay(opt->display.c_str())), screen(0),
    playerW(None), background(None), running(true),
    width(0), height(0), x(0), y(0), option(opt),
    playB(0), backwardB(0), forwardB(0), stopB(0),
    playBD(0), pauseBD(0), backwardBD(0), forwardBD(0), stopBD(0),
    backgroundT(0), releasedT(0), pressedT(0), img_ctrl(0),
    timeoutDelay(0l), timerStarted(false), cd_ctrl(0),
    wm_delete_window(0)
{
  if(! display) throw UnableToOpenDisplay();
  screen = DefaultScreen(display);

  wm_delete_window = XInternAtom(display, "WM_DELETE_WINDOW", false);

  makeGUI();

  cd_ctrl = new CD_Controler(option->cdrom_dev);

  // Curent status of CD Drive
  CD_Controler::AudioStatus as(cd_ctrl->getAudioStatus());
  switch(as) {
  case CD_Controler::Play:
    setDelay(1000000l);
    playBD->setPlaying(true);
    startTimer();
    break;
  case CD_Controler::Paused:
    setDelay(500000l);
    playB->setDrawer(pauseBD);
    startTimer();
    break;
  default:
    break;
  }
}

CD_GUI::~CD_GUI() {
  delete playB;
  delete stopB;
  if(backwardB) delete backwardB;
  if(forwardB) delete forwardB;
  delete playBD;
  delete stopBD;
  delete forwardBD;
  delete backwardBD;
  delete pauseBD;
  delete backgroundT;
  delete releasedT;
  delete pressedT;
  delete img_ctrl;
  cleanupGUI();
  delete cd_ctrl;
}

void CD_GUI::run() {
  int xfd = ConnectionNumber(display);
  XEvent xev;

  while(running) {
    if(!timerStarted || XPending(display)) {
      XNextEvent(display, &xev);
      switch(xev.type) {
      case Expose:
        processExpose(xev.xexpose);
        break;
      case ButtonPress:
        processButtonPress(xev.xbutton);
        break;
      case ButtonRelease:
        processButtonRelease(xev.xbutton);
        break;
      case ConfigureNotify:
        processConfigure(xev.xconfigure);
        break;
      case ClientMessage:
        processClientMessage(xev.xclient);
        break;
      }
    } else {
      bool expired(false);

      fd_set rfds;
      FD_ZERO(&rfds);
      FD_SET(xfd, &rfds);

      timeval now, timeout;
      gettimeofday(&now, 0);
      timeout.tv_sec = startTime.tv_sec - now.tv_sec;
      timeout.tv_usec = startTime.tv_usec + timeoutDelay - now.tv_usec;
      if(timeout.tv_usec >= 1000000) {
        timeout.tv_sec += timeout.tv_usec / 1000000;
        timeout.tv_usec = timeout.tv_usec % 1000000;
      } else if(timeout.tv_usec < 0 && timeout.tv_sec >= 0) {
        timeout.tv_sec -= timeout.tv_usec / 1000000 + 1;
        timeout.tv_usec = timeout.tv_usec % 1000000 + 1000000;
      }

      if(timeout.tv_sec >= 0 && timeout.tv_usec > 0) {
        expired = (select(xfd + 1, &rfds, 0, 0, &timeout) == 0);
      } else {
        expired = true;
      }

      if(expired) {
        gettimeofday(&startTime, 0);
        processTimeout();
      }

    } // if XPending / else
  } // while(running)
}

void CD_GUI::makeGUI() {
  // Set size hints
  XSizeHints szhints;
  computeSizeAndPosition(& szhints);

  // Create main Window:
  unsigned long wattr_mask = CWEventMask;
  XSetWindowAttributes wattr;
  wattr.event_mask = ButtonPressMask | ButtonReleaseMask | ExposureMask |
    StructureNotifyMask;
  playerW = XCreateWindow(display, DefaultRootWindow(display), x, y,
    width, height, 0, CopyFromParent, InputOutput, CopyFromParent,
    wattr_mask, &wattr);

  // Set WM hints
  XWMHints wmhints;
  if(option->withdrawn) {
    wmhints.initial_state = WithdrawnState;
  } else {
    wmhints.initial_state = NormalState;
  }
  wmhints.flags = IconWindowHint | StateHint;
  wmhints.icon_window = playerW;

  // Set class hints
  XClassHint classhints;
  classhints.res_name = "bbcd";
  classhints.res_class = "bbtools";

  XTextProperty wname;
  XStringListToTextProperty(&classhints.res_name, 1, &wname);

  XSetWMProperties(display, playerW, &wname, 0, option->argv, option->argc,
      &szhints, &wmhints, &classhints);

  // Set WM/BB atoms
  Atom wmproto[2];
  wmproto[0] = XInternAtom (display, "WM_DELETE_WINDOW", false);
  wmproto[1] = XInternAtom(display, "_BLACKBOX_STRUCTURE_MESSAGES", false);
  XSetWMProtocols(display, playerW, wmproto, 2);

  // Create GC for main window and drawers
  XGCValues xgcv;
  xgcv.foreground = BlackPixel(display, screen);
  xgcv.background = WhitePixel(display, screen);
  gc = XCreateGC(display, playerW, GCForeground|GCBackground, &xgcv);
  gcActive = XCreateGC(display, playerW, GCForeground|GCBackground, &xgcv);

  // Create Textures for main window and drawers
  img_ctrl = new BImageControl(display, screen);
  backgroundT = new BTexture(display, screen, img_ctrl);
  pressedT = new BTexture(display, screen, img_ctrl);
  releasedT = new BTexture(display, screen, img_ctrl);

  // Load styles (bbcd configuration + blackbox style, if available)
  readStyles();

  // Now we can set the background (color, pixmap, transparent...)
  setBackground();


  // Create drawers
  playBD = new PlayBDrawer(display, &gc, &gcActive, releasedT, pressedT);
  pauseBD = new PauseBDrawer(display, &gc, &gcActive, releasedT, pressedT);
  backwardBD = new BackwardBDrawer(display, &gc, &gcActive,
      releasedT, pressedT);
  forwardBD = new ForwardBDrawer(display, &gc, &gcActive,
      releasedT, pressedT);
  stopBD = new StopBDrawer(display, &gc, &gcActive, releasedT, pressedT);

  // Create buttons
  if(!option->compact) {
    backwardB = new Button(option->buttonSize, option->buttonSize, 1, 1,
        display, playerW, backwardBD);
    forwardB = new Button(option->buttonSize, option->buttonSize, 1, 1,
        display, playerW, forwardBD);
  }
  playB = new Button(option->buttonSize, option->buttonSize, 1, 1,
      display, playerW, playBD);
  stopB = new Button(option->buttonSize, option->buttonSize, 1, 1,
      display, playerW, stopBD);
  switch(option->layout) {
  case Horizontal:
    if(option->compact) {
      stopB->move(option->buttonSize+2, 1);
    } else {
      playB->move(option->buttonSize+2, 1);
      stopB->move((option->buttonSize+1)*2+1, 1);
      forwardB->move((option->buttonSize+1)*3+1, 1);
    }
    break;
  case Box:
    if(option->compact) {
      stopB->move(option->buttonSize+2, 1);
    } else {
      playB->move(1, option->buttonSize+2);
      stopB->move(option->buttonSize+2, option->buttonSize+2);
      forwardB->move(option->buttonSize+2, 1);
    }
    break;
  case Vertical:
    if(option->compact) {
      stopB->move(1, option->buttonSize+2);
    } else {
      playB->move(1, option->buttonSize+2);
      stopB->move(1, (option->buttonSize+1)*2+1);
      forwardB->move(1, (option->buttonSize+1)*3+1);
    }
    break;
  }

  XMapSubwindows(display, playerW);
  XMapWindow(display, playerW);
}

void CD_GUI::computeSizeAndPosition(XSizeHints * szh) {
  // Compute size, depending on options
  unsigned int nr(1), nc(1);
  switch(option->layout) {
  case Horizontal:
    nc = option->compact ? 2 : 4;
    break;
  case Box:
    nr = option->compact ? 1 : 2;
    nc = 2;
    break;
  case Vertical:
    nr = option->compact ? 2 : 4;
  }
  width = (option->buttonSize + 1) * nc + 1;
  height = (option->buttonSize + 1) * nr + 1;

  szh->max_width = szh->min_width = width;
  szh->max_height = szh->min_height = height;
  szh->x = szh->y = 0;
  szh->flags = USPosition | PMinSize | PMaxSize;

  // Parse position from options: first extract position info, then compute pos
  int gx(0), gy(0), grv(0), gw(0), gh(0);
  unsigned int ugw(0), ugh(0);
  int mask(XParseGeometry(option->geometry.c_str(), &gx, &gy, &ugw, &ugh));

  std::ostrstream user_geom;
  if((mask & XValue) && (mask & XNegative)) user_geom<<"-"<<-gx;
  else user_geom<<"+"<<gx;
  if((mask & YValue) && (mask & YNegative)) user_geom<<"-"<<-gy;
  else user_geom<<"+"<<gy;
  user_geom<<std::ends;
  user_geom.freeze();

  std::ostrstream def_geom;
  def_geom<<width<<"x"<<height<<"-0-0";
  mask = XWMGeometry(display, screen, user_geom.str(),
    def_geom.str(), 0, szh, &gx, &gy, &gw, &gh, &grv);
  szh->x = x = gx;
  szh->y = y = gy;
}

void CD_GUI::cleanupGUI() {
  XFreeGC(display, gc);
  XFreeGC(display, gcActive);
  if(background != None && background != ParentRelative) {
    XFreePixmap(display, background);
  }
  XDestroyWindow(display, playerW);
}

void CD_GUI::readStyles() {
  // Set the default style
  setDefaultStyle();

  // Set style infos from bbcd configuration file
  readBBCDStyle();

  // Set style from running blackbox
  readBBStyle();
}

void CD_GUI::readBBStyle() {
  if(option->bbstylefile == "")
    return;

  XrmDatabase stylerc = XrmGetFileDatabase(option->bbstylefile.c_str());
  if(!stylerc) {
    std::cerr<<"Style file not found !\n";
    return;
  }

  readStyle(stylerc);
}

void CD_GUI::readBBCDStyle() {
  std::string style = (option->bbcdrcfile == "") ?
    (option->defaultBbcdrcfile) : (option->bbcdrcfile);

  XrmDatabase stylerc = XrmGetFileDatabase(style.c_str());
  if(!stylerc) {
    if(style == option->bbcdrcfile) { // Let's try default one
      stylerc = XrmGetFileDatabase(option->defaultBbcdrcfile.c_str());
      if(!stylerc) {
        std::cerr<<"BBCD default configuration file not found: "
          <<option->defaultBbcdrcfile<<" !\n";
        return;
      }
    } else {
      std::cerr<<"BBCD configuration file not found !\n";
      return;
    }
  }

  readStyle(stylerc);
}

void CD_GUI::readStyle(const XrmDatabase & stylerc) {
  XrmValue value;
  char* value_type;

  // Toolbar
  if(XrmGetResource(stylerc, "toolbar.color", "Toolbar.Color", &value_type,
    &value)) {
    backgroundT->setColor(BColor(value.addr, display, screen));
  }
  if(XrmGetResource(stylerc, "toolbar.colorTo", "Toolbar.ColorTo",
    &value_type, &value)) {
    backgroundT->setColorTo(BColor(value.addr, display, screen));
  }
  if(XrmGetResource(stylerc, "toolbar", "Toolbar", &value_type, &value)) {
    backgroundT->setDescription(value.addr);
  }
  // Toolbar button
  if(XrmGetResource(stylerc, "toolbar.button.color", "Toolbar.Button.Color",
        &value_type, &value)) {
    releasedT->setColor(BColor(value.addr, display, screen));
  }
  if(XrmGetResource(stylerc, "toolbar.button.colorTo",
        "Toolbar.Button.ColorTo", &value_type, &value)) {
    releasedT->setColorTo(BColor(value.addr, display,
        screen));
  }
  if(XrmGetResource(stylerc, "toolbar.button", "Toolbar.Button",
        &value_type, &value)) {
    releasedT->setDescription(value.addr);
  }

  // Toolbar pressed button
  if(XrmGetResource(stylerc, "toolbar.button.pressed.color",
        "Toolbar.Button.Pressed.Color", &value_type, &value)) {
    pressedT->setColor(BColor(value.addr, display, screen));
  }
  if(XrmGetResource(stylerc, "toolbar.button.pressed.colorTo",
        "Toolbar.Button.Pressed.ColorTo", &value_type, &value)) {
    pressedT->setColorTo(BColor(value.addr, display,
        screen));
  }
  if(XrmGetResource(stylerc, "toolbar.button.pressed",
        "Toolbar.Button.Pressed", &value_type, &value)) {
    pressedT->setDescription(value.addr);
  }

  // Toolbar button picColor
  if(XrmGetResource(stylerc, "toolbar.button.picColor",
        "toolbar.button.picColor", &value_type, &value)) {
    XGCValues xgcv;
    xgcv.foreground =
        BColor(value.addr, display, screen).pixel();
    XChangeGC(display, gc, GCForeground, &xgcv);
  }

  // Active color
  if(XrmGetResource(stylerc, "bbcd.activeColor", "Bbcd.ActiveColor",
    &value_type, &value)) {
    XGCValues xgcv;
    xgcv.foreground = BColor(value.addr, display, screen).pixel();
    XChangeGC(display, gcActive, GCForeground, &xgcv);
  }

  XrmDestroyDatabase(stylerc);
}

void CD_GUI::setDefaultStyle() {
  BColor c1(std::string("slate grey"), display, screen);
  BColor c2(std::string("white"), display, screen);
  BColor c3(std::string("red"), display, screen);

  backgroundT->setDescription("flat solid");
  backgroundT->setColor(c1);

  pressedT->setDescription("gradient bevel1 diagonal raised");
  pressedT->setColor(c1);
  pressedT->setColorTo(c2);

  releasedT->setDescription("gradient bevel1 crossdiagonal sunken");
  releasedT->setColor(c2);
  releasedT->setColorTo(c1);

  XGCValues xgcv;
  xgcv.foreground = c3.pixel();
  XChangeGC(display, gcActive, GCForeground, &xgcv);
}






void CD_GUI::processExpose(const XExposeEvent &xe) {
  redraw();
}

void CD_GUI::setBackground() {
  Pixmap pm;
  if(option->shape) {
    pm = ParentRelative;
  } else {
    pm = backgroundT->render(width, height, background);
  }
  setPixmap(pm);
  if(pm == None) {
    XSetWindowBackground(display, playerW, pressedT->color().pixel());
  }
}

void CD_GUI::redraw(bool forced) {
  // Set background of main window
  if(forced){
    setBackground();
  }
  XClearWindow(display, playerW);

  // Draw buttons
  if(!option->compact) {
    backwardB->draw(backwardB->isPressed(), forced);
    forwardB->draw(forwardB->isPressed(), forced);
  }
  playB->draw(playB->isPressed(), forced);
  stopB->draw(stopB->isPressed(), forced);
}

void CD_GUI::processButtonPress(const XButtonPressedEvent &xe) {
  if(xe.subwindow == playB->getXWindow()) {
    restoreBD = playB->getDrawer();
    if(option->compact) {
      switch(xe.button) {
      case 1:
        playB->setDrawer(backwardBD);
        break;
      case 3:
        playB->setDrawer(forwardBD);
        break;
      case 2:
      default:
        CD_Controler::AudioStatus as(cd_ctrl->getAudioStatus());
        switch(as) {
        case CD_Controler::Play:
          playB->setDrawer(playBD);
          break;
        case CD_Controler::Paused:
          playB->setDrawer(pauseBD);
          break;
        }
        break;
      }
    }
    playB->draw(true);
  } else if(xe.subwindow == stopB->getXWindow()) {
    stopB->draw(true);
  } else if(!option->compact) {
    if(xe.subwindow == backwardB->getXWindow()) {
      backwardB->draw(true);
    } else if(xe.subwindow == forwardB->getXWindow()) {
      forwardB->draw(true);
    }
  }
}

void CD_GUI::processButtonRelease(const XButtonReleasedEvent &xe)
{
  if(xe.subwindow == playB->getXWindow()) {
    if(playB->containsPoint(xe.x, xe.y)) {
      if(option->compact) {
        switch(xe.button) {
        case 1:
          cd_ctrl->playPrevious();
          if(restoreBD) {
            playB->setDrawer(restoreBD);
            restoreBD = 0;
          } else playB->setDrawer(playBD);
          break;
        case 3:
          cd_ctrl->playNext();
          if(restoreBD) {
            playB->setDrawer(restoreBD);
            restoreBD = 0;
          } else playB->setDrawer(playBD);
          break;
        }
      }
      if(!option->compact || (xe.button != 1 && xe.button != 3)) {
        if(cd_ctrl->isActive()) {
          cd_ctrl->pause();
          if(cd_ctrl->isPaused()) {
            setDelay(500000l);
            playB->setDrawer(pauseBD);
          } else {
            startTimer(1000000l); // Really necessary ?
            playB->setDrawer(playBD);
            playBD->setPlaying(true); // Necessary because Bug Stop !
          }
        } else {
          cd_ctrl->play();
          if(cd_ctrl->isPlaying()) {
            startTimer(1000000l);
            playB->setDrawer(playBD);
            playBD->setPlaying(true);
          }
        }
      }
    } // if(playB->containsPoint(...))
  // End of "if(xe.subwindow == playB->getXWindow())"
  } else if(xe.subwindow == stopB->getXWindow()) {
    if(stopB->containsPoint(xe.x, xe.y)) {
      cd_ctrl->stop();
      stopTimer();
      playB->setDrawer(playBD);
      playBD->setPlaying(false);
    }
  } else if(!option->compact) {
    if(xe.subwindow == backwardB->getXWindow()) {
      if(backwardB->containsPoint(xe.x, xe.y)) {
        cd_ctrl->playPrevious();
      }
    } else if(xe.subwindow == forwardB->getXWindow()) {
      if(forwardB->containsPoint(xe.x, xe.y)) {
        cd_ctrl->playNext();
      }
    }
  } else if(restoreBD) {
    playB->setDrawer(restoreBD);
  }
  restoreBD = 0;

  if(!option->compact) {
    backwardB->draw();
    forwardB->draw();
  }
  playB->draw();
  stopB->draw();
}

void CD_GUI::processConfigure(const XConfigureEvent &xe) {
#ifdef DEBUG
  std::cerr<<"*** Processing configure event:"<<xe.serial<<"\tSend:"
    <<(xe.send_event ? "yes" : "no")<<"\n"
    <<"["<<xe.width<<"x"<<xe.height<<"] ("<<xe.x<<","<<xe.y<<") BW:"
    <<xe.border_width<<" Override_redirect:"
    <<(xe.override_redirect ? "yes" : "no")<<"\n";
#endif
  if((xe.window == playerW) && (xe.send_event)) {
    option->init();
    readStyles();
    redraw(true);
  }
}

void CD_GUI::processClientMessage(const XClientMessageEvent &xe) {
#ifdef DEBUG
  std::cerr<<"*** Processing client message:"<<xe.serial<<"\tSend:"
    <<(xe.send_event ? "yes" : "no")<<"\n";
#endif
  if(static_cast<unsigned int>(xe.data.l[0]) == wm_delete_window) {
    running = false;
  }
}

void CD_GUI::processTimeout() {
#ifdef DEBUG
//  std::cerr<<"*** Processing timeout\n";
#endif
  CD_Controler::AudioStatus as(cd_ctrl->getAudioStatus());
  switch(as) {
  case CD_Controler::Paused: // Pause button blinking
    if(playB->getDrawer() == pauseBD) { // Actually paused
      pauseBD->blink();
      playB->draw();
    } else { // Paused, but not asked by user -> End of play (Stop bug)
      stopTimer();
      playB->setDrawer(playBD);
      playBD->setPlaying(false);
    }
    break;
  case CD_Controler::Play:
    break;
  default: // End of play (this should be the normal situation)
    stopTimer();
    playB->setDrawer(playBD);
    playBD->setPlaying(false);
  }
}

void CD_GUI::startTimer(unsigned long delay) {
  gettimeofday(&startTime, 0);
  timerStarted = true;
  if(delay) setDelay(delay);
}
