/*
 * TemplateFilter.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1999-2004 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, drach, suhler.
 *
 * Version:  2.4
 * Created by suhler on 99/07/29
 * Last modified by suhler on 04/12/30 12:42:21
 */

package sunlabs.brazil.filter;

import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;
import sunlabs.brazil.util.http.MimeHeaders;
import sunlabs.brazil.template.TemplateRunner;
import java.io.UnsupportedEncodingException;

import java.util.Properties;

/**
 * The <code>TemplateFilter</code> sends HTML content through an
 * Html/XML parser to a set of <code>Template</code>s.  Each Html/XML tag may
 * dynamically invoke a Java method present in the <code>Template</code>s.
 * The dynamically-generated content from evaluating the Html/XML tags is
 * returned to the caller.
 * <p>
 * The following configuration parameters are used to initialize this
 * <code>Filter</code>.
 * <dl class=props>
 * <dt> <code>templates</code>
 * <dd> A list of template names.  For each name in the list, the property
 *      <code><i>name</i>.class</code> is examined to determine which class to
 *      use for each template.  Then <code>name</code> is used as the prefix
 *      for other template specific properties if any.  If
 *      <code><i>name</i>.class</code> does not exist, then <code>name</code>
 *      is assumed to be the class name, and there are no template specific
 *      properties for the template.  Methods in the template classes will be
 *      invoked to process the XML/HTML tags present in the content.
 * <dt> <code>session</code>
 * <dd> The request property that contains the session ID.  If no
 *	"session" property is found with the supplied prefix, then
 *	the global "session" property is used instead.  The default
 *	value is "SessionID".
 * <dt> <code>subtype</code>
 * <dd> Restrict this template to only handle specified text sub-types.
 *	defaults to the empty string, which implies any text sub-type.
 * <dt> <code>encoding</code>
 * <dd> The charset encoding to use to represent the content as text.
 *	If none is specified, the default encoding is used.
 * <dt> <code>outputEncoding</code>
 * <dd> The character encoding to use to interpret the template results.
 *	If no "outputEncoding" is specified, then "encoding" is used.
 *      Once template processing is complete, the results are converted
 *      into a byte stream for transmission to the next filter, using
 *	"outputEncoding", if specified.  If not specified then the
 *	default encoding is used.
 * <dt> <code>tagPrefix</code>
 * <dd> If specified, all tag names defined for each template class are prefixed with
 *      <i>tagPrefix</i>.
 *      This parameter only takes effect
 *      if the <code>tagPrefix</code> option is not specified for an individual template.
 * </dl>
 * The <code>TemplateHandler</code> class is similar, but not identical to
 * running a <code>FilterHandler</code> with the <code>FileHandler</code> and 
 * the <code>TemplateFilter</code>.  The differences between the two should
 * be resolved.
 * <p>
 * Note: The <code>templates</code> property accepts a list of
 * class names or <i>tokens</i> that could be used to represent
 * class names.  If class <i>names</i> are used, all
 * template classes share the TemplateHandler's properties prefix.
 *
 * @author	Stephen Uhler (stephen.uhler@sun.com)
 * @author	Colin Stevens (colin.stevens@sun.com)
 * @version	2.4
 */

public class TemplateFilter implements Filter {
    private static final String TEMPLATES = "templates"; 
    private static final String SESSION = "session";

    Server server;
    String prefix;    
    String subtype;		// media subtype to filter (must be text)

    String session = "SessionID";	// our session ID property name

    private TemplateRunner runner;	// The template object for our class

    public boolean
    init(Server server, String prefix)
    {
	Properties props = server.props;

	this.server = server;
	this.prefix = prefix;

	String name = props.getProperty(prefix + TEMPLATES, "");
	subtype = "text/" + props.getProperty(prefix + "subtype", "");
	try {
	    runner = new TemplateRunner(server, prefix, name);
	} catch (Exception e) {
	    server.log(Server.LOG_ERROR, prefix,
		    "Can't instantiate " + e.getMessage());
	    return false;
	}

	session = props.getProperty(prefix + SESSION, session);

	return true;
    }

    /**
     * No action before request is made
     */

    public boolean
    respond(Request request)
    {
	return false;
    }

    /**
     * Filters all HTML files, or files that are likely to be html files,
     * specifically, those whose "Content-Type" starts with "text/".
     */
    public boolean
    shouldFilter(Request request, MimeHeaders headers)
    {
	String type = headers.get("Content-Type");
	boolean filter= ((type != null) &&
		type.toLowerCase().startsWith("text/"));
	request.log(Server.LOG_DIAGNOSTIC, prefix + "type: " + type +
		    " filter?: " + filter);
	return filter;
    }

    /**
     * Evaluates the content as html/XML tags, if the file is (or has now been
     * converted to) "text/html".
     */
    public byte[]
    filter(Request request, MimeHeaders headers, byte[] content)
    {
	if (headers.get("Content-Type").toLowerCase().startsWith(subtype)
		    == false) {
	    request.log(Server.LOG_DIAGNOSTIC, prefix +
		    " Not " + subtype + ", skipping");
	    return content;
	}

	/*
	 * Get the cookie out of the request object (presumably from the
	 * sessionhandler).  If not there,  don't use cookies.
	 */

        String sessionId = request.props.getProperty(session);
	String encoding = request.props.getProperty(prefix + "encoding",
		request.props.getProperty("encoding"));
	String outputEncoding = request.props.getProperty(prefix +
		"outputEncoding", encoding);
	if (sessionId == null) {
	    sessionId = "noCookie";
	    request.log(Server.LOG_WARNING, "template not using cookies");
	}

	/*
	 * Process the content as a template.  If there is an error, 
	 * leave the content unchanged.
	 */

        String enc = null;
	if (encoding != null) {
	    try {
	       enc = new String(content, encoding);
	    } catch(UnsupportedEncodingException e) {
		request.log(Server.LOG_WARNING, "template Error: " +
		    e.getMessage());
	    }
	}
	if (enc == null) {
	    enc = new String(content);
	}

	String result = runner.process(request, enc, sessionId);
	if (result != null) {
	    if (outputEncoding != null) {
		try {
	            return result.getBytes(outputEncoding);
		} catch (UnsupportedEncodingException e) {
		    request.log(Server.LOG_WARNING, e.getMessage());
		}
	    }
	    return result.getBytes();
	} else {
	    request.log(Server.LOG_INFORMATIONAL, "template Error: " +
		    runner.getError());
	    return content;
	}
    }
}
