package remoteip

import (
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/stretchr/testify/require"
)

func TestRemoteIPParser(t *testing.T) {
	parser := NewRemoteIPParser(defaultTrustedProxies)

	tests := []struct {
		name        string
		remoteAddr  string
		headers     map[string]string
		expectedIP  string
		expectError bool
		errorType   error
	}{
		{
			name:        "Direct connection",
			remoteAddr:  "192.168.1.10:1234",
			headers:     map[string]string{},
			expectedIP:  "192.168.1.10",
			expectError: false,
		},
		{
			name:        "X-Forwarded-For single IP",
			remoteAddr:  "10.0.0.1:1234",
			headers:     map[string]string{"X-Forwarded-For": "203.0.113.7"},
			expectedIP:  "203.0.113.7",
			expectError: false,
		},
		{
			name:        "X-Forwarded-For multiple IPs",
			remoteAddr:  "10.0.0.1:1234",
			headers:     map[string]string{"X-Forwarded-For": "203.0.113.7, 203.0.113.8, 198.51.100.5"},
			expectedIP:  "203.0.113.7",
			expectError: false,
		},
		{
			name:        "X-Forwarded-For with mixed spaces and commas",
			remoteAddr:  "10.0.0.1:1234",
			headers:     map[string]string{"X-Forwarded-For": "203.0.113.7,203.0.113.8   198.51.100.5"},
			expectedIP:  "203.0.113.7",
			expectError: false,
		},
		{
			name:        "X-Forwarded-For with trusted proxies",
			remoteAddr:  "10.0.0.1:1234",
			headers:     map[string]string{"X-Forwarded-For": "10.0.0.5, 192.168.1.2, 203.0.113.7"},
			expectedIP:  "203.0.113.7",
			expectError: false,
		},
		{
			name:        "X-Forwarded-For with CIDR notation (invalid)",
			remoteAddr:  "10.0.0.1:1234",
			headers:     map[string]string{"X-Forwarded-For": "203.0.113.7/24, 10.0.0.5"},
			expectedIP:  "10.0.0.5",
			expectError: false,
		},
		{
			name:        "Client-IP header",
			remoteAddr:  "10.0.0.1:1234",
			headers:     map[string]string{"Client-Ip": "203.0.113.7"},
			expectedIP:  "203.0.113.7",
			expectError: false,
		},
		{
			name:       "Both headers consistent",
			remoteAddr: "10.0.0.1:1234",
			headers: map[string]string{
				"X-Forwarded-For": "203.0.113.7, 10.0.0.5",
				"Client-Ip":       "203.0.113.7",
			},
			expectedIP:  "203.0.113.7",
			expectError: false,
		},
		{
			name:       "Both headers inconsistent (spoofing)",
			remoteAddr: "10.0.0.1:1234",
			headers: map[string]string{
				"X-Forwarded-For": "203.0.113.7, 10.0.0.5",
				"Client-Ip":       "203.0.113.8",
			},
			expectedIP:  "",
			expectError: true,
			errorType:   ErrIPSpoofAttack,
		},
		{
			name:        "Multiple X-Forwarded-For headers (as comma-separated list)",
			remoteAddr:  "10.0.0.1:1234",
			headers:     map[string]string{"X-Forwarded-For": "203.0.113.7, 203.0.113.8, 203.0.113.9"},
			expectedIP:  "203.0.113.7",
			expectError: false,
		},
		{
			name:        "IPv6 address",
			remoteAddr:  "[2001:db8::1]:1234",
			headers:     map[string]string{},
			expectedIP:  "2001:db8::1",
			expectError: false,
		},
		{
			name:        "IPv6 in X-Forwarded-For",
			remoteAddr:  "10.0.0.1:1234",
			headers:     map[string]string{"X-Forwarded-For": "2001:db8::1"},
			expectedIP:  "2001:db8::1",
			expectError: false,
		},
		{
			name:        "IPv6 in X-Forwarded-For with brackets",
			remoteAddr:  "10.0.0.1:1234",
			headers:     map[string]string{"X-Forwarded-For": "[2001:db8::1]"},
			expectedIP:  "2001:db8::1",
			expectError: false,
		},
		{
			name:        "All trusted proxies",
			remoteAddr:  "127.0.0.1:1234",
			headers:     map[string]string{"X-Forwarded-For": "10.0.0.1, 192.168.1.1"},
			expectedIP:  "192.168.1.1",
			expectError: false,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			req := httptest.NewRequest(http.MethodGet, "http://example.com", nil)
			req.RemoteAddr = tt.remoteAddr

			// Add headers
			for name, value := range tt.headers {
				req.Header.Set(name, value)
			}

			ip, err := parser.GetIP(req)

			if tt.expectError {
				require.Error(t, err)
			} else {
				require.NoError(t, err)
			}

			require.Equal(t, tt.expectedIP, ip)
		})
	}
}

func TestParseIP(t *testing.T) {
	parser := NewRemoteIPParser(nil)

	tests := []struct {
		input    string
		expected string
	}{
		{"192.168.1.1", "192.168.1.1"},
		{"[2001:db8::1]", "2001:db8::1"},
		{"[2001:db8::1]:8080", "2001:db8::1"},
		{"192.168.1.1:8080", "192.168.1.1"},
		{"  192.168.1.1  ", "192.168.1.1"},
		{"192.168.1.1/24", ""}, // CIDR notation should be rejected
		{"not-an-ip", ""},
		{"256.256.256.256", ""}, // Invalid IP
		{"2001:db8::1", "2001:db8::1"},
	}

	for _, tt := range tests {
		t.Run(tt.input, func(t *testing.T) {
			result := parser.parseIP(tt.input)
			require.Equal(t, tt.expected, result)
		})
	}
}

func TestCustomTrustedProxies(t *testing.T) {
	parser := NewRemoteIPParser(nil)
	// Add a custom trusted proxy
	parser.TrustedProxies = append(parser.TrustedProxies, "203.0.113.0/24")

	req := httptest.NewRequest(http.MethodGet, "http://example.com", nil)
	req.RemoteAddr = "10.0.0.1:1234"
	req.Header.Set("X-Forwarded-For", "203.0.113.7, 198.51.100.5")

	ip, err := parser.GetIP(req)
	require.NoError(t, err)
	require.Equal(t, "198.51.100.5", ip, "Expected IP to be from the trusted proxy range")
}

func TestDisableIPSpoofingCheck(t *testing.T) {
	parser := NewRemoteIPParser(nil)
	parser.CheckIPSpoofing = false

	req := httptest.NewRequest(http.MethodGet, "http://example.com", nil)
	req.RemoteAddr = "10.0.0.1:1234"
	req.Header.Set("X-Forwarded-For", "203.0.113.7")
	req.Header.Set("Client-Ip", "198.51.100.5") // Different from X-Forwarded-For

	// This would normally trigger the spoofing detection
	ip, err := parser.GetIP(req)
	require.NoError(t, err)
	// With our implementation, X-Forwarded-For takes precedence
	require.Equal(t, "203.0.113.7", ip)
}
