/*
 * Decompiled with CFR 0.152.
 */
package org.asamk.signal.manager.storage.prekeys;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import org.asamk.signal.manager.config.ServiceConfig;
import org.asamk.signal.manager.storage.Database;
import org.asamk.signal.manager.storage.Utils;
import org.signal.libsignal.protocol.InvalidKeyException;
import org.signal.libsignal.protocol.InvalidKeyIdException;
import org.signal.libsignal.protocol.ecc.ECKeyPair;
import org.signal.libsignal.protocol.ecc.ECPrivateKey;
import org.signal.libsignal.protocol.ecc.ECPublicKey;
import org.signal.libsignal.protocol.state.SignedPreKeyRecord;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.whispersystems.signalservice.api.push.ServiceIdType;

public class SignedPreKeyStore
implements org.signal.libsignal.protocol.state.SignedPreKeyStore {
    private static final String TABLE_SIGNED_PRE_KEY = "signed_pre_key";
    private static final Logger logger = LoggerFactory.getLogger(SignedPreKeyStore.class);
    private final Database database;
    private final int accountIdType;

    public static void createSql(Connection connection) throws SQLException {
        try (Statement statement = connection.createStatement();){
            statement.executeUpdate("CREATE TABLE signed_pre_key (\n  _id INTEGER PRIMARY KEY,\n  account_id_type INTEGER NOT NULL,\n  key_id INTEGER NOT NULL,\n  public_key BLOB NOT NULL,\n  private_key BLOB NOT NULL,\n  signature BLOB NOT NULL,\n  timestamp INTEGER DEFAULT 0,\n  UNIQUE(account_id_type, key_id)\n) STRICT;\n");
        }
    }

    public SignedPreKeyStore(Database database, ServiceIdType serviceIdType) {
        this.database = database;
        this.accountIdType = Utils.getAccountIdType(serviceIdType);
    }

    public SignedPreKeyRecord loadSignedPreKey(int signedPreKeyId) throws InvalidKeyIdException {
        SignedPreKeyRecord signedPreKeyRecord = this.getSignedPreKey(signedPreKeyId);
        if (signedPreKeyRecord == null) {
            throw new InvalidKeyIdException("No such signed pre key record: " + signedPreKeyId);
        }
        return signedPreKeyRecord;
    }

    /*
     * Enabled aggressive exception aggregation
     */
    public List<SignedPreKeyRecord> loadSignedPreKeys() {
        String sql = "SELECT p.key_id, p.public_key, p.private_key, p.signature, p.timestamp\nFROM %s p\nWHERE p.account_id_type = ?\n".formatted(TABLE_SIGNED_PRE_KEY);
        try (Connection connection = this.database.getConnection();){
            List<SignedPreKeyRecord> list;
            block14: {
                PreparedStatement statement = connection.prepareStatement(sql);
                try {
                    statement.setInt(1, this.accountIdType);
                    list = Utils.executeQueryForStream(statement, this::getSignedPreKeyRecordFromResultSet).filter(Objects::nonNull).toList();
                    if (statement == null) break block14;
                }
                catch (Throwable throwable) {
                    if (statement != null) {
                        try {
                            statement.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                statement.close();
            }
            return list;
        }
        catch (SQLException e) {
            throw new RuntimeException("Failed read from signed_pre_key store", e);
        }
    }

    public void storeSignedPreKey(int signedPreKeyId, SignedPreKeyRecord record) {
        String sql = "INSERT INTO %s (account_id_type, key_id, public_key, private_key, signature, timestamp)\nVALUES (?, ?, ?, ?, ?, ?)\n".formatted(TABLE_SIGNED_PRE_KEY);
        try (Connection connection = this.database.getConnection();
             PreparedStatement statement = connection.prepareStatement(sql);){
            statement.setInt(1, this.accountIdType);
            statement.setInt(2, signedPreKeyId);
            try {
                ECKeyPair keyPair = record.getKeyPair();
                statement.setBytes(3, keyPair.getPublicKey().serialize());
                statement.setBytes(4, keyPair.getPrivateKey().serialize());
            }
            catch (InvalidKeyException e) {
                throw new AssertionError("unexpected invalid key", e);
            }
            statement.setBytes(5, record.getSignature());
            statement.setLong(6, record.getTimestamp());
            statement.executeUpdate();
        }
        catch (SQLException e) {
            throw new RuntimeException("Failed update signed_pre_key store", e);
        }
    }

    public boolean containsSignedPreKey(int signedPreKeyId) {
        return this.getSignedPreKey(signedPreKeyId) != null;
    }

    public void removeSignedPreKey(int signedPreKeyId) {
        String sql = "DELETE FROM %s AS p\nWHERE p.account_id_type = ? AND p.key_id = ?\n".formatted(TABLE_SIGNED_PRE_KEY);
        try (Connection connection = this.database.getConnection();
             PreparedStatement statement = connection.prepareStatement(sql);){
            statement.setInt(1, this.accountIdType);
            statement.setInt(2, signedPreKeyId);
            statement.executeUpdate();
        }
        catch (SQLException e) {
            throw new RuntimeException("Failed update signed_pre_key store", e);
        }
    }

    public void removeAllSignedPreKeys() {
        String sql = "DELETE FROM %s AS p\nWHERE p.account_id_type = ?\n".formatted(TABLE_SIGNED_PRE_KEY);
        try (Connection connection = this.database.getConnection();
             PreparedStatement statement = connection.prepareStatement(sql);){
            statement.setInt(1, this.accountIdType);
            statement.executeUpdate();
        }
        catch (SQLException e) {
            throw new RuntimeException("Failed update signed_pre_key store", e);
        }
    }

    public void removeOldSignedPreKeys(int activePreKeyId) {
        String sql = "DELETE FROM %s AS p\nWHERE p._id IN (\n    SELECT p._id\n    FROM %s AS p\n    WHERE p.account_id_type = ?\n        AND p.key_id != ?\n        AND p.timestamp < ?\n    ORDER BY p.timestamp DESC\n    LIMIT -1 OFFSET 1\n)\n".formatted(TABLE_SIGNED_PRE_KEY, TABLE_SIGNED_PRE_KEY);
        try (Connection connection = this.database.getConnection();
             PreparedStatement statement = connection.prepareStatement(sql);){
            statement.setInt(1, this.accountIdType);
            statement.setInt(2, activePreKeyId);
            statement.setLong(3, System.currentTimeMillis() - ServiceConfig.PREKEY_ARCHIVE_AGE);
            statement.executeUpdate();
        }
        catch (SQLException e) {
            throw new RuntimeException("Failed update signed_pre_key store", e);
        }
    }

    void addLegacySignedPreKeys(Collection<SignedPreKeyRecord> signedPreKeys) {
        logger.debug("Migrating legacy signedPreKeys to database");
        long start = System.nanoTime();
        String sql = "INSERT INTO %s (account_id_type, key_id, public_key, private_key, signature, timestamp)\nVALUES (?, ?, ?, ?, ?, ?)\n".formatted(TABLE_SIGNED_PRE_KEY);
        try (Connection connection = this.database.getConnection();){
            connection.setAutoCommit(false);
            String deleteSql = "DELETE FROM %s AS p WHERE p.account_id_type = ?".formatted(TABLE_SIGNED_PRE_KEY);
            try (PreparedStatement statement = connection.prepareStatement(deleteSql);){
                statement.setInt(1, this.accountIdType);
                statement.executeUpdate();
            }
            statement = connection.prepareStatement(sql);
            try {
                for (SignedPreKeyRecord record : signedPreKeys) {
                    statement.setInt(1, this.accountIdType);
                    statement.setInt(2, record.getId());
                    try {
                        ECKeyPair keyPair = record.getKeyPair();
                        statement.setBytes(3, keyPair.getPublicKey().serialize());
                        statement.setBytes(4, keyPair.getPrivateKey().serialize());
                    }
                    catch (InvalidKeyException e) {
                        throw new AssertionError("unexpected invalid key", e);
                    }
                    statement.setBytes(5, record.getSignature());
                    statement.setLong(6, record.getTimestamp());
                    statement.executeUpdate();
                }
            }
            finally {
                if (statement != null) {
                    statement.close();
                }
            }
            connection.commit();
        }
        catch (SQLException e) {
            throw new RuntimeException("Failed update signedPreKey store", e);
        }
        logger.debug("Complete signedPreKeys migration took {}ms", (Object)((System.nanoTime() - start) / 1000000L));
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private SignedPreKeyRecord getSignedPreKey(int signedPreKeyId) {
        String sql = "SELECT p.key_id, p.public_key, p.private_key, p.signature, p.timestamp\nFROM %s p\nWHERE p.account_id_type = ? AND p.key_id = ?\n".formatted(TABLE_SIGNED_PRE_KEY);
        try (Connection connection = this.database.getConnection();){
            SignedPreKeyRecord signedPreKeyRecord;
            block14: {
                PreparedStatement statement = connection.prepareStatement(sql);
                try {
                    statement.setInt(1, this.accountIdType);
                    statement.setInt(2, signedPreKeyId);
                    signedPreKeyRecord = Utils.executeQueryForOptional(statement, this::getSignedPreKeyRecordFromResultSet).orElse(null);
                    if (statement == null) break block14;
                }
                catch (Throwable throwable) {
                    if (statement != null) {
                        try {
                            statement.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                statement.close();
            }
            return signedPreKeyRecord;
        }
        catch (SQLException e) {
            throw new RuntimeException("Failed read from signed_pre_key store", e);
        }
    }

    private SignedPreKeyRecord getSignedPreKeyRecordFromResultSet(ResultSet resultSet) throws SQLException {
        try {
            int keyId = resultSet.getInt("key_id");
            ECPublicKey publicKey = new ECPublicKey(resultSet.getBytes("public_key"));
            ECPrivateKey privateKey = new ECPrivateKey(resultSet.getBytes("private_key"));
            byte[] signature = resultSet.getBytes("signature");
            long timestamp = resultSet.getLong("timestamp");
            return new SignedPreKeyRecord(keyId, timestamp, new ECKeyPair(publicKey, privateKey), signature);
        }
        catch (InvalidKeyException e) {
            return null;
        }
    }
}

