///////////////////////////////////////////////////////////////
//  Copyright 2011 - 2025 John Maddock.
//  Copyright Christopher Kormanyos 2021 - 2025.
//  Distributed under the Boost
//  Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at https://www.boost.org/LICENSE_1_0.txt
//
// This work is based on an earlier work:
// "Algorithm 910: A Portable C++ Multiple-Precision System for Special-Function Calculations",
// in ACM TOMS, {VOL 37, ISSUE 4, (February 2011)} (C) ACM, 2011. http://doi.acm.org/10.1145/1916461.1916469

#ifndef BOOST_MP_TEST_POW_DATA_REDUCED_HPP
   #define BOOST_MP_TEST_POW_DATA_REDUCED_HPP

   #include <array>
   #include <cstddef>

   namespace test_pow_data {

   using test_pow_data_array_type_reduced = std::array<std::array<const char*, 3>, std::size_t { UINT32_C(641) }>;

   static const test_pow_data_array_type_reduced data_reduced =
       {{
         {{"16.5291435510970359246130101382732391357421875", "0.0493652894786225093781695250072516500949859619140625", "1.148521901384917365622791205196683900811417787467438266815171811587512645713503049432985012846475645"}},
         {{"0.272264614486854572561469467473216354846954345703125", "0.0004869544345559033600434606370299661648459732532501220703125", "0.9993666822419621463685898095771200117231983084504331852993667325006778736374601559705472670997199143"}},
         {{"0.11586606975844959688259905306040309369564056396484375", "0.49125326275766578731918343692086637020111083984375", "0.3468689824884565657511674438813148406597435141734916120107707488773463981774893432376041608044139319"}},
         {{"472825.93831413052976131439208984375", "12.2893132739781236750786774791777133941650390625", "5472451570772698890910856961407955272822803346775691694879734192337962.501898438165602619651111960611"}},
         {{"12955.65722553341765888035297393798828125", "1.8152108329941842868038293090648949146270751953125", "29172989.94517362104368895879633524949699872653522942544421141498252973503009334031199616770782242578"}},
         {{"1743.059091451375934411771595478057861328125", "0.01557274597094950030484739045277819968760013580322265625", "1.123249231438217677150386839644175338290573417859310806799523418154085397751304050614658976128613204"}},
         {{"59.4723096329786358182900585234165191650390625", "0.0002479605849140214916814795031996254692785441875457763671875", "1.001013558956642657032539058008208343520129469781752139252898239002622264874758773076477538942197948"}},
         {{"4.7247047167980570883845381491283887953613884747028350830078125e-06", "19.31500975454144963805447332561016082763671875", "1.366138802626438027427562907363958506683175124665868711673216084241378675701086906262585073200438083e-103"}},
         {{"0.005420423329410971324815449179368442855775356292724609375", "0.02511748491388055004591706165228970348834991455078125", "0.8771717031277545026813102041257159416508017211549768638706992210723553351259665651826669250445987469"}},
         {{"0.1038149296564971901801754938787780702114105224609375", "4.0277553943286552827485191841105915955267846584320068359375e-05", "0.9999087696421757883861377789585393001376029184244102849520858740153367697267500475697348167661974098"}},
         {{"3826.35641630229656584560871124267578125", "0.00304696645144304778529686927868169732391834259033203125", "1.025455047189380943721027917279119301765060350016000180536444299910559186963190337307948520430216669"}},
         {{"0.1336724869064791487716092888149432837963104248046875", "12.0701621203230615719803608953952789306640625", "2.826087414359406560867225945615466460996347609181109595659777612088964182161665365619032222262149396e-11"}},
         {{"788.8038549080147276981733739376068115234375", "1.462785614257224864331874414347112178802490234375", "17284.0141730863663391876877521764355960382145920572562742914162972208384723706307292705275033115082"}},
         {{"60.29084450116994275958859361708164215087890625", "0.27315982450996312280722122523002326488494873046875", "3.064030972605630500818558006371664494959621647127192633308100863758834399703265306634456853462288623"}},
         {{"0.5218820792700160637878070701844990253448486328125", "0.10079734355056313432896786252968013286590576171875", "0.9365523254852778069643733313130988383332783304234553907158619732958763498887732678926811658761131768"}},
         {{"6.2117200304199542415517498739063739776611328125", "8.609194350208191940060191427175340095345745794475078582763671875e-07", "1.000001572417065356455596250811328980662533203641329917657490465675369153428136319204096428320190042"}},
         {{"55510.551939550437964498996734619140625", "0.220561294990389011871911861817352473735809326171875", "11.12821772880452010725075040887092862779782492663185103329098479289092263381451855952452939170863208"}},
         {{"2.6413937764460031047686339178426351281814277172088623046875e-05", "7.7853734449875281083131017823006914113648235797882080078125e-05", "0.99917963229955917131492670903076380973630913900537405289588693180767153245890405255614134213988581"}},
         {{"10420.736422527159447781741619110107421875", "0.000949096324860676521240065994788892567157745361328125", "1.008819277616181779650118456445741878037598983308017236099766066425669101291111507814639747211190438"}},
         {{"0.0003315193205662638094299854429891638574190437793731689453125", "5.6913035019043128025229005828578010550700128078460693359375e-05", "0.9995441266953241897466307374569467233610466613450278054078889710230266313399880563926428254787227923"}},
         {{"0.3954592905257090507120665279217064380645751953125", "2.67791671378427392183514677004296800078009255230426788330078125e-06", "0.9999975156798570057614405038860959278154063912444138352916248997973646156538959499028130007448517232"}},
         {{"4.1876570632091016932463389821350574493408203125", "2.1629180671474017572109005413949489593505859375", "22.1448401185251170746394861880464242649255630391558005989017991140563422704674077045117022763467562"}},
         {{"0.0012883408698555164073606960073448135517537593841552734375", "0.003507059232559610328916477328675682656466960906982421875", "0.9769328353726855233701717352870637264167672257348346158491345471096607787937993981690807259720945734"}},
         {{"0.0209720612044679288743509459891356527805328369140625", "0.045804008217355363097311737874406389892101287841796875", "0.837769288971832154015429547694831033608424622845142547611771987181607855769209591840464089347717993"}},
         {{"2.8634840909525142538076536435909247302333824336528778076171875e-06", "0.166147340815298694138846258283592760562896728515625", "0.1199576106959478769470183221201501434061422214693737579245197953451500069777734037303130233690793016"}},
         {{"143297.87334556807763874530792236328125", "1.4168124000936408216598516585094102993025444447994232177734375e-05", "1.00016822776212044529668140515271000711944889104836686684000447554242511520504646895662976301872272"}},
         {{"0.36009865314494415855506304069422185420989990234375", "2.033763392109324996681607444770634174346923828125", "0.125275538744154336055615419863736730746930928726962752118772852620054361808569416481449220736645771"}},
         {{"5.29540201171499924810703863187200113316066563129425048828125e-05", "0.00456136148696097143417915731333778239786624908447265625", "0.956082033932816309081231125453071826327896674917466727856568147854623107132514590973229074834141556"}},
         {{"1.4430180029842585298638368840329349040985107421875", "7.14308392953894662376346236243307430413551628589630126953125e-06", "1.000002619634857924168539271365691986823837938384543197296847349735390391823233061992353181718143609"}},
         {{"0.000622803809629584627372000937839402467943727970123291015625", "0.032509418723248473259701540882815606892108917236328125", "0.7866584695414039980349435008702430694768231324860077519908246946157828936442791737765125963341591426"}},
         {{"985.5536195585154928267002105712890625", "2.1937583537237941300190868787467479705810546875", "3693270.122937088111211449711752633700839101011695532780588598488770867090804195624475853557193488424"}},
         {{"0.0003587939221643085020707619747781791375018656253814697265625", "0.06918288445313602874620073635014705359935760498046875", "0.5776359908891251083096908528191127242622237366936469691316316110676124442824014710248458895266768701"}},
         {{"4961.46848614257760345935821533203125", "0.053691918265817595301570008814451284706592559814453125", "1.579153702119351947233623594416016717758587301144487628569235793702626818145087086748393717196770558"}},
         {{"4.2449562133582249465711677682833169455989263951778411865234375e-06", "0.241273133041338017079624478355981409549713134765625", "0.05056501488889488698113311209419633195414143758068226879838301382059384208622000640785765984698778312"}},
         {{"448640.4270322383381426334381103515625", "1.5754551253198957020025539055296803780947811901569366455078125e-06", "1.000020503146976076594310183596593611837133032127536131209025279043314981116031383943347574180548142"}},
         {{"6497.49570902381674386560916900634765625", "1.6723318686677810518403930473141372203826904296875", "2377807.240798402087110145336499915411094707965040843148351567593442149933028639528071715836607788039"}},
         {{"0.06096743671124205121714112465269863605499267578125", "1.570448414359829655655154123650163455749861896038055419921875e-06", "0.9999956068130994992676343894644386838500410608069083502693463211556728937611305772448191245938179842"}},
         {{"0.0757503455130639213876975190942175686359405517578125", "1.2565992789636808309693127849726579370326362550258636474609375e-05", "0.9999675763402359835297483228657576771861185541700236140502048816577587684411836729204242945686109523"}},
         {{"0.000723094922542264449683013793901409371756017208099365234375", "0.00015995763524068325710236759817917118198238313198089599609375", "0.9988438600173334890921105058993471305519760131474383219703534793262577934372238481044297078545616612"}},
         {{"22234.786633647046983242034912109375", "0.26757431496984462881982835824601352214813232421875", "14.55976509207003348964408642885867745521367447685339701510741492182728518296247645052394313354681315"}},
         {{"1.619697542041219871788992890060399076901376247406005859375e-05", "0.049843364953991387000797885775682516396045684814453125", "0.5770614553354390397780433591724677460252001467856302734078929498603140313062684473758713574169229803"}},
         {{"0.01107858028377593784252752584507106803357601165771484375", "1.1865222120755188430808735322585789617733098566532135009765625e-05", "0.9999465753962580845020188394462388465841107058128678776439244855857791131975120363177283401002854105"}},
         {{"4.009008561509338802904267584636954779853112995624542236328125e-06", "1.515324377685014537960918323822312459014938212931156158447265625e-06", "0.9999811692918874241577599322620672460667530449360861282539703158389898398171499141875816086885658065"}},
         {{"0.00011902135026373703886870547563603395246900618076324462890625", "16.829028854292829464611713774502277374267578125", "9.048524175406200099445660414537194672113902301494985160118195354407282469295381005765523834641739534e-67"}},
         {{"3.24276265467208501549166432820214822640991769731044769287109375e-06", "0.0004454089545421788080836744683210781658999621868133544921875", "0.9943862546876869469633223835521854920759773375319732139308380158382743763220448601371228105610678184"}},
         {{"0.61581061656620050825949874706566333770751953125", "0.031661505629107911108377493292209692299365997314453125", "0.9847672124420937304378061151264704452280334454617800199459967125384302760082293767621207779781835127"}},
         {{"49863.9800075090606696903705596923828125", "1.776107201703168634557457317146855757528101094067096710205078125e-06", "1.000019212432382371931838274893111705064467179933990565372823840270625261083378934886643803918491395"}},
         {{"1398.294781314763895352371037006378173828125", "3.434995305831383370642750918477759114466607570648193359375e-05", "1.000248827963445913854612398808452603846917768513070673763212128310433301197689262611526019784745353"}},
         {{"0.055382354272440259546073093588347546756267547607421875", "3.89513503025480504506072865211763200932182371616363525390625e-05", "0.9998873008428618351640174181134295599308044680190697986054820881419836652178154795031663413208072908"}},
         {{"141121.3613738338463008403778076171875", "0.9987541879788022214370357687585055828094482421875", "139052.0297965050871747990365720608342903396699362315030158719313557217224631617472077499308065163324"}},
         {{"2.89668134268413328424353336831842398169101215898990631103515625e-06", "0.1158653454600215848557809295016340911388397216796875", "0.2282055561174114081136528745643753299334234249353996060709751881650135493460434422720134694935941763"}},
         {{"6803.7045226678965263999998569488525390625", "0.0025558566754669553855450203627697192132472991943359375", "1.022812314043596121515273538979291803484787443288203330253325410708073327046561555403765599853129459"}},
         {{"4.62504113144313121035178237860918670776300132274627685546875e-05", "0.00015216342784673988867971683447422037716023623943328857421875", "0.9984823426526434719984718479182767296211152915276554074783473924406468107236717283071375597923894695"}},
         {{"45.08229321595757710383622907102108001708984375", "0.0010683947047554244458300587439225637353956699371337890625", "1.004077259574835588182392208924249486928765616120776910600071448009954803524517295090312929200218436"}},
         {{"0.0029740331900898926698051383255005930550396442413330078125", "0.386474774881857552344399664434604346752166748046875", "0.1055630406196574754240236326175534581442212606474717573494707696002632840688562630659839194855847224"}},
         {{"238.325080784211650097859092056751251220703125", "0.0016630246581186870769020202942556352354586124420166015625", "1.009144347417356333834779561466274825361713669580543328405511817842003446035824007315229599805438482"}},
         {{"17.142319020137080087806680239737033843994140625", "0.52277645345308965119102140306495130062103271484375", "4.417153734964563885393248108584096001949778638332554357826430578812952004868724813707271341102615228"}},
         {{"224202.18375039077363908290863037109375", "0.0002886061371389233264073137519289957708679139614105224609375", "1.003562044264646513183813533667666562751904638315035222395136521282950928976809498754331537084019662"}},
         {{"0.0668254924276967532392745852121151983737945556640625", "4.37453434835288135218434035778045654296875", "7.238842896154449769710588856604377874806273521375624274206312478810494372919421488420292670076863701e-06"}},
         {{"0.70873598960350125963714162935502827167510986328125", "49.3170577274373727050260640680789947509765625", "4.229945811788804284305883612198338835404634886754741043233539075117645045396251885162888263540933705e-08"}},
         {{"2320.14942323958166525699198246002197265625", "0.04448103101311928764260983371059410274028778076171875", "1.411567399249410711442221139916306831946201981464276523189788129459516875922174398176653851680046602"}},
         {{"185718.77773985941894352436065673828125", "5.2276076462150181827816253221641318305046297609806060791015625e-07", "1.000006342147884645778551894570844792315108121857671940374415817279755614698877235383412414884768484"}},
         {{"22.650946549776136862419662065804004669189453125", "0.010376251017939763787722995402873493731021881103515625", "1.032905800190428231297979472183774003203493095798873796480692196530445214963120221396333959754479731"}},
         {{"9707.988966673219692893326282501220703125", "0.0044424598045826046988082680400111712515354156494140625", "1.041628038100866990198113594516054534923271992036375883334430870167902365470760016129830249409692642"}},
         {{"4.6638922458981375666897672971344945835880935192108154296875e-05", "8.4159868321997493012531776912510395050048828125", "3.533820421397917895904598910743400667020601203836600589817073241704321977915466359845292805768774321e-37"}},
         {{"0.003598084156336257832808911416577757336199283599853515625", "0.38910607092369620119143291958607733249664306640625", "0.1119573247957651579017030299380639166628940007319959961969216015055195230326511521371482830982422674"}},
         {{"8.2954611281051593603408367272322720964439213275909423828125e-05", "2.583244664993120665030801319517195224761962890625", "2.866596582887405651220009597212702843049467204795792716515501036381612053674953716074433532572198064e-11"}},
         {{"0.0863334593891555979183749514049850404262542724609375", "1.53080323158965114771490334533154964447021484375", "0.02352337299735458698526970071607567498129753394976810402829990179950323826085507292015804227752201113"}},
         {{"0.049680253001626095699094776136917062103748321533203125", "0.00702519683572433439877613636781461536884307861328125", "0.9791301741310747126148247076064614731964243335674114097436091164147518161304893318572316845978174147"}},
         {{"12.470203355485182328266091644763946533203125", "1.71070595755497723185201142115374750574119389057159423828125e-05", "1.0000431678947787427762026348961382076965392439642922192519806552783214415214585515533021397145639"}},
         {{"8.1413098978146331875793695331822164007462561130523681640625e-06", "0.547674360353472167162180994637310504913330078125", "0.001631990959658180569038596683565699036065846193522934510105885634318076353769860549058978737841093877"}},
         {{"0.09483160475626417618144614607444964349269866943359375", "5.87318705342295180847622670938790179206989705562591552734375e-06", "0.9999861649076950494466991479999705033542605530009538398782751590764582617554287307635602166318516425"}},
         {{"0.000672439622398628665533859560810014954768121242523193359375", "7.25371945127648842799317208118736743927001953125", "9.74287216286274945633266572714083322962101975717027023442493643122162512912802620909159732325958549e-24"}},
         {{"230442.30926587688736617565155029296875", "5.10077615004599603520940576117936871014535427093505859375e-06", "1.000062985121897805334266213272922312846302403942863151288681766681803961566795057652316848688638476"}},
         {{"281337.2466064565815031528472900390625", "6.763448440708715210523127581154767540283501148223876953125e-05", "1.000848990990597377729138593655561180275810544072859547583290559905295293775110664558207907910918715"}},
         {{"0.0003349835238376486744316107291297157644294202327728271484375", "0.00320852082105692326674528658259077928960323333740234375", "0.9746539907825092435508361832010758233691591555196928467983144369306303465726901782454240148137116827"}},
         {{"117.365003664101777758332900702953338623046875", "0.378053489600925995972602322581224143505096435546875", "6.05893503437891990856263354837986270829833882333366791751057714437977103925035624856886729661425365"}},
         {{"120953.01074229204095900058746337890625", "0.18152862099916688976009027101099491119384765625", "8.368360821888521373891758043880742362137628148881451996189831654213425203181451205427193929634407006"}},
         {{"0.000502826864423656406943852203994538285769522190093994140625", "1.4914849761411505014996325968468227074481546878814697265625e-05", "0.9998867241849915062668222943526629142567162657028524506413684902274021187319110161822285235184858017"}},
         {{"1.776236057064855817892723433004675825941376388072967529296875e-05", "7.1688155984379040887299883166861036443151533603668212890625e-05", "0.999216151571584206550599285631558723279674057106881165894202136106399680298454020164284514110193651"}},
         {{"54.44707529935072898297221399843692779541015625", "7.894249943586238447480220303820175331566133536398410797119140625e-07", "1.00000315551756552929413349733123431754994533967419962917545562731445729245526256276165331606875912"}},
         {{"347.952865539182312204502522945404052734375", "0.00170647055379554400678099312926860875450074672698974609375", "1.010036410355161990037298717291721004727226837585799804065980035873230812409114661337977454663213752"}},
         {{"0.004896832471317287549528174395163659937679767608642578125", "1.96528625490362656162875165666292787136626429855823516845703125e-06", "0.9999895463694024884235941270721509764132278456092084650402242198052510524655263500382030354931840136"}},
         {{"1.4684657513408354290174173684135894291102886199951171875e-06", "1.139612957779291946280864067375659942626953125", "2.25155431004176641664360564432608271749851863826540763464893400702786658931519008803678187122457752e-07"}},
         {{"0.492258328635518438431972754187881946563720703125", "0.000216701285749932963387021089829431730322539806365966796875", "0.9998464244018230722113568274149981105013353753491598111618200980618319549421110550073126604159286363"}},
         {{"20.8891842024441984904115088284015655517578125", "0.1374511235522046792567607553792186081409454345703125", "1.518534594911014994620281531983079604640365293218624267690664862071661712964941704981795831350348551"}},
         {{"6.5969395189832624737391597591340541839599609375", "8.583905042851172444151597318295898730866611003875732421875e-05", "1.000161957566883307614914742736151364235525566145770361977557429966893360015123605952358900004419822"}},
         {{"47219.0302282203920185565948486328125", "0.1368726147956442762421147563145495951175689697265625", "4.362732893776740035495529565964282960068681945289024911729313300439165729337845954958245855833420794"}},
         {{"4.82034977757478931038870317848932245397008955478668212890625e-05", "0.000120114163951150231847420091213507475913502275943756103515625", "0.9988067681926094116743372676724240569600832324642713602291135420189602335726960082961281503383041726"}},
         {{"0.95419795742564250673467540764249861240386962890625", "0.00033430414202346991903169737270218320190906524658203125", "0.9999843265651398037729389902364010581174316619220455738462203806403496998327791230980585407530086229"}},
         {{"64.0960699263738433728576637804508209228515625", "100.8167320498315575605374760925769805908203125", "1.441575044221233060753076802461152797802690107379511733929223209030110262542612248107484347490333928e+182"}},
         {{"1.2231158900984038240267182118259370326995849609375", "0.0002104693491577123119762315894831772311590611934661865234375", "1.000042389764425264027879772573355015381861476260087805696928869643277269672508395569882503630440759"}},
         {{"6392.911184302080073393881320953369140625", "1.2839197583067516461596824228763580322265625", "76950.5481296349945509651870173004703548467117826619443587513918127052944692060859604078331125022112"}},
         {{"54.7317500559057634745840914547443389892578125", "0.01255694019060092647688264833050197921693325042724609375", "1.051542832171497399963223940731327321102859731771016721444762435367924032025546418413038216816741786"}},
         {{"0.06491238363825491131109401976573280990123748779296875", "0.0185020261717664669109950636993744410574436187744140625", "0.9506609466703063119143652794820950051733540412105718545771282443795835031411321586486677931223149025"}},
         {{"2.246927054764828106218910175329028788837604224681854248046875e-05", "3.5298269571832179281045682728290557861328125", "3.907630139589162503107039226285373953461442486143835000400961383754310391040020493675853907386713543e-17"}},
         {{"10377.631193883877131156623363494873046875", "0.0036775846103273106268982672872880357317626476287841796875", "1.03459301282359622046275858130909175858605738103500377882461906122638713902728757596922241637837678"}},
         {{"43.2141091390930114357615821063518524169921875", "0.017996255362703272862034964418853633105754852294921875", "1.070126540716327753325568534283574566351664811377536504016184144532280877519428694566691801123755073"}},
         {{"0.08851953952093294386571642462513409554958343505859375", "23.20149726040170889973524026572704315185546875", "3.713057134638627172363216136468936639263890707646199045636218247428562378896054856017802421856667151e-25"}},
         {{"5.11800358239917845513955452219079234055243432521820068359375e-05", "0.025319462580208895730038420879282057285308837890625", "0.7786758988158412514016037505778567567959504412454447228734522688964786089826834589419265031329115228"}},
         {{"0.0002327414661490319856829156464073093957267701625823974609375", "5.38640739832896395418437851532189597492106258869171142578125e-05", "0.9995494971625504384583044141613339207017817152688551450620338838829007501118195512492732528596639463"}},
         {{"288820.917341887019574642181396484375", "0.003726790793748768759741096801008097827434539794921875", "1.04797427152199505899776613380207153019163891696040089708565511495563600355422445163531206665239877"}},
         {{"2.999020797850647568338899873197078704833984375", "9.68497592645624978853480246954177346196956932544708251953125e-06", "1.000010636928440882104560024198550740111480054700791454716347881126282957717881908071234242335247598"}},
         {{"772854.459904651157557964324951171875", "0.0004489783485212080378634613708754841354675590991973876953125", "1.00610574384563796400321310101756992783528974702739913146260646865660943239706104030906315422031587"}},
         {{"92.97703728773058173828758299350738525390625", "8.352618724617581237590346887600389891304075717926025390625e-05", "1.000378641794619964909754016025149005775123619289098101351702622832114425001063348854464235953144215"}},
         {{"108.5939270419715967364027164876461029052734375", "0.0002542053285091486723723530616325660957954823970794677734375", "1.001192327091849506573473559005299436677765819365613330296428443229316518519229873966880704235135164"}},
         {{"608955.210127030499279499053955078125", "3.939976416002167525221011601388454437255859375", "61820121977766204913334.85449296850381452947984612338986694663682717269174497412093745623769088818053"}},
         {{"4.17002058199641822966441395692527294158935546875", "2.27638775324770278722132099691322082435362972319126129150390625e-06", "1.000003250507095081111088653043421782159546844144905445742011555455512255801354882591632733164942472"}},
         {{"7.6755008159808189686923451944267071667127311229705810546875e-05", "0.309358860541564251178670019726268947124481201171875", "0.05333610588887033129378772664097141008347485829100041029987795210401159444403597242199668984657171582"}},
         {{"911928.851435624994337558746337890625", "0.00077715027314583136341941127511745435185730457305908203125", "1.010722154435652974952128757578284311501734167865940779900112832536116840771719618579776585557729932"}},
         {{"54.14201226459181270911358296871185302734375", "1.895225928825032568093844620449317517341114580631256103515625e-05", "1.000075652897786756529841409793213990627577913714376036511252521795945865387284962808777954757182543"}},
         {{"0.08008799218859896651423468938446603715419769287109375", "96.5125145484174709054059349000453948974609375", "1.515421068819814917314894642087225594550802837295383163299780706932605853784153083953735585367023073e-106"}},
         {{"10.915699736235040973042487166821956634521484375", "0.0081087806268633766393350015277974307537078857421875", "1.019570667476289029409663025100929604885289252535402211726634712755034009907717895899166040685843344"}},
         {{"5.27981737437816178726279758848249912261962890625", "0.74197085209617963386108385748229920864105224609375", "3.436862532412935025430311642356805642804931139344257022905862596726552299194156344586974046879151412"}},
         {{"812277.0076296366751194000244140625", "4.92673741168140821855558897368609905242919921875", "130484996489149406381687564563.7794251784186920096700072441713213248888591234917788571602696263102591"}},
         {{"6.75063843580995381898901541717350482940673828125", "0.0010274848910674962143563249128419556654989719390869140625", "1.001964049474161743650304200060162036480605659559348069572061116378564324551728952016517278929585866"}},
         {{"35017.44441640540026128292083740234375", "0.000780282755451220007281154522615906898863613605499267578125", "1.008197988889011286296323179405657420325810447873105102983940512376107225479257794660830578523359308"}},
         {{"12.65649508534835376849514432251453399658203125", "1.55352677703511933278261725188684749809908680617809295654296875e-06", "1.000003943123655399895636254863316261379642738118347986950057800134957605381772171008050726735961906"}},
         {{"0.0021574588239726168181764620612739236094057559967041015625", "0.023790568334886519341608845934388227760791778564453125", "0.8641178516496138507695146254979778802789160783430375715424529106996086391993347162420872570336798134"}},
         {{"0.104145372167880889691105039673857390880584716796875", "8.3780218232517285059657297097146511077880859375", "5.885268524937381322391138731665392032631386720581279557069512621951537974730387076011012371960708863e-09"}},
         {{"0.30701255111511027706683307769708335399627685546875", "0.268879194147396471947786267264746129512786865234375", "0.7279590568113293433926640002454318162562521255013701906351083047350684147268690092539450944047474636"}},
         {{"38.78555527474190967041067779064178466796875", "0.03681239530031188422753984923474490642547607421875", "1.144149429906926564523417841257675903530642609072484678477405710161350499283211892611266100280988518"}},
         {{"0.007260045818669978723303159995339228771626949310302734375", "0.3064134107450371402592281810939311981201171875", "0.2210869630112627815579629160391624889926007891927175421314344521536011491706232791858074450111405922"}},
         {{"0.041128842235287788042796819354407489299774169921875", "2.51736412939929010690320865251123905181884765625", "0.0003245652885637518223312999726031794885503494683700671444467518988526440026167092162074081515065655893"}},
         {{"383413.786726422607898712158203125", "5.00384576840691153165607829578220844268798828125", "8705868811002419093650862818.281066153214377847175575998784043257176134045177228420638621511164324213"}},
         {{"1.4528763881028615607970966294715253752656280994415283203125e-05", "2.1268575193348029819633004944279264236683957278728485107421875e-06", "0.9999763084062013822965111956774540282139660505593869490610011761004586743065357666580757276498588619"}},
         {{"0.106060778784217379921983592794276773929595947265625", "3.665385550434487223203650074765391764231026172637939453125e-06", "0.9999917758507777695948054646821023920957663887249980065452828857430630655098402880390713539115218718"}},
         {{"5.880863770900304592081919707879933412186801433563232421875e-05", "0.0002289504148239272784959297268869704566895961761474609375", "0.9977722283995571159178975955531232000596177455253986310367479027162218252689170591308771482146913125"}},
         {{"1.8208843744294203048639246844686567783355712890625", "2.1817081930971879010112968177992343044024892151355743408203125e-06", "1.000001307547233783310033281085230776608904800861593774004217709161290602992963371338054933133287171"}},
         {{"1.817908642136924157739485963247716426849365234375", "4.11112317505943991591266239993274211883544921875", "11.67166160892608991586349276913114769402646894242708300388308034827618167791926644671131699865907169"}},
         {{"311682.2136415620334446430206298828125", "0.0029409858890547639631574128316060523502528667449951171875", "1.037903387806112408213643940961876890091514423377231276258839970001122248238515011889468127503204249"}},
         {{"0.00139087864126595343872150323250025394372642040252685546875", "0.3301027238081264414404358831234276294708251953125", "0.1140228773967218318422233460463354244702468238841374062337508591431647829100522731204529091592944248"}},
         {{"124.917144681230411151773296296596527099609375", "11.5140916555233019380466430447995662689208984375", "1382605995137734277459273.978414530423382482424569885174581477730195687046588538795159600040764022773"}},
         {{"3118.36231019214392290450632572174072265625", "0.020774588405401417379181339128990657627582550048828125", "1.181911304360897645398752760668243836000662781777979762687942316408666987895311156678567898287617268"}},
         {{"6.666206185963261649472799685867130392580293118953704833984375e-07", "0.0092127351538278945408677600426017306745052337646484375", "0.877204862888219369744788573660770287170392769001738688402155865708513288003188496251740638940642993"}},
         {{"2167.710504329414106905460357666015625", "0.000111416271213217389808580737309284813818521797657012939453125", "1.000856202266224709057328444179057579634630919916314686750456648876046969347391505143817774702119046"}},
         {{"8.6612927641114873140992802547089013387449085712432861328125e-05", "0.0026323047616202653598005412050042650662362575531005859375", "0.9756779263160872019929374341041722126351834020818453670509619073869829005288453573368554330200277019"}},
         {{"3.35057580340428966085564166377253059181384742259979248046875e-05", "3.16747650803829827736990409903228282928466796875", "6.697627268189159941026366696402486416952589165161782209516153325413584031698980897960855341082161816e-15"}},
         {{"6.53381930771882064867828177057162974961102008819580078125e-05", "5.88128239916411388321648701094090938568115234375", "2.442395769062951722951610647289013539750926290515902190768003779007138374032841367834312629774336669e-25"}},
         {{"9.804625448072497480300757644045006600208580493927001953125e-05", "0.1923800618186211064397639347589574754238128662109375", "0.1693679572154173665347950324140920449278981507986644695238195778874850524989755063536822290880318355"}},
         {{"4.22715824828159014714401564560830593109130859375", "173.23747338374550963635556399822235107421875", "2.851421816256107487275129676156086236423190898231808521165673387751554806253971407146484062288612475e+108"}},
         {{"88.7031657211726951572927646338939666748046875", "13.4070958469201997331765596754848957061767578125", "130679204065692425620221532.3219164101202809776658910189350996926779031428363657937133100741031831675"}},
         {{"0.00013889480019387587077239754052015996421687304973602294921875", "0.8288343702502256604702779441140592098236083984375", "0.000635222056499059860815119534190744388049132560559707635179974362835024292102863415735162094842301917"}},
         {{"1.0981694739727588497509025433629403778468258678913116455078125e-05", "5.5818394589962999044648050972483588338945992290973663330078125e-06", "0.9999362614393042122695937932746316210666785868979535945104734825146882672551743932435898719139400831"}},
         {{"0.2310969965168656603538011040654964745044708251953125", "0.80554981437270978261722120805643498897552490234375", "0.3072590032651657942319746509324710103251974110801147219043014571575470973343135677985348078823467414"}},
         {{"0.00849670416441904163828979790196171961724758148193359375", "5.0699807966672811622432792688641711720265448093414306640625e-05", "0.9997582886318732732262821458857417397574439248050273850965367005483419552233943883307385558084138152"}},
         {{"15731.13853871330502443015575408935546875", "3.28071775496995989116096048565651699391310103237628936767578125e-06", "1.00003170338187821543987107336746662729109295981087069294808638827112958850528655867141095917778641"}},
         {{"16463.485017001978121697902679443359375", "0.00158052312996124176558421225990969105623662471771240234375", "1.015463482525033628261071051983336009161238856163519297590950903706756838078533976985376710193485235"}},
         {{"1.2998477112373318042258529825261348378262482583522796630859375e-05", "5.48673306853607149757845462545446935109794139862060546875e-05", "0.999382895797564405052408771210317692764125583490813721738650031785719409851796983570805688722349823"}},
         {{"45.87692666851609146760893054306507110595703125", "0.000694008884841098259987912655333275324665009975433349609375", "1.002658780135145192968098652957058682173932468672255592456062271728932235226675835947243003223564883"}},
         {{"1.83659034529005538161072763614356517791748046875", "428.71771981883512125932611525058746337890625", "1.537269307679801346283514140400021636940049375383018780714923813066372654035932489743906872387459361e+113"}},
         {{"8.696561971106747429734185050875794331659562885761260986328125e-06", "0.01534705324715328333962105489263194613158702850341796875", "0.8362457008948993885558567754609553940486417423422233782035811094457903215446368575028693794076229319"}},
         {{"0.047786641269933871623010190887725912034511566162109375", "0.11684748733218441874015525172580964863300323486328125", "0.7009390954403092913368478024883128953850044300537727516422122034970679248708602695681663211486432475"}},
         {{"1549.637991964928005472756922245025634765625", "0.021120496095707597117296927535790018737316131591796875", "1.1678289731817137799033068988640290335505962819368411413282568405187575760779768993349107629918597"}},
         {{"247000.6794711612164974212646484375", "0.00152499082730789443929086957041363348253071308135986328125", "1.019116458045280602852950073655229019332443535712541084137763626429128356366092297333788770248289013"}},
         {{"1.7985639528641730322533476282842457294464111328125", "1.320591155767633669677017793642193055347888730466365814208984375e-06", "1.000000775172178238369352480778498926912998479034242652385246624176410276010521472711700020750346132"}},
         {{"0.4413559373541460217893472872674465179443359375", "34.26410181441593749696039594709873199462890625", "6.745485550365195207912762013228453058437669951258825333667587930483550435447779747931307180579325826e-13"}},
         {{"569.1719918299231721903197467327117919921875", "0.498043102542145899036540868110023438930511474609375", "23.56297005950916383958322134808882017270861535677566385627457442473645770787327728883910635718006969"}},
         {{"5.7695435601138136440246828584577087895013391971588134765625e-05", "0.5251118729145645147582399658858776092529296875", "0.005944633044458701558060258596677375842170297746590900912238651327061412040117935683740813836864974303"}},
         {{"0.128163238607435392424349629436619579792022705078125", "116.5241021552538995820214040577411651611328125", "1.078797195749260164944907474153237780121997055933948319435819382967774774883124253235060971546994592e-104"}},
         {{"787.202251453049029805697500705718994140625", "0.0639760572205021471603458849131129682064056396484375", "1.53207555977032989055018208938771579303277505887799197079806149426805566503858217166434829573392591"}},
         {{"185.413190872573295564507134258747100830078125", "0.56940684669212071611354986089281737804412841796875", "19.56567353686150675925228595129550396435748911985045493799983942843208146519455842649603155059750083"}},
         {{"0.0039995945084219941545455867526470683515071868896484375", "0.2867095783501181216479380964301526546478271484375", "0.2053406857179062707546056095305019324494433074745195535995249540216710590121769978916047343839356539"}},
         {{"0.00018467413957277728671357852618939432431943714618682861328125", "0.0225571659299317051061706251857685856521129608154296875", "0.823722070622293990830799081708429380180065314114356604329745073190591014971022286949735867992102291"}},
         {{"0.037126937469209309039541722086141817271709442138671875", "5.32704198530596029974130967143963744092616252601146697998046875e-07", "0.9999982455868751394899012716390283407137790591300270152767664184112832940065376278332550215728459387"}},
         {{"0.017010855343605968403863926141639240086078643798828125", "5.166075046837814558654822238992210259311832487583160400390625e-06", "0.9999789541297926226446500322690945743196319757774384867549036591509624640821181229454194105033231727"}},
         {{"209225.83666716678999364376068115234375", "0.0006856693744833435022467682529168087057769298553466796875", "1.008435632846997545287129615964500170354538603505885704100598710120112533798139336545294290453014345"}},
         {{"0.12171472754286838391379887980292551219463348388671875", "7.0402427498167407561580066754913787008263170719146728515625e-05", "0.9998517381802859896190137431226806202293604767383836723754572235961420245380325823341413637016852165"}},
         {{"5.07736820771065972511537633948819348006509244441986083984375e-05", "0.0039708981266413367894330121998791582882404327392578125", "0.9614961035315242921452699699043335081883641397081373292087590773020046510608038609970692318461465254"}},
         {{"1.0510709502402786341690443805418908596038818359375", "20.341134777936787259022821672260761260986328125", "2.754356184406962364238477587850392171916017873515490360228313989693788806443445517673284635059496913"}},
         {{"651.7447481409626561799086630344390869140625", "5.07710216336688000492681493991398156140348874032497406005859375e-07", "1.000003289791435074156856978114420962533632912979557206363854960271640546882941937625664821307458521"}},
         {{"0.004638844916386275496478219793061725795269012451171875", "15.8892435788971937427049851976335048675537109375", "8.337152608207913172699870647474123918003098499556020332172254094814005786678582659092626579399407382e-38"}},
         {{"1.490582857957906061548147835793542981264181435108184814453125e-05", "0.0001703175182588708390374332424244130379520356655120849609375", "0.9981089226220508270875464612988226708429730924795253226917599524755406602728683953306554288284490336"}},
         {{"0.75031526521941938057125298655591905117034912109375", "0.6062225177496873840254920651204884052276611328125", "0.8401754196895209649374453140513075651089614456936947230050433118609736483660685202570182857755328132"}},
         {{"4.26847036788872134273342684451790773891843855381011962890625e-05", "3.25843044022171965845742491740821833445806987583637237548828125e-06", "0.9999672152858525151920733955401379504825545405026689582383105759104833218794819526989280914936281645"}},
         {{"2.62667296496937296979012899100780487060546875", "6.9894343279332618069921156944701579050160944461822509765625e-07", "1.000000674982490937965541909063351504845370079053338058530847985220293630046608558468592618684821802"}},
         {{"6.70966669446914210084287333302199840545654296875", "9.163103845632905412876834105873058433644473552703857421875e-06", "1.000017442571818433691255569141649860657673777209996408887057934384303850271062350141130940557381867"}},
         {{"741203.37189983017742633819580078125", "1.08045243497875006967630373555522282913443632423877716064453125e-06", "1.000014603534503901097533202018489175796342895149378794844072588480048907447761699445979981752501175"}},
         {{"16.91592556379936240773531608283519744873046875", "0.00618026386927718329733494329047971405088901519775390625", "1.017633023482456494154402220140577551252885117519744394941185312384952123430847163565909001351504715"}},
         {{"0.47146052924976888931496432633139193058013916015625", "365.18882781649654134525917470455169677734375", "5.57039862156640741370099717884062397571736028728524050580037613318788846117662035095375103738860527e-120"}},
         {{"0.664982363517832908428317750804126262664794921875", "0.0002041248598992936695573074956655545975081622600555419921875", "0.9999167215946747473766005691888054036107492028791455873847974479348762985533914604402176144508116265"}},
         {{"11689.34417440040851943194866180419921875", "8.156786068593902427892317064106464385986328125", "1513888857055210920826764629553133.037709640424182548779072525079642344209160801037812701093300791238"}},
         {{"0.000976400625826371292015348757331594242714345455169677734375", "0.186442986632989526896153620327822864055633544921875", "0.2746232247983476012325655926919797110455806923306415679947562602008147400606696065465900622650747085"}},
         {{"18.786848284365163408438093028962612152099609375", "71.69298727328123277402482926845550537109375", "21202811937149403143033504464556470855614517606530464717380845404469701995016794105028347981.82529255"}},
         {{"61095.69348919205367565155029296875", "8.61245994668030334651120938360691070556640625", "165700792854128829481158381754556960783201.4464676038743964777536498999871997260503817641545878415793"}},
         {{"5.12940713132187653400251292623579502105712890625", "2.28564435019894255014284711879923861488350667059421539306640625e-06", "1.000003737012829737475209306551597962290969215415894639528026938488254417660234801188172243668282542"}},
         {{"5.969195279751538329537745297681539113909821026027202606201171875e-07", "1.571805821061507057946789178703994593888637609779834747314453125e-06", "0.9999774739444837618728283875810661840309688437463033387166359671280514829679268856041871202984683091"}},
         {{"0.368606214248608399230988652561791241168975830078125", "1.4926003796634690801332923104638439326663501560688018798828125e-05", "0.9999851035654931134844759221839998810892077896067423506985675166718233939259812399432594235564445823"}},
         {{"2.75108139709647048221086151897907257080078125", "45.0370430827189238698338158428668975830078125", "62220768465415809007.21661935576436411494819756526477017085303992773636646950643516106804685045572259"}},
         {{"18.1162455673292015490005724132061004638671875", "2.55260000441009051353087599967039977855165489017963409423828125e-06", "1.00000739442221122260724005033635770725844741914207974197964181739354726511926916714698090584379202"}},
         {{"0.89649350087997081715229796827770769596099853515625", "856.8751575411906742374412715435028076171875", "2.181864727587146157479355326189386951695875227256349943237647633613225418753384943792710053789813837e-41"}},
         {{"0.0001266128798089794817161024553797687985934317111968994140625", "0.0049087934026286195177135596168227493762969970703125", "0.9569028964817718791438773796826385278829018611902298197890427925469792383670543313377674982547416051"}},
         {{"1.05179746109396965946490565269044736851355992257595062255859375e-06", "6.593741007182075432313528295935611822642385959625244140625e-05", "0.9990927826623803247291636406553138487147561016481283476617607554723098851433984723029560678557101908"}},
         {{"2.266779318021509226355192367918789386749267578125", "5.01284500605095928449372877366840839385986328125", "60.4799524981909743425238268974274974124170089470250115095198088233403741098790024678291238778471336"}},
         {{"1.4764538383358800643918584682978689670562744140625", "834.7704459624865194200538098812103271484375", "1.818649531708333660010299576864570804797453892293431835079961872702560120321683477534406912892951636e+141"}},
         {{"0.6836904605625750974695620243437588214874267578125", "0.0038353076719810398620413849357646540738642215728759765625", "0.998542687144721504190991783229244229904437024428948393594123834827455412276977464755147963505927619"}},
         {{"51.52328028312382457443163730204105377197265625", "0.00360510257181055615571807493324740789830684661865234375", "1.014312898539510025325063776141689909480512803378866074368784945276456451477238596369403859954193562"}},
         {{"2.521039504810960595222735225462429298204369843006134033203125e-05", "3.045725696001661572491492346870245455647818744182586669921875e-05", "0.9996775628167446617235251536271244420529642317938757303249170631679100236321863238285968519208751561"}},
         {{"6.69572868673203203115917858667671680450439453125", "6.167632712180198752312254573126892864820547401905059814453125e-06", "1.000011727636192130333908634691625149237871370977312574783798584965289555837307976249907212704662925"}},
         {{"390.61115489594385508098639547824859619140625", "0.0020939281973425087624018914311818662099540233612060546875", "1.012574362389893644367400537857807489848777197110412493037945306776167208919946153350932208339401151"}},
         {{"18573.8968908478855155408382415771484375", "7.349004490823213796291124566550934105180203914642333984375e-06", "1.000072239740530297232713500392594121416360698064253867545090980255491210508376806927109432066846867"}},
         {{"17607.06151766344555653631687164306640625", "4.7488937086893798777964548207819461822509765625", "145316765019878240189.6024036804272238187703352232886081463479510949913051630367999744485766469141637"}},
         {{"8.5925717696667546115218971891636101645417511463165283203125e-05", "0.923640678060561270967809832654893398284912109375", "0.0001756267910550798562224123350796693807963976275000020198269216155919577246366381890515376789394381007"}},
         {{"1.183733667440531287808579141707099324776208959519863128662109375e-06", "1.673262776016292891260138497866449824869050644338130950927734375e-06", "0.9999771655163653186341540130980626760514829363406006765081150182863711743101942816649613883004847016"}},
         {{"0.002792970468375434445196248134379857219755649566650390625", "0.0002971350893620663784211455293871040339581668376922607421875", "0.998254178388410952430693242810343707992963411946313883778028696826023938537768767830025754426281554"}},
         {{"33.811385576253314866335131227970123291015625", "3.09301666014966266347874235265180686837993562221527099609375e-05", "1.000108904785945247090677399080223689170881328124499628763991659972119464383045048992914873178194723"}},
         {{"2.064259300831157162292583961971104145050048828125", "0.62232006873287470938294063671492040157318115234375", "1.569943813297441066451644012496390561641298501940151915247742357213493107493003456522297620558009403"}},
         {{"0.134087695405307183449394869967363774776458740234375", "0.001482734186166380363491867910852306522428989410400390625", "0.9970252330696136455774521556677077004315294051190460034593301547637086169241393339787119475172869104"}},
         {{"113.7958298375000367741449736058712005615234375", "2.03500201236379312419617904073021463773329742252826690673828125e-06", "1.000009634571897927480497447835096533036674313263648698184837048429483513806108756529794379886827463"}},
         {{"1.437339435152241756861457577798546481062658131122589111328125e-05", "2.3221483639344313587571377865970134735107421875", "5.690289295424826479104623331628579101889857848764075423876629317012952221842161334074206085577851636e-12"}},
         {{"140.0442727893359915469773113727569580078125", "0.0035374410478014872705188054169411770999431610107421875", "1.017635589786818103894055730881951384518985139460304208887561168735483569015400682728025954217975149"}},
         {{"0.03337706156360675802119430954917334020137786865234375", "51.63406910592811982496641576290130615234375", "5.749328791036172632876992694412305786575397522775091004981219928863634929275346548792301721865577718e-77"}},
         {{"4289.9626426868562703020870685577392578125", "0.005558434827351514673221544171610730700194835662841796875", "1.047588581601487495732076938650976112586825320535806447417083786076262995717414547814890711633319615"}},
         {{"0.01281103445960631026512288599406019784510135650634765625", "0.034341893135387768776212169541395269334316253662109375", "0.8610152801625812596116030412813122197147506720757123795196255207488379002828150935994696571024493295"}},
         {{"0.005166725179469845719015808072072104550898075103759765625", "1.803811850328186294702310921156396261721965856850147247314453125e-06", "0.999990502044552423124036840709042864832582112050272720529614683113869481837430803897286965523406403"}},
         {{"127899.90731678321026265621185302734375", "0.0005225360240903274833268454813151038251817226409912109375", "1.006163418993335513593283613927521623445772644672444725863707494143018405549611738291213464423688448"}},
         {{"71173.085018835379742085933685302734375", "9.6078807949550562757246030543001324986107647418975830078125e-05", "1.001074052413223820599539135968763334354548118716691311659665664286740695916640178540166495661177509"}},
         {{"10.542530559725719285779632627964019775390625", "0.0009412115616892306746432694808390806429088115692138671875", "1.002219405525173874569049409351536687686778034380180285870248923410852975495617106988737743786703299"}},
         {{"0.00020661668683597737665447180432920504244975745677947998046875", "0.0016951613389085302963099621820219908840954303741455078125", "0.9857200963930583463720848858938396063205546631694562706442490043104635007254572116576279989085351697"}},
         {{"0.00036440245111034851033249282181714079342782497406005859375", "0.0032045582435329088288877841250723577104508876800537109375", "0.9749478524386443489635940707794584862397293749247037580417065490211452699141027516296989496295504814"}},
         {{"9.292409184363591040140169241556833412687410600483417510986328125e-07", "1.1004153981903233017192178522236645221710205078125", "2.30375096076042540689175588676000039497925847630595558133397707131285132566953633275113891277016516e-07"}},
         {{"7.9011116867809083434692762892836981336586177349090576171875e-06", "2.501429356325690633866543066687881946563720703125", "1.725549731486626941667591571601661693032847363937811575143526844664778604007055375993926458754318659e-13"}},
         {{"0.002194702245385149463885454679257236421108245849609375", "0.4554029466081654931031152955256402492523193359375", "0.06155376440352307625112575016428910522748384288078827636591065830754795891039869053266161085677238176"}},
         {{"22.07301650212122012817417271435260772705078125", "1.068151388370592127542713090093684513703919947147369384765625e-05", "1.000033052951634522817949765296734844002030410121902696384761848213697880320785724906280559557839065"}},
         {{"1.2800269360474598383074773533252255219849757850170135498046875e-05", "0.82820384801033686272830891539342701435089111328125", "8.867067987565472730340232597483329986971141543604280930643988956421369453435122224438015094627883985e-05"}},
         {{"0.002002233797023321437702492175958468578755855560302734375", "0.36236548139569446647101358394138514995574951171875", "0.1052354607973999617068260125669320058455047494682482645401709676315349136753079791942675678834230644"}},
         {{"1.46975278087147759168741567581406570752733387053012847900390625e-06", "0.00058645898951331133786002425267724902369081974029541015625", "0.9921545490687015011993664397669822958083714823206308163244333794633812962026395804957892576387227609"}},
         {{"1.6986976932856716615560799255035817623138427734375", "0.0001094516643533479348382453366639310843311250209808349609375", "1.000057995947953725843579777685569446562531782495874803311459893448640342826663799443616110584671403"}},
         {{"3151.83975929332882515154778957366943359375", "0.00102384029084149653920920997052235179580748081207275390625", "1.008281899582170909456168292398893224228907524500917527686642948507446650968497487312143619177014888"}},
         {{"3.550874766190428744039309094659984111785888671875", "79.89569317877567300456576049327850341796875", "93201890815141841785987673285271804996235065.23421058504905831849012238052738430238396953548739940743"}},
         {{"1267.75488796815989189781248569488525390625", "0.00010050228579829597989891976084209090913645923137664794921875", "1.000718347002182882407769409001363751463635276990908474269810920227694055134802129113230570363820713"}},
         {{"815.2200481057616343605332076549530029296875", "5.750253113055042199219801801746143610216677188873291015625e-05", "1.000385540108100391707595259012300373326213777041133767823089694166719955199446313729637902265410758"}},
         {{"2.650279689769746482852497138082981109619140625", "10.0316549124626845923557993955910205841064453125", "17632.59960946628286511121612377889534158879523746701021343006538551132871252548303774239827320348992"}},
         {{"3651.82751295248817768879234790802001953125", "16.7556451265639907433069311082363128662109375", "492218138915478317609234268428449378609898504811886050861657.038624663939529613919692032879990877291"}},
         {{"3.060642348474054545183486997217414682381786406040191650390625e-06", "0.022032802794947847768725068817730061709880828857421875", "0.7559742382617084426756745965849446223803798606668886343566458753316252789580281517603266769098261298"}},
         {{"204040.79325440712273120880126953125", "2.862847267931666550566660589538514614105224609375", "1588213255256801.634651058475551520441905692003881167501122283389625981954923294974593850206130202563"}},
         {{"302.71606947669306464376859366893768310546875", "0.00604039959726672404105585201250505633652210235595703125", "1.035109860443134465787713172197441325109986903813526070172462735802476266956300469041602854531673831"}},
         {{"44.75525701797897681899485178291797637939453125", "44.249452204617455208790488541126251220703125", "11192964638753256897616859782891448808690727657602550976045898573096222461.9833220482536328707584889"}},
         {{"0.000254801173789754968612708552200274425558745861053466796875", "0.00515927661277336413458982633528648875653743743896484375", "0.9582053670079491698092799994611003609129028963999372720213326417706904508125449715329992757551327531"}},
         {{"18992.8750500089372508227825164794921875", "0.54107849552119002822792026563547551631927490234375", "206.5634245077146901677140426233375385552789635309965918449547133641936887156437580764585679826507816"}},
         {{"0.0291020477415979972324322488930192776024341583251953125", "3.1072802283957519885182474439488942152820527553558349609375e-05", "0.9998901031924130854410746678613156522493201966576113395704828570719416742096397098647038916658118846"}},
         {{"3148.72001537003598059527575969696044921875", "0.0034312511193348126103952466792179620824754238128662109375", "1.028023343470148041152032105250863401062218396072090858194872477430412552749445124856335471647742531"}},
         {{"7.575615636684095480744838280173780731274746358394622802734375e-06", "0.05507768740935425899607480459962971508502960205078125", "0.5223603226916152930093444425225769153654144101035675463624599568485396984810315297933949421483274752"}},
         {{"48.0817610372865829049260355532169342041015625", "38.36782919616922526984126307070255279541015625", "34192856433167851598147133882981140865900018119268600844398174911.57027693407220531747982920702905619"}},
         {{"3205.37873291325013269670307636260986328125", "9.1206363324713353468199683593553572791279293596744537353515625e-07", "1.000007362738794631737714416992032912962460375252148033834668194840649937178061922804439135855016025"}},
         {{"0.0003619121763468288906218095490885389153845608234405517578125", "4.52277102882139392207860861816470787744037806987762451171875e-05", "0.9996416749084297727277417160666925226843375869762362855267031231042443600919418994199014681946253249"}},
         {{"0.0547727150864758538517662600497715175151824951171875", "3.546226210285186703462245727536128470092080533504486083984375e-06", "0.9999896998152210471952377974154607456926816728837688359674498128520734857006266754618719587597885931"}},
         {{"3156.3743834170963964425027370452880859375", "0.47240784892373444137092519667930901050567626953125", "44.98255472866703197023809220501874152946184343758370061378924532031223319280470473064921720790785118"}},
         {{"343503.2949360641650855541229248046875", "0.00012725015014535098929748802021322262589819729328155517578125", "1.001623367790905822856685556175409909563751494167938708958226168823690790389637302936317245256218707"}},
         {{"1050.742451295334831229411065578460693359375", "1.44512744072382729528143674446738486949470825493335723876953125e-06", "1.000010054166738749706418748856610955038167172273790336768766127308653359898934495823454565434510811"}},
         {{"8.186579100861061182754785381376905206707306206226348876953125e-06", "0.000569535439208307230074534999175739358179271221160888671875", "0.9933512247464489969371135945049104018829943452077718003230497796912764701482139247234839552464360427"}},
         {{"0.000694796213109996792234301921098449383862316608428955078125", "40.10321896495196369869518093764781951904296875", "2.230064903422792175718061400776658869810994140445950934182959578609437202840869846657435780565682972e-127"}},
         {{"1105.9279458966411766596138477325439453125", "0.3252608218509838167165071354247629642486572265625", "9.772486766886219200446342974017864426453859413115251100028007419286004730671633850469615914557721611"}},
         {{"1.25391642141006403698803850499388090611319057643413543701171875e-06", "0.733579604520485872853896580636501312255859375", "4.683927953082485726528431353649697203821389437303399146363032990721504108681213785434084703909540109e-05"}},
         {{"0.1431247395427910351628497664933092892169952392578125", "0.62094204885090586998330763890407979488372802734375", "0.2990527234241506618869316490711039184526444605422983981656375319209152243858082272089640290104191215"}},
         {{"0.0034375157542059324444760903816131758503615856170654296875", "15.1206687051705870317164226435124874114990234375", "5.576635975644302773534483884173129856156262737039027688152206255908283562541479337058304006854112996e-38"}},
         {{"5.618568157807579300394053289924300997881800867617130279541015625e-07", "0.00129876093753134198849341629511400242336094379425048828125", "0.9814818163513155943847270078931865078594366446723310911676179280657886936633527815107466211600376074"}},
         {{"269.25200862256269829231314361095428466796875", "0.0115330514334751887961516558789298869669437408447265625", "1.066662797305684971981137242746780051557524981570464777820041783631923741644747218400403313120461153"}},
         {{"31.609259072748983498968300409615039825439453125", "0.034806499721284855919378742328262887895107269287109375", "1.127725203486160051153046779299620569191007025999045898769440002347276592287498349816004231004210097"}},
         {{"1.114175224784427742884262979572440599440596997737884521484375e-05", "0.2075156227794912044970487841055728495121002197265625", "0.09379192949551902383714066455506833314106216389170736561166845741990812550909109841150042253931379681"}},
         {{"0.000154818678360652516035056436294325976632535457611083984375", "5.620604719948978328196431686336609345744363963603973388671875e-06", "0.9999506902120058979591360972999171604820757815190440882082126765754530065876944594000834478821366651"}},
         {{"0.00022806771010807128617636774237098507001064717769622802734375", "4.2251646762978662362486570458486312418244779109954833984375e-05", "0.9996457460302684431214749122654565303870592408257755425039451175342943418526121120332630239944024971"}},
         {{"220.739083812885155566618777811527252197265625", "1.636354283488831591586963598583537304875790141522884368896484375e-06", "1.000008831412608547288837021755840157827029154994313002311834504744061191239476500120705424894650235"}},
         {{"0.00021111563919415225233644495261842166655696928501129150390625", "49.16111787911501096459687687456607818603515625", "2.038517492209277070961058380173579174050343403825283083208100378263913032667351582956697084464628592e-181"}},
         {{"238.63391634703702948172576725482940673828125", "24.73501562052791769019677303731441497802734375", "65049256219035845474834664910034266797505758084021762190682.84836047651567330430480587406882102626274"}},
         {{"4.164964901356960590150801948539083241485059261322021484375e-05", "30.7526112190653293509967625141143798828125", "1.957089704592745143537056133561650570315752294246407733117709674905605752624818549479386626309188698e-135"}},
         {{"5.6080787497200606205038919682692721835337579250335693359375e-05", "2.70244506350627981296208535155756180756725370883941650390625e-05", "0.9997355002794970945536403636952997012405250585934555412613947004698211116110567912994537320679549614"}},
         {{"535323.167813788168132305145263671875", "3.627785147504031471044072532095015048980712890625", "605603230256580909333.5126132561810860692133727780440808616204363762602451705033164413035817244033886"}},
         {{"1090.48941423813448636792600154876708984375", "0.46525477435368944867377649643458425998687744140625", "25.89809885035909605166373717380146607273491001944843963797254789940887347008035265729475407453080634"}},
         {{"244.743479435021299650543369352817535400390625", "0.0004829951125498422280502008874236707924865186214447021484375", "1.00266010667859458204491495940561046268243246346579041029260554228176172232635197593622622290922199"}},
         {{"15148.874966345727443695068359375", "1.0235427570266870576053130914573330301209352910518646240234375e-06", "1.000009852345164480198854738965617321451574435365208669115118979181187361867754731995877224877588171"}},
         {{"0.0026733138568757371100748088110776734538376331329345703125", "0.43776493021523865678545917035080492496490478515625", "0.07475714388623095337436248741361333665414547121275626631773995042866621833468941972468800431926296617"}},
         {{"0.0101808534444340903490910932305268943309783935546875", "0.02722118478369406346928371931426227092742919921875", "0.8826113849646053543235057956690569203393017633070899899484791309158057199548105178673708847857469047"}},
         {{"0.10073127665666070651440122674102894961833953857421875", "0.0051822898443548981450845758445211686193943023681640625", "0.9881755603469047267429306393986870863960363522242718185009369016853748859016487648923854950762881514"}},
         {{"44.182574749154809978790581226348876953125", "0.07798274622226319952034145899233408272266387939453125", "1.34369652134197483793288072833308732535658609465509415675669768410932243039179437886058214288267645"}},
         {{"0.1529684484566207203215526533313095569610595703125", "1.63460821261542546913893281246288324837223626673221588134765625e-06", "0.9999969309892166001917854502690088986343031812500669386860251904961790036389141122811543444938200936"}},
         {{"1545.906819698253457318060100078582763671875", "0.05807119444321440226985941990278661251068115234375", "1.531791602349608519319031418103904843389497097371934081929309304109127813081717274248427102355680812"}},
         {{"1.0614987542670878818522328668194631973165087401866912841796875e-05", "0.00782539004163816664938480016644462011754512786865234375", "0.9142729692254652899039260911271281368625915952503206976679064872257732777952092139089464357492981607"}},
         {{"0.0001709428596228042447624506650072362390346825122833251953125", "1.45014151369056887470876393564367390354163944721221923828125e-05", "0.9998742200082450068555423265585131564089195546432428879938186291931370802894569899254467963716796862"}},
         {{"6.891651291217998874516492691100211231969296932220458984375e-05", "0.5033007132627158597415473195724189281463623046875", "0.008043131066645385700988397477895521481316265618091701967466785636506164134262967640868794295521655704"}},
         {{"42794.643340434762649238109588623046875", "6.70089369397394089659936167180376287433318793773651123046875e-06", "1.000071462010852267529847887383576253078657670140276318033096409496805773766967244183761751362529138"}},
         {{"5.4016293236799147077509264391892429557628929615020751953125e-05", "0.82336215316034344624540608492679893970489501953125", "0.0003064237427472995741396526549805907880598228607734542143515003217810433185904587437905894326525531377"}},
         {{"69.848048293023339283536188304424285888671875", "4.82423550256237437978967808049901577760465443134307861328125e-05", "1.00020487356414235503163915728935712973913169358041240146118411664231468676920896629650395007895637"}},
         {{"1011.74486687951502972282469272613525390625", "0.028009603881921474322069798290613107383251190185546875", "1.213866283631119305678755013912282318327739712214699566189409943729193546014462712189774786349405877"}},
         {{"0.00458527642677075686350463001872412860393524169921875", "0.0003204268030303175467110587959496115217916667461395263671875", "0.9982760199066904986419792192270003547262561213092605405095503752563493463987390941966513945825468602"}},
         {{"0.0760517875978827806449089621310122311115264892578125", "0.0447867011136062753706710282131098210811614990234375", "0.8910223181817657501387098683300530238243699505885677857423893276783349612009874865828526516517499702"}},
         {{"8.744417875921096046701840032167041272259666584432125091552734375e-07", "9.16721737539795736814107751655456013395451009273529052734375e-06", "0.9998721284263699814921143836096320519523055215895368210860059904941160942921739237399835063760307132"}},
         {{"191138.73823602846823632717132568359375", "0.14187049368133752835774430423043668270111083984375", "5.613937191650890700842843023133994246799836751341635098525915904987358595309533689826819848299924975"}},
         {{"3959.358050945258582942187786102294921875", "9.663427982375548570825873895273616653867065906524658203125e-05", "1.000800823128083279885822095223783978520053045791721070851245373822971372250335578283439231397859081"}},
         {{"0.000952749482248151265129099130035683629103004932403564453125", "0.073664423725248173724367006798274815082550048828125", "0.5990432907444360807503671893117769628645148985822362180435460951450534260154039702759784703169550181"}},
         {{"2.48422895630792811018892896068877007564879022538661956787109375e-06", "0.09559931249894149107859675496001727879047393798828125", "0.2911951733590636704293594030485059787852193941513999794727700211168637358507539936870722738317835886"}},
         {{"494.9111898087649024091660976409912109375", "0.300486277593047912404244925710372626781463623046875", "6.451615404400454502239537837838944691443735031717683102993163112113967010396524762645480186320044909"}},
         {{"0.058683089882192118746928599648526869714260101318359375", "0.000515714500475533861589827466787028242833912372589111328125", "0.9985387067996500923468217690706620366417380826364790093725397332843475732293175641700165526988866946"}},
         {{"7835.0102671020358684472739696502685546875", "0.003387880407948816452456952674765489064157009124755859375", "1.030843033696537347594440791840187631101891452717573502907477891711204400119794389089560474379540612"}},
         {{"0.72404015738467020213420255458913743495941162109375", "0.003213975030617495509677183918029186315834522247314453125", "0.9989627187426011585453061048470420123735298945929408387441416012346998713580479352968692747175243391"}},
         {{"0.2377374918560193339800434841890819370746612548828125", "0.416670398708927525177614370477385818958282470703125", "0.5495894282178055678024473799708668513982175016086295801970678145286561918992437026323562116733605057"}},
         {{"30.16478740007227088426589034497737884521484375", "19.213646095318608786328695714473724365234375", "26705121745934660249647701056.04172517057127002540486390788392640603252247177626333464418003782408496"}},
         {{"0.0002635405558263971027843641792287598946131765842437744140625", "0.422820432867716089475607077474705874919891357421875", "0.03066628536663153845184547716889572738966400181663370609342305806648515816263701313662350452525154859"}},
         {{"0.022604606307534691467253651353530585765838623046875", "0.01362703892199722266465045095173991285264492034912109375", "0.9496696913569865970217132158200933792365138507136695381946551529313352037377916785584289328206318599"}},
         {{"0.0744369864652441037833341397345066070556640625", "0.07921380143369194382074738314258866012096405029296875", "0.8140106756691231834969215211168114277331354598246314749031896279695926909758612781687242667854315093"}},
         {{"5.9808461532545464933718903921544551849365234375", "7.5414170557308056789223849047942849210812710225582122802734375e-07", "1.000001348830148466576881318454851916118486742068356831542800613064645392839513376848134731774311321"}},
         {{"0.00069502607186987253040655332370079122483730316162109375", "7.312252397119359086097301059226083452813327312469482421875e-06", "0.9999468299227659327743471868272610891197195453728769391897616025483644806170298355140520447963393432"}},
         {{"0.0224073444824900158689473528283997438848018646240234375", "5.7201137248564493087941984494904090752243064343929290771484375e-06", "0.9999782731477103656185657274657500184342668600285527806345111582398204812001276373605760820822854938"}},
         {{"0.000102696336152198410170133247021340139326639473438262939453125", "0.00021179100195032186763899328951765710371546447277069091796875", "0.9980568580991655385804604522875412257533655542063138928684756880980722672647139738033293192400889673"}},
         {{"0.008670370528630788609092405749834142625331878662109375", "0.00015213392034167769369112921395981175010092556476593017578125", "0.9992779527184891986510356047475437435401156766517684177336888488317177236123262987656555888906706086"}},
         {{"2616.40599122827916289679706096649169921875", "10.473722683676584210843429900705814361572265625", "625310610377006493866619672909110501.4663707305171337918337460149572470053563031211416464620348042089"}},
         {{"0.0282002594439443032836578595379251055419445037841796875", "5.289033603944384687688129442761919563054107129573822021484375e-06", "0.9999811266631206639783154454181951783475188550900953273209289580000023800827579140827262300534231205"}},
         {{"1.055972373764280547447924618609249591827392578125", "0.97445680351053542977979304851032793521881103515625", "1.054504395822804702374192438485143339652087879346523937301679513729554941786064318122007283041395678"}},
         {{"0.1922389078095090297892966191284358501434326171875", "1.0834961286007661397136347947167678285040892660617828369140625e-05", "0.9999821331310967238463354777475712054585553605927723789898438317695399599613154801741867732108311882"}},
         {{"1046319.338600852526724338531494140625", "0.007476286580583475782635360928907175548374652862548828125", "1.109186908548771067909715657763675262161809839958190578036608644224061438476547809925690766905862409"}},
         {{"6.3625954889573433335969721014180322526954114437103271484375e-06", "12.2809772788395861198296188376843929290771484375", "1.525987823384502862861354504589545735984728026633273748116368002037009277867017743850338418375452517e-64"}},
         {{"2.13062266921418196562697244811346308779320679605007171630859375e-06", "23.853461403196860146636026911437511444091796875", "5.190984551180664985200137067544906775919844616968301811490359555356198742624740663740872542188095851e-136"}},
         {{"140.290401591987802021321840584278106689453125", "1.137226893357055956812473596073687076568603515625", "276.4763167470505288506920002462005195223604707169021813508585258545039437865122970202420242228965654"}},
         {{"2.82657718261440410630058739371861520339734852313995361328125e-05", "1.010658357224504329979243577586345281815738417208194732666015625e-06", "0.9999894145629343747335701239845374781938473759900235212029924045975643188091524419937277774180905565"}},
         {{"1.821315880654364970092640163557007326744496822357177734375e-06", "0.004665181994500729178998454926841077394783496856689453125", "0.9402073733739897500790795770946311996201673068036717064400223994911162719023784553422841649192511659"}},
         {{"830433.2876769490540027618408203125", "0.00252637571399967619800719376144115813076496124267578125", "1.035033455487311063387108121307250559865034235012530108375103243658892484450991076700563775812665939"}},
         {{"881076.109807957895100116729736328125", "0.0733372790631923177073758779442869126796722412109375", "2.728921912631523093684183984838587456151621489173172778889129587066720968236394139418167352017436109"}},
         {{"75.372902256075349214370362460613250732421875", "1.4499767405133117043926349598592651091166771948337554931640625e-06", "1.000006267468447228284449253059950995773138053015244041036864940090817435322352070193073391510830102"}},
         {{"92.919612531185748593998141586780548095703125", "0.0002873640759416726774000405697506721480749547481536865234375", "1.001303106071232426184936578840199070821790776984982762219099000162138918252427655342554212067985206"}},
         {{"14.372735353256985035841353237628936767578125", "2.29608176142756548188117449171841144561767578125", "454.7839934914958106504702804499474276233483470289814992814213367962805677794085061411095746841076958"}},
         {{"2.70759640224985531631318735890090465545654296875", "0.07305367315564137431493918484193272888660430908203125", "1.075478777674280834190961426045709117004853525616096795814660396718527136984527593296348690674017848"}},
         {{"4.8691145965921341629868701605943215326988138258457183837890625e-06", "0.464415295518658677309531412902288138866424560546875", "0.003410126737916470551594907106117979840197095971977662743199678111809404571280474170788132602083442642"}},
         {{"0.00784854690434243218444265721700503490865230560302734375", "0.0146294874434749200275973635143600404262542724609375", "0.9315407245636693073203224916671579739562006616514349195715933038461158638941036695134158333669999829"}},
         {{"37212.0262782922945916652679443359375", "3.949085307953822621129802428185939788818359375", "1122065843345042197.288466212649735532503027039467851718316456457218308940106752620185458150528400508"}},
         {{"0.00023717283574733820893098990012504145852290093898773193359375", "6.40757764503378617060302957497697207145392894744873046875e-05", "0.999465320336479719185464362393617871174305860472162937262848024567759736497677820188393809529906684"}},
         {{"8.310480898243693332933142159379968916255165822803974151611328125e-07", "0.00415539351718839655713821912286221049726009368896484375", "0.9434820759277230575314976634424952558862179256147137387099333783200051011655933934563302805926361997"}},
         {{"159.284463946466757988673634827136993408203125", "0.0237368596141571408519865826747263781726360321044921875", "1.127905413947902767329315745945927161684279632347468356914031755878260069824880955368222410707602456"}},
         {{"0.00101037463548088457532259099025395698845386505126953125", "38.33500847501835551156545989215373992919921875", "1.468279595154718987747630240799957061137167827507284691669983827382054746728745054083390709614549933e-115"}},
         {{"16.669556500429052903200499713420867919921875", "0.87180401566643439537074300460517406463623046875", "11.6219309185108283690353246115834277672387666007297244563945515894098174972467389980428784832971321"}},
         {{"12846.46022961163544096052646636962890625", "23.24576384293703767980332486331462860107421875", "324945391385140670639469572823424256394544549042741674612450176003078448450232834405677926416541.8277"}},
         {{"1.0433964906565103042575259895130557197262533009052276611328125e-05", "15.7348570830582303869960014708340167999267578125", "4.130596869080806051265644835082744904736471300635805480216299862776234065483837896338275696519482735e-79"}},
         {{"0.59852729275474647607779843383468687534332275390625", "5.9284261890584139843840549843889675685204565525054931640625e-05", "0.9999695708501038978522546024755341715400240980898885314916448855370226478686474552962143770976240785"}},
         {{"1.20170821629471988138682103652854493702761828899383544921875e-05", "1.8357317185857997543507735827006399631500244140625", "9.286050016464047991221232465833750178114014394092642422179564569241871479210731786017736401570075411e-10"}},
         {{"1790.27858130276217707432806491851806640625", "10.375458498238771198884933255612850189208984375", "5630446649950377508369878203743869.536465514250361869405158427515433119700704287731253663906210880485"}},
         {{"122585.961723861284554004669189453125", "1.0137266944681557292486095978034654763177968561649322509765625e-05", "1.000118781029283027791430364417618807082245324331473681680569312169681984061371641271751226949555368"}},
         {{"0.058259514964496383893077791071846149861812591552734375", "4.850587686387239954774984462826381559352739714086055755615234375e-07", "0.9999986210526718200102090822887221253771093230804421524345844409193776623376250605571317470202184133"}},
         {{"0.090112539554422443899284189683385193347930908203125", "0.821806070291085433154876227490603923797607421875", "0.1383681570613961293710906146635326172062571343983241441059747435603470381878519459599923618146021508"}},
         {{"0.440965439472245090968272052123211324214935302734375", "0.102453990431700336927178796031512320041656494140625", "0.9195340750972949100624273123649610248645277974837678636241714039283192268679765210961286929408475281"}},
         {{"0.019371794438632072843375908632879145443439483642578125", "0.070857426321694116921889872173778712749481201171875", "0.7561940641445182356670182244495886799768192963377831560904944510365230294839494483327639911775804926"}},
         {{"366.4798926816438324749469757080078125", "7.7114501159503703320297063328325748443603515625", "59230423624726834909.63125738933920453075456994119023172648801248379811643299350845605510822577781032"}},
         {{"132708.98535155109129846096038818359375", "6.5653183389401581666888763066936007817275822162628173828125e-05", "1.000774739256020978775836796988150180001102202479895903517450499147722586524896676835433119028501837"}},
         {{"35.27750452648962209423189051449298858642578125", "0.2202074412941776859753417738829739391803741455078125", "2.191646687126943627122510588262105645337920002449024384056311632288980114833615525307748356758904923"}},
         {{"526.7211505676968954503536224365234375", "0.0004025940211054459920425596664017575676552951335906982421875", "1.002526109642439961160420295814482345705071076193878934631042381909014010985612555643854330041376576"}},
         {{"2.853822326564237954471536795608699321746826171875", "0.121622084331017621394721572869457304477691650390625", "1.136030450901955167045965341753184015792079188415985871463245360135297845522059522038012944826962339"}},
         {{"0.003933816946332845121414578670737682841718196868896484375", "1.026405193197206244803965091705322265625", "0.003398636754343909897233510640112140375133982625471405777367628216077821119429488833370938567260750635"}},
         {{"1907.239732158193874056451022624969482421875", "5.129809540460706176502603259592660833732225000858306884765625e-06", "1.000038748317224297514967379229170044354799652572358164535915258684926708133192815411682757960180044"}},
         {{"208.352376339207012279075570404529571533203125", "5.773283722656519448691685159058550880217808298766613006591796875e-07", "1.000003082494156613101939359438621537023038759092453075294509120068218154533712525636427573660421718"}},
         {{"9717.2451368939946405589580535888671875", "1.536000016098639279170379179362271315767429769039154052734375e-05", "1.000141040204916659606427255475099747561169864969731479730561221948459715314064446066074530038669929"}},
         {{"1.39802629394119207304099383737394646232132799923419952392578125e-06", "0.035193664404874980267123874000390060245990753173828125", "0.6222418708971306201425537134442788963194148443011276284035504785553925967805540491923782015637477682"}},
         {{"0.00182765007732204644297357987170471460558474063873291015625", "48.43189502697572379474877379834651947021484375", "2.445071613348319759149763302092499027131463453499632019602497884289608023662822552298648548496643619e-133"}},
         {{"24.98722422892615213640965521335601806640625", "0.04704757082435817583387915874482132494449615478515625", "1.163480843921288726021700115612549350109766420096375481963354951657011865855364165977188069644618479"}},
         {{"9.737287112781810327101494895174482735455967485904693603515625e-07", "0.0649131637906343872401748740230686962604522705078125", "0.4071650476329476830932735890583298463670470573550467480200091025107800657667055865886676876462558282"}},
         {{"0.0004415583456706321900664402591019097599200904369354248046875", "4.02580267030248754346610784438098562532104551792144775390625e-06", "0.9999689003512318229504337393124284950728560387764861537033369519004881290276918967512863625258135369"}},
         {{"0.00169439965482008625397014611735357902944087982177734375", "0.002707242943732084128338755135700921528041362762451171875", "0.9828749639027349611452558506995759788767468262312673903320559520526221760608939432037417883046842148"}},
         {{"10337.6346181700355373322963714599609375", "0.00128989017209520627493812838793019182048738002777099609375", "1.011994523821528463744819510766311645825846616083228942905554078642279902075086398206177407152797972"}},
         {{"0.0311283305234331397759461879104492254555225372314453125", "4.0135347704642700995825155274587814346887171268463134765625e-06", "0.9999860745885194545606405216734429356419905543629560492710455306970880399438723293327702964414085153"}},
         {{"264401.5976317734457552433013916015625", "0.24851546821695524869255677913315594196319580078125", "22.25955589570639881250399962314132124577996850898098158786977874569488853007397604282337402933118357"}},
         {{"10.958211687916815435528405942022800445556640625", "0.5120627298060593801665163482539355754852294921875", "3.407312306071135778674181313650079538490975564330323666871242662656095346812970614682504109311945775"}},
         {{"28.76503292323553750975406728684902191162109375", "32.9636994307592203767853789031505584716796875", "1229088807878783381178701814266361748554445733223.493590872287239146328759463114918780349801450234419"}},
         {{"5.431923355811552165013644799529402007465250790119171142578125e-06", "7.05490815336499605694119130472330425618565641343593597412109375e-07", "0.9999914472181335984139208813911051792655685525793206179460018861436052537178251602210027707618330783"}},
         {{"0.0004244506207376568494848978474465184262953698635101318359375", "0.0242617657776718897988388334852061234414577484130859375", "0.8282951772972878150013071425721621537506689274800490299372219302551743107381929490669404965772527026"}},
         {{"0.00130733273631510917012210626353407860733568668365478515625", "5.2885872406397563066059175351796284303418360650539398193359375e-06", "0.9999648856332070729685807263347874221745185446024617211433836252248993680566658616499075817090494847"}},
         {{"5.0569331914835659834428494008307097828947007656097412109375e-05", "0.0037129598848615920358984254789902479387819766998291015625", "0.9639371218816808683038355443070848856743889082564237175107753506304638101612451572543160130648899457"}},
         {{"48442.3969203509041108191013336181640625", "0.000398413458911828048669701018980049411766231060028076171875", "1.004307386695580020314036591317448967283124853136314092310968669276316379938512474157353802224930247"}},
         {{"0.03707848036565419835852708274614997208118438720703125", "0.05525017089928641045304402723559178411960601806640625", "0.8335731973511686668879626312010293848826383236478361316629878138638797762152336056613221160502622244"}},
         {{"971.655514772415699553675949573516845703125", "0.459166384187756904111665789969265460968017578125", "23.53776538446583535690316796797352993709222329488105447630272127430481875799901641878930147939344105"}},
         {{"219.264585936548201061668805778026580810546875", "6.9672931617900194238741928831615268791210837662220001220703125e-06", "1.000037556360323018235379438130163565759024893141881450669162214945803926250814728169470242921618745"}},
         {{"0.00307913249304984926180139837015303783118724822998046875", "5.0546380401164157092352979816496372222900390625", "2.01796919571840276681288861157787148204747502682382876961304668410493347723226232435621763407251841e-13"}},
         {{"272.79362500860224827192723751068115234375", "0.034510787649445429092764925371739082038402557373046875", "1.2135636441470622599370931421778180361173455653459577718195140914264982313294947327716998248632074"}},
         {{"7.23608622250063859837609925307333469390869140625", "93.171384571694716214551590383052825927734375", "120546015073919711264373148066122758661068546232444031541084969085534241548251886.9305688136908098664"}},
         {{"2570.72303536429899395443499088287353515625", "0.0026979027499375897447553285246613086201250553131103515625", "1.021409746208889449303059594357997027288432896824493762149257508577830660068916490411587242272993211"}},
         {{"627071.378772734664380550384521484375", "9.772137051992151545419529323766028028330765664577484130859375e-06", "1.000130454964570163589023879023692488204817925056846047937823570034587402830120269539715125278322479"}},
         {{"0.12210937685213585357502097394899465143680572509765625", "2.1154363504782882081552142228275670277071185410022735595703125e-06", "0.9999955515897291603368049186766309113963405740560497665190685184277353610380068087005983790557375485"}},
         {{"33.35748803500194981097592972218990325927734375", "22.2409910922928020227118395268917083740234375", "7539273228900347423680173352272927.200057838212289418580033706290126943825571383922810898066597758137"}},
         {{"0.293886093222884170472752884961664676666259765625", "2.831521392109625297368868945824260663357563316822052001953125e-05", "0.9999653268371227893510658015136430193796917462344319787175450328825635431591290290005202380889115199"}},
         {{"1053.639003412403326365165412425994873046875", "0.045173341631881192004271952100680209696292877197265625", "1.369446565254178451738588975380261768795808805076146150664961284712205435387237675289919060383834468"}},
         {{"7.0442099359273142801692035863680985130486078560352325439453125e-07", "0.1835143848883531436655403012991882860660552978515625", "0.0742999851704960956730922360017793140547106959683768718211070193339741510527923658661871542385815055"}},
         {{"136453.344822830520570278167724609375", "0.0036416170660240575196464618557001813314855098724365234375", "1.043997950386359218978579582563166519774451525674537272225182845987697893919995952336617492741209614"}},
         {{"2965.6582425073720514774322509765625", "0.251611044873741196425953603466041386127471923828125", "7.475220683928678387324507085721359305671010055505554682672048090808828514531672589805073783792434153"}},
         {{"34957.188500732998363673686981201171875", "0.0004050304428750706176032281291554681956768035888671875", "1.004246370036370013465194451370479039855272987344442059988415130311437136505709888380616097225825151"}},
         {{"348.70435070784878917038440704345703125", "0.00059447045136969654810510377274113125167787075042724609375", "1.003486226239564266544197903655153850082045323087191202944580566914440716916275676374640938779066438"}},
         {{"0.156866354052483725212141507654450833797454833984375", "0.000292382444394148101329822253546808497048914432525634765625", "0.9994585487759416559856410495627812700434681522035800716173129525916247657168866454375169074510830256"}},
         {{"0.030255549669599812734332999752950854599475860595703125", "7.43476200926107111399687710218131542205810546875", "5.071624391121480193587722248775101689733991617173735368756844067969157544603081898435825618329967344e-12"}},
         {{"11330.10573697159998118877410888671875", "0.00016339149873991561275066164427016701665706932544708251953125", "1.001526459226896713053345331434475225318084874706366074726000900622820954273164745600679448382740738"}},
         {{"0.03576034314149090231893524105544202029705047607421875", "5.2592552770165985225915561418474908350617624819278717041015625e-06", "0.9999824820172953439232911408688245438618897060242677383259209409551636421685962773873823245962683139"}},
         {{"6378.9521531699574552476406097412109375", "1.336068644210004177814987702532789626275189220905303955078125e-05", "1.000117056606234250918057206171373181152231187397970333688295245721935118219795994973130691014878664"}},
         {{"0.0738104343240080229548993884236551821231842041015625", "2.78526096140715813593603800146780713475891388952732086181640625e-06", "0.9999927409255643306528065164207469389880101482396332751540764215716304547196956686301750263228676621"}},
         {{"22460.66664629729348234832286834716796875", "35.62248530015159531103563494980335235595703125", "1.01973918494886267810001894567464669163247766970377570638787946610322435627850446066260492108701968e+155"}},
         {{"0.000900800182710125019180935623808181844651699066162109375", "0.0009644327257653552420979536918821395374834537506103515625", "0.9932599951189437799131134508868032131573002364417591778083609799906703698137993976732290746910518598"}},
         {{"94182.00341389398090541362762451171875", "7.77419300288664260104552761188045906237675808370113372802734375e-07", "1.000008903810753849490815952091396964111994924168704174593640826878641136574899261056414781066683628"}},
         {{"3.736927319995179790079742332409296068362891674041748046875e-05", "5.54122149350758519403825630433857440948486328125", "2.926328139421667354692528652981751462893950530131934715325309027308425799275831053494494892189069183e-25"}},
         {{"49943.93526913248933851718902587890625", "38.2669140887688854490988887846469879150390625", "6.258007625751081761291379408943213692368030595354034742877877090491939999895538763511497912496071067e+179"}},
         {{"64.8539296127664783853106200695037841796875", "0.00419081769830710604285428644288913346827030181884765625", "1.017638419286460631799453114306197766721003662045459377314951399360771935340637483952708460812336436"}},
         {{"0.00126185358741126574244617586373351514339447021484375", "0.0004187078745637698613812904824271754478104412555694580078125", "0.9972089545056330839623488484181547364859448430625502453476122033045479655311315550252372864756908435"}},
         {{"0.9520791403690278542626401758752763271331787109375", "0.0010810790970678697819895575094051309861242771148681640625", "0.9999469127314662479725278900067545108957648199738538430006625837856408454465663885208619348339915409"}},
         {{"10.40906840130304544800310395658016204833984375", "193.673921485533583108917810022830963134765625", "1.112103386626915928196196649797265974491001632926835134791709616818898689279586301145184832904083108e+197"}},
         {{"0.197149801027129978336915883119218051433563232421875", "0.0050873959053111261763291395254782401025295257568359375", "0.9917731573614097817017154936637638665150904665616902894208911657701141641935806836868466461304389348"}},
         {{"279.97151154174071052693761885166168212890625", "0.000578721875202228321077857486898210481740534305572509765625", "1.003266239694909015664613594859548001828365159272002769886393253986637246844841765188300656832947231"}},
         {{"0.0647180277522192426431502099148929119110107421875", "3.54396046637904049703138298355042934417724609375", "6.113931592888489403153884843900739551737367851498945581160328866530885163519698181862568187320414631e-05"}},
         {{"0.01543914595519298160741783476623822934925556182861328125", "0.009302915020454383121517594190663658082485198974609375", "0.9619420643749070067584723239534196729126002888600943916950465384526584270014213568993868027817412338"}},
         {{"6.4800387592573969186560134403407573699951171875", "3.26121082452632900412770045051047418382950127124786376953125e-05", "1.000060944967697632788208723838178790361005294790189086304328364127163470762739111823942249093212008"}},
         {{"1.0225514626813350105294375680387020111083984375", "0.005899326910987552630150076993231778033077716827392578125", "1.000131569177670915525998922351292548534843989557700151538445658253491280846591049462933107663243451"}},
         {{"281.88812912132425481104291975498199462890625", "0.00407352602363188209810829221169115044176578521728515625", "1.023246932890744618060653146011305880994557485354832525251000297618810944830329810582156327094705469"}},
         {{"2.071854045279265398200874148670891372603364288806915283203125e-05", "5.352923007059786186800920404493808746337890625", "8.48850440195174823298359540121818302112603372945421421598869878434979555804675505735326375593302973e-26"}},
         {{"4.3193528604969547678160690651338882162235677242279052734375e-05", "2.67446027602891164636454701675205569699755869805812835693359375e-06", "0.9999731225171697588682986113938163156744687083824377840460130925178749341973501860053979617895652358"}},
         {{"5.607286595652558684721965265307375148040591739118099212646484375e-07", "6.38030853130188620203101790284705430167377926409244537353515625e-07", "0.9999908162077462976270002280900321499402993881659143049922805489179559382254436626926051375256660974"}},
         {{"457863.041241061873733997344970703125", "12.7996139284958445614392985589802265167236328125", "2852529115356798910733788359014798860899675762997894013856700157805516625.339900604515453462452066619"}},
         {{"2.08423524510303792567356140352785587310791015625", "1.35070210371325316600632504560053348541259765625", "2.696509948188848085514788517904213983308491109435218218898620544438327157424500959489947406585274773"}},
         {{"44477.756179801770485937595367431640625", "4.37314005329937316446375916711986064910888671875", "212314549862845492717.8775959647681155463251863397157644536049207612109965446237346519490412650768468"}},
         {{"6223.2583021543759969063103199005126953125", "0.0022522595279351638464238050119092804379761219024658203125", "1.019870694693727592704536289126427289542305135348330896049546619951717258016563809029634211449349061"}},
         {{"9.124004407016377861394662573957248241640627384185791015625e-05", "0.00023440055305491637634152102265261419233866035938262939453125", "0.9978219774869505831660847191615576896635683257900035223060211584177551856449366478783163682351855933"}},
         {{"13.379562781153111927778809331357479095458984375", "1.481838128525152987915829150278312908994848839938640594482421875e-06", "1.000003843492990457768628013604037323635457748437747016552002720806277399413864371758863796069027275"}},
         {{"0.1468094748199788579512414798955433070659637451171875", "0.00655150258058036227826192998691112734377384185791015625", "0.9875088290782914734963500607095999286273552030959134093921086368984711057148398313021479230252542598"}},
         {{"21.340342483575710730292485095560550689697265625", "0.00674881959702026967828913939229096285998821258544921875", "1.020870232322788986730250828211809433573362333100899876113874116672315215495756772397626008961327821"}},
         {{"0.007952338447526907661000450389110483229160308837890625", "0.284851678481346848315070019452832639217376708984375", "0.2523204412730889374827054551634518047280108530355707340248699896694722254000003239700145479435290709"}},
         {{"4.8646915561613240062965868304445393732748925685882568359375e-05", "0.005228774344446386612705879315399215556681156158447265625", "0.9493985966931214022325115517739462162389405948876565841123337216491283383092222557157330458309424395"}},
         {{"0.099706102361067916461934146354906260967254638671875", "58.80643552108921312537859193980693817138671875", "1.313392503053175861371032804352035402866412090419636869845649773176752678725464216300078555181042351e-59"}},
         {{"0.00925067228877758662708430392740410752594470977783203125", "0.0658041095700439004900772488326765596866607666015625", "0.7347944117032276407688345590371466780812374504971518086412846928572574050739212614458463071448398762"}},
         {{"1271.445111592622197349555790424346923828125", "0.06188363194499280073301861193613149225711822509765625", "1.556342621770853104552727770091255961034617664205557743532545769442888018868518369152039737013857958"}},
         {{"142.205871841051475712447427213191986083984375", "2.49175822061007323782177991233766078948974609375", "231499.4437694640650899356053754596830356428635750916142973928754575122979194424731470750159355284874"}},
         {{"179.8308464716837988817133009433746337890625", "5.1574074180122200274090094040957410470582544803619384765625e-05", "1.000267809307251477084251829056464334524202154640145209682615788870242738417800219799407910641436787"}},
         {{"14.1617493747205998033678042702376842498779296875", "0.00080665947340863124181975507553943316452205181121826171875", "1.002140374268426705516111473501470527521933451718025674223600340567195922737079678675509260511926502"}},
         {{"0.000303620445696295947202170140144517063163220882415771484375", "0.0177755559529355344761114565699244849383831024169921875", "0.8659074538096995094601474098299016759415278547888856303339792652146961638222475055950085287767767353"}},
         {{"0.88972758288691000672088193823583424091339111328125", "0.00104127996077174773958962106235048850066959857940673828125", "0.9998783443023582447222481441112224377626548657798117729357182132182151744165931165857655667811298977"}},
         {{"5.339277350309052741516614715155952808345318771898746490478515625e-07", "6.20766189867572767122939902950662371949874795973300933837890625e-07", "0.999991034310800110010707678195555035460840436574922962233018600570397473321611757701145927433415066"}},
         {{"21.999419507432406817315495572984218597412109375", "0.0030670748032577861696967858051721123047173023223876953125", "1.009525458623658556852190421156081451904552356132688205304061204893104190924210633765186616444065095"}},
         {{"9.49426131626506853745939285005306373932398855686187744140625e-06", "0.33365859318957813428596637095324695110321044921875", "0.02109535136370511679063624887039038982833249953133571609193273319396869217149037391922738409838473388"}},
         {{"1.582043812017442263595512486062943935394287109375", "1.69967088875234341601404286503651519524282775819301605224609375e-06", "1.000000779669191985361185060006040218483425874592332148125028253190361859271118426407204417784727621"}},
         {{"128909.708384213387034833431243896484375", "0.006930398774138209205819549652005662210285663604736328125", "1.084966470901606292628651762577514290477297440580644865587573677837124479812363381240118872937828003"}},
         {{"0.0018897933914625748841320529436416109092533588409423828125", "15.0746279790036936674368917010724544525146484375", "8.769184239911440383040214451934381732019420678192792376489005966955750242365854746240264604052109097e-42"}},
         {{"8.4151721236286078919874853454530239105224609375", "3.6469095409674991969950497150421142578125", "2363.856415558493358790477790545734505035312191432801865060598119525166028148020267989223174580223056"}},
         {{"77.9453915372945402850746177136898040771484375", "13.7570185974411316465193522162735462188720703125", "106022076559461302576262215.7242902035479202865813953648730746594780225759936468682235125871897113107"}},
         {{"153996.83080146810971200466156005859375", "0.01557539455806315753516599897920968942344188690185546875", "1.204474313743011751875116622397758862804421230575699891010487179656416641636839651225303803657816265"}},
         {{"158292.08564240694977343082427978515625", "2.398396583412239380095343221910297870635986328125", "2953691846308.993498353226004351957714305523939384033929204977432515050076937571630853902085000763474"}},
         {{"13400.79627897878526709973812103271484375", "0.00193078487352859630432622140006060362793505191802978515625", "1.018517748515570162676788969476734934269521469808385089561996257058687919783441016421791344126544682"}},
         {{"65.7565090460510646153124980628490447998046875", "29.534957910236840916695655323565006256103515625", "492883670959819008737323351550818895693909971573197862.6697562972335429000309509534807117101930531295"}},
         {{"1.10978898863228390113400134708854238851927220821380615234375e-05", "13.22193774039163827183074317872524261474609375", "3.079478675696583211944583805971812315656854455376470737481249943902024669095820211554407858476239991e-66"}},
         {{"0.060138853849309115151555715783615596592426300048828125", "2.60022931107700908527451122420615092778461985290050506591796875e-06", "0.9999926905242830920450417276662577122941704720556559836557891563069856035247063610358677624802944547"}},
         {{"0.0215534801432384071073755649194936268031597137451171875", "0.04058128146716322159903711508377455174922943115234375", "0.8557994453936024641449760831259619447020799824799242873747525739993421831400409189696987185570873497"}},
         {{"0.125", "0", "1"}},
         {{"-0.125", "0", "1"}},
         {{"0.75", "0", "1"}},
         {{"-0.75", "0", "1"}},
         {{"1", "0", "1"}},
         {{"-1", "0", "1"}},
         {{"1.25", "0", "1"}},
         {{"-1.25", "0", "1"}},
         {{"55.25", "0", "1"}},
         {{"-55.25", "0", "1"}},
         {{"0.125", "1", "0.125"}},
         {{"-0.125", "1", "-0.125"}},
         {{"0.75", "1", "0.75"}},
         {{"-0.75", "1", "-0.75"}},
         {{"1", "1", "1"}},
         {{"-1", "1", "-1"}},
         {{"1.25", "1", "1.25"}},
         {{"-1.25", "1", "-1.25"}},
         {{"55.25", "1", "55.25"}},
         {{"-55.25", "1", "-55.25"}},
         {{"0.125", "2", "0.015625"}},
         {{"-0.125", "2", "0.015625"}},
         {{"0.75", "2", "0.5625"}},
         {{"-0.75", "2", "0.5625"}},
         {{"1", "2", "1"}},
         {{"-1", "2", "1"}},
         {{"1.25", "2", "1.5625"}},
         {{"-1.25", "2", "1.5625"}},
         {{"55.25", "2", "3052.5625"}},
         {{"-55.25", "2", "3052.5625"}},
         {{"0.125", "3", "0.001953125"}},
         {{"-0.125", "3", "-0.001953125"}},
         {{"0.75", "3", "0.421875"}},
         {{"-0.75", "3", "-0.421875"}},
         {{"1", "3", "1"}},
         {{"-1", "3", "-1"}},
         {{"1.25", "3", "1.953125"}},
         {{"-1.25", "3", "-1.953125"}},
         {{"55.25", "3", "168654.078125"}},
         {{"-55.25", "3", "-168654.078125"}},
         {{"0.125", "4", "0.000244140625"}},
         {{"-0.125", "4", "0.000244140625"}},
         {{"0.75", "4", "0.31640625"}},
         {{"-0.75", "4", "0.31640625"}},
         {{"1", "4", "1"}},
         {{"-1", "4", "1"}},
         {{"1.25", "4", "2.44140625"}},
         {{"-1.25", "4", "2.44140625"}},
         {{"55.25", "4", "9318137.81640625"}},
         {{"-55.25", "4", "9318137.81640625"}},
         {{"0.125", "5", "3.0517578125e-05"}},
         {{"-0.125", "5", "-3.0517578125e-05"}},
         {{"0.75", "5", "0.2373046875"}},
         {{"-0.75", "5", "-0.2373046875"}},
         {{"1", "5", "1"}},
         {{"-1", "5", "-1"}},
         {{"1.25", "5", "3.0517578125"}},
         {{"-1.25", "5", "-3.0517578125"}},
         {{"55.25", "5", "514827114.3564453125"}},
         {{"-55.25", "5", "-514827114.3564453125"}},
         {{"0.125", "6", "3.814697265625e-06"}},
         {{"-0.125", "6", "3.814697265625e-06"}},
         {{"0.75", "6", "0.177978515625"}},
         {{"-0.75", "6", "0.177978515625"}},
         {{"1", "6", "1"}},
         {{"-1", "6", "1"}},
         {{"1.25", "6", "3.814697265625"}},
         {{"-1.25", "6", "3.814697265625"}},
         {{"55.25", "6", "28444198068.193603515625"}},
         {{"-55.25", "6", "28444198068.193603515625"}},
         {{"0.125", "7", "4.76837158203125e-07"}},
         {{"-0.125", "7", "-4.76837158203125e-07"}},
         {{"0.75", "7", "0.13348388671875"}},
         {{"-0.75", "7", "-0.13348388671875"}},
         {{"1", "7", "1"}},
         {{"-1", "7", "-1"}},
         {{"1.25", "7", "4.76837158203125"}},
         {{"-1.25", "7", "-4.76837158203125"}},
         {{"55.25", "7", "1571541943267.69659423828125"}},
         {{"-55.25", "7", "-1571541943267.69659423828125"}},
         {{"0.125", "8", "5.9604644775390625e-08"}},
         {{"-0.125", "8", "5.9604644775390625e-08"}},
         {{"0.75", "8", "0.1001129150390625"}},
         {{"-0.75", "8", "0.1001129150390625"}},
         {{"1", "8", "1"}},
         {{"-1", "8", "1"}},
         {{"1.25", "8", "5.9604644775390625"}},
         {{"-1.25", "8", "5.9604644775390625"}},
         {{"55.25", "8", "86827692365540.2368316650390625"}},
         {{"-55.25", "8", "86827692365540.2368316650390625"}},
         {{"0.125", "9", "7.450580596923828125e-09"}},
         {{"-0.125", "9", "-7.450580596923828125e-09"}},
         {{"0.75", "9", "0.075084686279296875"}},
         {{"-0.75", "9", "-0.075084686279296875"}},
         {{"1", "9", "1"}},
         {{"-1", "9", "-1"}},
         {{"1.25", "9", "7.450580596923828125"}},
         {{"-1.25", "9", "-7.450580596923828125"}},
         {{"55.25", "9", "4797230003196098.084949493408203125"}},
         {{"-55.25", "9", "-4797230003196098.084949493408203125"}},
         {{"0.125", "10", "9.31322574615478515625e-10"}},
         {{"-0.125", "10", "9.31322574615478515625e-10"}},
         {{"0.75", "10", "0.05631351470947265625"}},
         {{"-0.75", "10", "0.05631351470947265625"}},
         {{"1", "10", "1"}},
         {{"-1", "10", "1"}},
         {{"1.25", "10", "9.31322574615478515625"}},
         {{"-1.25", "10", "9.31322574615478515625"}},
         {{"55.25", "10", "265046957676584419.19345951080322265625"}},
         {{"-55.25", "10", "265046957676584419.19345951080322265625"}},
         {{"0.125", "11", "1.16415321826934814453125e-10"}},
         {{"-0.125", "11", "-1.16415321826934814453125e-10"}},
         {{"0.75", "11", "0.0422351360321044921875"}},
         {{"-0.75", "11", "-0.0422351360321044921875"}},
         {{"1", "11", "1"}},
         {{"-1", "11", "-1"}},
         {{"1.25", "11", "11.6415321826934814453125"}},
         {{"-1.25", "11", "-11.6415321826934814453125"}},
         {{"55.25", "11", "14643844411631289160.4386379718780517578125"}},
         {{"-55.25", "11", "-14643844411631289160.4386379718780517578125"}},
         {{"0.125", "12", "1.4551915228366851806640625e-11"}},
         {{"-0.125", "12", "1.4551915228366851806640625e-11"}},
         {{"0.75", "12", "0.031676352024078369140625"}},
         {{"-0.75", "12", "0.031676352024078369140625"}},
         {{"1", "12", "1"}},
         {{"-1", "12", "1"}},
         {{"1.25", "12", "14.551915228366851806640625"}},
         {{"-1.25", "12", "14.551915228366851806640625"}},
         {{"55.25", "12", "809072403742628726114.234747946262359619140625"}},
         {{"-55.25", "12", "809072403742628726114.234747946262359619140625"}},
         {{"0.125", "13", "1.818989403545856475830078125e-12"}},
         {{"-0.125", "13", "-1.818989403545856475830078125e-12"}},
         {{"0.75", "13", "0.02375726401805877685546875"}},
         {{"-0.75", "13", "-0.02375726401805877685546875"}},
         {{"1", "13", "1"}},
         {{"-1", "13", "-1"}},
         {{"1.25", "13", "18.18989403545856475830078125"}},
         {{"-1.25", "13", "-18.18989403545856475830078125"}},
         {{"55.25", "13", "44701250306780237117811.46982403099536895751953125"}},
         {{"-55.25", "13", "-44701250306780237117811.46982403099536895751953125"}},
         {{"0.125", "14", "2.27373675443232059478759765625e-13"}},
         {{"-0.125", "14", "2.27373675443232059478759765625e-13"}},
         {{"0.75", "14", "0.0178179480135440826416015625"}},
         {{"-0.75", "14", "0.0178179480135440826416015625"}},
         {{"1", "14", "1"}},
         {{"-1", "14", "1"}},
         {{"1.25", "14", "22.7373675443232059478759765625"}},
         {{"-1.25", "14", "22.7373675443232059478759765625"}},
         {{"55.25", "14", "2469744079449608100759083.7077777124941349029541015625"}},
         {{"-55.25", "14", "2469744079449608100759083.7077777124941349029541015625"}},
         {{"0.125", "15", "2.8421709430404007434844970703125e-14"}},
         {{"-0.125", "15", "-2.8421709430404007434844970703125e-14"}},
         {{"0.75", "15", "0.013363461010158061981201171875"}},
         {{"-0.75", "15", "-0.013363461010158061981201171875"}},
         {{"1", "15", "1"}},
         {{"-1", "15", "-1"}},
         {{"1.25", "15", "28.421709430404007434844970703125"}},
         {{"-1.25", "15", "-28.421709430404007434844970703125"}},
         {{"55.25", "15", "136453360389590847566939374.854718615300953388214111328125"}},
         {{"-55.25", "15", "-136453360389590847566939374.854718615300953388214111328125"}},
         {{"0.125", "16", "3.552713678800500929355621337890625e-15"}},
         {{"-0.125", "16", "3.552713678800500929355621337890625e-15"}},
         {{"0.75", "16", "0.01002259575761854648590087890625"}},
         {{"-0.75", "16", "0.01002259575761854648590087890625"}},
         {{"1", "16", "1"}},
         {{"-1", "16", "1"}},
         {{"1.25", "16", "35.52713678800500929355621337890625"}},
         {{"-1.25", "16", "35.52713678800500929355621337890625"}},
         {{"55.25", "16", "7539048161524894328073400460.72320349537767469882965087890625"}},
         {{"-55.25", "16", "7539048161524894328073400460.72320349537767469882965087890625"}},
         {{"0.125", "17", "4.44089209850062616169452667236328125e-16"}},
         {{"-0.125", "17", "-4.44089209850062616169452667236328125e-16"}},
         {{"0.75", "17", "0.0075169468182139098644256591796875"}},
         {{"-0.75", "17", "-0.0075169468182139098644256591796875"}},
         {{"1", "17", "1"}},
         {{"-1", "17", "-1"}},
         {{"1.25", "17", "44.4089209850062616169452667236328125"}},
         {{"-1.25", "17", "-44.4089209850062616169452667236328125"}},
         {{"55.25", "17", "416532410924250411626055375454.9569931196165271103382110595703125"}},
         {{"-55.25", "17", "-416532410924250411626055375454.9569931196165271103382110595703125"}},
         {{"0.125", "18", "5.5511151231257827021181583404541015625e-17"}},
         {{"-0.125", "18", "5.5511151231257827021181583404541015625e-17"}},
         {{"0.75", "18", "0.005637710113660432398319244384765625"}},
         {{"-0.75", "18", "0.005637710113660432398319244384765625"}},
         {{"1", "18", "1"}},
         {{"-1", "18", "1"}},
         {{"1.25", "18", "55.511151231257827021181583404541015625"}},
         {{"-1.25", "18", "55.511151231257827021181583404541015625"}},
         {{"55.25", "18", "23013415703564835242339559493886.373869858813122846186161041259765625"}},
         {{"-55.25", "18", "23013415703564835242339559493886.373869858813122846186161041259765625"}},
         {{"0.125", "19", "6.938893903907228377647697925567626953125e-18"}},
         {{"-0.125", "19", "-6.938893903907228377647697925567626953125e-18"}},
         {{"0.75", "19", "0.00422828258524532429873943328857421875"}},
         {{"-0.75", "19", "-0.00422828258524532429873943328857421875"}},
         {{"1", "19", "1"}},
         {{"-1", "19", "-1"}},
         {{"1.25", "19", "69.38893903907228377647697925567626953125"}},
         {{"-1.25", "19", "-69.38893903907228377647697925567626953125"}},
         {{"55.25", "19", "1271491217621957147139260662037222.15630969942503725178539752960205078125"}},
         {{"-55.25", "19", "-1271491217621957147139260662037222.15630969942503725178539752960205078125"}},
         {{"-1", "15", "-1"}},
         {{"-1", "14", "1"}},
      }};

   } // namespace test_pow_data

#endif // BOOST_MP_TEST_POW_DATA_REDUCED_HPP
