// Copyright 2019-2021 Authors of Hubble
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package observe

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"os"
	"os/signal"
	"strings"
	"text/tabwriter"
	"time"

	flowpb "github.com/cilium/cilium/api/v1/flow"
	"github.com/cilium/cilium/api/v1/observer"
	monitorAPI "github.com/cilium/cilium/pkg/monitor/api"
	"github.com/cilium/hubble/cmd/common/config"
	"github.com/cilium/hubble/cmd/common/conn"
	"github.com/cilium/hubble/cmd/common/template"
	"github.com/cilium/hubble/pkg/defaults"
	"github.com/cilium/hubble/pkg/logger"
	hubprinter "github.com/cilium/hubble/pkg/printer"
	hubtime "github.com/cilium/hubble/pkg/time"
	"github.com/spf13/cobra"
	"github.com/spf13/pflag"
	"github.com/spf13/viper"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
	"google.golang.org/protobuf/types/known/timestamppb"
	"gopkg.in/yaml.v2"
)

// see protocol filter in Hubble server code (there is unfortunately no
// list of supported protocols defined anywhere)
var protocols = []string{
	// L4
	"icmp", "icmpv4", "icmpv6",
	"tcp",
	"udp",
	// L7
	"dns",
	"http",
	"kafka",
}

var verdicts = []string{
	flowpb.Verdict_FORWARDED.String(),
	flowpb.Verdict_DROPPED.String(),
	flowpb.Verdict_AUDIT.String(),
	flowpb.Verdict_REDIRECTED.String(),
	flowpb.Verdict_ERROR.String(),
}

// flowEventTypes are the valid event types supported by observe. This corresponds
// to monitorAPI.MessageTypeNames, excluding MessageTypeNameAgent,
// MessageTypeNameDebug and MessageTypeNameRecCapture. These excluded message
// message types are not supported by `hubble observe flows` but have separate sub-commands.
var flowEventTypes = []string{
	monitorAPI.MessageTypeNameCapture,
	monitorAPI.MessageTypeNameDrop,
	monitorAPI.MessageTypeNameL7,
	monitorAPI.MessageTypeNamePolicyVerdict,
	monitorAPI.MessageTypeNameTrace,
}

// flowEventTypeSubtypes is a map message types and all their subtypes.
var flowEventTypeSubtypes = map[string][]string{
	monitorAPI.MessageTypeNameCapture: nil,
	monitorAPI.MessageTypeNameDrop:    nil,
	monitorAPI.MessageTypeNameTrace: {
		monitorAPI.TraceObservationPoints[monitorAPI.TraceFromLxc],
		monitorAPI.TraceObservationPoints[monitorAPI.TraceFromHost],
		monitorAPI.TraceObservationPoints[monitorAPI.TraceFromNetwork],
		monitorAPI.TraceObservationPoints[monitorAPI.TraceFromOverlay],
		monitorAPI.TraceObservationPoints[monitorAPI.TraceFromProxy],
		monitorAPI.TraceObservationPoints[monitorAPI.TraceFromStack],
		monitorAPI.TraceObservationPoints[monitorAPI.TraceToLxc],
		monitorAPI.TraceObservationPoints[monitorAPI.TraceToHost],
		monitorAPI.TraceObservationPoints[monitorAPI.TraceToNetwork],
		monitorAPI.TraceObservationPoints[monitorAPI.TraceToOverlay],
		monitorAPI.TraceObservationPoints[monitorAPI.TraceToProxy],
		monitorAPI.TraceObservationPoints[monitorAPI.TraceToStack],
	},
	monitorAPI.MessageTypeNameL7:            nil,
	monitorAPI.MessageTypeNamePolicyVerdict: nil,
}

const (
	allowlistFlag = "allowlist"
	denylistFlag  = "denylist"
)

// getFlowsFilters struct is only used for printing allowlist/denylist filters as YAML.
type getFlowsFilters struct {
	Allowlist []string `yaml:"allowlist,omitempty"`
	Denylist  []string `yaml:"denylist,omitempty"`
}

// getFlowFiltersYAML returns allowlist/denylist filters as a YAML string. This YAML can then be
// passed to hubble observe command via `--config` flag.
func getFlowFiltersYAML(req *observer.GetFlowsRequest) (string, error) {
	var allowlist, denylist []string
	for _, filter := range req.Whitelist {
		filterJSON, err := json.Marshal(filter)
		if err != nil {
			return "", err
		}
		allowlist = append(allowlist, string(filterJSON))

	}
	for _, filter := range req.Blacklist {
		filterJSON, err := json.Marshal(filter)
		if err != nil {
			return "", err
		}
		denylist = append(denylist, string(filterJSON))
	}
	filters := getFlowsFilters{
		Allowlist: allowlist,
		Denylist:  denylist,
	}
	out, err := yaml.Marshal(filters)
	if err != nil {
		return "", err
	}
	return string(out), nil
}

func newFlowsCmd(vp *viper.Viper, ofilter *flowFilter) *cobra.Command {
	observeCmd := &cobra.Command{
		Example: `* Piping flows to hubble observe

  Save output from 'hubble observe -o jsonpb' command to a file, and pipe it to
  the observe command later for offline processing. For example:

    hubble observe -o jsonpb --last 1000 > flows.json

  Then,

    cat flows.json | hubble observe

  Note that the observe command ignores --follow, --last, and server flags when it
  reads flows from stdin. The observe command processes and output flows in the same
  order they are read from stdin without sorting them by timestamp.`,
		Use:   "observe",
		Short: "Observe flows of a Hubble server",
		Long: `Observe provides visibility into flow information on the network and
application level. Rich filtering enable observing specific flows related to
individual pods, services, TCP connections, DNS queries, HTTP requests and
more.`,
		RunE: func(cmd *cobra.Command, args []string) error {
			debug := vp.GetBool(config.KeyDebug)
			if err := handleFlowArgs(ofilter, debug); err != nil {
				return err
			}
			req, err := getFlowsRequest(ofilter, vp.GetStringSlice(allowlistFlag), vp.GetStringSlice(denylistFlag))
			if err != nil {
				return err
			}
			if otherOpts.printRawFilters {
				filterYAML, err := getFlowFiltersYAML(req)
				if err != nil {
					return err
				}
				fmt.Print(filterYAML)
				return nil
			}

			ctx, cancel := signal.NotifyContext(context.Background(), os.Interrupt, os.Kill)
			defer cancel()

			var client observer.ObserverClient
			fi, err := os.Stdin.Stat()
			if err != nil {
				return err
			}
			if fi.Mode()&os.ModeNamedPipe != 0 {
				// read flows from stdin
				client = newIOReaderObserver(os.Stdin)
				logger.Logger.Debug("Reading flows from stdin")
			} else {
				// read flows from a hubble server
				hubbleConn, err := conn.New(ctx, vp.GetString(config.KeyServer), vp.GetDuration(config.KeyTimeout))
				if err != nil {
					return err
				}
				defer hubbleConn.Close()
				client = observer.NewObserverClient(hubbleConn)
			}

			logger.Logger.WithField("request", req).Debug("Sending GetFlows request")
			if err := getFlows(ctx, client, req); err != nil {
				msg := err.Error()
				// extract custom error message from failed grpc call
				if s, ok := status.FromError(err); ok && s.Code() == codes.Unknown {
					msg = s.Message()
				}
				return errors.New(msg)
			}
			return nil
		},
	}

	// selector flags
	selectorFlags := pflag.NewFlagSet("selectors", pflag.ContinueOnError)
	selectorFlags.BoolVar(&selectorOpts.all, "all", false, "Get all flows stored in Hubble's buffer")
	selectorFlags.Uint64Var(&selectorOpts.last, "last", 0, fmt.Sprintf("Get last N flows stored in Hubble's buffer (default %d)", defaults.FlowPrintCount))
	selectorFlags.Uint64Var(&selectorOpts.first, "first", 0, "Get first N flows stored in Hubble's buffer")
	selectorFlags.BoolVarP(&selectorOpts.follow, "follow", "f", false, "Follow flows output")
	selectorFlags.StringVar(&selectorOpts.since,
		"since", "",
		fmt.Sprintf(`Filter flows since a specific date. The format is relative (e.g. 3s, 4m, 1h43,, ...) or one of:
  StampMilli:             %s
  YearMonthDay:           %s
  YearMonthDayHour:       %s
  YearMonthDayHourMinute: %s
  RFC3339:                %s
  RFC3339Milli:           %s
  RFC3339Micro:           %s
  RFC3339Nano:            %s
  RFC1123Z:               %s
 `,
			time.StampMilli,
			hubtime.YearMonthDay,
			hubtime.YearMonthDayHour,
			hubtime.YearMonthDayHourMinute,
			time.RFC3339,
			hubtime.RFC3339Milli,
			hubtime.RFC3339Micro,
			time.RFC3339Nano,
			time.RFC1123Z,
		),
	)
	selectorFlags.StringVar(&selectorOpts.until,
		"until", "",
		fmt.Sprintf(`Filter flows until a specific date. The format is relative (e.g. 3s, 4m, 1h43,, ...) or one of:
  StampMilli:             %s
  YearMonthDay:           %s
  YearMonthDayHour:       %s
  YearMonthDayHourMinute: %s
  RFC3339:                %s
  RFC3339Milli:           %s
  RFC3339Micro:           %s
  RFC3339Nano:            %s
  RFC1123Z:               %s
 `,
			time.StampMilli,
			hubtime.YearMonthDay,
			hubtime.YearMonthDayHour,
			hubtime.YearMonthDayHourMinute,
			time.RFC3339,
			hubtime.RFC3339Milli,
			hubtime.RFC3339Micro,
			time.RFC3339Nano,
			time.RFC1123Z,
		),
	)
	observeCmd.PersistentFlags().AddFlagSet(selectorFlags)

	// filter flags
	filterFlags := pflag.NewFlagSet("filters", pflag.ContinueOnError)
	filterFlags.Var(filterVar(
		"not", ofilter,
		"Reverses the next filter to be blacklist i.e. --not --from-ip 2.2.2.2"))
	filterFlags.Var(filterVar(
		"node-name", ofilter,
		`Show all flows which match the given node names (e.g. "k8s*", "test-cluster/*.company.com")`))
	filterFlags.Var(filterVar(
		"protocol", ofilter,
		`Show only flows which match the given L4/L7 flow protocol (e.g. "udp", "http")`))
	filterFlags.Var(filterVar(
		"tcp-flags", ofilter,
		`Show only flows which match the given TCP flags (e.g. "syn", "ack", "fin")`))
	filterFlags.VarP(filterVarP(
		"type", "t", ofilter, []string{},
		fmt.Sprintf("Filter by event types TYPE[:SUBTYPE]. Available types and subtypes:\n%s", func() string {
			var b strings.Builder
			w := tabwriter.NewWriter(&b, 0, 0, 1, ' ', 0)
			fmt.Fprintln(w, "TYPE", "\t", "SUBTYPE")
			// we don't iterate the flowEventTypeSubtypes map as we want
			// consistent ordering
			for _, k := range flowEventTypes {
				v := flowEventTypeSubtypes[k]
				if len(v) > 0 {
					fmt.Fprintln(w, k, "\t", v[0])
					for i := 1; i < len(v); i++ {
						fmt.Fprintln(w, "\t", v[i])
					}
				} else {
					fmt.Fprintln(w, k, "\t", "n/a")
				}
			}
			w.Flush()
			return strings.TrimSpace(b.String())
		}())))
	filterFlags.Var(filterVar(
		"verdict", ofilter,
		fmt.Sprintf("Show only flows with this verdict [%s]", strings.Join(verdicts, ", ")),
	))

	filterFlags.Var(filterVar(
		"http-status", ofilter,
		`Show only flows which match this HTTP status code prefix (e.g. "404", "5+")`))
	filterFlags.Var(filterVar(
		"http-method", ofilter,
		`Show only flows which match this HTTP method (e.g. "get", "post")`))
	filterFlags.Var(filterVar(
		"http-path", ofilter,
		`Show only flows which match this HTTP path regular expressions (e.g. "/page/\\d+")`))

	filterFlags.Var(filterVar(
		"from-fqdn", ofilter,
		`Show all flows originating at the given fully qualified domain name (e.g. "*.cilium.io").`))
	filterFlags.Var(filterVar(
		"fqdn", ofilter,
		`Show all flows related to the given fully qualified domain name (e.g. "*.cilium.io").`))
	filterFlags.Var(filterVar(
		"to-fqdn", ofilter,
		`Show all flows terminating at the given fully qualified domain name (e.g. "*.cilium.io").`))

	filterFlags.Var(filterVar(
		"from-ip", ofilter,
		"Show all flows originating at the given IP address. Each of the source IPs can be specified as an exact match (e.g. '1.1.1.1') or as a CIDR range (e.g.'1.1.1.0/24')."))
	filterFlags.Var(filterVar(
		"ip", ofilter,
		"Show all flows related to the given IP address. Each of the IPs can be specified as an exact match (e.g. '1.1.1.1') or as a CIDR range (e.g.'1.1.1.0/24')."))
	filterFlags.Var(filterVar(
		"to-ip", ofilter,
		"Show all flows terminating at the given IP address. Each of the destination IPs can be specified as an exact match (e.g. '1.1.1.1') or as a CIDR range (e.g.'1.1.1.0/24')."))

	filterFlags.VarP(filterVarP(
		"ipv4", "4", ofilter, nil,
		`Show only IPv4 flows`))
	filterFlags.Lookup("ipv4").NoOptDefVal = "v4" // add default val so none is required to be provided
	filterFlags.VarP(filterVarP(
		"ipv6", "6", ofilter, nil,
		`Show only IPv6 flows`))
	filterFlags.Lookup("ipv6").NoOptDefVal = "v6" // add default val so none is required to be provided
	filterFlags.Var(filterVar(
		"ip-version", ofilter,
		`Show only IPv4, IPv6 flows or non IP flows (e.g. ARP packets) (ie: "none", "v4", "v6")`))

	filterFlags.Var(filterVar(
		"from-pod", ofilter,
		"Show all flows originating in the given pod name prefix([namespace/]<pod-name>). If namespace is not provided, 'default' is used"))
	filterFlags.Var(filterVar(
		"pod", ofilter,
		"Show all flows related to the given pod name prefix ([namespace/]<pod-name>). If namespace is not provided, 'default' is used."))
	filterFlags.Var(filterVar(
		"to-pod", ofilter,
		"Show all flows terminating in the given pod name prefix([namespace/]<pod-name>). If namespace is not provided, 'default' is used"))

	filterFlags.Var(filterVar(
		"from-namespace", ofilter,
		"Show all flows originating in the given Kubernetes namespace."))
	filterFlags.VarP(filterVarP(
		"namespace", "n", ofilter, nil,
		"Show all flows related to the given Kubernetes namespace."))
	filterFlags.Var(filterVar(
		"to-namespace", ofilter,
		"Show all flows terminating in the given Kubernetes namespace."))

	filterFlags.Var(filterVar(
		"from-label", ofilter,
		`Show only flows originating in an endpoint with the given labels (e.g. "key1=value1", "reserved:world")`))
	filterFlags.VarP(filterVarP(
		"label", "l", ofilter, nil,
		`Show only flows related to an endpoint with the given labels (e.g. "key1=value1", "reserved:world")`))
	filterFlags.Var(filterVar(
		"to-label", ofilter,
		`Show only flows terminating in an endpoint with given labels (e.g. "key1=value1", "reserved:world")`))

	filterFlags.Var(filterVar(
		"from-service", ofilter,
		"Shows flows where the source IP address matches the ClusterIP address of the given service name prefix([namespace/]<svc-name>). If namespace is not provided, 'default' is used"))
	filterFlags.Var(filterVar(
		"service", ofilter,
		"Shows flows where either the source or destination IP address matches the ClusterIP address of the given service name prefix ([namespace/]<svc-name>). If namespace is not provided, 'default' is used. "))
	filterFlags.Var(filterVar(
		"to-service", ofilter,
		"Shows flows where the destination IP address matches the ClusterIP address of the given service name prefix ([namespace/]<svc-name>). If namespace is not provided, 'default' is used"))

	filterFlags.Var(filterVar(
		"from-port", ofilter,
		"Show only flows with the given source port (e.g. 8080)"))
	filterFlags.Var(filterVar(
		"port", ofilter,
		"Show only flows with given port in either source or destination (e.g. 8080)"))
	filterFlags.Var(filterVar(
		"to-port", ofilter,
		"Show only flows with the given destination port (e.g. 8080)"))

	filterFlags.Var(filterVar(
		"from-identity", ofilter,
		"Show all flows originating at an endpoint with the given security identity"))
	filterFlags.Var(filterVar(
		"identity", ofilter,
		"Show all flows related to an endpoint with the given security identity"))
	filterFlags.Var(filterVar(
		"to-identity", ofilter,
		"Show all flows terminating at an endpoint with the given security identity"))
	observeCmd.Flags().AddFlagSet(filterFlags)

	rawFilterFlags := pflag.NewFlagSet("raw-filters", pflag.ContinueOnError)
	rawFilterFlags.StringArray(allowlistFlag, []string{}, "Specify allowlist as JSON encoded FlowFilters")
	rawFilterFlags.StringArray(denylistFlag, []string{}, "Specify denylist as JSON encoded FlowFilters")
	observeCmd.Flags().AddFlagSet(rawFilterFlags)
	// bind these flags to viper so that they can be specified as environment variables.
	vp.BindPFlags(rawFilterFlags)

	// formatting flags only to `hubble observe`, but not sub-commands. Will be added to
	// generic formatting flags below.
	observeFormattingFlags := pflag.NewFlagSet("", pflag.ContinueOnError)
	observeFormattingFlags.BoolVar(
		&formattingOpts.numeric,
		"numeric",
		false,
		"Display all information in numeric form",
	)
	observeFormattingFlags.BoolVar(
		&formattingOpts.enableIPTranslation,
		"ip-translation",
		true,
		"Translate IP addresses to logical names such as pod name, FQDN, ...",
	)
	observeFormattingFlags.StringVar(
		&formattingOpts.color,
		"color", "auto",
		"Colorize the output when the output format is one of 'compact' or 'dict'. The value is one of 'auto' (default), 'always' or 'never'",
	)
	observeCmd.Flags().AddFlagSet(observeFormattingFlags)

	// generic formatting flags, available to `hubble observe`, including sub-commands.
	formattingFlags := pflag.NewFlagSet("Formatting", pflag.ContinueOnError)
	formattingFlags.StringVarP(
		&formattingOpts.output, "output", "o", "compact",
		`Specify the output format, one of:
  compact:  Compact output
  dict:     Each flow is shown as KEY:VALUE pair
  json:     JSON encoding (DEPRECATED: use --output=jsonpb instead)
  jsonpb:   Output each GetFlowResponse according to proto3's JSON mapping
  table:    Tab-aligned columns
`)
	formattingFlags.BoolVarP(&formattingOpts.nodeName, "print-node-name", "", false, "Print node name in output")
	formattingFlags.StringVar(
		&formattingOpts.timeFormat, "time-format", "StampMilli",
		fmt.Sprintf(`Specify the time format for printing. This option does not apply to the json and jsonpb output type. One of:
  StampMilli:             %s
  YearMonthDay:           %s
  YearMonthDayHour:       %s
  YearMonthDayHourMinute: %s
  RFC3339:                %s
  RFC3339Milli:           %s
  RFC3339Micro:           %s
  RFC3339Nano:            %s
  RFC1123Z:               %s
 `,
			time.StampMilli,
			hubtime.YearMonthDay,
			hubtime.YearMonthDayHour,
			hubtime.YearMonthDayHourMinute,
			time.RFC3339,
			hubtime.RFC3339Milli,
			hubtime.RFC3339Micro,
			time.RFC3339Nano,
			time.RFC1123Z,
		),
	)
	observeCmd.PersistentFlags().AddFlagSet(formattingFlags)

	// other flags
	otherFlags := pflag.NewFlagSet("other", pflag.ContinueOnError)
	otherFlags.BoolVarP(&otherOpts.ignoreStderr,
		"silent-errors", "s", false,
		"Silently ignores errors and warnings")
	otherFlags.BoolVar(&otherOpts.printRawFilters,
		"print-raw-filters", false,
		"Print allowlist/denylist filters and exit without sending the request to Hubble server")
	observeCmd.PersistentFlags().AddFlagSet(otherFlags)

	// advanced completion for flags
	observeCmd.RegisterFlagCompletionFunc("ip-version", func(_ *cobra.Command, _ []string, _ string) ([]string, cobra.ShellCompDirective) {
		return []string{"none", "v4", "v6"}, cobra.ShellCompDirectiveDefault
	})
	observeCmd.RegisterFlagCompletionFunc("type", func(_ *cobra.Command, _ []string, _ string) ([]string, cobra.ShellCompDirective) {
		var completions []string
		for _, ftype := range flowEventTypes {
			completions = append(completions, ftype)
			for _, subtype := range flowEventTypeSubtypes[ftype] {
				completions = append(completions, fmt.Sprintf("%s:%s", ftype, subtype))
			}
		}
		return completions, cobra.ShellCompDirectiveDefault
	})
	observeCmd.RegisterFlagCompletionFunc("verdict", func(_ *cobra.Command, _ []string, _ string) ([]string, cobra.ShellCompDirective) {
		return verdicts, cobra.ShellCompDirectiveDefault
	})
	observeCmd.RegisterFlagCompletionFunc("protocol", func(_ *cobra.Command, _ []string, _ string) ([]string, cobra.ShellCompDirective) {
		return protocols, cobra.ShellCompDirectiveDefault
	})
	observeCmd.RegisterFlagCompletionFunc("http-status", func(_ *cobra.Command, _ []string, _ string) ([]string, cobra.ShellCompDirective) {
		httpStatus := []string{
			"100", "101", "102", "103",
			"200", "201", "202", "203", "204", "205", "206", "207", "208",
			"226",
			"300", "301", "302", "303", "304", "305", "307", "308",
			"400", "401", "402", "403", "404", "405", "406", "407", "408", "409",
			"410", "411", "412", "413", "414", "415", "416", "417", "418",
			"421", "422", "423", "424", "425", "426", "428", "429",
			"431",
			"451",
			"500", "501", "502", "503", "504", "505", "506", "507", "508",
			"510", "511",
		}
		return httpStatus, cobra.ShellCompDirectiveDefault
	})
	observeCmd.RegisterFlagCompletionFunc("http-method", func(_ *cobra.Command, _ []string, _ string) ([]string, cobra.ShellCompDirective) {
		return []string{
			http.MethodConnect,
			http.MethodDelete,
			http.MethodGet,
			http.MethodHead,
			http.MethodOptions,
			http.MethodPatch,
			http.MethodPost,
			http.MethodPut,
			http.MethodTrace,
		}, cobra.ShellCompDirectiveDefault
	})
	observeCmd.RegisterFlagCompletionFunc("identity", func(_ *cobra.Command, _ []string, _ string) ([]string, cobra.ShellCompDirective) {
		return reservedIdentitiesNames(), cobra.ShellCompDirectiveDefault
	})
	observeCmd.RegisterFlagCompletionFunc("to-identity", func(_ *cobra.Command, _ []string, _ string) ([]string, cobra.ShellCompDirective) {
		return reservedIdentitiesNames(), cobra.ShellCompDirectiveDefault
	})
	observeCmd.RegisterFlagCompletionFunc("from-identity", func(_ *cobra.Command, _ []string, _ string) ([]string, cobra.ShellCompDirective) {
		return reservedIdentitiesNames(), cobra.ShellCompDirectiveDefault
	})
	observeCmd.RegisterFlagCompletionFunc("output", func(_ *cobra.Command, _ []string, _ string) ([]string, cobra.ShellCompDirective) {
		return []string{
			"compact",
			"dict",
			"json",
			"jsonpb",
			"table",
		}, cobra.ShellCompDirectiveDefault
	})
	observeCmd.RegisterFlagCompletionFunc("color", func(_ *cobra.Command, _ []string, _ string) ([]string, cobra.ShellCompDirective) {
		return []string{"auto", "always", "never"}, cobra.ShellCompDirectiveDefault
	})
	observeCmd.RegisterFlagCompletionFunc("time-format", func(_ *cobra.Command, _ []string, _ string) ([]string, cobra.ShellCompDirective) {
		return hubtime.FormatNames, cobra.ShellCompDirectiveDefault
	})

	// default value for when the flag is on the command line without any options
	observeCmd.Flags().Lookup("not").NoOptDefVal = "true"

	observeCmd.AddCommand(
		newAgentEventsCommand(vp, selectorFlags, formattingFlags, config.ServerFlags, otherFlags),
		newDebugEventsCommand(vp, selectorFlags, formattingFlags, config.ServerFlags, otherFlags),
	)

	formattingFlags.AddFlagSet(observeFormattingFlags)

	template.RegisterFlagSets(observeCmd.Name(), selectorFlags, filterFlags, rawFilterFlags, formattingFlags, config.ServerFlags, otherFlags)

	return observeCmd
}

func handleFlowArgs(ofilter *flowFilter, debug bool) (err error) {
	if ofilter.blacklisting {
		return errors.New("trailing --not found in the arguments")
	}

	// initialize the printer with any options that were passed in
	var opts = []hubprinter.Option{
		hubprinter.WithTimeFormat(hubtime.FormatNameToLayout(formattingOpts.timeFormat)),
		hubprinter.WithColor(formattingOpts.color),
	}

	switch formattingOpts.output {
	case "compact":
		opts = append(opts, hubprinter.Compact())
	case "dict":
		opts = append(opts, hubprinter.Dict())
	case "json", "JSON":
		opts = append(opts, hubprinter.JSON())
	case "jsonpb":
		opts = append(opts, hubprinter.JSONPB())
	case "tab", "table":
		if selectorOpts.follow {
			return fmt.Errorf("table output format is not compatible with follow mode")
		}
		opts = append(opts, hubprinter.Tab())
	default:
		return fmt.Errorf("invalid output format: %s", formattingOpts.output)
	}
	if otherOpts.ignoreStderr {
		opts = append(opts, hubprinter.IgnoreStderr())
	}
	if formattingOpts.numeric {
		formattingOpts.enableIPTranslation = false
	}
	if formattingOpts.enableIPTranslation {
		opts = append(opts, hubprinter.WithIPTranslation())
	}
	if debug {
		opts = append(opts, hubprinter.WithDebug())
	}
	if formattingOpts.nodeName {
		opts = append(opts, hubprinter.WithNodeName())
	}
	printer = hubprinter.New(opts...)
	return nil
}

func parseRawFilters(filters []string) ([]*flowpb.FlowFilter, error) {
	var results []*flowpb.FlowFilter
	for _, f := range filters {
		dec := json.NewDecoder(strings.NewReader(f))
		for {
			var result flowpb.FlowFilter
			if err := dec.Decode(&result); err != nil {
				if err == io.EOF {
					break
				}
				return nil, fmt.Errorf("failed to decode '%v': %w", f, err)
			}
			results = append(results, &result)
		}
	}
	return results, nil
}

func getFlowsRequest(ofilter *flowFilter, allowlist []string, denylist []string) (*observer.GetFlowsRequest, error) {
	first := selectorOpts.first > 0
	last := selectorOpts.last > 0
	if first && last {
		return nil, fmt.Errorf("cannot set both --first and --last")
	}
	if first && selectorOpts.all {
		return nil, fmt.Errorf("cannot set both --first and --all")
	}
	if first && selectorOpts.follow {
		return nil, fmt.Errorf("cannot set both --first and --follow")
	}
	if last && selectorOpts.all {
		return nil, fmt.Errorf("cannot set both --last and --all")
	}

	// convert selectorOpts.since into a param for GetFlows
	var since, until *timestamppb.Timestamp
	if selectorOpts.since != "" {
		st, err := hubtime.FromString(selectorOpts.since)
		if err != nil {
			return nil, fmt.Errorf("failed to parse the since time: %v", err)
		}

		since = timestamppb.New(st)
		if err := since.CheckValid(); err != nil {
			return nil, fmt.Errorf("failed to convert `since` timestamp to proto: %v", err)
		}
		// Set the until field if both --since and --until options are specified and --follow
		// is not specified. If --since is specified but --until is not, the server sets the
		// --until option to the current timestamp.
		if selectorOpts.until != "" && !selectorOpts.follow {
			ut, err := hubtime.FromString(selectorOpts.until)
			if err != nil {
				return nil, fmt.Errorf("failed to parse the until time: %v", err)
			}
			until = timestamppb.New(ut)
			if err := until.CheckValid(); err != nil {
				return nil, fmt.Errorf("failed to convert `until` timestamp to proto: %v", err)
			}
		}
	}

	if since == nil && until == nil && !first {
		switch {
		case selectorOpts.all:
			// all is an alias for last=uint64_max
			selectorOpts.last = ^uint64(0)
		case selectorOpts.last == 0 && !selectorOpts.follow:
			// no specific parameters were provided, just a vanilla
			// `hubble observe` in non-follow mode
			selectorOpts.last = defaults.FlowPrintCount
		}
	}

	var (
		wl []*flowpb.FlowFilter
		bl []*flowpb.FlowFilter
	)
	if ofilter.whitelist != nil {
		wl = ofilter.whitelist.flowFilters()
	}
	if ofilter.blacklist != nil {
		bl = ofilter.blacklist.flowFilters()
	}

	// load filters from raw filter flags
	al, err := parseRawFilters(allowlist)
	if err != nil {
		return nil, fmt.Errorf("invalid --allowlist flag: %w", err)

	}
	wl = append(wl, al...)
	dl, err := parseRawFilters(denylist)
	if err != nil {
		return nil, fmt.Errorf("invalid --denylist flag: %w", err)
	}
	bl = append(bl, dl...)

	number := selectorOpts.last
	if first {
		number = selectorOpts.first
	}

	req := &observer.GetFlowsRequest{
		Number:    number,
		Follow:    selectorOpts.follow,
		Whitelist: wl,
		Blacklist: bl,
		Since:     since,
		Until:     until,
		First:     first,
	}

	return req, nil
}

func getFlows(ctx context.Context, client observer.ObserverClient, req *observer.GetFlowsRequest) error {
	b, err := client.GetFlows(ctx, req)
	if err != nil {
		return err
	}
	defer printer.Close()

	for {
		getFlowResponse, err := b.Recv()
		switch err {
		case io.EOF, context.Canceled:
			return nil
		case nil:
		default:
			if status.Code(err) == codes.Canceled {
				return nil
			}
			return err
		}

		if err = printer.WriteGetFlowsResponse(getFlowResponse); err != nil {
			return err
		}
	}
}
