/*  combinatorics.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "combinatorics.h"
#include "functions.h"
#include "yamlutils.h"

using namespace std;

namespace Reduze {

// Permutation members

// register type
namespace {
YAMLProxy<Permutation> dummyperm;
}

Permutation::Permutation(const std::map<int, int>& m) {
	vector<int> from, to;
	from.reserve(m.size());
	to.reserve(m.size());
	for (map<int, int>::const_iterator it = m.begin(); it != m.end(); ++it) {
		from.push_back(it->first);
		to.push_back(it->second);
	}
	Permutation perm(from, to);
	perm.perm_.swap(perm_);
}

Permutation::Permutation(const std::vector<int>& from,
		const std::vector<int>& to) {
	set<int> nodes1, nodes2;
	VERIFY(from.size() == to.size());
	nodes1.insert(from.begin(), from.end());
	nodes2.insert(to.begin(), to.end());
	VERIFY(nodes1 == nodes2);
	VERIFY(nodes1.size() == from.size());

	set<pair<int, int> > from_to;
	for (unsigned i = 0; i < from.size(); ++i)
		if (from[i] != to[i])
			from_to.insert(make_pair(from[i], to[i]));

	vector<vector<int> > res;
	while (!from_to.empty()) { // empty or at least 2 elements
		vector<int> cycle;
		cycle.push_back(from_to.begin()->first);
		int next = from_to.begin()->second;
		cycle.push_back(next);
		from_to.erase(from_to.begin());
		do {
			set<pair<int, int> >::iterator f = from_to.begin();
			for (; f != from_to.end(); ++f)
				if (f->first == next)
					break;
			ASSERT(f != from_to.end());
			next = f->second;
			from_to.erase(f);
			if (next != cycle.front()) {
				cycle.push_back(next);
			} else {
				res.push_back(cycle);
				break;
			}
		} while (true);
	}

	perm_.swap(res);

	// \todo remove this test
	Permutation test(perm_);
	VERIFY(*this == test);
}

Permutation Permutation::inverse() const {
	vector<vector<int> > inv;
	inv.reserve(perm_.size());
	vector<vector<int> >::const_reverse_iterator c;
	vector<int>::const_reverse_iterator i;
	for (c = perm_.rbegin(); c != perm_.rend(); ++c) {
		vector<int> tmp;
		tmp.reserve(c->size());
		for (i = c->rbegin(); i != c->rend(); ++i)
			tmp.push_back(*i);
		inv.push_back(tmp);
	}
	return Permutation(inv);
}

Permutation Permutation::compose(const Permutation& p1, const Permutation& p2) {
	vector<vector<int> > prod(p1.perm_);
	prod.insert(prod.end(), p2.perm_.begin(), p2.perm_.end());
	return Permutation(prod);
}

void Permutation::relabel(int offset_to_add) {
	for (vector<vector<int> >::iterator i = perm_.begin(); i != perm_.end(); ++i)
		for (vector<int>::iterator j = i->begin(); j != i->end(); ++j)
			*j += offset_to_add;
}

void Permutation::swap(Permutation& other) {
	perm_.swap(other.perm_);
}

void Permutation::erase(std::set<int> i) {
	vector<vector<int> > new_perm;
	new_perm.reserve(perm_.size());
	for (vector<vector<int> >::iterator c = perm_.begin(); c != perm_.end(); ++c) {
		vector<int> new_c;
		new_c.reserve(c->size());
		for (vector<int>::iterator n = c->begin(); n != c->end(); ++n)
			if (i.find(*n) == i.end())
				new_c.push_back(*n);
		if (!new_c.empty())
			new_perm.push_back(new_c);
	}
	new_perm.swap(perm_);
}

std::ostream& operator<<(std::ostream& os, const Permutation& p) {
	YAML::Emitter ye;
	p.print(ye);
	os << ye.c_str();
	return os;
}

void Permutation::print(YAML::Emitter& ye) const {
	ye << YAML::Flow << perm_;
}

void Permutation::print_to_onebased(YAML::Emitter& ye) const {
	Permutation pshift(*this);
	pshift.relabel(1);
	ye << YAML::Flow << pshift.perm_;
}

void Permutation::print_mma_to_onebased(std::ostream& os) const {
	vector<vector<int> >::const_iterator i;
	for (i = perm_.begin(); i != perm_.end(); ++i) {
		os << (i == perm_.begin() ? '{' : ',');
		vector<int>::const_iterator j;
		for (j = i->begin(); j != i->end(); ++j) {
			os << (j == i->begin() ?  '{' : ',') << (*j + 1);
		}
		os << '}';
	}
	os << '}';
}

void Permutation::read(const YAML::Node& node) {
	Reduze::read(node, perm_);
	init();
}

void Permutation::read_from_onebased(const YAML::Node& node) {
	Reduze::read(node, perm_);
	init();
	relabel(-1);
}

int Permutation::apply(int n) const {
	int res(n);
	vector<vector<int> >::const_reverse_iterator c;
	for (c = perm_.rbegin(); c != perm_.rend(); ++c) {
		if (c->size() < 2) // cycle of length 1 do nothing
			continue;
		if (c->back() == res) {
			res = c->front();
		} else {
			vector<int>::const_iterator n;
			for (n = c->begin(); n != c->end(); ++n) {
				if (*n == res) {
					res = *(n + 1);
					break;
				}
			}
		}
	}
	return res;
}

void Permutation::unique() {
	set<int> todo;
	for (vector<vector<int> >::iterator c = perm_.begin(); c != perm_.end(); ++c)
		todo.insert(c->begin(), c->end());

	vector<vector<int> > res;
	int start, next;
	while (!todo.empty()) {
		vector<int> cycle;
		start = *(todo.begin());
		next = start;
		do {
			todo.erase(next);
			cycle.push_back(next);
			next = apply(next);
		} while (next != start);

		if (cycle.size() > 1)
			res.push_back(cycle);
	}
	perm_.swap(res);
}

void Permutation::init() {
	// check for multiple occurrence of a number in the cycle
	vector<vector<int> > res;
	res.reserve(perm_.size());
	for (vector<vector<int> >::iterator c = perm_.begin(); c != perm_.end(); ++c) {
		set<int> values(c->begin(), c->end());
		if (values.size() != c->size())
			ABORT("Invalid permutation: " << *this);
		vector<int> cycle;
		if (c->size() > 1) {
			c->swap(cycle);
			res.push_back(cycle);
		}
	}
	perm_.swap(res);
	unique();
}

// PermutationSet members


void PermutationSet::make_pairs(std::set<Permutation>& s2) {
	set<Permutation> snew;
	set<Permutation>::const_iterator it1, it2;
	for (it1 = pset_.begin(); it1 != pset_.end(); ++it1) {
		for (it2 = s2.begin(); it2 != s2.end(); ++it2) {
			Permutation p = Permutation::compose(*it1, *it2);
			if (pset_.find(p) == pset_.end())
				snew.insert(p);
		}
	}
	pset_.insert(snew.begin(), snew.end());
	s2.swap(snew);
}

void PermutationSet::complete_permutations() {
	set<Permutation> tmp(pset_);
	while (!tmp.empty())
		make_pairs(tmp);
	// delete identity
	pset_.erase(Permutation());
}

// restricted growth string

restricted_growth_string::restricted_growth_string() :
		restrict_num_blocks_(0), min_block_size_(0), count_(1) {
}

restricted_growth_string::restricted_growth_string(int n,
		int restrict_num_blocks, int min_block_size) :
		rgs_(n > 0 ? n : 0, 0), max_(rgs_), restrict_num_blocks_(
				restrict_num_blocks), min_block_size_(min_block_size), count_(1) {
	if (n < 0)
		throw runtime_error("non-positive size n");
	if (restrict_num_blocks_ < 0 || restrict_num_blocks_ > n)
		throw runtime_error("invalid number of blocks");
	if (min_block_size_ < 0 || min_block_size_ > n
			|| min_block_size_ * restrict_num_blocks_ > n
			|| (min_block_size_ == 0 && n > 0))
		throw runtime_error("invalid minimal block size");
	if (restrict_num_blocks_ > 0)
		for (int i = n - restrict_num_blocks_ + 1; i < n; ++i)
			max_[i] = rgs_[i] = i - (n - restrict_num_blocks_);
	// filter partitions with block sizes greater or equal to the given minimum size
	bool good = true;
	if (!min_block_size_good())
		good = next();
	count_ = 1; // reset
	ASSERT(good); // partition still can be empty
}

bool restricted_growth_string::next() {
	bool good = true;
	do {
		good = next_no_filter();
	} while (!min_block_size_good() && good);
	if (!good)
		VERIFY(count_ == num_partitions());
	++count_;
	return good;
}

bool restricted_growth_string::min_block_size_good() const {
	if (min_block_size_ < 2)
		return true;
	std::vector<int> count(num_blocks(), 0);
	for (unsigned int i = 0; i < rgs_.size(); ++i)
		++count[rgs_[i]];
	for (int i = 0; i < num_blocks(); ++i)
		if (count[i] < min_block_size_)
			return false;
	return true;
}

bool restricted_growth_string::next_no_filter() {
	// construct next
	if (restrict_num_blocks_ == 0) {
		const int n = rgs_.size();
		for (int i = n - 1; i > 0; --i) {
			if (rgs_[i] <= max_[i - 1]) {
				max_[i] = std::max(max_[i], ++rgs_[i]);
				for (int j = i + 1; j < n; ++j) {
					rgs_[j] = 0;
					max_[j] = max_[i];
				}
				return true;
			}
		}
		return false;
	}

	// construct next with the same number of blocks
	const int p = num_blocks();
	const int n = rgs_.size();
	for (int i = n - 1; i > 0; --i) {
		if (rgs_[i] < p - 1 && rgs_[i] <= max_[i - 1]) {
			max_[i] = std::max(max_[i], ++rgs_[i]);
			int jj = n - (p - max_[i]) + 1;
			for (int j = i + 1; j < jj; ++j) {
				rgs_[j] = 0;
				max_[j] = max_[i];
			}
			for (int j = jj; j < n; ++j)
				rgs_[j] = max_[j] = p - (n - j);
			return true;
		}
	}
	return false;
}

int restricted_growth_string::num_blocks() const {
	if (rgs_.empty())
		return 0;
	ASSERT(max_[0] == 0);
	return max_[rgs_.size() - 1] + 1 /*- max_[0]*/;
}

unsigned long restricted_growth_string::num_partitions() const {
	unsigned long res = 0;
	if (restrict_num_blocks_ == 0) {
		for (unsigned int k = 0; k <= rgs_.size(); ++k)
			res += associated_stirling_2(rgs_.size(), k, min_block_size_);
	} else {
		res += associated_stirling_2(rgs_.size(), restrict_num_blocks_,
				min_block_size_);
	}
	return res;
}

} // namespace Reduze
