/*  ginacutils.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef GINACUTILS_H_
#define GINACUTILS_H_

#include <ginac/ginac.h>

namespace Reduze {

// basic GiNaC related helpers

/// helper function for user controlled output of conjugate()
DECLARE_FUNCTION_1P(ConjugateRedef)
// header to keep color structures together in some expression
DECLARE_FUNCTION_1P(Color)
/// header to keep some structures together (used by color and polarization)
DECLARE_FUNCTION_1P(Ident)
/// helper function to mark expressions, Tag(something) can be set to 1
/** currently used to indicate Feynman rules used for fermion loops **/
DECLARE_FUNCTION_1P(Tag)

/// returns true if the ex is a non negative integer and sets i to this value
bool nonneg_integer(const GiNaC::ex & e, int & i);

/// returns true if the ex is a non negative integer and sets i to this value
bool nonneg_integer(const GiNaC::ex & e, long & i);

/// converts an ex to int, aborts if not successful
int to_int(const GiNaC::ex & e);

/// converts a string to int, aborts if not successful
int to_int(const std::string & str);

/// converts a string to a non negative int, aborts if not successful
int to_nonneg_int(const std::string & str);

/// joins two lists
GiNaC::lst add_lst(const GiNaC::lst& l1, const GiNaC::lst& l2);

/// transforms an exset to a lst
GiNaC::lst set2lst(const GiNaC::exset & s);

/// transforms a lst to an exset
GiNaC::exset lst2set(const GiNaC::lst & l);

/// returns false if 'e' contains an element of 'l', true otherwise
bool freeof(const GiNaC::ex & e, const GiNaC::lst & l);

// comparison of exmap instances
int compare(const GiNaC::exmap&, const GiNaC::exmap&);

/// comparison of exmap instances
struct exmap_is_less {
	bool operator()(const GiNaC::exmap&, const GiNaC::exmap&) const;
};

/// returns true if the expression contains an object of type T
template<class T> bool has_object(const GiNaC::ex& e) {
	if (GiNaC::is_a<T>(e))
		return true;
	for (GiNaC::const_iterator i = e.begin(); i != e.end(); ++i)
		if (has_object<T> (*i))
			return true;
	return false;
}

/// appends all objects of type T to the set
template<class T> void gather_objects(const GiNaC::ex& e, GiNaC::exset& found) {
	if (GiNaC::is_a<T>(e))
		found.insert(e);
	for (GiNaC::const_iterator i = e.begin(); i != e.end(); ++i)
		gather_objects<T> (*i, found);
}

/// appends all objects of type T to the list (only once)
template<class T> void gather_objects(const GiNaC::ex& e, GiNaC::lst& found) {
	GiNaC::exset f;
	gather_objects<T> (e, f);
	for (GiNaC::exset::const_iterator s = f.begin(); s != f.end(); ++s)
		found.append(*s);
}

/// toggles the variance of all non-numeric varidx in the expression
GiNaC::ex toggle_non_numeric_varidx(const GiNaC::ex& e);

/// doc
bool has_symbol_name(const GiNaC::lst & l, const std::string & str);

/// provides alternative symbols, no clear() on output variables
void provide_alternative_symbols(const GiNaC::lst& symbols,
		GiNaC::lst& alt_symbols, GiNaC::exmap& sym2alt, GiNaC::exmap& alt2sym);

/// assert that name is non-empty and all characters are alpha-numerical
void assert_valid_symbol_name(const std::string& name);
// assert that name is non-empty and contains only lower case characters
void assert_valid_prefix_name(const std::string& name);

/// creates a new symbol with name 'name'
GiNaC::symbol create_symbol(const std::string& name);
/// creates a new real symbol with name 'name'
GiNaC::realsymbol create_real_symbol(const std::string& name);
/// creates a list with n new symbols with name "prefix" + "i", 0 < i <= n
GiNaC::lst create_symbols(const std::string& prefix, int n);
/// create new symbols with given names
GiNaC::lst create_symbols(const std::list<std::string>& s);
/// create new real symbols with given names
GiNaC::lst create_real_symbols(const std::list<std::string>& s);

GiNaC::lst subs_in_lst(const GiNaC::lst& v, const GiNaC::exmap& m,
		unsigned options = 0);

// operation on exvectors

GiNaC::exvector subs_in_exvector(const GiNaC::exvector& v,
		const GiNaC::exmap& m, unsigned options = 0);
GiNaC::exvector subs_in_exvector(const GiNaC::exvector& v, const GiNaC::ex& m,
		unsigned options = 0);
GiNaC::exvector conjugate_in_exvector(const GiNaC::exvector& v);

/// substitutes b in the right hand side of a
GiNaC::exmap substitute_rhs(const GiNaC::exmap& a, const GiNaC::exmap& b,
		unsigned options = 0);

/// substitutes b in the left hand side of a and returns a list of relationals
GiNaC::lst substitute_lhs(const GiNaC::exmap& a, const GiNaC::exmap& b,
		unsigned options = 0);

/// substitutes b in the left hand side of a and returns a map
GiNaC::exmap substitute_lhs_map(const GiNaC::exmap& a, const GiNaC::exmap& b,
		unsigned options = 0);

/// returns true if the any lhs of the exmap doesn't have e, false otherwise
bool lhs_freeof(const GiNaC::exmap& m, const GiNaC::ex& e);
/// returns true if the any rhs of the exmap doesn't have e, false otherwise
bool rhs_freeof(const GiNaC::exmap& m, const GiNaC::ex& e);

/// creates a one-to-one mapping from elements of a to elements of b
/** sizes of a and b must be equal **/
GiNaC::exmap make_translation(const GiNaC::lst& a, const GiNaC::lst& b);

/// tests whether the expression can be parsed with the symbols in the list
bool can_be_parsed_with(const GiNaC::ex& e, const GiNaC::lst& symbols);

/// throws if a symbol name appears more than once in the list
void assert_unique_symbols_and_valid_names(const GiNaC::lst& symb);

// algebraic manipulations

/// normalizes e, factors numerator and denominator
/** works also in presence of overall factor of I **/
GiNaC::ex factor_rational(const GiNaC::ex& e);

/// returns numerical factor and non-numerical factor as a two-element list
/** is able to extract numerical factor if e is a mul or a numerical quantity **/
GiNaC::lst extract_numerical_factor(const GiNaC::ex& e);

/// finds rules to substitute generic indeterminates, throws if not unique
/** e.g.: eqns = (sin(x)+sin(y)==u, sin(x)-sin(y)==v), pattern = sin(wild()) **/
GiNaC::exmap lsolve_pattern(const GiNaC::lst& eqns, const GiNaC::ex& pattern);

/// finds equations for indeterminates also for over-determined systems
/** if symbols is a lst, returns the form lst(var1==sol1,var2==sol2,...),
 ** NOTE: returned equations are necessary but may not be sufficient to solve
 ** the system, e.g. the method may just throw away some equations if it
 ** finds expressions for the indeterminates which are free of other indets.
 ** TODO: we might implement a cleaner method some day. **/
GiNaC::ex lsolve_overdetermined(const GiNaC::ex& eqns,
		const GiNaC::ex& symbols, unsigned options = 0);

GiNaC::exmap lsolve_result_to_rules(const GiNaC::ex& sol,
		const GiNaC::ex& symbols);

GiNaC::lst to_equations(const GiNaC::exmap&);

/// converts GinaC::lst of equations { a = b, c == d } to map a->b, c->d
/** if discard_identities is set, map entries a->a will be omitted **/
GiNaC::exmap equations_to_substitutions(const GiNaC::lst& eqs,
		GiNaC::exmap lhs_substitutions, bool discard_identities = false);

// compute jacobi determinant of a substitution
GiNaC::ex jacobi_determinant(const GiNaC::lst& old2new,
		const GiNaC::lst& new_vars);

// output

/// a map function for user controlled printout of conjugate()
/** This is a workaround since subs(conjugate(wild()) == ConjugateRedef(wild()))
 ** does not work as needed. */
struct subs_conjugate: public GiNaC::map_function {
	GiNaC::ex operator()(const GiNaC::ex &e);
};

class print_form: public GiNaC::print_dflt {
GINAC_DECLARE_PRINT_CONTEXT(print_form, print_dflt)
public:
	print_form(std::ostream & os, unsigned opt = 0) :
		print_dflt(os, opt) {
	}
};

class print_mma: public GiNaC::print_dflt {
GINAC_DECLARE_PRINT_CONTEXT(print_mma, print_dflt)
public:
	print_mma(std::ostream & os, unsigned opt = 0) :
		print_dflt(os, opt) {
	}
};

void print_as_mma(const GiNaC::exmap&, std::ostream&, bool linebreaks = false);

class print_maple: public GiNaC::print_dflt {
GINAC_DECLARE_PRINT_CONTEXT(print_maple, print_dflt)
public:
	print_maple(std::ostream & os, unsigned opt = 0) :
		print_dflt(os, opt) {
	}
};

void init_ginacutils();

} // namespace Reduze

#endif /* GINACUTILS_H_ */
