/*  graph.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef REDUZE_GRAPH_H_
#define REDUZE_GRAPH_H_

#include "topology.h"
#include "sector.h"

namespace Reduze {

class Crossing;
class Kinematics;

/// label for an Edge of a Graph
struct GraphEdge {
	GraphEdge(const GiNaC::ex& momentum, const GiNaC::ex& squaredmass) :
			momentum_(momentum), squaredmass_(squaredmass) {
	}
	GiNaC::ex momentum_;
	GiNaC::ex squaredmass_;
	friend std::ostream& operator<<(std::ostream&, const GraphEdge&);
};

/// graph with edges labeled by momenta and masses
class Graph: public Topology {
public:
	// construction and destruction
	Graph();
	Graph(const Topology&);
	virtual ~Graph();

	void swap(Graph& other);
	const GraphEdge& graphedge(int edge_id) const;

	// momenta

	/// sets symbols (e.g. k1, k2) for loop integration momenta
	void set_loop_momenta(const GiNaC::lst& symbols);
	/// returns symbols for loop integration momenta
	const GiNaC::lst& loop_momenta() const;
	/// clears momentum of a specific edge
	virtual void clear_momentum(int edge);
	/// clears momenta of internal edges (loop edges + 1-particle red. edges)
	virtual void clear_internal_momenta();
	/// tries to set momentum, throws runtime_error if momentum is not conserved
	virtual void set_momentum(int edge_id, const GiNaC::ex& mom);
	/// employs momentum conservation at node to set an undef. mom. if unique
	virtual void complete_momentum(int node_id, bool recurse = true);
	/// employs momentum conservation at all nodes to set undef. mom. if unique
	virtual void complete_momenta();
	/// sets internal momenta, loop momenta must be set before, sets all masses to sqmass
	virtual void set_internal_momenta();
	virtual void set_internal_masses(const GiNaC::ex& sqmass);
	/// returns momentum of an edge
	virtual GiNaC::ex get_momentum(int edge_id) const;
	std::set<int> get_edges_undefined_momentum() const;
	/// get the propagators of the edges which contain loop momenta (\todo remove)
	std::vector<Propagator> get_propagators_with_loop_momenta() const;

	const std::map<int, GraphEdge>& graphedges() const;
	virtual Edge insert_degree_2_node(int edge_id);
	void substitute_momenta(const GiNaC::ex& rules);
	void substitute_momenta(const GiNaC::exmap& rules);

	// squared masses

	virtual void set_squaredmass(int edge_id, const GiNaC::ex& mass);
	virtual void clear_squaredmasses();
	/// returns squared mass of an edge
	virtual GiNaC::ex get_squaredmass(int edge_id) const;
	virtual bool is_undefined_square_mass(int edge_id) const;
	/// returns an ordered list of different masses
	/** throws if some mass is undefined **/
	virtual GiNaC::lst get_different_masses() const;

	// graph construction

	/// tries to insert the edge with given edge_id via cleaving of node 'node' and respecting momentum conservation to the graph
	/** The node is required to have more than 3 adjacent edges.
	 ** All new graphs with the momentum of the graph edge inserted are appended to the result.
	 ** The input graph is still required not to have self-loops **/
	void insert_graph_edge_to_node(int edge_id, const GraphEdge& graphedge,
			int node, std::list<Graph>& graphlist) const;
	/// tries to add the graph edge to the graph, appends all matches to the result
	void insert_graph_edge(int edge_id, const GraphEdge& graphedge,
			std::list<Graph>& graphlist) const;
	/// tries to add all graph edges to the graph, appends all matches to the result
	void insert_graph_edge(const std::map<int, GraphEdge>& gedges,
			std::list<Graph>& graphlist) const;

	// graph manipulation

	/** Creates a tadpole graph by removing external legs and external nodes.
	 ** Potential degree 2 nodes will be removed.
	 ** The external momenta are set to zero. **/
	virtual void remove_external_legs();

	/** Merges multiple external legs entering the same vertex into one. **/
	virtual void merge_external_legs();

	/** Creates a tadpole graph by joining all external nodes in a new vertex.
	 ** potential degree 2 nodes will be removed. The number of QFT-loops is
	 ** increased by the number of independent external momenta. **/
	virtual void join_external_nodes();

	/// removes all but one edge with the same propagator, leaves external legs untouched
	void contract_multiple_internal_propagators();
	/// disconnects vacuum components from the rest of the graph
	void disconnect_vacuum_components();
	/// reverse edges and the momentum flow
	void reverse_edges_and_momentum_flow();

	/// changes underlying Topology by twists to minimize canonical label
	/** Afterwards, the canonical label of the Graph is unique for its matroid
	 ** isomorphism class and masses.
	 ** From the set of graphs with equal minimal canonical label (permuations
	 ** of external nodes allowed) a graph is chosen with the minimal
	 ** canonical label computed with fixed external nodes to remove
	 ** ambiguities of different crossings of external legs with the
	 ** same canonical label.
	 ** Assumption: graph contains at most one connected component with
	 ** external edges. Here, an external edge connects exclusively to some
	 ** node (momentum assignment is irrelevant). This restriction could
	 ** probably be removed rather easily if needed. */
	void minimize_by_twists();

	// shift

	/** Finds a shift and a crossing such that graph1 (transformed with the shift)
	 ** is equal to graph2 (with applied inverse crossing).
	 ** The kinematics is defined through the input crossing.
	 ** If the given node symmetries of graph2 are not empty then the shift
	 ** is chosen such that the crossing gets minimal. **/
	static GiNaC::ex find_shift(const Graph& graph1,
			const std::pair<CanonicalLabel, CanonicalRelabeling>& label1,
			const Graph& graph2,
			const std::pair<CanonicalLabel, CanonicalRelabeling>& label2,
			GiNaC::exmap& shift, Crossing& crossing,
			const std::list<std::map<int, int> >& sym2);
	/// get shifts of loop momenta to permute propagators
	/** compares with all node permutations from the symmetry group
	 ** and permutes multi-edges,
	 ** shifts have |det| = 1, only crossings which are identical to
	 ** identity crossing (leave kinematic invariant unchanged) are allowed */
	void find_symmetry_shifts(std::list<GiNaC::exmap>& shifts,
			const Kinematics* kin) const;

	/// determines the symmetry shifts from edge permutations via line graph
	void find_edge_symmetry_shifts(std::list<GiNaC::exmap>& shifts,
			const Kinematics* kin) const;

	// combinatorial matcher

	/// finds possible shifts of loop momenta to match an integral family
	/** inserts into shift (at most one for each sector if ambiguous),
	 ** shifts4sub[subtopo_id][sector] = shift for subtopology of graph
	 ** returns true if at least one match for the graph was found **/
	virtual bool match_to_integralfamily(const IntegralFamily* ic);
	void print_sector_matches() const;
	const std::pair<const Sector, GiNaC::exmap>* shift_to_sector() const;

	// output

	virtual void print_dot(const std::string& filename) const;
	virtual void print_dot(std::ostream& of) const;
	virtual void print_mma(std::ostream& of) const;
	friend YAML::Emitter& operator<<(YAML::Emitter&, const Graph&);
	friend std::ostream& operator<<(std::ostream&, const Graph&);
	void read(const YAML::Node& n, const GiNaC::lst& loopmoms,
			const GiNaC::lst& ext_kin);

	/// Returns the edge mass coloring and the masses
	/** For each edge the corresponding mass (masses are numbered starting
	 ** with 0) **/
	virtual std::pair<std::list<std::map<int, int> >, edge_attributes> get_edge_coloring() const;

protected:
	/// only for intermediate purposes, mom. conserv. is invariant of Topology
	/** Returns the sum of the momenta at internal node 'node'. Returns zero for external nodes
	 ** or nodes with attached legs with undefined momenta. **/
	GiNaC::ex momentum_sum(int node) const;
	/** Returns True if momentum is conserved at internal node 'node' and False otherwise.
	 ** If 'node' is internal or if 'node' contains attached legs with undefined momenta always True is returned. **/
	bool is_momentum_conserved(int node) const;
	virtual Graph* clone() const;
	std::list<Graph*> find_non_isomorphic_twists() const;
	std::list<Graph*> find_twists() const;

private:
	virtual void removed_edge(int id);
	/// clears the Graph members
	virtual void swapped();

private:
	GiNaC::symbol undefined;
	std::map<int, GraphEdge> graphedges_; // GraphEdges by egde id
	GiNaC::lst loop_momenta_;

	// members set by match_to_integralfamily(...)
	std::map<int, std::map<Sector, GiNaC::exmap> > subgraph_to_sector_;
	std::map<Sector, GiNaC::exmap> graph_to_sector_;
};

//
//
//

/// a graph with a sector
class SectorGraph: public Graph {
public:
	SectorGraph();
	virtual ~SectorGraph();
	SectorGraph(const Sector& sec, const Graph& graph);
	SectorGraph(const Sector& sec, const Graph& graph,
			const std::map<int, int>& edge_by_prop_pos);
	SectorGraph(const YAML::Node&);

	const Sector& sector() const;
	const std::map<int, int>& edge_id_by_prop_pos() const;

	std::set<int> get_different_propagator_types() const;

	void swap(SectorGraph& other);

	/// find shifts of loop momenta which permute propagators
	/** unless as in class Graph this version searches for symmetry shifts
	 ** of this graph and all twists of it which can in some examples give
	 ** more shifts **/
	void find_symmetry_shifts(std::list<GiNaC::exmap>& shifts,
			bool examine_twists) const;
	/// find the symmetry shifts via line graph
	void find_edge_symmetry_shifts(std::list<GiNaC::exmap>& shifts,
			bool examine_twists) const;

	friend YAML::Emitter& operator<<(YAML::Emitter&, const SectorGraph&);
	friend std::ostream& operator<<(std::ostream&, const SectorGraph&);

	virtual std::pair<std::list<std::map<int, int> >, edge_attributes> get_edge_coloring() const;

protected:
	virtual SectorGraph* clone() const;

private:
	void set_edge_id_by_prop_pos();
	void verify_edge_id_by_prop_pos() const;

	/// one element list for a sector (no default constructor)
	std::list<Sector> sector_;
	std::map<int, int> edge_id_by_prop_pos_;
};

/// helper class to store a sectorgraph with canonical label
class SectorGL {
public:

	SectorGL() {
	}
	explicit SectorGL(const SectorGraph& secgraph) :
			sectorgraph_(secgraph) {
		label_pair_ = sectorgraph_.find_canonical_label();
	}

	bool operator<(const SectorGL& other) const {
		return sectorgraph_.sector() < other.sectorgraph_.sector();
	}

	inline const SectorGraph& sectorgraph() const {
		return sectorgraph_;
	}

	inline const Sector& sector() const {
		return sectorgraph_.sector();
	}

	inline const std::pair<CanonicalLabel, CanonicalRelabeling> label_pair() const {
		return label_pair_;
	}

	void swap(SectorGL& other) {
		sectorgraph_.swap(other.sectorgraph_);
		label_pair_.first.swap(other.label_pair_.first);
		label_pair_.second.swap(other.label_pair_.second);
	}

	void swap(SectorGraph& sg) {
		sectorgraph_.swap(sg);
		label_pair_ = sectorgraph_.find_canonical_label();
	}

	SectorGL(const YAML::Node&);
	friend YAML::Emitter& operator<<(YAML::Emitter&, const SectorGL&);
	friend std::ostream& operator<<(std::ostream& os, const SectorGL& sg);

private:
	SectorGraph sectorgraph_;
	std::pair<CanonicalLabel, CanonicalRelabeling> label_pair_;
};

//
//
//

// global functions

/** creates all graphs by contracting one internal edge
 ** does not contract self-loops **/
void contract_internal_edges(const Graph& g,
		std::map<CanonicalLabel, Graph>& contracted);

/** for all input graphs: creates all graphs by contracting one internal edge
 ** does not contract self-loops **/
void contract_internal_edges(const std::map<CanonicalLabel, Graph>& g,
		std::map<CanonicalLabel, Graph>& contracted);

} // namespace Reduze

#endif /* REDUZE_GRAPH_H_ */
