/*  job_collectintegrals.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_collectintegrals.h"
#include "functions.h"
#include "sector.h"
#include "amplitude.h"
#include "equation.h"
#include "equationlist.h"
#include "files.h"
#include "filedata.h"
#include "sectormappings.h"

#include "ginacutils.h" // for streams
using namespace std;

namespace Reduze {

// register job type at JobFactory
namespace {
JobProxy<CollectIntegrals> dummy;
}

bool CollectIntegrals::find_dependencies(const set<string>& outothers,//
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	//find_dependencies_all_sectormappings(outothers, in, auxjobs);
	in.push_back(input_filename_);
	out.push_back(output_filename_);
	return true;
}

void CollectIntegrals::init() {
	if (input_filename_.empty() || output_filename_.empty())
		throw runtime_error("input or output file undefined");
	if (output_filename_ == input_filename_)
		ABORT("Input and output files are identical: " << input_filename_);
}

std::string CollectIntegrals::get_description() const {
	return string("collect integrals in ") + short_filename(input_filename_);
}

void CollectIntegrals::run_serial() {
	bool have_equations = true;
	LOG("Reading file " << input_filename_);
	set<INT> ints;
	try {
		LOGX("Assuming file contains linear combinations/equations");
		InFileLinearCombinations in(input_filename_);
		LinCombHLight lc;
		while (in.get_noabort(lc, true))
			lc.find_INT(ints);
		in.close();
	} catch (exception& e1) {
		ints.clear();
		LOGX("Failed to parse linear combinations:");
		LOGX(e1.what() << "\n");
		try {
			LOGX("Assuming file contains list of integrals.");
			InFileINTs in(input_filename_);
			in.get_all(ints);
			in.close();
		} catch (exception& e2) {
			ints.clear();
			LOGX("Failed to parse integrals:");
			LOGX(e2.what() << "\n");
			ERROR("Failed to parse integrals in file " << input_filename_);
		}
		have_equations = false;
	}
	LOGX("Succeeded to parse the file assuming it contains " << //
			(have_equations ? "lin.combs/equations" : "a list of integrals"));

	OutFileINTs out(output_filename_);
	LOG("  found " << ints.size() << " integrals");
	int selected = 0;
	for (set<INT>::const_iterator i = ints.begin(); i != ints.end(); ++i)
		if (filter_.contains(int(i->t()), *i)) {
			out << *i;
			++selected;
		}
	out.finalize();
	LOG("  wrote " << selected << " integral(s) to file " << output_filename_);
}

}
