/*  job_setupsectormappings.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef JOB_SETUPSECTORMAPPINGS_H_
#define JOB_SETUPSECTORMAPPINGS_H_

#include "job.h"
#include "sector.h"
#include "graph.h"
#include "yamlutils.h"

namespace Reduze {

class IntegralFamily;

/// Job to set up zero sectors and shifts between sectors via graphs
/** Setup:
 *
 *  Different types of zero sectors are considered
 *  zeros because of determinant condition,
 *  zeros with a graph,
 *  zeros without a graph
 *
 *  Zero test:
 *  The corner integral (all propagator exponents 0 or 1) of a sector
 *  is found to vanish using IBPs for various seed integrals in the sector.
 *
 *  Topologies that have non-vanishing subtopologies are defined as non-vanishing
 */

class SetupSectorMappings: public Job {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("setup_sector_mappings");
		s.set_short_description("Finds shifts between sectors via graphs.");
		s.set_long_description(""//
					"Job to set up sector mappings files for all defined integral"
					" families."
					" Typically, this should be the first job to run."
					" This job finds shifts between sectors by generating"
					" graphs and identifying isomorphic graphs."
					" Graphs without external legs can be minimized within"
					" their matroid isomorphism class by twisting."
					" Shifts of loop momenta are restricted to those with a"
					" modulus 1 Jacobi determinant, external momenta may be permuted."
					" For each representative sector of an isomorphism class"
					" an explicit zero test is performed: the algorithm reduces a"
					" small set of IBPs and if the scalar (corner) integral is found"
					" to reduce to zero, the sector is considered to be zero."
					" All results are stored in the corresponding sectormappings*.yaml"
					" file which is used to eliminate redundant and zero sectors in future"
					" calculations."
					" DOT files for the generated graphs are stored in the "
					" subdirectory 'graphs'. They can be used to generate "
					" images using e.g. the program 'dot' from the GraphViz "
					" package.");
		s.add_option("find_sector_relations", false, "boolean", ""//
					"Whether to find shifts of loop and external momenta,"
					" which map a sector to a lower sector."
					" Such sector relations are found by determining graph (or matroid)"
					" isomorphisms and deriving shifts of momenta from it."
					" See help for 'sector' to find out about sector ordering.");
		s.add_option("find_sector_symmetries", false, "boolean", ""//
					"Whether to find shifts of loop and external momenta,"
					" which map a sector onto itself."
					" Such symmetry shifts are found by checking the symmetry"
					" group of the vertices of the graph and considering all permutations"
					" of multi-edges with the same mass.  A possible crossing"
					" of external legs is required to leave the kinematic"
					" invariants unchanged.");
		s.add_option("examine_twists_for_symmetries", false, "boolean", ""//
					"In case find_sector_symmetries is true this keyword defines"
					" whether to find the symmetry shifts of all twisted graphs"
					" instead of only the selected graph representing the sector."
					" This can in some cases give more symmetry shifts.");
		s.add_option("find_zero_sectors", false, "boolean", ""//
					"Whether zero sectors for sectors with graphs should be found by"
					" an explicit IBP reduction. Sectors which vanish trivially since"
					" their number of independent propagators is less than the number"
					" of loop momenta are detected anyway.");
		s.add_option("find_graphless_zero_sectors", false, "boolean", ""//
					"Whether a zero test for the sectors without a graph should"
					" be performed. The test is only performed if the option"
					" \"find_zero_sectors\" is also set to true.");
		s.add_option("minimize_graphs_by_twists", false, "boolean", ""//
					"With this option set to true the generated graphs are twisted"
					" to a unique equivalent of its matroid class.");
		s.add_option("construct_minimal_graphs", false, "boolean", ""//
					"With this option set to true the graphs are constructed"
					" by inserting the propagators in all possible ways and the"
					" minimal graph is selected to represent the sector."
				    " This approach is a (slower) alternative to constructing"
				    " a single graph and minimize it by twists."
					" If set to true the option \"minimize_graphs_by_twists\""
					" can be set to false, but it should not lead to problems"
					" (other than speed) to enable both at the same time.");
		s.add_option("setup_crossings", false, "boolean", ""//
					"Whether crossings of external legs of the given"
					" kinematics should be setup (see job setup_crossings)."
					" In any case, crossed integral families are generated for"
					" all crossings defined in the crossings file.");
		s.add_option("minimize_target_crossings", false, "boolean", ""//
					"With this option set to true the shifts between sectors"
					" are derived in such a way that they lead to a minimal"
					" crossing of the target sector. For this, the node symmetry"
					" group of the graph of the target sector is calculated"
					" and the minimal crossing w.r.t the graph of the"
					" source sector derived.");
		s.add_option("allow_general_crossings", false, "boolean", ""//
					"With this option set to false shifts between sectors"
					" are rejected if they involve a crossing that changes"
					" the kinematic invariants.");
		s.add_option("verify_permutation_symmetries", false, "boolean", ""//
					"Whether the permutation symmetries of the integral families"
					" should be verified.");
		s.add_options(Job::yaml_spec());
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}
	SetupSectorMappings();
	virtual ~SetupSectorMappings() {
	}
	virtual void run_serial();
	virtual std::string get_description() const;
	virtual bool find_dependencies(const std::set<std::string>& outothers,//
			std::list<std::string>& in, std::list<std::string>& out, std::list<
					Job*>& auxjobs);
	static bool get_dependencies(
			const std::set<std::string>& outothers, //
			std::list<std::string>& in, std::list<std::string>& out,
			std::list<Job*>& auxjobs, bool setup_crossings, bool verify_perms,
			bool is_conditional);

protected:
	virtual void add_auto_options() {
		add_auto_io("find_sector_relations", find_sector_relations_);
		add_auto_io("find_sector_symmetries", find_sector_symmetries_);
		add_auto_io("examine_twists_for_symmetries", examine_twists_for_symmetries_);
		add_auto_io("find_zero_sectors", find_zero_sectors_);
		add_auto_io("find_graphless_zero_sectors", find_graphless_zero_sectors_);
		add_auto_io("minimize_graphs_by_twists", minimize_graphs_by_twists_);
		add_auto_io("construct_minimal_graphs", construct_minimal_graphs_);
		add_auto_io("setup_crossings", setup_crossings_);
		add_auto_io("minimize_target_crossings", minimize_target_crossings_);
		add_auto_io("allow_general_crossings", allow_general_crossings_);
		add_auto_io("verify_permutation_symmetries", verify_permutation_symmetries_);
	}
	virtual void init();

private:
	bool find_sector_relations_;
	bool find_sector_symmetries_;
	bool examine_twists_for_symmetries_;
	bool find_zero_sectors_;
	bool find_graphless_zero_sectors_;
	bool minimize_graphs_by_twists_;
	bool construct_minimal_graphs_;
	bool setup_crossings_;
	bool minimize_target_crossings_;
	bool allow_general_crossings_;
	bool verify_permutation_symmetries_;

private:
	/** sets up the sector mapping, updates the shift target sectors
	 ** determines the graphs of the sectors
	 ** finds shifts to sectors in 'shift_targets' (must be sectors from lower integral families) and to itself
	 ** finds zero sectors and sectors without a graph **/
	void setup(SectorMappings* sm, std::set<SectorGL>& shift_targets);
	void find_crossed_relations(const IntegralFamily* ic, const std::set<
			SectorGL>& sectors);
	void make_graph_directories(const IntegralFamily* ic, std::string& all_dir,
			std::string& shift_targets_dir) const;
	std::map<Sector, std::list<std::map<int, int> > > cached_node_perm_;
};

}

#endif /* JOB_SETUPSECTORMAPPINGS_H_ */
