/*  jobqueue.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "jobqueue.h"

#include <iomanip>
#include <unistd.h>
#include "job.h"
#include "functions.h"
#include "files.h"
#include "job_setupsectormappings.h"
#include "job_setupsectormappingsalt.h"
#include "job_setupcrossings.h"
#include "job_verifypermutationsymmetries.h"

using namespace std;

namespace Reduze {

JobQueue::JobQueue() :
	timeout_(0.), max_parallel_jobs_(0), time_interval_analysis_(30.),
			max_workers_release_(5), max_job_id_(0) {
}

JobQueue::~JobQueue() {
	for (list<Job*>::iterator j = jobs_.begin(); j != jobs_.end(); ++j)
		delete (*j);
}

void JobQueue::print_jobs(ostream& os) {
	os << "\nList of available job types:\n\n";
	map<string, YAMLSpec> specs;
	set<string> keys = YAMLFactory::instance().keywords();
	for (set<string>::const_iterator k = keys.begin(); k != keys.end(); ++k) {
		YAMLConfigurable* c = YAMLFactory::instance().create_object(*k);
		if (dynamic_cast<Job*> (c))
			specs[*k] = c->yaml_spec_link();
		delete c;
	}
	map<string, YAMLSpec>::const_iterator s;
	for (s = specs.begin(); s != specs.end(); ++s) {
		os << setw(35) << left << (s->first + ": ")
				<< s->second.short_description() << '\n';
	}
	os << '\n';
}

void JobQueue::insert(const std::string& keyword, const YAML::Node& job_node) {
	YAMLConfigurable* c = YAMLFactory::instance().create_object(keyword);
	Job* job = dynamic_cast<Job*>(c);
	if (!job)
		throw runtime_error(keyword + " is not job type");
	job->read(job_node);
	insert(job);
}

void JobQueue::insert(Job* job) {
	insert(job, jobs_.end());
	// resolve_dependencies();
}

list<Job*>::iterator JobQueue::insert(Job* job, list<Job*>::iterator pos) {
	ASSERT(job != 0);
	job->set_id(++max_job_id_);
	list<Job*>::iterator inserted = jobs_.insert(pos, job);
	job_by_id_[job->id()] = job;
	job_status_[job->id()] = Job::Pending;
	LOGX("Inserted new job in queue: " << job_string(job));
	return inserted;
}

void JobQueue::set_job_status(int id, Job::Status status) {
	if (job_by_id_.find(id) == job_by_id_.end())
		ABORT("Unknown job id " << id);
	job_status_[id] = status;
	if (status == Job::Done) {
		// check if all provided files really readable
		const list<string>& p = output_of_id_[id];
		for (list<string>::const_iterator f = p.begin(); f != p.end(); ++f)
			if (!is_readable_file(*f))
				ABORT("Can't read file " << *f << "\n"
						<< "which was supposed to be provided by "
						<< job_string(job_by_id_[id]));
		// resolve_dependencies();
	}
}

Job::Status JobQueue::job_status(int id) const {
	map<int, Job::Status>::const_iterator sit = job_status_.find(id);
	ASSERT(sit != job_status_.end());
	return sit->second;
}

string JobQueue::job_string(const Job* j) const {
	return j->to_short_string(max_job_id_);
}

size_t JobQueue::num_jobs() const {
	return jobs_.size();
}

void JobQueue::run() {
	///\todo: implement timeout feature in serial mode (?)
	while (insert_next_joblist()) {
		run_inserted_jobs();
		if (is_completed())
			LOG("Completed jobs in jobqueue\n");
		else
			ABORT("Failed to complete jobs for some unknown reason");
	}

	if (is_completed()) {
		LOG("Completed all jobs");
	} else {
		ABORT("Failed to complete all jobs for some unknown reason");
	}
}

bool JobQueue::insert_next_joblist() {
	if (sorted_job_lists_.empty())
		return false;
	list<pair<string, YAML::Node*> >& jobs = *sorted_job_lists_.begin();
	list<pair<string, YAML::Node*> >::const_iterator o;
	for (o = jobs.begin(); o != jobs.end(); ++o) {
		insert(o->first, *o->second);
		delete o->second;
	}
	jobs.clear();
	sorted_job_lists_.erase(sorted_job_lists_.begin());
	resolve_dependencies();
	LOG("Updated " << *this);
	return true;
}

void JobQueue::run_inserted_jobs() {
	///\todo: implement timeout feature in serial mode (?)
	while (Job* job = provide_runnable_job()) {
		Timer jobtime;
		LOGX("\nDate: " << time_string());
		LOGX("Run ID: " << RunIdentification::instance()->string() << "\n");
		LOG("Starting " << job_string(job));
		Timer timer;

		// set logging destination
		string jlogfn = Files::instance()->get_safe_log_filename_for_job(job);
		std::ofstream jlog(jlogfn.c_str(), ios_base::app);
		if (!jlog)
			ABORT("Can't open logging file " << jlogfn);
		streambuf* mainbuf = rlog2.rdbuf();
		rlog2.rdbuf(jlog.rdbuf()); // set job logging file
		rlog1 << "\n";
		LOGX("\nDate: " << time_string() << "\n");
		YAML::Emitter ys;
		job->print(ys);
		LOGX(ys.c_str() << "\n");

		// perform the job
		job->run_serial();

		// post process
		rlog2 << "\nJob done (total time: " << fixed << setprecision(0)
				<< jobtime.get_wall_time() << resetiosflags(ios::fixed)
				<< " s)\n";
		rlog2.rdbuf(mainbuf); // restore main logging destination
		set_job_status(job->id(), Job::Done);
		timer.pause();
		double tcpu = timer.get_cpu_time();
		double twall = timer.get_wall_time();
		LOG("Completed Job(" << job->id() << ")" << " [Time: " //
				<< (int)twall << "s, CPU: " << (int)tcpu << "s]");

		if (resolve_dependencies()) {
			LOG("Extended " << *this);
		}
	}

//	if (is_completed()) {
//		LOG("Completed all jobs");
//	} else {
//		ABORT("Failed to complete all jobs for some unknown reason");
//	}
}

namespace {
bool depends_on_sector_mappings(const Job* job) {
	return dynamic_cast<const SetupCrossings*>(job) == 0
			&& dynamic_cast<const VerifyPermutationSymmetries*>(job) == 0
			&& dynamic_cast<const SetupSectorMappings*>(job) == 0
			&& dynamic_cast<const SetupSectorMappingsAlt*>(job) == 0;
}
}

void JobQueue::waitforfile(const string& file, const Job* j, int timeout) {
	string errmsg;
	unsigned int delta = 500000;
	int ntries = timeout / (delta / 1000000.);
	bool first = true;
	while (!is_readable_file(file)) {
		if (--ntries <= 0)
			ERROR("could not resolve dependency problem:\n" <<
					"missing file '" << file << "' needed by job " << job_string(j));
		if (first)
			LOGN("waiting for file '" << file << "' to become available");
		else
			LOGN(".");
		first = false;
		usleep(delta);
	}
}

bool JobQueue::resolve_dependencies(int timeout) {
	LOGX("Updating job queue dependencies");
	bool created_new_jobs = false;
	set<string> allout; // all pending output files up to some position in queue
	bool changes_before = false;
	for (list<Job*>::iterator j = jobs_.begin(); j != jobs_.end();) {
		int id = (*j)->id();
		// query dependencies and create aux jobs if necessary
		if (input_of_id_.find(id) == input_of_id_.end()) {
			// virgin or incomplete previous lookup
			changes_before = true;
			list<string> in, out; // input and output of current job
			list<Job*> aux;
			// for now we let any job (other than setup jobs) depend on sector mappings by default
			bool deps_known;
			if (!depends_on_sector_mappings(*j))
				deps_known = (*j)->find_dependencies(allout, in, out, aux);
			else
				deps_known = //
						find_dependencies_all_sectormappings(allout, in, aux) && //
								(*j)->find_dependencies(allout, in, out, aux);
			// try to convert file names to a unique form
			for (list<string>::iterator f = in.begin(); f != in.end(); ++f)
				*f = get_canonical_filename(*f);
			for (list<string>::iterator f = out.begin(); f != out.end(); ++f)
				*f = get_canonical_filename(*f);
			if (deps_known) {
				input_of_id_[id] = in; // use also to remember lookup was complete
				output_of_id_[id] = out;
			}
			if (!aux.empty()) { // insert aux jobs before current job
				created_new_jobs = true;
				list<Job*>::const_reverse_iterator a;
				for (a = aux.rbegin(); a != aux.rend(); ++a)
					j = insert(*a, j);
				continue;
			}
			if (!deps_known) {
				LOGX("Incomplete dependencies for " << job_string(*j));
				break;
			}
		}
		if (changes_before) {
			// set up dependencies
			set<int>& deps = dependencies_[id];
			const list<string>& i = input_of_id_[id];
			const list<string>& o = output_of_id_[id];
			for (list<string>::const_iterator f = i.begin(); f != i.end(); ++f)
				if (job_for_file_.find(*f) != job_for_file_.end()) {
					deps.insert(job_for_file_[*f]);
					references_[job_for_file_[*f]].insert(id);
				} else if (job_status_[id] == Job::Pending) {
					waitforfile(*f, *j, timeout);
				}
			/// skip job if conditional and all output files are/will be available
			if ((job_status_[id] == Job::Pending && (*j)->is_conditional())) {
				list<string>::const_iterator f;
				for (f = o.begin(); f != o.end(); ++f) // find any new output file
					if (allout.find(*f) == allout.end() && !is_readable_file(*f)) {
						break;
					}
				if (f == o.end()) { // no new output file
					job_status_[id] = Job::Skipped;
					LOG("Skipping conditional " << job_string(*j));
				}
			}
			if (job_status_[id] != Job::Skipped)
				for (list<string>::const_iterator f = o.begin(); f != o.end(); ++f)
					job_for_file_[*f] = id;
			if (job_status_[id] == Job::Pending)
				allout.insert(output_of_id_[id].begin(), output_of_id_[id].end());
		}
		++j;
	}
	LOGXX("Done updating job queue dependencies");
	return created_new_jobs;
}

Job* JobQueue::find_most_referenced_runnable_job() {
	Job* best = 0;
	for (list<Job*>::iterator j = jobs_.begin(); j != jobs_.end(); ++j) {
		int id = (*j)->id();
		if (job_status_[id] != Job::Pending)
			continue;
		if (dependencies_.find(id) == dependencies_.end())
			continue;
		const set<int>& deps = dependencies_[id];
		set<int>::const_iterator d;
		for (d = deps.begin(); d != deps.end(); ++d) {
			bool is_pseudo_id = (*d <= 0);
			if (is_pseudo_id || job_status_[*d] == Job::Pending
					|| job_status_[*d] == Job::Running)
				break;
		}
		bool anydeps = (d != deps.end());
		if (anydeps)
			continue;
		if (best == 0 || (references_[id].size()
				> references_[best->id()].size()))
			best = *j;
	}
	return best;
}

Job* JobQueue::provide_runnable_job() {
	if (max_parallel_jobs_ > 0 && num_running_jobs() >= max_parallel_jobs_)
		return 0;
	return find_most_referenced_runnable_job();
}

bool JobQueue::is_completed() const {
	return !has_pending_jobs() && !has_running_jobs();
}

bool JobQueue::has_pending_jobs() const {
	for (list<Job*>::const_iterator j = jobs_.begin(); j != jobs_.end(); ++j)
		if (job_status((*j)->id()) == Job::Pending)
			return true;
	return false;
}

size_t JobQueue::num_running_jobs() const {
	size_t count = 0;
	for (list<Job*>::const_iterator j = jobs_.begin(); j != jobs_.end(); ++j)
		if (job_status((*j)->id()) == Job::Running)
			++count;
	return count;
}

bool JobQueue::has_running_jobs() const {
	for (list<Job*>::const_iterator j = jobs_.begin(); j != jobs_.end(); ++j)
		if (job_status((*j)->id()) == Job::Running)
			return true;
	return false;
}

double JobQueue::timeout() const {
	return timeout_;
}

size_t JobQueue::max_parallel_jobs() const {
	return max_parallel_jobs_;
}

double JobQueue::time_interval_analysis() const {
	return time_interval_analysis_;
}

size_t JobQueue::max_workers_release() const {
	return max_workers_release_;
}

std::ostream& operator<<(ostream& os, const JobQueue& q) {
	q.print(os, false);
	return os;
}

void JobQueue::print_job(const Job* job, std::ostream& os, bool verbose) const {
	int id = job->id();
	os << "  [";
	if (job_status(id) == Job::Pending)
		os << "p";
	else if (job_status(id) == Job::Running)
		os << "r";
	else if (job_status(id) == Job::Skipped)
		os << "s";
	else if (job_status(id) == Job::Done)
		os << "c";
	os << "] " << job_string(job);
	map<int, set<int> >::const_iterator deps = dependencies_.find(id);
	if (deps != dependencies_.end() /*&& !deps->second.empty()*/) {
		os << "  [dep: ";
		set<int>::const_iterator d;
		for (d = deps->second.begin(); d != deps->second.end(); ++d)
			os << (d == deps->second.begin() ? "" : ",") << *d;
		os << "]";
	}
	os << endl;
	if (verbose) {
		map<int, list<string> >::const_iterator p = output_of_id_.find(id);
		if (p != output_of_id_.end()) {
			list<string>::const_iterator f;
			for (f = p->second.begin(); f != p->second.end(); ++f)
				os << "       provides: " << *f << endl;
		}
	}
}

void JobQueue::print(ostream& os, bool verbose) const {
	os << "job queue:" << endl;
	for (list<Job*>::const_iterator j = jobs_.begin(); j != jobs_.end(); ++j)
		print_job(*j, os, verbose);
}

void JobQueue::read(const std::string& jobfile) {
	std::ifstream fin(jobfile.c_str());
	if (!fin)
		throw missing_file_error(string("Can't open file ") + jobfile);
	YAML::Parser parser(fin);
	YAML::Node doc;
	parser.GetNextDocument(doc);
	if (doc.FindValue("timeout")) {
		doc["timeout"] >> timeout_;
		if (timeout_ > 0.)
			LOG("Setting timeout limit to " << timeout_ << " s");
	}
	if (doc.FindValue("max_parallel_jobs"))
		doc["max_parallel_jobs"] >> max_parallel_jobs_;
	if (doc.FindValue("time_interval_analysis"))
		doc["time_interval_analysis"] >> time_interval_analysis_;
	if (doc.FindValue("max_workers_release"))
		doc["max_workers_release"] >> max_workers_release_;

	const YAML::Node& jobs_node = doc["jobs"];

	// sort and partition the jobs
	// sorted_job_lists_ = {
	//   {setup_crossing_1}, ..., {setup_crossing_n},
	//   {verify permutation symmetries jobs},
	//   {setup mapping_1 and auxiliary jobs},..., {setup mapping_n and auxiliary jobs},
	//   {remaining regular jobs}
	// }
	ASSERT(sorted_job_lists_.empty());
	list<list<pair<string, YAML::Node*> > > setup_crossings_jobs;
	list<pair<string, YAML::Node*> > verify_permsyms_jobs;
	list<list<pair<string, YAML::Node*> > > setup_mappings_jobs;
	list<pair<string, YAML::Node*> > regular_jobs;
	//list<pair<string, YAML::Node*> > ordered_setup_jobs;
	for (YAML::Iterator n = jobs_node.begin(); n != jobs_node.end(); ++n) {
		if (n->size() != 1)
			throw runtime_error("expected one job but got many entries");
		string key;
		n->begin().first() >> key;
		YAML::Node* p_node = n->begin().second().Clone().release();
		if (key == SetupCrossings::yaml_spec().keyword()) {
			list<pair<string, YAML::Node*> > sc;
			sc.push_back(make_pair(key, p_node));
			setup_crossings_jobs.push_back(sc);
		} else if (key == VerifyPermutationSymmetries::yaml_spec().keyword()) {
			verify_permsyms_jobs.push_back(make_pair(key, p_node));
		} else if (key == SetupSectorMappings::yaml_spec().keyword() || //
				key == SetupSectorMappingsAlt::yaml_spec().keyword()) {
			list<pair<string, YAML::Node*> > sm;
			sm.push_back(make_pair(key, p_node));
			setup_mappings_jobs.push_back(sm);
		} else {
			regular_jobs.push_back(make_pair(key, p_node));
		}
	}

	// for convenient upgrade of existing projects (2.0 -> 2.1 or later)
	const string xfile = Files::instance()->get_filename_crossings();
	if (setup_crossings_jobs.empty() && setup_mappings_jobs.empty()
			&& !is_readable_file(xfile)) {
		WARNING("did not find file: " << xfile << "\n"
				<< "Will automatically generate this crossing information file.\n"
				<< "You may omit specific crossings by modifying this file.");
		SetupCrossings sc;
		YAML::Node* n = sc.create_yaml_rep();
		list<pair<string, YAML::Node*> > scl;
		scl.push_back(make_pair(sc.yaml_spec_link().keyword(), n));
		setup_crossings_jobs.push_back(scl);
	}

	if (!setup_crossings_jobs.empty())
		sorted_job_lists_.insert(sorted_job_lists_.end(),
				setup_crossings_jobs.begin(), setup_crossings_jobs.end());
	if (!verify_permsyms_jobs.empty())
		sorted_job_lists_.push_back(verify_permsyms_jobs);
	if (!setup_mappings_jobs.empty())
		sorted_job_lists_.insert(sorted_job_lists_.end(),
				setup_mappings_jobs.begin(), setup_mappings_jobs.end());
	if (!regular_jobs.empty())
		sorted_job_lists_.push_back(regular_jobs);

	fin.close();
}

YAML::Emitter& operator<<(YAML::Emitter& os, const JobQueue&) {
	ABORT("unimplemented");
	return os;
}

void JobQueue::print_dot(const std::string& filename) const {
	ofstream file(filename.c_str());

	// header
	file << "digraph JobQueue {\n";

	// jobs are vertices, dependencies are directed edges
	map<int, set<int> >::const_iterator dep;
	for (dep = dependencies_.begin(); dep != dependencies_.end(); ++dep) {
		set<int>::const_iterator d;
		for (d = dep->second.begin(); d != dep->second.end(); ++d) {
			file << dep->first << " -> " << *d;
			Job::Status stat = job_status_.find(*d)->second;
			switch (stat) {
			case Job::Done:
			case Job::Skipped:
				file << "[color=green]";
				break;
			default:
				break;
			}
			file << ";\n";
		}
	}
	for (list<Job*>::const_iterator j = jobs_.begin(); j != jobs_.end(); ++j) {
		file << (*j)->id() << " [shape=box,";
		Job::Status stat = job_status_.find((*j)->id())->second;
		switch (stat) {
		case Job::Skipped:
			file << "color=yellow,";
			break;
		case Job::Done:
			file << "color=green,";
			break;
		case Job::Running:
			file << "color=blue,";
			break;
		default:
			if (dependencies_.find((*j)->id()) == dependencies_.end())
				file << "color=red,";
			break;
		}
		/// \todo: escape possible quotation marks in the string
		file << "label=\"" << job_string(*j) << "\"];\n";
	}

	// trailer
	file << "}\n";

	file.close();
}

} // namespace Reduze
