/*  kinematics.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef KINEMATICS_H_
#define KINEMATICS_H_

#include <ginac/ginac.h>
#include "yamlconfigurable.h"

namespace Reduze {

class Crossing;

/// External kinematics of the process
class Kinematics: public YAMLConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("kinematics");
		s.set_short_description("Declaration of the kinematics.");
		s.set_long_description(s.short_description());
		s.add_option("name", false, "string", "" //
						"A name for the kinematics.");
		s.add_option("incoming_momenta", true, "sequence of strings", "" //
						"The incoming external momenta.");
		s.add_option("outgoing_momenta", true, "sequence of strings", "" //
						"The outgoing external momenta.");
		s.add_option("momentum_conservation", false, "two-element sequence", "" //
						"The external momentum (first entry) that has to be"
						" replaced by a combination of the other external"
						" momenta (second entry).");
		s.add_option("kinematic_invariants", true, "sequence of strings", "" //
						"Each invariant must be given as a sequence of"
						" a symbol and its mass dimension.");
		s.add_option("scalarproduct_rules", true, "sequence", "" //
						"Rules how to replace scalar products of independent"
						" external momenta by kinematic invariants."
						" The first entry of such a rule is a two-element sequence"
						" (scalar product of combination of independent"
						" external momenta) and the second entry is a combination"
						" of kinematic invariants");
		s.add_option("dimension", false, "string", "" //
						"For backward compatibility: a symbol for the space time"
						" dimension. If defined here it will be used instead"
						" of the space time dimension symbol defined in \"global.yaml\"."
						" See the online help for \"global_symbols\".");
		s.add_option("symbol_to_replace_by_one", false, "string", "" //
						"A kinematic invariant that will be set to one during"
						" a reduction and in the reduction results. The explicit"
						" dependence on this symbol of the reduction result"
						" is restored in the output of the job \"export\"."
				        " Please note that changing this symbol will lead to"
				        " an inconsistent interpretation of previously computed"
				        " reductions.");
		s.add_option("loop_momentum_prefix", false, "string", "" //
						"A symbol, e.g. \"k\", from which loop momenta"
						" \"k1\", \"k2\", ... are constructed."
						" They are only used for reading diagrams and"
						" not for defining integral families.");
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	/// construct empty kinematics
	Kinematics();
	virtual ~Kinematics();
	/// create derived Kinematics using suffix rules
    static Kinematics* create_derived_kinematics(const std::string& name);
    /// analyzes if name contains suffix for implicit dimension shift rule
	static bool search_crossing_suffix(const std::string& name,
			std::string& basename, std::string& suffix, Crossing& cross);
    /// analyzes if name contains suffix for implicit dimension shift rule
	static bool search_dimension_shift_suffix(const std::string& name,
			std::string& basename, std::string& suffix, int& shift);
	/// compose new label for dimension shifted kinematics
	static std::string dimension_shifted_label(const std::string& base,
			int shift);

	virtual void read(const YAML::Node& node);
	virtual void print(YAML::Emitter& os) const;

	const std::string& name() const {
		return name_;
	}
	/// get the external momenta
	inline const GiNaC::lst& external_momenta() const;
	/// get the independent external momenta
	inline const GiNaC::lst& independent_external_momenta() const;
	/// get the incoming momenta (lin.comb. of ext.mom.)
	inline const GiNaC::lst& incoming_momenta() const;
	/// get the outgoing momenta (lin.comb. of ext.mom.)
	inline const GiNaC::lst& outgoing_momenta() const;
	/// get the kinematic invariants
	inline const GiNaC::lst& kinematic_invariants() const;
	/// access the kinematic invariants together with the dimension
	inline const GiNaC::lst& kinematic_invariants_and_dimension() const;
	/// access the invariants, the dimension and global symbols defined in the Feynman rules file
	inline const GiNaC::lst& invariants_dimension_feynman_rules_globals() const;
	/// get the dimension
	const GiNaC::ex& dimension() const;
	/// external momenta, kinematic invariants, dimension
	inline const GiNaC::lst& all_symbols() const;

	/// get external momenta and the kinematic invariants
	GiNaC::lst get_external_momenta_and_kinematics() const;

	/// get the loop momentum prefix
	const std::string& loop_momentum_prefix() const;

	const GiNaC::scalar_products& momenta_to_invariants() const {
		return momenta_to_invariants_;
	}
	const GiNaC::exmap& rules_sp_to_invariants() const {
		return rules_sp_to_invariants_;
	}

	/// crossing by name
	const Crossing& crossing(const std::string& name) const;

	/// try to invert the scalar product to invariant rules
	/** Problems arise if non-linear dependencies between scalar products
	 * and invariants are specified, simple cases with e.g. m instead of m^2
	 * should be covered. AVOID using this function if possible ! **/
	GiNaC::exmap find_rules_invariants_to_sp() const;

	const GiNaC::exmap& rule_momentum_conservation() const {
		return rule_momentum_conservation_;
	}

	const GiNaC::symbol* symbol_to_replace_by_one() const {
		if (symbol_to_replace_by_one_.get_name() == "undefined")
			return 0;
		return &symbol_to_replace_by_one_;
	}

	GiNaC::exmap get_rule_symbol_to_replace_by_one() const {
		GiNaC::exmap rule;
		if (symbol_to_replace_by_one_.get_name() != "undefined")
			rule[symbol_to_replace_by_one_] = 1;
		return rule;
	}

	int find_mass_dimension(const GiNaC::ex&) const;

	const std::map<std::string, GiNaC::symbol>& shared_symbols() const {
		return shared_symbols_;
	}
	void set_shared_symbols(const std::map<std::string, GiNaC::symbol>& symbs) {
		shared_symbols_ = symbs;
	}

	// compare kinematics
	int id() const;
	void set_id(int id);
	bool operator<(const Kinematics& other) const;
	bool operator==(const Kinematics& other) const;
    bool operator!=(const Kinematics& other) const;

    void print_info() const;

private:
	/// name of the Kinematics
	std::string name_;
	/// external momenta
	GiNaC::lst external_momenta_, independent_external_momenta_;
	/// incoming and outgoing momenta
	GiNaC::lst incoming_momenta_, outgoing_momenta_;
	/// kinematic invariants
	GiNaC::lst kinematic_invariants_;
	/// all symbols
	GiNaC::lst all_symbols_;
	/// dimension
	GiNaC::ex dimension_;
	/// kinematic invariants and dimension
	GiNaC::lst kinematic_invariants_and_dimension_;
	/// kinematic invariants and dimension and the symbols from FeynmanRulesGlobals
	GiNaC::lst invariants_dimension_feynman_rules_globals_;
	/// symbol to replace by one
	GiNaC::symbol symbol_to_replace_by_one_;
	/// mass dimensions of kinematic invariants
	std::map<GiNaC::symbol, int, GiNaC::ex_is_less> mass_dimension_;
	/// map of scalar product ScalarProduct(pi,pj) of external momenta to kinematic invariants
	GiNaC::exmap rules_sp_to_invariants_;
	///  GiNaC::scalar_products of external momenta to kinematic invariants
	GiNaC::scalar_products momenta_to_invariants_;

	/// map of kinematic invariant to scalar products pi*pj of external momenta
	GiNaC::exmap rules_invariants_to_sp_;
	/// substitution of an external momentum employing global mom. conserv.
	GiNaC::exmap rule_momentum_conservation_;

	/// prefix for not explicitly specified loop momenta // \todo remove
	std::string loop_momentum_prefix_;

	/// shared symbols
	/** a symbol (like kinematic invariants) which should be generated from a
	 ** string is taken from this map if its name is present as a key.
	 **  This map must be filled before the Kinematics is read from file. **/
	std::map<std::string, GiNaC::symbol> shared_symbols_;

	int id_;
};

inline void operator>>(const YAML::Node& n, Kinematics& r) {
	r.read(n);
}
inline YAML::Emitter& operator<<(YAML::Emitter& ye, const Kinematics& r) {
	r.print(ye);
	return ye;
}

const GiNaC::lst& Kinematics::external_momenta() const {
	return external_momenta_;
}
const GiNaC::lst& Kinematics::independent_external_momenta() const {
	return independent_external_momenta_;
}
const GiNaC::lst& Kinematics::incoming_momenta() const {
	return incoming_momenta_;
}
const GiNaC::lst& Kinematics::outgoing_momenta() const {
	return outgoing_momenta_;
}
const GiNaC::lst& Kinematics::kinematic_invariants() const {
	return kinematic_invariants_;
}
const GiNaC::lst& Kinematics::kinematic_invariants_and_dimension() const {
	return kinematic_invariants_and_dimension_;
}
const GiNaC::lst& Kinematics::invariants_dimension_feynman_rules_globals() const {
	return invariants_dimension_feynman_rules_globals_;
}

const GiNaC::lst& Kinematics::all_symbols() const {
	return all_symbols_;
}

} // namespace Reduze

#endif /* KINEMATICS_H_ */
