/*  topology.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef TOPOLOGY_H_
#define TOPOLOGY_H_

#include <string>
#include <list>
#include <set>
#include <vector>
#include <ostream>
#include <ginac/ginac.h>
#include "canonicallabel.h"
#include "propagator.h"

namespace YAML {
class Node;
class Emitter;
}

namespace Reduze {

/// directed graph
/** node ids: negative integer for external node, non-negative for internal
 ** edge ids: may be integer, sign has no meaning to Topology */
class Topology : public IFindCanonicalLabel {
public:
	// construction and destruction
	Topology();
	virtual ~Topology();

	// general
	std::string name() const;
	void set_name(const std::string& name);

	/// swaps the topologies
	void swap(Topology& other, bool call_back = true);

	/// returns an unused edge id which can be used for next edge insertion
	int next_edge_id() const;
	/// inserts edge, returns assigned id
	int insert_edge(int from, int to);
	/// inserts edge, returns id
	int insert_edge(const Edge& e);
	/// insert node, node < 0 means external node
	void insert_node(int node);
	/// inserts a new node on the edge 'edge_id'
	// returns the new edge, which goes from the new node to the to(edge_id)
	// except if the leg is outgoing, then the new edge goes from the from(edge_id)
	// to the new node.
	Edge insert_degree_2_node(int edge_id);
	/// removes nodes which are connected by two different edges
	void remove_degree_2_nodes();
	/// removes an edge (no automatic removal of adjacent nodes)
	void remove_edge(int edge_id);
	/// contract edge (by default: node with larger id will be deleted)
	void contract_edge(int edge_id, bool remove_greater_node = true);
	/// removes bridges by contracting them
	/** note: "external" edges will not be contracted **/
	void contract_bridges();
	/// removes the node (introduces new nodes)
	void remove_node(int node);
	/// removes the isolated node (aborts if it is not isolated)
	void remove_isolated_node(int node);
	/// identifies two nodes: relabel the node 'from' to 'to'
	void identify_nodes(int from, int to);
	/// flips orientation of Edge
	void reverse_edge(int edge_id);
	/// removes external edges and external nodes, creates tadpole topology (no removal of degree 2 vertices)
	virtual void remove_external_legs();
	/// joins all external nodes in a new vertex, creates tadpole topology (no removal of degree 2 vertices)
	virtual void join_external_nodes();
	/// cleaves the node 'node'
	/** creates a new node 'new_node',
	 *  adds an edge 'new_edge' from the new generated node 'new_node' to 'node'
	 *  (or opposite direction if sign <= 0)
	 *  reassigns the selected edges 'edges' from 'node' to 'new_node'
	 *  returns the pair (new_node, new_edge) */
	virtual std::pair<unsigned int, unsigned int>	//
	cleave(int node, const std::set<Edge>& edges, int sign, int edge_id);

	unsigned int num_edges() const;
	unsigned int num_nodes() const;
	unsigned int num_external_nodes() const;
	unsigned int num_internal_edges() const;
	bool contains_node(int node) const;
	bool contains_edge(int edge) const;
	/// returns whether one or both nodes the given edge connects is external
	bool is_external_edge(int edge) const;
	/// returns whether the given node is external
	bool is_external_node(int node) const;
	virtual const std::set<int>& nodes() const; // specification IFindCanonicalLabel
	virtual const std::map<int, Edge>& edges() const; // specification IFindCanonicalLabel
	const std::list<Edge>& edges_of_node(int node) const;
	const std::map<int, std::list<Edge> >& edges_of_nodes() const;
	const Edge& edge(int edge_id) const;
	std::list<Edge> find_edges_between(int node1, int node2) const;
	/// return adjacent nodes (not including node itself)
	std::set<int> neighbors(int node) const;
	std::set<int> internal_nodes() const;
	virtual std::set<int> external_nodes() const; // specification IFindCanonicalLabel
	std::list<Edge> internal_edges() const;
	virtual std::list<Edge> external_edges() const; // specification IFindCanonicalLabel
	/// returns internal edges which belong to some closed loops
	std::set<int> find_loop_edges() const;
	/// returns edges from the spanning tree (cycles) of the connected component containing 'node'
	std::pair<std::set<int>, std::set<int> >
	spanning_tree_and_cycles(int node) const;
	/// returns a set of fundamental cycles
	/** size of returned set is cyclomatic number (first Betti number) */
	std::set<int> find_cycles() const;
	// topological analysis
	bool is_connected() const;
	bool is_one_particle_irreducible() const;
	/// returns the component connected to given node
	/** disabled_edge is considered absent when determining connnectedness */
	Topology connected_component(int node, int disabled_edge = 0) const;
	std::list<Topology> connected_components() const;
	Topology biconnected_component(int node) const;
	/// returns 2-vertex-connected components and single edges (bridges)
	/** algorithm by Hopcroft, Tarjan (1971) via DFT
	 ** (could be linear time, O(n), with n = V+E,
	 **  current implementation is slower because of arbitrary numbers
	 **  for vertices => uses map[vertex] instead of array[vertex];
	 **  also performance could be improved by tracking cut_nodes e.g.
	 **  with some kind of tree of biconnected_components),
	 ** if cut_nodes is non-zero, the ids of cut nodes are returned **/
	std::list<Topology>	//
	biconnected_components(std::set<int>* cut_nodes = 0) const;
	/// returns true biconnected components through which no momentum could flow
	/** all other biconnected components are stored in 'other', single edges
	 ** may also occur as "biconnected" vacuum component **/
	std::list<Topology>	//
	biconnected_vacuum_components(std::list<Topology>& other) const;
	/// returns one-particle irreducible components
	std::list<Topology> one_pi_components(std::set<int>& inter_edges) const;
	/// returns number of edges belonging to some cycle
	int num_loop_edges() const;

	/// colors for edges
	virtual std::pair<std::list<std::map<int, int> >, edge_attributes> get_edge_coloring() const;
	/// coloring of the (external) nodes
	/** If the bool 'permute_external_nodes' is set to false then external
	 ** nodes all get different negativ color numbers **/
	virtual std::list<std::map<int, int> > get_node_coloring(bool permute_external_nodes =
			true) const;

	// output

	virtual void print_dot(const std::string& filename) const;
	virtual void print_postscript(const std::string& filename) const;
	friend YAML::Emitter& operator<<(YAML::Emitter& ye, const Topology& t);
	friend std::ostream& operator<<(std::ostream& os, const Topology& t);

private:

	// call-back type methods for bookkeeping of derived classes

	/// is called whenever a new node is inserted
	virtual void inserted_node(int id);
	/// is called whenever a new edge is inserted
	virtual void inserted_edge(int id);
	/// is called whenever a node is removed
	virtual void removed_node(int id);
	/// is called whenever an edge is removed
	virtual void removed_edge(int id);
	/// is called when topologies are swapped
	virtual void swapped();

private:
	/// name of the topology
	std::string name_;
	/// edges by their edge id
	std::map<int, Edge> edges_;
	/// the nodes (vertices)
	std::set<int> nodes_;
	/// list of edges adjacent to a node
	std::map<int, std::list<Edge> > edges_of_node_;
};


int identify_external_nodes(Topology& t);
//bool split_identified_external_nodes(Topology& g, const Topology& nonvac);
void reverse_all_edges(Topology& t);


} // namespace Reduze

#endif /* TOPOLOGY_H_ */
