/**
 * Copyright (c) 2025, Fabian Groffen. All rights reserved.
 *
 * See LICENSE for the licence.
 */

#include <stdio.h>
#include <unistd.h>
#include <strings.h>
#include <stdbool.h>
#include <stdlib.h>
#include <limits.h>

#include <ldns/ldns.h>

#include "util.h"

typedef struct rdflist {
    ldns_rdf       *d;
    ldns_rr_type   *r;
    struct rdflist *next;
} rdf_list;

typedef struct hostopts {
    uint8_t        verbose:1;
    uint8_t        doafxr:1;
    uint8_t        doip6int:1;
    uint8_t        recurse:1;
    uint8_t        failfast:1;
    uint8_t        checkrev:1;
    uint8_t        implicit:1;
    uint8_t        colour:1;
    uint8_t        ipmode:2;
#define HOST_IP_UNSET 0
#define HOST_IP_4     1
#define HOST_IP_6     2
    uint8_t        tcpudp:2;
#define HOST_MODE_UNSET 0
#define HOST_MODE_TCP   1
#define HOST_MODE_UDP   2
    uint8_t        ndots;
    uint8_t        nretries;
    uint8_t        timeout;
    uint16_t       port;
    ldns_rr_class  qclass;
} host_opts;

static int
host_print_record
(
    ldns_rr_list *ans
)
{
    size_t i;
    size_t j;
    int    ret = EXIT_FAILURE;

    for (i = 0; i < ldns_rr_list_rr_count(ans); i++)
    {
        ldns_rr      *rr   = ldns_rr_list_rr(ans, i);
        ldns_rdf     *base = ldns_rr_owner(rr);
        ldns_rr_type  tpe  = ldns_rr_get_type(rr);
        char         *own;
        char         *trg;
        uint32_t      prio;

        /* drop trailing dot */
        ldns_rdf_set_size(base, ldns_rdf_size(base) - 1);
        own = ldns_rdf2str(base);

        if (tpe == LDNS_RR_TYPE_A ||
            tpe == LDNS_RR_TYPE_AAAA)
        {
            ldns_rdf *targ = ldns_rr_rdf(rr, 0);
            char     *dest = ldns_rdf2str(targ);

            fprintf(stdout, "%s%s%s has %saddress %s%s%s\n",
                    FGCOLOUR_BLUE, own, FGCOLOUR_NORM,
                    tpe == LDNS_RR_TYPE_AAAA ? "IPv6 " : "",
                    FGCOLOUR_BRGREEN, dest, FGCOLOUR_NORM);
            ret = EXIT_SUCCESS;

            free(dest);
        }
        else if (tpe == LDNS_RR_TYPE_CNAME)
        {
            ldns_rdf *targ = ldns_rr_rdf(rr, 0);
            char     *dest = ldns_rdf2str(targ);

            fprintf(stdout, "%s%s%s is an alias for %s%s%s\n",
                    FGCOLOUR_BLUE, own, FGCOLOUR_NORM,
                    FGCOLOUR_BRBLUE, dest, FGCOLOUR_NORM);
            ret = EXIT_SUCCESS;

            free(dest);
        }
        else if (tpe == LDNS_RR_TYPE_MX)
        {
            trg = NULL;
            for (j = 0; j < ldns_rr_rd_count(rr); j++)
            {
                ldns_rdf      *targ = ldns_rr_rdf(rr, j);
                ldns_rdf_type  t    = ldns_rdf_get_type(targ);

                switch (t)
                {
                    case LDNS_RDF_TYPE_DNAME:
                        trg = ldns_rdf2str(targ);
                        break;
                    case LDNS_RDF_TYPE_INT8:
                        prio = (uint32_t)ldns_rdf2native_int8(targ);
                        break;
                    case LDNS_RDF_TYPE_INT16:
                        prio = (uint32_t)ldns_rdf2native_int16(targ);
                        break;
                    case LDNS_RDF_TYPE_INT32:
                        prio = ldns_rdf2native_int32(targ);
                        break;
                    default:
                        /* ignore this RD */
                        break;
                }
            }
            if (trg != NULL)
            {
                fprintf(stdout, "%s%s%s mail is handled by %s%u %s%s%s\n",
                        FGCOLOUR_BLUE, own, FGCOLOUR_NORM,
                        FGCOLOUR_BRYELLOW, prio,
                        FGCOLOUR_BRCYAN, trg, FGCOLOUR_NORM);
                ret = EXIT_SUCCESS;
            }

            free(trg);
        }
        else if (tpe == LDNS_RR_TYPE_PTR)
        {
            ldns_rdf *targ = ldns_rr_rdf(rr, 0);
            char     *dest = ldns_rdf2str(targ);

            fprintf(stdout, "%s%s%s domain name pointer %s%s%s\n",
                    FGCOLOUR_GREEN, own, FGCOLOUR_NORM,
                    FGCOLOUR_BRBLUE, dest, FGCOLOUR_NORM);
            ret = EXIT_SUCCESS;

            free(dest);
        }
        else if (tpe == LDNS_RR_TYPE_TXT)
        {
            ldns_rdf *targ = ldns_rr_rdf(rr, 0);
            char     *dest = ldns_rdf2str(targ);

            fprintf(stdout, "%s%s%s descriptive text %s%s%s\n",
                    FGCOLOUR_BLUE, own, FGCOLOUR_NORM,
                    FGCOLOUR_BRMAGENTA, dest, FGCOLOUR_NORM);
            ret = EXIT_SUCCESS;

            free(dest);
        }
        else if (tpe == LDNS_RR_TYPE_NS)
        {
            ldns_rdf *targ = ldns_rr_rdf(rr, 0);
            char     *dest = ldns_rdf2str(targ);

            fprintf(stdout, "%s%s%s name server %s%s%s\n",
                    FGCOLOUR_BLUE, own, FGCOLOUR_NORM,
                    FGCOLOUR_BRGREEN, dest, FGCOLOUR_NORM);
            ret = EXIT_SUCCESS;

            free(dest);
        }
        else if (tpe == LDNS_RR_TYPE_SOA)
        {
            ldns_rdf *targ = ldns_rr_rdf(rr, 0);
            char     *dest = ldns_rdf2str(targ);

            fprintf(stdout, "%s%s%s has SOA record %s%s%s",
                    FGCOLOUR_BLUE, own, FGCOLOUR_NORM,
                    FGCOLOUR_BRMAGENTA, dest, FGCOLOUR_NORM);
            free(dest);
            for (j = 1; j < ldns_rr_rd_count(rr); j++)
            {
                ldns_rdf_type  t;

                targ = ldns_rr_rdf(rr, j);
                t    = ldns_rdf_get_type(targ);

                switch (t)
                {
                    case LDNS_RDF_TYPE_DNAME:
                    case LDNS_RDF_TYPE_STR:
                        dest = ldns_rdf2str(targ);
                        fprintf(stdout, " %s%s%s",
                                FGCOLOUR_BRCYAN, dest, FGCOLOUR_NORM);
                        free(dest);
                        break;
                    case LDNS_RDF_TYPE_INT8:
                        fprintf(stdout, " %s%u%s",
                                FGCOLOUR_BRYELLOW,
                                (uint32_t)ldns_rdf2native_int8(targ),
                                FGCOLOUR_NORM);
                        break;
                    case LDNS_RDF_TYPE_INT16:
                        fprintf(stdout, " %s%u%s",
                                FGCOLOUR_YELLOW,
                                (uint32_t)ldns_rdf2native_int16(targ),
                                FGCOLOUR_NORM);
                        break;
                    case LDNS_RDF_TYPE_INT32:
                    case LDNS_RDF_TYPE_PERIOD:
                        fprintf(stdout, " %s%u%s",
                                FGCOLOUR_GREEN,
                                ldns_rdf2native_int32(targ),
                                FGCOLOUR_NORM);
                        break;
                    default:
                        /* ignore this RD */
                        break;
                }
            }
            fprintf(stdout, "\n");

            ret = EXIT_SUCCESS;
        }

        free(own);
    }

    return ret;
}

static int
host_func
(
    ldns_resolver *res,
    host_opts     *opts,
    ldns_rdf      *domain,
    ldns_rr_type  *types
)
{
    ldns_pkt      *pkt;
    ldns_rr_list  *ans;
    ldns_rr_type   tpe;
    ldns_rr_type   lasttpe = (ldns_rr_type)0;
    bool           found   = false;

    while ((uint32_t)(tpe = *types++) != 0)
    {
        ldns_status resret =
            ldns_resolver_search_status(&pkt, res, domain,
                                        tpe, opts->qclass, LDNS_RD);

        switch (resret)
        {
            case LDNS_STATUS_NETWORK_ERR:
            case LDNS_STATUS_ADDRESS_ERR:
            case LDNS_STATUS_SOCKET_ERROR:
                fprintf(stderr,
                        "%s;; %s%s%s; no servers could be reached%s\n",
                        FGCOLOUR_RED, FGCOLOUR_BRRED,
                        ldns_get_errorstr_by_id(resret),
                        FGCOLOUR_RED, FGCOLOUR_NORM);

                return EXIT_FAILURE;
            case LDNS_STATUS_OK:
                /* all good, let's do something */
                /* note: pkt can be NULL in this case :( */
                break;
            default:
                fprintf(stderr, "%s;; %s%s%s\n",
                        FGCOLOUR_RED, FGCOLOUR_BRRED,
                        ldns_get_errorstr_by_id(resret),
                        FGCOLOUR_NORM);
                return EXIT_FAILURE;
        }

        if (pkt == NULL ||
            (ldns_pkt_reply_type(pkt) != LDNS_PACKET_ANSWER &&
             ldns_pkt_reply_type(pkt) != LDNS_PACKET_NODATA))
        {
            char           *dom;
            char           *rcd;
            ldns_pkt_rcode  rc;

            rc = LDNS_RCODE_NXDOMAIN;
            if (pkt != NULL)
                rc = ldns_pkt_get_rcode(pkt);

            if (ldns_dname_absolute(domain))
                ldns_rdf_set_size(domain, ldns_rdf_size(domain) - 1);
            dom = ldns_rdf2str(domain);
            rcd = ldns_pkt_rcode2str(rc);

            fprintf(stderr, "%sHost %s%s%s not found: %s%u(%s)%s\n",
                    FGCOLOUR_RED, FGCOLOUR_BRRED, dom, FGCOLOUR_RED,
                    FGCOLOUR_BRBLUE, (uint32_t)rc, rcd, FGCOLOUR_NORM);

            free(dom);
            free(rcd);
            if (pkt != NULL)
                ldns_pkt_free(pkt);

            return EXIT_FAILURE;
        }

        if (opts->verbose)
        {
            char *digd = ldns_pkt2str(pkt);
            fprintf(stdout, "%s\n", digd);
            free(digd);
            found = true;
        }
        else
        {
            ans = ldns_pkt_rr_list_by_type(pkt, tpe, LDNS_SECTION_ANSWER);
            if (tpe == LDNS_RR_TYPE_MX)
                ldns_rr_list_sort(ans);
            if (host_print_record(ans) == EXIT_SUCCESS)
                found = true;
            ldns_rr_list_free(ans);
        }

        ldns_pkt_free(pkt);

        lasttpe = tpe;
    }

    if (!opts->implicit &&
        !found &&
        lasttpe != (ldns_rr_type)0)
    {
        char *dom;
        char *rr;

        if (ldns_dname_absolute(domain))
            ldns_rdf_set_size(domain, ldns_rdf_size(domain) - 1);
        dom = ldns_rdf2str(domain);

        /* take last type to report on */
        rr  = ldns_rr_type2str(lasttpe);

        fprintf(stderr, "%s%s%s has no %s%s%s record%s\n",
                FGCOLOUR_BRRED, dom, FGCOLOUR_RED,
                FGCOLOUR_BRBLUE, rr, FGCOLOUR_RED, FGCOLOUR_NORM);

        free(dom);
        free(rr);

        return EXIT_FAILURE;
    }

    return EXIT_SUCCESS;
}

static int
host_checkptr
(
    ldns_resolver *res,
    host_opts     *opts,
    rdf_list      *domains,
    rdf_list      *seen
)
{
    ldns_pkt      *pkt;
    ldns_rr_list  *ans;
    ldns_rr_type  *qtypes = domains->r;
    ldns_rr_type  *types;
    ldns_rr_type   tpe;
    ldns_rdf      *revq;
    rdf_list      *seent;
    rdf_list      *todot = domains;
    rdf_list      *nextd;
    int            i;

    for (seent = seen; seent->next != NULL; seent = seent->next)
        ;

    /* switch to default if input is an address */
    if (qtypes[0] == LDNS_RR_TYPE_PTR)
    {
        qtypes = calloc(3, sizeof(qtypes[0]));
        qtypes[0] = LDNS_RR_TYPE_A;
        qtypes[1] = LDNS_RR_TYPE_AAAA;
    }

    while (domains != NULL)
    {
        bool addseen = true;

        types = domains->r;

        while ((uint32_t)(tpe = *types++) != 0)
        {
            ldns_status resret =
                ldns_resolver_search_status(&pkt, res, domains->d,
                                            tpe, opts->qclass, LDNS_RD);

            switch (resret)
            {
                case LDNS_STATUS_NETWORK_ERR:
                case LDNS_STATUS_ADDRESS_ERR:
                case LDNS_STATUS_SOCKET_ERROR:
                    fprintf(stderr,
                            "%s;; %s%s%s; no servers could be reached%s\n",
                            FGCOLOUR_RED, FGCOLOUR_BRRED,
                            ldns_get_errorstr_by_id(resret),
                            FGCOLOUR_RED, FGCOLOUR_NORM);

                    return EXIT_FAILURE;
                case LDNS_STATUS_OK:
                    /* all good, let's do something */
                    /* note: pkt can be NULL in this case :( */
                    break;
                default:
                    fprintf(stderr, "%s;; %s%s%s\n",
                            FGCOLOUR_RED, FGCOLOUR_BRRED,
                            ldns_get_errorstr_by_id(resret),
                            FGCOLOUR_NORM);
                    return EXIT_FAILURE;
            }

            ans = NULL;
            if (pkt == NULL ||
                (ldns_pkt_reply_type(pkt) != LDNS_PACKET_ANSWER &&
                 ldns_pkt_reply_type(pkt) != LDNS_PACKET_NODATA))
            {
                char           *dom;
                char           *rcd;
                ldns_pkt_rcode  rc;

                rc = LDNS_RCODE_NXDOMAIN;
                if (pkt != NULL)
                    rc = ldns_pkt_get_rcode(pkt);

                if (ldns_dname_absolute(domains->d))
                    ldns_rdf_set_size(domains->d,
                                      ldns_rdf_size(domains->d) - 1);
                dom = ldns_rdf2str(domains->d);
                rcd = ldns_pkt_rcode2str(rc);

                fprintf(stderr, "%sHost %s%s%s not found: %s%u(%s)%s\n",
                        FGCOLOUR_RED, FGCOLOUR_BRRED, dom, FGCOLOUR_RED,
                        FGCOLOUR_BRBLUE, (uint32_t)rc, rcd, FGCOLOUR_NORM);

                free(dom);
                free(rcd);
            }
            else
            {
                ans = ldns_pkt_rr_list_by_type(pkt, tpe, LDNS_SECTION_ANSWER);
            }

            if (ans != NULL)
            {
                if (addseen)
                {
                    ldns_rr *rr = ldns_rr_list_rr(ans, 0);

                    /* append this domain to the "seen" list, we delay this
                     * until we have a resolve, because transformations may
                     * happen due to non-absolute input, and added search
                     * domains, by delaying until here, we can properly
                     * ignore re-resolving the same domain names lateron */
                    seent    = seent->next = calloc(1, sizeof(*seent));
                    seent->d = ldns_rdf_clone(ldns_rr_owner(rr));

                    addseen  = false;
                }

                if (host_print_record(ans) == EXIT_SUCCESS)
                {
                    for (i = 0; i < ldns_rr_list_rr_count(ans); i++)
                    {
                        ldns_rr       *rr   = ldns_rr_list_rr(ans, i);
                        ldns_rr_type   tpe  = ldns_rr_get_type(rr);
                        ldns_rdf      *rrdf = ldns_rr_rdf(rr, 0);
                        ldns_rr_type  *ntpe;

                        switch (tpe)
                        {
                            case LDNS_RR_TYPE_A:
                            case LDNS_RR_TYPE_AAAA:
                                revq = util_addr2dname(rrdf, false);
                                ntpe = calloc(2, sizeof(ntpe[0]));
                                ntpe[0] = LDNS_RR_TYPE_PTR;
                                break;
                            case LDNS_RR_TYPE_PTR:
                                revq = rrdf;
                                ntpe = qtypes;
                                break;
                            default:
                                exit(EXIT_FAILURE);
                        }

                        for (nextd = seen->next;
                             nextd != NULL;
                             nextd = nextd->next)
                        {
                            if (ldns_rdf_compare(nextd->d, revq) == 0)
                            {
                                /* already seen, skip */
                                ldns_rdf_deep_free(revq);
                                if (ntpe != qtypes)
                                    free(ntpe);
                                revq = NULL;
                                break;
                            }
                        }
                        for (nextd = domains;
                             nextd != NULL;
                             nextd = nextd->next)
                        {
                            if (ldns_rdf_compare(nextd->d, revq) == 0)
                            {
                                /* already scheduled, skip */
                                ldns_rdf_deep_free(revq);
                                if (ntpe != qtypes)
                                    free(ntpe);
                                revq = NULL;
                                break;
                            }
                        }

                        if (revq != NULL)
                        {
                            todot = todot->next = calloc(1, sizeof(*todot));
                            todot->d = revq;
                            todot->r = ntpe;
                        }
                    }
                }
                ldns_rr_list_free(ans);
            }

            ldns_pkt_free(pkt);
        }

        if (domains->r != qtypes)
            free(domains->r);
        free(domains->d);
        nextd = domains->next;
        free(domains);
        domains = nextd;
    }

    free(qtypes);

    return EXIT_SUCCESS;
}

static int
host_afxr
(
    ldns_resolver *res,
    ldns_rdf      *domain,
    ldns_rr_type  *types,
    host_opts     *opts
)
{
    ldns_rr_list *ans;

    if (ldns_axfr_start(res, domain, opts->qclass) != LDNS_STATUS_OK)
        return EXIT_FAILURE;

    ans = ldns_rr_list_new();

    while (ldns_axfr_complete(res) == 0)
    {
        ldns_rr        *rr  = ldns_axfr_next(res);
        ldns_rr_type   tpe  = ldns_rr_get_type(rr);
        ldns_rr_type  *tpew = NULL;

        if (rr == NULL)
        {
            char           *dom;
            char           *rcd;
            ldns_pkt       *pkt;
            ldns_pkt_rcode  rc;

            pkt = ldns_axfr_last_pkt(res);

            rc = LDNS_RCODE_NOTAUTH;
            if (pkt != NULL)
                rc = ldns_pkt_get_rcode(pkt);

            dom = ldns_rdf2str(domain);
            rcd = ldns_pkt_rcode2str(rc);

            fprintf(stderr, "%sHost %s%s%s not found: %s%u(%s)%s\n",
                    FGCOLOUR_RED, FGCOLOUR_BRRED, dom, FGCOLOUR_RED,
                    FGCOLOUR_BRBLUE, (uint32_t)rc, rcd, FGCOLOUR_NORM);

            free(dom);
            free(rcd);
            if (pkt != NULL)
                ldns_pkt_free(pkt);
            ldns_rr_list_free(ans);

            return EXIT_FAILURE;
        }

        for (tpew = types; (uint32_t)*tpew != 0; tpew++)
        {
            if (*tpew == tpe)
            {
                ldns_rr_list_push_rr(ans, rr);
                break;
            }
        }
    }

    if (opts->verbose)
    {
        size_t i;
        for (i = 0; i < ldns_rr_list_rr_count(ans); i++)
        {
            char *digd = ldns_rr2str(ldns_rr_list_rr(ans, i));
            fprintf(stdout, "%s\n", digd);
            free(digd);
        }
    }
    else
    {
        host_print_record(ans);
    }

    ldns_rr_list_free(ans);

    return EXIT_SUCCESS;
}

static void
do_usage
(
    FILE *dst
)
{
    fprintf(dst,
"Usage: host [-aCdilrTvVw] [-c class] [-N ndots] [-t type] [-W time]\n"
"            [-R number] [-m flag] [-p port] hostname [server]\n"
"    -a is equivalent to -v -t ANY\n"
"    -A is like -a but omits RRSIG, NSEC, NSEC3\n"
"    -c specifies query class for non-IN data\n"
"    -d is equivalent to -v\n"
"    -i reverse lookups of IPv6 addresses use ip6.int domain\n"
"    -l lists all hosts in a domain, using AXFR\n"
"    -N changes the number of dots allowed before root lookup is done\n"
"    -p specifies the port on the server to query\n"
"    -r disables recursive processing\n"
"    -R specifies number of retries for UDP packets\n"
"    -s a SERVFAIL response should stop query\n"
"    -t specifies the query type\n"
"    -T enables TCP/IP mode\n"
"    -U enables UDP mode\n"
"    -v enables verbose output\n"
"    -V print version number and exit\n"
"    -w specifies to wait forever for a reply\n"
"    -W specifies how long to wait for a reply\n"
"    -4 use IPv4 query transport only\n"
"    -6 use IPv6 query transport only\n"
"  ldns-tools extensions\n"
"    -h this help screen\n"
"    -C use colours (defaults to true on ttys, respects NO_COLOR envvar)\n"
"    -P resolve resulting address and pointer answers (A/AAAA/PTR consistency)\n"
);
}

int
main(int argc, char **argv)
{
    ldns_resolver *res;
    ldns_rdf      *domain;
    ldns_rr_type  *qtypes   = NULL;
    ldns_status    s;
    char          *typestr  = NULL;
    char          *classstr = NULL;
    char          *nocolour = getenv("NO_COLOR");
    int            opt;
    host_opts      opts = {
        .ndots    = 1,
        .nretries = 1,
        .timeout  = 1,  /* second */
        .recurse  = true,
        .colour   = true,
        .qclass   = LDNS_RR_CLASS_IN
    };

    /* https://no-color.org/ */
    if (nocolour != NULL &&
        *nocolour != '\0')
        opts.colour = false;

    if (!isatty(fileno(stdout)))
        opts.colour = false;

    while ((opt = getopt(argc, argv, "46aAc:CdhilN:p:PsrR:t:TUvVwW:")) != -1)
    {
        switch (opt) {
            case '4':
                opts.ipmode = HOST_IP_4;
                break;
            case '6':
                opts.ipmode = HOST_IP_6;
                break;
            case 'a':
            case 'A':  /* should be a - [RRSIG, NSEC, NSEC3] */
                typestr = "any";
                /* fall through */
            case 'd':
            case 'v':
                opts.verbose = true;
                break;
            case 'c':
                classstr = optarg;
                break;
            case 'C':
                opts.colour = true;
                break;
            case 'h':
                do_usage(stdout);
                exit(EXIT_SUCCESS);
            case 'i':
                opts.doip6int = true;
                break;
            case 'l':
                opts.doafxr = true;
                break;
            case 'N':
                {
                    char *retp;
                    long  val;

                    val = strtol(optarg, &retp, 10);
                    if (retp == optarg ||
                        *retp != '\0' ||
                        val < 0)
                    {
                        fprintf(stderr,
                                "host: invalid argument for -N: %s\n",
                                optarg);
                        exit(EXIT_FAILURE);
                    }
                    if (val > UCHAR_MAX)
                    {
                        fprintf(stderr,
                                "host: value for -N too large: %ld\n",
                                val);
                        exit(EXIT_FAILURE);
                    }

                    opts.ndots = (uint8_t)val;
                }
                break;
            case 'p':
                {
                    char *retp;
                    long  val;

                    val = strtol(optarg, &retp, 10);
                    if (retp == optarg ||
                        *retp != '\0' ||
                        val <= 0)
                    {
                        fprintf(stderr,
                                "host: invalid argument for -p: %s\n",
                                optarg);
                        exit(EXIT_FAILURE);
                    }
                    if (val > USHRT_MAX)
                    {
                        fprintf(stderr,
                                "host: value for -p too large: %ld\n",
                                val);
                        exit(EXIT_FAILURE);
                    }

                    opts.port = (uint16_t)val;
                }
                break;
            case 'P':
                opts.checkrev = true;
                break;
            case 'r':
                opts.recurse = false;
                break;
            case 'R':
                {
                    char *retp;
                    long  val;

                    val = strtol(optarg, &retp, 10);
                    if (retp == optarg ||
                        *retp != '\0')
                    {
                        fprintf(stderr,
                                "host: invalid argument for -R: %s\n",
                                optarg);
                        exit(EXIT_FAILURE);
                    }
                    if (val > UCHAR_MAX)
                    {
                        fprintf(stderr,
                                "host: value for -R too large: %ld\n",
                                val);
                        exit(EXIT_FAILURE);
                    }

                    if (val <= 0)
                        opts.nretries = 1;
                    else
                        opts.nretries = (uint8_t)val;
                }
                break;
            case 's':
                opts.failfast = true;
                break;
            case 't':
                typestr = optarg;
                break;
            case 'U':
                opts.tcpudp = HOST_MODE_UDP;
                break;
            case 'T':
                opts.tcpudp = HOST_MODE_TCP;
                break;
            case 'V':
                fprintf(stdout, "host %s from ldns-tools (using ldns %s)\n",
                        LDNS_TOOLS_VERSION, ldns_version());
                exit(EXIT_SUCCESS);
            case 'w':
                opts.timeout = 0;  /* forever */
                break;
            case 'W':
                {
                    char *retp;
                    long  val;

                    val = strtol(optarg, &retp, 10);
                    if (retp == optarg ||
                        *retp != '\0')
                    {
                        fprintf(stderr,
                                "host: invalid argument for -W: %s\n",
                                optarg);
                        exit(EXIT_FAILURE);
                    }
                    if (val > UCHAR_MAX)
                    {
                        fprintf(stderr,
                                "host: value for -W too large: %ld\n",
                                val);
                        exit(EXIT_FAILURE);
                    }

                    if (val <= 0)
                        opts.timeout = 1;
                    else
                        opts.timeout = (uint8_t)val;
                }
                break;
            default:
                do_usage(stderr);
                exit(EXIT_FAILURE);
        }
    }

    util_enable_colours(opts.colour);

    if (optind >= argc)
    {
        do_usage(stdout);
        exit(EXIT_FAILURE);
    }

    if (typestr != NULL)
    {
        ldns_rr_type rrtype = ldns_get_rr_type_by_name(typestr);

        switch (rrtype)
        {
            case (ldns_rr_type)0:
                fprintf(stderr, "host: invalid type: %s\n", typestr);
                exit(EXIT_FAILURE);
            case LDNS_RR_TYPE_A:
            case LDNS_RR_TYPE_AAAA:
            case LDNS_RR_TYPE_TXT:
                qtypes = calloc(sizeof(qtypes[0]), 3);
                qtypes[0] = LDNS_RR_TYPE_CNAME;
                qtypes[1] = rrtype;
                break;
            case LDNS_RR_TYPE_ANY:
                qtypes = calloc(sizeof(qtypes[0]), 8);
                qtypes[0] = LDNS_RR_TYPE_SOA;
                qtypes[1] = LDNS_RR_TYPE_NS;
                qtypes[2] = LDNS_RR_TYPE_CNAME;
                qtypes[3] = LDNS_RR_TYPE_A;
                qtypes[4] = LDNS_RR_TYPE_AAAA;
                qtypes[5] = LDNS_RR_TYPE_MX;
                qtypes[6] = LDNS_RR_TYPE_TXT;
                break;
            default:
                qtypes = calloc(sizeof(qtypes[0]), 2);
                qtypes[0] = rrtype;
                break;
        }

        if (opts.checkrev)
        {
            switch (rrtype)
            {
                case LDNS_RR_TYPE_A:
                case LDNS_RR_TYPE_AAAA:
                case LDNS_RR_TYPE_PTR:
                    /* ok, accept this */
                    break;
                default:
                    fprintf(stderr, "host: -P requires A, AAAA or PTR type\n");
                    exit(EXIT_FAILURE);
            }
        }
    }

    if (classstr != NULL)
    {
        if (strcasecmp(classstr, "in") == 0)
            opts.qclass = LDNS_RR_CLASS_IN;
        else if (strcasecmp(classstr, "ch") == 0)
            opts.qclass = LDNS_RR_CLASS_CH;
        else if (strcasecmp(classstr, "hs") == 0)
            opts.qclass = LDNS_RR_CLASS_HS;
        else
        {
            fprintf(stderr, "host: invalid class: %s\n", classstr);
            exit(EXIT_FAILURE);
        }
    }

    /* create resolver from /etc/resolv.conf */
    s = ldns_resolver_new_frm_file(&res, NULL);

    /* take second argument as DNS server to use */
    if (argc - optind > 1)
    {
        ldns_rdf **dns = util_addr_frm_str(res, argv[optind + 1], opts.ndots);

        if (dns == NULL)
        {
            if (s == LDNS_STATUS_OK)
            {
                fprintf(stderr,
                        "host: couldn't get address for '%s': not found\n",
                        argv[optind + 1]);
                exit(EXIT_FAILURE);
            }
            /* else handled below */
        }
        else
        {
            ldns_rdf *swalk;
            size_t    i;

            if (s == LDNS_STATUS_OK)
                ldns_resolver_deep_free(res);

            res = ldns_resolver_new();
            for (i = 0; (swalk = dns[i]) != NULL; i++)
            {
                ldns_resolver_push_nameserver(res, swalk);
                ldns_rdf_deep_free(swalk);
            }
            free(dns);

            /* in case /etc/resolv.conf could not be read and an IP was
             * given, ensure the check below won't fire */
            s = LDNS_STATUS_OK;
        }
    }

    if (s != LDNS_STATUS_OK)
    {
        fprintf(stderr, "host: could not open/parse /etc/resolv.conf\n");
        exit(EXIT_FAILURE);
    }

    if (opts.port > 0)
        ldns_resolver_set_port(res, opts.port);
    ldns_resolver_set_recursive(res, opts.recurse);
    ldns_resolver_set_retry(res, opts.nretries);
    ldns_resolver_set_ip6(res, opts.ipmode);
    {
        struct timeval tvtimeout = {
            .tv_sec = opts.timeout == 0 ? INT_MAX : opts.timeout
        };
        ldns_resolver_set_timeout(res, tvtimeout);
        ldns_resolver_set_retrans(res, opts.timeout);
    }
    if (opts.tcpudp > 0)
        ldns_resolver_set_usevc(res,
                                opts.tcpudp == HOST_MODE_TCP ? true : false);
    ldns_resolver_set_fail(res, opts.failfast);

    s = ldns_str2rdf_a(&domain, argv[optind]);
    if (s != LDNS_STATUS_OK)
        s = ldns_str2rdf_aaaa(&domain, argv[optind]);
    if (s == LDNS_STATUS_OK)
    {
        if (!opts.doafxr &&
            qtypes == NULL)
        {
            qtypes = calloc(sizeof(qtypes[0]), 2);
            qtypes[0] = LDNS_RR_TYPE_PTR;
        }

        domain = util_addr2dname(domain, opts.doip6int);
    }
    else
    {
        domain = util_dname_frm_str(argv[optind], opts.ndots, opts.doip6int);
    }

    if (opts.verbose)
    {
        char *dom = ldns_rdf2str(domain);
        fprintf(stdout, "Trying \"%s\"\n", dom);
        free(dom);
    }
    if (opts.doafxr)
    {
        if (qtypes == NULL)
        {
            qtypes = calloc(sizeof(qtypes[0]), 5);
            qtypes[0] = LDNS_RR_TYPE_NS;
            qtypes[1] = LDNS_RR_TYPE_A;
            qtypes[2] = LDNS_RR_TYPE_AAAA;
            qtypes[3] = LDNS_RR_TYPE_PTR;
        }
        host_afxr(res, domain, qtypes, &opts);
    }
    else if (opts.checkrev)
    {
        rdf_list  seenlist = { .d = NULL, .next = NULL };
        rdf_list *seenw    = &seenlist;
        rdf_list *todo;

        if (qtypes == NULL)
        {
            qtypes = calloc(sizeof(qtypes[0]), 3);
            qtypes[0] = LDNS_RR_TYPE_A;
            qtypes[1] = LDNS_RR_TYPE_AAAA;
        }

        todo = calloc(1, sizeof(*todo));
        todo->d    = domain;
        todo->r    = qtypes;
        todo->next = NULL;
        domain     = NULL;
        qtypes     = NULL;

        host_checkptr(res, &opts, todo, seenw);

        seenw = seenw->next;
        while (seenw != NULL)
        {
            rdf_list *next = seenw->next;
            if (seenw->d != domain)
                ldns_rdf_deep_free(seenw->d);
            free(seenw);
            seenw = next;
        }
    }
    else
    {
        if (qtypes == NULL)
        {
            qtypes        = calloc(sizeof(qtypes[0]), 5);
            qtypes[0]     = LDNS_RR_TYPE_CNAME;
            qtypes[1]     = LDNS_RR_TYPE_A;
            qtypes[2]     = LDNS_RR_TYPE_AAAA;
            qtypes[3]     = LDNS_RR_TYPE_MX;
            opts.implicit = true;
        }
        host_func(res, &opts, domain, qtypes);
    }

    free(qtypes);
    if (domain != NULL)
        ldns_rdf_deep_free(domain);
    ldns_resolver_deep_free(res);

    return EXIT_SUCCESS;
}

/* vim: set ts=4 sw=4 expandtab cinoptions=(0,u0,U1,W2s,l1: */
