/*
 *    WMAcpiLoad - A dockapp to monitor ACPI status
 *    Copyright (C) 2002  Thomas Nemeth <tnemeth@free.fr>
 *
 *    Patch by Alan Carriou <cariou_alan@yahoo.fr> (C) 2004-2005
 *
 *    Based on work by Seiichi SATO <ssato@sh.rim.or.jp>
 *    Copyright (C) 2001,2002  Seiichi SATO <ssato@sh.rim.or.jp>
 *    and on work by Mark Staggs <me@markstaggs.net>
 *    Copyright (C) 2002  Mark Staggs <me@markstaggs.net>
 
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <dirent.h>
#include <stdio.h>
#include "dockapp.h"
#include "lib_acpi.h"
#include "draw.h"

/************************************************************************
 * Includes the XPM pictures
 ************************************************************************/

#include "backlight_on.xpm"
#include "backlight_off.xpm"
#include "parts.xpm"

/************************************************************************
 * Macro definitions
 ************************************************************************/

#define BACKGROUND_PIXMAP(d) \
    ((d)->backlight == LIGHTON) ? (d)->backdrop_on : (d)->backdrop_off
#define SIZE                      58
#define WINDOWED_BG    ". c #AEAAAE"

/************************************************************************
 * Functions definitions
 ************************************************************************/

int
draw_init(DrawConfig *drw, int argc, char **argv)
{
    XpmColorSymbol colors[2] = { {"Back0", NULL, 0}, {"Back1", NULL, 0} };
    int ncolor;

    /* Initialize X window */
    if (dockapp_open_window(drw->display_name, PACKAGE,
                            SIZE, SIZE, argc, argv) != 0) {
        return 1;
    }

    dockapp_set_eventmask(ButtonPressMask);

    if (drw->light_color) {
        colors[0].pixel = dockapp_getcolor(drw->light_color);
        colors[1].pixel =
            dockapp_blendedcolor(drw->light_color, -24, -24, -24, 1.0);
        ncolor = 2;
    } else {
        ncolor = 0;
    }

    /* change raw xpm data to pixmap */
    if (dockapp_iswindowed)
        backlight_on_xpm[1] = backlight_off_xpm[1] = WINDOWED_BG;

    if (!dockapp_xpm2pixmap(backlight_on_xpm, &drw->backdrop_on,
                            &drw->mask, colors, ncolor)) {
        fprintf(stderr,
                "%s : Error initializing backlit background image.\n",
                PACKAGE);
        return 1;
    }
    if (!dockapp_xpm2pixmap(backlight_off_xpm, &drw->backdrop_off,
                            NULL, NULL, 0)) {
        fprintf(stderr,
                "%s : Error initializing background image.\n", PACKAGE);
        return 1;
    }
    if (!dockapp_xpm2pixmap(parts_xpm, &drw->parts, NULL, colors, ncolor)) {
        fprintf(stderr, "%s : Error initializing parts image.\n", PACKAGE);
        return 1;
    }

    /* shape window */
    if (!dockapp_iswindowed)
        dockapp_setshape(drw->mask, 0, 0);

    if (drw->mask)
        dockapp_XFreePixmap(drw->mask);

    /* pixmap : draw area */
    drw->pixmap = dockapp_XCreatePixmap(SIZE, SIZE);

    /* Initialize pixmap */
    if (drw->backlight == LIGHTON)
        dockapp_copyarea(drw->backdrop_on, drw->pixmap, 0, 0, SIZE, SIZE,
                         0, 0);
    else
        dockapp_copyarea(drw->backdrop_off, drw->pixmap, 0, 0, SIZE, SIZE,
                         0, 0);

    dockapp_set_background(drw->pixmap);
    dockapp_show();
    return 0;
}

void
draw_window(DrawConfig *d, AcpiInfos *i)
{
    draw_background(d, d->backlight);

    /* redraw foreground */
    draw_tempdigit(d, i);
    draw_pcdigit(d, i);
    draw_statusdigit(d, i);
    draw_pcgraph(d, i);
    draw_power_icon(d, i);

    /* show */
    dockapp_copy2window(d->pixmap);
}

void
switch_light(DrawConfig *d, AcpiInfos *i)
{
    switch (d->backlight) {
    case LIGHTOFF:
        d->backlight = LIGHTON;
        break;
    case LIGHTON:
        d->backlight = LIGHTOFF;
        break;
    }

    draw_window(d, i);
}

void
draw_tempdigit(DrawConfig *d, AcpiInfos *infos)
{
    if (infos && infos->thermal && d) {
        int y;

        if (d->backlight == LIGHTON)
            y = 20;
        else
            y = 0;

        dockapp_copyarea(d->parts, d->pixmap,
                         (infos->thermal->temp / 10) * 10, y, 10, 20, 5,
                         7);
        dockapp_copyarea(d->parts, d->pixmap,
                         (infos->thermal->temp % 10) * 10, y, 10, 20, 17,
                         7);

#if 0                           /* ? */
        int time_left, hour_left, min_left;

        hour_left = time_left / 60;
        min_left = time_left % 60;
        dockapp_copyarea(parts, pixmap, (hour_left / 10) * 10, y, 10, 20,
                         5, 7);
        dockapp_copyarea(parts, pixmap, (hour_left % 10) * 10, y, 10, 20,
                         17, 7);
        dockapp_copyarea(parts, pixmap, (min_left / 10) * 10, y, 10, 20,
                         32, 7);
        dockapp_copyarea(parts, pixmap, (min_left % 10) * 10, y, 10, 20,
                         44, 7);
#endif
    }
}

void
draw_pcdigit(DrawConfig *drw, AcpiInfos *infos)
{
    int d, u, xd, num;

    if (drw->backlight == LIGHTON)
        xd = 50;
    else
        xd = 0;

    if (infos && infos->bat && infos->bat->plugged) {

        if (infos->bat->percentage < 0)
            num = 0;
        else
            num = infos->bat->percentage;

        d = (num % 100) / 10;
        u = num % 10;

        /* draw digits */
        dockapp_copyarea(drw->parts, drw->pixmap, u * 5 + xd, 40, 5, 9, 17,
                         45);

        if (num >= 10)
            dockapp_copyarea(drw->parts, drw->pixmap, d * 5 + xd, 40, 5, 9,
                             11, 45);
        else
            dockapp_copyarea(BACKGROUND_PIXMAP(drw), drw->pixmap, 11, 45,
                             5, 9, 11, 45);

        if (num >= 100) {
            dockapp_copyarea(drw->parts, drw->pixmap, 5 + xd, 40, 5, 9, 5,
                             45);
            dockapp_copyarea(drw->parts, drw->pixmap, 0 + xd, 40, 5, 9, 11,
                             45);
        } else {
            dockapp_copyarea(BACKGROUND_PIXMAP(drw), drw->pixmap, 11, 5,
                             5, 9, 5, 45);
        }
    } else {
        /* battery unplugged or no battery slot at all: display "---%" */
        dockapp_copyarea(drw->parts, drw->pixmap, 10 + xd, 49, 5, 9, 5,
                         45);
        dockapp_copyarea(drw->parts, drw->pixmap, 10 + xd, 49, 5, 9, 11,
                         45);
        dockapp_copyarea(drw->parts, drw->pixmap, 10 + xd, 49, 5, 9, 17,
                         45);
    }
}

void
draw_statusdigit(DrawConfig *d, AcpiInfos *infos)
{
    int xd, y;

    if (d->backlight == LIGHTON) {
        y = 40;
        xd = 50;
    } else {
        y = 31;
        xd = 0;
    }

    /* draw digit */
    if (infos && infos->bat && infos->bat->plugged) {
        if (infos->bat->status == Charging)
            dockapp_copyarea(d->parts, d->pixmap, 100, y, 4, 9, 41, 45);
        else
            /* draw background */
            dockapp_copyarea(BACKGROUND_PIXMAP(d), d->pixmap, 41, 45,
                             4, 9, 41, 45);
    } else {
        /*
         * no battery slot or battery unplugged: shows '-' as charging
         * state 
         */
        dockapp_copyarea(d->parts, d->pixmap, 10 + xd, 49, 5, 9, 41, 45);
    }
}

void
draw_power_icon(DrawConfig *d, AcpiInfos *infos)
{
    if (infos != NULL) {
        int xd;

        xd = (d->backlight == LIGHTON) ? 50 : 0;

        if (infos->AC_power == On_line)
            dockapp_copyarea(d->parts, d->pixmap, 0 + xd, 49, 5, 9, 34,
                             45);
        else if (infos->bat && infos->bat->status == Charging)
            dockapp_copyarea(d->parts, d->pixmap, 5 + xd, 49, 5, 9, 48,
                             45);
    }
}

void
draw_pcgraph(DrawConfig *d, AcpiInfos *infos)
{
    if (infos && infos->bat) {
        int xd, nb, num;

        num = infos->bat->percentage / 6.25;
        if (num < 0 || !infos->bat->plugged)
            num = 0;

        if (d->backlight == LIGHTON)
            xd = 102;
        else
            xd = 100;

        /* draw digit */
        for (nb = 0; nb < num; nb++)
            dockapp_copyarea(d->parts, d->pixmap, xd, 0, 2, 9, 6 + nb * 3,
                             33);
        /* draw the rest */
        if (num < 16)
            dockapp_copyarea(BACKGROUND_PIXMAP(d), d->pixmap, 6 + num * 3,
                             33, 46 - num * 3, 9, 6 + num * 3, 33);
    }
}

void
draw_background(DrawConfig *d, light l)
{
    if (l == LIGHTON)
        dockapp_copyarea(d->backdrop_on, d->pixmap, 0, 0, 58, 58, 0, 0);
    else
        dockapp_copyarea(d->backdrop_off, d->pixmap, 0, 0, 58, 58, 0, 0);
}
