const expect = chai.expect;

import RFB from '../core/rfb.js';

import FakeWebSocket from './fake.websocket.js';

function fakeGetRandomValues(arr) {
    if (arr.length === 16) {
        arr.set(new Uint8Array([
            0x1c, 0x08, 0xfe, 0x21, 0x78, 0xef, 0x4e, 0xf9,
            0x3f, 0x05, 0xec, 0xea, 0xd4, 0x6b, 0xa5, 0xd5,
        ]));
    } else {
        arr.set(new Uint8Array([
            0xee, 0xe2, 0xf1, 0x5a, 0x3c, 0xa7, 0xbe, 0x95,
            0x6f, 0x2a, 0x75, 0xfd, 0x62, 0x01, 0xcb, 0xbf,
            0x43, 0x74, 0xca, 0x47, 0x4d, 0xfb, 0x0f, 0xcf,
            0x3a, 0x6d, 0x55, 0x6b, 0x59, 0x3a, 0xf6, 0x87,
            0xcb, 0x03, 0xb7, 0x28, 0x35, 0x7b, 0x15, 0x8e,
            0xb6, 0xc8, 0x8f, 0x2d, 0x5e, 0x7b, 0x1c, 0x9a,
            0x32, 0x55, 0xe7, 0x64, 0x36, 0x25, 0x7b, 0xa3,
            0xe9, 0x4f, 0x6f, 0x97, 0xdc, 0xa4, 0xd4, 0x62,
            0x6d, 0x7f, 0xab, 0x02, 0x6b, 0x13, 0x56, 0x69,
            0xfb, 0xd0, 0xd4, 0x13, 0x76, 0xcd, 0x0d, 0xd0,
            0x1f, 0xd1, 0x0c, 0x63, 0x3a, 0x34, 0x20, 0x6c,
            0xbb, 0x60, 0x45, 0x82, 0x23, 0xfd, 0x7c, 0x77,
            0x6d, 0xcc, 0x5e, 0xaa, 0xc3, 0x0c, 0x43, 0xb7,
            0x8d, 0xc0, 0x27, 0x6e, 0xeb, 0x1d, 0x6c, 0x5f,
            0xd8, 0x1c, 0x3c, 0x1c, 0x60, 0x2e, 0x82, 0x15,
            0xfd, 0x2e, 0x5f, 0x3a, 0x15, 0x53, 0x14, 0x70,
            0x4f, 0xe1, 0x65, 0x68, 0x35, 0x6d, 0xc7, 0x64,
            0xdb, 0xdd, 0x09, 0x31, 0x4f, 0x7b, 0x6d, 0x6c,
            0x77, 0x59, 0x5e, 0x1e, 0xfa, 0x4b, 0x06, 0x14,
            0xbe, 0xdc, 0x9c, 0x3d, 0x7b, 0xed, 0xf3, 0x2b,
            0x19, 0x26, 0x11, 0x8e, 0x3f, 0xab, 0x73, 0x9a,
            0x0a, 0x3a, 0xaa, 0x85, 0x06, 0xd5, 0xca, 0x3f,
            0xc3, 0xe2, 0x33, 0x7f, 0x97, 0x74, 0x98, 0x8f,
            0x2f, 0xa5, 0xfc, 0x7e, 0xb1, 0x77, 0x71, 0x58,
            0xf0, 0xbc, 0x04, 0x59, 0xbb, 0xb4, 0xc6, 0xcc,
            0x0f, 0x06, 0xcd, 0xa2, 0xd5, 0x01, 0x2f, 0xb2,
            0x22, 0x0b, 0xfc, 0x1e, 0x59, 0x9f, 0xd3, 0x4f,
            0x30, 0x95, 0xc6, 0x80, 0x0f, 0x69, 0xf3, 0x4a,
            0xd4, 0x36, 0xb6, 0x5a, 0x0b, 0x16, 0x0d, 0x81,
            0x31, 0xb0, 0x69, 0xd4, 0x4e,
        ]));
    }
}

async function fakeGeneratekey() {
    let key = JSON.parse('{"alg":"RSA-OAEP-256","d":"B7QR2yI8sXjo8vQhJpX9odqqR\
6wIuPrTM1B1JJEKVeSrr7OYcc1FRJ52Vap9LIAU-ezigs9QDvWMxknB8motLnG69Wck37nt9_z4s8l\
FQp0nROA-oaR92HW34KNL1b2fEVWGI0N86h730MvTJC5O2cmKeMezIG-oNqbbfFyP8AW-WLdDlgZm1\
1-FjzhbVpb0Bc7nRSgBPSV-EY6Sl-LuglxDx4LaTdQW7QE_WXoRUt-GYGfTseuFQQK5WeoyX3yBtQy\
dpauW6rrgyWdtP4hDFIoZsX6w1i-UMWMMwlIB5FdnUSi26igVGADGpV_vGMP36bv-EHp0bY-Qp0gpI\
fLfgQ","dp":"Z1v5UceFfV2bhmbG19eGYb30jFxqoRBq36PKNY7IunMs1keYy0FpLbyGhtgMZ1Ymm\
c8wEzGYsCPEP-ykcun_rlyu7YxmcnyC9YQqTqLyqvO-7rUqDvk9TMfdqWFP6heADRhKZmEbmcau6_m\
2MwwK9kOkMKWvpqp8_TpJMnAH7zE","dq":"OBacRE15aY3NtCR4cvP5os3sT70JbDdDLHT3IHZM6r\
E35CYNpLDia2chm_wnMcYvKFW9zC2ajRZ15i9c_VXQzS7ZlTaQYBFyMt7kVhxMEMFsPv1crD6t3uEI\
j0LNuNYyy0jkon_LPZKQFK654CiL-L2YaNXOH4HbHP02dWeVQIE","e":"AQAB","ext":true,"ke\
y_ops":["decrypt"],"kty":"RSA","n":"m1c92ZFk9ZI6l_O4YFiNxbv0Ng94SB3yThy1P_mcqr\
GDQkRiGVdcTxAk38T9PgLztmspF-6U5TAHO-gSmmW88AC9m6f1Mspps6r7zl-M_OG-TwvGzf3BDz8z\
Eg1FPbZV7whO1M4TCAZ0PqwG7qCc6nK1WiAhaKrSpzuPdL1igfNBsX7qu5wgw4ZTTGSLbVC_LfULQ5\
FADgFTRXUSaxm1F8C_Lwy6a2e4nTcXilmtN2IHUjHegzm-Tq2HizmR3ARdWJpESYIW5-AXoiqj29tD\
rqCmu2WPkB2psVp83IzZfaQNQzjNfvA8GpimkcDCkP5VMRrtKCcG4ZAFnO-A3NBX_Q","p":"2Q_lN\
L7vCOBzAppYzCZo3WSh0hX-MOZyPUznks5U2TjmfdNZoL6_FJRiGyyLvwSiZFdEAAvpAyESFfFigng\
AqMLSf448nPg15VUGj533CotsEM0WpoEr1JCgqdUbgDAfJQIBcwOmegBqd7lWm7uzEnRCvouB70ybk\
JfpdprhkVE","q":"tzTt-F3g2u_3Ctj26Ho9iN_wC_W0lXGzslLt5nLmss8JqdLoDDrijjU-gjeRh\
7lgiuHdUc3dorfFKbaMNOjoW3QKqt9oZ1JM0HKeRw0X2PnWW_0WK6DK5ASWDTXbMq2sUZqJvYEyL74\
H2Zrt0RPAux7XQLEVgND6ROdXnMJ70O0","qi":"qfl4cXQkz4BNqa2De0-PfdU-8d1w3onnaGqx1D\
s2fHzD_SJ4cNghn2TksoT9Qo64b3pUjH9igi2pyEjomk6D12N6FG0e10u7vFKv3W5YqUOgTpYdbcWH\
dZ2qZWJU0XQZIrF8jLGTOO4GYP6_9sJ5R7Wk_0MdqQy8qvixWD4zLcY"}');
    key = await window.crypto.subtle.importKey("jwk", key, {
        name: "RSA-OAEP",
        hash: {name: "SHA-256"}
    }, true, ["decrypt"]);
    return {privateKey: key};
}

const receiveData = new Uint8Array([
    // server public key
    0x00, 0x00, 0x08, 0x00, 0xac, 0x1a, 0xbc, 0x42,
    0x8a, 0x2a, 0x69, 0x65, 0x54, 0xf8, 0x9a, 0xe6,
    0x43, 0xaa, 0xf7, 0x27, 0xf6, 0x2a, 0xf8, 0x8f,
    0x36, 0xd4, 0xae, 0x54, 0x0f, 0x16, 0x28, 0x08,
    0xc2, 0x5b, 0xca, 0x23, 0xdc, 0x27, 0x88, 0x1a,
    0x12, 0x82, 0xa8, 0x54, 0xea, 0x00, 0x99, 0x8d,
    0x02, 0x1d, 0x77, 0x4a, 0xeb, 0xd0, 0x93, 0x40,
    0x79, 0x86, 0xcb, 0x37, 0xd4, 0xb2, 0xc7, 0xcd,
    0x93, 0xe1, 0x00, 0x4d, 0x86, 0xff, 0x97, 0x33,
    0x0c, 0xad, 0x51, 0x47, 0x45, 0x85, 0x56, 0x07,
    0x65, 0x21, 0x7c, 0x57, 0x6d, 0x68, 0x7d, 0xd7,
    0x00, 0x43, 0x0c, 0x9d, 0x3b, 0xa1, 0x5a, 0x11,
    0xed, 0x51, 0x77, 0xf9, 0xd1, 0x5b, 0x33, 0xd7,
    0x1a, 0xeb, 0x65, 0x57, 0xc0, 0x01, 0x51, 0xff,
    0x9b, 0x82, 0xb3, 0xeb, 0x82, 0xc2, 0x1f, 0xca,
    0x47, 0xc0, 0x6a, 0x09, 0xe0, 0xf7, 0xda, 0x39,
    0x85, 0x12, 0xe7, 0x45, 0x8d, 0xb4, 0x1a, 0xda,
    0xcb, 0x86, 0x58, 0x52, 0x37, 0x66, 0x9d, 0x8a,
    0xce, 0xf2, 0x18, 0x78, 0x7d, 0x7f, 0xf0, 0x07,
    0x94, 0x8e, 0x6b, 0x17, 0xd9, 0x00, 0x2a, 0x3a,
    0xb9, 0xd4, 0x77, 0xde, 0x70, 0x85, 0xc4, 0x3a,
    0x62, 0x10, 0x02, 0xee, 0xba, 0xd8, 0xc0, 0x62,
    0xd0, 0x8e, 0xc1, 0x98, 0x19, 0x8e, 0x39, 0x0f,
    0x3e, 0x1d, 0x61, 0xb1, 0x93, 0x13, 0x59, 0x39,
    0xcb, 0x96, 0xf2, 0x17, 0xc9, 0xe1, 0x41, 0xd3,
    0x20, 0xdd, 0x62, 0x5e, 0x7d, 0x53, 0xd6, 0xb7,
    0x1d, 0xfe, 0x02, 0x18, 0x1f, 0xe0, 0xef, 0x3d,
    0x94, 0xe3, 0x0a, 0x9c, 0x59, 0x54, 0xd8, 0x98,
    0x16, 0x9c, 0x31, 0xda, 0x41, 0x0f, 0x2e, 0x71,
    0x68, 0xe0, 0xa2, 0x62, 0x3e, 0xe5, 0x25, 0x31,
    0xcf, 0xfc, 0x67, 0x63, 0xc3, 0xb0, 0xda, 0x3f,
    0x7b, 0x59, 0xbe, 0x7e, 0x9e, 0xa8, 0xd0, 0x01,
    0x4f, 0x43, 0x7f, 0x8d, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x01,
    // server random
    0x01, 0x00, 0x5b, 0x58, 0x2a, 0x96, 0x2d, 0xbb,
    0x88, 0xec, 0xc3, 0x54, 0x00, 0xf3, 0xbb, 0xbe,
    0x17, 0xa3, 0x84, 0xd3, 0xef, 0xd8, 0x4a, 0x31,
    0x09, 0x20, 0xdd, 0xbc, 0x16, 0x9d, 0xc9, 0x5b,
    0x99, 0x62, 0x86, 0xfe, 0x0b, 0x28, 0x4b, 0xfe,
    0x5b, 0x56, 0x2d, 0xcb, 0x6e, 0x6f, 0xec, 0xf0,
    0x53, 0x0c, 0x33, 0x84, 0x93, 0xc9, 0xbf, 0x79,
    0xde, 0xb3, 0xb9, 0x29, 0x60, 0x78, 0xde, 0xe6,
    0x1d, 0xa7, 0x89, 0x48, 0x3f, 0xd1, 0x58, 0x66,
    0x27, 0x9c, 0xd4, 0x6e, 0x72, 0x9c, 0x6e, 0x4a,
    0xc0, 0x69, 0x79, 0x6f, 0x79, 0x0f, 0x13, 0xc4,
    0x20, 0xcf, 0xa6, 0xbb, 0xce, 0x18, 0x6d, 0xd5,
    0x9e, 0xd9, 0x67, 0xbe, 0x61, 0x43, 0x67, 0x11,
    0x76, 0x2f, 0xfd, 0x78, 0x75, 0x2b, 0x89, 0x35,
    0xdd, 0x0f, 0x13, 0x7f, 0xee, 0x78, 0xad, 0x32,
    0x56, 0x21, 0x81, 0x08, 0x1f, 0xcf, 0x4c, 0x29,
    0xa3, 0xeb, 0x89, 0x2d, 0xbe, 0xba, 0x8d, 0xe4,
    0x69, 0x28, 0xba, 0x53, 0x82, 0xce, 0x5c, 0xf6,
    0x5e, 0x5e, 0xa5, 0xb3, 0x88, 0xd8, 0x3d, 0xab,
    0xf4, 0x24, 0x9e, 0x3f, 0x04, 0xaf, 0xdc, 0x48,
    0x90, 0x53, 0x37, 0xe6, 0x82, 0x1d, 0xe0, 0x15,
    0x91, 0xa1, 0xc6, 0xa9, 0x54, 0xe5, 0x2a, 0xb5,
    0x64, 0x2d, 0x93, 0xc0, 0xc0, 0xe1, 0x0f, 0x6a,
    0x4b, 0xdb, 0x77, 0xf8, 0x4a, 0x0f, 0x83, 0x36,
    0xdd, 0x5e, 0x1e, 0xdd, 0x39, 0x65, 0xa2, 0x11,
    0xc2, 0xcf, 0x56, 0x1e, 0xa1, 0x29, 0xae, 0x11,
    0x9f, 0x3a, 0x82, 0xc7, 0xbd, 0x89, 0x6e, 0x59,
    0xb8, 0x59, 0x17, 0xcb, 0x65, 0xa0, 0x4b, 0x4d,
    0xbe, 0x33, 0x32, 0x85, 0x9c, 0xca, 0x5e, 0x95,
    0xc2, 0x5a, 0xd0, 0xc9, 0x8b, 0xf1, 0xf5, 0x14,
    0xcf, 0x76, 0x80, 0xc2, 0x24, 0x0a, 0x39, 0x7e,
    0x60, 0x64, 0xce, 0xd9, 0xb8, 0xad, 0x24, 0xa8,
    0xdf, 0xcb,
    // server hash
    0x00, 0x14, 0x39, 0x30, 0x66, 0xb5, 0x66, 0x8a,
    0xcd, 0xb9, 0xda, 0xe0, 0xde, 0xcb, 0xf6, 0x47,
    0x5f, 0x54, 0x66, 0xe0, 0xbc, 0x49, 0x37, 0x01,
    0xf2, 0x9e, 0xef, 0xcc, 0xcd, 0x4d, 0x6c, 0x0e,
    0xc6, 0xab, 0x28, 0xd4, 0x7b, 0x13,
    // subtype
    0x00, 0x01, 0x30, 0x2a, 0xc3, 0x0b, 0xc2, 0x1c,
    0xeb, 0x02, 0x44, 0x92, 0x5d, 0xfd, 0xf9, 0xa7,
    0x94, 0xd0, 0x19,
]);

const sendData = new Uint8Array([
    // client public key
    0x00, 0x00, 0x08, 0x00, 0x9b, 0x57, 0x3d, 0xd9,
    0x91, 0x64, 0xf5, 0x92, 0x3a, 0x97, 0xf3, 0xb8,
    0x60, 0x58, 0x8d, 0xc5, 0xbb, 0xf4, 0x36, 0x0f,
    0x78, 0x48, 0x1d, 0xf2, 0x4e, 0x1c, 0xb5, 0x3f,
    0xf9, 0x9c, 0xaa, 0xb1, 0x83, 0x42, 0x44, 0x62,
    0x19, 0x57, 0x5c, 0x4f, 0x10, 0x24, 0xdf, 0xc4,
    0xfd, 0x3e, 0x02, 0xf3, 0xb6, 0x6b, 0x29, 0x17,
    0xee, 0x94, 0xe5, 0x30, 0x07, 0x3b, 0xe8, 0x12,
    0x9a, 0x65, 0xbc, 0xf0, 0x00, 0xbd, 0x9b, 0xa7,
    0xf5, 0x32, 0xca, 0x69, 0xb3, 0xaa, 0xfb, 0xce,
    0x5f, 0x8c, 0xfc, 0xe1, 0xbe, 0x4f, 0x0b, 0xc6,
    0xcd, 0xfd, 0xc1, 0x0f, 0x3f, 0x33, 0x12, 0x0d,
    0x45, 0x3d, 0xb6, 0x55, 0xef, 0x08, 0x4e, 0xd4,
    0xce, 0x13, 0x08, 0x06, 0x74, 0x3e, 0xac, 0x06,
    0xee, 0xa0, 0x9c, 0xea, 0x72, 0xb5, 0x5a, 0x20,
    0x21, 0x68, 0xaa, 0xd2, 0xa7, 0x3b, 0x8f, 0x74,
    0xbd, 0x62, 0x81, 0xf3, 0x41, 0xb1, 0x7e, 0xea,
    0xbb, 0x9c, 0x20, 0xc3, 0x86, 0x53, 0x4c, 0x64,
    0x8b, 0x6d, 0x50, 0xbf, 0x2d, 0xf5, 0x0b, 0x43,
    0x91, 0x40, 0x0e, 0x01, 0x53, 0x45, 0x75, 0x12,
    0x6b, 0x19, 0xb5, 0x17, 0xc0, 0xbf, 0x2f, 0x0c,
    0xba, 0x6b, 0x67, 0xb8, 0x9d, 0x37, 0x17, 0x8a,
    0x59, 0xad, 0x37, 0x62, 0x07, 0x52, 0x31, 0xde,
    0x83, 0x39, 0xbe, 0x4e, 0xad, 0x87, 0x8b, 0x39,
    0x91, 0xdc, 0x04, 0x5d, 0x58, 0x9a, 0x44, 0x49,
    0x82, 0x16, 0xe7, 0xe0, 0x17, 0xa2, 0x2a, 0xa3,
    0xdb, 0xdb, 0x43, 0xae, 0xa0, 0xa6, 0xbb, 0x65,
    0x8f, 0x90, 0x1d, 0xa9, 0xb1, 0x5a, 0x7c, 0xdc,
    0x8c, 0xd9, 0x7d, 0xa4, 0x0d, 0x43, 0x38, 0xcd,
    0x7e, 0xf0, 0x3c, 0x1a, 0x98, 0xa6, 0x91, 0xc0,
    0xc2, 0x90, 0xfe, 0x55, 0x31, 0x1a, 0xed, 0x28,
    0x27, 0x06, 0xe1, 0x90, 0x05, 0x9c, 0xef, 0x80,
    0xdc, 0xd0, 0x57, 0xfd, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x01,
    // client random
    0x01, 0x00, 0x84, 0x7f, 0x26, 0x54, 0x74, 0xf6,
    0x47, 0xaf, 0x33, 0x64, 0x0d, 0xa6, 0xe5, 0x30,
    0xba, 0xe6, 0xe4, 0x8e, 0x50, 0x40, 0x71, 0x1c,
    0x0e, 0x06, 0x63, 0xf5, 0x07, 0x2a, 0x26, 0x68,
    0xd6, 0xcf, 0xa6, 0x80, 0x84, 0x5e, 0x64, 0xd4,
    0x5e, 0x62, 0x31, 0xfe, 0x44, 0x51, 0x0b, 0x7c,
    0x4d, 0x55, 0xc5, 0x4a, 0x7e, 0x0d, 0x4d, 0x9b,
    0x84, 0xb4, 0x32, 0x2b, 0x4d, 0x8a, 0x34, 0x8d,
    0xc8, 0xcf, 0x19, 0x3b, 0x64, 0x82, 0x27, 0x9e,
    0xa7, 0x70, 0x2a, 0xc1, 0xb8, 0xf3, 0x6a, 0x3a,
    0xf2, 0x75, 0x6e, 0x1d, 0xeb, 0xb6, 0x70, 0x7a,
    0x15, 0x18, 0x38, 0x00, 0xb4, 0x4f, 0x55, 0xb5,
    0xd8, 0x03, 0x4e, 0xb8, 0x53, 0xff, 0x80, 0x62,
    0xf1, 0x9d, 0x27, 0xe8, 0x2a, 0x3d, 0x98, 0x19,
    0x32, 0x09, 0x7e, 0x9a, 0xb0, 0xc7, 0x46, 0x23,
    0x10, 0x85, 0x35, 0x00, 0x96, 0xce, 0xb3, 0x2c,
    0x84, 0x8d, 0xf4, 0x9e, 0xa8, 0x42, 0x67, 0xed,
    0x09, 0xa6, 0x09, 0x97, 0xb3, 0x64, 0x26, 0xfb,
    0x71, 0x11, 0x9b, 0x3f, 0xbb, 0x57, 0xb8, 0x5b,
    0x2e, 0xc5, 0x2d, 0x8c, 0x5c, 0xf7, 0xef, 0x27,
    0x25, 0x88, 0x42, 0x45, 0x43, 0xa4, 0xe7, 0xde,
    0xea, 0xf9, 0x15, 0x7b, 0x5d, 0x66, 0x24, 0xce,
    0xf7, 0xc8, 0x2f, 0xc5, 0xc0, 0x3d, 0xcd, 0xf2,
    0x62, 0xfc, 0x1a, 0x5e, 0xec, 0xff, 0xf1, 0x1b,
    0xc8, 0xdb, 0xc1, 0x0f, 0x54, 0x66, 0x9e, 0xfd,
    0x99, 0x9b, 0x23, 0x70, 0x62, 0x37, 0x80, 0xad,
    0x91, 0x6b, 0x84, 0x85, 0x6a, 0x4c, 0x80, 0x9e,
    0x60, 0x8a, 0x93, 0xa3, 0xc8, 0x8e, 0xc4, 0x4b,
    0x4d, 0xb4, 0x8e, 0x3e, 0xaf, 0xce, 0xcd, 0x83,
    0xe5, 0x21, 0x90, 0x95, 0x20, 0x3c, 0x82, 0xb4,
    0x7c, 0xab, 0x63, 0x9c, 0xae, 0xc3, 0xc9, 0x71,
    0x1a, 0xec, 0x34, 0x18, 0x47, 0xec, 0x5c, 0x4d,
    0xed, 0x84,
    // client hash
    0x00, 0x14, 0x9c, 0x91, 0x9e, 0x76, 0xcf, 0x1e,
    0x66, 0x87, 0x5e, 0x29, 0xf1, 0x13, 0x80, 0xea,
    0x7d, 0xec, 0xae, 0xf9, 0x60, 0x01, 0xd3, 0x6f,
    0xb7, 0x9e, 0xb2, 0xcd, 0x2d, 0xc8, 0xf8, 0x84,
    0xb2, 0x9f, 0xc3, 0x7e, 0xb4, 0xbe,
    // credentials
    0x00, 0x08, 0x9d, 0xc8, 0x3a, 0xb8, 0x80, 0x4f,
    0xe3, 0x52, 0xdb, 0x62, 0x9e, 0x97, 0x64, 0x82,
    0xa8, 0xa1, 0x6b, 0x7e, 0x4d, 0x68, 0x8c, 0x29,
    0x91, 0x38,
]);

describe('RA2 handshake', function () {
    let sock;
    let rfb;
    let sentData;

    before(() => {
        FakeWebSocket.replace();
        sinon.stub(window.crypto, "getRandomValues").callsFake(fakeGetRandomValues);
        sinon.stub(window.crypto.subtle, "generateKey").callsFake(fakeGeneratekey);
    });
    after(() => {
        FakeWebSocket.restore();
        window.crypto.getRandomValues.restore();
        window.crypto.subtle.generateKey.restore();
    });

    it('should fire the serververification event', function (done) {
        sentData = new Uint8Array();
        rfb = new RFB(document.createElement('div'), "ws://example.com");
        sock = rfb._sock;
        sock.send = (data) => {
            let res = new Uint8Array(sentData.length + data.length);
            res.set(sentData);
            res.set(data, sentData.length);
            sentData = res;
        };
        rfb._rfbInitState = "Security";
        rfb._rfbVersion = 3.8;
        sock._websocket._receiveData(new Uint8Array([1, 6]));
        rfb.addEventListener("serververification", (e) => {
            expect(e.detail.publickey).to.eql(receiveData.slice(0, 516));
            done();
        });
        sock._websocket._receiveData(receiveData);
    });

    it('should handle approveServer and fire the credentialsrequired event', function (done) {
        rfb.addEventListener("credentialsrequired", (e) => {
            expect(e.detail.types).to.eql(["password"]);
            done();
        });
        rfb.approveServer();
    });

    it('should match sendData after sending credentials', function (done) {
        rfb.addEventListener("securityresult", (event) => {
            expect(sentData.slice(1)).to.eql(sendData);
            done();
        });
        rfb.sendCredentials({ "password": "123456" });
    });
});
