// Copyright 2022 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "generator/internal/stub_factory_rest_generator.h"
#include "generator/internal/codegen_utils.h"
#include "generator/internal/printer.h"
#include <google/protobuf/descriptor.h>

namespace google {
namespace cloud {
namespace generator_internal {

StubFactoryRestGenerator::StubFactoryRestGenerator(
    google::protobuf::ServiceDescriptor const* service_descriptor,
    VarsDictionary service_vars,
    std::map<std::string, VarsDictionary> service_method_vars,
    google::protobuf::compiler::GeneratorContext* context,
    std::vector<MixinMethod> const& mixin_methods)
    : ServiceCodeGenerator(
          "stub_factory_rest_header_path", "stub_factory_rest_cc_path",
          service_descriptor, std::move(service_vars),
          std::move(service_method_vars), context, mixin_methods) {}

Status StubFactoryRestGenerator::GenerateHeader() {
  HeaderPrint(CopyrightLicenseFileHeader());
  HeaderPrint(R"""(
// Generated by the Codegen C++ plugin.
// If you make any local changes, they will be lost.
// source: $proto_file_name$

#ifndef $header_include_guard$
#define $header_include_guard$

)""");

  // includes
  HeaderLocalIncludes({vars("stub_rest_header_path"), "google/cloud/options.h",
                       "google/cloud/version.h"});
  HeaderSystemIncludes({"memory"});

  auto result = HeaderOpenNamespaces(NamespaceType::kInternal);
  if (!result.ok()) return result;

  HeaderPrint(R"""(
std::shared_ptr<$stub_rest_class_name$> CreateDefault$stub_rest_class_name$(
    Options const& options);
)""");

  HeaderCloseNamespaces();
  // close header guard
  HeaderPrint("\n#endif  // $header_include_guard$\n");
  return {};
}

Status StubFactoryRestGenerator::GenerateCc() {
  CcPrint(CopyrightLicenseFileHeader());
  CcPrint(R"""(
// Generated by the Codegen C++ plugin.
// If you make any local changes, they will be lost.
// source: $proto_file_name$

)""");

  // includes
  CcLocalIncludes(
      {vars("stub_factory_rest_header_path"), vars("logging_rest_header_path"),
       vars("metadata_rest_header_path"), vars("stub_rest_header_path"),
       "google/cloud/common_options.h",
       "google/cloud/internal/populate_rest_options.h",
       "google/cloud/rest_options.h", "google/cloud/internal/algorithm.h",
       "google/cloud/options.h", "google/cloud/log.h", "absl/strings/match.h"});
  CcSystemIncludes({"memory", "utility"});

  auto result = CcOpenNamespaces(NamespaceType::kInternal);
  if (!result.ok()) return result;

  // factory function implementation
  CcPrint(R"""(
std::shared_ptr<$stub_rest_class_name$>
CreateDefault$stub_rest_class_name$(Options const& options) {
  auto opts = internal::PopulateRestOptions(options);
  std::shared_ptr<$stub_rest_class_name$> stub =
      std::make_shared<Default$stub_rest_class_name$>(std::move(opts));
  stub = std::make_shared<$metadata_rest_class_name$>(std::move(stub));
  if (internal::Contains(
      options.get<LoggingComponentsOption>(), "rpc")) {
    GCP_LOG(INFO) << "Enabled logging for REST rpc calls";
    stub = std::make_shared<$logging_rest_class_name$>(
        std::move(stub),
        options.get<RestTracingOptionsOption>(),
        options.get<LoggingComponentsOption>());
  }
  return stub;
}
)""");

  CcCloseNamespaces();
  return {};
}

}  // namespace generator_internal
}  // namespace cloud
}  // namespace google
