#! /bin/sh
#

#
# Include config variables
#
. ./config

#
# FIND defaultpathname searchpath name1 name2 ...
# This function tries to find an executable program.
# If the first argument is an executable file then that is returned.
# Otherwise, the PATH and searchpath are searched for an executable with
# one of the names name1, name2, etc.
# The first executable file found is returned.
#
FIND()
{
  SAVED_IFS="${IFS}"
  user_supplied=$1
  search_path=$2
  if test -f "$user_supplied" -a -x "$user_supplied"
  then
    echo "$user_supplied"
  else
    shift; shift
    for prog in $*
    do
      IFS=":"
      for path in $PATH $search_path
      do
        if test -f "$path/$prog" -a -x "$path/$prog"
        then
          echo "$path/$prog"
          IFS="${SAVED_IFS}"
          return
        fi
      done
      IFS="${SAVED_IFS}"
    done
  fi
}

#
# Try to find wish8.3, tclsh8.3 and openssl
#
WISH=`FIND "$WISH" "" wish8.4 wish8.3 wish`
TCLSH=`FIND "$TCLSH" "" tclsh8.4 tclsh8.3 tclsh`
OPENSSL=`FIND "$OPENSSL" "$OPENSSL_SEARCH_PATH" openssl ssl`

#
# Bail out if no wish...
#
if test -x "$WISH"
then
	echo "Found wish program: $WISH"
else
	echo "No wish8.3 or wish shell found."
	echo "Please install Tcl/Tk or edit the file 'config' by hand."
	exit 1
fi

#
# ...and if no tclsh.
#
if test -x "$TCLSH"
then
	echo "Found tclsh program: $TCLSH"
else
	echo "No tclsh8.3 or tclsh shell found."
        echo "Please install Tcl/Tk or edit the file 'config' by hand."
        exit 1
fi

#
# Build the tkpasman executable
#
sed -e "s,@wish@,$WISH,"		\
    -e "s,@tclsh@,$TCLSH,"		\
    -e "s,@configfile@,$CONFIGFILE,"	\
    -e "s,@pwdfile@,$PWDFILE,"	\
    -e "s,@version@,$VERSION,"	\
    tkpasman-hdr.tcl > $PROGNAME

#
# If requested, find and initialize OpenSSL
#
if test "$USE_OPENSSL" = "true" -a -x "$OPENSSL"
then
  echo "Found OpenSSL program: $OPENSSL"
  ENC=`$OPENSSL list-cipher-commands|sort|egrep "^($ENC)\$"|head -1`
  if test -z "$ENC"
  then
    echo "Could not find a suitable encryption algorithm."
    echo "Try: \"$OPENSSL list-cipher-commands\" to view a list of supported"
    echo "encryption algorithms, read the openssl enc.1 manpage, make up your"
    echo "mind, and put an algorithm name in the \"config\" file."
    echo "Disabling OpenSSL support for now."
  elif echo bla|$OPENSSL $ENC -e -pass stdin|$OPENSSL $ENC -d -k bla
  then
    # It Works!
    echo "set ssl(cmd) {$OPENSSL}"	>> $PROGNAME
    echo "set ssl(enc) {$ENC}"		>> $PROGNAME
    echo "set ssl(passwd) {}"		>> $PROGNAME
    echo "Enabling OpenSSL support:"
    echo "  openssl: $OPENSSL"
    echo "  enc:     $ENC"
    if echo bla|$OPENSSL $ENC -salt -pass stdin|$OPENSSL $ENC -d -salt -k bla
    then
      # Salt option works!
      echo "set ssl(salt) -salt"	>> $PROGNAME
      echo "  salt:    yes"
    else
      echo "set ssl(salt) {}"		>> $PROGNAME
      echo "  salt:    no"
    fi
    cat WARNING
  else
    echo "OpenSSL test failed, not enabling OpenSSL support"
  fi
fi

#
# Build in Help file
#
echo "set Help {"	>> $PROGNAME
cat README		>> $PROGNAME
echo "}"		>> $PROGNAME

#
# Assemble the rest of the program, leaving out comments and empty lines
#
sed -e '/^[[:space:]]*#.*$/d' \
    -e '/^[[:space:]]*$/d'    \
    -e 's/^[[:space:]][[:space:]]*//' \
    -e 's/[[:space:]][[:space:]]*/ /g' \
    cooltk.tcl tkpasman.tcl	>> $PROGNAME
chmod 0755 $PROGNAME
echo "Built $PROGNAME succesfully."
