/*
 *   dspgpc.c -- Routines that provide all GPC (general purpose connection)
 *           related functionality
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "dspmgr.h"

#include "dspgpc.h"                    // Include function prototypes
#include "dsppcmem.h"                  // Include function prototypes
#include "dspmem.h"                    // Include function prototypes
#include "dspstruc.h"                  // Include function prototypes
#include "dspglist.h"                  // Include CLL handling
#include "dspactiv.h"                  // Include Activate Task routines
#include "dspparse.h"                  // String compare routine
#include "dspbios.h"                   // Include function prototypes
#include "dspfxutl.h"                  // Include function prototypes
#include "dspload.h"                   // Include function prototypes
#include "dspfree.h"                   // Include function prototypes
#include "dspstcmp.h"                  // Include function prototypes
#include "dspispos.h"                  // CH01 Include Mwave/OS interface
#include "dspquery.h"                  // CH01 New loc for NameToXXX fns.

typedef struct gpccbtemplate {      /* gpc control block structure       */
   USHORT     GPCCB_usGetPutPtr;
   USHORT     GPCCB_usSize;
   USHORT     GPCCB_usMaxWPF;
   USHORT     GPCCB_usOtherPtr;
   USHORT     GPCCB_usProtocol;
} RGPCCB;

#define  MAX(value1,value2) ( ((value1) > (value2)) ? (value1) : (value2) )
#define  GPCCB_OTHERPTR FIELDOFFSET(RGPCCB,GPCCB_usOtherPtr)
#define  GPCCB_MINSIZE FIELDOFFSET(RGPCCB,GPCCB_usSize)

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AllocateGPCBuf                                          */
/*                                                                          */
/* FUNCTION: This routines will traverse the GPC connection list of a       */
/*           module and allocate memory in DSP for the GPC buffer and also  */
/*           fill in the Owner and User GPC data structure pointers.        */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRGPC prgpc - Ptr to the OWNER GPC.  Connect the buffer to this   */
/*                      GPC.                                                */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC AllocateGPCBuf(PRGPC prgpc)

{
   RC         ulRC;                    // Error return code
   PRMEM      prmem;                 // ptr to MEM structure of GPC buffer
   PVOID      pBuffer;
   LONG       lTempSize;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspgpc::AllocateGPCBuf entry prgpc %x\n",(int)prgpc);

   /*************************************************************************/
   /* Make sure that the GPC is an OWNER, determine the                     */
   /* size and Allocate the Buffer.                                         */
   /*************************************************************************/

   if (prgpc->GPC_usflgKind != GPC_OWNER) {
      ulRC = DSP_INV_GPC_TYPE;
      MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspgpc::AllocateGPCBuf error ulRC %lx Owner: %s\n",ulRC, prgpc->GPC_pszName); 
      return (ulRC);
   }

   lTempSize = prgpc->GPC_lBufferSize;
   if ((ulRC = AllocateMem(prgpc->GPC_prseg->SEG_prtsk->TSK_prdsp, lTempSize,
      MEM_DATA, lTempSize, 0, &prmem)) != DSP_NOERROR)

      return (ulRC);

   lTempSize *= 2;                     /* lTempSize now in bytes            */
   ulRC = AllocateTemp(lTempSize, 1, &pBuffer);
   if (DSP_NOERROR != ulRC)
      return (ulRC);

   memset(pBuffer, 0, lTempSize);      /* Write zeros to buffer             */

   /* Write Zeros to DSP                                                    */

   PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspgpc::AllocateGPCBuf");
   PRINT_DEBUG(EMBEDFILEPTR, "  prgpc (0x%X)\n", prgpc);
   PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR, "WRITE41:  Zeros to DSP",
        EOF, lTempSize >> 1,
        prgpc->GPC_prseg->SEG_prtsk->TSK_prdsp, prmem->MEM_ulSegAddr,
        lTempSize >> 1, pBuffer);

   ulRC = DspBIOS_D_Write(prgpc->GPC_prseg->SEG_prtsk->TSK_prdsp,
      prmem->MEM_ulSegAddr, lTempSize >> 1, pBuffer);

   if (DSP_NOERROR != FreeTemp(pBuffer, lTempSize))
      return (DSP_INTERNAL_ERROR);

   if (DSP_NOERROR != ulRC)
      return (ulRC);

   /* Connect the GPC structure with MEM structure for buffer               */

   prgpc->GPC_prmemBuffer = prmem;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspgpc::AllocateGPCBuf exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: ProcessGPCCL                                            */
/*                                                                          */
/* FUNCTION: This routines will traverse the GPC connection list of a       */
/*           module, connect the Owner and User GPCs, and also              */
/*           fill in the Owner and User GPC data structure pointers.        */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRMOD prmod - Ptr to module. Connect all GPC's in this module.    */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: GPCCL deleted for the module since its no longer needed.   */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC ProcessGPCCL(PRMOD prmod)

{
   RC         ulRC=DSP_NOERROR;        /* code assumes initialization */
   PRGPCCL    prgpccl;
   PRTSK      prOwnertask;
   PRTSK      prUsertask;


   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspgpc::ProcessGPCCL entry prmod %x\n",(int)prmod);

   /*************************************************************************/
   /* Traverse the GPC Connection list for this module.                     */
   /*************************************************************************/

   while (prmod->MOD_prgpcclTail != NULL)/* list empty?                     */
      {
      prgpccl = prmod->MOD_prgpcclTail;

      /**********************************************************************/
      /* for the OWNER gpc......                                            */
      /* If the module name is 'BIOS' set prOwnertask to NULL,              */
      /* else ignore module name, search module for task.                   */
      /**********************************************************************/

      if ((mwstrcmpnull(prgpccl->GPCCL_pszGpc1ModName, "BIOS\0") == 0))
         prOwnertask = NULL;
      else {
         if (prgpccl->GPCCL_pszGpc1TaskName == NULL)
            ulRC = DSP_NAME_NOT_FOUND;
         else {
            ulRC = NameToTaskHandle((HMOD)prmod,
               prgpccl->GPCCL_pszGpc1TaskName, (HTASK *)&prOwnertask);
         }
         if (ulRC != DSP_NOERROR) {
	   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspgpc::ProcessGPCCL error ulRC %lx Owner Task: %s\n",ulRC,
               prgpccl->GPCCL_pszGpc1TaskName);            /* LBM */
            return (ulRC);
         }
      }

      /**********************************************************************/
      /* for the USER gpc......                                             */
      /* If the module name is 'BIOS' set prUsertask to NULL,               */
      /* else ignore module name, search module for task.                   */
      /**********************************************************************/

      if ((mwstrcmpnull(prgpccl->GPCCL_pszGpc2ModName, "BIOS\0") == 0))
         prUsertask = NULL;
      else {
         if (prgpccl->GPCCL_pszGpc2TaskName == NULL)
            ulRC = DSP_NAME_NOT_FOUND;
         else {
            ulRC = NameToTaskHandle((HMOD)prmod,
               prgpccl->GPCCL_pszGpc2TaskName, (HTASK *)&prUsertask);
         }
         if (ulRC != DSP_NOERROR) {
	   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspgpc::ProcessGPCCL error ulRC %lx User task: %s\n",ulRC,
               prgpccl->GPCCL_pszGpc2TaskName);   
            return (ulRC);
         }
      }

      /**********************************************************************/
      /* make the connection and Delete the GPCCL.                          */
      /**********************************************************************/

      if ((ulRC = ConnectGPC(prOwnertask, prgpccl->GPCCL_pszGpc1GPCName,
         prUsertask, prgpccl->GPCCL_pszGpc2GPCName, NULL)) != DSP_NOERROR)
      {
	MW_SYSLOG_4(TRACE_MANAGER_CORE,"dspgpc::ProcessGPCCL error ulRC %lx %s <-> %s\n",ulRC,
            prgpccl->GPCCL_pszGpc1GPCName,
            prgpccl->GPCCL_pszGpc2GPCName);
         return (ulRC);
      }

      if ((ulRC = RemoveGPCCL(prmod, prgpccl)) != DSP_NOERROR)
         return (ulRC);
   }                                  /* end while                         */
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspgpc::ProcessGPCCL exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveGPCCL                                             */
/*                                                                          */
/* FUNCTION: This routines will delete One GPC Control Block from the       */
/*           Connection list.                                               */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRMOD   prmod - Pointer to the Module with the Connection list    */
/*        PRGPCCL prgpccl - Pointer to GPCCL node to delete.                */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveGPCCL(PRMOD prmod,PRGPCCL prgpccl)

{
   RC         ulRC;
   PRGPCCL    prgpcclPrev;
   PRGPCCL    prgpcclCur;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspgpc::RemoveGPCCL entry prmod %x\n",(int)prmod);
   /*************************************************************************/
   /* Check to see if there are any GPCCL blocks to process.                */
   /*************************************************************************/

   if (prmod->MOD_prgpcclTail == NULL)
      return (DSP_NOERROR);

   /*************************************************************************/
   /* Remove..... takes the GPCCL off the linked-list and recovers          */
   /* the structures memory.  So find the right pointers for                */
   /* manipulation; Namely the previous to the current GPCCL                */
   /*************************************************************************/

   prgpcclPrev = prmod->MOD_prgpcclTail;
   prgpcclCur = prgpcclPrev->GPCCL_prgpcclNext;

   while (prgpcclCur != prgpccl) {
      prgpcclPrev = prgpcclCur;
      prgpcclCur = prgpcclCur->GPCCL_prgpcclNext;
   }                                   /* endwhile                          */
   ulRC = RemoveRGPCConnList(&(prmod->MOD_prgpcclTail), prgpccl, prgpcclPrev);

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspgpc::RemoveGPCCL exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: ConnectGPC                                              */
/*                                                                          */
/* FUNCTION: This routines will connect two GPCs.  This includes setting    */
/*           both connectors to the same buffer and                         */
/*           filling in the Owner and User GPC data structure pointers.     */
/*                                                                          */
/* INPUT:                                                                   */
/*        HTASK  hOwnerTask - Handle of Owner Task for GPC.                 */
/*        PSZ   pszOwner   - Name of Owner GPC.                             */
/*        HTASK  hUserTask  - Handle of User Task for GPC.                  */
/*        PSZ   pszUser    - Name of User GPC.                              */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC ConnectGPC(PRTSK prtskOwner,PSZ pszOwner,PRTSK prtskUser,PSZ pszUser,PULONG
               pulDSPAddr)

{
   RC         ulRC=DSP_NOERROR;        // Error return code. CH04 Initialized.
   PRGPC      prgpcOwner;              // Pointer to owner GPC
   PRGPC      prgpcUser;               // Pointer to User GPC
   ULONG      ulAddrOwnerCB;           /* dsp addr of owner control block   */
   ULONG      ulAddrUserCB;            /* dsp addr of user control block    */
   PRBIOT     prbiot;
   USHORT     usBuf;                   /* temperary buffer                  */


   MW_SYSLOG_5(TRACE_MANAGER_CORE,"dspgpc::ConnectGPC entry prtskOwner %x pszOwner %s prtskUser %x pszUser %s\n",
	   (int)prtskOwner,pszOwner,(int)prtskUser,pszUser);
   /************************************************************************/
   /*  CH03 If either prtsk is not NULL it better be valid!  MTS #2675     */
   /************************************************************************/

   if ((prtskUser!=NULL) && (prtskUser->TSK_achValid!=TSKVALIDATE))
      return (DSP_INV_HANDLE); /* CH03 Chk before using MTS #2675 */

   if ((prtskOwner!=NULL) && (prtskOwner->TSK_achValid != TSKVALIDATE))
      return (DSP_INV_HANDLE); /* CH03 Chk before using MTS #2675 */

   /*************************************************************************
    * If either the Owner or User Tasks are NULL.                           *
    *   Call ConnectorToBIOS() to find the bios task                        *
    *   Call GetBIOSTask to Load the BIOS task.                             *
    *                                                                       *
    * If both User and Owner handles are null, ie. BIOS tasks, then return  *
    * DSP_INV_HANDLE because we don't know which DSP to load them on.       *
    * This also prevents a trap which occurred when attempting to access a  *
    * task structure with a null prtskUser pointer                          *
    *************************************************************************/

   if (prtskOwner == NULL) {
      if (prtskUser == NULL)
         ulRC = DSP_INV_HANDLE;
      else {
         ulRC = ConnectorToBIOS(pszOwner, pg->prmgrTail, &prbiot);
         if (ulRC == DSP_NOERROR) {
            ulRC = GetBIOSTask(prbiot->BT_pszFilename, prbiot->BT_pszTaskname,
               prtskUser->TSK_prdsp->DSP_prmodOS, /* CH04 */
               prtskUser->TSK_prdsp,
               &prtskOwner);
         }
      }
   }

   else if (prtskUser == NULL) { /* CH04 Added "else" clause */
      ulRC = ConnectorToBIOS(pszUser, pg->prmgrTail, &prbiot);
      if (ulRC == DSP_NOERROR) {
         ulRC = GetBIOSTask(prbiot->BT_pszFilename, prbiot->BT_pszTaskname,
            prtskOwner->TSK_prdsp->DSP_prmodOS, /* CH04 */
            prtskOwner->TSK_prdsp,
            &prtskUser);
      }
   }

   if (ulRC != DSP_NOERROR) /* CH04 Combined several tests into this one */
      return (ulRC);

   if (prtskOwner->TSK_prdsp != prtskUser->TSK_prdsp)
      return (DSP_INV_GPC_CONNECT);

   if ((ulRC=CheckDSPStatus(prtskOwner->TSK_prdsp)))
      return (ulRC);  /* CH05 Check for card unplugged */

   /*************************************************************************/
   /* Verify that tasks are in standby or active state                      */
   /*************************************************************************/

   if ((prtskOwner->TSK_usflgState == TSK_INACTIVE) ||
      (prtskUser->TSK_usflgState == TSK_INACTIVE))
      return (DSP_INV_TASK_STATE);

   /*************************************************************************/
   /* Verify that there are Data Segments attached to the tasks.            */
   /* Verify that there are GPCs attached to Segments.                      */
   /* The GPCs are in the Data Segments.                                    */
   /*************************************************************************/

   if ((prtskOwner->TSK_prsegDTail == NULL) ||
       (prtskUser->TSK_prsegDTail == NULL))
      return (DSP_NO_DATA);

   /*************************************************************************/
   /* Find the GPCs within the Tasks. Search the GPCs until                 */
   /* a match is found or the entire list has been checked.                 */
   /* Start at Tail->Next and go thru Tail.                                 */
   /* If a match is found, bFound will be set to TRUE and                   */
   /* the loop will end with prgpcNode pointing to the                      */
   /* correct GPC.                                                          */
   /*************************************************************************/
   /*************************************************************************/
   /* Find the Owner GPC.                                                   */
   /*************************************************************************/

   if ((ulRC = FindGPC(prtskOwner, pszOwner, &prgpcOwner)) != DSP_NOERROR)
      return (ulRC);

   if (prgpcOwner->GPC_usflgKind != GPC_OWNER) {
      ulRC = DSP_INV_GPC_TYPE;
      MW_SYSLOG_4(TRACE_MANAGER_CORE,"dspgpc::ConnectGPC error ulRC %lx GPC: %s.%s\n",ulRC, prtskOwner->TSK_pszVirtName, pszOwner);
      return (ulRC);
   }

   /*************************************************************************/
   /* Find the User GPC.                                                    */
   /*************************************************************************/

   if ((ulRC = FindGPC(prtskUser, pszUser, &prgpcUser)) != DSP_NOERROR)
      return (ulRC);

   if ((prgpcUser->GPC_usflgKind&GPC_KINDMASK) != GPC_USER) {
      ulRC = DSP_INV_GPC_TYPE;
      MW_SYSLOG_4(TRACE_MANAGER_CORE,"dspgpc::ConnectGPC error ulRC %lx User GPC: %s.%s\n",ulRC, prtskUser->TSK_pszVirtName, pszUser);
      return (ulRC);
   }

   /*************************************************************************/
   /* Verify that the user GPC is not already connected.                    */
   /* Verify that a User or Safe Data Driven Owner GPC is                   */
   /* not already connected.                                                */
   /*************************************************************************/

   if (prgpcUser->GPC_prgpcOwner != NULL)
      return (DSP_GPC_CONNECTED);
   if ((prgpcOwner->GPC_usUseCount > 0) &&
      (prgpcOwner->GPC_usProtocol&(GPC_USERDD|GPC_SAFEDD)))
      return (DSP_GPC_CONNECTED);

   /**BEGIN CH02 DEC MTS #1528 **********************************************/
   /*************************************************************************/
   /* Check that the GPC buffer sizes are compatible.                       */
   /*************************************************************************/

   if ((prgpcUser->GPC_lBufferSize) < 0) { /* If user GPC is MINSIZE        */
      /**********************************************************************/
      /* MINSIZE is indicated by a negative GPC_lBufferSize, with the       */
      /* absolute value of GPC_lBufferSize being the actual buffer size.    */
      /* (NOTE: Only a user GPC can be specified as MINSIZE)                */
      /* The connection can be made only if owner GPC buffer size is equal  */
      /* to or greater than the user MINIMUM GPC buffer size.  If the       */
      /* addition of the positve owner GPC buffer size with the negative    */
      /* user MINIMUM GPC buffer size results in a negative number the user */
      /* MINIMUM GPC buffer size is greater than the owner GPC buffer size  */
      /* and the connection can not be made.                                */
      /**********************************************************************/
      if ((prgpcOwner->GPC_lBufferSize + prgpcUser->GPC_lBufferSize) < 0)
         return (DSP_INV_GPC_SIZE);

   } else {                                /* If user GPC is NOT MINIZE     */
      if (prgpcOwner->GPC_lBufferSize != (prgpcUser->GPC_lBufferSize))
         return (DSP_INV_GPC_SIZE);
   } /* endif */

   /*************************************************************************/
   /* Check that the addressing modes match: 1) both the                    */
   /* same or 2) either UNIVERSAL.                                          */
   /*************************************************************************/

   /**END CH02 DEC MTS #1528 ************************************************/


   if ((prgpcOwner->GPC_usMode != GPC_UNIVERSAL) && (prgpcUser->GPC_usMode !=
      GPC_UNIVERSAL) && (prgpcUser->GPC_usMode != prgpcOwner->GPC_usMode))
      return (DSP_INV_GPC_MODE);

   /*************************************************************************/
   /* Validate the protocol specifications and their combinations           */
   /*************************************************************************/

   switch (prgpcOwner->GPC_usProtocol) /* allow only valid protocols        */
      {
      case (GPC_SYNC) :                /* SYNC owner can connect to SYNC or */
                                       /* ODD user                          */
         if ((prgpcUser->GPC_usProtocol != GPC_SYNC) &&
            (prgpcUser->GPC_usProtocol != GPC_OWNERDD))
            return (DSP_INV_GPC_PROTOCOL);
         break;
      case (GPC_SAFEDD) :              /* SAFE owner can connect to SAFE or */
                                       /* ODD user                          */
         if ((prgpcUser->GPC_usProtocol != GPC_SAFEDD) &&
            (prgpcUser->GPC_usProtocol != GPC_OWNERDD))
            return (DSP_INV_GPC_PROTOCOL);
         break;
      case (GPC_OWNERDD) :             /* ODD owner can connect to ODD user */
         if (prgpcUser->GPC_usProtocol != GPC_OWNERDD)
            return (DSP_INV_GPC_PROTOCOL);
         break;
      case (GPC_USERDD) :              /* UDD owner can connect to all users*/
         if ((prgpcUser->GPC_usProtocol != GPC_SYNC) &&
            (prgpcUser->GPC_usProtocol != GPC_OWNERDD) &&
            (prgpcUser->GPC_usProtocol != GPC_SAFEDD) &&
            (prgpcUser->GPC_usProtocol != GPC_USERDD))
            return (DSP_INV_GPC_PROTOCOL);
         break;
      default  :
         return (DSP_INV_GPC_PROTOCOL);
   }

   /* For any combination, SAMPRATEs must be equal or at least one must     */
   /* be zero.                                                              */

   if ((prgpcOwner->GPC_ulSamplerate != 0) && (prgpcUser->GPC_ulSamplerate !=
      0) && (prgpcOwner->GPC_ulSamplerate != prgpcUser->GPC_ulSamplerate))
      return (DSP_GPC_SAMPLE_RATES);

   /* IF both User and Owner are SYNC, the strides must be equal CH01 DEC   */

   if ((prgpcOwner->GPC_usProtocol == GPC_SYNC) &&
       (prgpcUser->GPC_usProtocol == GPC_SYNC)) {
      if ((prgpcOwner->GPC_usIntegerStride != prgpcUser->GPC_usIntegerStride )
          || (prgpcOwner->GPC_usFractStride != prgpcUser->GPC_usFractStride ))
         return (DSP_INV_GPC_STRIDE);
   }

   /* compute dsp address of Owner GPC's Control Block                      */

   ulAddrOwnerCB = (prgpcOwner->GPC_prseg->SEG_prmem->MEM_ulSegAddr)+
      (prgpcOwner->GPC_ulOffset);

   /*************************************************************************/
   /* Process Virtual User GPC                                              */
   /*************************************************************************/

   if (prgpcUser->GPC_usflgKind == GPC_VIRTUSER) {

      if (prgpcUser->GPC_prvgpcTail != NULL)/* if not 1st connect           */

         /*Allocate VGPC cycles                                             */

         if ((ulRC = AllocateVGPC_CPS(prtskUser, prgpcUser)) != DSP_NOERROR)
            return (ulRC);

      /**********************************************************************/
      /* If the GPC is a Virtual GPC with a USER-DEFINED                    */
      /* Area, and the API issuer wants the Address of the                  */
      /* Area, CreateVGPC will return the DSP Address of the                */
      /* USER-DEFINED Area to the API issuer via pulDSPAddr.                */
      /*                                                                    */
      /* CreateVGPC will return prgpcUser for the VGPC it                   */
      /* creates.                                                           */
      /**********************************************************************/

      if ((ulRC = CreateVGPC(&prgpcUser, prgpcOwner,
         &ulAddrUserCB, pulDSPAddr)) != DSP_NOERROR)
         return (ulRC);
   }

   else {
      ulAddrUserCB = (prgpcUser->GPC_prseg->SEG_prmem->MEM_ulSegAddr)+
         (prgpcUser->GPC_ulOffset);

      /**********************************************************************/
      /* If the Issuer Wants the address but the GPC is not a               */
      /* VGPC, Return 0.                                                    */
      /**********************************************************************/

      if (pulDSPAddr != NULL)
         *pulDSPAddr = 0;
   }

   ulRC = GPCSync(prgpcOwner, prtskOwner->TSK_prfm, ulAddrOwnerCB,

      prgpcUser, prtskUser->TSK_prfm, ulAddrUserCB);
   prgpcOwner->GPC_usUseCount += 1;

   /* Add the User to the Owner's list of Users                             */

   (VOID)CLLInsTail((PVOID)prgpcUser,
      (PVOID *)&(prgpcUser->GPC_prgpcUserListNext), (PVOID *)
      &(prgpcOwner->GPC_prgpcUserListNext->GPC_prgpcUserListNext), (PVOID *)
      &(prgpcOwner->GPC_prgpcUserListNext));

   prgpcUser->GPC_prgpcOwner = prgpcOwner;/* do for all users now           */
  if ((prgpcUser->GPC_usflgKind == GPC_VIRTUSER) && (ulRC == DSP_NOERROR))

      {                                /* now write 'mute' flag             */
      usBuf = 0xFFFF;

      PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspgpc::ConnectGPC");
      PRINT_DEBUG(EMBEDFILEPTR, "  prtskOwner (0x%X)", prtskOwner);
      PRINT_DEBUG(EMBEDFILEPTR, "  pszOwner \"%s\"\n", pszOwner);
      PRINT_DEBUG(EMBEDFILEPTR, "  prtskUser (0x%X)", prtskUser);
      PRINT_DEBUG(EMBEDFILEPTR, "  pszUser \"%s\"", pszUser);
      PRINT_DEBUG(EMBEDFILEPTR, "  pulDSPAddr (0x%X)\n", pulDSPAddr);
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR, "WRITE42:  Zeros to DSP",
           EOF, (ULONG)1,
           prtskUser->TSK_prdsp, ulAddrUserCB-2, (ULONG)1, &usBuf);

      ulRC = DspBIOS_D_Write(prtskUser->TSK_prdsp,
         ulAddrUserCB-2,
         (ULONG)1, &usBuf);
   }
  MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspgpc::ConnectGPC exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: DisconnectGPC                                           */
/*                                                                          */
/* FUNCTION: This routines will disconnect two GPC.                         */
/*                                                                          */
/* INPUT:                                                                   */
/*      HTASK  hOwnerTask - Handle of Owner Task for GPC.                   */
/*      PSZ   pszOwner   - Name of Owner GPC.                               */
/*      HTASK  hUserTask  - Handle of User Task for GPC.                    */
/*      PSZ   pszUser    - Name of User GPC.                                */
/*      USHORT usInhibit - Indicate whether Freeing of BIOS tasks is allowed*/
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC DisconnectGPC(PRTSK prtskOwner,PSZ pszOwner,PRTSK prtskUser,PSZ pszUser,
                  USHORT usInhibit)

{
   RC         ulRC=DSP_NOERROR;        // Error return code. CH02 Initialized.
   PRGPC      prgpcOwner;              // Pointer to owner GPC
   PRGPC      prgpcUser;               // Pointer to User GPC
   PRGPC      prgpcNode;               // General Pointer to a GPC Node
   PRGPC      prgpcPrev=0;               // General Pointer to a GPC Node
   PRGPC      prvgpcRoot=0;
   PRBIOT     prbiot;
   RC         ulRC_user = 0;           // Freed User Task RC
   RC         ulRC_owner = 0;          // Freed Owner Task RC
   PRDSP      prdsp;                   // CH03 DSP that GPC is on
   ULONG      ulAddrCB;                // CH03 Temp addr for control block
   USHORT     usAddrCB;                // CH03 Temp addr for control block
   BOOL       bFound;

   MW_SYSLOG_5(TRACE_MANAGER_CORE,"dspgpc::DisconnectGPC entry prtskOwner %x pszOwner %s prtskUser %x pszUser %s\n",
	   (int)prtskOwner,pszOwner,(int)prtskUser,pszUser);
   /************************************************************************/
   /*  CH01 If either prtsk is not NULL it better be valid!  MTS #2675     */
   /************************************************************************/

   if ((prtskUser!=NULL) && (prtskUser->TSK_achValid!=TSKVALIDATE))
      return (DSP_INV_HANDLE); /* CH03 Chk before using MTS #2675 */

   if ((prtskOwner!=NULL) && (prtskOwner->TSK_achValid != TSKVALIDATE))
      return (DSP_INV_HANDLE); /* CH03 Chk before using MTS #2675 */


   /************************************************************************
    * Check the pointers for the Tasks for validity.                       *
    * If either of the Tasks comes in set to NULL,                         *
    * this must be a disconnect for a BIOS task.                           *
    * Find the BIOS Task that is used in the disconnect.                   *
    *                                                                      *
    * If both User and Owner handles are null, ie. BIOS tasks, then return *
    * DSP_INV_HANDLE because we don't know which DSP they are loaded on.   *
    * This also prevents a trap which occurred when attempting to get the  *
    * DSP_prmodOS with a null prtskUser.                                   *
    ************************************************************************/

   if (prtskOwner == NULL) {
      if (prtskUser == NULL)
         ulRC = DSP_INV_HANDLE;
      else {
         ulRC = ConnectorToBIOS(pszOwner, pg->prmgrTail, &prbiot);
         if (ulRC == DSP_NOERROR) {
            ulRC = NameToTaskHandle(prtskUser->TSK_prdsp->DSP_prmodOS,
                                    prbiot->BT_pszTaskname,
                                    &prtskOwner);  /* CH02 */
            if (ulRC)
	      MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspgpc::DisconnectGPC error ulRC %lx Bios Task: %s\n",ulRC, prbiot->BT_pszTaskname);
         }
      }
   }

   else if (prtskUser == NULL) { /* CH02 added "else" clause */
      ulRC = ConnectorToBIOS(pszUser, pg->prmgrTail, &prbiot);
      if (ulRC == DSP_NOERROR) {
         ulRC = NameToTaskHandle(prtskOwner->TSK_prdsp->DSP_prmodOS,
                                 prbiot->BT_pszTaskname,
                                 &prtskUser);      /* CH02 */
         if (ulRC)
	   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspgpc::DisconnectGPC error ulRC %lx Bios Task: %s\n",ulRC, prbiot->BT_pszTaskname);
      }
   }

   if (ulRC != DSP_NOERROR) /* CH02 Combined several tests into this one */
      return (ulRC);

   /*************************************************************************/
   /* Verify that there are Data Segments attached to the                   */
   /* tasks.                                                                */
   /*************************************************************************/

   if ((prtskOwner->TSK_prsegDTail == NULL) || (prtskUser->TSK_prsegDTail ==
      NULL))
      return (DSP_NO_DATA);

   /*************************************************************************/
   /* Find the GPCs within the Tasks. Search the GPCs until                 */
   /* a match is found or the entire list has been checked.                 */
   /* Start at Tail->Next and go thru Tail.                                 */
   /* If a match is found, bFound will be set to TRUE and                   */
   /* the loop will end with prgpcNode pointing to the                      */
   /* correct GPC.                                                          */
   /*                                                                       */
   /* Find the Owner GPC.                                                   */
   /*************************************************************************/

   if ((ulRC = FindGPC(prtskOwner, pszOwner, &prgpcOwner)) != DSP_NOERROR)
      return (ulRC);

   if (prgpcOwner->GPC_usflgKind != GPC_OWNER) {
      ulRC = DSP_INV_GPC_TYPE;
      MW_SYSLOG_4(TRACE_MANAGER_CORE,"dspgpc::DisconnectGPC error ulRC %lx Owner GPC: %s.%s\n",ulRC,
               prtskOwner->TSK_pszVirtName, pszOwner);
      return (ulRC);
   }

   if ((ulRC=CheckDSPStatus(prtskOwner->TSK_prdsp)))
      return (ulRC);  /* CH04 Check for card unplugged */

   /*************************************************************************/
   /* Find the User GPC.                                                    */
   /*************************************************************************/

   if ((ulRC = FindGPC(prtskUser, pszUser, &prgpcUser)) != DSP_NOERROR)
      return (ulRC);

   if ((prgpcUser->GPC_usflgKind&GPC_KINDMASK) != GPC_USER) {
      ulRC = DSP_INV_GPC_TYPE;
      MW_SYSLOG_4(TRACE_MANAGER_CORE,"dspgpc::DisconnectGPC error ulRC %lx User GPC: %s.%s\n",ulRC,
               prtskUser->TSK_pszVirtName, pszUser);
      return (ulRC);
   }

   /*************************************************************************/
   /* If User is a Virtual GPC, find the correct instance                   */
   /*************************************************************************/

   if (prgpcUser->GPC_usflgKind == GPC_VIRTUSER)
   {
      prvgpcRoot = prgpcUser;
      ulRC = FindVGPC(&prgpcUser, prgpcOwner);
   }
   if (ulRC != DSP_NOERROR)
   {                                   /* Virtual user not found ...they're*/
                                       /* not connected!                    */
     MW_SYSLOG_6(TRACE_MANAGER_CORE,"dspgpc::DisconnectGPC error ulRC %lx %s.%s <-> %s.%s\n",ulRC,
         prtskOwner->TSK_pszVirtName, pszOwner,
         prtskUser->TSK_pszVirtName, pszUser);
      return (ulRC);
   }

   /*************************************************************************/
   /* Remove the User from the Owners UserList and                          */
   /* decrement the UseCount.                                               */
   /*                                                                       */
   /* Find the Previous so you can delete the User from ...UserList         */
   /*************************************************************************/

   bFound = FALSE;
   prgpcNode = prgpcOwner->GPC_prgpcUserListNext;/* Tail of List            */
   if (prgpcNode != NULL)
      do {
         prgpcPrev = prgpcNode;
         prgpcNode = prgpcNode->GPC_prgpcUserListNext;/* Current Node       */
         if (prgpcNode == prgpcUser) {
            bFound = TRUE;
            break;
         }
      }  while (prgpcNode != prgpcOwner->GPC_prgpcUserListNext);

   if (bFound != TRUE) {               /* user not found on owner's         */
                                       /* list...they're not connected!    */
      ulRC = DSP_GPC_NOTCONNECTED;
      MW_SYSLOG_6(TRACE_MANAGER_CORE,"dspgpc::DisconnectGPC error ulRC %lx %s.%s <-> %s.%s\n",ulRC,
         prtskOwner->TSK_pszVirtName, pszOwner,
         prtskUser->TSK_pszVirtName, pszUser);
      return (ulRC);
   }

   /* they were connected, so it's OK to disconnect them                    */

   CLLDelNode((PVOID *)&(prgpcUser->GPC_prgpcUserListNext), (PVOID *)
      &(prgpcPrev->GPC_prgpcUserListNext), (PVOID)prgpcPrev, (PVOID *)
      &(prgpcOwner->GPC_prgpcUserListNext));

   prgpcOwner->GPC_usUseCount -= 1;

   /**CH03********************************************************/
   /* Set the GPC ADDR field to point to itself MTS #1956, #2867 */
   /* This is done only for data driven protocols.               */
   /**************************************************************/
   prdsp=prtskOwner->TSK_prdsp;  /* DSP that GPC is on */

   if (GPC_SYNC != prgpcOwner->GPC_usProtocol) {   /* if not SYNC */
      ulAddrCB = (prgpcOwner->GPC_prseg->SEG_prmem->MEM_ulSegAddr)+
         (prgpcOwner->GPC_ulOffset);           /* Addr of control block */
      usAddrCB = (USHORT) ulAddrCB;            /* Convert to 16 bits    */

      PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspgpc::DisconnectGPC");
      PRINT_DEBUG(EMBEDFILEPTR, "  prtskOwner (0x%X)", prtskOwner);
      PRINT_DEBUG(EMBEDFILEPTR, "  pszOwner \"%s\"\n", pszOwner);
      PRINT_DEBUG(EMBEDFILEPTR, "  prtskUser (0x%X)", prtskUser);
      PRINT_DEBUG(EMBEDFILEPTR, "  pszUser \"%s\"", pszUser);
      PRINT_DEBUG(EMBEDFILEPTR, "  usInhibit (0x%X)\n", usInhibit);
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
           "WRITE43:  Set GPC Addr to point to itself for data driven",
           EOF, (ULONG)1,
           prdsp, ulAddrCB+GPCCB_OTHERPTR, (ULONG)1 , &usAddrCB);

      if ((ulRC = DspBIOS_D_Write(prdsp, ulAddrCB+GPCCB_OTHERPTR,
         (ULONG)1 , &usAddrCB)) != DSP_NOERROR)
         return (ulRC);
   }  /* End CH03 MTS #1956, #2867 */

   /*************************************************************************/
   /* if user gpc is virtual, remove the virtual connection                 */
   /*  else delete owner from user data structure                           */
   /*************************************************************************/

   if (prgpcUser->GPC_usflgKind == GPC_VIRTUSER) {

      ulRC = RemoveVGPC(prgpcUser, prvgpcRoot);
      if (ulRC != DSP_NOERROR)
         return (ulRC);

      if (prvgpcRoot->GPC_prvgpcTail != NULL)  /* if not first connection */
         if ((ulRC = FreeVGPC_CPS(prtskUser, prvgpcRoot)) != DSP_NOERROR)
            return (ulRC); /* Free VGPC cycles */
   }

   else { /* Begin CH03 For normal GPCs only, Not VGPCs */
      prgpcUser->GPC_prgpcOwner = NULL;  /* invalidate connection */

      /**CH03********************************************************/
      /* Set the GPC ADDR field to point to itself MTS #1956, #2867 */
      /* This is done only for data driven protocols.               */
      /**************************************************************/
      if (GPC_SYNC != prgpcUser->GPC_usProtocol) {   /* if not SYNC */
         ulAddrCB = (prgpcUser->GPC_prseg->SEG_prmem->MEM_ulSegAddr)+
            (prgpcUser->GPC_ulOffset);            /* Addr of control block */
         usAddrCB = (USHORT) ulAddrCB;            /* Convert to 16 bits    */

         PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspgpc::DisconnectGPC");
         PRINT_DEBUG(EMBEDFILEPTR, "  prtskOwner (0x%X)", prtskOwner);
         PRINT_DEBUG(EMBEDFILEPTR, "  pszOwner \"%s\"\n", pszOwner);
         PRINT_DEBUG(EMBEDFILEPTR, "  prtskUser (0x%X)", prtskUser);
         PRINT_DEBUG(EMBEDFILEPTR, "  pszUser \"%s\"", pszUser);
         PRINT_DEBUG(EMBEDFILEPTR, "  usInhibit (0x%X)\n", usInhibit);
         PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
              "WRITE44:  Set GPC Addr to point to itself for data driven",
              EOF, (ULONG)1,
              prdsp, ulAddrCB+GPCCB_OTHERPTR, (ULONG)1 , &usAddrCB);

         if ((ulRC = DspBIOS_D_Write(prdsp, ulAddrCB+GPCCB_OTHERPTR,
            (ULONG)1 , &usAddrCB)) != DSP_NOERROR)
            return (ulRC);
      }  /* End CH03 MTS #1956, #2867 */
   }

   /*************************************************************************/
   /* Call Check Connections to check the connections                       */
   /* for the BIOS tasks.                                                   */
   /* If no connections are left, Free the BIOS Task.                       */
   /*************************************************************************/

   if (((prtskOwner->TSK_usflgType&TSK_BIOSMASK) == TSK_BIOS) && ((usInhibit
      &DONT_FREE_OWNER_BIOS) == 0))
      {
      ulRC = CheckConnections(prtskOwner);
      if (ulRC == DSP_NO_CONNECTIONS) {
         ulRC = FreeTask(prtskOwner);
         if (ulRC == DSP_NOERROR)
            ulRC_owner = DSP_OWNER_TASK_FREE;
         else
            return (ulRC);
      }                                /* endif                             */
   }
   if (((prtskUser->TSK_usflgType&TSK_BIOSMASK) == TSK_BIOS) && ((usInhibit
      &DONT_FREE_USER_BIOS) == 0))
      {
      ulRC = CheckConnections(prtskUser);
      if (ulRC == DSP_NO_CONNECTIONS) {
         ulRC = FreeTask(prtskUser);
         if (ulRC == DSP_NOERROR)
            ulRC_user = DSP_USER_TASK_FREE;
         else
            return (ulRC);
      }                                /* endif                             */
   }

   /*************************************************************************/
   /* removed the bitwise exclusive OR of ulRC_user with ulRC_owmer         */
   /* which was causing an erroneous ulRC of 3, "DSP_FILE_NOT_FOUND"        */
   /*************************************************************************/

   if (ulRC == DSP_NOERROR)
      {
      if (ulRC_user && ulRC_owner)
         ulRC = DSP_OW_US_TASK_FREE;
      else
         if (ulRC_user)

            ulRC = DSP_USER_TASK_FREE;
         else
            if (ulRC_owner)

               ulRC = DSP_OWNER_TASK_FREE;
   }


   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspgpc::DisconnectGPC exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: CheckConnections                                        */
/*                                                                          */
/* FUNCTION: This routine will check a BIOS task for connections            */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRTSK  prtsk - BIOS Task to check out.                            */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              DSP_NO_CONNECTIONS if no connections.       */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC CheckConnections(PRTSK prtsk)

{
   RC         ulRC = DSP_NOERROR;      // Assume DSP_NOERROR
   PRGPC      prgpc;                   // Pointer to User GPC
   PRITCB     pritcb;                  // Pointer to User GPC
   BOOL       bfound = FALSE;          // Assume FALSE

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspgpc::CheckConnections entry prtsk %x\n",(int)prtsk);
   /********************************************************
    * Verify the task is OK.
    *******************************************************                 */

   if (prtsk == NULL)
      return (DSP_INV_PARAMETER);

   /********************************************************
    * Check the GPC list for the task.
    * If its an Owner GPC, checks its User list ptr.
    * If its a Virtual User check its VGPC list.
    * Defualt is a User GPC, checks its Owner list ptr.
    * If a connection exists, set ulRC to DSP_NOERROR.
    *******************************************************                 */

   if (prtsk->TSK_prgpcTail != NULL) {
      prgpc = prtsk->TSK_prgpcTail;
      do {
         switch (prgpc->GPC_usflgKind) {
            case  GPC_OWNER :
               if (prgpc->GPC_prgpcUserListNext == NULL)
                  ulRC = DSP_NO_CONNECTIONS;
               else
                  bfound = TRUE;
               break;

            case  GPC_VIRTUSER :
               if (prgpc->GPC_prvgpcTail == NULL)
                  ulRC = DSP_NO_CONNECTIONS;
               else
                  bfound = TRUE;

               break;
            default  :
               if (prgpc->GPC_prgpcOwner == NULL)
                  ulRC = DSP_NO_CONNECTIONS;
               else
                  bfound = TRUE;
               break;
         }                             /* endswitch                         */

         prgpc = prgpc->GPC_prgpcNext;
      }  while ((prgpc != prtsk->TSK_prgpcTail) && (bfound == FALSE));
                                       /* enddo                             */
   }
   else
      ulRC = DSP_NO_CONNECTIONS;

   /********************************************************
    * If nothing has been found, check the ITCB list.
    *******************************************************                 */

   if (bfound == FALSE) {
      if (prtsk->TSK_pritcbTail != NULL) {
         pritcb = prtsk->TSK_pritcbTail;
         do {     /* CH01 Add KINDMASK to next line */
            if ((pritcb->ITCB_usflgKind&ITCB_KINDMASK) == ITCB_PRIMARY)
               if (pritcb->ITCB_pritcbSecListNext == NULL)
                  ulRC = DSP_NO_CONNECTIONS;
               else
                  bfound = TRUE;

            else /* Secondary ITCBs */
               if (pritcb->ITCB_pritcbPrimary == NULL)
                  ulRC = DSP_NO_CONNECTIONS;
               else
                  bfound = TRUE;

            pritcb = pritcb->ITCB_pritcbNext;
        }  while ((pritcb != prtsk->TSK_pritcbTail) && (bfound == FALSE));
                                       /* enddo                             */
      }
      else
         ulRC = DSP_NO_CONNECTIONS;
   }
   if (bfound == TRUE)
      ulRC = DSP_NOERROR;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspgpc::CheckConnections exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: CreateVGPC                                              */
/*                                                                          */
/* FUNCTION: This routine creates an instance of a virtual GPC.             */
/*           It allocates dsp data storage, creates PC data struc, links    */
/*           this instance to linked list, writes gpccb in dsp memory.      */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRGPC prgpcUser  - Ptr to user gpc data struc                     */
/*        PRGPC prgpcOwner - Ptr to owner gpc data struc                    */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC CreateVGPC(PRGPC *pprgpcUser,PRGPC prgpcOwner,PULONG pulAddrUserCB,PULONG
               pulDSPAddr)
{
   RC         ulRC = DSP_NOERROR;      // Code assumes initial condition
   PRGPC      prvgpcInst,prvgpcRoot;
   ULONG      ulAddrNextVGPC=0,offset;
   PRDSP      prdsp;
   RVGPCINS   *pImageBuf = NULL;
   USHORT     usStep;
   USHORT     usTemp;  /* CH01 */
   ULONG      baseplus_size=0;

#define  VGPCINS_NEXTVGPC FIELDOFFSET(RVGPCINS,VGPCINS_usNextVGPC)
   typedef    enum {
      ALLOC_STRUC,LINK_STRUC,ALLOC_IMAGE_BUF,ZERO_IMAGE_BUF,ALLOC_DSP_MEM,
         WRITE_GPC,INSERT_GPC
   }

   step_type;

   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspgpc::CreateVGPC entry\n");

   prvgpcRoot = *pprgpcUser;           /* set VGPC root                     */
   prdsp = prvgpcRoot->GPC_prseg->SEG_prtsk->TSK_prdsp;

   for (usStep = ALLOC_STRUC; usStep <= INSERT_GPC && ulRC == DSP_NOERROR;
      usStep++) {
      switch (usStep) {
         case  ALLOC_STRUC :           /* Allocate memory for vgpc data     */
                                       /* structure                         */
           ulRC = AllocatePerm((ULONG)sizeof(RGPC), (PVOID *)&prvgpcInst);

            *pprgpcUser = prvgpcInst;  /* give user new prgpc for shadow    */
            break;
         case  LINK_STRUC :            /* Add Virtual GPC Instance and link */
            memcpy((PVOID)prvgpcInst, (PVOID)prvgpcRoot, sizeof(RGPC));
            prvgpcInst->GPC_prgpcNext = prvgpcRoot->GPC_prgpcNext;
            prvgpcRoot->GPC_prgpcNext = prvgpcInst;

         /* save dsp address of previous vgpc tail's NextVGPC               */

            if (prvgpcRoot->GPC_prvgpcTail == NULL) {
               ulAddrNextVGPC =
                  prvgpcRoot->GPC_prseg->SEG_prmem->MEM_ulSegAddr+
                  prvgpcRoot->GPC_ulOffset;
            }
            else {
               ulAddrNextVGPC =
                  prvgpcRoot->GPC_prvgpcTail->GPC_prmemBuffer->MEM_ulSegAddr+
                  VGPCINS_NEXTVGPC;
            }
            prvgpcInst->GPC_prgpcOwner = prgpcOwner;
            ulRC = CLLInsTail((PVOID)prvgpcInst,
               (PVOID *)&prvgpcInst->GPC_prvgpcTail,
               (PVOID *)&(prvgpcRoot->GPC_prvgpcTail->GPC_prvgpcTail),

               (PVOID *)&prvgpcRoot->GPC_prvgpcTail);
            break;
         case  ALLOC_IMAGE_BUF :       /* Allocate pc mem for vgpc image    */
            ulRC = AllocateTemp(prvgpcRoot->GPC_ulVAllocSize << 1, 1, (PVOID
               *)&pImageBuf);
            break;
         case  ZERO_IMAGE_BUF :        /* Fill image buffer with zeros      */
            memset((PVOID *)pImageBuf, 0, ((size_t)
               (prvgpcRoot->GPC_ulVAllocSize))*2);
            break;
         case  ALLOC_DSP_MEM :         /* allocate dsp data memory for vgpc */
                                       /* instance                          */
            ulRC = AllocateMem(prdsp, prvgpcRoot->GPC_ulVAllocSize, MEM_DATA,
               1L, 0, &prvgpcInst->GPC_prmemBuffer);
            if (DSP_NOERROR == ulRC) {  /* CH01 DEC MTS #1578, Don't access */
                                        /* prvgpcInst->... if error occured */
               *pulAddrUserCB = (prvgpcInst->GPC_prmemBuffer->MEM_ulSegAddr)
                                 +FIELDOFFSET(RVGPCINS, VGPCINS_usPutGetPtr);

               baseplus_size = (prvgpcInst->GPC_prmemBuffer->MEM_ulSegAddr) + (prvgpcRoot->GPC_ulVAllocSize << 1);
            } /* endif */
            break;

/* write parameters to vgpc image buffer, then write image to data memory   */

         case  WRITE_GPC : {

      /* write GPC control block area of VGPC                               */

               pImageBuf->VGPCINS_usPutGetPtr = 0xFFFF;
               offset = 2;
            }

      /* write if ( MinimumSize) or !SYNCH                                  */

            if ((prvgpcRoot->GPC_lBufferSize < 0L) ||
               (prvgpcRoot->GPC_usProtocol != GPC_SYNC)) {
               pImageBuf->VGPCINS_usSize = (USHORT)labs
                  (prvgpcRoot->GPC_lBufferSize);
               if (prvgpcRoot->GPC_usAddrMode == GPC_BYTE)
                  pImageBuf->VGPCINS_usSize <<= 1;/* 2x if byte             */
               offset = 4;
            }

      /* write if  !sync protolcol                                          */

            if (prvgpcRoot->GPC_usProtocol != GPC_SYNC) {
               pImageBuf->VGPCINS_usMaxWPF = prvgpcRoot->GPC_usMaxWPF;
               pImageBuf->VGPCINS_usPtrOtherPtr = (USHORT)*pulAddrUserCB;
               offset = 8;
            }

            if (pulDSPAddr != NULL) {
               if ( (*pulAddrUserCB + offset) == baseplus_size )
                  *pulDSPAddr = 0;                          /* no user data return 0 */
               else
                  *pulDSPAddr = *pulAddrUserCB + offset;    /* pointer to data */
	    }

            PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspgpc::CreateVGPC");
            PRINT_DEBUG(EMBEDFILEPTR, "  *pprgpcUser (0x%X)", *pprgpcUser);
            PRINT_DEBUG(EMBEDFILEPTR, "  prgpcOwner (0x%X)\n", prgpcOwner);
            PRINT_DEBUG(EMBEDFILEPTR, "  pulAddrUserCB (0x%X)", pulAddrUserCB);
            PRINT_DEBUG(EMBEDFILEPTR, "  pulDSPAddr (0x%X)\n", pulDSPAddr);
            PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
                 "WRITE45:  write GPC control block area of VGPC",
                 EOF, prvgpcRoot->GPC_ulVAllocSize,
                 prdsp, prvgpcInst->GPC_prmemBuffer->MEM_ulSegAddr,
                 prvgpcRoot->GPC_ulVAllocSize, pImageBuf);

            ulRC = DspBIOS_D_Write(prdsp,
               prvgpcInst->GPC_prmemBuffer->MEM_ulSegAddr,
               prvgpcRoot->GPC_ulVAllocSize, pImageBuf);
            break;
         case  INSERT_GPC :            /* insert vgpc at end-of-list (in dsp*/
                                       /* memory)                           */
            /* Begin CH02: Writes of size 1 must use a USHORT for AIX */
            usTemp = (USHORT) prvgpcInst->GPC_prmemBuffer->MEM_ulSegAddr;

            PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspgpc::CreateVGPC");
            PRINT_DEBUG(EMBEDFILEPTR, "  *pprgpcUser (0x%X)", *pprgpcUser);
            PRINT_DEBUG(EMBEDFILEPTR, "  prgpcOwner (0x%X)\n",prgpcOwner);
            PRINT_DEBUG(EMBEDFILEPTR, "  pulAddrUserCB (0x%X)", pulAddrUserCB);
            PRINT_DEBUG(EMBEDFILEPTR, "  pulDSPAddr (0x%X)\n", pulDSPAddr);
            PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
                 "WRITE46:  insert vgpc at end-of-list (in dsp memory)",
                 EOF, (ULONG)1, prdsp, ulAddrNextVGPC, (ULONG)1, &usTemp);

            ulRC = DspBIOS_D_Write(prdsp, ulAddrNextVGPC, (ULONG)1,
               &usTemp);
            break;
      }                                /* end switch                        */
   }                                   /* endfor                            */
   if (pImageBuf != NULL)
      FreeTemp((PVOID)pImageBuf, prvgpcRoot->GPC_ulVAllocSize << 1);

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspgpc::CreateVGPC exit ulRc %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveVGPC                                              */
/*                                                                          */
/* FUNCTION: This routine removes an instance of a virtual GPC.             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRGPC prgpcUser  - Ptr to user gpc data struc                     */
/*        PRGPC prgpcOwner - Ptr to owner gpc data struc                    */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveVGPC(PRGPC prvgpcInst,PRGPC prvgpcRoot)
{
   RC         ulRC=DSP_NOERROR;        /* Initialization assumed by code    */
   RC         ulRCtemp=DSP_NOERROR;    /* Init assumed by code         CH01 */
   PRDSP      prdsp = prvgpcRoot->GPC_prseg->SEG_prtsk->TSK_prdsp;
   PRGPC      prvgpcNode=0,prvgpcPrev=0;
   ULONG      ulDspAddrNextVGPC;       /* DSP address of next VGPC          */
   USHORT     usNextVGPC;
   USHORT     usStep;
   BOOL       bFound;

   typedef    enum {
      FIND_STRUC,REMOVE_VGPC,WAIT_FOR_NRT,FREE_DSP_MEM,DELETE_N_FREE_STRUC
   }
   step_type;

   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspgpc::RemoveVGPC entry\n");

   for (usStep = FIND_STRUC; usStep <= DELETE_N_FREE_STRUC && ulRC ==
      DSP_NOERROR; usStep++) {
      switch (usStep) {
         case  FIND_STRUC :            /* find Instance for this vgpc       */
                                       /* connection                        */
            bFound = FALSE;
            prvgpcNode = prvgpcRoot->GPC_prvgpcTail;/* Tail of List         */
            do {                       /*  NEXT                           */
               prvgpcPrev = prvgpcNode;/* V                                 */
               prvgpcNode = prvgpcNode->GPC_prvgpcTail;/* Current Node      */
               if (prvgpcNode == prvgpcInst) {
                  bFound = TRUE;
                  break;
               }
            }  while (prvgpcNode != prvgpcRoot->GPC_prvgpcTail);
            if (bFound != TRUE)
               ulRC = DSP_GPC_NOTCONNECTED;
            break;
         case  REMOVE_VGPC :           /* remove vgpc from task's list (in  */
                                       /* dsp memory)                       */
            if (prvgpcPrev == prvgpcRoot->GPC_prvgpcTail)
               ulDspAddrNextVGPC =
                  prvgpcRoot->GPC_prseg->SEG_prmem->MEM_ulSegAddr+
                  prvgpcRoot->GPC_ulOffset;
            else
               ulDspAddrNextVGPC = prvgpcPrev->GPC_prmemBuffer->MEM_ulSegAddr

                  +VGPCINS_NEXTVGPC;
            if (prvgpcNode == prvgpcRoot->GPC_prvgpcTail)
               usNextVGPC = 0;         /*  NEXT                           */
            else                       /* V                                 */
               usNextVGPC = (USHORT)
                prvgpcNode->GPC_prvgpcTail->GPC_prmemBuffer->MEM_ulSegAddr/**/

                  +VGPCINS_NEXTVGPC;

            PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspgpc::RemoveVGPC");
            PRINT_DEBUG(EMBEDFILEPTR, "  prvgpcInst (0x%X)", prvgpcInst);
            PRINT_DEBUG(EMBEDFILEPTR, "  prvgpcRoot (0x%X)\n", prvgpcRoot);
            PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
                 "WRITE47:  remove vgpc from task's list (in dsp memory)",
                 EOF, (ULONG)1,
                 prdsp, ulDspAddrNextVGPC, (ULONG)1, &usNextVGPC);

            ulRC = DspBIOS_D_Write(prdsp, ulDspAddrNextVGPC, (ULONG)1,
               &usNextVGPC);
            break;

         case  WAIT_FOR_NRT :          /* wait for ISPOS to enter           */
                                       /* non-realtime mode                 */
            ulRCtemp = Wait_DONERT(prdsp); /* CH01 */
            break;

         case  FREE_DSP_MEM :          /* free dsp data memory for vgpc     */
                                       /* instance                          */
            ulRC = FreeMem(prdsp, prvgpcNode->GPC_prmemBuffer, MEM_DATA);
            break;

         case  DELETE_N_FREE_STRUC :

      /* delete data struc from Virtual GPC linked-list                     */

            CLLDelNode((PVOID *)&(prvgpcNode->GPC_prvgpcTail), (PVOID *)
               &(prvgpcPrev->GPC_prvgpcTail),
               (PVOID)prvgpcPrev,
               (PVOID *)&(prvgpcRoot->GPC_prvgpcTail));

      /* add delete data struc from task's GPC linked-list        */

            prvgpcNode = prvgpcRoot;
            do {                       /* Starting at prvgpcRoot, find the  */
                                       /* GPC previous to prvgpcInst        */
               prvgpcPrev = prvgpcNode;
               prvgpcNode = prvgpcNode->GPC_prgpcNext;
            }  while ((prvgpcNode != prvgpcInst) && (prvgpcNode != prvgpcRoot)
               );

            if (prvgpcNode != prvgpcInst)/* if search leads back to root    */
               return (DSP_INTERNAL_CORRUPT);/* then the list is corrupt    */
            prvgpcPrev->GPC_prgpcNext = prvgpcInst->GPC_prgpcNext;/* Un-Link*/

      /* free memory for vgpc data structure                                */

            ulRC = FreePerm((PVOID)prvgpcNode, (ULONG)sizeof(RGPC));
            break;
      }                                /* end switch                        */
   }                                   /* endfor                            */
   if (ulRC==DSP_NOERROR) {            /* CH01 Report RC from MWAVEOS       */
      ulRC=ulRCtemp;                   /* Return nonfatal RC from DONERT    */
   } /* endif */
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspgpc::RemoveVGPC exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FindGPC                                                 */
/*                                                                          */
/* FUNCTION: This routines will take a task and GPC name and return the     */
/*           GPC.                                                           */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRTSK prtsk      - Task                                           */
/*        PSZ   pszgpc     - Name of GPC                                    */
/*        PRGPC *pprgpc    - GPC node if found                              */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error or not found.                   */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC FindGPC(PRTSK prtsk,PSZ pszgpc,PRGPC *pprgpc)
{
   RC         ulRC = DSP_NOERROR;
   PRGPC      prgpc=0;
   BOOL       bFound;

   bFound = FALSE;

   /*************************************************************************/
   /* Traverse all GPC lists for the Task until                             */
   /* the GPC is found.  Always check the tail first to                     */
   /* avoid the VGPC shadows.                                               */
   /*************************************************************************/
   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspgpc::FindGPC entry prtsk  %x, pszgpc %s\n",(int)prtsk,pszgpc);

   if (prtsk->TSK_prgpcTail != NULL) {
      prgpc = prtsk->TSK_prgpcTail;
      do {

         if (mwstrcmpnull(pszgpc, prgpc->GPC_pszName) == 0)
            bFound = TRUE;
         else
            prgpc = prgpc->GPC_prgpcNext;
      }  while ((bFound == FALSE) && (prgpc != prtsk->TSK_prgpcTail));
                                       /* enddo                             */
   }                                   /* endif                             */
   if (bFound == TRUE)
      *pprgpc = prgpc;
   else {
      ulRC = DSP_NAME_NOT_FOUND;
      MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspgpc::FindGPC error %lx GPC: %s\n", ulRC,pszgpc);
   }

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspgpc::FindGPC exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FindVGPC                                                */
/*                                                                          */
/* FUNCTION: This routines will take a VGPC Root and an Owner GPC and       */
/*            return the the VGPC Instance.                                 */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRGPC prvgpcRoot  - VGPC Root                                     */
/*        PRGPC prgpcOwner  - GPC Owner                                     */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error or not found.                   */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC FindVGPC(PRGPC *pprvgpcRoot,PRGPC prgpcOwner)
{
   RC         ulRC = DSP_NOERROR;
   PRGPC      prvgpc,prvgpcTail;

   /*************************************************************************/
   /* Traverse the Roots VGPC list until the VGPC Instance                  */
   /* that has the given Owner is found.                                    */
   /*************************************************************************/

   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspgpc::FindVGPC entry\n");

   if ((*pprvgpcRoot)->GPC_prvgpcTail != NULL) {
      prvgpcTail = prvgpc = (*pprvgpcRoot)->GPC_prvgpcTail;
      do {
         prvgpc = prvgpc->GPC_prvgpcTail;

                                    /*  Next                           */

      }  while ((prvgpc->GPC_prgpcOwner != prgpcOwner) && (prvgpc !=
         prvgpcTail));

      if (prvgpc->GPC_prgpcOwner == prgpcOwner)
         *pprvgpcRoot = prvgpc;
      else
         ulRC = DSP_GPC_NOTCONNECTED;
   }                                   /* endif                             */
   else
      ulRC = DSP_GPC_NOTCONNECTED;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspgpc::FindVGPC exit ulRC %lx\n",ulRC);
   return (ulRC);
}


/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: GPCSYNC                                                 */
/*                                                                          */
/* FUNCTION: This fucntion contains the dsp manager code that is needed to  */
/*           initiate GPC synchronization.                                  */
/*                                                                          */
/* INPUT:  ptrs to owner and user GPC structures                            */
/*         ptrs to owner and user FMCB structures                           */
/*                                                                          */
/* OUTPUT:  error code                                                      */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC GPCSync(PRGPC prgpcOwner,PRFM prfmOwner,ULONG ulAddrOwnerCB,
           PRGPC prgpcUser,PRFM prfmUser,ULONG ulAddrUserCB)

{
   RC         ulRC;                    /* return code                       */
   PRDSP      prdsp;                   /* ptr to DSP                        */
   ULONG      ulAddrSyncCB;            /* dsp addr of sync control block    */
   ULONG      ulAddrGpcAct;            /* dsp addr of GPCact variable       */
   ULONG      ulGPCSCB_Size;           /* size of GPC Sync control block    */
   PVOID      pGPCSCB;                 /* ptr to GPC Sync CB structure      */
   RGPCTMP    rgpctmp;
   PRGPCTMP   prgpctmp;
   USHORT     usTemp;                  /* CH03  */
   SHORT      sBufferSize;             /* CH03 Chg buffersize long->short   */
   BOOL       fAtomicSync;             /* flag indicating new Sync          */


   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspgpc::GPCSync entry\n");
/* setup pointer to dsp struc...assumption: GPCs are on same DSP.           */

   prdsp = prgpcOwner->GPC_prseg->SEG_prtsk->TSK_prdsp;

/* Determine version of MwaveOS, then set Control block size and pointer    */
/* accordingly.  Version 3 and above support Atomic sync. CH04 MTS #3250    */

   if (prdsp->DSP_prmodOS->MOD_rver.bVersion > 2) {

      fAtomicSync = TRUE;
      ulGPCSCB_Size = (ULONG)(sizeof(rgpctmp)/2);
      pGPCSCB = &rgpctmp;

      /* get DSP address of GPC sync control block for atomic sync operation */

      if ((ulRC = IsposLabelToAddress(prdsp, GPCATOMICLABEL, &ulAddrSyncCB)) !=
         DSP_NOERROR)
         return (ulRC);

         ulAddrGpcAct = ulAddrSyncCB+FIELDOFFSET(RGPCTMP,usGpcAct);

   } else { /* support original sync where manager fixed up ptrs to ptrs    */

      fAtomicSync = FALSE;
      ulGPCSCB_Size = (ULONG)((sizeof(rgpctmp) -
                               FIELDOFFSET(RGPCTMP,usOwnerFM))/2);
      pGPCSCB = &rgpctmp.usOwnerFM;

      /* get DSP address of GPC sync control block                          */

      if ((ulRC = IsposLabelToAddress(prdsp, GPCLABEL, &ulAddrSyncCB)) !=
         DSP_NOERROR)
         return (ulRC);

      ulAddrGpcAct = ulAddrSyncCB + FIELDOFFSET(RGPCTMP,usGpcAct) -
                               FIELDOFFSET(RGPCTMP,usOwnerFM);
   } /* endif */

/* build GPC sync control block                                             */

   prgpctmp = &rgpctmp;

/* If fAtomicSync is TRUE:                                                  */
/*   copy the address of the user's ptr to owner ptr into the GPC Sync      */
/*   Control Block and vice versa.  If GPC protocol is GPC_SYNC, there is   */
/*   no valid address within the GPC control block to write to, so set      */
/*   the ptr to ptr address to point to itself such that MwaveOS will       */
/*   harmlessly overwrite this location when it attempts the pointer fixup  */
/* If fAtomicSync is FALSE:                                                 */
/*   write address of Owner GPC's put pointer to User GPC's control         */
/*   block and vice versa. do only if that GPC has that space allocated     */
/*   where space allocated == (!sync protolcol || multiple protocols)       */

   if ((prgpcUser->GPC_usProtocol^GPC_SYNC) != 0) {

      usTemp = (USHORT) ulAddrOwnerCB;   /* CH03 */

      if (fAtomicSync) {        /* CH04 MTS #3250 */
         prgpctmp->usPtrOwnerPtrAddr = (USHORT)(ulAddrUserCB+GPCCB_OTHERPTR);
      } else {
         PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspgpc::GPCSync");
         PRINT_DEBUG(EMBEDFILEPTR, "  prgpcOwner (0x%X)", prgpcOwner);
         PRINT_DEBUG(EMBEDFILEPTR, "  prfmOwner (0x%X)", prfmOwner);
         PRINT_DEBUG(EMBEDFILEPTR, "  ulAddrOwnerCB (0x%X)\n",ulAddrOwnerCB);
         PRINT_DEBUG(EMBEDFILEPTR, "  prgpcUser (0x%X)", prgpcUser);
         PRINT_DEBUG(EMBEDFILEPTR, "  prfmUser (0x%X)", prfmUser);
         PRINT_DEBUG(EMBEDFILEPTR, "  ulAddrUserCB (0x%X)\n", ulAddrUserCB);
         PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
              "WRITE48:  addr of Owner's GPC's put ptr to User GPC's cntl block",
              EOF, (ULONG)1,
              prdsp, ulAddrUserCB+GPCCB_OTHERPTR, (ULONG)1 , &usTemp);

         if ((ulRC = DspBIOS_D_Write(prdsp, ulAddrUserCB+GPCCB_OTHERPTR,
             (ULONG)1 , &usTemp)) != DSP_NOERROR)  /* CH03 */
            return (ulRC);
      } /* endif */

   } else { /* set address of ptrOwnerptr to itself (such that it will       */
            /* be overwritten) because MwaveOS needs a safe place to write   */
      prgpctmp->usPtrOwnerPtrAddr = ulAddrSyncCB +
                                    FIELDOFFSET(RGPCTMP,usPtrOwnerPtrAddr);
   } /* endif */

   if ((prgpcOwner->GPC_usProtocol^GPC_SYNC) != 0) {

      usTemp = (USHORT) ulAddrUserCB;   /* CH03 */

      if (fAtomicSync) {        /* CH04 MTS #3250 */
         prgpctmp->usPtrUserPtrAddr = (USHORT)(ulAddrOwnerCB+GPCCB_OTHERPTR);
      } else {
         PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspgpc::GPCSync");
         PRINT_DEBUG(EMBEDFILEPTR, "  prgpcOwner (0x%X)", prgpcOwner);
         PRINT_DEBUG(EMBEDFILEPTR, "  prfmOwner (0x%X)", prfmOwner);
         PRINT_DEBUG(EMBEDFILEPTR, "  ulAddrOwnerCB (0x%X)\n", ulAddrOwnerCB);
         PRINT_DEBUG(EMBEDFILEPTR, "  prgpcUser (0x%X)", prgpcUser);
         PRINT_DEBUG(EMBEDFILEPTR, "  prfmUser (0x%X)", prfmUser);
         PRINT_DEBUG(EMBEDFILEPTR, "  ulAddrUserCB (0x%X)\n", ulAddrUserCB);
         PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
              "WRITE49:  addr of User's GPC's put ptr to Owner GPC's cntl block",
              EOF, (ULONG)1,
              prdsp, ulAddrOwnerCB+GPCCB_OTHERPTR, (ULONG)1 , &usTemp);

         if ((ulRC = DspBIOS_D_Write(prdsp, ulAddrOwnerCB+GPCCB_OTHERPTR, (ULONG)
            1, &usTemp)) != DSP_NOERROR) /* CH03 */
            return (ulRC);
      } /* endif */

   } else { /* set address of ptrUserptr to itself (such that it will        */
            /* be overwritten) because MwaveOS needs a safe place to write   */
      prgpctmp->usPtrUserPtrAddr = ulAddrSyncCB +
                                   FIELDOFFSET(RGPCTMP,usPtrUserPtrAddr);
   } /* endif */

   /*************************************************************************/
   /* If User gpc has specified MINSIZE, then write owner buffer size       */
   /* (2x if byte addressing) to user gpc control block                     */
   /*************************************************************************/

   if (prgpcUser->GPC_lBufferSize < 0L)
      {
      sBufferSize = prgpcOwner->GPC_lBufferSize;  /* CH03 */
      if (prgpcOwner->GPC_usAddrMode == GPC_BYTE)
         sBufferSize <<= 1;

      PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspgpc::GPCSync");
      PRINT_DEBUG(EMBEDFILEPTR, "  prgpcOwner (0x%X)", prgpcOwner);
      PRINT_DEBUG(EMBEDFILEPTR, "  prfmOwner (0x%X)", prfmOwner);
      PRINT_DEBUG(EMBEDFILEPTR, "  ulAddrOwnerCB (0x%X)\n", ulAddrOwnerCB);
      PRINT_DEBUG(EMBEDFILEPTR, "  prgpcUser (0x%X)", prgpcUser);
      PRINT_DEBUG(EMBEDFILEPTR, "  prfmUser (0x%X)", prfmUser);
      PRINT_DEBUG(EMBEDFILEPTR, "  ulAddrUserCB (0x%X)\n", ulAddrUserCB);
      PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
           "WRITE50:  owner buffer size to user GPC's cntl block if MINSIZE",
           EOF, (ULONG)1,
           prdsp, ulAddrUserCB + GPCCB_MINSIZE, (ULONG)1, &sBufferSize);

      if ((ulRC = DspBIOS_D_Write(prdsp, ulAddrUserCB
         +GPCCB_MINSIZE, (ULONG)1,
         &sBufferSize)) != DSP_NOERROR)
         return (ulRC);
   }

/* write the addresses of the owner & user's frame manager control block    */

   if (prfmOwner != NULL)
      prgpctmp->usOwnerFM = (USHORT)(prfmOwner->FM_prmemFMCB->MEM_ulSegAddr);
   if (prfmUser != NULL)
      prgpctmp->usUserFM = (USHORT)(prfmUser->FM_prmemFMCB->MEM_ulSegAddr);

/* write Maximum of minimum pointer separation and Stride                   */

   prgpctmp->usMaxPtrSep = MAX(prgpcOwner->GPC_usMinPtrSep,
      prgpcUser->GPC_usMinPtrSep);

/* fill in modulo mask based on size of GPC buffer                          */
/* modmask = (BufferSize * 2) - 1 BufferSize must be positive number !      */

   prgpctmp->usModMask = (((USHORT)labs(prgpcOwner->GPC_lBufferSize))*(USHORT)
      2)-(USHORT)1;

/* fill in dsp address of Owner & User GPCs' Control Blocks                 */

   prgpctmp->usOwnPtrAddr = (USHORT)ulAddrOwnerCB;
   prgpctmp->usUserPtrAddr = (USHORT)ulAddrUserCB;

/* fill in owner and user addressing modes                                  */

   prgpctmp->usOwnAddMode = prgpcOwner->GPC_usAddrMode;
   prgpctmp->usUserAddMode = prgpcUser->GPC_usAddrMode;

/****************************************************************************/
/* Write stride value and set activate flag:    Begin CH01 DEC              */
/****************************************************************************/
/*  For NON-REAL-TIME tasks:                                                */
/*   If either Task is NRT, The GPCACT should get set to -1 to signal that  */
/*   an NRT task is being synched and stride should be set to 0.            */
/*  For REAL-TIME tasks:                                                    */
/*   GPCACT should get set to 1 and...                                      */
/*   If interrupt sources are not the same, set stride to zero              */
/*   Otherwise:                                                             */
/*   If Owner is GPC_SYNC and User is GPC_ODD, let user define stride       */
/*   If User is GPC_SYNC and Owner is GPC_UDD, let Owner define stride      */
/*   If Owner is GPC_SYNC and User is GPC_SYNC, User and Owner strides have */
/*    been verified to be equal (in ConnectGPC), it does not matter which   */
/*    one defines stride. For coding simplicity, let User define stride     */
/*   Else assume strides are equal, default to Owner defined stride         */
/****************************************************************************/

   if ((prfmOwner == NULL) || (prfmUser == NULL)) {
      prgpctmp->usIntegerStride = 0;   /* Reset Stride                      */
      prgpctmp->usFractStride = 0;     /* Reset Stride                      */
      prgpctmp->usGpcAct = -1;         /* NRT Synching                      */
   }
   else {
      prgpctmp->usGpcAct = 1;
      if (prfmOwner->FM_hwid == prfmUser->FM_hwid) {
         if (prgpcOwner->GPC_usProtocol == GPC_SYNC) {
            prgpctmp->usIntegerStride = prgpcUser->GPC_usIntegerStride;
            prgpctmp->usFractStride = prgpcUser->GPC_usFractStride;
         } else {
            prgpctmp->usIntegerStride = prgpcOwner->GPC_usIntegerStride;
            prgpctmp->usFractStride = prgpcOwner->GPC_usFractStride;
         } /* endif */
      } else {
         prgpctmp->usIntegerStride = 0;
         prgpctmp->usFractStride = 0;
      } /* endif */
   } /* endif */
/****************************************************************************/
/*  END CH01 DEC                                                            */
/****************************************************************************/

/* write GPC template into DSP memory                                       */

   PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspgpc::GPCSync");
   PRINT_DEBUG(EMBEDFILEPTR, "  prgpcOwner (0x%X)", prgpcOwner);
   PRINT_DEBUG(EMBEDFILEPTR, "  prfmOwner (0x%X)", prfmOwner);
   PRINT_DEBUG(EMBEDFILEPTR, "  ulAddrOwnerCB (0x%X)\n", ulAddrOwnerCB);
   PRINT_DEBUG(EMBEDFILEPTR, "  prgpcUser (0x%X)", prgpcUser);
   PRINT_DEBUG(EMBEDFILEPTR, "  prfmUser (0x%X)", prfmUser);
   PRINT_DEBUG(EMBEDFILEPTR, "  ulAddrUserCB (0x%X)\n", ulAddrUserCB);
   PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
        "WRITE51:  GPC template into DSP memory",
        EOF, ulGPCSCB_Size,
        prdsp, ulAddrSyncCB, ulGPCSCB_Size, pGPCSCB);

   if ((ulRC = DspBIOS_D_Write(prdsp, ulAddrSyncCB, ulGPCSCB_Size,
      pGPCSCB)) != DSP_NOERROR)
      return (ulRC);

/****************************************************************************/
/* CH02  Wait/test for MWAVEOS to complete GPC synchronization:            */
/****************************************************************************/
   ulRC = WaitISPOSLocZero( prdsp,
            ulAddrGpcAct); /* CH02 */

   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspgpc::GPCSync exit\n");
   return (DSP_NOERROR);

}                                      /* end of GPCSYNC                    */
