/*
 *   mwaveapi.c -- Mwave Manager API entry points 
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "dspmgr.h"   
#include "dspmgri.h"
#include "mwavepub.h" 
#include "mwaveapi.h"
#include "dspbios.h"
#include "dspsrv.h"
#include "mwcload.h"
extern char *ini_file;
#define MW_USE_REG_FOR_INI_SETTINGS
//#include "ini2reg.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <pthread.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <port_types.h>


#define  RESBUFLEN  128

HANDLE hDrv=0;       
pthread_mutex_t g_hIPCMutex;  
pthread_mutex_t g_hSerializeMgrEntry;

#define MWAPI_MaxIPCs       16            // Maximum number of IPCs we presently support    
BOOL g_bIPCSupportActive = FALSE;         // True if our IPC support layer is initialized and active
unsigned g_uIPCsInUse;                    // One bit per IPC in use for this subsystem
BOOL g_abCancellingIPCs[MWAPI_MaxIPCs];   // TRUE if we're attempting to cancel an IPC thread
HANDLE g_ahIPCThreads[MWAPI_MaxIPCs];     // Thread handle of IPC thread
PFN g_pfnIPCNotice[MWAPI_MaxIPCs];

BOOL   bDspDisabled = FALSE;

BYTE   pRequestBuffer[REQUEST_BUFFER_SIZE];
BYTE   pAnswerBuffer[ANSWER_BUFFER_SIZE];

void dspIPCThread(int uIPCNum);
PSZ   DspFindFile(PSZ pszFileName,PSZ ResultBuffer,int iSizeBuf);
ULONG SendToServer( ULONG ulCode, PVOID prParams, DWORD dwParamSize,
                    PSZ pszBuf1, PSZ pszBuf2, PSZ pszBuf3,
                    PDWORD pRetHandle, PBYTE pRetBuffer, PUSHORT pBufSize );
BOOL InitializeIPCSupport( void );
void ShutdownIPCSupport( void );
void DestroyAllIPCConnections( void );

/****************************************************************************/
/*                                                                          */
/* Function Name: : MwaveAPIInit                                             */
/*                                                                          */
/* FUNCTION: This function initializes the manager library.                 */
/*           It is also called on termination.                              */
/*                                                                          */
/* INPUT:   ulReason   - Why function was called.(TRUE=Start,FALSE=Terminate*/
/*                                                                          */
/* OUTPUT:  Return Codes:  FALSE (0) - Initialization failed                */
/*                         TRUE  (1) - Initialization successful            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/

BOOL MwaveAPIInit(DWORD ulReason)
{
  int rc;
  char mwave_device[32];
  
  switch(ulReason) {
    
  case TRUE:
    MW_SYSLOG_1(TRACE_MANAGER_API,"mwaveapi::MwaveAPIInit calling StartMwaveSubsystem\n");
    if ( !StartMwaveSubsystem() ) { 
      MW_SYSLOG_ERROR(LOG_ERR,"MwaveAPIInit - StartMwaveSubsystem failed\n");
      bDspDisabled = TRUE;
      return FALSE;
    } else {
      MW_SYSLOG_1(TRACE_MANAGER_API,"mwaveapi::MwaveAPIInit StartMwaveSubsystem OK\n");
    }

    GetPrivateProfileString("STARTUP", "DEVICE", "/dev/modems/mwave",
		mwave_device, sizeof(mwave_device), ini_file);
    hDrv=open(mwave_device,O_RDWR);
    if (hDrv == -1) {
      MW_SYSLOG_ERROR(LOG_ERR,"MwaveAPIInit - Open mwave device driver failed\n");
      bDspDisabled = TRUE;
      return FALSE;
    }
    MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::MwaveAPIInit opened %s hDrv %x\n",mwave_device,(int)hDrv);
    
    rc=pthread_mutex_init(&g_hIPCMutex,0);
    if (rc) {
      MW_SYSLOG_ERROR(LOG_ERR,"MwaveAPIInit - Create IPCSerializationMutex failed with error %d\n", errno);
      bDspDisabled = TRUE;
      return FALSE;
    }
    MW_SYSLOG_1(TRACE_MANAGER_API,"mwaveapi::MwaveAPIInit::Attach created IPC Mutex\n");
    
    rc=pthread_mutex_init(&g_hSerializeMgrEntry,0);
    if (rc) {
      MW_SYSLOG_ERROR(LOG_ERR,"MwaveAPIInit -  Create g_hSerializeMgrEntry failed with error %d\n",errno);
      bDspDisabled = TRUE;
      return FALSE;
    }        
    MW_SYSLOG_1(TRACE_MANAGER_API,"mwaveapi::MwaveAPIInit Attach created SerializeMgrEntry Mutex\n");
    
    MW_SYSLOG_1(TRACE_MANAGER_API,"mwaveapi::MwaveAPIInit Attach calling InitializeIPCSupport()\n");
    InitializeIPCSupport();
    MW_SYSLOG_1(TRACE_MANAGER_API,"mwaveapi::MwaveAPIInit Attach return from InitializeIPCSupport\n");
    break;
    
    
    
  case FALSE:
    MW_SYSLOG_1(TRACE_MANAGER_API,"mwaveapi::MwaveAPIInit, Detach calling ShutdownIPCSupport\n");
    ShutdownIPCSupport();
    
    if (hDrv) close(hDrv);
    
    pthread_mutex_destroy(&g_hIPCMutex);
    pthread_mutex_destroy(&g_hSerializeMgrEntry);
    break;
    
  default:
    break;
  }
  
  MW_SYSLOG_1(TRACE_MANAGER_API,"mwaveapi::MwaveAPIInit, exiting return TRUE\n");
  return TRUE;
}



BOOL InitializeIPCSupport( void )
{
  g_uIPCsInUse = 0;
  g_bIPCSupportActive = TRUE;
  return TRUE;
}

void ShutdownIPCSupport( void )
{
  if (g_bIPCSupportActive) {
    g_bIPCSupportActive = FALSE;
    DestroyAllIPCConnections();
  }
}


void DestroyAllIPCConnections( void )
{
    unsigned uIPCIndex;

    for (uIPCIndex = 0; uIPCIndex < MWAPI_MaxIPCs; uIPCIndex++) {
        if (g_uIPCsInUse & (1 << uIPCIndex)) {
	  g_abCancellingIPCs[uIPCIndex] = TRUE;
	  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::DestroyAllIPCConnections call DspBIOS_UnregisterIPCEvent usIpcNum %x\n",
		 uIPCIndex);
	  DspBIOS_UnregisterIPCEvent(hDrv, uIPCIndex);
        }
    }
}





/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspAbilities                                            */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue call to the      */
/*           routine.                                                       */
/*                                                                          */
/* INPUT:   hDSP    - Handle to DSP                                         */
/*          pusSize - Size of DSPInfoBuf (place for DSP Info)               */
/*          pszDSPInfoBuf - Buffer that gets info                           */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INS_BUFFER                                   */
/*                         DSP_INV_PARAMETER                                */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/****************************************************************************/

RC APIENTRY main_dspAbilities(HDSP hDSP,PUSHORT pusSize,PVOID pDSPQueryBuf)
{
   RC              ulRC;
   RPARM_ABILITIES rparm;

   if ( bDspDisabled ) return DSP_NOT_AVAILABLE;

   pthread_mutex_lock(&g_hSerializeMgrEntry);

   if (hDSP == NULL) {
     pthread_mutex_unlock(&g_hSerializeMgrEntry);     
     return DSP_INV_HANDLE;
   }

   rparm.ABILITIES_hDsp = hDSP;
   rparm.ABILITIES_pusSize = pusSize;
   rparm.ABILITIES_pDSPQueryBuf = pDSPQueryBuf;
   rparm.ABILITIES_ulRC = DSP_NOERROR;

   ulRC = SendToServer( DSPMDD_DspAbilities,         /* function      */
                        (PVOID)&rparm,               /* pParams       */
                        sizeof(RPARM_ABILITIES),     /* Params size   */
                        NULL,                        /* pg, buffer1   */
                        NULL,                        /* pg, buffer2   */
                        NULL,                        /* pg, buffer3   */
                        NULL,                        /* return Handle */
                        pDSPQueryBuf,                /* return buffer */
                        pusSize );                   /* buffer size   */
   
   pthread_mutex_unlock(&g_hSerializeMgrEntry);      
   
   return (ulRC);
}
RC APIENTRY dspAbilities(HDSP hDSP,PUSHORT pusSize,PVOID pDSPQueryBuf){
  RC rc;
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspAbilities entry hDSP %x , usSize %x\n",
	 (int)hDSP, *pusSize);
  rc = main_dspAbilities( hDSP, pusSize, pDSPQueryBuf );
  
  MW_SYSLOG_4(TRACE_MANAGER_API,"mwaveapi::dspAbilities exit rc %lx hDSP %x , usSize %x\n",
	 rc,(int)hDSP, *pusSize);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspAllocateSegment                                      */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:                                                                   */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/****************************************************************************/

RC APIENTRY main_dspAllocateSegment(HTASK hTask,PSZ pszInstanceName,ULONG ulSize,
                                    ULONG ulAlign,USHORT usFLAGS,ULONG ulDSPAddr,
                                    PHSEG phSeg)
{
   RC             ulRC;                        
   RPARM_ALLOCSEG rparm;

   if ( bDspDisabled ) return DSP_NOT_AVAILABLE;

   pthread_mutex_lock(&g_hSerializeMgrEntry);

   if (hTask == NULL) {
     pthread_mutex_unlock(&g_hSerializeMgrEntry);    
     return DSP_INV_HANDLE;
   }

   rparm.ALLOCSEG_hTask = hTask;
   rparm.ALLOCSEG_pszInstanceName = pszInstanceName;
   rparm.ALLOCSEG_ulSize = ulSize;
   rparm.ALLOCSEG_ulAlign = ulAlign;
   rparm.ALLOCSEG_usFlags = usFLAGS;
   rparm.ALLOCSEG_ulDSPAddr = ulDSPAddr;
   rparm.ALLOCSEG_phSeg = phSeg;
   rparm.ALLOCSEG_ulRC = DSP_NOERROR;
   
   ulRC = SendToServer( DSPMDD_DspAllocateSegment,   /* function      */
                        (PVOID)&rparm,               /* pParams       */
                        sizeof(RPARM_ALLOCSEG),      /* Params size   */
                        pszInstanceName,             /* pg, buffer1   */
                        NULL,                        /* pg, buffer2   */
                        NULL,                        /* pg, buffer3   */
                        (PULONG)phSeg,               /* return Handle */
                        NULL,                        /* return buffer */
                        (PUSHORT)0 );                         /* buffer size   */
   
   if (ulRC != DSP_NOERROR)
     *phSeg = (HSEG)NULL;

   pthread_mutex_unlock(&g_hSerializeMgrEntry);     

   return (ulRC);
}
RC APIENTRY dspAllocateSegment(HTASK hTask,PSZ pszInstanceName,ULONG ulSize,
			       ULONG ulAlign,USHORT usFLAGS,ULONG ulDSPAddr,
			       PHSEG phSeg)
{
  RC rc;
  MW_SYSLOG_6(TRACE_MANAGER_API,"mwaveapi::dspAllocateSegment entry hTask %x pszInstanceName %s ulSize %x ulAlign %x usFLAGS %x ulDSPAddr\n",
	 (int)hTask,pszInstanceName,(int)ulSize,(int)ulAlign,(int)ulDSPAddr);
  rc = main_dspAllocateSegment( hTask, pszInstanceName, ulSize, ulAlign, usFLAGS, ulDSPAddr, phSeg );
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspAllocateSegment exit rc %lx hSeg %x\n",rc,(int)*phSeg);
  return rc;
}


/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspChangeCPF                                            */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue call to the      */
/*           routine.                                                       */
/*                                                                          */
/* INPUT:   hTask   - Handle to Task                                        */
/*          lCPF    - The additional (or neg) value to change Cycles Per/F  */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/****************************************************************************/
RC APIENTRY main_dspChangeCPF(HTASK hTask,LONG lCPF)
{
  RC          ulRC;
  RPARM_CHCPF rparm;
    
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;
  
  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hTask == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);    
    return DSP_INV_HANDLE;
  }

  rparm.CHCPF_hTask = hTask;
  rparm.CHCPF_lCPF = lCPF;
  rparm.CHCPF_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspChangeCPF,         /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_CHCPF),         /* Params size   */
		       NULL,                        /* pg, buffer1   */
		       NULL,                        /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       NULL,                        /* return Handle */
		       NULL,                        /* return buffer */
		       0 );                         /* buffer size   */
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);      
  
  return (ulRC);
}
RC APIENTRY dspChangeCPF(HTASK hTask,LONG lCPF)
{
  RC rc;
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspChangeCPF entry hTask %x lCPF %x\n",
	 (int)hTask,(int)lCPF);
  rc = main_dspChangeCPF( hTask, lCPF );
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspChangeCPF exit rc %lx\n",rc);
  return rc;
}


/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspChangeModuleState                                    */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/****************************************************************************/
RC APIENTRY main_dspChangeModuleState(HMOD hMod,USHORT usFlags)
{
   RC               ulRC;                   
   RPARM_CHMODSTATE rparm;

   if ( bDspDisabled ) return DSP_NOT_AVAILABLE;

   pthread_mutex_lock(&g_hSerializeMgrEntry);

   if (hMod == NULL) {
     pthread_mutex_unlock(&g_hSerializeMgrEntry);    
     return DSP_INV_HANDLE;
   }
   
   rparm.CHMODSTATE_hMOD = hMod;
   rparm.CHMODSTATE_usFlags = usFlags;
   rparm.CHMODSTATE_ulRC = DSP_NOERROR;
   
   ulRC = SendToServer( DSPMDD_DspChangeModuleState, /* function      */
                        (PVOID)&rparm,               /* pParams       */
                        sizeof(RPARM_CHMODSTATE),    /* Params size   */
                        NULL,                        /* pg, buffer1   */
                        NULL,                        /* pg, buffer2   */
                        NULL,                        /* pg, buffer3   */
                        NULL,                        /* return Handle */
                        NULL,                        /* return buffer */
                        0 );                         /* buffer size   */
   
   pthread_mutex_unlock(&g_hSerializeMgrEntry);     

   return (ulRC);
}
RC APIENTRY dspChangeModuleState(HMOD hMod,USHORT usFlags)
{
  RC rc;
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspChangeModuleState entry hMod %x usFlags %x\n",
	 (int)hMod,usFlags);
  rc = main_dspChangeModuleState( hMod, usFlags );
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspChangeModuleState exit rc %lx\n",rc);
  return rc;
}


/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspChangeTaskState                                      */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/****************************************************************************/
RC APIENTRY main_dspChangeTaskState(HTASK hTask,USHORT usFlags)
{
  RC                ulRC;
  RPARM_CHTASKSTATE rparm;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;

  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hTask == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);    
    return DSP_INV_HANDLE;
  }
  
  rparm.CHTASKSTATE_hTask = hTask;
  rparm.CHTASKSTATE_usFlags = usFlags;
  rparm.CHTASKSTATE_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspChangeTaskState,   /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_CHTASKSTATE),   /* Params size   */
		       NULL,                        /* pg, buffer1   */
		       NULL,                        /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       NULL,                        /* return Handle */
		       NULL,                        /* return buffer */
		       0 );                         /* buffer size   */
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);      

  return (ulRC);
}
RC APIENTRY dspChangeTaskState(HTASK hTask,USHORT usFlags)
{
  RC rc;
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspChangeTaskState entry hTask %x usFlags %x\n",
	 (int)hTask,usFlags);
  rc = main_dspChangeTaskState( hTask, usFlags );
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspChangeTaskState exit rc %lx\n",rc);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspConnectGPC                                           */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/****************************************************************************/
RC APIENTRY main_dspConnectGPC(HTASK hOwnerTask,PSZ pszOwner,HTASK hUserTask,PSZ
                               pszUser,USHORT usProtocol,PULONG pulDSPAddr)
{
  RC            ulRC;
  RPARM_CONNGPC rparm;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;
  
  pthread_mutex_lock(&g_hSerializeMgrEntry);

  rparm.CONNGPC_hOwnerTask = hOwnerTask;
  rparm.CONNGPC_pszOwner = pszOwner;
  rparm.CONNGPC_hUserTask = hUserTask;
  rparm.CONNGPC_pszUser = pszUser;
  rparm.CONNGPC_usProtocol = usProtocol;
  rparm.CONNGPC_pulDSPAddr = pulDSPAddr;
  rparm.CONNGPC_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspConnectGPC,        /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_CONNGPC),       /* Params size   */
		       pszOwner,                    /* pg, buffer1   */
		       pszUser,                     /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       pulDSPAddr,                  /* return Handle */
		       NULL,                        /* return buffer */
		       (PUSHORT)0 );                /* buffer size   */
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);      
  
  return (ulRC);
}
RC APIENTRY dspConnectGPC(HTASK hOwnerTask,PSZ pszOwner,HTASK hUserTask,PSZ
			  pszUser,USHORT usProtocol,PULONG pulDSPAddr)
{
  RC rc;
  
  MW_SYSLOG_6(TRACE_MANAGER_API,"mwaveapi::dspConnectGPC entry hOwnerTask %x pszOwner %s hUserTask %x pszUser %s usProtocol %x \n",
	 (int)hOwnerTask,pszOwner,(int)hUserTask,pszUser,usProtocol);
  rc = main_dspConnectGPC( hOwnerTask, pszOwner, hUserTask, pszUser, usProtocol, pulDSPAddr );
  
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspConnectGPC exit rc %lx\n",rc);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspConnectIPC                                           */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/****************************************************************************/

RC APIENTRY main_dspConnectIPC(HTASK hTask,PFN pfnIPCNotice,PHIPC phIPC)
{
  RC            ulRC;
  RPARM_CONNIPC rparm;
  USHORT        usMask;
  USHORT        usMaskSize = sizeof(USHORT);
  pthread_t     ThreadID;

  UINT          usIpcNum;
  USHORT        usTempMask;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;

  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hTask == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);     
    return DSP_INV_HANDLE;
  }

  rparm.CONNIPC_hTask = hTask;
  rparm.CONNIPC_pfnIPCNotice = pfnIPCNotice;
  rparm.CONNIPC_phIPC = phIPC;
  rparm.CONNIPC_usType = IPCTE_RING3CALL;
  rparm.CONNIPC_ulRC = DSP_NOERROR;

  ulRC = SendToServer( DSPMDD_DspConnectIPC,        /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_CONNIPC),       /* Params size   */
		       NULL,                        /* pg, buffer1   */
		       NULL,                        /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       (PULONG)phIPC,               /* return Handle */
		       (PBYTE)&usMask,              /* return buffer */
		       (PUSHORT)&usMaskSize);                /* buffer size   */
  
  if (ulRC != DSP_NOERROR) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);     
    return ulRC;
  }

  /**********************************************/
  /* Find IPC index.                            */
  /**********************************************/
  usTempMask = 1;
  for (usIpcNum = 0; usIpcNum < MWAPI_MaxIPCs; usIpcNum++) {
    if (usTempMask == usMask)
      break;
    usTempMask<<=1;
  }
  
  if (usIpcNum > 15) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);    
    return DSP_INTERNAL_ERROR;
  }

  g_abCancellingIPCs[usIpcNum] = FALSE;
  g_pfnIPCNotice[usIpcNum]=pfnIPCNotice;


  ulRC=pthread_create(&ThreadID,0,(void *)dspIPCThread,(void *)usIpcNum);

  if (ulRC) {
    MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::main_dspConnectIPC error %x from pthread_create\n",errno);
    g_pfnIPCNotice[usIpcNum]=0;
    return DSP_INTERNAL_ERROR;
  }
  g_ahIPCThreads[usIpcNum]=ThreadID;
  g_uIPCsInUse |= (1 << usIpcNum);

  pthread_mutex_unlock(&g_hSerializeMgrEntry);      

  return (ulRC);
}
RC APIENTRY dspConnectIPC(HTASK hTask,PFN pfnIPCNotice,PHIPC phIPC)
{
  RC rc;
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspConnectIPC entry hTask %x pfnIPCNotice %x\n",
	 (int)hTask,(int)pfnIPCNotice);
  rc = main_dspConnectIPC( hTask, pfnIPCNotice, phIPC );
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspConnectIPC exit rc %lx hIPC %x\n",rc,(int)*phIPC);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspIPCThread                                            */
/*                                                                          */
/* FUNCTION: This routine will be a thread that gets created by             */
/*           dspConnectIPC.                                                 */
/*                                                                          */
/*           It will perform the callback to the Ring 3 App that does the   */
/*           IPCConnect.                                                    */
/* INPUT:                                                                   */
/*                                                                          */
/* OUTPUT:                                                                  */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/****************************************************************************/

void dspIPCThread(int usIpcNum)
{
  RC         ulRC;
  PFN        pfnCallBack;
  BOOL       bIPCEventRegistered;
  
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspIPCThread entry usIpcNum %x\n",usIpcNum);
  /* ulRC=setpriority(PRIO_PROCESS,0,-20);
  if (ulRC) {
    MW_SYSLOG_ERROR(LOG_ERR,"mwaveapi::dspIPCThread Unable to set thread priority\n");
  } */

  pfnCallBack=g_pfnIPCNotice[usIpcNum];
  bIPCEventRegistered = FALSE;

  if (pfnCallBack == 0) {
    MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspIPCThread DspBIOS_RegisterIPCEvent invalid pfnCallback %x\n",(int)pfnCallBack);
    goto cleanup;
  }

  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspIPCThread calling DspBIOS_RegisterIPCEvent hDrv %x usIpcNum %x\n",
	 (int)hDrv,usIpcNum);
//	 gtid(),(int)hDrv,usIpcNum);
  ulRC=DspBIOS_RegisterIPCEvent(hDrv, usIpcNum);

  if (ulRC != DSP_NOERROR) {
    MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspIPCThread DspBIOS_RegisterIPCEvent error %lx\n",ulRC);
    goto cleanup;
  }

  bIPCEventRegistered = TRUE;

  while (TRUE) {
    MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspIPCThread calling DspBIOS_GetIPCEvent usIpcNum %x\n", usIpcNum);
    ulRC = DspBIOS_GetIPCEvent(hDrv, usIpcNum);
    
    if (ulRC != DSP_NOERROR) {
      MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspIPCThread return from DspBIOS_GetIPCEvent error %x\n",errno);
      break;
    }

    if (g_abCancellingIPCs[usIpcNum] == TRUE) {
      bIPCEventRegistered=FALSE; /* whomever set the abort to true issued the unregister */
      MW_SYSLOG_1(TRACE_MANAGER_API,"mwaveapi::dspIPCThread DspBIOS_GetIPCEvent IPC cancelled\n");
      break;
    }

    if ((PVOID)(*pfnCallBack) != NULL) {
      MW_SYSLOG_1(TRACE_MANAGER_API,"mwaveapi::dspIPCThread locking global IPC Mutex\n");
      pthread_mutex_lock(&g_hIPCMutex);
      MW_SYSLOG_1(TRACE_MANAGER_API,"mwaveapi::dspIPCThread calling pfnCallback\n");
      (*pfnCallBack)();
      MW_SYSLOG_1(TRACE_MANAGER_API,"mwaveapi::dspIPCThread return from pfnCallback\n");
      pthread_mutex_unlock(&g_hIPCMutex);
    }
  }
  
 cleanup:
  if (bIPCEventRegistered) {
    MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspIPCThread call DspBIOS_UnregisterIPCEvent usIpcNum %x\n",
	   usIpcNum);
    ulRC=DspBIOS_UnregisterIPCEvent( hDrv, usIpcNum);
    if (ulRC != DSP_NOERROR) {
      MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspIPCThread return from DspBIOS_UnregisterIPCEvent error %x\n",errno);
    }
  }

  g_ahIPCThreads[usIpcNum]=0;  
  g_uIPCsInUse &= ~(1 << usIpcNum);
  g_pfnIPCNotice[usIpcNum]=0;
  g_abCancellingIPCs[usIpcNum]=FALSE;

  MW_SYSLOG_1(TRACE_MANAGER_API,"mwaveapi::dspIPCThread thread exiting\n");
  return;
}



/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspConnectITCB                                          */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
RC APIENTRY main_dspConnectITCB(HTASK hPrimaryTask,PSZ pszPrimary,HTASK
                                hSecondaryTask,PSZ pszSecondary)
{
  RC             ulRC;
  RPARM_CONNITCB rparm;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;

  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  rparm.CONNITCB_hPTask = hPrimaryTask;
  rparm.CONNITCB_pszPName = pszPrimary;
  rparm.CONNITCB_hSTask = hSecondaryTask;
  rparm.CONNITCB_pszSName = pszSecondary;
  rparm.CONNITCB_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspConnectITCB,       /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_CONNITCB),      /* Params size   */
		       pszPrimary,                  /* pg, buffer1   */
		       pszSecondary,                /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       NULL,                        /* return Handle */
		       NULL,                        /* return buffer */
		       0 );                         /* buffer size   */
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);     
  return (ulRC);
}
RC APIENTRY dspConnectITCB(HTASK hPrimaryTask,PSZ pszPrimary,HTASK
			   hSecondaryTask,PSZ pszSecondary)
{
  RC rc;
  MW_SYSLOG_5(TRACE_MANAGER_API,"mwaveapi::dspConnectITCB entry hPrimaryTask %x pszPrimary %s hSecondaryTask %x pszSecondary %s\n",
	 (int)hPrimaryTask,pszPrimary,(int)hSecondaryTask,pszSecondary);
  rc = main_dspConnectITCB( hPrimaryTask, pszPrimary, hSecondaryTask, pszSecondary );
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspConnectITCB exit rc %lx\n",rc);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspDisconnectIPC                                        */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
RC APIENTRY main_dspDisconnectIPC(HIPC hIPC)
{
  RC            ulRC;
  USHORT        usMask;
  USHORT        usMaskSize = sizeof(USHORT);
  RPARM_DISCIPC rparm;
  USHORT        usIpcNum,usTempMask;
    
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;
  
  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hIPC == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);      
    return DSP_INV_HANDLE;
  }

  rparm.DISCIPC_hIPC = hIPC;
  rparm.DISCIPC_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspDisconnectIPC,     /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_DISCIPC),       /* Params size   */
		       NULL,                        /* pg, buffer1   */
		       NULL,                        /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       NULL,                        /* return Handle */
		       (PBYTE)&usMask,              /* return buffer */
		       &usMaskSize);                /* buffer size   */
  
  /****************************************************************************/
  /*  Kill the Thread, table, etc.                                            */
  /****************************************************************************/
  if (ulRC == DSP_NOERROR) {
    usTempMask = 1;
    for (usIpcNum = 0; usIpcNum < MWAPI_MaxIPCs; usIpcNum++) {
      if (usTempMask == usMask)
	break;
      usTempMask<<=1;
    }
    
    if (usIpcNum > 15) {
      pthread_mutex_unlock(&g_hSerializeMgrEntry);  
      return DSP_INTERNAL_ERROR;
    }

    g_abCancellingIPCs[usIpcNum]=TRUE;
    ulRC=DspBIOS_UnregisterIPCEvent( hDrv, usIpcNum);
    if (ulRC != DSP_NOERROR) {
      MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspIPCThread return from DspBIOS_UnregisterIPCEvent error %x\n",errno);
    }
  }

  pthread_mutex_unlock(&g_hSerializeMgrEntry);   
  return (ulRC);
}
RC APIENTRY dspDisconnectIPC(HIPC hIPC)
{
    RC rc;
    MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspDisconnectITCB entry hIPC %x \n",
	    (int)hIPC);
    rc = main_dspDisconnectIPC( hIPC );
    MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspDisconnectITCB exit rc %lx \n",rc);
    return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspDisconnectGPC                                        */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/

RC APIENTRY main_dspDisconnectGPC(HTASK hOwnerTask,PSZ pszOwner,HTASK hUserTask,PSZ
                                  pszUser)
{
  RC            ulRC;
  RPARM_DISCGPC rparm;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;

  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  rparm.DISCGPC_hOwnerTask = hOwnerTask;
  rparm.DISCGPC_pszOwner = pszOwner;
  rparm.DISCGPC_hUserTask = hUserTask;
  rparm.DISCGPC_pszUser = pszUser;
  rparm.DISCGPC_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspDisconnectGPC,     /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_DISCGPC),       /* Params size   */
		       pszOwner,                    /* pg, buffer1   */
		       pszUser,                     /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       NULL,                        /* return Handle */
		       NULL,                        /* return buffer */
		       0 );                         /* buffer size   */
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);     
  return (ulRC);
}
RC APIENTRY dspDisconnectGPC(HTASK hOwnerTask,PSZ pszOwner,HTASK hUserTask,PSZ
			     pszUser)
{
  RC rc;
  MW_SYSLOG_5(TRACE_MANAGER_API,"mwaveapi::dspDisconnectGPC entry hOwnerTask %x pszOwner %s hUserTask %x pszUser %s\n",
	 (int)hOwnerTask,pszOwner,(int)hUserTask,pszUser);
  rc = main_dspDisconnectGPC( hOwnerTask, pszOwner, hUserTask, pszUser );
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspDisconnectGPC exit rc %lx\n", rc);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspDisconnectITCB                                       */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:                                                                   */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
RC APIENTRY main_dspDisconnectITCB(HTASK hPrimaryTask,PSZ pszPrimary,HTASK
                                   hSecondaryTask,PSZ pszSecondary)
{
  RC         ulRC;
  RPARM_DISCITCB rparm;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;
  
  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  rparm.DISCITCB_hPTask = hPrimaryTask;
  rparm.DISCITCB_pszPName = pszPrimary;
  rparm.DISCITCB_hSTask = hSecondaryTask;
  rparm.DISCITCB_pszSName = pszSecondary;
  rparm.DISCITCB_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspDisconnectITCB,    /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_DISCITCB),      /* Params size   */
		       pszPrimary,                  /* pg, buffer1   */
		       pszSecondary,                /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       NULL,                        /* return Handle */
		       NULL,                        /* return buffer */
		       0 );                         /* buffer size   */

  pthread_mutex_unlock(&g_hSerializeMgrEntry);      
  return (ulRC);
}
RC APIENTRY dspDisconnectITCB(HTASK hPrimaryTask,PSZ pszPrimary,HTASK
			      hSecondaryTask,PSZ pszSecondary)
{
  RC rc;
  MW_SYSLOG_5(TRACE_MANAGER_API,"mwaveapi::dspDisconnectITCB entry hPrimaryTask %x pszPrimary %s hSecondaryTask %x pszSecondary %s\n",
	 (int)hPrimaryTask,pszPrimary,(int)hSecondaryTask,pszSecondary);
  rc = main_dspDisconnectITCB( hPrimaryTask, pszPrimary, hSecondaryTask, pszSecondary );
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspDisconnectITCB exit rc %lx\n",rc);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspFreeModule                                           */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
RC APIENTRY main_dspFreeModule(HMOD hMod)
{
  RC         ulRC;
  RPARM_FREEMOD rparm;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;
  
  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hMod == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);     
    return DSP_INV_HANDLE;
  }
  
  rparm.FREEMOD_hMOD = hMod;
  rparm.FREEMOD_usFlags = DSP_ACTIVATE_INACTIVE;
  rparm.FREEMOD_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspFreeModule,        /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_FREEMOD),       /* Params size   */
		       NULL,                        /* pg, buffer1   */
		       NULL,                        /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       NULL,                        /* return Handle */
		       NULL,                        /* return buffer */
		       0 );                         /* buffer size   */
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);      // Tal - 02/05/96 
  
  return (ulRC);
}
RC APIENTRY dspFreeModule(HMOD hMod)
{
  RC rc;
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspFreeModule entry hMod %x\n",(int)hMod);
  rc = main_dspFreeModule( hMod );
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspFreeModule exit rc %lx\n",rc);
  return rc;
}



/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspFreeSegment                                          */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/****************************************************************************/
RC APIENTRY main_dspFreeSegment(HSEG hSeg)
{
  RC             ulRC;
  RPARM_FREESEG  rparm;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;
  
  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hSeg == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);     
    return DSP_INV_HANDLE;
  }

  rparm.FREESEG_hSeg = hSeg;
  rparm.FREESEG_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspFreeSegment,       /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_FREESEG),       /* Params size   */
		       NULL,                        /* pg, buffer1   */
		       NULL,                        /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       NULL,                        /* return Handle */
		       NULL,                        /* return buffer */
		       0 );                         /* buffer size   */
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);     
  
  return (ulRC);
}
RC APIENTRY dspFreeSegment(HSEG hSeg)
{
  RC rc;
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspFreeSegment entry hSeg %x\n",(int)hSeg);
  rc = main_dspFreeSegment( hSeg );
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspFreeSegment exit rc %lx\n",rc);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspFreeTask                                             */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
RC APIENTRY main_dspFreeTask(HTASK hTask)
{
  RC             ulRC;
  RPARM_FREETASK rparm;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;
  
  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hTask == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);      
    return DSP_INV_HANDLE;
  }
  
  rparm.FREETASK_hTask = hTask;
  rparm.FREETASK_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspFreeTask,          /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_FREETASK),      /* Params size   */
		       NULL,                        /* pg, buffer1   */
		       NULL,                        /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       NULL,                        /* return Handle */
		       NULL,                        /* return buffer */
		       0 );                         /* buffer size   */
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);    

  return (ulRC);
}
RC APIENTRY dspFreeTask(HTASK hTask)
{
  RC rc;
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspFreeTask entry hTask %x\n",(int)hTask);
  rc = main_dspFreeTask( hTask );
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspFreeTask rc %lx\n",rc);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspInit                                                 */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
RC APIENTRY main_dspInit(HDSP hdsp,PSZ pszFileName)
{
  RC         ulRC;                              
  RPARM_INIT rparm;
  BYTE       ResultBuffer[RESBUFLEN];         
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;
  
  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  pszFileName = DspFindFile(pszFileName, ResultBuffer, RESBUFLEN);
  
  rparm.INIT_hDSP = hdsp;
  rparm.INIT_pszFileName = pszFileName;
  rparm.INIT_usState = 0;
  rparm.INIT_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspInit,              /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_INIT),          /* Params size   */
		       pszFileName,                 /* pg, buffer1   */
		       NULL,                        /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       NULL,                        /* return Handle */
		       NULL,                        /* return buffer */
		       0 );                         /* buffer size   */
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);    
  return (ulRC);
}
RC APIENTRY dspInit(HDSP hdsp,PSZ pszFileName)
{
  RC rc;
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspInit entry hdsp %x pszFileName %s\n",(int)hdsp,pszFileName);
  rc = main_dspInit( hdsp, pszFileName );
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspInit exit rc %lx\n",rc);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspLabelToAddress                                       */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/****************************************************************************/
RC APIENTRY main_dspLabelToAddress(HTASK hTask,PSZ pszName,PULONG pulDSPAddr)
{
  RC             ulRC;
  RPARM_LABTOADD rparm;

  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;
  
  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hTask == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);   
    return DSP_INV_HANDLE;
  }

  rparm.LABTOADD_hTask = hTask;
  rparm.LABTOADD_pszName = pszName;
  rparm.LABTOADD_pulDSPAddr = (PULONG)pulDSPAddr;
  rparm.LABTOADD_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspLabelToAddr,       /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_LABTOADD),      /* Params size   */
		       pszName,                     /* pg, buffer1   */
		       NULL,                        /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       pulDSPAddr,                  /* return Handle */
		       NULL,                        /* return buffer */
		       (PUSHORT)0 );                         /* buffer size   */
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);    
  
  return (ulRC);
}
RC APIENTRY dspLabelToAddress(HTASK hTask,PSZ pszName,PULONG pulDSPAddr)
{
  RC rc;
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspLabelToAddress entry hTask %x pszName %s\n",(int)hTask,pszName);
  rc = main_dspLabelToAddress( hTask, pszName, pulDSPAddr );
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspLabelToAddress exit rc %lx ulDSPAddr %x\n",rc,(int)*pulDSPAddr);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspLoadModule                                           */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
RC APIENTRY main_dspLoadModule(HDSP hDSP,PSZ pszFileName,PSZ pszInstanceName,USHORT
                               usFlags,PHMOD phMod)
{
  RC            ulRC;                          
  RPARM_LOADMOD rparm;                         
  BYTE          ResultBuffer[RESBUFLEN];       
    
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;

  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hDSP == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);      
    return DSP_INV_HANDLE;
  }
  
  if ( (usFlags != DSP_ACTIVATE_INACTIVE) &&
       (usFlags != DSP_ACTIVATE_STANDBY)  &&
       (usFlags != DSP_ACTIVATE_ACTIVE) ) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);    
    return DSP_INV_FLAG;
  }
  
  pszFileName = DspFindFile(pszFileName, ResultBuffer, RESBUFLEN);
  
  rparm.LOADMOD_hDSP = hDSP;
  rparm.LOADMOD_pszFileName = pszFileName;
  rparm.LOADMOD_pszInstanceName = pszInstanceName;
  rparm.LOADMOD_usFlags = usFlags;
  rparm.LOADMOD_phMOD = phMod;
  rparm.LOADMOD_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspLoadModule,        /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_LOADMOD),       /* Params size   */
		       pszFileName,                 /* pg, buffer1   */
		       pszInstanceName,             /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       (PULONG)phMod,               /* return Handle */
		       NULL,                        /* return buffer */
		       (PUSHORT)0 );                         /* buffer size   */
  
  if (ulRC != DSP_NOERROR)
    *phMod = (HMOD)NULL;
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);   
  
  return (ulRC);
}
RC APIENTRY dspLoadModule(HDSP hDSP,PSZ pszFileName,PSZ pszInstanceName,USHORT
			  usFlags,PHMOD phMod)
{
  RC rc;
  MW_SYSLOG_5(TRACE_MANAGER_API,"mwaveapi::dspLoadModule entry hDSP %x pszFileName %s pszInstanceName %s usFlags %x\n",
	 (int)hDSP,pszFileName,pszInstanceName,usFlags);
  rc = main_dspLoadModule( hDSP, pszFileName, pszInstanceName, usFlags, phMod );
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspLoadModule exit rc %lx hMod %x\n",rc,(int)*phMod);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspLoadSegment                                          */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:                                                                   */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/****************************************************************************/
RC APIENTRY main_dspLoadSegment(HTASK hTask,PSZ pszFileName,PSZ pszSegmentName,PSZ
                                pszInstanceName,ULONG ulDSPAddr,PHSEG phSeg)
{
  RC            ulRC;                             
  RPARM_LOADSEG rparm;
  BYTE          ResultBuffer[RESBUFLEN];         

  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;

  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hTask == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);    
    return DSP_INV_HANDLE;
  }
  
  pszFileName = DspFindFile(pszFileName, ResultBuffer, RESBUFLEN);
  
  rparm.LOADSEG_hTask = hTask;
  rparm.LOADSEG_pszFileName = pszFileName;
  rparm.LOADSEG_pszSegmentName = pszSegmentName;
  rparm.LOADSEG_pszInstanceName = pszInstanceName;
  rparm.LOADSEG_ulDSPAddr = ulDSPAddr;
  rparm.LOADSEG_phSeg = phSeg;
  rparm.LOADSEG_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspLoadSegment,       /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_LOADSEG),       /* Params size   */
		       pszFileName,                 /* pg, buffer1   */
		       pszSegmentName,              /* pg, buffer2   */
		       pszInstanceName,             /* pg, buffer3   */
		       (PULONG)phSeg,               /* return Handle */
		       NULL,                        /* return buffer */
		       (PUSHORT)0 );                         /* buffer size   */
  
  if (ulRC != DSP_NOERROR)
    *phSeg = (HSEG)NULL;
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);     
  return (ulRC);
}
RC APIENTRY dspLoadSegment(HTASK hTask,PSZ pszFileName,PSZ pszSegmentName,PSZ
			   pszInstanceName,ULONG ulDSPAddr,PHSEG phSeg)
{
  RC rc;
  MW_SYSLOG_6(TRACE_MANAGER_API,"mwaveapi::dspLoadSegment entry hTask %x pszFileName %s pszSegmentName %s pszInstanceName %s, ulDSPAddr %x\n",
	 (int)hTask,pszFileName,pszSegmentName,pszInstanceName,(int)ulDSPAddr);
  rc = main_dspLoadSegment( hTask, pszFileName, pszSegmentName, pszInstanceName, ulDSPAddr, phSeg );
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspLoadModule exit rc %lx hSeg %x\n",
	 rc,(int)*phSeg);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspLoadTask                                             */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue call to the      */
/*           routine.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          pszFileName - FileName for .DSP file                            */
/*          pszRealName - Real Name                                         */
/*          pszVirtName - Virt Name                                         */
/*          usFlags     - Flags                                             */
/*          pRQHWBuf    - HW Request Buf                                    */
/*          phTask      - Pointer to location for new task loaded.          */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*          phTask      - Pointer to location for new task loaded.          */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/****************************************************************************/
RC APIENTRY main_dspLoadTask(HMOD hMod,HDSP hDSP,PSZ pszFileName,PSZ pszTaskName,
                             PSZ pszInstanceName,USHORT usFlags,PVOID pRQHWBuf,
                             PHTASK phTask)
{
  RC             ulRC;
  RPARM_LOADTASK rparm;                         
  BYTE           ResultBuffer[RESBUFLEN];       

  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;
  
  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hMod == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);    
    return DSP_INV_HANDLE;
  }

  if ( (usFlags != DSP_ACTIVATE_INACTIVE) &&
       (usFlags != DSP_ACTIVATE_STANDBY) &&
       (usFlags != DSP_ACTIVATE_ACTIVE) ) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);     
    return (ulRC = DSP_INV_FLAG);
  }

  pszFileName = DspFindFile(pszFileName, ResultBuffer, RESBUFLEN);
  
  rparm.LOADTASK_hMod = hMod;
  rparm.LOADTASK_hDSP = hDSP;
  rparm.LOADTASK_pszFileName = pszFileName;
  rparm.LOADTASK_pszTaskName = pszTaskName;
  rparm.LOADTASK_pszInstanceName = pszInstanceName;
  rparm.LOADTASK_usFlags = usFlags;
  rparm.LOADTASK_pRQHWBuf = pRQHWBuf;
  rparm.LOADTASK_phTask = phTask;
  rparm.LOADTASK_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspLoadTask,          /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_LOADTASK),      /* Params size   */
		       pszFileName,                 /* pg, buffer1   */
		       pszTaskName,                 /* pg, buffer2   */
		       pszInstanceName,             /* pg, buffer3   */
		       (PULONG)phTask,              /* return Handle */
		       NULL,                        /* return buffer */
		       (PUSHORT)0 );                         /* buffer size   */
  
  if (ulRC != DSP_NOERROR)
    *phTask = (HTASK)NULL;
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);    

  return (ulRC);
}
RC APIENTRY dspLoadTask(HMOD hMod,HDSP hDSP,PSZ pszFileName,PSZ pszTaskName,
			PSZ pszInstanceName,USHORT usFlags,PVOID pRQHWBuf,
			PHTASK phTask)
{
  RC rc;
  MW_SYSLOG_7(TRACE_MANAGER_API,"mwaveapi::dspLoadTask entry hMod %x hDsp %x pszFileName %s pszTaskName %s pszInstanceName %s, usFlags %x\n",
	 (int)hMod,(int)hDSP,pszFileName,pszTaskName,pszInstanceName,usFlags);
  rc = main_dspLoadTask( hMod, hDSP, pszFileName, pszTaskName, pszInstanceName, usFlags, pRQHWBuf, phTask );
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspLoadTask exit rc %lx hTask %x\n",
	 rc,(int)*phTask);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspMemTransfer                                          */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
RC APIENTRY main_dspMemTransfer(HDSP hDSP,ULONG ulDSPAddr,PVOID pPCAddress,USHORT
                                usNumWords,USHORT usFlags)
{
  RC         ulRC;
  USHORT     usWordSize;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;
  
  if (hDSP == NULL)
    return DSP_INV_HANDLE;
  
  if (usNumWords == 0)
    return DSP_INV_XFER_SIZE;
  
  if ((usFlags == DSP_MEMXFER_PROG_WRITE) ||
      (usFlags == DSP_MEMXFER_PROG_READ))
    usWordSize = 4;
  else
    usWordSize = 2;
  
  switch(usFlags) {
  case DSP_MEMXFER_DATA_READ:
    ulRC = DspBIOS_D_Read(hDSP,ulDSPAddr,usNumWords,pPCAddress);
    break;
  case DSP_MEMXFER_DATA_WRITE:
    ulRC = DspBIOS_D_Write(hDSP,ulDSPAddr,usNumWords,pPCAddress);
    break;
  case DSP_MEMXFER_PROG_READ:
    ulRC = DspBIOS_I_Read(hDSP,ulDSPAddr,usNumWords,pPCAddress);
    break;
  case DSP_MEMXFER_PROG_WRITE:
    ulRC = DspBIOS_I_Write(hDSP,ulDSPAddr,usNumWords,pPCAddress);
    break;
  default:
    ulRC = DSP_INV_FLAG;
  }
  return (ulRC);
}
RC APIENTRY dspMemTransfer(HDSP hDSP,ULONG ulDSPAddr,PVOID pPCAddress,USHORT
			   usNumWords,USHORT usFlags)
{
  RC rc;
  USHORT usNumBytes;
  USHORT i;
  
  MW_SYSLOG_6(TRACE_MANAGER_API,"mwaveapi::dspMemTransfer entry hDSP %x ulDSPAddr %x usNumWords %x usFlags %x pPCAddress %p\n",
	 (int)hDSP,(int)ulDSPAddr,usNumWords,usFlags,pPCAddress);

  if (pPCAddress == 0) {
    MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspMemTransfer error pPCAddress %p\n",pPCAddress);
    return DSP_INS_BUFFER;
  }

  if ((usFlags == DSP_MEMXFER_PROG_WRITE) || (usFlags == DSP_MEMXFER_PROG_READ))
    usNumBytes=usNumWords*4;
  else
    usNumBytes=usNumWords*2;

  if (usFlags == DSP_MEMXFER_DATA_WRITE) {
      MW_SYSLOG_1(TRACE_MANAGER_API_DATA,"MEMXFER_DATA_WRITE: ");
      for(i=0;i<usNumBytes;i++) {
	MW_SYSLOG_2(TRACE_MANAGER_API_DATA,":%x",((CHAR *)pPCAddress)[i]);
      }
      MW_SYSLOG_1(TRACE_MANAGER_API_DATA,"\n");
  }
  rc = main_dspMemTransfer( hDSP, ulDSPAddr, pPCAddress, usNumWords, usFlags );
  if (usFlags == DSP_MEMXFER_DATA_READ) {
    MW_SYSLOG_1(TRACE_MANAGER_API_DATA,"MEMXFER_DATA_READ: ");
    for(i=0;i<usNumBytes;i++) {
      MW_SYSLOG_2(TRACE_MANAGER_API_DATA,":%x",((CHAR *)pPCAddress)[i]);
    }
    MW_SYSLOG_1(TRACE_MANAGER_API_DATA,"\n");
  }
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspMemTransfer exit rc %lx\n",rc);
  return rc;
}



/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspNameToModuleHandle                                   */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
RC APIENTRY main_dspNameToModuleHandle(PSZ pszInstanceName,PHMOD phMod)
{
  RC                ulRC;
  RPARM_NAMEMODULEH rparm;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;
  
  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  rparm.NAMEMODULEH_pszInstanceName = pszInstanceName;
  rparm.NAMEMODULEH_phModule = phMod;
  rparm.NAMEMODULEH_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspNameToModuleHandle,/* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_NAMEMODULEH),   /* Params size   */
		       pszInstanceName,             /* pg, buffer1   */
		       NULL,                        /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       (PULONG)phMod,               /* return Handle */
		       NULL,                        /* return buffer */
		       (PUSHORT)0 );                         /* buffer size   */
  
  if (ulRC != DSP_NOERROR)
    *phMod = (HMOD)NULL;
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);    
  
  return (ulRC);
}
RC APIENTRY dspNameToModuleHandle(PSZ pszInstanceName,PHMOD phMod)
{
  RC rc;
  
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspNameToModuleHandle entry pszInstanceName %s\n",
	 pszInstanceName);
  rc = main_dspNameToModuleHandle( pszInstanceName, phMod );
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspNameToModuleHandle exit rc %lx hMod %x\n",
	 rc,(int)*phMod);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspNameToSegmentHandle                                  */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
RC APIENTRY main_dspNameToSegmentHandle(HTASK hTask,PSZ pszInstanceName,PHSEG phSeg)
{
  RC             ulRC;
  RPARM_NAMESEGH rparm;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;
  
  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hTask == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);    
    return DSP_INV_HANDLE;
  }

  rparm.NAMESEGH_hTask = hTask;
  rparm.NAMESEGH_pszInstanceName = pszInstanceName;
  rparm.NAMESEGH_phSeg = phSeg;
  rparm.NAMESEGH_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspNameToSegmentHandle, /* function      */
		       (PVOID)&rparm,                 /* pParams       */
		       sizeof(RPARM_NAMESEGH),        /* Params size   */
		       pszInstanceName,               /* pg, buffer1   */
		       NULL,                          /* pg, buffer2   */
		       NULL,                          /* pg, buffer3   */
		       (PULONG)phSeg,                 /* return Handle */
		       NULL,                          /* return buffer */
		       (PUSHORT)0 );                           /* buffer size   */
  
  if (ulRC != DSP_NOERROR)
    *phSeg = (HSEG)NULL;
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);    
  return (ulRC);
}
RC APIENTRY dspNameToSegmentHandle(HTASK hTask,PSZ pszInstanceName,PHSEG phSeg)
{
  RC rc;
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspNameToSegmentHandle entry hTask %x pszInstanceName %s\n",
	 (int)hTask,pszInstanceName);
  rc = main_dspNameToSegmentHandle( hTask, pszInstanceName, phSeg );
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspNameToSegmentHandle exit rc %lx, hSeg %x\n",
	 rc,(int)*phSeg);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspNameToTaskHandle                                     */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
RC APIENTRY main_dspNameToTaskHandle(HMOD hMod,PSZ pszInstanceName,PHTASK phTask)
{
  RC              ulRC;
  RPARM_NAMETASKH rparm;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;

  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hMod == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);     
    return DSP_INV_HANDLE;
  }
  
  rparm.NAMETASKH_hMod = hMod;
  rparm.NAMETASKH_pszInstanceName = pszInstanceName;
  rparm.NAMETASKH_phTask = phTask;
  rparm.NAMETASKH_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspNameToTaskHandle,  /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_NAMETASKH),     /* Params size   */
		       pszInstanceName,             /* pg, buffer1   */
		       NULL,                        /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       (PULONG)phTask,              /* return Handle */
		       NULL,                        /* return buffer */
		       (PUSHORT)0 );                         /* buffer size   */
  
  if (ulRC != DSP_NOERROR)
    *phTask = (HTASK)NULL;
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);   
  return (ulRC);
}
RC APIENTRY dspNameToTaskHandle(HMOD hMod,PSZ pszInstanceName,PHTASK phTask)
{
  RC rc;
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspNameToTaskHandle entry hMod %x pszInstanceName %s\n",
	 (int)hMod,pszInstanceName);
  rc = main_dspNameToTaskHandle( hMod, pszInstanceName, phTask );
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspNameToTaskHandle exit rc %lx hTask %x\n",
	 rc,(int)*phTask);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspQueryDSPInfo                                         */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue call to the      */
/*           routine.                                                       */
/*                                                                          */
/* INPUT:   hDSP        - Handle to a DSP                                   */
/*          puSize      - Size of Buffer that gets info                     */
/*          pDSPInfoBuf - Buffer                                            */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                         DSP_INS_BUFFER                                   */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
RC APIENTRY main_dspQueryDSPInfo(HDSP hDSP,PUSHORT pusSize,PVOID pDSPInfoBuf)
{
  RC         ulRC;
  RPARM_QDSPINFO rparm;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;

  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hDSP == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);    
    return DSP_INV_HANDLE;
  }
  
  rparm.QDSPINFO_hDsp = hDSP;
  rparm.QDSPINFO_pusSize = pusSize;
  rparm.QDSPINFO_pDspInfoBuf = pDSPInfoBuf;
  rparm.QDSPINFO_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspQueryDSPInfo,      /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_QDSPINFO),      /* Params size   */
		       NULL ,                       /* pg, buffer1   */
		       NULL,                        /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       NULL,                        /* return Handle */
		       pDSPInfoBuf,                 /* return buffer */
		       pusSize );                   /* buffer size   */
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);    
  return (ulRC);
}
RC APIENTRY dspQueryDSPInfo(HDSP hDSP,PUSHORT pusSize,PVOID pDSPInfoBuf)
{
  RC rc;
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspQueryDSPInfo entry hDSP %x\n",
	 (int)hDSP);
  rc = main_dspQueryDSPInfo( hDSP, pusSize, pDSPInfoBuf );
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspQueryDSPInfo exit rc %lx\n",rc);
  
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspQueryMangerInfo                                      */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue call to the      */
/*           routine.                                                       */
/*                                                                          */
/* INPUT:   hMgr        - Handle to a DSP Manager                           */
/*          puSize      - Size of Buffer that gets info                     */
/*          pMgrInfoBuf - Buffer                                            */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                         DSP_INS_BUFFER                                   */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
RC APIENTRY main_dspQueryManagerInfo(PUSHORT pusSize,PVOID pMgrInfoBuf)
{
  RC             ulRC;
  RPARM_QMANINFO rparm;
  
  if ( bDspDisabled ) {
    MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::main_dspQueryManagerInfo returns DSP_NOT_AVAILABLE %x\n",
	DSP_NOT_AVAILABLE);
//	gtid(),DSP_NOT_AVAILABLE);
    return DSP_NOT_AVAILABLE;
  } 

  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  rparm.QMANINFO_pusSize = pusSize;
  rparm.QMANINFO_pMgrInfoBuf = pMgrInfoBuf;
  rparm.QMANINFO_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspQueryManagerInfo,  /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_QMANINFO),      /* Params size   */
		       NULL,                        /* pg, buffer1   */
		       NULL,                        /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       NULL,                        /* return Handle */
		       pMgrInfoBuf,                 /* return buffer */
		       pusSize );                   /* buffer size   */
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);    
  return (ulRC);
}
RC APIENTRY dspQueryManagerInfo(PUSHORT pusSize,PVOID pMgrInfoBuf)
{
  RC rc;
  MW_SYSLOG_1(TRACE_MANAGER_API,"mwaveapi::dspQueryManagerInfo entry\n");
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspQueryManagerInfo pusSize %x, pMgrInfoBuf %p\n",*pusSize,pMgrInfoBuf); // @TBD

  rc = main_dspQueryManagerInfo( pusSize, pMgrInfoBuf );
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspQueryManagerInfo exit rc %lx\n",rc);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspQueryMiscInfo                                        */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue call to the      */
/*           routine.                                                       */
/*                                                                          */
/* INPUT:   usType       - Type of info requested                           */
/*          hMisc        - Handle to element                                */
/*          puSize       - Size of Buffer that gets info                    */
/*          pMiscInfoBuf - Buffer                                           */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                         DSP_INS_BUFFER                                   */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
RC APIENTRY main_dspQueryMiscInfo(USHORT usType,HMISC hMisc,PUSHORT pusSize,PVOID
                                  pMiscInfoBuf)
{
  RC              ulRC;
  RPARM_QMISCINFO rparm;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;

  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hMisc == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);   
    return DSP_INV_HANDLE;
  }
  
  rparm.QMISCINFO_usType = usType;
  rparm.QMISCINFO_hMisc = hMisc;
  rparm.QMISCINFO_pusSize = pusSize;
  rparm.QMISCINFO_pMiscInfoBuf = pMiscInfoBuf;
  rparm.QMISCINFO_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspQueryMiscInfo,     /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_QMISCINFO),     /* Params size   */
		       NULL,                        /* pg, buffer1   */
		       NULL,                        /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       NULL,                        /* return Handle */
		       pMiscInfoBuf,                /* return buffer */
		       pusSize );                   /* buffer size   */
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);      
  return (ulRC);
}
RC APIENTRY dspQueryMiscInfo(USHORT usType,HMISC hMisc,PUSHORT pusSize,PVOID
			     pMiscInfoBuf)
{
  RC rc;
  MW_SYSLOG_3(TRACE_MANAGER_API,"mwaveapi::dspQueryMiscInfo entry usType %x hMisc %x\n",usType,(int)hMisc);
  rc = main_dspQueryMiscInfo( usType, hMisc, pusSize, pMiscInfoBuf );
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspQueryMiscInfo exit rc %lx\n",rc);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspQueryModuleInfo                                      */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue call to the      */
/*           routine.                                                       */
/*                                                                          */
/* INPUT:   hMod        - Handle to a DSP Module                            */
/*          puSize      - Size of Buffer that gets info                     */
/*          pModInfoBuf - Buffer                                            */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                         DSP_INS_BUFFER                                   */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
RC APIENTRY main_dspQueryModuleInfo(HMOD hMod,PUSHORT pusSize,PVOID pModInfoBuf)
{
  RC             ulRC;
  RPARM_QMODINFO rparm;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;
  
  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hMod == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);    
    return DSP_INV_HANDLE;
  }
  
  rparm.QMODINFO_hMod = hMod;
  rparm.QMODINFO_pusSize = pusSize;
  rparm.QMODINFO_pModInfoBuf = pModInfoBuf;
  rparm.QMODINFO_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspQueryModuleInfo,   /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_QMODINFO),      /* Params size   */
		       NULL,                        /* pg, buffer1   */
		       NULL,                        /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       NULL,                        /* return Handle */
		       pModInfoBuf,                 /* return buffer */
		       pusSize );                   /* buffer size   */
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);     
  return (ulRC);
}
RC APIENTRY dspQueryModuleInfo(HMOD hMod,PUSHORT pusSize,PVOID pModInfoBuf)
{
  RC rc;
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspQueryModuleInfo entry hMod %x\n",(int)hMod);
  rc = main_dspQueryModuleInfo( hMod, pusSize, pModInfoBuf );
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspQueryModuleInfo exit %lx\n",rc);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspQueryTaskInfo                                        */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue call to the      */
/*           routine.                                                       */
/*                                                                          */
/* INPUT:   hTask        - Handle to a Task                                 */
/*          puSize       - Size of Buffer that gets info                    */
/*          pTaskInfoBuf - Buffer                                           */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                         DSP_INS_BUFFER                                   */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
RC APIENTRY main_dspQueryTaskInfo(HTASK hTask,PUSHORT pusSize,PVOID pTaskInfoBuf)
{
  RC              ulRC;
  RPARM_QTASKINFO rparm;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;

  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hTask == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);     
    return DSP_INV_HANDLE;
  }
  
  rparm.QTASKINFO_hTask = hTask;
  rparm.QTASKINFO_pusSize = pusSize;
  rparm.QTASKINFO_pTaskInfoBuf = pTaskInfoBuf;
  rparm.QTASKINFO_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspQueryTaskInfo,     /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_QTASKINFO),     /* Params size   */
		       NULL,                        /* pg, buffer1   */
		       NULL,                        /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       NULL,                        /* return Handle */
		       pTaskInfoBuf,                /* return buffer */
		       pusSize );                   /* buffer size   */
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);     
  return (ulRC);
}
RC APIENTRY dspQueryTaskInfo(HTASK hTask,PUSHORT pusSize,PVOID pTaskInfoBuf)
{
  RC rc;
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspQueryTaskInfo entry hTask %x\n",(int)hTask);
  rc = main_dspQueryTaskInfo( hTask, pusSize, pTaskInfoBuf );
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspQueryTaskInfo exit rc %lx\n",rc);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspReset                                                */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/****************************************************************************/
RC APIENTRY main_dspReset(HDSP hdsp)
{
  RC             ulRC;                          // Error return code
  RPARM_RESETDSP rparm;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;

  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hdsp == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);   
    return DSP_INV_HANDLE;
  }
  
  rparm.RESETDSP_hDsp = hdsp;
  rparm.RESETDSP_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspReset,             /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_RESETDSP),      /* Params size   */
		       NULL,                        /* pg, buffer1   */
		       NULL,                        /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       NULL,                        /* return Handle */
		       NULL,                        /* return buffer */
		       0 );                         /* buffer size   */
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);  
  return (ulRC);
}
RC APIENTRY dspReset(HDSP hdsp)
{
  RC rc;
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspReset entry hdsp %x\n",(int)hdsp);
  rc = main_dspReset( hdsp );
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspReset exit rc %lx\n",rc);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: dspRun                                                  */
/*                                                                          */
/* FUNCTION: This routine will verify parameters and issue and IOCtl to     */
/*           the PDD.                                                       */
/*                                                                          */
/* INPUT:   hMod    - Handle to Module                                      */
/*          usFlags - Flags                                                 */
/*                                                                          */
/* OUTPUT:  Return Codes:  DSP_NOERROR                                      */
/*                         DSP_INV_HANDLE                                   */
/*                         DSP_INV_FLAG                                     */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
RC APIENTRY main_dspRun(HDSP hdsp)
{
  RC           ulRC;                            
  RPARM_RUNDSP rparm;
  
  if ( bDspDisabled ) return DSP_NOT_AVAILABLE;

  pthread_mutex_lock(&g_hSerializeMgrEntry);
  
  if (hdsp == NULL) {
    pthread_mutex_unlock(&g_hSerializeMgrEntry);     
    return DSP_INV_HANDLE;
  }
  
  rparm.RUNDSP_hDsp = hdsp;
  rparm.RUNDSP_ulRC = DSP_NOERROR;
  
  ulRC = SendToServer( DSPMDD_DspRun,               /* function      */
		       (PVOID)&rparm,               /* pParams       */
		       sizeof(RPARM_RUNDSP),        /* Params size   */
		       NULL,                        /* pg, buffer1   */
		       NULL,                        /* pg, buffer2   */
		       NULL,                        /* pg, buffer3   */
		       NULL,                        /* return Handle */
		       NULL,                        /* return buffer */
		       0 );                         /* buffer size   */
  
  pthread_mutex_unlock(&g_hSerializeMgrEntry);    
  return (ulRC);
}
RC APIENTRY dspRun(HDSP hdsp)
{
  RC rc;
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspRun entry hdsp %x\n",(int)hdsp);
  rc = main_dspRun( hdsp );
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspRun exit rc %lx\n",rc);
  return rc;
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME:  DspFindFile                                            */
/*                                                                          */
/* DISCRIPTIVE NAME: Qualify a file name with the entire path.              */
/*                                                                          */
/* FUNCTION:         Same                                                   */
/*                                                                          */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   NONE                                                   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/* ENTRY POINTS:     DspFindFile                                            */
/*                                                                          */
/* INPUT:            pszFileName                                            */
/*                   ResultBuffer[]                                         */
/*                   Size of Buffer (integer)                               */
/*                                                                          */
/* EXIT-NORMAL:      pszFileName has fully qualified filename ulRC = 0      */
/* EXIT-ABNORMAL:    pszFileName is returned unchanged        ulRC = !0     */
/*                                                                          */
/* EXTERNAL REFERENCES: System Calls                                        */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
PSZ DspFindFile(PSZ pszFileName,PSZ ResultBuffer,int iSizeBuf)
{
  /* @TBD for now assume DSP files are in current directory */
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::DspFindFile entry pszFileName %s\n",pszFileName);
  //ResultBuffer[0]='.';
  //ResultBuffer[1]='/';
  //strcpy(ResultBuffer+2,pszFileName);
  strcpy(ResultBuffer,pszFileName);
  MW_SYSLOG_2(TRACE_MANAGER_API,"mwaveapi::dspFindFile exit pszFileName %s\n",ResultBuffer);
  return (ResultBuffer);
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME:  SendToServer                                           */
/*                                                                          */
/* DISCRIPTIVE NAME: Send Application request to Server and                 */
/*                   return the Reply                                       */
/*                                                                          */
/* FUNCTION:         Same                                                   */
/*                                                                          */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   NONE                                                   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/* ENTRY POINTS:     SendToServer                                           */
/*                                                                          */
/* INPUT:            ulCode      - function                                 */
/*                   prParams    - pParams                                  */
/*                   dwParamSize - Params size                              */
/*                   pszBuf1     - pg, buffer1                              */
/*                   pszBuf2     - pg, buffer2                              */
/*                   pszBuf3     - pg, buffer3                              */
/*                   pRetHandle  - return Handle                            */
/*                   pRetBuffer  - return buffer                            */
/*                   pBufSize    - buffer size                              */
/*                                                                          */
/* EXTERNAL REFERENCES: System Calls                                        */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/
ULONG SendToServer( ULONG ulCode, PVOID prParams, DWORD dwParamSize,
                    PSZ pszBuf1, PSZ pszBuf2, PSZ pszBuf3,
                    PDWORD pRetHandle, PBYTE pRetBuffer, PUSHORT pBufSize )
{
  ULONG                 ulRC;
  PMWMGR_REQUEST_HEAD   pRequestHead;
  PMWMGR_ANSWER_HEAD    pAnswerHead;
  DWORD                 dwBuf1Size,dwBuf2Size,dwBuf3Size;
  LPBYTE                lpbRequest;
  DWORD                 dwRequestSize,dwAnswerSize;
  BYTE                * pbServerAnswerBuffer;
  
  
  /* Calculate actual size of Request packet. */
  
  dwBuf1Size = dwBuf2Size = dwBuf3Size = 0;
  
  if (pszBuf1)
    dwBuf1Size = strlen(pszBuf1) + 1;
  
  if (pszBuf2)
    dwBuf2Size = strlen(pszBuf2) + 1;
  
  if (pszBuf3)
    dwBuf3Size = strlen(pszBuf3) + 1;
  
  dwRequestSize = sizeof( MWMGR_REQUEST_HEAD ) + dwParamSize +
    dwBuf1Size + dwBuf2Size + dwBuf3Size;
  
  
  /* Set Request pointers to begining of buffer */
  
  lpbRequest   = pRequestBuffer;
  pRequestHead = (MWMGR_REQUEST_HEAD *) lpbRequest;
  
  
  /* Fill the Request header */
  
  pRequestHead->dwRequestCode = ulCode;
  pRequestHead->dwPacketSize  = dwParamSize;
  pRequestHead->dwPgBuf1Size  = dwBuf1Size;
  pRequestHead->dwPgBuf2Size  = dwBuf2Size;
  pRequestHead->dwPgBuf3Size  = dwBuf3Size;
  
  if (pBufSize != NULL)
    pRequestHead->dwRetBufSize = (DWORD) *pBufSize;
  else
    pRequestHead->dwRetBufSize = 0;
  
  lpbRequest += sizeof( MWMGR_REQUEST_HEAD );
  
  
  /* Add Command packet */
  
  if (dwParamSize > 0) {
    memcpy( (PVOID)lpbRequest, prParams, dwParamSize );
    lpbRequest += dwParamSize;
  }
  
  
  /* Add the three string arguments (if any) */
  
  if (dwBuf1Size > 0) {
    memcpy( (PVOID)lpbRequest, pszBuf1, dwBuf1Size );
    lpbRequest += dwBuf1Size;
  }
  
  if (dwBuf2Size > 0) {
    memcpy( (PVOID)lpbRequest, pszBuf2, dwBuf2Size );
    lpbRequest += dwBuf2Size;
  }
  
  if (dwBuf3Size > 0)
    memcpy( (PVOID)lpbRequest, pszBuf3, dwBuf3Size );
  
  
  //
  //  Send the request to the Mwave API server
  //
  if (!IssueMwaveAPIRequest( pRequestBuffer, dwRequestSize, &pbServerAnswerBuffer, (unsigned *)&dwAnswerSize ))
    return DSP_NO_MWAVEDM;
  
  memcpy( pAnswerBuffer, pbServerAnswerBuffer, dwAnswerSize );
  FreeMwaveAPIAnswerBuffer( pbServerAnswerBuffer );
  
  
  /* Unpack the Answer */
  
  pAnswerHead = (PMWMGR_ANSWER_HEAD) pAnswerBuffer;
  
  ulRC = (DWORD) pAnswerHead->dwReturnCode;
  
  if (ulRC == DSP_NOERROR) {
    
    if (pRetHandle != NULL)
      *pRetHandle = pAnswerHead->dwHandle;
    
    if ((pRetBuffer != NULL) && (pBufSize != NULL)) {
      memcpy( (PVOID)pRetBuffer,
		  (PVOID)(pAnswerBuffer + sizeof(MWMGR_ANSWER_HEAD)),
		  pAnswerHead->dwBufSize );
      *pBufSize = (USHORT) pAnswerHead->dwBufSize;
    }
  }
  
  if ((ulRC == DSP_INS_BUFFER) && (pBufSize != NULL))
    *pBufSize = (USHORT) pAnswerHead->dwBufSize;
  
  return (ulRC);
}


