/*
 *   mwmcmsg.c -- Mwave Modem AT Command Parser
 *
 *  Written By: Paul Schroeder IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#include <mwmspcfc.h>

static char szThisFile[] = "MWMCMSG.C";

ULONG mwmcmsgWriteConnectMessage(PMWM_DSPINFO pmwmDspInfo,MWM_MODEM_STATUS  *pModemStatus)
{
  ULONG ulRC = 0;
  REGISTERS Registers;
  USHORT    usUseLineSpeed = 1;
  USHORT    usUseMNPResults = 1;
  ULONG     ulAddress = 0;
  USHORT    usDefaultMessageCase = 0; // Show that we took the Default Case For Message #
  USHORT    usConnectMessageNumber = 0;
  USHORT    usConnectSpeed;
  USHORT    usWValue=0;
  USHORT    usTempValue = 0;
  ULONG     ulComPortBPS=0;

  char  achSpeedString[10]      = "\0";
  char  achConnectMessage[80]   = "CONNECT ";
  char  achProtocolMessage[]    = "ECL";
  char  achCompressionMessage[] = "C";

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmcmsg::mwmcmsgWriteConnectMessage entry\n");  }


  /***************************************************************************/
  /* Check to see if the Speed will be used in the connect message.          */
  /* The X0 setting should prevent the Line Speed from showing in the connect*/
  /* Message.                                                                */
  /***************************************************************************/
  ulRC = mwmParseQueryModemRegisters(&Registers);
  if (ulRC)
    return (ulRC);

  /***************************************************************************/
  /* X is bits 6-4 of S22                                                    */
  /* Check for X0                                                            */
  /***************************************************************************/
  if ( !(Registers.S[22] & 0x0070) )
    usUseLineSpeed = 0;


  /***************************************************************************/
  /*  									     */
  /***************************************************************************/
  if ( Registers.S[14] & 0x0010 )
    usUseMNPResults = 0;

  /***************************************************************************/
  /* Add the linespeed if we are supposed to show it.                        */
  /***************************************************************************/
  usConnectSpeed = pModemStatus->usLineSpeed;

  /**********************************************************
  ** If we've got an x2 connection, we need to compute the
  ** actual connect speed that we will report.
  ***********************************************************/
  if (pmwmDspInfo->usPCMConnection) {
    usConnectSpeed = mwmcmsgConvertX2ConnectSpeed(pModemStatus->usLineSpeed);
  } /* endif */

  if (usUseLineSpeed)
  {
    usWValue = mwmParseGetWValue();
    if (usWValue == 0)
    {
      /***************************************************************************/
      /* Get Address of UART registers...                                        */
      /***************************************************************************/
      ulRC = dspLabelToAddress(pmwmDspInfo->hmctlTask, "UARTREGS", &ulAddress);
      if (ulRC!=DSP_NOERROR)
      {
        mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                       MWM_DSP_ERROR, ulRC);
        return(ulRC);
      }

      /***************************************************************************/
      /* Read UART_DL location.                                                  */
      /***************************************************************************/
      ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddress + 4,
                            &usTempValue, 1,
                            DSP_MEMXFER_DATA_READ);
      if (ulRC!=DSP_NOERROR)
      {
        mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                       MWM_DSP_ERROR, ulRC);
        return(ulRC);
      }

      ulComPortBPS = 115200 / usTempValue;
      sprintf(achSpeedString, "%ld", ulComPortBPS);
      strcat(achConnectMessage,achSpeedString);
    }
    else
    {
      sprintf(achSpeedString,"%u", usConnectSpeed);
      strcat(achConnectMessage,achSpeedString);
    }
  }


  /***************************************************************************/
  /* Now Check for Protocols...                                              */
  /***************************************************************************/
  if ((pModemStatus->usProtocol & (MWM_STATUS_V42_CONNECT | MWM_STATUS_MNP_CONNECT) ) &&
      (usUseMNPResults))
  {
    if (usUseLineSpeed)
    {
      strcat(achConnectMessage,"/");
    }

    strcat(achConnectMessage,achProtocolMessage);

    /***************************************************************************/
    /* Now Check for Compression...                                            */
    /***************************************************************************/
    if (pModemStatus->usProtocol & (MWM_STATUS_V42BIS_CONNECT | MWM_STATUS_MNP5_CONNECT) )
    {
      strcat(achConnectMessage,achCompressionMessage);
    }
  }


  /***************************************************************************/
  /* Find the connect message number                                         */
  /***************************************************************************/

  /***************************************************************************/
  /* Get the base "non-ecl" message number.                                  */
  /***************************************************************************/
  if (usUseLineSpeed)
  {
    if (usWValue == 0)
      usConnectSpeed = ulComPortBPS;
    switch (usConnectSpeed)
    {
      case 1200:
        usConnectMessageNumber = 5;
        if ((pModemStatus->usProtocol & (MWM_STATUS_V42_CONNECT | MWM_STATUS_MNP_CONNECT) ) &&
            (usUseMNPResults))
        {
          if  (pModemStatus->usProtocol & (MWM_STATUS_V42BIS_CONNECT | MWM_STATUS_MNP5_CONNECT) )
            usConnectMessageNumber = 26;
          else
            usConnectMessageNumber = 25;
        }
        break;
      case 2400:
        usConnectMessageNumber = 10;
        if ((pModemStatus->usProtocol & (MWM_STATUS_V42_CONNECT | MWM_STATUS_MNP_CONNECT) ) &&
            (usUseMNPResults))
        {
          if  (pModemStatus->usProtocol & (MWM_STATUS_V42BIS_CONNECT | MWM_STATUS_MNP5_CONNECT) )
            usConnectMessageNumber = 31;
          else
            usConnectMessageNumber = 30;
        }
        break;
      case 4800:
        usConnectMessageNumber = 33;
        break;
      case 7200:
        usConnectMessageNumber = 37;
        break;
      case 9600:
        usConnectMessageNumber = 41;
        break;
      case 12000:
        usConnectMessageNumber = 45;
        break;
      case 14400:
        usConnectMessageNumber = 49;
        break;
      case 16800:
        usConnectMessageNumber = 53;
        break;
      case 19200:
        usConnectMessageNumber = 57;
        break;
      case 21600:
        usConnectMessageNumber = 61;
        break;
      case 24000:
        usConnectMessageNumber = 64;
        break;
      case 26400:
        usConnectMessageNumber = 67;
        break;
      case 28800:
        usConnectMessageNumber = 71;
        break;
      case 29333:
        usConnectMessageNumber = 74;
        break;
      case 30000:
        usConnectMessageNumber = 77;
        break;
      case 30666:
        usConnectMessageNumber = 80;
        break;
      case 31200:
        usConnectMessageNumber = 83;
        break;
      case 32000:
        usConnectMessageNumber = 86;
        break;
      case 33333:
        usConnectMessageNumber = 89;
        break;
      case 33600:
        usConnectMessageNumber = 92;
        break;
      case 34000:
        usConnectMessageNumber = 95;
        break;
      case 34666:
        usConnectMessageNumber = 98;
        break;
      case 36000:
        usConnectMessageNumber = 101;
        break;
      case 37333:
        usConnectMessageNumber = 104;
        break;
      case 38000:
        usConnectMessageNumber = 107;
        break;
      case 38666:
        usConnectMessageNumber = 110;
        break;
      case 40000:
        usConnectMessageNumber = 113;
        break;
      case 41333:
        usConnectMessageNumber = 116;
        break;
      case 42000:
        usConnectMessageNumber = 119;
        break;
      case 42666:
        usConnectMessageNumber = 122;
        break;
      case 44000:
        usConnectMessageNumber = 125;
        break;
      case 45333:
        usConnectMessageNumber = 128;
        break;
      case 46000:
        usConnectMessageNumber = 131;
        break;
      case 46666:
        usConnectMessageNumber = 134;
        break;
      case 48000:
        usConnectMessageNumber = 137;
        break;
      case 49333:
        usConnectMessageNumber = 140;
        break;
      case 50000:
        usConnectMessageNumber = 143;
        break;
      case 50666:
        usConnectMessageNumber = 146;
        break;
      case 52000:
        usConnectMessageNumber = 149;
        break;
      case 53333:
        usConnectMessageNumber = 152;
        break;
      case 54000:
        usConnectMessageNumber = 155;
        break;
      case 54666:
        usConnectMessageNumber = 158;
        break;
      case 56000:
        usConnectMessageNumber = 161;
        break;
      case 57333:
        usConnectMessageNumber = 164;
        break;
      case 64000:
        usConnectMessageNumber = 167;
        break;
      case 38400:
        usConnectMessageNumber = 170;
        break;
      case 57600:
        usConnectMessageNumber = 173;
        break;
      case 115200:
        usConnectMessageNumber = 176;
        break;
      default:
        usConnectMessageNumber = 1;
        if ((pModemStatus->usProtocol & (MWM_STATUS_V42_CONNECT | MWM_STATUS_MNP_CONNECT) ) &&
            (usUseMNPResults))
        {
          if  (pModemStatus->usProtocol & (MWM_STATUS_V42BIS_CONNECT | MWM_STATUS_MNP5_CONNECT) )
            usConnectMessageNumber = 20;
          else
            usConnectMessageNumber = 19;
        }
        usDefaultMessageCase = 1;  // show that we took the default case....
        break;
    }


    /***************************************************************************/
    /* Now, make the correct modifications to message numbers if               */
    /* Protocol or compression.                                                */
    /***************************************************************************/
    if ((pModemStatus->usLineSpeed > 2400) &&
        (!usDefaultMessageCase) &&
        (usUseMNPResults))
    {
      if (pModemStatus->usProtocol & (MWM_STATUS_V42_CONNECT | MWM_STATUS_MNP_CONNECT) )
      {
        usConnectMessageNumber++;
        /***************************************************************************/
        /* Now Check for Compression...                                            */
        /***************************************************************************/
        if (pModemStatus->usProtocol & (MWM_STATUS_V42BIS_CONNECT | MWM_STATUS_MNP5_CONNECT) )
        {
          usConnectMessageNumber++;
        }
      }
    }
  } // if we are to use the line speed information in the connect message.
  else
  {
    usConnectMessageNumber = 1;
    if ((pModemStatus->usProtocol & (MWM_STATUS_V42_CONNECT | MWM_STATUS_MNP_CONNECT) ) &&
        (usUseMNPResults))
    {
      usConnectMessageNumber = 19;
      if (pModemStatus->usProtocol & (MWM_STATUS_V42BIS_CONNECT | MWM_STATUS_MNP5_CONNECT) )
      {
        usConnectMessageNumber++;
      }

    }
  }



  /***************************************************************************/
  /* Put the connect message number in CMSGNUM                               */
  /***************************************************************************/
  ulRC = dspLabelToAddress(pmwmDspInfo->hmctlTask, "CMSGNUM", &ulAddress);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }



  ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                        &usConnectMessageNumber, 1,
                        DSP_MEMXFER_DATA_WRITE);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }



  /***************************************************************************/
  /* Put the final connect message in ATRLCBUF                               */
  /***************************************************************************/
  ulRC = mwmrspEchoStringToATRLCBUF(pmwmDspInfo, achConnectMessage);
  if (ulRC)
    return(ulRC);

  { MW_SYSLOG_1(TRACE_MWMLW32,"mwmcmsg::mwmcmsgWriteConnectMessage exit\n");  }


  return 0;
}

/**********************************************************
** Function: mwmcmsgConvertX2ConnectSpeed
**
** If we've got an x2 connection, we need to compute the
** actual connect speed that we will report. The data
** pump cannot compute the precise connect speed, so we
** need to determine the actual speed as follows:
** - Get the speed reported by modem control.
** - If the speed is less than the lowest possible
**   x2 speed, or higher than the highest possible,
**   use the lowest (or highest) x2 speed.
** - If the speed exactly matches an x2 connect speed,
**   then we'll keep that as the actual speed.
** - If the speed doesn't match, we'll determine if
**   the reported speed is higher than the midpoint
**   between two valid speeds. If so, report the higher
**   speed. If not, report the lower speed.
***********************************************************/
USHORT MWM_ENTRY mwmcmsgConvertX2ConnectSpeed(USHORT usLineSpeed)
{
  int    i;
  USHORT usConnectSpeed=0;
  BOOL   bX2SpeedFound = FALSE;
  USHORT usArraySize;
  USHORT ausX2ConnectSpeeds[] = {29333, 30666, 32000, 33333, 34666, 36000,
                                 37333, 38666, 40000, 41333, 42666, 44000,
                                 45333, 46666, 48000, 49333, 50666, 52000,
                                 53333, 54666, 56000, 57333 };

  { MW_SYSLOG_1(TRACE_MWMLW32,"mwmcmsg::mwmcmsgConvertX2ConnectSpeed entry\n");  }


  usArraySize = sizeof(ausX2ConnectSpeeds) / sizeof(USHORT);

  if (usLineSpeed < ausX2ConnectSpeeds[0]) {
    usConnectSpeed = ausX2ConnectSpeeds[0];
  } else if (usLineSpeed > ausX2ConnectSpeeds[usArraySize-1]) {
    usConnectSpeed = ausX2ConnectSpeeds[usArraySize-1];
  } else {

    for (i=0; (i < usArraySize) && !bX2SpeedFound; i++) {
      if (usLineSpeed == ausX2ConnectSpeeds[i]) {
        bX2SpeedFound = TRUE;
        usConnectSpeed = usLineSpeed;
      } else if ( usLineSpeed > ausX2ConnectSpeeds[i] &&
                  usLineSpeed < ausX2ConnectSpeeds[i+1] ) {
        if (usLineSpeed >= ausX2ConnectSpeeds[i]+(1333/2)) {
          usConnectSpeed = ausX2ConnectSpeeds[i+1];
        } else {
          usConnectSpeed = ausX2ConnectSpeeds[i];
        } /* endif */
        bX2SpeedFound = TRUE;
      } /* endif */
    } /* endfor */
  } /* endif */

  { MW_SYSLOG_1(TRACE_MWMLW32,"mwmcmsg::mwmcmsgConvertX2ConnectSpeed exit\n");  }


  return(usConnectSpeed);
}

