/*  Pasang Emas. Enjoy a unique traditional game of Brunei.
    Copyright (C) 2010  Nor Jaidi Tuah

    This file is part of Pasang Emas.
      
    Pasang Emas is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
namespace Pasang {

class PatternSelector : Gtk.ScrolledWindow {
    /**
     * Emitted whenever a pattern is selected.
     */
    public signal void pattern_changed_signal (string? move);

    /**
     * List of core patterns: randomizers and traditional patterns.
     * This list is fixed: no items are added or removed from it.
     */
    Pattern[] core_patterns = {
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Ajong Dengan Sauhnya"),
            "HPHHPPPHHPH PPHPPHPPHPP HHPHHPHHPHH HPHPHPHPHPH PPHHPHPHHPP PHPPH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Awan Berturus"),
            "PHPPHHHPPHP HPHPPPPPHPH PHPHHPHHPHP PPHPHHHPHPP HPHHPHPHHPH HPPHH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Awan Mengandung Hujan"),
            "PPHPHPPHHHP HPPHPHPPHPP HHHPHPHHHPH HPHHHPPHPHP PPHPPHPHHPH PHPPH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Awan Tunjung"),
            "PHHPPHPPHPP PPHHPHPHPPH HPPHHPHHPHH PHHHHPPHHHP PPHPPHPHHPP HHPPH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Bangkawat / Awan Menurus"),
            "PHHPPHHPPHP HHPPHHPPHHH HPPHHPPHPHP PPHHPPHHHPP PHHPPHPHPPH HHPPH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Bangkawat Kudung"),
            "PPHHPPHHHHP HPPHHPPHHPP HHPPHHPHPPH HHHPPHHPPHH HPPHPPPPHHP PPHHP"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Bintang Terhambur"),
            "HPHPHPHPHPH PHPHPHPHPHP HPHPHPHPHPH PHPHPHPHPHP HPHPHPHPHPH PHPHP"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Bunga Bercucuk"),
            "PPHHHHHHHPP PPPHHPHHPPP HPPPHPHPPPH HHPPPHPPPHH HHHPPHPPHHH HPPHH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Bunga Kambang Kiapu"),
            "HPHHPPPHHPH PPHPHPHPHPP HHPHPHPHPHH HPHPHPHPHPH PHPHPHPHPHP PPHPH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Bunga Kembang Berhujan"),
            "PPHPPHPPHHP HPHHPHHPHPP HHPPHHPHPHH PPHHPPHHPHP PHPHHPHPHPP HHHPP"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Bunga Sepapan"),
            "PPHPPHPPHPP PHHHPHPHHHP HHPPHHHPPHH PHPHHPPHPHP PPHPPHPHHPP HHHPH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Galap Madam"),
            "PPPPPHHHHHP HHHHHPPPPHP HPPPPHHHPHP HPHHHPPHPHP HPHPPHPHPHP HPHPH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Ikan Bernang"),
            "PHHHPPPHHHP HPPHHPHHPPH HPPPHPHPPPH HHPPHPHPPHH PHHHPHPHHHP PPPPH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Kalabutan"),
            "HPPHHPHHPPH PPHPHHHPHPP PHPHPHPHPHP HPHPPHPPHPH HHPPHPHPPHH PHHHP"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Kalabutan Berdarah Hitam"),
            "HHPPHPHPPHH HPHHPHPHHPH PHPPHPHPPHP PHPPHPHPPHP HPHHPHPHHPH PHPPH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Kalajangking 1"),
            "HPHPHPHPHPH PPHPPHPPHPP HHPHHPHHPHH PPHPPHPPHPP HPHPHHHPHPH PHPHH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Kalajangking 2"),
            "HPHHPHPHHPH PPHPPHPPHPP HHPHHPHHPHH HPHPPHPPHPH PPHPHPHPHPP HHPHP"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Kedudukan Dian"),
            "PHPPHPHPPHP HPHPHPHPHPH PHPHPHPHPHP PPHPHHHPHPP HHPHPHPHPHH PPHHH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Kina Putar"),
            "HPPHHPPHPHH HHPPHHPPHHP PHHPPHHPHPP HPPHPPHHPPH PPHHHPHPPHH PHHPP"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Lasung-Lasung"),
            "HPPPPHHHHPH PHHHHPPPHHP HHPPPHHPPHP HPPHHPHHPHP HPHHPPPHPHP HPHPP"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Lungun Sipugut"),
            "PPHPHPHPHPP PPHHPHPHHPP HHPHHPHHPHH PHHPPHPPHHP HPHPHPHPHPH PHPHP"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Mahligai Bergantung Bertali"),
            "PHPPHPHPPHP HHHPPHPPHHH PHPHPHPHPHP PPHHHPHHHPP HPPHPHPHPPH PHHPH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Mayang Piasau"),
            "PHHPPHHHPPP PHPHPPPHPHH PPHHPHPHHPH HHHPHPHPHHP HPPHPHPHPPP HPHPH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Pagar Batu"),
            "PHHPPHPPHHP HPPHHPHHPPH HPPHPHPHPPH PHHPHPHPHHP PHPHPHPHPHP HPHPH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Paha Ayam"),
            "PPPPPHHHHHP HPPPPHHHHPP HHPPPHHHPPP HHHPPHHPPPP HHHHPHPPPPP HHHHH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Paita"),
            "HPPHPHPHPPH PPHHHPHHHPP PHHPPHPPHHP HHPPPHPPPHH PHPPHHHPPHP HPHHH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Patah-Patah"),
            "HPHPPHPPHPH PPHPHHHPHPP HHHPHPHPHHH PPPHPHPHPPP PHHPPHPPHHP HHPHH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Peria Bulat"),
            "PPHHHPHHHPP PPPHHPHHPPP HPPPHHHPPPH HHPPPHPPPHH HHHPPHPPHHH PPHHH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Peria Buruk Hatinya"),
            "HPPHPHPHPPH PHPHHPHHPHP PPHPPHPPHPP HHPHPHPHPHH PHPPHHHPPHP HPHHH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Peria Buruk Sepanggal"),
            "HPPHPPPHHPH PPPHHHHHPPP HPHHHPPHHPP HHHPHPPPHHH PHPPPHPHHHP PHPPH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Peria Sejambangan"),
            "HPPHPHPHPPH PPHHHPHHHPP PHPPHPHPPHP HHPHPHPHPHH PHHPPHPPHHP HPPHH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Pucuk Rebung"),
            "HHPHPPPHPHH HPPHHPHHPPH PPPPHPHPPPP HHPPHHHPPHH PHHHPHPHHHP PPPHH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Puncak Mahligai"),
            "PHPHHPHHPHP HPHPPPPPHPH PHHHPPPHHHP HPHHHPHHHPH HPPHHPHHPPH PPPPP"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Saputangan Jatuh Ke Laut"),
            "HPPHHPHHPPH PPHHPHPHHPP PHPPHPHPPHP HHPHHPPHPHH HPHPPHPHHPH PHPPH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Sarimbangun"),
            "HPPHPHPHPPH PPHPPHPPHPP PHPHHHHHPHP HPHPPHPPHPH PPHPHHHPHPP HHHHH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Selidan Putar"),
            "PHHPHPPHHPP PPHHPPHHPPH HPPHPHHPPHH HHPPHHPPHHP PHHPPHPHPPH PPHHH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Si Pugut Menunggu Beringin Buruk Hatinya"),
            "HPHPPHPPHPH PHPHHPHHPHP HPHPHPHPHPH PHPHPHPHPHP PHHPPHPPHHP HPPHH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Sisir Sepapan"),
            "HPPHHPPHPPH PPHHPPHHHPP PHPHPHHPPHP HHPPHHPPHHH PHHPPHPHPPH PPHHH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Talapok Berantai"),
            "HPHHHPHHHPH PPPHPHPHPPP HPHPHPHPHPH HHPHPPPHPHH HPHPHPHPHPH PHPPP"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Talapok Sepapan"),
            "HPHHHPHHHPH PPPHPPPHPPP HPHPHPHPHPH HHPPPHPPPHH HPHPHHHPHPH PPPHH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Tekuyong Naik Pasang"),
            "PHPHPHPHPHP HPHPHPHPHPH PHPHPHPHPHP HPHPPHPPHPH PHPPHHHPPHP HPHHH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Tikus Naik Ke Durong"),
            "HPHHPPPHHPH PPPHPHPHPPP HPPHHPHHPPH HHHPHPHPHHH PPHHPHPHHPP PHPPH"
        ),
        new Pattern (
            // TRANSLATORS: Pattern name. Don't translate.
            N_("Tudung Dulang"),
            "PHPHPHPHPHP HPPHPHPHPPH PPHHPHPHHPP HHHPPHPPHHH PPPPHHHPPPP HHHHH"
        ),
        // TRANSLATORS: Pattern symmetry. You may translate to whatever conjures a highly symmetrical arrangement.
        new RandomPattern (N_("Gardens"),           " J J J O X OfJf"),
        // TRANSLATORS: Pattern symmetry. The foreground and the background form the same shape.
        new RandomPattern (N_("Day and Night"),     " d"),
        // TRANSLATORS: Pattern symmetry. You may translate to whatever conjures rotational symmetry.
        new RandomPattern (N_("Fans"),              " R R R * *fRf Rf*f"),
        // TRANSLATORS: Pattern symmetry. You may translate to whatever conjures reflectional symmetry.
        new RandomPattern (N_("Folds"),             " Rt Rt Rt *t RtJf JfRt"),
        // TRANSLATORS: Pattern symmetry. You may translate to whatever conjures diagonal symmetry.
        new RandomPattern (N_("Oblique Folds"),     " JL JL Xs JLJf JfJL"),
        // TRANSLATORS: Pattern symmetry.
        new RandomPattern (N_("Whatever"),          " RR RR RR *? ** *?*? d")
    };

    /**
     * Actual list of patterns. This list contains at least core_patterns.
     * Custom patterns can be added/removed through add_patterns.
     *
     * The patterns are listed in the following order :
     *      traditional patterns
     *      random patterns
     *      custom patterns
     *      blank pattern
     */
    Pattern[] patterns;

    /**
     * Most recently clicked pattern. Used to determine if a RandomPattern
     * is clicked more than once in sucession.
     */
    private RandomPattern? recent_pattern;

    /**
     * Ticket to ensure that at most one RandomPattern is hot and is cooling down.
     */
    private int cooling_ticket = 0;

    /**
     * Similar to gtk sensitive property.
     */
    private bool responsive = true;

    /**
     * True when programatically tweaking the pattern buttons.
     */
    private bool ignore_signal = false;

    /**
     * Invisible button to support unselect.
     */
    private Gtk.RadioButton invisible_button;

    /**
     * Visual container to hold list of patterns.
     */
    private Gtk.FlowBox pattern_box;

    public PatternSelector () {
        patterns = {};
        pattern_box = new Gtk.FlowBox ();
        pattern_box.set_vadjustment (this.vadjustment);
        pattern_box.selection_mode = Gtk.SelectionMode.NONE;
        pattern_box.valign = Gtk.Align.CENTER;
        pattern_box.max_children_per_line = 10;
        add (pattern_box);
        invisible_button = new Gtk.RadioButton (null);
        foreach (var pattern in core_patterns) {
            add_pattern (pattern);
        }
        set_policy (Gtk.PolicyType.NEVER, Gtk.PolicyType.ALWAYS);
    }

    /**
     * Change the size of all buttons according to the width of the container.
     * NOTE: overriding size_allocate is more reliable than connecting
     * to size_allocate signal.
     */
    int prev_dot_width;

    public override void size_allocate (Gtk.Allocation allocation) {
        var width = allocation.width;
        // Calculate dot_width to prefer 5 buttons per row
        int dot_width = (width / 5 / (BOARD_WIDTH + 4)).clamp (2, 12);
        if (dot_width != prev_dot_width) {
            prev_dot_width = dot_width;
            foreach (var pattern in patterns) {
                pattern.set_dot_width (dot_width);
            }
        }
        base.size_allocate (allocation);
    }

    /**
     * Add custom patterns, replacing any previous ones.
     */
    public void add_patterns (Pattern[] additional_patterns) {
        // Remove previous ones
        for (int i = core_patterns.length; i < patterns.length; i++) {
            // Remove the FlowBoxChild adapter component
            pattern_box. remove (pattern_box.get_child_at_index (core_patterns.length));
            patterns[i] = null;
        }
        patterns.length = core_patterns.length;
        
        // Add new ones
        foreach (var pattern in additional_patterns) {
            add_pattern (pattern);
        }

        // Add a blank custom pattern
        add_pattern (new BlankPattern ());

        pattern_box.show_all ();
    }

    /**
     * Add pattern to patterns
     * Initialise and add pattern.button to pattern_box
     */
    private void add_pattern (Pattern pattern) {
        var button = pattern.button;
        if (patterns.length > 0) pattern.set_dot_width (patterns[0].dot_width);
        button.set_group (invisible_button.get_group ());
        button.set_mode (false);  // On/off only. No intermediate state.
        button.border_width = 0;
        button.relief = Gtk.ReliefStyle.NONE;
        button.halign = Gtk.Align.CENTER;
        button.valign = Gtk.Align.CENTER;
        button.expand = false;
        // Use signal_clicked rather than signal_toggled to allow a pattern randomizer
        // to be clicked again (despite being already selected)
        var index = patterns.length;
        button.clicked.connect ((s) => {on_selection (index);});
        button.tooltip_text =
            (pattern is RandomPattern) ? _("Random Pattern: %s").printf (_(pattern.name)) :
            (pattern is CustomPattern) ? _("Custom Pattern") :
                                         _("Pattern: \"%s\"").printf (_(pattern.name));
        patterns += pattern;
        pattern_box.add (button);
    }

    /**
     * Select a random pattern. This can be a traditional pattern, or a pattern
     * generated by a randomizer, or a custom pattern.
     */
    public void random () {
        // Click on any pattern button except the last blank one
        int rnd = Random.int_range (0, patterns.length - 1);
        var pattern = patterns[rnd];
        unselect ();
        // Show selection visually
        ignore_signal = true;
        pattern.button.clicked ();
        scroll_to_view (pattern.button);
        ignore_signal = false;
        // Activate selection
        if (pattern is RandomPattern) (pattern as RandomPattern).next_random ();
        pattern_changed_signal (pattern.dots);
    }

    /**
     * Handler for button click.
     */
    private void on_selection (int pattern_num) {
        assert (pattern_num >= 0 && pattern_num < patterns.length);
        if (ignore_signal || !responsive) return;
        var pattern = patterns[pattern_num];
        bool hot = (pattern == recent_pattern);
        recent_pattern = pattern as RandomPattern;

        // Ignore de-selection event. (Is this ever triggered?)
        if (pattern.button.active == false) return;

        // Turn on the selected random-pattern button, and randomize if it is selected while "hot"
        if (pattern is RandomPattern) {
            if (hot) (pattern as RandomPattern).next_random ();
            cool_down.begin (pattern as RandomPattern);
            // Random pattern will only emit its change signal when it is cooled to 0.
        }
        else if (pattern is BlankPattern) {
            pattern_changed_signal (null);
        }
        else {
            pattern_changed_signal (pattern.dots);
        }
    }

    private async void cool_down (RandomPattern pattern) {
        var timer = new Timer ();
        var my_cooling_ticket = ++cooling_ticket;
        while (true) {
            var temperature = 3.0 - double.min (double.max (timer.elapsed (), 2.0), 3.0);
            if (pattern != recent_pattern) temperature = 0;
            pattern.temperature = temperature;
            // Random pattern will only emit its change signal when it is cooled to 0.
            if (temperature == 0) {
                if (my_cooling_ticket == cooling_ticket && pattern == recent_pattern) {
                    recent_pattern = null;
                    pattern_changed_signal (pattern.dots);
                }
                return;
            }
            if (my_cooling_ticket != cooling_ticket) return;
            yield Util.wait_async (200);
        }
    }

    /**
     * Unselect all buttons by selecting the invisible one.
     */
    public void unselect () {
        ignore_signal = true;
        invisible_button.set_active (true);
        ignore_signal = false;
    }

    /**
     * Ignore click signals from the user, but not self-clicks.
     */
    public new void set_sensitive (bool b) {
        recent_pattern = null;
        responsive = b;
        foreach (var pattern in patterns) pattern.button.set_sensitive (true);
    }

    /**
     * Apply set_sensitive to all child, except the currently selected pattern.
     * This has the effect of highlighting the current selection more prominently
     * then the typical pressed look.
     */
    public void highlight_selected () {
        foreach (var pattern in patterns) {
            pattern.button.set_sensitive (pattern.button.get_active ());
        }
    }

    /**
     * Find a pattern. Set the selection to this pattern.
     * Trigger pattern_changed_signal, even if the pattern is not found.
     */
    public void find_and_click (string dots) {
        var compressed_dots = dots.replace (" ", "");
        for (int i=0; i < patterns.length; i++) {
            if (!(patterns[i] is RandomPattern) && patterns[i].dots.replace (" ", "") == compressed_dots) {
                // Select the pattern
                ignore_signal = true;
                patterns[i].button.set_active (true);
                ignore_signal = false;
                scroll_to_view (patterns[i].button);
                break;
            }
        }
        pattern_changed_signal (dots);
    }

    /**
     * Scroll the given widget to view. If possible show it at the centre.
     */
    void scroll_to_view (Gtk.Widget widget) {
        int middle = widget.get_allocated_height () / 2;
        int x, y;
        widget.translate_coordinates (pattern_box, 0, middle, out x, out y);
        vadjustment.clamp_page (y - vadjustment.page_size / 2, y + vadjustment.page_size / 2);
    }
}//class
}//namespace
// vim: tabstop=4: expandtab: textwidth=100: autoindent:
