/*******************************************************************************
*                                                                              *
* Copyright (C) 1997 Steve Haehn                                               *
*                                                                              *
* This is free software; you can redistribute it and/or modify it under the    *
* terms of the GNU General Public License as published by the Free Software    *
* Foundation; either version 2 of the License, or (at your option) any later   *
* version.                                                                     *
*                                                                              *
* This software is distributed in the hope that it will be useful, but WITHOUT *
* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or        *
* FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License *
* for more details.                                                            *
*                                                                              *
* You should have received a copy of the GNU General Public License along with *
* software; if not, write to the Free Software Foundation, Inc., 59 Temple     *
* Place, Suite 330, Boston, MA  02111-1307 USA                                 *
*                                                                              *
*******************************************************************************/

/*==============================================================================
**
** "Where" is given file located?
**
** DESCRIPTION:  
**
**  This program attempts to locate the given files and print out the complete
**  path to those files. Unless the user specifies a specific environment 
**  variable to use as fuel for where to look for the existence of the files,
**  it uses "path" as its default. The path values may be separated by
**  space, tab, or colon characters. The program also will ignore any "-I"
**  flags prepended to the paths, in case one wants to use compiler argument
**  include file syntax.
**
*/
#if 0
** REVISION HISTORY:
**
**       Date       Author/Comment
**   Dec 01, 2000   Steven Haehn
**                  Now allow environment variables in directory path names.
**
**   Nov 17, 1997   Steven Haehn
**                  Original version.
**==============================================================================
#endif

/*============================================================================*/
/*====================  OPERATING SYSTEM INCLUDE FILES  ======================*/
/*============================================================================*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>

typedef int bool_t;

#define FALSE (bool_t)0
#define TRUE  (bool_t)!FALSE

#define SEPARATOR " \t:"
#define UB_STRING 255
#define EOS '\0'
#define FPATH_SEPARATOR '/'
#define ENVIRON_IND     "$"
#define ENV_NAME "_ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789abcdefghijklmnopqrstuvwxyz"

/*============================================================================*/
/*                             PROGRAM PROTOTYPES                             */
/*============================================================================*/
 
#ifdef NEED_GETOPT_PROTO_TYPE
int   getopt( int, char **, char * );
#endif

/*============================================================================*/
/*================================= PROGRAMS =================================*/
/*============================================================================*/

#ifdef NEED_STRTOK_R
 
char * strtok_r
(
    char  *inoutp_src,       /* Source string for tokenizing, or NULL */
    char  *inp_separators,   /* List of token separators              */
    char **inoutpp_next_pos  /* Remainder of initial source string    */
)
{
    char           *crnt_token, *next_separator;
    int             sepLen;
 
    /*------------------------------------------
    * Use given source string, or when NULL use
    * remaining string from previous call.
    *------------------------------------------*/
    crnt_token = (inoutp_src) ? inoutp_src : *inoutpp_next_pos;
    
    /*---------------------------------------------------
    * When there is a possibility of a token present ...
    *---------------------------------------------------*/
    if (crnt_token != NULL)
    {
        /*-------------------------------------------
        * ... skip over all leading token separators 
        *     prior to determining position of token.
        *-------------------------------------------*/
        sepLen = strspn(crnt_token, inp_separators);
        
        if (crnt_token[sepLen] == EOS)
            crnt_token = *inoutpp_next_pos = NULL;
        else
            crnt_token = crnt_token + sepLen;
 
        /*---------------------------------------------------------
        * When there is a token present, locate the next separator
        * (if any) and prepare for retrieval of subsequent tokens.
        *--------------------------------------------------------*/
        if (crnt_token != NULL)
        {
            next_separator = strpbrk(crnt_token, inp_separators);
    
            if (next_separator == NULL)
                *inoutpp_next_pos = NULL;  /* no more tokens */
            else
            {
                *inoutpp_next_pos = next_separator + 1;
                *next_separator   = EOS;   /* terminate token string */
            }
        }
    }

    return ( crnt_token );
}

#endif /* NEED_STRTOK_R */

/*----------------------------------------------------------------------------*/

void eval_environ_names(

    char * inp_file,
    char * outp_filename,
    int    in_name_size
)
{
    char * cp     = inp_file;
    char * name_p = outp_filename;
    char * end_p  = name_p + in_name_size;
    
    while( cp && *cp != EOS )
    {
        char * found_env = strpbrk( cp, ENVIRON_IND );
        
        if( found_env == NULL )
        {
            /*---------------------------------------
            * Copy remainder of given name to result.
            *---------------------------------------*/
            while( *cp != EOS )
                *(name_p++) = *(cp++);
        }
        else
        {
            char env_name[UB_STRING];
            int i;
            
            /*-----------------------------------------------------
            * Copy characters up to environment variable indicator.
            *-----------------------------------------------------*/
            while( cp < found_env && name_p < end_p )
                *(name_p++) = *(cp++);
            
            found_env++;
            cp = found_env + strspn( found_env, ENV_NAME );
            
            /*----------------------------------
            * Extract environment variable name.
            *----------------------------------*/
            for( i=0; found_env < cp; i++ )
                env_name[i] = *(found_env++);
            env_name[i] = EOS;
            
            /*------------------------------
            * Evaluate environment variable.
            *------------------------------*/
            found_env = getenv( env_name );
            
            if( found_env != NULL )
            {
                /*---------------------------------
                * Copy environment value to result.
                *---------------------------------*/
                while( *found_env != EOS && name_p < end_p )
                    *(name_p++) = *(found_env++);
            }
        }
    }
    
    *name_p = EOS;
}

/*----------------------------------------------------------------------------*/

char *
remove_file_path
(
    char * inp_file_path
)
{
    char * file_name = strrchr( inp_file_path, FPATH_SEPARATOR );
    
    return ( file_name ) ? file_name+1 : inp_file_path;
}

/*----------------------------------------------------------------------------*/

void show_usage(

    char * inp_pgm_name
)
{
    fprintf( stderr, "\nUsage: %s [-f][-p environmentName] file ...\n", inp_pgm_name );
    fprintf( stderr, "\n" );
    fprintf( stderr, "    -f means only report on first location of file.\n" );
    fprintf( stderr, "    -p is for specifying an alternate path environment variable.\n" );
    fprintf( stderr, "\n" );
    fprintf( stderr, "    The default environment variable used for file paths\n" );
    fprintf( stderr, "    is 'PATH'. Values of the file path environment variable\n" );
    fprintf( stderr, "    can be separated by spaces or colons. The -I syntax typically\n" );
    fprintf( stderr, "    used for compiler \"include\" files is also understood.\n" );
    fprintf( stderr, "\n" );
    exit( 1 );

}

/*----------------------------------------------------------------------------*/

int main( int argc, char * argv[] )
{
    extern char *optarg;
    extern int   optind;
    
    char * pgm_name   = remove_file_path( argv[0] );
    char * paths      = getenv( "PATH" );
    int    result     = 1;
    bool_t first_only = FALSE;
    int    c;

    /*----------------------------
    * Gather command line options.
    *----------------------------*/
    while( (c = getopt( argc, argv, "fp:" )) != -1 )
    {
        switch( c )
        {
          case 'f':
            first_only = TRUE;
            break;
          
          case 'p':
            paths = getenv( optarg );

            if( paths == NULL )
            {
                fprintf( stderr,
                   "%s: %s is unavailable in program environment.\n", 
                   pgm_name, optarg );
                exit( 1 );
            }
            break;

          case '?':
            show_usage( pgm_name );
            break;
        }
    }
        
    if( optind >= argc || paths == NULL )
    {
        if( paths == NULL )
        {
            fprintf( stderr, 
               "%s: default \"path\" environment variable is unavailable.\n",
               pgm_name );
        }
        
        show_usage( pgm_name );  /* Not enough arguments given */
    }
    else
    {
        /*------------------------------------------
        * For each file found on the command line...
        *------------------------------------------*/
        while( optind < argc )
        {
            char * filename  = argv[ optind++ ];
            char * path_copy = strdup( paths );
            char * remainder = path_copy;
            char * path;
            
            /*---------------------------------------------------------
            * ... try to locate it in one of the mentioned directories.
            *---------------------------------------------------------*/
            while( ( path = strtok_r( NULL, SEPARATOR, &remainder ) ) )
            {
                struct stat status;
                char   full_file_name[ UB_STRING ];
                char   full_path[ UB_STRING ];
                
                if( strncmp( path, "-I", 2 ) == 0 )  /* skip over -I */
                    path += 2;
                
                eval_environ_names( path, full_path, UB_STRING );
                strcpy( full_file_name, full_path );
                strcat( full_file_name, "/" );
                strcat( full_file_name, filename );
                
                /*------------------------------------
                * If file exists, print full path out.
                *------------------------------------*/
                if( stat( full_file_name, &status ) == 0 )
                {
                    printf( "%s\n", full_file_name );
                    result = 0; /* found a file */
                    if( first_only ) break;
                }
            }
            
            free ( path_copy );
        }
    }
    
    return result;
}
