/*
 * stats.c
 *
 * Copyright (C) 2002 Thomas Graf <tgr@reeler.org>
 *
 * This file belongs to the nstats package, see COPYING for more information.
 *
 */

#include <stdio.h>
#include <curses.h>
#include <netinet/in.h>
#include <netinet/ether.h>
#include <netinet/ip.h>
#include <netinet/ip6.h>
#include <netinet/icmp6.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <netinet/ip_icmp.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <sys/ioctl.h>

#include "stats.h"
#include "curs_util.h"
#include "util.h"

#define ST stats
#define IPV4 stats.ip.ipv4
#define IPV6 stats.ip.ipv6
#define ICMP6 IPV6.icmp
#define DESTU6 ICMP6.dst_unreach
#define TEXC6 ICMP6.time_excd
#define PARA6 ICMP6.param_prob
#define ICMP4 IPV4.icmp
#define ICMP4DU ICMP4.types.dest_unreach
#define TEX ICMP4.types.time_exceeded
#define TCP stats.ip.tcp
#define UDP stats.ip.udp
#define TOS stats.ip.tos
#define ECN stats.ip.ecn
#define IRED ICMP4.types.redirect
#define FRAG stats.ip.frag

enum {
    ethernet,
    ip_overview,
    ecn,
    tcp,
    ip4,
    ip6
} m_ip = ip_overview;

int
stats_handle_input(int ch)
{

    switch (ch) {

        case 'e': /* Ethernet */
            clear();
            m_ip = ethernet;
            return 1;
        
        case 'i':   /* IP */
            clear();
            m_ip = ip_overview;
            return 1;

        case 'E':    /* ECN */
            clear();
            m_ip = ecn;
            return 1;

        case 't':  /* TCP */
            clear();
            m_ip = tcp;
            return 1;

        case '4': /* IPv4 */
            clear();
            m_ip = ip4;
            return 1;

        case '6': /* IPv6 */
            clear();
            m_ip = ip6;
            return 1;
    }

    return 0;
}

void
print_header(const char *str)
{
    print_full_line("%-24s #      %%        bytes      %%     avg         rate", str);
}

void
print_line(const char *str, cnt_t p, cnt_t pt, cnt_t b, cnt_t bt, cnt_t rate)
{
    char *u;
    float res;

    res = sumup(rate, &u);
    
    print_full_line("%-16s%10llu %6.2f %12llu %6.2f %7.1f %9.1f %s",
        str,
        p, (p*100) / (float) pt,
        b, (b*100) / (float) bt,
        (p != 0) ? b / (float) p : 0,
        res, u);
}

void
print_ether(void)
{
    NEXT_ROW;
    print_header("Ethernet");
    NEXT_ROW;
    print_sep();
    NEXT_ROW;
    print_line("IP:", ST.ip.cnt, ST.cnt, ST.ip.bs, ST.bs, ST.ip.bs_rate);
    NEXT_ROW;
    print_line("ARP:", ST.arp.cnt, ST.cnt, ST.arp.bs, ST.bs, ST.arp.bs_rate);
    NEXT_ROW;
    print_line("RARP", ST.rarp.cnt, ST.cnt, ST.rarp.bs, ST.bs, ST.rarp.bs_rate);
    NEXT_ROW;
    print_line("Other:", ST.other.cnt, ST.cnt, ST.other.bs, ST.bs, ST.other.bs_rate);
    NEXT_ROW;
    print_sep();
    NEXT_ROW;
    print_line("Total:", ST.cnt, ST.cnt, ST.bs, ST.bs, ST.bs_rate);
}

void
print_ip(void)
{
    NEXT_ROW;
    print_header("IP");
    NEXT_ROW;
    print_sep();
    NEXT_ROW;
    print_line("IPv4:", IPV4.cnt, ST.ip.cnt, IPV4.bs, ST.ip.bs, IPV4.bs_rate);
    NEXT_ROW;
    print_line("IPv6:", IPV6.cnt, ST.ip.cnt, IPV6.bs, ST.ip.bs, IPV6.bs_rate);
    NEXT_ROW;
    print_full_line("");
    NEXT_ROW;
    print_line("Total:", ST.ip.cnt, ST.ip.cnt, ST.ip.bs, ST.ip.bs, ST.ip.bs_rate);
    
    NEXT_ROW;
    NEXT_ROW;
    print_header("Protocols");
    NEXT_ROW;
    print_sep();
    NEXT_ROW;
    print_line("TCP:", TCP.cnt, IPV4.cnt, TCP.bs, IPV4.bs, TCP.bs_rate);
    NEXT_ROW;
    print_line("UDP:", UDP.cnt, IPV4.cnt, UDP.bs, IPV4.bs, UDP.bs_rate);
    NEXT_ROW;
    print_line("ICMP:", ICMP4.cnt, IPV4.cnt, ICMP4.bs, IPV4.bs, ICMP4.bs_rate);
    NEXT_ROW;
    print_line("ICMP6:", ICMP6.cnt, IPV6.cnt, ICMP6.bs, IPV6.bs, ICMP6.bs_rate);
    NEXT_ROW;
    print_line("Other:", ST.ip.other.cnt, IPV4.cnt, ST.ip.other.bs, IPV4.bs, ST.ip.other.bs_rate);

    NEXT_ROW;
    NEXT_ROW;
    print_header("Type of Service");
    NEXT_ROW;
    print_sep();
    NEXT_ROW;
    print_line("Low-Delay:", TOS.lowdelay, IPV4.cnt, TOS.s_lowdelay, IPV4.bs, .0);
    NEXT_ROW;
    print_line("Throughput:", TOS.maxthroughput, IPV4.cnt, TOS.s_maxthroughput, IPV4.bs, .0);
    NEXT_ROW;
    print_line("Reliability:", TOS.reliability, IPV4.cnt, TOS.s_reliability, IPV4.bs, .0);
    NEXT_ROW;
    print_line("Low-Cost:", TOS.lowcost, IPV4.cnt, TOS.s_lowcost, IPV4.bs, .0);
    NEXT_ROW;
    print_line("None:", TOS.none, IPV4.cnt, TOS.s_none, IPV4.bs, .0);
}


void
print_ecn(void)
{
    NEXT_ROW;
    print_header("ECN");
    NEXT_ROW;
    print_sep();
    NEXT_ROW;
    print_line("Not-ECT:", ECN.not_ect, ST.ip.cnt, ECN.s_not_ect, ST.ip.bs, .0);
    NEXT_ROW;
    print_line("ECT(1):", ECN.ect_1, ST.ip.cnt, ECN.s_ect_1, ST.ip.bs, .0);
    NEXT_ROW;
    print_line("ECT(0):", ECN.ect_0, ST.ip.cnt, ECN.s_ect_0, ST.ip.bs, .0);
    NEXT_ROW;
    print_line("CE:", ECN.ce, ST.ip.cnt, ECN.s_ce, ST.ip.bs, .0);
    NEXT_ROW;
    print_line("CWR (TCP):", ECN.cwr, ST.ip.cnt, ECN.s_cwr, ST.ip.bs, .0);
    NEXT_ROW;
    print_line("ECE (TCP):", ECN.ece, ST.ip.cnt, ECN.s_ece, ST.ip.bs, .0);
    NEXT_ROW;
    print_line("ECN-Setup (TCP):", ECN.ecn_setup, ST.ip.cnt, ECN.s_ecn_setup,
        ST.ip.bs, .0);
}

void
print_tcp(void)
{
    NEXT_ROW;
    print_header("TCP Options (IPv4)");
    NEXT_ROW;
    print_sep();
    NEXT_ROW;
    print_line("URG:", TCP.opts.urg, TCP.cnt, TCP.opts.s_urg, TCP.bs, .0);
    NEXT_ROW;
    print_line("ACK:", TCP.opts.ack, TCP.cnt, TCP.opts.s_ack, TCP.bs, .0);
    NEXT_ROW;
    print_line("PSH:", TCP.opts.psh, TCP.cnt, TCP.opts.s_psh, TCP.bs, .0);
    NEXT_ROW;
    print_line("RST:", TCP.opts.rst, TCP.cnt, TCP.opts.s_rst, TCP.bs, .0);
    NEXT_ROW;
    print_line("SYN:", TCP.opts.syn, TCP.cnt, TCP.opts.s_syn, TCP.bs, .0);
    NEXT_ROW;
    print_line("FIN:", TCP.opts.fin, TCP.cnt, TCP.opts.s_fin, TCP.bs, .0);
}


void
print_ip4(void)
{
    NEXT_ROW;
    print_header("ICMP Types (IPv4)");
    NEXT_ROW;
    print_sep();
    NEXT_ROW;
    print_line("Echo-Request:", ICMP4.types.echo_request, ICMP4.cnt,
            ICMP4.types.s_echo_request, ICMP4.bs, .0);
    NEXT_ROW;
    print_line("Echo-Reply:", ICMP4.types.echo_reply, ICMP4.cnt,
        ICMP4.types.s_echo_reply, ICMP4.bs, .0);
    NEXT_ROW;
    print_line("Dest Unreach:", ICMP4DU.cnt, ICMP4.cnt,
            ICMP4DU.bs, ICMP4.bs, .0);
    NEXT_ROW;
    print_line("Source Quench:", ICMP4.types.source_quench, ICMP4.cnt,
             ICMP4.types.s_source_quench, ICMP4.bs, .0);
    NEXT_ROW;
    print_line("Redirect:", ICMP4.types.redirect.cnt, ICMP4.cnt,
             ICMP4.types.redirect.bs, ICMP4.bs, .0);
    NEXT_ROW;
    print_line("Time Exceeded:", ICMP4.types.time_exceeded.cnt, ICMP4.cnt,
             ICMP4.types.time_exceeded.bs, ICMP4.bs, .0);
    NEXT_ROW;
    print_line("Parameter Prob:", ICMP4.types.para_prob, ICMP4.cnt,
             ICMP4.types.s_para_prob, ICMP4.bs, .0);
    NEXT_ROW;
    print_line("Timetstamp:", ICMP4.types.timestamp, ICMP4.cnt,
             ICMP4.types.s_timestamp, ICMP4.bs, .0);
    NEXT_ROW;
    print_line("Timetstamp Rpy:", ICMP4.types.timestamp_reply, ICMP4.cnt,
             ICMP4.types.s_timestamp_reply, ICMP4.bs, .0);
    NEXT_ROW;
    print_line("Info Request:", ICMP4.types.info_request, ICMP4.cnt,
             ICMP4.types.s_info_request, ICMP4.bs, .0);
    NEXT_ROW;
    print_line("Info Reply:", ICMP4.types.info_reply, ICMP4.cnt,
             ICMP4.types.s_info_reply, ICMP4.bs, .0);
    NEXT_ROW;
    print_line("Address:", ICMP4.types.address, ICMP4.cnt,
             ICMP4.types.s_address, ICMP4.bs, .0);
    NEXT_ROW;
    print_line("Address Reply:", ICMP4.types.address_reply, ICMP4.cnt,
             ICMP4.types.s_address_reply, ICMP4.bs, .0);
    NEXT_ROW;
    print_line("Other:", ICMP4.types.other, ICMP4.cnt,
             ICMP4.types.s_other, ICMP4.bs, .0);
}

void
print_ip6(void)
{
    NEXT_ROW;
    print_header("IPv6");
    NEXT_ROW;
    print_sep();
    NEXT_ROW;
    print_line("Flow-Label:", IPV6.fl, IPV6.cnt, IPV6.s_fl, IPV6.bs, .0);
    NEXT_ROW;
    print_line("Traffic-Class:", IPV6.tc, IPV6.cnt, IPV6.s_tc, IPV6.bs, .0);
    NEXT_ROW;
    NEXT_ROW;
    print_header("ICMP6 (IPv6)");
    NEXT_ROW;
    print_sep();
    NEXT_ROW;
    print_line("Dest. Unreach.", ICMP6.dst_unreach.cnt, ICMP6.cnt,
            ICMP6.dst_unreach.bs, ICMP6.bs, .0);
    NEXT_ROW;
    print_line("Time Exceeded:", ICMP6.time_excd.cnt, ICMP6.cnt,
            ICMP6.time_excd.bs, ICMP6.bs, .0);
    NEXT_ROW;
    print_line("Param. Prob:", ICMP6.param_prob.cnt, ICMP6.cnt,
            ICMP6.param_prob.bs, ICMP6.bs, .0);
    NEXT_ROW;
    print_line("Pkt too big:", ICMP6.pkt_too_big, ICMP6.cnt,
            ICMP6.s_pkt_too_big, ICMP6.bs, .0);
    NEXT_ROW;
    print_line("Echo Request:", ICMP6.echo_request, ICMP6.cnt,
            ICMP6.s_echo_request, ICMP6.bs, .0);
    NEXT_ROW;
    print_line("Echo Reply:", ICMP6.echo_reply, ICMP6.cnt,
            ICMP6.s_echo_reply, ICMP6.bs, .0);
    NEXT_ROW;
    print_line("MbrShip Query:", ICMP6.mbrship_query, ICMP6.cnt,
            ICMP6.s_mbrship_query, ICMP6.bs, .0);
    NEXT_ROW;
    print_line("MbrShip Report:", ICMP6.mbrship_report, ICMP6.cnt,
            ICMP6.s_mbrship_report, ICMP6.bs, .0);
    NEXT_ROW;
    print_line("MbrShip Reduct.:", ICMP6.mbrship_reduction, ICMP6.cnt,
            ICMP6.s_mbrship_reduction, ICMP6.bs, .0);
    NEXT_ROW;
    print_line("ND Rtr Solicit:", ICMP6.nd_router_solicit, ICMP6.cnt,
            ICMP6.s_nd_router_solicit, ICMP6.bs, .0);
    NEXT_ROW;
    print_line("ND Rtr Advert:", ICMP6.nd_router_advert, ICMP6.cnt,
            ICMP6.s_nd_router_advert, ICMP6.bs, .0);
    NEXT_ROW;
    print_line("ND Neig Solicit:", ICMP6.nd_neighbor_solicit, ICMP6.cnt,
            ICMP6.s_nd_neighbor_solicit, ICMP6.bs, .0);
    NEXT_ROW;
    print_line("ND Neig Advert:", ICMP6.nd_neighbor_advert, ICMP6.cnt,
            ICMP6.s_nd_neighbor_advert, ICMP6.bs, .0);
    NEXT_ROW;
    print_line("ND Redirect:", ICMP6.nd_redirect, ICMP6.cnt,
            ICMP6.s_nd_redirect, ICMP6.bs, .0);
    NEXT_ROW;
    print_line("Other:", ICMP6.other, ICMP6.cnt, ICMP6.s_other, ICMP6.bs, .0);
}

void
stats_draw(void)
{
    switch (m_ip) {

        case ethernet:
            print_top("Ethernet");
            print_ether();
            break;
        
        case ip_overview:
            print_top("Internet Protocol");
            print_ip();
            break;

        case ecn:
            print_top("ECN");
            print_ecn();
            break;

        case tcp:
            print_top("TCP");
            print_tcp();
            break;

        case ip4:
            print_top("IP4");
            print_ip4();
            break;

        case ip6:
            print_top("IP6");
            print_ip6();
            break;
    }

    print_to_end();

    move ( LINES-2, 0);
    print_full_line("Modes: i:IP e:Ethernet t:TCP E:ECN 4:IPv4 6:IPv6");
}

void
stats_calc_rate(void)
{
    time_t now = time(0);
    time_t delta;

    if (!stats.last_rate_refresh)
        stats.last_rate_refresh = now;

    delta = (now - stats.last_rate_refresh);

    if (delta >= 3)
    {

#define REFRESH_RATE(prefix) {                           \
    if (!prefix.bs_old)                                  \
        prefix.bs_old = prefix.bs;                       \
    prefix.bs_rate += (prefix.bs - prefix.bs_old);       \
    prefix.bs_rate /= (delta + 1);                       \
    prefix.bs_old = prefix.bs;                           \
}

        REFRESH_RATE ( stats                  );
        REFRESH_RATE ( stats.arp              );
        REFRESH_RATE ( stats.rarp             );
        REFRESH_RATE ( stats.other            );
        REFRESH_RATE ( stats.ip               );
        REFRESH_RATE ( stats.ip.tcp           );
        REFRESH_RATE ( stats.ip.udp           );
        REFRESH_RATE ( stats.ip.other         );
        REFRESH_RATE ( stats.ip.ipv4          );
        REFRESH_RATE ( stats.ip.ipv4.icmp     );
        REFRESH_RATE ( stats.ip.ipv6          );
        REFRESH_RATE ( stats.ip.ipv6.icmp     );

        stats.last_rate_refresh = now;
    }
}
