/*
 * packet.c
 *
 * Copyright (C) 2002 Thomas Graf <tgr@reeler.org>
 *
 * This file belongs to the nstats package, see COPYING for more information.
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <pcap.h>
#include <netdb.h>
#include <unistd.h>
#include <signal.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netinet/ether.h>
#include <netinet/ip.h>
#include <netinet/ip6.h>
#include <netinet/icmp6.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <netinet/ip_icmp.h>
#include <arpa/inet.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <net/if.h>

#include "packet.h"
#include "conn.h"
#include "stats.h"

struct ifconf ifc;

void print_output(void);

void
handle_tcp(const struct pcap_pkthdr *hdr, const u_char *packet, int offset,
        char *saddr, char *daddr, uint16_t type)
{
    int sp, dp, e1, e2;
    struct tcphdr *tptr;

    stats.ip.tcp.cnt++;
    stats.ip.tcp.bs += hdr->len;
    tptr = (struct tcphdr *) (packet + offset);
    sp = ntohs(tptr->source);
    dp = ntohs(tptr->dest);

    handle_port(sp, dp, saddr, daddr, type, hdr);

    if (tptr->urg) {
        stats.ip.tcp.opts.urg++;
        stats.ip.tcp.opts.s_urg += hdr->len;
    }

    if (tptr->ack) {
        stats.ip.tcp.opts.ack++;
        stats.ip.tcp.opts.s_ack += hdr->len;
    }

    if (tptr->psh) {
        stats.ip.tcp.opts.psh++;
        stats.ip.tcp.opts.s_psh += hdr->len;
    }

    if (tptr->rst) {
        stats.ip.tcp.opts.rst++;
        stats.ip.tcp.opts.s_rst += hdr->len;
    }

    if (tptr->syn) {
        stats.ip.tcp.opts.syn++;
        stats.ip.tcp.opts.s_syn += hdr->len;
    }

    if (tptr->fin) {
        stats.ip.tcp.opts.fin++;
        stats.ip.tcp.opts.s_fin += hdr->len;
    }

#if __BYTE_ORDER == __LITTLE_ENDIAN
    e1 = 0x01;
    e2 = 0x03;
#else
    e1 = 0x03;
    e2 = 0x01;
#endif

    if (tptr->res2 & e1 && tptr->res2 & e2) {
        stats.ip.ecn.ecn_setup++;
        stats.ip.ecn.s_ecn_setup += hdr->len;
    } else if (tptr->res2 & e1) {
        stats.ip.ecn.cwr++;
        stats.ip.ecn.s_cwr += hdr->len;
    } else if (tptr->res2 & e2) {
        stats.ip.ecn.ece++;
        stats.ip.ecn.s_ece += hdr->len;
    }
}

void
handle_udp(const struct pcap_pkthdr *hdr, const u_char *packet, int offset,
        char *saddr, char *daddr, uint16_t type)
{
    int sp, dp;
    struct udphdr *uptr;

    stats.ip.udp.cnt++;
    stats.ip.udp.bs += hdr->len;
    uptr = (struct udphdr *) (packet + offset);
    sp = ntohs(uptr->source);
    dp = ntohs(uptr->dest);

    if (sp == 3)
        printf("%d\n",dp);

    if (dp == 3)
        printf("%d\n", sp);
    handle_port(sp, dp, saddr, daddr, type, hdr);
}

void
handle_icmp6(const struct pcap_pkthdr *hdr, const u_char *packet, int offset)
{
    struct icmp6_hdr *icptr;

    icptr = (struct icmp6_hdr *) (packet + offset);

    stats.ip.ipv6.icmp.cnt++;
    stats.ip.ipv6.icmp.bs += hdr->len;

    switch(icptr->icmp6_type) {
        case ICMP6_DST_UNREACH:
            stats.ip.ipv6.icmp.dst_unreach.cnt++;
            stats.ip.ipv6.icmp.dst_unreach.bs += hdr->len;

            switch(icptr->icmp6_code) {
                case ICMP6_DST_UNREACH_NOROUTE:
                    stats.ip.ipv6.icmp.dst_unreach.noroute++;
                    stats.ip.ipv6.icmp.dst_unreach.s_noroute += hdr->len;
                    break;

                case ICMP6_DST_UNREACH_ADMIN:
                    stats.ip.ipv6.icmp.dst_unreach.admin++;
                    stats.ip.ipv6.icmp.dst_unreach.s_admin += hdr->len;
                    break;

                case ICMP6_DST_UNREACH_NOTNEIGHBOR:
                    stats.ip.ipv6.icmp.dst_unreach.notneighbor++;
                    stats.ip.ipv6.icmp.dst_unreach.s_notneighbor += hdr->len;
                    break;

                case ICMP6_DST_UNREACH_ADDR:
                    stats.ip.ipv6.icmp.dst_unreach.addr++;
                    stats.ip.ipv6.icmp.dst_unreach.s_addr += hdr->len;
                    break;

                case ICMP6_DST_UNREACH_NOPORT:
                    stats.ip.ipv6.icmp.dst_unreach.noport++;
                    stats.ip.ipv6.icmp.dst_unreach.s_noport += hdr->len;
                    break;
            }
            break;

        case ICMP6_TIME_EXCEEDED:
            stats.ip.ipv6.icmp.time_excd.cnt++;
            stats.ip.ipv6.icmp.time_excd.bs += hdr->len;

            switch (icptr->icmp6_code) {
                case ICMP6_TIME_EXCEED_TRANSIT:
                    stats.ip.ipv6.icmp.time_excd.transit++;
                    stats.ip.ipv6.icmp.time_excd.s_transit += hdr->len;
                    break;

                case ICMP6_TIME_EXCEED_REASSEMBLY:
                    stats.ip.ipv6.icmp.time_excd.reassembly++;
                    stats.ip.ipv6.icmp.time_excd.s_reassembly += hdr->len;
                    break;
            }
            break;

        case ICMP6_PARAM_PROB:
            stats.ip.ipv6.icmp.param_prob.cnt++;
            stats.ip.ipv6.icmp.param_prob.bs += hdr->len;

            switch(icptr->icmp6_code) {
                case ICMP6_PARAMPROB_HEADER:
                    stats.ip.ipv6.icmp.param_prob.header++;
                    stats.ip.ipv6.icmp.param_prob.s_header += hdr->len;
                    break;

                case ICMP6_PARAMPROB_NEXTHEADER:
                    stats.ip.ipv6.icmp.param_prob.nextheader++;
                    stats.ip.ipv6.icmp.param_prob.s_nextheader += hdr->len;
                    break;

                case ICMP6_PARAMPROB_OPTION:
                    stats.ip.ipv6.icmp.param_prob.option++;
                    stats.ip.ipv6.icmp.param_prob.s_option += hdr->len;
                    break;
            }
            break;

        case ICMP6_PACKET_TOO_BIG:
            stats.ip.ipv6.icmp.pkt_too_big++;
            stats.ip.ipv6.icmp.s_pkt_too_big += hdr->len;
            break;

        case ICMP6_ECHO_REQUEST:
            stats.ip.ipv6.icmp.echo_request++;
            stats.ip.ipv6.icmp.s_echo_request += hdr->len;
            break;

        case ICMP6_ECHO_REPLY:
            stats.ip.ipv6.icmp.echo_reply++;
            stats.ip.ipv6.icmp.s_echo_reply += hdr->len;
            break;

        case ICMP6_MEMBERSHIP_QUERY:
            stats.ip.ipv6.icmp.mbrship_query++;
            stats.ip.ipv6.icmp.s_mbrship_query += hdr->len;
            break;

        case ICMP6_MEMBERSHIP_REPORT:
            stats.ip.ipv6.icmp.mbrship_report++;
            stats.ip.ipv6.icmp.s_mbrship_report += hdr->len;
            break;

        case ICMP6_MEMBERSHIP_REDUCTION:
            stats.ip.ipv6.icmp.mbrship_reduction++;
            stats.ip.ipv6.icmp.s_mbrship_reduction += hdr->len;
            break;

        case ND_ROUTER_SOLICIT:
            stats.ip.ipv6.icmp.nd_router_solicit++;
            stats.ip.ipv6.icmp.s_nd_router_solicit += hdr->len;
            break;

        case ND_ROUTER_ADVERT:
            stats.ip.ipv6.icmp.nd_router_advert++;
            stats.ip.ipv6.icmp.s_nd_router_advert += hdr->len;
            break;

        case ND_NEIGHBOR_SOLICIT:
            stats.ip.ipv6.icmp.nd_neighbor_solicit++;
            stats.ip.ipv6.icmp.s_nd_neighbor_solicit += hdr->len;
            break;

        case ND_NEIGHBOR_ADVERT:
            stats.ip.ipv6.icmp.nd_neighbor_advert++;
            stats.ip.ipv6.icmp.s_nd_neighbor_advert += hdr->len;
            break;

        case ND_REDIRECT:
            stats.ip.ipv6.icmp.nd_redirect++;
            stats.ip.ipv6.icmp.s_nd_redirect += hdr->len;
            break;

        default:
            printf("UNK ICMP6 type: %d\n", icptr->icmp6_type);
            stats.ip.ipv6.icmp.other++;
            stats.ip.ipv6.icmp.s_other += hdr->len;
            break;
    }
}

void
handle_icmp(const struct pcap_pkthdr *hdr, const u_char *packet, int offset)
{
    struct icmphdr *icptr;

    stats.ip.ipv4.icmp.cnt++;
    stats.ip.ipv4.icmp.bs += hdr->len;
    icptr = (struct icmphdr *) (packet + offset);
    switch (icptr->type) {
        case ICMP_ECHOREPLY:
            stats.ip.ipv4.icmp.types.echo_reply++;
            stats.ip.ipv4.icmp.types.s_echo_reply += hdr->len;
            break;

        case ICMP_DEST_UNREACH:
            stats.ip.ipv4.icmp.types.dest_unreach.cnt++;
            stats.ip.ipv4.icmp.types.dest_unreach.bs += hdr->len;
            
            switch (icptr->code) {
                case ICMP_NET_UNREACH:
                    stats.ip.ipv4.icmp.types.dest_unreach.net_unreach++;
                    stats.ip.ipv4.icmp.types.dest_unreach.s_net_unreach += hdr->len;
                    break;

                case ICMP_HOST_UNREACH:
                    stats.ip.ipv4.icmp.types.dest_unreach.host_unreach++;
                    stats.ip.ipv4.icmp.types.dest_unreach.s_host_unreach += hdr->len;
                    break;

                case ICMP_PROT_UNREACH:
                    stats.ip.ipv4.icmp.types.dest_unreach.prot_unreach++;
                    stats.ip.ipv4.icmp.types.dest_unreach.s_prot_unreach += hdr->len;
                    break;

                case ICMP_FRAG_NEEDED:
                    stats.ip.ipv4.icmp.types.dest_unreach.frag_needed++;
                    stats.ip.ipv4.icmp.types.dest_unreach.s_frag_needed += hdr->len;
                    break;

                case ICMP_SR_FAILED:
                    stats.ip.ipv4.icmp.types.dest_unreach.sr_failed++;
                    stats.ip.ipv4.icmp.types.dest_unreach.s_sr_failed += hdr->len;
                    break;

                case ICMP_NET_UNKNOWN:
                    stats.ip.ipv4.icmp.types.dest_unreach.net_unknown++;
                    stats.ip.ipv4.icmp.types.dest_unreach.s_net_unknown += hdr->len;
                    break;

                case ICMP_HOST_UNKNOWN:
                    stats.ip.ipv4.icmp.types.dest_unreach.host_unknown++;
                    stats.ip.ipv4.icmp.types.dest_unreach.s_host_unknown += hdr->len;
                    break;

                case ICMP_HOST_ISOLATED:
                    stats.ip.ipv4.icmp.types.dest_unreach.host_isolated++;
                    stats.ip.ipv4.icmp.types.dest_unreach.s_host_isolated += hdr->len;
                    break;

                case ICMP_NET_ANO:
                    stats.ip.ipv4.icmp.types.dest_unreach.net_ano++;
                    stats.ip.ipv4.icmp.types.dest_unreach.s_net_ano += hdr->len;
                    break;

                case ICMP_HOST_ANO:
                    stats.ip.ipv4.icmp.types.dest_unreach.host_ano++;
                    stats.ip.ipv4.icmp.types.dest_unreach.s_host_ano += hdr->len;
                    break;

                case ICMP_NET_UNR_TOS:
                    stats.ip.ipv4.icmp.types.dest_unreach.net_unr_tos++;
                    stats.ip.ipv4.icmp.types.dest_unreach.s_net_unr_tos += hdr->len;
                    break;

                case ICMP_HOST_UNR_TOS:
                    stats.ip.ipv4.icmp.types.dest_unreach.host_unr_tos++;
                    stats.ip.ipv4.icmp.types.dest_unreach.s_host_unr_tos += hdr->len;
                    break;

                case ICMP_PKT_FILTERED:
                    stats.ip.ipv4.icmp.types.dest_unreach.pkt_filtered++;
                    stats.ip.ipv4.icmp.types.dest_unreach.s_pkt_filtered += hdr->len;
                    break;

                case ICMP_PREC_VIOLATION:
                    stats.ip.ipv4.icmp.types.dest_unreach.prec_violation++;
                    stats.ip.ipv4.icmp.types.dest_unreach.s_prec_violation += hdr->len;
                    break;

                case ICMP_PREC_CUTOFF:
                    stats.ip.ipv4.icmp.types.dest_unreach.prec_cutoff++;
                    stats.ip.ipv4.icmp.types.dest_unreach.s_prec_cutoff += hdr->len;
                    break;

                default:
                    stats.ip.ipv4.icmp.types.dest_unreach.other++;
                    stats.ip.ipv4.icmp.types.dest_unreach.s_other += hdr->len;
            }
            break;

        case ICMP_SOURCE_QUENCH:
            stats.ip.ipv4.icmp.types.source_quench++;
            stats.ip.ipv4.icmp.types.s_source_quench += hdr->len;
            break;

        case ICMP_REDIRECT:
            stats.ip.ipv4.icmp.types.redirect.cnt++;
            stats.ip.ipv4.icmp.types.redirect.bs += hdr->len;

            switch (icptr->code) {
                case ICMP_REDIR_NET:
                    stats.ip.ipv4.icmp.types.redirect.net++;
                    stats.ip.ipv4.icmp.types.redirect.s_net += hdr->len;
                    break;

                case ICMP_REDIR_HOST:
                    stats.ip.ipv4.icmp.types.redirect.host++;
                    stats.ip.ipv4.icmp.types.redirect.s_host += hdr->len;
                    break;

                case ICMP_REDIR_NETTOS:
                    stats.ip.ipv4.icmp.types.redirect.nettos++;
                    stats.ip.ipv4.icmp.types.redirect.s_nettos += hdr->len;
                    break;

                case ICMP_REDIR_HOSTTOS:
                    stats.ip.ipv4.icmp.types.redirect.hosttos++;
                    stats.ip.ipv4.icmp.types.redirect.s_hosttos += hdr->len;
                    break;

                default:
                    stats.ip.ipv4.icmp.types.redirect.other++;
                    stats.ip.ipv4.icmp.types.redirect.s_other += hdr->len;
                    break;
            }
            break;

        case ICMP_ECHO:
            stats.ip.ipv4.icmp.types.echo_request++;
            stats.ip.ipv4.icmp.types.s_echo_request += hdr->len;
            break;

        case ICMP_TIME_EXCEEDED:
            stats.ip.ipv4.icmp.types.time_exceeded.cnt++;
            stats.ip.ipv4.icmp.types.time_exceeded.bs += hdr->len;

            switch(icptr->code) {
                case ICMP_EXC_TTL:
                    stats.ip.ipv4.icmp.types.time_exceeded.ttl++;
                    stats.ip.ipv4.icmp.types.time_exceeded.s_ttl += hdr->len;
                    break;

                case ICMP_EXC_FRAGTIME:
                    stats.ip.ipv4.icmp.types.time_exceeded.fragtime++;
                    stats.ip.ipv4.icmp.types.time_exceeded.s_fragtime += hdr->len;
                    break;

                default:
                    stats.ip.ipv4.icmp.types.time_exceeded.other++;
                    stats.ip.ipv4.icmp.types.time_exceeded.s_other += hdr->len;
                    break;
            }
            break;

        case ICMP_PARAMETERPROB:
            stats.ip.ipv4.icmp.types.para_prob++;
            stats.ip.ipv4.icmp.types.s_para_prob += hdr->len;
            break;

        case ICMP_TIMESTAMP:
            stats.ip.ipv4.icmp.types.timestamp++;
            stats.ip.ipv4.icmp.types.s_timestamp += hdr->len;
            break;
        
        case ICMP_TIMESTAMPREPLY:
            stats.ip.ipv4.icmp.types.timestamp_reply++;
            stats.ip.ipv4.icmp.types.s_timestamp_reply += hdr->len;
            break;

        case ICMP_INFO_REQUEST:
            stats.ip.ipv4.icmp.types.info_request++;
            stats.ip.ipv4.icmp.types.s_info_request += hdr->len;
            break;

        case ICMP_INFO_REPLY:
            stats.ip.ipv4.icmp.types.info_reply++;
            stats.ip.ipv4.icmp.types.s_info_reply += hdr->len;
            break;

        case ICMP_ADDRESS:
            stats.ip.ipv4.icmp.types.address++;
            stats.ip.ipv4.icmp.types.s_address += hdr->len;
            break;

        case ICMP_ADDRESSREPLY:
            stats.ip.ipv4.icmp.types.address_reply++;
            stats.ip.ipv4.icmp.types.s_address_reply += hdr->len;
            break;

        default:
            stats.ip.ipv4.icmp.types.other++;
            stats.ip.ipv4.icmp.types.s_other += hdr->len;
            break;
    }
}

void
look_at_packet(u_char *p, const struct pcap_pkthdr *hdr, const u_char *packet)
{
    struct ether_header *eptr = (struct ether_header *) packet;
    struct iphdr *iptr;
    int fragid, mf, offset;

    switch ( ntohs(eptr->ether_type) ) {
        case ETHERTYPE_IP:
        case 0x86dd:          /* IPv6 */

            iptr = (struct iphdr *) (packet + sizeof(struct ether_header));

            stats.ip.cnt++;
            stats.ip.bs += hdr->len;

            if (iptr->version == 4) {
                stats.ip.ipv4.cnt++;
                stats.ip.ipv4.bs += hdr->len;

                fragid = ntohs(iptr->id);
                mf = (ntohs(iptr->frag_off) & 0x2000) >> 13;
                offset = ntohs(iptr->frag_off) & 0x1fff;

#if 0
                printf("ID: %u, MF: %u Offset: %u\n", fragid, mf, offset);
#endif

                if (mf || (!mf && offset)) {

                    stats.ip.frag.frag++;
                    stats.ip.frag.s_frag += hdr->len;

                } else {
                    stats.ip.frag.nfrag++;
                    stats.ip.frag.s_nfrag += hdr->len;
                }

                if (offset) {
#if 0
    if ( sp < 0 || dp < 0 || sp > 65536 || dp > 65536 )
        return;

    if ( dp && sp ) {
        int f = 0;
        struct ip_conn_s *c = stats.ip.conns;

        if ( (cc = conn_lookup[dp]) ) {
            while (cc) {
                if (match(cc->conn, sp, dp, saddr, daddr, type)) {
                    cc->conn->cnt++;
                    cc->conn->bs += hdr->len;
                    f = 1;
                    break;
                }
                cc = cc->next;
            }
        }
#endif
                    /*
                     * fragmented packet (2nd and following) we need to
                     * join them before parsing them further.
                     */
                    goto ignore_packet;
                }

                switch ( IPTOS_TOS(iptr->tos) ) {
                    case IPTOS_LOWDELAY:
                        stats.ip.tos.lowdelay++;
                        stats.ip.tos.s_lowdelay += hdr->len;
                        break;
                    case IPTOS_THROUGHPUT:
                        stats.ip.tos.maxthroughput++;
                        stats.ip.tos.s_maxthroughput += hdr->len;
                        break;
                    case IPTOS_RELIABILITY:
                        stats.ip.tos.reliability++;
                        stats.ip.tos.s_reliability += hdr->len;
                        break;
                    case IPTOS_LOWCOST:
                        stats.ip.tos.lowcost++;
                        stats.ip.tos.s_lowcost += hdr->len;
                        break;
                    default:
                        stats.ip.tos.none++;
                        stats.ip.tos.s_none += hdr->len;
                        break;
                }

                switch ( iptr->tos & 0x3 ) {
                    case 0:
                        stats.ip.ecn.not_ect++;
                        stats.ip.ecn.s_not_ect += hdr->len;
                        break;

                    case 1:
                        stats.ip.ecn.ect_1++;
                        stats.ip.ecn.s_ect_1 += hdr->len;
                        break;

                    case 2:
                        stats.ip.ecn.ect_0++;
                        stats.ip.ecn.s_ect_0 += hdr->len;
                        break;

                    case 3:
                        stats.ip.ecn.ce++;
                        stats.ip.ecn.s_ce += hdr->len;
                        break;

                    default:
                        printf("hmm?\n");
                        break;
                }

                switch (iptr->protocol) {
                    case 6:
                        handle_tcp(hdr, packet, sizeof(struct ether_header)
                                + (iptr->ihl * 4),
                                (char *) &iptr->saddr,
                                (char *) &iptr->daddr,
                                ntohs(eptr->ether_type));
                        break;

                    case 17:
                        handle_udp(hdr, packet, sizeof(struct ether_header)
                                + (iptr->ihl * 4),
                                (char *) &iptr->saddr,
                                (char *) &iptr->daddr,
                                ntohs(eptr->ether_type));
                        break;
                    case 1:
                        handle_icmp(hdr, packet, sizeof(struct ether_header)
                                + (iptr->ihl * 4));
                        break;

                    default:
                        stats.ip.other.cnt++;
                        stats.ip.other.bs += hdr->len;
                        break;
                }
            } else if (iptr->version == 6) {
                struct ip6_hdr *i6ptr;

                stats.ip.ipv6.cnt++;
                stats.ip.ipv6.bs += hdr->len;

                i6ptr = (struct ip6_hdr *) (packet + sizeof(struct ether_header));

                stats.ip.ipv6.hoplimit += i6ptr->ip6_hops;

                if ( (i6ptr->ip6_flow & 0xfffff) != 0 ) {
                    stats.ip.ipv6.fl++;
                    stats.ip.ipv6.s_fl += hdr->len;
                }

                if ( ((i6ptr->ip6_flow & 0xff00000) >> 20) != 0 ) {
                    stats.ip.ipv6.tc++;
                    stats.ip.ipv6.s_tc += hdr->len;
                }

                switch (i6ptr->ip6_nxt) {
                    case 6:
                        handle_tcp(hdr, packet, hdr->len - ntohs(i6ptr->ip6_plen),
                                (char *) &i6ptr->ip6_src,
                                (char *) &i6ptr->ip6_dst,
                                ntohs(eptr->ether_type));
                        break;

                    case 17:
                        handle_udp(hdr, packet, hdr->len - ntohs(i6ptr->ip6_plen),
                                (char *) &i6ptr->ip6_src,
                                (char *) &i6ptr->ip6_dst,
                                ntohs(eptr->ether_type));
                        break;

                    case 58:
                        handle_icmp6(hdr, packet, hdr->len - ntohs(i6ptr->ip6_plen));
                        break;

                    default:
                        printf("UNK IPV6 Proto: %d\n", i6ptr->ip6_nxt);
                        stats.ip.other.cnt++;
                        stats.ip.other.bs += hdr->len;
                        break;
                }
            }
            break;
        case ETHERTYPE_REVARP:
            stats.rarp.cnt++;
            stats.rarp.bs += hdr->len;
            break;
        case ETHERTYPE_ARP:
            stats.arp.cnt++;
            stats.arp.bs += hdr->len;
            break;
        default:
            stats.other.cnt++;
            stats.other.bs += hdr->len;
            break;
    }

ignore_packet:

    stats.bs += hdr->len;
    stats.cnt++;
}

