#! /usr/bin/env sscm -C UTF-8
;; -*- buffer-file-coding-system: utf-8 -*-

;;  Filename : test-formal-syntax.scm
;;  About    : unit test for R5RS formal syntax
;;
;;  Copyright (C) 2005-2006 YAMAMOTO Kengo <yamaken AT bp.iij4u.or.jp>
;;  Copyright (c) 2007-2008 SigScheme Project <uim-en AT googlegroups.com>
;;
;;  All rights reserved.
;;
;;  Redistribution and use in source and binary forms, with or without
;;  modification, are permitted provided that the following conditions
;;  are met:
;;
;;  1. Redistributions of source code must retain the above copyright
;;     notice, this list of conditions and the following disclaimer.
;;  2. Redistributions in binary form must reproduce the above copyright
;;     notice, this list of conditions and the following disclaimer in the
;;     documentation and/or other materials provided with the distribution.
;;  3. Neither the name of authors nor the names of its contributors
;;     may be used to endorse or promote products derived from this software
;;     without specific prior written permission.
;;
;;  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
;;  IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
;;  THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;;  PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
;;  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
;;  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
;;  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
;;  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
;;  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

(require-extension (unittest))

(define tn test-name)
(define *test-track-progress* #f)

;; See "7.1 Formal syntax" of R5RS
;; See also test-number-literal.scm

(tn "invalid boolean")
(if sigscheme?
    (begin
      (assert-parse-error (tn) "#F")
      (assert-parse-error (tn) "#T"))
    (begin
      (assert-true (tn) (boolean? (string-read "#F")))
      (assert-true (tn) (boolean? (string-read "#T")))))

(tn "boolean")
(assert-true (tn) (boolean? (string-read "#f")))
(assert-true (tn) (boolean? (string-read "#t")))

(tn "invalid identifier")
(assert-parse-error (tn) "#")
(assert-parse-error (tn) ".")
(assert-parse-error (tn) "..")
(assert-parse-error (tn) "....")
(assert-parse-error (tn) ".a")
(assert-parse-error (tn) "+a")
(assert-parse-error (tn) "++")
(assert-parse-error (tn) "--")
(assert-parse-error (tn) "-=")
(assert-parse-error (tn) "-$")
(assert-parse-error (tn) "-.")
(assert-parse-error (tn) "-@")
(assert-parse-error (tn) "@")
(assert-parse-error (tn) "1a")
(assert-parse-error (tn) "-a")

(tn "special initial identifier")
(assert-true (tn) (symbol? (string-read "!")))
(assert-true (tn) (symbol? (string-read "$")))
(assert-true (tn) (symbol? (string-read "%")))
(assert-true (tn) (symbol? (string-read "&")))
(assert-true (tn) (symbol? (string-read "*")))
(assert-true (tn) (symbol? (string-read "/")))
(assert-true (tn) (symbol? (string-read ":")))
(assert-true (tn) (symbol? (string-read "<")))
(assert-true (tn) (symbol? (string-read "=")))
(assert-true (tn) (symbol? (string-read ">")))
(assert-true (tn) (symbol? (string-read "?")))
(assert-true (tn) (symbol? (string-read "^")))
(assert-true (tn) (symbol? (string-read "_")))
(assert-true (tn) (symbol? (string-read "~")))

(tn "special initial identifier + number")
(assert-true (tn) (symbol? (string-read "!1")))
(assert-true (tn) (symbol? (string-read "$1")))
(assert-true (tn) (symbol? (string-read "%1")))
(assert-true (tn) (symbol? (string-read "&1")))
(assert-true (tn) (symbol? (string-read "*1")))
(assert-true (tn) (symbol? (string-read "/1")))
(assert-true (tn) (symbol? (string-read ":1")))
(assert-true (tn) (symbol? (string-read "<1")))
(assert-true (tn) (symbol? (string-read "=1")))
(assert-true (tn) (symbol? (string-read ">1")))
(assert-true (tn) (symbol? (string-read "?1")))
(assert-true (tn) (symbol? (string-read "^1")))
(assert-true (tn) (symbol? (string-read "_1")))
(assert-true (tn) (symbol? (string-read "~1")))

(tn "special initial identifier + letter")
(assert-true (tn) (symbol? (string-read "!a")))
(assert-true (tn) (symbol? (string-read "$a")))
(assert-true (tn) (symbol? (string-read "%a")))
(assert-true (tn) (symbol? (string-read "&a")))
(assert-true (tn) (symbol? (string-read "*a")))
(assert-true (tn) (symbol? (string-read "/a")))
(assert-true (tn) (symbol? (string-read ":a")))
(assert-true (tn) (symbol? (string-read "<a")))
(assert-true (tn) (symbol? (string-read "=a")))
(assert-true (tn) (symbol? (string-read ">a")))
(assert-true (tn) (symbol? (string-read "?a")))
(assert-true (tn) (symbol? (string-read "^a")))
(assert-true (tn) (symbol? (string-read "_a")))
(assert-true (tn) (symbol? (string-read "~a")))

(tn "identifier")
(assert-true (tn) (symbol? (string-read "...")))
(assert-true (tn) (symbol? (string-read "+")))
(assert-true (tn) (symbol? (string-read "-")))
(assert-true (tn) (symbol? (string-read "a.")))
(assert-true (tn) (symbol? (string-read "a+")))
(assert-true (tn) (symbol? (string-read "a-")))
(assert-true (tn) (symbol? (string-read "a@")))
(assert-true (tn) (symbol? (string-read "a1")))
;; SigScheme 0.7.0 and later disallows initial hyphen of an identifier.
(if sigscheme?
    (begin
      (assert-error (tn) (lambda () (symbol? (string-read "-a"))))
      (assert-true  (tn)            (symbol? (string->symbol "-a")))))

(tn "invalid dot pair")
(assert-parse-error (tn) "( . )")
(assert-parse-error (tn) "( . \"foo\")")
(assert-parse-error (tn) "( . \"foo\" \"bar\")")
(assert-parse-error (tn) "(\"foo\" . )")
(assert-parse-error (tn) "(\"foo\" \"bar\" . )")
(assert-parse-error (tn) "(\"foo\" . \"bar\" \"baz\")")
(assert-parse-error (tn) "(\"foo\" \"bar\" . \"baz\" \"quux\")")

(tn "invalid dot pair without left space")
(assert-parse-error (tn) "(. )")
(assert-parse-error (tn) "(. \"foo\")")
(assert-parse-error (tn) "(. \"foo\" \"bar\")")
(assert-parse-error (tn) "(\"foo\". )")
(assert-parse-error (tn) "(\"foo\" \"bar\". )")
(assert-parse-error (tn) "(\"foo\". \"bar\" \"baz\")")
(assert-parse-error (tn) "(\"foo\" \"bar\". \"baz\" \"quux\")")

(tn "dot pair")
(assert-parseable (tn) "(\"foo\" . \"bar\")")
(assert-parseable (tn) "(\"foo\" \"bar\" . \"baz\")")

(tn "dot pair without left space")
(assert-parseable (tn) "(\"foo\". \"bar\")")
(assert-parseable (tn) "(\"foo\" \"bar\". \"baz\")")

(let ((assert (if (and (provided? "sigscheme")
                       (not (provided? "strict-r5rs")))
                  assert-parse-error
                  assert-parseable)))
  (tn "invalid dot pair without right space")
  (assert (tn) "( .)")
  (assert (tn) "( .\"foo\")")
  (assert (tn) "( .\"foo\" \"bar\")")
  (assert (tn) "(\"foo\" .)")
  (assert (tn) "(\"foo\" \"bar\" .)")
  (assert (tn) "(\"foo\" .\"bar\" \"baz\")")
  (assert (tn) "(\"foo\" \"bar\" .\"baz\" \"quux\")")

  (tn "invalid dot pair without both space")
  (assert (tn) "(.)")
  (assert (tn) "(.\"foo\")")
  (assert (tn) "(.\"foo\" \"bar\")")
  (assert (tn) "(\"foo\".)")
  (assert (tn) "(\"foo\" \"bar\".)")
  (assert (tn) "(\"foo\".\"bar\" \"baz\")")
  (assert (tn) "(\"foo\" \"bar\".\"baz\" \"quux\")")

  (tn "dot pair without right space")
  (assert (tn) "(\"foo\" .\"bar\")")
  (assert (tn) "(\"foo\" \"bar\" .\"baz\")")

  (tn "dot pair without both space")
  (assert (tn) "(\"foo\".\"bar\")")
  (assert (tn) "(\"foo\" \"bar\".\"baz\")"))

(assert-error "invalid function calling: boolean" (lambda () (#t)))
(assert-error "invalid function calling: integer" (lambda () (1)))
(assert-error "invalid function calling: null"    (lambda () ('())))
(assert-error "invalid function calling: pair"    (lambda () ('(1 2))))
(assert-error "invalid function calling: char"    (lambda () (#\a)))
(assert-error "invalid function calling: string"  (lambda () ("a")))
(assert-error "invalid function calling: vector"  (lambda () (#(1))))

(tn "function calling fixed_0")
(define f (lambda () #t))
(assert-equal? (tn) #t         (f))
(assert-error  (tn) (lambda () (f . #t)))
(assert-error  (tn) (lambda () (f #t)))
(assert-error  (tn) (lambda () (f #t . #t)))
(assert-error  (tn) (lambda () (f #t #t)))
(assert-error  (tn) (lambda () (f #t #t . #t)))
(assert-error  (tn) (lambda () (f #t #t #t)))
(assert-error  (tn) (lambda () (f #t #t #t . #t)))
(tn "function calling variadic_0")
(define f (lambda args args))
(assert-equal? (tn) '()         (f))
(assert-error  (tn) (lambda ()  (f . #t)))
(assert-equal? (tn) '(#t)       (f #t))
(assert-error  (tn) (lambda ()  (f #t . #t)))
(assert-equal? (tn) '(#t #t)    (f #t #t))
(assert-error  (tn) (lambda ()  (f #t #t . #t)))
(assert-equal? (tn) '(#t #t #t) (f #t #t #t))
(assert-error  (tn) (lambda ()  (f #t #t #t . #t)))
(tn "function calling fixed_1")
(define f (lambda (x) x))
(assert-error  (tn) (lambda () (f)))
(assert-error  (tn) (lambda () (f . #t)))
(assert-equal? (tn) #t         (f #t))
(assert-error  (tn) (lambda () (f #t . #t)))
(assert-error  (tn) (lambda () (f #t #t)))
(assert-error  (tn) (lambda () (f #t #t . #t)))
(assert-error  (tn) (lambda () (f #t #t #t)))
(assert-error  (tn) (lambda () (f #t #t #t . #t)))
(tn "function calling variadic_1")
(define f (lambda (x . rest) (list x rest)))
(assert-error  (tn) (lambda ()    (f)))
(assert-error  (tn) (lambda ()    (f . #t)))
(assert-equal? (tn) '(#t ())      (f #t))
(assert-error  (tn) (lambda ()    (f #t . #t)))
(assert-equal? (tn) '(#t (#t))    (f #t #t))
(assert-error  (tn) (lambda ()    (f #t #t . #t)))
(assert-equal? (tn) '(#t (#t #t)) (f #t #t #t))
(assert-error  (tn) (lambda ()    (f #t #t #t . #t)))
(tn "function calling fixed_2")
(define f (lambda (x y) (list x y)))
(assert-error  (tn) (lambda ()    (f)))
(assert-error  (tn) (lambda ()    (f . #t)))
(assert-error  (tn) (lambda ()    (f #t)))
(assert-error  (tn) (lambda ()    (f #t . #t)))
(assert-equal? (tn) '(#t #t)      (f #t #t))
(assert-error  (tn) (lambda ()    (f #t #t . #t)))
(assert-error  (tn) (lambda ()    (f #t #t #t)))
(assert-error  (tn) (lambda ()    (f #t #t #t . #t)))
(tn "function calling variadic_2")
(define f (lambda (x y . rest) (list x y rest)))
(assert-error  (tn) (lambda ()    (f)))
(assert-error  (tn) (lambda ()    (f . #t)))
(assert-error  (tn) (lambda ()    (f #t)))
(assert-error  (tn) (lambda ()    (f #t . #t)))
(assert-equal? (tn) '(#t #t ())   (f #t #t))
(assert-error  (tn) (lambda ()    (f #t #t . #t)))
(assert-equal? (tn) '(#t #t (#t)) (f #t #t #t))
(assert-error  (tn) (lambda ()    (f #t #t #t . #t)))

(tn "function calling fixed_0 for define-created closure")
(define (f) #t)
(assert-equal? (tn) #t         (f))
(assert-error  (tn) (lambda () (f . #t)))
(assert-error  (tn) (lambda () (f #t)))
(assert-error  (tn) (lambda () (f #t . #t)))
(assert-error  (tn) (lambda () (f #t #t)))
(assert-error  (tn) (lambda () (f #t #t . #t)))
(assert-error  (tn) (lambda () (f #t #t #t)))
(assert-error  (tn) (lambda () (f #t #t #t . #t)))
(tn "function calling variadic_0 for define-created closure")
(define (f . args) args)
(assert-equal? (tn) '()         (f))
(assert-error  (tn) (lambda ()  (f . #t)))
(assert-equal? (tn) '(#t)       (f #t))
(assert-error  (tn) (lambda ()  (f #t . #t)))
(assert-equal? (tn) '(#t #t)    (f #t #t))
(assert-error  (tn) (lambda ()  (f #t #t . #t)))
(assert-equal? (tn) '(#t #t #t) (f #t #t #t))
(assert-error  (tn) (lambda ()  (f #t #t #t . #t)))
(tn "function calling fixed_1 for define-created closure")
(define (f x) x)
(assert-error  (tn) (lambda () (f)))
(assert-error  (tn) (lambda () (f . #t)))
(assert-equal? (tn) #t         (f #t))
(assert-error  (tn) (lambda () (f #t . #t)))
(assert-error  (tn) (lambda () (f #t #t)))
(assert-error  (tn) (lambda () (f #t #t . #t)))
(assert-error  (tn) (lambda () (f #t #t #t)))
(assert-error  (tn) (lambda () (f #t #t #t . #t)))
(tn "function calling variadic_1 for define-created closure")
(define (f x . rest) (list x rest))
(assert-error  (tn) (lambda ()    (f)))
(assert-error  (tn) (lambda ()    (f . #t)))
(assert-equal? (tn) '(#t ())      (f #t))
(assert-error  (tn) (lambda ()    (f #t . #t)))
(assert-equal? (tn) '(#t (#t))    (f #t #t))
(assert-error  (tn) (lambda ()    (f #t #t . #t)))
(assert-equal? (tn) '(#t (#t #t)) (f #t #t #t))
(assert-error  (tn) (lambda ()    (f #t #t #t . #t)))
(tn "function calling fixed_2 for define-created closure")
(define (f x y) (list x y))
(assert-error  (tn) (lambda ()    (f)))
(assert-error  (tn) (lambda ()    (f . #t)))
(assert-error  (tn) (lambda ()    (f #t)))
(assert-error  (tn) (lambda ()    (f #t . #t)))
(assert-equal? (tn) '(#t #t)      (f #t #t))
(assert-error  (tn) (lambda ()    (f #t #t . #t)))
(assert-error  (tn) (lambda ()    (f #t #t #t)))
(assert-error  (tn) (lambda ()    (f #t #t #t . #t)))
(tn "function calling variadic_2 for define-created closure")
(define (f x y . rest) (list x y rest))
(assert-error  (tn) (lambda ()    (f)))
(assert-error  (tn) (lambda ()    (f . #t)))
(assert-error  (tn) (lambda ()    (f #t)))
(assert-error  (tn) (lambda ()    (f #t . #t)))
(assert-equal? (tn) '(#t #t ())   (f #t #t))
(assert-error  (tn) (lambda ()    (f #t #t . #t)))
(assert-equal? (tn) '(#t #t (#t)) (f #t #t #t))
(assert-error  (tn) (lambda ()    (f #t #t #t . #t)))

;; Although SigScheme's eval facility itself does not ensure properness of
;; syntax args, each syntax implementation must check it. These tests only
;; indicate what should be done.
(tn "syntax application fixed_0")
;; FIXME: no syntax with syntax_fixed_0
(assert-equal? (tn) #t         ((lambda () #t)))
(assert-error  (tn) (lambda () ((lambda () #t) . #t)))
(assert-error  (tn) (lambda () ((lambda () #t) #t)))
(assert-error  (tn) (lambda () ((lambda () #t) #t . #t)))
(assert-error  (tn) (lambda () ((lambda () #t) #t #t)))
(assert-error  (tn) (lambda () ((lambda () #t) #t #t . #t)))
(assert-error  (tn) (lambda () ((lambda () #t) #t #t #t)))
(assert-error  (tn) (lambda () ((lambda () #t) #t #t #t . #t)))
(tn "syntax application variadic_0")
(assert-equal? (tn) #t          (and))
(assert-error  (tn) (lambda ()  (and . #t)))
(assert-equal? (tn) #t          (and #t))
(assert-error  (tn) (lambda ()  (and #t . #t)))
(assert-equal? (tn) #t          (and #t #t))
(assert-error  (tn) (lambda ()  (and #t #t . #t)))
(assert-equal? (tn) #t          (and #t #t #t))
(assert-error  (tn) (lambda ()  (and #t #t #t . #t)))
(tn "syntax application fixed_1")
(assert-error  (tn) (lambda () (quote)))
(assert-error  (tn) (lambda () (quote . #t)))
(assert-equal? (tn) #t         (quote #t))
(assert-error  (tn) (lambda () (quote #t . #t)))
(assert-error  (tn) (lambda () (quote #t #t)))
(assert-error  (tn) (lambda () (quote #t #t . #t)))
(assert-error  (tn) (lambda () (quote #t #t #t)))
(assert-error  (tn) (lambda () (quote #t #t #t . #t)))
(tn "syntax application variadic_1")
(assert-error  (tn) (lambda ()    (let*)))
(assert-error  (tn) (lambda ()    (let* . #t)))
(assert-error  (tn) (lambda ()    (let* ())))
(assert-error  (tn) (lambda ()    (let* #t . #t)))
(assert-equal? (tn) #t            (let* () #t))
(assert-error  (tn) (lambda ()    (let* #t #t . #t)))
(assert-equal? (tn) #t            (let* () #t #t))
(assert-error  (tn) (lambda ()    (let* #t #t #t . #t)))
(tn "syntax application fixed_2")
(define foo #f)
(assert-error  (tn) (lambda ()    (set!)))
(assert-error  (tn) (lambda ()    (set! . #t)))
(assert-error  (tn) (lambda ()    (set! #t)))
(assert-error  (tn) (lambda ()    (set! #t . #t)))
(if (and (provided? "sigscheme")
         (provided? "strict-r5rs"))
    (assert-equal? (tn) (undef)   (set! foo #t))
    (assert-equal? (tn) #t        (set! foo #t)))
(assert-error  (tn) (lambda ()    (set! #t #t . #t)))
(assert-error  (tn) (lambda ()    (set! #t #t #t)))
(assert-error  (tn) (lambda ()    (set! #t #t #t . #t)))
(tn "syntax application variadic_2")
(assert-error  (tn) (lambda ()    (if)))
(assert-error  (tn) (lambda ()    (if . #t)))
(assert-error  (tn) (lambda ()    (if #t)))
(assert-error  (tn) (lambda ()    (if #t . #t)))
(assert-equal? (tn) #t            (if #t #t))
(assert-error  (tn) (lambda ()    (if #t #t . #t)))
(assert-equal? (tn) #t            (if #t #t #t))
(assert-error  (tn) (lambda ()    (if #t #t #t . #t)))

(tn "EOF immediately after quoter")
;; (quote #<eof>) is invalid
(assert-error  (tn) (lambda () (string-read "'")))
(assert-error  (tn) (lambda () (string-read "`")))
(assert-error  (tn) (lambda () (string-read ",")))
(assert-error  (tn) (lambda () (string-read ",@")))

(total-report)
