"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isEntryNested = exports.getUpdatedEntriesOnDelete = exports.getOperatorType = exports.getOperatorOptions = exports.getNewExceptionItem = exports.getFormattedBuilderEntry = exports.getFormattedBuilderEntries = exports.getFilteredIndexPatterns = exports.getExceptionOperatorSelect = exports.getEntryValue = exports.getEntryOnOperatorChange = exports.getEntryOnMatchChange = exports.getEntryOnMatchAnyChange = exports.getEntryOnListChange = exports.getEntryOnFieldChange = exports.getEntryFromOperator = exports.getDefaultNestedEmptyEntry = exports.getDefaultEmptyEntry = exports.getCorrespondingKeywordField = exports.filterExceptionItems = exports.containsValueListEntry = exports.addIdToEntries = void 0;
var _uuid = _interopRequireDefault(require("uuid"));
var _securitysolutionUtils = require("@kbn/securitysolution-utils");
var _securitysolutionIoTsUtils = require("@kbn/securitysolution-io-ts-utils");
var _securitysolutionIoTsListTypes = require("@kbn/securitysolution-io-ts-list-types");
var _esQuery = require("@kbn/es-query");
var _autocomplete_operators = require("../autocomplete_operators");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const isEntryNested = item => {
  return item.entries != null;
};
exports.isEntryNested = isEntryNested;
const filterExceptionItems = exceptions => {
  return exceptions.reduce((acc, exception) => {
    const entries = exception.entries.reduce((nestedAcc, singleEntry) => {
      const strippedSingleEntry = (0, _securitysolutionUtils.removeIdFromItem)(singleEntry);
      if (_securitysolutionIoTsListTypes.entriesNested.is(strippedSingleEntry)) {
        const nestedEntriesArray = strippedSingleEntry.entries.filter(singleNestedEntry => {
          const noIdSingleNestedEntry = (0, _securitysolutionUtils.removeIdFromItem)(singleNestedEntry);
          const [validatedNestedEntry] = (0, _securitysolutionIoTsUtils.validate)(noIdSingleNestedEntry, _securitysolutionIoTsListTypes.nestedEntryItem);
          return validatedNestedEntry != null;
        });
        const noIdNestedEntries = nestedEntriesArray.map(singleNestedEntry => (0, _securitysolutionUtils.removeIdFromItem)(singleNestedEntry));
        const [validatedNestedEntry] = (0, _securitysolutionIoTsUtils.validate)({
          ...strippedSingleEntry,
          entries: noIdNestedEntries
        }, _securitysolutionIoTsListTypes.entriesNested);
        if (validatedNestedEntry != null) {
          return [...nestedAcc, {
            ...singleEntry,
            entries: nestedEntriesArray
          }];
        }
        return nestedAcc;
      } else {
        const [validatedEntry] = (0, _securitysolutionIoTsUtils.validate)(strippedSingleEntry, _securitysolutionIoTsListTypes.entry);
        if (validatedEntry != null) {
          return [...nestedAcc, singleEntry];
        }
        return nestedAcc;
      }
    }, []);
    if (entries.length === 0) {
      return acc;
    }
    const item = {
      ...exception,
      entries
    };
    if (_securitysolutionIoTsListTypes.exceptionListItemSchema.is(item)) {
      return [...acc, item];
    } else if (_securitysolutionIoTsListTypes.createExceptionListItemSchema.is(item)) {
      const {
        meta,
        ...rest
      } = item;
      const itemSansMetaId = {
        ...rest,
        meta: undefined
      };
      return [...acc, itemSansMetaId];
    } else {
      return acc;
    }
  }, []);
};
exports.filterExceptionItems = filterExceptionItems;
const addIdToEntries = entries => {
  return entries.map(singleEntry => {
    if (singleEntry.type === 'nested') {
      return (0, _securitysolutionUtils.addIdToItem)({
        ...singleEntry,
        entries: singleEntry.entries.map(nestedEntry => (0, _securitysolutionUtils.addIdToItem)(nestedEntry))
      });
    } else {
      return (0, _securitysolutionUtils.addIdToItem)(singleEntry);
    }
  });
};
exports.addIdToEntries = addIdToEntries;
const getNewExceptionItem = ({
  listId,
  namespaceType,
  ruleName
}) => {
  return {
    comments: [],
    description: `${ruleName} - exception list item`,
    entries: addIdToEntries([{
      field: '',
      operator: 'included',
      type: 'match',
      value: ''
    }]),
    item_id: undefined,
    list_id: listId,
    meta: {
      temporaryUuid: _uuid.default.v4()
    },
    name: `${ruleName} - exception list item`,
    namespace_type: namespaceType,
    tags: [],
    type: 'simple'
  };
};

/**
 * Returns the operator type, may not need this if using io-ts types
 *
 * @param item a single ExceptionItem entry
 */
exports.getNewExceptionItem = getNewExceptionItem;
const getOperatorType = item => {
  switch (item.type) {
    case 'match':
      return _securitysolutionIoTsListTypes.ListOperatorTypeEnum.MATCH;
    case 'match_any':
      return _securitysolutionIoTsListTypes.ListOperatorTypeEnum.MATCH_ANY;
    case 'list':
      return _securitysolutionIoTsListTypes.ListOperatorTypeEnum.LIST;
    default:
      return _securitysolutionIoTsListTypes.ListOperatorTypeEnum.EXISTS;
  }
};

/**
 * Determines operator selection (is/is not/is one of, etc.)
 * Default operator is "is"
 *
 * @param item a single ExceptionItem entry
 */
exports.getOperatorType = getOperatorType;
const getExceptionOperatorSelect = item => {
  if (item.type === 'nested') {
    return _autocomplete_operators.isOperator;
  } else {
    const operatorType = getOperatorType(item);
    const foundOperator = _autocomplete_operators.EXCEPTION_OPERATORS.find(operatorOption => {
      return item.operator === operatorOption.operator && operatorType === operatorOption.type;
    });
    return foundOperator != null ? foundOperator : _autocomplete_operators.isOperator;
  }
};

/**
 * Returns the fields corresponding value for an entry
 *
 * @param item a single ExceptionItem entry
 */
exports.getExceptionOperatorSelect = getExceptionOperatorSelect;
const getEntryValue = item => {
  switch (item.type) {
    case _securitysolutionIoTsListTypes.ListOperatorTypeEnum.MATCH:
    case _securitysolutionIoTsListTypes.ListOperatorTypeEnum.MATCH_ANY:
      return item.value;
    case _securitysolutionIoTsListTypes.ListOperatorTypeEnum.EXISTS:
      return undefined;
    case _securitysolutionIoTsListTypes.ListOperatorTypeEnum.LIST:
      return item.list.id;
    default:
      return undefined;
  }
};

/**
 * Determines whether an entire entry, exception item, or entry within a nested
 * entry needs to be removed
 *
 * @param exceptionItem
 * @param entryIndex index of given entry, for nested entries, this will correspond
 * to their parent index
 * @param nestedEntryIndex index of nested entry
 *
 */
exports.getEntryValue = getEntryValue;
const getUpdatedEntriesOnDelete = (exceptionItem, entryIndex, nestedParentIndex) => {
  const itemOfInterest = exceptionItem.entries[nestedParentIndex != null ? nestedParentIndex : entryIndex];
  if (nestedParentIndex != null && itemOfInterest.type === _securitysolutionIoTsListTypes.ListOperatorTypeEnum.NESTED) {
    const updatedEntryEntries = [...itemOfInterest.entries.slice(0, entryIndex), ...itemOfInterest.entries.slice(entryIndex + 1)];
    if (updatedEntryEntries.length === 0) {
      return {
        ...exceptionItem,
        entries: [...exceptionItem.entries.slice(0, nestedParentIndex), ...exceptionItem.entries.slice(nestedParentIndex + 1)]
      };
    } else {
      const {
        field
      } = itemOfInterest;
      const updatedItemOfInterest = {
        entries: updatedEntryEntries,
        field,
        id: itemOfInterest.id != null ? itemOfInterest.id : `${entryIndex}`,
        type: _securitysolutionIoTsListTypes.ListOperatorTypeEnum.NESTED
      };
      return {
        ...exceptionItem,
        entries: [...exceptionItem.entries.slice(0, nestedParentIndex), updatedItemOfInterest, ...exceptionItem.entries.slice(nestedParentIndex + 1)]
      };
    }
  } else {
    return {
      ...exceptionItem,
      entries: [...exceptionItem.entries.slice(0, entryIndex), ...exceptionItem.entries.slice(entryIndex + 1)]
    };
  }
};

/**
 * Returns filtered index patterns based on the field - if a user selects to
 * add nested entry, should only show nested fields, if item is the parent
 * field of a nested entry, we only display the parent field
 *
 * @param patterns IndexPatternBase containing available fields on rule index
 * @param item exception item entry
 * set to add a nested field
 */
exports.getUpdatedEntriesOnDelete = getUpdatedEntriesOnDelete;
const getFilteredIndexPatterns = (patterns, item, type, preFilter, osTypes) => {
  const indexPatterns = preFilter != null ? preFilter(patterns, type, osTypes) : patterns;
  if (item.nested === 'child' && item.parent != null) {
    // when user has selected a nested entry, only fields with the common parent are shown
    return {
      ...indexPatterns,
      fields: indexPatterns.fields.filter(indexField => {
        const subTypeNested = (0, _esQuery.getDataViewFieldSubtypeNested)(indexField);
        const fieldHasCommonParentPath = subTypeNested && item.parent != null && subTypeNested.nested.path === item.parent.parent.field;
        return fieldHasCommonParentPath;
      }).map(f => {
        const [fieldNameWithoutParentPath] = f.name.split('.').slice(-1);
        return {
          ...f,
          name: fieldNameWithoutParentPath
        };
      })
    };
  } else if (item.nested === 'parent' && item.field != null) {
    // when user has selected a nested entry, right above it we show the common parent
    return {
      ...indexPatterns,
      fields: [item.field]
    };
  } else if (item.nested === 'parent' && item.field == null) {
    // when user selects to add a nested entry, only nested fields are shown as options
    return {
      ...indexPatterns,
      fields: indexPatterns.fields.filter(field => (0, _esQuery.isDataViewFieldSubtypeNested)(field))
    };
  } else {
    return indexPatterns;
  }
};

/**
 * Determines proper entry update when user selects new field
 *
 * @param item - current exception item entry values
 * @param newField - newly selected field
 *
 */
exports.getFilteredIndexPatterns = getFilteredIndexPatterns;
const getEntryOnFieldChange = (item, newField) => {
  const {
    parent,
    entryIndex,
    nested
  } = item;
  const newChildFieldValue = newField != null ? newField.name.split('.').slice(-1)[0] : '';
  if (nested === 'parent') {
    // For nested entries, when user first selects to add a nested
    // entry, they first see a row similar to what is shown for when
    // a user selects "exists", as soon as they make a selection
    // we can now identify the 'parent' and 'child' this is where
    // we first convert the entry into type "nested"
    const subTypeNested = (0, _esQuery.getDataViewFieldSubtypeNested)(newField);
    const newParentFieldValue = (subTypeNested === null || subTypeNested === void 0 ? void 0 : subTypeNested.nested.path) || '';
    return {
      index: entryIndex,
      updatedEntry: {
        entries: [(0, _securitysolutionUtils.addIdToItem)({
          field: newChildFieldValue != null ? newChildFieldValue : '',
          operator: _autocomplete_operators.isOperator.operator,
          type: _securitysolutionIoTsListTypes.ListOperatorTypeEnum.MATCH,
          value: ''
        })],
        field: newParentFieldValue,
        id: item.id,
        type: _securitysolutionIoTsListTypes.ListOperatorTypeEnum.NESTED
      }
    };
  } else if (nested === 'child' && parent != null) {
    return {
      index: parent.parentIndex,
      updatedEntry: {
        ...parent.parent,
        entries: [...parent.parent.entries.slice(0, entryIndex), {
          field: newChildFieldValue != null ? newChildFieldValue : '',
          id: item.id,
          operator: _autocomplete_operators.isOperator.operator,
          type: _securitysolutionIoTsListTypes.ListOperatorTypeEnum.MATCH,
          value: ''
        }, ...parent.parent.entries.slice(entryIndex + 1)]
      }
    };
  } else {
    return {
      index: entryIndex,
      updatedEntry: {
        field: newField != null ? newField.name : '',
        id: item.id,
        operator: _autocomplete_operators.isOperator.operator,
        type: _securitysolutionIoTsListTypes.ListOperatorTypeEnum.MATCH,
        value: ''
      }
    };
  }
};

/**
 * Determines proper entry update when user updates value
 * when operator is of type "list"
 *
 * @param item - current exception item entry values
 * @param newField - newly selected list
 *
 */
exports.getEntryOnFieldChange = getEntryOnFieldChange;
const getEntryOnListChange = (item, newField) => {
  const {
    entryIndex,
    field,
    operator
  } = item;
  const {
    id,
    type
  } = newField;
  return {
    index: entryIndex,
    updatedEntry: {
      field: field != null ? field.name : '',
      id: item.id,
      list: {
        id,
        type
      },
      operator: operator.operator,
      type: _securitysolutionIoTsListTypes.ListOperatorTypeEnum.LIST
    }
  };
};

/**
 * Determines proper entry update when user updates value
 * when operator is of type "match_any"
 *
 * @param item - current exception item entry values
 * @param newField - newly entered value
 *
 */
exports.getEntryOnListChange = getEntryOnListChange;
const getEntryOnMatchAnyChange = (item, newField) => {
  const {
    nested,
    parent,
    entryIndex,
    field,
    operator
  } = item;
  if (nested != null && parent != null) {
    const fieldName = field != null ? field.name.split('.').slice(-1)[0] : '';
    return {
      index: parent.parentIndex,
      updatedEntry: {
        ...parent.parent,
        entries: [...parent.parent.entries.slice(0, entryIndex), {
          field: fieldName,
          id: item.id,
          operator: operator.operator,
          type: _securitysolutionIoTsListTypes.ListOperatorTypeEnum.MATCH_ANY,
          value: newField
        }, ...parent.parent.entries.slice(entryIndex + 1)]
      }
    };
  } else {
    return {
      index: entryIndex,
      updatedEntry: {
        field: field != null ? field.name : '',
        id: item.id,
        operator: operator.operator,
        type: _securitysolutionIoTsListTypes.ListOperatorTypeEnum.MATCH_ANY,
        value: newField
      }
    };
  }
};

/**
 * Determines proper entry update when user updates value
 * when operator is of type "match"
 *
 * @param item - current exception item entry values
 * @param newField - newly entered value
 *
 */
exports.getEntryOnMatchAnyChange = getEntryOnMatchAnyChange;
const getEntryOnMatchChange = (item, newField) => {
  const {
    nested,
    parent,
    entryIndex,
    field,
    operator
  } = item;
  if (nested != null && parent != null) {
    const fieldName = field != null ? field.name.split('.').slice(-1)[0] : '';
    return {
      index: parent.parentIndex,
      updatedEntry: {
        ...parent.parent,
        entries: [...parent.parent.entries.slice(0, entryIndex), {
          field: fieldName,
          id: item.id,
          operator: operator.operator,
          type: _securitysolutionIoTsListTypes.ListOperatorTypeEnum.MATCH,
          value: newField
        }, ...parent.parent.entries.slice(entryIndex + 1)]
      }
    };
  } else {
    return {
      index: entryIndex,
      updatedEntry: {
        field: field != null ? field.name : '',
        id: item.id,
        operator: operator.operator,
        type: _securitysolutionIoTsListTypes.ListOperatorTypeEnum.MATCH,
        value: newField
      }
    };
  }
};

/**
 * On operator change, determines whether value needs to be cleared or not
 *
 * @param field
 * @param selectedOperator
 * @param currentEntry
 *
 */
exports.getEntryOnMatchChange = getEntryOnMatchChange;
const getEntryFromOperator = (selectedOperator, currentEntry) => {
  const isSameOperatorType = currentEntry.operator.type === selectedOperator.type;
  const fieldValue = currentEntry.field != null ? currentEntry.field.name : '';
  switch (selectedOperator.type) {
    case 'match':
      return {
        field: fieldValue,
        id: currentEntry.id,
        operator: selectedOperator.operator,
        type: _securitysolutionIoTsListTypes.ListOperatorTypeEnum.MATCH,
        value: isSameOperatorType && typeof currentEntry.value === 'string' ? currentEntry.value : ''
      };
    case 'match_any':
      return {
        field: fieldValue,
        id: currentEntry.id,
        operator: selectedOperator.operator,
        type: _securitysolutionIoTsListTypes.ListOperatorTypeEnum.MATCH_ANY,
        value: isSameOperatorType && Array.isArray(currentEntry.value) ? currentEntry.value : []
      };
    case 'list':
      return {
        field: fieldValue,
        id: currentEntry.id,
        list: {
          id: '',
          type: 'ip'
        },
        operator: selectedOperator.operator,
        type: _securitysolutionIoTsListTypes.ListOperatorTypeEnum.LIST
      };
    default:
      return {
        field: fieldValue,
        id: currentEntry.id,
        operator: selectedOperator.operator,
        type: _securitysolutionIoTsListTypes.ListOperatorTypeEnum.EXISTS
      };
  }
};

/**
 * Determines proper entry update when user selects new operator
 *
 * @param item - current exception item entry values
 * @param newOperator - newly selected operator
 *
 */
exports.getEntryFromOperator = getEntryFromOperator;
const getEntryOnOperatorChange = (item, newOperator) => {
  const {
    parent,
    entryIndex,
    field,
    nested
  } = item;
  const newEntry = getEntryFromOperator(newOperator, item);
  if (!_securitysolutionIoTsListTypes.entriesList.is(newEntry) && nested != null && parent != null) {
    return {
      index: parent.parentIndex,
      updatedEntry: {
        ...parent.parent,
        entries: [...parent.parent.entries.slice(0, entryIndex), {
          ...newEntry,
          field: field != null ? field.name.split('.').slice(-1)[0] : ''
        }, ...parent.parent.entries.slice(entryIndex + 1)]
      }
    };
  } else {
    return {
      index: entryIndex,
      updatedEntry: newEntry
    };
  }
};

/**
 * Determines which operators to make available
 *
 * @param item
 * @param listType
 * @param isBoolean
 * @param includeValueListOperators whether or not to include the 'is in list' and 'is not in list' operators
 */
exports.getEntryOnOperatorChange = getEntryOnOperatorChange;
const getOperatorOptions = (item, listType, isBoolean, includeValueListOperators = true) => {
  if (item.nested === 'parent' || item.field == null) {
    return [_autocomplete_operators.isOperator];
  } else if (item.nested != null && listType === 'endpoint' || listType === 'endpoint') {
    return isBoolean ? [_autocomplete_operators.isOperator] : [_autocomplete_operators.isOperator, _autocomplete_operators.isOneOfOperator];
  } else if (item.nested != null && listType === 'detection') {
    return isBoolean ? [_autocomplete_operators.isOperator, _autocomplete_operators.existsOperator] : [_autocomplete_operators.isOperator, _autocomplete_operators.isOneOfOperator, _autocomplete_operators.existsOperator];
  } else {
    return isBoolean ? [_autocomplete_operators.isOperator, _autocomplete_operators.isNotOperator, _autocomplete_operators.existsOperator, _autocomplete_operators.doesNotExistOperator] : includeValueListOperators ? _autocomplete_operators.EXCEPTION_OPERATORS : _autocomplete_operators.EXCEPTION_OPERATORS_SANS_LISTS;
  }
};

/**
 * Fields of type 'text' do not generate autocomplete values, we want
 * to find it's corresponding keyword type (if available) which does
 * generate autocomplete values
 *
 * @param fields IFieldType fields
 * @param selectedField the field name that was selected
 * @param isTextType we only want a corresponding keyword field if
 * the selected field is of type 'text'
 *
 */
exports.getOperatorOptions = getOperatorOptions;
const getCorrespondingKeywordField = ({
  fields,
  selectedField
}) => {
  const selectedFieldBits = selectedField != null && selectedField !== '' ? selectedField.split('.') : [];
  const selectedFieldIsTextType = selectedFieldBits.slice(-1)[0] === 'text';
  if (selectedFieldIsTextType && selectedFieldBits.length > 0) {
    const keywordField = selectedFieldBits.slice(0, selectedFieldBits.length - 1).join('.');
    const [foundKeywordField] = fields.filter(({
      name
    }) => keywordField !== '' && keywordField === name);
    return foundKeywordField;
  }
  return undefined;
};

/**
 * Formats the entry into one that is easily usable for the UI, most of the
 * complexity was introduced with nested fields
 *
 * @param patterns IndexPatternBase containing available fields on rule index
 * @param item exception item entry
 * @param itemIndex entry index
 * @param parent nested entries hold copy of their parent for use in various logic
 * @param parentIndex corresponds to the entry index, this might seem obvious, but
 * was added to ensure that nested items could be identified with their parent entry
 */
exports.getCorrespondingKeywordField = getCorrespondingKeywordField;
const getFormattedBuilderEntry = (indexPattern, item, itemIndex, parent, parentIndex) => {
  const {
    fields
  } = indexPattern;
  const field = parent != null ? `${parent.field}.${item.field}` : item.field;
  const [foundField] = fields.filter(({
    name
  }) => field != null && field === name);
  const correspondingKeywordField = getCorrespondingKeywordField({
    fields,
    selectedField: field
  });
  if (parent != null && parentIndex != null) {
    return {
      correspondingKeywordField,
      entryIndex: itemIndex,
      field: foundField != null ? {
        ...foundField,
        name: foundField.name.split('.').slice(-1)[0]
      } : foundField,
      id: item.id != null ? item.id : `${itemIndex}`,
      nested: 'child',
      operator: getExceptionOperatorSelect(item),
      parent: {
        parent,
        parentIndex
      },
      value: getEntryValue(item)
    };
  } else {
    return {
      correspondingKeywordField,
      entryIndex: itemIndex,
      field: foundField,
      id: item.id != null ? item.id : `${itemIndex}`,
      nested: undefined,
      operator: getExceptionOperatorSelect(item),
      parent: undefined,
      value: getEntryValue(item)
    };
  }
};

/**
 * Formats the entries to be easily usable for the UI, most of the
 * complexity was introduced with nested fields
 *
 * @param patterns IndexPatternBase containing available fields on rule index
 * @param entries exception item entries
 * @param addNested boolean noting whether or not UI is currently
 * set to add a nested field
 * @param parent nested entries hold copy of their parent for use in various logic
 * @param parentIndex corresponds to the entry index, this might seem obvious, but
 * was added to ensure that nested items could be identified with their parent entry
 */
exports.getFormattedBuilderEntry = getFormattedBuilderEntry;
const getFormattedBuilderEntries = (indexPattern, entries, parent, parentIndex) => {
  return entries.reduce((acc, item, index) => {
    const isNewNestedEntry = item.type === 'nested' && item.entries.length === 0;
    if (item.type !== 'nested' && !isNewNestedEntry) {
      const newItemEntry = getFormattedBuilderEntry(indexPattern, item, index, parent, parentIndex);
      return [...acc, newItemEntry];
    } else {
      const parentEntry = {
        correspondingKeywordField: undefined,
        entryIndex: index,
        field: isNewNestedEntry ? undefined :
        // This type below is really a FieldSpec type from "src/plugins/data/common/index_patterns/fields/types.ts", we cast it here to keep using the IndexPatternFieldBase interface
        {
          aggregatable: false,
          esTypes: ['nested'],
          name: item.field != null ? item.field : '',
          searchable: false,
          type: 'string'
        },
        id: item.id != null ? item.id : `${index}`,
        nested: 'parent',
        operator: _autocomplete_operators.isOperator,
        parent: undefined,
        value: undefined
      };

      // User has selected to add a nested field, but not yet selected the field
      if (isNewNestedEntry) {
        return [...acc, parentEntry];
      }
      if (isEntryNested(item)) {
        const nestedItems = getFormattedBuilderEntries(indexPattern, item.entries, item, index);
        return [...acc, parentEntry, ...nestedItems];
      }
      return [...acc];
    }
  }, []);
};
exports.getFormattedBuilderEntries = getFormattedBuilderEntries;
const getDefaultEmptyEntry = () => ({
  field: '',
  id: _uuid.default.v4(),
  operator: _securitysolutionIoTsListTypes.ListOperatorEnum.INCLUDED,
  type: _securitysolutionIoTsListTypes.ListOperatorTypeEnum.MATCH,
  value: ''
});
exports.getDefaultEmptyEntry = getDefaultEmptyEntry;
const getDefaultNestedEmptyEntry = () => ({
  entries: [],
  field: '',
  id: _uuid.default.v4(),
  type: _securitysolutionIoTsListTypes.ListOperatorTypeEnum.NESTED
});
exports.getDefaultNestedEmptyEntry = getDefaultNestedEmptyEntry;
const containsValueListEntry = items => items.some(item => item.entries.some(({
  type
}) => type === _securitysolutionIoTsListTypes.ListOperatorTypeEnum.LIST));
exports.containsValueListEntry = containsValueListEntry;