// Copyright 2016 VMware, Inc. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package kvstore

import (
	"bytes"
	"context"
	"io"
	"io/ioutil"
	"os"

	"github.com/stretchr/testify/mock"
)

type MockBackend struct {
	buf []byte
}

// Creates path and ovewrites whatever is there
func (m *MockBackend) Save(ctx context.Context, r io.Reader, pth string) error {
	buf, err := ioutil.ReadAll(r)
	if err != nil {
		return err
	}

	m.buf = buf

	return nil
}

func (m *MockBackend) Load(ctx context.Context, pth string) (io.ReadCloser, error) {
	if len(m.buf) == 0 {
		return nil, os.ErrNotExist
	}

	return ioutil.NopCloser(bytes.NewReader(m.buf)), nil
}

// MockKeyValueStore is an autogenerated mock type for the KeyValueStore type
type MockKeyValueStore struct {
	mock.Mock
}

// Delete provides a mock function with given fields: ctx, key
func (_m *MockKeyValueStore) Delete(ctx context.Context, key string) error {
	ret := _m.Called(ctx, key)

	var r0 error
	if rf, ok := ret.Get(0).(func(context.Context, string) error); ok {
		r0 = rf(ctx, key)
	} else {
		r0 = ret.Error(0)
	}

	return r0
}

// Get provides a mock function with given fields: key
func (_m *MockKeyValueStore) Get(key string) ([]byte, error) {
	ret := _m.Called(key)

	var r0 []byte
	if rf, ok := ret.Get(0).(func(string) []byte); ok {
		r0 = rf(key)
	} else {
		if ret.Get(0) != nil {
			r0 = ret.Get(0).([]byte)
		}
	}

	var r1 error
	if rf, ok := ret.Get(1).(func(string) error); ok {
		r1 = rf(key)
	} else {
		r1 = ret.Error(1)
	}

	return r0, r1
}

// List provides a mock function with given fields: re
func (_m *MockKeyValueStore) List(re string) (map[string][]byte, error) {
	ret := _m.Called(re)

	var r0 map[string][]byte
	if rf, ok := ret.Get(0).(func(string) map[string][]byte); ok {
		r0 = rf(re)
	} else {
		if ret.Get(0) != nil {
			r0 = ret.Get(0).(map[string][]byte)
		}
	}

	var r1 error
	if rf, ok := ret.Get(1).(func(string) error); ok {
		r1 = rf(re)
	} else {
		r1 = ret.Error(1)
	}

	return r0, r1
}

// Name provides a mock function with given fields:
func (_m *MockKeyValueStore) Name() string {
	ret := _m.Called()

	var r0 string
	if rf, ok := ret.Get(0).(func() string); ok {
		r0 = rf()
	} else {
		r0 = ret.Get(0).(string)
	}

	return r0
}

// Save provides a mock function with given fields: ctx
func (_m *MockKeyValueStore) Save(ctx context.Context) error {
	ret := _m.Called(ctx)

	var r0 error
	if rf, ok := ret.Get(0).(func(context.Context) error); ok {
		r0 = rf(ctx)
	} else {
		r0 = ret.Error(0)
	}

	return r0
}

// Set provides a mock function with given fields: ctx, key, value
func (_m *MockKeyValueStore) Put(ctx context.Context, key string, value []byte) error {
	ret := _m.Called(ctx, key, value)

	var r0 error
	if rf, ok := ret.Get(0).(func(context.Context, string, []byte) error); ok {
		r0 = rf(ctx, key, value)
	} else {
		r0 = ret.Error(0)
	}

	return r0
}

var _ KeyValueStore = (*MockKeyValueStore)(nil)
