
:- export open_input_stream/2.
:- export read_stream/5.
:- export close_input_stream/1.

:- export read_octet/3.
:- export read_octets_till/5.
:- export read_octets/5.
:- export eos/1.
:- export read_file/2.

eos( -1 ).

% Prolog Streams
%open_input_stream( stream(FileName), stream(Stream) ) :-
%	open( FileName, read, Stream ).

open_input_stream( stream(FileName), stream(FileName) ) :-
	see( FileName ).

% Character Lists 
open_input_stream( [], [] ).
open_input_stream( [List|RestList], [List|RestList] ).

open_input_stream( S1 + S2, OS1 + OS2 ) :- !,
	open_input_stream( S1, OS1 ),
	open_input_stream( S2, OS2 ).

% URIREFS
open_input_stream( uri(URI), _ ) :- !, fail.


% Atoms
open_input_stream( A, Codes ) :- 
	atom( A ), !, 
	atom_codes( A, Codes ).

close_input_stream( [] ) :- !.
close_input_stream( [_|_] ) :- !.
close_input_stream( S1 + S2 ) :- !,
	close_input_stream( S1 ),
	close_input_stream( S2 ).
%close_input_stream( stream(Stream) ) :- 
%	close( Stream ).
close_input_stream( stream(_) ) :- 
	seen.

/*
reset_stream( [] ) :- !.
reset_stream( [_|_] ) :- !.
reset_stream( S1 + S2 ) :- !,
	reset_stream( S1 ),
	reset_stream( S2 ).
reset_stream( Stream ) :- !,
	seek( Stream, 0, 0, _ ).
*/

% Read one octet from the input stream
read_octet( [Octet|RestOctets], Octet, RestOctets ) :- !.
read_octet( S1 + S2, Octet, NS1 + S2 ) :-
	read_octet( S1, Octet, NS1 ), !.
read_octet( _ + S2, Octet, NS2 ) :- !,
	read_octet( S2, Octet, NS2 ).	
%read_octet( stream(Stream), Octet, stream(Stream) ) :- !, get_byte(Stream,Octet), Octet \= -1.
read_octet( stream(Stream), Octet, stream(Stream) ) :- !, get0(Octet), Octet \= -1.


% Read N octets from the input stream
read_octets( 0, Stream, RestOctets, RestOctets, Stream ) :- !.
read_octets( N, Stream, [Octet|NextOctets], RestOctets, RestStream ) :-
	N > 0, 
	N1 is N - 1,
	read_octet( Stream, Octet, AuxStream ), !,
	read_octets( N1, AuxStream, NextOctets, RestOctets, RestStream ).	


read_octets_till( Stream, EOS, [Octet|NextOctets], RestOctets, RestStream ) :-
	read_octet( Stream, Octet, AuxStream ), !,
	( Octet = EOS -> RestStream = Stream, RestOctets = NextOctets;
                       read_octets_till( AuxStream, EOS, NextOctets, RestOctets, RestStream )
	). 


% Reads a complete US-ASCII stream, returning a list of characters
read_stream( 'US-ASCII', Stream, Codes, RestCodes, RestStream ) :- !,
	read_ascii_stream( Stream, Codes, RestCodes, RestStream ).

read_ascii_stream( Stream, [Code|NextCodes], RestCodes, RestStream ) :-
	read_octet( Stream, Code, AuxStream ),
	Code < 128, !,
	read_ascii_stream( AuxStream, NextCodes, RestCodes, RestStream ).

read_ascii_stream( Stream, RestCodes, RestCodes, Stream ) :- !.


% Reads a single US-ASCII character
read_code( 'US-ASCII', Stream, Code, RestStream ) :- !,
	read_octet( Stream, Code, RestStream ), Code < 128.


% Reads a complete ISO-8859-1 stream, returning a list of characters
read_stream( 'ISO-8859-1', Stream, Codes, RestCodes, RestStream ) :- !,
	read_iso88591_stream( Stream, Codes, RestCodes, RestStream ).

read_iso88591_stream( Stream, [Code|NextCodes], RestCodes, RestStream ) :-
	read_octet( Stream, Code, AuxStream ), !,
	read_iso88591_stream( AuxStream, NextCodes, RestCodes, RestStream ).

read_iso88591_stream( Stream, RestCodes, RestCodes, Stream ) :- !.


% Reads a single ISO-8859-1 character
read_code( 'ISO-8859-1', Stream, Code, RestStream ) :- !,
	read_octet( Stream, Code, RestStream ).	

get_byte( Stream, Byte ) :-
	get_code( Stream, Byte  ).


read_file( File, _ ) :-
	open( File, read, Stream ),
      read_all( Stream ),
	close( Stream ).

read_all( Stream ) :-
	get_code( Stream, Byte ),
	( Byte = -1 -> true; read_all( Stream ) ).



%% Support for UTF-8
% Reads a complete stream, returning a list of characters
read_stream( 'UTF-8', Stream, Codes, RestCodes, RestStream ) :- !,
	read_utf8_stream( Stream, ReadCodes, RestCodes, RestStream ),
	skip_bom( ReadCodes, Codes ).

skip_bom( [65279|Codes], Codes ) :- !.
skip_bom( Codes, Codes ) :- !.

read_utf8_stream( Stream, [Code|NextCodes], RestCodes, RestStream ) :-
	read_utf8_code( Stream, Code, AuxStream ), !,
	read_utf8_stream( AuxStream, NextCodes, RestCodes, RestStream ).
read_utf8_stream( Stream, RestCodes, RestCodes, Stream ) :- !.


/*
read_utf8_stream( Stream, OutCodes, RestCodes, RestStream ) :-
	read_utf8_code( Stream, Code, AuxStream ) -> 
		OutCodes = [Code|NextCodes], !, read_utf8_stream( AuxStream, NextCodes, RestCodes, RestStream )
	;
		OutCodes = RestCodes, RestStream = Stream	
	.
*/

read_code( 'UTF-8', Stream, Code, RestStream ) :- !,
	read_utf8_code( Stream, Code, RestStream ).

read_utf8_code( Stream, Code, RestStream ) :-
	read_octet( Stream, Oct1, AuxStream ), !,
	read_utf8_code( Oct1, AuxStream, Code, RestStream ).


read_utf8_code( O1, Stream, O1, Stream ) :-
	O1 < 16'80, !.
read_utf8_code( O1, Stream, Code, RestStream ) :-
	O1 >= 16'C0, O1 =< 16'DF, !,
	read_octet( Stream, O2, RestStream ),
	Code is (( O1 /\ 16'1F ) << 6) \/ ( O2 /\ 16'3F).
read_utf8_code( O1, Stream, Code, RestStream  ) :-
	O1 =< 16'EF, !,					% O1 >= 16'E0
	read_octet( Stream, O2, AuxStream ),
	read_octet( AuxStream, O3, RestStream ),
	Code is (( O1 /\ 16'0F ) << 12) \/ (( O2 /\ 16'3F ) << 6) \/ ( O3 /\ 16'3F).
read_utf8_code( O1, Stream, Code, RestStream  ) :-
	O1 =< 16'F7, !,					% O1 >= 16'F0
	read_octet( Stream, O2, AuxStream1 ),
	read_octet( AuxStream1, O3, AuxStream2 ),
	read_octet( AuxStream2, O4, RestStream ),
	Code  is (( O1 /\ 16'07 ) << 18) \/ (( O2 /\ 16'3F ) << 12) \/ (( O3 /\ 16'3F ) << 6) \/ ( O4 /\ 16'3F).

/*
read_utf8_code( O1, Stream, Code, RestStream ) :-
	O1 < 16'80, !, Code = O1, Stream = RestStream 
; 
	O1 >= 16'C0, O1 =< 16'DF, !,
	read_octet( Stream, O2, RestStream ),
	Code is (( O1 /\ 16'1F ) << 6) \/ ( O2 /\ 16'3F)
;
	O1 =< 16'EF, !,					% O1 >= 16'E0
	read_octet( Stream, O2, AuxStream ),
	read_octet( AuxStream, O3, RestStream ),
	Code is (( O1 /\ 16'0F ) << 12) \/ (( O2 /\ 16'3F ) << 6) \/ ( O3 /\ 16'3F)
;
	O1 =< 16'F7, !,					% O1 >= 16'F0
	read_octet( Stream, O2, AuxStream1 ),
	read_octet( AuxStream1, O3, AuxStream2 ),
	read_octet( AuxStream2, O4, RestStream ),
	Code  is (( O1 /\ 16'07 ) << 18) \/ (( O2 /\ 16'3F ) << 12) \/ (( O3 /\ 16'3F ) << 6) \/ ( O4 /\ 16'3F).
*/

%% Support for UTF-16 Big Endian
read_stream( 'UTF-16BE', Stream, Codes, RestCodes, RestStream ) :- !,
	read_utf16be_stream( Stream, ReadCodes, RestCodes, RestStream ),
	skip_bom( ReadCodes, Codes ).


read_code( 'UTF-16BE', Stream, Code, RestStream ) :- !,
	read_utf16be_code( Stream, Code, RestStream ).

read_utf16be_stream( Stream, [Code|NextCodes], RestCodes, RestStream ) :-
	read_utf16be_code( Stream, Code, AuxStream ), !,
	read_utf16be_stream( AuxStream, NextCodes, RestCodes, RestStream ).

read_utf16be_stream( Stream, RestCodes, RestCodes, Stream ) :- !.


read_utf16be_code( Stream, Code, RestStream ) :-
	read_16be( Stream, Code1, AuxStream ),
	read_utf16be_code( Code1, AuxStream, Code, RestStream ).


read_utf16be_code( Code, Stream, Code, Stream ) :-
	(Code < 16'D800 ; Code > 16'DFFF, Code < 16'FFFE), !. 
read_utf16be_code( W1, Stream, Code, RestStream ) :-
	W1 >= 16'D800, W1 =< 16'DBFF, !,
	read_16be( Stream, W2, RestStream ),
	W2 >= 16'DC00, W2 =< 16'DFFF,
	Code is (((W1 /\ 16'03FF) << 10 ) \/ ( W2 /\16'03FF )) + 16'10000.

read_16be( Stream, Code, RestStream ) :-
	read_octet( Stream, Big, AuxStream ),
	read_octet( AuxStream, Little, RestStream ),
	Code is Big << 8 \/ Little.


%% Support for UTF-16 Little Endian

read_stream( 'UTF-16LE', Stream, Codes, RestCodes, RestStream ) :- !,
	read_utf16le_stream( Stream, ReadCodes, RestCodes, RestStream ),
	skip_bom( ReadCodes, Codes ).

read_code( 'UTF-16LE', Stream, Code, RestStream ) :- !,
	read_utf16be_code( Stream, Code, RestStream ).


read_utf16le_stream( Stream, [Code|NextCodes], RestCodes, RestStream ) :-
	read_utf16le_code( Stream, Code, AuxStream ), !,
	read_utf16le_stream( AuxStream, NextCodes, RestCodes, RestStream ).

read_utf16le_stream( Stream, RestCodes, RestCodes, Stream ) :- !.

read_utf16le_code( Stream, Code, RestStream ) :-
	read_16le( Stream, Code1, AuxStream ),
	read_utf16le_code( Code1, AuxStream, Code, RestStream ).

read_utf16le_code( Code, Stream, Code, Stream ) :-
	(Code < 16'D800 ; Code > 16'DFFF, Code < 16'FFFE), !. 
read_utf16le_code( W1, Stream, Code, RestStream ) :-
	W1 >= 16'D800, W1 =< 16'DBFF, !,
	read_16le( Stream, W2, RestStream ),
	W2 >= 16'DC00, W2 =< 16'DFFF,
	Code is (((W1 /\ 16'03FF) << 10 ) \/ ( W2 /\16'03FF )) + 16'10000.

read_16le( Stream, Code, RestStream ) :-
	read_octet( Stream, Little, AuxStream ),
	read_octet( AuxStream, Big, RestStream ),
	Code is Big << 8 \/ Little.


%% Support for UTF-16 using BOMs

read_stream( 'UTF-16', Stream, Codes, RestCodes, RestStream ) :- !,
	read_16le( Stream, BOM, AuxStream ),
	read_utf16_stream( BOM, AuxStream, Codes, RestCodes, RestStream ).

read_utf16_stream( 65279, Stream, Codes, RestCodes, RestStream ) :- !,
	read_utf16le_stream( Stream, Codes, RestCodes, RestStream ) .
read_utf16_stream( 65534, Stream, Codes, RestCodes, RestStream ) :- !,
	read_utf16be_stream( Stream, Codes, RestCodes, RestStream ) .
read_utf16_stream(_,_,_,_,_) :- 
	throw( iostream( 'Expecting BOM in UTF-16 stream' ) ).



