/*
 * GQradio
 * (C) 2004 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqradio.h"
#include "preferences.h"

#include "display.h"
#include "io_radio.h"
#include "mixer.h"
#include "preset.h"
#include "ui2_editor.h"
#include "ui2_main.h"
#include "ui_fileops.h"
#include "ui_menu.h"
#include "ui_misc.h"
#include "ui_tabcomp.h"
#include "ui_utildlg.h"
#include "ui_tree_edit.h"


#define PREF_WINDOW_DEFAULT_WIDTH  350
#define PREF_WINDOW_DEFAULT_HEIGHT 450


enum {
	SKIN_COLUMN_PATH = 0,
	SKIN_COLUMN_NAME,
	SKIN_COLUMN_COUNT
};

enum {
	PRESET_COLUMN_NUMBER = 0,
	PRESET_COLUMN_FREQUENCY,
	PRESET_COLUMN_DESCRIPTION,
	PRESET_COLUMN_PATH,
	PRESET_COLUMN_COUNT
};


static GtkWidget *configwindow;
static GtkWidget *mixer_entry;
static GtkWidget *default_skin_entry;
static GtkWidget *preset_view;
static GtkWidget *device_entry;

static gint c_slik_smart_placement;
static gint c_slik_remember_position;
static gint c_mixer_device_id;
static gint c_wm_decorations;
static gint c_slik_double_size;
static gint c_description_scrolls_always;
static gint c_skinned_menus_enable;

static gint c_slik_colorshift_on;
static gint c_slik_colorshift_r;
static gint c_slik_colorshift_g;
static gint c_slik_colorshift_b;
static gint c_slik_colorshift_a;

static gint c_slik_transparency_force;
static gint c_slik_transparency_force_a;

static gint c_use_reported_freq_limits;

static gint c_freq_step;

static gint c_radio_volume_boost;


static void create_config_window(gint start_tab);


static void hscale_centered_cb(GtkObject *adj, gpointer data)
{
	gint *c_n = data;

	*c_n = (gint)(GTK_ADJUSTMENT(adj)->value / 200.0 * 255.0 + 128.0);
}

static void hscale_cb(GtkObject *adj, gpointer data)
{
	gint *c_n = data;

	*c_n = (gint)(GTK_ADJUSTMENT(adj)->value / 100.0 * 255.0);
}

static GtkWidget *hscale_new(GtkWidget *box, const gchar *text, gint n, gint *c_n, gint centered)
{
	GtkObject *adj;
        GtkWidget *hscale;

	*c_n = n;

	if (centered)
		{
		hscale = ui_edit_spin_new_with_scale(box, text, -100, 100, &adj);
		gtk_adjustment_set_value(GTK_ADJUSTMENT(adj), ((float)n - 128.0) / 255.0 * 200.0);
		}
	else
		{
		hscale = ui_edit_spin_new_with_scale(box, text, 0, 100, &adj);
		gtk_adjustment_set_value(GTK_ADJUSTMENT(adj), (float)n / 255.0 * 100.0);
		}

	if (centered)
		{
	        g_signal_connect(G_OBJECT(adj), "value_changed",
				 G_CALLBACK(hscale_centered_cb), c_n);
		}
	else
		{
		g_signal_connect(G_OBJECT(adj), "value_changed",
				 G_CALLBACK(hscale_cb), c_n);
		}

	return hscale;
}

static gint preset_view_frequency_cb(TreeEditData *ted, const gchar *oldname, const gchar *newname, gpointer data)
{
	GtkTreeModel *store;
	GtkTreeIter iter;
	guint32 freq;
	gchar *old_path;

	freq = radio_freq_from_text(newname);

	store = gtk_tree_view_get_model(ted->tree);
	gtk_tree_model_get_iter(store, &iter, ted->path);
	gtk_tree_model_get(store, &iter, PRESET_COLUMN_PATH, &old_path, -1);
	if (freq < radio_limit_get_lower() || freq > radio_limit_get_upper())
		{
		gtk_list_store_set(GTK_LIST_STORE(store), &iter,
				   PRESET_COLUMN_FREQUENCY, NULL,
				   PRESET_COLUMN_DESCRIPTION, NULL,
				   PRESET_COLUMN_PATH, "clear", -1);
		}
	else
		{
		gchar *new_path;
		gchar *freq_text;
		gchar *tmp;

		if (old_path)
			{
			tmp = radio_path_to_title(old_path);
			}
		else
			{
			tmp = g_strdup(preset_get_description(tree_path_to_row(ted->path)));
			}
		new_path = radio_freq_to_path(freq, tmp);
		g_free(tmp);

		freq_text = radio_freq_to_text(freq);
		gtk_list_store_set(GTK_LIST_STORE(store), &iter,
				   PRESET_COLUMN_FREQUENCY, freq_text,
				   PRESET_COLUMN_PATH, new_path, -1);
		g_free(freq_text);
		g_free(new_path);
		}

	g_free(old_path);

	return FALSE;
}

static gint preset_view_description_cb(TreeEditData *ted, const gchar *oldname, const gchar *newname, gpointer data)
{
	GtkTreeModel *store;
	GtkTreeIter iter;
	gchar *old_path;
	gchar *new_path;

	store = gtk_tree_view_get_model(ted->tree);
	gtk_tree_model_get_iter(store, &iter, ted->path);
	gtk_tree_model_get(store, &iter, PRESET_COLUMN_PATH, &old_path, -1);
	if (old_path)
		{
		new_path = radio_freq_to_path(radio_path_to_freq(old_path), newname);
		}
	else
		{
		new_path = radio_freq_to_path(preset_get_freq(tree_path_to_row(ted->path)), newname);
		}

	gtk_list_store_set(GTK_LIST_STORE(store), &iter,
			   PRESET_COLUMN_DESCRIPTION, newname,
			   PRESET_COLUMN_PATH, new_path, -1);

	g_free(old_path);
	g_free(new_path);

	return FALSE;
}

static gint preset_view_press_cb(GtkWidget *view, GdkEventButton *event, gpointer data)
{
	GtkTreeModel *store;
	GtkTreeViewColumn *tcolumn;
	GtkTreePath *tpath;
	GtkTreeIter iter;
	GList *list;
	gint row = -1;
	gint column;
	gint ret = FALSE;

	if (!gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(view), event->x, event->y,
					   &tpath, &tcolumn, NULL, NULL))
		{
		return FALSE;
		}

	row = tree_path_to_row(tpath);

	list = gtk_tree_view_get_columns(GTK_TREE_VIEW(view));
	column = g_list_index(list, tcolumn);
	g_list_free(list);

	store = gtk_tree_view_get_model(GTK_TREE_VIEW(view));
	gtk_tree_model_get_iter(store, &iter, tpath);

	if (column == PRESET_COLUMN_FREQUENCY)
		{
		gchar *text;
		gchar *old_path;

		gtk_tree_model_get(store, &iter,
				   PRESET_COLUMN_FREQUENCY, &text,
				   PRESET_COLUMN_PATH, &old_path, -1);

		if (event->button == 1)
			{
			tree_edit_by_path(GTK_TREE_VIEW(view), tpath, column, text,
					  preset_view_frequency_cb, NULL);

			ret = TRUE;
			}
		else if (event->button == 2 || event->button == 3 ||
			 event->button == 4 || event->button == 5)
			{
			guint32 freq;
			gchar *new_path;
			gchar *tmp;

			freq = (old_path) ? radio_path_to_freq(old_path) : preset_get_freq(row);

			if (event->button == 4 || event->button == 3)
				{
				freq += RADIO_SCAN_INCREMENT;
				}
			else
				{
				freq -= RADIO_SCAN_INCREMENT;
				}
			freq = radio_freq_clamp_to_increment(freq, RADIO_SCAN_INCREMENT);
			freq = CLAMP(freq, radio_limit_get_lower(), radio_limit_get_upper());

			tmp = (old_path) ? radio_path_to_title(old_path) : g_strdup(preset_get_description(row));
			new_path = radio_freq_to_path(freq, tmp);
			g_free(tmp);

			tmp = radio_freq_to_text(freq);
			gtk_list_store_set(GTK_LIST_STORE(store), &iter,
					   PRESET_COLUMN_FREQUENCY, tmp,
					   PRESET_COLUMN_PATH, new_path, -1);
			g_free(tmp);
			g_free(new_path);
			g_free(text);
			g_free(old_path);

			ret = TRUE;
			}
		}
	else if (column == 2)
		{
		if (event->button == 1)
			{
			gchar *text;

			gtk_tree_model_get(store, &iter, PRESET_COLUMN_DESCRIPTION, &text, -1);
			tree_edit_by_path(GTK_TREE_VIEW(view), tpath, column, text,
                                          preset_view_description_cb, NULL);
			g_free(text);

			ret = TRUE;
			}
		}

	gtk_tree_path_free(tpath);

	return ret;
}

static void preset_list_apply(GtkWidget *view)
{
	GtkTreeModel *store;
	GtkTreeIter iter;
	gint valid;
	gint i;
	gint change = FALSE;

	store = gtk_tree_view_get_model(GTK_TREE_VIEW(view));

	valid = gtk_tree_model_get_iter_first(store, &iter);
	i = 0;
	while (i < PRESET_LIST_SIZE && valid)
		{
		gchar *path;

		gtk_tree_model_get(store, &iter, PRESET_COLUMN_PATH, &path, -1);
		if (path)
			{
			if (strcmp(path, "clear") == 0)
				{
				preset_clear(i);
				}
			else
				{
				preset_set_by_path(i, path);
				}
			change = TRUE;
			g_free(path);
			}

		valid = gtk_tree_model_iter_next(store, &iter);
		i++;
		}

	if (change)
		{
		if (!preset_is_freq(preset, frequency))
			{
			preset = preset_find_freq(frequency);
			display_update_preset(FALSE, preset);
			}
		display_preset_list_update(-1);
		}
}

static void populate_preset_list(GtkListStore *store)
{
	gint i;

	for (i = 0; i < PRESET_LIST_SIZE; i++)
		{
		GtkTreeIter iter;
		guint32 freq;
		gchar *freq_text = NULL;
		const gchar *desc;

		freq = preset_get_freq(i);
		desc = preset_get_description(i);

		if (freq > 0)
			{
			freq_text = radio_freq_to_text(preset_get_freq(i));
			}

		gtk_list_store_append(store, &iter);
		gtk_list_store_set(store, &iter,
				   PRESET_COLUMN_NUMBER, i + 1,
				   PRESET_COLUMN_FREQUENCY, freq_text,
				   PRESET_COLUMN_DESCRIPTION, desc,
				   PRESET_COLUMN_PATH, NULL, -1);


		g_free(freq_text);
		}
}

static void mixer_device_id_cb(GtkWidget *combo, gpointer data)
{
	GtkTreeModel *store;
	GtkTreeIter iter;
	gchar *dev_text;
	gint d;

	store = gtk_combo_box_get_model(GTK_COMBO_BOX(combo));
	if (!gtk_combo_box_get_active_iter(GTK_COMBO_BOX(combo), &iter)) return;

	gtk_tree_model_get(store, &iter, 0, &dev_text, -1);
	if (!dev_text) return;

	d = get_mixer_device_id(dev_text);
	if (d >= 0) c_mixer_device_id = d;

	g_free(dev_text);
}

static void frequency_spin_cb(GtkWidget *spin, gpointer data)
{
	gint *c_n = data;

	*c_n = (gint)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spin)) * 10000;
}

static void skin_item_selected(GtkTreeSelection *selection, gpointer data)
{
	GtkTreeModel *store;
	GtkTreeIter iter;
	gchar *path;

	if (!gtk_tree_selection_get_selected(selection, &store, &iter)) return;

	gtk_tree_model_get(store, &iter, SKIN_COLUMN_PATH, &path, -1);
	gtk_entry_set_text(GTK_ENTRY(default_skin_entry), (path) ? path : "");
	g_free(path);
}

static gint sort_skin_list_cb(void *a, void *b)
{
	const gchar *fa = filename_from_path((gchar *)a);
	const gchar *fb = filename_from_path((gchar *)b);
	if (!fa || !fb) return 0;
	return strcmp(fa, fb);
}

static GList *skin_list_add_dir(GList *list, const gchar *skinpath)
{
	GList *dirs;

	if (!path_list(skinpath, NULL, &dirs)) return list;

	return g_list_concat(list, dirs);
}

static void populate_skin_list(GtkListStore *store)
{
	gchar *skinpath;
	GList *skin_list = NULL;
	GList *work;
	GtkTreeIter iter;

	/* add default skin */
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter,
			   SKIN_COLUMN_PATH, NULL,
			   SKIN_COLUMN_NAME, _("None (default)"), -1);

	/* add home dir skins */
	skinpath = g_strconcat(homedir(), "/", GQRADIO_RC_DIR_SKIN, NULL);
	skin_list = skin_list_add_dir(skin_list, skinpath);
	g_free(skinpath);

	/* add system dir skins */
	skin_list = skin_list_add_dir(skin_list, GQRADIO_SKINDIR);

	skin_list = g_list_sort(skin_list, (GCompareFunc) sort_skin_list_cb);

	work = skin_list;
	while (work)
		{
		gchar *path = work->data;
		gchar *skindata_file;

		work = work->next;

		if (strcmp(path, GQRADIO_SKINDIR "/default") == 0)
			{
			continue;
			}

		skindata_file = g_strconcat(path, "/skindata", NULL);
		if (isfile(skindata_file))
			{
			gtk_list_store_append(store, &iter);
			gtk_list_store_set(store, &iter,
					   SKIN_COLUMN_PATH, path,
					   SKIN_COLUMN_NAME, filename_from_path(path), -1);
			}
		g_free(skindata_file);
		}

	path_list_free(skin_list);
}

void config_window_show(gint start_tab)
{
	if (configwindow)
		{
		gdk_window_raise(configwindow->window);
		return;
		}

	create_config_window(start_tab);
	gtk_widget_show_all(configwindow);
}

static void config_window_hide(void)
{
	gtk_widget_destroy(configwindow);
	configwindow = NULL;
}

static void config_window_close_cb(GtkWidget *widget, gpointer data)
{
	config_window_hide();
}

static gint config_window_delete_cb(GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	config_window_hide();
	return TRUE;
}

static void config_window_apply_cb(GtkWidget *widget, gpointer data)
{
	const gchar *tmp;
	gchar *buf = NULL;
	gint skin_changed = FALSE;
	gint need_init = FALSE;

	if (slik_double_size != c_slik_double_size) skin_changed = TRUE;

	if (slik_colorshift_on != c_slik_colorshift_on)
		{
		slik_colorshift_on = c_slik_colorshift_on;
		skin_changed = TRUE;
		}

	if (slik_colorshift_r != c_slik_colorshift_r ||
	    slik_colorshift_g != c_slik_colorshift_g ||
	    slik_colorshift_b != c_slik_colorshift_b ||
	    slik_colorshift_a != c_slik_colorshift_a )
		{
		slik_colorshift_r = c_slik_colorshift_r;
		slik_colorshift_g = c_slik_colorshift_g;
		slik_colorshift_b = c_slik_colorshift_b;
		slik_colorshift_a = c_slik_colorshift_a;
		if (slik_colorshift_on) skin_changed = TRUE;
		}

	if (slik_transparency_force != c_slik_transparency_force)
		{
		slik_transparency_force = c_slik_transparency_force;
		skin_changed = TRUE;
		}

	if (slik_transparency_force_a != c_slik_transparency_force_a)
		{
		slik_transparency_force_a = c_slik_transparency_force_a;
		if (slik_transparency_force) skin_changed = TRUE;
		}

	slik_double_size = c_slik_double_size;

	buf = remove_trailing_slash(gtk_entry_get_text(GTK_ENTRY(default_skin_entry)));
	if ( (buf && main_window->skin_path && strcmp(buf, main_window->skin_path) != 0 && strlen(buf) > 0) ||
		(!main_window->skin_path && buf && strlen(buf) > 0) )
		{
		/* change to the new skin */
		ui_skin_load(main_window, buf, NULL);
		tab_completion_append_to_history(default_skin_entry, buf);
		}
	else if (main_window->skin_path && buf && strlen(buf) == 0)
		{
		ui_skin_load(main_window, NULL, NULL);
		}
	else if (skin_changed)
		{
		ui_skin_load(main_window, main_window->skin_path, main_window->skin_mode_key);
		}
	g_free(buf);

	preset_list_apply(preset_view);

	tmp = gtk_entry_get_text(GTK_ENTRY(device_entry));
	if (( (strlen(tmp) > 0) != (radio_custom_device != NULL) ) ||
	    (radio_custom_device && strcmp(tmp, radio_custom_device) != 0) )
		{
		g_free(radio_custom_device);
		if (strlen(tmp) > 0)
			{
			radio_custom_device = g_strdup(tmp);
			}
		else
			{
			radio_custom_device = NULL;
			}
		need_init = TRUE;
		}

	freq_step = c_freq_step;

	if (mixer_device_id != c_mixer_device_id)
		set_mixer_device(c_mixer_device_id);
	mixer_device_id = c_mixer_device_id;

	if (mixer_command)
		{
		g_free(mixer_command);
		mixer_command = NULL;
		}
	tmp = gtk_entry_get_text(GTK_ENTRY(mixer_entry));
	if (tmp && strlen(tmp) > 0) mixer_command = g_strdup(tmp);

	slik_smart_placement = c_slik_smart_placement;
	slik_remember_position = c_slik_remember_position;
	wm_decorations = c_wm_decorations;

	if (description_scrolls_always != c_description_scrolls_always)
		{
		description_scrolls_always = c_description_scrolls_always;
		display_set_description(NULL);
		}

	skinned_menus_enable = c_skinned_menus_enable;

	gdk_window_set_decorations (main_window->window->window, wm_decorations);

	radio_limit_by_card_set(c_use_reported_freq_limits);

	if (need_init)
		{
		radio_shutdown();
		radio_set_device(radio_custom_device);
		if (radio_test(TRUE))
			{
			radio_startup(frequency);
			if (main_window->skin_mode_key && strcmp(main_window->skin_mode_key, "skindata_error") == 0)
				{	
				ui_skin_mode_set(main_window, NULL);
				}
			}
		else
			{
			buf = g_strdup_printf(_("Failed to start %s"), (radio_custom_device) ? radio_custom_device : RADIO_DEVICE);
			warning_dialog(_("Radio initialization error"), buf,
				       GTK_STOCK_DIALOG_ERROR, widget);
			g_free(buf);
			}
		}

	if (radio_volume_boost != c_radio_volume_boost)
		{
		radio_volume_boost = c_radio_volume_boost;

		if (radio_volume_boost && !muted)
			{
			mute_set(TRUE);
			mute_set(FALSE);
			}
		}
}

static void config_window_ok_cb(GtkWidget *widget, gpointer data)
{
	config_window_apply_cb(widget, data);
	config_window_hide();
}

static void create_config_window(gint start_tab)
{
	GtkWidget *box;
	GtkWidget *notebook;
	GtkWidget *group;
	GtkWidget *button;
	GtkWidget *ct_button;
	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkWidget *label;
	GtkListStore *store;
	GtkWidget *skin_view;
	GtkWidget *combo;
	GList *devices;
	GList *list;
	gint lh, cnt;
	GtkWidget *tabcomp;
	gchar *text;

	configwindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_type_hint(GTK_WINDOW(configwindow), GDK_WINDOW_TYPE_HINT_DIALOG);
	gtk_window_set_wmclass(GTK_WINDOW(configwindow), "preferences", "GQradio");
	g_signal_connect(G_OBJECT(configwindow), "delete_event",
			 G_CALLBACK(config_window_delete_cb), NULL);
	gtk_window_set_resizable(GTK_WINDOW(configwindow), TRUE);
	gtk_window_set_default_size(GTK_WINDOW(configwindow), PREF_WINDOW_DEFAULT_WIDTH, PREF_WINDOW_DEFAULT_HEIGHT);
	gtk_window_set_title(GTK_WINDOW(configwindow), _("GQradio preferences"));
	gtk_container_set_border_width(GTK_CONTAINER(configwindow), PREF_PAD_BORDER);

#if 0
	window_set_icon(configwindow, NULL, NULL);
#endif

	box = gtk_vbox_new(FALSE, PREF_PAD_BUTTON_SPACE);
	gtk_container_add(GTK_CONTAINER(configwindow), box);
	gtk_widget_show(box);

	hbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
	gtk_box_set_spacing(GTK_BOX(hbox), PREF_PAD_BUTTON_GAP);
	gtk_box_pack_end(GTK_BOX(box), hbox, FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	button = pref_button_new(NULL, GTK_STOCK_OK, NULL, FALSE,
				 G_CALLBACK(config_window_ok_cb), NULL);
	gtk_container_add(GTK_CONTAINER(hbox), button);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button);
	gtk_widget_show(button);

	ct_button = button;

	button = pref_button_new(NULL, GTK_STOCK_APPLY, NULL, FALSE,
				 G_CALLBACK(config_window_apply_cb), NULL);
	gtk_container_add(GTK_CONTAINER(hbox), button);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_show(button);

	button = pref_button_new(NULL, GTK_STOCK_CANCEL, NULL, FALSE,
			 	 G_CALLBACK(config_window_close_cb), NULL);
	gtk_container_add(GTK_CONTAINER(hbox), button);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_show(button);

	if (!generic_dialog_get_alternative_button_order(configwindow))
		{
		gtk_box_reorder_child(GTK_BOX(hbox), ct_button, -1);
		}

	notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(box), notebook, TRUE, TRUE, 0);

	/* general tab */

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), PREF_PAD_BORDER);
	label = gtk_label_new(_("General"));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, label);
	gtk_widget_show(vbox);

	group = pref_group_new(vbox, FALSE, _("Window"), GTK_ORIENTATION_VERTICAL);

	pref_checkbox_new_int(group, _("Smart window placement"),
			      slik_smart_placement, &c_slik_smart_placement);
	pref_checkbox_new_int(group, _("Remember window position"),
			      slik_remember_position, &c_slik_remember_position);
	pref_checkbox_new_int(group, _("Enable window manager decorations"),
			      wm_decorations, &c_wm_decorations);
	pref_checkbox_new_int(group, _("Double sized window"),
			      slik_double_size, &c_slik_double_size);

	group = pref_group_new(vbox, FALSE, _("Text display"), GTK_ORIENTATION_VERTICAL);

	pref_checkbox_new_int(group, _("Always scroll description"),
			      description_scrolls_always, &c_description_scrolls_always);

	group = pref_group_new(vbox, FALSE, _("Volume"), GTK_ORIENTATION_VERTICAL);

	hbox = pref_box_new(group, FALSE, GTK_ORIENTATION_HORIZONTAL, PREF_PAD_SPACE);
	pref_label_new(hbox, _("Mixer launch command:"));
	tabcomp = tab_completion_new(&mixer_entry, mixer_command, NULL, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), tabcomp, TRUE, TRUE, 0);
	gtk_widget_show(tabcomp);

	c_mixer_device_id = mixer_device_id;

	hbox = pref_box_new(group, FALSE, GTK_ORIENTATION_HORIZONTAL, PREF_PAD_SPACE);
	pref_label_new(hbox, _("Volume controls device:"));

	combo = gtk_combo_box_new_text();

	devices = get_mixer_device_list();

	list = devices;

	lh = 0;
	cnt = 0;
	while (list)
		{
		gchar *dev_text = list->data;

		gtk_combo_box_append_text(GTK_COMBO_BOX(combo), dev_text);
		if (get_mixer_device_id(dev_text) == mixer_device_id) lh = cnt;
		cnt++;
		list = list->next;
		}

	g_list_free(devices);

	gtk_combo_box_set_active(GTK_COMBO_BOX(combo), lh);
	g_signal_connect(G_OBJECT(combo), "changed",
			 G_CALLBACK(mixer_device_id_cb), NULL);
        gtk_box_pack_start(GTK_BOX(hbox), combo, TRUE, TRUE, 0);
        gtk_widget_show(combo);

	/* tuner tab */

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), PREF_PAD_BORDER);
	label = gtk_label_new(_("Tuner"));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, label);
	gtk_widget_show(vbox);

	group = pref_group_new(vbox, FALSE, _("Device"), GTK_ORIENTATION_VERTICAL);

	hbox = pref_box_new(group, FALSE, GTK_ORIENTATION_HORIZONTAL, PREF_PAD_SPACE);
	pref_label_new(hbox, _("Device:"));

	device_entry = gtk_entry_new();
	if (radio_custom_device) gtk_entry_set_text(GTK_ENTRY(device_entry), radio_custom_device);
	gtk_box_pack_start(GTK_BOX(hbox), device_entry, TRUE, TRUE, 0);
	gtk_widget_show(device_entry);

	group = pref_group_new(vbox, FALSE, _("Frequency control"), GTK_ORIENTATION_VERTICAL);

	pref_checkbox_new_int(group, _("Use frequency limits reported by driver"),
			      radio_limit_by_card_get(), &c_use_reported_freq_limits);
	c_freq_step = freq_step;
	pref_spin_new(group, _("Frequency step:"), "(x10Khz)",
		      1.0, 100.0, 1.0, 0,
		     (gdouble)freq_step / 10000.0,
		     G_CALLBACK(frequency_spin_cb), &c_freq_step);

	group = pref_group_new(vbox, FALSE, _("Volume"), GTK_ORIENTATION_VERTICAL);

	pref_checkbox_new_int(group, _("Increase tuner volume"),
			      radio_volume_boost, &c_radio_volume_boost);

	/* preset tab */

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), PREF_PAD_BORDER);
	label = gtk_label_new(_("Presets"));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, label);
	gtk_widget_show(vbox);

	store = gtk_list_store_new(4, G_TYPE_INT, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);
	preset_view = ui_edit_list_new(vbox, store, NULL, NULL);
	g_object_unref(store);

	ui_edit_list_add_column(preset_view, _("#"), PRESET_COLUMN_NUMBER, FALSE, FALSE);
	ui_edit_list_add_column(preset_view, _("Frequency"), PRESET_COLUMN_FREQUENCY, FALSE, FALSE);
	ui_edit_list_add_column(preset_view, _("Description"), PRESET_COLUMN_DESCRIPTION, FALSE, FALSE);

	g_signal_connect(G_OBJECT(preset_view), "button_press_event",
			 G_CALLBACK(preset_view_press_cb), NULL);

	populate_preset_list(store);

	/* skin tab */

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), PREF_PAD_BORDER);
	label = gtk_label_new(_("Skins"));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, label);
	gtk_widget_show(vbox);

	group = pref_group_new(vbox, TRUE, _("Skin"), GTK_ORIENTATION_VERTICAL);

	hbox = pref_box_new(group, FALSE, GTK_ORIENTATION_HORIZONTAL, PREF_PAD_SPACE);
	pref_label_new(hbox, _("Skin:"));

	tabcomp = tab_completion_new_with_history(&default_skin_entry,
						  main_window->skin_path ? main_window->skin_path : "",
						  "select_skin", 16, NULL, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), tabcomp, TRUE, TRUE, 0);
	gtk_widget_show(tabcomp);

	/* skin list */
	store = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
	skin_view = ui_edit_list_new(group, store,
				     G_CALLBACK(skin_item_selected), NULL);
	g_object_unref(store);

	ui_edit_list_add_column(skin_view, _("Available skins:"), SKIN_COLUMN_NAME, FALSE, FALSE);

	populate_skin_list(store);

	pref_checkbox_new_int(group, _("Enable menu skins"),
			      skinned_menus_enable, &c_skinned_menus_enable);

	pref_line(group, 0);

	text = g_strconcat(homedir(), "/", GQRADIO_RC_DIR_SKIN, NULL);
	pref_label_new(group, text);

	/* color tab */

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), PREF_PAD_BORDER);
	label = gtk_label_new(_("Color"));
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, label);
	gtk_widget_show(vbox);

	group = pref_group_new(vbox, FALSE, _("Skin Adjustment"), GTK_ORIENTATION_VERTICAL);

	hbox = pref_box_new(group, FALSE, GTK_ORIENTATION_HORIZONTAL, PREF_PAD_SPACE);

	pref_checkbox_new_int(hbox, _("Color tint:"),
			      slik_colorshift_on, &c_slik_colorshift_on);
	hscale_new(hbox, NULL, slik_colorshift_a, &c_slik_colorshift_a, FALSE);
	hscale_new(group, "R", slik_colorshift_r, &c_slik_colorshift_r, TRUE);
	hscale_new(group, "G", slik_colorshift_g, &c_slik_colorshift_g, TRUE);
	hscale_new(group, "B", slik_colorshift_b, &c_slik_colorshift_b, TRUE);

	hbox = pref_box_new(group, FALSE, GTK_ORIENTATION_HORIZONTAL, PREF_PAD_SPACE);

	pref_checkbox_new_int(hbox, _("Transparent:"),
			      slik_transparency_force, &c_slik_transparency_force);
	hscale_new(hbox, NULL, slik_transparency_force_a, &c_slik_transparency_force_a, FALSE);

	gtk_notebook_set_current_page(GTK_NOTEBOOK(notebook), start_tab);
	gtk_widget_show(notebook);
}

