// Copyright 2020-2025 Consensys Software Inc.
// Licensed under the Apache License, Version 2.0. See the LICENSE file for details.

// Code generated by consensys/gnark-crypto DO NOT EDIT

package bls12377

import (
	"github.com/consensys/gnark-crypto/ecc/bls12-377/fp"
	"github.com/consensys/gnark-crypto/ecc/bls12-377/internal/fptower"

	"math/big"
)

//Note: This only works for simple extensions

func g2IsogenyXNumerator(dst *fptower.E2, x *fptower.E2) {
	g2EvalPolynomial(dst,
		false,
		[]fptower.E2{
			{
				A0: fp.Element{3551783286045471771, 15672698349814166255, 7201714524012399751, 9685135133462022557, 11459791422433132438, 34279211894444158},
				A1: fp.Element{6733784119909728882, 8027365598504339614, 9826395261252013434, 17766961605401961078, 16718790361550578585, 8827310132881948},
			},
			{
				A0: fp.Element{10039326048219096853, 8025585753053690704, 15692757884719051200, 14081267914923412694, 12483400733579637594, 41501995942887693},
				A1: fp.Element{3285050121746765179, 16424976010740556635, 13525960835401060630, 8230390587856081588, 332312595686915068, 58693565636984574},
			},
			{
				A0: fp.Element{17569899329826403508, 9637548884483664645, 11559603533194429416, 509473447889017775, 16843450937194425803, 11820413515158522},
				A1: fp.Element{18082143759519379917, 7350976797508953919, 8718515184478260262, 11424641547646470649, 12610734036362352161, 109931236078585740},
			},
			{
				A0: fp.Element{14344095877598929896, 14770204065590636976, 13870643972022067464, 14327299415926938990, 2100218127689809229, 111985606372347998},
				A1: fp.Element{18166912950538149334, 7903342274102247275, 11235613814926762637, 6048582781848067329, 9865957781737849443, 72257053590751229},
			},
			{
				A0: fp.Element{9289693438943775070, 611273684478921638, 3161020385710416475, 2286858861996231773, 14631078872284738786, 101751379483637100},
				A1: fp.Element{218371821390517888, 3239557307852648611, 7956123978234311251, 4433191957359481551, 7602230667186897987, 36183482319693410},
			},
			{
				A0: fp.Element{7524026848147356443, 15899508093607509001, 9269755160329214834, 12526651159477248728, 3756002781809695765, 101164747683440023},
				A1: fp.Element{6220885675233469595, 1070578225745722143, 9028649589122227273, 241265531361573072, 17722097355684345537, 59426317598728635},
			},
			{
				A0: fp.Element{1147337684740884862, 11995598217907408439, 6334092051568104396, 4490620767408292574, 16484486820571077628, 32056264099725884},
				A1: fp.Element{1626504798254052117, 15843967556282260399, 3222912561813979091, 1092398256542226755, 15613117580878270463, 35759440311789519},
			},
			{
				A0: fp.Element{9456818710433496904, 8738575976265225627, 17481785322204872502, 10252666647792359788, 13007033241788380867, 54062177864773120},
				A1: fp.Element{7857657292788377886, 18159261954362575139, 6957043223229770024, 101887390030524213, 7411469664095682342, 17462521019270966},
			},
			{
				A0: fp.Element{5935437080311313022, 4869272223104979307, 14892623520420170949, 6274040211921387638, 190450687675494048, 100326942952217018},
				A1: fp.Element{4080141112519796596, 14389619698824585061, 15572076811388758383, 5573262239497616930, 5714216936392065098, 10910945353280475},
			},
			{
				A0: fp.Element{12050383421129197508, 7679000367015579641, 4181895471589351098, 4488864303009131705, 1755297417051974713, 33780283701258651},
				A1: fp.Element{11393221592097145268, 10835918896706663346, 4746287363995169177, 6169630568067228482, 11259555703140136465, 52771734216395635},
			},
			{
				A0: fp.Element{4577617767559140265, 568504000295320317, 4739467389388831592, 9537638546299567945, 11669151734363428413, 119097732794758712},
				A1: fp.Element{5691235310984308451, 13601281267571106288, 7819292534586961743, 11652278682059157182, 2231701566242469994, 110966193728844398},
			},
			{
				A0: fp.Element{8957158313166077105, 5669155713645675267, 17333896861619738764, 4892240835897020648, 8492829473755858526, 70767002641220602},
				A1: fp.Element{8355305174234995113, 16726408794609999189, 6819868380250046496, 9302398329327482182, 3371252083110614225, 32903157732774708},
			},
			{
				A0: fp.Element{3562726361589542410, 15642076536163677636, 3828009540728039550, 5513538145598570884, 1202255355797680138, 19209968154720986},
				A1: fp.Element{2433342667414904247, 14430646666116795259, 7881503288178112304, 10086912537277758921, 14321155618236575013, 84224128485096326},
			},
			{
				A0: fp.Element{7543845460144717594, 14101022520017632284, 6419288061290617259, 11326768478076341964, 18432026940412127132, 85359846879027793},
				A1: fp.Element{17142157839755119212, 5159976764710014065, 6393633352893279233, 8672509349035392886, 7859431465567482570, 62163334379266093},
			},
			{
				A0: fp.Element{2954881934412060022, 10898916794193723070, 10915833338735408379, 703491535562714344, 14589964434689495439, 90051679935520087},
				A1: fp.Element{18417998714611085978, 11039054240270137506, 9972891257774348246, 13263552626589315210, 4522668809484918556, 63885995997825101},
			},
			{
				A0: fp.Element{9864134077459603571, 12525567282624341376, 5205345924937688700, 12649124091757575439, 14636003906379491067, 96025183135833306},
				A1: fp.Element{10316403063861314867, 10750711173072110600, 4513675589861596212, 2635673197095740125, 16720951430549947037, 31390298180493148},
			},
			{
				A0: fp.Element{3003952238586646516, 3250841642932517220, 7061834734009117554, 16259415476002355236, 1626926148029432162, 77189086665928784},
				A1: fp.Element{14450051585993059472, 11823315281584113748, 6929304997500454604, 15344892292748160673, 15787768514545706202, 69813057053620435},
			},
			{
				A0: fp.Element{4228110392233474072, 18010581486575392207, 13699345898545483049, 17704744337446085874, 5487913134051621310, 112205105149111207},
				A1: fp.Element{14824585485470590037, 7637124239293942071, 1041102438278370701, 4207950142422409777, 9639439542389544094, 106832755917361143},
			},
			{
				A0: fp.Element{10744145886747796703, 8741979218876153119, 4537642647264646058, 9645243825719833866, 16643801652280184094, 57528843479739428},
				A1: fp.Element{5785935421797206370, 847985697064985249, 11478142027129258160, 12741057482356268413, 10148591557028647803, 106797667835022069},
			},
			{
				A0: fp.Element{6454832842704992412, 9977685523482313420, 413111938383295655, 13594952949000658414, 17807487788385728582, 108984464456649540},
				A1: fp.Element{17048359867690489456, 2859252502285541521, 10818623701995947153, 7550564651763690537, 459333235977057482, 37663478480017988},
			},
			{
				A0: fp.Element{8658783637463168087, 13532094408327228336, 12875492560745818922, 6809856611120372218, 2853890452345629690, 87034840621524077},
				A1: fp.Element{16301307707120111515, 9104654412985804807, 12475785366789695268, 4024450921325678435, 5626908908910088752, 56687303925216568},
			},
			{
				A0: fp.Element{9729408098285946449, 14379177692720606450, 15077053755146607368, 1299576162980711320, 13990524355831736308, 88070216003513722},
				A1: fp.Element{5766436769337206504, 1093294963802231624, 1118315683159380206, 16238633443172785277, 5463682062430385614, 116508026946667814},
			},
			{
				A0: fp.Element{16584249698770136319, 8326278446225484560, 3131917383401484830, 4756633391835977312, 7642636045510739113, 6547192373501023},
				A1: fp.Element{10803045440816594384, 6133821182275761752, 7762705812675926549, 15035799326051880159, 10201360843527298396, 76693252438359195},
			},
			{
				A0: fp.Element{3621702609341817994, 1179514540952803843, 9695126383219869545, 4861853798003230532, 15648444733987506481, 103088924877589738},
				A1: fp.Element{0},
			},
		},
		x)
}

func g2IsogenyXDenominator(dst *fptower.E2, x *fptower.E2) {
	g2EvalPolynomial(dst,
		true,
		[]fptower.E2{
			{
				A0: fp.Element{1063048776114699222, 13419136991291290443, 17670140655952814712, 17007170270485437006, 2714055472280753035, 14919040757258909},
				A1: fp.Element{16666868668043867029, 10413023948165423527, 2513282340300795191, 5135056029366772344, 18074639060500180041, 66260525510488187},
			},
			{
				A0: fp.Element{10940415603117103310, 2845175453419864190, 16971099885047235913, 4517542951822462583, 12015180195047358700, 101962474939260879},
				A1: fp.Element{1812172074217037271, 9040376510937171745, 1548369375982775200, 5323713323104515663, 11625954855074087334, 4220230277540083},
			},
			{
				A0: fp.Element{8797262280793960976, 17802450974932240081, 10157003283304584770, 593225980605733121, 14184353532500093055, 108979035901112461},
				A1: fp.Element{15554837140147424903, 1512405638789788986, 3262333938884419786, 4638305243452530609, 139542405126620667, 54024907639584882},
			},
			{
				A0: fp.Element{11424716963860540790, 13456329776215064369, 1929697779149270213, 6306593112502705131, 13859057472975507250, 41785102954052035},
				A1: fp.Element{7602541488494927932, 4897344463908346766, 17005994653424089316, 17216735712046963938, 3756018940504096168, 5402674048264985},
			},
			{
				A0: fp.Element{2416535722849524790, 4831233808757576698, 17297774284520319797, 6772473332127607735, 18174962172090050489, 58994994344686536},
				A1: fp.Element{15610410799617712282, 9543692017702705957, 8970286522052539731, 12714219237879632746, 2449975609997710437, 10121059733973512},
			},
			{
				A0: fp.Element{15388608950644940213, 15178518612850901922, 4663566445208062329, 3999751164791233677, 7358895960008222342, 33044898311505685},
				A1: fp.Element{2356569524239497521, 4634121806982763800, 13294827248503187097, 18379906191200424608, 13949207972645393879, 37350653623873291},
			},
			{
				A0: fp.Element{4970487283568561993, 8451944303783250587, 10744185545939488302, 297910826254460501, 5663064319185246782, 55864728573479562},
				A1: fp.Element{13676619861655804765, 9740179349809417716, 1447466405783296044, 7262347140551810932, 4460517809397706328, 48696693541469882},
			},
			{
				A0: fp.Element{4218670501850515232, 1300837062036343562, 6322288902222626865, 2517640049285419442, 15933997662514683752, 35742655751559900},
				A1: fp.Element{8671399843829082486, 10727571055999201132, 18066412728811459184, 2077243532292929295, 18402906919639961112, 11053564656556137},
			},
			{
				A0: fp.Element{13549293427847064273, 13454861990004702789, 12625716182603551974, 16289223817658875114, 709786698748164395, 79373748066056979},
				A1: fp.Element{9755261456181601166, 8062867867952015070, 11667298511884909423, 12783693965971962594, 5335701901003645771, 118924945769569072},
			},
			{
				A0: fp.Element{5024330816113629597, 9162185537450801251, 14637535063833510048, 5042964231794706299, 10987285991634226322, 46336617111585333},
				A1: fp.Element{17451698037621354790, 10238258568486351103, 10306732172443683782, 16783673474705190959, 2366024509224094980, 27102949281362657},
			},
			{
				A0: fp.Element{1779719694787807439, 13682444363499832102, 319805876265464201, 10878143779945294209, 5260570358490406259, 111384964548942529},
				A1: fp.Element{10364815279125342799, 5955681787042171859, 16259776976357711022, 698420282628335564, 14458917752432688041, 9951241155641633},
			},
			{
				A0: fp.Element{9860169129499874274, 4710132861939245083, 16729683163954203081, 1163919786098698325, 16999533791398931846, 45007426249965870},
				A1: fp.Element{1554156753268998897, 13001788039766734041, 13083055690099472212, 15457335577166095794, 9225717367159961098, 40091861438122274},
			},
			{
				A0: fp.Element{15170394446649694794, 515688257282406708, 7546053921359572147, 2040402108618036352, 14910633907967741865, 51620635462170312},
				A1: fp.Element{14097816726424606264, 12505549891408832791, 13993079436736795338, 10172463092702817360, 608991345474995671, 70034669163571313},
			},
			{
				A0: fp.Element{4643108199199003526, 15736620484932148216, 15681664113334307244, 17299685843716562967, 13906356132799386736, 93346316071232156},
				A1: fp.Element{4605618257264761423, 18021103018327472901, 16108946406338519358, 9045268489748282166, 5059321869053749600, 110536556169650965},
			},
			{
				A0: fp.Element{11517108407780568371, 7978619811691229504, 9264608640258152436, 8451802924690124465, 4675285626878377699, 109463398958344863},
				A1: fp.Element{3597960821476001565, 3759680844169152876, 14302414818654496990, 17433096654117785124, 17967041042193057544, 85366167313641495},
			},
			{
				A0: fp.Element{9939822412604413185, 13452127979624463736, 5130248585009642508, 3885932386715663181, 5051687816649505884, 77901780572240613},
				A1: fp.Element{519763984047258436, 9600915485628319569, 15140529797299450996, 9971542775239334883, 8807015558507490608, 948222705021672},
			},
			{
				A0: fp.Element{2827339587428175511, 6895834130469522434, 13331059522798479027, 13955187059915735579, 14378880524037078149, 37849146151734053},
				A1: fp.Element{5659129353366385472, 1315670479643534676, 5263307416551198333, 6925817959331280727, 2077163856293267360, 78059303625906606},
			},
			{
				A0: fp.Element{12063007979636530410, 11856928215255593909, 11875394835884852884, 7779285203935234969, 6621016507099994054, 58081312732574672},
				A1: fp.Element{16883486087598545310, 13020319043706888256, 13790070886127375971, 1584348143126996741, 14269751476430075034, 39520851290632316},
			},
			{
				A0: fp.Element{10496651096160116529, 17447330314090714314, 7901980568892205616, 12523643767284114259, 14047697587913173436, 75967257029771325},
				A1: fp.Element{928850053641029162, 14269106311960337185, 16222409674828338261, 5805191570224440459, 18405225593952050465, 81611267238126250},
			},
			{
				A0: fp.Element{11963245856892367752, 10425756598646938656, 18335238837348744770, 15086772981218061512, 16514870314421827966, 31505685309500160},
				A1: fp.Element{12503518650914401765, 9253053408485518576, 7574198415595259890, 9578781703355827570, 4997060354564847196, 4129967821132897},
			},
			{
				A0: fp.Element{2031421031641935199, 11357963626711833272, 3216176135850906142, 18356175605205399499, 14930266308542898024, 45166487627178747},
				A1: fp.Element{4224679465464892278, 5162816687557914011, 9625001313214472999, 9646089844316603379, 3042432097393437616, 116654205222221531},
			},
			{
				A0: fp.Element{707631102761073363, 4675040476129639901, 7917747947488915690, 10109499984363985170, 8993410474198507338, 72712009800189820},
				A1: fp.Element{13149412355826661761, 13738163990683470943, 2035497293527985654, 18119267602401018212, 2631647798265895027, 2795783216290358},
			},
		},
		x)
}

func g2IsogenyYNumerator(dst *fptower.E2, x *fptower.E2, y *fptower.E2) {
	var _dst fptower.E2
	g2EvalPolynomial(&_dst,
		false,
		[]fptower.E2{
			{
				A0: fp.Element{17926225976816550695, 14023720841551579195, 6357178813752170559, 1429363592569423041, 10398153225153858948, 33711798988254397},
				A1: fp.Element{2451854115792992988, 8983506616720050336, 2910564589437158732, 14075622914381394491, 9210830493684175792, 3356683084380210},
			},
			{
				A0: fp.Element{5363401642911389912, 1675875184098052872, 15320530934239994732, 16287584047609698426, 10234580471621189795, 40152057551620421},
				A1: fp.Element{2205557896318444105, 4065007897235623968, 4049591920488634456, 16004168804099107709, 13290522222905374988, 70089339901218536},
			},
			{
				A0: fp.Element{1887285394871164447, 7142690945990485012, 11190732658400833066, 6881717282640585612, 12116088968278807379, 103071704289665226},
				A1: fp.Element{15968649428224979513, 622837296484333259, 15527184444320989860, 2221190225062639140, 16647213690505955661, 110449865605275384},
			},
			{
				A0: fp.Element{8732294600695513394, 7924365878303510912, 12588020556237861366, 10068029016225118132, 785053188876688927, 120347068207419939},
				A1: fp.Element{14514622768737464893, 5553264648606662629, 12707457796658055665, 6066303778837734141, 8279024849745683367, 3564406469625657},
			},
			{
				A0: fp.Element{10035254538193759291, 9758866910217654439, 2724217928072676653, 3087232989313988901, 14980280964270815877, 34590022796322467},
				A1: fp.Element{15702474365778911803, 7256912742043165018, 1566344877835261304, 4035729625101537726, 16346625826270990512, 84087754076446931},
			},
			{
				A0: fp.Element{8353555275402615800, 6017666668033757053, 13076202439893933084, 1913164921831891521, 13296563472388407395, 43485447354470561},
				A1: fp.Element{12571296156797641372, 13988198528186614994, 5672291450663514913, 7033607850615758657, 16846880911077910300, 44471894806759326},
			},
			{
				A0: fp.Element{1319289899879823718, 1591088360172441240, 7805677496653365276, 5569897284498525554, 5635591756513279404, 74901024143303203},
				A1: fp.Element{10251017843654697682, 10395107644570958416, 3071938417266745181, 16127666859700570668, 5111665418914357408, 2168588371926498},
			},
			{
				A0: fp.Element{7692831217085618076, 6702179445852008930, 11308520252707392151, 12038365346701529390, 7201289518723110646, 24757234241788495},
				A1: fp.Element{17948859968330001914, 10144279887227452625, 5167544611537672341, 4094514978127885079, 17912079766649616973, 45691468580901020},
			},
			{
				A0: fp.Element{3911440855998194160, 13198838254872822648, 13301641437413064797, 11598620320741753383, 6260523097595092689, 78284230929812985},
				A1: fp.Element{7876366075466990529, 13523468790275433139, 982512625724968021, 2122595334378906057, 14803785928219626498, 71415583741911053},
			},
			{
				A0: fp.Element{14923926944529083273, 3172840012894527710, 8293434476071337387, 16713256045335958267, 14057165773167995662, 57626475506313562},
				A1: fp.Element{1362417042314121750, 3304417561071796103, 17626406775105189491, 11748219015558348173, 14151421547261463616, 82460535821434588},
			},
			{
				A0: fp.Element{3641967463568066103, 8959416625663108732, 5147508997828480363, 16811352377666989046, 1441182181356889676, 52012247235371457},
				A1: fp.Element{17686146190561162997, 11457769513035421935, 11629039572857129752, 16013548565551254584, 3489389447040062088, 108111902919606578},
			},
			{
				A0: fp.Element{3810472718885394006, 2981020169366520013, 2816385682378197235, 153866317221550159, 11847618666936524337, 45551005605881971},
				A1: fp.Element{3468095351662354134, 4642197759139584052, 4233923780028877603, 17495173774535575422, 6042084483130094534, 67695926977382460},
			},
			{
				A0: fp.Element{1324953063484934482, 15924484690572898301, 11397027945012096698, 2650928770069584769, 15779309286157100138, 107280353846472153},
				A1: fp.Element{7825813464319090460, 9031312139767459777, 142189155675192148, 6582610291000324889, 17102396003811978688, 22084094988625284},
			},
			{
				A0: fp.Element{5210064249141309135, 1705543797894948713, 2628665872375881350, 13536211673960271710, 18420806283098729881, 115824256193502087},
				A1: fp.Element{15082907751180860242, 6568697878451242210, 16244491021091296231, 17790448340613041754, 7733097786923840729, 14504831353209381},
			},
			{
				A0: fp.Element{1708526764431582271, 13958279355051552323, 14990211071762970223, 7884242526085975541, 16332397605884981291, 80533848446738849},
				A1: fp.Element{14979957683309612745, 6077136825354362762, 8344075647158254715, 10162044015152162839, 5892600246835629906, 87516868768986919},
			},
			{
				A0: fp.Element{13823384149985104375, 400062563487168190, 14946453239900597346, 3063239780002983931, 12307299790663558215, 82575345472847758},
				A1: fp.Element{4955387593349539956, 11580263215419679285, 13810204272372323220, 15118104627613044122, 5709821153764726112, 97792497530186865},
			},
			{
				A0: fp.Element{8029106480647627463, 9339835209992361362, 2257747803917390435, 13841766612482081060, 11846105036367819521, 10985089103756089},
				A1: fp.Element{12896564185217823813, 11446288085903243988, 7990576940036837900, 3323202217483830000, 8230734762253073878, 36590510406546600},
			},
			{
				A0: fp.Element{9258620980666570289, 14799769271847025124, 3337623733744503313, 7847288847664005088, 4443979768963902018, 60018570132140318},
				A1: fp.Element{2675246192261018596, 15150250319428322656, 8044595947793549351, 17819017498503740634, 8168003399719773701, 50802205070212383},
			},
			{
				A0: fp.Element{6416675551757596528, 11137110755893373387, 12196234615749865580, 10261229930898283794, 16443034629854739148, 92762661836396101},
				A1: fp.Element{2617757450501858457, 13481841765161333192, 14399351126797435540, 14845648777279341476, 16479218442290092360, 58694504406226241},
			},
			{
				A0: fp.Element{2498292483400110656, 17577335584861186499, 17782334663901894811, 2829649086165738601, 1498570879458573752, 54126946483698861},
				A1: fp.Element{13664305368410221366, 15519373467383236285, 6930373704919785768, 5035259077705758702, 15118622066815350587, 76218461077450180},
			},
			{
				A0: fp.Element{2359464294290896887, 15126005627433822176, 8745471049496239338, 16249169944251666409, 5887779353961924474, 9756279022859315},
				A1: fp.Element{4564346538121733135, 7083224900235365477, 12708889282498498077, 7850548098138279688, 14178174888234998222, 79663491154927524},
			},
			{
				A0: fp.Element{2363371748841007386, 8201543480932182326, 3606722496785934427, 11335897361905574349, 7238564335142183540, 28370763379089425},
				A1: fp.Element{17773120842166679454, 6968817275085296949, 589078002303466881, 11901496965136831973, 17508389603594408067, 105726710734816064},
			},
			{
				A0: fp.Element{7295712044514929446, 1419749005841863626, 15921158613862149232, 9464988326326595083, 4254037448365833139, 69144288462579473},
				A1: fp.Element{2330521880128496868, 6942731841460529291, 16753201799444057524, 5125438220420299042, 743707329901356982, 86418732453789215},
			},
			{
				A0: fp.Element{10073672469351267894, 7530218602938011234, 12150950127635720924, 14699101300080070173, 3657462378365608060, 50677269975209252},
				A1: fp.Element{16217326474471800173, 12182932688121705224, 9523557196391803719, 13559107473982584173, 14264388955497449506, 89155196038187526},
			},
			{
				A0: fp.Element{8500524953883552338, 13471246252006481381, 399342016929624192, 15321780621361720165, 1735807610194144505, 81584295376527434},
				A1: fp.Element{3028140852425956403, 9738549527114127103, 13145547273810927, 15439064114192138046, 15332022320720552951, 84308032823666865},
			},
			{
				A0: fp.Element{8795699974103596314, 642987394252844125, 12559698238980671421, 15439596853334509309, 8192837603484177265, 86858193154220713},
				A1: fp.Element{13665577425741482529, 6924526015867702055, 8773433633434605845, 4385776193759960181, 14751123844375383386, 84259094466106596},
			},
			{
				A0: fp.Element{6308743764871173820, 14529376135552483358, 3992864522868188832, 16018150786687814926, 14942376479240309869, 30721880050281254},
				A1: fp.Element{8640235552523037016, 9835096537876469025, 789797926152341591, 17554386444425767744, 2184317346571194421, 105569965705365467},
			},
			{
				A0: fp.Element{10895141504918788686, 5671269070898752172, 9721970862384110947, 6958416614840799556, 18087813302866953828, 120526621462965167},
				A1: fp.Element{4966785024859028542, 18182687130036955400, 10092534947477547130, 11367367723010839926, 7417818378683193783, 94369912047147779},
			},
			{
				A0: fp.Element{5709922971951703245, 10382852257937442526, 3453842328747730539, 16133368957829378910, 8271517063962590774, 9539719803485949},
				A1: fp.Element{1704900793320796784, 8995901783485359023, 9994857694012530400, 7408202244508772902, 18055801701001909838, 109149704128086904},
			},
			{
				A0: fp.Element{8357460663115985094, 5765296628618444602, 6421674075164890879, 1651079036919805888, 1287520506307076832, 98975482931648514},
				A1: fp.Element{8078944849704442787, 4038716179863104913, 8864619430523621449, 11983583689047803099, 2541042365160408900, 28281336919305267},
			},
			{
				A0: fp.Element{13472601104608613638, 521231156844377864, 4649313698209759510, 5796371833735044995, 8684999192663632207, 51033152603009675},
				A1: fp.Element{3409558318672010377, 10145443448574304058, 3995874611835909177, 15021984820089990008, 15242143999191686238, 73656911605031936},
			},
			{
				A0: fp.Element{9770462122700591808, 17547477902406858790, 14010124614813851222, 4165622522809648336, 16982670942141592785, 28856170353703120},
				A1: fp.Element{15290128055618535720, 17454664547186830394, 2442682438806360163, 18129275600450347134, 4187898517238597724, 22281223126461925},
			},
			{
				A0: fp.Element{12998303454550796060, 5752293956714303316, 6752814987597331313, 3166668351495264669, 9929418606104908953, 29385282975911842},
				A1: fp.Element{8756413506556179286, 143408982899135549, 2705277708630136288, 5154523209628123430, 6318572504564936509, 15532021916109921},
			},
			{
				A0: fp.Element{18250508560718013179, 7486164413457419330, 8732750857092323232, 457033717118918321, 12167888022606617966, 20277559260742340},
				A1: fp.Element{0},
			},
		},
		x)

	dst.Mul(&_dst, y)
}

func g2IsogenyYDenominator(dst *fptower.E2, x *fptower.E2) {
	g2EvalPolynomial(dst,
		true,
		[]fptower.E2{
			{
				A0: fp.Element{2775408832476871526, 9008699192344519496, 1102884431771657931, 11532306452895462867, 5856674524343862704, 99430919144638985},
				A1: fp.Element{15551301223123894338, 11614654532478001117, 17328204268627498271, 15544698294678786409, 1484472732893154418, 94605104690421825},
			},
			{
				A0: fp.Element{15538486723908094425, 10978766348102143722, 15056964343960721360, 8732109610906448146, 8620899699042055528, 1351681984895437},
				A1: fp.Element{3175104032835843731, 9314571151837830613, 14059118678096680542, 2500500275627160283, 18190796245603721369, 49058560079762280},
			},
			{
				A0: fp.Element{2177050294381443304, 18124514285214412099, 9381797198372716589, 8046190315005422985, 9931273159159670369, 8036418831901820},
				A1: fp.Element{1549742439238596983, 15619417821567290543, 9431498181016104480, 2850048082163302555, 17545915875775834651, 72687252788290726},
			},
			{
				A0: fp.Element{16572119218800158686, 13113809656757589233, 12872678318679566545, 3255385198038960565, 17070551903094984362, 19742086936927308},
				A1: fp.Element{12635717383520653743, 12204869003999147728, 8256611698395114158, 12382254302890131233, 603685913325391887, 60927550204070150},
			},
			{
				A0: fp.Element{15497795759006280655, 16847324196958114585, 8218328297664216257, 15353718428023978640, 6357440186573265200, 109435672289072692},
				A1: fp.Element{17126430308003109280, 8510055147921463837, 2424009275722614739, 9284383313721206596, 12867548350984534889, 46750493171897961},
			},
			{
				A0: fp.Element{6993808529266055670, 10024278910497662817, 6638820395273369627, 17550512930522052164, 11971278025880956390, 4218694435301565},
				A1: fp.Element{4117072311900938869, 8052759146924035127, 1883914237959498468, 843437547616490150, 1332392274725871932, 100467415201018114},
			},
			{
				A0: fp.Element{10031273074038578964, 6635740043384123318, 2357760728051263554, 12405057037224522557, 2967360385882286162, 53130165628915609},
				A1: fp.Element{9340276074750471546, 8779557686584984578, 12118682911992514942, 8792287004994786286, 11729922744948342197, 97115621599174349},
			},
			{
				A0: fp.Element{8243258075199662251, 789646305137795307, 9501755430432007632, 15339245131080115010, 3036491630055907252, 4720358405852701},
				A1: fp.Element{6272873484523042114, 18180794113363272037, 10135093694274252446, 13006427779172872075, 12612518697452888675, 4088970499189038},
			},
			{
				A0: fp.Element{12559327402496405011, 13162954855686166920, 2622361684062280170, 16985128559432625018, 10671934355632922492, 22106668982430516},
				A1: fp.Element{10342048742210368049, 4070662490021521195, 9050590024293164750, 6778744574336170333, 16385669267342466637, 110676722503289309},
			},
			{
				A0: fp.Element{8257142434361782318, 11625188549524762434, 4162174183904813140, 15534648919830235837, 15345786270187761745, 49274740272907617},
				A1: fp.Element{17452699565973497082, 1589725784862892127, 17590303103109791779, 8233530623701537904, 717929863606521126, 42679722481449639},
			},
			{
				A0: fp.Element{17135289411523873787, 11725292909156152853, 16347014946116696110, 13323937426977246609, 8703182288833044255, 99079409785168386},
				A1: fp.Element{9130446173903866415, 12010725389601806440, 9146179391205715125, 12338366065383252573, 9136251064810045627, 19315728226188373},
			},
			{
				A0: fp.Element{13754907156191096138, 22846764546901886, 362622052532339515, 17013010775786408901, 13860043181928645305, 41171875858186406},
				A1: fp.Element{16141340011263075417, 18360728019638818576, 16264128300543356196, 12473458415555386384, 18336837302801391285, 63727493440743041},
			},
			{
				A0: fp.Element{3658804449581456923, 1347747705877184398, 3978302900333357541, 5081865331785059868, 1329643100050532471, 64102076251113639},
				A1: fp.Element{6980160750786278877, 11782099681251246419, 5400025369843657828, 9848695260591786723, 7987487093370334558, 27115517650078156},
			},
			{
				A0: fp.Element{408044901411159465, 18068234941175928745, 2992890619264445487, 8810813216534328625, 4005157550725594837, 26065038549738560},
				A1: fp.Element{10975300981822504330, 15715038812214651197, 7602692888794350386, 12917547374269268270, 422938878523833779, 36305463980543648},
			},
			{
				A0: fp.Element{10382180122959422967, 15807168734599653808, 18372044240449882372, 17642941170379477011, 10340644338271517361, 36063527972088465},
				A1: fp.Element{17969075460977832752, 9177853432360197657, 2917850475625504299, 7079850467696515295, 787036529903063845, 107077168732108508},
			},
			{
				A0: fp.Element{6502963492633373835, 9890663567118850708, 12445720805224431135, 6172862156806685987, 16576315346783950860, 5856337016358393},
				A1: fp.Element{4476660199518722374, 3771101137683024451, 489075951782192448, 6489873046594013732, 280944977367484653, 69709094171715534},
			},
			{
				A0: fp.Element{5629815848909521275, 5316587566230943622, 7982019375390215350, 15543803108203970386, 5941855484214597918, 5142365312974746},
				A1: fp.Element{3442509802156923890, 6170315030905476396, 2122926024914179804, 17368567939581660282, 12200117156652989113, 7492215668086454},
			},
			{
				A0: fp.Element{15017092692882720799, 2306085001940265068, 6863750251390498522, 772425548835188093, 7866794496176459304, 119478713276832042},
				A1: fp.Element{539473290914229032, 10680532665300974652, 2401493774971219183, 771791041055281045, 17266300769954562072, 46203199265737132},
			},
			{
				A0: fp.Element{12080185159744170157, 6121786867780957883, 13378966825252880343, 16978399570563241468, 13189121794372052505, 31306179382417939},
				A1: fp.Element{6033582013845463960, 10201637616554513673, 3729832524646428556, 2564427389283182369, 662893124657004215, 101996692818942248},
			},
			{
				A0: fp.Element{14864583376459179927, 16267452113314442715, 14135040057928255187, 712462212063179204, 12089188474131830930, 36841165809084721},
				A1: fp.Element{13089068048776542239, 1568107234484844315, 4262841373517201534, 572974140393742986, 13114372614372436015, 221271375458892},
			},
			{
				A0: fp.Element{15352358256255559128, 14776149476170502093, 13834021062045579807, 16108249515581661111, 7563707564996631205, 86582638494403858},
				A1: fp.Element{16813570010660423536, 3065722476932153407, 8938713923763210470, 2255995557158728394, 8016474455897300271, 84052925693202818},
			},
			{
				A0: fp.Element{15615825726181497195, 5658715128964929728, 13877560309241249627, 15409847795806183219, 382506108092986341, 33112756833083889},
				A1: fp.Element{4921115267284233431, 2093038862301396370, 11859221373400371788, 14840702906540650688, 3187022540654844811, 12940014803022939},
			},
			{
				A0: fp.Element{10653597201928750109, 13006094245282229535, 13295987796930539448, 16777781019019743602, 516306931828031556, 114717773318043797},
				A1: fp.Element{235309942988776203, 14860563794932685398, 4559874204394395321, 16639699308077583607, 9227151754747780541, 15908917446924874},
			},
			{
				A0: fp.Element{17337820088369903748, 8586614709689721762, 1507576901396006569, 12700849903501888588, 17971589925911161601, 108264042385722694},
				A1: fp.Element{16945398252920060844, 12661842631743740793, 3757011696927843364, 15267617233875676717, 4747339870779259550, 34661428542359791},
			},
			{
				A0: fp.Element{16466466374773138846, 14279763770124955377, 2253148650834359538, 7990858085823320889, 7124676177871440280, 94151727797031734},
				A1: fp.Element{15321139925728056718, 15122461222407649886, 17533799169860262777, 16804941505997951982, 1600681788854461369, 44486672743339990},
			},
			{
				A0: fp.Element{8867142699562737491, 18199028041645162481, 3517239929170429351, 775467166380995197, 5868381756625215392, 6197719424154602},
				A1: fp.Element{9737125075096738524, 9629792455358261596, 11564279494282885105, 5089237230157463720, 17189088638807565425, 106712839297101083},
			},
			{
				A0: fp.Element{10270843696704620770, 13223598872017291793, 9010388952516938805, 15912131807459901749, 6846913031975448418, 86431140578472953},
				A1: fp.Element{15439335117817990542, 18208862128889198852, 1680174721576182500, 13180416124629130962, 7418066627384568449, 3653571306187636},
			},
			{
				A0: fp.Element{1131060365675191416, 10374247235014096023, 5274169106847399340, 9643591707517384578, 8354961201546942085, 16860653124975415},
				A1: fp.Element{7591061314547569259, 3687653994280978709, 14906524746128876973, 7376139658567228596, 3304769414542873433, 39389809689964413},
			},
			{
				A0: fp.Element{15061500520004024506, 13781868913987378153, 1311923405823490306, 13614698003779137726, 2734567429318555742, 41565578486180068},
				A1: fp.Element{2264606496492496938, 7463181801259856780, 4974231603582406689, 12631973597673125932, 13768142234619696863, 69859376569568629},
			},
			{
				A0: fp.Element{12977806950220507699, 9598682745345891240, 12727041082031765282, 13876739454320532788, 16731345162096153922, 52377912619410421},
				A1: fp.Element{9749487977108607359, 9452745135768007871, 17734899000824388188, 6032906020241168466, 3891046964558379200, 42965172988847872},
			},
			{
				A0: fp.Element{15134142718037669741, 2650609953202022541, 18444582952265559358, 10072051383647105272, 1614233522361858494, 6353896085976033},
				A1: fp.Element{1244819074686382314, 16827930012669440062, 7409202602064068605, 7833164804348414003, 14093967246945573989, 91886691830462491},
			},
			{
				A0: fp.Element{3481568382353274479, 1454015438048276221, 18321294391133087854, 11721714833000613979, 10732204074831352562, 71241856315356756},
				A1: fp.Element{2091942538387325776, 9875355486415603002, 12623384858601965052, 7559030465493899268, 7674739998354242578, 75767137192853634},
			},
			{
				A0: fp.Element{5491757234451068988, 6182298996664147147, 1538132785577256287, 4999224323735886804, 6348107227722746450, 48518858347037381},
				A1: fp.Element{15293807953430533698, 2990763629845967503, 13391735075948095730, 9673810945847290845, 11089480180973857099, 64742831177682886},
			},
		},
		x)
}

func g2Isogeny(p *G2Affine) {

	den := make([]fptower.E2, 2)

	g2IsogenyYDenominator(&den[1], &p.X)
	g2IsogenyXDenominator(&den[0], &p.X)

	g2IsogenyYNumerator(&p.Y, &p.X, &p.Y)
	g2IsogenyXNumerator(&p.X, &p.X)

	den = fptower.BatchInvertE2(den)

	p.X.Mul(&p.X, &den[0])
	p.Y.Mul(&p.Y, &den[1])
}

// g2SqrtRatio computes the square root of u/v and returns 0 iff u/v was indeed a quadratic residue
// if not, we get sqrt(Z * u / v). Recall that Z is non-residue
// If v = 0, u/v is meaningless and the output is unspecified, without raising an error.
// The main idea is that since the computation of the square root involves taking large powers of u/v, the inversion of v can be avoided
func g2SqrtRatio(z *fptower.E2, u *fptower.E2, v *fptower.E2) uint64 {

	// https://www.ietf.org/archive/id/draft-irtf-cfrg-hash-to-curve-16.html#name-sqrt_ratio-for-any-field

	tv1 := fptower.E2{
		A0: fp.Element{0},
		A1: fp.Element{9669507733633691314, 13074846433352412541, 11948543988290594071, 11104573675215509690, 878729975203857826, 120237394476367878},
	} //tv1 = c6

	var tv2, tv3, tv4, tv5 fptower.E2
	var exp big.Int
	// c4 = 140737488355327 = 2⁴⁷ - 1
	// q is odd so c1 is at least 1.
	exp.SetBytes([]byte{127, 255, 255, 255, 255, 255})

	tv2.Exp(*v, &exp) // 2. tv2 = vᶜ⁴
	tv3.Square(&tv2)  // 3. tv3 = tv2²
	tv3.Mul(&tv3, v)  // 4. tv3 = tv3 * v
	tv5.Mul(u, &tv3)  // 5. tv5 = u * tv3

	// c3 = 237702427642072544657662731610863241996908072983433720914596829155825880635712864411696995402952020023758239370111403151139377997388748144480691770738487127695940799749981367718443673617185911789718419957467908625
	exp.SetBytes([]byte{2, 211, 7, 208, 187, 175, 251, 34, 86, 145, 59, 179, 97, 38, 60, 75, 184, 184, 125, 164, 174, 233, 63, 31, 94, 113, 65, 61, 218, 77, 92, 9, 208, 24, 175, 185, 6, 96, 205, 192, 20, 231, 18, 80, 42, 77, 108, 70, 10, 170, 170, 139, 183, 10, 224, 49, 131, 36, 185, 88, 99, 140, 157, 107, 203, 251, 210, 53, 241, 192, 154, 74, 218, 38, 143, 46, 27, 216, 0, 115, 56, 210, 84, 240, 0, 0, 1, 10, 17})

	tv5.Exp(tv5, &exp)  // 6. tv5 = tv5ᶜ³
	tv5.Mul(&tv5, &tv2) // 7. tv5 = tv5 * tv2
	tv2.Mul(&tv5, v)    // 8. tv2 = tv5 * v
	tv3.Mul(&tv5, u)    // 9. tv3 = tv5 * u
	tv4.Mul(&tv3, &tv2) // 10. tv4 = tv3 * tv2

	// c5 = 70368744177664
	exp.SetBytes([]byte{64, 0, 0, 0, 0, 0})
	tv5.Exp(tv4, &exp)      // 11. tv5 = tv4ᶜ⁵
	isQNr := g2NotOne(&tv5) // 12. isQR = tv5 == 1
	c7 := fptower.E2{
		A0: fp.Element{1479358275892676257, 2814264704614556731, 13691179386454739330, 7530671302001941842, 60362263363904715, 37906327945374822},
		A1: fp.Element{5350190547200862053, 10822704806123199611, 5122684409451163826, 10616884767534481491, 1436196917100294910, 20226740120672211},
	}
	tv2.Mul(&tv3, &c7)                 // 13. tv2 = tv3 * c7
	tv5.Mul(&tv4, &tv1)                // 14. tv5 = tv4 * tv1
	tv3.Select(int(isQNr), &tv3, &tv2) // 15. tv3 = CMOV(tv2, tv3, isQR)
	tv4.Select(int(isQNr), &tv4, &tv5) // 16. tv4 = CMOV(tv5, tv4, isQR)
	exp.Lsh(big.NewInt(1), 47-2)       // 18, 19: tv5 = 2ⁱ⁻² for i = c1

	for i := 47; i >= 2; i-- { // 17. for i in (c1, c1 - 1, ..., 2):

		tv5.Exp(tv4, &exp)               // 20.    tv5 = tv4ᵗᵛ⁵
		nE1 := g2NotOne(&tv5)            // 21.    e1 = tv5 == 1
		tv2.Mul(&tv3, &tv1)              // 22.    tv2 = tv3 * tv1
		tv1.Mul(&tv1, &tv1)              // 23.    tv1 = tv1 * tv1    Why not write square?
		tv5.Mul(&tv4, &tv1)              // 24.    tv5 = tv4 * tv1
		tv3.Select(int(nE1), &tv3, &tv2) // 25.    tv3 = CMOV(tv2, tv3, e1)
		tv4.Select(int(nE1), &tv4, &tv5) // 26.    tv4 = CMOV(tv5, tv4, e1)

		if i > 2 {
			exp.Rsh(&exp, 1) // 18, 19. tv5 = 2ⁱ⁻²
		}
	}

	*z = tv3
	return isQNr
}

func g2NotOne(x *fptower.E2) uint64 {

	//Assuming hash is implemented for G1 and that the curve is over Fp
	var one fp.Element
	return one.SetOne().NotEqual(&x.A0) | g1NotZero(&x.A1)

}

// g2MulByZ multiplies x by [12, 1] and stores the result in z
func g2MulByZ(z *fptower.E2, x *fptower.E2) {

	z.Mul(x, &fptower.E2{
		A0: fp.Element{10560307807486212317, 9936456306313395274, 2092561269709285211, 8738829082964617622, 5243865315912343348, 114311569748804731},
		A1: fp.Element{202099033278250856, 5854854902718660529, 11492539364873682930, 8885205928937022213, 5545221690922665192, 39800542322357402},
	})

}

// https://www.ietf.org/archive/id/draft-irtf-cfrg-hash-to-curve-16.html#name-simplified-swu-method
// MapToCurve2 implements the SSWU map
// No cofactor clearing or isogeny
func MapToCurve2(u *fptower.E2) G2Affine {

	var sswuIsoCurveCoeffA = fptower.E2{
		A0: fp.Element{4274545572028848265, 14157081418478689358, 13123833976752631407, 4466041663276938746, 9062541850312583986, 90030181981586611},
		A1: fp.Element{4627353644986202063, 14941155654691983603, 14266958733709189881, 10264689865410103271, 10052798319587953375, 111844286035220969},
	}
	var sswuIsoCurveCoeffB = fptower.E2{
		A0: fp.Element{10237434857876739089, 8476639787604822147, 6641637803208190023, 1721529389316620686, 8656544759275761743, 38999476160258021},
		A1: fp.Element{2360755569119276357, 10390833517265838837, 12467133771585386911, 8219721226907645480, 3130947551623757939, 83517800164149569},
	}

	var tv1 fptower.E2
	tv1.Square(u) // 1.  tv1 = u²

	//mul tv1 by Z
	g2MulByZ(&tv1, &tv1) // 2.  tv1 = Z * tv1

	var tv2 fptower.E2
	tv2.Square(&tv1)    // 3.  tv2 = tv1²
	tv2.Add(&tv2, &tv1) // 4.  tv2 = tv2 + tv1

	var tv3 fptower.E2
	var tv4 fptower.E2
	tv4.SetOne()
	tv3.Add(&tv2, &tv4)                // 5.  tv3 = tv2 + 1
	tv3.Mul(&tv3, &sswuIsoCurveCoeffB) // 6.  tv3 = B * tv3

	tv2NZero := g2NotZero(&tv2)

	// tv4 = Z
	tv4 = fptower.E2{
		A0: fp.Element{10560307807486212317, 9936456306313395274, 2092561269709285211, 8738829082964617622, 5243865315912343348, 114311569748804731},
		A1: fp.Element{202099033278250856, 5854854902718660529, 11492539364873682930, 8885205928937022213, 5545221690922665192, 39800542322357402},
	}

	tv2.Neg(&tv2)
	tv4.Select(int(tv2NZero), &tv4, &tv2) // 7.  tv4 = CMOV(Z, -tv2, tv2 != 0)
	tv4.Mul(&tv4, &sswuIsoCurveCoeffA)    // 8.  tv4 = A * tv4

	tv2.Square(&tv3) // 9.  tv2 = tv3²

	var tv6 fptower.E2
	tv6.Square(&tv4) // 10. tv6 = tv4²

	var tv5 fptower.E2
	tv5.Mul(&tv6, &sswuIsoCurveCoeffA) // 11. tv5 = A * tv6

	tv2.Add(&tv2, &tv5) // 12. tv2 = tv2 + tv5
	tv2.Mul(&tv2, &tv3) // 13. tv2 = tv2 * tv3
	tv6.Mul(&tv6, &tv4) // 14. tv6 = tv6 * tv4

	tv5.Mul(&tv6, &sswuIsoCurveCoeffB) // 15. tv5 = B * tv6
	tv2.Add(&tv2, &tv5)                // 16. tv2 = tv2 + tv5

	var x fptower.E2
	x.Mul(&tv1, &tv3) // 17.   x = tv1 * tv3

	var y1 fptower.E2
	gx1NSquare := g2SqrtRatio(&y1, &tv2, &tv6) // 18. (is_gx1_square, y1) = sqrt_ratio(tv2, tv6)

	var y fptower.E2
	y.Mul(&tv1, u) // 19.   y = tv1 * u

	y.Mul(&y, &y1) // 20.   y = y * y1

	x.Select(int(gx1NSquare), &tv3, &x) // 21.   x = CMOV(x, tv3, is_gx1_square)
	y.Select(int(gx1NSquare), &y1, &y)  // 22.   y = CMOV(y, y1, is_gx1_square)

	y1.Neg(&y)
	y.Select(int(g2Sgn0(u)^g2Sgn0(&y)), &y, &y1)

	// 23.  e1 = sgn0(u) == sgn0(y)
	// 24.   y = CMOV(-y, y, e1)

	x.Div(&x, &tv4) // 25.   x = x / tv4

	return G2Affine{x, y}
}

func g2EvalPolynomial(z *fptower.E2, monic bool, coefficients []fptower.E2, x *fptower.E2) {
	dst := coefficients[len(coefficients)-1]

	if monic {
		dst.Add(&dst, x)
	}

	for i := len(coefficients) - 2; i >= 0; i-- {
		dst.Mul(&dst, x)
		dst.Add(&dst, &coefficients[i])
	}

	z.Set(&dst)
}

// g2Sgn0 is an algebraic substitute for the notion of sign in ordered fields
// Namely, every non-zero quadratic residue in a finite field of characteristic =/= 2 has exactly two square roots, one of each sign
// https://www.ietf.org/archive/id/draft-irtf-cfrg-hash-to-curve-16.html#name-the-sgn0-function
// The sign of an element is not obviously related to that of its Montgomery form
func g2Sgn0(z *fptower.E2) uint64 {

	nonMont := z.Bits()

	sign := uint64(0) // 1. sign = 0
	zero := uint64(1) // 2. zero = 1
	var signI uint64
	var zeroI uint64

	// 3. i = 1
	signI = nonMont.A0[0] % 2 // 4.   sign_i = x_i mod 2
	zeroI = g1NotZero(&nonMont.A0)
	zeroI = 1 ^ (zeroI|-zeroI)>>63 // 5.   zero_i = x_i == 0
	sign = sign | (zero & signI)   // 6.   sign = sign OR (zero AND sign_i) # Avoid short-circuit logic ops
	zero = zero & zeroI            // 7.   zero = zero AND zero_i
	// 3. i = 2
	signI = nonMont.A1[0] % 2 // 4.   sign_i = x_i mod 2
	// 5.   zero_i = x_i == 0
	sign = sign | (zero & signI) // 6.   sign = sign OR (zero AND sign_i) # Avoid short-circuit logic ops
	// 7.   zero = zero AND zero_i
	return sign

}

// MapToG2 invokes the SSWU map, and guarantees that the result is in g2
func MapToG2(u fptower.E2) G2Affine {
	res := MapToCurve2(&u)
	//this is in an isogenous curve
	g2Isogeny(&res)
	res.ClearCofactor(&res)
	return res
}

// EncodeToG2 hashes a message to a point on the G2 curve using the SSWU map.
// It is faster than HashToG2, but the result is not uniformly distributed. Unsuitable as a random oracle.
// dst stands for "domain separation tag", a string unique to the construction using the hash function
// https://www.ietf.org/archive/id/draft-irtf-cfrg-hash-to-curve-16.html#roadmap
func EncodeToG2(msg, dst []byte) (G2Affine, error) {

	var res G2Affine
	u, err := fp.Hash(msg, dst, 2)
	if err != nil {
		return res, err
	}

	res = MapToCurve2(&fptower.E2{
		A0: u[0],
		A1: u[1],
	})

	//this is in an isogenous curve
	g2Isogeny(&res)
	res.ClearCofactor(&res)
	return res, nil
}

// HashToG2 hashes a message to a point on the G2 curve using the SSWU map.
// Slower than EncodeToG2, but usable as a random oracle.
// dst stands for "domain separation tag", a string unique to the construction using the hash function
// https://www.ietf.org/archive/id/draft-irtf-cfrg-hash-to-curve-16.html#roadmap
func HashToG2(msg, dst []byte) (G2Affine, error) {
	u, err := fp.Hash(msg, dst, 2*2)
	if err != nil {
		return G2Affine{}, err
	}

	Q0 := MapToCurve2(&fptower.E2{
		A0: u[0],
		A1: u[1],
	})
	Q1 := MapToCurve2(&fptower.E2{
		A0: u[2+0],
		A1: u[2+1],
	})

	//TODO (perf): Add in E' first, then apply isogeny
	g2Isogeny(&Q0)
	g2Isogeny(&Q1)

	var _Q0, _Q1 G2Jac
	_Q0.FromAffine(&Q0)
	_Q1.FromAffine(&Q1).AddAssign(&_Q0)

	_Q1.ClearCofactor(&_Q1)

	Q1.FromJacobian(&_Q1)
	return Q1, nil
}

func g2NotZero(x *fptower.E2) uint64 {
	//Assuming G1 is over Fp and that if hashing is available for G2, it also is for G1
	return g1NotZero(&x.A0) | g1NotZero(&x.A1)

}
