/*++

Copyright (C) 2018 Autodesk Inc. (Original Author)

All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

--*/

//////////////////////////////////////////////////////////////////////////////////////////////////////
// buildbindinggo.go
// functions to generate Go-bindings of a library's API.
//////////////////////////////////////////////////////////////////////////////////////////////////////

package main

import (
	"errors"
	"fmt"
	"log"
	"path"
	"strings"
)

// BuildBindingGo builds Go-bindings of a library's API
func BuildBindingGo(component ComponentDefinition, outputFolder string, outputFolderExample string) error {
	forceRecreation := false

	NameSpace := component.NameSpace
	LibraryName := component.LibraryName
	BaseName := component.BaseName

	GoIntfName := path.Join(outputFolder, BaseName+".go")
	log.Printf("Creating \"%s\"", GoIntfName)
	gofile, err := CreateLanguageFile(GoIntfName, "	")
	if err != nil {
		return err
	}

	GoImplName := path.Join(outputFolder, BaseName+"_impl.go")
	log.Printf("Creating \"%s\"", GoImplName)
	goimplfile, err := CreateLanguageFile(GoImplName, "	")
	if err != nil {
		return err
	}

	gofile.WriteCLicenseHeader(component,
		fmt.Sprintf("This is an autogenerated Go wrapper file in order to allow an easy\n use of %s", LibraryName),
		true)
	goimplfile.WriteCLicenseHeader(component,
		fmt.Sprintf("This is an autogenerated Go implementation file in order to allow an easy\n use of %s", LibraryName),
		true)

	err = buildGoWrapper(component, gofile, goimplfile)
	if err != nil {
		return err
	}

	if len(outputFolderExample) > 0 {
		goExample := path.Join(outputFolderExample, NameSpace+"_example"+".go")
		if forceRecreation || !FileExists(goExample) {
			log.Printf("Creating \"%s\"", goExample)
			goExampleFile, err := CreateLanguageFile(goExample, "	")
			if err != nil {
				return err
			}
			goExampleFile.WriteCLicenseHeader(component,
				fmt.Sprintf("This is an autogenerated Go application that demonstrates the\n usage of the Go bindings of %s", LibraryName),
				true)
			buildGoExample(component, goExampleFile, outputFolder)
		} else {
			log.Printf("Omitting recreation of Go example file \"%s\"", goExample)
		}
	}
	return nil
}

func buildGoExample(component ComponentDefinition, w LanguageWriter, outputFolder string) {
	packageName := component.BaseName
	NameSpace := component.NameSpace

	w.Writeln("")
	w.Writeln("package main")
	w.Writeln("")
	w.Writeln("import (")
	w.Writeln("  \"fmt\"")
	w.Writeln("  \"log\"")
	w.Writeln("  \"../../Bindings/Go\"")
	w.Writeln(")")
	w.Writeln("")
	w.Writeln("func main() {")
	w.Writeln("	 wrapper, err := %s.%sLoadWrapper(\"../../Implementations/Cpp/build/Debug/%s.dll\") // TODO: add-path here", packageName, NameSpace, component.BaseName)
	w.Writeln("  if (err != nil) {")
	w.Writeln("    log.Fatal(err)")
	w.Writeln("  }")
	w.Writeln("  ")
	w.Writeln("  nMajor, nMinor, nMicro, err := wrapper.%s()", component.Global.VersionMethod)
	w.Writeln("  if (err != nil) {")
	w.Writeln("    log.Fatal(err)")
	w.Writeln("  }")
	w.Writeln("  versionString := fmt.Sprintf(\"%s.version = %s\", nMajor, nMinor, nMicro)", component.BaseName, "%d.%d.%d")
	w.Writeln("  ")
	if len(component.Global.PrereleaseMethod) > 0 {
		w.Writeln("  hasInfo, preReleaseInfo, err := wrapper.%s()", component.Global.PrereleaseMethod)
		w.Writeln("  if (err != nil) {")
		w.Writeln("    log.Fatal(err)")
		w.Writeln("  }")
		w.Writeln("  if (hasInfo) {")
		w.Writeln("    versionString += \"-\"+preReleaseInfo")
		w.Writeln("  }")
		w.Writeln("")
	}
	if len(component.Global.BuildinfoMethod) > 0 {
		w.Writeln("  hasInfo, buildInfo, err := wrapper.%s()", component.Global.BuildinfoMethod)
		w.Writeln("  if (err != nil) {")
		w.Writeln("    log.Fatal(err)")
		w.Writeln("  }")
		w.Writeln("  if (hasInfo) {")
		w.Writeln("    versionString += \"+\"+buildInfo")
		w.Writeln("  }")
		w.Writeln("")
	}

	w.Writeln("  fmt.Println(versionString)")
	w.Writeln("}")
	w.Writeln("")
}


func buildGoEnums(component ComponentDefinition, w LanguageWriter) {
	if len(component.Enums) <= 0 {
		return
	}
	NameSpace := component.NameSpace

	w.Writeln("")
	w.Writeln("/*************************************************************************************************************************")
	w.Writeln(" Declaration of enums")
	w.Writeln("**************************************************************************************************************************/")
	w.Writeln("")
	for i := 0; i < len(component.Enums); i++ {
		enum := component.Enums[i]
		w.Writeln("type E%s%s int", NameSpace, enum.Name)

		w.Writeln("const (")

		for j := 0; j < len(enum.Options); j++ {

			option := enum.Options[j]
			w.Writeln("    e%s_%s = %d", enum.Name, option.Name, option.Value)
		}
		w.Writeln(")")
		w.Writeln("")
	}
	w.Writeln("")
}

func buildGoStructs(component ComponentDefinition, w LanguageWriter) (error) {
	if len(component.Structs) <= 0 {
		return nil
	}
	NameSpace := component.NameSpace

	w.Writeln("/*************************************************************************************************************************")
	w.Writeln(" Declaration of structs")
	w.Writeln("**************************************************************************************************************************/")
	w.Writeln("")

	for i := 0; i < len(component.Structs); i++ {
		structinfo := component.Structs[i]
		w.Writeln("type s%s%s struct {", NameSpace, structinfo.Name)

		for j := 0; j < len(structinfo.Members); j++ {

			member := structinfo.Members[j]

			arraysuffix := ""
			if member.Rows > 0 {
				if member.Columns > 0 {
					arraysuffix = fmt.Sprintf("[%d][%d]", member.Columns, member.Rows)
				} else {
					arraysuffix = fmt.Sprintf("[%d]", member.Rows)
				}
			}

			switch member.Type {
			case "uint8":
				w.Writeln("    %s%s uint8;", member.Name, arraysuffix)
			case "uint16":
				w.Writeln("    %s%s uint16;", member.Name, arraysuffix)
			case "uint32":
				w.Writeln("    %s%s uint32;", member.Name, arraysuffix)
			case "uint64":
				w.Writeln("    %s%s uint64;", member.Name, arraysuffix)
			case "int8":
				w.Writeln("    %s%s int8;", member.Name, arraysuffix)
			case "int16":
				w.Writeln("    %s%s int16;", member.Name, arraysuffix)
			case "int32":
				w.Writeln("    %s%s int32;", member.Name, arraysuffix)
			case "int64":
				w.Writeln("    %s%s int64;", member.Name, arraysuffix)
			case "bool":
				w.Writeln("    %s%s bool;", member.Name, arraysuffix)
			case "single":
				w.Writeln("    %s%s float32;", member.Name, arraysuffix)
			case "double":
				w.Writeln("    %s%s float64;", member.Name, arraysuffix)
			case "pointer":
				w.Writeln("    %s%s uint64;", member.Name, arraysuffix)
			case "string":
				return fmt.Errorf("it is not possible for struct s%s%s to contain a string value", NameSpace, structinfo.Name)
			case "class", "optionalclass":
				return fmt.Errorf("it is not possible for struct s%s%s to contain a handle value", NameSpace, structinfo.Name)
			case "enum":
				w.Writeln("    %s%s E%s%s;", member.Name, arraysuffix, NameSpace, member.Class)
			}

		}
		w.Writeln("}")
		w.Writeln("")
	}
	w.Writeln("")
	return nil
}

func buildGoInterfaces(component ComponentDefinition, w LanguageWriter) {
	w.Writeln("/*************************************************************************************************************************")
	w.Writeln(" Declaration of interfaces")
	w.Writeln("**************************************************************************************************************************/")
	w.Writeln("")
	w.Writeln("type %sHandle interface {", component.NameSpace)
	w.Writeln("    Close() error")
	w.Writeln("    IsValid() bool")
	w.Writeln("}")
	w.Writeln("")
}

func buildGoImplementation(component ComponentDefinition, implw LanguageWriter) {
	NameSpace := component.NameSpace

	implw.Writeln("type %sImplementation struct {", NameSpace)
	implw.Writeln("  Initialized bool")
	implw.Writeln("  DLLHandle syscall.Handle")

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			implw.Writeln("  %s_%s_%s uintptr", NameSpace, strings.ToLower(class.ClassName), strings.ToLower(method.MethodName))
		}
	}

	for j := 0; j < len(component.Global.Methods); j++ {
		method := component.Global.Methods[j]
		implw.Writeln("  %s_%s uintptr", NameSpace, strings.ToLower(method.MethodName))
	}
	implw.Writeln("}")
	implw.Writeln("")
}

func buildGoImplementationHandle(component ComponentDefinition, implw LanguageWriter) {	
	NameSpace := component.NameSpace
	implw.Writeln("type %sImplementationHandle interface {", NameSpace)
	implw.Writeln("  %sHandle", NameSpace)
	implw.Writeln("")
	implw.Writeln("  GetDLLInHandle() (uintptr)")
	implw.Writeln("  GetDLLOutHandle() (uintptr)")
	implw.Writeln("  GetWrapper() (*%sImplementation)", NameSpace)
	implw.Writeln("}")
	implw.Writeln("")
	implw.Writeln("type %sImplementationHandleStruct struct {", NameSpace)
	implw.Writeln("  Implementation * %sImplementation", NameSpace)
	implw.Writeln("  DLLhandle uintptr")
	implw.Writeln("}")
	implw.Writeln("")
	implw.Writeln("func (handle *%sImplementationHandleStruct) Close() (error) {", NameSpace)
	implw.Writeln("  if (handle.DLLhandle != 0) {")
	implw.Writeln("    if (handle.Implementation == nil) {")
	implw.Writeln("      return errors.New(\"Uninitialized DLL Implementation Handle\")")
	implw.Writeln("    }")
	implw.Writeln("    ")
	implw.Writeln("    dllhandle := handle.DLLhandle")
	implw.Writeln("    handle.DLLhandle = 0;")
	implw.Writeln("    ")
	implw.Writeln("    return handle.Implementation.CallFunction(handle.Implementation.%s_%s, dllhandle)", NameSpace, strings.ToLower(component.Global.ReleaseMethod))
	implw.Writeln("  }")
	implw.Writeln("  ")
	implw.Writeln("  return nil")
	implw.Writeln("}")
	implw.Writeln("")
	implw.Writeln("func (handle *%sImplementationHandleStruct) IsValid() (bool) {", NameSpace)
	implw.Writeln("  return (handle.DLLhandle != 0)")
	implw.Writeln("}")
	implw.Writeln("")
	implw.Writeln("func (handle *%sImplementationHandleStruct) GetDLLInHandle() (uintptr) {", NameSpace)
	implw.Writeln("  return handle.DLLhandle;")
	implw.Writeln("}")
	implw.Writeln("")
	implw.Writeln("func (handle *%sImplementationHandleStruct) GetDLLOutHandle() (uintptr) {", NameSpace)
	implw.Writeln("  return uintptr(unsafe.Pointer(&handle.DLLhandle));")
	implw.Writeln("}")
	implw.Writeln("")
	implw.Writeln("func (handle *%sImplementationHandleStruct) GetWrapper() (*%sImplementation) {", NameSpace, NameSpace)
	implw.Writeln("  return handle.Implementation;")
	implw.Writeln("}")
	implw.Writeln("")
}

func buildGoHelperFunctions(implw LanguageWriter) {
	for _, theIntType := range [2]string{"Int", "UInt"} {
		for _, theWidth := range [4]string{"8", "16", "32", "64"} {
			implw.Writeln("func %s%sOutValue(reference * %s%s) uintptr {", theIntType, theWidth, strings.ToLower(theIntType), theWidth)
			implw.Writeln("  return uintptr(unsafe.Pointer(reference))")
			implw.Writeln("}")

			implw.Writeln("func %s%sInValue(value %s%s) uintptr {", theIntType, theWidth, strings.ToLower(theIntType), theWidth)
			implw.Writeln("  return uintptr(value)")
			implw.Writeln("}")
		}
	}

	for _, theFloatType := range [1]string{"Float"} {
		for _, theWidth := range [2]string{"32", "64"} {
			implw.Writeln("func %s%sOutValue(reference * %s%s) uintptr {", theFloatType, theWidth, strings.ToLower(theFloatType), theWidth)
			implw.Writeln("  return uintptr(unsafe.Pointer(reference))")
			implw.Writeln("}")

			implw.Writeln("func %s%sInValue(value %s%s) uintptr {", theFloatType, theWidth, strings.ToLower(theFloatType), theWidth)
			implw.Writeln("  return uintptr(value)")
			implw.Writeln("}")
		}
	}

	implw.Writeln("func StringInValue (value string) uintptr {")
	implw.Writeln("  bytePtr, err := syscall.BytePtrFromString(value)")
	implw.Writeln("  if err != nil {")
	implw.Writeln("    return 0")
	implw.Writeln("  }")
	implw.Writeln("  return uintptr(unsafe.Pointer(bytePtr))")
	implw.Writeln("}")
	implw.Writeln("")
	implw.Writeln("func PtrOutValue(ptr * uintptr) uintptr {")
	implw.Writeln("    return uintptr(unsafe.Pointer(ptr))")
	implw.Writeln("}")
	implw.Writeln("")
	implw.Writeln("func BytesOutValue(bytePtr * []byte) uintptr {")
	implw.Writeln("    return uintptr(unsafe.Pointer(bytePtr))")
	implw.Writeln("}")
	implw.Writeln("")
}

func buildGoErrorHandling(component ComponentDefinition, implw LanguageWriter) {
	implw.Writeln("")
	implw.Writeln("func Get%sErrorMessage(errorcode uint32) (string) {", component.NameSpace)
	implw.Writeln("  switch (errorcode) {")

	for i := 0; i < len(component.Errors.Errors); i++ {
		errorcode := component.Errors.Errors[i]
		implw.Writeln("  case %d: return \"%s\";", errorcode.Code, errorcode.Name)
	}
	implw.Writeln("  default:")
	implw.Writeln("    return \"unknown\";")
	implw.Writeln("  }")
	implw.Writeln("}")
	implw.Writeln("")
}

func buildGoInitialize(component ComponentDefinition, implw LanguageWriter) {
	NameSpace := component.NameSpace
	global := component.Global

	implw.Writeln("func (implementation *%sImplementation) Initialize(DLLFileName string) error {", NameSpace)
	implw.Writeln("  implementation.Initialized = false;")
	implw.Writeln("  implementation.DLLHandle = 0;")
	implw.Writeln("")
	implw.Writeln("  dllHandle, err := syscall.LoadLibrary(DLLFileName);")
	implw.Writeln("  if (err != nil) {")
	implw.Writeln("    return err;")
	implw.Writeln("  }")
	implw.Writeln("")

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]
		for j := 0; j < len(class.Methods); j++ {
			method := class.Methods[j]
			functionName := fmt.Sprintf("%s_%s_%s", strings.ToLower(NameSpace), strings.ToLower(class.ClassName), strings.ToLower(method.MethodName))
			implw.Writeln("  implementation.%s_%s_%s, err = syscall.GetProcAddress(dllHandle, \"%s\")", NameSpace, strings.ToLower(class.ClassName), strings.ToLower(method.MethodName), functionName)
			implw.Writeln("  if (err != nil) {")
			implw.Writeln("    return errors.New(\"Could not get function %s: \" + err.Error())", functionName)
			implw.Writeln("  }")
			implw.Writeln("  ")
		}
	}

	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]

		functionName := fmt.Sprintf("%s_%s", strings.ToLower(NameSpace), strings.ToLower(method.MethodName))
		implw.Writeln("  implementation.%s_%s, err = syscall.GetProcAddress(dllHandle, \"%s\")", NameSpace, strings.ToLower(method.MethodName), functionName)
		implw.Writeln("  if (err != nil) {")
		implw.Writeln("    return errors.New(\"Could not get function %s: \" + err.Error())", functionName)
		implw.Writeln("  }")
		implw.Writeln("  ")
	}

	implw.Writeln("  implementation.DLLHandle =  dllHandle")
	implw.Writeln("  implementation.Initialized = true")
	implw.Writeln("  return nil")
	implw.Writeln("}")
}

func buildGoCallFunction(component ComponentDefinition, implw LanguageWriter) {
	NameSpace := component.NameSpace

	implw.Writeln("func (implementation *%sImplementation) CallFunction(funcptr uintptr, parameters ... uintptr) (error) {", NameSpace)
	implw.Writeln("  var ret uintptr;")
	implw.Writeln("  if (!implementation.Initialized) {")
	implw.Writeln("    return errors.New(\"%s Implementation has not been initialized!\")", NameSpace)
	implw.Writeln("  }")
	implw.Writeln("  ")
	implw.Writeln("  switch len(parameters) { ")
	implw.Writeln("    case 0: ret, _, _ = syscall.Syscall(funcptr, 0, 0, 0, 0)")
	implw.Writeln("    case 1: ret, _, _ = syscall.Syscall(funcptr, 1, uintptr(parameters[0]), 0, 0)")
	implw.Writeln("    case 2: ret, _, _ = syscall.Syscall(funcptr, 2, uintptr(parameters[0]), uintptr(parameters[1]), 0)")
	implw.Writeln("    case 3: ret, _, _ = syscall.Syscall(funcptr, 3, uintptr(parameters[0]), uintptr(parameters[1]), uintptr(parameters[2]))")
	implw.Writeln("    case 4: ret, _, _ = syscall.Syscall6(funcptr, 4, uintptr(parameters[0]), uintptr(parameters[1]), uintptr(parameters[2]), uintptr(parameters[3]), 0, 0)")
	implw.Writeln("    case 5: ret, _, _ = syscall.Syscall6(funcptr, 5, uintptr(parameters[0]), uintptr(parameters[1]), uintptr(parameters[2]), uintptr(parameters[3]), uintptr(parameters[4]), 0)")
	implw.Writeln("    case 6: ret, _, _ = syscall.Syscall6(funcptr, 6, uintptr(parameters[0]), uintptr(parameters[1]), uintptr(parameters[2]), uintptr(parameters[3]), uintptr(parameters[4]), uintptr(parameters[5]))")
	implw.Writeln("    case 7: ret, _, _ = syscall.Syscall9(funcptr, 7, uintptr(parameters[0]), uintptr(parameters[1]), uintptr(parameters[2]), uintptr(parameters[3]), uintptr(parameters[4]), uintptr(parameters[5]), uintptr(parameters[6]), 0, 0)")
	implw.Writeln("    case 8: ret, _, _ = syscall.Syscall9(funcptr, 8, uintptr(parameters[0]), uintptr(parameters[1]), uintptr(parameters[2]), uintptr(parameters[3]), uintptr(parameters[4]), uintptr(parameters[5]), uintptr(parameters[6]), uintptr(parameters[7]), 0)")
	implw.Writeln("    case 9: ret, _, _ = syscall.Syscall9(funcptr, 9, uintptr(parameters[0]), uintptr(parameters[1]), uintptr(parameters[2]), uintptr(parameters[3]), uintptr(parameters[4]), uintptr(parameters[5]), uintptr(parameters[6]), uintptr(parameters[7]), uintptr(parameters[8]))")
	implw.Writeln("    case 10: ret, _, _ = syscall.Syscall12(funcptr, 10, uintptr(parameters[0]), uintptr(parameters[1]), uintptr(parameters[2]), uintptr(parameters[3]), uintptr(parameters[4]), uintptr(parameters[5]), uintptr(parameters[6]), uintptr(parameters[7]), uintptr(parameters[8]), uintptr(parameters[9]), 0, 0)")
	implw.Writeln("    case 11: ret, _, _ = syscall.Syscall12(funcptr, 11, uintptr(parameters[0]), uintptr(parameters[1]), uintptr(parameters[2]), uintptr(parameters[3]), uintptr(parameters[4]), uintptr(parameters[5]), uintptr(parameters[6]), uintptr(parameters[7]), uintptr(parameters[8]), uintptr(parameters[9]), uintptr(parameters[10]), 0)")
	implw.Writeln("    case 12: ret, _, _ = syscall.Syscall12(funcptr, 12, uintptr(parameters[0]), uintptr(parameters[1]), uintptr(parameters[2]), uintptr(parameters[3]), uintptr(parameters[4]), uintptr(parameters[5]), uintptr(parameters[6]), uintptr(parameters[7]), uintptr(parameters[8]), uintptr(parameters[9]), uintptr(parameters[10]), uintptr(parameters[11]))")
	implw.Writeln("    default: ")
	implw.Writeln("      return errors.New(\"Invalid DLL function parameter count!\");")
	implw.Writeln("  }")
	implw.Writeln("  ")
	implw.Writeln("  if (int(ret) != 0) {")
	implw.Writeln("    return errors.New(fmt.Sprintf(\"%s Error: %%.04x (%%s)\", int(ret), Get%sErrorMessage(uint32(ret))))", NameSpace, NameSpace)
	implw.Writeln("  }")
	implw.Writeln("  ")
	implw.Writeln("  return nil")
	implw.Writeln("}")
	implw.Writeln("")
}


func buildGoClass(component ComponentDefinition, class ComponentDefinitionClass, w LanguageWriter, implw LanguageWriter, NameSpace string, classdefinitions* []string) (error) {
	*classdefinitions = append(*classdefinitions, fmt.Sprintf(""))
	*classdefinitions = append(*classdefinitions, fmt.Sprintf("/*************************************************************************************************************************"))
	*classdefinitions = append(*classdefinitions, fmt.Sprintf("Class definition %s%s", NameSpace, class.ClassName))
	*classdefinitions = append(*classdefinitions, fmt.Sprintf("**************************************************************************************************************************/"))
	*classdefinitions = append(*classdefinitions, fmt.Sprintf(""))
	*classdefinitions = append(*classdefinitions, fmt.Sprintf("type %s%s struct {",  NameSpace, class.ClassName))

	if (component.Global.BaseClassName == class.ClassName) {
		*classdefinitions = append(*classdefinitions, fmt.Sprintf("  Interface %sGoInterface",  NameSpace))
		*classdefinitions = append(*classdefinitions, fmt.Sprintf("  Handle %sHandle", NameSpace))
	} else {
		if len(class.ParentClass)>0 {
			*classdefinitions = append(*classdefinitions, fmt.Sprintf("  %s%s", NameSpace, class.ParentClass))
		} else {
			*classdefinitions = append(*classdefinitions, fmt.Sprintf("  %s%s", NameSpace, component.Global.BaseClassName))
		}
	}
	
	*classdefinitions = append(*classdefinitions, fmt.Sprintf("}"))
	*classdefinitions = append(*classdefinitions, fmt.Sprintf(""))
	*classdefinitions = append(*classdefinitions, fmt.Sprintf("func (instance *%s%s) Close() (error) {", NameSpace, class.ClassName))
	*classdefinitions = append(*classdefinitions, fmt.Sprintf("  return instance.Handle.Close()"))
	*classdefinitions = append(*classdefinitions, fmt.Sprintf("}"))
	*classdefinitions = append(*classdefinitions, fmt.Sprintf(""))

	for j := 0; j < len(class.Methods); j++ {
		method := class.Methods[j]

		err := writeGoMethod(method, w, implw, NameSpace, class.ClassName, false, classdefinitions)
		if err != nil {
			return err
		}
	}
	return nil
}

func buildGoWrapper(component ComponentDefinition, w LanguageWriter, implw LanguageWriter) error {
	NameSpace := component.NameSpace
	global := component.Global
	packageName := strings.ToLower(component.BaseName)

	w.Writeln("")
	w.Writeln("package %s", packageName)
	w.Writeln("")

	buildGoEnums(component, w)
	err := buildGoStructs(component, w)
	if (err != nil) {
		return err
	}
	buildGoInterfaces(component, w)

	implw.Writeln("")
	implw.Writeln("package %s", packageName)
	implw.Writeln("")
	implw.Writeln("// #include <string.h>")
	implw.Writeln("import \"C\"")
	implw.Writeln("")
	implw.Writeln("import (")
	implw.Writeln("    \"fmt\"")
	implw.Writeln("    \"errors\"")
	implw.Writeln("    \"syscall\"")
	implw.Writeln("    \"unsafe\"")
	implw.Writeln(")")
	implw.Writeln("")

	buildGoImplementation(component, implw)

	buildGoImplementationHandle(component, implw)

	buildGoHelperFunctions(implw)

	buildGoErrorHandling(component, implw)
	
	implw.Writeln("")
	implw.Writeln("func (implementation *%sImplementation) GetWrapperHandle(handle %sHandle) (%sImplementationHandle, error) {", NameSpace, NameSpace, NameSpace)
	implw.Writeln("  implementation_handle, ok := handle.(%sImplementationHandle)", NameSpace)
	implw.Writeln("  if ok {")
	implw.Writeln("    handle_implementation := implementation_handle.GetWrapper()")
	implw.Writeln("    if (handle_implementation == implementation) {")
	implw.Writeln("      return implementation_handle, nil")
	implw.Writeln("    }")
	implw.Writeln("    return nil, errors.New(\"Invalid Implementation for DLL handle.\")")
	implw.Writeln("  }")
	implw.Writeln("  return nil, errors.New(\"Could not cast DLL handle.\")")
	implw.Writeln("}")
	implw.Writeln("")

	buildGoInitialize(component, implw)

	implw.Writeln("")
	implw.Writeln("func (implementation *%sImplementation) NewHandle() (%sImplementationHandle) {", NameSpace, NameSpace)
	implw.Writeln("  handle := new (%sImplementationHandleStruct)", NameSpace)
	implw.Writeln("  handle.Implementation = implementation")
	implw.Writeln("  handle.DLLhandle = 0")
	implw.Writeln("  return handle")
	implw.Writeln("}")
	implw.Writeln("")

	buildGoCallFunction(component, implw)
	implw.Writeln("")

	var classdefinitions []string
	w.Writeln("type %sGoInterface interface {", NameSpace)

	for i := 0; i < len(component.Classes); i++ {
		class := component.Classes[i]

		err = buildGoClass(component, class, w, implw, NameSpace, &classdefinitions)
		if err != nil {
			return err
		}
	}

	implw.Writeln("")
	implw.Writeln("/*************************************************************************************************************************")
	implw.Writeln("  Class definition %sWrapper", NameSpace)
	implw.Writeln("**************************************************************************************************************************/")
	implw.Writeln("type %sWrapper struct {", NameSpace)
	implw.Writeln("  Interface %sGoInterface", NameSpace)
	implw.Writeln("}")

	global = component.Global
	for j := 0; j < len(global.Methods); j++ {
		method := global.Methods[j]

		err := writeGoMethod(method, w, implw, NameSpace, "Wrapper", true, &classdefinitions)
		if err != nil {
			return err
		}
	}

	w.Writeln("")
	w.Writeln("}")
	w.Writeln("")

	w.Writelns("", classdefinitions)

	implw.Writeln("")
	implw.Writeln("func (implementation *%sImplementation) checkBinaryVersion() (error) {", NameSpace)
	implw.Writeln("  var nBindingMajor uint32 = %d;", majorVersion(component.Version))
	implw.Writeln("  var nBindingMinor uint32 = %d;", minorVersion(component.Version))
	implw.Writeln("  nMajor, nMinor, _, err := implementation.%s()", global.VersionMethod)
	implw.Writeln("  if (err != nil) {")
	implw.Writeln("      return err;")
	implw.Writeln("  }")
	implw.Writeln("  if ( (nMajor != nBindingMajor) || (nMinor < nBindingMinor) ) {")
	implw.Writeln("    return fmt.Errorf(\"%s Error: %.04x (%s)\", int(0), Get%sErrorMessage(uint32(0)));", NameSpace, "%", "%s", NameSpace)
	implw.Writeln("  }")
	implw.Writeln("  return nil")
	implw.Writeln("}")
	implw.Writeln("")

	implw.Writeln("func %sLoadWrapper(DllFileName string) (%sWrapper, error) {", NameSpace, NameSpace)
	implw.Writeln("  var Wrapper %sWrapper;", NameSpace)
	implw.Writeln("  var Instance %sImplementation;", NameSpace)
	implw.Writeln("  ")
	implw.Writeln("  err := Instance.Initialize(DllFileName);")
	implw.Writeln("  if (err != nil) {")
	implw.Writeln("      return Wrapper, err;")
	implw.Writeln("  }")

	implw.Writeln("  err = Instance.checkBinaryVersion()")
	implw.Writeln("  if (err != nil) {")
	implw.Writeln("      return Wrapper, err;")
	implw.Writeln("  }")

	implw.Writeln("  Wrapper.Interface = &Instance;")
	implw.Writeln("  ")

	implw.Writeln("  return Wrapper, nil;")
	implw.Writeln("}")
	implw.Writeln("")

	return nil
}

func getGoBasicType(paramType string) (string, error) {
	switch paramType {
	case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "bool":
		return paramType, nil
	case "single":
		return "float32", nil
	case "double":
		return "float64", nil
	case "pointer":
		return "uint64", nil
	}
	return "", errors.New("Invalid basic type: " + paramType)
}

func paramFunction(paramType string, paramPass string) (string, error) {
	paramFunctionStr := ""
	switch paramType {
	case "uint8":		paramFunctionStr = "UInt8"
	case "uint16":		paramFunctionStr = "UInt16"
	case "uint32":		paramFunctionStr = "UInt32"
	case "uint64":		paramFunctionStr = "UInt64"
	case "int8":		paramFunctionStr = "Int8"
	case "int16":		paramFunctionStr = "Int16"
	case "int32":		paramFunctionStr = "Int32"
	case "int64":		paramFunctionStr = "Int64"
	case "bool":		paramFunctionStr = "bool"
	case "single":		paramFunctionStr = "Float32"
	case "double":		paramFunctionStr = "Float64"
	case "pointer":		paramFunctionStr = "UInt64"
	default:
		return "", errors.New("Invalid basic type: " + paramType)
	}

	if paramPass == "in" {
		paramFunctionStr += "InValue"
	} else {
		paramFunctionStr += "OutValue"
	}
	return paramFunctionStr , nil
}

func writeGoMethod(method ComponentDefinitionMethod, w LanguageWriter, implw LanguageWriter, NameSpace string, ClassName string, isGlobal bool, classdefinitions* []string) error {

	parameters := ""
	callparameters := ""
	returnvalues := ""
	
	var comments []string

	var implDeclarations []string
	implDeclarations = append(implDeclarations, fmt.Sprintf("var err error = nil"))
	var implCasts []string

	implReturnValues := ""

	errorReturn := ""
	for k := 0; k < len(method.Params); k++ {
		param := method.Params[k]

		if (param.ParamPass == "out") || (param.ParamPass == "return") {
			switch param.ParamType {
			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "single", "double":
				errorReturn = errorReturn + fmt.Sprintf("0, ")
			case "pointer":
				errorReturn = errorReturn + fmt.Sprintf("0, ")
			case "enum":
				errorReturn = errorReturn + fmt.Sprintf("0, ")
			case "bool":
				errorReturn = errorReturn + fmt.Sprintf("false, ")
			case "string":
				errorReturn = errorReturn + fmt.Sprintf("\"\", ")
			case "struct":
				errorReturn = errorReturn + fmt.Sprintf("s%s, ", param.ParamName)
			case "class", "optionalclass":
				errorReturn = errorReturn + fmt.Sprintf("h%s, ", param.ParamName)
			case "functiontype":
				errorReturn = errorReturn + fmt.Sprintf("0, ")
			case "basicarray":
				basicType, err := getGoBasicType(param.ParamClass)
				if err != nil {
					return err
				}
				errorReturn = errorReturn + fmt.Sprintf("make([]%s, 0), ", basicType)
			case "structarray":
				errorReturn = errorReturn + fmt.Sprintf("make([]s%s%s, 0), ", NameSpace, param.ParamClass)
			default:
				return fmt.Errorf("invalid method parameter type \"%s\" for %s.%s(%s)", param.ParamType, ClassName, method.MethodName, param.ParamName)
			}
		}
	}
	errorReturn = errorReturn + "err"

	if !isGlobal {
		implCasts = append(implCasts, fmt.Sprintf(""))
		implCasts = append(implCasts, fmt.Sprintf("implementation_%s, err := implementation.GetWrapperHandle(%s)", strings.ToLower(ClassName), ClassName))
		implCasts = append(implCasts, fmt.Sprintf("if (err != nil) {"))
		implCasts = append(implCasts, fmt.Sprintf("  return %s", errorReturn))
		implCasts = append(implCasts, fmt.Sprintf("}"))
	}

	requiresInitCall := false
	implCallParameters := ""
	implInitCallParameters := ""
	var implInitCallLines []string

	var classReturnImplementation []string
	classReturnVariables := ""
	classReturnString := ""
	classReturnTypes := ""

	for k := 0; k < len(method.Params); k++ {
		param := method.Params[k]
		thisImplCallParamter := ""
		thisInitImplCallParamter := ""
		switch param.ParamPass {
		case "in":
			if parameters != "" {
				parameters = parameters + ", "
			}
			if callparameters != "" {
				callparameters = callparameters + ", "
			}

			switch param.ParamType {
			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64":
				goParamName := "n"+param.ParamName
				comments = append(comments, fmt.Sprintf("* @param[in] %s - %s", goParamName, param.ParamDescription))
				parameters = parameters + fmt.Sprintf("%s %s", goParamName, param.ParamType)
				goParamFunction, err := paramFunction(param.ParamType, param.ParamPass)
				if err != nil {
					return err
				}
				thisImplCallParamter = fmt.Sprintf(", %s(%s)", goParamFunction, goParamName)
				callparameters = callparameters + goParamName

			case "bool":
				comments = append(comments, fmt.Sprintf("* @param[in] b%s - %s", param.ParamName, param.ParamDescription))
				implDeclarations = append(implDeclarations, fmt.Sprintf("var n%s uint8 = 0", param.ParamName))
				implDeclarations = append(implDeclarations, fmt.Sprintf("if (b%s) {", param.ParamName))
				implDeclarations = append(implDeclarations, fmt.Sprintf("  n%s = 1", param.ParamName))
				implDeclarations = append(implDeclarations, fmt.Sprintf("}"))
				implDeclarations = append(implDeclarations, fmt.Sprintf(""))
				parameters = parameters + fmt.Sprintf("b%s bool", param.ParamName)
				thisImplCallParamter = fmt.Sprintf(", UInt8InValue(n%s)", param.ParamName)
				callparameters = callparameters + "b" + param.ParamName

			case "single":
				comments = append(comments, fmt.Sprintf("* @param[in] f%s - %s", param.ParamName, param.ParamDescription))
				parameters = parameters + fmt.Sprintf("f%s float32", param.ParamName)
				thisImplCallParamter = fmt.Sprintf(", Float32InValue(f%s)", param.ParamName)
				callparameters = callparameters + "f" + param.ParamName

			case "double":
				comments = append(comments, fmt.Sprintf("* @param[in] d%s - %s", param.ParamName, param.ParamDescription))
				parameters = parameters + fmt.Sprintf("d%s float64", param.ParamName)
				thisImplCallParamter = fmt.Sprintf(", Float64InValue(d%s)", param.ParamName)
				callparameters = callparameters + "d" + param.ParamName

			case "pointer":
				comments = append(comments, fmt.Sprintf("* @param[in] n%s - %s", param.ParamName, param.ParamDescription))
				parameters = parameters + fmt.Sprintf("n%s uint64", param.ParamName)
				thisImplCallParamter = fmt.Sprintf(", UInt64InValue(n%s)", param.ParamName)
				callparameters = callparameters + "n" + param.ParamName

			case "string":
				comments = append(comments, fmt.Sprintf("* @param[in] s%s - %s", param.ParamName, param.ParamDescription))
				parameters = parameters + fmt.Sprintf("s%s string", param.ParamName)
				thisImplCallParamter = fmt.Sprintf(", StringInValue(s%s)", param.ParamName)
				callparameters = callparameters + "s" + param.ParamName

			case "enum":
				comments = append(comments, fmt.Sprintf("* @param[in] e%s - %s", param.ParamName, param.ParamDescription))
				parameters = parameters + fmt.Sprintf("e%s E%s%s", param.ParamName, NameSpace, param.ParamClass)
				thisImplCallParamter = fmt.Sprintf(", uintptr(e%s)", param.ParamName)
				callparameters = callparameters + "e" + param.ParamName

			case "struct":
				comments = append(comments, fmt.Sprintf("* @param[in] s%s - %s", param.ParamName, param.ParamDescription))
				parameters = parameters + fmt.Sprintf("s%s s%s%s", param.ParamName, NameSpace, param.ParamClass)
				thisImplCallParamter = fmt.Sprintf(", uintptr(unsafe.Pointer(&s%s))", param.ParamName)
				callparameters = callparameters + "s" + param.ParamName

			case "basicarray":
				basicType, err := getGoBasicType(param.ParamClass)
				if err != nil {
					return err
				}
				comments = append(comments, fmt.Sprintf("* @param[in] %s - %s", param.ParamName, param.ParamDescription))
				parameters = parameters + fmt.Sprintf("%s []%s", param.ParamName, basicType)
				thisImplCallParamter = fmt.Sprintf(", 0, 0")
				callparameters = callparameters + param.ParamName

			case "structarray":
				comments = append(comments, fmt.Sprintf("* @param[in] %s - %s", param.ParamName, param.ParamDescription))
				parameters = parameters + fmt.Sprintf("%s []s%s%s", param.ParamName, NameSpace, param.ParamClass)
				thisImplCallParamter = fmt.Sprintf(", 0, 0")
				callparameters = callparameters + param.ParamName

			case "functiontype":
				comments = append(comments, fmt.Sprintf("* @param[in] p%s - %s", param.ParamName, param.ParamDescription))
				parameters = parameters + fmt.Sprintf("p%s int64", param.ParamName)
				thisImplCallParamter = fmt.Sprintf(", 0")
				callparameters = callparameters + "p" + param.ParamName

			case "class", "optionalclass":
				comments = append(comments, fmt.Sprintf("* @param[in] %s - %s", param.ParamName, param.ParamDescription))
				parameters = parameters + fmt.Sprintf("%s %sHandle", param.ParamName, NameSpace)

				implCasts = append(implCasts, fmt.Sprintf("implementation_%s, err := implementation.GetWrapperHandle(%s)", strings.ToLower(param.ParamName), param.ParamName))
				implCasts = append(implCasts, fmt.Sprintf("if (err != nil) {"))
				implCasts = append(implCasts, fmt.Sprintf("  return %s", errorReturn))
				implCasts = append(implCasts, fmt.Sprintf("}"))
				implCasts = append(implCasts, fmt.Sprintf(""))

				implCasts = append(implCasts, fmt.Sprintf("%sDLLHandle := implementation_%s.GetDLLInHandle()", param.ParamName, strings.ToLower(param.ParamName)))
				if (param.ParamType == "class") {
					implCasts = append(implCasts, fmt.Sprintf("if (%sDLLHandle == 0) {", param.ParamName))
					implCasts = append(implCasts, fmt.Sprintf("  err := fmt.Errorf(\"Handle must not be 0.\")", ))
					implCasts = append(implCasts, fmt.Sprintf("  return %s", errorReturn))
					implCasts = append(implCasts, fmt.Sprintf("}"))
				}
				thisImplCallParamter = fmt.Sprintf(", %sDLLHandle", param.ParamName)
				callparameters = callparameters + param.ParamName

			default:
				return fmt.Errorf("invalid method parameter type \"%s\" for %s.%s (%s)", param.ParamType, ClassName, method.MethodName, param.ParamName)
			}

			thisInitImplCallParamter = thisImplCallParamter

		case "out", "return":
			comments = append(comments, fmt.Sprintf("* @return %s", param.ParamDescription))

			switch param.ParamType {
			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64":
				basicType, err := getGoBasicType(param.ParamType)
				if err != nil {
					return err
				}
				goParamFunction, err := paramFunction(param.ParamType, param.ParamPass)
				if err != nil {
					return err
				}
				goParamName := "n"+param.ParamName
				returnvalues = returnvalues + fmt.Sprintf("%s, ", basicType)
				implDeclarations = append(implDeclarations, fmt.Sprintf("var %s %s = 0", goParamName, param.ParamType))
				implReturnValues = implReturnValues + fmt.Sprintf("%s(%s), ", basicType, goParamName)

				thisImplCallParamter = fmt.Sprintf(", %s(&%s)", goParamFunction, goParamName)
				thisInitImplCallParamter = thisImplCallParamter

				classReturnVariables = classReturnVariables + goParamName + ", "
				classReturnString = classReturnString + goParamName + ", "
				classReturnTypes = classReturnTypes + fmt.Sprintf("%s, ", basicType)

			case "pointer":
				returnvalues = returnvalues + fmt.Sprintf("uint64, ")
				implDeclarations = append(implDeclarations, fmt.Sprintf("var n%s uint64 = 0", param.ParamName))
				implReturnValues = implReturnValues + fmt.Sprintf("n%s, ", param.ParamName)

				thisImplCallParamter = fmt.Sprintf(", UInt64OutValue(&n%s)", param.ParamName)
				thisInitImplCallParamter = thisImplCallParamter

				classReturnVariables = classReturnVariables + "n" + param.ParamName + ", "
				classReturnString = classReturnString + "n" + param.ParamName + ", "
				classReturnTypes = classReturnTypes + fmt.Sprintf("uint64, ")

			case "bool":
				returnvalues = returnvalues + fmt.Sprintf("bool, ")
				implDeclarations = append(implDeclarations, fmt.Sprintf("var b%s int64 = 0", param.ParamName))
				implReturnValues = implReturnValues + fmt.Sprintf("(b%s != 0), ", param.ParamName)

				thisImplCallParamter = fmt.Sprintf(", Int64OutValue(&b%s)", param.ParamName)
				thisInitImplCallParamter = thisImplCallParamter

				classReturnVariables = classReturnVariables + "b" + param.ParamName + ", "
				classReturnString = classReturnString + "b" + param.ParamName + ", "
				classReturnTypes = classReturnTypes + fmt.Sprintf("bool, ")

			case "single":
				returnvalues = returnvalues + fmt.Sprintf("float32, ")
				implDeclarations = append(implDeclarations, fmt.Sprintf("var f%s float32 = 0", param.ParamName))
				implReturnValues = implReturnValues + fmt.Sprintf("f%s, ", param.ParamName)

				thisImplCallParamter = fmt.Sprintf(", Float32OutValue(&f%s)", param.ParamName)
				thisInitImplCallParamter = thisImplCallParamter

				classReturnVariables = classReturnVariables + "f" + param.ParamName + ", "
				classReturnString = classReturnString + "f" + param.ParamName + ", "
				classReturnTypes = classReturnTypes + fmt.Sprintf("float32, ")

			case "double":
				returnvalues = returnvalues + fmt.Sprintf("float64, ")
				implDeclarations = append(implDeclarations, fmt.Sprintf("var d%s float64 = 0", param.ParamName))
				implReturnValues = implReturnValues + fmt.Sprintf("d%s, ", param.ParamName)

				thisImplCallParamter = fmt.Sprintf(", Float64OutValue(&d%s)", param.ParamName)
				thisInitImplCallParamter = thisImplCallParamter

				classReturnVariables = classReturnVariables + "d" + param.ParamName + ", "
				classReturnString = classReturnString + "d" + param.ParamName + ", "
				classReturnTypes = classReturnTypes + fmt.Sprintf("float64, ")

			case "string":
				requiresInitCall = true
				implDeclarations = append(implDeclarations, fmt.Sprintf("var neededfor%s int64 = 0", param.ParamName))
				implDeclarations = append(implDeclarations, fmt.Sprintf("var filledin%s int64 = 0", param.ParamName))
				
				thisInitImplCallParamter = fmt.Sprintf(", Int64InValue(0), Int64OutValue(&neededfor%s), Int64InValue(0)", param.ParamName)

				implInitCallLines = append(implInitCallLines, fmt.Sprintf("bufferSize%s := neededfor%s", param.ParamName, param.ParamName))
				implInitCallLines = append(implInitCallLines, fmt.Sprintf("buffer%s := make([]byte, bufferSize%s)", param.ParamName, param.ParamName))

				thisImplCallParamter = fmt.Sprintf(", Int64InValue(bufferSize%s), Int64OutValue(&filledin%s), uintptr(unsafe.Pointer(&buffer%s[0]))", param.ParamName, param.ParamName, param.ParamName)

				implReturnValues = implReturnValues + fmt.Sprintf("string(buffer%s[:(filledin%s-1)]), ", param.ParamName, param.ParamName)

				returnvalues = returnvalues + fmt.Sprintf("string, ")
				classReturnVariables = classReturnVariables + "s" + param.ParamName + ", "
				classReturnString = classReturnString + "s" + param.ParamName + ", "
				classReturnTypes = classReturnTypes + fmt.Sprintf("string, ")

			case "enum":
				returnvalues = returnvalues + fmt.Sprintf("E%s%s, ", NameSpace, param.ParamClass)
				implDeclarations = append(implDeclarations, fmt.Sprintf("var e%s uint64 = 0", param.ParamName))
				implReturnValues = implReturnValues + fmt.Sprintf("E%s%s (e%s), ", NameSpace, param.ParamClass, param.ParamName)
				thisImplCallParamter = fmt.Sprintf(", UInt64OutValue(&e%s)", param.ParamName)
				thisInitImplCallParamter = thisImplCallParamter

				classReturnVariables = classReturnVariables + "e" + param.ParamName + ", "
				classReturnString = classReturnString + "e" + param.ParamName + ", "
				classReturnTypes = classReturnTypes + fmt.Sprintf("E%s%s, ", NameSpace, param.ParamClass)

			case "basicarray":
				requiresInitCall = true
				basicType, err := getGoBasicType(param.ParamClass)
				if err != nil {
					return err
				}
				implDeclarations = append(implDeclarations, fmt.Sprintf("var neededfor%s int64 = 0", param.ParamName))
				implDeclarations = append(implDeclarations, fmt.Sprintf("var filledin%s int64 = 0", param.ParamName))

				bufferName := fmt.Sprintf("buffer%s", param.ParamName)
				implDeclarations = append(implDeclarations, fmt.Sprintf("%s := make([]%s, 0)", bufferName, basicType))

				thisInitImplCallParamter = fmt.Sprintf(", Int64InValue(0), Int64OutValue(&neededfor%s), Int64InValue(0)", param.ParamName)

				implInitCallLines = append(implInitCallLines, fmt.Sprintf("bufferSize%s := neededfor%s", param.ParamName, param.ParamName))
				implInitCallLines = append(implInitCallLines, fmt.Sprintf("%s = make([]%s, bufferSize%s)", bufferName, basicType, param.ParamName))

				thisImplCallParamter = fmt.Sprintf(", Int64InValue(bufferSize%s), Int64OutValue(&filledin%s), uintptr(unsafe.Pointer(&%s[0]))", param.ParamName, param.ParamName, bufferName)

				returnvalues = returnvalues + fmt.Sprintf("[]%s, ", basicType)
				implReturnValues = implReturnValues + fmt.Sprintf("%s, ", bufferName)

				classReturnVariables = classReturnVariables + bufferName + ", "
				classReturnString = classReturnString + bufferName + ", "
				classReturnTypes = classReturnTypes + fmt.Sprintf("[]%s, ", basicType)

			case "structarray":
				requiresInitCall = true
				returnvalues = returnvalues + fmt.Sprintf("[]s%s%s, ", NameSpace, param.ParamClass)
				implDeclarations = append(implDeclarations, fmt.Sprintf("array%s := make([]s%s%s, 0)", param.ParamName, NameSpace, param.ParamClass))
				implReturnValues = implReturnValues + fmt.Sprintf("array%s, ", param.ParamName)
				thisImplCallParamter = fmt.Sprintf(", 0, 0, 0")
				thisInitImplCallParamter = thisImplCallParamter

				classReturnVariables = classReturnVariables + "array" + param.ParamName + ", "
				classReturnString = classReturnString + "array" + param.ParamName + ", "
				classReturnTypes = classReturnTypes + fmt.Sprintf("[]s%s%s, ", NameSpace, param.ParamClass)

			case "functiontype":
				returnvalues = returnvalues + fmt.Sprintf("uint64, ")
				implDeclarations = append(implDeclarations, fmt.Sprintf("var p%s uint64 = 0", param.ParamName))
				implReturnValues = implReturnValues + fmt.Sprintf("p%s, ", param.ParamName)
				thisImplCallParamter = fmt.Sprintf(", UInt64OutValue(&p%s)", param.ParamName)
				thisInitImplCallParamter = thisImplCallParamter
				
				classReturnVariables = classReturnVariables + "p" + param.ParamName + ", "
				classReturnString = classReturnString + "p" + param.ParamName + ", "
				classReturnTypes = classReturnTypes + fmt.Sprintf("uint64, ")

			case "struct":
				returnvalues = returnvalues + fmt.Sprintf("s%s%s, ", NameSpace, param.ParamClass)
				implDeclarations = append(implDeclarations, fmt.Sprintf("var s%s s%s%s", param.ParamName, NameSpace, param.ParamClass))
				implReturnValues = implReturnValues + fmt.Sprintf("s%s, ", param.ParamName)
				thisImplCallParamter = fmt.Sprintf(", 0")
				thisInitImplCallParamter = thisImplCallParamter

				classReturnVariables = classReturnVariables + "s" + param.ParamName + ", "
				classReturnString = classReturnString + "s" + param.ParamName + ", "
				classReturnTypes = classReturnTypes + fmt.Sprintf("s%s%s, ", NameSpace, param.ParamClass)

			case "class", "optionalclass":
				returnvalues = returnvalues + fmt.Sprintf("%sHandle, ", NameSpace)
				implDeclarations = append(implDeclarations, fmt.Sprintf("h%s := implementation.NewHandle()", param.ParamName))

				thisImplCallParamter = fmt.Sprintf(", h%s.GetDLLOutHandle()", param.ParamName)
				thisInitImplCallParamter = thisImplCallParamter

				implReturnValues = implReturnValues + fmt.Sprintf("h%s, ", param.ParamName)
				classReturnVariables = classReturnVariables + "h" + param.ParamName + ", "
				classReturnImplementation = append(classReturnImplementation, fmt.Sprintf("var c%s %s%s", param.ParamName, NameSpace, param.ParamClass))
				classReturnImplementation = append(classReturnImplementation, fmt.Sprintf("c%s.Interface = instance.Interface", param.ParamName))
				classReturnImplementation = append(classReturnImplementation, fmt.Sprintf("c%s.Handle = h%s", param.ParamName, param.ParamName))

				classReturnString = classReturnString + "c" + param.ParamName + ", "
				classReturnTypes = classReturnTypes + fmt.Sprintf("%s%s, ", NameSpace, param.ParamClass)

			default:
				return fmt.Errorf("invalid method parameter type \"%s\" for %s.%s (%s)", param.ParamType, ClassName, method.MethodName, param.ParamName)
			}

		default:
			return fmt.Errorf("invalid method parameter passing \"%s\" for %s.%s (%s)", param.ParamPass, ClassName, method.MethodName, param.ParamName)
		}

		implCallParameters += thisImplCallParamter 
		implInitCallParameters += thisInitImplCallParamter 
	}

	w.Writeln("")
	w.Writeln("  /**")
	w.Writeln("  * %s", method.MethodDescription)
	w.Writeln("  *")
	w.Writeln("  * @param[in] %s - %s instance.", ClassName, ClassName)
	w.Writelns("  ", comments)
	w.Writeln("  */")

	handleparameter := ""
	if isGlobal {
		w.Writeln("  %s(%s) (%serror)\n", method.MethodName, parameters, returnvalues)
	} else {
		handleparameter = fmt.Sprintf("%s %sHandle", ClassName, NameSpace)
		if parameters != "" {
			handleparameter = handleparameter + ", "
		}
		w.Writeln("  %s_%s(%s%s) (%serror)\n", ClassName, method.MethodName, handleparameter, parameters, returnvalues)
	}


	// Implementation
	implmethodname := "implementation." + NameSpace + "_"
	implGetHandleFunction := ""
	if !isGlobal {
		implmethodname += strings.ToLower(ClassName) + "_"
		implGetHandleFunction = fmt.Sprintf(", implementation_%s.GetDLLInHandle()", strings.ToLower(ClassName))
	}
	implmethodname += strings.ToLower(method.MethodName)
	
	if isGlobal {
		implw.Writeln("func (implementation *%sImplementation) %s(%s%s) (%serror) {", NameSpace, method.MethodName, handleparameter, parameters, returnvalues)
	} else {
		implw.Writeln("func (implementation *%sImplementation) %s_%s(%s%s) (%serror) {", NameSpace, ClassName, method.MethodName, handleparameter, parameters, returnvalues)
	}
	implw.Writelns("  ", implDeclarations)
	implw.Writelns("  ", implCasts)
	implw.Writeln("")
	if requiresInitCall {
		implw.Writeln("  err = implementation.CallFunction(%s%s%s)", implmethodname, implGetHandleFunction, implInitCallParameters)
		implw.Writeln("  if (err != nil) {")
		implw.Writeln("    return %s", errorReturn)
		implw.Writeln("  }")
		implw.Writelns("  ", implInitCallLines)
	}
	implw.Writeln("  err = implementation.CallFunction(%s%s%s)", implmethodname, implGetHandleFunction, implCallParameters)

	implw.Writeln("  if (err != nil) {")
	implw.Writeln("    return %s", errorReturn)
	implw.Writeln("  }")
	implw.Writeln("  ")
	implw.Writeln("  return %serr", implReturnValues)
	implw.Writeln("}")
	implw.Writeln("")

	if isGlobal {
		*classdefinitions = append(*classdefinitions, fmt.Sprintf("func (instance *%s%s) %s(%s) (%serror) {", NameSpace, ClassName, method.MethodName, parameters, classReturnTypes))
		*classdefinitions = append(*classdefinitions, fmt.Sprintf("  %serror := instance.Interface.%s(%s)", classReturnVariables, method.MethodName, callparameters))
	} else {
		if callparameters != "" {
			callparameters = ", " + callparameters
		}
		*classdefinitions = append(*classdefinitions, fmt.Sprintf("func (instance *%s%s) %s(%s) (%serror) {", NameSpace, ClassName, method.MethodName, parameters, classReturnTypes))
		*classdefinitions = append(*classdefinitions, fmt.Sprintf("  %serror := instance.Interface.%s_%s(instance.Handle%s)", classReturnVariables, ClassName, method.MethodName, callparameters))
	}
	for _, line := range classReturnImplementation {
		*classdefinitions = append(*classdefinitions, fmt.Sprintf("  %s", line ))
	}
	*classdefinitions = append(*classdefinitions, fmt.Sprintf("  return %serror", classReturnString))
	*classdefinitions = append(*classdefinitions, fmt.Sprintf("}"))
	*classdefinitions = append(*classdefinitions, fmt.Sprintf(""))

	return nil
}

