/*++

Copyright (C) 2018 Autodesk Inc. (Original Author)

All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

--*/

//////////////////////////////////////////////////////////////////////////////////////////////////////
// buildbindingpython.go
// functions to generate dynamic Python3-bindings of a library's API in form of explicitly loaded
// function handles.
//////////////////////////////////////////////////////////////////////////////////////////////////////

package main

import (
	"fmt"
	"log"
	"path"
	"strings"
)

// BuildBindingPythonDynamic builds dynamic Python bindings of a library's API in form of explicitly loaded
// functions handles.
func BuildBindingPythonDynamic(componentdefinition ComponentDefinition, outputFolder string, outputFolderExample string, indentString string) error {
	forceRecreation := false

	namespace := componentdefinition.NameSpace
	libraryname := componentdefinition.LibraryName
	
	DynamicPythonImpl := path.Join(outputFolder, namespace+".py");
	log.Printf("Creating \"%s\"", DynamicPythonImpl)
	dynpythonfile, err := CreateLanguageFile (DynamicPythonImpl, indentString)
	if err != nil {
		return err;
	}

	dynpythonfile.WritePythonLicenseHeader(componentdefinition,
		fmt.Sprintf("This is an autogenerated Python file in order to allow an easy\n use of %s", libraryname),
		true)
	
	err = buildDynamicPythonImplementation(componentdefinition, dynpythonfile)
	if err != nil {
		return err;
	}
	
	if (len(outputFolderExample) > 0) {
		DynamicPythonExample := path.Join(outputFolderExample, namespace+"_Example"+".py");
		if (forceRecreation || !FileExists(DynamicPythonExample)) {
			log.Printf("Creating \"%s\"", DynamicPythonExample)
			dynpythonexamplefile, err := CreateLanguageFile (DynamicPythonExample, indentString)
			dynpythonexamplefile.WritePythonLicenseHeader(componentdefinition,
				fmt.Sprintf("This is an autogenerated Python application that demonstrates the\n usage of the Python bindings of %s", libraryname),
				true)
			err = buildDynamicPythonExample(componentdefinition, dynpythonexamplefile, outputFolder)
			if err != nil {
				return err;
			}
		} else {
			log.Printf("Omitting recreation of Python example \"%s\"", DynamicPythonExample)
		}
	}
	

	return nil;
}

func buildDynamicPythonImplementation(componentdefinition ComponentDefinition, w LanguageWriter) error {

	NameSpace := componentdefinition.NameSpace
	BaseName := componentdefinition.BaseName

	w.Writeln("")
	w.Writeln("import ctypes")
	w.Writeln("import platform")
	w.Writeln("import enum")
	w.Writeln("import os")
	w.Writeln("")


	if len(componentdefinition.ImportedComponentDefinitions) > 0 {
		w.Writeln("import sys")
		w.Writeln("# Injected Components")
		for _, subComponent := range(componentdefinition.ImportedComponentDefinitions) {
			subNameSpace := subComponent.NameSpace
			w.Writeln("sys.path.append(os.path.join(os.path.dirname(os.path.realpath(__file__)), \"..\", \"..\", \"..\", \"%s_component\", \"Bindings\", \"Python\"))", subNameSpace)
			w.Writeln("import %s", subNameSpace)
		}
		w.Writeln("")
	}

	w.Writeln("name = \"%s\"", BaseName)
	w.Writeln("")

	w.Writeln("'''Definition of domain specific exception")
	w.Writeln("'''")
	w.Writeln("class E%sException(Exception):", NameSpace)
	w.Writeln("  def __init__(self, code, message = ''):")
	w.Writeln("    self._code = code")
	w.Writeln("    self._message = message")
	w.Writeln("  ")
	w.Writeln("  def __str__(self):")
	w.Writeln("    if self._message:")
	w.Writeln("      return '%sException ' + str(self._code) + ': '+ str(self._message)", NameSpace)
	w.Writeln("    return '%sException ' + str(self._code)", NameSpace)
	w.Writeln("")


	w.Writeln("'''Definition of binding API version")
	w.Writeln("'''")
	w.Writeln("class BindingVersion(enum.IntEnum):")
	w.Writeln("  MAJOR = %d", majorVersion(componentdefinition.Version))
	w.Writeln("  MINOR = %d", minorVersion(componentdefinition.Version))
	w.Writeln("  MICRO = %d", microVersion(componentdefinition.Version))
	w.Writeln("")

	w.Writeln("'''Definition Error Codes")
	w.Writeln("'''")
	w.Writeln("class ErrorCodes(enum.IntEnum):")
	w.Writeln("  SUCCESS = 0")
	for i := 0; i<len(componentdefinition.Errors.Errors); i++ {
		merror := componentdefinition.Errors.Errors[i]
		w.Writeln("  %s = %d", merror.Name, merror.Code)
	}
	w.Writeln("")

	w.Writeln("'''Definition of Function Table")
	w.Writeln("'''")
	w.Writeln("class FunctionTable:")
	for j:=0; j<len(componentdefinition.Global.Methods); j++ {
		method := componentdefinition.Global.Methods[j]
		w.Writeln("  %s_%s = None", strings.ToLower(NameSpace), strings.ToLower(method.MethodName))
	}
	for i:=0; i<len(componentdefinition.Classes); i++ {
		class := componentdefinition.Classes[i]
		for j:=0; j<len(class.Methods); j++ {
			method := class.Methods[j]
			w.Writeln("  %s_%s_%s = None", strings.ToLower(NameSpace), strings.ToLower(class.ClassName), strings.ToLower(method.MethodName))
		}
	}
	w.Writeln("")

	if (len(componentdefinition.Enums) > 0) {
		w.Writeln("'''Definition of Enumerations")
		w.Writeln("'''")
		w.Writeln("")
		w.Writeln("'''Definition of base enumeration for ctypes")
		w.Writeln("'''")
		w.Writeln("class CTypesEnum(enum.IntEnum):")
		w.Writeln("  def from_param(obj):")
		w.Writeln("    return int(obj)")
		w.Writeln("")

		for i := 0; i<len(componentdefinition.Enums); i++ {
			enum := componentdefinition.Enums[i]
			w.Writeln("'''Definition of %s", enum.Name)
			w.Writeln("'''")
			w.Writeln("class %s(CTypesEnum):", enum.Name)
			for j:= 0; j<len(enum.Options); j++ {
				option := enum.Options[j]
				w.Writeln("  %s = %d", option.Name, option.Value)
			}
		}
		w.Writeln("")
	}
	
	if (len(componentdefinition.Structs) > 0) {
		w.Writeln("'''Definition of Structs")
		w.Writeln("'''")
		for i := 0; i<len(componentdefinition.Structs); i++ {
			_struct := componentdefinition.Structs[i]
			w.Writeln("'''Definition of %s", _struct.Name)
			w.Writeln("'''")
			w.Writeln("class %s(ctypes.Structure):", _struct.Name)
			if (len(_struct.Members) > 0) {
				w.Writeln("  _pack_ = 1")
				w.Writeln("  _fields_ = [")
				for j:= 0; j<len(_struct.Members); j++ {
					member := _struct.Members[j]
					comma := ""
					if (j < len(_struct.Members) -1  ) {
						comma = ", "
					}
					memberType, err:= getCTypesParameterTypeName(member.Type, NameSpace, member.Class, true)
					if (err != nil) {
						return err
					}
					if (member.Type == "enum") {
						memberType = "ctypes.c_int32"
					}
					typeFormatter := "%s"
					if (member.Rows > 0) {
						memberType = fmt.Sprintf(typeFormatter + " * %d", memberType, member.Rows)
						typeFormatter = "(%s)"
					}
					if (member.Columns > 0) {
						memberType = fmt.Sprintf(typeFormatter + " * %d", memberType, member.Columns)
						typeFormatter = "(%s)"
					}
					w.Writeln("    (\"%s\", %s)%s", member.Name, memberType, comma)
				}
				w.Writeln("  ]")
			}
		}
		w.Writeln("")
	}

	if (len(componentdefinition.Functions) > 0) {
		w.Writeln("'''Definition of Function Types")
		w.Writeln("'''")
		for i := 0; i<len(componentdefinition.Functions); i++ {
			_func := componentdefinition.Functions[i]
			w.Writeln("'''Definition of %s", _func.FunctionName)
			w.Writeln("    %s", _func.FunctionDescription)
			w.Writeln("'''")
			arguments := "ctypes.c_void_p"
			for j := 0; j<len(_func.Params); j++ {
				param := _func.Params[j]
				if (arguments != "") {
					arguments = arguments + ", "
				}
				cParams, err := generateCTypesParameter(param, "", _func.FunctionName, NameSpace)
				if (err != nil) {
					return err
				}
				arguments = arguments + cParams[0].ParamType
			}
			w.Writeln("%s = ctypes.CFUNCTYPE(%s)", _func.FunctionName, arguments)
		}
		w.Writeln("")
	}

	w.Writeln("")
	w.Writeln("'''Wrapper Class Implementation")
	w.Writeln("'''")
	w.Writeln("class Wrapper:")
	w.Writeln("")
	
	if len(componentdefinition.ImportedComponentDefinitions) > 0 {
		w.Writeln("# Injected Components")
		for _, subComponent := range(componentdefinition.ImportedComponentDefinitions) {
			subNameSpace := subComponent.NameSpace
			w.Writeln("  _%sWrapper = None", subNameSpace)
		}
		w.Writeln("")
	}

	w.Writeln("  def __init__(self, libraryName = None, symbolLookupMethodAddress = None):")
	w.Writeln("    ending = ''")
	w.Writeln("    if platform.system() == 'Windows':")
	w.Writeln("      ending = 'dll'")
	w.Writeln("    elif platform.system() == 'Linux':")
	w.Writeln("      ending = 'so'")
	w.Writeln("    elif platform.system() == 'Darwin':")
	w.Writeln("      ending = 'dylib'")
	w.Writeln("    else:")
	w.Writeln("      raise E%sException(ErrorCodes.COULDNOTLOADLIBRARY)", NameSpace)
	w.Writeln("    ")
	w.Writeln("    if (not libraryName) and (not symbolLookupMethodAddress):")
	w.Writeln("      libraryName = os.path.join(os.path.dirname(os.path.realpath(__file__)),'%s')", BaseName)
	w.Writeln("    ")
	w.Writeln("    if libraryName is not None:")
	w.Writeln("      path = libraryName + '.' + ending")
	w.Writeln("      try:")
	w.Writeln("        self.lib = ctypes.CDLL(path)")
	w.Writeln("      except Exception as e:")
	w.Writeln("        raise E%sException(ErrorCodes.COULDNOTLOADLIBRARY, str(e) + '| \"'+path + '\"' )", NameSpace )
	w.Writeln("      ")
	w.Writeln("      self._loadFunctionTable()")
	w.Writeln("    elif symbolLookupMethodAddress is not None:")
	w.Writeln("        self.lib = FunctionTable()")
	w.Writeln("        self._loadFunctionTableFromMethod(symbolLookupMethodAddress)")
	w.Writeln("    else:")
	w.Writeln("      raise E%sException(ErrorCodes.COULDNOTLOADLIBRARY, str(e))", NameSpace )
	w.Writeln("    ")
	w.Writeln("    self._checkBinaryVersion()")
	w.Writeln("  ")

	w.Writeln("  def _loadFunctionTableFromMethod(self, symbolLookupMethodAddress):")
	w.Writeln("    try:")
	w.AddIndentationLevel(3)
	err := loadFunctionTableFromMethod(componentdefinition, w)
	if (err!=nil) {
		return err
	}
	w.AddIndentationLevel(-3)
	w.Writeln("    except AttributeError as ae:")
	w.Writeln("      raise E%sException(ErrorCodes.COULDNOTFINDLIBRARYEXPORT, ae.args[0])", NameSpace)
	w.Writeln("    ")

	w.Writeln("  def _loadFunctionTable(self):")
	w.Writeln("    try:")
	err = loadFunctionTable(componentdefinition, w)
	if (err!=nil) {
		return err
	}
	w.Writeln("    except AttributeError as ae:")
	w.Writeln("      raise E%sException(ErrorCodes.COULDNOTFINDLIBRARYEXPORT, ae.args[0])", NameSpace)
	w.Writeln("  ")

	w.Writeln("  def _checkBinaryVersion(self):")
	w.Writeln("    nMajor, nMinor, _ = self.%s()", componentdefinition.Global.VersionMethod)
	w.Writeln("    if (nMajor != BindingVersion.MAJOR) or (nMinor < BindingVersion.MINOR):")
	w.Writeln("      raise E%sException(ErrorCodes.INCOMPATIBLEBINARYVERSION)", NameSpace)
	w.Writeln("  ")

	
	w.Writeln("  def checkError(self, instance, errorCode):")
	w.Writeln("    if errorCode != ErrorCodes.SUCCESS.value:")
	w.Writeln("      if instance:")
	w.Writeln("        if instance._wrapper != self:")
	w.Writeln("          raise E%sException(ErrorCodes.INVALIDCAST, 'invalid wrapper call')", NameSpace)
	w.Writeln("      message,_ = self.%s(instance)", componentdefinition.Global.ErrorMethod)
	w.Writeln("      raise E%sException(errorCode, message)", NameSpace)
	w.Writeln("  ")
	
	for j:=0; j<len(componentdefinition.Global.Methods); j++ {
		method := componentdefinition.Global.Methods[j]

		isSpecialFunction, err := CheckHeaderSpecialFunction(method, componentdefinition.Global);
		if err != nil {
			return err
		}
		
		implementationLines := make([]string, 0)
		if (isSpecialFunction == eSpecialMethodInjection) {
			implementationLines = append(implementationLines, "bNameSpaceFound = False")
			sParamName := method.Params[0].ParamName
			for _, subComponent := range(componentdefinition.ImportedComponentDefinitions) {
				theNameSpace := subComponent.NameSpace
				implementationLines = append(implementationLines, fmt.Sprintf("if %s == \"%s\":", sParamName, theNameSpace))
				implementationLines = append(implementationLines, fmt.Sprintf("  if self._%sWrapper is not None:", theNameSpace))
				implementationLines = append(implementationLines, fmt.Sprintf("    raise E%sException(ErrorCodes.COULDNOTLOADLIBRARY, \"Library with namespace \" + %s + \" is already registered.\")", NameSpace, sParamName) )
				implementationLines = append(implementationLines, fmt.Sprintf("  self._%sWrapper = %s.Wrapper(symbolLookupMethodAddress = %s)", theNameSpace, theNameSpace, method.Params[1].ParamName))
				implementationLines = append(implementationLines, fmt.Sprintf("  bNameSpaceFound = True"))
			}
			implementationLines = append(implementationLines, "if not bNameSpaceFound:")
			implementationLines = append(implementationLines, fmt.Sprintf("  raise E%sException(ErrorCodes.COULDNOTLOADLIBRARY, \"Unknown namespace \" + %s)", NameSpace, sParamName ))
		}

		err = writeMethod(method, w, NameSpace, "Wrapper", implementationLines, true)
		if (err!=nil) {
			return err
		}
	}

	for i:=0; i<len(componentdefinition.Classes); i++ {
		w.Writeln("")
		w.Writeln("")
		err = writePythonClass(componentdefinition, componentdefinition.Classes[i], w, NameSpace)
		if (err!=nil) {
			return err
		}
	}

	w.Writeln("    ")
	
	return nil
}


func generateCTypeParemeters(method ComponentDefinitionMethod, w LanguageWriter, NameSpace string, ClassName string, isGlobal bool)([]ctypesParameter, error) {
	parameters := []ctypesParameter{}
	if (!isGlobal) {
		cParams := make([]ctypesParameter,1)
		cParams[0].ParamName = "Object"
		cParams[0].ParamType = "ctypes.c_void_p"
		cParams[0].ParamComment = "#"
		parameters = append(parameters, cParams...);
	}
	for k:=0; k<len(method.Params); k++ {
		cParams, err := generateCTypesParameter(method.Params[k], ClassName, method.MethodName, NameSpace)
		if (err != nil) {
			return nil, err
		}
		parameters = append(parameters, cParams...);
	}
	return parameters, nil
}

func writeCDLLFunctionTableMethod(method ComponentDefinitionMethod, w LanguageWriter, NameSpace string, ClassName string, isGlobal bool) error {
	exportName := GetCExportName(NameSpace, ClassName, method, isGlobal)
	w.Writeln("      self.lib.%s.restype = ctypes.c_int32", exportName)
	parameters, err := getMethodCParams(method, NameSpace, ClassName, isGlobal)
	if err != nil {
		return err
	}
	w.Writeln("      self.lib.%s.argtypes = [%s]", exportName, parameters)
	w.Writeln("      ")
	return nil
}

func writeFunctionTableMethod(method ComponentDefinitionMethod, w LanguageWriter, NameSpace string, ClassName string, isGlobal bool) error {
	linearMethodName := ""
	if (isGlobal) {
		linearMethodName = strings.ToLower(NameSpace + "_" + method.MethodName)
	} else {
		linearMethodName = strings.ToLower(NameSpace + "_" + ClassName + "_" + method.MethodName)
	}
	
	params, err := getMethodCParams(method, NameSpace, ClassName, isGlobal)
	if err != nil {
		return err
	}

	w.Writeln("err = symbolLookupMethod(ctypes.c_char_p(str.encode(\"%s\")), methodAddress)", linearMethodName)
	w.Writeln("if err != 0:")
	w.Writeln("  raise E%sException(ErrorCodes.COULDNOTLOADLIBRARY, str(err))", NameSpace)
	w.Writeln("methodType = ctypes.CFUNCTYPE(ctypes.c_int32, " + params + ")")
	w.Writeln("self.lib.%s = methodType(int(methodAddress.value))", linearMethodName)
	w.Writeln("")
	return nil
}

func getMethodCParams(method ComponentDefinitionMethod, NameSpace string, ClassName string, isGlobal bool) (string, error) {
	parameters := ""
	if (!isGlobal) {
		parameters = "ctypes.c_void_p"
	}
	for k:=0; k<len(method.Params); k++ {
		param := method.Params[k]
		cparams, err := generateCTypesParameter(param, ClassName, method.MethodName, NameSpace)
		if (err != nil) {
			return "", err
		}
		for _, cparam := range cparams {
			if (parameters != "") {
				parameters = parameters + ", ";
			}
			parameters = parameters + cparam.ParamType
		}
	}

	return parameters, nil
}

func loadFunctionTableFromMethod(componentdefinition ComponentDefinition, w LanguageWriter) error {
	w.Writeln("symbolLookupMethodType = ctypes.CFUNCTYPE(ctypes.c_int32, ctypes.c_char_p, ctypes.POINTER(ctypes.c_void_p))")
	w.Writeln("symbolLookupMethod = symbolLookupMethodType(int(symbolLookupMethodAddress))")
	w.Writeln("")
	w.Writeln("methodAddress = ctypes.c_void_p()")
	w.Writeln("")

	for j:=0; j<len(componentdefinition.Global.Methods); j++ {
		method := componentdefinition.Global.Methods[j]
		err := writeFunctionTableMethod(method, w, componentdefinition.NameSpace, "Wrapper", true)
		if err != nil {
			return err
		}
	}

	for i:=0; i<len(componentdefinition.Classes); i++ {
		class := componentdefinition.Classes[i]
		for j:=0; j<len(class.Methods); j++ {
			err := writeFunctionTableMethod(class.Methods[j], w, componentdefinition.NameSpace, class.ClassName, false)
			if err != nil {
				return err
			}
		}
	}

	return nil
}

func loadFunctionTable(componentdefinition ComponentDefinition, w LanguageWriter) error {
	for j:=0; j<len(componentdefinition.Global.Methods); j++ {
		err := writeCDLLFunctionTableMethod(componentdefinition.Global.Methods[j], w, componentdefinition.NameSpace, "Wrapper", true)
		if err != nil {
			return err
		}
	}

	for i:=0; i<len(componentdefinition.Classes); i++ {
		class := componentdefinition.Classes[i]
		for j:=0; j<len(class.Methods); j++ {
			err := writeCDLLFunctionTableMethod(class.Methods[j], w, componentdefinition.NameSpace, class.ClassName, false)
			if err != nil {
				return err
			}
		}
	}
	return nil
}

func getCTypesParameterTypeName(ParamTypeName string, NameSpace string, ParamClass string, isPlain bool)(string, error) {
	CTypesParamTypeName := "";
	switch (ParamTypeName) {
		case "uint8":
			CTypesParamTypeName = "ctypes.c_uint8";
		case "uint16":
			CTypesParamTypeName = "ctypes.c_uint16";
		case "uint32":
			CTypesParamTypeName = "ctypes.c_uint32";
		case "uint64":
			CTypesParamTypeName = "ctypes.c_uint64";
		case "int8":
			CTypesParamTypeName = "ctypes.c_int8";
		case "int16":
			CTypesParamTypeName = "ctypes.c_int16";
		case "int32":
			CTypesParamTypeName = "ctypes.c_int32";
		case "int64":
			CTypesParamTypeName = "ctypes.c_int64";
		case "bool":
			CTypesParamTypeName = "ctypes.c_bool";
		case "single":
			CTypesParamTypeName = "ctypes.c_float";
		case "double":
			CTypesParamTypeName = "ctypes.c_double";
		case "pointer":
			CTypesParamTypeName = "ctypes.c_void_p";
		case "string":
			CTypesParamTypeName = "ctypes.c_char_p";
		case "basicarray":
			dummy, err := getCTypesParameterTypeName(ParamClass, NameSpace, "", isPlain)
			if (err != nil) {
				return "", err
			}
			CTypesParamTypeName = dummy
		case "enum":
			return fmt.Sprintf("%s", ParamClass), nil
		case "struct":
			return fmt.Sprintf("%s", ParamClass), nil
		case "structarray":
			return fmt.Sprintf("%s", ParamClass), nil
		case "functiontype":
			return fmt.Sprintf("%s", ParamClass), nil
		case "class", "optionalclass":
			CTypesParamTypeName = "ctypes.c_void_p";
		default:
			return "", fmt.Errorf ("invalid parameter type \"%s\" for Python parameter", ParamTypeName);
	}
	
	return CTypesParamTypeName, nil;
}

type ctypesParameter struct {
	ParamType string
	ParamCallType string
	ParamName string
	ParamComment string
}

func generateCTypesParameter(param ComponentDefinitionParam, className string, methodName string, NameSpace string) ([]ctypesParameter, error) {
	cParams := make([]ctypesParameter,1)
	cParamTypeName, err := getCTypesParameterTypeName(param.ParamType, NameSpace, param.ParamClass, true);
	if (err != nil) {
		return nil, err;
	}

	switch (param.ParamPass) {
	case "in":
		switch (param.ParamType) {
			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64":
				cParams[0].ParamType = cParamTypeName;
				cParams[0].ParamCallType = cParamTypeName;
				cParams[0].ParamName = "n" + param.ParamName;
				cParams[0].ParamComment = fmt.Sprintf("* @param[in] %s - %s", cParams[0].ParamName, param.ParamDescription);

			case "bool":
				cParams[0].ParamType = cParamTypeName;
				cParams[0].ParamCallType = cParamTypeName;
				cParams[0].ParamName = "b" + param.ParamName;
				cParams[0].ParamComment = fmt.Sprintf("* @param[in] %s - %s", cParams[0].ParamName, param.ParamDescription);
				
			case "single":
				cParams[0].ParamType = cParamTypeName;
				cParams[0].ParamCallType = cParamTypeName;
				cParams[0].ParamName = "f" + param.ParamName;
				cParams[0].ParamComment = fmt.Sprintf("* @param[in] %s - %s", cParams[0].ParamName, param.ParamDescription);

			case "double":
				cParams[0].ParamType = cParamTypeName;
				cParams[0].ParamCallType = cParamTypeName;
				cParams[0].ParamName = "d" + param.ParamName;
				cParams[0].ParamComment = fmt.Sprintf("* @param[in] %s - %s", cParams[0].ParamName, param.ParamDescription);
			
			case "pointer":
				cParams[0].ParamType = cParamTypeName;
				cParams[0].ParamCallType = cParamTypeName;
				cParams[0].ParamName = "p" + param.ParamName;
				cParams[0].ParamComment = fmt.Sprintf("* @param[in] %s - %s", cParams[0].ParamName, param.ParamDescription);
				
			case "string":
				cParams[0].ParamType = cParamTypeName;
				cParams[0].ParamCallType = cParamTypeName;
				cParams[0].ParamName = "p" + param.ParamName;
				cParams[0].ParamComment = fmt.Sprintf("* @param[in] %s - %s", cParams[0].ParamName, param.ParamDescription);

			case "enum":
				cParams[0].ParamType = cParamTypeName;
				cParams[0].ParamCallType = cParamTypeName;
				cParams[0].ParamName = "e" + param.ParamName;
				cParams[0].ParamComment = fmt.Sprintf("* @param[in] %s - %s", cParams[0].ParamName, param.ParamDescription);

			case "struct":
				cParams[0].ParamType = "ctypes.POINTER("+cParamTypeName+")";
				cParams[0].ParamCallType = cParamTypeName;
				cParams[0].ParamName = "p" + param.ParamName;
				cParams[0].ParamComment = fmt.Sprintf("* @param[in] %s - %s", cParams[0].ParamName, param.ParamDescription);

			case "basicarray", "structarray":
				cParams = make([]ctypesParameter,2)
				cParams[0].ParamType = "ctypes.c_uint64";
				cParams[0].ParamCallType = "ctypes.c_uint64";
				cParams[0].ParamName = "n" + param.ParamName + "Count";
				cParams[0].ParamComment = fmt.Sprintf("* @param[in] %s - Number of elements in buffer", cParams[0].ParamName);

				cParams[1].ParamType = "ctypes.POINTER(" + cParamTypeName + ")"
				cParams[1].ParamCallType = cParamTypeName
				cParams[1].ParamName = "p" + param.ParamName + "Buffer";
				cParams[1].ParamComment = fmt.Sprintf("* @param[in] %s - %s buffer of %s", cParams[1].ParamName, param.ParamClass, param.ParamDescription);

			case "class", "optionalclass":
				cParams[0].ParamType = cParamTypeName;
				cParams[0].ParamCallType = cParamTypeName;
				cParams[0].ParamName = "p" + param.ParamName;
				cParams[0].ParamComment = fmt.Sprintf("* @param[in] %s - %s", cParams[0].ParamName, param.ParamDescription);

			case "functiontype":
				cParams[0].ParamType = cParamTypeName;
				cParams[0].ParamCallType = cParamTypeName;
				cParams[0].ParamName = "p" + param.ParamName;
				cParams[0].ParamComment = fmt.Sprintf("* @param[in] %s - %s", cParams[0].ParamName, param.ParamDescription);
			default:
				return nil, fmt.Errorf ("invalid method parameter type \"%s\" for %s.%s (%s)", param.ParamType, className, methodName, param.ParamName);
		}
	
	case "out", "return":
	
		switch (param.ParamType) {
		
			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "bool", "single", "double", "pointer":
				cParams[0].ParamType = "ctypes.POINTER("+cParamTypeName+")";
				cParams[0].ParamCallType = cParamTypeName;
				cParams[0].ParamName = "p" + param.ParamName;
				cParams[0].ParamComment = fmt.Sprintf("* @param[out] %s - %s", cParams[0].ParamName, param.ParamDescription);
			
			case "enum":
				cParams[0].ParamType = "ctypes.POINTER(ctypes.c_int32)";
				cParams[0].ParamCallType = cParamTypeName;
				cParams[0].ParamName = "p" + param.ParamName;
				cParams[0].ParamComment = fmt.Sprintf("* @param[out] %s - %s", cParams[0].ParamName, param.ParamDescription);
			
			case "struct":
				cParams[0].ParamType = "ctypes.POINTER("+cParamTypeName+")";
				cParams[0].ParamCallType = cParamTypeName;
				cParams[0].ParamName = "p" + param.ParamName;
				cParams[0].ParamComment = fmt.Sprintf("* @param[out] %s - %s", cParams[0].ParamName, param.ParamDescription);
				
			case "basicarray", "structarray":
				cParams = make([]ctypesParameter,3)
				cParams[0].ParamType = "ctypes.c_uint64";
				cParams[0].ParamCallType = "ctypes.c_uint64";
				cParams[0].ParamName = "n" + param.ParamName + "Count";
				cParams[0].ParamComment = fmt.Sprintf("* @param[in] %s - Number of elements in buffer", cParams[0].ParamName);

				cParams[1].ParamType = "ctypes.POINTER(ctypes.c_uint64)";
				cParams[1].ParamCallType = "ctypes.c_uint64";
				cParams[1].ParamName = "n" + param.ParamName + "NeededCount";
				cParams[1].ParamComment = fmt.Sprintf("* @param[out] %s - will be filled with the count of the written elements, or needed buffer size.", cParams[1].ParamName);

				cParams[2].ParamType = "ctypes.POINTER("+ cParamTypeName+")";
				cParams[2].ParamCallType = cParamTypeName;
				cParams[2].ParamName = "p" + param.ParamName + "Buffer";
				cParams[2].ParamComment = fmt.Sprintf("* @param[out] %s - %s buffer of %s", cParams[2].ParamName, param.ParamClass, param.ParamDescription);

			case "string":
				cParams = make([]ctypesParameter,3)
				cParams[0].ParamType = "ctypes.c_uint64";
				cParams[0].ParamCallType = "ctypes.c_uint64";
				cParams[0].ParamName = "n" + param.ParamName + "BufferSize";
				cParams[0].ParamComment = fmt.Sprintf("* @param[in] %s - size of the buffer (including trailing 0)", cParams[0].ParamName);

				cParams[1].ParamType = "ctypes.POINTER(ctypes.c_uint64)";
				cParams[1].ParamCallType = "ctypes.c_uint64"
				cParams[1].ParamName = "n" + param.ParamName + "NeededChars";
				cParams[1].ParamComment = fmt.Sprintf("* @param[out] %s - will be filled with the count of the written bytes, or needed buffer size.", cParams[1].ParamName);

				cParams[2].ParamType = cParamTypeName
				cParams[2].ParamCallType = cParamTypeName;
				cParams[2].ParamName = "p" + param.ParamName + "Buffer";
				cParams[2].ParamComment = fmt.Sprintf("* @param[out] %s - %s buffer of %s, may be NULL", cParams[2].ParamName, param.ParamClass, param.ParamDescription);

			case "class", "optionalclass":
				cParams[0].ParamType = "ctypes.POINTER("+cParamTypeName +")";
				cParams[0].ParamCallType = cParamTypeName;
				cParams[0].ParamName = "p" + param.ParamName;
				cParams[0].ParamComment = fmt.Sprintf("* @param[out] %s - %s", cParams[0].ParamName, param.ParamDescription);
	
			default:
				return nil, fmt.Errorf ("invalid method parameter type \"%s\" for %s.%s (%s)", param.ParamType, className, methodName, param.ParamName);
		}
		
	default:
		return nil, fmt.Errorf ("invalid method parameter passing \"%s\" for %s.%s (%s)", param.ParamPass, className, methodName, param.ParamName);
	}

	return cParams, nil;
}


func writePythonClass(component ComponentDefinition, class ComponentDefinitionClass, w LanguageWriter, NameSpace string) error {
	pythonBaseClassName := fmt.Sprintf("%s", component.Global.BaseClassName)

	w.Writeln("''' Class Implementation for %s",  class.ClassName)
	w.Writeln("'''")
	
	parentClass := ""
	if (!component.isBaseClass(class)) {
		if (class.ParentClass != "") {
			parentClass = fmt.Sprintf("%s", class.ParentClass)
		} else {
			parentClass = pythonBaseClassName
		}
		w.Writeln("class %s(%s):", class.ClassName, parentClass)
		w.Writeln("  def __init__(self, handle, wrapper):")
		w.Writeln("    %s.__init__(self, handle, wrapper)", parentClass)

	} else {
		w.Writeln("class %s:", class.ClassName)
		w.Writeln("  def __init__(self, handle, wrapper):")
		w.Writeln("    if not handle or not wrapper:")
		w.Writeln("      raise E%sException(ErrorCodes.INVALIDPARAM)", NameSpace)
		w.Writeln("    self._handle = handle")
		w.Writeln("    self._wrapper = wrapper")
		w.Writeln("  ")
		w.Writeln("  def __del__(self):")
		w.Writeln("    self._wrapper.%s(self)", component.Global.ReleaseMethod)
	}

	for i:=0; i<len(class.Methods); i++ {
		err := writeMethod(class.Methods[i], w, NameSpace, class.ClassName, []string{}, false)
		if (err != nil) {
			return err
		}
	}
	return nil
}

func writeMethod(method ComponentDefinitionMethod, w LanguageWriter, NameSpace string, ClassName string, implementationLines []string, isGlobal bool) error {
	preCallLines := []string{}
	checkCallLines := []string{}
	postCallLines := []string{}
	
	retVals := ""
	pythonInParams := ""

	wrapperReference := "self"
	selfReference := "None"
	if (!isGlobal) {
		wrapperReference = "self._wrapper"
		selfReference = "self"
	}
	cArguments := ""
	if (!isGlobal) {
		cArguments = "self._handle"
	}
	cCheckArguments := ""
	if (!isGlobal) {
		cCheckArguments = "self._handle"
	}
	doCheckCall := false
	for k:=0; k<len(method.Params); k++ {
		param := method.Params[k]
		
		cParams, err := generateCTypesParameter(param, ClassName, method.MethodName, NameSpace)
		if (err != nil) {
			return err
		}

		switch param.ParamPass {
			case  "out", "return":
			if (cArguments != "") {
				cArguments = cArguments + ", ";
			}
			if (cCheckArguments != "") {
				cCheckArguments = cCheckArguments + ", "
			}
			switch param.ParamType {
			case "class", "optionalclass": {
				if (retVals != "") {
					retVals = retVals + ", ";
				}
				preCallLines = append(preCallLines, fmt.Sprintf("%sHandle = %s()", param.ParamName, cParams[0].ParamCallType))
				newArgument := fmt.Sprintf("%sHandle", param.ParamName)
				cArguments = cArguments + newArgument
				cCheckArguments = cCheckArguments + newArgument

				theWrapperReference := wrapperReference
				subNameSpace, paramClassName, _ := decomposeParamClassName(param.ParamClass)
				if len(subNameSpace) > 0 {
					theWrapperReference = theWrapperReference + "._" + subNameSpace + "Wrapper"
					subNameSpace = subNameSpace + "."
				}
				postCallLines = append(postCallLines, fmt.Sprintf("if %sHandle:", param.ParamName))
				postCallLines = append(postCallLines,
					fmt.Sprintf("  %sObject = %s%s(%sHandle, %s)",
					param.ParamName, subNameSpace, paramClassName, param.ParamName, theWrapperReference))
				postCallLines = append(postCallLines, fmt.Sprintf("else:"))
				if (param.ParamType == "optionalclass") {
					postCallLines = append(postCallLines, fmt.Sprintf("  %sObject = None", param.ParamName))
				} else {
					postCallLines = append(postCallLines, fmt.Sprintf("  raise E%sException(ErrorCodes.INVALIDCAST, 'Invalid return/output value')", NameSpace))
				}

				
				retVals = retVals + fmt.Sprintf("%sObject", param.ParamName)
			}
			case "string": {
				if (retVals != "") {
					retVals = retVals + ", ";
				}
				preCallLines = append(preCallLines, fmt.Sprintf("%s = %s(0)", cParams[0].ParamName, cParams[0].ParamCallType))
				preCallLines = append(preCallLines, fmt.Sprintf("%s = %s(0)", cParams[1].ParamName, cParams[1].ParamCallType))
				preCallLines = append(preCallLines, fmt.Sprintf("%s = %s(None)", cParams[2].ParamName, cParams[2].ParamCallType))
				
				cCheckArguments = cCheckArguments + cParams[0].ParamName + ", " + cParams[1].ParamName + ", " + cParams[2].ParamName
				checkCallLines = append(checkCallLines, fmt.Sprintf("%s = %s(%s.value)", cParams[0].ParamName, cParams[0].ParamCallType, cParams[1].ParamName))
				checkCallLines = append(checkCallLines, fmt.Sprintf("%s = (ctypes.c_char * (%s.value))()", cParams[2].ParamName, cParams[1].ParamName))
				doCheckCall = true
				cArguments = cArguments + cParams[0].ParamName + ", " + cParams[1].ParamName + ", " + cParams[2].ParamName
				retVals = retVals + cParams[2].ParamName + ".value.decode()"
			}
			case "basicarray": {
				if (retVals != "") {
					retVals = retVals + ", ";
				}
				preCallLines = append(preCallLines, fmt.Sprintf("%s = %s(0)", cParams[0].ParamName, cParams[0].ParamCallType))
				preCallLines = append(preCallLines, fmt.Sprintf("%s = %s(0)", cParams[1].ParamName, cParams[1].ParamCallType))
				preCallLines = append(preCallLines, fmt.Sprintf("%s = (%s*0)()", cParams[2].ParamName, cParams[2].ParamCallType))

				cCheckArguments = cCheckArguments + cParams[0].ParamName + ", " + cParams[1].ParamName + ", " + cParams[2].ParamName
				checkCallLines = append(checkCallLines, fmt.Sprintf("%s = %s(%s.value)", cParams[0].ParamName, cParams[0].ParamCallType, cParams[1].ParamName))
				checkCallLines = append(checkCallLines, fmt.Sprintf("%s = (%s * %s.value)()", cParams[2].ParamName, cParams[2].ParamCallType, cParams[1].ParamName))
				doCheckCall = true

				cArguments = cArguments + cParams[0].ParamName + ", " + cParams[1].ParamName + ", " + cParams[2].ParamName
				retVals = retVals + fmt.Sprintf("[%s[i] for i in range(%s.value)]", cParams[2].ParamName, cParams[1].ParamName)
			}
			case "structarray": {
				if (retVals != "") {
					retVals = retVals + ", ";
				}
				preCallLines = append(preCallLines, fmt.Sprintf("%s = %s(0)", cParams[0].ParamName, cParams[0].ParamCallType))
				preCallLines = append(preCallLines, fmt.Sprintf("%s = %s(0)", cParams[1].ParamName, cParams[1].ParamCallType))
				preCallLines = append(preCallLines, fmt.Sprintf("%s = (%s*0)()", cParams[2].ParamName, cParams[2].ParamCallType))

				cCheckArguments = cCheckArguments + cParams[0].ParamName + ", " + cParams[1].ParamName + ", " + cParams[2].ParamName
				checkCallLines = append(checkCallLines, fmt.Sprintf("%s = %s(%s.value)", cParams[0].ParamName, cParams[0].ParamCallType, cParams[1].ParamName))
				checkCallLines = append(checkCallLines, fmt.Sprintf("%s = (%s * %s.value)()", cParams[2].ParamName, cParams[2].ParamCallType, cParams[1].ParamName))
				doCheckCall = true

				cArguments = cArguments + cParams[0].ParamName + ", " + cParams[1].ParamName + ", " + cParams[2].ParamName
				retVals = retVals + fmt.Sprintf("[%s[i] for i in range(%s.value)]", cParams[2].ParamName, cParams[1].ParamName)
			}
			case "enum": {
				if (retVals != "") {
					retVals = retVals + ", ";
				}
				preCallLines = append(preCallLines, fmt.Sprintf("%s = ctypes.c_int32()", cParams[0].ParamName))
				cArguments = cArguments + cParams[0].ParamName
				cCheckArguments = cCheckArguments  + cParams[0].ParamName
				retVals = retVals + fmt.Sprintf("%s(%s.value)", cParams[0].ParamCallType, cParams[0].ParamName)
			}
			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "single", "double", "bool", "pointer":
				if (retVals != "") {
					retVals = retVals + ", ";
				}
				preCallLines = append(preCallLines, fmt.Sprintf("%s = %s()", cParams[0].ParamName, cParams[0].ParamCallType))
				cArguments = cArguments + cParams[0].ParamName
				cCheckArguments = cCheckArguments  + cParams[0].ParamName
				retVals = retVals + cParams[0].ParamName + ".value"
			case "struct": {
				if (retVals != "") {
					retVals = retVals + ", ";
				}
				preCallLines = append(preCallLines, fmt.Sprintf("%s = %s()", cParams[0].ParamName, cParams[0].ParamCallType))
				cArguments = cArguments + cParams[0].ParamName
				cCheckArguments = cCheckArguments  + cParams[0].ParamName
				retVals = retVals + fmt.Sprintf("%s", cParams[0].ParamName)
			}
			default:
				return fmt.Errorf("Invalid parameter of type \"%s\" used as pass=\"%s\"", param.ParamType, param.ParamPass)
			}
		case "in":
			if (cArguments != "") {
				cArguments = cArguments + ", ";
			}
			if (cCheckArguments != "") {
				cCheckArguments = cCheckArguments + ", "
			}
			pythonInParams = pythonInParams + ", ";
			
			switch param.ParamType {
			case "uint8", "uint16", "uint32", "uint64", "int8", "int16", "int32", "int64", "single", "double", "bool", "pointer":
				preCallLines = append(preCallLines, fmt.Sprintf("%s = %s(%s)", cParams[0].ParamName, cParams[0].ParamCallType, param.ParamName))
				pythonInParams = pythonInParams + param.ParamName
				cArguments = cArguments + cParams[0].ParamName
				cCheckArguments = cCheckArguments  + cParams[0].ParamName
			case "enum": {
				pythonInParams = pythonInParams + param.ParamName
				cArguments = cArguments + param.ParamName
				cCheckArguments = cCheckArguments  + param.ParamName
			}
			case "string": {
				preCallLines = append(preCallLines, fmt.Sprintf("%s = %s(str.encode(%s))", cParams[0].ParamName, cParams[0].ParamCallType, param.ParamName))
				pythonInParams = pythonInParams + param.ParamName
				cArguments = cArguments + cParams[0].ParamName
				cCheckArguments = cCheckArguments  + cParams[0].ParamName
			}
			case "basicarray": {
				preCallLines = append(preCallLines, fmt.Sprintf("%s = %s(len(%s))", cParams[0].ParamName, cParams[0].ParamCallType, param.ParamName))
				preCallLines = append(preCallLines, fmt.Sprintf("%s = (%s*len(%s))(*%s)", cParams[1].ParamName, cParams[1].ParamCallType, param.ParamName, param.ParamName))
				pythonInParams = pythonInParams + param.ParamName
				cArguments = cArguments + cParams[0].ParamName + ", " + cParams[1].ParamName
				cCheckArguments = cCheckArguments  + cParams[0].ParamName + ", " + cParams[1].ParamName
			}
			case "structarray": {
				preCallLines = append(preCallLines, fmt.Sprintf("%s = %s(len(%s))", cParams[0].ParamName, cParams[0].ParamCallType, param.ParamName))
				preCallLines = append(preCallLines, fmt.Sprintf("%s = (%s*len(%s))(*%s)", cParams[1].ParamName, cParams[1].ParamCallType, param.ParamName, param.ParamName))
				pythonInParams = pythonInParams + param.ParamName
				cArguments = cArguments + cParams[0].ParamName + ", " + cParams[1].ParamName
				cCheckArguments = cCheckArguments  + cParams[0].ParamName + ", " + cParams[1].ParamName
			}
			case "struct": {
				pythonInParams = pythonInParams + param.ParamName
				cArguments = cArguments + param.ParamName
				cCheckArguments = cCheckArguments  + param.ParamName
			}
			case "class", "optionalclass": {
				pythonInParams = pythonInParams + param.ParamName + "Object"
				preCallLines = append(preCallLines, fmt.Sprintf("%sHandle = None", param.ParamName))
				preCallLines = append(preCallLines, fmt.Sprintf("if %sObject:", param.ParamName))
				preCallLines = append(preCallLines, fmt.Sprintf("  %sHandle = %sObject._handle", param.ParamName, param.ParamName))
				if (param.ParamType == "class") {
					preCallLines = append(preCallLines, fmt.Sprintf("else:"))
					preCallLines = append(preCallLines, fmt.Sprintf("  raise E%sException(ErrorCodes.INVALIDPARAM, 'Invalid return/output value')", NameSpace))
				}
				cArguments = cArguments + param.ParamName + "Handle"
				cCheckArguments = cCheckArguments  + param.ParamName + "Handle"
			}
			case "functiontype": {
				pythonInParams = pythonInParams + param.ParamName + "Func"
				cArguments = cArguments + param.ParamName + "Func"
				cCheckArguments = cCheckArguments  + param.ParamName + "Func"
			}
			default:
				return fmt.Errorf("Invalid parameter of type \"%s\" used as pass=\"%s\"", param.ParamType, param.ParamPass)
			}

		}
	}
	
	exportName := GetCExportName(NameSpace, ClassName, method, isGlobal)
	
	w.Writeln("  def %s(self%s):", method.MethodName, pythonInParams)
	w.Writelns("    ", preCallLines)
	if (doCheckCall) {
		w.Writeln("    %s.checkError(%s, %s.lib.%s(%s))", wrapperReference, selfReference, wrapperReference, exportName, cCheckArguments)
		w.Writelns("    ", checkCallLines)
	}
	w.Writeln("    %s.checkError(%s, %s.lib.%s(%s))", wrapperReference, selfReference, wrapperReference, exportName, cArguments)
	w.Writelns("    ", postCallLines)
	w.Writeln("    ")

	if len(implementationLines) > 0 {
		w.Writelns("    ", implementationLines)
		w.Writeln("    ")
	}

	if len(retVals) > 0 {
		w.Writeln("    return %s", retVals)
	}
	w.Writeln("  ")

	return nil
}

func buildDynamicPythonExample(componentdefinition ComponentDefinition, w LanguageWriter, outputFolder string) error {
	NameSpace := componentdefinition.NameSpace
	BaseName := componentdefinition.BaseName

	w.Writeln("")
	w.Writeln("import os")
	w.Writeln("import sys")
	w.Writeln("sys.path.append(os.path.join(os.path.dirname(os.path.realpath(__file__)), \"..\", \"..\", \"Bindings\", \"Python\"))")
	w.Writeln("import %s",NameSpace )
	w.Writeln("")
	w.Writeln("")

	w.Writeln("def main():")
	w.Writeln("  libpath = '' # TODO add the location of the shared library binary here")
	w.Writeln("  wrapper = %s.Wrapper(libraryName = os.path.join(libpath, \"%s\"))", NameSpace, BaseName)
	w.Writeln("  ")
	w.Writeln("  major, minor, micro = wrapper.%s()", componentdefinition.Global.VersionMethod)
	w.Writeln("  print(\"%s version: {:d}.{:d}.{:d}\".format(major, minor, micro), end=\"\")", NameSpace)
	if len(componentdefinition.Global.PrereleaseMethod)>0 {
		w.Writeln("  hasInfo, prereleaseinfo = wrapper.%s()", componentdefinition.Global.PrereleaseMethod)
		w.Writeln("  if hasInfo:")
		w.Writeln("    print(\"-\"+prereleaseinfo, end=\"\")")
	}
	if len(componentdefinition.Global.BuildinfoMethod)>0 {
		w.Writeln("  hasInfo, buildinfo = wrapper.%s()", componentdefinition.Global.BuildinfoMethod)
		w.Writeln("  if hasInfo:")
		w.Writeln("    print(\"+\"+buildinfo, end=\"\")")
	}
	w.Writeln("  print(\"\")")
	w.Writeln("")
	w.Writeln("")

	w.Writeln("if __name__ == \"__main__\":")
	w.Writeln("  try:")
	w.Writeln("    main()")
	w.Writeln("  except %s.E%sException as e:", NameSpace, NameSpace)
	w.Writeln("    print(e)")

	return nil
}
