/*
 * This file is part of the KubeVirt project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Copyright The KubeVirt Authors.
 */

package apply_test

import (
	. "github.com/onsi/ginkgo/v2"
	. "github.com/onsi/gomega"

	k8sfield "k8s.io/apimachinery/pkg/util/validation/field"

	virtv1 "kubevirt.io/api/core/v1"
	v1beta1 "kubevirt.io/api/instancetype/v1beta1"

	"kubevirt.io/kubevirt/pkg/instancetype/apply"
	"kubevirt.io/kubevirt/pkg/libvmi"
	"kubevirt.io/kubevirt/pkg/pointer"
)

var _ = Describe("Preference.PreferredTerminationGracePeriodSeconds", func() {
	var (
		vmi              *virtv1.VirtualMachineInstance
		instancetypeSpec *v1beta1.VirtualMachineInstancetypeSpec
		preferenceSpec   *v1beta1.VirtualMachinePreferenceSpec

		field      = k8sfield.NewPath("spec", "template", "spec")
		vmiApplier = apply.NewVMIApplier()
	)

	BeforeEach(func() {
		vmi = libvmi.New()
		// delete spec.TerminationGracePeriodSeconds set in VM
		vmi.Spec.TerminationGracePeriodSeconds = nil
	})

	It("should apply to VMI", func() {
		preferenceSpec = &v1beta1.VirtualMachinePreferenceSpec{
			PreferredTerminationGracePeriodSeconds: pointer.P(int64(180)),
		}
		Expect(vmiApplier.ApplyToVMI(field, instancetypeSpec, preferenceSpec, &vmi.Spec, &vmi.ObjectMeta)).To(Succeed())
		Expect(vmi.Spec.TerminationGracePeriodSeconds).To(HaveValue(Equal(*preferenceSpec.PreferredTerminationGracePeriodSeconds)))
	})

	It("should not overwrite user defined value", func() {
		const userDefinedValue = int64(100)
		vmi.Spec.TerminationGracePeriodSeconds = pointer.P(userDefinedValue)
		preferenceSpec = &v1beta1.VirtualMachinePreferenceSpec{
			PreferredTerminationGracePeriodSeconds: pointer.P(int64(180)),
		}
		Expect(vmiApplier.ApplyToVMI(field, instancetypeSpec, preferenceSpec, &vmi.Spec, &vmi.ObjectMeta)).To(Succeed())
		Expect(vmi.Spec.TerminationGracePeriodSeconds).To(HaveValue(Equal(userDefinedValue)))
	})
})
