using System.Globalization;
using Jint.Native.Generator;
using Jint.Native.Object;
using Jint.Native.RegExp;
using Jint.Runtime;

namespace Jint.Native.Iterator;

internal abstract class IteratorInstance : ObjectInstance
{
    protected IteratorInstance(Engine engine) : base(engine)
    {
        _prototype = engine.Realm.Intrinsics.ArrayIteratorPrototype;
    }

    public override object ToObject()
    {
        Throw.NotImplementedException();
        return null;
    }

    public abstract bool TryIteratorStep(out ObjectInstance nextItem);

    public virtual void Close(CompletionType completion)
    {
    }

    /// <summary>
    /// Gets the underlying iterator object instance.
    /// For object iterators, this is the wrapped object. For built-in iterators, this is self.
    /// Used by yield* to call methods like "return" and "throw" on the iterator.
    /// </summary>
    public virtual ObjectInstance Instance => this;

    /// <summary>
    /// https://tc39.es/ecma262/#sec-createiterresultobject
    /// </summary>
    private IteratorResult CreateIterResultObject(JsValue value, bool done)
    {
        return new IteratorResult(_engine, value, JsBoolean.Create(done));
    }

    internal sealed class ObjectIterator : IteratorInstance
    {
        private readonly ObjectInstance _target;
        private readonly ICallable? _nextMethod;

        public override ObjectInstance Instance => _target;

        public ObjectIterator(ObjectInstance target) : base(target.Engine)
        {
            _target = target;
            // Don't check for 'next' method here - it's only required when actually iterating
            // This allows iterators with only 'return' method to be created (e.g., for closing)
            if (target.Get(CommonProperties.Next) is ICallable callable)
            {
                _nextMethod = callable;
            }
        }

        public override bool TryIteratorStep(out ObjectInstance result)
        {
            result = IteratorNext();

            var done = result.Get(CommonProperties.Done);
            if (!done.IsUndefined() && TypeConverter.ToBoolean(done))
            {
                return false;
            }

            return true;
        }

        private ObjectInstance IteratorNext()
        {
            // Check for 'next' method when actually trying to iterate
            if (_nextMethod is null)
            {
                Throw.TypeError(_target.Engine.Realm, "Iterator does not have a next method");
                return null!;
            }

            var jsValue = _nextMethod.Call(_target, Arguments.Empty);
            var instance = jsValue as ObjectInstance;
            if (instance is null)
            {
                Throw.TypeError(_target.Engine.Realm, $"Iterator result {jsValue} is not an object");
            }

            return instance;
        }

        public override void Close(CompletionType completion)
        {
            var callable = _target.GetMethod(CommonProperties.Return);
            if (callable is null)
            {
                return;
            }

            JsValue innerResult;
            try
            {
                innerResult = callable.Call(_target, Arguments.Empty);
            }
            catch (JavaScriptException) when (completion == CompletionType.Throw)
            {
                return;
            }

            if (completion != CompletionType.Throw && !innerResult.IsObject())
            {
                Throw.TypeError(_target.Engine.Realm, "Iterator returned non-object");
            }
        }
    }

    internal sealed class StringIterator : IteratorInstance
    {
        private readonly TextElementEnumerator _iterator;

        public StringIterator(Engine engine, string str) : base(engine)
        {
            _iterator = StringInfo.GetTextElementEnumerator(str);
        }

        public override bool TryIteratorStep(out ObjectInstance nextItem)
        {
            if (_iterator.MoveNext())
            {
                nextItem = IteratorResult.CreateValueIteratorPosition(_engine, (string) _iterator.Current);
                return true;
            }

            nextItem = IteratorResult.CreateKeyValueIteratorPosition(_engine);
            return false;
        }
    }

    internal sealed class RegExpStringIterator : IteratorInstance
    {
        private readonly JsRegExp _iteratingRegExp;
        private readonly string _s;
        private readonly bool _global;
        private readonly bool _unicode;

        private bool _done;

        public RegExpStringIterator(Engine engine, ObjectInstance iteratingRegExp, string iteratedString, bool global, bool unicode) : base(engine)
        {
            var r = iteratingRegExp as JsRegExp;
            if (r is null)
            {
                Throw.TypeError(engine.Realm);
            }

            _iteratingRegExp = r;
            _s = iteratedString;
            _global = global;
            _unicode = unicode;
        }

        public override bool TryIteratorStep(out ObjectInstance nextItem)
        {
            if (_done)
            {
                nextItem = CreateIterResultObject(Undefined, true);
                return false;
            }

            var match = RegExpPrototype.RegExpExec(_iteratingRegExp, _s);
            if (match.IsNull())
            {
                _done = true;
                nextItem = CreateIterResultObject(Undefined, true);
                return false;
            }

            if (_global)
            {
                var macthStr = TypeConverter.ToString(match.Get(JsString.NumberZeroString));
                if (macthStr == "")
                {
                    var thisIndex = TypeConverter.ToLength(_iteratingRegExp.Get(JsRegExp.PropertyLastIndex));
                    var nextIndex = thisIndex + 1;
                    _iteratingRegExp.Set(JsRegExp.PropertyLastIndex, nextIndex, true);
                }
            }
            else
            {
                _done = true;
            }

            nextItem = CreateIterResultObject(match, false);
            return true;
        }
    }

    internal sealed class EnumerableIterator : IteratorInstance
    {
        private readonly IEnumerator<JsValue> _enumerable;

        public EnumerableIterator(Engine engine, IEnumerable<JsValue> obj) : base(engine)
        {
            _enumerable = obj.GetEnumerator();
        }

        public override bool TryIteratorStep(out ObjectInstance nextItem)
        {
            if (_enumerable.MoveNext())
            {
                nextItem = IteratorResult.CreateValueIteratorPosition(_engine, _enumerable.Current);
                return true;
            }

            nextItem = IteratorResult.CreateValueIteratorPosition(_engine, done: JsBoolean.True);
            return false;
        }
    }

    internal sealed class GeneratorIterator : IteratorInstance
    {
        private readonly GeneratorInstance _generator;

        public GeneratorIterator(Engine engine, GeneratorInstance generator) : base(engine)
        {
            _generator = generator;
        }

        public override bool TryIteratorStep(out ObjectInstance nextItem)
        {
            nextItem = IteratorResult.CreateValueIteratorPosition(_engine, done: JsBoolean.True);
            return false;
        }
    }

}
