/*
  unhxd.c -- ascii (hexadecimal) -> binary converter
  Copyright (C) 1998,1999 Steffen Solyga <solyga@absinth.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 *	$Id: unhxd.c,v 1.5 2004/04/08 21:01:41 solyga Exp $
 */


#include	"unhxd.h"


int
display_help( char* pn ) {
  fprintf( HELP_CHANNEL, "%s v%s (%s): ", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( HELP_CHANNEL, "Ascii (hex) -> binary conversion\n" );
  fprintf( HELP_CHANNEL, "Flowers & bug reports to %s.\n", MY_EMAIL_ADDRESS );
  fprintf( HELP_CHANNEL, "Usage: %s [options] [in-file [out-file]]\n", pn );
  fprintf( HELP_CHANNEL, "switches:\n" );
  fprintf( HELP_CHANNEL, "  -h\t write this info to %s and exit sucessfully\n",HELP_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "  -v\t raise verbosity level on %s\n", VERBOSE_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "  -V\t write version and compilation info to %s and exit sucessfully\n",VERSION_CHANNEL==stdout?"stdout":"stderr" );
  return( 0 );
}


int
display_version( char* pn ) {
  fprintf( HELP_CHANNEL, "%s v%s (%s)\n", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( VERSION_CHANNEL, "  BEGIN_CHAR         :\t%c\n", BEGIN_CHAR );
  fprintf( VERSION_CHANNEL, "  END_CHAR           :\t%c\n", END_CHAR );
  fprintf( VERSION_CHANNEL, "  REMARKER           :\t%c\n", REMARKER );
  fprintf( VERSION_CHANNEL, "  HAVE_LONGLONG      :\t" );
#ifdef  HAVE_LONGLONG
  fprintf( VERSION_CHANNEL, "defined\n" );
#else
  fprintf( VERSION_CHANNEL, "undefined\n" );
#endif
  return( 0 );
}


char
*ss_str_dsc( char* str1, char* str2) {
/* returns pointer to first char of str1 not contained in str2 */
/* 1997-02-09, 1999-07-10 */
  char *p1, *p2;
  for ( p1=str1; ; p1++ ) {
    if ( *p1 == '\0' ) return( p1 );
    for ( p2=str2; ; p2++ ) {
      if ( *p2 == '\0' ) return( p1 );
      if ( *p2 == *p1 ) break;
    }
  }
}


ssize_t
my_read( int fd, void* buf, size_t nbtr ) {
/* returns number of bytes read or -1 on error */
/* like read(2) but tnbr<nbtr only if eof reached */
  ssize_t nbr;
  ssize_t tnbr= 0;
  size_t rem= nbtr;
  unsigned char *p= (unsigned char*)buf;
  do {
    if( (nbr=read(fd,p+tnbr,rem)) == -1 ) return( -1 );
    tnbr+= nbr;
    rem= nbtr - tnbr;
  }
  while( nbr>0 && rem>0 );
  return( tnbr );
}


ssize_t
my_write( int fd, void *buf, size_t count ) {
/* write exactly count bytes to fd (else error) */
  ssize_t nbw;
  ssize_t tnbw= 0;
  size_t rem= count;
  unsigned char *p= (unsigned char*)buf;
  do {
    if( (nbw= write( fd, p+tnbw, rem )) == -1 ) return( -1 );
    tnbw+= nbw;
    rem-= nbw;
  } while( nbw>0 && rem>0 );
  return( tnbw );
}


int
main( int argc, char** argv) {
/*
SOLYGA --------------------
SOLYGA main( int argc, char** argv) unhxd
SOLYGA hexadecimal undump utility

SOLYGA started      : 1996/11/11 @beast
*/
  int retval= RETVAL_OK;
  int c;				/* option char */
  int verbose= 0;
/*
  state takes one of 3 states:
  STATE_WAIT    - wait_for_BEGIN_CHAR
  STATE_CONVERT - (try to) convert
  STATE_IGNORE  - ignore input until EOL
*/
  int state;
  int low;				/* flag: low 4-bits expected */
  unsigned char byte;
  char* in_fn= NULL;
  int in_fd= STDIN_FILENO;
  char* out_fn= NULL;
  int out_fd= STDOUT_FILENO;
  char ibuf[IN_BUFFER_SIZE];
  unsigned char obuf[OUT_BUFFER_SIZE];
  ssize_t nbr;				/* number of bytes read */
  ssize_t nbw;				/* number of bytes written */
  size_t nbtr;				/* number of bytes to read */
  size_t nbtw;				/* number of bytes to write */
  OFF_T tnbr;				/* total number of bytes read */
  OFF_T tnbw;				/* total number of bytes written */
  OFF_T line;				/* line number in [1,inf) */
  OFF_T col;				/* column number in [1,inf] */
#if defined OS_TYPE_AIX
  extern int errno;
  extern char* sys_errlist[];
#endif


  /* process options */
  while( (c=getopt(argc,argv,"hvV")) != EOF )
    switch( c ) {
      case 'h': /* display help */
        display_help( *argv );
        retval= RETVAL_OK; goto DIE_NOW;
      case 'v': /* raise verbosity level */
        verbose++;
        break;
      case 'V': /* display version */
        display_version( *argv );
        retval= RETVAL_OK; goto DIE_NOW;
      case '?': /* refer to -h */
        fprintf( ERROR_CHANNEL, "%s: Try `%s -h' for more information.\n",
                 *argv, *argv );
        retval= RETVAL_ERROR; goto DIE_NOW;
      default: /* program error */
        fprintf( ERROR_CHANNEL,
                 "%s: Bug detected! Please send a mail to %s.\n",
                 *argv, MY_EMAIL_ADDRESS );
        retval= RETVAL_BUG; goto DIE_NOW;
    }


  /* open input and output */
  if( argc>optind && strcmp(argv[optind],"-")!=0 ) {
    in_fn= argv[optind];
    if( (in_fd=open(in_fn,O_RDONLY)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot open `%s' read-only. %s.\n",
               *argv, in_fn, strerror(errno) );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
  }
  if( argc>optind+1 && strcmp(argv[optind+1],"-")!=0 ) {
    out_fn= argv[optind+1];
    if( (out_fd=open(out_fn,O_CREAT|O_WRONLY|O_TRUNC,PERMISSIONS)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot open `%s' write-only. %s.\n",
               *argv, out_fn, strerror(errno) );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
  }


  /* undump */
  tnbr= 0; tnbw= 0;
  nbtr= IN_BUFFER_SIZE;
  state= STATE_WAIT;
  low= 0;
  line= 1; col= 1;
  while ( 1 ) { /* start: process one input buffer */
    long i;
    /* fill input buffer (read) */
    if( (nbr=my_read(in_fd,ibuf,nbtr)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot read from ", *argv );
      if( in_fd == STDIN_FILENO ) fprintf( ERROR_CHANNEL, "stdin" );
      else fprintf( ERROR_CHANNEL, "`%s'", in_fn );
#ifdef  HAVE_LONGLONG
      fprintf( ERROR_CHANNEL, "at offset 0x%08Lx", tnbr );
#else
      fprintf( ERROR_CHANNEL, "at offset 0x%08lx", tnbr );
#endif
      fprintf( ERROR_CHANNEL, ". %s.\n", strerror(errno) );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    nbtw= 0;
    /* fill output buffer */
    for( i=0; i<nbr; i++ ) {
      switch( state ) {
        case STATE_WAIT:
          if( ibuf[i] == BEGIN_CHAR ) {
            state= STATE_CONVERT;
          }
          else if( ibuf[i] == END_CHAR || ibuf[i] == REMARKER ) {
            state= STATE_IGNORE;
          }
          else if( ibuf[i] == '\n' && verbose >= 2 ) {
#ifdef	HAVE_LONGLONG
            fprintf( VERBOSE_CHANNEL, "%s: Line %Ld ignored.\n", *argv, line );
#else
            fprintf( VERBOSE_CHANNEL, "%s: Line %ld ignored.\n", *argv, line );
#endif
          }
          break;
        case STATE_CONVERT:
          if( ibuf[i] == END_CHAR || ibuf[i] == REMARKER || ibuf[i] == '\n') {
            if( low ) {
#ifdef	HAVE_LONGLONG
              fprintf( ERROR_CHANNEL,
                       "%s: Runaway halfbyte at line %Ld column %Ld.\n",
#else
              fprintf( ERROR_CHANNEL,
                       "%s: Runaway halfbyte at line %ld column %ld.\n",
#endif
                       *argv, line, col );
              retval= RETVAL_ERROR; goto DIE_NOW;
            }
            state= STATE_IGNORE;
            if( ibuf[i] == '\n' ) {
              state= STATE_WAIT;
              line++; col= 0;
            }
            break;
          }
          if( ibuf[i] == ' ' || ibuf[i] == '\t' )
            break;
          if( !IS_HEX_DIGIT(ibuf[i]) ) {
#ifdef	HAVE_LONGLONG
            fprintf( ERROR_CHANNEL,
                     "%s: Bad hexadecimal digit `%c' at line %Ld column %Ld.\n",
                     *argv, ibuf[i], line, col );
#else
            fprintf( ERROR_CHANNEL,
                     "%s: Bad hexadecimal digit `%c' at line %ld column %ld.\n",
                     *argv, ibuf[i], line, col );
#endif
            retval= RETVAL_ERROR; goto DIE_NOW;
          }
          if( low ) { /* lower 4-bits */
            obuf[nbtw++]= byte | CHAR2BYTE(ibuf[i]);
          }
          else { /* higher 4-bits */
            byte= 0x00 | CHAR2BYTE(ibuf[i]) << 4;
          }
          FLIP( low );
          break;
        case STATE_IGNORE:
          if( ibuf[i] == '\n' )
            state= STATE_WAIT;
            line++; col= 0;
          break;
        default: /* program error: unknown state */
          fprintf( ERROR_CHANNEL,
                   "%s: Bug detected! Please send a mail to %s.\n",
                   *argv, MY_EMAIL_ADDRESS );
          retval= RETVAL_BUG; goto DIE_NOW;
      }
      col++;
    }
    /* write output */
    if( (nbw=my_write(out_fd,obuf,nbtw)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot write to ", *argv );
      if( out_fd == STDOUT_FILENO ) fprintf( ERROR_CHANNEL, "stdout" );
      else fprintf( ERROR_CHANNEL, "`%s'", in_fn );
      fprintf( ERROR_CHANNEL, ". %s.\n", strerror(errno) );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    tnbr+= nbr;
    tnbw+= nbw;
    if( nbr < nbtr ) break;
  } /* end: process one input buffer */


  /* close input and output */
  if( out_fd!=STDOUT_FILENO && close(out_fd)==-1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot close `%s'. %s.\n",
             *argv, out_fn, strerror(errno) );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }
  if( in_fd!=STDIN_FILENO && close(in_fd)==-1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot close `%s'. %s.\n",
             *argv, in_fn, strerror(errno) );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }


  if( verbose )
#ifdef	HAVE_LONGLONG
    fprintf( VERBOSE_CHANNEL, "%s: %Ld/%Ld bytes read/written.\n",
             *argv, tnbr, tnbw );
#else
    fprintf( VERBOSE_CHANNEL, "%s: %ld/%ld bytes read/written.\n",
             *argv, tnbr, tnbw );
#endif

DIE_NOW:
  close( out_fd );
  close( in_fd );
  exit( retval );
}
