; Event Macros Library
;
; Copyright (C) 2022 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (coglio@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "ACL2")

(include-book "kestrel/std/system/pseudo-event-formp" :dir :system)
(include-book "std/util/define" :dir :system)
(include-book "xdoc/defxdoc-plus" :dir :system)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ event-macro-event-generators
  :parents (event-macros)
  :short "Utilities to generate events in event macros."
  :long
  (xdoc::topstring
   (xdoc::p
    "These utilities return two results:
     a local event form, and a non-local one.
     The assumption is that the event macro generates an @(tsee encapsulate)
     inside which the events generated by these utilities:
     thus, the local event is local to the encapsulate,
     while the non-local one is exported from the encapsulate.
     The local event includes proof hints, while the exported one doesn't:
     this way, the ACL2 history after the encapsulate is ``clean'',
     without hints that may refer to local theorems, in particular.")
   (xdoc::p
    "A caller of these utilities may use these utilities
     also to generate a local-only event,
     simply by ignoring the second result.
     In cases in which the local and exported events are the same
     (except for the @('(local ...)') wrapper,
     a caller of these utilities can ignore the first result."))
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define evmac-generate-defun ((name symbolp)
                              &key
                              ((formals symbol-listp) ':absent)
                              ((guard "A term.") 't)
                              ((body "A term.") ':absent)
                              ((verify-guards booleanp) ':absent)
                              ((enable booleanp) ':absent)
                              ((guard-hints true-listp) 'nil)
                              ((guard-simplify
                                (member-eq guard-simplify '(t :limited)))
                               't)
                              ((measure "A term.") 'nil)
                              ((well-founded-relation symbolp) 'nil)
                              ((hints true-listp) 'nil))
  :returns (mv (local-event pseudo-event-formp)
               (exported-event pseudo-event-formp))
  :short "Generate a @('defun') or @('defund') function definition
          with the specified attributes."
  (b* (((when (eq formals :absent))
        (raise "Internal error: :FORMALS must be always supplied.")
        (mv '(irrelevant) '(irrelevant)))
       ((when (eq body :absent))
        (raise "Internal error: :BODY must be always supplied.")
        (mv '(irrelevant) '(irrelevant)))
       ((when (eq verify-guards :absent))
        (raise "Internal error: :VERIFY-GUARDS must be always supplied.")
        (mv '(irrelevant) '(irrelevant)))
       ((when (eq enable :absent))
        (raise "Internal error: :ENABLE must be always supplied.")
        (mv '(irrelevant) '(irrelevant)))
       (macro (if enable 'defun 'defund))
       (measure (and measure (list :measure measure)))
       (well-founded-relation (and well-founded-relation
                                   (list :well-founded-relation
                                     well-founded-relation)))
       (hints (and measure hints (list :hints hints)))
       (guard (list :guard guard))
       (guard-hints (and guard-hints
                         verify-guards
                         (list :guard-hints guard-hints)))
       (guard-simplify (and (not (eq guard-simplify t))
                            (list :guard-simplify guard-simplify)))
       (verify-guards (list :verify-guards verify-guards))
       (local-event
        `(local
          (,macro ,name ,formals
                  (declare (xargs ,@measure
                                  ,@well-founded-relation
                                  ,@hints
                                  ,@guard
                                  ,@verify-guards
                                  ,@guard-hints
                                  ,@guard-simplify))
                  ,body)))
       (exported-event
        `(,macro ,name ,formals
                 (declare (xargs ,@measure
                                 ,@well-founded-relation
                                 ,@guard
                                 ,@verify-guards))
                 ,body)))
    (mv local-event exported-event)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define evmac-generate-defthm ((name symbolp)
                               &key
                               ((formula "A term.") ':absent)
                               ((rule-classes "Rule classes.") ':rewrite)
                               ((enable (or (booleanp enable)
                                            (eq enable :absent)))
                                ':absent)
                               ((hints true-listp) 'nil)
                               ((instructions true-listp) 'nil)
                                ((otf-flg booleanp) 'nil))
  :guard (implies (not rule-classes) (eq enable :absent))
  :returns (mv (local-event pseudo-event-formp)
               (exported-event pseudo-event-formp))
  :short "Generate a @('defthm') or @('defthmd') theorem
          with the specified attributes."
  (b* (((when (eq formula :absent))
        (raise "Internal error: :FORMULA must be always supplied.")
        (mv '(irrelevant) '(irrelevant)))
       ((unless (iff (eq rule-classes nil)
                     (eq enable :absent)))
        (prog2$
         (if rule-classes
             (raise "Internal error: :ENABLE must be supplied ~
                                      when :RULE-CLASSES is not NIL.")
           (raise "Internal error: :ENABLE must not be supplied ~
                                    when :RULE-CLASSES is NIL."))
         (mv '(irrelevant) '(irrelevant))))
       ((when (and hints instructions))
        (raise "Internal error: at most one of :HINTS and :INSTRUCTIONS ~
                                may be non-NIL.")
        (mv '(irrelevant) '(irrelevant)))
       (macro (if enable 'defthm 'defthmd))
       (rule-classes (and rule-classes (list :rule-classes rule-classes)))
       (hints (and hints (list :hints hints)))
       (instructions (and instructions (list :instructions instructions)))
       (otf-flg (and otf-flg (list :otf-flg t)))
       (local-event
        `(local
          (,macro ,name
                  ,formula
                  ,@rule-classes
                  ,@hints
                  ,@instructions
                  ,@otf-flg)))
       (exported-event
        `(,macro ,name
                 ,formula
                 ,@rule-classes)))
    (mv local-event exported-event)))
